var count = (function minus(BigNumber) {
    var start = +new Date(),
        log,
        error,
        undefined,
        passed = 0,
        total = 0,
        n = 'null',
        N = 'NaN',
        I = 'Infinity';

    if (typeof window === 'undefined') {
        log = console.log;
        error = console.error;
    } else {
        log = function (str) { document.body.innerHTML += str.replace('\n', '<br>') };
        error = function (str) { document.body.innerHTML += '<div style="color: red">' +
          str.replace('\n', '<br>') + '</div>' };
    }

    if (!BigNumber && typeof require === 'function') BigNumber = require('../bignumber');

    function assert(expected, actual) {
        total++;
        if (expected !== actual) {
           error('\n Test number: ' + total + ' failed');
           error(' Expected: ' + expected);
           error(' Actual:   ' + actual);
           //process.exit();
        }
        else {
            passed++;
            //log('\n Expected and actual: ' + actual);
        }
    }

    function T(minuend, subtrahend, expected) {
        assert(String(expected), String(new BigNumber(minuend).minus(subtrahend)));
        assert(String(expected), String(new BigNumber(minuend).minus(new BigNumber(subtrahend))));
    }

    function isMinusZero(n) {
        return n.toString() === '0' && n.s == -1;
    }

    log('\n Testing minus...');

    BigNumber.config({
        DECIMAL_PLACES: 20,
        ROUNDING_MODE: 4,
        ERRORS: false,
        RANGE: 1E9,
        EXPONENTIAL_AT: [-7, 21]
    });

    T(1, 0, 1);
    T(1, -0, 1);
    T(-1, 0, -1);
    T(-1, -0, -1);
    T(1, N, N);
    T(-1, N, N);
    T(1, I, -I);
    T(1, -I, I);
    T(-1, I, -I);
    T(-1, -I, I);
    T(0, 1, -1);
    T(0, -1, 1);
    T(-0, 1, -1);
    T(-0, -1, 1);

    /*
      IEEE 754 - 2008 section 6.3
      When the difference of two operands with like signs is exactly zero, the
      sign of that difference shall be +0 in all rounding-direction attributes
      except roundTowardNegative; under that attribute, the sign of an exact
      difference shall be −0.
      However, x + x = x −(−x) retains the same sign as x even when x is zero.
    */
    BigNumber.config( {ROUNDING_MODE: 3} );
    assert(false, isMinusZero(new BigNumber(0).minus(-0)));     //   0 - -0 =  0
    assert(true, isMinusZero(new BigNumber(-0).minus(0)));      //  -0 -  0 = -0
    assert(true, isMinusZero(new BigNumber(0).minus(0)));       //   0 -  0 = -0
    assert(true, isMinusZero(new BigNumber(-0).minus(-0)));     //  -0 - -0 = -0
    assert(true, isMinusZero(new BigNumber(1).minus(1)));       //   1 -  1 = -0
    assert(true, isMinusZero(new BigNumber(-1).minus(-1)));     //  -1 - -1 = -0

    BigNumber.config( {ROUNDING_MODE: 4} );
    assert(false, isMinusZero(new BigNumber(0).minus(-0)));     //   0 - -0 =  0
    assert(true, isMinusZero(new BigNumber(-0).minus(0)));      //  -0 -  0 = -0
    assert(false, isMinusZero(new BigNumber(0).minus(0)));      //   0 -  0 =  0
    assert(false, isMinusZero(new BigNumber(-0).minus(-0)));    //  -0 - -0 =  0
    assert(false, isMinusZero(new BigNumber(1).minus(1)));      //   1 -  1 =  0
    assert(false, isMinusZero(new BigNumber(-1).minus(-1)));    //  -1 - -1 =  0

    T(0, N, N);
    T(-0, N, N);
    T(0, I, -I);
    T(0, -I, I);
    T(-0, I, -I);
    T(-0, -I, I);
    T(N, 1, N);
    T(N, -1, N);
    T(N, 0, N);
    T(N, -0, N);
    T(N, N, N);
    T(N, I, N);
    T(N, -I, N);
    T(I, 1, I);
    T(I, -1, I);
    T(-I, 1, -I);
    T(-I, -1, -I);
    T(I, 0, I);
    T(I, -0, I);
    T(-I, 0, -I);
    T(-I, -0, -I);
    T(I, N, N);
    T(-I, N, N);
    T(I, I, N);
    T(I, -I, I);
    T(-I, I, -I);
    T(-I, -I, N);

    T(1, '0', '1');
    T(1, '1', '0');
    T(1, '-45', '46');
    T(1, '22', '-21');
    T(1, 0144, '-99');
    T(1, '0144', '-143');
    T(1, '6.1915', '-5.1915');
    T(1, '-1.02', '2.02');
    T(1, '0.09', '0.91');
    T(1, '-0.0001', '1.0001');
    T(1, '8e5', '-799999');
    T(1, '9E12', '-8999999999999');
    T(1, '1e-14', '0.99999999999999');
    T(1, '3.345E-9', '0.999999996655');
    T(1, '-345.43e+4', '3454301');
    T(1, '-94.12E+0', '95.12');
    T(1, '', N);
    T(1, '    ', N);
    T(1, '\t\t', N);
    T(1, 'ertgrt546', N);
    T(1, 'qweqwdewee', N);
    T(1, true, N);
    T(1, false, N);
    T(1, 'e 4.3', N);
    T(1, '4 .3', N);
    T(1, '4.0 01e', N);
    T(1, ' 4.001', '-3.001');
    T(1, '4.001 ', '-3.001');
    T(1, ' 4.001 ', '-3.001');
    T(1, '    4.001', '-3.001');
    T(1, ' 4.0 01', N);
    T(1, '4. 001', N);
    T(1, '4. 001 ', N);
    T(1, '  4.001e ', N);
    T(1, ' 4 .001 e ', N);
    T(1, undefined, N);
    T(1, null, N);
    T(1, Number.POSITIVE_INFINITY, -I);
    T(1, Number.NEGATIVE_INFINITY, I);
    T(1, new Date(2012, 11, 4), N);
    T(1, new Object(), N);
    T(1, function () {}, N);
    T('0', 0, '0');
    T(0, '+0', '0');
    T('0', '0', '0');
    T(3, -0, '3');
    T(9.654, 0, '9.654');
    T(0, '0.001', '-0.001');
    T(0, '111.1111111110000', '-111.111111111');
    T(-1, 1, '-2');
    T(-0.01, 0.01, '-0.02');
    T(54, -54, '108');
    T(9.99, '-9.99', '19.98');
    T('0.0000023432495704937', '-0.0000023432495704937', '0.0000046864991409874');
    T(NaN, NaN, N);
    T(NaN, N, N);
    T(N, NaN, N);
    T(N, 4, N);
    T(N, '4534534.45435435', N);
    T(N, 99999.999, N);
    T(Infinity, '354.345341', I);
    T(3, -I, I);
    T(-Infinity, -I, N);
    T(-I, -Infinity, N);
    T(I, '-999e999', I);
    T('1.21123e43', -I, I);
    T('-999.0', I, -I);
    T('657.342e-45', -I, I);
    T(I, 123, I);
    T(-0, I, -I);
    T(100, 100, '0');
    T(-999.99, '0.01', '-1000');
    T('10 ', 4, '6');
    T('03.333', -4, '7.333');
    T(-1, -0.1, '-0.9');
    T(43534.5435, '0.054645', '43534.488855');
    T('99999', '1', '99998');
    T('3e', 8, N);
    T('-3..0', 13, N);
    T('0 0', -0.4, N);
    T(' +3e0', 4, '-1');
    T(9.9806, '+ 1', N);
    T(' +2 0', '1e1', N);
    T('e3', 4, N);
    T(' ', 0, N);
    T(323, null, N);
    T(undefined, undefined, N);
    T('undefined', undefined, N);
    T(null, null, N);

    T('-0.00000020', '-1.5', '1.4999998');
    T('-5', '0', '-5');
    T('-2', '1.5', '-3.5');
    T('7', '-3', '10');
    T('0', '0', '0');
    T('0', '0', '0');
    T('0', '-9', '9');
    T('-15', '-7', '-8');
    T('-3', '0', '-3');
    T('3', '-5', '8');
    T('4', '-2', '6');
    T('2', '23', '-21');
    T('0', '0', '0');
    T('0.00000000022', '0', '2.2e-10');
    T('3', '-7', '10');
    T('0', '1.1', '-1.1');
    T('-2', '0', '-2');
    T('0', '-1', '1');
    T('2.3', '-3', '5.3');
    T('-3', '1', '-4');
    T('0.000040', '3', '-2.99996');
    T('0', '1', '-1');
    T('0', '2', '-2');
    T('0', '0', '0');
    T('3', '1.8', '1.2');
    T('0', '1.4', '-1.4');
    T('-1', '1', '-2');
    T('-3', '-1.0', '-2');
    T('1', '-4.5', '5.5');
    T('-1.8', '4', '-5.8');
    T('0', '-1.6', '1.6');
    T('1', '3.1', '-2.1');
    T('0.000000000000000000013', '-7', '7.000000000000000000013');
    T('0', '0', '0');
    T('-3', '2.4', '-5.4');
    T('-3.0', '2.2', '-5.2');
    T('0', '-2', '2');
    T('-1', '1.0', '-2');
    T('0', '5', '-5');
    T('5', '0', '5');
    T('-1', '0.0000014', '-1.0000014');
    T('3', '0', '3');
    T('0', '4.1', '-4.1');
    T('0', '-2.7', '2.7');
    T('-1', '0', '-1');
    T('-0.0000000000000030', '3', '-3.000000000000003');
    T('-2', '-1.8', '-0.2');
    T('0', '1', '-1');
    T('-1', '-1', '0');
    T('0', '-0.0000000011', '1.1e-9');
    T('6', '-15', '21');
    T('-2.2', '-1.6', '-0.6');
    T('0', '0', '0');
    T('0', '-5', '5');
    T('-6', '5', '-11');
    T('-3.1', '0.000000000000029', '-3.100000000000029');
    T('0', '3.6', '-3.6');
    T('0', '-7', '7');
    T('-5', '-1', '-4');
    T('0', '1.5', '-1.5');
    T('1', '2', '-1');
    T('-2', '0', '-2');
    T('-9', '-13', '4');
    T('-2', '0', '-2');
    T('1.1', '0', '1.1');
    T('8', '-0.000000000027', '8.000000000027');
    T('-0.000036', '-4', '3.999964');
    T('0', '5', '-5');
    T('0', '1', '-1');
    T('-7', '-5.9', '-1.1');
    T('-7', '-1', '-6');
    T('-1', '0', '-1');
    T('-0.000000036', '9', '-9.000000036');
    T('-6', '0', '-6');
    T('-9', '-6', '-3');
    T('2', '3.1', '-1.1');
    T('0', '-2', '2');
    T('0', '7', '-7');
    T('2.8', '3', '-0.2');
    T('-2', '0', '-2');
    T('0', '-3', '3');
    T('-3.7', '-5', '1.3');
    T('-1.3', '2.0', '-3.3');
    T('-4.7', '-3.0', '-1.7');
    T('-1.5', '0', '-1.5');
    T('4', '0', '4');
    T('-4', '0', '-4');
    T('-6', '-3', '-3');
    T('-1', '0', '-1');
    T('3', '-3', '6');
    T('0', '-3', '3');
    T('-2.6', '-1', '-1.6');
    T('0', '-0.0000000000000000013', '1.3e-18');
    T('2', '-3', '5');
    T('7', '0', '7');
    T('-1.8', '-3', '1.2');
    T('-5', '0', '-5');
    T('0', '0', '0');
    T('1', '0', '1');
    T('6', '-3.1', '9.1');
    T('1', '-3', '4');
    T('-1', '7', '-8');
    T('-2', '-2', '0');
    T('3.0', '-1.8', '4.8');
    T('0', '0', '0');
    T('0', '-9', '9');
    T('-4', '0', '-4');
    T('-1', '-1.9', '0.9');
    T('0', '-2', '2');
    T('-6', '1', '-7');
    T('1', '1', '0');
    T('-3.8', '3.4', '-7.2');
    T('5', '-2', '7');
    T('0', '0', '0');
    T('-5', '1', '-6');
    T('2', '0', '2');
    T('-6', '-32', '26');
    T('-3', '7', '-10');
    T('0', '0', '0');
    T('-1', '3', '-4');
    T('-5.6', '-0.00000000023', '-5.59999999977');
    T('1.1', '-3', '4.1');
    T('0', '3', '-3');
    T('0.0000000028', '1', '-0.9999999972');
    T('1', '0', '1');
    T('-2.9', '0', '-2.9');
    T('-4', '-3', '-1');
    T('-1', '0', '-1');
    T('0', '-8', '8');
    T('5', '-1', '6');
    T('50', '0', '50');
    T('1.4', '-0.023', '1.423');
    T('0', '0', '0');
    T('0', '-0.000000000000000000044', '4.4e-20');
    T('6', '1.0', '5');
    T('0', '0', '0');
    T('-5', '0', '-5');
    T('1', '4', '-3');
    T('5', '8', '-3');
    T('-1', '0', '-1');
    T('4', '1', '3');
    T('1.1', '0', '1.1');
    T('2', '-1.8', '3.8');
    T('1.2', '3', '-1.8');
    T('1.0', '-2', '3');
    T('0', '-26', '26');
    T('-1.2', '-0.000021', '-1.199979');
    T('0.00000000000000022', '0', '2.2e-16');

    T('-0.00000000000000121361969403227661937847795811', '574992758622232775968317372974458171501.4254915170813', '-5.7499275862223277596831737297445817150142549151708130121361969403227661937847795811e+38');
    T('-4550837208408065075504522665550667460', '5.751431060476645646485961575720', '-4.55083720840806507550452266555066746575143106047664564648596157572e+36');
    T('7996662362107395829425.18720176386230', '-8731907507950', '7.9966623708393033373751872017638623e+21');
    T('-68461838113448256548677344232782026104.040843480437532', '0.0000142986280419', '-6.84618381134482565486773442327820261040408577790655739e+37');
    T('-665.7', '3908931.49090108734150120238488', '-3909597.19090108734150120238488');
    T('338109.679317396058', '-45.6', '338155.279317396058');
    T('-2471514788.354942554630483008', '-0.0000000055765369280717362958649740', '-2471514788.354942549053946079928263704135026');
    T('-18642529601026.27209290620938355635', '67022255553746002366834404466320.198', '-6.702225555374600238547693406734647009290620938355635e+31');
    T('27195199183.93214817879349469171272068644528813652023534', '-1', '27195199184.93214817879349469171272068644528813652023534');
    T('41026854231898.0486363115993557', '-0.0000000000000029972604673995461216855', '41026854231898.0486363115993586972604673995461216855');
    T('-1220181643016279028.87348578653647757882815849', '-4124297626266.931179306588915876637', '-1220177518718652761.94230647994756170219115849');
    T('174593431895507873748465653530769377174', '7.3340', '1.74593431895507873748465653530769377166666e+38');
    T('93091029642275068666781.0284945435443997658445', '-31928424886539383422.9555351175713959', '9.31229580671616080502039840296611157956658445e+22');
    T('-86027968259520771550471521850.6', '0.0000000000000000292728157559323008342389754688224199225762190199185607', '-8.60279682595207715504715218506000000000000000292728157559323008342389754688224199225762190199185607e+28');
    T('-1.44467944259873814286', '2.3245669450269604739750258313435250093720066817981549442', '-3.7692463876256986168350258313435250093720066817981549442');
    T('15548437865455556312866591689712167475961', '2.2', '1.55484378654555563128665916897121674759588e+40');
    T('0.00000000000000000020770604235293769845', '-0.0000992358273674621622485217917158420', '0.00009923582736746236995456414465354045');
    T('-1626846234967158413103510196475626.88719324087757539708', '-0.0000000000000007877989096708003891755677772229305473607860191', '-1.6268462349671584131035101964756268871932408775746092810903291996108244322227770694526392139809e+33');
    T('-0.00001339836', '0.0000333909797773140', '-0.000046789339777314');
    T('-20.6409', '-0.0000000000017673756783209754', '-20.6408999999982326243216790246');
    T('-0.00000000164445378704494553248164602947761915692700', '-294719705242476177981296942472301246209096', '2.94719705242476177981296942472301246209095999999998355546212955054467518353970522380843073e+41');
    T('-714.4854312', '479.74301995618099492271043882', '-1194.22845115618099492271043882');
    T('41306640988462651378.04879814073', '2.3', '41306640988462651375.74879814073');
    T('1888287331013305459263552414.6839815', '7369526522887481233382690934873900068542595569442820447065', '-7.3695265228874812333826909348720117812115822639835568946503160185e+57');
    T('-63087590127015461742418723247183314165549180801856.76047', '511852', '-6.308759012701546174241872324718331416554918131370876047e+49');
    T('-712813331685105224867937788768200257082.35597637750550', '-0.000000154423320652795188516788493', '-7.12813331685105224867937788768200257082355976223082179347204811483211507e+38');
    T('-0.00000021267799883027266680922532294584', '-99920149812553578396', '99920149812553578395.99999978732200116972733319077467705416');
    T('-0.000001314853272479802673030806326616332901672901446530', '-0.00000000000000177806', '-0.00000131485327070174267303080632661633290167290144653');
    T('-335141130.448', '-622097386577809642069744047.28826330', '6.220973865778096417346029168402633e+26');
    T('0', '120769246650024506936432254.2242696', '-1.207692466500245069364322542242696e+26');
    T('-23641982115679784801339.39566245781597254522', '-3682962980152156410.3125514', '-2.363829915269963264492908311105781597254522e+22');
    T('-88042183866221727883616143734328.731116339192172335', '2124601502534008501755.584348081323605468287', '-8.8042183868346329386150152236084315464420515777803287e+31');
    T('11947261350807745084450506573219.467267625392669956554', '-2144324041379.38432776094911354677701674948588', '1.194726135080774508659483061459885159538634178350333101674948588e+31');
    T('230257242957.393', '-10607111174738890296557505.142420583396378', '1.0607111174739120553800462535420583396378e+25');
    T('-0.000000000000000000205207122559582205445490420', '-137363469389767969588812337824785302440844', '1.3736346938976796958881233782478530244084399999999999999999979479287744041779455450958e+41');
    T('-681.369980', '279143637047358960283640320234319673269492015123776', '-2.7914363704735896028364032023431967326949201512445736998e+50');
    T('0.00000000000047867734749297313578318166203844289133575435501552', '5318894.3577269473404992938443574038899509659386283', '-5318894.35772694734002061649686443075416778427658985710866424564498448');
    T('-7009640457950142150404723076242641700234100735651466.001', '450572184269898130184881623816.351532207', '-7.009640457950142150405173648426911598364285617275282352532207e+51');
    T('-667761.1964938970806540832766776998292', '-11916.0', '-655845.1964938970806540832766776998292');
    T('78115005042419615473330500516096634.176852324566877077', '-6798.68296696083172787385428', '7.811500504241961547333050051610343285981928539860495085428e+34');
    T('7810338218505304105029156703754.942106797', '0.000001731538457890332023362604451262238656250891', '7.810338218505304105029156703754942105065461542109667976637395548737761343749109e+30');
    T('147524.3448475445608', '1648294473575921', '-1648294473428396.6551524554392');
    T('-0.000000000000000000171463070071732948056011420608183352219588685', '586793187554936.5855', '-586793187554936.585500000000000000171463070071732948056011420608183352219588685');
    T('-1211390378858441999502927549647140419603384681145414251351213', '-635082911540249607321.024949083796644', '-1.211390378858441999502927549647140419602749598233874001743891975050916203356e+60');
    T('4', '16059489423456265750163100012.9030479691340920432572860034', '-1.60594894234562657501631000089030479691340920432572860034e+28');
    T('-0.02386494924652569606758900911891223878669277', '6954997.21231337279682439457198586266365943551328684831121', '-6954997.23617832204335009063957487178257167429997961831121');
    T('-77097949257283524913898.9005058281976253594672', '2548220758876398059247570217121711835.85372529', '-2.5482207588764751571968275006466257347542311181976253594672e+36');
    T('0.0000000000000000003167303237768130560', '-1774472.102377282878491377352513811904', '1774472.102377282878491377669244135680813056');
    T('12826152962883951576635.9617817', '-85603180142227733048283563048771.16275698594', '8.560318015505388601116751462540712453868594e+31');
    T('-28594.58153422332574547631803496831493', '0.00138135692454804974058929153874335145729995858764536203', '-28594.58291558025029352605862425985367335145729995858764536203');
    T('-1076543179517781327152214.4430186084047103180550303835484', '-0.000001966501043041903595634052091', '-1.076543179517781327152214443016641903667276151434749496309e+24');
    T('1.1303763207006847408087108827484230273165561756244577060659', '-5226422630308737810960949550589969908179555303', '5.2264226303087378109609495505899699081795553041303763207006847408087108827484230273165561756244577060659e+45');
    T('4.6379', '0.0010851657258206953367908525441350095857171922501796819281707', '4.6368148342741793046632091474558649904142828077498203180718293');
    T('-0.00000121970222487531743738910465210', '14700571395.710868072772989809', '-14700571395.7108692924752146843174373891046521');
    T('-0.00948647116781336833163259991', '-1.7768824919899484548781987321907266047229337854', '1.7673960208221350865465661322807266047229337854');
    T('0.0000000000294045914362167819882890883773282671860065407908497565282', '1203484199859679.515373248387944', '-1203484199859679.5153732483585394085637832180117109116226717328139934592091502434718');
    T('370605558535696550704041748.54', '2859.002553198485573190388217618142019', '3.70605558535696550704038889537446801514426809611782381857981e+26');
    T('23063425385264062114.4182185521', '919024.6882123634392645395934284705089', '23063425385263143089.7300061886607354604065715294911');
    T('196575005.9706725', '-5209445831094830', '5209446027669835.9706725');
    T('-749632.806770304627392347831545996165984090513451938727374', '772302148117130.7584834', '-772302148866763.565253704627392347831545996165984090513451938727374');
    T('218023.954642124004155088474898834365173675390209344675040', '-4493284.9', '4711308.85464212400415508847489883436517367539020934467504');
    T('371184371226569100503051893.30461921952559443426660043299229', '-0.0000000034906715619289934', '3.7118437122656910050305189330461922301626599619559383299229e+26');
    T('220152166905051656324095658827324313008647.0', '-0.00000541198892301696315819055681172535', '2.2015216690505165632409565882732431300864700000541198892301696315819055681172535e+41');
    T('343183254236262724.51785020', '1', '343183254236262723.5178502');
    T('0.0000000003890086401641242121857013479571090262584101373', '-8317.38612508391889137944463821522953', '8317.3861250843079000196087624274152313479571090262584101373');
    T('0.00000000000000000086155332', '-17.6333243552751', '17.63332435527510000086155332');
    T('0', '268666137918342629663843421660.888578', '-2.68666137918342629663843421660888578e+29');
    T('0.00000000000000135738708106732310272812674949101792', '53327465.2335089327976851709827349825', '-53327465.23350893279768381359565391517689727187325050898208');
    T('266910608197788439217725496348217', '-0.00000000000000138658', '2.6691060819778843921772549634821700000000000000138658e+32');
    T('11641358', '81237.72625754123497510364150080', '11560120.2737424587650248963584992');
    T('-10397358465137390.59112438352080341792626708545', '0.000000000000035771048469575519305632795707', '-10397358465137390.591124383520839188974736660969305632795707');
    T('-319067503684881132718867038113132023386552638920.93464409992', '-4362498854097189202568295213849864755626616334057699.89553', '4.36217978659350432143557634681175162360322978141877896088590008e+51');
    T('-291431002924776541728754603601.36970805037321548645', '5964565940.126384157441377', '-2.9143100292477654173471916954149609220781459248645e+29');
    T('-1.2311860880', '-596880833224', '596880833222.768813912');
    T('87.132412112577000', '3', '84.132412112577');
    T('14496251944930895122386093.97997766', '-0.0186085382187367427216451737443431075455317', '1.44962519449308951223860939985861982187367427216451737443431075455317e+25');
    T('-1454167.8755560906631572', '-14023057721.47963687', '14021603553.6040807793368428');
    T('-48.0205403500832716158268160170078138883305085421632055636', '-4225012867.41212544381914188804', '4225012819.3915850937358702722131839829921861116694914578367944364');
    T('132233805697430308711156924448348175516180746442', '0.0000017428839153838195472106854742425800', '1.3223380569743030871115692444834817551618074644199999825711608461618045278931452575742e+47');
    T('-3464742138515.3596070', '-116196292944709774661.6869498643733175605', '116196289479967636146.3273428643733175605');
    T('959618941323760340533285165929739125596307241332873169.21229', '66867302503354526158135738979.180422740', '9.5961894132376034053328509906243662224178108319713419003186726e+53');
    T('272744684991153578922570226355.46751565943', '2120.8826649154579918903365041556839175862', '2.727446849911535789225702242345848507439720081096634958443160824138e+29');
    T('392752713096323327908999', '0.000000017921157295538031059674698531743345633301', '3.92752713096323327908998999999982078842704461968940325301468256654366699e+23');
    T('98160222492281761742.521218301', '-0.0285627038777634002456158926936381577455032644551906387514964', '98160222492281761742.5497810048777634002456158926936381577455032644551906387514964');
    T('-4.36', '-271625870935.20401097518', '271625870930.84401097518');
    T('-19.4212927874167898', '-3296800120589635669055398465088086245282862075.96730', '3.2968001205896356690553984650880862452828620565460072125832102e+45');
    T('-7384530397856089', '-45.9', '-7384530397856043.1');
    T('-774820079197336364734223154.8', '-10.942584818509498694779756530950038148', '-7.74820079197336364734223143857415181490501305220243469049961852e+26');
    T('-150596940796161658996.388271502068869', '2985065345825248339763675882859244909723097343219', '-2.985065345825248339763675883009841850519259002215388271502068869e+48');
    T('-23847.667194770678602066965772568166575932862070476568359149', '1.05838941', '-23848.725584180678602066965772568166575932862070476568359149');
    T('-139.90436152895151552565739375535355738822610296926957375853', '0.2047189', '-140.10908042895151552565739375535355738822610296926957375853');
    T('-4859551136289853679597736', '5705851489', '-4.859551136289859385449225e+24');
    T('-2.822932759210967942204799418357885484503668027660403642776', '-21260029158.7755772147552', '21260029155.952644455544232057795200581642114515496331972339596357224');
    T('70214727623926969557083.633194040081188', '0.000000000000000008876781900', '7.02147276239269695570836331940400811879911232181e+22');
    T('168167915122787.2659904', '-0.000000000000000000022424096865237242831921499442051317229853874001', '168167915122787.265990400000000000022424096865237242831921499442051317229853874001');
    T('-10381081602312', '0.000000000000000923228830', '-10381081602312.00000000000000092322883');
    T('6481217378435019509.3', '204127.50354634007763', '6481217378434815381.79645365992237');
    T('-1055871579008468883600015600194814.219411847', '0.00000000000000001993933675383625869609055753063233531835884', '-1.05587157900846888360001560019481421941184700000001993933675383625869609055753063233531835884e+33');
    T('-11.9', '2190021917.5', '-2190021929.4');
    T('8893568.397103781249', '-7.29674059550', '8893575.693844376749');
    T('-270074164732402092470707995826826345208506171395062650454324', '392935540317', '-2.70074164732402092470707995826826345208506171395455585994641e+59');
    T('-582332724559151988564243629866.793819328', '-752.316045', '-5.82332724559151988564243629114477774328e+29');
    T('115780050433.9744723375949160', '-10354559440726926933085330907464207489638', '1.0354559440726926933085330907579987540071974472337594916e+40');
    T('-0.0000000000092082909772982168202901287881437', '-140307240110991278811078363131.3533385306060728599114476021', '1.403072401109912788110783631313533385305968645689341493852797098712118563e+29');
    T('-0.0000000000000000001302905868005538222253', '-687601306119.30280105580919933093896015091959269610032847933', '687601306119.30280105580919933080866956411903887387502847933');
    T('-17406776.9106520', '7404147683526942010449859408523376672505043.20', '-7.404147683526942010449859408523376689911820110652e+42');
    T('-5810490488292821.088550524158200456214711210120', '51309126128019006928075619352.35509020203161', '-5.130912612802481741856391217344364072618981045621471121012e+28');
    T('-244663279887797845457244.69627014023746468381042256674045148', '-0.383335594341020704911762197811406655', '-2.44663279887797845457244312934545896443978898660368929044825e+23');
    T('316759637646274728459769194971270681520143070356211369', '138969888208.552504340', '3.1675963764627472845976919497127068152014293138632316044749566e+53');
    T('-1026716565990818152.338103221159', '1255719213990793975090997507278', '-1.255719213991820691656988325430338103221159e+30');
    T('854.93693123842152090961298359199223034349519195749', '-424117414357060.756799', '424117414357915.69373023842152090961298359199223034349519195749');
    T('-0.006379478941040566506054', '3928.2672878523130519347199700529360289380801729479368', '-3928.2736673312540925012260240529360289380801729479368');
    T('3469779053.85027069235104893994561560034151457638008273176647', '25676509132014403666611211903.3702668974', '-2.567650913201440366314143284951999620504895106005438439965848542361991726823353e+28');
    T('-6.0694', '-820322384540258597408240140073275834913353.69', '8.203223845402585974082401400732758349133476206e+41');
    T('-0.000000000000030532', '24.896880201507626275851', '-24.896880201507656807851');
    T('105468873', '0', '105468873');
    T('256760156309110476', '-13837063158475039.0608439897688680023912472959404758275635521', '270597219467585515.0608439897688680023912472959404758275635521');
    T('0.000000000000000001698545937726615224762953174', '0.0091708187464', '-0.009170818746399998301454062273384775237046826');
    T('-21254127772.055477', '170724844816140294005708794935062810958442.2386', '-1.70724844816140294005708794935084065086214294077e+41');
    T('4.3', '-42289375032753833461292867654449379918122659582300511814.86', '4.228937503275383346129286765444937991812265958230051181916e+55');
    T('-10.6945731348037109472689588888585520230967031058063284652', '-0.000000000000000013225637814061559914861522643297461025', '-10.6945731348037109340433210747969921082351804625088674402');
    T('42022129692143246218.588174513905523856665669864568', '-6688196881698', '42022136380340127916.588174513905523856665669864568');
    T('59567595164759708273.58488961999', '-2882669546962777869333', '2.94223714212753757760658488961999e+21');
    T('-0.00000000000000023219819555088939', '0.000002699155773875942009505196424', '-0.000002699155774108140205056085814');
    T('69.02', '1019.384525', '-950.364525');
    T('0.28885545083047505669261757814370521673472269407769277', '-54409284625846293530958014777988170910931971864508243085507', '5.440928462584629353095801477798817091093197186450824308550728885545083047505669261757814370521673472269407769277e+58');
    T('357051970556175609969558033601717918622832633809282', '-0.00000000000000009191759009979', '3.5705197055617560996955803360171791862283263380928200000000000000009191759009979e+50');
    T('55324032341233413075839541.33979738365146913154063', '18778204875511803067.37488033228813311299318998557567838806', '5.532401356302853756403647396491705136333601854744001442432161194e+25');
    T('-954279324346950395396438207814989843', '5855883956282.7418', '-9.542793243469503953964440636989461257418e+35');
    T('0.0000559372400719154055', '36326.3773033065', '-36326.3772473692599280845945');
    T('-2308273733231941938445560342695.8870897860340173552415859451', '0.0000000000000000000176744410303783123790738765240224970704250496463086', '-2.3082737332319419384455603426958870897860340173552592603861303783123790738765240224970704250496463086e+30');
    T('-0.0000000000000075315134692867358694825', '0.000000000020673253575755655053286884627683943900054889', '-2.0680785089224941789156367127683943900054889e-11');
    T('2141565935846165335053366646564.55', '249139386942.433521867737894', '2.141565935846165334804227259622116478132262106e+30');
    T('0.0000000000000020997422982353397148204786811844837507999862174', '0.000000000000000002356388173646211525529238', '2.0973859100616935032949494431844837507999862174e-15');
    T('-9.3', '1210724787708353076304704165555.510002672635727895531335', '-1.210724787708353076304704165564810002672635727895531335e+30');
    T('-94455687.8', '0.0000002761251584122631449121019167818539664249453145710', '-94455687.800000276125158412263144912101916781853966424945314571');
    T('-527306114203076623189237999707479.9688776820253214193899164', '1599.728954415768000917144378511032', '-5.27306114203076623189237999709079697832097793322336534294911032e+32');
    T('5201144349263847987496751693.2718392256495115509', '-0.000000000000000008062522688625856069134486136704806619173555611278', '5.201144349263847987496751693271839225649511558962522688625856069134486136704806619173555611278e+27');
    T('-769134537.983940345', '-596619628089684709170992.484', '5.96619628089683940036454500059655e+23');
    T('16103.67287682796494953', '-465781.44458474726', '481885.11746157522494953');
    T('972019.0403266398942', '0.00000000032060736406868701231737725067236', '972019.04032663957359263593131298768262274932764');
    T('-3722457663697211882835707', '-16.8', '-3.7224576636972118828356902e+24');
    T('140260.1', '5', '140255.1');
    T('5314287735775902656123286734.69857230', '-33935993.716814133855691839888466392152019494030707469085', '5.314287735775902656157222728415386433855691839888466392152019494030707469085e+27');
    T('68283872710781941018806771046574852133', '0.00000000075', '6.828387271078194101880677104657485213299999999925e+37');
    T('-369081.4527', '141567.2246', '-510648.6773');
    T('20446533321512161063774845264.22916016420704184590862638', '-68295029.764030715', '2.044653332151216106384314029399319087920704184590862638e+28');
    T('-600307.74138395225787263', '-0.0000001304496382302165936930', '-600307.741383821808234399783406307');
    T('24411161239.85690978507516121', '2051270884829.8383684099295195107651592681789671', '-2026859723589.9814586248543583007651592681789671');
    T('4384414740.5595825362370080404420', '-0.00000000000000554366638630081310095841078569927213406536019558277955', '4384414740.55958253623701358410838630081310095841078569927213406536019558277955');
    T('9025031.8925028', '-32.8', '9025064.6925028');
    T('-256722752342173962991.91', '-0.0000000000000000002123046323462613566369', '-256722752342173962991.9099999999999999997876953676537386433631');
    T('43542362222453910743776206071468.7163717435431410983843', '1100168.647375874421732648073262226', '4.3542362222453910743776204971300068995869121408450311037774e+31');
    T('0.062784590888327314154797', '1057478188.877253410081536', '-1057478188.814468819193208685845203');
    T('-7050', '45104384760478621637071269060904128288820257014210.74', '-4.510438476047862163707126906090412828882025702126074e+49');
    T('1592658310474883023', '-399176738358380589054069565962.3840352823659556138516', '3.991767383599732473645444489853840352823659556138516e+29');
    T('206141261', '-1266053.469', '207407314.469');
    T('644.1', '-0.4369751366969488640109929651726886244920517832668019', '644.5369751366969488640109929651726886244920517832668019');
    T('1612132.216890744513316472', '-51082267675271', '51082269287403.216890744513316472');
    T('1230427083.97', '0.000758744063330274578234716889288876558038637793691373850', '1230427083.96924125593666972542176528311071112344196136220630862615');
    T('-35776247491956058344815123524285', '0.000000000012958363043907', '-3.5776247491956058344815123524285000000000012958363043907e+31');
    T('8661462327754244989542.365', '0', '8.661462327754244989542365e+21');
    T('-1629190886816579351740029513722257473580386199203949.0', '-6496265115300408904692604.738158', '-1.629190886816579351740029507225992358279977294511344261842e+51');
    T('13.891128134595', '4', '9.891128134595');
    T('-11113.7', '0.000000007958782556879962582439914573325191682999786', '-11113.700000007958782556879962582439914573325191682999786');
    T('-1118611298789753369562561', '0', '-1.118611298789753369562561e+24');
    T('0.000000000590336252071681886600391810609976', '-67.75', '67.750000000590336252071681886600391810609976');
    T('-169823638653318606537007998653864233599165.937', '-281597273362859424.39994', '-1.6982363865331860653700771705659087073974153706e+41');
    T('-3.156805980919999726868943', '546306412638131947574220', '-5.46306412638131947574223156805980919999726868943e+23');
    T('-0.000000000961866927553387950929926693447189622664803906118589565996', '1599939073440130339796992146.295983955944260728', '-1.599939073440130339796992146295983956906127655553387950929926693447189622664803906118589565996e+27');
    T('0.4283', '0.000000000020729812246752207692169547780166634734001', '0.428299999979270187753247792307830452219833365265999');
    T('-0.0004573434550153087210903', '0.03770996737891493754829068954377697580116', '-0.03816731083393024626938098954377697580116');
    T('-3960.59579547', '-1814.0503015604326701', '-2146.5454939095673299');
    T('228667830414358772925710849158.07453', '-242389246', '2.2866783041435877292595323840407453e+29');
    T('-4938.8', '33796487947361', '-33796487952299.8');
    T('23066858157696468038.419700247057090437730118', '1017294462916594953213682499527318061014200', '-1.017294462916594953213659432669160364546161580299752942909562269882e+42');
    T('-3.329', '7129476962505466189.958258', '-7129476962505466193.287258');
    T('0.0002134952691176147868498212012808766187', '-55.418', '55.4182134952691176147868498212012808766187');
    T('-287906196200519721661337752259844531254735020373.400102226', '-149964438772134655329086295.489409993989261332871176761', '-2.87906196200519721661187787821072396599405934077910692232010738667128823239e+47');
    T('0.879756603251906828679553532880817', '-169.6608007559773549752480442', '170.540557359229261803927597732880817');
    T('0.0000006245253019711867486517080491', '-0.000000000000000016145400714914217811', '6.24525301987332149366622266911e-7');
    T('-0.00000000000000000028355442083', '-6.7960400', '6.79603999999999999971644557917');
    T('-327828.933845', '-0.000000000016720812786354347397949935318536433299320234', '-327828.933844999983279187213645652602050064681463566700679766');
    T('-12062798839897370982166084543665094512857864041322339898381', '-983855787277604624519041110446627116266868551757862371483', '-1.1078943052619766357647043433218467396590995489564477526898e+58');
    T('-368029067460.235618031440780659671986274176149669', '-7.8327366781', '-368029067452.402881353340780659671986274176149669');
    T('48.841', '621799398797787989614515.034591400113887912474171355', '-6.21799398797787989614466193591400113887912474171355e+23');
    T('0.00000000000000000001028447389', '11411023852013652', '-11411023852013651.99999999999999999998971552611');
    T('109388.053063770057103', '-0.000161002185201484294184231320866015688631673966536', '109388.053224772242304484294184231320866015688631673966536');
    T('0.0000000075454081720580498999564801', '-6133333234722476.50731423128929112324554345104797981', '6133333234722476.50731423883469929530359335100445991');
    T('0.0000000000001748707964040132704465271', '0.0014559085438085897564402233344521317116113630490391191639', '-0.0014559085436337189600362100640056046116113630490391191639');
    T('169.5777437575868731193679671977293144862017912', '171384684233482289557.248837', '-171384684233482289387.6710932424131268806320328022706855137982088');
    T('0.000000000000000963994117406293262728543', '10931977.50377877', '-10931977.503778769999999036005882593706737271457');
    T('-0.0000000000000004943772275105155307931235866139', '-6331812416762098838522', '6.3318124167620988385219999999999999995056227724894844692068764133861e+21');
    T('551506392806.03421875426724909838', '1494156988342161962977365.44', '-1.49415698834161045658455940578124573275090162e+24');
    T('-35954803866862.46955421251697497767880', '-96620255925.814702354', '-35858183610936.6548518585169749776788');
    T('-8973170535214416154002860437001664', '8701360172863868015378154133091486710755411398837050.58411296', '-8.70136017286386802435132466830590286475827183583871458411296e+51');
    T('369.0854266274373143948330319405738975782757717917', '0', '369.0854266274373143948330319405738975782757717917');
    T('4130252362617611707187141280611348999706', '-120506138511555615072992404.1670872857058427291541946', '4.1302523626177322133256528362264219921101670872857058427291541946e+39');
    T('0.00000000000000000129518837', '-43970554119955228415415630801336595257.433109069593', '4.397055411995522841541563080133659525743310906959300000129518837e+37');
    T('-199832949667155587976206363450860330500926252.4790', '88697015896564.601775678484183991355335780830', '-1.9983294966715558797620636345094902751682281708077567848418399135533578083e+44');
    T('-3', '-94608429043423790061', '94608429043423790058');
    T('5914252851', '-32833159514298.717701562542', '32839073767149.717701562542');
    T('0.00000000000000000062135622742614277', '-934153618.26014440731275', '934153618.26014440731275000062135622742614277');
    T('-414018564163156440', '82550425859087829706198633745536278044153640.5895796189250', '-8.2550425859087829706198634159554842207310080589579618925e+43');
    T('-1496454100585258488602.0286', '0.000000000109895026640912465560937708538864597016913853723419507', '-1.496454100585258488602028600000109895026640912465560937708538864597016913853723419507e+21');
    T('-1341525946245998', '193723560793329369351971060', '-1.93723560794670895298217058e+26');
    T('0.000000000000004972247161753329501987430882626554770098611002363504786', '-0.02206405830101097017174889990515767580456922875023374', '0.022064058301015942418910653234659663235451855305003838611002363504786');
    T('759412612479446413', '-0.001641148192720654679893469509982581527584330378', '759412612479446413.001641148192720654679893469509982581527584330378');
    T('202578904.71133021395689295350439448588680928447617886', '-35337424341284.346281238517', '35337626920189.05761145247389295350439448588680928447617886');
    T('10862970759795016587884612569718575353781870555253901512816', '-214.533', '1.0862970759795016587884612569718575353781870555253901513030533e+58');
    T('-0.00000000000000040285962950086', '-216829085197233503.866070375365413688040', '216829085197233503.86607037536541328518037049914');
    T('550250782173491663208128526734.569', '-0.000000000100710017655581949422409688681756709469258865537626639492015', '5.50250782173491663208128526734569000000100710017655581949422409688681756709469258865537626639492015e+29');
    T('-19678891452829871158.36095', '1040.72243146362968914160338864', '-19678891452829872199.08338146362968914160338864');
    T('-20972447940296172836322634.7', '0.00000000000000150351', '-2.097244794029617283632263470000000000000150351e+25');
    T('-135716917862073377635780105702430442287902.2441615427680', '-156170394359696385001989148011019228708', '-1.35560747467713681250778116554419423059194244161542768e+41');
    T('249073042918039345', '758712100158049101656321168.88542968', '-7.5871209990897605873828182388542968e+26');
    T('9228932554571685674.6101721970', '93.5210831751023', '9228932554571685581.0890890218977');
    T('0', '-3271.372', '3271.372');
    T('712617017414340344456425371594378983', '-7013575085098427106742320.320014442', '7.12617017421353919541523798701121303320014442e+35');
    T('0.00000000000000991278822903249', '-266685953910734027663766948621954.4074654249748', '2.6668595391073402766376694862195440746542497480991278822903249e+32');
    T('-0.00001168408090912846185222706894568927004955366630406', '-0.00000357928456', '-0.00000810479634912846185222706894568927004955366630406');
    T('-0.0000000000000000001077428', '-18065.9571', '18065.9570999999999999998922572');
    T('-26767075956743559361260276966383.60643853154', '9.811418917439919669145337064909359580107696', '-2.6767075956743559361260276966393417857448979919669145337064909359580107696e+31');
    T('-94992614046521719511194550871785575096581461042589', '-7.24', '-9.499261404652171951119455087178557509658146104258176e+49');
    T('249.84717238354744614', '-58760885296179353551418799.1637574', '5.876088529617935355141904901092978354744614e+25');
    T('0.190077788951733727544', '3029470118670840', '-3029470118670839.809922211048266272456');
    T('-535577308373493988698.28', '-0.00000000000000000001223589264699363887684193', '-535577308373493988698.27999999999999999998776410735300636112315807');
    T('17330374908582279821881.6419631817630053975265540001599', '51572037568522929067270779857332002426120', '-5.15720375685229290499404049487497226042383580368182369946024734459998401e+40');
    T('2620791577.5388', '-429.387', '2620792006.9258');
    T('24955342165062415125441804.9102804783908687387110', '-894.26', '2.4955342165062415125442699170280478390868738711e+25');
    T('-30.966199737786598010412005660223057114288056999548471748', '7554891.786144582480621940890219161695392942927639518', '-7554922.752344320267219951302224821918450057215696517548471748');
    T('1.1132', '33358298757744778319077792505.510848143559223', '-3.3358298757744778319077792504397648143559223e+28');
    T('-71806617.15', '5002475659678496285425142082140754487', '-5.00247565967849628542514208221256110415e+36');
    T('-64642.3969', '-45742744308293235253.79', '45742744308293170611.3931');
    T('0.000000000000000005061114767099631770343151056312', '-0.1380692319876694309061865652400348112805369059470451819398816', '0.1380692319876694359673013323396665816236879622590451819398816');
    T('1104962688677222230.43026613036133967074026481598059', '65279233245714944288887', '-6.527812828302626706665656973386963866032925973518401941e+22');
    T('-244.20751029974608088458301628723627482919450688930317756', '-10.309887906765666581', '-233.89762239298041430358301628723627482919450688930317756');
    T('-374204737.17485473467771059091311676198780910101246261', '-1772169195775224156804629688653654033.48913943457', '1.77216919577522415680462968827944929631428469989228940908688323801219089898753739e+36');
    T('0.0000825267296819574292144866422', '0.000000024644261884116746400488936364031', '0.000082502085420073312468086153263635969');
    T('-337528893876584865851246910311878729078289', '-181706307425668740481.5172', '-3.375288938765848658510652040044530603378074828e+41');
    T('338.68219932619', '36769458628.200', '-36769458289.51780067381');
    T('946213.052924553279580429320', '0.0000000000000000000164588680615834011158188287840390933880557701437361142926', '946213.0529245532795804293035411319384165988841811712159609066119442298562638857074');
    T('-2', '-1595717583274440094311744086944287786728005322', '1.59571758327444009431174408694428778672800532e+45');
    T('17453173.964073494242189989440985005742212919109806776575', '-14.66', '17453188.624073494242189989440985005742212919109806776575');
    T('9035579398295.9420488582066606273524126698350184209', '966939914253463733672441.61004035647', '-9.669399142444281542741456679914982633393726475873301649815791e+23');
    T('7', '-27403953633424153.2922529596537493', '27403953633424160.2922529596537493');
    T('-0.00000000000000000065661940373886034771780807724464', '23789019501708804655829.7720420134937860512', '-2.378901950170880465582977204201349378605185661940373886034771780807724464e+22');
    T('-379831', '1792454931672118909115.7219790011', '-1.7924549316721192889467219790011e+21');
    T('-6815476482314.81368548838834442636132177246866027753', '-23737403998114260637862634', '2.373740399810744516138031918631451161165557363867822753133972247e+25');
    T('-277091361.834', '1255049423087969154842.0583316773528577200623', '-1.2550494230882462462038923316773528577200623e+21');
    T('0.000527964207481201439802251215793030527669347056468169977214382', '1104312503520806883476040.604659402909367', '-1.104312503520806883476040604131438701885798560197748784206969472330652943531830022785618e+24');
    T('416.19049862551799054889649850446495', '25962521611212.630583656', '-25962521610796.44008503048200945110350149553505');
    T('-2', '0.00000023172759362622993578493256103424786936904115653889542088', '-2.00000023172759362622993578493256103424786936904115653889542088');
    T('2322661483960492703.7985297249359258447971802752303052984194', '-0.00285627578605593051554020854314614896', '2322661483960492703.8013860007219817753127204837734514473794');
    T('-50114375544024117743137634.7132631195193', '-4518136285230626801', '-5.01143710258878325125108337132631195193e+25');
    T('491790857283052164040388091239524266252647468540.602280', '0.0000000000000000002505634', '4.917908572830521640403880912395242662526474685406022799999999999997494366e+47');
    T('223757557.31976336704487143430860', '-30.36327', '223757587.6830333670448714343086');
    T('2266.004651', '-0.0000005867989135639875708', '2266.0046515867989135639875708');
    T('6958762418946097.285', '8704860092930499346311.824744308693', '-8.704853134168080400214539744308693e+21');
    T('-0.0000000000008758100333932644717298947984578724622849096677225', '-310962285610.30017170574463655670078', '310962285610.3001717057437607466673867355282701052015421275377150903322775');
    T('3654335458448.908519039918424', '-1017862207107897070244694254273184418309706541.32954379163', '1.017862207107897070244694254273188072645164990238062831548424e+45');
    T('0.000000028485591', '0', '2.8485591e-8');
    T('5073947900606220328306097118306928028865584', '-0.0000610672338767500', '5.07394790060622032830609711830692802886558400006106723387675e+42');
    T('184963915.618593586643741543203', '-1677.631219', '184965593.249812586643741543203');
    T('396871997862363101002483892.26', '-0.0000000000000000066839646413390330134202944396013877478232108451615', '3.968719978623631010024838922600000000000000066839646413390330134202944396013877478232108451615e+26');
    T('-0.001443', '-0.0000000000001480307586020440332395145414988692178570219362536734359922', '-0.0014429999998519692413979559667604854585011307821429780637463265640078');
    T('5173379226320319729203764178125730829947033.85', '16568260.329616', '5.173379226320319729203764178125730813378773520384e+42');
    T('-378684740425288590329.865832078814181361689411770555828033', '21039423253539461585612290662', '-2.1039423632224202010900880991865832078814181361689411770555828033e+28');
    T('-143411025.19904252040830704437889996439197644180', '0', '-143411025.1990425204083070443788999643919764418');
    T('-3971034326547', '-0.000000000000001192201124343158918732899783', '-3971034326546.999999999999998807798875656841081267100217');
    T('-1957074390.30', '-785.9373', '-1957073604.3627');
    T('11798157981084854910522713.726423390731883', '0.00000000000000000794673726058684351969247016507', '1.179815798108485491052271372642339073188299205326273941315648030752983493e+25');
    T('0', '63718006.533955912499520687', '-63718006.533955912499520687');
    T('-106057718559494874723085290836.38939560519223757224643923', '68938204335.59', '-1.0605771855949487479202349517197939560519223757224643923e+29');
    T('765027217313815006', '160135.21207763126955482', '765027217313654870.78792236873044518');
    T('-821923031708250104665643576287477708843314087.882590093', '-3990962158191136912693.079142196669532', '-8.21923031708250104665639585325319517706401394803447896330468e+44');
    T('0.00509812696199828345610871074161', '696247014515864996317842', '-6.9624701451586499631784199490187303800171654389128925839e+23');
    T('29732795045547659.0996914254', '-51709086127799933076710172937271558434987133.720', '5.17090861277999330767101729670043534805347928196914254e+43');
    T('-16286.209377407275812137531392451949', '0.00000000000000000022372394617291', '-16286.20937740727581213775511639812191');
    T('-1375253307827927096349134575050603.94211767852698652', '905513058712.3724033', '-1.37525330782792709635004008810931631452097852698652e+33');
    T('6307.78968779279695539093771472627844783336', '-0.00002778260088313361218945039359426278958850789168', '6307.78971557539783852454990417667204209614958850789168');
    T('-1251971126793648464770.62302133464300', '4108332118551588855518884176084', '-4.108332119803559982312532640854623021334643e+30');
    T('796875815810006347157872416846682346.70724147', '-26887.93843', '7.9687581581000634715787241684670923464567147e+35');
    T('503761901876.1', '-0.00000058591916804106576202', '503761901876.10000058591916804106576202');
    T('-0.0000144343551936630598170853542', '0.00000000000000001330848692526361142604488661622', '-0.00001443435519367636830401061781142604488661622');
    T('333807319855762444667280144179919.21709315488049995404', '3710.87428669634', '3.3380731985576244466728014417620834280645854049995404e+32');
    T('0.000000000000007381560468758912676992936863885310325961189', '5277874186685074734.78128632893251930850699235193165881271', '-5277874186685074734.781286328932511926946523593018981819773136114689674038811');
    T('0.002045907391481002055126364741722746123920492354', '-134940106198957372161636571.3843066120486149838380841998784', '1.34940106198957372161636571386352519440095985893210564620122746123920492354e+26');
    T('-30130112088.993827003962835076', '188322529121371266385077234579252182224.5691709790', '-1.88322529121371266385077234609382294313562997982962835076e+38');
    T('-3', '-8324641628446525083548695251230914153428', '8.324641628446525083548695251230914153425e+39');
    T('0.00018606293964295302172654549272224527271134692398', '-282281050085963.9114203115670420384745855159215452', '282281050085963.91160637450668499149631206141426744527271134692398');
    T('-16333583942997041272867370326132870504131195981.4696057800513', '45720082683202081441524403533698410.5545453845767', '-1.6333583943042761355550572407574394907664894392024151164628e+46');
    T('279328203.5', '-5825567.71080931239385137020833', '285153771.21080931239385137020833');
    T('-269224.94927224666095', '0.0000000000000000003064416316790457', '-269224.9492722466609500003064416316790457');
    T('6744812299069336942789392301581837129883725551922493.40', '-5703915992.9167440851668499293951029', '6.7448122990693369427893923015818371298837312558384863167440851668499293951029e+51');
    T('-0.000000000000000066706723472682', '-145217233354840.9452974499221403003294897671707915', '145217233354840.9452974499221402336227662944887915');
    T('8627053077274630737609487236548.4979774', '11.791146312838460410035604930189', '8.627053077274630737609487236536706831087161539589964395069811e+30');
    T('-0.01202422952668513636643178716923354493558618790', '-0.4635434098404790609527188243911111255589856688195077910287', '0.4515191803137939245862870372218775806233994809195077910287');
    T('-22486311958570.75', '136623992244353739977111844218335966565067420559172408797', '-1.3662399224435373997711184421833596656506744304548436736775e+56');
    T('-252554701524493.902565999316640', '0', '-252554701524493.90256599931664');
    T('0.00000000010355292', '-13318465915882548061708659853990135442387095003099676083', '1.331846591588254806170865985399013544238709500309967608300000000010355292e+55');
    T('0.000000000000003588941991478575635246817587941', '5787662416433534641981775919801763640029.288211579210799', '-5.787662416433534641981775919801763640029288211579210795411058008521424364753182412059e+39');
    T('3803854629.170758287238', '722557244.90221176531910610529831590', '3081297384.2685465219188938947016841');
    T('-4035292286478852880912536818781650501388972481375613031', '7023934966.329481438434165', '-4.035292286478852880912536818781650501388972488399547997329481438434165e+54');
    T('0', '-4651.5712076', '4651.5712076');
    T('943569.4471', '368852398555888115298707.33698897234448443580026', '-3.6885239855588811435513788988897234448443580026e+23');
    T('-7688139997698910283021899776208658703126', '-3014.021539', '-7.688139997698910283021899776208658700111978461e+39');
    T('-76014.328438738930802423876495451104480710964006', '0.16299839827610198847716746', '-76014.491437137206904412353662911104480710964006');
    T('425793955', '236826394', '188967561');
    T('1912.281988', '112194.49296301061246', '-110282.21097501061246');
    T('-5003494177.4861', '-1.82', '-5003494175.6661');
    T('7617479678.376905694802432279901930587745665869', '-947377.04', '7618427055.416905694802432279901930587745665869');
    T('367903789829906246.71158889842', '21491233153686072381957715968967883669600234520102964736911', '-2.149123315368607238195771596896788366959986661631313483066428841110158e+58');
    T('-6397756530845969250791946194979007883084.20444410377877774', '-2', '-6.39775653084596925079194619497900788308220444410377877774e+39');
    T('-75490132843212222631302495635041908895.42', '121330908238673', '-7.549013284321222263130261696595014756842e+37');
    T('7452613620257648176992750760.263566991938225', '519063023319381552888', '7.452613101194624857611197872263566991938225e+27');
    T('0.00000000000000778133736110171456441934891601883471609303002832912018885', '322394945530.295120804275213239014033', '-322394945530.29512080427520545767667189828543558065108398116528390696997167087981115');
    T('-8453.41542', '-92872449307.96646595226774421156553041473428714203142', '92872440854.55104595226774421156553041473428714203142');
    T('4184618260709992776470144081019630641.9546', '-39747088643178.4900618612667154842919288568428', '4.1846182607099927764701838281082738204446618612667154842919288568428e+36');
    T('-1002876199682238092689386871396', '-115235.191932983', '-1.002876199682238092689386756160808067017e+30');
    T('-0.0000020', '-67044918017341867873741', '6.7044918017341867873740999998e+22');
    T('19249573172273106187.51981325183516777379', '-0.0000036330230998383752637622876226209213994133789721626367167406', '19249573172273106187.5198168848582676121652637622876226209213994133789721626367167406');
    T('2367.8562797376185225678048', '-336.407580836131941', '2704.2638605737504635678048');
    T('15788198606260330370566880001150524384989538847.1566', '-13709725199190579602641841302109970817776478707296', '1.37255133977968399330124081821111213421614682461431566e+49');
    T('-0.000031568319907', '-319812227.101518430949692784775684487114739', '319812227.101486862629785784775684487114739');
    T('12598796115', '780974770970176277984213873835856961457.264132549460821743', '-7.80974770970176277984213873823258165342264132549460821743e+38');
    T('-12.05916350311631637603038624139252323932639510735991686', '-5079694354454213499772.98021638032478529', '5.07969435445421349976092105287720846891396961375860747676067360489264008314e+21');
    T('71001042332266', '-3.8749502633208489001742955080741839268852411926', '71001042332269.8749502633208489001742955080741839268852411926');
    T('-820079175.6236', '2602133034534013517476745071429', '-2.6021330345340135174775651506046236e+30');
    T('-0.05761', '-39324478925.37411997', '39324478925.31650997');
    T('-62331428327357976057401629977344374544913756472193', '5', '-6.2331428327357976057401629977344374544913756472198e+49');
    T('0.0000000079919621468803275713173314506', '-858.3109258585612540534304739091057092569004843368', '858.3109258665532162003108014804230407075004843368');
    T('0', '49228741196127252', '-49228741196127252');
    T('-0.000034775952537402974515789650718361816884555', '-81545435078.9081293428693463384918512495662737197672903499', '81545435078.9080945669168089355173354599155553579504057949');
    T('16831812489134711005.277009209496663', '-2711976951712313.504224327994229016719141489', '16834524466086423318.781233537490892016719141489');
    T('-14.478', '-7657287359215.379', '7657287359200.901');
    T('-5818291298159242639134804410045447574555164535.312859', '-2.318', '-5.818291298159242639134804410045447574555164532994859e+45');
    T('-0.00000000000000000538849432020724', '-0.00000000000043292954381309', '4.3292415531876979276e-13');
    T('129925455598327874.011965', '-373481502322636.54947846', '130298937100650510.56144346');
    T('23460218954598.255822499367', '5031071127.77603001713489698449235', '23455187883470.47979248223210301550765');
    T('0.1925927960745748035942288444958429531342391265261594074', '0.000000000001459509170829311723098341575950975186409687619', '0.192592796073115294423399532772744611558288151339749719781');
    T('-117656959944939983221909881756492.60506693656', '614774667011953263151051375859904824607146.03088402', '-6.1477466712961022309599135908181470636363863595095656e+41');
    T('4636798651708977942823954.838902837557757807933209543802058', '90429847774248975608629.657240991460781465643663', '4.546368803934728967215325181661846096976342289546543802058e+24');
    T('-8547.67216', '695927494328.0531238602508437', '-695927502875.7252838602508437');
    T('0.000000000000000000014395480531584764422752572199140185540212150740674066683', '0.00000000000000000165514253473158542904', '-1.640747054200000664617247427800859814459787849259325933317e-18');
    T('-14004821946.12197759187', '504302757065358604146449888186847067296187319675184252.5', '-5.0430275706535860414644988818684706729618733368000619862197759187e+53');
    T('0.00020661761959205061422959604052997905', '5983723321093859965577864973.03971707767767374', '-5.98372332109385996557786497303951046005808168938577040395947002095e+27');
    T('46.214922256501145', '-19450859033.49527912435693482', '19450859079.71020138085807982');
    T('607064966949689390411116362805267199510935776380', '-326141246118.898400528715804016912422852185420861463709', '6.07064966949689390411116362805267199837077022498898400528715804016912422852185420861463709e+47');
    T('7237360928.529978130333768130950', '52337694872297307366400191176848518321.9397556503425061', '-5.233769487229730736640019116961115739340977752000873796905e+37');
    T('-8.77', '0.00000250309032382179368', '-8.77000250309032382179368');
    T('2238967687307501052.8777593247573250304131289359965647930', '0.0000000006178212737146909670918694', '2238967687307501052.877759324139503756698437968904695393');
    T('4800872.6343266910471003503', '-43.630262', '4800916.2645886910471003503');
    T('6647.98125718514930441819193526467280173179366', '-4671450984642064474363.0922986578251558474619141444116721601', '4.67145098464206448101107355584297446026565384940908447389189366e+21');
    T('5213460994088.42423086209586241417259053025217796299975798215', '-4383637172725.279', '9597098166813.70323086209586241417259053025217796299975798215');
    T('-3.853', '20.661', '-24.514');
    T('-14271230745287310945.6312400262684102', '0.00000029645344606', '-14271230745287310945.63124032272185626');
    T('0.00000000032289911998', '92366796259.63491126551', '-92366796259.63491126518710088002');
    T('0.0000366129951186090146374619139990790117703732293398409315027110', '37573094701656757193581695.06285323469', '-3.7573094701656757193581695062816621694881390985362538086000920988229626770660159068497289e+25');
    T('-0.5052577098964168325539487540838485290578969474', '-2.469847', '1.9645892901035831674460512459161514709421030526');
    T('4157486.219788', '-0.000000000000000030', '4157486.21978800000000003');
    T('-18128173766.45100920958', '0.0000000270000492658439841803', '-18128173766.4510092365800492658439841803');
    T('0.00000011629092179596057280755013975516', '-21061915294588585007285269886442442.89790253758527', '2.106191529458858500728526988644244289790265387619179596057280755013975516e+34');
    T('1628917.78', '-21383191736538.31532452805283554', '21383193365456.09532452805283554');
    T('1559380098949430706272504015810.69555717100498119630529074', '165173284.748129', '1.55938009894943070627233884252594742817100498119630529074e+30');
    T('36374023831462155387508503692769818148680320937241486', '-1.031827911563637271237851439355833437004912', '3.6374023831462155387508503692769818148680320937241487031827911563637271237851439355833437004912e+52');
    T('4.8081', '0.0003519196291075952295892328224194785771237221553', '4.8077480803708924047704107671775805214228762778447');
    T('-3782511514044035022728785099618757257160775562240791309177', '-0.000000000000795400939', '-3.782511514044035022728785099618757257160775562240791309176999999999999204599061e+57');
    T('-0.00005330668607181480434', '71992567276897950313737317415196.7780236926093645024753', '-7.199256727689795031373731741519677807699929543631727964e+31');
    T('-336.6', '0.000000127038712398089934486863716910205972688100425270367714213640', '-336.60000012703871239808993448686371691020597268810042527036771421364');
    T('0.000000002105290992087417133622746', '534450465748570290833221890995266975.81084848298', '-5.34450465748570290833221890995266975810848480874709007912582866377254e+35');
    T('364174039086712731', '1199801.7839398845629477125742176401101168032886363303997296', '364174039085512929.2160601154370522874257823598898831967113636696002704');
    T('53877990578308734625625639885060125168528280779536508786', '103516583128.45068629304281452459340', '5.38779905783087346256256398850601251685282806760199256575493137069571854754066e+55');
    T('3.754616214', '-371.3860189917370620745548727151697481689878945779565644', '375.1406352057370620745548727151697481689878945779565644');
    T('-437926015567097757255992979555892654.76', '-176655130153455.0051', '-4.379260155670977572558163244257391997549e+35');
    T('-0.1515104603550', '4834028992891471292.78', '-4834028992891471292.931510460355');
    T('2.585364', '23561717378268.26567934', '-23561717378265.68031534');
    T('-0.00000091447444444639', '-14131860317888699752291325342793026389759140', '1.413186031788869975229132534279302638975913999999908552555555361e+43');
    T('-16752710965652120653891175893', '2680897440943339017130745026.27310', '-1.94336084065954596710219209192731e+28');
    T('-7690583538728476436923091784910770916580375622563360.6', '4501948.1501019307848784354357334781717072047589062130609366', '-7.6905835387284764369230917849107709165803756270653087501019307848784354357334781717072047589062130609366e+51');
    T('-210025330495963043.1263117974098', '4.0408', '-210025330495963047.1671117974098');
    T('0.000000000000000479418012427570596658200595436786', '-0.000000049076192452', '4.9076192931418012427570596658200595436786e-8');
    T('-4960517.52677', '-0.20046636670669982714317739081794719428221226519297075795', '-4960517.32630363329330017285682260918205280571778773480702924205');
    T('0.00000000000000288393352959940194003685227429', '-8.192356703062703306084975083202', '8.19235670306270619001850468260394003685227429');
    T('-0.000000028227155496211712', '-0.056083781', '0.056083752772844503788288');
    T('-839454439636.987095', '775436321034387281261238203040959529197645939448222.14142187', '-7.7543632103438728126123820304095952919848539388785912851687e+50');
    T('0', '0.002509727311267980178283', '-0.002509727311267980178283');
    T('138194000652491447330479606912366232', '51353410500300307.8188174125', '1.381940006524914472791261964120659241811825875e+35');
    T('-0.0000086433365715041098678710807336961726259747239597811694878263', '-145268834.890490551072993367', '145268834.8904819077364218628901321289192663038273740252760402188305121737');
    T('19818.037591821397251595707488', '-14488431569410083120931256684496228387192907.10558580383653', '1.4488431569410083120931256684496228387212725143177625233781595707488e+43');
    T('-18346363094160301954515801173824097.520003457764168574', '-132648009061.542335857', '-1.8346363094160301954515668525815035977667600764168574e+34');
    T('0.000000097667699203314068161184', '-350287455.727299', '350287455.727299097667699203314068161184');
    T('15800829403166795.5956516579102376267581193644', '29.54785', '15800829403166766.0478016579102376267581193644');
    T('-195433133492107651439595.4648774', '135838131242484112792610949874.2647657738877701854426', '-1.358383266756176049002623894697296431738877701854426e+29');
    T('-2', '-331826679660519916423352740752709.621460214', '3.31826679660519916423352740752707621460214e+32');
    T('412805093083.82448845946357429243448308860021150924', '-4809048739065577.4431952808766327302', '4809461544158661.26768374034020702263448308860021150924');
    T('-0.00000000000235767643592123724694037105632168276114465', '-631328471724074335984942302107.380800425094037515623490', '6.3132847172407433598494230210738080042509167983918756876275305962894367831723885535e+29');
    T('-0.00000000000000220460849133312061988', '-323463691145097046132093042761677037355966973.1622930610540', '3.2346369114509704613209304276167703735596697316229306105399779539150866687938012e+44');
    T('-253226218107370.687861922377273342910306', '-52546031966435521927925716976', '5.2546031966435268701707609605312138077622726657089694e+28');
    T('-148850237194.787856', '1335742711780796103930791.895449687698338781835275', '-1.335742711780944954167986683305687698338781835275e+24');
    T('3481814465412.29117434619146758762840', '57.076809325489408713318479840', '3481814465355.21436502070205887430992016');
    T('-15060251059.6808167001235294317015914788204', '-0.00000000000523357091851957064551', '-15060251059.68081670011829586078307190817489');
    T('29418890050482267.571', '1512904619571947700639.092531808891009', '-1.512875200681897218371521531808891009e+21');
    T('-142805667193382230472426114.689990961689308909', '-519593843130556.62290843106349543226083561151272', '-1.4280566719286263662929555806708253062581347673916438848728e+26');
    T('-358638707277120721995733548.388859943630638806306978', '-12.513', '-3.58638707277120721995733535875859943630638806306978e+26');
    T('-0.000000000957413518520', '154616552.5166', '-154616552.51660000095741351852');
    T('0.0000000000001317378310642388269736805357842209597616683098', '-379.8345851707018360080', '379.8345851707019677458310642388269736805357842209597616683098');
    T('1372405.3778516179192341', '0.000000000000002163', '1372405.377851617919231937');
    T('61969275419603.4874854962928918', '5690625807302485253835751613390387468762288135', '-5.6906258073024852538357516133903254994868685315125145037071082e+45');
    T('0.000000000000000000034542917777656343403295447877924', '-61797209717624749784326365240476675880321846180162004605.08', '6.1797209717624749784326365240476675880321846180162004605080000000000000000034542917777656343403295447877924e+55');
    T('149340986382607394618252', '-348422861056802876633.244173070463082440851567934244', '1.49689409243664197494885244173070463082440851567934244e+23');
    T('21.0305752637461994', '-0.00000000000000000018', '21.03057526374619940018');
    T('0.000000000000245763391729127989068640916678750088674', '3273.6525', '-3273.652499999999754236608270872010931359083321249911326');
    T('8345432957379892013764872946202.3591286636257', '775586.72252749606', '8.3454329573798920137648721706156366011675657e+30');
    T('0.0000000009480848572374298346871921803725380637', '0.00000000000000911217413808095850740763197861728244451', '9.4807574506329175372868477274055944641755549e-10');
    T('70880626.4544855056079364096562061524443476869494980365', '89384389228261065.70802344171649', '-89384389157380439.2535379361085535903437938475556523130505019635');
    T('0.000000045482894532333129301508476587210437088400596510400692048', '24.795178', '-24.795177954517105467666870698491523412789562911599403489599307952');
    T('12.868', '-781064.3974167751464737017896493722545380625', '781077.2654167751464737017896493722545380625');
    T('25861.13776', '-13581866495119551449671.4104186135980', '1.3581866495119551475532548178613598e+22');
    T('0.0000000279851374783866294881264604138618', '0', '2.79851374783866294881264604138618e-8');
    T('0.00000000000023206437663193708130977441410825096207', '0.000000000000000000425087532539980230979512', '2.3206395154440454132954343459625096207e-13');
    T('-0.0000000000000009639012508328042527698', '-81245998951741764.163321', '81245998951741764.1633209999999990360987491671957472302');
    T('144172855150098.8332813431897919012738325392504726164868382730', '0.00000000000000673875', '144172855150098.833281343189785162523832539250472616486838273');
    T('85529799866102780989277206081042484508661294846.1678649', '0.000000006974145550375375', '8.5529799866102780989277206081042484508661294846167864893025854449624625e+46');
    T('0.000000000000000000012367964247205144574', '-24.9923163', '24.992316300000000000012367964247205144574');
    T('-44089358978.731833149676054401997', '-3455102051335455388472971892.2931', '3.455102051335455344383612913561266850323945598003e+27');
    T('2736316.76004751504759105989955248293683757', '8624.126564458951823396869380153', '2727692.63348305609576766303017232993683757');
    T('-7512235696480.75815974687820', '4322425432060440.112621890781', '-4329937667756920.8707816376592');
    T('1041115705901246425717958893168', '192120533687402856766617056619385', '-1.91079417981501610340899097726217e+32');
    T('0', '2999934433827046995221465.734220582881837016702755103103369', '-2.999934433827046995221465734220582881837016702755103103369e+24');
    T('9132760.7806176704193036626898196556957735479237133471', '-22308955.789058912', '31441716.5696765824193036626898196556957735479237133471');
    T('3', '2510378.8302', '-2510375.8302');
    T('-601681273.341517', '0.000000241648614891714420578', '-601681273.341517241648614891714420578');
    T('7267886123144958232.238682838203', '6452575385170169.8619732325801597156212862621827', '7261433547759788062.3767096056228402843787137378173');
    T('1078231543180', '-48380974651269509996197011777331635750068803914.42093022312', '4.838097465126950999619701177733163682830034709442093022312e+46');
    T('-1565742468.40376463', '0.0000000000010050045763134196571949420320006727965700414589703', '-1565742468.4037646300010050045763134196571949420320006727965700414589703');
    T('-38.8392135498095373389405034839742338375432749653', '0.0000000000002005139891413163385623', '-38.8392135498097378529296448003127961375432749653');
    T('-548507159746966309439370232114407604859278960168.292101', '199162629302104211276079352651591547730479754027.137688159010', '-7.4766978904907052071544958476599915258975871419542978915901e+47');
    T('-133758998429281991365.53', '66349112060274991956125.55718817444465', '-6.648287105870427394749108718817444465e+22');
    T('-39898963484560669919956991873123027250358.023847', '473431260689632044598244688486253.757', '-3.9898963957991930609589036471367715736611780847e+40');
    T('0.000000048220620061585093302526746146', '11163285812734658091858.1', '-1.1163285812734658091858099999951779379938414906697473253854e+22');
    T('122477777.735604874', '1468489954', '-1346012176.264395126');
    T('9276934317177450323278091141313420338701', '0.00000000001807544083251601944163916296938991', '9.27693431717745032327809114131342033870099999999998192455916748398055836083703061009e+39');
    T('-726473085520805.68375', '-0.00000123946378123648562805', '-726473085520805.68374876053621876351437195');
    T('-862282453600744233837865549353.0199427', '-778120.1206481', '-8.622824536007442338378647712328992946e+29');
    T('181724674499441806352104852243619274064166.468709530648193', '87301023839580890632.74008889', '1.81724674499441806352017551219779693173533728620640648193e+41');
    T('6267361382130301.64571563169641350047', '-26946623311829959264900239615329966386.978158853097020829667', '2.6946623311829959264906506976712096688623874484793434330137e+37');
    T('9277033429827', '-11814312598907199463437424804147161225245724986.3549313', '1.18143125989071994634374248041471705022791548133549313e+46');
    T('0.00000000000000002212994406752', '506633182883598094135216', '-5.0663318288359809413521599999999999999997787005593248e+23');
    T('-1416.8382390011', '-616915584637554', '616915584636137.1617609989');
    T('173.67513129697677560331958844251459', '1761111548897687055316514264676906704871', '-1.76111154889768705531651426467690670469732486870302322439668041155748541e+39');
    T('-67316184421669659743.09478', '-247379520759715899369976143088432782692.1213428655', '2.473795207597158993026599586667631229490265628655e+38');
    T('-754359424425.325316286', '0.0000000000583', '-754359424425.3253162860583');
    T('-10745411.84893952612941343946024114284555985539357467110', '-2141617925741482283282941658235570196', '2.1416179257414822832829416582248247841510604738705865605397588571544401446064253289e+36');
    T('0.0000000005131824448956120259143731094456835119548321563889982', '0.0000000000261047578282225790207950702922733665519445636593133768591747', '4.870776870673894468935780391534101454028875927296848231408253e-10');
    T('0.0004478755432082113312483869192', '0.005599339808981958070363083539134166424719944', '-0.005151464265773746739114696619934166424719944');
    T('-0.000000000000001512526283647', '-0.0001153048049432909602774', '0.000115304804941778433993753');
    T('88012539757.9404775939244071255700825395712721930057171', '0.000000000008035455856306225847392238301065068', '88012539757.940477593916371669713776313723879954704652032');
    T('0.00000000000000000073487390383290696928688042616356700', '-0.0000000000000000019730868218008762078515130641383111305700506367751468707226', '2.7079607256337831771383934903018781305700506367751468707226e-18');
    T('54733464273.704359284996374313377912882194031', '0.0000000000000066', '54733464273.704359284996367713377912882194031');
    T('0.000000000000002583138166590499538768478879737734800', '-1493063867598060178500693.39747107', '1.4930638675980601785006933974710700000025831381665904995387684788797377348e+24');
    T('0.0000000000035', '16749083849855920068346', '-1.67490838498559200683459999999999965e+22');
    T('-0.000078422658166136557055297306793562873739079876517', '7.9665125435', '-7.966590966158166136557055297306793562873739079876517');
    T('0.000000000000000004026190822483046169679424015359624776872534298', '-6085341714.088449383656', '6085341714.088449383656000004026190822483046169679424015359624776872534298');
    T('-5.0471797394110', '-3508848877366497564225222.8325257807827', '3.5088488773664975642252177853460413717e+24');
    T('-1495955602856711738.322374249597444877184247', '0.00000337196217180', '-1495955602856711738.322377621559616677184247');
    T('-104.2132489251390', '445169760230242591585531000760.7493281224701823786204109196', '-4.451697602302425915855310008649625770476091823786204109196e+29');
    T('-11595.7728900', '0.000000077', '-11595.772890077');
    T('5693501895857569.824083133994650783219625725263527941848014', '38554113375996.91610257332', '5654947782481572.907980560674650783219625725263527941848014');
    T('-30342286.086738328729', '-0.003035138441', '-30342286.083703190288');
    T('23.0418', '-21373402684893068914382744570171080639', '2.13734026848930689143827445701710806620418e+37');
    T('1', '1792759503737685636350819178488081433275948', '-1.792759503737685636350819178488081433275947e+42');
    T('-38470355434482826491837564169', '-5414067362036587155177091588060858998697264.4', '5.4140673620365486848216571052343671611330954e+42');
    T('5376267017268469803391742461123.1484717294535169', '-3003564661167.71329437830040953302813675659856575', '5.37626701726846980639530712229086176610775392643302813675659856575e+30');
    T('242777368241805037089.180775943923462029927', '-0.0000000000102172171156058275649815', '242777368241805037089.1807759439336792470426058275649815');
    T('-0.00000000000000008656604284642653008106731777826037099', '-6154.026848568460638', '6154.02684856846063791343395715357346991893268222173962901');
    T('-25.0959689', '0', '-25.0959689');
    T('4.4106469958140159750720', '-2.7', '7.110646995814015975072');
    T('-1191115481.24960789843322513194152993', '101480579638.7', '-102671695119.94960789843322513194152993');
    T('-64335581449382102148651961778', '-13922618962022310.38526581442116512370091', '-6.433558144936817952968993946761473418557883487629909e+28');
    T('0.000002419556771906375756783331125070178732526767', '-704659820.24047391470481450387646', '704659820.240476334261586410252216783331125070178732526767');
    T('91086229113018291.85868623557295042563793418491', '-6412976188024708199931810226.6257', '6.41297618811579442904482851848438623557295042563793418491e+27');
    T('-166376886435906071.671676003286240157545425223534', '-0.000000000000091700715807851859225796752901241413659885', '-166376886435906071.671676003286148456829617371674774203247098758586340115');
    T('-1604178602185.15391140031008758283', '-0.003382166434467926234482400431856589755224770', '-1604178602185.15052923387561965659551759956814341024477523');
    T('-1550.5067982', '1755233709470197458.5321953822', '-1755233709470199009.0389935822');
    T('0.3014', '43133762452399944536671349566322158.93626365', '-4.313376245239994453667134956632215863486365e+34');
    T('-64.5979', '-0.00000150928436566222094395688472086656530077', '-64.59789849071563433777905604311527913343469923');
    T('-771717.603678878', '0.0000000000000011883865777330973023055360', '-771717.603678878000001188386577733097302305536');
    T('0.0000066980807778368470000446256117729524345121272595145987', '-32254507023.88915752082', '32254507023.8891642189007778368470000446256117729524345121272595145987');
    T('-3.896651617783010917182197706278524524091', '-234356.4491086908087268932525516', '234352.552457073025715976070353893721475475909');
    T('0.000000000000000000365309384905118249080774978958', '-2593982058025640086344149662711100252281819944.0726056015', '2.593982058025640086344149662711100252281819944072605601500000000365309384905118249080774978958e+45');
    T('-807268846295783.66565769288073589114', '-9279.26445753', '-807268846286504.40120016288073589114');
    T('-0.000205066477824679946712948821846487595173765417878', '-11856312001.2579', '11856312001.257694933522175320053287051178153512404826234582122');
    T('21715853560267741681005732.8864129839650731630056482243802', '-26518465662155265718064844202342', '2.65184873780088259858065252080748864129839650731630056482243802e+31');
    T('-0.0000000002358578440972167', '214498383035810423', '-214498383035810423.0000000002358578440972167');
    T('4361467034560145704861429904045767902851.92573820902038', '3.59', '4.36146703456014570486142990404576790284833573820902038e+39');
    T('-5399354467819638876391446017237518290882302', '-9418844852716336037742898633771821467185538647360652378656.5', '9.4188448527163306383884308141329450757395214098423614963545e+57');
    T('-0.0000001178659835758080442880526500344288949201205201989828562345873', '3789033099896128122043510040864191.7055425399', '-3.7890330998961281220435100408641917055426577659835758080442880526500344288949201205201989828562345873e+33');
    T('0.00000000005982976503399938136849324', '0.00000000000075755295405642132036', '5.907221207994296004813324e-11');

        T('-421450565450862733739882741263410125987400848651432120569319841269419221990711610801409.8070215252564874', '-243392659607204009035334731041415676417559363520670299047012358737705418584457146.868335409', '-4.214503220582031265358737059286790845717244310920685998990207942570604842852930263442629386861162564874e+86');
    T('2995340172136827282681082940837304688534605164044247935816574334297096539991453139359346773494887259928780399192576191569246077843624156140842225', '-1221176047834651784581078792412765867711628546241044712073657285378727907841444910656147964990491084587511386617.262995074491', '2.995340172136827282681082940837305909710652998696032516895366747062964251619999380404058847152172638656688240637486847717211068334708743652228842262995074491e+144');
    T('66496777.1048248699226846854182647353383780982905985351644739', '611110887295764011339.52942021906792320951632532924734194110935652615437372802282890281281736794257440312255694791', '-611110887295697514562.42459534914523852409806059390896384281875799098989982802282890281281736794257440312255694791');
    T('-1008448844147', '-0.0000000000000000019476287', '-1008448844146.9999999999999999980523713');
    T('-0.000000000000000000516432305268991528777659423596262', '-298372128925675105602022884944348.2680420889095373857666394', '2.98372128925675105602022884944348268042088909537385250207094731008471222340576403738e+32');
    T('-123500072490134997.714568930708754033518940686118181893673033346987867', '-2578011852826744661373892702884.2218394002814041283307474989342835625859515222796606255126639775600582467053535947189955421841414538276875776982', '2.5780118528266211613014025678865072704695726500948118068128161016689129181752917936255126639775600582467053535947189955421841414538276875776982e+30');
    T('21874922820040.11669694092724059036160007405283774815020181046771169772206306814891484445951224460904841652605502865721442462130288010', '31200659.3720643364048896004458800281137277425602565342', '21874891619380.7446326045223509899157200459391100055899452762677116977220630681489148444595122446090484165260550286572144246213028801');
    T('-67826336382297566597618786205691264038459363425175950127924836604587803884808989.730966693870183767', '-0.000010934566312291377545830845045333824309160616881090991', '-6.7826336382297566597618786205691264038459363425175950127924836604587803884808989730955759303871475622454169154954666175690839383118909009e+79');
    T('-86697937667412', '-0.000000000000000010885508014854389736820644838727265143960231771135151634179626636837681706789395306987969756304048180103464002627364375565746495636426525896', '-86697937667411.999999999999999989114491985145610263179355161272734856039768228864848365820373363162318293210604693012030243695951819896535997372635624434253504363573474104');
    T('13239420394588114639023745289048683419202338196760858236726250820537420580804213017510790278292290014782026362194', '-0.0000000000000006743804370173243035185881087573', '1.32394203945881146390237452890486834192023381967608582367262508205374205808042130175107902782922900147820263621940000000000000006743804370173243035185881087573e+112');
    T('15390.4', '4140709577543085939068096954669836524172838073868529566429367862425997374799588343038360402928322145938978599798868.9860038387559873', '-4.1407095775430859390680969546698365241728380738685295664293678624259973747995883430383604029283221459389785997834785860038387559873e+114');
    T('1068041657610774618407195645889793312307387843651159034936031674459010136784601252596838684187431642727503640875306262502315681768438', '-270043310707338335874517438680428125320070134440535322728713924276969', '1.068041657610774618407195645889793312307387843651159034936031674729053447491939588471356122867859768047573775315841585231029606045407e+132');
    T('-154248552738364025256732966577155380446860806518778173860697213762291995224.05508159477619681374', '-0.004301153450587648414307336514788194741568895974277211381969307271', '-1.54248552738364025256732966577155380446860806518778173860697213762291995224050780441325609165325692663485211805258431104025722788618030692729e+74');
    T('-63378626295.9', '3581805238164539487164037240742370503938629203589842975477397652', '-3.5818052381645394871640372407423705039386292035898430388560239479e+63');
    T('-4278738506861855303803.685941371999517217112263860961', '1', '-4.278738506861855303804685941371999517217112263860961e+21');
    T('4722957382126750592971015303833730398703112.9325022739644', '135935655079909511034147686928535747252480216156484264244569583366838078412.147497624', '-1.359356550799095110341476869285310242950980894058912932292657496364393752992149953500356e+74');
    T('-0.085968464304838528608760871311794870568574888285345150515011250264989219693613274665433658244439289382746661487914586830547', '-1063290105646753333057649979045695307527992816411892112479790721669979250549466052710783805392449284495837713080894121068105045830138816290.52', '1.063290105646753333057649979045695307527992816411892112479790721669979250549466052710783805392449284495837713080894121068105045830138816290434031535695161471391239128688205129431425111714654849484988749735010780306386725334566341755560710617253338512085413169453e+138');
    T('16124498471436271836.174203105207953106915', '-0.0000000000000000024763633167036725188984815540228360744116272407116469322488952920669576143580628085795995666173708903799390536974903000256740969', '16124498471436271836.1742031052079531093913633167036725188984815540228360744116272407116469322488952920669576143580628085795995666173708903799390536974903000256740969');
    T('13060329629573076844480853604358182661217396692808889824.8934224473655695659109683301950879985310417264401844104664883', '2655975253497367378557314381561996.880511919813781979361114586185190856887597866439252779127105', '1.30603296295730768444781976291046852938388393784273278280129105275517875865498537440098971416434438600009316313393833e+55');
    T('7828918777589631958673609756697717233158380319992892621665', '664080.907156046062276770226566166774715944049', '7.828918777589631958673609756697717233158380319992891957584092843953937723229773433833225284055951e+57');
    T('1102045632452.9793048891721127647873110178015535129097555045251534115765524554358921107344361810504392212504634634', '10480858160855531623861683.3953304547', '-1.04808581608544295782292304160255655278872352126889821984464870902444954748465884234475445641078892655638189495607787495365366e+25');
    T('1588237583004474241570674574155806135977994054265831583868733852477199751.6125369120506144977', '-1758643150655173823782138162406249816325114275744255398376729480619194524', '3.3468807336596480653528127365620559523031083300100869822454633330963942756125369120506144977e+72');
    T('-121908802598585866222616306604158404572834073196639.30608547355363660580248376236860407669299263592702', '-47300202299798563445786841751110283088211407088073898239866395', '4.730020229967665464318825588488766678160724868350106416666975569391452644636339419751623763139592330700736407298e+61');
    T('0.00001018', '-0.0000000000000000003775949557336735085158573036212360452188352127143755001043595772762253543341426632340961625924634823303829450325116', '0.0000101800000000003775949557336735085158573036212360452188352127143755001043595772762253543341426632340961625924634823303829450325116');
    T('-397079706875731796310420230640137723941921196836019.63', '-0.041045084177321369958383091589758576', '-3.97079706875731796310420230640137723941921196836019588954915822678630041616908410241424e+50');
    T('-25255670718458145585125947408178.20003200206718595937', '23790086654614644315060006023753790313128255720789244411931242762019828811933845880517294907262', '-2.379008665461464431506000602375379031312825572078924441193124278727549953039199146564324231544020003200206718595937e+94');
    T('388216212.4067014280868903953571410637325737919396528867014154838280455505188062329013057244757878', '606945158458429249401712430149203335884502730947097871424092354090910701814671093294150735256.311971299391789036587867972923', '-6.069451584584292494017124301492033358845027309470978714240923540909107018146710932937625190439052698713048986412307269091904262080603471132985845161719544494811937670986942755242122e+92');
    T('-13758420895788683271912900568609892935336486526537239092874365', '0.0000000427931529718203244548288038739877932413390488514893154906501275029696961218603307975431172274454', '-1.37584208957886832719129005686098929353364865265372390928743650000000427931529718203244548288038739877932413390488514893154906501275029696961218603307975431172274454e+61');
    T('0.000370061363293', '-0.00000000729718323650351200936805011200230897331673895494582965292153458641292454418451631', '0.00037006866047623650351200936805011200230897331673895494582965292153458641292454418451631');
    T('-0.000000000000000037267063363570380515693697474002406633393933', '5371503530945116760368005450503842267224225935.443693351128547707505116811332630555129489463135', '-5.371503530945116760368005450503842267224225935443693351128547744772180174903011070823186937137406633393933e+45');
    T('987277696994412694084034600111639245624814061790126142560388376636239445837.3', '0.0000021235425818843833696026604635378067565443474', '9.872776969944126940840346001116392456248140617901261425603883766362394458372999978764574181156166303973395364621932434556526e+74');
    T('-0.0000220122231289032322357369232053897035669971251749680777769003202659812319155442107', '3509368133759059638669289822772173467.5789662072733463078733831264876196643926418400220747139696586172005930587312837', '-3.5093681337590596386692898227721734675789882194964752111056188634108250540962088371472496820474355175208590399631992442107e+36');
    T('-0.0000000000000035350393604865398332629110742215182474841758918373895373399045442910854670022381868595891655052656143931905987096001246764946455093272', '0.000000000000000024560182', '-3.5595995424865398332629110742215182474841758918373895373399045442910854670022381868595891655052656143931905987096001246764946455093272e-15');
    T('-207299473460686051638799737978676287687756682811506270896244029691640707953363231798114058314798954715497689478762298979167444328.2154321', '84.9875755884409473552243470559913666018369306750346853705848251516641143427438599427140088', '-2.072994734606860516387997379786762876877566828115062708962440296916407079533632317981140583147989547154976894787622989791674444132030076884409473552243470559913666018369306750346853705848251516641143427438599427140088e+128');
    T('3271753269852982202221073139870131965618400215989200201303890204830390736963230557181963499423.97161923384826121516522802454125216009692783106615', '610709896713545517320.61325351307180550727524165198', '3.27175326985298220222107313987013196561840021598920020130389020483039073635252066046841798210335836572077645570788998637256125216009692783106615e+93');
    T('-171890925468345318289557520417686524897898778387.7049747769377054087', '0.0000000000008626202220397871613368552810353647018975957787995640929353071154146588942452406480447534761421336774021631535188253352941985865', '-1.718909254683453182895575204176865248978987783877049747769385680289220397871613368552810353647018975957787995640929353071154146588942452406480447534761421336774021631535188253352941985865e+47');
    T('11797561306382075049308870675243395524960617768982687731904993854819896455053370294329059.5', '-9266409282787978360416645397432756233517921427389.714082569117296693989915074785029952731114155890581220089683737330882', '1.1797561306382075049308870675243395524969884178265475710265410500217329211286888215756449214082569117296693989915074785029952731114155890581220089683737330882e+88');
    T('-424430636249988989578724500467105386916790551644026356512230425657196415575958762209685893722637592.42515458907522545588092531969', '-699146481762025356180846176956775322378040371282973135438056771990065583612223969221925059327626424485', '6.9872205112577536719126745245630821699112358073132910908154454156440838719664801045971537343390378689257484541092477454411907468031e+101');
    T('-0.00000130798527635604695258712844244105224418951215845097932279572456369133399455', '0.000000176284029965375880004181656007589949467118245220868298039483413025781635383510797', '-0.000001484269306321422832591310098448642193656630403671847620835207976717115629933510797');
    T('0.000000000000462071916734680776281054', '1143549992221513568696244482022549056707895554082064.282542', '-1.143549992221513568696244482022549056707895554082064282541999999537928083265319223718946e+51');
    T('0.0000000000000000426800518382365970695662273350350877364291904795717019319799963342380112501925257351654', '0.0000000000000013278679842106035272342072', '-1.2851879323723669301646409726649649122635708095204282980680200036657619887498074742648346e-15');
    T('-0.000000000000000000016354802329513866', '122022209857.15312653353983161651403936785710765909662481656738699983151473856575442647972705932905488645338136045162567147688725', '-122022209857.15312653353983161653039417018662152509662481656738699983151473856575442647972705932905488645338136045162567147688725');
    T('25801628293435628816490256345279207687083725145006124892029160063238265267.3431144744384935936819525999160077311803590506825603145972954326789291', '-285119248451148500201319196762968888442368918233718192676490082873043334950267626673646944070692473524519642', '2.851192484511485002013191967629689142439972116693470091667464281522510220339927716797718360998525367627849093431144744384935936819525999160077311803590506825603145972954326789291e+107');
    T('0.1423', '2429833169520732752372574762485088466742598273119469284544035996929667546050321271916945887077199411341346293601.72939677', '-2.42983316952073275237257476248508846674259827311946928454403599692966754605032127191694588707719941134134629360158709677e+111');
    T('3.80188793676049024809', '0.0000000000002833272682244473387924044230539081036171682212105326649350597590901815021244717391629977491943712364070741264536987011350709308', '3.8018879367602069208217755526612075955769460918963828317787894673350649402409098184978755282608370022508056287635929258735463012988649290692');
    T('-322814450973877874279521.0', '263039808493736512406179831713394337242295861157184.672295063', '-2.63039808493736512406179832036208788216173735436705672295063e+50');
    T('398742236640963', '44.00495510581548851408966', '398742236640918.99504489418451148591034');
    T('-3573583864.539627856711706646067988209079651520670851811891852561', '-0.00000000000000000329278234017133237246586831901282091949913572717292793996445894343939371558967511917774705488676891269560643860123317086289974802538004355597691349812', '-3573583864.53962785671170664277520586890831914820498349287903164150086427282707206003554105656060628441032488082225294511323108730439356139876682913710025197461995644402308650188');
    T('1912764785977618050454430552593203249088615699', '-485515908236945825.82230967655356283052002508081428895192743749813173660196480495243134130534734417501263634', '1.91276478597761805045443055307871915732556152482230967655356283052002508081428895192743749813173660196480495243134130534734417501263634e+45');
    T('-209.9480474748934376775111457605595206860865832017099805042786198516620696960966458436798888919348', '-539891303089085349109418789373221841481476900927418909970947.98689276127478239299357632782372174032445906', '5.398913030890853491094187893732218414814769009274189099707380388452863813447154824305672642010542378758582900194957213801483379303039033541563201111080652e+59');
    T('14686906472232406033983442617897900163963974602', '-1420622706271964656354950704.1697326665068929806041828344182380198712418608178840095762', '1.46869064722324060354040653241698648203189253061697326665068929806041828344182380198712418608178840095762e+46');
    T('86543050569726858572005888237520502257116260484789853410061111403475802158093598826908247486.836335533846125875491403509186859062493992', '-0.0000051065665739422395774587551282612336173295903645593309699764793560943989004198255241', '8.65430505697268585720058882375205022571162604847898534100611114034758021580935988269082474868363406404126998177309809679419873237276093295903645593309699764793560943989004198255241e+91');
    T('342895773116306829110212528967784741562164292.01513115088269038806033385850', '0.00000118862709647506650871551520859000440835170689555307064827601186026157168885360722702508608962915698632770928960656597545', '3.4289577311630682911021252896778474156216429201512996225559391299382514298479140999559164829310444692935172398813973842831114639277297491391037084301367229071039343402455e+44');
    T('739924109312422522091016571038779963072584', '-64775259.6826186518884824872925596155743958419605122483453848814970771390948590513454670', '7.39924109312422522091016571038780027847843682618651888482487292559615574395841960512248345384881497077139094859051345467e+41');
    T('6609838981181176735659252.7740697649501200112', '-11286186232989566466114315553164377878561114785839243669268249646267418129841210.35980783715095457778535932', '1.128618623298956646611431555316437787856111478583924367587808862744859486550046313387760210107458898535932e+79');
    T('-187391162221219921070460320744240172407912239353798545547332717684757305669457677296318215340489442645.28366529601614598850373', '-9113.9748812203681488412701629366868978775543399390741272418220574874035442918413045437197807171343403867576380368103250028', '-1.873911622212199210704603207442401724079122393537985455473327176847573056694576772963182153404894335313087840756479971472335670633131021224456600609258727581779425125964557081586954562802192828656596132423619631896749972e+101');
    T('545567297425658900059033345039628434103460646144205348817463850197715173654.760035446992603845914884479975954101703', '-3.25531678994006', '5.45567297425658900059033345039628434103460646144205348817463850197715173658015352236932663845914884479975954101703e+74');
    T('41005341883885124910253307373863933024572386393999832504926014424.04068230757553061685965194104847256614930594539431445372085538790197', '-0.0000003064573352368766798045131330999970376084101941252529063279240676746738279894937999916288137325237148909161534147848901003719756893653182826', '4.10053418838851249102533073738639330245723863939998325049260144240406826140328658537363317455616056661463435538045085789737617158260376746738279894937999916288137325237148909161534147848901003719756893653182826e+64');
    T('-21772543213224719483282057502996460271243910021555859843788194322560764800215622961743791', '-219538453980808949925972347106117120906813214436418773865178519764872595507494270091275.549469440429227', '-2.1553004759243910533356085155890343150337096807119441069923015802795892204708128691652515450530559570773e+88');
    T('4843564632.07325685271287267587642074472329955', '-2253.1', '4843566885.17325685271287267587642074472329955');
    T('-1465639293224611258.1640919252320447', '0.0000000000000009272840616057150840199792579106702667313583800294803029238640296064513396998901049886143310419234155012846386', '-1465639293224611258.1640919252320456272840616057150840199792579106702667313583800294803029238640296064513396998901049886143310419234155012846386');
    T('46668145279017817857972034174120636924960629989668564040596683624271865534.57', '84.9', '4.666814527901781785797203417412063692496062998966856404059668362427186544967e+73');
    T('-738241543348122522391845849281775440907729860333.816380311486149269839379', '0.0265469560323666758057467119830695293620004914159121944165101243889371879961', '-7.382415433481225223918458492817754409077298603338429272675185159456451257119830695293620004914159121944165101243889371879961e+47');
    T('0.000000000000000001734597550329925170683988212040626455377015149950764555153937498632499946117886619477443', '21522778596700039662643518332810996834742527241604682606746145945739181821479030002316968608237.843101668183269553916654985900396859343464324', '-2.1522778596700039662643518332810996834742527241604682606746145945739181821479030002316968608237843101668183269552182057435570471688659476111959373544622984850049235444846062501367500053882113380522557e+94');
    T('-0.0000000000000000000309703805445398610474900212495574298677865146433319735295565686883503979155963415371001608918561542864325055703887850430784945642120545934153', '-5.6227', '5.6226999999999999999690296194554601389525099787504425701322134853566680264704434313116496020844036584628998391081438457135674944296112149569215054357879454065847');
    T('-5064143323681064652393632163111.93089333677771516967546169174265515779023117868686275683397034493611565', '-0.00000000001073777696437370169', '-5.06414332368106465239363216311193089333676697739271108799005265515779023117868686275683397034493611565e+30');
    T('-1396826091243475193.1', '-1288002403964300782016794952204121336647057494337679233.713597178033979199448274404277214801719513817499902870656', '1.288002403964300782016794952204121335250231403094204040613597178033979199448274404277214801719513817499902870656e+54');
    T('364339635.011212286283196681723183929801097146881219006560017', '-1005720700996298', '1005721065335933.011212286283196681723183929801097146881219006560017');
    T('-257562976090545385234669132851624833292427282439090020.719343974047857466853161243072384425509047149775370947901720407174956846928296202473095472', '-1623127319151265475871704129654550871405407416881602056343569964910663809884633229508196329223823041.7206810159931522079791856822', '1.623127319151265475871704129654550871405407416624039080253024579675994677033008396215769046784733021001337041945294741126024439127615574490952850224629052098279592825043153071703797526904528e+99');
    T('0.0000000000000220607316968906929903970883560258897819751208683268360364851788263', '0.0000000000000000600784639775121406482942932372611353', '2.20006532329131808497487940627886286466751208683268360364851788263e-14');
    T('-0.000000000000043989086', '-44309230851677894984939938565089867097397779157566647630330915058596891092311151485774600807340241541392574864911717487.862518042789404', '4.4309230851677894984939938565089867097397779157566647630330915058596891092311151485774600807340241541392574864911717487862518042789360010914e+118');
    T('-0.000000482926598434254495862080721294034240317086724705402508463332112037991339240729385845354673866033075192672349090063032508', '-131360031110406516192037002195365556732148135932795271282547111275387791270050177023234968875866030325638578', '1.31360031110406516192037002195365556732148135932795271282547111275387791270050177023234968875866030325638577999999517073401565745504137919278705965759682913275294597491536667887962008660759270614154645326133966924807327650909936967492e+107');
    T('90662704837359221983067383218111894.9575101269344703639471125782669325117231999424752935395342', '-1003082878257466499301963750949455756319801133590.9096038135671589349331563057210', '1.0030828782575571620068011101714388237030192454858671139405016292988802688839879325117231999424752935395342e+48');
    T('8122241424803915907910045579802249996527160493328189939983695513719757201277358369302040281397660429815708520244075564900409812', '5953815721349504215933972982145270510517473063697276921387486344144210713.78', '8.12224142480391590791004557980224999652716049332818993402987979237025298534338538715676977088018736611843159885658922075619909822e+126');
    T('7991350014483385014452400954462003012335870376818921107933891647581751944961549909381574328015589839.2322422374080381874344210390470431707892', '-18688.878545993875113185968540055115951056413820404', '7.991350014483385014452400954462003012335870376818921107933891647581751944961549909381574328015608528110788231283151373402961094162994227203020404e+99');
    T('-1201549207285023471623288363.093077137984596708144478749068', '0.00000026539824642010892262104442818872329795607002102749766710877993859556577972466454691281569600833952883562418043758499574880445814224717667678262548541', '-1.20154920728502347162328836309307740338284312825340137011242818872329795607002102749766710877993859556577972466454691281569600833952883562418043758499574880445814224717667678262548541e+27');
    T('75311239722806498991168677625201999153196464897386996233710029214675693542594584.030032275297208734980242411305757844200195419550995808', '0.00000000000000174652609458505387986677857979923', '7.5311239722806498991168677625201999153196464897386996233710029214675693542594584030032275297206988454147826251877977421615620320995808e+79');
    T('0.0000000000000353682715117343396008139392193804645279990939015187625700163231712251406332', '0.000000000113523402513085710775666504535102628539977357555114115140', '-1.134880342415739764360656905958832480754493584612125963774299836768287748593668e-10');
    T('25393267156020697622507966272559237.59641792718917124986994778365558647150342934845113943261397901868166533972121', '-8953602751519594072912851200714566873449464820520095981074552864073057275.64400277107712961954031710933646037', '8.95360275151959407291285120071456687347485808767611667869706083034561651324042069826630086941026489299204684150342934845113943261397901868166533972121e+72');
    T('-109621598899109119.681443742531046712086897', '-374581278609124761376087334567960408592166724187316443183587025428047356270871954503065666641000369697417943372656053785.43872597385309997628624757', '3.7458127860912476137608733456796040859216672418731644318358702542804735627087195450306566664100036969730832177375694466575728223132205326419935057e+119');
    T('-1', '-975353903601003155097156088969595825652369092261504414412601751527209186035985001687896975402318843359120945984358141908733802607901210061', '9.7535390360100315509715608896959582565236909226150441441260175152720918603598500168789697540231884335912094598435814190873380260790121006e+137');
    T('-1010180736624357632304278472376562', '-2976630172749484868944762912267709771588766925127519991889274779878786428243958065464677092268718803221719459273939687', '2.976630172749484868944762912267709771588766925127519991889274779878786428243958065463666911532094445589415180801563125e+117');
    T('-0.0000000000000000003534066629774631966159394411462082995873351999578646015320944827942055231355490842935810', '-0.036044356747079611060903437812032142854350', '0.036044356747079610707496774834568946238410558853791700412664800042135398467905517205794476864450915706419');
    T('-1526096527808476160349609179852946562211.685903646022496376055878254', '1362591651000463812653063829143557737447628829546619971189589.203149513988762118328955262250680582456193660886095627130648934202517770563176', '-1.362591651000463812654589925671366213607978438726472917751800889053160011258494384833516250680582456193660886095627130648934202517770563176e+60');
    T('19320521819464336609965537393303850154554779478967237541.181740342436742642132602701909555424833647068652587', '36.95235866', '1.9320521819464336609965537393303850154554779478967237504229381682436742642132602701909555424833647068652587e+55');
    T('0.0000000000000000059086539273106107233962061788234561229771196283927754983525', '0.000010966647043776077878565685051717652096966', '-0.0000109666470437701692246383744409942558907871765438770228803716072245016475');
    T('0.000000000000000291508913251611559233670098989956434773070295291467160805172311376194998291058162533782938726179866516047606', '-1628909277725099787508683923508579707615799.2781649599495440150562819682935976323342667567117971951187701859394514400', '1.628909277725099787508683923508579707615799278164959949544306565195219905156866004365746668231968189065477406612245172311376194998291058162533782938726179866516047606e+42');
    T('2456565085671291252533213649183332261726393175508794379993650690041356484115074939702355', '-792071839545569033370633444973336689417838.95905325719912926858163081', '2.45656508567129125253321364918333226172639317630086621953921972341198992908841162912019395905325719912926858163081e+87');
    T('36242208914619642339819083618221621528937544636695183870528988863103659500324120453213846411263678533206476169383042170996012205255558591125924287', '-8527637110089077004296133078.548689219750368210567156616', '3.6242208914619642339819083618221621528937544636695183870528988863103659500324120453213846411263678533206476169383042179523649315344635595422057365548689219750368210567156616e+145');
    T('-288139905642.85133972865494566177561360699121828453190729669018251776515265250985806198704228574404406409996058809130695', '57.685745719', '-288139905700.53708544765494566177561360699121828453190729669018251776515265250985806198704228574404406409996058809130695');
    T('-680590848077096656729099299986105268010775', '-3940756250582632953280085589688456490450830441053358639711015662685.5876458545849231450334623183020123887046847733341586636887427156443214389386142', '3.9407562505826329532800849090976084133541737119540586536057476519105876458545849231450334623183020123887046847733341586636887427156443214389386142e+66');
    T('-0.00000000000000464798477570881362597831391087848642171588956612352664397214025637282503146289569312188996809201235391721184701297091963', '229510668379945446635992190232641551178111333311952316902120658018936303610', '-2.2951066837994544663599219023264155117811133331195231690212065801893630361000000000000000464798477570881362597831391087848642171588956612352664397214025637282503146289569312188996809201235391721184701297091963e+74');
    T('-47431205694992565418930842840793703282717398.191947035122622', '-38262831081463216678995758515.24', '-4.7431205694992527156099761377577024286958882951947035122622e+43');
    T('-919364307978.8988017767137102', '0.054658', '-919364307978.9534597767137102');
    T('-143225400724940647154298336931676918152131842385395309001703075947743779850991836523285204182327721048431248004372787336203288188136955101.42539412', '-0.0000000000036444268460743503614664927361907234', '-1.432254007249406471542983369316769181521318423853953090017030759477437798509918365232852041823277210484312480043727873362032881881369551014253941199963555731539256496385335072638092766e+137');
    T('632955343367277745405583809562259843947443640083859269919841228715604160541151817585859720.153766976201457179988669391500248183047854', '-3980620775419876921723363242387478932830746531218598561304844371368103759418106738578.2191228007189231296352120987296973541145408118698522', '6.329593239880531652825055329255022314263764708303904885184025335599755286449112356925982983728897769203803096238814902299455371623948118698522e+89');
    T('-11431356965911717667550228258416437331889929786766103155708832250235968861451582992625848777060561188238182761522625940473897303', '-21494014792784787595184007767480630.4718359202897210704378425198', '-1.14313569659117176675502282584164373318899297867661031557088322502359688614515829926258487770390671734453979739274419327064166725281640797102789295621574802e+127');
    T('167445439608039555943133.2565349346255919159524753710331727927482546974522844127', '-170610177.84658737421848366816682443401980132194941329175375443204315', '1.6744543960803972655331110312230884407558411929980505297411469766798920603884474315e+23');
    T('13588050303471.14204224509147709748772701071961428', '-2846670239.21559765880387140454090103138633524132996006051956914300933861577438157749445114659138365637629', '13590896973710.35763990389534850202862804210594952132996006051956914300933861577438157749445114659138365637629');
    T('2884217633259642787319342000051603653521516691886608156996898524479398460095322784150599567666195414054636506165396248262312499110274037149829793', '-0.49', '2.88421763325964278731934200005160365352151669188660815699689852447939846009532278415059956766619541405463650616539624826231249911027403714982979349e+144');
    T('1628348053926578504775310683640594840.18', '-5', '1.62834805392657850477531068364059484518e+36');
    T('0', '25.6085', '-25.6085');
    T('0.686217890943093321456902957432427930920753888456197188131905279831038714006759313970367467076944122805245768167433186925922801013913340', '-280.9846437544474336300817147012964188574151285470284551305', '281.67086164539052695153861765872884678833588243548465231863190527983103871400675931397036746707694412280524576816743318692592280101391334');
    T('0.00000000000340332021329324173250202191356270977362746852052557847139210569709283686364254219951529694529651271453552821650417', '-164368911717325046920.73329482756545681454251205873812192944414945415279875808501852199453462518417817132233', '164368911717325046920.73329482756886013475580530047062395135771216392642622660554410046592673088127100818597254219951529694529651271453552821650417');
    T('-516534324070139884.7741879765292957505952198282777909600935487202', '146754452557367377599239860534956890285095395520562623.08414402490725339577004922478313417', '-1.467544525573673775992398605349568908016297195907025078583320014365491463652690530609251300935487202e+53');
    T('-7862150161740.333843293297844', '-3489304238554590513247507065916226966675791294808848752.7395893237178003837558347838402377186885033242010881848399380565235778', '3.4893042385545905132475070659162269666757834326586870124057460304199563837558347838402377186885033242010881848399380565235778e+54');
    T('-76425877353723218923193769976112627809694461006515127338.70903900591342267672', '-81405', '-7.642587735372321892319376997611262780969446100651504593370903900591342267672e+55');
    T('448890821004569959750117.3985925529062982149919', '-0.00000000000006771592275943991478905618644801247941193384606583159209550989858234', '4.4889082100456995975011739859255290636593091465943991478905618644801247941193384606583159209550989858234e+23');
    T('0.00000011406978978113385456966935146505617316701596378729845687370006219616672043338275771066022032618558072376690445085965016785383519759578849021601322', '0.000000001074333633497324260534', '1.1299545614763653030913535146505617316701596378729845687370006219616672043338275771066022032618558072376690445085965016785383519759578849021601322e-7');
    T('0.0000000000000000098388959587739740341162303378089561994855515112821239635081705976158309039469451054135', '-0.000001410700377949704801346056617673576599569410838070841791880586705430927605302815864034641907656163728488162489705472103253731031782190814249810704', '0.000001410700377959543697304830591707692829907219794270327343391868829394435775900431694938588852761577228488162489705472103253731031782190814249810704');
    T('16965', '0', '16965');
    T('-173296003170746074411974480464449849275789060677253187906655130.35621428918277091401054198685991445606', '9987307250642568848222273488.07237677103631', '-1.7329600317074607441197448046444985926309631131982203612892861842859106021908091401054198685991445606e+62');
    T('56313995097156205794032855788026957442153004380906', '1959197690.09', '5.631399509715620579403285578802695744215104518321591e+49');
    T('-16265987991789427405120519655124290086694897782465483955109881892312790.0', '-8472377582734657144678697107349408612484052002424435997452474401137339657179.8354793580669182195980931547215677234501775415', '8.4723613167466653552512919868297534881939653075266535319685192912554473443898354793580669182195980931547215677234501775415e+75');
    T('554861276750786050195093192911576646864838216939018115253478076548406847106024415049015009323070447826851.1693209493824199114881022365592076838779457', '1729203992886560182769799176627696.7030206', '5.548612767507860501950931929115766468648382169390181152534780765484068453768204221624548265532712711991544663003493824199114881022365592076838779457e+104');
    T('2350978155540287801691924042190620635285451442925634874069224914124883396043040648506312047176852810899548676455764006316837438', '-2551506628086094864840341408447994216741146364868142569612506063507487425025110.309931396625986', '2.350978155540287801691924042190620635285451442928186380697311008989723737451488642723053193541720953469161182519271493741862548309931396625986e+126');
    T('13203137085009109554034420.10176018304241579430683160045', '170019441677688.0741576755619011964973604245532745397', '1.32031370848390901123567320276025074805145978094711758967254603e+25');
    T('-91119940320533621629925088838381377761994531140379743426003837742546140176935348401012460941486552655906857745993505840563823144054409171683778779229', '-74652873494135054265048456139803801421904108545479487454387811725432723023321219093564112176252137605.48139182125841144031282511', '-9.111994032053362162992508883838137776199453114030509055250970268828109172079554459959055683294107316845246993426807311754050192496084505950752664162351860817874158855968717489e+148');
    T('8687124396055842976264601096501451863929462701537236230130347130957533866751062788203.6801997865475627493037980272028364235779819787945862854388786', '4393.0547331217320860218042817496323025348168504', '8.6871243960558429762646010965014518639294627015372362301303471309575338667510627838106254666648154767274995162775705338887611315787945862854388786e+84');
    T('4227267383837391491108630568817648.18', '4703737939182568540209790636363.1253794769484641462244847000439730177702702662024029476513485225861537356914086556572097', '4.2225636458982089225684207781812850546205230515358537755152999560269822297297337975970523486514774138462643085913443427903e+33');
    T('1331599892524734478741104974374901736796', '8613986863.5286325031971100693336720522422497083861729312475851625078455', '1.3315998925247344787411049743662877499324713674968028899306663279477577502916138270687524148374921545e+39');
    T('-1435605.5713', '-0.000000000018538927081499603430462441110384554897082244', '-1435605.571299999981461072918500396569537558889615445102917756');
    T('-0.000000000133889283781216077153647008712974175960714836447518299614317834783467849261908546492568082004751707979306122', '-0.0000000000000035759233716787482258438019525200', '-1.33885707857844398405421164911021655960714836447518299614317834783467849261908546492568082004751707979306122e-10');
    T('87163032789315876139411128215473910733429571507063141054764287035611190214823258259', '0.0000755980441879469673888177662328951349071414731847083920686253624381293723357131370215782774531128617325392065045408062', '8.71630327893158761394111282154739107334295715070631410547642870356111902148232582589999244019558120530326111822337671048650928585268152916079313746375618706276642868629784217225468871382674607934954591938e+82');
    T('-9968522225042314632060285988443730732405555397698281720591792230663115171346412535.174789294146936609880713903210346686578323', '0.0000000021461329196842167039414796333960450676424674430116711196401669424125946978994613357227', '-9.9685222250423146320602859884437307324055553976982817205917922306631151713464125351747892962930695295649306071518263199743680676424674430116711196401669424125946978994613357227e+81');
    T('-682416717526953753240801236778041538122363074270539580771563.1595987724242158270047634677372763495475148', '-3227642122668496595741822002740302413422234499976652746605683904.9121550427442753468836876331877102668', '3.2269597059509696419885812015035243718841121369023822070249123417525562703200595198789241654504339172524852e+63');
    T('-429867232476019516593307236395662379525520042732934988570902.8316836760759159804393112331039824296654196079561094919062415600057', '-0.000000000000000026333381057307903712946311446867749470440652084025735627892221995697998435370429702', '-4.29867232476019516593307236395662379525520042732934988570902831683676075915954105930175796078716719108161088360021465589475979964372107778004302001564629570298e+59');
    T('-138287645002871107111044106385113952028931497798857994040661009094288998455256.8095564239091729600518464971437959597906', '-0.02071931654118840583447405976401564732916585950777930', '-1.382876450028711071110441063851139520289314977988579940406610090942889984552567888371073679845542173724373797803124614341404922207e+77');
    T('4607551890089700279547745482119886413441062288238756662240999999639177353304250157030', '43307073.38', '4.60755189008970027954774548211988641344106228823875666224099999963917735330420684995662e+84');
    T('6920.57190940428', '-8645401812213044670689632263.29035180132881948349690692829976371388993201328638', '8.64540181221304467068963918386226120560881948349690692829976371388993201328638e+27');
    T('500991793', '-251557191089121012500206645723710630.646511742483974550401389816576313099323805855573024993', '2.51557191089121012500206646224702423646511742483974550401389816576313099323805855573024993e+35');
    T('-115233062337818810389000461509795282853985222.0275735411960280352874012910144768141872396745625756737723537749444300561710511610476184357904253808912', '0.0000000000495099894662293514089120418920723309611170282344708775045419868213594847406514716827907876160549261', '-1.152330623378188103890004615097952828539852220275735412455380247536306424233888560793120055236927020068246524489720429924106457882699074732161685072549261e+44');
    T('1084723297738639869084299710762779016169159497808468673580004946719115763988516377.060148266799130410609492416453157', '1150603671275361.430584849644', '1.084723297738639869084299710762779016169159497808468673580004946717965160317241015629563417155130410609492416453157e+81');
    T('-620198612430107265937225529159584510224233045.598796327230401974796505862412', '8699552316331245145807826624182809542207885635402715254490655987896376746007483640710686.1', '-8.699552316331245145807826624182809542207886255601327684597921925121905905591993864943731698796327230401974796505862412e+87');
    T('-0.00000000000000020169002470441637778693075769563628101156942009213003841435019513553874336624464221898908333827', '55.3331815493788993877220026432941015', '-55.33318154937889958941202734771047928693075769563628101156942009213003841435019513553874336624464221898908333827');
    T('-509764.93911892369554928129540502547911386352816580281788234048871931011148766546995671957100520016074479741', '-7382417043131145488539352159357244152353681306894423489936116601393190269054624158211831.7', '7.38241704313114548853935215935724415235368130689442348993611660139319026905462415770206676088107630445071870459497452088613647183419718211765951128068988851233453004328042899479983925520259e+87');
    T('-893000341.075147812975752038206750463966592844149263520700585107620473591684006742845795349014', '0.00000776091913729118336285439866565205907149064403785475', '-893000341.075155573894889329390113318365258496208335011344622962370473591684006742845795349014');
    T('-0.000000000000000002814616587779740535288495495412599887159946708410461905984819284638731116086722791161143897920235719807012684318952369879227930494144', '-0.0000000000000000005799625258433885817982609943558405190973445644397869297994874177616229666024278363086227571999810218116665', '-2.234654061936351953490234501056759368062602143970674976185331866877108149484294954852521140720254697995346184318952369879227930494144e-18');
    T('1834227012352.2713316845164117139085412957218639331749015268185070505', '-4.100048574136744539426673770238786618959827409', '1834227012356.3713802586531562533352150659606505521347289358185070505');
    T('0.00000000000000000142961844636281549156124640702213622371103004167390847447921367714972088995760542647850152690627680982452493981026', '6.1225395082722753161364268570886553216', '-6.12253950827227531470680841072583983003875359297786377628896995832609152552078632285027911004239457352149847309372319017547506018974');
    T('290235134201682761299.48447329578802943221996178987974147165456654679438349900205350412258435388669557', '-5.251662064778328732803891', '290235134201682761304.73613536056635816502385278987974147165456654679438349900205350412258435388669557');
    T('52669199650648.0733153757733982313403970813057', '10386465556240559678192244.497235287453859060119988808620945435183307855210', '-1.038646555618789047854159642391991168046082877959172731524543518330785521e+25');
    T('564130335207659724622920169067214809519472975333569366119272839995955208278726184985111100693242720484069747466014', '57536812827159741113793203777862678446119346340872225907713497972953215.80310853441834229459084096442571657822478233621137898187840818727731', '5.6413033520765972462292016906721480951947291779675653895953172620275143041604773886576475982101681277057177451279819689146558165770540915903557428342177521766378862101812159181272269e+113');
    T('0.00000043656429426791027383754468210558259906392852085133878453458581222463232577586181', '-271425626170236597587007634813280769267131068497092506238139897701753645335886637971522821911055750072613148529', '2.7142562617023659758700763481328076926713106849709250623813989770175364533588663797152282191105575007261314852900000043656429426791027383754468210558259906392852085133878453458581222463232577586181e+110');
    T('6481867806858740818821154.19518001214840199197053504721', '-24870963461424.9780802848031479299059282084582297393626139502421237942537429692805914399653118951', '6.4818678068836117822825791732602969515499218764632556682297393626139502421237942537429692805914399653118951e+24');
    T('148.5411044804108834287889704020144411630326474907357936225017786074238828250511258990554797855720803620532362291284421872991756289113870701', '1687016782820477407264422479587.9602119652053424651802657', '-1.6870167828204774072644224794394191074847944590363912952979855588369673525092642063774982213925761171749488741009445202144279196379467637708715578127008243710886129299e+30');
    T('8753847156263969.74982484420314345673976831768407096355073194364180012848383603365570684241235747026243411927098077087739548', '-0.00000000000000003613762', '8753847156263969.74982484420314349287738831768407096355073194364180012848383603365570684241235747026243411927098077087739548');
    T('-16837319431920685704407321.888557412698092013196681699716686208311450572004672272353233944129089212943388330973909123531051036641', '-1434889199750833697003122178022.058644773719134292792949141816179232326040948745', '1.434872362431401776317417770700170087361021042279596267442099493024014590376740327727646766055870910787056611669026090876468948963359e+30');
    T('2735023455757108232743.584761329', '-883406375899961097762446633306214261625385142201740151414521222441704673205539113483156696740358847547521951238312862.0627294', '8.83406375899961097762446633306214261625385142201740151414521222441704673205539113483156696740361582570977708346545605647490729e+116');
    T('-689135077159362593015820247134066.74813269929304775244252204', '62605354599199881779688698278184513063808732', '-6.260535459988901685684806087120033331094279874813269929304775244252204e+43');
    T('-93750448497700582903413063583342246551210153127015625232845923539378736289601064020685503346328943587228412139711305446249370169912156093', '53953865201297384977039465.35433748887805', '-9.375044849770058290341306358334224655121015312701562523284592353937873628960106402068550334632894358722841213976525931145066755488919555835433748887805e+136');
    T('-0.0057925051239501905861387596709549007647236435494415395249013033629943873073216811979342031615889626092942975371394702866419123165012397470964059557', '246259164947571633614812263148098657772268077205108363195576642177873196268402803807976843988029200739623402923', '-2.462591649475716336148122631480986577722680772051083631955766421778731962684028038079768439880292007396234029230057925051239501905861387596709549007647236435494415395249013033629943873073216811979342031615889626092942975371394702866419123165012397470964059557e+110');
    T('202392866176', '-8676236136205102485012807', '8.676236136205304877878983e+24');
    T('-191.8725060079318845278193570544207471357649633582270807761589989403533787967818943544130158154895636289', '5967581749760718357075978833.8340341311556837206664', '-5.9675817497607183570759790257065401390875682484857570544207471357649633582270807761589989403533787967818943544130158154895636289e+27');
    T('-87951143371583656362303007410293120018882465816507444269402753743368383718', '-2822.68023786349119776278475695671489424412910651386641750260764', '-8.795114337158365636230300741029312001888246581650744426940275374336838089531976213650880223721524304328510575587089348613358249739236e+73');
    T('-2049698402682368740429778132608', '3963994567.44431117756210251278243787772', '-2.04969840268236874043374212717544431117756210251278243787772e+30');
    T('0.00000074347728103607719356135364801379323744725864451372933321009322616176605367628546021691519809986', '-8540850540817997066546115303931551402143341270504721504778718364110693.89400762910442310131822804681639793', '8.54085054081799706654611530393155140214334127050472150477871836411069389400837258170413739542160817004594379323744725864451372933321009322616176605367628546021691519809986e+69');
    T('585159105540713468004754266887.654351367900', '1854705740580.2225867', '5.851591055407134661500485263074317646679e+29');
    T('162041637982.2683403774133705140156618173084938487183540577160265469', '-2197013308.144', '164238651290.4123403774133705140156618173084938487183540577160265469');
    T('174785435.552717047585130654711', '0.00000671119928322725418238685556928705875440728878357795977246824794908748816696845271345930860829862726899160137025675180272592756862877515530420748', '174785435.55271033638584742745681761314443071294124559271121642204022753175205091251183303154728654069139170137273100839862974324819727407243137122484469579252');
    T('4747036426204027106781637076.99097717205130208396664801692006667885526524697426', '391523438182724274674956108257313659034734187677930.91506053303114035059545228819956', '-3.9152343818272427467495136122088745500762740604085392408336097983826662880427127949332114473475302574e+50');
    T('-10745175155350240219231799201005463136151.98855739210592020709205390996178000687517676358804699378042357529248908687167266714727400279', '-0.0000002750441382972253188972707268540147394058399897719491538526', '-1.074517515535024021923179920100546313615198855711706178190986673501269105315286043735774805722183126972269248908687167266714727400279e+40');
    T('-2016841186074253812287860201579902794714901016418893309168903524134146272466067614888435632384131092578883301477097742275', '-48984', '-2.016841186074253812287860201579902794714901016418893309168903524134146272466067614888435632384131092578883301477097693291e+120');
    T('0.00000000000000000075207122050490580445284855903892869462082699466860383714628003131667711449129', '6753.26798748946603087152519732149922074864703600110831025894212287136691857176209', '-6753.26798748946603087077312610099431494419418744206938156432129587669831473461580996868332288550871');
    T('-308768135249863901090901133014590589993514630400.7468363567730076836979817211331706945342868110424488561974871997974143', '-8486713.9935588503385776986188852583327833173309260920238377195', '-3.087681352498639010909011330145905899935061436867532775064344299850790964628003873772033607190186111366974871997974143e+47');
    T('-0.000000001028796916762577891829', '1261021215925756356473356031384588949462200699880284631995758278856051', '-1.261021215925756356473356031384588949462200699880284631995758278856051000000001028796916762577891829e+69');
    T('-2903534117347456791838333880169161579106798055500855912397515413217562006547074531951197755159', '3307709092093724676970726712003.7206946987943424741314293246854810621', '-2.9035341173474567918383338801691615791067980555008559123975154165252710986407992089219244671627206946987943424741314293246854810621e+93');
    T('308731033791553222538788495713125564554678655165.2376522348907919208222504963793054507680223355575585755605662130362013509014505533795599', '-169.6211818204352308680313496650435350420', '3.087310337915532225387884957131255645546786553348588340553260227888536001614228404927680223355575585755605662130362013509014505533795599e+47');
    T('46348756895149965005917587370622806277924635266696727', '-0.00000000000000002398533517999676328004879683029567483442514435593265672823703129497223737239', '4.634875689514996500591758737062280627792463526669672700000000000000002398533517999676328004879683029567483442514435593265672823703129497223737239e+52');
    T('0.00000003006579713187753751924205324740743488561146075520720007968290383452647952341024387158387162086926340315695048074352907454', '1105061311685464009276168371271826257860.49712297124331664597', '-1.10506131168546400927616837127182625786049712294117751951409246248075794675259256511438853924479279992031709616547352047658975612841612837913073659684304951925647092546e+39');
    T('-3.6658927729870468', '-285149436709749.875158993845', '285149436709746.2092662208579532');
    T('1013170470745994100692839107539318414591831785271578048439737249808144480351311573254823428168517820353394587009371292.096734997794012533488133881722531', '41152002543729538774558905430724841814570756403811876570254628123254482638637.6488377939839037328832602707121007518768718906728163', '1.0131704707459941006928391075393184145917906332690343189009626909027137555094970024984196162919475657252713325267326544478972038101088006048736110104302481231281093271837e+117');
    T('-33927165312151896961929125486531.3854145399412684039819801379567554380', '-6267032.778812', '-3.3927165312151896961929119219498606602539941268403981980137956755438e+31');
    T('-4363114224107555270076478029292860456660849819228669713810028502419476164665171', '-36729829795867171018424681856707243732121421681', '-4.36311422410755527007647802929282372683105395205765128912817179517574404324349e+78');
    T('-29943384747492648907943443.256412350894815929125401', '901747579048273816602550102529926854.35549646674102528482365191696613584123624344078663288749130490435297134720727576379819588420174171101450551687', '-9.0174757907821720135004275143787029761190881763584121394905291696613584123624344078663288749130490435297134720727576379819588420174171101450551687e+35');
    T('952557.246599587', '-797618074571855160635860103922751796538514948531.1378948', '7.97618074571855160635860103922751796538515901088384494387e+47');
    T('0.0000142901236857676679196257053214336506759362362251650293753952658734900913009470163591148854555849408371350202869477370839162018128614', '7274936081690054967838223208959684283904860805239531936252446172884944195802432384674846535126802169513987351992902768514276949331803322.9', '-7.2749360816900549678382232089596842839048608052395319362524461728849441958024323846748465351268021695139873519929027685142769493318033228999857098763142323320803742946785663493240637637748349706246047341265099086990529836408851145444150591628649797130522629160837981871386e+135');
    T('0.000900929560788911182381071562865432668409466182204594172033162144812128331193077475077423880166489', '-21408597957321978981807151751183219484279556440', '2.1408597957321978981807151751183219484279556440000900929560788911182381071562865432668409466182204594172033162144812128331193077475077423880166489e+46');
    T('-0.000000000000722138907853996119863088603030135349132652185245300184344322987517460715495745561630491364910862090541897814776826750915053339581143330860038', '209219551544.876995097117350588163000906011435189490577683711751207988391411412107868148561383392176930291264602055541628014525', '-209219551544.876995097118072727070854902131298278093607819060883860173636711596452191136078844107672675852895093420452490105066897814776826750915053339581143330860038');
    T('-0.000000000030345983499647050827143736470118525842662060717', '-140869900251790823527224455558474696.4221968423139', '1.40869900251790823527224455558474696422196842283554016500352949172856263529881474157337939283e+35');
    T('1388628543041917754379950101913898878882776054395652240051594608900168911', '-634307381507440042083604924255544396160446890170874345571896298473856984277777.4362135602582999033', '6.343087701359830840013593042056463100593257729469287412241363500684658844466884362135602582999033e+77');
    T('-99178063079827.224433662117625116489799642020572811986957525604609123159620902928779438653713181794557115326708644398427245972390297968858510824615', '0.00000504045524028117298744681753139617460039787354788645269348683810171985366129261406243335125957027159417139707333860235220380988', '-99178063079827.224438702572865397662787088838104208161557923478157009612314389766881158507374474408619548677968214670021417369463636571210714634495');
    T('-7964344017627.65729896967420320480697541366008605972016325458796553432374728674811327', '363448465190438134042900077564563226373185360490340453089461130955156779944343856777497645145316462888016452339862253.6350461018654269710898806', '-3.6344846519043813404290007756456322637318536049034045308946113095515677994434385677749764514531646288802441668387988129234507153963017589685601366008605972016325458796553432374728674811327e+116');
    T('22398969506188688342902196074268860817610249888096.96109576852487881868553737058841014408233230362878332132', '150517970635382784547231206380342822229.6234502535074912839215712571451888428718057053541076659669932226957916203900023162576692335713154540338973281941', '2.23989695060381703722668132897216296112299070658673376455150173875347639661134432213012105265982746756553530067773042083796099976837423307664286845459661026718059e+49');
    T('319606363926195232923049.458147549841648', '9555894221042120529332254896056451521165707003469529318473253597473821865716922096418529662756642900117543029527432853335.21880839479040', '-9.555894221042120529332254896056451521165707003469529318473253597473821865716922096418529662756642580511179103332199930285760660844948752e+120');
    T('-1.7470999677375', '-0.000000009390885937723498768042482633173355191224894879999587923088', '-1.747099958346614062276501231957517366826644808775105120000412076912');
    T('0.000031694970', '0.0003489109755744784300779153529542987938823096920725044639439934321377263762937670754333912101552258897473271605646851768747294780640640656588128251', '-0.0003172160055744784300779153529542987938823096920725044639439934321377263762937670754333912101552258897473271605646851768747294780640640656588128251');
    T('15171023450028723096297246476058654442156062896548968046928610610466495463708888690638445384052.5655402936580877003227652909723023235', '300381994693137917085900.481742', '1.51710234500287230962972464760586544421560628965489680469286106104664951633268939975005282981520837982936580877003227652909723023235e+94');
    T('-2510422771394805479647755240376039069.235025776988445880809314981927377977492349', '-4325650526595464202944513201707597153545181637871972811769276403384719489760146217058876642443138050881786664883495941202481550', '4.325650526595464202944513201707597153545181637871972811769276403384719489760146217058876639932715279486981185235740700826442480764974223011554119190685018072622022507651e+126');
    T('11989030014394748093925142998.905998360', '4503.1829930104967074062224350092529380543376637410353675591178914025067457206266908222070994034406954750129621950217244', '1.19890300143947480939251384957230053495032925937775649907470619456623362589646324408821085974932542793733091777929005965593045249870378049782756e+28');
    T('-19622700098117555355413025589212248976721079203363291412', '0.0000000000000000047154963464125735069117227717041172013254203483060822909726449561031204371411673028528388174607696692799950770173459572692814680509896623815282932', '-1.96227000981175553554130255892122489767210792033632914120000000000000000047154963464125735069117227717041172013254203483060822909726449561031204371411673028528388174607696692799950770173459572692814680509896623815282932e+55');
    T('971668743073894893699159113569468037426157854834.4859245308812504877580159133281293', '2155437400819398246908875466354226177837114235441458903251149825.33', '-2.1554374008193972752401323924593324786780006659734214770932949908440754691187495122419840866718707e+63');
    T('-75219128342703758653804581528133672565697676079415639891282318101552836007137960680191708674953779692.830850412286175003379303061502973114', '8890697637237157877394293', '-7.5219128342703758653804581528133672565697676079415639891282318101552836007146851377828945832831173985830850412286175003379303061502973114e+100');
    T('-2207826365359691933623472386278119104.10928610148472698325737986521030447640095006599795584977677429732704747959908983', '-0.0000000000000000020914676583255446799691346083866342797576131478040257513927024455', '-2.2078263653596919336234723862781191041092861014847269811659122068847597964318154576113215700191611495230217282063873845e+36');
    T('0.00000000000000003529871868734909395544492984890736625012421342', '-102670940276960616548987079748287921392363616947', '1.0267094027696061654898707974828792139236361694700000000000000003529871868734909395544492984890736625012421342e+47');
    T('-16411029.5930820', '1443249803759620996.044533632703181774392190634172289219280976531099613177406267663822437005138144965669407971461692413376915', '-1443249803776032025.637615632703181774392190634172289219280976531099613177406267663822437005138144965669407971461692413376915');
    T('4764961911712179295885207525501495839960553554868719797653093551996007318270912309189493238056884734776455.4857499673', '40.70', '4.7649619117121792958852075255014958399605535548687197976530935519960073182709123091894932380568847347764147857499673e+105');
    T('5188138747138380958829180.51675828531601306918974036835433546892123023643482179063194941268993890524469204530549406272435329728', '-218356026557832392846658731388880742380232559055047578182115570398', '2.1835602655783239284665873138888074238023774719379471656307439957851675828531601306918974036835433546892123023643482179063194941268993890524469204530549406272435329728e+65');
    T('-284221651007245305082697851875743201255770321818102228785943924728887176146409104121604051647768402471354721655179789550685066', '129802078488939350592312997875682679945769088382.2879783911513234655169373537741125669105835698742', '-2.842216510072453050826978518757432012557703218181022287859439247288871761464092339236825405871189947843525973378597353197734482879783911513234655169373537741125669105835698742e+125');
    T('124593177725250773807727158694606.48780660352148825658767811933255518150599696025419899514439334', '-6627581651662715635950762070180.79290709812889445314806559015208127150071', '1.3122075937691348944367792076478728071370165038270973574370948463645300670696025419899514439334e+32');
    T('31366660201652105746754295566374870763019559240421554119188881694316784695785923310969085305946601572113197265575586214263032', '-0.0000185119', '3.13666602016521057467542955663748707630195592404215541191888816943167846957859233109690853059466015721131972655755862142630320000185119e+124');
    T('-0.00000000123441941', '20577.06632', '-20577.06632000123441941');
    T('-0.000000000000000000030367929295105668550777381180120786905515151003716045995461753738312176999386', '-8296319183891479588134987898667542872737578839488792548834441415271860494822651335094309535757878796690.933873784093259915387382991882725', '8.296319183891479588134987898667542872737578839488792548834441415271860494822651335094309535757878796690933873784093259915357015062587619331449222618819879213094484848996283954004538246261687823000614e+102');
    T('-40308545430969720235947527422528929192769782793205800044070633029892426929802242614838.13803009279797050440596035398833519975310207504', '1959239464986228954812427023.535609294269733799545341029686339869114158840301903524765921157569249146361346526131', '-4.0308545430969720235947527422528929192769782793205800044072592269357413158757055041861673639387067704303951301383674675068867260915341903524765921157569249146361346526131e+85');
    T('672932217082947936267931802569635295900508909649104706736777590283845640515821796262381421.68141583026180417434416456', '95.0', '6.7293221708294793626793180256963529590050890964910470673677759028384564051582179626238132668141583026180417434416456e+89');
    T('-0.000000000000000002099033004183951058449767693649425035305130331498778862127487553169994535996', '-2133980231896312773907666627097548168415.7102448765553114470129805463990038', '2.133980231896312773907666627097548168415710244876555311444913947542215052741550232306350574964694869668501221137872512446830005464004e+39');
    T('580445381354905284048807368831006691784.546059063403641312301109310676306914676494597505919617349947441960647278533345556960169308273917366276', '3.604904939285817554324885708126246946469619276624534189637434215912939543315825766859553647518276673500996166212799836185950984610359767998387151515', '5.80445381354905284048807368831006691780941154124117823757976223602550059968206875320881385427712513226047707735217519790100615660755640692775003833787200163814049015389640232001612848485e+38');
    T('-712252720953571012672166069903348092161944796663143300420882328496228974.04618394348904074586863813372820528', '-0.00000000000000002092030775039740453919504630522275260893114360423604255725599892060864590', '-7.122527209535710126721660699033480921619447966631433004208823284962289740461839434890407249483303833308007408049536947772473910688563957639574427440010793913541e+71');
    T('-0.0002570816455207099536036261314936090157674967926099800795', '-0.0000000000000000130674933467131551680491886017016042522946101681008', '-0.0002570816455206968861102794183384409665788950910057277848898318992');
    T('348881507908033302985005.68757522', '1549290074464248686919664413535348476523309010487683201418540220237090492636459013101704289528591313898108.981495311', '-1.549290074464248686919664413535348476523309010487683201418540220237090492636459012752822781620558010913103293920091e+105');
    T('8637413948644495520046181627871244127949368908598113230287818578.76219', '172462329166.2925867381763494812410830828530111105472625398762231331365955149682132409286466618708916', '8.6374139486444955200461816278712441279493689085981130578254894124696032618236505187589169171469888894527374601237768668634044850317867590713533381291084e+63');
    T('-28059829346271297089846697823619339730697463927882137285411047187621.3365094609441835662258292', '1732861001433499169623889562015115612698521681606694360771023126224313336415239339086113519268958723711', '-1.7328610014334991696238895620151156407583510278779914506177209498436530671127032669682508046800059113323365094609441835662258292e+102');
    T('0.000000000000000000279920475007074858021168221493722708848631480438782694735575852706998', '-36266411773095115580689400782807334211196172564607435961552009343283932464.9837695007775892543051564275268186', '3.6266411773095115580689400782807334211196172564607435961552009343283932464983769500777589254585076902533893458021168221493722708848631480438782694735575852706998e+73');
    T('2118046918816158.19627388128567915392498196852916971435774168271017092737124387676986128523446', '700283033083138972861857057593773868572323954863349193882758861910219422329615290916944812790080302082455986032622356459796411429994571952107', '-7.0028303308313897286185705759377386857232395486334919388275886191021942232961529091694481279008030208245598603262235645979640931194765313594880372611871432084607501803147083028564225831728982907262875612323013871476554e+140');
    T('339538851650247942.278697506630118637945095728103', '-18620906912720892893391275141765114456962630526288263883096512988776949390578497946714236563676541486658', '1.8620906912720892893391275141765114456962630526288263883096512988776949390578497946714576102528191734600278697506630118637945095728103e+103');
    T('0.0000000006904532369558150510262488281807825581586827251', '305468661281838707389527955487753.82749543444315394761867888706263', '-3.054686612818387073895279554877538274954337527007106628638360363811718192174418413172749e+32');
    T('-908820873408880798933264474895850726464711747913205439443550022010444648198141527361971583612034255103784.08541958237725093777442080304887210', '-137777781478673113932594166168320795125870691409923883994778273765714588733515698043941904144625098636345868758', '1.377768726577997050517952329038458992751442266981759707893388302156925782888674999024145421730414866020907649739145804176227490622255791969511279e+110');
    T('374355450809837247.5174995', '-10.2', '374355450809837257.7174995');
    T('-7.24216105787', '41335238394851454012288137625393602319.3264096053286738641962006640465495687544256', '-4.13352383948514540122881376253936023265685706631986738641962006640465495687544256e+37');
    T('66326977.9050567217', '-0.0000000000000000000209', '66326977.9050567217000000000209');
    T('0.0000000000000000000265737965021', '-81974150526768898581207134037679.2068387354409294052459', '8.19741505267688985812071340376792068387354409294052724737965021e+31');
    T('6007018972384004427335278306100547732730414161628343888635552404650.58282054220706207456023620752991471499255', '-146656810017560772593186152454856504300187695304259360339772717751599966472', '1.4665681602457974497719057979013481040073542803467352196811660638715237112258282054220706207456023620752991471499255e+74');
    T('-34292722587178232282501917441324831663740801129878989387480983625715484621782.0919', '-55883431183450605660044015172082536043911915919486680333854135673827331840347136897324', '5.58834311491578830728657828895806186025870842557458792039751462863463482146316522755419081e+85');
    T('-3985805244044837084411977422757376378060.290541664260691735184074947', '-870047741853.19289603162319510263029755253365086206461589052739480724332358851742990324', '-3.98580524404483708441197742188732863620709764563263749663255377739446634913793538410947260519275667641148257009676e+39');
    T('-73929816747605180484918003597961235850171978225512788042904593650909580403443769374087641707696975944827828354020150979988740026532938', '385361544903059555711806156019448314064782', '-7.392981674760518048491800359796123585017197822551278804290459365090958040344376937408764170808233748973088790973195713600818834059772e+133');
    T('-12.31698991', '-751188829485947331552.960312839846836094417204848662429224188702772253892719659281557251306277105093575994993846127669453550207016018564227856026', '751188829485947331540.643322929846836094417204848662429224188702772253892719659281557251306277105093575994993846127669453550207016018564227856026');
    T('-3628724754364816284699558478736662262101049.8686961811050797854712833273940271907427771082320587042061269470622', '-36853113191655566744.0036748111113230617191140153379097532999540758924150850150462749161630182517728624293985456773', '-3.6287247543648162846995216256234706065343058650213699937567237521693120561174374428230323396436191910806721460369817482271375706014543227e+42');
    T('-0.000000000000000215', '19894059835635097553.40565542831442512355858226792638029459899976138082', '-19894059835635097553.40565542831442533855858226792638029459899976138082');
    T('-30106887927.73937362847', '-102.030', '-30106887825.70937362847');
    T('-1218248850456.97353951682445947974274913192207119277095702137346638965795765469780537397426794516539138118094328539', '0.00000000007851693763154494785066647692511713030801839675154976889840604918825877872638771285849514900835419977768857575607733', '-1218248850456.97353951690297641737429407977273766969607415168148478640950742359621142316252672389177909403943843439835419977768857575607733');
    T('178340800930010683157306838387114508530123729280296661983521452605952899303508793164498075.18', '124104.19653839526727306978867606725200681644881825667', '1.7834080093001068315730683838711450853012372928029666198352145260595289930350879316437397098346160473272693021132393274799318355118174333e+89');
    T('-124972459343817845451907386538316.88444543539384522567165512709192478339963512253604376341607496802805641627423480880098070424177758544304960846746', '-12151860990400073737429990488961466727732472556130861379738694459231286354909360090762808804561520156.93885466', '1.215186099040007373742999048896146672773247255613086137973869445923116138245001627291735689717498184005440922460615477432834487290807521660036487746395623658392503197194358372576519119901929575822241455695039153254e+100');
    T('242305619480044.117669571', '0.000000000004954617457909921566083069296578532961927959570975042077881955980821090098114246814822601513881827860466408504361049103', '242305619480044.117669570995045382542090078433916930703421467038072040429024957922118044019178909901885753185177398486118172139533591495638950897');
    T('-149393121723.8', '0.000031123464017479325434907565167758533432136444707393019391355781893113282833929057474217941299740843262816249', '-149393121723.800031123464017479325434907565167758533432136444707393019391355781893113282833929057474217941299740843262816249');
    T('304991568028921434085139022483844876733665.542738390605319884', '0.00000000000000138699174160296898158944175519528763014061938962295885233194249035736092848113286092759686854653852969146', '3.0499156802892143408513902248384487673366554273839060531849700825839703101841055824480471236985938061037704114766805750964263907151886713907240313145346147030854e+41');
    T('0.000000000000439684683303579860356147180119042386845099669639708374008856922269915124921301748', '-155818221549284043593101823943330354192243873661833929849877145194258036053339831371360183.4319505793302857955106411234496205383496363144', '1.55818221549284043593101823943330354192243873661833929849877145194258036053339831371360183431950579330725480193944703309976685529755356786845099669639708374008856922269915124921301748e+89');
    T('-13.155103645301', '-0.000000000000000007137064', '-13.155103645300999992862936');
    T('-10356428764148893775275.9673266649810109703809707584633512809703095548147864827191283851408290', '116469121259.580', '-1.0356428764265362896535547326664981010970380970758463351280970309554814786482719128385140829e+22');
    T('-3963571363397152992273906802160286886086920884936747622634365849406904936980386.45359892835130720540151945758060955916317802717566903116864480481972522', '-0.000070996697013405007160606340379272860030484625815078099084045955348142553', '-3.963571363397152992273906802160286886086920884936747622634365849406904936980386453527931654293800394358851240230286303147542549853953069560758864377077447e+78');
    T('-2002.01719868890096247452400760373', '1.8', '-2003.81719868890096247452400760373');
    T('126616415747207305303537223662274177712840.6700589197974', '12097.7020794393208690215', '1.266164157472073053035372236622741777007429679794804765309785e+41');
    T('0.00000000000000221572441427972677866185400843138814827125078372356950505019583272816338112', '86011656827173.81', '-86011656827173.80999999999999778427558572027322133814599156861185172874921627643049494980416727183661888');
    T('45543191029347952618469386523689325602984824853133335610432465910372937520283489097581912486080942672641037', '-15716512380188479.932083641807869052147091043255091861011843759620263027387952842807874862983089866', '4.5543191029347952618469386523689325602984824853133335610432465910372937520283489097581912501797455052829516932083641807869052147091043255091861011843759620263027387952842807874862983089866e+106');
    T('0.0000000000000000035315502420788508529985856314419012272238207473944358204104290394926106351779', '-5624156772170117.1657476003837553942295755292258482770658649647601920538662932208435924708861640276207503440293532949', '5624156772170117.1657476003837553977611257713046991300644505962020932810901139682380282912965930671133609792072532949');
    T('-1374562994880051964271.27898098906188407287205169338', '0.00000000000000031865339921207143913309344392566436421', '-1.37456299488005196427127898098906188439152545090545143913309344392566436421e+21');
    T('13691023568703828954478002891170478555913926785952721.6937350094794685112055123701357309217624134750110846096769', '0.00000000202148677107507460297752891791329356574084754670408775226519607249962621030745245954851134740234779971411203608026098975', '1.369102356870382895447800289117047855591392678595272169373500745798174013043776715820200384911990927023706297281224773480392750037378969254754045148865259765220028588796391973901025e+52');
    T('-125671888607778034275154792009202764081572882956209693373934.74566972561590141388688468937614644443796008254', '107551721884298521156095810104481388698517320600299802047', '-1.2577944032966233279631088781930724547027140027680999317598174566972561590141388688468937614644443796008254e+59');
    T('-0.00000000003969378933740708691300122226819421581066663873920896781273585494818171435695407330811294202', '107041184511163885.04271388462119128657386073279129422270006', '-107041184511163885.04271388466088507591126781970429544496825421581066663873920896781273585494818171435695407330811294202');
    T('-2951965402440738579279759050624852782460407526269383738169962367847588117867127018487856370629062877269116277468600261229135554565114176766056702', '0.000000000006287363483881253518833462795616481179', '-2.951965402440738579279759050624852782460407526269383738169962367847588117867127018487856370629062877269116277468600261229135554565114176766056702000000000006287363483881253518833462795616481179e+144');
    T('-0.00000000000000019636276967708885671764139154128434128626403182943490041425988637275339748810419236540254220706284324560538931703795739814', '0.00000000004481901971403795542991680781566338421560653872041361060', '-4.481921607680763251877352545705492549994782498444544003490041425988637275339748810419236540254220706284324560538931703795739814e-11');
    T('-354719950361379273301359534196300717853.0186108408149835703024', '294514396703359984682344528266458432292891368613559732314480483030279392081064357195630432442505889982.13440538511210586133603914935112321050428', '-2.9451439670335998468234452826645843229289136861355973231448048338499934244244363049698996663880660783515301622592708943163843914935112321050428e+101');
    T('-33927540600930859899252442105999310587077967493761773461091899630388700204633168728544588442837280913459638158456686740859615.1414', '0.000000001894808120791736519518300794339416227', '-3.3927540600930859899252442105999310587077967493761773461091899630388700204633168728544588442837280913459638158456686740859615141400001894808120791736519518300794339416227e+124');
    T('-104666170827444788671824735700977493877060949970738269558671964530958.55363526374', '-2386824408518725780122101607313193461624217723024418681790730644945410612236759422226411833632517989926779141564329843840707189394654888', '2.38682440851872578012210160731319346162421772302441868179073064494530594606593197743774000889681701243290208061435910557114851743012392944636473626e+135');
    T('179421.5289096829953298', '34342876175696462977913441783684065522884009795140039066541213786441.821375732575578', '-3.43428761756964629779134417836840655228840097951400390665412136070202924660495802482e+67');
    T('-41426466350219117773090054766644394486314258063933223051089090219912755660263417233021.9855607046947587791892004394939235', '-170862420310795633399339976259565748191370.489727472434638', '-4.14264663502191177730900547666443944863142578930708027402934568205727794006976690416514958332322601207791892004394939235e+85');
    T('4.356', '-14.51838553571377330631321826342372952104419616215052089825233325146382745555567077756959361343373549160815949781902924303746486223707861', '18.87438553571377330631321826342372952104419616215052089825233325146382745555567077756959361343373549160815949781902924303746486223707861');
    T('-0.00000000007415111125701712334659040023188433045533322445707355817794327911316470827191624585971', '-89496941499485282617443687.11', '8.949694149948528261744368710999999992584888874298287665340959976811566954466677554292644182205672088683529172808375414029e+25');
    T('395406829706311296589271889719583140635568635127665', '264737687.61137854553903553659981221568900541026391672130446196896447248', '3.9540682970631129658927188971958314063556837038997738862145446096446340018778431099458973608327869553803103552752e+50');
    T('-91265516152809240955984710790601792219643671438700946275373285010553537355097749911888722836472150250555502817.7083486', '0.00000000005908292684015754961053591782663005124668197067163418657692319967418105412597307502992726349390479329489028174099287376424396743766', '-9.126551615280924095598471079060179221964367143870094627537328501055353735509774991188872283647215025055550281770834860005908292684015754961053591782663005124668197067163418657692319967418105412597307502992726349390479329489028174099287376424396743766e+109');
    T('1393668664736340396506094893169761479672219453971818718110678752454.93557171027859666584700501067765267442071267740729381267372577', '-7.26506721396234349631410058152182568345755302761014440038909778541514253256903438374', '1.39366866473634039650609489316976147967221945397181871811067875246220063892424094016216110559219947835787826570501743821306282355541514253256903438374e+66');
    T('154487157373823493137066274.85878615066004288744973873503106160140296864897001622023987434759035195222460780528406318750', '13509881005355988338518293330867947988093581878486275639404094210458245268756845889504384099465156294654099130595946018501027161729896016', '-1.35098810053559883385182933308679479880935818784862756394040942104582452687568458895043840994651562946540991304414588611272036685928297411412138493399571125502612649689383985970313510299837797601256524096480477753921947159368125e+136');
    T('-0.0924532442681963280236579804820783805412208183173216', '437714061304750576304069895787070747586930126317209775578680048.64905430778651377650282955909271100702270378704840241396672781384650762', '-4.3771406130475057630406989578707074758693012631720977557868004874150755205471010452648753957478938756392460536572401396672781384650762e+62');
    T('-2903867214462079147781503.43110967041878698272228130248023004255630098538325966386117388084280644624246779280709038813631', '-0.0000000000029001819398679', '-2.90386721446207914778150343110967041588680078241340248023004255630098538325966386117388084280644624246779280709038813631e+24');
    T('-68478988612026757357956003683332744549804329487951781618243422133228484235053139292.63859031920959545863', '-0.0000000000000206209573312295241705394462020368371700010471324470113265516966420178900', '-6.847898861202675735795600368333274454980432948795178161824342213322848423505313929263859031920957483767266877047582946055379796316282999895286755298867344830335798211e+82');
    T('2383784.15779081', '-0.000000000001342350812195198392532374798854977110090674918085941866638293587449553312', '2383784.157790810001342350812195198392532374798854977110090674918085941866638293587449553312');
    T('-930945859272044558989235909016.68900733503225770344925642072767208409841301004231082154342176269488246587383976591788', '-5242498654293148790270310863665439318251858135048956.2717470020461119420226047241249204047819006900600244190387', '5.24249865429314879026937991780616727369286889913993958273966701385423857334830339724832068348768001771359749527823730511753412616023408212e+51');
    T('313632278451489633607271056572839346933772283691577954375047188993463512.092647131238398999623', '-959538307079783685461423074240027842581631060789324748920778178110453051', '1.273170585531273319068694130812867189515403344480902703295825367103916563092647131238398999623e+72');
    T('-1288931819347065003984018264113291238692657741.766349736546125023857675298415334704473602112840757695193462502084969449569542319748464264978', '539036429230664384725011772246635819519599525128.174476503860', '-5.40325361050011449728995790510749110758292182869940826240406125023857675298415334704473602112840757695193462502084969449569542319748464264978e+47');
    T('-778185145541293282976352747700447673380643846202878141.94156682254735142708475873940521', '-1743702845400679249.985436319726446594414323812859185183027678188541351470985', '-7.78185145541293282976352747700447671636941000802198891956130502820904832670434926546024816972321811458648529015e+53');
    T('-34198990286492455559791232025411563407581563098497220768937957961783704810043975778308343224151175569504114685192665.30536869253744218089551619675327', '-24305945127441184832105158951553331385997091194202426787946078772275521880625289746661831499392830801691448841424.759', '-3.417468434136501437495912686646001007619556600730301834215001188301142928816335048856168139265178273870242323635124054636869253744218089551619675327e+115');
    T('0.0000000000000018763276930197981120089551282335569865999686905948962583363764311923118368944319393384148082395872202253088856212394945596283231828525914885594', '0.35819808711067029153542041472676295936698327566168897', '-0.3581980871106684152077273949286509504118550421047023700313094051037416636235688076881631055680606615851917604127797746911143787605054403716768171474085114406');
    T('1418031575151863438477712278537563020829712238913927232796816808973295891102.55296374071457002140814551720742132636073542069579726675621573941382622600', '-3488182224093174883184', '1.418031575151863438477712278537563020829712238913927236284999033066470774286552963740714570021408145517207421326360735420695797266756215739413826226e+75');
    T('-0.000000000000000205390684486171825460536006353563548135276787747301988476146287694114350379000036906338511378609', '-0.000405', '0.000404999999999794609315513828174539463993646436451864723212252698011523853712305885649620999963093661488621391');
    T('-3202016.0694803128322148239145232929204107555210732105469003170128506695838996492', '0.0000000000014', '-3202016.0694803128336148239145232929204107555210732105469003170128506695838996492');
    T('-628423720237204906120199547323758489801025.51874281555091178803625021347340455594130', '-81477111818306314666306661562', '-6.284237202371234290083812410090921831394635187428155509117880362502134734045559413e+41');
    T('-0.000000000000000000047888974392267564390331997035456851568895821244693231371197999015834461798775529541', '947453380002', '-947453380002.000000000000000000047888974392267564390331997035456851568895821244693231371197999015834461798775529541');
    T('773747345', '-1335821513605028779157', '1.335821513605802526502e+21');
    T('-99273.596604127', '-1154952503563378596462693693250.5187439232425471346', '1.1549525035633785964626935939769221397962425471346e+30');
    T('1395668702641787254510091198553284', '1578568229336237285301359008898017.5404224057562993167719876179720474361379874854903455492277573607378863860373466275770813782848272509058803097', '-1.828995266944500307912678103447335404224057562993167719876179720474361379874854903455492277573607378863860373466275770813782848272509058803097e+32');
    T('5386712177299027464798225011978949139476122', '469103363732559983685.648370', '5.38671217729902746479775590861521657949243635163e+42');
    T('0', '419192442362435629465014450272844841376382155269323588979357043630935966384025897855088784350716859.22651317', '-4.1919244236243562946501445027284484137638215526932358897935704363093596638402589785508878435071685922651317e+98');
    T('15748527253971616615055630358104959795592326115279195892785474069003661098024748903045029419699640409113503023237945492.3252432550999839712', '0', '1.57485272539716166150556303581049597955923261152791958927854740690036610980247489030450294196996404091135030232379454923252432550999839712e+118');
    T('0.000000002333438210472843133760719971852', '919118976014042057478336651114029860028401924997726.37482700254255653557019028924295996133470278160287964', '-9.1911897601404205747833665111402986002840192499772637482700020911832509734715548223998948270278160287964e+50');
    T('79', '108926743127104368181585148150.21181757229942036662043384365317289250377588166675035927', '-1.0892674312710436818158514807121181757229942036662043384365317289250377588166675035927e+29');
    T('66999786317143790144990981463093917201197515876881754571980957877868116674749519019018943370951009463.847271877093748797400', '-476103985245129257652983725336203640462984911959107831794737195876658142.344', '6.69997863171437901449909814635700211864426451345347382973171615183311015867086268508136805668276676061912718770937487974e+100');
    T('-1405033234114', '-783177164833364354044558842295870625108912898123626390937631256466.542569106951119275629030077420445312292985', '7.83177164833364354044558842295870625108912898123626389532598022352542569106951119275629030077420445312292985e+65');
    T('-426205767054867887.03542915655127780054256765661941922641508457335701462240270994743442393487518141488173295859', '-8298.4660255243403790577839381771224207771986', '-426205767054859588.56940363221089874275862947949699844921648457335701462240270994743442393487518141488173295859');
    T('-464425272555675063916245905964590494968787633124183600713545419950133228198836596648857626820065', '-2237730394988417178529397.358188196946174766853203174768301504251', '-4.64425272555675063916245905964590494968787633124183600713545419950133225961106201660440448290667641811803053825233146796825231698495749e+95');
    T('-0.0000062461634807333371859314155489802574', '-8181353643032489942675123036461953565314428349623.6339704961199', '8.1813536430324899426751230364619535653144283496236339642499564192666628140685844510197426e+48');
    T('269031915782887321035697046572774089182793600032731720080010568383408337.749033224541645681839966449764521621943733799440', '0.404132605183270165863226749047082654939280193886231529898349131828226447630704284', '2.69031915782887321035697046572774089182793600032731720080010568383408337344900619358375515976739700717438967004453605553768470101650868171773552369295716e+71');
    T('988459584582389858260070961884.0298077770057303924476954237850807378083146026121046406109614', '548657365885761906.16130546365', '9.884595845818412008941851999778685023133557303924476954237850807378083146026121046406109614e+29');
    T('-4579867386321723251708179657431459719237903598.4249552191197039811767', '1700354336203076885308678693.5509603461600890657301710589520939346267614885400052674765795335958411640877837509557489194202307107946', '-4.5798673863217232534085339936345366045465822919759155652797930469068710589520939346267614885400052674765795335958411640877837509557489194202307107946e+45');
    T('-39518020854854496682.408531833821299513675968633019378670635010897754414243511999567768675730681912611391338379426802378786621', '-69488349149463743032009.4871074472497372391041043969145996', '6.9448831128608888535327078575613428437725428135763895220929364989102245585756488000432231324269318087388608661620573197621213379e+22');
    T('-280346355147300855628404.763216', '2.0516001318909', '-2.803463551473008556284068148161318909e+23');
    T('0.000000000000000099623328', '0.000000000015534855199084239745828402971988998230014287963399588591303464021723633191191744223224494006485956657062304557069407148666768780677134976225521', '-1.5534755575756239745828402971988998230014287963399588591303464021723633191191744223224494006485956657062304557069407148666768780677134976225521e-11');
    T('-334431356812439.13448916221758530842995', '1667521702832986056260091238367720285408933716412273493571950628545108649803910938622937226194497229999330474159307225451757494492442', '-1.66752170283298605626009123836772028540893371641227349357195062854510864980391093862293722619449722999933047415930722578618885130488113448916221758530842995e+132');
    T('-25720.3501395114447395112201', '-0.2953998138840', '-25720.0547396975607395112201');
    T('116226570322143124260726476527797976180664474951141971508306422788718375438939254351135212826259379114908655', '-160024900712467287748021197823552179871601569963013469207712918690534434663724462868758130292404825253945347600017444751054592974832', '1.60024900712467287748021314050122502014725830689489997005689099355009385805695971175180919010780264193199698735230271010433707883487e+131');
    T('2324224807171911990671839000597607829497219555298.55014120924636442879633120922286793218', '-1270380349602863576806275283032958213074398270334091077777630627191054790255851313419272408473587843637275886291610.642667419393', '1.27038034960286357680627528303295821307439827033409107777763062719337901506302322540994424747418545146677310584690919280862863936442879633120922286793218e+114');
    T('-10622788500960972991004261.8032858098042872288506798796510154272226938496951067890622789073880670340432116295332571771677661755224600880395', '17077138128499655920.4093059483819429666765577064075794109485305714940654287018169021314700994059258023402097770797726365433681288588963390967078', '-1.06228055780991014906601822125917581862301955272375860585948381712244211891722177640958095195371334491374318734669542475388120658282168983963390967078e+25');
    T('-6336513.23', '-5249172002022980268812313222482147319.6010671183072060237015049011583118322', '5.2491720020229802688123132224758108063710671183072060237015049011583118322e+36');
    T('267635446505753748225432544372279094104011987689607870636.88308855576649357656458127950511', '0.000000007151544482605747363599261338379766369120662716754539', '2.67635446505753748225432544372279094104011987689607870636883088548614949093958833915905848661620233630879337283245461e+56');
    T('0.0000000000000000251680172', '-43996141.9473547548529', '43996141.9473547548529000251680172');
    T('3008681171818278308404.634698376660136232243907341147927120093956', '-72008045405579488.929013843', '3.008753179863683887893563712219660136232243907341147927120093956e+21');
    T('-1917507321972333152063718.3', '-458382.6165374681213015', '-1.9175073219723331516053356834625318786985e+24');
    T('-0.00000000162489680974735997523039312016450827505959711537297137706530', '-731118718225310016148587418433198514412521393862386540689205275178569598255098419588588653874560984.082237617343418975656930427', '7.311187182253100161485874184331985144125213938623865406892052751785695982550984195885886538745609840822376157185221659095704517696068798354917249404028846270286229347e+98');
    T('531802532279637137552808013110951053732.907624514862128773520547166755212879477296231276825579342', '-173067154812932973797393419951029156470968384903254747150744.1037243073944890748534755651124752', '1.73067154812932973797925222483308793608521192916365698204477011348822256617848374022731867688079477296231276825579342e+59');
    T('1216376563.861151456186911150794802999960874196553469479145822420208291551214685836322120517079008', '85984.5', '1216290579.361151456186911150794802999960874196553469479145822420208291551214685836322120517079008');
    T('0.000000000000000000054924120831599544196056464059913345528362581686003585823080198063175104748720429835436816', '-33.965305825679', '33.965305825679000000054924120831599544196056464059913345528362581686003585823080198063175104748720429835436816');
    T('-0.0000000000000000272691295425523253281545245364330746310034056347651726374661392748936831102659438676749529105824808034929101186122342434701063245841476', '-781187391210758142167459761699751749533058942483806705247714720005729302232618042462.31429718942883993867507', '7.811873912107581421674597616997517495330589424838067052477147200057293022326180424623142971894288399114059404574476746718454754635669253689965943652348273625338607251063168897340561323250470894175191965070898813877657565298936754158524e+83');
    T('127262997433251959079279503780471.05197845683869545951325487141398044178736403887617606907597695630472759165663283084977628891449702428789338877110708', '77396683518158379744711.66156663437342404789854794363811642267757202778894836789743946370198310026950193022769015766161296030', '1.2726299735585527556112112403575939041182246527141161470692777586401910979201108722770117853749260274449138713090062208613125288406398789338877110708e+32');
    T('42170499815256385540791846972137805290009126085642915432609561244148971544358403270767061193697589382522357089597879438170922900180458360491', '-0.000003885040489695371692757168739195827258540410745579', '4.2170499815256385540791846972137805290009126085642915432609561244148971544358403270767061193697589382522357089597879438170922900180458360491000003885040489695371692757168739195827258540410745579e+139');
    T('-1484024306291646967346493622090638440659128585952793958990883359092640546575462579828476965972418873274552.1834807695558477417523', '-10598014380036531229136810438900071681432863341249945982490785444176829679842459382408061192080712.9619817785969053235790359984721057762058', '-1.4840242956936325873099623929538280017590569045199306177409373766018551023986328999860175835643576811938392214989909589424181732640015278942237942e+105');
    T('-0.0000000000000037', '-3076174224740955436355989748079441650915310480031800204994005115124341562063700871463141434110179138410587089277839517152320944517479716270543461', '3.0761742247409554363559897480794416509153104800318002049940051151243415620637008714631414341101791384105870892778395171523209445174797162705434609999999999999963e+144');
    T('-0.000000075027040201482330625619859591861141930627870540731155446710', '0.0139177672366169698378876807351533299376331910835826839154061965617972722046119104334991574172', '-0.0139178422636571713202183063550129217987751217114532246465616432717972722046119104334991574172');
    T('8875143592690748989920787895291661147029298230059201908543304359430935440199.867134812405756511257745234125626824', '-247943204197117001912763303824992424614854547911.72151922913414922441109501', '8.875143592690748989920787895539604351226415231971965212368296784045789988111588654041539905735668840244125626824e+75');
    T('31916591334815493867348.2006986155618932404016660295722', '4606925311680888622.612954414237689607109445591833340796649297129233762211446696674', '3.1911984409503812978725587744201324203633292220437738859203350702870766237788553303326e+22');
    T('381592731768551523450237299269065558626439762823266272728608770256434241300666186155782732703413979594892722766059796378113553693016425636137853', '1074098218589615684162333302525523', '3.8159273176855152345023729926906555862643976282326627272860877025643424130066618615578273270341397959489272276498569815952393800885409233361233e+143');
    T('859384792671165504880.591270312184098707804090920437508527546076998804061888741623409867', '-415064661757507462508064979252020173304919.38161995361360609605709210292378740874075259401137388397242347603728910288128847206', '4.1506466175750746250892436404469133880979997289026579770480386118302336129593628682959281543577271404688590428910288128847206e+41');
    T('-0.0000000378511781638932', '-0.0000000019415746608194397271645430475998139434698262967689419183607208269227560180148712767202698791460535585478881851701240', '-3.5909603503073760272835456952400186056530173703231058081639279173077243981985128723279730120853946441452111814829876e-8');
    T('-127323.654151371587544406942297034389812264638323249811302292394967515069458109985654766167188775393537479581494034', '653407290188829294787949470104324404483492727890', '-6.53407290188829294787949470104324404483492855213654151371587544406942297034389812264638323249811302292394967515069458109985654766167188775393537479581494034e+47');
    T('155323920605544966037456890780901792411.6779803510786020326988033806935198', '159029239238266232.68574849991657262205375600388916', '1.553239206055449660372978615416635261789922318511620294106450473768043598e+38');
    T('13156801663791593515197335348715600238117690700.815038836887843190319660754853410804197810482520222445623353150435061783014391903811607138', '-8270199589564900477356715352108242301640427858623530223929697653745713628878.564288094121549802323184040403763027925657974', '8.270199589564900477356715352121399103304219452138727559278413253983831319579379326931009392992642844795257173832123468456520222445623353150435061783014391903811607138e+75');
    T('0.00000000000000003786195460340598104288682558070063337088309567282201351032563', '-1912699796464583227379572473541475383216013821339026581757100140817849255701729993948439667349733408806253839792.24198692029', '1.91269979646458322737957247354147538321601382133902658175710014081784925570172999394843966734973340880625383979224198692029000003786195460340598104288682558070063337088309567282201351032563e+111');
    T('0.0000678213457017134314852892364803864629433537599025120942836845115758304505087119211392119756099777668193068326923413902250456637918108722842255', '1140640116158968963485566592.5893331949954139196192113429058069640694322621110308576696441720269550777832332421', '-1.1406401161589689634855665925892653736497122061877260536693265776064889083511283455753604875153792473327245301788607880243900222331806931673076586097749543362081891277157745e+27');
    T('-5941423076793531487369004038913872115546949940523795.371014687155306260796761670579766889905', '47861971864566209979133077736766504352908819881729160935697807580950039529832250339875991.2370', '-4.7861971864566209979133077736766504358850242958522692423066811619863911645379200280399786608014687155306260796761670579766889905e+88');
    T('-3651590804629633244384272036497219152794961178826414247756327968228195689150408706412981706033290645060605', '-0.000000000000000000047076028716961617745970472461943359903971686105244537457258691', '-3.651590804629633244384272036497219152794961178826414247756327968228195689150408706412981706033290645060604999999999999999999952923971283038382254029527538056640096028313894755462542741309e+105');
    T('118687168472.901776101483448', '6575654819663787.704095301960878738190', '-6575536132495314.80231920047743073819');
    T('0.000000000000000008221087096832270706747915015998320970007508295513800923805028534762897555917424732521734252235383609982', '-4730231.56580380830953', '4730231.565803808309530008221087096832270706747915015998320970007508295513800923805028534762897555917424732521734252235383609982');
    T('0.21789670842440974974661994228192699055349', '0.01334643986427991665741664557', '0.20455026856012983308920329671192699055349');
    T('-14226117087.28617301017414275549732687575166423932', '-2570477277243155.1204160224690347778553830976260385383374348745734315475200813314035', '2570463051126067.8342430122948920223580562218743742990174348745734315475200813314035');
    T('-46.941057107072660652688121977370', '-1198710434479146302024548885128439972868624935885330601645871874157901.96708543469474', '1.19871043447914630202454888512843997286862493588533060164587187415785502602832762207934731187802263e+69');
    T('-295140171894972088951187786.0843050337025558841174717700928362793074491331276762385630102550855368430612108831357', '-0.0000000000000004883812758401282299689103133449934936', '-2.951401718949720889511877860843050337025553957361959299646063103971357881341826385630102550855368430612108831357e+26');
    T('1.4711261754068130281842564186342696000895948066', '-120648633751429663556459972182655021410865.5086632989898240206360208470238848290763965597192371216857894570350195365591255381749962060', '1.20648633751429663556459972182655021410866979789474396637048820277265658154429165991366319237121685789457035019536559125538174996206e+41');
    T('-144700', '-0.18213538140165154164724446677239877238828512236771523708810552188227956132323566115125955382011533702944784', '-144699.81786461859834845835275553322760122761171487763228476291189447811772043867676433884874044617988466297055216');
    T('7527305921883997190334173981195704877744988728728518609052340094791.8855544226349862816292745519085545', '-0.0000000007048946656101980003207741963970650160207507827240902', '7.5273059218839971903341739811957048777449887287285186090523400947918855544233398809472394725522293286963970650160207507827240902e+66');
    T('-63795532.1247756163974543523664090934199701938480314046575430119409403191645390756181006417970791', '-0.0062137513829573043880447774252661092604135729387774810554', '-63795532.1185618650144970479783643159947040845876178317187655308855403191645390756181006417970791');
    T('9170218742863146.63960565263545342233167920145227196559581870281019754919446058190890025166080356692023953226567583', '94899.938506789077785525289197268929739161960202702966987511034', '9170218742768246.70109886355766789704248193252253280363561599984321003816046058190890025166080356692023953226567583');
    T('3901879734193765914331129847202500304665157801252541780432220949839069127986525493675829826983735746021109803224777662529', '5885103402359590328286381776411514113770718249033564454051018031288461.42208859547727201936136', '3.90187973419376591433112984720250030466515780125253589532881859024874084160474908216171605626548671245665575220674637406757791140452272798063864e+120');
    T('3', '-15200.9143662493055675379993', '15203.9143662493055675379993');
    T('15423485096047599053364024548519496681508881945096830176039851680666630029365886063521259842', '-0.000000000283965132096', '1.5423485096047599053364024548519496681508881945096830176039851680666630029365886063521259842000000000283965132096e+91');
    T('20462143', '63769737', '-43307594');
    T('0.000000000002655335762566545159279103266301165139940149572346903456821981448916315558904', '760001783034845256482634525599227420841592883976982090372844443290413192545720519296', '-7.60001783034845256482634525599227420841592883976982090372844443290413192545720519295999999999997344664237433454840720896733698834860059850427653096543178018551083684441096e+83');
    T('-2859.72', '1143457536770062100322750086623.71845', '-1.14345753677006210032275008948343845e+30');
    T('-0.00000000000000035489705537003133966656620934689', '11987350492635913981351083990815685367027572166189009953407790102342640597311.916155924353548989207704206322071476178972779', '-1.198735049263591398135108399081568536702757216618900995340779010234264059731191615592435354934410475957635341114274518212589e+76');
    T('-672459565845298082592172835281835774315217847427856401542010053256271699439933658929866947464739169479833929233830480664939232420953.2923', '63108751.15749592458552', '-6.7245956584529808259217283528183577431521784742785640154201005325627169943993365892986694746473916947983392923383048066493929552970444979592458552e+131');
    T('-0.000010035206280282747019162655750927410618908635896952580', '230.1907049834484816952377627084616053336290', '-230.19071501865476197798478187111735626103961890863589695258');
    T('7173044628.5777843255026455413348266', '74453983.680083199805556', '7098590644.8977011256970895413348266');
    T('-83919871511965703755476466.256569440941669204141450851398993', '-120420004260233575347420874147255645547454760214523', '1.20420004260233575347420790227384133581751004738056743430559058330795858549148601007e+50');
    T('3630900902246671948646521726957218461983828034217333961862682876897350087684.6822', '142378828011194667522298164078675050366271553.18418507871350239937970327468837520064392740785265', '3.63090090224667194864652172695707608315581683954981166369860420184698381613149801492128649760062029672531162479935607259214735e+75');
    T('3600368521700307314019929721242984506937722877903512324371675383642175079745764808500491934491199396', '-8889151549.0980820465205798627062897257998560429872362749013588868330096', '3.6003685217003073140199297212429845069377228779035123243716753836421750797457648085004919433803509450980820465205798627062897257998560429872362749013588868330096e+99');
    T('2566124149510189054603184804420483142674551293246048640739644229580149479831221856889682947040898.1947', '2.26262852', '2.56612414951018905460318480442048314267455129324604864073964422958014947983122185688968294704089593207148e+96');
    T('-6445706060883314535319320109991761267536435348960454296790760967040816468929', '-0.00000000000000000062830932861674764819252613026097705872828029129848817631757479456721702066162270528458013184133179280907256876980768150732125981012639579785215', '-6.44570606088331453531932010999176126753643534896045429679076096704081646892899999999999999999937169067138325235180747386973902294127171970870151182368242520543278297933837729471541986815866820719092743123019231849267874018987360420214785e+75');
    T('0.00000000000600072079222315465580504023456367837739567604666764878391621677969104485271956013728552504195645646316020084441518304737259573663289004508147', '39.31302175525738768291245783088445596818951069182891044427517151104661708832', '-39.31302175525138696212023467622865092795494701345151476822850386226270087154030895514728043986271447495804354353683979915558481695262740426336710995491853');
    T('-0.00000000000000000294054960556608929', '-0.366150388110291163127790', '0.36615038811029116018724039443391071');
    T('-511163388228633237873316822527313608282340841918478362004266090071119793101871.464037584689475734', '-329989907679436788948852901850804713300444041228309095926110784330762590667953827054759693', '3.29989907678925625560624268612931396477916727620026755084192305968758324577882707261657821535962415310524266e+89');
    T('0.0000000000001704747491801544304246616272150314276800251600985090', '356912525185464485672340637548121974961395954891152868968788980713248525881582063315953007660073165537954814817936395362149647051889219299067882', '-3.56912525185464485672340637548121974961395954891152868968788980713248525881582063315953007660073165537954814817936395362149647051889219299067881999999999999829525250819845569575338372784968572319974839901491e+143');
    T('252850202046876977851994.143212', '-848887102457721130097.091770', '2.53699089149334698982091234982e+23');
    T('-1463813561256.991439146629821901956687441988620850205985146015744308635116738069577049349761881704662355647698634718200189276416926210554813240', '62029849111463410013756762844374812825897194275000462631388966790212235003405628390', '-6.202984911146341001375676284437481282589719427500046263138896679021223646721918964699143914662982190195668744198862085020598514601574430863511673806957704934976188170466235564769863471820018927641692621055481324e+82');
    T('69782305662663437094541084757082159939270157247.9716354', '-89147785350539844445.416910718922580580536969939488837234964042787197740548106624', '6.9782305662663437094541084846229945289810001693388546118922580580536969939488837234964042787197740548106624e+46');
    T('-131.76', '-0.000000000005696147108564746426888775570848202211810884752865344375891071070092502829538629785436541334365692051020683595375875811133291302478345676258', '-131.759999999994303852891435253573111224429151797788189115247134655624108928929907497170461370214563458665634307948979316404624124188866708697521654323742');
    T('-497777803060949043131155159968723958.16915576601446255882706227712009561', '187513374620216567870246128064.585091639963330830047759047069515276559563684958977723723813958088783235081595183826887', '-4.97777990574323663347723030214852022754247405977793388874821324189610886559563684958977723723813958088783235081595183826887e+35');
    T('4125226628898214776459043043848665640891754611330056829115889332318900550787802063162881102.0172', '-5654133954935729315244416.571910565', '4.125226628898214776459043043848665640891754611330056829115889332324554684742737792478125518589110565e+90');
    T('0.00000000004139915941037474807032518234750457019529957219537984823906929653706764566737605928162402285281158810275719639', '4918252046919178557340409576205071237879887503291090041', '-4.91825204691917855734040957620507123787988750329109004099999999995860084058962525192967481765249542980470042780462015176093070346293235433262394071837597714718841189724280361e+54');
    T('-2311168251436595165228357269442780578123112.83612714285098893733488905613344909999353', '-23179010559927745251596254686184413186603531856508883463422590120878079108990895937457179980442401237385839673669330286893540161257', '2.317901055992774525159625468618441318660353185650888346342259012087807910899089593745717766927414980079067444531206084411296203814416387285714901106266511094386655090000647e+130');
    T('-157110395920267638794958655839617698095333501445314955918734069161689933642356923466527500.842491896021313687363006263896039621', '-0.00000169938824017905253390365814790480115170799502101643798', '-1.5711039592026763879495865583961769809533350144531495591873406916168993364235692346652750084249019663307350831047236023789171619884829200497898356202e+89');
    T('-0.0000000034695699166026892143077647500761510769012638170656915865714501717320327171654999280961297567557115531', '0.000003285105858090918101513244579', '-0.0000032885754280075207907275523437500761510769012638170656915865714501717320327171654999280961297567557115531');
    T('12.821444033432329487', '-828945494613921320615913033922430384667832091835552867448637507842855527762415897847274731250460560.5938743219195296432763484012308615936905854362', '8.289454946139213206159130339224303846678320918355528674486375078428555277624158978472747312504605734153183553518591302763484012308615936905854362e+98');
    T('-21780810408046000685162929853231193847862416557439280736028379007620.9726981547505', '-139252299485951365181945740150018259705534184833100450.519901854594606176500', '-2.17808104080458614328634439018660119021222665391795752018435459071704527963001558938235e+67');
    T('31153778331.4387887187891452717104214859599795399160108427720346733063507', '179728021.06909075407924714', '30974050310.3696979647098981317104214859599795399160108427720346733063507');
    T('-618834912531652.0057377', '21797574580198291897359627810662681102637063422664387863754038865817', '-2.17975745801982918973596278106626811026370634226643884825889513974690057377e+67');
    T('-0.000000000000000765821929389877370458202611860322225362885792434335153360684490327702934795208517771383256418915784282787289167843225530943806979468309', '451287395769559434110582968934970.685236', '-4.51287395769559434110582968934970685236000000000765821929389877370458202611860322225362885792434335153360684490327702934795208517771383256418915784282787289167843225530943806979468309e+32');
    T('77106705055908475310440141267552925652440318989464386727898512296975567.66061423617574863098323698323360502040356558', '-0.000000011450841720290860862448039550281180968988484941121896581469341885197489647153563705690545985946332080104331971976300', '7.71067050559084753104401412675529256524403189894643867278985122969755676606142476265903512740978456816445706847465489884849411218965814693418851974896471535637056905459859463320801043319719763e+70');
    T('-0.000000000000019290717114050931338105101245194653742552644960059', '143276467308241499701', '-143276467308241499701.000000000000019290717114050931338105101245194653742552644960059');
    T('-0.0000000000001540616672147681280171217297593042431394881995494063140060573330749436996881403638596216426697967870302615524370008926698907505446249500632', '-0.28350757173542186862807156137406823418555996167492030058802005151707375796930334126928268135255', '0.2835075717352678069608567932460511124558006574317808123884706452030677006362283975695945409886903783573302032129697384475629991073301092494553750499368');
    T('0.00000005655991707149037399922054071139097387281061191477648236574838149302923575757471824009608753090882473645396585695414457', '-1037735687954213696569384673332890401637635622739449765500.241575915381331', '1.03773568795421369656938467333289040163763562273944976550024157597194124807149037399922054071139097387281061191477648236574838149302923575757471824009608753090882473645396585695414457e+57');
    T('4855675332.7083066004691748608', '-9495842049673468279.5671213956484193849367821937814670521168063407441607221529038456882314074643242900887127616369767306383357140885', '9495842054529143612.2754279961175942457367821937814670521168063407441607221529038456882314074643242900887127616369767306383357140885');
    T('-371727694604000265531518957397404745190308397247774643342809559479423488145460577873367071667140515223960430330052550077105059687.602928315818224370992', '-0.006182613654799728503894831274202426819463433943221582526744062754609114781332193519619', '-3.71727694604000265531518957397404745190308397247774643342809559479423488145460577873367071667140515223960430330052550077105059687596745702163424642488105168725797573180536566056778417473255937245390885218667806480381e+128');
    T('252.7286', '-1048727383.8417648117635129245612611075251', '1048727636.5703648117635129245612611075251');
    T('246339105236279993029.779136671622836638254622844315', '-0.2081466910740537', '246339105236279993029.987283362696890338254622844315');
    T('-0.000000000004251821155298564603972810714958089479168033107886959502196802756203470788176960484264054952475484390527833088494710819240861104331195786410885', '72374649981010622768585141052423674921404658971075010108219334219091239517373927587321641489790577909684263', '-7.2374649981010622768585141052423674921404658971075010108219334219091239517373927587321641489790577909684263000000000004251821155298564603972810714958089479168033107886959502196802756203470788176960484264054952475484390527833088494710819240861104331195786410885e+106');
    T('183837148.90443961940430047231989366651993476031567037971215956651037637016802721769559763173816796542803144211780111934219050428648', '-1132210802667233013885301883421427506588959233625', '1.13221080266723301388530188342142750658914307077390443961940430047231989366651993476031567037971215956651037637016802721769559763173816796542803144211780111934219050428648e+48');
    T('-3691822058.57085474396080945369673728122984439428444257734562382467780353804871181415538969082586191879205496710', '3926432506300249210509179487381770792289628403911395731250431112551381231067577307.7886693710023579159088025939730616110630998092011', '-3.9264325063002492105091794873817707922896284039113957312504311125513812347593993663595241149631673696055398752029060053475423865467238246778035380487118141553896908258619187920549671e+81');
    T('161380629205776417370619978.94790585532571175965001173657717968258603', '15770163883113132525352877130764618385076935573552026524962864893082340267956294317512981350074440117.51141761065457763829', '-1.577016388311313252535287713076461838507693557355202652496286489308234026779491368830720493270382013856351175532886587863998826342282031741397e+100');
    T('-0.003870041060244155089', '372037566.3292566800057303692867308995967514456947136761173', '-372037566.3331267210659745243757308995967514456947136761173');
    T('306847039405846004795255377789835522736966045.703419256307644158563670046841211782545782850044162871280281083966845034443161', '-0.0000000000920455239477610995683838741187048041416049264294738878526378270582', '3.06847039405846004795255377789835522736966045703419256399689682511431146409595656664487654185767797709754971819482861501361e+44');
    T('-0.0000000000003210953781062772150098343898602436843086564236327052301688', '-150338459473158557923119665782760876423441993048051543691089147086917525625936551761033445707803557045155557.52837300270019703', '1.503384594731585579231196657827608764234419930480515436910891470869175256259365517610334457078035570451555575283730026998759346218937227849901656101397563156913435763672947698312e+107');
    T('25983124363595065.33092768399411430713290111808632249366', '-580933674475894898982997155309901918649485043287834362560222468094831906478743818165029199620003568770150378311086272184208826293576318808290', '5.8093367447589489898299715530990191864948504328783436256022246809483190647874381816502919962000356877015037831108627218420885227670068240335533092768399411430713290111808632249366e+140');
    T('1850916610962229939602303480362676023908431072215702014137566112296384421153424607176.8059784587131', '512243500267', '1.8509166109622299396023034803626760239084310722157020141375661122963844206411811069098059784587131e+84');
    T('-184877142422294703520975479796306025809263499292.7931910621756669188662102660938494746987999101249089061195470207030', '-0.00000000000000008498222206556406262243165', '-1.84877142422294703520975479796306025809263499292793191062175666833883988200529786852267149910124908906119547020703e+47');
    T('-0.000000000002591577406696730110925816397500154615', '-0.000000008203755062608231746294494108119453489077150713983075114286087573203872407396100452280836', '8.201163485201535016183568291721953334462150713983075114286087573203872407396100452280836e-9');
    T('0.00113187652077011811375354713438890135407233573999645640576361426763004959545048824419375996447732201405034000719887068357917006193372968721', '-9513299452685933145884287122288569823742580460859259721427216624858272956972621512882828978401205032497576349955400997444307472.127288502040606', '9.51329945268593314588428712228856982374258046085925972142721662485827295697262151288282897840120503249757634995540099744430747212842037856137611811375354713438890135407233573999645640576361426763004959545048824419375996447732201405034000719887068357917006193372968721e+126');
    T('-101282948616724272484953545953349716593478543806497.286950890059245094039', '117309932329495122737411487914013724789825355301759777928575644736371090499526769006843968707622513899074453', '-1.17309932329495122737411487914013724789825355301759777928676927684987814772011722552797318424215992442880950286950890059245094039e+107');
    T('-3527672542958095940230507255558332.170340607904850914382797250686015987003437982374864974619427360612046545786860047', '-44308074539', '-3.527672542958095940230462947483793170340607904850914382797250686015987003437982374864974619427360612046545786860047e+33');
    T('696251462779098640018848016194974848273984539897182732503721195581.022', '-0.0009913757384911224166061061275859102001742166932949065339677297902168319985124401', '6.962514627790986400188480161949748482739845398971827325037211955810229913757384911224166061061275859102001742166932949065339677297902168319985124401e+65');
    T('1078800798224.734663928613388302956878421495374656333024393408174361403843900153053669034118433549172155326377609', '-395534581529124125726253950708301388351136671971313.269513765972391964435573515235', '3.95534581529124125726253950708301388352215472769538004177694585780267392451936730374656333024393408174361403843900153053669034118433549172155326377609e+50');
    T('157736254008627834038860309559683387724555', '-4594454316276846.22889', '1.5773625400862783403886031415413770400140122889e+41');
    T('192386447850938643364980037157714078.667670', '-851657218576.302225280595205926373252086357245881159099186004946924709955966414529108970202216172176', '1.92386447850938643364980888814932654969895280595205926373252086357245881159099186004946924709955966414529108970202216172176e+35');
    T('-30904996899995566378174385484494931717271463370895674716598098724288017981274999017442756424.6379761510688077348608197361985605065677866468436805', '4689937374232446.45257401892489335752', '-3.09049968999955663781743854844949317172714633708956747165980987242880179812796889548169888710905501699937010923808197361985605065677866468436805e+91');
    T('-75737124093497081558918300645262008779192868684414555820427607278946300576673', '217541673539103', '-7.5737124093497081558918300645262008779192868684414555820427607496487974115776e+76');
    T('-0.0000000000178582787097936410955952527724904571812233290700799389147161616449340030526614581183800172797335354127416587536811082238416196766809706109306282257', '1655811940148466388814205.23320907612777562208494378505895383608907847845502485386735176120340958223402858437920118040340355068270079322722548', '-1.6558119401484663888142052332090761456339007947374261545490888615689356362481829374317001181257438789625874318626385217835679624343286399671387536811082238416196766809706109306282257e+24');
    T('0.000000002919991993491021882536587348', '-505934880332731792510691865564845809059631032976000339923495992940889398540650687707453464781582591628461285346207100158557879217', '5.05934880332731792510691865564845809059631032976000339923495992940889398540650687707453464781582591628461285346207100158557879217000000002919991993491021882536587348e+128');
    T('-33466744637598795529381545892043857804617468.46401319673766053647481787577214595481626050900894623520', '-0.0000000000003357237902324', '-3.34667446375987955293815458920438578046174684640131967373248126845854757721459548162605090089462352e+43');
    T('-89425417162201175465400546339403148500149133957193.96578038786595026758317811953966733498932978601', '-1405803301520003345575789184231943451603588638985.9376135626367435873857739001195431353742', '-8.801961386068117211982475715517120504854554531820802816682522920668019740421942012419961512978601e+49');
    T('0.000000000000001871288647830989494290689415161953067408619362137083942908468831270334812418581424545772814063266068263844492510500380710872894374', '4.23013879231167686677612360843795550900659989406713647936612', '-4.230138792311674995487475777448461218317184732114069070746757862916057091531168729665187581418575454227185936733931736155507489499619289127105626');
    T('1081898192142823325334768391585696724.266551471539993', '-17659238921005307.917413084214192739', '1.081898192142823325352427630506702032183964555754185739e+36');
    T('-211840291024698604.48893683835289524386164142885412599596', '0.956798615642965592754457212313533228615333073536496635490759649687294643223525742190828313263626448735964', '-211840291024698605.445735453995860836616098641167659224575333073536496635490759649687294643223525742190828313263626448735964');
    T('221425316830895877056683698880899917852135343813023998859360273819797052290640.9315231787425484197016892424623525', '0.00000400616385054508', '2.214253168308958770566836988808999178521353438130239988593602738197970522906409315191725786978746216892424623525e+77');
    T('0.0000000074648516794219838184240130975016099', '116876.42749590099927445693850031173280282361261639', '-116876.42749589353442277751651649330878972611100649');
    T('-73443010540402954576842026', '-4926832937510632043653806478767750613389877126019642141260618065698.0792895022817212282107503047772151899589664246413340', '4.926832937510632043653806478767750613389803683009101738306041223672079289502281721228210750304777215189958966424641334e+66');
    T('2784502990835628144358253950083873358740549170601580432289611124376226555234877.9858200557236237337577147596', '4917661361731789584921195447270.5587547070375957130008047176901629887458255', '2.7845029908356281443582539500838733587405491705966627709278793347913053597876074270653486860280207569100419098370112541745e+78');
    T('0.0000000278504063853447146762310784743311219577950698679342514425094994531041925550104863191773566381681464090705194734640', '327544799712156512794784451228653130604976681932044866.8130716684653513802', '-3.27544799712156512794784451228653130604976681932044866813071640614944994855285323768921525668878042204930132065748557490500546895807444989513680822643361831853590929480526536e+53');
    T('-0.00000000000002008111880513429242547', '-1168142699198895682872256551220621216397424308459153677585114.96438975713066891515', '1.16814269919889568287225655122062121639742430845915367758511496438975713064883403119486570757453e+60');
    T('-0.0020759804305965617', '764484378925575955998262271107967177799.3921673495985811141655562538838430225703', '-7.644843789255759559982622711079671777993942433300291776758655562538838430225703e+38');
    T('314538242246288.42720726996770214230667062396125055947593471536', '-5709766296171650602.24185600104198360224', '5710080834413896890.66906327100968574454667062396125055947593471536');
    T('2601655151900642206636222997530561092879117314164702943580485229395095933.8148647041', '0.0000000000000000000545923963200601881583792553271167476403317094011785783473945906028127827591476470085229569639102315240902527107858916390', '2.601655151900642206636222997530561092879117314164702943580485229395095933814864704099999999945407603679939811841620744672883252359668290598821421652605409397187217240852352991477043036089768475909747289214108361e+72');
    T('-43151301919298268068196451304198990983037212765635867863105642809532901509070161236071836104869066552364969311762463975552', '-65792506329.162401577209876729615122943184875102458456675147552153', '-4.3151301919298268068196451304198990983037212765635867863105642809532901509070161236071836104869066552364969311696671469222837598422790123270384877056815124897541543324852447847e+121');
    T('-58145525166565890.04122722148517396992268609808559449424081643957465013', '36895', '-58145525166602785.04122722148517396992268609808559449424081643957465013');
    T('-1380019075368410665434910685837646231061500678571273.437', '-708783781474372956418809.1668488022806394799579905891256533778023871539', '-1.3800190753684106654349106851288624495871277221524642701511977193605200420094108743466221976128461e+51');
    T('1963393100153283900680602317090841091868.7057877321407368918', '-4558805287590516608986267171325288983366964400655106', '4.5588052875924800020864204552259695856840552417469747057877321407368918e+51');
    T('0.0000001407996470540256884206972291699170231269478445264415596334101718170629891348240406473704424192652557212601717979646372575017642378452081', '-68319808.85143904948777163237579395976723632411831309906501022', '68319808.8514391902874186864014823804644654940353362260128547464415596334101718170629891348240406473704424192652557212601717979646372575017642378452081');
    T('-0.00000000000000000520745300386927858263802847774818906513849609279791649851542455109794569658396745790033', '-0.0092550980435850490662646493376107', '0.00925509804358504385881164546833211736197152225181093486150390720208350148457544890205430341603254209967');
    T('2967617678347088013.1738970622321751596070351867397208390655655108226821804468393275334586999342460367780260036922367170801495', '-1037314302.0266874363156528872956184434218993246886249535339781421', '2967617679384402315.2005844985478280469026536301616201637541904643566603225468393275334586999342460367780260036922367170801495');
    T('-49291697.79', '3118337286372049189132894.361136097492458179780194692374756783544285588810310527347173746912677367611366854072067678627774', '-3.118337286372049238424592151136097492458179780194692374756783544285588810310527347173746912677367611366854072067678627774e+24');
    T('-339.2', '163572547291591069133936914034534.567315194', '-1.63572547291591069133936914034873767315194e+32');
    T('34325086548698898487887478294.36420', '355576739939045466327.4975269472620813909920831535481189862294384252210442757667064506332182539172503591866982', '3.43250861931221585488420119668666730527379186090079168464518810137705615747789557242332935493667817460827496408133018e+28');
    T('370044764.3013082042', '1790005633588538461650641328587108706658340072380027626477.0132722745841414452316360238830847702114898024038942988300305', '-1.7900056335885384616506413285871087066583400723796575817127119640703841414452316360238830847702114898024038942988300305e+57');
    T('4263702202353975994071.31690964522111184782225643263309765852647473593539755902567196', '-19975397854626091996286240550959532193241549861480938890275634702569032237325096179631367809302680197661729288714.051', '1.997539785462609199628624055095953219324154986148093889027563470256903223732509617963136781356638240001570528278536790964522111184782225643263309765852647473593539755902567196e+112');
    T('3486961.94860938', '155083657058384127930580314746.0523483468034310636373779', '-1.550836570583841279305768277841037389668034310636373779e+29');
    T('-12500272340430247337584871234769182598217741493536064909830797164011240022929666553931.1010', '320228573980378863122568709145317117449935889878174725217701802157098986085371.51305672546876049053871', '-1.250027266065882131796373435733789174353485894347195478800552238171304218002865263930261405672546876049053871e+85');
    T('-5051726177322.81786601098426577', '-0.478083315601904193517321798543685421468736554729083533236145832716201463299848292219431026180384620242431752321517', '-5051726177322.339782695382361576482678201456314578531263445270916466763854167283798536700151707780568973819615379757568247678483');
    T('-1805640145024569891681224418968926516432842084069.4683478', '-0.00004666227081', '-1.80564014502456989168122441896892651643284208406946830113772919e+48');
    T('-122539612725006940436479306393463901576107037860603468764716.442090206833953614902041846195976605726465853757952136575397703', '-41270858.6670990535407801373074922648949382808920775900799658210502034041462604039074761', '-1.225396127250069404364793063934639015761070378606034274938577749911532931734775945495813010383248343882636779863155251942988537395960925239e+59');
    T('-122359.1727490376', '1581678069.9607', '-1581800429.1334490376');
    T('0.00000828479828485256539017422162536712777837952542406358440363757753554784788015690875096756272853061031753883063876226018540022889614127275718921', '-0.00000000000979390831128660928240121877046', '0.00000828480807876087667678350402658589823837952542406358440363757753554784788015690875096756272853061031753883063876226018540022889614127275718921');
    T('3014937410709080363048635820022478067970287003612206702018715047141006381003373200865760747279383102148885920325786', '13192212031830204018622500889173518071465445462648474880006710438901406028.19078998512989422200926796249911378252', '3.01493741070908036304863582002247806797027381140017487181469642464011720748530173542029809880450309543844701891975780921001487010577799073203750088621748e+114');
    T('1137652429855.7006716974270280155270950682974135967673015740628328197884', '0.000005536906484', '1137652429855.7006661605205440155270950682974135967673015740628328197884');
    T('17516195002079504046599923580222888468136191973844988762823377013157696208127912781416616872247424539619546427766', '0.000000049332322771448227773362567337764784013258792663644857484303435435122827459514247309673313863866397145377917359757', '1.7516195002079504046599923580222888468136191973844988762823377013157696208127912781416616872247424539619546427765999999950667677228551772226637432662235215986741207336355142515696564564877172540485752690326686136133602854622082640243e+112');
    T('-27506767197655948181.328949804790883947367925537829924105775693473950222413500805587179604', '851744208518189337306972955526150694905469056618627352385522013393688952141414396074470925490405490728735', '-8.51744208518189337306972955526150694905469056618627352385522013393688952141414396074498432257603146676916328949804790883947367925537829924105775693473950222413500805587179604e+104');
    T('0.00000000000000016100516695902975851532468630731065388238733922125157107578982740360720', '0.000000000000005453287371976994277320664460255897903401402435852268978047976350868347882510607594', '-5.292282205017964518805339773948587249519015096631017406972186523464740682510607594e-15');
    T('-1015.36', '28726679251696495519418305.7', '-2.872667925169649551941932106e+25');
    T('-936968911546704282843988971022862859429380575863600004134648546127147385.550916104277091139359959595538873413416181058503593874140', '-175348106986507342335754762602245018364666640954631672307547275757550482650641067', '1.7534810604953843078905047975825604734180378152525109644394727162290193652349368144908389572290886064004040446112658658381894149640612586e+80');
    T('-82283247317913068455250030332808481250588157.87622854768457973434714154261139038109409280493164157480405984455365143662342695995148270', '44.99825878', '-8.22832473179130684552500303328084812505882028744873276845797343471415426113903810940928049316415748040598445536514366234269599514827e+43');
    T('0.0000000008490168', '-0.0000000000000006074865058947945839', '8.490174074865058947945839e-10');
    T('3636426464211118305138698215541505022990147742911418561057930702808150291', '1598908059965309335026038743126561583060668905575689367.6003785995795', '3.6364264642111183035397901555761956879641089997848569779972617972324609233996214004205e+72');
    T('-7516190.940861499379106729192131043068573493293233100655140480083907054072822505940742766302343169414520157351216195387299511801647', '-0.0000000000000454932933336808575963301305396992632885880696675679107651681040430359684592346731492202438', '-7516190.940861499379061235898797362210977163162693401391851892014239486162057337836699730333883934741370937107416195387299511801647');
    T('-48288936.78722249318487555690454563461', '-2005716261952974336303267852338923837806513', '2.00571626195297433630326785233892378951757621277750681512444309545436539e+42');
    T('0.00000000000000361164834854643173545096303609810033142721680028195071060833143531948647509219803554613154933367271433460345060095402441586344758203', '0', '3.61164834854643173545096303609810033142721680028195071060833143531948647509219803554613154933367271433460345060095402441586344758203e-15');
    T('117527597286243844925820166915403846128409126690409737.7024153118718614783138378277032681506289005899206380178136648075347527401515', '142970923259618812743278564725226081502378163356150859956591622347575726634881287085', '-1.429709232596188127432785647251085539050919195112250397896762185014473175081908773472975846881281385216861621722967318493710994100793619821863351924652472598485e+83');
    T('-421.637794298766899659968770626491660939483442168172530357951266709389524009618302536438221', '0.000007500440138930182530717119147674010713308857022125249921960367734502038756006365022302482030874542048185964246356551809776026365', '-421.637801799207038590151301343610808613494155477029552483201188669757258511657058542803243302482030874542048185964246356551809776026365');
    T('151539803714.55419760516058121061664', '5947110489174350865159529800495269979412122032546964623633750977098107', '-5.94711048917435086515952980049526997941212203254696462363359943729439244580239483941878938336e+69');
    T('42.96514137222747126649680518556598169535979854687275322179628273846143520146782958673105379865087129307601', '-10953889685.72652645743016845401321258839294061151336635450954240593892170965006938629511080', '10953889728.69166782965763972051001777395892230687316490138229562773520444811150458776294038673105379865087129307601');
    T('-45444452978530864728152294556239654908077032829329359349565438897.0872421377694', '-1.10543198001234853494574224133819939759899818427770986357398073461833864778527689', '-4.544445297853086472815229455623965490807703282932935934956543889598181015775705146505425775866180060240100181572229013642601926538166135221472311e+64');
    T('-0.002845911806687586456195667795987390886482178862015476630127308203573595922990347328439736545942442607248575854261113227721222844030682403654028296609', '-108143910626361371525060838061026913900811972840772563215348121329650489043833335691285421668618034618407054685798027697197635.65860759495683373744', '1.08143910626361371525060838061026913900811972840772563215348121329650489043833335691285421668618034618407054685798027697197635655761683150146150983804332204012609113517821137984523369872691796426404077009652671560263454057557392751424145738886772278777155969317596345971703391e+125');
    T('-46907095890372173836389327440139250674757510384063869086116234641758602924747249181765303804967251303089507.19103700711514447125694845740498171264', '25808803835091891711851699685937630228367986645198551990544154370479382.0551364316906891141232071494', '-4.690709589037217383638932744013925070056631421915576079796793432769623315311523582696385579551140567356888924617343880583358538015560680498171264e+106');
    T('774303554476944464976310906.2756072907223780702981916825924935009102144113269340555014106355954', '-2936542523893120558694770993613502855976036487473.1185405114318897552959636978670899057597113825111455591', '2.9365425238931205586955452971679798004410127983793941478021542678255941553804595834066699257938380796146014106355954e+48');
    T('-13275374194717318544585743586295562757313566313778528439681338744852144432953462009763652326380221990178967224217813656923840876936', '-1268893501255453921460255787670811403994557320916562978493176785265067911028531055.964036945884980267232', '-1.3275374194717318544585743586295562757313566313777259546180083290930684177165791198359657769059305427200474047432548589012812345880035963054115019732768e+130');
    T('335836320670736795110387481821779195576314622771385546058961565', '-6308825096439610198958861602887495693205.4798888075506048902810900464069359363711381750141774362863', '3.358363206707367951103937906468756351865135816329884335546547704798888075506048902810900464069359363711381750141774362863e+62');
    T('0.0000021372107920', '-1492049097078082888491204608304758854999.3774548350456232694485449355735093537220396664845726422284909165974459140941684162404595576799509302', '1.4920490970780828884912046083047588549993774569722564152694485449355735093537220396664845726422284909165974459140941684162404595576799509302e+39');
    T('-64732225450134517358652.7727736714979370370316688953979348962291764', '2981903640111957881620728082679423538602.35210787691735540826362359673812680870549799515376723672419104726837070300471', '-2.98190364011195794635295353281394089725512488154841529244529529249213606170493467439515376723672419104726837070300471e+39');
    T('0.00000000010852624286329189173169519346885793702035699273525590064499030', '0.00136476424465186406412883677012902', '-0.0013647641361256212008369450384338265311420629796430072647440993550097');
    T('-177975785070132616812935423102873085249499780810541192549343263252072028435998023073647418141301', '-283854219686234199819511886405040807512965424302362972904060004217971747993015757990079005629446149391437513331803368676845228.9308370137200554682048', '2.838542196862341998195118864048628317278952916855500374809571311327222482122052167975296623661940773630015153087297212587039279308370137200554682048e+125');
    T('11674840520046183.070270802775802015272346477', '171117028337514616542538412107099681879.7091235075363784803038814937481135947601822871', '-1.711170283375146165425267372665796356966388527047605764650315350167481135947601822871e+38');
    T('11685421494153882328823406', '99823703745248486246555543362896398057.69654850969729974206782841252942985600643', '-9.982370374523680082506138948056757465169654850969729974206782841252942985600643e+37');
    T('-9109482616.16307635880678540039795900931968117400633485165763268347733490749858174229181208845630928397440346561525883931665961736391868364592211620', '14097919301831317803788298625489630505329242071596241458071703375084752395181642884783601132188.8522', '-1.40979193018313178037882986254896305053292420715962414580717033750847523951816428847927106148050152763588067854003979590093196811740063348516576326834773349074985817422918120884563092839744034656152588393166596173639186836459221162e+94');
    T('-396517877066594063060677506383119536072199294985913999200669.59905258559791428967320074613407023609823115620484119798490058332639', '17256241436287752756476467398810586324749967766162129140817034689021019365.9302352418415133924725065381333905577352346319497471430577092460069', '-1.725624143628814927435353399287364700225635088569820134011202060302022003552928782743942768214570728426746079383346578815458834104260982933329e+73');
    T('-0.0000000000000006853990620674437', '-11653311815829554561363099986.3318015068509425148506115888737905', '1.16533118158295545613630999863318015068509418294515495214300905e+28');
    T('-5723499348320995600234845601896898858316706679233.56868503185838526295669696287365718201389967796395316338461191538328404398163', '461418995024299794224540170173617.270340', '-5.72349934832099606165384062619669308285687685285083902503185838526295669696287365718201389967796395316338461191538328404398163e+48');
    T('-0.0000000000000101517398894041224975316', '30047274158786647140078597868700853194123877763708264122482641575288270215565704601867490408221403474129054083514278658302219', '-3.00472741587866471400785978687008531941238777637082641224826415752882702155657046018674904082214034741290540835142786583022190000000000000101517398894041224975316e+124');
    T('-618567708703808887087390708806987', '-0.0015997812131968024699920112', '-6.185677087038088870873907088069869984002187868031975300079888e+32');
    T('34635015.52962864534284905231551787970767779289829087969225962977505217622119009888696504123088304563966631377931959000', '65036143001432807857013922660931089988524905416395272216298294489190408991210531327960173961048126960813108032096.485609144979688026094', '-6.503614300143280785701392266093108998852490541639527221629829448919040899121053132796017396104812696081307339708095598049963683897377848212029232220710170912030774037022494782377880990111303495876911695436033368622068041e+112');
    T('0.00000545779896722128', '75.894260707608627618281140481123324654999311980', '-75.89425524980966039700114048112332465499931198');
    T('-25447415.284973', '-0.0000000000000007754252238300503245755256660265406480381126460350812317568910629452911860373986555887795038942931814558679883067258919646131547862678299152', '-25447415.2849729999999992245747761699496754244743339734593519618873539649187682431089370547088139626013444112204961057068185441320116932741080353868452137321700848');
    T('-0.001075197560340105849714564474713709665034764454198356529111553747592026589852279831398339755093155322360635646225772353512853661478291013155199651968', '-0.00020617706103921965422517798875780714829530744843284313739680833048742357175207280371959115967601565', '-0.000869020499300886195489386485955902516739457005765513391714745417104603018100207027678748595417139672360635646225772353512853661478291013155199651968');
    T('-116637095502756610832663.0820790492356971286520755389123518333397635564353785857413082439598928', '1333189986026163882571196149909544777510081036404785123663838051125806079402280.39239585', '-1.3331899860261638825711961499095447775100810364047851237804751466285626902349434744748992356971286520755389123518333397635564353785857413082439598928e+78');
    T('7263206331639956734034338857142.2483005287993760580093202768989231263952047014145290402299553496171737360348099042', '930482152339662', '7.2632063316399558035521865174802483005287993760580093202768989231263952047014145290402299553496171737360348099042e+30');
    T('17084', '-9268741940429543879.25831048455993577463', '9268741940429560963.25831048455993577463');
    T('14223703.25068993203', '-337469327718493482402348015976790428942562', '3.3746932771849348240234801597679044316626525068993203e+41');
    T('3449947415518601.021', '967.0', '3449947415517634.021');
    T('-0.00000001950779962429255860674296257770624055952071967804562271969656131634150181850350309', '0.0000000008381662576798806596505469925921281919496917780384885026235741778910169613942109570101961434140020955862069084082830238300586284', '-2.03459658819724392663935095702983687514704114560841112223201354942325187798977140470101961434140020955862069084082830238300586284e-8');
    T('-150022.550348329907362699413739505634058542', '184370701561492907256959190967889134766063594155869181275951.2113720965937284915580147334609117040504470778054254305016213147502', '-1.843707015614929072569591909678891347660635941558691814259737617204265010911909717542390949702460504470778054254305016213147502e+59');
    T('0.000003620654231399775474818103408872051818273000913919007645231626869893019046908014555613658', '0.68604446001049997675748396664984795835553205960118589677575278373113193853373621258895597212465646303658933', '-0.68604083935626857698200914854643908630371378660027197776810755210426204551468930457440035846665646303658933');
    T('65368354350846125525740295379107787645781698927087650085.02206520625654961397', '-0.0000000674457908966051035778148933613021264363709209074145574586026651385058583013192568248506496786927539106989162985222622021758286164962220899099', '6.53683543508461255257402953791077876457816989270876500850220652737023405105751035778148933613021264363709209074145574586026651385058583013192568248506496786927539106989162985222622021758286164962220899099e+55');
    T('-8920208628291.3008573264554796861275279433556810380829567356987428005869325303620053400734014217983382463883243235802100794975758488782407', '-513373869275942293084285976984505491137681087793006719363115134289902581030528408559950531152647492241975261279494', '5.133738692759422930842859769845054911376810877930067193631151342899025810305284085599505311526474922330550526512026991426735445203138724720566443189619170432643012571994130674696379946599265985782016617536116756764197899205024241511217593e+113');
    T('-0.00000000000183230276994692835786255816165075667312', '-229380247132858507707379441535866.9842134679934568352180933906622411983573128385632545016159055477', '2.293802471328585077073794415358669842134679916245324481464623043786401956620818901345016159055477e+32');
    T('-287936424370618182.780178943170029610095984105232897190551066507742776730608050074096160389428747382606461428337968', '-0.0000000000000553002065739653836900325094668610744480924050098363944976572396316708026350418744470424301014068736481205320698956807538835', '-287936424370618182.7801789431699743098894101398492071580415996466683286382030402377016627321891157118038263864635209575698985931263518794679301043192461165');
    T('-22.334064568139055296413453550456558402422719815', '-5916832358056988948884.3608360136502313', '5.916832358056988948862026771445511176003586546449543441597577280185e+21');
    T('39536714308273957618181661117092495089825904732491104299979330463137078095213622807057621623.41662', '589200756584346355393503066.219095388726950444141250942', '3.9536714308273957618181661117092495089825904732491104299979330462547877338629276451664118557197524611273049555858749058e+91');
    T('3332373913416111574992795772547417643215092643096621602439954355776074197.0830941294911902023033935298448733337875606874978778273299805496488198', '-376783721571530269384664901.84367020991', '3.3323739134161115749927957725474176432150926434734053240114846251607390989267643394011902023033935298448733337875606874978778273299805496488198e+72');
    T('344530122785201375772271590504206290220784446141965110231600995362970158247376180772409197123197131294077.186098277324338477562529348577734786615', '-6800118879080455193015443336230714183574736068660393152304.8553', '3.44530122785201375772271590504206290220784446148765229110681450555985601583606894955983933191857524446382041398277324338477562529348577734786615e+104');
    T('-3163.9', '542978947164157706891911715882635312937995954746776936', '-5.429789471641577068919117158826353129379959547467800999e+53');
    T('-0.0105476924518869569128457836748478061179109848158054941752355692339310906027532526097315426003502305592037541108758447196738444856373715790727005274', '-6071446248.8516018389629874549426069280533669184418994141964930724996411053034540030731460065', '6071446248.8410541465111004980297611443785191123239884293806875783244055360695229124703927538902684573996497694407962458891241552803261555143626284209272994726');
    T('7010541707164752', '193282362072373636518312100314810747000413165206769770654226368116684704540011812177387120657825195495894912200085910730717115', '-1.93282362072373636518312100314810747000413165206769770654226368116684704540011812177387120657825195495894912193075369023552363e+125');
    T('-0.000000023007226851065128925630962810852833918316878559874', '-2227361414820256464065922780424658.1090', '2.227361414820256464065922780424658108999976992773148934871074369037189147166081683121440126e+33');
    T('-1299409497083890403177181021135821640390762146063059101862994.03242662743092991583924105899423080674085051582932994688405941727718256319443172611542', '15608748.5795516673514981732595600003601066848923630692039049676154725134079753969829680892279555729548167054550621674232927404', '-1.2994094970838904031771810211358216403907621460630591174717426119782947824280890988010593543374916332135850332349144995319306851579601773998153433755729548167054550621674232927404e+60');
    T('0.0010233218531674933882798080038538892677859219827', '0.00000000000000000045199509493736627851797824870803', '0.00102332185316749293628471306648761074980767327467');
    T('1948739934183785636586602970179911734844812119742663347882584684214752603558036589222.0466138568409029422628086739352361053669148', '1', '1.9487399341837856365866029701799117348448121197426633478825846842147526035580365892210466138568409029422628086739352361053669148e+84');
    T('-44988.17201344296154165057134154894617', '-974065266153241716711223.01990357382515241927388516358056793645871008786729345524788381378024038040395896194', '9.7406526615324171666623484789013086361076870254361463439793645871008786729345524788381378024038040395896194e+23');
    T('745292098682291838946082762537313754587386896855334901324378413687611576868785947032214747734817.29211388635775339523', '1603867082712698768798395795990225766099634680717512610131434577189024930073228282709539770294753232846533372678799230422356361956.600981654582985832103', '-1.603867082712698768798395795990225020807535998425673664048672039875270342686331427374638445916339545234956503892852198207608627139308867768225232436873e+129');
    T('631756437278888751214121898961751903012419248661678700380667.3292365506360170752783293706267811846643253079106924335724182257177006394590934297614', '2.5458577', '6.317564372788887512141218989617519030124192486616787003806647833788506360170752783293706267811846643253079106924335724182257177006394590934297614e+59');

    T('1.28687824652704e-1', '-6.698774e-7', '0.128688494530104');
    T('1.96882453e+4', '-2.36884e+4', '43376.6453');
    T('4.808718317e+1', '-8.925037868396601e+10', '89250378732.05319317');
    T('4.46523409841128641e+17', '-1.95389255e+8', '446523410036517896');
    T('1.6093613199311543706483e+9', '-3.73783629619292e+10', '38987724281.8603543706483');
    T('7.725438045407340759132811929e+20', '2.138563034e+3', '772543804540734073774.7181589');
    T('-1.87913317632093687e+17', '-1.463e+3', '-187913317632092224');
    T('-4.050502080705989793958e+18', '-8.7328619183422188241e+8', '-4050502079832703602.12377811759');
    T('-1.487e+0', '-1.966e+1', '18.173');
    T('6.1045011712250705486e-8', '-4.629661371271963538730865e-17', '6.104501175854731919871963538730865e-8');
    T('8.04812092e+3', '2.72507554517550342397e-11', '8048.1209199999727492445482449657603');
    T('5.5494e+0', '9.98100066134314965301158e+17', '-998100066134314959.751758');
    T('-7.4960224131954428972698482953e+0', '-1.796223184996831375e+7', '17962224.3539459005545571027301517047');
    T('1e+0', '1.01572540062334032198739881e+12', '-1015725400622.34032198739881');
    T('-3.075828909877630559405653e+24', '-6.20852829406318715e+15', '-3.07582890366910226534246585e+24');
    T('-7.3885363256616208126763e+18', '-4.22e-11', '-7388536325661620812.6762999999578');
    T('1.3633033e+8', '1.098216485205929e-1', '136330329.8901783514794071');
    T('-2.199452313170283112273e+20', '-5.39263091194725e+13', '-219945177390719191754.8');
    T('-8.56e+2', '0e+0', '-856');
    T('1.2877828562642718847e+5', '4.31337491e+4', '85644.53652642718847');
    T('1.4061037483985e+10', '-6.6330709216531850592433e-6', '14061037483.9850066330709216531850592433');
    T('-8.4483135e-12', '-4.113751474103595929387e-2', '0.04113751473258764579387');
    T('-1.03739443149798e+6', '4.343428478300989093e+4', '-1080828.71628098989093');
    T('1e+0', '-1.263607183e+1', '13.63607183');
    T('-2.2320214496959336032e+19', '-1.634037313014e+12', '-22320212862922023018');
    T('7.10073836719e+5', '-1e+0', '710074.836719');
    T('-2.49072186445097e+13', '9e+0', '-24907218644518.7');
    T('-2.9e-20', '-2.15971658258839e+6', '2159716.582588389999999999971');
    T('2.963920096335082831168e+18', '2.2582253282839e+6', '2963920096332824605.8397161');
    T('-4.14586e+3', '-2.9154876269e+4', '25009.016269');
    T('-1.03004879281543026e+15', '-3.78177957668791144e+3', '-1030048792811648.48042331208856');
    T('-9e+0', '4.022232761275766154688145458e+1', '-49.22232761275766154688145458');
    T('1.86836560622539908029690893881e+22', '7.662554282002068984274562e-7', '1.86836560622539908029690893873337445717997931015725438e+22');
    T('3.97689874072e+11', '-3.5744e+3', '397689877646.4');
    T('0e+0', '5.0942844226755638584371e-20', '-5.0942844226755638584371e-20');
    T('-4.12770106201259e+14', '-5.0659330520967113815e+2', '-412770106200752.40669479032886185');
    T('0e+0', '-2.4438628e+7', '24438628');
    T('4.8284508891459500047917633538e+2', '1.0650081449195084887e+2', '376.34427442264415160917633538');
    T('3.0960998932675992e+13', '1.5217314605145612561e+19', '-15217283644146679885.008');
    T('-3.731852849155459781176419989e+27', '4.33183367677e+9', '-3.73185284915545978550825366577e+27');
    T('9.08814632201479508422122537138e+1', '4.73770550569e+2', '-382.8890873488520491577877462862');
    T('3.810846909868531e+7', '2.4444336909e+2', '38108224.65531622');
    T('-2.000986521711238411058857571e+5', '-7.048740793695688315e+18', '7048740793695488216.3478288761588941142429');
    T('2.09309765830355290819000023e-3', '5e+0', '-4.99790690234169644709180999977');
    T('7.70724770820309161581e-5', '2.533e-12', '0.0000770724745490309161581');
    T('2.53375068e+6', '4.00994583e+0', '2533746.67005417');
    T('2e+0', '-6.264387065771998085033e+17', '626438706577199810.5033');
    T('-1.2407705424252222353e+19', '1.8244036019e-1', '-12407705424252222353.18244036019');
    T('-1.799721324042e+12', '-1.3447368530231020851721e+12', '-454984471018.8979148279');
    T('-5.6651404522751594041444107472e+20', '-7.535689111154473052048e+13', '-566513969870624828869.71055424');
    T('3.769112233878e+8', '-1e-10', '376911223.3878000001');
    T('-1.839105258587571e+0', '1.20177991131103601214234e+23', '-1.20177991131103601214235839105258587571e+23');
    T('1.082600317037849190993749e+19', '1.4938307025455867644875688e-4', '10826003170378491909.93734061692974544132355124312');
    T('1.12e+1', '3e+0', '8.2');
    T('6.526872092482495666296825e+9', '4.8645966080716540006e-18', '6526872092.4824956662968249951354033919283459994');
    T('1.339857855971657033752952445e+27', '-4e+0', '1.339857855971657033752952449e+27');
    T('-2.419044558390159646056e-13', '-1.3079097339e+2', '130.7909733899997580955441609840353944');
    T('-7.1705677513767136563e-17', '-6.3081315e-10', '6.30813078294322486232863437e-10');
    T('-2.124781842799053e-2', '5.5030007661157553e+16', '-55030007661157553.02124781842799053');
    T('1.55629535242e+4', '-5.1428769625e-1', '15563.46781189625');
    T('1.01504354636013042e+15', '2.63791456408046734975e+15', '-1622871017720336.92975');
    T('-9.4294e+4', '1.8866986e+7', '-18961280');
    T('1.71775e+1', '-9.97178848809e+7', '99717902.0584');
    T('-6.51108703897071e-2', '2.0481127e+7', '-20481127.0651108703897071');
    T('2.5575417677e+10', '-9.875210228621188311e+18', '9875210254196605988');
    T('-1.515600255e+1', '1.78e+0', '-16.93600255');
    T('-4.934811484526990303505e+15', '3.903941767e-17', '-4934811484526990.30350500000000003903941767');
    T('2.660689025e+9', '1.210751379395e-18', '2660689024.999999999999999998789248620605');
    T('-1.3039e+3', '3.693e+0', '-1307.593');
    T('-8.150211374792494231850488e+10', '-2.4154e+4', '-81502089593.92494231850488');
    T('-7.04276965355095765787e+2', '5.12083556215696675027418e-3', '-704.28208619065792275375027418');
    T('4.560203906e+6', '6.662833107173476e+6', '-2102629.201173476');
    T('1.18219623035280292278e+16', '1.35881201163097673230180989e+16', '-1766157812781738.0952180989');
    T('1.01145921064769030827e+12', '1.0162177943917298766993291463e+4', '1011459200485.512364352701233006708537');
    T('4.478831496443e+12', '-3e+0', '4478831496446');
    T('-4.8033663475504622462769283e+6', '2.400585121791519891706e+17', '-240058512183955355.5181504622462769283');
    T('-1.197167677719e+4', '-9.5568526467142828986e+3', '-2414.8241304757171014');
    T('1.0454e+4', '-7.5783e-13', '10454.00000000000075783');
    T('1.497160982390283668405594719e+2', '3.01769088202242341433267860826e+29', '-3.017690882022423414332678606762839017609716331594405281e+29');
    T('-6.8836180118422e+2', '-4.8579187557539e+5', '485103.51377420578');
    T('-2.2015868446618e+0', '1.112092601534267192772461445e-1', '-2.3127961048152267192772461445');
    T('-1.72703850655851149109833572e+24', '0e+0', '-1.72703850655851149109833572e+24');
    T('-1.722156947575281448505e+21', '7.2324954621329520621898740974e+7', '-1.722156947575353773459621329520621898740974e+21');
    T('1.5403804214964843217980824e+9', '1.2891e+1', '1540380408.6054843217980824');
    T('-5.945326523e+4', '4.669485020980197203323689e+4', '-106148.11543980197203323689');
    T('5.55266637914837652416872269693e+21', '1.471120167304946e+14', '5.55266623203635979367412269693e+21');
    T('3.73778817609658865295139e+1', '-9.87441939032866e+6', '9874456.7682104209658865295139');
    T('5.1012538663005271824613969722e+22', '-1.50156e+0', '5.1012538663005271824615471282e+22');
    T('-2.058e+2', '2.8646830031089679744e+17', '-286468300310897003.24');
    T('-1.632e-16', '9.9938831828547468539e+19', '-99938831828547468539.0000000000000001632');
    T('3.74855475772666e+1', '4.636917906044210284685212e+14', '-463691790604383.5429209439334');
    T('-6.125e+1', '-1.01695009327434527205612623e+7', '10169439.6827434527205612623');
    T('-3.713055373107057485e+8', '-1.579459775785685628895042e+12', '1579088470248.374923146542');
    T('-4.973637071239865e+8', '9.700610204149955956e+18', '-9700610204647319663.1239865');
    T('0e+0', '-4.4905032171787016875848189267e+20', '449050321717870168758.48189267');
    T('-1.4666228937e+5', '4.268680884193908e-17', '-146662.28937000000000004268680884193908');
    T('5.70462240242e+5', '1.140990056118581205e+15', '-1140990055548118.964758');
    T('3.4806661e-11', '8.29856853e+2', '-829.856852999965193339');
    T('-9.086912118595249999201e+21', '3.898423213142569952135e+9', '-9.086912118599148422414142569952135e+21');
    T('-1.61819145378513602550953401e+8', '2.6469734905241672957e+10', '-26631554050.620186559550953401');
    T('1.8497588e+7', '3.133737672348e-9', '18497587.999999996866262327652');
    T('-6.21844625e+6', '6.448980631426448042e-2', '-6218446.31448980631426448042');
    T('2.0553505882011009665e+19', '6.92603e+4', '20553505882010940404.7');
    T('1.92248242840598741949091e+13', '1.339870445733828207038e+20', '-133987025348558536643.9258050909');
    T('4.7736392e-14', '-8.243397452128704948729866e+11', '824339745212.870494872986647736392');
    T('-1.7e+1', '-7.923074334739511792397e-16', '-16.9999999999999992076925665260488207603');
    T('4.9635811440987476026476e-17', '-1.466e-1', '0.146600000000000049635811440987476026476');
    T('2.512230419e+4', '2.5397165598196246028285880532e-1', '25122.05021834401803753971714119468');
    T('-1.11299e+2', '1.11035558611694808056e-18', '-111.29900000000000000111035558611694808056');
    T('2.026822870015760981220174257e-5', '-2.047169266002425550414624e-19', '0.0000202682287001578145291283428125550414624');
    T('-4.0648049768873318277e+6', '-3e+0', '-4064801.9768873318277');
    T('-2.588724223892656991e-16', '3.0098072257229e-6', '-0.0000030098072259817724223892656991');
    T('1.56373140247e+4', '-3.91671625230328e+8', '391687262.5443527');
    T('1.00341736976978e-7', '2.304833813547170392924601625e+22', '-2.3048338135471703929246016249899658263023022e+22');
    T('8.35221698180089026e-19', '-9.32578907000664374e+2', '932.578907000664374000835221698180089026');
    T('-1.08764251456624192e+8', '5.0280989399438845291162488994e+9', '-5136863191.4005087211162488994');
    T('2.46130217148349070169911254e+26', '-1.5624487714497915185829268064e-20', '2.46130217148349070169911254000000000000000000015624487714497915185829268064e+26');
    T('1.01340342215606347222e+6', '-9.22914210865038825420171003e-5', '1013403.4222483548933065038825420171003');
    T('-3.4384e+4', '2.364774267367323611e-9', '-34384.000000002364774267367323611');
    T('2.754440021463e+8', '-8.87772958126791e+2', '275444889.919258126791');
    T('7.866841036008389399178e+3', '2.250087110372669759e+8', '-225000844.196230967510600822');
    T('-1.048e+3', '2e+0', '-1050');
    T('1.51258199632721102661464e-18', '1.37889e+3', '-1378.88999999999999999848741800367278897338536');
    T('-1.247e+3', '-4.576980640479189083e+3', '3329.980640479189083');
    T('2.01135499e+5', '-2e+0', '201137.499');
    T('7.03046195059871843437588e+23', '-2.434388173985239e+12', '7.03046195062306231611573239e+23');
    T('-3.54e+2', '-2.0062114120746252479542e+4', '19708.114120746252479542');
    T('-1.703025e-18', '-3.4292979189222437121973e+21', '3.429297918922243712197299999999999999998296975e+21');
    T('4.87547962612395e+6', '-3.6652187394898226692963e+9', '3670094219.1159466192963');
    T('-2.0143114813870112462786921e-17', '3.767411e+0', '-3.767411000000000020143114813870112462786921');
    T('3.26586277019e+6', '-2.38239842063e+11', '238243107925.77019');
    T('6.097001595334722255599827169e+1', '-1.27e+1', '73.67001595334722255599827169');
    T('2.5862383503225371e+9', '-5.1214922610895542e-1', '2586238350.83468632610895542');
    T('2.231e+1', '2.117312218972605e+5', '-211708.9118972605');
    T('3.80021425650817317250555e+13', '-1.93695171769468e+11', '38195837736851.1997250555');
    T('-7.810946417020212022651917386e+18', '2.290482727614761942741662e+7', '-7810946417043116849.92806500542741662');
    T('3e+0', '-2.2205773655663464012097e-6', '3.0000022205773655663464012097');
    T('3.64246615622912558467016935e+26', '0e+0', '3.64246615622912558467016935e+26');
    T('3.77e+3', '-1e+0', '3771');
    T('-3.58605723986826101196397318e+26', '-1.485e+2', '-3.586057239868261011963971695e+26');
    T('-3e+0', '1.40899579022e+11', '-140899579025');
    T('-1.45909430175366456235718665e+19', '-3.4261617579351795158e+18', '-11164781259601466107.7718665');
    T('-2.348189590117898492617e+8', '2.878559430244312e+14', '-287856177843390.2117898492617');
    T('-9.7e+0', '2.690189646188e+1', '-36.60189646188');
    T('1.36455632e-1', '2.93006388540946905134174872e+13', '-29300638854094.5540577854872');
    T('4.048592176739e-20', '-6.005127e+2', '600.51270000000000000004048592176739');
    T('-2.566085007096806352945241e+6', '1.0191e-13', '-2566085.007096806353047151');
    T('-3.31696e+2', '6.3e+1', '-394.696');
    T('4.1079014482683004e+8', '6.4446707e+7', '346343437.82683004');
    T('-1.577640941527783e+9', '1.27337662718184094697659300437e-5', '-1577640941.5277957337662718184094697659300437');
    T('2.7420244617713377e+16', '-8.623235881e-16', '27420244617713377.0000000000000008623235881');
    T('2.13592440096179687295287e+23', '-5.24330594e+5', '2.13592440096179687819617594e+23');
    T('4.5966088142543734664499633e-4', '1.9622997228567688328e+13', '-19622997228567.68786833911857456265335500367');
    T('-2.003763953536177099e+9', '9.92e-20', '-2003763953.5361770990000000000992');
    T('1.549844513583e+5', '2.12886113900670758986381e+14', '-212886113745686.307628081');
    T('-5.035786855374e+12', '1.5195491474751e+13', '-20231278330125');
    T('-8.837922375208113241898942499e+23', '4.91312103789241e-17', '-8.837922375208113241898942499000000000000491312103789241e+23');
    T('5.28121115884713e+6', '9.35111e+2', '5280276.04784713');
    T('7.0534112016532323024600701578e+21', '2.158e+0', '7.0534112016532323024579121578e+21');
    T('-1.22304768148016734601138489e+20', '9.08133282381327051184228774e+26', '-9.08133404686095199200963375138489e+26');
    T('1.899e-6', '2.153419180764601e+10', '-21534191807.646008101');
    T('-5.650839046342472094e+5', '5.0808365974271307182e+8', '-508648743.6473473190294');
    T('-9.4e+1', '-2.6927013e+0', '-91.3072987');
    T('-6.571708892881e+12', '-8.545294322652905079217675482e+19', '85452936654820157911.17675482');
    T('-1.76782224366755e+10', '-4.13948649746403e+10', '23716642537.9648');
    T('-5.54107604714186288144437e+7', '-4.96270174556976430531733658389e+25', '4.96270174556976429977626053674813711855563e+25');
    T('-9.978651982580698767e+16', '-6.954508412732601e+0', '-99786519825806980.715491587267399');
    T('1.8693160127470428298008436e-4', '1.547218677067e+11', '-154721867706.69981306839872529571701991564');
    T('-1.38945385393184793207e+20', '1.37289934375618017208481128e+20', '-276235319768802810415.481128');
    T('-2.03189937609562003767063412e-3', '1.50517775495e+11', '-150517775495.00203189937609562003767063412');
    T('5.37526674392168e+11', '6.281346735299e+9', '531245327656.869');
    T('-3.1901487e+4', '-2.47278901189761333644390687e-17', '-31901.4869999999999999752721098810238666355609313');
    T('1.911067e+3', '1.63809954792658e+10', '-16380993568.1988');
    T('-6.262e+0', '-4.732755159790421866759181099e+12', '4732755159784.159866759181099');
    T('1.9922595737406200707054959707e+19', '1.5757171e+0', '19922595737406200705.479242607');
    T('-4.20463196e+6', '-5.39927e+4', '-4150639.26');
    T('5.698457e+5', '0e+0', '569845.7');
    T('-2.3755793e+6', '-3.03634360026652225878481594e-5', '-2375579.2999696365639973347774121518406');
    T('-1.2540612364355e+13', '-9.7056758273427691e-14', '-12540612364354.999999999999902943241726572309');
    T('7e+0', '-2.1217135735760987306431493773e+10', '21217135742.760987306431493773');
    T('-1.878187232e+1', '1e+0', '-19.78187232');
    T('4.083686618644527491e+14', '-1.0207e-8', '408368661864452.749100010207');
    T('1.936437398920283252217215e+7', '-1.426953028e+1', '19364388.25873311252217215');
    T('-5.36e+1', '-3.270440850977315671331954027e+27', '3.2704408509773156713319539734e+27');
    T('-5e+0', '3.9771e+4', '-39776');
    T('-1.41165283690117849025039415809e+15', '9.2653033582808e+1', '-1411652836901271.14328397696609');
    T('1e+0', '5.853518072e+8', '-585351806.2');
    T('9.28866302211484528e-16', '3.42924e+3', '-3429.239999999999999071133697788515472');
    T('1.5141918664046020576449895893e+12', '-1.61e+2', '1514191866565.6020576449895893');
    T('1.38289322288900523715e+1', '7.080258718068754274828878e+11', '-708025871793.0464952539977476285');
    T('4.021120497e+9', '-3.58323864641440176097e+12', '3587259766911.40176097');
    T('3.9109896247281e+5', '-7.7768225519466054995224860986e+28', '7.776822551946605499522525208496247281e+28');
    T('5.52e-6', '1.34849494404812613034319555618e+29', '-1.3484949440481261303431955561799999448e+29');
    T('2.0537e-20', '-6.4758411334071e+7', '64758411.334071000000000000020537');
    T('-3.1417778482733028111928e-15', '3.3188180849003762977741480641e+28', '-3.31881808490037629777414806410000000000000031417778482733028111928e+28');
    T('-8.744874784408647631e+0', '2.135742081446470378868598e+24', '-2.135742081446470378868606744874784408647631e+24');
    T('4.84800483e+2', '3.91e+1', '445.700483');
    T('-3.7016741e-12', '-4.1497617308e-1', '0.4149761730762983259');
    T('-6e+0', '-6.8805897379338e+13', '68805897379332');
    T('6.250477943675354e+3', '-2e+0', '6252.477943675354');
    T('6.3749589779023191251153515e+1', '-5.20173903691687372543e+3', '5265.488626695896916681153515');
    T('-6.6824849945959e+13', '3.901733743171141119056850874e+27', '-3.901733743171207943906796833e+27');
    T('-2.0817240249744817154064994e+21', '5.985e+1', '-2.0817240249744817154663494e+21');
    T('-2.9323306082438653e-1', '-4.051822836352475688965270021e+15', '4051822836352475.39573220919661347');
    T('1.41083866e+7', '1.222938e+6', '12885448.6');
    T('-2.685513393949e+6', '3.6665913622890279695898073188e+8', '-369344649.62285179695898073188');
    T('2.1722453e+3', '5.27433180346789078179e+12', '-5274331801295.64548179');
    T('1.82945984e+1', '2.84080338471485422824162e+7', '-28408015.5525501422824162');
    T('3.8282192270741792760092388e-2', '-1.034837658465e+2', '103.522048038770741792760092388');
    T('5.90588e+0', '3.041191e+2', '-298.21322');
    T('-3e+0', '-2.22554559e+2', '219.554559');
    T('-1.58686397805828293e-1', '-2.50678032123148961445e+20', '250678032123148961444.841313602194171707');
    T('6.9731426202279009749874e+16', '-6.8033025527322349e+3', '69731426202285813.0524267322349');
    T('-3.5288065255382459e+8', '4.319711887733e-5', '-352880652.55386778711887733');
    T('-3.861521213463773165e+10', '2.1786e+4', '-38615233920.63773165');
    T('1.712e+2', '1.34100827765345675353094e-1', '171.065899172234654324646906');
    T('5.78089955830190857610305894e+1', '6.36549849502761e+4', '-63597.1759546930809142389694106');
    T('-2.044680654381125e+11', '-1.96641728023668398742736257646e+29', '1.966417280236683985382681922078875e+29');
    T('-2.9076e+1', '-2.7871249e+7', '27871219.924');
    T('1.0798349049680377701191303e+1', '-1.28969281e+7', '12896938.898349049680377701191303');
    T('-5.84e+0', '4.096887717702998692e-5', '-5.84004096887717702998692');
    T('-4.444975675033721e-1', '-3.1452254562577476e+0', '2.7007278887543755');
    T('1.183597e-16', '-8.2003e+2', '820.0300000000000001183597');
    T('4.08955925538021395e-9', '0e+0', '4.08955925538021395e-9');
    T('-8.518844894710593442e+7', '8.021e+1', '-85188529.15710593442');
    T('2.68232609980658991729575880176e+0', '0e+0', '2.68232609980658991729575880176');
    T('4.4477770435926438470217122e+8', '5.12429681263e+7', '393534736.23296438470217122');
    T('1.6841448787702267666418174e-17', '-1.2182119751289e+1', '12.182119751289000016841448787702267666418174');
    T('-2.2729761e+5', '-4.46452153e+0', '-227293.14547847');
    T('1.0246412734413008e-11', '4.66232758476e-16', '1.0245946501654532e-11');
    T('-2.68821719678017623378e-19', '3.39024822961193e-2', '-0.033902482296119300268821719678017623378');
    T('1.853752578428008999076e+6', '2.3969844952707427e-18', '1853752.5784280089990759976030155047292573');
    T('7.61e+2', '2.4131847562072898e+13', '-24131847561311.898');
    T('3.109821644875197306e-16', '1.606639839e-19', '3.108215005036197306e-16');
    T('-6.1011e+3', '3.906986073489232797471e+9', '-3906992174.589232797471');
    T('9e+0', '6.09778743145589590112609339e-2', '8.9390221256854410409887390661');
    T('-2.33021e-12', '-5.805032816747e+12', '5805032816746.99999999999766979');
    T('5.785870917e+2', '2.8947e+4', '-28368.4129083');
    T('4.9138011093118555140264508438e+1', '-2.688188222207754803e-17', '49.13801109311855516714639066007754803');
    T('6.86328788e-20', '-1.0903e-4', '0.0001090300000000000686328788');
    T('-1.5222667065790299e-10', '-5.30045567313202177385949422e+8', '530045567.31320217723372275134209701');
    T('2.102091e+0', '-2.69376496330958751240504762e+26', '2.69376496330958751240504764102091e+26');
    T('3.944367498734539718e+1', '-1.46339226240020591588421e+15', '1463392262400245.35955919734539718');
    T('5.7233801647184e+10', '-6.5e+0', '57233801653.684');
    T('-3.803540018773522987078e-15', '-4.8838714337004857737078494e+4', '48838.714337004857733274953981226477012922');
    T('2.4743428039e+9', '-7.150348863482e+2', '2474343518.9348863482');
    T('-4.72e+0', '-2.74711687758923547052379498e+0', '-1.97288312241076452947620502');
    T('-7.65e+1', '-1.3497715e+5', '134900.65');
    T('-1.302284775e+7', '-5.259e-14', '-13022847.74999999999994741');
    T('-7.9577e+4', '-2e+0', '-79575');
    T('6.36947759756484191820773706e+10', '-5.37489619395631884896608e+10', '117443737915.2116076717381706');
    T('2.8137158805278468362788e-15', '1.45644e+3', '-1456.4399999999999971862841194721531637212');
    T('2.214e+0', '-7.164325571694427826653861996e+15', '7164325571694430.040653861996');
    T('-3.80573224245e+11', '6.30327705991065191e+18', '-6303277440483876155');
    T('-1.3156481902267089e+3', '-3.987e+3', '2671.3518097732911');
    T('-2.473947689670568922810605185e+3', '-5.9864037611282e+7', '59861563.663592329431077189394815');
    T('1.1001731e+7', '-9.2848140928651653621884e+0', '11001740.2848140928651653621884');
    T('-9.8044989570620753370567373e+7', '-1.17e+1', '-98044977.870620753370567373');
    T('-3.7000443862e+1', '-1.9192804071866287509957721855e+12', '1919280407149.6283071337721855');
    T('-9.5557565112682694e+1', '-1.139475090734929511787e+21', '1.139475090734929511691442434887317306e+21');
    T('-9.36659e+4', '-2.2488368141052655e-10', '-93665.89999999977511631858947345');
    T('0e+0', '-1.96659668503522757503562e+15', '1966596685035227.57503562');
    T('-2.46588281577379345e+9', '-1.52735567584678502576392632012e+9', '-938527139.92700842423607367988');
    T('1.116693556411266004e+13', '8.181470064283092308441666e+0', '11166935564104.478569935716907691558334');
    T('1.42867858509369e+11', '6.36335708310839028e+14', '-636192840452329.659');
    T('9.46345944e-20', '4.09009002345797e+0', '-4.0900900234579699999053654056');
    T('0e+0', '-5.752093e-16', '5.752093e-16');
    T('0e+0', '-7e+0', '7');
    T('6.671e+1', '-5.22802396973925929e+6', '5228090.67973925929');
    T('3.80635453384970742358475002e+21', '1.6514766472479832640394018164e+1', '3.806354533849707423568235253527520167359605981836e+21');
    T('3.909e+1', '-3.0330657786e-7', '39.09000030330657786');
    T('-4.177266320643e+2', '4.36481347232e+5', '-436899.0738640643');
    T('-1.93e-3', '4.44897e+4', '-44489.70193');
    T('-8.493203254439607467238e+13', '3.780960689802534388741349e+4', '-84932032582205.68157040534388741349');
    T('-8.62e+2', '-1.21242637514496e+9', '1212425513.14496');
    T('-2.034125439802066e+1', '1.041223699e+0', '-21.38247809702066');
    T('-1.8756615731747938145167676e+9', '-5.9079882582580697796179e+17', '590798823950145404.7869961854832324');
    T('-7.68665407063362e-12', '2.68347734852e+3', '-2683.47734852000768665407063362');
    T('-2.5e-9', '1.0094051112011317e+2', '-100.94051112261317');
    T('-2.273663293572256347483e+16', '-2.0246e-14', '-22736632935722563.474829999999979754');
    T('-4.8958e+1', '-1.18656711e+8', '118656662.042');
    T('-1.1921912260356353311433328e+25', '-6.7981248360658545755979997e+4', '-1.1921912260356353311365346751639341454244020003e+25');
    T('-6.849149772761089815191e+18', '5.24560558802513e+14', '-6849674333319892328.191');
    T('2.99306425443292501e+17', '2.8e+1', '299306425443292473');
    T('2.5471860004e+9', '-4.98536550694988897317422126e-12', '2547186000.40000000000498536550694988897317422126');
    T('-1e+0', '2.54e+1', '-26.4');
    T('1.814452890048444838634591e+17', '3.50190331311817e+6', '181445289001342580.55034093');
    T('-3.740775006407299187613971e+21', '-6.411975563570869469e+1', '-3.74077500640729918754985124436429130531e+21');
    T('7.68508730852e+3', '-6e+0', '7691.08730852');
    T('-4.97430855e+5', '-1.74704104775e+3', '-495683.81395225');
    T('-2.25986972303734e-18', '-3.563603e+5', '356360.29999999999999999774013027696266');
    T('7.300705399e+0', '-2.68400174895358537123619e+2', '275.700880294358537123619');
    T('-7.4583833263680087e+11', '9.53468014076291e+9', '-755373012777.56378');
    T('8.9104e+4', '-6.4430091336482802107567729773e+3', '95547.0091336482802107567729773');
    T('2.34280052861242466e-1', '0e+0', '0.234280052861242466');
    T('7.269051098e+8', '-3.4459676199e+4', '726939569.476199');
    T('-3.211997876e+3', '-9.38450118638793709019229e+14', '938450118635581.711143229');
    T('7.7969891679461230711262e+22', '7.69e+0', '7.796989167946123071125431e+22');
    T('1.097e+0', '-4.6249580639699552947992543e-20', '1.097000000000000000046249580639699552947992543');
    T('-6e+0', '-1.456695321775709567231919891e+20', '145669532177570956717.1919891');
    T('3.8834077979e+6', '-1.95675144721034531902e-6', '3883407.79790195675144721034531902');
    T('-1.6658487619383064e+2', '3.100133628146717733340869e+7', '-31001502.86634337116404869');
    T('-1.4005754211993309661536063e-14', '-8.84275e+4', '88427.499999999999985994245788006690338463937');
    T('-1.3403661333569183042528493e+25', '-2.16701427914547231e+16', '-1.34036613118990402510737699e+25');
    T('2.6104592333650733834147661141e+8', '5.5581589300093249719048e+21', '-5.55815893000906392598146349266165852338859e+21');
    T('5.315518589225007574739350338e-12', '5.299157680175e-14', '5.262527012423257574739350338e-12');
    T('-2.17607896075487810392093064e+0', '-4.372277792215178893233e+16', '43722777922151786.75625103924512189607906936');
    T('-1.1360680787648690671166707184e+0', '6.64626014583885222275e+5', '-664627.1506519639871440671166707184');
    T('-4.322698247e+9', '7.5666e+0', '-4322698254.5666');
    T('7.329582e+2', '1.1e+0', '731.8582');
    T('4.70556124361e+0', '1.9128794148091e+7', '-19128789.44252975639');
    T('-1.664718e-15', '-2e+0', '1.999999999999998335282');
    T('-2.3441520698096713166992980212e+28', '-2.45124495841653227069e-11', '-2.34415206980967131669929802119999999999754875504158346772931e+28');
    T('1.08447643843924835e+14', '4.629055171533282139112789e-9', '108447643843924.834999995370944828466717860887211');
    T('-2.56258604619015315966e+3', '-2.033704783173576458375994e+19', '20337047831735762021.17389380984684034');
    T('-5.86438097577803988896572e-1', '3.9380601967502959955e+11', '-393806019675.616037647577803988896572');
    T('0e+0', '3.50862e+6', '-3508620');
    T('5.16326e-10', '-4.653750284661785074950340187e-4', '0.0004653755447921785074950340187');
    T('2.448434809857e+3', '9.861166640454948250573581e+13', '-98611666402101.04769587881');
    T('-1.377207013806422897185e+8', '2.93868995004852419030725393e+26', '-2.938689950048524191684460943806422897185e+26');
    T('-1.21087028473668e-15', '-1.79392219447388957457557920332e+16', '17939221944738895.74575579203319878912971526332');
    T('7.1194812461460298471228e+22', '-3.11548104971e+6', '7.119481246146030158670904971e+22');
    T('-7.9973363706063079e+1', '-4.18e+1', '-38.173363706063079');
    T('-3.4658e+3', '-1.50428792912533108828785e+14', '150428792909067.308828785');
    T('-5.43e+1', '1.08e+1', '-65.1');
    T('-3.958612981624798584713e+21', '1.905323655927183e+11', '-3.9586129818153309503057183e+21');
    T('-6.05e+0', '7.113109139228e+12', '-7113109139234.05');
    T('-4.668265354520068e+15', '-3.4465582208309591e-8', '-4668265354520067.999999965534417791690409');
    T('-2.771846365e+5', '-2.5244589e+4', '-251940.0475');
    T('1.11530656435193641253e+18', '-3e+0', '1115306564351936415.53');
    T('-1.05624e+6', '-3.9048304233341053516492008879e-20', '-1056239.999999999999999999960951695766658946483507991121');
    T('1.0127e+0', '-3.135359936645349731157475112e-13', '1.0127000000003135359936645349731157475112');
    T('2.2181732450662751e+7', '4.985e+2', '22181233.950662751');
    T('-1.458218486287662421604e+16', '8.6772071348685837314014362447e+3', '-14582184862885301.4231748685837314014362447');
    T('-5.2949825314874e+13', '-1.34587e+5', '-52949825180287');
    T('2.7853732e+3', '-2.8992603343141478e+3', '5684.6335343141478');
    T('1.0093686351168e-16', '2e+0', '-1.99999999999999989906313648832');
    T('-2.69782596422e+5', '9.281e+1', '-269875.406422');
    T('-8.4892243734618044988822527303e+28', '-1.093399419157518929e+15', '-8.4892243734616951589403369784071e+28');
    T('-2e+0', '-9.8226109e+7', '98226107');
    T('-7.91355730171712559764046812857e+29', '3.738879817007584795015696e-16', '-7.913557301717125597640468128570000000000000003738879817007584795015696e+29');
    T('-7.20113991704e+6', '3.766628446721078627562e+2', '-7201516.5798846721078627562');
    T('8.09195711497727858010623376e+2', '-2.299005549455463322634739396e-2', '809.21870155322241264384972339396');
    T('7.1e-16', '1.34172774450117955431064868e+26', '-1.3417277445011795543106486799999999999999929e+26');
    T('2.479213128129e+12', '-4.3266851790425848916602875204e+19', '43266854269638977045.602875204');
    T('-5.167953131767e-17', '1.089636317069144688e+0', '-1.08963631706914473967953131767');
    T('-1.043599533780483997e+8', '-8.4604453857096e+13', '84604349497142.6219516003');
    T('-9.86113e-3', '5.25012650031541380643072e+14', '-525012650031541.390504202');
    T('-4.607800742854309e-16', '6.629214050478863773e+7', '-66292140.5047886377300004607800742854309');
    T('-7.0726129234663936285e+0', '5.1813062685268139626360921992e+0', '-12.2539191919932075911360921992');
    T('-9.59114856809630214548e-16', '8.19552038958943023267524e+23', '-8.19552038958943023267524000000000000000959114856809630214548e+23');
    T('-6.1160705538571976049e+19', '-1.5e-14', '-61160705538571976048.999999999999985');
    T('-5.20340090331936061e-19', '-6.8907407213853599371e+12', '6890740721385.359937099999999999479659909668063939');
    T('3.266e-6', '-1.178352424764155283425353405e+3', '1178.352428030155283425353405');
    T('-4.88224490036702e+6', '2.067e+2', '-4882451.60036702');
    T('1.01802113642363671778919360155e+10', '-6.088e-5', '10180211364.2364280578919360155');
    T('3.210380386593524862226e+16', '2.8740812364241127382e+17', '-255304319776476025.19774');
    T('-1.465851233021161325672097e+24', '1.2772576319289909484700907499e+28', '-1.2774042170522930646026579596e+28');
    T('-3.7269879590593e-13', '-1.89917687e+2', '189.91768699999962730120409407');
    T('-3.54751669e-4', '1.507681824540198379007387131e-20', '-0.00035475166900000001507681824540198379007387131');
    T('8.6e+0', '-1.718010447132363e+14', '171801044713244.9');
    T('-1.523861475830427411616557095e-1', '-1.4263256386185266172813e-16', '-0.15238614758304259852909184764733827187');
    T('-3.0492e+4', '-3.19247471953735496043e+16', '31924747195343057.6043');
    T('1.64e+1', '-2.300091101743972533e+2', '246.4091101743972533');
    T('-8.4166616e-14', '-1.08365043381515665671178336e+3', '1083.65043381515657254516736');
    T('2.834233434423066e-3', '1.5147922394810634456e+19', '-15147922394810634455.997165766565576934');
    T('9.809683833546843226149927904e+7', '-7.04744e+1', '98096908.80986843226149927904');
    T('-2.78659733581828216291e-6', '3e+0', '-3.00000278659733581828216291');
    T('1.88550609258935040490277736481e+21', '-2.0139e+4', '1.88550609258935042504177736481e+21');
    T('-2.961956555914393326e+9', '-1.736319970053323220680261e+17', '173631994043375766.153632774');
    T('1.1511839112e+10', '6.1439877305045904627491e+21', '-6.1439877304930786236371e+21');
    T('-2.6e+0', '-2.49048e-15', '-2.59999999999999750952');
    T('-5.48e+0', '2.4e+0', '-7.88');
    T('-8.281668732400133e-7', '3.4802271248601324e+13', '-34802271248601.3240008281668732400133');
    T('5.4692910497907010278816248427e-19', '-3.995182600549255e+6', '3995182.60054925500000000054692910497907010278816248427');
    T('-4e+0', '-2.6796842621783387684139e-8', '-3.999999973203157378216612315861');
    T('-1.2503809566311e-14', '-4.194893531803037022861e+13', '41948935318030.370228609999987496190433689');
    T('8.20101101549315479737983e+23', '-6.21887189572000039915995509e+29', '6.21888009673101589231475246983e+29');
    T('-8.49630557904938058714202e+2', '-9.656431846752105646935065749e+12', '9656431845902.475089030127690285798');
    T('4.709804775734875967339134e-13', '-5.238417348338395e+4', '52384.1734833839504709804775734875967339134');
    T('3.215642768985439973e+18', '-6.47876e+2', '3215642768985440620.876');
    T('1.057e+3', '5.4e-14', '1056.999999999999946');
    T('-1.27406478713275073021e+15', '-4.95098112170149342507601184272e+29', '4.9509811217014806844281405152126979e+29');
    T('-2.47741050648e-6', '2.867648067351452617036e+9', '-2867648067.35145509444650648');
    T('9.683330132790537859047707237e+27', '1.4e-13', '9.68333013279053785904770723699999999999986e+27');
    T('1.8191377569793603e+1', '-2.22262592888890757e+16', '22226259288889093.891377569793603');
    T('-2.647537436000328921472e+19', '1.73756170770793102269901e-4', '-26475374360003289214.720173756170770793102269901');
    T('3.9166725e+1', '-3e+0', '42.166725');
    T('-6.896e+2', '-2.4697e+3', '1780.1');
    T('3.049991467813178e-17', '9.921359715790652629887e-6', '-0.00000992135971576015271520886822');
    T('0e+0', '-1.014702489558798e+11', '101470248955.8798');
    T('3.63e-1', '-9.504114905372600685531012901e+7', '95041149.41672600685531012901');
    T('1.43e+0', '1.032179739e+4', '-10320.36739');
    T('6.002271250385225215266404e-20', '1.198673608430859644679301e+24', '-1.19867360843085964467930099999999999999999993997728749614774784733596e+24');
    T('-8.165072327377418555e+1', '1.15946801209912e+14', '-115946801209993.65072327377418555');
    T('-2.855963912187238e+15', '-2.34813677339491877e-20', '-2855963912187237.9999999999999999999765186322660508123');
    T('4.7139819451281756523160792425e+8', '1.711815969456703e+12', '-1711344571262.19018243476839207575');
    T('3.51917952e+0', '-1.9090638600782859881987727e-9', '3.5191795219090638600782859881987727');
    T('-1.101278668075627123510003e+3', '3.23115293065e-14', '-1101.2786680756271558215323065');
    T('3.1909723769e+4', '-1.7298288176e+4', '49208.011945');
    T('-9.30120609629212846377709e-16', '8.055569687891446e+0', '-8.055569687891446930120609629212846377709');
    T('-7.474064e+1', '-4.271718880894195123e-6', '-74.740635728281119105804877');
    T('-1.621114105166716832660982e+9', '-7.814354217e+7', '-1542970562.996716832660982');
    T('-3.338943287283e+12', '1.245463771082e+7', '-3338955741920.71082');
    T('2.1732853867827159984017e-12', '-5.54912539966658289e+7', '55491253.9966658289021732853867827159984017');
    T('-2.878598492994e+9', '-5.57847e+3', '-2878592914.524');
    T('-1.12e+3', '3.5382811477716371e-9', '-1120.0000000035382811477716371');
    T('3.6386011498e-2', '-5.6384769154648814041207571e-18', '0.0363860114980000056384769154648814041207571');
    T('-1.890178944623951306311749095e+16', '-1.01868789996777383e+16', '-8714910446561774.76311749095');
    T('8.76907e+2', '-3.0262476312e+6', '3027124.5382');
    T('-4.2809356122824992293797e-6', '-9.5432581914885e-3', '0.0095389772558762175007706203');
    T('2.613634e+6', '0e+0', '2613634');
    T('-2.31839833569e-6', '1.324426936e-19', '-0.0000023183983356901324426936');
    T('-1.10399938484009e+1', '1.85186916549461617452156387962e+2', '-196.226910397862517452156387962');
    T('-2.75009407098166054593418e+23', '-4.97e+1', '-2.750094070981660545933683e+23');
    T('6.61314773892e+8', '1.16976318594169000300606933902e+3', '661313604.12881405830999699393066098');
    T('-1.65790748026635119525e+19', '1.13625143013338763e-18', '-16579074802663511952.50000000000000000113625143013338763');
    T('1.63212008e+0', '-6.5032932718736077532455e+5', '650330.95930744077532455');
    T('-8.590257420137321035e+2', '-1.4660197718643934781936e+1', '-844.365544295088168718064');
    T('-3.6e+0', '-6.453e-9', '-3.599999993547');
    T('6.875e+2', '2.496601459911533840095e-1', '687.2503398540088466159905');
    T('-3.4091253800019e+10', '5.06219863404e+11', '-540311117204.019');
    T('2.387640687847751e-9', '-7.507348694854873e+1', '75.073486950936370687847751');
    T('2.358337603147636672593e+20', '-5.29206788288316214838405e+3', '235833760314763672551.36788288316214838405');
    T('8.4460769897709104577e+2', '-1.15924234770358e+7', '11593268.08473477709104577');
    T('-2.8e+0', '-1.5677449559023e+12', '1567744955899.5');
    T('-3.826911067801817992605e+13', '-2.1058817772376e-17', '-38269110678018.179926049999999978941182227624');
    T('-1.69888440840703020274433621314e+3', '1.062612513780848e+14', '-106261251379783.68440840703020274433621314');
    T('1.648578064349849e+13', '-1.21870444531163389687261e+5', '16485780765368.934531163389687261');
    T('2.800963824434122005e+17', '7.48550615366110606948944e+18', '-7205409771217693868.98944');
    T('-1.204700704881831924047852e+4', '4.1180094e+8', '-411812987.00704881831924047852');
    T('-6.25e+0', '-2.76872444100293e-3', '-6.24723127555899707');
    T('-1.017210130108320604980095901e+6', '4.176006246137492e+5', '-1434810.754722069804980095901');
    T('4.1443694206107e+13', '6.12963489e+6', '41443688076472.11');
    T('1.9e-17', '-7.420644757780278e+11', '742064475778.027800000000000019');
    T('8.1905699267317e+2', '-4.97714998177821e+13', '49771499818601.15699267317');
    T('-9.30893198737591725124852e-6', '4.7833151233848e+8', '-478331512.33848930893198737591725124852');
    T('-1.81958867206789289436468142851e+23', '-3.097e-2', '-1.81958867206789289436468111881e+23');
    T('-1.444361606243966e+3', '-3.956649026995563806968275e+24', '3.956649026995563806966830638393756034e+24');
    T('-5e+0', '1.12518838778955247e+6', '-1125193.38778955247');
    T('1.63068443484005465679998399962e+10', '5.74911714044959689e+11', '-558604869696.5591424320001600038');
    T('2.76768430312670672e+11', '-3.509356457052e+6', '276771939669.127724');
    T('-4.7752e+2', '-1.66411766445e-11', '-477.5199999999833588233555');
    T('2.92062139592881e+4', '7.5417235168554886240845487e+25', '-7.54172351685548862408162807860407119e+25');
    T('-3.44544937179490458706e+21', '-6.4975e-20', '-3.445449371794904587059999999999999999999935025e+21');
    T('1.67e+0', '4.0077e+3', '-4006.03');
    T('1.6243825105e+10', '-1.0098e-8', '16243825105.000000010098');
    T('-5.9619584679887e-7', '9.66507394754761256582e+0', '-9.66507454374345936469');
    T('1.80193e+0', '5.50148888976681094630325187e+26', '-5.5014888897668109463032518519807e+26');
    T('1.160439596351493417e+12', '0e+0', '1160439596351.493417');
    T('4.0943062e+3', '-1.663541448179706e+1', '4110.94161448179706');
    T('-1e+0', '-3.35949335221661870210955798738e+27', '3.35949335221661870210955798638e+27');
    T('9.26520428366e-14', '2e+0', '-1.9999999999999073479571634');
    T('-1.824416684e+5', '3.45961e+4', '-217037.7684');
    T('1.13072307e-4', '-3.34360798215590565e+15', '3343607982155905.650113072307');
    T('-2.17632297389949094797657e+23', '1.19e+0', '-2.1763229738994909479765819e+23');
    T('-1.573702e+6', '-1.25528655078116284599e+4', '-1561149.1344921883715401');
    T('6.64914341532954197034212171e+13', '-2e+0', '66491434153297.4197034212171');
    T('4.83091120908653225429424458e+26', '-1.187019616662875213797e+16', '4.8309112092052342159605321013797e+26');
    T('-2.2287372197e+10', '-6.1022242482e+5', '-22286761974.57518');
    T('9.86041489707e+10', '6.466463546e+2', '98604148324.0536454');
    T('1.576597918713765858286e+19', '5.733014936878637e-11', '15765979187137658582.85999999994266985063121363');
    T('-1.66885015827009e+9', '-1.0080560673990663464633e+2', '-1668850057.46448326009336535367');
    T('9.464436245828899351275554369e+26', '1.096e+3', '9.464436245828899351275543409e+26');
    T('2.443594208008465567246e+15', '-4.554332290621012278987216e+6', '2443594212562797.857867012278987216');
    T('5.633400747825031776e+9', '3.7814892610756903304424493614e+28', '-3.7814892610756903298791092866174968224e+28');
    T('4.58055e-16', '-2e+1', '20.000000000000000458055');
    T('5.8468767369534498129149668101e+21', '-6.216283e+1', '5.8468767369534498129771296401e+21');
    T('2.821e+1', '-9.2972318416976897120959e-7', '28.21000092972318416976897120959');
    T('1.1716e+4', '-3.117484326301445402577599e+5', '323464.4326301445402577599');
    T('3.99770350169601009505e+9', '1.24807913187467085161e+12', '-1244081428372.97484151495');
    T('-4.33494867e+1', '1.86891793238107724209212e+23', '-1.868917932381077242092553494867e+23');
    T('-1.49454787799250031e-14', '3.530543e+4', '-35305.4300000000000149454787799250031');
    T('4.17405829e+7', '-1.101329481059979917182595274247e+30', '1.1013294810599799171826370148299e+30');
    T('-8.233e-4', '-4.3503021e-3', '0.0035270021');
    T('1.54205169188e+11', '-1.184e+0', '154205169189.184');
    T('8.47800873e+2', '-1.782e+2', '1026.000873');
    T('-6.3788050181391747328e-14', '-1.37296254009260334434e+20', '137296254009260334433.999999999999936211949818608252672');
    T('-2.3187272390455607805797367e-15', '3.03283266313882322447430852e-7', '-3.032832686326095614929916325797367e-7');
    T('4.3489421504381358410994764309e-17', '4.84138351196e-18', '3.8648037992421358410994764309e-17');
    T('-4.5630374165368095e+10', '8.62482790372860322202392e+23', '-8.62482790372905952576557368095e+23');
    T('-1.25355946295e+11', '-3.311539940149343915869e+1', '-125355946261.88460059850656084131');
    T('-4.7415711963385615707e+16', '-1e+0', '-47415711963385614.707');
    T('4.458641e-16', '-3.8021524504141994601653134206e+4', '38021.524504141994602098998306');
    T('3.8057808011e-1', '9.59582e-8', '0.3805779841518');
    T('2.340942415924217e-10', '-3e+0', '3.0000000002340942415924217');
    T('0e+0', '-1.3748e+4', '13748');
    T('-3.634008711669113e-7', '-5.865299497e+9', '5865299496.9999996365991288330887');
    T('1.39690641e+3', '7.12382644256362362011e-19', '1396.906409999999999999287617355743637637989');
    T('-6.52e-7', '-5.4715e+4', '54714.999999348');
    T('6.56e+0', '-8.1504302966189587144e+16', '81504302966189593.704');
    T('4.2093554363975272071207e-17', '1.223e+3', '-1222.999999999999999957906445636024727928793');
    T('-3.739622748318e+4', '-2.01200115180528303803076e+9', '2011963755.57779985803076');
    T('0e+0', '8.987465875243974e+15', '-8987465875243974');
    T('-6.4436e-16', '-1.9979e-1', '0.19978999999999935564');
    T('-1.01268732158001903838205690742e+4', '-4.7518687460936e+6', '4741741.8728777998096161794309258');
    T('0e+0', '-1.3030144588487978649e+10', '13030144588.487978649');
    T('-3.9e+0', '5.24670125e+8', '-524670128.9');
    T('3.8313670568933092257440033774e+17', '-5.741523524290048567e+2', '383136705689331496.7267527667448567');
    T('-2.56218040669566121e+12', '1.83637202203675e+8', '-2562364043897.864885');

    log('\n ' + passed + ' of ' + total + ' tests passed in ' + (+new Date() - start) + ' ms \n');
    return [passed, total];
})(this.BigNumber);
if (typeof module !== 'undefined' && module.exports) module.exports = count;
