var count = (function pow(BigNumber) {
    var start = +new Date(),
        log,
        error,
        undefined,
        passed = 0,
        total = 0,
        MAX_POWER = Math.pow(2, 53) - 1;

    if (typeof window === 'undefined') {
        log = console.log;
        error = console.error;
    } else {
        log = function (str) { document.body.innerHTML += str.replace('\n', '<br>') };
        error = function (str) { document.body.innerHTML += '<div style="color: red">' +
          str.replace('\n', '<br>') + '</div>' };
    }

    if (!BigNumber && typeof require === 'function') BigNumber = require('../bignumber.js');

    function assert(expected, actual) {
        total++;
        if (expected !== actual) {
           error('\n Test number: ' + total + ' failed');
           error(' Expected: ' + expected);
           error(' Actual:   ' + actual);
           //process.exit();
        } else {
            passed++;
            //log('\n Expected and actual: ' + actual);
        }
    }

    function assertException(func, message) {
        var actual;
        total++;
        try {
            func();
        } catch (e) {
            actual = e;
        }
        if (actual && actual.name == 'BigNumber Error') {
            passed++;
            //log('\n Expected and actual: ' + actual);
        } else {
            error('\n Test number: ' + total + ' failed');
            error('\n Expected: ' + message + ' to raise a BigNumber Error.');
            error(' Actual:   ' + (actual || 'no exception'));
            //process.exit();
        }
    }

    var T = function (expected, n, exp) {
        assert(String(expected), String(new BigNumber(n).pow(exp)));
    };

    log('\n Testing pow...');

    BigNumber.config({
        POW_PRECISION: 0,
        DECIMAL_PLACES: 20,
        ROUNDING_MODE: 4,
        ERRORS: false,
        RANGE: 1E9,
        EXPONENTIAL_AT: [-7, 21]
    });

    // ---------------------------------------------------------------- v8 start
    // Tests the special cases specified by ES 15.8.2.13


    T(4, 2, 2);
    T(2147483648, 2, 31);
    T(0.25, 2, -2);
    T(0.0625, 2, -4);
    T(1, 1, 100);
    T(0, 0, 1000);

    T(NaN, 2, NaN);
    T(NaN, +0, NaN);
    T(NaN, -0, NaN);
    T(NaN, Infinity, NaN);
    T(NaN, -Infinity, NaN);

    T(1, NaN, +0);
    T(1, NaN, -0);

    T(NaN, NaN, NaN);
    T(NaN, NaN, 2.2);
    T(NaN, NaN, 1);
    T(NaN, NaN, -1);
    T(NaN, NaN, -2.2);
    T(NaN, NaN, Infinity);
    T(NaN, NaN, -Infinity);

    T(Infinity, 1.1, Infinity);
    T(Infinity, -1.1, Infinity);
    T(Infinity, 2, Infinity);
    T(Infinity, -2, Infinity);
    T(0, 2, -Infinity);
    T(0, -2, -Infinity);

    T(+Infinity, 1/1.1, -Infinity);
    T(+Infinity, 1/-1.1, -Infinity);
    T(+Infinity, 1/2, -Infinity);
    T(+Infinity, 1/-2, -Infinity);

    T(NaN, 1, Infinity);
    T(NaN, 1, -Infinity);
    T(NaN, -1, Infinity);
    T(NaN, -1, -Infinity);

    T(+0, 0.1, Infinity);
    T(+0, -0.1, Infinity);
    T(+0, 0.999, Infinity);
    T(+0, -0.999, Infinity);

    T(Infinity, 0.1, -Infinity);
    T(Infinity, -0.1, -Infinity);
    T(Infinity, 0.999, -Infinity);
    T(Infinity, -0.999, -Infinity);

    T(Infinity, Infinity, 2);
    T(+Infinity, 1/Infinity, -2);
    T(Infinity, -Infinity, 2);

    T(-Infinity, -Infinity, 3);
    T(-Infinity, -Infinity, 13);

    T(-Infinity, 1/-Infinity, -3);
    T(-Infinity, 1/-Infinity, -13);
    T(-Infinity, 1/-0, 3);
    T(-Infinity, 1/-0, 13);
    T(-Infinity, -0, -3);
    T(-Infinity, -0, -13);
    T(-Infinity, -0, -1);

    // Out of range exponents become +-Infinity, so the result will not match Math.pow.
    T(Infinity, -0, -MAX_POWER - 1);            // Math.pow(-0, -10000000001) == -Infinity, Math.pow(-0, -Infinity) == Infinity
    T(+Infinity, 1/-Infinity, -2);

    T(+Infinity, 1/+0, 1.1);
    T(+Infinity, 1/+0, 2);

    T(Infinity, +0, -1.1);
    T(Infinity, +0, -2);

    T(+Infinity, 1/-0, 2);
    T(Infinity, -0, -2);

    // As ERRORS is false, these exponents will be truncated to 1.
    T(-0.00001, -0.00001, 1.1);                 //    T(NaN, -0.00001, 1.1);
    T(-100000, -0.00001, -1.1);                 //    T(NaN, -0.00001, -1.1);

    T(-Infinity, 1/-0, 1);
    T(Infinity, 1/-0, MAX_POWER + 1);           // Math.pow(1/-0, 10000000001) == -Infinity

    T(NaN, 2, null);                            // Math.pow(2, null) == 1
    T(NaN, 2, undefined);
    T(NaN, 2, NaN);
    T(NaN, 2, 'erger');
    T(NaN, 2, {});
    T(NaN, 2, []);                              // Math.pow(2, []) == 1
    T(Infinity, 2,  MAX_POWER + 1);
    T(0, 2, - MAX_POWER - 1);
    T(Infinity, -2,  MAX_POWER + 1);            // Math.pow(-2, MAX_POWER + 1) == -Infinity
    T(0, -2, - MAX_POWER - 1);
    T(0, -0.2,  MAX_POWER + 1);
    T(Infinity, -0.2, - MAX_POWER - 1);         // Math.pow(-0.2, -MAX_POWER - 1) == -Infinity
    T(0, 0.2,  MAX_POWER + 1);
    T(Infinity, 0.2, - MAX_POWER - 1);

    // The exponent will be truncated to 0, so the result will always be 1.
    T(1, 1/-0, 0.5);                            // Math.pow(1/-0, 0.5) == Infinity
    T(1, -0, -0.5);                             // Math.pow(-0, -0.5) == Infinity
    T(1, 1/-0, 0.6);                            // Math.pow(1/-0, 0.6) == Infinity
    T(1, -0, -0.6);                             // Math.pow(-0, -0.6) == Infinity
    T(1, Infinity, 0.1);                        // Math.pow(Infinity, 0.1) == Infinity
    T(1, 1/Infinity, -0.1);                     // Math.pow(1/Infinity, -0.1) == Infinity

    // ------------------------------------------------------------------ v8 end

    T(16, 2, 4.01);
    T(8, 2, 3.99);

    T('4096', '8', 4);
    T('-1.331', '-1.1', 3);
    T('5.125696', '-2.264', 2);
    T('6.7266e-7', '6.7266E-7', 1);
    T('1', '-1', 8);
    T('4142779.6499215776', '21.06', 5);
    T('731.1616', '-5.2', 4);
    T('1', '61818', 0);
    T('3.2', '3.2', 1);
    T('5.4139923025768140625e-41', '-1.945E-7', 6);
    T('1280630.81718016', '5.8', 8);
    T('3965.318943552', '15.828', 3);
    T('53.721', '53.721', 1);
    T('1', '-1.9', 0);
    T('4.58357323731267363492522744606954913014016e-39', '-0.0000161306', 8);
    T('-8560814308.6108448224', '-96.94', 5);
    T('4.4127502627834341562081e-74', '-6.77E-10', 8);
    T('79327261142.56790234534719652175488087744161', '23.0371', 8);
    T('3.101121e-38', '1.761E-19', 2);
    T('1', '-1', 4);
    T('6.23201296e-28', '-1.58E-7', 4);
    T('-8.50893402176e-19', '-9.476E-7', 3);
    T('16', '4', 2);
    T('90368789.0625', '-97.5', 4);
    T('1', '-112.8', 0);
    T('4.122181458338334221291398681640625e+27', '40042.5', 6);
    T('5.94467302491009e+21', '1290.0', 7);
    T('5.6599836943004175019970957e+25', '141437', 5);
    T('9', '3', 2);
    T('69.75757441', '1.7', 8);
    T('1e-42', '0.0000010', 7);

    T('8', '8', 1);
    T('-2.5090696333749305038864384e+25', '-664', 9);
    T('24794.911296', '5.4', 6);
    T('7077398515.1515538432', '93.32', 5);
    T('-1.4520042511984659693722813984375e-130', '-2.8295E-19', 7);
    T('4', '4', 1);
    T('372088627687.312953646321', '781.019', 4);
    T('-5.3864523289963490660381317787272961329536e-23', '-0.000658806', 7);
    T('-1.8', '-1.8', 1);
    T('1', '-5.9', 0);
    T('1', '2', 0);
    T('-6.4097340625', '-1.45', 5);
    T('170859375', '15', 7);
    T('595732589817199.440265999920799232', '43.82', 9);
    T('1.7080198121677824e+36', '-4200', 10);
    T('1', '658.8', 0);
    T('136.460505366756569881', '2.269', 6);
    T('-1', '-1', 3);
    T('236007.159691873761', '-22.041', 4);
    T('1', '-1.1', 0);
    T('287803125756.085809269657829376', '18.76', 9);
    T('4.1069049025e-34', '-2.02655E-17', 2);
    T('-8', '-2', 3);
    T('-5.0787635527751e-52', '-5.51E-11', 5);
    T('-8', '-2', 3);
    T('1704883919.2576', '203.2', 4);
    T('22106814.0740608', '11.2', 7);
    T('3.7481851264119295287828498195966142655968201e+23', '227.71', 10);
    T('-3', '-3', 1);
    T('5.308416e-54', '4.8E-14', 4);
    T('1', '-1', 8);
    T('1', '-4.1', 0);
    T('398', '398', 1);
    T('1', '1.17981E-18', 0);
    T('255896509713547.45824', '761.4', 5);
    T('16807', '7', 5);
    T('1', '232.6', 0);
    T('2.67066142562472466573674890357652039841e+30', '-6358.1', 8);
    T('390625', '-5', 8);
    T('85766121', '21.0', 6);
    T('2206809.987903620081317314341735861401', '4.309', 10);
    T('6.1917364224e-80', '-1.2E-8', 10);
    T('64', '-2', 6);
    T('-1', '-1', 7);
    T('-39.8778220049', '-2.09', 5);
    T('892496818.033068251283537321', '-31.029', 6);
    T('1.1289646949223432899980166202016362758071452681629968849e+25', '320.087', 10);
    T('1932991.455312009', '124.569', 3);
    T('635307227133823.258624', '-293.2', 6);
    T('784', '-28', 2);
    T('-43361725294765860565.175323475675341312', '-152.02', 9);
    T('1589.6036888689492933115234375', '4.36775', 5);
    T('1', '-18.4', 0);
    T('20832.8532313216810321678408500840248440324096', '3.46612', 8);
    T('6.5536', '-1.6', 4);
    T('174729381067.247616', '5590.56', 3);
    T('-4.084101e-14', '-0.0021', 5);
    T('7.46848810699576790739263937619996819197249e+21', '153.93', 10);
    T('2.09881105970752e-16', '0.0000059428', 3);
    T('23298085122481', '-169', 6);
    T('1', '-4.095E-11', 0);
    T('30016915.773120638290557721', '17.629', 6);
    T('16807', '7', 5);
    T('130666515.5580240243056896', '-10.34', 8);
    T('-5.97080224872032e-91', '-9.02E-19', 5);
    T('-3450.25251', '-5.1', 5);
    T('43046721', '-9', 8);
    T('1', '1', 1);
    T('3545.66529682492339392399', '5.1279', 5);
    T('3396.171616714297', '15.0313', 3);
    T('4.622674003397519975764019459730496e+33', '-408146', 6);
    T('1', '918.0', 0);
    T('7.59375', '1.5', 5);
    T('-67822.3072849', '-4.90', 7);
    T('3.7588592026706176e-104', '-1.18E-13', 8);
    T('1', '1', 3);
    T('5237990.22862336', '47.84', 4);
    T('11698.5856', '10.4', 4);
    T('110075314176', '24', 8);
    T('3.24210716131761936e-27', '2.38620E-7', 4);
    T('9', '3', 2);
    T('14641', '121', 2);
    T('62.81386652381601821841', '2.81523', 4);
    T('1', '8', 0);
    T('4.29981696e-160', '-1.2E-20', 8);
    T('1.692652673311521044295612689449216e-7', '-0.14242', 8);
    T('1', '-19.5', 0);
    T('1', '8621.8', 0);
    T('7.907e-7', '7.907E-7', 1);
    T('1.61051', '1.1', 5);
    T('1.553e-14', '1.553E-14', 1);
    T('858461358961485360081', '171171', 4);
    T('64', '-2', 6);
    T('-2.9', '-2.9', 1);
    T('-2.0661046784e-179', '-1.4E-20', 9);
    T('39.0625', '-2.5', 4);
    T('-391460020121.8781037', '-45.3', 7);
    T('-80.1', '-80.1', 1);
    T('3.5831808', '1.2', 7);
    T('41.08469075197275390625', '-1.45', 10);
    T('-128', '-2', 7);
    T('-5277043158.170301334052958483', '-12.03', 9);
    T('100.31024025', '10.0155', 2);
    T('262144', '4', 9);
    T('-2.66450474490105494016e-70', '-1.86E-8', 9);
    T('1.58541338622748269695158625134808009096449e+21', '-131.83', 10);
    T('1.69e-14', '1.3E-7', 2);
    T('81', '-3', 4);
    T('1.3979045762098993055105649e+25', '327', 10);
    T('-2.8334269484119140625e-17', '-0.0145', 9);
    T('8455365204.69607', '96.7', 5);
    T('4.826809', '-1.3', 6);
    T('-4.2027e-14', '-4.2027E-14', 1);
    T('-3671885391606530844.02199', '-5163.9', 5);
    T('1.4833', '1.4833', 1);
    T('1078702060.96', '32843.6', 2);
    T('16384', '4', 7);
    T('0.000041046707114327285209693732389121', '-0.185729', 6);
    T('9.04141586678594150656e-76', '1.73404E-19', 4);
    T('7', '7', 1);
    T('-5.408864751631992324037382349788164206309e+39', '-25989', 9);
    T('3.6213725246884329693766314512921496510839580087890625e-58', '0.00000180255', 10);
    T('51.58686976', '-2.68', 4);
    T('32562222.2784028467808485507564561204461862636001', '5.6399', 10);
    T('1.5269627878770126091369423832739776731492122624e-24', '-0.0041532', 10);
    T('2.27128515349184346452713121698128944001e-82', '-6.849E-9', 10);
    T('788.12330352545906108118941549114886272', '2.59298', 7);
    T('-128', '-2', 7);
    T('2.43585613160298150834272926486523151679435546875e+29', '1841.55', 9);
    T('3573226485.213841', '39.1', 6);
    T('5.0912', '5.0912', 1);
    T('1.302260124847515625e-102', '1.0450E-17', 6);
    T('-1097881.796860068547323829', '-4.69', 9);
    T('24.137569', '1.70', 6);
    T('67.937289638464', '-2.02', 6);
    T('-91125', '-45.0', 3);
    T('3.5658406477912053139330818066558837890625e-96', '1.17225E-12', 8);
    T('5.11226325150500959599107782635087831282590275214649e-160', '-1.17723E-16', 10);
    T('67.937289638464', '-2.02', 6);
    T('-3125', '-5', 5);
    T('-3125', '-5', 5);
    T('-2983765.0756983032103435524997723277', '-8.4133', 7);
    T('4535877.137495584829386816', '-12.866', 6);
    T('256', '-2', 8);
    T('-7.9', '-7.9', 1);
    T('4.18161601', '1.430', 4);
    T('2.3591116836e-38', '-1.53594E-19', 2);
    T('6648326359.9150104576', '9.6', 10);
    T('65536', '-4', 8);
    T('22777216155500625', '-12285', 4);
    T('1', '-1.14E-9', 0);
    T('1', '7.4', 0);
    T('1.21', '1.1', 2);
    T('0.248', '0.248', 1);
    T('-2187', '-3', 7);
    T('1.48996e-23', '-3.86E-12', 2);
    T('2.49173e-14', '2.49173E-14', 1);
    T('-2738.124199', '-13.99', 3);
    T('-2.7044661231722633428249e-63', '-3.0649E-13', 5);
    T('0.0000998001', '-0.00999', 2);
    T('9', '3', 2);
    T('-331.370935156703232', '-3.192', 5);
    T('2.611824102393094023763', '1.147', 7);
    T('-527', '-527', 1);
    T('27', '3', 3);
    T('2.313441e-34', '3.90E-9', 4);
    T('1', '-8.00148E-16', 0);
    T('6.427669793976056365481141274099681e+25', '-1682.7', 8);
    T('-27', '-3', 3);
    T('256', '-2', 8);
    T('59049', '-3', 10);
    T('8.7713801081173367580884490451432713332434379776e+26', '494.66', 10);
    T('20234327779106168.4159474599729911316138226499265176167424', '42.7178', 10);
    T('-27', '-3', 3);
    T('0.009411328144', '-0.097012', 2);
    T('349044.64', '-590.8', 2);
    T('4750.104241', '-4.1', 6);
    T('-173928970714601588962.8103429', '-778.9', 7);
    T('3656158440062976', '-36', 10);
    T('6.12220032e+22', '1800', 7);
    T('1', '3.4', 0);
    T('9.99997000003000028999940000030000299999700000000001e+29', '9999990000.0000001', 3);
    T('9.70097321773035876225154576358130561e-37', '0.000031503', 8);
    T('9.851127637605409e-117', '-4.63E-20', 6);
    T('1', '-1.4E-9', 0);
    T('-4.52470110728381829e-7', '-0.00767709', 3);
    T('282475249', '7.0', 10);
    T('16651779441254092864', '1598', 6);
    T('729', '3', 6);
    T('6.90946358996746999370693799867018948630842395035736169049e-104', '-4.82997E-11', 10);
    T('684.84796416', '26.1696', 2);
    T('-1', '-1', 5);
    T('1', '-6', 0);
    T('0.0000078310985281', '-0.23', 8);
    T('2.274064', '-1.508', 2);
    T('3.04122555034158459939649e-47', '-0.0000223', 10);
    T('696452982073287.871843324909094936562277581368328192', '44.5872', 9);
    T('-7.5151448e-29', '-4.22E-10', 3);
    T('1', '6', 0);
    T('-12.708', '-12.708', 1);
    T('70', '70', 1);
    T('2187', '3', 7);
    T('-1.628413597910449e-174', '-4.9E-20', 9);
    T('-6.312457192887951173182283446626151709594241e-66', '-5.6961E-8', 9);
    T('0.000299069043583441159212021318210939', '0.4059', 9);
    T('15.995353981724701', '1.741', 5);
    T('1', '97857', 0);
    T('0.221585791441', '-0.470729', 2);
    T('1', '52.0023', 0);
    T('-26015680550432', '-482', 5);
    T('22903.7956', '151.34', 2);
    T('6.561e-17', '-8.1E-9', 2);
    T('-93189625398108974.44387339079552', '-265.58', 7);
    T('-3086125724460522.34468469084288', '-163.22', 7);
    T('0.0000051', '0.0000051', 1);
    T('19683', '3', 9);
    T('-2.4507253464874125175251e+22', '-30051', 5);
    T('-38.907147732329430154371456512', '-1.502', 9);
    T('4.68733667816790896452997907438948199465216e+41', '161758', 8);
    T('64', '-2', 6);
    T('27541696650101.74182563635028557824', '-22.08', 10);
    T('1.352771823277379852935844331805603817270014056494234420224e-23', '-0.00516562', 10);
    T('1', '1', 4);
    T('-2350072823968', '-298', 5);
    T('8941.5936', '-94.56', 2);
    T('97241.5', '97241.5', 1);
    T('6.93207873781332929149e-50', '9.49E-8', 7);
    T('-59049', '-9', 5);
    T('-8882.144880086754771640625', '-3.665', 7);
    T('8100981934880.472', '20083.8', 3);
    T('16', '4.0', 2);
    T('6.561e-49', '-9.00E-13', 4);
    T('-446592952.353220867168501', '-53.701', 5);
    T('-3.2466525536194576823494051652634346881e-125', '-1.47210E-14', 9);
    T('3.89751', '3.89751', 1);
    T('3.8443359375e-107', '1.5E-12', 9);
    T('493.039', '7.9', 3);
    T('-16807', '-7', 5);
    T('-4.7471727261300877329502034895519256591796875e+34', '-7127.5', 9);
    T('49', '-7', 2);
    T('1.27e-12', '1.27E-12', 1);
    T('30373391.06115737305088', '11.72', 7);
    T('1', '-1', 6);
    T('-19349.17632', '-7.2', 5);
    T('361', '19.0', 2);
    T('1', '0', 0);
    T('-10077696', '-6', 9);
    T('42180.533641', '5.9', 6);
    T('38.443359375', '1.5', 9);
    T('1643032', '118', 3);
    T('16', '-2', 4);
    T('1962.49', '-44.3', 2);
    T('640.09', '-25.3', 2);
    T('8693.75638483077329986588967330173480782272550321318049', '-2.47697', 10);
    T('8105445187812500000', '6050', 5);
    T('3106365092236328976', '41982', 4);
    T('1.9e-18', '1.9E-18', 1);
    T('1388437454689.837102639637015104', '105.622', 6);
    T('217.67', '217.67', 1);
    T('-1', '-1', 1);
    T('-40.84101', '-2.1', 5);
    T('1.542564e-12', '-0.0000012420', 2);
    T('-1.18009517625e-37', '-4.9050E-13', 3);
    T('5549069137356750625', '-48535', 4);
    T('1172.7599043051', '4.11', 5);
    T('1.024e-39', '-3.2E-20', 2);
    T('64', '-2', 6);
    T('9765625', '-5', 10);
    T('10174708482.4576', '317.6', 4);
    T('4596.544052624786837880243001942570369016159210753', '2.55233', 9);
    T('4096', '8', 4);
    T('729', '-3', 6);
    T('19034163', '267.0', 3);
    T('7.13342911662882601e-183', '-6.1E-19', 10);
    T('-8', '-2', 3);
    T('7.458836077824580594735138491702651752566987e+24', '580.27', 9);
    T('-1.488656882033767469315065509517197312e-126', '-1.0452E-14', 9);
    T('-22512044836648447.5871232', '-216.8', 7);
    T('1', '1.0', 2);
    T('-1048576', '-16.0', 5);
    T('2.96566225198439749053708901e+26', '196981', 5);
    T('387361397731.0852349972153162834577841407848071071001', '-14.4149', 10);
    T('2.5937424601', '1.1', 10);
    T('1', '-132.84', 0);
    T('1', '4642.2', 0);
    T('-7.3499661205554469210497482752e+28', '-1612', 9);
    T('2.56745903401e-35', '-5.06701E-18', 2);
    T('-8.0416926477372443174045490389e-14', '-0.013469', 7);
    T('481.890304', '2.8', 6);
    T('-2.0079766520165347109375e+22', '-1535', 7);
    T('-3.34255384e-7', '-0.00694', 3);
    T('65536', '-4', 8);
    T('15700.150160096030724096', '5.004', 6);
    T('3', '3', 1);
    T('-1024', '-4.0', 5);
    T('1', '-0.0000026', 0);
    T('3.75e-11', '3.75E-11', 1);
    T('-2.197e-33', '-1.3E-11', 3);
    T('850260.560009615038237114368', '7.032', 7);
    T('1.165611915415579117119882256009e-48', '-1.02587E-8', 6);
    T('1.874746414628249416823686235041e+30', '-6083', 8);
    T('3541035030023585410.9685579776', '-71.6', 10);
    T('159873594510682.812023861622308942042241', '59.631', 8);
    T('964', '964', 1);
    T('1', '7', 0);
    T('1', '1', 7);
    T('959554720546.9939335182868736', '-31.46', 8);
    T('390625', '5', 8);
    T('8.58117156760783434066136732224e-67', '9.7482E-12', 6);
    T('1', '1.10E-19', 0);
    T('2.7206534396294947e-38', '0.000067', 9);
    T('-2.7993930117296371229848579851719540736e+30', '-22365.6', 7);
    T('5531947943217503125', '5605', 5);
    T('2.77921878692682183940201e+33', '2210', 10);
    T('-1.34775901066369285240390603125e+29', '-669765', 5);
    T('1', '1', 8);
    T('-187.539', '-187.539', 1);
    T('114983.567789585767578125', '3.65', 9);
    T('1', '1', 6);
    T('9.5563709286130016454092290324007041e+28', '-67615.9', 6);
    T('2.39576999392237432790890516398336', '-1.1154', 8);
    T('-23.924567789936824982851522063', '-1.423', 9);
    T('39.4384', '-6.28', 2);
    T('46411484401953', '33', 9);
    T('3864', '3864', 1);
    T('20554002898923904', '214', 7);
    T('1', '-481', 0);
    T('1073741824', '8', 10);
    T('64', '-2', 6);
    T('-15633814.156853823', '-6.3', 9);
    T('7.9', '7.9', 1);
    T('-284.68', '-284.68', 1);
    T('1', '92.6', 0);
    T('256', '4', 4);
    T('-4233837892.359375', '-1617.75', 3);
    T('1.01054975167057821704275308297863156222443369140625e+40', '10010.5', 10);
    T('4.15867265447116306999621451776e-171', '-9.16E-18', 10);
    T('14796.346375', '24.55', 3);
    T('0.000003596774583043934673428448834277192142749696', '-0.208684', 8);
    T('19619412024963200000', '7220', 5);
    T('203.33219575276960518721', '3.77617', 4);
    T('2187', '3', 7);
    T('-512', '-2', 9);
    T('1', '2.4E-15', 0);
    T('-79', '-79', 1);
    T('-281399112371155271', '-311', 7);
    T('2.0441457156547956272121e-32', '-0.000005229', 6);
    T('164546156906.7655853425459503979807203014092043691201', '13.2321', 10);
    T('-46.13', '-46.13', 1);
    T('191707312997281', '-61', 8);
    T('1019.4513038883292709315124224', '3.99644', 5);
    T('1', '3', 0);
    T('1', '-44.4', 0);
    T('545516.0701056', '6.6', 7);
    T('-4177', '-4177', 1);
    T('358050', '358050', 1);
    T('229.97659811731444699830505796029057408', '2.17462', 7);
    T('2.6108288368639387032909799e-70', '1.21159E-14', 5);
    T('4325169905.757', '1629.3', 3);
    T('122.5', '122.50', 1);
    T('1', '-0.0001858', 0);
    T('3.17145729137553882999930496108165023687245824e+34', '2819.2', 10);
    T('-1', '-1', 3);
    T('-49.358166606568', '-3.6682', 3);
    T('-3.0228559084421978498330001408', '-1.17120', 7);
    T('24.76099', '1.9', 5);
    T('-17.576', '-2.6', 3);
    T('-20', '-20', 1);
    T('-1', '-1', 1);
    T('30804.095140913337149030892729572601', '2.811', 10);
    T('15625', '-5', 6);
    T('117649', '7', 6);
    T('27', '3', 3);
    T('-65939.264', '-40.4', 3);
    T('4.56976e-27', '2.6E-7', 4);
    T('3052847.6176', '41.8', 4);
    T('18440755681.001536', '51.4', 6);
    T('693.43957', '3.7', 5);
    T('-10187688835060.27636096761856', '-72.16', 7);
    T('-290.1', '-290.1', 1);
    T('-2097152', '-8', 7);
    T('63.582', '63.582', 1);
    T('-92713.643576', '-45.26', 3);
    T('58778272623218534616.288838935169', '1971.83', 6);
    T('606071.1605323', '6.7', 7);

    T('49478904727733663575.051582184770446498259493718971258958961046708278746324459568256251259157238318185790764612200843144821137580796334089638532287323488178607687532035618686192601481827902773410791645057647691925965756808571011539462511641266331263543939501901006735272661302011976870015625991111560481188335365459566760212844848008439578745211448299643484693328447464460429576345111356618402802188681216', '-1.60380', 96);
    T('1.17647340482075248419254427344376045314316013640615098803852731448350843468375614975229837000370025634765625e+81', '-1312.5', 26);
    T('7.0251816660378376802360594609084621021532608383100803495211258733288152327693358768741786747381360859889479870288549139760601303377485234176e+87', '-2391.76', 26);
    T('85.717123455378345835968064', '2.0998', 6);
    T('7.8189586726154755713303078223442071754329e+40', '187', 18);
    T('9.145283009860369082115124525217171723626500086794105722560960199131059600008740516579605574000306840952542591838726900925804876261491646795282472377021036430500854987560433039755667787412917103148092128371355207976203112911565783359385741850905996481e+41', '-6.4113', 52);
    T('2.05766289512932241998482631797034621069017517962263802542480365981262536498089637707776e+86', '26', 61);
    T('98190573319064.214163360748608660888346857427032348178060040012159036566957731914573312588336568078808326155328828092023492584783227218123097647282496144318320115860427194595696623235777670393347299594686642872330179664094829207439052010438267983369137644871417464586087393161049', '-1.6293', 66);
    T('1.392598192044227355988772015943279484446048049140235246881310663971517522995991944491655264930301742862780134225258915488062891063359354742689683631308848403923615202977361e+171', '109', 84);
    T('1.5469996446227372120183605762031536360861258626528255060075983309687394111102266369973688377324297491658050323998067994446097153176237864040180207896595918522219595889946775556217e+64', '13.37', 57);
    T('417.53905413413116367045797', '1.3', 23);
    T('1.624290817967129588779052613030846258081e+39', '79677', 8);
    T('6.613998972307900952958925063693292646912001305426127894831260851822286968476028114198011923029274945784114379928252947652708965396453170282038903005249536e-263', '0.00000908', 52);
    T('17179869184', '2', 34);
    T('9.7756260861181096053516970094092017016032995016954767790504110039917368643756356818044285001644705953440430468819149794081183248976623767084182482857941562203266736374606395800621259352114936429301555533971456e+141', '131.6', 67);
    T('4294967296', '2', 32);
    T('131621703842267136', '-6', 22);
    T('7.9228162514264337593543950336e+28', '2', 96);
    T('-1', '-1', 41);
    T('5.0096959237841976223015468271587016074738912992534254294320102857326166341642003475731604090433896328441157235303061782495282575910872792923981113856240569836427242032573810996210128117467500761653330557379347005097932959928243111015057764620250811725789666461853189934772359267843261626881e+197', '140.9', 92);
    T('2.404907604760405225358828131112281116032698930082119547604265954848982041717359046106827774346003151904701415424e+111', '32', 74);
    T('2.69721605590607563262106870407286853611938890184108047911269431464974473521e+22', '-2.7', 52);
    T('18.174', '18.174', 1);
    T('1.36891479058588375991326027382088315966463695625337436471480190078368997177499076593800206155688941388250484440597994042813512732765695774566001e+43', '2.7', 100);
    T('1.25668660115116491467662710973545031109788152235501170924313646703423681882164573441433515503588152722789494580469301248e+72', '34.2', 47);
    T('1.56438420672167084519979701497174364597412396629019101951505612890666817698608365177006994887516584111152450449887451995069033040411213741933147153316594951835865202053237255800944727011696760985308614176976019463960094165747239428884158901702435389388132623055120422260406418255249330452909476806656e+299', '1308', 96);
    T('-1.366174308182025187777528918308146941320736787278460889575749404867695325446359807161410602138908197731224033169833047036778978022709170976735608988099868310813030127497812540494011203545816852288666658682985428600019476831917843e+58', '-4.83', 85);
    T('2.966806088806219456880539423158152815514806645521282752514016241570099725287916962770395912230014801025390625e+28', '-5.15', 40);
    T('8589934592', '2', 33);
    T('3.92012913845865034844198476089476040042936801910400390625e+28', '-10.5', 28);
    T('6.26204859896956162110746451442214931102399028194694661966016701109504299345041406486245804277657094315831017177064948219797627444240188709011456e+72', '10.6', 71);
    T('-7.4010922571797388655581604684527426699437277457499992187268655958229686891933340740793991046504324130585438416249947232728346081088006867719609592595610095069331022367447133421906806310032697675369409918188139600222703781482594772971289070262621142441874895437059567046831661495957273728351051362944953711082346089727330717918651409489163662706555003193741825901442633007312863953564385259087744338361955671878877893732351367361843585968017578125e+364', '-31956.5', 81);
    T('6.192736416792149755132475219493405669399778858602560762871794455317449717448580836229845787580888290473782381059983387805505374173265513677435445890655671110840202944429668728225878544903602239e-270', '0.000319', 77);
    T('2.321827472731847506187210910989207010589651530892302311791322352855446782760907718473019283821056753242967949096791603718944064726033664384082861996009850744961116407128920163494805886357836270766853975184523023642990429569615539676822828070446559161161124635674389704743948383107212919966543969471039720464085404567773913503635201792672811032312063265454094042341369554833714443413857449688164664657173315113983032808001e+220', '159.83', 100);
    T('4.953736650985698001275587761883016173298229503362193331584010106486052779761329981195411366329068322677550960044762649449732919438910119549694010717647780952298320287365138145210431380601324547389442337520392258009e+27', '-2.797', 62);
    T('-338678221586896.02233451034424755534030583977124338116841050703894756829829935320421996887', '-2.47', 37);
    T('-3.181403788572786716059998378326698266679069780899509454959934125355133265193e-226', '-0.0000057', 43);
    T('2.2481244795444847556680083178853571141700204725943661497539185751725919215990484669445118877448657555202573745174750533635052209260608084995835891707958259234271442982681066779696130155696868114346639852814575962457281749995256623713554214237458259176913852401064878760784275382146658972801541462348779288776844691139086889351786721e+179', '229.03', 76);
    T('1.09544203269304121665791452848994634135137143229447033238750572549632571476402641497730897157401184093614916314911584167715115647604065370616785146866296916632437569033863034458711783186506727022529378159297811462454675308057682780534190256280910865229376785975491743196806996272790525705427972427376758443312371576466422700240638756574402393677191376501159420836822933903938571935476091044876631814554274289087939783553959776485376e+31', '-2.0436', 100);
    T('-18437563379.178327736384102280592359424', '-2.4', 27);
    T('-9.9938258857146531850367031e-85', '-0.0000000231', 11);
    T('2.45673281757908854552870925433471410070214400334169452520459965621921664416713287959754208329284439619509099451538227140937811535553399003066102795766261859069268099803838998995339331311912275460846260697085073312049264485545427543086425664188963659692922005683716104197951717376e+78', '-6.08', 100);
    T('-1.270210714458885172439348525985048001607770314656078008119108251089872343340175575576722621917724609375e+24', '-4.15', 39);
    T('1209.5', '1209.5', 1);
    T('1220703125', '5.0', 13);
    T('2.2282376678496302739664556233420633e-74', '0.000000006553', 9);
    T('2.218088039378445352751296336694731578146959285984085545221611060271486214348079661041003283535646898758718143100202167951881147633680620698026648392938984778649e+159', '893', 54);
    T('2.10030968607618842407969732186669026418164521383378412734005003194384320968659491239802049483617846390172315534146347011074332799904332281963786542263761412332465321180532568441958976108339460464558490863953503079336356338135760894630619724493454777575270501253639608281424341291617698518141921861556244699796561591915201261004739209197698527393965924844797945393999012105081714167479170123104969e-75', '-0.16053', 94);
    T('2.5812157101662325924336748344070521698410946762506012782923924916938485004644329931151074914373301798478548152505612780265485671672098712055026947247254600014458282820627468534250894082132928621697659949721616819635951012296371246281717642419683893063297904005186874144169940879165882368e+286', '25488', 65);
    T('-15.625', '-2.5', 3);
    T('2.9729685996261878754343622542880008372085710279050331084451543295302017053107644303792930816e+59', '-72.2', 32);
    T('8.15730721', '1.3', 8);
    T('1027.334860999330573407176938442293022889', '-1.47', 18);
    T('1.20109289910050514343148045475506579676136202628631781424931463646123976128634333627993215283736800191296195442718065238702508520055513255346146658609301480565930380776363581062628797012612519067602295957218821773367819208495782017345400726347727499503218384384326417098721424389373157748956399698521853695814977704822352568748499324367082691064737392226159651276162776412093080580234527587890625e+315', '8679.5', 80);
    T('-6.00774319642841528092711102659123449286352750007573133369794363392e-33', '-0.22', 49);
    T('-4.399348626310072691340857398671747903319767685984244866658805429802943497233040561059795990923598510826890908258017493538766494536483606174431755682553269424176946343755577041745496403493430254087811474250251864009392456889456030217048416123130922140883015639602300329038985615284964386359900403921860549312983015992252637174331659470716194350399038561704290046291e+177', '-81.31', 93);
    T('-3.05410827582876268390180635074883196222812916959850956275590047122226930793718597160851022493537771838978253152932907298258251555751167476560688772053768410645587233357374279876239597797393798828125e-247', '-0.000000217485', 37);
    T('5.94029530519133658573841651381893916937282220672559984930445801765696530117833330901747112098660456061962952244422396324466169433806822249922825538552986841011e-19', '0.491', 59);
    T('6.30523820693302485362185385323378612453984181696300485861970499164041182778517073604560519798249286847423997356860449832519236757706844668669858983416671702949907296869514603475482872905487176008793424936405673085021973514041591620544995041842415714193527926259387876594710020672351902248919941484928131103515625e+217', '-207.5', 94);
    T('1.3437482898194084408359108790569717431056936319064040016470306278420649755238589129250716839755182586596297106623509972801915917688676116852207e-198', '0.0047', 85);
    T('-7.2125976851811538835264180669311252876757309817127885777282268797579465235996423577991265862898834402822899090972405912847104748143261578733778979456054998346007864343435284069799588577484912513898046820531311312862161018665360468415091050892814436289818311468501837802070896183140379745282036670643116477155731121958797307187300620339307163520531060733946755523082115115646976e+279', '-767.6', 97);
    T('-3.6014175419419889979825978713404525121917028354184157603813105843373968813713708575013260529677555144144724689493141661197266911743582631754736205113266715503341523389605232359965730457026715491821102337726896541765297511282920487936650122950956509166185123335946923710632464810273024751e+116', '-23.51', 85);
    T('864441744884.215763637320205869663958329635842919446190153261333597518371076208416885489806126587561695103959519298009135858834566699504679568405218950836778578735139166130606578479646072479355927241854902595925015466294859320621226069690609744261681250304', '1.404', 81);
    T('2.1740698194376449374483147468467027620784304276072222988616865620688896e+52', '808.4', 18);
    T('3.8358611506121121577937e+22', '113', 11);
    T('7.708142065054490966774434895299862856570563111111152865942369472254266622057906176e-375', '0.0000000000000002582', 24);
    T('1.697054670829812124918589801369405630774787179949418285317226086899030607907838200923128944336540480137932154797404155642362243064347816887752994688507537824596616056930370261936449615792564115353680658971035397858675806419291627327666883921628766762017e+153', '35.3', 99);
    T('-4.710128697246244834921603689e+27', '-9', 29);
    T('3.2505902718449668133084058626546515621706592677748264541998335302067693284031047914414370910433910917853346703702752017311701370087768406221174636457161634693311184118695477205673579866365261461786447054994132502457715569269165081396243597525386428206972507983546973520886724081775555138991647659604473279e+52', '6.8159', 63);
    T('10125003344964.40019736846338763496828575459544161517257186895064610033021488387585533013795387142569984', '1.4', 89);
    T('-1.4631650542473900277009650505078819321358806229e+27', '-26.90', 19);
    T('0.000005295825289418576524939548312590198218162176', '0.468', 16);
    T('-2.13825209215512828440269455240169347772084407562319954177735966744864590012014618496355742013223900851293500036195697397558882293948621657334016332630442341312358031641684225772703692879077286212410249112292259617578818732032e+224', '-203479518', 27);
    T('5.444517870735015415413993718908291383296e+39', '8', 44);
    T('-33338.549041729', '-33338.549041729', 1);
    T('2.7214030202002235259872217354276868322505756236796101744750922254072943468398158090938361760503196297567571421573431913968089480080443473498108606049915577803053551131894096917430267992139228904833591260925102332702863889641131058350845466180073805176058062062511496305436593986927276878340795001904960142635221849217162104418149855782368706167456898339731322626805338533602069310753044118499679070391659043014863756212342671519561753648641528938961249781667596503442752009505056776106357574462890625e+355', '25403380.775', 48);
    T('-134217728', '-8.00', 9);
    T('6.4919612772480684272077477598407081955951769084986853540654908040236664132468736e+31', '4.6', 48);
    T('-408518.488', '-74.2', 3);
    T('2.6122383559387081296537383480965114437482497294603331038492821930754511334614082451247995976011754175948033208350877260508339210350470971773483611262598975948388081226982520139805939801386768827832059816233575763932242012655210850575522139572300856461682889553829554870249034765972111622144e-53', '0.119534', 57);
    T('1.88522188856624562372708154656751441041381726119427387599765763695834615013232142421520586436232009071932931843172032205056661409003338693915273301223126561630347970098203184279906247655542800284362476149094060315694708604230353242909365032002466875120342181047689475694706227284673985251088619705863125778538982501735383260557349198668384156297528069830916927729430864813333996813440323955895348081117644429414429531206777610934118661932545955356203081687958272101e+350', '1396953.01', 57);
    T('-7.4641079962956705920034912695642465787383566977879750308559196923222402311665962335871104200586964218306687527948988363660881738310704671349709669487610144905689251123467238632699575375039822099609504292687413249132266915915320573775450823119373280252201183083672896166447786672561808861768782076443312543759437244643146889964363933455672502577964296155349186564340392078021387512970362139779496580009767098175116221996336815276032e+365', '-425417.2', 65);
    T('1.915651048048760989675223006710502095760272088541579446290639477213455428121307995153274075329786132157611746651e+72', '363227.051', 13);
    T('-7.65032026817518248876564402449082201253180407179105772287243705588834467106246226864795794350056707139677395342984049031608196167e-158', '-0.0000000000008205956327', 13);
    T('1.5286700631942576193765185769276826401e+37', '7', 44);
    T('1073741824', '8', 10);
    T('-7.492788722116264058568890205770570774859100309722464309788118049412229345772411480224300523702767256086276612012745150464681862935165491623598382139203570285583278591609354972796097254467143774807748614976801272704624819215123157317922738427454525033948883547809409740884056108275497786621497433099431568320476275959806764951584862378486525479494452910448127181400467328943711415916384184330110378180608e+38', '-2.6742', 91);
    T('-9.9615483925456314514941835282258027490657130453558630944504898425429565393128463181577359582035968e+22', '-34.13672', 15);
    T('1.399922200750534682219817715626042346444265577310152832656021878942015248035909712686518235139333930615671437276362060231490721869741275050613538630628001507198150040959125681675660226363983554975062957910826317184614904091530578807538850671607301864207908455029708464592014690257839027200102495492869235566493012545340996194182688131148892605629158333606816661843824439253963260085811270982958376407623291015625e+181', '8668.79595', 46);
    T('1.1972515182562019788602740026717047105681e+40', '9', 42);
    T('2.809433795080632851709145012225734908409054034941321241601987773815883609706743541544402166343924517939889529050536360240983658675834106822309237371584880050176e+67', '29.26', 46);
    T('-1.130764362193076554708917348657069436161383119992010510285626112959278426786936465605907875370782367006311207257151205709430547608789956273155866459033031412425747753017041228269946019068798157376004812291000927158419091251680264057450083682494989856683271885476193403159681085354715517423876371812000883161101039915298940406738346701704184093773206563783365396974283025554248166993067633181013835776e+229', '-495.16', 85);
    T('-3.077228040283419501485435148888847716813106712632276671229710377509021026817676725408321326107948661263691709915698886286976007892051135335651197784824163323643144277812710153968406419716183905139635176333083510237302643593683953953477494067043109636077672667948180821005422658784541128359007025402810248257138220199195378601727145592714535208968044372704483391015842523268884989073259915535384596690721656759730286732238685451797332150695395077171985430070293527840123294365665570804795301448909420756992e+418', '-5398620298.48', 43);
    T('2591500643510.8703986391408896', '35.620', 8);
    T('-3.35390966929685775566428232543003229321430942684482327393890454463616e+26', '-6156.933746', 7);
    T('-4911940958770.721866380729238693889243243167048192', '-25.7122', 9);
    T('2.09670982511886308779364382461226064522913244291274975553909045464953985730555995547122852179186171376596687978705841751285402054060384366505333092240660435368377833186306191605049e+179', '2093', 54);
    T('-1.3120135409024675389197377e-32', '-0.0000000000235855553', 3);
    T('3.06580286249339590556181770635869764259023245764469206004664101e-78', '0.0061', 35);
    T('-4.429293345196932276984549040219058245905488651609667018402382745119170674779904528756935495114384894434070998056621967747005022674509457845620395626349316891933712941619646377506486625368600115937614507993729672492431012578899815005421568e+37', '-32.051870380', 25);
    T('1.299270565684257922619246441017034698039805547412943905460754496572097300952165266370194652051376805296279609746312644347199238836765289306640625e+56', '-18.85', 44);
    T('1.566108714385843340192620380546546555641735293434509574033302065853656477829384746549085411912657151882140111651595251043453318950397645765893613409097202898027341727466585518405768283017471302953847629952837298920882176e+24', '41.0192888808596', 15);
    T('2.61958676888190130160363553014940478261307337527924200329289138176e+65', '-1866', 20);
    T('3.23590932372253866812859949454563891250217856922745576253641197253435665205998361592799809e+89', '2474977', 14);
    T('-9.7368504802272205153595678239454304952368475046930119079817985916928e+67', '-12', 63);
    T('3.44552147465294110719732986332367243247925798357929806000836849e+62', '-7', 74);
    T('128', '2', 7);
    T('5.65251025822228531444194278761233641822281792819659230058333528631972497846690920646507426317458140864925211864848848044857090776165664263450974899867329254381660849739991179791628529582217569168185463633540183106805961127540063207424e+251', '-96880334819820', 18);
    T('1.214529036103353599078303402353988354368901290518483617131872964733929620706476271152496337890625e+43', '6.5', 53);
    T('-1.3089738102615705488950036718911116878301386483323342409513651601949943841095689259436086715966084434967103664037924413474102289403813212913777522181382481044473580351805322191321973e+181', '-78533', 37);
    T('1.252453015827223091648143056289e+30', '47', 18);
    T('4.593884321465082753417847006029094939767153045428871371185146539475234934788078659649525017869031479813190380591953477672897146298354193774199686172307595609685062957351072442717423824419651156326307969688849329e+172', '34973.7', 38);
    T('6.4993779802563991313942147088910855038980304493628711994552936552675080091715543868582664478495863312478503560216160496462563144213423384672637506819444007554298505039521124189159003567749810227911510054190116940785207230365438230686589958633589792785739898681640625e+25', '19.527362557915', 20);
    T('22876792454961', '3', 28);
    T('429643228077338878294.90980773552148447232', '13385.0992', 5);
    T('-7316.89', '-7316.89', 1);
    T('5234016485162380478.15409743331356736233224871362317808951664029477472141754368438049197212877764328120319129016485361', '2.41', 49);
    T('-9.18448171920669942200338050009727274073573579465100492416464784644248864677562331387086995022599687737054074140424187879986024804042070022603490146718362719056788563440317890910682257768632925460866552260450415793672740716743856864948011399578402033107962721324762764871499139470157293091799845778141752390211181570286270362089589942764421437922022663206638528993662330911966459538905107421403529415706249287725923857606476849002798755812680628308770502507877888997449832630062999974358780733e+247', '-11613.6333', 61);
    T('2.648409294963988844015616e-24', '0.0000000138355056', 3);
    T('2.40863472207038829716951461596425356200891165327233287498341065017430044564718221477455813374032856667895504085743031089347856425830213625711815509685491800628989456526993157007511653813393067316060298253585492583958097263354394562221211900373157618798878275619835293896301261289133749847496154802330721946407650402052956106639135229248180394944200385403412146150497882700273381421471626181859523581299015700836664867216504968658889499860010034200157321932124862289179191239642591047175122517733014555125735104472403712501002006088016431555682009e+357', '6337.73', 94);
    T('-3502571449.82200575261531309080576', '-2.6', 23);
    T('1', '1', 25);
    T('-1.1809125003067061836785344667120764389443931749265317457525459548456128442105786285468877390219672806038351135283671645849621670658638427296075188535296e+27', '-7.46960', 31);
    T('-7804725.584345565904628551916716032', '-1.8', 27);
    T('652.68343537143736977925527144667849979675453569710942641839519790994481', '-1.1', 68);
    T('1', '-3', 0);
    T('4.26825223812027400796974891518773732342988745354489429495479078935112929549619739019072139340757097296812815466676129830954465240517595242384015591919845376e+55', '3.6', 100);
    T('1.143346541001259437465016483752642875340200408207372315398374410086504054045958634479945641547514904516533919702060746490155830653988883705469429546657675670673347514271976875109918113055399954802405065425692486675365016543448426931378922954782957026636988416e+190', '-388997.14', 34);
    T('1.391479860226959780852496779461260634580951060711489905798047445581119629509685004732118558920853318033429289719854192097911392274604744812177542476869031306237630906988638098552100901480110526866015997536096860367785128698535455722265589251495705036009334110148089566472594008624482378925581538366245059147829461639938451131334656e+162', '785.994', 56);
    T('-2.039429275458086101347617297850948854294429450037773018607110681185721614063201210356599410953e+93', '-27753', 21);
    T('1.8072708076715752612606017458825273480777271963492548142293548724970849500539908571180569517655791008990836891664536992680448e+43', '64022.984420938', 9);
    T('-155568095557812224', '-14', 15);
    T('2.18753957001465798602093911444285519704647324430365134831536108015616e+44', '4954.46', 12);

    BigNumber.config({EXPONENTIAL_AT: 0});

    T('2e+0', 2, '1.0');
    T('1.6e+1', 2, '4.00000000');
    T('6.25e-2', 2, -4);
    T('5.0600621890668482322956892808849303e+20', '907.27', 7);
    T('-7e+0', '-7', 1);
    T('-9.01e+2', '-901', 1);
    T('1.016984074247269470395836690098169093010136836967e+39', '21584.7', 9);
    T('-8.983272e+1', '-89.83272', 1);
    T('5.308416e+6', '-48', 4);
    T('3.83432050166120236679168e+23', '52088', 5);
    T('-2.679971527468745095582058350756311201706813294321409e+51', '-517889', 9);
    T('5.067853299870089529116832768e+2', '3.47508', 5);
    T('3.48822062687911109850066182676769e+32', '4129', 9);
    T('1e+0', '-429.32321', 0);
    T('-4.2773e+0', '-4.2773', 1);
    T('-5.8169306081172252508071119604378757744768e+12', '-66.6082', 7);
    T('1e+0', '-7.0654', 0);
    T('-1.51655708279450944384385164853883404204414169862685507e+46', '-3956084.3', 7);
    T('8.1e+1', '-3', 4);
    T('1.296e+3', '-6', 4);
    T('2.9e+0', '2.9', 1);
    T('1.764e+3', '-42', 2);
    T('-9.3418332730097368870513138581415704704611459349313e+49', '-356673', 9);
    T('1.517108809906561e+15', '79', 8);
    T('3.1063e+4', '31063', 1);
    T('1e+0', '-21914.49416', 0);
    T('1.4586250332983909737249e+10', '49.43', 6);
    T('-4.208092749838142546109102616048103440952842087045576022556672e+46', '-4577028.48', 7);
    T('3e+0', '3', 1);
    T('1.4301918832998497740081358663795354088849575358074712812593899025686055664154241e+71', '784195396.7', 8);
    T('1.914994057877243921847839041839022064004621019747303000421532785660701999616e+57', '2315715.86', 9);
    T('1.40379741624014849127481344e+26', '804', 9);
    T('2.1305364420464979969795870610432e+21', '18437.32', 5);
    T('1e+0', '-48', 0);
    T('7.018e+3', '7018', 1);
    T('-5.8149737003040059690390169e+25', '-729', 9);
    T('2.537640625e+9', '50375', 2);
    T('1e+0', '-287', 0);
    T('-1.879616311308566413901e+11', '-179.81', 5);
    T('-3.73314280039567349e+17', '-3269', 5);
    T('8.503056e+6', '-54', 4);
    T('1.114538684361769559136325232186960408417483078419495901986816e+28', '3205.4344', 8);
    T('-4.27504205e+3', '-4275.04205', 1);
    T('-4.18195493e+8', '-53', 5);
    T('-1.32651e+5', '-51', 3);
    T('-4.309348e+6', '-4309348', 1);
    T('5.2697770766776504576e+19', '-7259323024', 2);
    T('7.334473e+6', '7334473', 1);

    T('4.2587208337796928798976e+22', '-674', 8);
    T('2.5398714208994744315474847375553672678449979949471296663472852498581058264323e+76', '82219173067', 7);
    T('3.8595867893817789e+16', '38595867893817789', 1);
    T('1e+0', '6607929.612', 0);
    T('2.03571124747179512510686558355453870501244709050625e+34', '-377727882.4895', 4);
    T('2.45293306092383728169089251608498030232760591201e+47', '-838901', 8);
    T('1e+0', '-3', 0);
    T('3.834566459566834602823771544700625e+33', '248845085', 4);
    T('9.9603189655699799982630929090395077799549456e+43', '-99900649454', 4);
    T('4.96981290961e+11', '-89', 6);
    T('1.58714182711801961914035004098749014540213215369889902124332186866896819339082286680313155541911581006508901525889227579394470766224134369e+74', '175595288.8066529', 9);
    T('1.326409999413464946341631126231535779147856072448189797080406801e+19', '-60348.91227522707', 4);
    T('-2.050107090202653994550415295865311714194868710425929507925126964581376e+69', '-72837546696276', 5);
    T('2.0550048143314549586011369057890711537772093948321e+49', '-2129135057417', 4);
    T('1e+0', '3943801.89538088', 0);
    T('2.17438998607457e+14', '737', 5);
    T('-2.273581752472e+10', '-22735817524.72', 1);
    T('1.51891070159203e+12', '1518910701592.03', 1);
    T('-6.42199562432576e+14', '-86276', 3);
    T('1.33010089e+8', '11533', 2);
    T('5.1662108991396663099278667856407361e+26', '-4767525.59', 4);
    T('-2.197e+3', '-13', 3);
    T('5.7261215932713209368576e+22', '489176', 4);
    T('2.63105908732834720740862027188940646962896811537209872829318157804817881e+71', '800488505731', 6);
    T('-7.8151742291e+2', '-781.51742291', 1);
    T('-2.5619022164869008875e+19', '-2947955', 3);
    T('1.888497394256858387595532951034741052100428552247961627495355202904723558656e+3', '-6.592180609971056972', 4);
    T('1.227102111503512992112190463e+27', '1023', 9);
    T('4.762033342402709161761620616017059035607e+39', '86210167', 5);
    T('1.0526899962331857933889886363874543386374239042307e+34', '6374705.107', 5);
    T('8.3043861038534085480857730707894983974895341475449922046323943309856240866914720437665233e+61','7584431.953', 9);
    T('5.4787291089024129877918140548999194405505068584642366052046899220357579072497678119620025119214429570421679076944647727607529032984620150444553526120397e+151', '72419131838243117', 9);
    T('6.1203577224967e+13', '61203577224967', 1);
    T('-2.6742593337474927167986914400257603e+34', '-82827', 7);
    T('3.490938536648870448335810684618787855050175354916141007545163425555957335829e+75', '247548469', 9);
    T('6.634204312890625e+15', '-95', 8);
    T('-9.171271e+1', '-91.71271', 1);
    T('5.65625765123605544067809018296482219675982471958093705254464e+59', '-9093998202', 6);
    T('3.518743761e+9', '-39', 6);
    T('1e+0', '-5420353171006060062', 0);
    T('5.93548365959371477646876280183482286894691258676416e+26', '840398698.65610156', 3);
    T('3.05021119744369888239209417566581849921624834657246971302432870103990316674041818082170961e+89', '-820455414011161', 6);
    T('3.2696530375744e+11', '571808.8', 2);
    T('7.5976962512577721e+4', '-275.639189', 2);
    T('1.902906615566371112511312310657143013376e+4', '7.1760176', 5);
    T('1.518464318469541965652429354308518011195184133938752135708801e+60', '33317743', 8);
    T('2.13200587786233626937431579432326390786198510967574168821917803373975072742648456640669209424089001389646413824e+92', '2444169945946508.968', 6);
    T('2.22147484530834915265639755288996e+8', '-14904.612860817114', 2);
    T('1.492078384275163354935132877205588401e+36', '-1105217899', 4);
    T('-8.7063159741318127407528723363285743789464660953004359884056134456089069164858324456719131936164339926111276712486987242162931090900645752178720048529774274761e+31', '-3538.91793390607241', 9);
    T('-2.13718835678657210508559759338095142848187133522908355458407047497607984468725768078407393313580673815584615486794665109272908382172581587476837516398285148210296241056810064518192239263716914808770258390627580625498767684881468120398755080836616073670778327363332334149416019505078840253645081995145611223104498789608205428495349224542916922658293232780298744501314853052243306816366148311919597221018850456061527891844821902500302159789434488225792e+125', '-43837.525307806238', 27);
    T('6.6067183217034216835242910438119195789788339709493349261334009277024675568767116522521737472677079426635748603259327817717050761351617610783486587612028204471198915744801916514515768618923473461233676712402753179616115705829361991013069927369820940249862144e+28', '-63581.52864975395137421365938883267810362492', 6);
    T('3.4979655423042386318207008930737637967359866053138998526505479068249644854876701482811561341347105355107695596776164641704434681969517211923069874207364828070027467521475982678466251216378786468053405366050900916905682920934733855175647035328921497719019616796867272653837232009633511918699584196136643032274244642212313884180837128156688381283842300678798685499010807063746785776492193558002382725297304258913361e+62', '860658355.93419774133511168973596889241553275604551215135281', 7);
    T('2.43e+2', '3', 5);
    T('1.1341259418236998571758990926811782086077524757263871423863079766086030645673726799115988350322336274469528121736526520039192242333771040454545272536238688950013853620343757910936524944674880346697808001951639020756685105416346426508389541123017097529060283829118103883424462596062875341251599735321937114348037712815556934333412355272212045907182873831909566029976608100685207198148793211027456e+142', '-77969657.57799062937836', 18);
    T('-3.0028686351108853762578194984762919284354267811356443985818417209466959135294437327385632801156377236905276696018118450080884435331856383482308738976558186955588239868161702337334759747439195397392938819582577110829644281561085620841415307488010481547467558616309027516654381610456065415065653113913265920029870297942479155862011879884534002975129045737955826661617772520189976935251896803639204375428428030749e+58', '-3144155.647351179442684823966727126493552586749', 9);
    T('1.201753576366394485531985516556867241747342491824062593983537252565487129409802473156809713616241453319630220923725569628522288147931699557284946119375022985149050727430775477724150730989713693887368264655960182781852240604839836052804518341222619522792226084796948765314193322314769946060633722353531243331728258530762514582945950418550784e+31', '27540.61485935537676544614263765574563131573245204', 7);
    T('3.37155040678778244943012545203918986928745669295171850695394544390939711301031039954614562269468815944072319615611367890859596537480727045421148118535227427436434942469760108682572253320836320356162649649114580786938367825230347818905248996593031355662611333376e+4', '-3.68110925219710489918033269313682', 8);
    T('-1.157385562678413511184308898769846329872668259271078544423253372613586833033423503791374542602879075132311305638124240471079826030985837104460703581522437292662638238646090030670166015625e+186', '-7784665', 27);
    T('5.7216584769709576633639728428105243709762039786762015993329350164736e+39', '-8697220540.1137988', 4);
    T('3.605080945893673836738483946060969335788919921875e+3', '2.48435', 9);
    T('-1.36240450766284865053749603054987406024313651234922549391481734582321604854817272341996695098685037495173906240642212299142117181990034526636150309000822666516664160854969363947955605850108340373472205128364217170231566407701356867606602700453971968987168686696650431666690097496345998516797552915475256643599049488322243247996920458277719083301674288561258518312772229949771006312491204843363917607453589630126953125e+161', '-3009460118.872263767875045', 17);
    T('-3.8433712973156659829312033596061261378003902713922379084688365643813353176050190118421109324160544394293788636677094907930777317141076841010276342345029919613746057639577024242309077555840877645790649554834063514938596292845365250946008190566774814236719875110926196028928250892121252859736673222597350006754961294389605594296221696e+157', '-271616.155606', 29);
    T('8.880125173267286184293229956929e+30', '-143903', 6);
    T('1.2933690030633029978317864458950890700698349292432976269804895466563737201880939364145851524416113479708518067001609042030295271670130750045121e+14', '-7.62048571', 16);
    T('4.49118544044091499255717578905704133184433268994940305778505055836352880272044837700029253390614132345790751503571600332872e+26', '765808758.66956931514481639498062905791538', 3);
    T('-1.128193052103617375313057818469912250487118062091774976e+29', '-646363.29196', 5);
    T('-3.852842458277771128223673975490918109e+4', '-38528.42458277771128223673975490918109', 1);
    T('4.47317529320544012507678955001652121720459770030481180787405405093720278140090456738247550862238112656659162732461713512251836889602737214995151906586537236627456e+105', '5932.48', 28);
    T('1e+0', '-13.97065588562499', 0);
    T('3.600545803275794460618862354918801e+11', '600045.48188248151', 2);
    T('2.569380011313067359691205257942517240115628560689496193366261158908199626359277748729466511294505469537479575614097255886064905027804073982599354300474897943e+129', '62080.7', 27);
    T('5.818461494039866039726986100228248575472782020261098417686712139322808092966875943575012778303872663798273362332843148103605110550575434794509775866235877112057573531799569068637521409971342270836350080123911710082519119058044202449706075905917444188007277340785315892319100567144025511009943278859996309146349108129588009524947393206783018441414022970002633906397294399684375635228021017590940658971148436650082905540838555648e+102', '6398.880593717242', 27);
    T('1.34971356455758975292534311187056647652453780619004003032217331555704261585125261846040712896856946037144900053640398406244356726653396169868958721875222540716653887768486266703330919098362921e+11', '4.1528548909', 18);
    T('-1.77396388128500114222945974047586725531284545911409446577e+56', '-913', 19);
    T('-1.23493440106890737225414566980096565722401440419412087747352443276574932419181598097379536554237254841727093836607763912665203669933311010821309594074994067411700657618950544400833319706872315553946358432623522564358319794505880365894233180227408889879677018966524371664896e+230', '-90520083747.46', 21);
    T('-2.17525001942067723150294558531604730954322658381064353285681098867925681081506083705983156066163186746091443633027643554689317160117967955473175500964155968361524273945120598810972722792061e+14', '-3.121741', 29);
    T('1e+0', '-562610.16179592649796590823394093366548', 0);
    T('5.5399516545866639955830015026176e+11', '14.94', 10);
    T('1.494189395849269188211255039709933309086424259778445906419464942576097148044216376789735318980392750336285644804638743600807550074206128272345650029255016954321611264002141919840462369550905098763723254901675135787504979910497931539962354019230845564318816091666473025536e+126', '-32698949771.110178432792', 12);
    T('1.1504940871276742926708823617505372960241390892442547940194260102743306128298973371802547471453755938653944600792141533514422490556963535378521856840746722206160260148176604222079226186281680715577736316488196108040509176925372372483300649927835887344415603493563915019264675154039059770309142781930141352329958156926976e+15', '-8.7358521345995835476', 16);

    BigNumber.config({DECIMAL_PLACES: 1000});

    T('5.5626846462680034577255817933310101605480399511558295763833185422180110870347954896357078975312775514101683493275895275128810854038836502721400309634442970528269449838300058261990253686064590901798039126173562593355209381270166265416453973718012279499214790991212515897719252957621869994522193843748736289511290126272884996414561770466127838448395124802899527144151299810833802858809753719892490239782222290074816037776586657834841586939662825734294051183140794537141608771803070715941051121170285190347786926570042246331102750604036185540464179153763503857127117918822547579033069472418242684328083352174724579376695971173152319349449321466491373527284227385153411689217559966957882267024615e-309', 2, -1024);

    BigNumber.config({EXPONENTIAL_AT: 1000});

    T('179769313486231590772930519078902473361797697894230657273430081157732675805500963132708477322407536021120113879871393357658789768814416622492847430639474124377767893424865485276302219601246094119453082952085005768838150682342462881473913110540827237163350510684586298239947245938479716304835356329624224137216', 2, 1024);

    BigNumber.config({ERRORS: true});

    T(2, 2, '1.0');
    T(27, 3, new BigNumber(3));
    T(2048, 2, '1.1e1');
    T(0.0625, 0.5, '   +0.04e2    ');

    assertException(function () {new BigNumber(2).pow(4.4)}, ".pow(4.4)");
    assertException(function () {new BigNumber(2).pow('5.5')}, ".pow('5.5')");
    assertException(function () {new BigNumber(2).pow(-2.1)}, ".pow(-2.1)");
    assertException(function () {new BigNumber(2).pow('-11.5')}, ".pow('-11.5')");
    assertException(function () {new BigNumber(2).pow(0.99)}, ".pow(0.99)");
    assertException(function () {new BigNumber(2).pow('-0.044e2')}, ".pow('-0.044e2')");

    assertException(function () {new BigNumber('12.345').pow(undefined)}, ".pow(undefined)");
    assertException(function () {new BigNumber('12.345').pow(null)}, ".pow(null)");
    assertException(function () {new BigNumber('12.345').pow(true)}, ".pow(true)");
    assertException(function () {new BigNumber('12.345').pow(false)}, ".pow(false)");
    assertException(function () {new BigNumber('12.345').pow(NaN)}, ".pow(NaN)");
    assertException(function () {new BigNumber('12.345').pow('NaN')}, ".pow('NaN')");
    assertException(function () {new BigNumber('12.345').pow([])}, ".pow([])");
    assertException(function () {new BigNumber('12.345').pow({})}, ".pow({})");
    assertException(function () {new BigNumber('12.345').pow('')}, ".pow('')");
    assertException(function () {new BigNumber('12.345').pow(' ')}, ".pow(' ')");
    assertException(function () {new BigNumber('12.345').pow('2.66e+1')}, ".pow('2.66e+1')");
    assertException(function () {new BigNumber('12.345').pow('4e')}, ".pow('4e')");
    assertException(function () {new BigNumber('12.345').pow('hello')}, ".pow('hello')");
    assertException(function () {new BigNumber('12.345').pow('\t')}, ".pow('\t')");
    assertException(function () {new BigNumber('12.345').pow(new Date)}, ".pow(new Date)");
    assertException(function () {new BigNumber('12.345').pow(new RegExp)}, ".pow(new RegExp)");
    assertException(function () {new BigNumber('12.345').pow(function (){})}, ".pow(function (){})");
    assertException(function () {new BigNumber('12.345').pow(7.5)}, ".pow(7.5)");
    assertException(function () {new BigNumber('12.345').pow('-1.123e1')}, ".pow('-1.123e1')");
    assertException(function () {new BigNumber('12.345').pow('-0.01')}, ".pow('-0.01')");
    assertException(function () {new BigNumber('12.345').pow('-1e-1')}, ".pow('-1e-1')");
    assertException(function () {new BigNumber('12.345').pow(Infinity)}, ".pow(Infinity)");
    assertException(function () {new BigNumber('12.345').pow('-Infinity')}, ".pow('-Infinity')");

    assertException(function () {new BigNumber('2').pow(Infinity)}, ".pow(Infinity)");
    assertException(function () {new BigNumber('2').pow('-Infinity')}, ".pow('-Infinity')");

    assertException(function () {new BigNumber(1).pow(MAX_POWER + 1)}, ".pow(MAX_POWER + 1)");

    T(1, 1, MAX_POWER);
    T('16', 2, '4.00000000');
    T('0.0625', 2, -4);

    // As negative exponents involve a division, the result depends on the decimal places and rounding mode specified:

    BigNumber.config({DECIMAL_PLACES: 20, ROUNDING_MODE: 0, EXPONENTIAL_AT: 0});

    T('-5.196101e-14', '-453.8', -5);
    T('1e-20', '8308633', -17);
    T('1e-20', '834', -23);
    T('1.6666666666666666667e-1', '6', -1);
    T('-1e-20', '-71', -29);
    T('9.765625e-4', '-2', -10);
    T('1e-20', '62.8159321', -18);
    T('-6.333754988633e-8', '-2.0557545', -23);
    T('1e-20', '33', -24);
    T('-1e-20', '-5235.7923', -27);
    T('-1.29728108e-12', '-9169', -3);
    T('2.323057312542e-8', '3', -16);
    T('1e-20', '1523620.62', -20);
    T('1e+0', '1', -16);
    T('1e-20', '13', -25);
    T('1.5241579027587259e-4', '-9', -4);
    T('1e-20', '531385.4', -11);
    T('8.4322648810503e-7', '-33', -4);
    T('1e-20', '480546.8181', -13);
    T('-1e-20', '-83591', -9);
    T('1e-20', '457.54', -29);

    BigNumber.config({DECIMAL_PLACES: 40, ROUNDING_MODE: 1});

    T('0e+0', '41', -25);
    T('0e+0', '-26403', -25);
    T('-1.83965573171075e-25', '-28.5112', -17);
    T('1.3846500590693220280355384e-15', '72', -8);
    T('0e+0', '388528.736', -15);
    T('3.934117957191277521704056558e-13', '9', -13);
    T('0e+0', '523512', -7);
    T('0e+0', '65', -30);
    T('0e+0', '-43284153', -10);
    T('0e+0', '825797.7867', -9);
    T('1.45171e-35', '14.49674917', -30);
    T('4.11323023e-32', '713.2735', -11);
    T('6.75896391888332043299090622098e-11', '-2.8992', -22);
    T('0e+0', '9006558087', -9);
    T('2.35898248759e-29', '9', -30);
    T('4.2316349570524877e-24', '-835', -8);
    T('2.306295e-33', '-18', -26);
    T('2.735111227791253388712174e-16', '6', -20);
    T('2.11e-38', '-5855.32565', -10);
    T('0e+0', '-3338', -20);

    BigNumber.config({DECIMAL_PLACES: 2, ROUNDING_MODE: 2});

    T('1e-2', '95636', -15);
    T('1e-2', '802.942', -28);
    T('0e+0', '-31645', -27);
    T('1e-2', '-839791.83', -2);
    T('1e-2', '-2.8383287', -6);
    T('1e-2', '61689855', -2);
    T('1e-2', '7.404284', -20);
    T('1e-2', '898183.004', -11);
    T('1e-2', '-771825.7331', -16);
    T('1e-2', '-33080.258', -12);
    T('2.6e-1', '1.4064', -4);
    T('1e-2', '-21411.3', -8);
    T('1e-2', '-444', -8);
    T('1e-2', '-456', -26);
    T('0e+0', '-84450.1821', -15);
    T('0e+0', '-393', -15);
    T('2e-1', '5', -1);
    T('0e+0', '-7902377.6', -27);
    T('1e-2', '4695.5', -19);
    T('0e+0', '-198', -25);

    BigNumber.config({DECIMAL_PLACES: 50, ROUNDING_MODE: 3});

    T('2.5224883818467056168924068720414334436865e-10', '-62963', -2);
    T('0e+0', '231061', -19);
    T('-1.33787022089433001581801823890617687241090613775e-2', '-74.74566549', -1);
    T('-1e-50', '-64577145', -15);
    T('-9.2e-49', '-7293822', -7);
    T('0e+0', '6853973492', -16);
    T('5.2813339504e-40', '61', -22);
    T('0e+0', '-376283', -30);
    T('0e+0', '728444.059', -27);
    T('-1e-50', '-368631.5722', -23);
    T('4.704e-47', '52631521', -6);
    T('3.57224508459076360310928212162780064014631915e-6', '6', -7);
    T('4.406926288364964664194824943180564261e-14', '469', -5);
    T('1.4551915228366851806640625e-11', '-4', -18);
    T('2.44140625e-4', '8', -4);
    T('0e+0', '32065.3618', -15);
    T('3.7252902984619140625e-9', '-4', -14);
    T('0e+0', '911', -18);
    T('-4e-50', '-226', -21);
    T('0e+0', '6584.1', -26);

    BigNumber.config({DECIMAL_PLACES: 101, ROUNDING_MODE: 4});

    T('2.59086221967070203169073279093835370593438251373865163612424184820637293172653704e-21', '7280902.7', -3);
    T('0e+0', '-864898946', -18);
    T('-6.92056804675353e-87', '-4243081.8', -13);
    T('4.535096358699077993843952445171547362594555572598969066238446195262298595680218449724146e-14', '28041.71', -3);
    T('3.935198147655009521015158343912785620124855179789168611080864740903996079598197380173314783859e-8', '-71', -4);
    T('0e+0', '-22174.6', -24);
    T('-2.99258857055507737893205013175602880569082447633970771174973412126870110482831422782576004e-12', '-44.287279', -7);
    T('0e+0', '470977.3', -28);
    T('0e+0', '1491202647', -13);
    T('7.6416831024439568243705975297501568471970365586580909307933089678e-37', '-63.95', -20);
    T('1.265999063160693261086986795629771233969286862727721581486029700338021749863905100710225474433149e-5', '78989', -1);
    T('0e+0', '4669863', -27);
    T('0e+0', '68895.202', -25);
    T('2.739979564368301888555652244200021058884694833454583774293368045917370205983074157e-20', '8174.2993', -5);
    T('0e+0', '-84777.31264', -24);
    T('-1.61732012052302022569648421623255682672403613813784e-51', '-8072.642', -13);
    T('9.226701330968798029567410873712667474511623e-59', '86', -30);
    T('-4.52671289980544818356121416933990316474525113390656357838131753149833770190909867e-21', '-805.966', -7);
    T('0e+0', '-62548432.8', -25);
    T('1.2208830071779439347812238188621035387222053043582168736013775635140091964690008930489382362e-10', '-90503', -2);

    BigNumber.config({DECIMAL_PLACES: 77, ROUNDING_MODE: 5});

    T('0e+0', '66772.53301', -27);
    T('3.40661780943015419152e-57', '368.73348', -22);
    T('5.601099885e-68', '5310955.57', -10);
    T('4.7583024e-70', '4452', -19);
    T('6.90016290197904917102412334010616805829465226173165163729397834725e-12', '-617', -4);
    T('0e+0', '-107429', -28);
    T('3.31141522e-69', '2654.601393', -20);
    T('0e+0', '29989989', -29);
    T('0e+0', '86252993', -26);
    T('2.526578217600621656723144619575640216185522e-35', '27.64147', -24);
    T('0e+0', '67591', -22);
    T('1.045378162048104058782115833482674315266588732330848431111612381555126e-8', '95659163', -1);
    T('0e+0', '-442', -30);
    T('-1.1560137793600081456221291815651129339464282155352978388454e-19', '-13', -17);
    T('5.58677799866283804991416971229e-48', '8664.361199', -12);
    T('2.279303152972397504104114985089222523674913654876994600999395e-17', '352695', -3);
    T('0e+0', '-1058', -27);
    T('-1.59341022326498995406233262095617818e-42', '-6306.5', -11);
    T('7.1812368392200578e-61', '-5741', -16);
    T('-4.13699730669298062229277514676940165486322151495730381776e-21', '-71.29', -11);

    BigNumber.config({DECIMAL_PLACES: 99, ROUNDING_MODE: 6});

    T('3.50565137890907848896561225350273375702045873464e-52', '282', -21);
    T('2.5679471225603104166379068675724780241775210532444313569170923793768408333314758e-20', '91', -10);
    T('-1.41295214664272077547931481358044650743763505172248724988734056104337e-30', '-57', -17);
    T('-1.6987939248665576039846745813760545846750230030808816733060274510838e-32', '-47', -19);
    T('2.5752032499330351923455285237362946185406503583972342814435847681120726508860096355483248328e-8', '78.94', -4);
    T('0e+0', '171036487', -28);
    T('-2.69509379734942915218278341741127616464435811759556667850711208302182540858969514716020661667e-7', '-3710446', -1);
    T('0e+0', '186138.741', -19);
    T('4.37024265074487259941007405227545667043223030296145105953223847971330338431363825001208e-13', '-5.922', -16);
    T('0e+0', '-68229265', -23);
    T('4.928268737732714614617566124770385834839673627880400464580256256e-35', '-522257', -6);
    T('-2.022506359274224265530332563554654257153128338531e-51', '-7935', -13);
    T('4.113745803364219006888865973815476523221392611155744131624798603545707151167823836246769e-12', '-79', -6);
    T('1.58e-97', '7449', -25);
    T('1.118728735197209316850322750834837614213911286885421936356799842846357177424981773587541512e-9', '19', -7);
    T('2.30221163139014605760922065192648652054327053388946752033272296649e-32', '-433', -12);
    T('-2.267272190443616852157926557261723579010324347431719319355102519288715860877837553741191e-12', '-7612', -3);
    T('3.9978176310189055673570151190218519691760260637389486960244942965129919180092262453024314407659703e-2', '5.001364539', -2);
    T('-7.116521605575577249405567849234479247439447514e-54', '-804088', -9);
    T('8.903896183702542287639673e-75', '-294', -30);
    T('8.07793566946316088741610050849573099185363389551639556884765625e-28', '8', -30);
    T('0e+0', '-81529053.21', -19);
    T('6.5536e-12', '-5', -16);
    T('0e+0', '-393954461', -21);
    T('8.6647113997049045695e-80', '-80579504', -10);
    T('0e+0', '-148396', -20);
    T('0e+0', '-9359', -27);
    T('9.90263871916064254974009912780157486232613002968707056446243636671e-34', '-562.8', -12);
    T('4.3492544377486037018080360300067123809538178904353011549513518579352205e-29', '-53325', -6);
    T('-2.4321805935844144232493980416059793684e-61', '-323865', -11);
    T('7.85857243441531254691813393990245178596730427987853282250959743319065590613e-25', '-52.694', -14);
    T('2.0976497180691992906625338620920916459004443120221572748409028790029453e-29', '-62', -16);
    T('2.364680619496191078868743412738756777204213983827334426304480201936157926992330323938307469e-8', '-6503', -2);
    T('-1.6469853898171534943494115141107888422500959997099727812322359810844298906326507134302e-14', '-34', -9);
    T('-2.8509481595561976509829554676872915624841643712635341972335222140599199298237e-23', '-32285.1', -5);
    T('1.146039e-93', '2086', -28);
    T('1.4679476870451975270004793115264473958785611246906228067920825392511892301766856e-20', '21', -15);
    T('-2.7632000420131244245e-80', '-884.33', -27);
    T('1.28e-5', '5', -7);
    T('0e+0', '465056.3', -25);
    T('-3.385374643100272485246768762282984762e-63', '-8732826', -9);
    T('8.1099148118448025737743300061125240950246836242140801097075869941095902018e-26', '-24.771', -18);
    T('0e+0', '41616', -24);
    T('-4.600396873004638534957365480807567e-66', '-693.02361', -23);
    T('4.743101317987169079986565102514233588322445597602857643390328860354941191426667037e-18', '-54', -10);
    T('1.25187781672508763144717075613420130195292939409113670505758637956935403104656985478217325988983475e-1', '7.988', -1);
    T('0e+0', '-28456394.82', -20);
    T('0e+0', '65420941.8', -28);
    T('6.936158419792753387019531369763661408e-63', '8063848', -9);
    T('7.76868869438579094965029841442548135292087504293659500457724060495468797e-28', '8.0104161', -30);
    T('1.418903330951799073514333098017321859067165230705198534199808126932321545716946e-21', '8899161', -3);
    T('5.5856834219770636694175782782449695904056e-59', '-668794', -10);
    T('1.459854014598540145985401459854014598540145985401459854014598540145985401459854014598540145985401e-3', '685', -1);
    T('4.71680980075571e-85', '-186379', -16);
    T('8.28459140195479963018738579860475551841921130300932e-49', '-2719', -14);
    T('2.67077966735439243101042271765185051646201817465563634664049302592659362084276452403367853161e-7', '1935', -2);
    T('0e+0', '3635', -29);
    T('0e+0', '2554338', -24);
    T('2.5540033620655455476185589070642885151916557690202314061872180456116463869710334413505216200293094e-2', '6.257333', -2);
    T('3.90625e-3', '4', -4);
    T('1.950390958369440429127151141521415628594542650755102593292842679774e-33', '282908.755', -6);
    T('7.055018227999164719047034313928978751513528911694565147012740459916e-32', '19.86', -24);
    T('-1.7926074978022214012194558178183164876939431544374262895950448410755196050436473373785e-14', '-92', -7);
    T('4.17170159620302267465579950126198740043495589902397687214801e-40', '422', -15);
    T('9.094947017729282379150390625e-13', '32', -8);
    T('3.840702799092530418345764876794193332866602413557110156e-45', '-597', -16);
    T('2.86244662448576320862533497809376104296326075367736978e-46', '56.45', -26);
    T('-1.9579266376951475915762057927633263263525293205616726853270204e-38', '-32.2350348', -25);
    T('1.25868312332880899610888409696274260282720434321518772464e-43', '52', -25);
    T('0e+0', '262233', -24);

    BigNumber.config({DECIMAL_PLACES: 0, ROUNDING_MODE: 6});

    T('0e+0', '-8645', -30);
    T('0e+0', '7.24173993', -23);

    // toPower with modulus

    BigNumber.config({EXPONENTIAL_AT: 1E9, ERRORS: false, DECIMAL_PLACES: 17, ROUNDING_MODE: 4});

    T = function (expected, n, exp, mod) {
        assert(String(expected), String(new BigNumber(n).pow(exp, mod)));
    };

    T('0', '-1', -1, '-1');
    T('-1', '-1', -1, '-3');
    T('-1', '-1', -1, '-5.7');
    T('-1', '-1', -1, '-Infinity');
    T('NaN', '-1', -1, '0');
    T('0', '-1', -1, '1');
    T('-1', '-1', -1, '3');
    T('-1', '-1', -1, '5.7');
    T('-1', '-1', -1, 'Infinity');
    T('NaN', '-1', -1, 'NaN');
    T('0', '-1', -3, '-1');
    T('-1', '-1', -3, '-3');
    T('-1', '-1', -3, '-5.7');
    T('-1', '-1', -3, '-Infinity');
    T('NaN', '-1', -3, '0');
    T('0', '-1', -3, '1');
    T('-1', '-1', -3, '3');
    T('-1', '-1', -3, '5.7');
    T('-1', '-1', -3, 'Infinity');
    T('NaN', '-1', -3, 'NaN');
    T('0', '-1', -5.7, '-1');
    T('-1', '-1', -5.7, '-3');
    T('-1', '-1', -5.7, '-5.7');
    T('-1', '-1', -5.7, '-Infinity');
    T('NaN', '-1', -5.7, '0');
    T('0', '-1', -5.7, '1');
    T('-1', '-1', -5.7, '3');
    T('-1', '-1', -5.7, '5.7');
    T('-1', '-1', -5.7, 'Infinity');
    T('NaN', '-1', -5.7, 'NaN');
    T('NaN', '-1', -Infinity, '-1');
    T('NaN', '-1', -Infinity, '-3');
    T('NaN', '-1', -Infinity, '-5.7');
    T('NaN', '-1', -Infinity, '-Infinity');
    T('NaN', '-1', -Infinity, '0');
    T('NaN', '-1', -Infinity, '1');
    T('NaN', '-1', -Infinity, '3');
    T('NaN', '-1', -Infinity, '5.7');
    T('NaN', '-1', -Infinity, 'Infinity');
    T('NaN', '-1', -Infinity, 'NaN');
    T('0', '-1', 0, '-1');
    T('1', '-1', 0, '-3');
    T('1', '-1', 0, '-5.7');
    T('1', '-1', 0, '-Infinity');
    T('NaN', '-1', 0, '0');
    T('0', '-1', 0, '1');
    T('1', '-1', 0, '3');
    T('1', '-1', 0, '5.7');
    T('1', '-1', 0, 'Infinity');
    T('NaN', '-1', 0, 'NaN');
    T('0', '-1', 1, '-1');
    T('-1', '-1', 1, '-3');
    T('-1', '-1', 1, '-5.7');
    T('-1', '-1', 1, '-Infinity');
    T('NaN', '-1', 1, '0');
    T('0', '-1', 1, '1');
    T('-1', '-1', 1, '3');
    T('-1', '-1', 1, '5.7');
    T('-1', '-1', 1, 'Infinity');
    T('NaN', '-1', 1, 'NaN');
    T('0', '-1', 3, '-1');
    T('-1', '-1', 3, '-3');
    T('-1', '-1', 3, '-5.7');
    T('-1', '-1', 3, '-Infinity');
    T('NaN', '-1', 3, '0');
    T('0', '-1', 3, '1');
    T('-1', '-1', 3, '3');
    T('-1', '-1', 3, '5.7');
    T('-1', '-1', 3, 'Infinity');
    T('NaN', '-1', 3, 'NaN');
    T('0', '-1', 5.7, '-1');
    T('-1', '-1', 5.7, '-3');
    T('-1', '-1', 5.7, '-5.7');
    T('-1', '-1', 5.7, '-Infinity');
    T('NaN', '-1', 5.7, '0');
    T('0', '-1', 5.7, '1');
    T('-1', '-1', 5.7, '3');
    T('-1', '-1', 5.7, '5.7');
    T('-1', '-1', 5.7, 'Infinity');
    T('NaN', '-1', 5.7, 'NaN');
    T('NaN', '-1', Infinity, '-1');
    T('NaN', '-1', Infinity, '-3');
    T('NaN', '-1', Infinity, '-5.7');
    T('NaN', '-1', Infinity, '-Infinity');
    T('NaN', '-1', Infinity, '0');
    T('NaN', '-1', Infinity, '1');
    T('NaN', '-1', Infinity, '3');
    T('NaN', '-1', Infinity, '5.7');
    T('NaN', '-1', Infinity, 'Infinity');
    T('NaN', '-1', Infinity, 'NaN');
    T('NaN', '-1', NaN, '-1');
    T('NaN', '-1', NaN, '-3');
    T('NaN', '-1', NaN, '-5.7');
    T('NaN', '-1', NaN, '-Infinity');
    T('NaN', '-1', NaN, '0');
    T('NaN', '-1', NaN, '1');
    T('NaN', '-1', NaN, '3');
    T('NaN', '-1', NaN, '5.7');
    T('NaN', '-1', NaN, 'Infinity');
    T('NaN', '-1', NaN, 'NaN');
    T('-0.33333333333333333', '-3', -1, '-1');
    T('-0.33333333333333333', '-3', -1, '-3');
    T('-0.33333333333333333', '-3', -1, '-5.7');
    T('-0.33333333333333333', '-3', -1, '-Infinity');
    T('NaN', '-3', -1, '0');
    T('-0.33333333333333333', '-3', -1, '1');
    T('-0.33333333333333333', '-3', -1, '3');
    T('-0.33333333333333333', '-3', -1, '5.7');
    T('-0.33333333333333333', '-3', -1, 'Infinity');
    T('NaN', '-3', -1, 'NaN');
    T('-0.03703703703703704', '-3', -3, '-1');
    T('-0.03703703703703704', '-3', -3, '-3');
    T('-0.03703703703703704', '-3', -3, '-5.7');
    T('-0.03703703703703704', '-3', -3, '-Infinity');
    T('NaN', '-3', -3, '0');
    T('-0.03703703703703704', '-3', -3, '1');
    T('-0.03703703703703704', '-3', -3, '3');
    T('-0.03703703703703704', '-3', -3, '5.7');
    T('-0.03703703703703704', '-3', -3, 'Infinity');
    T('NaN', '-3', -3, 'NaN');
    T('-0.00411522633744856', '-3', -5.7, '-1');
    T('-0.00411522633744856', '-3', -5.7, '-3');
    T('-0.00411522633744856', '-3', -5.7, '-5.7');
    T('-0.00411522633744856', '-3', -5.7, '-Infinity');
    T('NaN', '-3', -5.7, '0');
    T('-0.00411522633744856', '-3', -5.7, '1');
    T('-0.00411522633744856', '-3', -5.7, '3');
    T('-0.00411522633744856', '-3', -5.7, '5.7');
    T('-0.00411522633744856', '-3', -5.7, 'Infinity');
    T('NaN', '-3', -5.7, 'NaN');
    T('0', '-3', -Infinity, '-1');
    T('0', '-3', -Infinity, '-3');
    T('0', '-3', -Infinity, '-5.7');
    T('0', '-3', -Infinity, '-Infinity');
    T('NaN', '-3', -Infinity, '0');
    T('0', '-3', -Infinity, '1');
    T('0', '-3', -Infinity, '3');
    T('0', '-3', -Infinity, '5.7');
    T('0', '-3', -Infinity, 'Infinity');
    T('NaN', '-3', -Infinity, 'NaN');
    T('0', '-3', 0, '-1');
    T('1', '-3', 0, '-3');
    T('1', '-3', 0, '-5.7');
    T('1', '-3', 0, '-Infinity');
    T('NaN', '-3', 0, '0');
    T('0', '-3', 0, '1');
    T('1', '-3', 0, '3');
    T('1', '-3', 0, '5.7');
    T('1', '-3', 0, 'Infinity');
    T('NaN', '-3', 0, 'NaN');
    T('0', '-3', 1, '-1');
    T('0', '-3', 1, '-3');
    T('-3', '-3', 1, '-5.7');
    T('-3', '-3', 1, '-Infinity');
    T('NaN', '-3', 1, '0');
    T('0', '-3', 1, '1');
    T('0', '-3', 1, '3');
    T('-3', '-3', 1, '5.7');
    T('-3', '-3', 1, 'Infinity');
    T('NaN', '-3', 1, 'NaN');
    T('0', '-3', 3, '-1');
    T('0', '-3', 3, '-3');
    T('-4.2', '-3', 3, '-5.7');
    T('-27', '-3', 3, '-Infinity');
    T('NaN', '-3', 3, '0');
    T('0', '-3', 3, '1');
    T('0', '-3', 3, '3');
    T('-4.2', '-3', 3, '5.7');
    T('-27', '-3', 3, 'Infinity');
    T('NaN', '-3', 3, 'NaN');
    T('0', '-3', 5.7, '-1');
    T('0', '-3', 5.7, '-3');
    T('-3.6', '-3', 5.7, '-5.7');
    T('-243', '-3', 5.7, '-Infinity');
    T('NaN', '-3', 5.7, '0');
    T('0', '-3', 5.7, '1');
    T('0', '-3', 5.7, '3');
    T('-3.6', '-3', 5.7, '5.7');
    T('-243', '-3', 5.7, 'Infinity');
    T('NaN', '-3', 5.7, 'NaN');
    T('NaN', '-3', Infinity, '-1');
    T('NaN', '-3', Infinity, '-3');
    T('NaN', '-3', Infinity, '-5.7');
    T('NaN', '-3', Infinity, '-Infinity');
    T('NaN', '-3', Infinity, '0');
    T('NaN', '-3', Infinity, '1');
    T('NaN', '-3', Infinity, '3');
    T('NaN', '-3', Infinity, '5.7');
    T('NaN', '-3', Infinity, 'Infinity');
    T('NaN', '-3', Infinity, 'NaN');
    T('NaN', '-3', NaN, '-1');
    T('NaN', '-3', NaN, '-3');
    T('NaN', '-3', NaN, '-5.7');
    T('NaN', '-3', NaN, '-Infinity');
    T('NaN', '-3', NaN, '0');
    T('NaN', '-3', NaN, '1');
    T('NaN', '-3', NaN, '3');
    T('NaN', '-3', NaN, '5.7');
    T('NaN', '-3', NaN, 'Infinity');
    T('NaN', '-3', NaN, 'NaN');
    T('-0.17543859649122807', '-5.7', -1, '-1');
    T('-0.17543859649122807', '-5.7', -1, '-3');
    T('-0.17543859649122807', '-5.7', -1, '-5.7');
    T('-0.17543859649122807', '-5.7', -1, '-Infinity');
    T('NaN', '-5.7', -1, '0');
    T('-0.17543859649122807', '-5.7', -1, '1');
    T('-0.17543859649122807', '-5.7', -1, '3');
    T('-0.17543859649122807', '-5.7', -1, '5.7');
    T('-0.17543859649122807', '-5.7', -1, 'Infinity');
    T('NaN', '-5.7', -1, 'NaN');
    T('-0.00539977212961613', '-5.7', -3, '-1');
    T('-0.00539977212961613', '-5.7', -3, '-3');
    T('-0.00539977212961613', '-5.7', -3, '-5.7');
    T('-0.00539977212961613', '-5.7', -3, '-Infinity');
    T('NaN', '-5.7', -3, '0');
    T('-0.00539977212961613', '-5.7', -3, '1');
    T('-0.00539977212961613', '-5.7', -3, '3');
    T('-0.00539977212961613', '-5.7', -3, '5.7');
    T('-0.00539977212961613', '-5.7', -3, 'Infinity');
    T('NaN', '-5.7', -3, 'NaN');
    T('-0.00016619797259514', '-5.7', -5.7, '-1');
    T('-0.00016619797259514', '-5.7', -5.7, '-3');
    T('-0.00016619797259514', '-5.7', -5.7, '-5.7');
    T('-0.00016619797259514', '-5.7', -5.7, '-Infinity');
    T('NaN', '-5.7', -5.7, '0');
    T('-0.00016619797259514', '-5.7', -5.7, '1');
    T('-0.00016619797259514', '-5.7', -5.7, '3');
    T('-0.00016619797259514', '-5.7', -5.7, '5.7');
    T('-0.00016619797259514', '-5.7', -5.7, 'Infinity');
    T('NaN', '-5.7', -5.7, 'NaN');
    T('0', '-5.7', -Infinity, '-1');
    T('0', '-5.7', -Infinity, '-3');
    T('0', '-5.7', -Infinity, '-5.7');
    T('0', '-5.7', -Infinity, '-Infinity');
    T('NaN', '-5.7', -Infinity, '0');
    T('0', '-5.7', -Infinity, '1');
    T('0', '-5.7', -Infinity, '3');
    T('0', '-5.7', -Infinity, '5.7');
    T('0', '-5.7', -Infinity, 'Infinity');
    T('NaN', '-5.7', -Infinity, 'NaN');
    T('0', '-5.7', 0, '-1');
    T('1', '-5.7', 0, '-3');
    T('1', '-5.7', 0, '-5.7');
    T('1', '-5.7', 0, '-Infinity');
    T('NaN', '-5.7', 0, '0');
    T('0', '-5.7', 0, '1');
    T('1', '-5.7', 0, '3');
    T('1', '-5.7', 0, '5.7');
    T('1', '-5.7', 0, 'Infinity');
    T('NaN', '-5.7', 0, 'NaN');
    T('-0.7', '-5.7', 1, '-1');
    T('-2.7', '-5.7', 1, '-3');
    T('0', '-5.7', 1, '-5.7');
    T('-5.7', '-5.7', 1, '-Infinity');
    T('NaN', '-5.7', 1, '0');
    T('-0.7', '-5.7', 1, '1');
    T('-2.7', '-5.7', 1, '3');
    T('0', '-5.7', 1, '5.7');
    T('-5.7', '-5.7', 1, 'Infinity');
    T('NaN', '-5.7', 1, 'NaN');
    T('-0.193', '-5.7', 3, '-1');
    T('-2.193', '-5.7', 3, '-3');
    T('-2.793', '-5.7', 3, '-5.7');
    T('-185.193', '-5.7', 3, '-Infinity');
    T('NaN', '-5.7', 3, '0');
    T('-0.193', '-5.7', 3, '1');
    T('-2.193', '-5.7', 3, '3');
    T('-2.793', '-5.7', 3, '5.7');
    T('-185.193', '-5.7', 3, 'Infinity');
    T('NaN', '-5.7', 3, 'NaN');
    T('-0.92057', '-5.7', 5.7, '-1');
    T('-1.92057', '-5.7', 5.7, '-3');
    T('-3.42057', '-5.7', 5.7, '-5.7');
    T('-6016.92057', '-5.7', 5.7, '-Infinity');
    T('NaN', '-5.7', 5.7, '0');
    T('-0.92057', '-5.7', 5.7, '1');
    T('-1.92057', '-5.7', 5.7, '3');
    T('-3.42057', '-5.7', 5.7, '5.7');
    T('-6016.92057', '-5.7', 5.7, 'Infinity');
    T('NaN', '-5.7', 5.7, 'NaN');
    T('NaN', '-5.7', Infinity, '-1');
    T('NaN', '-5.7', Infinity, '-3');
    T('NaN', '-5.7', Infinity, '-5.7');
    T('NaN', '-5.7', Infinity, '-Infinity');
    T('NaN', '-5.7', Infinity, '0');
    T('NaN', '-5.7', Infinity, '1');
    T('NaN', '-5.7', Infinity, '3');
    T('NaN', '-5.7', Infinity, '5.7');
    T('NaN', '-5.7', Infinity, 'Infinity');
    T('NaN', '-5.7', Infinity, 'NaN');
    T('NaN', '-5.7', NaN, '-1');
    T('NaN', '-5.7', NaN, '-3');
    T('NaN', '-5.7', NaN, '-5.7');
    T('NaN', '-5.7', NaN, '-Infinity');
    T('NaN', '-5.7', NaN, '0');
    T('NaN', '-5.7', NaN, '1');
    T('NaN', '-5.7', NaN, '3');
    T('NaN', '-5.7', NaN, '5.7');
    T('NaN', '-5.7', NaN, 'Infinity');
    T('NaN', '-5.7', NaN, 'NaN');
    T('0', '-Infinity', -1, '-1');
    T('0', '-Infinity', -1, '-3');
    T('0', '-Infinity', -1, '-5.7');
    T('0', '-Infinity', -1, '-Infinity');
    T('NaN', '-Infinity', -1, '0');
    T('0', '-Infinity', -1, '1');
    T('0', '-Infinity', -1, '3');
    T('0', '-Infinity', -1, '5.7');
    T('0', '-Infinity', -1, 'Infinity');
    T('NaN', '-Infinity', -1, 'NaN');
    T('0', '-Infinity', -3, '-1');
    T('0', '-Infinity', -3, '-3');
    T('0', '-Infinity', -3, '-5.7');
    T('0', '-Infinity', -3, '-Infinity');
    T('NaN', '-Infinity', -3, '0');
    T('0', '-Infinity', -3, '1');
    T('0', '-Infinity', -3, '3');
    T('0', '-Infinity', -3, '5.7');
    T('0', '-Infinity', -3, 'Infinity');
    T('NaN', '-Infinity', -3, 'NaN');
    T('0', '-Infinity', -5.7, '-1');
    T('0', '-Infinity', -5.7, '-3');
    T('0', '-Infinity', -5.7, '-5.7');
    T('0', '-Infinity', -5.7, '-Infinity');
    T('NaN', '-Infinity', -5.7, '0');
    T('0', '-Infinity', -5.7, '1');
    T('0', '-Infinity', -5.7, '3');
    T('0', '-Infinity', -5.7, '5.7');
    T('0', '-Infinity', -5.7, 'Infinity');
    T('NaN', '-Infinity', -5.7, 'NaN');
    T('0', '-Infinity', -Infinity, '-1');
    T('0', '-Infinity', -Infinity, '-3');
    T('0', '-Infinity', -Infinity, '-5.7');
    T('0', '-Infinity', -Infinity, '-Infinity');
    T('NaN', '-Infinity', -Infinity, '0');
    T('0', '-Infinity', -Infinity, '1');
    T('0', '-Infinity', -Infinity, '3');
    T('0', '-Infinity', -Infinity, '5.7');
    T('0', '-Infinity', -Infinity, 'Infinity');
    T('NaN', '-Infinity', -Infinity, 'NaN');
    T('0', '-Infinity', 0, '-1');
    T('1', '-Infinity', 0, '-3');
    T('1', '-Infinity', 0, '-5.7');
    T('1', '-Infinity', 0, '-Infinity');
    T('NaN', '-Infinity', 0, '0');
    T('0', '-Infinity', 0, '1');
    T('1', '-Infinity', 0, '3');
    T('1', '-Infinity', 0, '5.7');
    T('1', '-Infinity', 0, 'Infinity');
    T('NaN', '-Infinity', 0, 'NaN');
    T('NaN', '-Infinity', 1, '-1');
    T('NaN', '-Infinity', 1, '-3');
    T('NaN', '-Infinity', 1, '-5.7');
    T('NaN', '-Infinity', 1, '-Infinity');
    T('NaN', '-Infinity', 1, '0');
    T('NaN', '-Infinity', 1, '1');
    T('NaN', '-Infinity', 1, '3');
    T('NaN', '-Infinity', 1, '5.7');
    T('NaN', '-Infinity', 1, 'Infinity');
    T('NaN', '-Infinity', 1, 'NaN');
    T('NaN', '-Infinity', 3, '-1');
    T('NaN', '-Infinity', 3, '-3');
    T('NaN', '-Infinity', 3, '-5.7');
    T('NaN', '-Infinity', 3, '-Infinity');
    T('NaN', '-Infinity', 3, '0');
    T('NaN', '-Infinity', 3, '1');
    T('NaN', '-Infinity', 3, '3');
    T('NaN', '-Infinity', 3, '5.7');
    T('NaN', '-Infinity', 3, 'Infinity');
    T('NaN', '-Infinity', 3, 'NaN');
    T('NaN', '-Infinity', 5.7, '-1');
    T('NaN', '-Infinity', 5.7, '-3');
    T('NaN', '-Infinity', 5.7, '-5.7');
    T('NaN', '-Infinity', 5.7, '-Infinity');
    T('NaN', '-Infinity', 5.7, '0');
    T('NaN', '-Infinity', 5.7, '1');
    T('NaN', '-Infinity', 5.7, '3');
    T('NaN', '-Infinity', 5.7, '5.7');
    T('NaN', '-Infinity', 5.7, 'Infinity');
    T('NaN', '-Infinity', 5.7, 'NaN');
    T('NaN', '-Infinity', Infinity, '-1');
    T('NaN', '-Infinity', Infinity, '-3');
    T('NaN', '-Infinity', Infinity, '-5.7');
    T('NaN', '-Infinity', Infinity, '-Infinity');
    T('NaN', '-Infinity', Infinity, '0');
    T('NaN', '-Infinity', Infinity, '1');
    T('NaN', '-Infinity', Infinity, '3');
    T('NaN', '-Infinity', Infinity, '5.7');
    T('NaN', '-Infinity', Infinity, 'Infinity');
    T('NaN', '-Infinity', Infinity, 'NaN');
    T('NaN', '-Infinity', NaN, '-1');
    T('NaN', '-Infinity', NaN, '-3');
    T('NaN', '-Infinity', NaN, '-5.7');
    T('NaN', '-Infinity', NaN, '-Infinity');
    T('NaN', '-Infinity', NaN, '0');
    T('NaN', '-Infinity', NaN, '1');
    T('NaN', '-Infinity', NaN, '3');
    T('NaN', '-Infinity', NaN, '5.7');
    T('NaN', '-Infinity', NaN, 'Infinity');
    T('NaN', '-Infinity', NaN, 'NaN');
    T('NaN', '0', -1, '-1');
    T('NaN', '0', -1, '-3');
    T('NaN', '0', -1, '-5.7');
    T('NaN', '0', -1, '-Infinity');
    T('NaN', '0', -1, '0');
    T('NaN', '0', -1, '1');
    T('NaN', '0', -1, '3');
    T('NaN', '0', -1, '5.7');
    T('NaN', '0', -1, 'Infinity');
    T('NaN', '0', -1, 'NaN');
    T('NaN', '0', -3, '-1');
    T('NaN', '0', -3, '-3');
    T('NaN', '0', -3, '-5.7');
    T('NaN', '0', -3, '-Infinity');
    T('NaN', '0', -3, '0');
    T('NaN', '0', -3, '1');
    T('NaN', '0', -3, '3');
    T('NaN', '0', -3, '5.7');
    T('NaN', '0', -3, 'Infinity');
    T('NaN', '0', -3, 'NaN');
    T('NaN', '0', -5.7, '-1');
    T('NaN', '0', -5.7, '-3');
    T('NaN', '0', -5.7, '-5.7');
    T('NaN', '0', -5.7, '-Infinity');
    T('NaN', '0', -5.7, '0');
    T('NaN', '0', -5.7, '1');
    T('NaN', '0', -5.7, '3');
    T('NaN', '0', -5.7, '5.7');
    T('NaN', '0', -5.7, 'Infinity');
    T('NaN', '0', -5.7, 'NaN');
    T('NaN', '0', -Infinity, '-1');
    T('NaN', '0', -Infinity, '-3');
    T('NaN', '0', -Infinity, '-5.7');
    T('NaN', '0', -Infinity, '-Infinity');
    T('NaN', '0', -Infinity, '0');
    T('NaN', '0', -Infinity, '1');
    T('NaN', '0', -Infinity, '3');
    T('NaN', '0', -Infinity, '5.7');
    T('NaN', '0', -Infinity, 'Infinity');
    T('NaN', '0', -Infinity, 'NaN');
    T('0', '0', 0, '-1');
    T('1', '0', 0, '-3');
    T('1', '0', 0, '-5.7');
    T('1', '0', 0, '-Infinity');
    T('NaN', '0', 0, '0');
    T('0', '0', 0, '1');
    T('1', '0', 0, '3');
    T('1', '0', 0, '5.7');
    T('1', '0', 0, 'Infinity');
    T('NaN', '0', 0, 'NaN');
    T('0', '0', 1, '-1');
    T('0', '0', 1, '-3');
    T('0', '0', 1, '-5.7');
    T('0', '0', 1, '-Infinity');
    T('NaN', '0', 1, '0');
    T('0', '0', 1, '1');
    T('0', '0', 1, '3');
    T('0', '0', 1, '5.7');
    T('0', '0', 1, 'Infinity');
    T('NaN', '0', 1, 'NaN');
    T('0', '0', 3, '-1');
    T('0', '0', 3, '-3');
    T('0', '0', 3, '-5.7');
    T('0', '0', 3, '-Infinity');
    T('NaN', '0', 3, '0');
    T('0', '0', 3, '1');
    T('0', '0', 3, '3');
    T('0', '0', 3, '5.7');
    T('0', '0', 3, 'Infinity');
    T('NaN', '0', 3, 'NaN');
    T('0', '0', 5.7, '-1');
    T('0', '0', 5.7, '-3');
    T('0', '0', 5.7, '-5.7');
    T('0', '0', 5.7, '-Infinity');
    T('NaN', '0', 5.7, '0');
    T('0', '0', 5.7, '1');
    T('0', '0', 5.7, '3');
    T('0', '0', 5.7, '5.7');
    T('0', '0', 5.7, 'Infinity');
    T('NaN', '0', 5.7, 'NaN');
    T('0', '0', Infinity, '-1');
    T('0', '0', Infinity, '-3');
    T('0', '0', Infinity, '-5.7');
    T('0', '0', Infinity, '-Infinity');
    T('NaN', '0', Infinity, '0');
    T('0', '0', Infinity, '1');
    T('0', '0', Infinity, '3');
    T('0', '0', Infinity, '5.7');
    T('0', '0', Infinity, 'Infinity');
    T('NaN', '0', Infinity, 'NaN');
    T('NaN', '0', NaN, '-1');
    T('NaN', '0', NaN, '-3');
    T('NaN', '0', NaN, '-5.7');
    T('NaN', '0', NaN, '-Infinity');
    T('NaN', '0', NaN, '0');
    T('NaN', '0', NaN, '1');
    T('NaN', '0', NaN, '3');
    T('NaN', '0', NaN, '5.7');
    T('NaN', '0', NaN, 'Infinity');
    T('NaN', '0', NaN, 'NaN');
    T('0', '1', -1, '-1');
    T('1', '1', -1, '-3');
    T('1', '1', -1, '-5.7');
    T('1', '1', -1, '-Infinity');
    T('NaN', '1', -1, '0');
    T('0', '1', -1, '1');
    T('1', '1', -1, '3');
    T('1', '1', -1, '5.7');
    T('1', '1', -1, 'Infinity');
    T('NaN', '1', -1, 'NaN');
    T('0', '1', -3, '-1');
    T('1', '1', -3, '-3');
    T('1', '1', -3, '-5.7');
    T('1', '1', -3, '-Infinity');
    T('NaN', '1', -3, '0');
    T('0', '1', -3, '1');
    T('1', '1', -3, '3');
    T('1', '1', -3, '5.7');
    T('1', '1', -3, 'Infinity');
    T('NaN', '1', -3, 'NaN');
    T('0', '1', -5.7, '-1');
    T('1', '1', -5.7, '-3');
    T('1', '1', -5.7, '-5.7');
    T('1', '1', -5.7, '-Infinity');
    T('NaN', '1', -5.7, '0');
    T('0', '1', -5.7, '1');
    T('1', '1', -5.7, '3');
    T('1', '1', -5.7, '5.7');
    T('1', '1', -5.7, 'Infinity');
    T('NaN', '1', -5.7, 'NaN');
    T('NaN', '1', -Infinity, '-1');
    T('NaN', '1', -Infinity, '-3');
    T('NaN', '1', -Infinity, '-5.7');
    T('NaN', '1', -Infinity, '-Infinity');
    T('NaN', '1', -Infinity, '0');
    T('NaN', '1', -Infinity, '1');
    T('NaN', '1', -Infinity, '3');
    T('NaN', '1', -Infinity, '5.7');
    T('NaN', '1', -Infinity, 'Infinity');
    T('NaN', '1', -Infinity, 'NaN');
    T('0', '1', 0, '-1');
    T('1', '1', 0, '-3');
    T('1', '1', 0, '-5.7');
    T('1', '1', 0, '-Infinity');
    T('NaN', '1', 0, '0');
    T('0', '1', 0, '1');
    T('1', '1', 0, '3');
    T('1', '1', 0, '5.7');
    T('1', '1', 0, 'Infinity');
    T('NaN', '1', 0, 'NaN');
    T('0', '1', 1, '-1');
    T('1', '1', 1, '-3');
    T('1', '1', 1, '-5.7');
    T('1', '1', 1, '-Infinity');
    T('NaN', '1', 1, '0');
    T('0', '1', 1, '1');
    T('1', '1', 1, '3');
    T('1', '1', 1, '5.7');
    T('1', '1', 1, 'Infinity');
    T('NaN', '1', 1, 'NaN');
    T('0', '1', 3, '-1');
    T('1', '1', 3, '-3');
    T('1', '1', 3, '-5.7');
    T('1', '1', 3, '-Infinity');
    T('NaN', '1', 3, '0');
    T('0', '1', 3, '1');
    T('1', '1', 3, '3');
    T('1', '1', 3, '5.7');
    T('1', '1', 3, 'Infinity');
    T('NaN', '1', 3, 'NaN');
    T('0', '1', 5.7, '-1');
    T('1', '1', 5.7, '-3');
    T('1', '1', 5.7, '-5.7');
    T('1', '1', 5.7, '-Infinity');
    T('NaN', '1', 5.7, '0');
    T('0', '1', 5.7, '1');
    T('1', '1', 5.7, '3');
    T('1', '1', 5.7, '5.7');
    T('1', '1', 5.7, 'Infinity');
    T('NaN', '1', 5.7, 'NaN');
    T('NaN', '1', Infinity, '-1');
    T('NaN', '1', Infinity, '-3');
    T('NaN', '1', Infinity, '-5.7');
    T('NaN', '1', Infinity, '-Infinity');
    T('NaN', '1', Infinity, '0');
    T('NaN', '1', Infinity, '1');
    T('NaN', '1', Infinity, '3');
    T('NaN', '1', Infinity, '5.7');
    T('NaN', '1', Infinity, 'Infinity');
    T('NaN', '1', Infinity, 'NaN');
    T('NaN', '1', NaN, '-1');
    T('NaN', '1', NaN, '-3');
    T('NaN', '1', NaN, '-5.7');
    T('NaN', '1', NaN, '-Infinity');
    T('NaN', '1', NaN, '0');
    T('NaN', '1', NaN, '1');
    T('NaN', '1', NaN, '3');
    T('NaN', '1', NaN, '5.7');
    T('NaN', '1', NaN, 'Infinity');
    T('NaN', '1', NaN, 'NaN');
    T('0.33333333333333333', '3', -1, '-1');
    T('0.33333333333333333', '3', -1, '-3');
    T('0.33333333333333333', '3', -1, '-5.7');
    T('0.33333333333333333', '3', -1, '-Infinity');
    T('NaN', '3', -1, '0');
    T('0.33333333333333333', '3', -1, '1');
    T('0.33333333333333333', '3', -1, '3');
    T('0.33333333333333333', '3', -1, '5.7');
    T('0.33333333333333333', '3', -1, 'Infinity');
    T('NaN', '3', -1, 'NaN');
    T('0.03703703703703704', '3', -3, '-1');
    T('0.03703703703703704', '3', -3, '-3');
    T('0.03703703703703704', '3', -3, '-5.7');
    T('0.03703703703703704', '3', -3, '-Infinity');
    T('NaN', '3', -3, '0');
    T('0.03703703703703704', '3', -3, '1');
    T('0.03703703703703704', '3', -3, '3');
    T('0.03703703703703704', '3', -3, '5.7');
    T('0.03703703703703704', '3', -3, 'Infinity');
    T('NaN', '3', -3, 'NaN');
    T('0.00411522633744856', '3', -5.7, '-1');
    T('0.00411522633744856', '3', -5.7, '-3');
    T('0.00411522633744856', '3', -5.7, '-5.7');
    T('0.00411522633744856', '3', -5.7, '-Infinity');
    T('NaN', '3', -5.7, '0');
    T('0.00411522633744856', '3', -5.7, '1');
    T('0.00411522633744856', '3', -5.7, '3');
    T('0.00411522633744856', '3', -5.7, '5.7');
    T('0.00411522633744856', '3', -5.7, 'Infinity');
    T('NaN', '3', -5.7, 'NaN');
    T('0', '3', -Infinity, '-1');
    T('0', '3', -Infinity, '-3');
    T('0', '3', -Infinity, '-5.7');
    T('0', '3', -Infinity, '-Infinity');
    T('NaN', '3', -Infinity, '0');
    T('0', '3', -Infinity, '1');
    T('0', '3', -Infinity, '3');
    T('0', '3', -Infinity, '5.7');
    T('0', '3', -Infinity, 'Infinity');
    T('NaN', '3', -Infinity, 'NaN');
    T('0', '3', 0, '-1');
    T('1', '3', 0, '-3');
    T('1', '3', 0, '-5.7');
    T('1', '3', 0, '-Infinity');
    T('NaN', '3', 0, '0');
    T('0', '3', 0, '1');
    T('1', '3', 0, '3');
    T('1', '3', 0, '5.7');
    T('1', '3', 0, 'Infinity');
    T('NaN', '3', 0, 'NaN');
    T('0', '3', 1, '-1');
    T('0', '3', 1, '-3');
    T('3', '3', 1, '-5.7');
    T('3', '3', 1, '-Infinity');
    T('NaN', '3', 1, '0');
    T('0', '3', 1, '1');
    T('0', '3', 1, '3');
    T('3', '3', 1, '5.7');
    T('3', '3', 1, 'Infinity');
    T('NaN', '3', 1, 'NaN');
    T('0', '3', 3, '-1');
    T('0', '3', 3, '-3');
    T('4.2', '3', 3, '-5.7');
    T('27', '3', 3, '-Infinity');
    T('NaN', '3', 3, '0');
    T('0', '3', 3, '1');
    T('0', '3', 3, '3');
    T('4.2', '3', 3, '5.7');
    T('27', '3', 3, 'Infinity');
    T('NaN', '3', 3, 'NaN');
    T('0', '3', 5.7, '-1');
    T('0', '3', 5.7, '-3');
    T('3.6', '3', 5.7, '-5.7');
    T('243', '3', 5.7, '-Infinity');
    T('NaN', '3', 5.7, '0');
    T('0', '3', 5.7, '1');
    T('0', '3', 5.7, '3');
    T('3.6', '3', 5.7, '5.7');
    T('243', '3', 5.7, 'Infinity');
    T('NaN', '3', 5.7, 'NaN');
    T('NaN', '3', Infinity, '-1');
    T('NaN', '3', Infinity, '-3');
    T('NaN', '3', Infinity, '-5.7');
    T('NaN', '3', Infinity, '-Infinity');
    T('NaN', '3', Infinity, '0');
    T('NaN', '3', Infinity, '1');
    T('NaN', '3', Infinity, '3');
    T('NaN', '3', Infinity, '5.7');
    T('NaN', '3', Infinity, 'Infinity');
    T('NaN', '3', Infinity, 'NaN');
    T('NaN', '3', NaN, '-1');
    T('NaN', '3', NaN, '-3');
    T('NaN', '3', NaN, '-5.7');
    T('NaN', '3', NaN, '-Infinity');
    T('NaN', '3', NaN, '0');
    T('NaN', '3', NaN, '1');
    T('NaN', '3', NaN, '3');
    T('NaN', '3', NaN, '5.7');
    T('NaN', '3', NaN, 'Infinity');
    T('NaN', '3', NaN, 'NaN');
    T('0.17543859649122807', '5.7', -1, '-1');
    T('0.17543859649122807', '5.7', -1, '-3');
    T('0.17543859649122807', '5.7', -1, '-5.7');
    T('0.17543859649122807', '5.7', -1, '-Infinity');
    T('NaN', '5.7', -1, '0');
    T('0.17543859649122807', '5.7', -1, '1');
    T('0.17543859649122807', '5.7', -1, '3');
    T('0.17543859649122807', '5.7', -1, '5.7');
    T('0.17543859649122807', '5.7', -1, 'Infinity');
    T('NaN', '5.7', -1, 'NaN');
    T('0.00539977212961613', '5.7', -3, '-1');
    T('0.00539977212961613', '5.7', -3, '-3');
    T('0.00539977212961613', '5.7', -3, '-5.7');
    T('0.00539977212961613', '5.7', -3, '-Infinity');
    T('NaN', '5.7', -3, '0');
    T('0.00539977212961613', '5.7', -3, '1');
    T('0.00539977212961613', '5.7', -3, '3');
    T('0.00539977212961613', '5.7', -3, '5.7');
    T('0.00539977212961613', '5.7', -3, 'Infinity');
    T('NaN', '5.7', -3, 'NaN');
    T('0.00016619797259514', '5.7', -5.7, '-1');
    T('0.00016619797259514', '5.7', -5.7, '-3');
    T('0.00016619797259514', '5.7', -5.7, '-5.7');
    T('0.00016619797259514', '5.7', -5.7, '-Infinity');
    T('NaN', '5.7', -5.7, '0');
    T('0.00016619797259514', '5.7', -5.7, '1');
    T('0.00016619797259514', '5.7', -5.7, '3');
    T('0.00016619797259514', '5.7', -5.7, '5.7');
    T('0.00016619797259514', '5.7', -5.7, 'Infinity');
    T('NaN', '5.7', -5.7, 'NaN');
    T('0', '5.7', -Infinity, '-1');
    T('0', '5.7', -Infinity, '-3');
    T('0', '5.7', -Infinity, '-5.7');
    T('0', '5.7', -Infinity, '-Infinity');
    T('NaN', '5.7', -Infinity, '0');
    T('0', '5.7', -Infinity, '1');
    T('0', '5.7', -Infinity, '3');
    T('0', '5.7', -Infinity, '5.7');
    T('0', '5.7', -Infinity, 'Infinity');
    T('NaN', '5.7', -Infinity, 'NaN');
    T('0', '5.7', 0, '-1');
    T('1', '5.7', 0, '-3');
    T('1', '5.7', 0, '-5.7');
    T('1', '5.7', 0, '-Infinity');
    T('NaN', '5.7', 0, '0');
    T('0', '5.7', 0, '1');
    T('1', '5.7', 0, '3');
    T('1', '5.7', 0, '5.7');
    T('1', '5.7', 0, 'Infinity');
    T('NaN', '5.7', 0, 'NaN');
    T('0.7', '5.7', 1, '-1');
    T('2.7', '5.7', 1, '-3');
    T('0', '5.7', 1, '-5.7');
    T('5.7', '5.7', 1, '-Infinity');
    T('NaN', '5.7', 1, '0');
    T('0.7', '5.7', 1, '1');
    T('2.7', '5.7', 1, '3');
    T('0', '5.7', 1, '5.7');
    T('5.7', '5.7', 1, 'Infinity');
    T('NaN', '5.7', 1, 'NaN');
    T('0.193', '5.7', 3, '-1');
    T('2.193', '5.7', 3, '-3');
    T('2.793', '5.7', 3, '-5.7');
    T('185.193', '5.7', 3, '-Infinity');
    T('NaN', '5.7', 3, '0');
    T('0.193', '5.7', 3, '1');
    T('2.193', '5.7', 3, '3');
    T('2.793', '5.7', 3, '5.7');
    T('185.193', '5.7', 3, 'Infinity');
    T('NaN', '5.7', 3, 'NaN');
    T('0.92057', '5.7', 5.7, '-1');
    T('1.92057', '5.7', 5.7, '-3');
    T('3.42057', '5.7', 5.7, '-5.7');
    T('6016.92057', '5.7', 5.7, '-Infinity');
    T('NaN', '5.7', 5.7, '0');
    T('0.92057', '5.7', 5.7, '1');
    T('1.92057', '5.7', 5.7, '3');
    T('3.42057', '5.7', 5.7, '5.7');
    T('6016.92057', '5.7', 5.7, 'Infinity');
    T('NaN', '5.7', 5.7, 'NaN');
    T('NaN', '5.7', Infinity, '-1');
    T('NaN', '5.7', Infinity, '-3');
    T('NaN', '5.7', Infinity, '-5.7');
    T('NaN', '5.7', Infinity, '-Infinity');
    T('NaN', '5.7', Infinity, '0');
    T('NaN', '5.7', Infinity, '1');
    T('NaN', '5.7', Infinity, '3');
    T('NaN', '5.7', Infinity, '5.7');
    T('NaN', '5.7', Infinity, 'Infinity');
    T('NaN', '5.7', Infinity, 'NaN');
    T('NaN', '5.7', NaN, '-1');
    T('NaN', '5.7', NaN, '-3');
    T('NaN', '5.7', NaN, '-5.7');
    T('NaN', '5.7', NaN, '-Infinity');
    T('NaN', '5.7', NaN, '0');
    T('NaN', '5.7', NaN, '1');
    T('NaN', '5.7', NaN, '3');
    T('NaN', '5.7', NaN, '5.7');
    T('NaN', '5.7', NaN, 'Infinity');
    T('NaN', '5.7', NaN, 'NaN');
    T('0', 'Infinity', -1, '-1');
    T('0', 'Infinity', -1, '-3');
    T('0', 'Infinity', -1, '-5.7');
    T('0', 'Infinity', -1, '-Infinity');
    T('NaN', 'Infinity', -1, '0');
    T('0', 'Infinity', -1, '1');
    T('0', 'Infinity', -1, '3');
    T('0', 'Infinity', -1, '5.7');
    T('0', 'Infinity', -1, 'Infinity');
    T('NaN', 'Infinity', -1, 'NaN');
    T('0', 'Infinity', -3, '-1');
    T('0', 'Infinity', -3, '-3');
    T('0', 'Infinity', -3, '-5.7');
    T('0', 'Infinity', -3, '-Infinity');
    T('NaN', 'Infinity', -3, '0');
    T('0', 'Infinity', -3, '1');
    T('0', 'Infinity', -3, '3');
    T('0', 'Infinity', -3, '5.7');
    T('0', 'Infinity', -3, 'Infinity');
    T('NaN', 'Infinity', -3, 'NaN');
    T('0', 'Infinity', -5.7, '-1');
    T('0', 'Infinity', -5.7, '-3');
    T('0', 'Infinity', -5.7, '-5.7');
    T('0', 'Infinity', -5.7, '-Infinity');
    T('NaN', 'Infinity', -5.7, '0');
    T('0', 'Infinity', -5.7, '1');
    T('0', 'Infinity', -5.7, '3');
    T('0', 'Infinity', -5.7, '5.7');
    T('0', 'Infinity', -5.7, 'Infinity');
    T('NaN', 'Infinity', -5.7, 'NaN');
    T('0', 'Infinity', -Infinity, '-1');
    T('0', 'Infinity', -Infinity, '-3');
    T('0', 'Infinity', -Infinity, '-5.7');
    T('0', 'Infinity', -Infinity, '-Infinity');
    T('NaN', 'Infinity', -Infinity, '0');
    T('0', 'Infinity', -Infinity, '1');
    T('0', 'Infinity', -Infinity, '3');
    T('0', 'Infinity', -Infinity, '5.7');
    T('0', 'Infinity', -Infinity, 'Infinity');
    T('NaN', 'Infinity', -Infinity, 'NaN');
    T('0', 'Infinity', 0, '-1');
    T('1', 'Infinity', 0, '-3');
    T('1', 'Infinity', 0, '-5.7');
    T('1', 'Infinity', 0, '-Infinity');
    T('NaN', 'Infinity', 0, '0');
    T('0', 'Infinity', 0, '1');
    T('1', 'Infinity', 0, '3');
    T('1', 'Infinity', 0, '5.7');
    T('1', 'Infinity', 0, 'Infinity');
    T('NaN', 'Infinity', 0, 'NaN');
    T('NaN', 'Infinity', 1, '-1');
    T('NaN', 'Infinity', 1, '-3');
    T('NaN', 'Infinity', 1, '-5.7');
    T('NaN', 'Infinity', 1, '-Infinity');
    T('NaN', 'Infinity', 1, '0');
    T('NaN', 'Infinity', 1, '1');
    T('NaN', 'Infinity', 1, '3');
    T('NaN', 'Infinity', 1, '5.7');
    T('NaN', 'Infinity', 1, 'Infinity');
    T('NaN', 'Infinity', 1, 'NaN');
    T('NaN', 'Infinity', 3, '-1');
    T('NaN', 'Infinity', 3, '-3');
    T('NaN', 'Infinity', 3, '-5.7');
    T('NaN', 'Infinity', 3, '-Infinity');
    T('NaN', 'Infinity', 3, '0');
    T('NaN', 'Infinity', 3, '1');
    T('NaN', 'Infinity', 3, '3');
    T('NaN', 'Infinity', 3, '5.7');
    T('NaN', 'Infinity', 3, 'Infinity');
    T('NaN', 'Infinity', 3, 'NaN');
    T('NaN', 'Infinity', 5.7, '-1');
    T('NaN', 'Infinity', 5.7, '-3');
    T('NaN', 'Infinity', 5.7, '-5.7');
    T('NaN', 'Infinity', 5.7, '-Infinity');
    T('NaN', 'Infinity', 5.7, '0');
    T('NaN', 'Infinity', 5.7, '1');
    T('NaN', 'Infinity', 5.7, '3');
    T('NaN', 'Infinity', 5.7, '5.7');
    T('NaN', 'Infinity', 5.7, 'Infinity');
    T('NaN', 'Infinity', 5.7, 'NaN');
    T('NaN', 'Infinity', Infinity, '-1');
    T('NaN', 'Infinity', Infinity, '-3');
    T('NaN', 'Infinity', Infinity, '-5.7');
    T('NaN', 'Infinity', Infinity, '-Infinity');
    T('NaN', 'Infinity', Infinity, '0');
    T('NaN', 'Infinity', Infinity, '1');
    T('NaN', 'Infinity', Infinity, '3');
    T('NaN', 'Infinity', Infinity, '5.7');
    T('NaN', 'Infinity', Infinity, 'Infinity');
    T('NaN', 'Infinity', Infinity, 'NaN');
    T('NaN', 'Infinity', NaN, '-1');
    T('NaN', 'Infinity', NaN, '-3');
    T('NaN', 'Infinity', NaN, '-5.7');
    T('NaN', 'Infinity', NaN, '-Infinity');
    T('NaN', 'Infinity', NaN, '0');
    T('NaN', 'Infinity', NaN, '1');
    T('NaN', 'Infinity', NaN, '3');
    T('NaN', 'Infinity', NaN, '5.7');
    T('NaN', 'Infinity', NaN, 'Infinity');
    T('NaN', 'Infinity', NaN, 'NaN');
    T('NaN', 'NaN', -1, '-1');
    T('NaN', 'NaN', -1, '-3');
    T('NaN', 'NaN', -1, '-5.7');
    T('NaN', 'NaN', -1, '-Infinity');
    T('NaN', 'NaN', -1, '0');
    T('NaN', 'NaN', -1, '1');
    T('NaN', 'NaN', -1, '3');
    T('NaN', 'NaN', -1, '5.7');
    T('NaN', 'NaN', -1, 'Infinity');
    T('NaN', 'NaN', -1, 'NaN');
    T('NaN', 'NaN', -3, '-1');
    T('NaN', 'NaN', -3, '-3');
    T('NaN', 'NaN', -3, '-5.7');
    T('NaN', 'NaN', -3, '-Infinity');
    T('NaN', 'NaN', -3, '0');
    T('NaN', 'NaN', -3, '1');
    T('NaN', 'NaN', -3, '3');
    T('NaN', 'NaN', -3, '5.7');
    T('NaN', 'NaN', -3, 'Infinity');
    T('NaN', 'NaN', -3, 'NaN');
    T('NaN', 'NaN', -5.7, '-1');
    T('NaN', 'NaN', -5.7, '-3');
    T('NaN', 'NaN', -5.7, '-5.7');
    T('NaN', 'NaN', -5.7, '-Infinity');
    T('NaN', 'NaN', -5.7, '0');
    T('NaN', 'NaN', -5.7, '1');
    T('NaN', 'NaN', -5.7, '3');
    T('NaN', 'NaN', -5.7, '5.7');
    T('NaN', 'NaN', -5.7, 'Infinity');
    T('NaN', 'NaN', -5.7, 'NaN');
    T('NaN', 'NaN', -Infinity, '-1');
    T('NaN', 'NaN', -Infinity, '-3');
    T('NaN', 'NaN', -Infinity, '-5.7');
    T('NaN', 'NaN', -Infinity, '-Infinity');
    T('NaN', 'NaN', -Infinity, '0');
    T('NaN', 'NaN', -Infinity, '1');
    T('NaN', 'NaN', -Infinity, '3');
    T('NaN', 'NaN', -Infinity, '5.7');
    T('NaN', 'NaN', -Infinity, 'Infinity');
    T('NaN', 'NaN', -Infinity, 'NaN');
    T('0', 'NaN', 0, '-1');
    T('1', 'NaN', 0, '-3');
    T('1', 'NaN', 0, '-5.7');
    T('1', 'NaN', 0, '-Infinity');
    T('NaN', 'NaN', 0, '0');
    T('0', 'NaN', 0, '1');
    T('1', 'NaN', 0, '3');
    T('1', 'NaN', 0, '5.7');
    T('1', 'NaN', 0, 'Infinity');
    T('NaN', 'NaN', 0, 'NaN');
    T('NaN', 'NaN', 1, '-1');
    T('NaN', 'NaN', 1, '-3');
    T('NaN', 'NaN', 1, '-5.7');
    T('NaN', 'NaN', 1, '-Infinity');
    T('NaN', 'NaN', 1, '0');
    T('NaN', 'NaN', 1, '1');
    T('NaN', 'NaN', 1, '3');
    T('NaN', 'NaN', 1, '5.7');
    T('NaN', 'NaN', 1, 'Infinity');
    T('NaN', 'NaN', 1, 'NaN');
    T('NaN', 'NaN', 3, '-1');
    T('NaN', 'NaN', 3, '-3');
    T('NaN', 'NaN', 3, '-5.7');
    T('NaN', 'NaN', 3, '-Infinity');
    T('NaN', 'NaN', 3, '0');
    T('NaN', 'NaN', 3, '1');
    T('NaN', 'NaN', 3, '3');
    T('NaN', 'NaN', 3, '5.7');
    T('NaN', 'NaN', 3, 'Infinity');
    T('NaN', 'NaN', 3, 'NaN');
    T('NaN', 'NaN', 5.7, '-1');
    T('NaN', 'NaN', 5.7, '-3');
    T('NaN', 'NaN', 5.7, '-5.7');
    T('NaN', 'NaN', 5.7, '-Infinity');
    T('NaN', 'NaN', 5.7, '0');
    T('NaN', 'NaN', 5.7, '1');
    T('NaN', 'NaN', 5.7, '3');
    T('NaN', 'NaN', 5.7, '5.7');
    T('NaN', 'NaN', 5.7, 'Infinity');
    T('NaN', 'NaN', 5.7, 'NaN');
    T('NaN', 'NaN', Infinity, '-1');
    T('NaN', 'NaN', Infinity, '-3');
    T('NaN', 'NaN', Infinity, '-5.7');
    T('NaN', 'NaN', Infinity, '-Infinity');
    T('NaN', 'NaN', Infinity, '0');
    T('NaN', 'NaN', Infinity, '1');
    T('NaN', 'NaN', Infinity, '3');
    T('NaN', 'NaN', Infinity, '5.7');
    T('NaN', 'NaN', Infinity, 'Infinity');
    T('NaN', 'NaN', Infinity, 'NaN');
    T('NaN', 'NaN', NaN, '-1');
    T('NaN', 'NaN', NaN, '-3');
    T('NaN', 'NaN', NaN, '-5.7');
    T('NaN', 'NaN', NaN, '-Infinity');
    T('NaN', 'NaN', NaN, '0');
    T('NaN', 'NaN', NaN, '1');
    T('NaN', 'NaN', NaN, '3');
    T('NaN', 'NaN', NaN, '5.7');
    T('NaN', 'NaN', NaN, 'Infinity');
    T('NaN', 'NaN', NaN, 'NaN');

    T('1', '8431', 398674124, '6');
    T('7110', '1929384198', 564, '12627');
    T('1', '421', 969026510, '50');
    T('18279', '503236662', 7865967220250664, '146017');
    T('212243931', '26283', 559977785516, '438161990');
    T('3986011', '309905614', 303248194641, '6707349');
    T('18', '255', 235075187670817, '33');
    T('757606', '85279549', 1, '4971879');
    T('493022001', '1221', 339684, '1227120430');
    T('4928', '308', 862705533, '245399');

    T('1', '2655619', 712076, '33');
    T('3249', '36597', 10, '4814');
    T('24693574779382', '64756899620069', 929, '31837505671417');
    T('4', '16138', 25, '6');
    T('803688476775598957', '23434498424', 1004504153400006, '1552170429770308303');
    T('2372169341110', '626', 2909120945494472, '2627700430731');
    T('763331421', '8920455', 5401, '887776639');
    T('43484276', '95602644', 47268, '79441540');
    T('5184', '552490380863549826', 771646862212, '48411');
    T('249062220104231038', '8165474757220190', 48336800, '275469974740761006');

    T('4533624538711', '71095', 84895497, '8511056291816');
    T('179616962799673689600', '30', 5993842570087, '3846357299605272267200');
    T('2635639014', '948664855786186969995462', 991073259299, '6492710818');
    T('25207202', '7062946640447932397849934745', 9179924349024, '131689081');
    T('46215552578796', '9226321147260318', 465662072, '82890635248305');
    T('2381', '67677914606828678309445778836', 379808346, '6545');
    T('3194', '9430054', 33266765435198, '8086');
    T('489391901079659061061615631695', '61', 532791104437751, '575349759754184186263817284386');
    T('53895639062204578043630854379', '7625677354305052466464542', 384495887925, '92309679232802956732606796453');
    T('2556', '2556', 474, '4260');

    T('7454833596345226', '6627640799561604435379549901208146893914', 738172551780, '9927201293379475');
    T('180534305804480057935271006266', '30118923384576', 1724057305, '217106906249791126796683884757');
    T('40198606849673093876708023432368286', '46018377546879392378', 9583035, '97268554010071292786749920599930246');
    T('119594913310192', '3876146908085398', 6310557700917, '460635614054751');
    T('124164038822218003447569', '6324437543304', 5145398258501, '185457603447664741003595');
    T('3090727470454298372533496615424', '9043956512017564199358', 812419, '24686499729814129954773423938808');
    T('196432807299128', '85644819061206872677771002', 2149475, '2390467883263880');
    T('21038128225272112935718937889', '10201699689699', 975439792, '552736938894180564012881397228');
    T('48815091877022922', '819772006700', 27, '92801123950463822');
    T('3428327713', '4783442049811351657', 763298409739419, '4611091548');

    T('13332959', '50223861853248559483171199', 433657796167153, '14302445');
    T('1671780604656974600625', '5262390185', 237568007998, '2996619244329632717320');
    T('163214427140868907633628729899294561', '69729', 5547, '348074057682137097650725573316845631');
    T('1431482931095017928787410350478464311370783', '324036022719', 741179, '1926651417082887424580613612432908230284256');
    T('2066482089348451048557491058810559698377957', '485499449712405070079531', 26619244686, '5198031511981114390451723050087908159382379');
    T('230132104292091', '51108', 72, '300839508488483');
    T('77425469643841456', '683612', 1358712956, '87679547089722516');
    T('19704951483426394285654701761857', '648220087103027691231', 951, '51871772124455644390524635948707');
    T('49664718770671', '90754235859251032779427547330178026902879259879149', 56465, '88662672119247');
    T('5850177638872804389781733677', '43833993637925773026276875024526870636161009', 18680, '7143233335762749555439764903');

    T('148766490582493984', '637303893625341624581231866', 80158754948186, '277200469688808288');
    T('461156083195', '7410440324698529628156609999534999680075097798739066525', 5, '497057036562');
    T('64916883599108541736776193', '612994182961138850333413794352070459841441623351', 60767786944, '98752274335195294154105792');
    T('10540459703800', '64273975302329668592061012520727877789083279767399625069872', 47928, '11135116378808');
    T('51323389055913840647', '8812207212196306768767581580252769675186591389274669815', 318348861090, '64421250202609658903');
    T('29757808741764208562941153498135845503485664433483002816', '867758291086087441092815266', 87972360, '35296135515471412857388210439806001268345767694668654815');
    T('12216915747251127577023017426296915245567146', '5', 630681763, '99691500608302115525875055527858805680711469');
    T('39256229035626291077932595741439945459', '884788691267481486036549962208626653751575112606547', 395, '74611250766351101157218135270578882456');
    T('1566585758309290833383165985121', '5518513093291031', 84076954280784, '3268027301397216624639210698650');
    T('23244239019466456', '726961684629206', 44302850171068, '25825428077885724');

    T('576202045348', '2', 86, '1993487574093');
    T('214887739391325311421203390352640473539330809857', '530920695787544208524163946341577905732500119397445776874457', 92074232456, '298234073905137329443605619833742500919938582584');
    T('65987551079170180991', '15727645504455027240711713610867307628685087714766285006179', 5415994008046072, '98093654946405312851');
    T('235755348', '97895915562280371546579179382663613522080', 6864027244330, '2914481959');
    T('25', '2385833031024324749500218699098192497329791351618465248', 76, '43');
    T('60978709428230482055794', '633426981727026457056243235', 2573352678, '68892104926357370406321');
    T('2205305143872260015', '68666113504059034715448345087743557663421209680169', 966912, '91335033334933977914');
    T('1773010325526997630667956162969199', '50236266447725', 1011051, '1815196566132403387261756476229021');
    T('21651713309307397647418967909940906495359918883652297119748967664610', '2116690032950796724931519528504418', 56377918425202, '72530284342935017506382060155224822052591722104751066167988598475621');
    T('424342025184319173350839545801827251126937962398522383162485691860', '7887123908856684587220682858582136758248493694412325668235690671821222', 78263291426, '606621751096553135780007544328033731512868463777097140904367236076');

    T('263494656066740331512505119529952169855176897305760218583136', '66196629771558316669009697587278986920237144718211014622', 82975788577, '276995756096460875347969999989044425131478330494164864612372');
    T('3233881936397621380726', '54662037196287984527444181218823835370969115893018818932457726246761952463863247', 7978980, '8616858825085971720675');
    T('638258565634367399562728805308653408227469872980835260151', '271260887707883899257906058181413820', 83084825904971, '911022277353533247041821679426958551977223586921062184599');
    T('4929004454697476334020070969568981614084143361021890853851227932576', '52595660817705323426336739112374', 82, '84588413633625400636331211734490327208866927410873351741019781325200');
    T('1318', '52222440815620913414283909971331735692219726452073974', 31969869852707, '1743');
    T('14198375573817127297631', '8216191940103223703953921417098533793768836648251711587683532494485283139', 1148021101052, '33840057260506501760990');
    T('21060027143743696716588231173430226739686253140085459901957546396844213341461', '929861823123874973319343426480313452550331811680088489136411', 2724613045067, '41704674452090069450347101739468855285700640202119997571378502239544273783510');
    T('4443992', '15541126534638718868684758085506345122523183909428377824025206962', 260040175408, '25089386');
    T('69108491603450670146194732329929457413687007477897110583837256484', '403009613248326350380989310363291291192333753504', 62476, '79005045873240215031140013298308923107449817612581444589282697702');
    T('17236718288049536002844483635703049705951027651789224052123191026028473', '2094473204798008783226910505910335184920132719939788443730442642435545859265', 45508995, '82091072999723696400188424215432751998089672273801746338001474566467528');

    T('789232110429757050346575421789740938023359099985821439285059579046301609631489', '81969532915716492451825251138141557551221798800067', 9374132001566, '6151970522916703120074639237189829128382929132326931047111863128679908831947160');
    T('309967512158129367241503417', '5036401011755282976501651891', 8, '579409771218647975769237062');
    T('1005177723064444533676930215329318398408005233744147409253092051884720439872045691', '34143228467709965637133329397909447638094686489341571291939014959027173', 5908294874, '1948118218782552504717113674377900788819505447144982442644571967109889114978555094');
    T('2674983886895543103385933707806021292069592798960403352624723368307164005844515', '630121260445826051', 543924007975, '5393527392056050378946327047321386307679397116474720103645794120383930696591597');
    T('34282424398363091595264858358407576878748130711657', '2465523511004443813182991155444943123114440980033643073427530034623', 40040, '53406881113973714252481843982237377938283267787704');
    T('56697362235159637435394616', '13578442879343644788292568854324', 3213346, '64112922323162194609121303');
    T('1719024107812073742324652892209591438916478681803041822032', '741729184', 15707757516391, '2841684181135919788850393385226690862963456286187597289743');
    T('77235876127631492414224275788312346586010414227977024436242959514761441403681', '9779956774206680777933282330040609', 848907522, '80469529418181750667389985753151188249008367167903257882428958848063106959425');
    T('818237407', '8647307267571638478062009936214941341887730783', 72115, '868503288');
    T('16656646685269365251966621943766388357818276618200', '90718594921015553491234286035964355061165665', 413647015, '3608930136060789794621435623631749086641253604872025');

    T('4200771934599417598043614250678407408742790405352069', '13887167479352054887893441108734976256983355293630326688937701987456059128531183047285855439', 4878021, '5174782460386365636916415446777806773439498506957659');
    T('229321232335228179325607264299805007162580253510', '88842733861662854338989696299185544317548185717287774', 1884553431751485, '511271768851992491206847877157605201694070726149');
    T('2782154434549194158424475393312481728934235', '548551139346831388798410342333799565824514031', 21440165013, '28690744453818561525474569067146373783311411');
    T('231885589525566300', '41452536764738341533654876175985845652783844060455944861412', 504, '882872312288902804');
    T('3526070056991550931813749027', '67955640858421297373955388629490294119793212927955296181464396292269', 56248954153, '30688081810027133462513567962');
    T('131014', '2089950142999411392048663', 51275898, '658471');
    T('1563352013492150412336688693007169330972429493240982436018814920117063643496545477705204209349110', '9397910631929603594996958575220998774', 84406064, '4909019378673255436084701488614137765928330710577696608363266704298169838609295204783331338958374');
    T('1044297548212903', '42525084216745706252327916418', 42291, '6457910534907511');
    T('405167196487546464980337225371932484221676531564558020674296714011859111057380', '3400756652952034869667491286822430771347620199472', 26886850142633, '611549802518085362254974981183236948088206027646591513228667566697306287449399');
    T('7794452159255315351790298031593675', '3374396899581350491330711739601350455629429581990670970624900984868802087092615683985175', 4600499328250313, '9412479648183932289355382618210050');

    log('\n ' + passed + ' of ' + total + ' tests passed in ' + (+new Date() - start) + ' ms \n');
    return [passed, total];
})(this.BigNumber);
if (typeof module !== 'undefined' && module.exports) module.exports = count;

/*

Notes:

n to the power of 1 is n
Anything to the power of 0 is 1
Math.pow(2, null) = 1                  // NaN for BigNumber
Math.pow(2, undefined) = NaN
Math.pow(2, NaN) = NaN
Math.pow(2, 'rferf') = NaN
Math.pow(2, []) = 1                    // NaN for BigNumber
Math.pow(2, {}) = NaN                  // NaN for BigNumber
Math.pow(2, 1e4) = Infinity
Math.pow(2, -1e4) = 0
Math.pow(2, 1e-16) = 1
Math.pow(2, -1e-17) = 1
A negative number to a non-integer power is NaN
0 to negative power is Infinity

--------------------------------------------------------------------------------

http://es5.github.com/#x15.8.2.13
15.8.2.13 pow (x, y)
Returns an implementation-dependent approximation to the result of raising x to the power y.

If y is NaN, the result is NaN.
If y is +0, the result is 1, even if x is NaN.
If y is -0, the result is 1, even if x is NaN.
If x is NaN and y is nonzero, the result is NaN.
If abs(x)>1 and y is +8, the result is +8.
If abs(x)>1 and y is -8, the result is +0.
If abs(x)==1 and y is +8, the result is NaN.
If abs(x)==1 and y is -8, the result is NaN.
If abs(x)<1 and y is +8, the result is +0.
If abs(x)<1 and y is -8, the result is +8.
If x is +8 and y>0, the result is +8.
If x is +8 and y<0, the result is +0.
If x is -8 and y>0 and y is an odd integer, the result is -8.
If x is -8 and y>0 and y is not an odd integer, the result is +8.
If x is -8 and y<0 and y is an odd integer, the result is -0.
If x is -8 and y<0 and y is not an odd integer, the result is +0.
If x is +0 and y>0, the result is +0.
If x is +0 and y<0, the result is +8.
If x is -0 and y>0 and y is an odd integer, the result is -0.
If x is -0 and y>0 and y is not an odd integer, the result is +0.
If x is -0 and y<0 and y is an odd integer, the result is -8.
If x is -0 and y<0 and y is not an odd integer, the result is +8.
If x<0 and x is finite and y is finite and y is not an integer, the result is NaN.

*/
