var count = (function toExponential(BigNumber) {
    var start = +new Date(),
        log,
        error,
        undefined,
        passed = 0,
        total = 0,
        MAX = 1e9;

    if (typeof window === 'undefined') {
        log = console.log;
        error = console.error;
    } else {
        log = function (str) { document.body.innerHTML += str.replace('\n', '<br>') };
        error = function (str) { document.body.innerHTML += '<div style="color: red">' +
          str.replace('\n', '<br>') + '</div>' };
    }

    if (!BigNumber && typeof require === 'function') BigNumber = require('../bignumber');

    function assert(expected, actual) {
        total++;
        if (expected !== actual) {
           error('\n Test number: ' + total + ' failed');
           error(' Expected: ' + expected);
           error(' Actual:   ' + actual);
           //process.exit();
        }
        else {
            passed++;
            //log('\n Expected and actual: ' + actual);
        }
    }

    function assertException(func, message) {
        var actual;
        total++;
        try {
            func();
        } catch (e) {
            actual = e;
        }
        if (actual && actual.name == 'BigNumber Error') {
            passed++;
            //log('\n Expected and actual: ' + actual);
        } else {
            error('\n Test number: ' + total + ' failed');
            error('\n Expected: ' + message + ' to raise a BigNumber Error.');
            error(' Actual:   ' + (actual || 'no exception'));
            //process.exit();
        }
    }

    function T(expected, value, decimalPlaces){
        assert(String(expected), new BigNumber(value).toExponential(decimalPlaces));
    }

    log('\n Testing toExponential...');

    BigNumber.config({
        DECIMAL_PLACES: 20,
        ROUNDING_MODE: 4,
        ERRORS: true,
        RANGE: 1E9,
        EXPONENTIAL_AT: 1E9
    });

    T('1e+0', 1, undefined);
    T('1.1e+1', 11, undefined);
    T('1.12e+2', 112, undefined);

    T('1e+0', 1, 0);
    T('1e+1', 11, 0);
    T('1e+2', 112, 0);
    T('1.0e+0', 1, 1);
    T('1.1e+1', 11, 1);
    T('1.1e+2', 112, 1);
    T('1.00e+0', 1, 2);
    T('1.10e+1', 11, 2);
    T('1.12e+2', 112, 2);
    T('1.000e+0', 1, 3);
    T('1.100e+1', 11, 3);
    T('1.120e+2', 112, 3);
    T('1e-1', 0.1, undefined);
    T('1.1e-1', 0.11, undefined);
    T('1.12e-1', 0.112, undefined);
    T('1e-1', 0.1, 0);
    T('1e-1', 0.11, 0);
    T('1e-1', 0.112, 0);
    T('1.0e-1', 0.1, 1);
    T('1.1e-1', 0.11, 1);
    T('1.1e-1', 0.112, 1);
    T('1.00e-1', 0.1, 2);
    T('1.10e-1', 0.11, 2);
    T('1.12e-1', 0.112, 2);
    T('1.000e-1', 0.1, 3);
    T('1.100e-1', 0.11, 3);
    T('1.120e-1', 0.112, 3);

    T('-1e+0', -1, undefined);
    T('-1.1e+1', -11, undefined);
    T('-1.12e+2', -112, undefined);
    T('-1e+0', -1, 0);
    T('-1e+1', -11, 0);
    T('-1e+2', -112, 0);
    T('-1.0e+0', -1, 1);
    T('-1.1e+1', -11, 1);
    T('-1.1e+2', -112, 1);
    T('-1.00e+0', -1, 2);
    T('-1.10e+1', -11, 2);
    T('-1.12e+2', -112, 2);
    T('-1.000e+0', -1, 3);
    T('-1.100e+1', -11, 3);
    T('-1.120e+2', -112, 3);
    T('-1e-1', -0.1, undefined);
    T('-1.1e-1', -0.11, undefined);
    T('-1.12e-1', -0.112, undefined);
    T('-1e-1', -0.1, 0);
    T('-1e-1', -0.11, 0);
    T('-1e-1', -0.112, 0);
    T('-1.0e-1', -0.1, 1);
    T('-1.1e-1', -0.11, 1);
    T('-1.1e-1', -0.112, 1);
    T('-1.00e-1', -0.1, 2);
    T('-1.10e-1', -0.11, 2);
    T('-1.12e-1', -0.112, 2);
    T('-1.000e-1', -0.1, 3);
    T('-1.100e-1', -0.11, 3);
    T('-1.120e-1', -0.112, 3);

    T('NaN', NaN, undefined);
    T('NaN', -NaN, 2);
    T('Infinity', Infinity, undefined);
    T('Infinity', Infinity, 10);
    T('-Infinity', -Infinity, 0);
    T('-Infinity', -Infinity, 1);
    T('0e+0', 0, undefined);
    T('0e+0', -0, undefined);
    T('-5.0e-1', -0.5, 1);
    T('0.00e+0', 0, 2);
    T('1e+1', 11.2356, 0);
    T('1.1236e+1', 11.2356, 4);
    T('1.1236e-4', 0.000112356, 4);
    T('-1.1236e-4', -0.000112356, 4);
    T('1.12356e-4', 0.000112356, undefined);
    T('-1.12356e-4', -0.000112356, undefined);

    T('1.00e+0', 0.99976, 2);
    T('1.00e+2', 99.9979, 2);
    T('1.00e+5', '99991.27839', 2);
    T('1.000e+2', '99.999', 3);
    T('1.000e+7', '9999512.8', 3);
    T('1.00e+9', '999702726', 2);
    T('1.000e+3', '999.964717', 3);

    BigNumber.config({ROUNDING_MODE: 0});

    T('-5.3453435435e+8', '-53453.435435E4', undefined);
    T('-8.8254658100092746334967191957167916942544e+17', '-882546581000927463.34967191957167916942543286', 40);
    T('-4.794121828559674450610889008537305783490457e-9', '-0.00000000479412182855967445061088900853730578349045628396662493370334888944406719979291547717079', 42);
    T('3.6149e+33', '3614844933096444884855774574994631.0106397808', 4);
    T('5.582954000000000e-12', '0.000000000005582954', 15);
    T('-3.88740271991885914774802363151163005925700000000000000000e-24', '-0.000000000000000000000003887402719918859147748023631511630059257', 56);
    T('-6.87079645872437277236913190316306435274902613151676421e-20', '-0.00000000000000000006870796458724372772369131903163064352749026131516764202733298056929060151437', 53);
    T('3.8181874087278104533737313621586530711155405443818235503358935045749888900678e+35', '381818740872781045337373136215865307.11155405443818235503358935045749888900677769535371296063', 76);
    T('-7.11375441247784115059912118586189732891550e+20', '-711375441247784115059.91211858618973289154952986', 41);
    T('6.5783e+24', '6578282366667302135641813.7249573246362582', 4);
    T('6.000000000000000000000e-20', '0.00000000000000000006', 21);
    T('-5.3799672107777e+13', '-53799672107777', 13);
    T('-6.949e-23', '-0.00000000000000000000006948849870723', 3);
    T('-8.073585184316705309757817e+25', '-80735851843167053097578169.623098209399637950843019109979317', 24);
    T('-4.2956483e-12', '-0.0000000000042956482047751', 7);
    T('-6.1162155721951440801240154580459651167830311633e+15', '-6116215572195144.0801240154580459651167830311633', 46);
    T('-7.263265230767e-21', '-0.000000000000000000007263265230766073544739', 12);
    T('-2.3013406115701776345891815e+18', '-2301340611570177634.5891814408272260224632', 25);
    T('-6.0299793663e+30', '-6029979366232747481609455093247.705001183386474', 10);
    T('-2.97544304967e+21', '-2975443049668038511693.75547178021412', 11);
    T('-4.1471192639160032e+10', '-41471192639.1600315953295208128538183546', 16);
    T('-3.61201776785294987e+27', '-3612017767852949869824542721.1595027189', 17);
    T('-6.9983494044506115115e+17', '-699834940445061151.14676', 19);
    T('-1.4580700323629245038287e+20', '-145807003236292450382.86958174', 22);
    T('-8.54e+10', '-85390930743', 2);
    T('-2.715269856970717e+19', '-27152698569707163435', 15);
    T('-5.67681004e+20', '-567681003999187989540.627303416332508226276308449233', 8);
    T('-2.06809e+27', '-2068085084336615438842661921.06985539576218546524301', 5);
    T('-2.92273061370427012250925e+14', '-292273061370427.0122509240087955481845060858420928631', 23);
    T('-4.3355e-17', '-0.0000000000000000433542', 4);
    T('-3.491610942584e+21', '-3491610942583064798345', 12);
    T('-8.701944635985129980360621e+16', '-87019446359851299.8036062002728328', 24);
    T('-4.9e-10', '-0.000000000486409475991', 1);
    T('-4.82125e+19', '-48212433366063403866', 5);
    T('-7.95593941e-20', '-0.000000000000000000079559394098236', 8);
    T('-2.00563e-10', '-0.0000000002005622924388', 5);
    T('-6.9777057921142634382521825e+16', '-69777057921142634.3825218243275152606161149381', 25);
    T('-8.42591e+14', '-842590769172062', 5);
    T('-6.35123264409e+27', '-6351232644080754054285724566', 11);
    T('-5.508835492577586495894259979e-28', '-0.00000000000000000000000000055088354925775864958942599785412', 27);
    T('-2.667451876e+12', '-2667451875964', 9);
    T('-6.6444610474323616283e+26', '-664446104743236162820999716', 19);
    T('-2.419775049243e+12', '-2419775049242.726', 12);
    T('-5.32e-18', '-0.000000000000000005319', 2);
    T('-8.63030355223e-26', '-0.000000000000000000000000086303035522286938593814060049934', 11);
    T('-2.5046920981956385048538613818080285657602718e+17', '-250469209819563850.48538613818080285657602717018', 43);
    T('-3.78e+15', '-3779392491464393.04412843034387404882622864039', 2);
    T('-3.3883802002818774e-21', '-0.0000000000000000000033883802002818773261717', 16);
    T('-3.57205e+19', '-35720468100481047658.74549510716', 5);
    T('-5.23810604e+18', '-5238106039196464333.4164490675655417554216049', 8);
    T('-8.9851705212202749156714435676788925065e+21', '-8985170521220274915671.443567678892506483244', 37);
    T('-4.8002620797467441513113e+15', '-4800262079746744.151311270846595944560084461404058322669896', 22);
    T('-7.6602835119619761973713784765241687426415076035234065319212e+19', '-76602835119619761973.713784765241687426415076035234065319212', 58);
    T('-5.381812197644510770977641728943e+29', '-538181219764451077097764172894.2045958494', 30);
    T('-6.04171e+30', '-6041702557251805571827972925970.859227', 5);
    T('-3.995516696587253269e+28', '-39955166965872532681529528721.070757896455736015403', 18);
    T('-7.597966e+15', '-7597965080819292', 6);
    T('-5.302339e+10', '-53023381796.8478', 6);
    T('-9.02545540564356e+13', '-90254554056435.587103358700012', 14);
    T('-4.90010261765297775855e+21', '-4900102617652977758549.72018756787751358174277326416937', 20);
    T('-5.078904359675664732215233579164e+14', '-507890435967566.473221523357916309238214', 30);
    T('-5.521012629302366870801695374639196986679745208450805993e+22', '-55210126293023668708016.9537463919698667974520845080599201807', 54);
    T('-5.2937835496774926e-19', '-0.00000000000000000052937835496774925027979577384249493104941', 16);
    T('-2.3554653675126963e+18', '-2355465367512696228', 16);
    T('-2.891052510655698093e-17', '-0.000000000000000028910525106556980924149216708779185331', 18);
    T('-3.68377e-16', '-0.0000000000000003683765961604816288244373051', 5);
    T('-3.95708e+25', '-39570783738574043219687566.965221194063889914', 5);
    T('-3.584456985168021826814122e-17', '-0.0000000000000000358445698516802182681412158196413726', 24);
    T('-8.556316744104688591686120874555554808035e+28', '-85563167441046885916861208745.55554808034964', 39);
    T('-6.02219e+18', '-6022186164465021650.884475588', 5);
    T('-7.790612428288e+18', '-7790612428287383595.5394047', 12);

    BigNumber.config({ROUNDING_MODE: 1});

    T('0e+0', '-0.0E-0', undefined);
    T('-2.856376815219143184897347685012382222462687620998915470135915e+6', '-2856376.815219143184897347685012382222462687620998915470135915511363444', 60);
    T('7.75700e-24', '0.000000000000000000000007757', 5);
    T('7.0e-1', '0.7', 1);
    T('5.2109749078977455423107465583658126e+37', '52109749078977455423107465583658126637', 34);
    T('3.631093819552528994444977110063007461579154042777868294000e-29', '0.00000000000000000000000000003631093819552528994444977110063007461579154042777868294', 57);
    T('-9.893937860425888e+8', '-989393786.042588804219191', 15);
    T('8.7978043622607467e+42', '8797804362260746751563912625017414439944006.5804807', 16);
    T('-4.6561702764394602621e-7', '-0.000000465617027643946026213823955447791862428108248596086901464075785390015', 19);
    T('-2.542770482242902215596924884302407e+8', '-254277048.224290221559692488430240765024783', 33);
    T('2.70000000e-8', '0.000000027', 8);
    T('-8.0291821891769794408790934252924453237e+16', '-80291821891769794.408790934252924453237503615825249362166', 37);
    T('-8.05295923004057358545854771e-16', '-0.0000000000000008052959230040573585458547716514262', 26);
    T('-2.786758e-21', '-0.00000000000000000000278675879025858093817787290334306', 6);
    T('-8.0160835624737225803853824687641777660406527e+20', '-801608356247372258038.538246876417776604065270622886204812876', 43);
    T('-7.2849054887999144694619191770897589e+27', '-7284905488799914469461919177.08975892527524', 34);
    T('-7.586e-17', '-0.00000000000000007586908', 3);
    T('-5.9508150933636580674249602941673984254864e+20', '-595081509336365806742.496029416739842548642249', 40);
    T('-3.526911897e-18', '-0.000000000000000003526911897770082481187', 9);
    T('-5.774e-22', '-0.0000000000000000000005774729035676859', 3);
    T('-6.4700957007714124190210074e-13', '-0.00000000000064700957007714124190210074383', 25);
    T('-5.610492e+21', '-5610492566512449795573', 6);
    T('-6.015e+23', '-601556443593022914280678', 3);
    T('-6.0673361553344e+11', '-606733615533.448288878', 13);
    T('-3.1e+26', '-315617199368461055533962323.071668327669249', 1);
    T('-9.1391079512104562032343e+24', '-9139107951210456203234346', 22);
    T('-2.0441e+21', '-2044198307917443182711', 4);
    T('-8.21283723216249535240085606500821783973097233e+23', '-821283723216249535240085.606500821783973097233814324', 44);
    T('-6.375e+14', '-637540984314799.4', 3);
    T('-2.17797482005219478530856429744726e+29', '-217797482005219478530856429744.7268928676963181', 32);
    T('-3.9547e+11', '-395476721391', 4);
    T('-6.8927e+21', '-6892798573971046301111', 4);
    T('-6.33842141402916538926e-12', '-0.000000000006338421414029165389261335065112712777', 20);
    T('-4.5727e-30', '-0.000000000000000000000000000004572725511159166', 4);
    T('-7.8847457779026882221249217577974e-17', '-0.000000000000000078847457779026882221249217577974', 31);
    T('-2.64916231640264927e+12', '-2649162316402.649271824', 17);
    T('-1.73604404e+28', '-17360440496948254515028685124.37795415803082546457797184294', 8);
    T('-8.680224985623e+16', '-86802249856236148.11694273469092873', 12);
    T('-4.3e-19', '-0.00000000000000000043859841576346037715462713764211635', 1);
    T('-7.68867535389098159141717105e-11', '-0.000000000076886753538909815914171710501337139', 26);
    T('-5.24325038611090505928389422325001606e+21', '-5243250386110905059283.894223250016067979080420266', 35);
    T('-1.38e-21', '-0.0000000000000000000013874592057586367688528204069850262406', 2);
    T('-7.308601949094508589445770582074109410615037e+24', '-7308601949094508589445770.5820741094106150373221910779', 42);
    T('-3.2638e+13', '-32638405387645.3309565877781780222317335852159983', 4);
    T('-3.55454737448094719019291183206515059962378e+22', '-35545473744809471901929.118320651505996237856336054914', 41);
    T('-5.3906242252792e-11', '-0.00000000005390624225279268530907215395611', 13);
    T('-8.86760873811213105078e+15', '-8867608738112131.050787', 20);
    T('-4.78129254835567e-23', '-0.00000000000000000000004781292548355671480462711435866243551', 14);
    T('-6.4694208834502691835879021438795583630205e-19', '-0.00000000000000000064694208834502691835879021438795583630205', 40);
    T('-9.324e-25', '-0.00000000000000000000000093242969', 3);
    T('-6.922220589076408182786e+19', '-69222205890764081827.8655148459740694252038421', 21);
    T('-4.193207546161458e+19', '-41932075461614585862.215078', 15);
    T('-7.98e+20', '-798827417648620333729.80696458197', 2);
    T('-2.53e-27', '-0.0000000000000000000000000025361014542495516754818606153', 2);
    T('-1.4930677606201e-20', '-0.0000000000000000000149306776062013560263804', 13);
    T('-2.4385708957357e+19', '-24385708957357294486.03887038886025345320045340124898971786', 13);
    T('-2.3170650157672525597815028610843e+18', '-2317065015767252559.781502861084367708776250552', 31);
    T('-6.9178198e+18', '-6917819884210952360.76327902290237387108459707859893972', 7);
    T('-5.8557793e-24', '-0.000000000000000000000005855779377', 7);
    T('-2.9760848e-12', '-0.00000000000297608486674725722', 7);
    T('-5.994209456542723342157e+23', '-599420945654272334215750.2697081334512770109182770472941827', 21);
    T('-2.176318765141873189550724e+24', '-2176318765141873189550724', 24);
    T('-3.015068240172763167642991583362591462e+17', '-301506824017276316.76429915833625914624', 36);
    T('-4.092360120459492827213341546580282588568024330771e+25', '-40923601204594928272133415.465802825885680243307714368088538', 48);
    T('-1.241037736e-28', '-0.00000000000000000000000000012410377364', 9);

    BigNumber.config({ROUNDING_MODE: 2});

    T('0e+0', '0E0000000000', undefined);
    T('0e+0', '-0E01', undefined);
    T('0.00e+0', '-0E00000000001', 2);
    T('3.0465655253692145345165166442116e-14', '0.0000000000000304656552536921453451651664421156', 31);
    T('9.0573943842008592406279608542923313381394286641978907203396551e+22', '90573943842008592406279.60854292331338139428664197890720339655043720040907662489784', 61);
    T('-1.17181502970008783734855040984899000e-1', '-0.117181502970008783734855040984899', 35);
    T('-5.28860565e-16', '-0.00000000000000052886056528317233012115396784629214632', 8);
    T('6.4114675970838738000e-18', '0.0000000000000000064114675970838738', 19);
    T('8.00000000000000000000e-20', '0.00000000000000000008', 20);
    T('2.74000064578288771723078597711103520450391668117802304078152085625023633681179e+24', '2740000645782887717230785.977111035204503916681178023040781520856250236336811781347278', 77);
    T('8.1936742669491704846805837777816457628e-16', '0.00000000000000081936742669491704846805837777816457628', 37);
    T('-7.2157448e+14', '-721574484716710.00141299844961546', 7);
    T('-5.321807464703650000000e-15', '-0.00000000000000532180746470365', 21);
    T('-4.449e+27', '-4449471658582621135143349142.228707647170080816912435271162', 3);
    T('-4.922915821313919623758e+19', '-49229158213139196237.584', 21);
    T('-6.996668225774098e-14', '-0.000000000000069966682257740984029052', 15);
    T('-8.6856039174822133942616012424795168e+11', '-868560391748.2213394261601242479516861829472792', 34);
    T('-8.461e+21', '-8461810373307862460504', 3);
    T('-3.898716627703194625824411967e+25', '-38987166277031946258244119.67718', 27);
    T('-2.821935496755e+26', '-282193549675582402670759843.23655', 12);
    T('-3.49e-22', '-0.0000000000000000000003491662482987', 2);
    T('-3.362111778576231615366457333e-14', '-0.0000000000000336211177857623161536645733316587527475522615', 27);
    T('-5.9933e-13', '-0.00000000000059933412636903331', 4);
    T('-2.77927721e+29', '-277927721100404435781172100113.4136636412460458083951', 8);
    T('-1.876833722329e-10', '-0.0000000001876833722329987477942', 12);
    T('-6.5e+14', '-653341175209856', 1);
    T('-8.627291840173867961e+14', '-862729184017386.7961', 18);
    T('-3.9137457165597668391301218029e-11', '-0.00000000003913745716559766839130121802935022889', 28);
    T('-8.95e+10', '-89532775488', 2);
    T('-2.1395541875015568986238e-17', '-0.000000000000000021395541875015568986238771696', 22);
    T('-4.98575853353890809143399546448630559732119628e-12', '-0.00000000000498575853353890809143399546448630559732119628509', 44);
    T('-8.99e+16', '-89989591559494822', 2);
    T('-3.49346327e+22', '-34934632714180035424463', 8);
    T('-3.5699537605753905457597e-14', '-0.00000000000003569953760575390545759785014980652333323889116', 22);
    T('-2.9892536880349975618286e+12', '-2989253688034.9975618286212199904979534461637613', 22);
    T('-3.04383919217904949618e+10', '-30438391921.790494961888803732171', 20);
    T('-8.232411544e+17', '-823241154405701456', 9);
    T('-5.809151226990464016815e-16', '-0.00000000000000058091512269904640168152354', 21);
    T('-8.522042397326932431e+13', '-85220423973269.324312660179132118', 18);
    T('-7.5210942e-22', '-0.000000000000000000000752109428925015', 7);
    T('-5.2018321449543e+23', '-520183214495439298725191.09', 13);
    T('-6.04084045453711395629268198016245611021901815e+21', '-6040840454537113956292.68198016245611021901815486929628647', 44);
    T('-1.495478178996755138125934544343674798e-13', '-0.00000000000014954781789967551381259345443436747983317353423', 36);
    T('-6.881484497510733524151245220630282259985306546537e+16', '-68814844975107335.241512452206302822599853065465371507616758', 48);
    T('-4.7121389019956e-14', '-0.00000000000004712138901995619', 13);
    T('-8.8332728504053108443425344711e-15', '-0.00000000000000883327285040531084434253447119282', 28);
    T('-8.2e+14', '-822000812447305', 1);
    T('-7.772164697477093877214551050634072755e+21', '-7772164697477093877214.55105063407275517068350805', 36);
    T('-3.9087122838427126623505550357872e+10', '-39087122838.4271266235055503578721071128', 31);
    T('-2.312032777966762704192668904908578897e+20', '-231203277796676270419.266890490857889726891117', 36);
    T('-6.145717261905789834140342e+10', '-61457172619.0578983414034269108488849155084479', 24);
    T('-1.22122395777234009028954105999904e+23', '-122122395777234009028954.10599990431', 32);
    T('-8.11092557e-19', '-0.000000000000000000811092557221182808185409783', 8);
    T('-2.0148183904e+12', '-2014818390421', 10);
    T('-8.5895e+12', '-8589543094837', 4);
    T('-4.52948430169449249063e+13', '-45294843016944.92490631367483828208567689248', 20);
    T('-5.5627328016242253171e+18', '-5562732801624225317.15482034912', 19);
    T('-2.299e+22', '-22994771657263381474221.8393766046648504992', 3);
    T('-4.886104291748549e+15', '-4886104291748549.177', 15);
    T('-3.7192656464776e-11', '-0.00000000003719265646477604172611', 13);
    T('-6.135956620537e+25', '-61359566205370067856449153.5', 12);
    T('-3.35703853285800120218674208960269655701e-14', '-0.000000000000033570385328580012021867420896026965570155917', 38);
    T('-8.713884178791321311224703e+22', '-87138841787913213112247.03564225163096', 24);
    T('-7.073358e+12', '-7073358766762', 6);
    T('-6.829e+30', '-6829360758600890577632541121747.862424035', 3);
    T('-3.05687463293e+22', '-30568746329329110731433.300963185825462157574537899186', 11);
    T('-8.761781e+24', '-8761781624975891699172893.0141633817001124644', 6);
    T('-1.477e+12', '-1477134517234.0307742', 3);
    T('-5.78904078729758522168774487851811e+16', '-57890407872975852.216877448785181168776187771353947582', 32);
    T('-7.74939714942520320266429137e+12', '-7749397149425.20320266429137053013', 26);
    T('-8.3224649681672648581370532515e-14', '-0.00000000000008322464968167264858137053251586527433370546682', 28);
    T('-7.04146154016765195683657078079536e-22', '-0.000000000000000000000704146154016765195683657078079536', 32);
    T('-1.914289454756549529781861916925090389e+16', '-19142894547565495.29781861916925090389840210707205', 36);
    T('-8.840670154325523051759462672e+27', '-8840670154325523051759462672.142803216', 27);
    T('-2.823e-11', '-0.00000000002823852806134378210195515771768582269146178698', 3);
    T('-1.5186417607496557534159723950506e+29', '-151864176074965575341597239505.06547275781526923', 31);
    T('-7.397218e+16', '-73972184449181471.152912157', 6);
    T('-3.581193819284374099989e+22', '-35811938192843740999895.1573225646377886389016478830802218237', 21);
    T('-4.563585432210043885759681337791545e+29', '-456358543221004388575968133779.154510217739887576756399', 33);
    T('-1.8176465832459836335875e-18', '-0.0000000000000000018176465832459836335875559', 22);
    T('-3.784854627631e+20', '-378485462763141736445.9462626829154663', 12);
    T('-3.8510536744200399243363367786406618e+23', '-385105367442003992433633.6778640661831754293129488068676868', 34);
    T('-8.7323e+22', '-87323916569164596208111.88101028771355420576029037973', 4);
    T('-7.578e+11', '-757882758255.76', 3);
    T('-8.5977102122033e+20', '-859771021220338061040.041580289025031', 13);
    T('-7.3998697893908579137880913e+17', '-739986978939085791.37880913509684330685', 25);
    T('-6.71117252123290052432148305375254e-19', '-0.00000000000000000067111725212329005243214830537525418', 32);
    T('-6.6762993760195471322427204620426381935201299178096e+11', '-667629937601.9547132242720462042638193520129917809665', 49);
    T('-2.852022020015364818597602e+15', '-2852022020015364.818597602673413917443', 24);
    T('-3.151044e+30', '-3151044929117854676102403114231.56823295246', 6);
    T('-4.47120537692951873038916592e+10', '-44712053769.29518730389165927694', 26);
    T('-7.4041969e+23', '-740419699691346150775964.049522110341852844412207474667958', 7);
    T('-6.311838e-10', '-0.000000000631183849892543191', 6);
    T('-7.2570104326587672213e+16', '-72570104326587672.213076838263780308795144628367752', 19);
    T('-4.445769230869049803541e+15', '-4445769230869049.80354196820931591782233498498378174385', 21);

    BigNumber.config({ROUNDING_MODE: 3});

    T('-9.99999999000000009e+8', '-999999999.000000009e-0', undefined);
    T('-3.99764422903251220452704763278376060858663250289320247532595e+24', '-3997644229032512204527047.63278376060858663250289320247532594416986984981431156065660613', 59);
    T('5.534083545686157907280686578717428772e+12', '5534083545686.157907280686578717428772', 36);
    T('5.00000000e-9', '0.000000005', 8);
    T('-4.08363116583051e+14', '-408363116583051', 14);
    T('9.278230415634296945273818e+19', '92782304156342969452.738186255580532649103987374718221928871873054827841260470670536425', 24);
    T('-1.08732508998603085454662e-12', '-0.000000000001087325089986030854546619968259691229662152159029641023997866843605032534351388775075', 23);
    T('3.5288804517377606688698e+32', '352888045173776066886981811418233.218955856086', 22);
    T('4.32188781438877554e+16', '43218878143887755.42593887518334667202', 17);
    T('-8.15e+2', '-815', 2);
    T('1.515077312590223222678749527e+18', '1515077312590223222.678749527895871363186918977679783240817218232896076765321818445939718165', 27);
    T('-8.0538186421664536509917032729912932814374102e+20', '-805381864216645365099.17032729912932814374101821', 43);
    T('-3.4367097301002099047381e+14', '-343670973010020.990473804391071456587732173', 22);
    T('-5.3421e-12', '-0.0000000000053420288504', 4);
    T('-2.6320052e+23', '-263200517731973006983184.60341959097016190770542276', 7);
    T('-4.5e-11', '-0.000000000044673422483', 1);
    T('-7.232463101115829118145025733451801e-17', '-0.00000000000000007232463101115829118145025733451800457178', 33);
    T('-1.18320100044154762448545914170978206041022039e+22', '-11832010004415476244854.5914170978206041022038489', 44);
    T('-7.745237371276392645711e+21', '-7745237371276392645710.0521930569226728841707200771', 21);
    T('-4.431559500053255695643e-10', '-0.000000000443155950005325569564213010993378905', 21);
    T('-2.5e-24', '-0.000000000000000000000002443', 1);
    T('-5.005027028439023958391203127005503621542e-11', '-0.0000000000500502702843902395839120312700550362154137', 39);
    T('-6.453525377934213334367e-22', '-0.00000000000000000000064535253779342133343665123283565', 21);
    T('-4.5594370326121718626850982373529e+13', '-45594370326121.71862685098237352845979966987', 31);
    T('-1.709e+16', '-17088248121660259', 3);
    T('-3.9047581533864713e+16', '-39047581533864712.6574405', 16);
    T('-2.08804202e-17', '-0.000000000000000020880420127397564274443250271135', 8);
    T('-6.801694635944774655689008216925036e+15', '-6801694635944774.65568900821692503508025', 33);
    T('-8.7691286374104240967931800593734e+19', '-87691286374104240967.93180059373367907299683816381677816389', 31);
    T('-2.802257731715238453e-29', '-0.000000000000000000000000000028022577317152384526775320012', 18);
    T('-4.4705e+22', '-44704405768781565005877.813010169083', 4);
    T('-4.17374908496486449232e-10', '-0.00000000041737490849648644923105632500267064', 20);
    T('-2.2707e-10', '-0.00000000022706134122862417334386435', 4);
    T('-2.85432e-24', '-0.0000000000000000000000028543100839983854161', 5);
    T('-5.79188949e+12', '-5791889489461.643555240257', 8);
    T('-7.46e+15', '-7459701910718662.03421293892346992893463534702', 2);
    T('-1.0535086280629e+25', '-10535086280628995915087428.2423609320023833125322801559606', 13);
    T('-2.9074412651647188367106e+30', '-2907441265164718836710598468491.31550321772', 22);
    T('-5.010945976711327691649e+27', '-5010945976711327691648509517.2305', 21);
    T('-8.8633960213386533e-20', '-0.0000000000000000000886339602133865324283362544', 16);
    T('-3.1891844834898211661452730714015664837805e+19', '-31891844834898211661.45273071401566483780434051217', 40);
    T('-5.083380976014365533843229882526437e+28', '-50833809760143655338432298825.264367948359', 33);
    T('-6.8e-16', '-0.000000000000000678534987604148025611184', 1);
    T('-7.9e+30', '-7838656097386639584904346062976.9346038436', 1);
    T('-6.30535781e+20', '-630535780834495012856', 8);
    T('-9.663e-30', '-0.00000000000000000000000000000966289400023904753107633012', 3);
    T('-2.315198482309e+12', '-2315198482308.7361348', 12);
    T('-8.158235289416e+18', '-8158235289415958939', 12);
    T('-4.1618890517404316933699206360639988582832624525e+23', '-416188905174043169336992.063606399885828326245241437', 46);
    T('-5.97550716981833990839441709632e+21', '-5975507169818339908394.41709631281058258352209', 29);
    T('-6.3372e-18', '-0.000000000000000006337122571683959413228', 4);
    T('-8.9189088e+18', '-8918908714500548003.38400978696756078013348', 7);
    T('-2.30738494e+15', '-2307384939629592.5507643557167543121437', 8);
    T('-5.5187220703008771818558364e+20', '-551872207030087718185.58363308126401300424', 25);
    T('-6.6221540532808e+16', '-66221540532807215', 13);
    T('-7.52280140768218860970644149216497725e+28', '-75228014076821886097064414921.6497724655', 35);
    T('-4.50815289e-10', '-0.0000000004508152886241127131780051700309401', 8);
    T('-8.05636473909e+28', '-80563647390897795982047004786.9809587987299506647489380735', 11);
    T('-8.3e-22', '-0.00000000000000000000082867896643314771124884886449603747139', 1);
    T('-8.3783e+13', '-83782644902152', 4);
    T('-1.1939712427296807e+16', '-11939712427296807', 16);
    T('-6.520492185955083727143468903725e+24', '-6520492185955083727143468.90372469799639', 30);
    T('-5.468441290352576854e+22', '-54684412903525768532358.76123265640787599117379', 18);
    T('-6.3213239044187e-12', '-0.000000000006321323904418628', 13);
    T('-6.80758136e+10', '-68075813559.812083737218313494618879237157412', 8);
    T('-2.32394435705096500766e+20', '-232394435705096500765.423311444507670516532857314906', 20);
    T('-5.35396744204815374979010975360864002355e+14', '-535396744204815.374979010975360864002354465685768494008245896', 38);
    T('-1.8388340153656061115e-24', '-0.0000000000000000000000018388340153656061114681', 19);
    T('-2.09349812455746e+24', '-2093498124557455120865520.476275227', 14);
    T('-2.888450139093515656e-25', '-0.0000000000000000000000002888450139093515656', 18);
    T('-6.97756838052316890676e+30', '-6977568380523168906759075718628.73360426401485819654038588804', 20);
    T('-8.05604538646883624239398132377048820023e+24', '-8056045386468836242393981.323770488200227820839', 38);
    T('-4.13045948e+29', '-413045947014551860341804907208.7067642881758676', 8);
    T('-7.990552461602111454165337515e+23', '-799055246160211145416533.75144940262265224221931', 27);
    T('-7.84498851993324e+11', '-784498851993.323271787115869178093231451893938531755482687806', 14);
    T('-8.63875584973951951712658379e-21', '-0.000000000000000000008638755849739519517126583785754757065', 26);
    T('-8.61609302272300237447639006834635e-14', '-0.00000000000008616093022723002374476390068346342187746', 32);
    T('-7.01300801762e+17', '-701300801761204790.177590913310762', 11);
    T('-8.0318131135482342451545e-11', '-0.0000000000803181311354823424515442372680533', 22);
    T('-8.310034087753417316659936093943321e+25', '-83100340877534173166599360.9394332099174859', 33);
    T('-7.716088095718838665380730070082633435173897567e+30', '-7716088095718838665380730070082.6334351738975662966', 45);
    T('-6.5207000918869e-14', '-0.00000000000006520700091886862177', 13);
    T('-6.579884485936605389e+14', '-657988448593660.538847871', 18);
    T('-5.31961604251455760419e+30', '-5319616042514557604183392605338.36600372994596807972708', 20);
    T('-7.87765329352729e+16', '-78776532935272856.77806', 14);
    T('-8.23e+11', '-822427564609', 2);
    T('-1.2946e+16', '-12945401038582508.297183225785515084520662225', 4);
    T('-4.3885535805231634787626423119240512694696e+14', '-438855358052316.347876264231192405126946952', 40);
    T('-6.4067449547192616381924351e-29', '-0.00000000000000000000000000006406744954719261638192435066816', 25);
    T('-9.41834953e+18', '-9418349527156084224.2', 8);
    T('-3.19716162829318952418046452988e+13', '-31971616282931.895241804645298754890905582545633', 29);

    BigNumber.config({ROUNDING_MODE: 4});

    T('-5.002239116605888927178702930656e-39', '-0.00000000000000000000000000000000000000500223911660588892717870293065633642', 30);
    T('-8.52292947230244775435e+29', '-852292947230244775434968241532.494643593912804433318745222587246680109833509655450267792446', 20);
    T('-6.1169514510867e+10', '-61169514510.8673382', 13);
    T('-8.05745763527307676170759722175169266017831695215e+48', '-8057457635273076761707597221751692660178316952146', 47);
    T('-4.923572102098e+10', '-49235721020.9847017846898652687600227388412980598816', 12);
    T('-7.981341661715027117746906076515945e+41', '-798134166171502711774690607651594491039629', 33);
    T('-8.00e-3', '-0.008', 2);
    T('8.517466793430899278197016892000000000000e-15', '0.000000000000008517466793430899278197016892', 39);
    T('-3.032293512e+0', '-3.0322935124071923328711934463341802038', 9);
    T('-2.60682904403489305678908771323995810138267385200000000e-20', '-0.00000000000000000002606829044034893056789087713239958101382673852', 53);
    T('-3.935816927273980e+20', '-393581692727398036652.850960055902271', 15);
    T('-2.98297216346e-27', '-0.00000000000000000000000000298297216346039288935575576076143', 11);
    T('-3.01319315e+23', '-301319315398414808376087.572306433', 8);
    T('-8.870698526921188e-12', '-0.00000000000887069852692118832284144110732', 15);
    T('-3.27e+23', '-326739927744903524706793.652546266488323001284674736489440831', 2);
    T('-8.614e+12', '-8613828413581', 3);
    T('-6.1382445990593346026804e+12', '-6138244599059.3346026803630253203', 22);
    T('-7.9111971e+12', '-7911197130975', 7);
    T('-8.5902152501051e+29', '-859021525010507210136559039003.689834129033952321238', 13);
    T('-7.24491e-30', '-0.00000000000000000000000000000724490826045045451271534', 5);
    T('-8.4948070285349193974989221504919380656715136165603325e+24', '-8494807028534919397498922.15049193806567151361656033246', 52);
    T('-6.3295239596e-17', '-0.00000000000000006329523959626011114164', 10);
    T('-3.1725692353e+30', '-3172569235260846783669130724638.711', 10);
    T('-4.065727077e+11', '-406572707673.336570352310681187663765', 9);
    T('-6.82883869249998075574247223155497e+18', '-6828838692499980755.7424722315549682855987375899188309581152', 32);
    T('-2.56144400427045214943786338e+24', '-2561444004270452149437863.38354535663028539', 26);
    T('-4.97637439956044400125498868e+23', '-497637439956044400125498.8682100590602459937304614141772', 26);
    T('-4.307891929198702822746534506143e+29', '-430789192919870282274653450614.349564081', 30);
    T('-8.55e-27', '-0.00000000000000000000000000855367295711812079', 2);
    T('-7.906e+11', '-790612526329.410459220189562', 3);
    T('-3.1841363e-22', '-0.00000000000000000000031841363', 7);
    T('-6.2068049304845006e+20', '-620680493048450055389.3227069760888437941041', 16);
    T('-8.4809476e+18', '-8480947614295114807.320148688', 7);
    T('-2.287988570734255855e+23', '-228798857073425585542366.399034916953775', 18);
    T('-8.148647139762925073276164486240320698e+21', '-8148647139762925073276.1644862403206980851079', 36);
    T('-6.87643138785664756e-12', '-0.0000000000068764313878566475604352570287089535238582267443', 17);
    T('-3.709587e+18', '-3709586618852569033.55141868', 6);
    T('-6.8086794224e+28', '-68086794224433270564431694468.814537646575833889824621540849', 10);
    T('-4.966301085179e+19', '-49663010851788946007', 12);
    T('-5.34439184068052811184219234494114e+26', '-534439184068052811184219234.494113670484623394', 32);
    T('-2.798732412e+16', '-27987324119455299', 9);
    T('-1.554430791885961957e+15', '-1554430791885961.956863404519493346081223', 18);
    T('-6.90619083822075003978e+24', '-6906190838220750039778836.289105048686876596', 20);
    T('-1.108034176809770578315e+12', '-1108034176809.7705783154', 21);
    T('-1.43e+22', '-14266566332440117777110.63461224926682073525873105', 2);
    T('-9.15e+13', '-91477543307040.916791223', 2);
    T('-1.1001e+26', '-110010856476508992391958436.9355559264588205214557001854', 4);
    T('-1.2e+16', '-12148027447349021', 1);
    T('-4.4e+13', '-44268551660889.40880208546489742632181832780494', 1);
    T('-8.62058920338555484081691e+19', '-86205892033855548408.169086865949596390775', 23);
    T('-5.2e-13', '-0.00000000000051876025261394172', 1);
    T('-4.88063953404884862027221562057786242658496407473e-11', '-0.0000000000488063953404884862027221562057786242658496407473', 47);
    T('-5.255e+18', '-5254530327311322805.9528217', 3);
    T('-6.4630488003995117e-11', '-0.0000000000646304880039951167486', 16);
    T('-3.15214e-23', '-0.00000000000000000000003152137339126187', 5);
    T('-8.86563136e+11', '-886563136251.626990531858472111699416852', 8);
    T('-8.638990742871e-16', '-0.0000000000000008638990742870608', 12);
    T('-1.57817750020560815944470062e+12', '-1578177500205.60815944470062002898187', 26);
    T('-3.6558384593093900422637e-27', '-0.00000000000000000000000000365583845930939004226367940618', 22);
    T('-7.5e+12', '-7540535487033', 1);
    T('-6.7647935206791247e+19', '-67647935206791246567', 16);
    T('-3.0204818086245915027e+30', '-3020481808624591502749101182536.872936744534671794', 19);
    T('-8.40498662e+12', '-8404986622734.85', 8);
    T('-2.944135296894e-18', '-0.0000000000000000029441352968942548971', 12);
    T('-8.826099694855290261753e+11', '-882609969485.52902617534731', 21);
    T('-1.9717565867734925e-13', '-0.000000000000197175658677349252855292223369', 16);
    T('-4.91451975824866130376722e+20', '-491451975824866130376.722358803861287205044883122152013315', 23);
    T('-5.111649e+17', '-511164947156144375', 6);
    T('-9.496473458673099e+11', '-949647345867.30987953779868637405061', 15);
    T('-2.1903308925764762892e+21', '-2190330892576476289225', 19);
    T('-3.47598363e+25', '-34759836338593591584288059.755482689269713', 8);
    T('-2.9192144584989753156762701431e-24', '-0.0000000000000000000000029192144584989753156762701431', 28);
    T('-4.0456517973466503588734928438425e+23', '-404565179734665035887349.28438424933669843', 31);
    T('-1.297871549154944904150929e+17', '-129787154915494490.4150929407633398', 24);
    T('-1.4566530316908752e+18', '-1456653031690875152.6306667', 16);
    T('-3.5521e-12', '-0.00000000000355210483', 4);
    T('-9.1838324864110351307221525161e+17', '-918383248641103513.07221525161442', 28);
    T('-8.33245633316304149287131334e-22', '-0.00000000000000000000083324563331630414928713133382', 26);
    T('-4.593824606634605622464043606094613988489104e+15', '-4593824606634605.62246404360609461398848910424547985108092894', 42);
    T('-5.232e-26', '-0.0000000000000000000000000523185958604202852', 3);
    T('-3.8319390497954462e+25', '-38319390497954461897251251.444', 16);
    T('-1.00157678068191049988073716749599603712e+17', '-100157678068191049.9880737167495996037119953003896147', 38);
    T('-4.169977410059689809645035174132294864e+20', '-416997741005968980964.50351741322948635363513285839302', 36);
    T('-7.121660153198989278372512656775647e-11', '-0.0000000000712166015319898927837251265677564651728358', 33);
    T('-7.98924570545536548623603750084330391943e+19', '-79892457054553654862.360375008433039194317394396964358522', 38);

    BigNumber.config({ROUNDING_MODE: 5});

    T('4.95474614815842e+38', '495474614815842191683004449862568813538.573064401156', 14);
    T('-8.9667567079038139e+16', '-89667567079038139', 16);
    T('-7.0e+2', '-703', 1);
    T('-2.6249e+33', '-2624861185343559570287214983819906', 4);
    T('-6.510119186347371697501169416839709631422185436811698613000000000000000000000000000000e-31', '-0.0000000000000000000000000000006510119186347371697501169416839709631422185436811698613', 84);
    T('7.73e+3', '7729', 2);
    T('1.4393781011009257793117531801549e+4', '14393.781011009257793117531801548751', 31);
    T('8.4e+6', '8404542', 1);
    T('8.471284625267663009248667391059202502589207637435209861233007389000000000000000e-35', '0.00000000000000000000000000000000008471284625267663009248667391059202502589207637435209861233007389', 78);
    T('-5.26079297227015e+31', '-52607929722701509263909039511536.9266822991', 14);
    T('-4.63550600857003551411914120562163394e+15', '-4635506008570035.51411914120562163394396594237358863897062', 35);
    T('-7.8219563406482338767189100434751303552919130625101491e+27', '-7821956340648233876718910043.4751303552919130625101491', 52);
    T('-6.977184098e+17', '-697718409782854734', 9);
    T('-8.1e+15', '-8092701222454628.9934935902179330839653799891168', 1);
    T('-3.872944373744596915691884729973e+15', '-3872944373744596.91569188472997336351132980366520033057011287', 30);
    T('-1.389676e+11', '-138967565295.146055555208419143848718279114979831585', 6);
    T('-2.218316993130903882223e+19', '-22183169931309038822.22612', 21);
    T('-3.370809304e-25', '-0.000000000000000000000000337080930401566', 9);
    T('-6.1503e+19', '-61503417721509415792.24703', 4);
    T('-3.13657134e-22', '-0.00000000000000000000031365713378439345', 8);
    T('-1.9e-10', '-0.000000000187981', 1);
    T('-2.596508353714425677970049724e+28', '-25965083537144256779700497237.5841327343962292316215149169', 27);
    T('-4.151454545748277604112308101174917062e+11', '-415145454574.827760411230810117491706171981266892178', 36);
    T('-1.3e-18', '-0.000000000000000001319061308619561567664259803361817', 1);
    T('-1.5294854487046553159e+24', '-1529485448704655315921667', 19);
    T('-1.9365487654708143765583400538310103350799e-13', '-0.000000000000193654876547081437655834005383101033507988', 40);
    T('-3.88128259276357427027515474e+25', '-38812825927635742702751547.353', 26);
    T('-5.64525474904155517374289736218e-11', '-0.00000000005645254749041555173742897362182099811344', 29);
    T('-8.94963385755006409131430087734467745e+22', '-89496338575500640913143.0087734467744538', 35);
    T('-3.7551731901764025e+17', '-375517319017640249', 16);
    T('-7.601921e-16', '-0.00000000000000076019214974360137746140339586742455753', 6);
    T('-6.93501087055e+20', '-693501087055377288564', 11);
    T('-1.283656440009563e+24', '-1283656440009563292695670.575360580373829197017512', 15);
    T('-4.9556506e+13', '-49556505932168.7211084603', 7);
    T('-8.133584588946e+26', '-813358458894586332533196788.490201803951456991010654609646', 12);
    T('-3.824207296e+22', '-38242072955850210158724', 9);
    T('-4.2168087e-12', '-0.00000000000421680868317080291', 7);
    T('-7.152812829e+15', '-7152812829336253.782723153403637377960530795', 9);
    T('-8.0469635248612874571e+16', '-80469635248612874.5712104436', 19);
    T('-2.726549954018643349550392804e+11', '-272654995401.8643349550392803783934819148125595437353472547', 27);
    T('-2.477986360297097033217143e+30', '-2477986360297097033217143442370.539404', 24);
    T('-2.7620555408e+15', '-2762055540757162', 10);
    T('-5.044e+10', '-50436788962', 3);
    T('-1.51176171306898543927009427965761639e+17', '-151176171306898543.9270094279657616389483779413616294465635', 35);
    T('-3.76233131039732974161231568e+13', '-37623313103973.2974161231567776787873083163171', 26);
    T('-1.77876313221062362e+17', '-177876313221062362.01', 17);
    T('-4.28033364715744300662536e+13', '-42803336471574.430066253616', 23);
    T('-6.053e-13', '-0.00000000000060527568964627046163209582', 3);
    T('-3.9447068214322315685949701607748761e+16', '-39447068214322315.685949701607748760885392781169754754427622', 34);
    T('-4.76203665586552028e+15', '-4762036655865520.285', 17);
    T('-7.442141482296791204320219247230530359e+24', '-7442141482296791204320219.2472305303585223494415', 36);
    T('-5.96279453376966633e+23', '-596279453376966633175009.6', 17);
    T('-3.393419405169789e+24', '-3393419405169788742460001.267', 15);
    T('-5.3001e+12', '-5300055380607', 4);
    T('-5.6075017651299255742594578e+24', '-5607501765129925574259457.7938331743229', 25);
    T('-1.7016332185618e-12', '-0.000000000001701633218561829307163951183908', 13);
    T('-8.2586539997288574125e-29', '-0.0000000000000000000000000000825865399972885741250631446', 19);
    T('-6.867e+11', '-686673700185', 3);
    T('-6.77934398386662123284e+26', '-677934398386662123284378302.457585912', 20);
    T('-1.68708254641574159341563239757e+14', '-168708254641574.159341563239757201959', 29);
    T('-7.969791397195291274332017902569730510486538e+16', '-79697913971952912.74332017902569730510486538476172', 42);
    T('-8.35460490386e+14', '-835460490386401.159749305581999482', 11);
    T('-3.4904587e+10', '-34904586685.65531405315150234636', 7);
    T('-7.655476116917648649e-10', '-0.0000000007655476116917648649345', 18);
    T('-3.035704337e+17', '-303570433749270293', 9);
    T('-1.4902739431686400585e-18', '-0.000000000000000001490273943168640058452103113', 19);
    T('-2.57617086126164572e+17', '-257617086126164572', 17);
    T('-6.9708e+16', '-69708261331391628', 4);
    T('-8.61400120130585599610136e-12', '-0.00000000000861400120130585599610136066', 23);
    T('-9.0670988886e-19', '-0.000000000000000000906709888862126926', 10);
    T('-2.889463982215818248e-26', '-0.00000000000000000000000002889463982215818248', 18);
    T('-3.7376459408597195073982491e+26', '-373764594085971950739824910.4572745527', 25);
    T('-6.21372353850510695881280108179e-12', '-0.0000000000062137235385051069588128010817907', 29);
    T('-2.4240953581712173951958e-21', '-0.00000000000000000000242409535817121739519585', 22);
    T('-8.3687559027615173415e+18', '-8368755902761517341.46477685623835786273991', 19);
    T('-7.18294352e-11', '-0.0000000000718294352479105', 8);
    T('-3.52454012503419773886785e-25', '-0.000000000000000000000000352454012503419773886785342913143', 23);

    BigNumber.config({ROUNDING_MODE: 6});

    T('-4.3502707501164e+36', '-4350270750116411997402439304498892819', 13);
    T('9.5e-21', '0.0000000000000000000094520280724178734152', 1);
    T('1.39631186750554172785676012693418617250072200744214625994656047727270672248243741907e+34', '13963118675055417278567601269341861.725007220074421462599465604772727067224824374190703237660781', 83);
    T('5.9446570e-26', '0.00000000000000000000000005944657036540768164877637239177740419063920648', 7);
    T('7.00000e-12', '0.000000000007', 5);
    T('-2.87e+14', '-287060740776209.3950381715', 2);
    T('3.411740542875509329e+24', '3411740542875509328514044', 18);
    T('-6.20235112738687046118395830000000000000000000000e-29', '-0.000000000000000000000000000062023511273868704611839583', 47);
    T('2.94349130121570276626863135396717336528655493e+19', '29434913012157027662.686313539671733652865549279174', 44);
    T('4.01255076512828067130306533670644537832e-10', '0.000000000401255076512828067130306533670644537831678294548', 38);
    T('-5.4277306444432e+11', '-542773064444.317654960431120452254700391693837992', 13);
    T('-4.355706886680889557797360814402e+30', '-4355706886680889557797360814401.536556745674646509159280626', 30);
    T('-1.29e-15', '-0.00000000000000128978312277001609181774216296380783932', 2);
    T('-1.0588973816292989769e+25', '-10588973816292989768709129.1767038708798755780352204', 19);
    T('-3.210569596e+10', '-32105695962.8803639621', 9);
    T('-7.18504270173744681360682714959e+28', '-71850427017374468136068271495.87', 29);
    T('-4.29794333519778779150824479010034817077204e-10', '-0.0000000004297943335197787791508244790100348170772040392', 41);
    T('-4.615682142828269066227773895179987062919e+20', '-461568214282826906622.7773895179987062919071922', 39);
    T('-1.3864477517287155526073e+13', '-13864477517287.15552607265', 22);
    T('-6.793120028e+13', '-67931200280922.72252141789646787475433427482', 9);
    T('-8.075e-18', '-0.000000000000000008074975073002274636799975', 3);
    T('-8.360228691054180854419062530687032074820667001e+24', '-8360228691054180854419062.530687032074820667001120752628', 45);
    T('-3.0763956760417194035216e-12', '-0.000000000003076395676041719403521594', 22);
    T('-2.5288383e+25', '-25288383009460922631988717.84659997837058450749', 7);
    T('-4.554185192e+29', '-455418519247311560996997520087.98189', 9);
    T('-9.135175372324138467397264e+11', '-913517537232.413846739726417', 24);
    T('-8.257259383044471855222900534859251889332388855848e-10', '-0.0000000008257259383044471855222900534859251889332388855848', 48);
    T('-7.651597268450922707e-13', '-0.000000000000765159726845092270720405167100094', 18);
    T('-8.952011763950994514e+26', '-895201176395099451377549961.34870447', 18);
    T('-2.7395479569618982298152060567357e-10', '-0.00000000027395479569618982298152060567357', 31);
    T('-1.31151451700453378841431e+24', '-1311514517004533788414313', 23);
    T('-5.915297930316863891e-10', '-0.0000000005915297930316863890707686339684395', 18);
    T('-1.449e-27', '-0.0000000000000000000000000014487033279693402845128265141859', 3);
    T('-3.7e+10', '-36919550406.826974442743517918128', 1);
    T('-3.945347688940382499631779106638865e+13', '-39453476889403.824996317791066388653', 33);
    T('-8.547704e-29', '-0.0000000000000000000000000000854770378842608635356', 6);
    T('-3.76e+25', '-37618296325402619735777629.467812385256281737441412', 2);
    T('-8.031066086398624e+28', '-80310660863986235667567286452', 15);
    T('-4.038276256088135496e-17', '-0.000000000000000040382762560881354955896694823328777602811', 18);
    T('-1.77173574740860868e+25', '-17717357474086086837250852', 17);
    T('-1.421967649e+21', '-1421967648805122645888', 9);
    T('-4.7e+11', '-469485715327', 1);
    T('-7.372223291560455075681748682810527006883e+16', '-73722232915604550.75681748682810527006882666313809409', 39);
    T('-8.9539396357e+14', '-895393963565598', 10);
    T('-8.14646103854802172250414801405e+10', '-81464610385.48021722504148014045579178726', 29);
    T('-1.2053415734425581e+12', '-1205341573442.5581371841633131879', 16);
    T('-8.35214176861046133596101313170854966756043001e+28', '-83521417686104613359610131317.0854966756043001041619492', 44);
    T('-3.7610694152e-28', '-0.00000000000000000000000000037610694151517628351', 10);
    T('-6.71e-12', '-0.00000000000670729337105720320122353', 2);
    T('-4.005517304396006251e+13', '-40055173043960.0625088492324182094858', 18);
    T('-6.0206e+28', '-60205974155921075891080012488.4566490314762809', 4);
    T('-6.36287561326e+11', '-636287561325.9124444291802472', 11);
    T('-3.11336117e-16', '-0.000000000000000311336117052129384933053792', 8);
    T('-5.3927134886536e+30', '-5392713488653639958906162302264.424436642808', 13);
    T('-3.82395446711276e-10', '-0.0000000003823954467112758458806849565215407952986440811', 14);
    T('-4.2858082253423e-27', '-0.0000000000000000000000000042858082253422975', 13);
    T('-2.9918792622984137284399075479267066e+14', '-299187926229841.3728439907547926706557', 34);
    T('-3.1949909651023223034303544498737e+27', '-3194990965102322303430354449.8737', 31);
    T('-9.1e-27', '-0.0000000000000000000000000090531861025', 1);
    T('-2.8e+11', '-279301037794', 1);
    T('-7.126913661498270214611054421e+13', '-71269136614982.70214611054420849', 27);
    T('-4.86337579169293342736515180299340135e+13', '-48633757916929.334273651518029934013479777304', 35);
    T('-3.406744915848058125e+25', '-34067449158480581246177934.3445612265793', 18);
    T('-5.542902272865090080311949446460659235171860088660477e+16', '-55429022728650900.803119494464606592351718600886604770155246', 51);
    T('-8.26224854264697737938997145336e+12', '-8262248542646.9773793899714533620028598662842221171', 29);
    T('-3.16331e+18', '-3163306186318700887', 5);
    T('-9.087531707575372e+25', '-90875317075753723792666377.6466517495', 15);
    T('-8.758548512438e+14', '-875854851243824.87435', 12);
    T('-3.9e-11', '-0.0000000000387093', 1);
    T('-3.987015017148130889206385341736666180313e+11', '-398701501714.813088920638534173666618031251290587', 39);
    T('-2.493129998e-11', '-0.00000000002493129997889845697168462', 9);
    T('-7.0892393575673871055576e+17', '-708923935756738710.5557595392277447617', 22);
    T('-4.931821627225927773384e-20', '-0.00000000000000000004931821627225927773384063578', 21);
    T('-5.245261764976094777313893054196562e-17', '-0.0000000000000000524526176497609477731389305419656234', 33);
    T('-6.66625797221972034223428591e+23', '-666625797221972034223428.590606426470365', 26);
    T('-4.06575860462e+17', '-406575860461750182.91372176567693718', 11);
    T('-8.90585675951e+19', '-89058567595113495345', 11);

    BigNumber.config({ROUNDING_MODE: 4});

    T('-2.033619450856645241153977e+0', '-2.03361945085664524115397653636144859', 24);
    T('1.130e+8', '112955590.0430616', 3);
    T('-2.1366468193419876852426155614364269e+10', '-21366468193.419876852426155614364269', 34);
    T('5.82086615659566151529e+7', '58208661.56595661515285734890860077163', 20);
    T('9.1615809372817426111208e+6', '9161580.937281742611120838868847823478250167882379624', 22);
    T('3.8976506901061164197e+1', '38.97650690106116419699490320634490920742414', 19);
    T('9.0994914931570087194607344641722310104e+6', '9099491.4931570087194607344641722310103895224905', 37);
    T('6.06e+5', '605633', 2);
    T('2.6999974790473705518992117e+1', '26.9999747904737055189921170044987', 25);
    T('6.7108801361722e+6', '6710880.136172156342982663450743452', 13);
    T('-8.0e+0', '-8', 1);
    T('3.000e-2', '0.03', 3);
    T('-4.7e+2', '-469', 1);
    T('-6.3000e+0', '-6.3', 4);
    T('-5.4e+2', '-542', 1);
    T('-5.2000e+0', '-5.2', 4);
    T('-9.00000e-2', '-0.09', 5);
    T('-3.1000e-1', '-0.31', 4);
    T('-4.4e+2', '-436', 1);
    T('-3.00e+0', '-3', 2);
    T('-5.00e-2', '-0.05', 2);
    T('1.00e-2', '0.01', 2);

    T('1.23000000000e+2', '12.3e1', '1.1e1');
    T('1.230e+2', '12.3e1', new BigNumber('3'));
    T('1.230e+2', '12.3e1', 3);
    T('1.23e+2', '12.3e1', null);
    T('1.23e+2', '12.3e1', undefined);
    T('1e+2', '12.3e1', '0');
    T('1e+2', '12.3e1', '-0');
    T('1e+2', '12.3e1', '-0.000000');
    T('1e+2', '12.3e1', 0);
    T('1e+2', '12.3e1', -0);

    assertException(function () {new BigNumber('1.23').toExponential(NaN)}, "('1.23').toExponential(NaN)");
    assertException(function () {new BigNumber('1.23').toExponential('NaN')}, "('1.23').toExponential('NaN')");
    assertException(function () {new BigNumber('1.23').toExponential([])}, "('1.23').toExponential([])");
    assertException(function () {new BigNumber('1.23').toExponential({})}, "('1.23').toExponential({})");
    assertException(function () {new BigNumber('1.23').toExponential('')}, "('1.23').toExponential('')");
    assertException(function () {new BigNumber('1.23').toExponential(' ')}, "('1.23').toExponential(' ')");
    assertException(function () {new BigNumber('1.23').toExponential('hello')}, "('1.23').toExponential('hello')");
    assertException(function () {new BigNumber('1.23').toExponential('\t')}, "('1.23').toExponential('\t')");
    assertException(function () {new BigNumber('1.23').toExponential(new Date)}, "('1.23').toExponential(new Date)");
    assertException(function () {new BigNumber('1.23').toExponential(new RegExp)}, "('1.23').toExponential(new RegExp)");
    assertException(function () {new BigNumber('1.23').toExponential(2.01)}, "('1.23').toExponential(2.01)");
    assertException(function () {new BigNumber('1.23').toExponential(10.5)}, "('1.23').toExponential(10.5)");
    assertException(function () {new BigNumber('1.23').toExponential('-1.1e1')}, "('1.23').toExponential('-1.1e1')");
    assertException(function () {new BigNumber('1.23').toExponential(true)}, "('1.23').toExponential(true)");
    assertException(function () {new BigNumber('1.23').toExponential(false)}, "('1.23').toExponential(false)");
    assertException(function () {new BigNumber('1.23').toExponential(function (){})}, "('1.23').toExponential(function (){})");

    assertException(function () {new BigNumber(12.3).toExponential('-1')}, ".toExponential('-1')");
    assertException(function () {new BigNumber(12.3).toExponential(-23)}, ".toExponential(-23)");
    assertException(function () {new BigNumber(12.3).toExponential(MAX + 1)}, ".toExponential(MAX + 1)");
    assertException(function () {new BigNumber(12.3).toExponential(MAX + 0.1)}, ".toExponential(MAX + 0.1)");
    assertException(function () {new BigNumber(12.3).toExponential(-0.01)}, ".toExponential(-0.01)");
    assertException(function () {new BigNumber(12.3).toExponential('-1e-1')}, ".toExponential('-1e-1')");
    assertException(function () {new BigNumber(12.3).toExponential(Infinity)}, ".toExponential(Infinity)");
    assertException(function () {new BigNumber(12.3).toExponential('-Infinity')}, ".toExponential('-Infinity')");

    BigNumber.config({ERRORS: false});

    T('Infinity', Infinity, 0);
    T('Infinity', Infinity, NaN);
    T('Infinity', Infinity, null);
    T('Infinity', Infinity, Infinity);
    T('NaN', NaN, -Infinity);

    T('1.230e+2', '12.3e1', BigNumber(3));
    T('1.23e+2', '12.3e1', null);
    T('1.23e+2', '12.3e1', undefined);
    T('1.23e+2', '12.3e1', NaN);
    T('1.23e+2', '12.3e1', 'NaN');
    T('1.23e+2', '12.3e1', []);
    T('1.23e+2', '12.3e1', {});
    T('1.23e+2', '12.3e1', '');
    T('1.23e+2', '12.3e1', ' ');
    T('1.23e+2', '12.3e1', 'hello');
    T('1.23e+2', '12.3e1', '\t');
    T('1.23e+2', '12.3e1', ' ');
    T('1.23e+2', '12.3e1', new Date);
    T('1.23e+2', '12.3e1', new RegExp);

    T('1e+2', '12.3e1', -0);
    T('1.2e+2', '12.3e1', 1.999);
    T('1.2300000e+2', '12.3e1', 7.5);
    T('1.23000000000e+2', '12.3e1', '1.1e1');

    T('1.23e+2', '12.3e1', '-1');
    T('1.23e+2', '12.3e1', -23);
    T('1.23e+2', '12.3e1', MAX + 1);
    T('1e+2', '12.3e1', -0.01);
    T('1e+2', '12.3e1', '-1e-1');
    T('1.23e+2', '12.3e1', Infinity);
    T('1.23e+2', '12.3e1', '-Infinity');

    log('\n ' + passed + ' of ' + total + ' tests passed in ' + (+new Date() - start) + ' ms \n');
    return [passed, total];
})(this.BigNumber);
if (typeof module !== 'undefined' && module.exports) module.exports = count;