/**
 * @fileoverview Implements WysiwygCommand
 * @author Sungho Kim(sungho-kim@nhnent.com) FE Development Team/NHN Ent.
 * @author Junghwan Park(junghwan.park@nhnent.com) FE Development Team/NHN Ent.
 */


import CommandManager from '../commandManager';
import domUtil from '../domUtils';

/**
 * RemoveRow
 * remove Row to selected table
 * @exports RemoveRow
 * @augments Command
 * @augments WysiwygCommand
 */
const RemoveRow = CommandManager.command('wysiwyg', /** @lends RemoveRow */{
    name: 'RemoveRow',
    /**
     *  커맨드 핸들러
     *  @param {WysiwygEditor} wwe WYsiwygEditor instance
     */
    exec(wwe) {
        const sq = wwe.getEditor();
        const range = sq.getSelection().cloneRange();
        const $table = $(range.startContainer).parents('table');
        const selectionMgr = wwe.getManager('tableSelection');
        const tableMgr = wwe.getManager('table');
        const $tr = getTrs(range, selectionMgr, $table);
        const tbodyRowLength = $table.find('tbody tr').length;

        sq.focus();

        if ((sq.hasFormat('TD') || sq.hasFormat('TABLE')) && tbodyRowLength > 1) {
            sq.saveUndoState(range);
            const $nextFocus = $tr.last().next()[0] ? $tr.last().next() : $tr.first().prev();

            if ($nextFocus.length) {
                focusToFirstTd(sq, range, $nextFocus, tableMgr);
            }
            $tr.remove();
        }
        selectionMgr.removeClassAttrbuteFromAllCellsIfNeed();
    }
});

/**
 * Focus to first TD in given TR
 * @param {SquireExt} sq Squire instance
 * @param {Range} range Range object
 * @param {jQuery} $tr jQuery wrapped TR
 * @param {object} tableMgr Table manager
 */
function focusToFirstTd(sq, range, $tr, tableMgr) {
    const nextFocusCell = $tr.find('td')[0];
    range.setStart(nextFocusCell, 0);
    range.collapse(true);

    tableMgr.setLastCellNode(nextFocusCell);
    sq.setSelection(range);
}

/**
 * Get start, end row index from current range
 * @param {HTMLElement} firstSelectedCell Range object
 * @param {object} rangeInformation Range information object
 * @param {jQuery} $table jquery wrapped TABLE
 * @returns {jQuery}
 */
function getSelectedRows(firstSelectedCell, rangeInformation, $table) {
    const tbodyRowLength = $table.find('tbody tr').length;
    const isStartContainerInThead = $(firstSelectedCell).parents('thead').length;
    let startRowIndex = rangeInformation.from.row;
    let endRowIndex = rangeInformation.to.row;

    if (isStartContainerInThead) {
        startRowIndex += 1;
    }

    const isWholeTbodySelected = (startRowIndex === 1 || isStartContainerInThead) && endRowIndex === tbodyRowLength;

    if (isWholeTbodySelected) {
        endRowIndex -= 1;
    }

    return $table.find('tr').slice(startRowIndex, endRowIndex + 1);
}

/**
 * Get TRs
 * @param {Range} range Range object
 * @param {object} selectionMgr Table selection manager
 * @param {jQuery} $table current table
 * @returns {jQuery}
 */
function getTrs(range, selectionMgr, $table) {
    const selectedCells = selectionMgr.getSelectedCells();
    let rangeInformation, trs, startCell, endCell;

    if (selectedCells.length) {
        rangeInformation = selectionMgr.getSelectionRangeFromTable(selectedCells.first()[0],
            selectedCells.last()[0]);
        trs = getSelectedRows(selectedCells.first()[0], rangeInformation, $table);
    } else {
        const startContainer = range.startContainer;
        const endContainer = range.endContainer;

        startCell = domUtil.isTextNode(startContainer) ? $(startContainer).parent('td,th')[0] : startContainer;
        endCell = domUtil.isTextNode(endContainer) ? $(endContainer).parent('td,th')[0] : endContainer;
        rangeInformation = selectionMgr.getSelectionRangeFromTable(startCell, endCell);
        trs = getSelectedRows(startCell, rangeInformation, $table);
    }

    return trs;
}
module.exports = RemoveRow;
