
/*!
 * @name jQuery Apply Field plugin
 * @version 1.0.0
 * @requires:
 *   jquery >= 1.7
 *   jquery-extends >= 1.0.0
 */

(function() {
  (function(factory) {
    if (typeof define === 'function' && define.amd) {
      return define(['jquery-extends'], function($) {
        return factory($);
      });
    } else if (typeof exports === 'object') {
      return factory(require('jquery-extends'));
    } else {
      return factory(jQuery);
    }
  })(function($) {
    'use strict';

    /**
     * @description Converter Store.
     */
    var ConverterProvider, converterStore, normalizeConverters;
    converterStore = {
      log: function() {
        console.log(this);
      },
      array: function() {
        this.array = true;
      },
      ignoreEmpty: function() {
        return (this.value != null) && this.value !== '';
      },
      ignoreBlank: function() {
        return (this.value != null) && this.value.trim() !== '';
      }
    };

    /**
     * @description Normalize Converters
     * @param {string|function|array} converters then converters.
     */
    normalizeConverters = function(converters) {
      if (!$.isArray(converters)) {
        converters = typeof converters === 'string' ? converters.split(/\s+/) : [converters];
      }
      return $.map(converters, function(converter) {
        if (typeof converter !== 'string') {
          return converter;
        }
        if (converter !== '') {
          if (!(converter in converterStore)) {
            $.error("Converter not found : " + converter);
          }
          return converterStore[converter];
        }
      });
    };

    /**
     * @description
     *
     * Converter Provider.
     * Converter 우선순위: general &gt; regex &gt; defaults
     * ex1) { '*': 'ignoreEmpty array' }
     * ex2) { '*': ['ignoreEmpty', 'array', function(){}]
     *
     * @param {object} rawConverters 가공되지 않은 Converters
     */
    ConverterProvider = (function() {
      function ConverterProvider(rawConverters) {
        var converters, flag, key, lsp, pattern, value;
        if (!$.isPlainObject(rawConverters)) {
          $.error("RawConverters must be plain object");
        }
        this.general = {};
        this.regex = [];
        this.defaults = [];
        for (key in rawConverters) {
          value = rawConverters[key];
          converters = normalizeConverters(value);
          if (/^\*+$/.test(key)) {
            $.merge(this.defaults, converters);
          } else if (key.charAt(0) === '/' && key.charAt(1) !== '/' && (lsp = key.lastIndexOf('/')) > 1) {
            pattern = key.slice(1, lsp);
            flag = key.slice(lsp + 1);
            this.regex.push({
              regex: new RegExp(pattern, flag),
              converters: converters
            });
          } else if (key.contains('*')) {
            this.regex.push({
              regex: new RegExp('^' + key.replace(/\*+/g, '.*') + '$'),
              converters: converters
            });
          } else {
            this.general[key] = converters;
          }
        }
      }


      /**
       * @description
       * 파라미터이름으로 Converter를 찾는다.
       *
       * @param {string} param 파라미터 이름
       * @param {string} [root] 파라미터 루트
       * @returns {*}
       */

      ConverterProvider.prototype.find = function(param, root) {
        var converters, r, _i, _len, _ref;
        if (typeof param !== 'string') {
          $.error("Param must be string");
        }
        if (root) {
          if (!param.startsWith(root)) {
            $.error("Param must be starts with '" + root + "'");
          }
          param = param.slice(root.length);
          if (param.charAt(0) === '.') {
            param = param.slice(1);
          }
        }
        converters = [];
        if (param in this.general) {
          $.merge(converters, this.general[param]);
        }
        _ref = this.regex;
        for (_i = 0, _len = _ref.length; _i < _len; _i++) {
          r = _ref[_i];
          if (r.regex.test(param)) {
            $.merge(converters, r.converters);
          }
        }
        return $.merge(converters, this.defaults);
      };

      return ConverterProvider;

    })();

    /**
     * @param {boolean} [isDefault] 기본값 사용 여부 (default is false)
     * @param {string} [root] 파라미터 루트 (default is "")
     * @param {object} data 데이타
     * @param {object|ConverterProvider} [converterProvider] Converters
     * @param {function} [callback] Callback (default is $.noop)
     * @param {boolean} [isDebug] 디버깅 메시지 출력 여부 (default is false)
     */
    $.fn.applyField = function(isDefault, root, data, converterProvider, callback, isDebug) {
      var PROP_NAME, args;
      if (!(this.length > 0)) {
        return this;
      }
      PROP_NAME = '$applyField';
      args = $.makeArray(arguments);
      isDefault = typeof args[0] === 'boolean' ? args.shift() : false;
      root = typeof args[0] === 'string' ? args.shift() : '';
      data = args.shift();
      converterProvider = args[0] instanceof ConverterProvider ? args.shift() : new ConverterProvider($.isPlainObject(args[0]) ? args.shift() : {});
      callback = $.isFunction(args[0]) ? args.shift() : $.noop;
      isDebug = typeof args[0] === 'boolean' ? args.shift() : false;
      this.filter('form').each(function() {
        var $this, applyField;
        $this = $(this);
        if (isDefault) {
          $this.data(PROP_NAME, [root, data, converterProvider, callback]);
          $this.off('reset.' + PROP_NAME);
          $this.on('reset.' + PROP_NAME, function() {
            $.nextTick(function() {
              data = $this.data(PROP_NAME);
              if (data) {
                return $this.applyField.apply($this, data);
              }
            });
          });
        }
        return (applyField = function(form, param) {
          var converter, elements, i, k, key, multiples, name, textbox, textboxes, v, val, value, _i, _j, _k, _len, _len1, _len2, _ref;
          _ref = converterProvider.find(param.name, root);
          for (_i = 0, _len = _ref.length; _i < _len; _i++) {
            converter = _ref[_i];
            if ($.isFunction(converter)) {
              converter = converter.call(param, root, data, form);
            }
            if (converter === false) {
              return;
            }
            if (converter === true) {
              continue;
            }
            if (typeof converter !== 'undefined') {
              param.value = converter;
            }
          }
          name = param.name, value = param.value;
          if (param.array) {
            if (!$.isArray(value)) {
              value = [value];
            }
            for (i = _j = 0, _len1 = value.length; _j < _len1; i = ++_j) {
              val = value[i];
              if ($.isPlainObject(val)) {
                for (k in val) {
                  v = val[k];
                  $.nextTick(function() {
                    return applyField(form, {
                      name: name ? "" + name + "[" + i + "]." + k : "" + k + "[" + i + "]",
                      value: v
                    });
                  });
                }
              } else {
                $.nextTick(function() {
                  return applyField(form, {
                    name: "" + name + "[" + i + "]",
                    value: val
                  });
                });
              }
            }
          } else if ($.isPlainObject(value)) {
            for (key in value) {
              val = value[key];
              $.nextTick(function() {
                return applyField(form, {
                  name: (name ? name + '.' : '') + key,
                  value: val
                });
              });
            }
          } else {
            elements = $(form).findByName(name);
            multiples = elements.filter(':checkbox, :radio, select');
            textboxes = elements.not(multiples);
            if (multiples.length > 0) {
              multiples.val($.isArray(value) ? value : [value]);
            }
            if (textboxes.length === 1) {
              if (typeof value !== 'undefined') {
                textboxes.val(value);
              }
            } else if (textboxes.length > 1) {
              if ($.isArray(value)) {
                for (i = _k = 0, _len2 = textboxes.length; _k < _len2; i = ++_k) {
                  textbox = textboxes[i];
                  if (!(i < value.length)) {
                    break;
                  }
                  if (typeof value[i] !== 'undefined') {
                    $(textbox).val(value[i]);
                  }
                }
              } else if (typeof value !== 'undefined') {
                textboxes.first().val(value);
              }
            }
            if (elements.length > 0) {
              callback.call(elements, elements, value);
            }
            if (isDebug) {
              console.log("%s: %o, %o", name, value, elements.get());
            }
          }
        })(this, {
          name: root,
          value: data
        });
      });
      return this;
    };
    return $.applyField = {
      converters: converterStore,
      ConverterProvider: ConverterProvider
    };
  });

}).call(this);

//# sourceMappingURL=jquery-applyfield.js.map
