
/*!
 * @name jQuery Hangul plugin
 * @version 1.0.0
 * @requires:
 *   jquery >= 1.7
 * @description
 *   한글 처리를 위한 유틸 클래스
 *   유니코드 2.0 한글의 범위 * AC00 ~ D7A3
 */

(function() {
  (function(factory) {
    if (typeof define === 'function' && define.amd) {
      return define(['jquery'], function($) {
        return factory($);
      });
    } else if (typeof exports === 'object') {
      return factory(require('jquery'));
    } else {
      return factory(jQuery);
    }
  })(function($) {
    'use strict';

    /**
     * @type {string[]} 초성
     */
    var FINAL_CONSONANTS, INITIAL_CONSONANTS, LARGE_NUMBERS, NUMBERS, VOWELS;
    INITIAL_CONSONANTS = ['ㄱ', 'ㄲ', 'ㄴ', 'ㄷ', 'ㄸ', 'ㄹ', 'ㅁ', 'ㅂ', 'ㅃ', 'ㅅ', 'ㅆ', 'ㅇ', 'ㅈ', 'ㅉ', 'ㅊ', 'ㅋ', 'ㅌ', 'ㅍ', 'ㅎ'];

    /**
     * @type {string[]} 중성
     */
    VOWELS = ['ㅏ', 'ㅐ', 'ㅑ', 'ㅒ', 'ㅓ', 'ㅔ', 'ㅕ', 'ㅖ', 'ㅗ', 'ㅘ', 'ㅙ', 'ㅚ', 'ㅛ', 'ㅜ', 'ㅝ', 'ㅞ', 'ㅟ', 'ㅠ', 'ㅡ', 'ㅢ', 'ㅣ'];

    /**
     * @type {string[]} 종성
     */
    FINAL_CONSONANTS = [' ', 'ㄱ', 'ㄲ', 'ㄳ', 'ㄴ', 'ㄵ', 'ㄶ', 'ㄷ', 'ㄹ', 'ㄺ', 'ㄻ', 'ㄼ', 'ㄽ', 'ㄾ', 'ㄿ', 'ㅀ', 'ㅁ', 'ㅂ', 'ㅄ', 'ㅅ', 'ㅆ', 'ㅇ', 'ㅈ', 'ㅊ', 'ㅋ', 'ㅌ', 'ㅍ', 'ㅎ'];

    /**
     * @type {string[]} 한글 숫자
     */
    NUMBERS = ['영', '일', '이', '삼', '사', '오', '육', '칠', '팔', '구'];

    /**
     * @description
     *
     * 숫자 단위 테이블
     *
     * @type {string[][]} 숫자 단위
     */
    LARGE_NUMBERS = [['십', '백', '천'], ['만', '억', '조', '경']];
    $.hangul = {

      /**
       * @description 문자열이 한글인지 검사한다
       * @param {string} s 문자열
       * @param {number} [l] 검사길이
       * @return {boolean} 검사결과
       */
      isHangul: function(s, l) {
        var c, i, _i;
        if (!(typeof s === 'string' && s.length > 0)) {
          return false;
        }
        if (!l || l < 0 || l > s.length) {
          l = s.length;
        }
        for (i = _i = 0; 0 <= l ? _i < l : _i > l; i = 0 <= l ? ++_i : --_i) {
          c = s.charCodeAt(i);
          if (c < 0xAC00 || c > 0xD7A3) {
            return false;
          }
        }
        return true;
      },

      /**
       * @description 한글 한문자에서 초성을 분리해서 가져온다
       * @param {string} s 문자
       * @return {string} 초성
       */
      initial: function(s) {
        if (!$.hangul.isHangul(s, 1)) {
          return (typeof s === 'string' ? s.charAt(0) : s);
        }
        return INITIAL_CONSONANTS[parseInt(((s.charCodeAt(0) & 0xFFFF) - 0xAC00) / (21 * 28))];
      },

      /**
       * @description 한글 한문자에서 중성을 분리해서 가져온다
       * @param {string} s 문자
       * @return {string} 중성
       */
      vowel: function(s) {
        if (!$.hangul.isHangul(s, 1)) {
          return (typeof s === 'string' ? s.charAt(0) : s);
        }
        return VOWELS[parseInt((((s.charCodeAt(0) & 0xFFFF) - 0xAC00) % (21 * 28)) / 28)];
      },

      /**
       * @description 한글 한문자에서 종성을 분리해서 가져온다
       * @param {string} s 문자
       * @return {string} 종성
       */
      under: function(s) {
        if (!$.hangul.isHangul(s, 1)) {
          return (typeof s === 'string' ? s.charAt(0) : s);
        }
        return FINAL_CONSONANTS[parseInt((((s.charCodeAt(0) & 0xFFFF) - 0xAC00) % (21 * 28)) % 28)];
      },

      /**
       * @description 한문자의 종성의 유무를 검사한다
       * @param {string} s 문자
       * @return {boolean} 종성의 유무
       */
      hasUnder: function(s) {
        return $.hangul.isHangul(s, 1) && $.hangul.under(s) !== ' ';
      },

      /**
       * @description 문자열의 마지막 문자의 종성 유무를 검사한다
       * @param {string} s 문자열
       * @return {boolean} 마지막 문자의 종성 유무
       */
      hasLastUnder: function(s) {
        if (!(typeof s === 'string' && s.length > 0)) {
          return false;
        }
        return $.hangul.hasUnder(s.charAt(s.length - 1));
      },

      /**
       * @description 숫자를 한글형식으로 바꿔준다
       * @param {number} number 숫자
       * @param {string} [delimiter] 단위 구분자
       * @return {string} 한글숫자
       */
      toHangulNumber: function(number, delimiter) {
        var hnum, hnums, i, j, lunit, n, nums, unit, _i, _ref;
        if (typeof number === 'undefined') {
          $.error("'number' is undefined");
        }
        if (typeof number !== 'string') {
          number = number.toString();
        }
        if (!/^[0-9]+$/.test(number)) {
          return number;
        }
        if (number.charAt(0) === '0') {
          for (i = _i = 1, _ref = number.length; 1 <= _ref ? _i < _ref : _i > _ref; i = 1 <= _ref ? ++_i : --_i) {
            if (number.charAt(i) === '0') {
              continue;
            }
            number = number.substring(i);
            break;
          }
        }
        if (number.length === 0) {
          return NUMBERS[0];
        }
        if (number.length === 1) {
          return NUMBERS[number.charAt(0)];
        }
        return ((function() {
          var _j, _len, _ref1, _results;
          _ref1 = number.split('').map(function(a) {
            return parseInt(a);
          }).reverse().grouped(4);
          _results = [];
          for (i = _j = 0, _len = _ref1.length; _j < _len; i = ++_j) {
            nums = _ref1[i];
            hnums = ((function() {
              var _k, _len1, _results1;
              _results1 = [];
              for (j = _k = 0, _len1 = nums.length; _k < _len1; j = ++_k) {
                n = nums[j];
                hnum = n > 1 || j === 0 && n === 1 ? NUMBERS[n] : '';
                unit = n > 0 && j > 0 ? LARGE_NUMBERS[0][j - 1] : '';
                _results1.push(hnum + unit);
              }
              return _results1;
            })()).reduce(function(a, b) {
              return b + a;
            });
            lunit = i > 0 && hnums.length > 0 ? LARGE_NUMBERS[1][(i - 1) % 4] : '';
            _results.push(hnums + lunit);
          }
          return _results;
        })()).filter(function(a) {
          return a.length > 0;
        }).reduce(function(a, b) {
          return b + (delimiter || '') + a;
        });
      }
    };
    String.prototype.isHangul = function(l) {
      return $.hangul.isHangul(this.toString(), l);
    };
    String.prototype.hangulInitial = function() {
      return $.hangul.initial(this.toString());
    };
    String.prototype.hangulVowel = function() {
      return $.hangul.vowel(this.toString());
    };
    String.prototype.hangulUnder = function() {
      return $.hangul.under(this.toString());
    };
    String.prototype.hasHangulUnder = function() {
      return $.hangul.hasUnder(this.toString());
    };
    String.prototype.hasLastHangulUnder = function() {
      return $.hangul.hasLastUnder(this.toString());
    };
    String.prototype.toHangulNumber = function(d) {
      return $.hangul.toHangulNumber(this.toString(), d);
    };
    Number.prototype.toHangul = function(d) {
      return $.hangul.toHangulNumber(this.toString(), d);
    };
    return $.hangul;
  });

}).call(this);

//# sourceMappingURL=jquery-hangul.js.map
