###!
# @name jQuery Hangul plugin
# @version 1.0.0
# @requires:
#   jquery >= 1.7
# @description
#   한글 처리를 위한 유틸 클래스
#   유니코드 2.0 한글의 범위 * AC00 ~ D7A3
###

((factory) ->
  if typeof define is 'function' and define.amd
    # AMD. Register as an anonymous module.
    define ['jquery'], ($) -> factory $
  else if typeof exports is 'object'
    # CommonJS
    factory require('jquery')
  else
    # Browser globals
    factory jQuery
) ($) ->
  'use strict'

  ###*
  # @type {string[]} 초성
  ###
  INITIAL_CONSONANTS = ['ㄱ','ㄲ','ㄴ','ㄷ','ㄸ','ㄹ','ㅁ','ㅂ','ㅃ','ㅅ','ㅆ','ㅇ','ㅈ','ㅉ','ㅊ','ㅋ','ㅌ','ㅍ','ㅎ']
  ###*
  # @type {string[]} 중성
  ###
  VOWELS = ['ㅏ','ㅐ','ㅑ','ㅒ','ㅓ','ㅔ','ㅕ','ㅖ','ㅗ','ㅘ','ㅙ','ㅚ','ㅛ','ㅜ','ㅝ','ㅞ','ㅟ','ㅠ','ㅡ','ㅢ','ㅣ']
  ###*
  # @type {string[]} 종성
  ###
  FINAL_CONSONANTS = [' ','ㄱ','ㄲ','ㄳ','ㄴ','ㄵ','ㄶ','ㄷ','ㄹ','ㄺ','ㄻ','ㄼ','ㄽ','ㄾ','ㄿ','ㅀ','ㅁ','ㅂ','ㅄ','ㅅ','ㅆ','ㅇ','ㅈ','ㅊ','ㅋ','ㅌ','ㅍ','ㅎ']

  ###*
  # @type {string[]} 한글 숫자
  ###
  NUMBERS = ['영','일','이','삼','사','오','육','칠','팔','구']

  ###*
  # @description
  #
  # 숫자 단위 테이블
  #
  # @type {string[][]} 숫자 단위
  ###
  LARGE_NUMBERS = [['십','백','천'],['만','억','조','경']]

  $.hangul =

    ###*
    # @description 문자열이 한글인지 검사한다
    # @param {string} s 문자열
    # @param {number} [l] 검사길이
    # @return {boolean} 검사결과
    ###
    isHangul: (s, l) ->
      return false  unless typeof s is 'string' and s.length > 0
      l = s.length  if not l or l < 0 or l > s.length
      for i in [0...l]
        c = s.charCodeAt(i)
        return false  if c < 0xAC00 or c > 0xD7A3
      true

    ###*
    # @description 한글 한문자에서 초성을 분리해서 가져온다
    # @param {string} s 문자
    # @return {string} 초성
    ###
    initial: (s) ->
      return (if typeof s is 'string' then s.charAt(0) else s)  unless $.hangul.isHangul(s, 1)
      return INITIAL_CONSONANTS[parseInt ((s.charCodeAt(0) & 0xFFFF) - 0xAC00) / (21 * 28)]

    ###*
    # @description 한글 한문자에서 중성을 분리해서 가져온다
    # @param {string} s 문자
    # @return {string} 중성
    ###
    vowel: (s) ->
      return (if typeof s is 'string' then s.charAt(0) else s)  unless $.hangul.isHangul(s, 1)
      return VOWELS[parseInt (((s.charCodeAt(0) & 0xFFFF) - 0xAC00) % (21 * 28)) / 28]

    ###*
    # @description 한글 한문자에서 종성을 분리해서 가져온다
    # @param {string} s 문자
    # @return {string} 종성
    ###
    under: (s) ->
      return (if typeof s is 'string' then s.charAt(0) else s)  unless $.hangul.isHangul(s, 1)
      return FINAL_CONSONANTS[parseInt (((s.charCodeAt(0) & 0xFFFF) - 0xAC00) % (21 * 28)) % 28]

    ###*
    # @description 한문자의 종성의 유무를 검사한다
    # @param {string} s 문자
    # @return {boolean} 종성의 유무
    ###
    hasUnder: (s) -> $.hangul.isHangul(s, 1) and $.hangul.under(s) isnt ' '

    ###*
    # @description 문자열의 마지막 문자의 종성 유무를 검사한다
    # @param {string} s 문자열
    # @return {boolean} 마지막 문자의 종성 유무
    ###
    hasLastUnder: (s) ->
      return false  unless typeof s is 'string' and s.length > 0
      return $.hangul.hasUnder(s.charAt(s.length - 1))

    ###*
    # @description 숫자를 한글형식으로 바꿔준다
    # @param {number} number 숫자
    # @param {string} [delimiter] 단위 구분자
    # @return {string} 한글숫자
    ###
    toHangulNumber: (number, delimiter) ->
      $.error "'number' is undefined"  if typeof number is 'undefined'
      number = number.toString()  unless typeof number is 'string'
      return number  unless /^[0-9]+$/.test number

      if number.charAt(0) is '0'
        for i in [1...number.length]
          continue  if number.charAt(i) is '0'
          number = number.substring(i)
          break

      return NUMBERS[0]  if number.length is 0
      return NUMBERS[number.charAt(0)]  if number.length is 1

      (for nums, i in number.split('').map((a) -> parseInt(a)).reverse().grouped(4)
        hnums = (for n, j in nums
          hnum = if n > 1 or j is 0 and n is 1 then NUMBERS[n] else ''
          unit = if n > 0 and j > 0 then LARGE_NUMBERS[0][j - 1] else ''
          hnum + unit
        ).reduce((a, b) -> b + a)
        lunit = if i > 0 and hnums.length > 0 then LARGE_NUMBERS[1][(i - 1) % 4] else ''
        hnums + lunit
      ).filter((a) -> a.length > 0).reduce((a, b) -> b + (delimiter or '') + a)
    #end of toHangulNumber: (number, delimiter) ->

  #end of $.hangul

  # 문자열 객체의 프로토타입 정의
  String::isHangul = (l) -> $.hangul.isHangul(@toString(), l)
  String::hangulInitial = -> $.hangul.initial(@toString())
  String::hangulVowel = -> $.hangul.vowel(@toString())
  String::hangulUnder = -> $.hangul.under(@toString())
  String::hasHangulUnder = -> $.hangul.hasUnder(@toString())
  String::hasLastHangulUnder = -> $.hangul.hasLastUnder(@toString())
  String::toHangulNumber = (d) -> $.hangul.toHangulNumber(@toString(), d)

  # 숫자형식의 객체의 프로토타입 정의
  Number::toHangul = (d) -> $.hangul.toHangulNumber(@toString(), d)

  # Exports
  $.hangul
