define(['./internal/baseIteratee', './internal/baseWhile'], function(baseIteratee, baseWhile) {

  /**
   * Creates a slice of `array` excluding elements dropped from the end.
   * Elements are dropped until `predicate` returns falsey. The predicate is
   * invoked with three arguments: (value, index, array).
   *
   * @static
   * @memberOf _
   * @category Array
   * @param {Array} array The array to query.
   * @param {Function|Object|string} [predicate=_.identity] The function invoked per iteration.
   * @returns {Array} Returns the slice of `array`.
   * @example
   *
   * var users = [
   *   { 'user': 'barney',  'active': true },
   *   { 'user': 'fred',    'active': false },
   *   { 'user': 'pebbles', 'active': false }
   * ];
   *
   * _.dropRightWhile(users, function(o) { return !o.active; });
   * // => objects for ['barney']
   *
   * // using the `_.matches` iteratee shorthand
   * _.dropRightWhile(users, { 'user': 'pebbles', 'active': false });
   * // => objects for ['barney', 'fred']
   *
   * // using the `_.matchesProperty` iteratee shorthand
   * _.dropRightWhile(users, ['active', false]);
   * // => objects for ['barney']
   *
   * // using the `_.property` iteratee shorthand
   * _.dropRightWhile(users, 'active');
   * // => objects for ['barney', 'fred', 'pebbles']
   */
  function dropRightWhile(array, predicate) {
    return (array && array.length)
      ? baseWhile(array, baseIteratee(predicate, 3), true, true)
      : [];
  }

  return dropRightWhile;
});
