#!/usr/bin/env node
/*eslint no-console:0*/

'use strict';


var fs = require('fs');
var argparse = require('argparse');


////////////////////////////////////////////////////////////////////////////////

var cli = new argparse.ArgumentParser({
  prog: 'markdown-it',
  version: require('../package.json').version,
  addHelp: true
});

cli.addArgument([ '--no-html' ], {
  help:   'Disable embedded HTML',
  action: 'storeTrue'
});

cli.addArgument([ '-l', '--linkify' ], {
  help:   'Autolink text',
  action: 'storeTrue'
});

cli.addArgument([ '-t', '--typographer' ], {
  help:   'Enable smartquotes and other typographic replacements',
  action: 'storeTrue'
});

cli.addArgument([ '--trace' ], {
  help:   'Show stack trace on error',
  action: 'storeTrue'
});

cli.addArgument([ 'file' ], {
  help: 'File to read',
  nargs: '?',
  defaultValue: '-'
});

var options = cli.parseArgs();


function readFile(filename, encoding, callback) {
  if (options.file === '-') {
    // read from stdin
    var chunks = [];

    process.stdin.on('data', function(chunk) { chunks.push(chunk); });

    process.stdin.on('end', function() {
      return callback(null, Buffer.concat(chunks).toString(encoding));
    });
  } else {
    fs.readFile(filename, encoding, callback);
  }
}


////////////////////////////////////////////////////////////////////////////////

readFile(options.file, 'utf8', function (err, input) {
  var output, md;

  if (err) {
    if (err.code === 'ENOENT') {
      console.error('File not found: ' + options.file);
      process.exit(2);
    }

    console.error(
      options.trace && err.stack ||
      err.message ||
      String(err));

    process.exit(1);
  }

  md = require('..')({
    html: !options['no-html'],
    xhtmlOut: false,
    typographer: options.typographer,
    linkify: options.linkify
  });

  try {
    output = md.render(input);

  } catch (e) {
    console.error(
      options.trace && e.stack ||
      e.message ||
      String(e));

    process.exit(1);
  }

  process.stdout.write(output);

  process.exit(0);
});
