require('./helper');

// A map of templates to their expected token output. Tokens are in the format:
// [type, value, startIndex, endIndex, subTokens].
var expectations = {
  ''                                        : [],
  '{{hi}}'                                  : [ [ 'name', 'hi', 0, 6 ] ],
  '{{hi.world}}'                            : [ [ 'name', 'hi.world', 0, 12 ] ],
  '{{hi . world}}'                          : [ [ 'name', 'hi . world', 0, 14 ] ],
  '{{ hi}}'                                 : [ [ 'name', 'hi', 0, 7 ] ],
  '{{hi }}'                                 : [ [ 'name', 'hi', 0, 7 ] ],
  '{{ hi }}'                                : [ [ 'name', 'hi', 0, 8 ] ],
  '{{{hi}}}'                                : [ [ '&', 'hi', 0, 8 ] ],
  '{{!hi}}'                                 : [ [ '!', 'hi', 0, 7 ] ],
  '{{! hi}}'                                : [ [ '!', 'hi', 0, 8 ] ],
  '{{! hi }}'                               : [ [ '!', 'hi', 0, 9 ] ],
  '{{ !hi}}'                                : [ [ '!', 'hi', 0, 8 ] ],
  '{{ ! hi}}'                               : [ [ '!', 'hi', 0, 9 ] ],
  '{{ ! hi }}'                              : [ [ '!', 'hi', 0, 10 ] ],
  'a\n b'                                   : [ [ 'text', 'a\n b', 0, 4 ] ],
  'a{{hi}}'                                 : [ [ 'text', 'a', 0, 1 ], [ 'name', 'hi', 1, 7 ] ],
  'a {{hi}}'                                : [ [ 'text', 'a ', 0, 2 ], [ 'name', 'hi', 2, 8 ] ],
  ' a{{hi}}'                                : [ [ 'text', ' a', 0, 2 ], [ 'name', 'hi', 2, 8 ] ],
  ' a {{hi}}'                               : [ [ 'text', ' a ', 0, 3 ], [ 'name', 'hi', 3, 9 ] ],
  'a{{hi}}b'                                : [ [ 'text', 'a', 0, 1 ], [ 'name', 'hi', 1, 7 ], [ 'text', 'b', 7, 8 ] ],
  'a{{hi}} b'                               : [ [ 'text', 'a', 0, 1 ], [ 'name', 'hi', 1, 7 ], [ 'text', ' b', 7, 9 ] ],
  'a{{hi}}b '                               : [ [ 'text', 'a', 0, 1 ], [ 'name', 'hi', 1, 7 ], [ 'text', 'b ', 7, 9 ] ],
  'a\n{{hi}} b \n'                          : [ [ 'text', 'a\n', 0, 2 ], [ 'name', 'hi', 2, 8 ], [ 'text', ' b \n', 8, 12 ] ],
  'a\n {{hi}} \nb'                          : [ [ 'text', 'a\n ', 0, 3 ], [ 'name', 'hi', 3, 9 ], [ 'text', ' \nb', 9, 12 ] ],
  'a\n {{!hi}} \nb'                         : [ [ 'text', 'a\n', 0, 2 ], [ '!', 'hi', 3, 10 ], [ 'text', 'b', 12, 13 ] ],
  'a\n{{#a}}{{/a}}\nb'                      : [ [ 'text', 'a\n', 0, 2 ], [ '#', 'a', 2, 8, [], 8 ], [ 'text', 'b', 15, 16 ] ],
  'a\n {{#a}}{{/a}}\nb'                     : [ [ 'text', 'a\n', 0, 2 ], [ '#', 'a', 3, 9, [], 9 ], [ 'text', 'b', 16, 17 ] ],
  'a\n {{#a}}{{/a}} \nb'                    : [ [ 'text', 'a\n', 0, 2 ], [ '#', 'a', 3, 9, [], 9 ], [ 'text', 'b', 17, 18 ] ],
  'a\n{{#a}}\n{{/a}}\nb'                    : [ [ 'text', 'a\n', 0, 2 ], [ '#', 'a', 2, 8, [], 9 ], [ 'text', 'b', 16, 17 ] ],
  'a\n {{#a}}\n{{/a}}\nb'                   : [ [ 'text', 'a\n', 0, 2 ], [ '#', 'a', 3, 9, [], 10 ], [ 'text', 'b', 17, 18 ] ],
  'a\n {{#a}}\n{{/a}} \nb'                  : [ [ 'text', 'a\n', 0, 2 ], [ '#', 'a', 3, 9, [], 10 ], [ 'text', 'b', 18, 19 ] ],
  'a\n{{#a}}\n{{/a}}\n{{#b}}\n{{/b}}\nb'    : [ [ 'text', 'a\n', 0, 2 ], [ '#', 'a', 2, 8, [], 9 ], [ '#', 'b', 16, 22, [], 23 ], [ 'text', 'b', 30, 31 ] ],
  'a\n {{#a}}\n{{/a}}\n{{#b}}\n{{/b}}\nb'   : [ [ 'text', 'a\n', 0, 2 ], [ '#', 'a', 3, 9, [], 10 ], [ '#', 'b', 17, 23, [], 24 ], [ 'text', 'b', 31, 32 ] ],
  'a\n {{#a}}\n{{/a}}\n{{#b}}\n{{/b}} \nb'  : [ [ 'text', 'a\n', 0, 2 ], [ '#', 'a', 3, 9, [], 10 ], [ '#', 'b', 17, 23, [], 24 ], [ 'text', 'b', 32, 33 ] ],
  'a\n{{#a}}\n{{#b}}\n{{/b}}\n{{/a}}\nb'    : [ [ 'text', 'a\n', 0, 2 ], [ '#', 'a', 2, 8, [ [ '#', 'b', 9, 15, [], 16 ] ], 23 ], [ 'text', 'b', 30, 31 ] ],
  'a\n {{#a}}\n{{#b}}\n{{/b}}\n{{/a}}\nb'   : [ [ 'text', 'a\n', 0, 2 ], [ '#', 'a', 3, 9, [ [ '#', 'b', 10, 16, [], 17 ] ], 24 ], [ 'text', 'b', 31, 32 ] ],
  'a\n {{#a}}\n{{#b}}\n{{/b}}\n{{/a}} \nb'  : [ [ 'text', 'a\n', 0, 2 ], [ '#', 'a', 3, 9, [ [ '#', 'b', 10, 16, [], 17 ] ], 24 ], [ 'text', 'b', 32, 33 ] ],
  '{{>abc}}'                                : [ [ '>', 'abc', 0, 8, '', 0, false ] ],
  '{{> abc }}'                              : [ [ '>', 'abc', 0, 10, '', 0, false ] ],
  '{{ > abc }}'                             : [ [ '>', 'abc', 0, 11, '', 0, false ] ],
  '  {{> abc }}\n'                          : [ [ '>', 'abc', 2, 12, '  ', 0, false ] ],
  '  {{> abc }} {{> abc }}\n'               : [ [ '>', 'abc', 2, 12, '  ', 0, false ], [ '>', 'abc', 13, 23, '   ', 1, false ] ],
  '{{ > abc }}'                             : [ [ '>', 'abc', 0, 11, '', 0, false ] ],
  '{{=<% %>=}}'                             : [ [ '=', '<% %>', 0, 11 ] ],
  '{{= <% %> =}}'                           : [ [ '=', '<% %>', 0, 13 ] ],
  '{{=<% %>=}}<%={{ }}=%>'                  : [ [ '=', '<% %>', 0, 11 ], [ '=', '{{ }}', 11, 22 ] ],
  '{{=<% %>=}}<%hi%>'                       : [ [ '=', '<% %>', 0, 11 ], [ 'name', 'hi', 11, 17 ] ],
  '{{#a}}{{/a}}hi{{#b}}{{/b}}\n'            : [ [ '#', 'a', 0, 6, [], 6 ], [ 'text', 'hi', 12, 14 ], [ '#', 'b', 14, 20, [], 20 ], [ 'text', '\n', 26, 27 ] ],
  '{{a}}\n{{b}}\n\n{{#c}}\n{{/c}}\n'        : [ [ 'name', 'a', 0, 5 ], [ 'text', '\n', 5, 6 ], [ 'name', 'b', 6, 11 ], [ 'text', '\n\n', 11, 13 ], [ '#', 'c', 13, 19, [], 20 ] ],
  '{{#foo}}\n  {{#a}}\n    {{b}}\n  {{/a}}\n{{/foo}}\n'
                                            : [ [ '#', 'foo', 0, 8, [ [ '#', 'a', 11, 17, [ [ 'text', '    ', 18, 22 ], [ 'name', 'b', 22, 27 ], [ 'text', '\n', 27, 28 ] ], 30 ] ], 37 ] ]
};

var originalTemplateCache;
before(function () {
  originalTemplateCache = Mustache.templateCache;
});

beforeEach(function (){
  Mustache.clearCache();
  Mustache.templateCache = originalTemplateCache;
});

describe('Mustache.parse', function () {

  for (var template in expectations) {
    (function (template, tokens) {
      it('knows how to parse ' + JSON.stringify(template), function () {
        assert.deepEqual(Mustache.parse(template), tokens);
      });
    })(template, expectations[template]);
  }

  describe('when there is an unclosed tag', function () {
    it('throws an error', function () {
      assert.throws(function () {
        Mustache.parse('My name is {{name');
      }, /unclosed tag at 17/i);
    });
  });

  describe('when there is an unclosed section', function () {
    it('throws an error', function () {
      assert.throws(function () {
        Mustache.parse('A list: {{#people}}{{name}}');
      }, /unclosed section "people" at 27/i);
    });
  });

  describe('when there is an unopened section', function () {
    it('throws an error', function () {
      assert.throws(function () {
        Mustache.parse('The end of the list! {{/people}}');
      }, /unopened section "people" at 21/i);
    });
  });

  describe('when invalid tags are given as an argument', function () {
    it('throws an error', function () {
      assert.throws(function () {
        Mustache.parse('A template <% name %>', [ '<%' ]);
      }, /invalid tags/i);
    });
  });

  describe('when the template contains invalid tags', function () {
    it('throws an error', function () {
      assert.throws(function () {
        Mustache.parse('A template {{=<%=}}');
      }, /invalid tags/i);
    });
  });

  describe('when parsing a template without tags specified followed by the same template with tags specified', function () {
    it('returns different tokens for the latter parse', function () {
      var template = '{{foo}}[bar]';
      var parsedWithBraces = Mustache.parse(template);
      var parsedWithBrackets = Mustache.parse(template, ['[', ']']);
      assert.notDeepEqual(parsedWithBrackets, parsedWithBraces);
    });
  });

  describe('when parsing a template with tags specified followed by the same template with different tags specified', function () {
    it('returns different tokens for the latter parse', function () {
      var template = '(foo)[bar]';
      var parsedWithParens = Mustache.parse(template, ['(', ')']);
      var parsedWithBrackets = Mustache.parse(template, ['[', ']']);
      assert.notDeepEqual(parsedWithBrackets, parsedWithParens);
    });
  });

  describe('when parsing a template after already having parsed that template with a different Mustache.tags', function () {
    it('returns different tokens for the latter parse', function () {
      var template = '{{foo}}[bar]';
      var parsedWithBraces = Mustache.parse(template);

      var oldTags = Mustache.tags;
      Mustache.tags = ['[', ']'];
      var parsedWithBrackets = Mustache.parse(template);
      Mustache.tags = oldTags;

      assert.notDeepEqual(parsedWithBrackets, parsedWithBraces);
    });
  });

  describe('when parsing a template with the same tags second time, return the cached tokens', function () {
    it('returns the same tokens for the latter parse', function () {
      var template = '{{foo}}[bar]';
      var parsedResult1 = Mustache.parse(template);
      var parsedResult2 = Mustache.parse(template);

      assert.deepEqual(parsedResult1, parsedResult2);
      assert.ok(parsedResult1 === parsedResult2);
    });
  });

  describe('when parsing a template with caching disabled and the same tags second time, do not return the cached tokens', function () {
    it('returns different tokens for the latter parse', function () {
      Mustache.templateCache = undefined;
      var template = '{{foo}}[bar]';
      var parsedResult1 = Mustache.parse(template);
      var parsedResult2 = Mustache.parse(template);

      assert.deepEqual(parsedResult1, parsedResult2);
      assert.ok(parsedResult1 !== parsedResult2);
    });
  });

  describe('when parsing a template with custom caching and the same tags second time, do not return the cached tokens', function () {
    it('returns the same tokens for the latter parse', function () {
      Mustache.templateCache = {
        _cache: [],
        set: function set (key, value) {
          this._cache.push([key, value]);
        },
        get: function get (key) {
          var cacheLength = this._cache.length;
          for (var i = 0; i < cacheLength; i++) {
            var entry = this._cache[i];
            if (entry[0] === key) {
              return entry[1];
            }
          }
          return undefined;
        },
        clear: function clear () {
          this._cache = [];
        }
      };

      var template = '{{foo}}[bar]';
      var parsedResult1 = Mustache.parse(template);
      var parsedResult2 = Mustache.parse(template);

      assert.deepEqual(parsedResult1, parsedResult2);
      assert.ok(parsedResult1 === parsedResult2);
    });
  });

});
