/**
 * @author Mat Groves http://matgroves.com/ @Doormat23
 */

/**
 * The base class for all objects that are rendered on the screen.
 * This is an abstract class and should not be used on its own rather it should be extended.
 *
 * @class DisplayObject
 * @constructor
 */
PIXI.DisplayObject = function()
{
    /**
     * The coordinate of the object relative to the local coordinates of the parent.
     *
     * @property position
     * @type Point
     */
    this.position = new PIXI.Point(0, 0);

    /**
     * The scale factor of the object.
     *
     * @property scale
     * @type Point
     */
    this.scale = new PIXI.Point(1, 1);

    /**
     * The pivot point of the displayObject that it rotates around
     *
     * @property pivot
     * @type Point
     */
    this.pivot = new PIXI.Point(0, 0);

    /**
     * The rotation of the object in radians.
     *
     * @property rotation
     * @type Number
     */
    this.rotation = 0;

    /**
     * The opacity of the object.
     *
     * @property alpha
     * @type Number
     */
    this.alpha = 1;

    /**
     * The visibility of the object.
     *
     * @property visible
     * @type Boolean
     */
    this.visible = true;

    /**
     * This is the defined area that will pick up mouse / touch events. It is null by default.
     * Setting it is a neat way of optimising the hitTest function that the interactionManager will use (as it will not need to hit test all the children)
     *
     * @property hitArea
     * @type Rectangle|Circle|Ellipse|Polygon
     */
    this.hitArea = null;

    /**
     * Can this object be rendered
     *
     * @property renderable
     * @type Boolean
     */
    this.renderable = false;

    /**
     * [read-only] The display object container that contains this display object.
     *
     * @property parent
     * @type DisplayObjectContainer
     * @readOnly
     */
    this.parent = null;

    /**
     * [read-only] The stage the display object is connected to, or undefined if it is not connected to the stage.
     *
     * @property stage
     * @type Stage
     * @readOnly
     */
    this.stage = null;

    /**
     * [read-only] The multiplied alpha of the displayObject
     *
     * @property worldAlpha
     * @type Number
     * @readOnly
     */
    this.worldAlpha = 1;

    /**
     * [read-only] Current transform of the object based on world (parent) factors
     *
     * @property worldTransform
     * @type Matrix
     * @readOnly
     * @private
     */
    this.worldTransform = new PIXI.Matrix();

    /**
     * The position of the Display Object based on the world transform.
     * This value is updated at the end of updateTransform and takes all parent transforms into account.
     *
     * @property worldPosition
     * @type Point
     * @readOnly
     */
    this.worldPosition = new PIXI.Point(0, 0);

    /**
     * The scale of the Display Object based on the world transform.
     * This value is updated at the end of updateTransform and takes all parent transforms into account.
     *
     * @property worldScale
     * @type Point
     * @readOnly
     */
    this.worldScale = new PIXI.Point(1, 1);

    /**
     * The rotation of the Display Object, in radians, based on the world transform.
     * This value is updated at the end of updateTransform and takes all parent transforms into account.
     *
     * @property worldRotation
     * @type Number
     * @readOnly
     */
    this.worldRotation = 0;

    /**
     * cached sin rotation and cos rotation
     *
     * @property _sr
     * @type Number
     * @private
     */
    this._sr = 0;

    /**
     * cached sin rotation and cos rotation
     *
     * @property _cr
     * @type Number
     * @private
     */
    this._cr = 1;

    /**
     * The area the filter is applied to like the hitArea this is used as more of an optimisation
     * rather than figuring out the dimensions of the displayObject each frame you can set this rectangle
     *
     * @property filterArea
     * @type Rectangle
     */
    this.filterArea = null;

    /**
     * The original, cached bounds of the object
     *
     * @property _bounds
     * @type Rectangle
     * @private
     */
    this._bounds = new PIXI.Rectangle(0, 0, 1, 1);

    /**
     * The most up-to-date bounds of the object
     *
     * @property _currentBounds
     * @type Rectangle
     * @private
     */
    this._currentBounds = null;

    /**
     * The original, cached mask of the object
     *
     * @property _mask
     * @type Rectangle
     * @private
     */
    this._mask = null;

    /**
     * Cached internal flag.
     *
     * @property _cacheAsBitmap
     * @type Boolean
     * @private
     */
    this._cacheAsBitmap = false;

    /**
     * Cached internal flag.
     *
     * @property _cacheIsDirty
     * @type Boolean
     * @private
     */
    this._cacheIsDirty = false;

};

// constructor
PIXI.DisplayObject.prototype.constructor = PIXI.DisplayObject;

/**
 * Destroy this DisplayObject.
 * Removes all references to transformCallbacks, its parent, the stage, filters, bounds, mask and cached Sprites.
 *
 * @method destroy
 */
PIXI.DisplayObject.prototype.destroy = function()
{
    if (this.children)
    {
        var i = this.children.length;

        while (i--)
        {
            this.children[i].destroy();
        }

        this.children = [];
    }

    this.hitArea = null;
    this.parent = null;
    this.stage = null;
    this.worldTransform = null;
    this.filterArea = null;
    this._bounds = null;
    this._currentBounds = null;
    this._mask = null;

    //  In case Pixi is still going to try and render it even though destroyed
    this.renderable = false;

    this._destroyCachedSprite();
};

/**
 * [read-only] Indicates if the sprite is globally visible.
 *
 * @property worldVisible
 * @type Boolean
 */
Object.defineProperty(PIXI.DisplayObject.prototype, 'worldVisible', {

    get: function() {

        var item = this;

        do
        {
            if (!item.visible) return false;
            item = item.parent;
        }
        while(item);

        return true;
    }

});

/**
 * Sets a mask for the displayObject. A mask is an object that limits the visibility of an object to the shape of the mask applied to it.
 * In PIXI a regular mask must be a PIXI.Graphics object. This allows for much faster masking in canvas as it utilises shape clipping.
 * To remove a mask, set this property to null.
 *
 * @property mask
 * @type Graphics
 */
Object.defineProperty(PIXI.DisplayObject.prototype, 'mask', {

    get: function() {
        return this._mask;
    },

    set: function(value) {

        if (this._mask) this._mask.isMask = false;

        this._mask = value;

        if (this._mask) this._mask.isMask = true;
    }

});

/**
 * Sets the filters for the displayObject.
 * IMPORTANT: This is a webGL only feature and will be ignored by the Canvas renderer.
 * 
 * To remove filters simply set this property to 'null'.
 * 
 * You cannot have a filter and a multiply blend mode active at the same time. Setting a filter will reset
 * this objects blend mode to NORMAL.
 * 
 * @property filters
 * @type Array(Filter)
 */
Object.defineProperty(PIXI.DisplayObject.prototype, 'filters', {

    get: function() {
        return this._filters;
    },

    set: function(value) {

        if (value)
        {
            // now put all the passes in one place..
            var passes = [];

            for (var i = 0; i < value.length; i++)
            {
                var filterPasses = value[i].passes;

                for (var j = 0; j < filterPasses.length; j++)
                {
                    passes.push(filterPasses[j]);
                }
            }

            // TODO change this as it is legacy
            this._filterBlock = { target: this, filterPasses: passes };
        }

        this._filters = value;

        if (this.blendMode && this.blendMode === PIXI.blendModes.MULTIPLY)
        {
            this.blendMode = PIXI.blendModes.NORMAL;
        }
    }
});

/**
 * Set if this display object is cached as a bitmap.
 * This basically takes a snap shot of the display object as it is at that moment. It can provide a performance benefit for complex static displayObjects.
 * To remove simply set this property to 'null'
 * @property cacheAsBitmap
 * @type Boolean
 */
Object.defineProperty(PIXI.DisplayObject.prototype, 'cacheAsBitmap', {

    get: function() {
        return  this._cacheAsBitmap;
    },

    set: function(value) {

        if (this._cacheAsBitmap === value)
        {
            return;
        }

        if (value)
        {
            this._generateCachedSprite();
        }
        else
        {
            this._destroyCachedSprite();
        }

        this._cacheAsBitmap = value;
    }

});

/*
 * Updates the object transform for rendering.
 *
 * If the object has no parent, and no parent parameter is provided, it will default to Phaser.Game.World as the parent.
 * If that is unavailable the transform fails to take place.
 *
 * The `parent` parameter has priority over the actual parent. Use it as a parent override.
 * Setting it does **not** change the actual parent of this DisplayObject, it just uses the parent for the transform update.
 *
 * @method updateTransform
 * @param {DisplayObject} [parent] - Optional parent to parent this DisplayObject transform from.
 */
PIXI.DisplayObject.prototype.updateTransform = function(parent)
{
    if (!parent && !this.parent && !this.game)
    {
        return;
    }

    var p = this.parent;

    if (parent)
    {
        p = parent;
    }
    else if (!this.parent)
    {
        p = this.game.world;
    }

    // create some matrix refs for easy access
    var pt = p.worldTransform;
    var wt = this.worldTransform;

    // temporary matrix variables
    var a, b, c, d, tx, ty;

    // so if rotation is between 0 then we can simplify the multiplication process..
    if (this.rotation % PIXI.PI_2)
    {
        // check to see if the rotation is the same as the previous render. This means we only need to use sin and cos when rotation actually changes
        if (this.rotation !== this.rotationCache)
        {
            this.rotationCache = this.rotation;
            this._sr = Math.sin(this.rotation);
            this._cr = Math.cos(this.rotation);
        }

        // get the matrix values of the displayobject based on its transform properties..
        a  =  this._cr * this.scale.x;
        b  =  this._sr * this.scale.x;
        c  = -this._sr * this.scale.y;
        d  =  this._cr * this.scale.y;
        tx =  this.position.x;
        ty =  this.position.y;
        
        // check for pivot.. not often used so geared towards that fact!
        if (this.pivot.x || this.pivot.y)
        {
            tx -= this.pivot.x * a + this.pivot.y * c;
            ty -= this.pivot.x * b + this.pivot.y * d;
        }

        // concat the parent matrix with the objects transform.
        wt.a  = a  * pt.a + b  * pt.c;
        wt.b  = a  * pt.b + b  * pt.d;
        wt.c  = c  * pt.a + d  * pt.c;
        wt.d  = c  * pt.b + d  * pt.d;
        wt.tx = tx * pt.a + ty * pt.c + pt.tx;
        wt.ty = tx * pt.b + ty * pt.d + pt.ty;
    }
    else
    {
        // lets do the fast version as we know there is no rotation..
        a  = this.scale.x;
        d  = this.scale.y;

        tx = this.position.x - this.pivot.x * a;
        ty = this.position.y - this.pivot.y * d;

        wt.a  = a  * pt.a;
        wt.b  = a  * pt.b;
        wt.c  = d  * pt.c;
        wt.d  = d  * pt.d;
        wt.tx = tx * pt.a + ty * pt.c + pt.tx;
        wt.ty = tx * pt.b + ty * pt.d + pt.ty;
    }

    // multiply the alphas..
    this.worldAlpha = this.alpha * p.worldAlpha;

    this.worldPosition.set(wt.tx, wt.ty);
    this.worldScale.set(Math.sqrt(wt.a * wt.a + wt.b * wt.b), Math.sqrt(wt.c * wt.c + wt.d * wt.d));
    this.worldRotation = Math.atan2(-wt.c, wt.d);

    // reset the bounds each time this is called!
    this._currentBounds = null;

    //  Custom callback?
    if (this.transformCallback)
    {
        this.transformCallback.call(this.transformCallbackContext, wt, pt);
    }

};

// performance increase to avoid using call.. (10x faster)
PIXI.DisplayObject.prototype.displayObjectUpdateTransform = PIXI.DisplayObject.prototype.updateTransform;

/**
 * Retrieves the bounds of the displayObject as a rectangle object
 *
 * @method getBounds
 * @param matrix {Matrix}
 * @return {Rectangle} the rectangular bounding area
 */
PIXI.DisplayObject.prototype.getBounds = function(matrix)
{
    matrix = matrix;//just to get passed js hinting (and preserve inheritance)
    return PIXI.EmptyRectangle;
};

/**
 * Retrieves the local bounds of the displayObject as a rectangle object
 *
 * @method getLocalBounds
 * @return {Rectangle} the rectangular bounding area
 */
PIXI.DisplayObject.prototype.getLocalBounds = function()
{
    return this.getBounds(PIXI.identityMatrix);///PIXI.EmptyRectangle();
};

/**
 * Sets the object's stage reference, the stage this object is connected to
 *
 * @method setStageReference
 * @param stage {Stage} the stage that the object will have as its current stage reference
 */
PIXI.DisplayObject.prototype.setStageReference = function(stage)
{
    this.stage = stage;
};

/**
 * Empty, to be overridden by classes that require it.
 *
 * @method preUpdate
 */
PIXI.DisplayObject.prototype.preUpdate = function()
{
};

/**
 * Useful function that returns a texture of the displayObject object that can then be used to create sprites
 * This can be quite useful if your displayObject is static / complicated and needs to be reused multiple times.
 *
 * @method generateTexture
 * @param resolution {Number} The resolution of the texture being generated
 * @param scaleMode {Number} See {{#crossLink "PIXI/scaleModes:property"}}PIXI.scaleModes{{/crossLink}} for possible values
 * @param renderer {CanvasRenderer|WebGLRenderer} The renderer used to generate the texture.
 * @return {Texture} a texture of the graphics object
 */
PIXI.DisplayObject.prototype.generateTexture = function(resolution, scaleMode, renderer)
{
    var bounds = this.getLocalBounds();

    var renderTexture = new PIXI.RenderTexture(bounds.width | 0, bounds.height | 0, renderer, scaleMode, resolution);
    
    PIXI.DisplayObject._tempMatrix.tx = -bounds.x;
    PIXI.DisplayObject._tempMatrix.ty = -bounds.y;
    
    renderTexture.render(this, PIXI.DisplayObject._tempMatrix);

    return renderTexture;
};

/**
 * Generates and updates the cached sprite for this object.
 *
 * @method updateCache
 */
PIXI.DisplayObject.prototype.updateCache = function()
{
    this._generateCachedSprite();
};

/**
 * Calculates the global position of the display object
 *
 * @method toGlobal
 * @param position {Point} The world origin to calculate from
 * @return {Point} A point object representing the position of this object
 */
PIXI.DisplayObject.prototype.toGlobal = function(position)
{
    // don't need to u[date the lot
    this.displayObjectUpdateTransform();
    return this.worldTransform.apply(position);
};

/**
 * Calculates the local position of the display object relative to another point
 *
 * @method toLocal
 * @param position {Point} The world origin to calculate from
 * @param [from] {DisplayObject} The DisplayObject to calculate the global position from
 * @return {Point} A point object representing the position of this object
 */
PIXI.DisplayObject.prototype.toLocal = function(position, from)
{
    if (from)
    {
        position = from.toGlobal(position);
    }

    // don't need to u[date the lot
    this.displayObjectUpdateTransform();

    return this.worldTransform.applyInverse(position);
};

/**
 * Internal method.
 *
 * @method _renderCachedSprite
 * @param renderSession {Object} The render session
 * @private
 */
PIXI.DisplayObject.prototype._renderCachedSprite = function(renderSession)
{
    this._cachedSprite.worldAlpha = this.worldAlpha;

    if (renderSession.gl)
    {
        PIXI.Sprite.prototype._renderWebGL.call(this._cachedSprite, renderSession);
    }
    else
    {
        PIXI.Sprite.prototype._renderCanvas.call(this._cachedSprite, renderSession);
    }
};

/**
 * Internal method.
 *
 * @method _generateCachedSprite
 * @private
 */
PIXI.DisplayObject.prototype._generateCachedSprite = function()
{
    this._cacheAsBitmap = false;

    var bounds = this.getLocalBounds();

    this.updateTransform();

    if (!this._cachedSprite)
    {
        var renderTexture = new PIXI.RenderTexture(bounds.width | 1, bounds.height | 1);
        this._cachedSprite = new PIXI.Sprite(renderTexture);
        this._cachedSprite.worldTransform = this.worldTransform;
    }
    else
    {
        this._cachedSprite.texture.resize(bounds.width | 1, bounds.height | 1);
    }

    //  Remove filters
    var tempFilters = this._filters;
    this._filters = null;
    this._cachedSprite.filters = tempFilters;

    // PIXI.DisplayObject._tempMatrix.identity();
    PIXI.DisplayObject._tempMatrix.tx = -bounds.x;
    PIXI.DisplayObject._tempMatrix.ty = -bounds.y;

    this._cachedSprite.texture.render(this, PIXI.DisplayObject._tempMatrix, true);
    this._cachedSprite.anchor.x = -( bounds.x / bounds.width );
    this._cachedSprite.anchor.y = -( bounds.y / bounds.height );

    this._filters = tempFilters;

    this._cacheAsBitmap = true;
};

/**
* Destroys the cached sprite.
*
* @method _destroyCachedSprite
* @private
*/
PIXI.DisplayObject.prototype._destroyCachedSprite = function()
{
    if (!this._cachedSprite) return;

    this._cachedSprite.texture.destroy(true);

    // TODO could be object pooled!
    this._cachedSprite = null;
};

/**
* Renders the object using the WebGL renderer
*
* @method _renderWebGL
* @param renderSession {RenderSession}
* @private
*/
PIXI.DisplayObject.prototype._renderWebGL = function(renderSession)
{
    // OVERWRITE;
    // this line is just here to pass jshinting :)
    renderSession = renderSession;
};

/**
* Renders the object using the Canvas renderer
*
* @method _renderCanvas
* @param renderSession {RenderSession}
* @private
*/
PIXI.DisplayObject.prototype._renderCanvas = function(renderSession)
{
    // OVERWRITE;
    // this line is just here to pass jshinting :)
    renderSession = renderSession;
};

/**
 * The position of the displayObject on the x axis relative to the local coordinates of the parent.
 *
 * @property x
 * @type Number
 */
Object.defineProperty(PIXI.DisplayObject.prototype, 'x', {

    get: function() {
        return  this.position.x;
    },

    set: function(value) {
        this.position.x = value;
    }

});

/**
 * The position of the displayObject on the y axis relative to the local coordinates of the parent.
 *
 * @property y
 * @type Number
 */
Object.defineProperty(PIXI.DisplayObject.prototype, 'y', {

    get: function() {
        return  this.position.y;
    },

    set: function(value) {
        this.position.y = value;
    }

});
