var core = require('../../core');
// @see https://github.com/substack/brfs/issues/25
var fs = require('fs');

/**
 * The NormalMapFilter class uses the pixel values from the specified texture (called the normal map)
 * to project lighting onto an object.
 *
 * @class
 * @extends PIXI.AbstractFilter
 * @memberof PIXI.filters
 * @param texture {Texture} The texture used for the normal map, must be power of 2 texture at the moment
 */
function NormalMapFilter(texture)
{
    core.AbstractFilter.call(this,
        // vertex shader
        null,
        // fragment shader
        fs.readFileSync(__dirname + '/normalMap.frag', 'utf8'),
        // custom uniforms
        {
            displacementMap:  { type: 'sampler2D', value: texture },
            scale:            { type: '2f', value: { x: 15, y: 15 } },
            offset:           { type: '2f', value: { x: 0,  y: 0 } },
            mapDimensions:    { type: '2f', value: { x: 1,  y: 1 } },
            dimensions:       { type: '4f', value: [0, 0, 0, 0] },
            // LightDir:         { type: 'f3', value: [0, 1, 0] },
            LightPos:         { type: '3f', value: [0, 1, 0] }
        }
    );

    texture.baseTexture._powerOf2 = true;

    if (texture.baseTexture.hasLoaded)
    {
        this.onTextureLoaded();
    }
    else
    {
        texture.baseTexture.once('loaded', this.onTextureLoaded, this);
    }
}

NormalMapFilter.prototype = Object.create(core.AbstractFilter.prototype);
NormalMapFilter.prototype.constructor = NormalMapFilter;
module.exports = NormalMapFilter;

/**
 * Sets the map dimensions uniforms when the texture becomes available.
 *
 * @private
 */
NormalMapFilter.prototype.onTextureLoaded = function ()
{
    this.uniforms.mapDimensions.value.x = this.uniforms.displacementMap.value.width;
    this.uniforms.mapDimensions.value.y = this.uniforms.displacementMap.value.height;
};

Object.defineProperties(NormalMapFilter.prototype, {
    /**
     * The texture used for the displacement map. Must be power of 2 texture.
     *
     * @member {Texture}
     * @memberof NormalMapFilter#
     */
    map: {
        get: function ()
        {
            return this.uniforms.displacementMap.value;
        },
        set: function (value)
        {
            this.uniforms.displacementMap.value = value;
        }
    },

    /**
     * The multiplier used to scale the displacement result from the map calculation.
     *
     * @member {Point}
     * @memberof NormalMapFilter#
     */
    scale: {
        get: function ()
        {
            return this.uniforms.scale.value;
        },
        set: function (value)
        {
            this.uniforms.scale.value = value;
        }
    },

    /**
     * The offset used to move the displacement map.
     *
     * @member {Point}
     * @memberof NormalMapFilter#
     */
    offset: {
        get: function ()
        {
            return this.uniforms.offset.value;
        },
        set: function (value)
        {
            this.uniforms.offset.value = value;
        }
    }
});
