var _curry2 = require('./internal/_curry2');
var _slice = require('./internal/_slice');
var assoc = require('./assoc');
var dissoc = require('./dissoc');


/**
 * Makes a shallow clone of an object, omitting the property at the
 * given path. Note that this copies and flattens prototype properties
 * onto the new object as well.  All non-primitive properties are copied
 * by reference.
 *
 * @func
 * @memberOf R
 * @category Object
 * @sig [String] -> {k: v} -> {k: v}
 * @param {Array} path the path to set
 * @param {Object} obj the object to clone
 * @return {Object} a new object without the property at path
 * @see R.assocPath
 * @example
 *
 *      R.dissocPath(['a', 'b', 'c'], {a: {b: {c: 42}}}); //=> {a: {b: {}}}
 */
module.exports = _curry2(function dissocPath(path, obj) {
  switch (path.length) {
    case 0:
      return obj;
    case 1:
      return dissoc(path[0], obj);
    default:
      var head = path[0];
      var tail = _slice(path, 1);
      return obj[head] == null ? obj : assoc(head, dissocPath(tail, obj[head]), obj);
  }
});
