var _curry3 = require('./internal/_curry3');


/**
 * The mapAccumRight function behaves like a combination of map and reduce; it applies a
 * function to each element of a list, passing an accumulating parameter from right
 * to left, and returning a final value of this accumulator together with the new list.
 *
 * Similar to `mapAccum`, except moves through the input list from the right to the
 * left.
 *
 * The iterator function receives two arguments, *acc* and *value*, and should return
 * a tuple *[acc, value]*.
 *
 * @func
 * @memberOf R
 * @category List
 * @sig (acc -> x -> (acc, y)) -> acc -> [x] -> (acc, [y])
 * @param {Function} fn The function to be called on every element of the input `list`.
 * @param {*} acc The accumulator value.
 * @param {Array} list The list to iterate over.
 * @return {*} The final, accumulated value.
 * @example
 *
 *      var digits = ['1', '2', '3', '4'];
 *      var append = function(a, b) {
 *        return [a + b, a + b];
 *      }
 *
 *      R.mapAccumRight(append, 0, digits); //=> ['04321', ['04321', '0432', '043', '04']]
 */
module.exports = _curry3(function mapAccumRight(fn, acc, list) {
  var idx = list.length - 1, result = [], tuple = [acc];
  while (idx >= 0) {
    tuple = fn(tuple[0], list[idx]);
    result[idx] = tuple[1];
    idx -= 1;
  }
  return [tuple[0], result];
});
