## Shepherd

Guide your users through a tour of your app.

### Dependencies

[Popper](https://popper.js.org/)

### Install

__npm__
```bash
npm install shepherd.js --save
```

__yarn__
```bash
yarn add shepherd.js
```

### Install with Eager

**Note: Eager is now Cloudflare Apps**

We will eventually make this work again, but it probably currently does not.

<!--The easiest way to add a Shepherd tour to your site is with [Eager](http://eager.io).
Click Install to create a tour right on your site with no coding required.-->

<!--iframe style="height: 48px; width: 180px" src="//install.eager.io?appId=AalP5veMma6s" allowtransparency="true" scroll="no" frameBorder="0"></iframe-->

### Usage

First create a new `Tour` instance for your tour:

```javascript
const tour = new Shepherd.Tour({
  defaults: {
    classes: 'shepherd-theme-arrows',
    scrollTo: true
  }
});
```

The `defaults` option allows you to specify any options which should be applied
to all this tour's steps by default.

Next, add your steps:

```javascript
tour.addStep('example-step', {
  text: 'This step is attached to the bottom of the <code>.example-css-selector</code> element.',
  attachTo: '.example-css-selector bottom',
  classes: 'example-step-extra-class',
  buttons: [
    {
      text: 'Next',
      action: tour.next
    }
  ]
});
```

Finally, to start the tour, just call `start` on your `Tour` instance:

```javascript
tour.start();
```

### API

#### Global Shepherd Object

Shepherd exposes a single object onto the window, `Shepherd`.

That global object fires several events to let you link up actions with events
occurring in _any_ tour:

##### Methods

- `Shepherd.on(eventName, handler, [context])`: Bind an event
- `Shepherd.off(eventName, [handler])`: Unbind an event
- `Shepherd.once(eventName, handler, [context])`: Bind just the next instance of an event

##### Events

The global Shepherd fires the following events whenever a `Tour` instance fires them.  It adds to the object passed to the
event handlers a `tour` key pointing to the instance which fired the event:

- `complete`
- `cancel`
- `hide`
- `show`
- `start`
- `active`
- `inactive`

##### Current Tour

The global `Shepherd` includes a property which is always set to the currently active tour, or null if there is no active tour:

- `Shepherd.activeTour`

#### Tour Instances

##### Creation

You create a `Tour` object for each tour you'd like to create.

Tour's constructor accepts a hash of options:

```javascript
const myTour = new Shepherd.Tour(options);
```

##### Tour Options

- `steps`: An array of Step instances to initialize the tour with
- `defaults`: Default options for Steps created through `addStep`
- `confirmCancel`: If true, will issue a window.confirm before cancelling
- `confirmCancelMessage`: The message to display in the confirm dialog

##### Tour Methods

- `addStep(id, options)`: Creates a new Step object with options, and returns the `Step` instance it created.  If you'd like you can also just pass an options hash which includes `id` as a key.
If the options hash doesn't include an `id`, one will be generated.
You can also pass an existing `Step` instance rather than `options`, but note that Shepherd does not support a Step being attached to multiple Tours.
- `getById(id)`: Return a step with a specific id
- `next()`: Advance to the next step, in the order they were added
- `back()`: Show the previous step, in the order they were added
- `cancel()`: Trigger cancel on the current step, hiding it without advancing
- `hide()`: Hide the current step
- `show([id])`: Show the step specified by id (if it's a string), or index (if it's a number) provided.  Defaults to the first step.
- `start()`: Show the first step and begin the tour
- `getCurrentStep()`: Returns the currently shown step
- `on(eventName, handler, [context])`: Bind an event
- `off(eventName, [handler])`: Unbind an event
- `once(eventName, handler, [context])`: Bind just the next instance of an event

##### Tour Events

- `complete`: Triggered when the last step is advanced
- `cancel`
- `hide`
- `show`: Triggered with a hash of the `step` and the `previous` step
- `start`

Steps are instances of the Step object.  They are generally created by the `Tour::addStep` method, which returns the `Step` instance it
created.

#### Steps

##### Step Options

- `text`: The text in the body of the step.  It can be one of four types:
  - HTML string
  - Array of HTML strings
  - `HTMLElement` object
  - Function to be executed when the step is built. It must return one the three options above.
- `title`: The steps title.  It becomes an `h3` at the top of the step.
- `attachTo`: What element the step should be attached to on the page.  It can either be a string of the form `"element on"`, or an object with those properties.  For example: `".some #element left"`, or `{element: '.some #element',
on: 'left'}`.  If you use the object syntax, `element` can also be a DOM element.  If you don't specify an `attachTo`
the element will appear in the middle of the screen.
- `beforeShowPromise`: A function that returns a promise. When the promise resolves, the rest of the `show` code for
the step will execute.
- `classes`: Extra classes to add to the step.  `shepherd-theme-arrows` will give you our theme.
- `buttons`: An array of buttons to add to the step.  By default we add a Next button which triggers `next()`, set this to false
to disable.  Each button in the array is an object of the format:
  - `text`: The HTML text of the button
  - `classes`: Extra classes to apply to the `<a>`
  - `action`: A function executed when the button is clicked on
  - `events`: A hash of events to bind onto the button, for example `{'mouseover': function(){}}`.  Adding a click event to `events` when you
  already have an `action` specified is not supported.
  You can use `events` to skip steps or navigate to specific steps, with something like:
  ```javascript
  events: {  
    click: function() {  
      return Shepherd.activeTour.show('some_step_name');  
    }  
  }
  ```
- `advanceOn`: An action on the page which should advance shepherd to the next step.  It can be of the form `"selector event"`, or an object with those
properties.  For example: `".some-element click"`, or `{selector: '.some-element', event: 'click'}`.  It doesn't have to be an event inside
the tour, it can be any event fired on any element on the page.  You can also always manually advance the Tour by calling `myTour.next()`.
- `renderLocation`: An `HTMLElement` or selector string of the element you want the tour step to render in. Most of the time, you will
not need to pass anything, and it will default to `document.body`, but this is needed for `<dialog>` and might as well support passing anything.
- `showCancelLink`: Should a cancel "✕" be shown in the header of the step?
- `showOn`: A function that, when it returns true, will show the step. If it returns false, the step will be skipped.
- `scrollTo`: Should the element be scrolled to when this step is shown?
- `scrollToHandler`: A function that lets you override the default `scrollTo` behavior and define a custom action to do the scrolling,
and possibly other logic.
- `when`: You can define show, hide, etc events inside when. For example:
```javascript
when: {
  show: function() {
    window.scrollTo(0, 0);
  }
}
```
- `popperOptions`: Extra options to pass to [popper.js](https://github.com/FezVrasta/popper.js)

##### Step Methods

- `show()`: Show this step
- `hide()`: Hide this step
- `cancel()`: Hide this step and trigger the `cancel` event
- `complete()`: Hide this step and trigger the `complete` event
- `scrollTo()`: Scroll to this step's element
- `isOpen()`: Returns true if the step is currently shown
- `destroy()`: Remove the element
- `on(eventName, handler, [context])`: Bind an event
- `off(eventName, [handler])`: Unbind an event
- `once(eventName, handler, [context])`: Bind just the next instance of an event

##### Step Events

- `before-show`
- `show`
- `before-hide`
- `hide`
- `complete`
- `cancel`
- `destroy`

Please note that `complete` and `cancel` are only ever triggered if you call the associated methods in your code.

### Advancing on Actions

You can use the `advanceOn` option, or the Next button, to advance steps.  If you would like however to have a step advance on a
complex user action, you can do the following:

```javascript
const myStep = myTour.addStep('my-step', options);

yourApp.on('some-event', () => {
  if (myStep.isOpen()){
    Shepherd.activeTour.next();
  }
});
```

It's strongly recommended that you use some sort of event mediator to connect your app's actions with Shepherd, to prevent
having to sprinkle Shepherd code throughout your codebase, and to keep things loosely coupled.  You can create a basic
mediator if need be using the `Evented` object which is provided with Shepherd:

```javascript
const mediator = new Shepherd.Evented();
```

You can then trigger events in one part of your app:

```javascript
mediator.trigger('user-create');
```

And listen for them in other areas:

```javascript
mediator.on('user-create', () => {});
```

### Browser Support

IE9+ and all modern browsers

## Using Themes
We deliver some predefined themes (e.g. shepherd-theme-default or shepherd-theme-square). You are welcome to use them and embed them into your page as additional stylesheets.

```html
<head>
  <link rel="stylesheet" href="shepherd-theme-default.css">
</head>
```

Individual customizations to the standard themes must be made within the CSS file. Furthermore, individual CSS classes can be passed via the tour instance or for each step individually, to overwritte or supplement  existing style properties.

```javascript
let tour = new Shepherd.Tour({
  defaults: {
    classes: 'shepherd-theme-custom'
  }
});
```

### Custom Themeing
We use [SASS](https://sass-lang.com/) as pre-processor for CSS. In connection with SASS there are extensive possibilities to generate CSS. For example, SASS can calculate or increase the saturation of color values. In addition, variables can be defined (similar to a scripting language), which ultimately end up as values in the CSS result. We make use of these extended possibilities by extracting themeing-relevant values as variables (__variables.scss__). This makes it easy to individualise colors and shapes.

Basically, there are only a handful of variables that need to be adjusted. A distinction is made between color settings and other options.

**Color Settings**

| Variable | Purpose | Default
|---|---|---
| $shepherd-theme-primary | Primary or brand color. Primary button gets this color. | #3288e6
| $shepherd-theme-secondary | Secondary color. If it is not set explicitly, it is calculated using the primary color. | desaturate(lighten($shepherd-theme-primary, 40), 70)
| $shepherd-text-background | Background color of the text area. | #ffffff
| $shepherd-header-background | Background color of the header element. If it is not set explicitly, it is calculated using the text background color. | darken($shepherd-text-background, 10)

**Options**

| Variable | Purpose | Default
|---|---|---
| $shepherd-element-width | Width of the step element | 400px
| $shepherd-element-border-radius | Set radius of rounded corners. 0 means (sharp) pointed corners. | 5px
| $shepherd-element-max-height | Maximum height of the element | 100%
| $shepherd-element-max-width | Maximum width of the element | 100%
| $shepherd-element-z-index | Move the element forward or backward | 9999
| $shepherd-text-line-height | Determine the line height of the body text | 1.3em
| $shepherd-button-border-radius | Decide whether the buttons should have rounded or pointed corners. 0 means (sharp) pointed corners. | 3px
| $use-drop-shadow | The element casts a shadow | true

The example below is intended to illustrate the individual customizations. Please make sure that the values are set before the import of the _base-theme_.

```scss
$shepherd-theme-primary: #9b59b6 !default;
$shepherd-theme-secondary: desaturate(lighten($shepherd-theme-primary, 30), 70) !default;
$shepherd-header-background: #eeeeee !default;
$shepherd-element-border-radius: 0 !default;
$shepherd-button-border-radius: 0 !default;
$use-drop-shadow: true !default;

@import 'base';
```

As a result you get a squared theme with a purple primary button. The individual steps cast a shadow on the underlying elements. The header is colored in a light gray tone.

## Projects Using Shepherd

Here we showcase some of the awesome libraries built using Shepherd.

### [ember-shepherd](https://github.com/shipshapecode/ember-shepherd)

Ember addon for the site tour library Shepherd

### [SimplePlanner](https://simpleplanner.io)

[SimplePlanner](https://simpleplanner.io) uses Shepherd to help new users get familiar with its collaborative scheduling approach. You do need to sign up via OAuth or email to see the scheduling tour. Check out the [Envato Tuts+ Startup Series on its codebase](https://code.tutsplus.com/series/building-your-startup-with-php--cms-742) which describes how Simple Planner was built.

### Your Project Here

If you have a cool open-source library built on Shepherd, PR this doc.
