var distance = require('@turf/distance');
var segmentReduce = require('@turf/meta').segmentReduce;

/**
 * Takes a {@link GeoJSON} and measures its length in the specified units, {@link (Multi)Point|Point}'s distance are ignored.
 *
 * @name lineDistance
 * @param {FeatureCollection|Feature|Geometry} geojson GeoJSON to measure
 * @param {string} [units=kilometers] can be degrees, radians, miles, or kilometers
 * @returns {number} length of GeoJSON
 * @example
 * var line = turf.lineString([[115, -32], [131, -22], [143, -25], [150, -34]]);
 * var length = turf.lineDistance(line, 'miles');
 *
 * //addToMap
 * var addToMap = [line];
 * line.properties.distance = length;
 */
module.exports = function lineDistance(geojson, units) {
    // Input Validation
    if (!geojson) throw new Error('geojson is required');

    // Calculate distance from 2-vertex line segements
    return segmentReduce(geojson, function (previousValue, segment) {
        var coords = segment.geometry.coordinates;
        return previousValue + distance(coords[0], coords[1], units);
    }, 0);
};
