'use strict';

var bullseye = require('bullseye');
var ranchored = /^[#>*-]$/;
var rboundary = /^[*_`#-]$/;
var rbulletafter = /^\d+\. /;
var rbulletline = /^\s*\d+\.$/;
var rbulletleft = /^\s*\d+$/;
var rbulletright = /\d|\./;
var rwhitespace = /^\s*$/;
var rhr = /^---+$/;
var rend = /^$|\s|\n/;
var rfootnotedeclaration = /^\[[^\]]+\]\s*:\s*[A-z\/]/;
var rfootnotebegin = /^\s*\[[^\]]*$/;
var rfootnotebegan = /^\s*\[[^\]]+$/;
var rfootnoteleft = /^\s*\[[^\]]+\]\s*$/;
var rfootnoteanchor = /^\s*\[[^\]]+\]\s*:$/;
var rfootnotelink = /^\s*\[[^\]]+\]\s*:\s*[A-z\/]/;
var rfootnotefull = /^\s*\[[^\]]+\]\s*:\s*[A-z\/].*\s*"[^"]*"/;
var rspaceorquote = /\s|"/;
var rspaceorcolon = /\s|:/;
var rempty = /^(<p><\/p>)?\n?$/i;

function rememberSelection (history) {
  var code = Math.random().toString(18).substr(2).replace(/\d+/g, '');
  var open = 'WoofmarkSelectionOpenMarker' + code;
  var close = 'WoofmarkSelectionCloseMarker' + code;
  var rmarkers = new RegExp(open + '|' + close, 'g');
  mark();
  return unmark;

  function mark () {
    var state = history.reset().inputState;
    var chunks = state.getChunks();
    var mode = state.mode;
    var all = chunks.before + chunks.selection + chunks.after;
    if (rempty.test(all)) {
      return;
    }
    if (mode === 'markdown') {
      updateMarkdownChunks(chunks);
    } else {
      updateHTMLChunks(chunks);
    }
    state.setChunks(chunks);
    state.restore(false);
  }

  function unmark () {
    var state = history.inputState;
    var chunks = state.getChunks();
    var all = chunks.before + chunks.selection + chunks.after;
    var start = all.lastIndexOf(open);
    var end = all.lastIndexOf(close) + close.length;
    var selectionStart = start === -1 ? 0 : start;
    var selectionEnd = end === -1 ? 0 : end;
    chunks.before = all.substr(0, selectionStart).replace(rmarkers, '');
    chunks.selection = all.substr(selectionStart, selectionEnd - selectionStart).replace(rmarkers, '');
    chunks.after = all.substr(end).replace(rmarkers, '');
    var el = history.surface.current(history.inputMode);
    var eye = bullseye(el, {
      caret: true, autoupdateToCaret: false, tracking: false
    });
    state.setChunks(chunks);
    state.restore(false);
    state.scrollTop = el.scrollTop = eye.read().y - el.getBoundingClientRect().top - 50;
    eye.destroy();
  }

  function updateMarkdownChunks (chunks) {
    var all = chunks.before + chunks.selection + chunks.after;
    var originalStart = chunks.before.length;
    var originalEnd = originalStart + chunks.selection.length;
    var selectionStart = move(originalStart, 1);
    var selectionEnd = move(originalEnd, -1);
    var moved = originalStart !== selectionStart || originalEnd !== selectionEnd;

    updateSelection(chunks, all, selectionStart, selectionEnd, moved);

    function move (p, offset) {
      var prev = all[p - 1] || '';
      var next = all[p] || '';
      var line = backtrace(p - 1, '\n');
      var jumps = prev === '' || prev === '\n';

      if (next === ' ' && (jumps || prev === ' ')) {
        return again();
      }

      var close = backtrace(p - 1, ']');
      var reopened = close.indexOf('[');

      // these two handle anchored references '[foo][1]', or even '[bar]  \n [2]'
      if (reopened !== -1 && rwhitespace.test(close.substr(0, reopened))) {
        return again(-close.length);
      } else {
        reopened = all.substr(p).indexOf('[');
        if (reopened !== -1 && rwhitespace.test(all.substr(p, reopened))) {
          return again(-1);
        }
      }

      // the seven following rules together handle footnote references
      if ((jumps || rwhitespace.test(line)) && rfootnotedeclaration.test(all.substr(p))) {
        return again(); // started with '', '\n', or '  ' and continued with '[a-1]: h'
      }
      if (rfootnotebegin.test(line) && next !== ']') {
        return again(); // started with '[' and continued with 'a-1'
      }
      if (rfootnotebegan.test(line) && next === ']') {
        return again(); // started with '[a-1' and continued with ']: h'
      }
      if (rfootnoteleft.test(line) && rspaceorcolon.test(next)) {
        return again(); // started with '[a-1]  ' and continued with ':'
      }
      if (rfootnoteanchor.test(line) && next === ' ') {
        return again(); // started with '[a-1]  :' and continued with ' '
      }
      if (rfootnotelink.test(line) && prev === ' ' && rspaceorquote.test(next) && offset === 1) {
        return again(); // started with '[a-1]  :' and continued with ' ', or '"', on the left
      }
      if (rfootnotefull.test(line) && rend.test(next)) {
        return again(-1); // started with '[a-1]  : something "something"' and continued with '', ' ', or '\n'
      }

      // the three following rules together handle ordered list items: '\n1. foo\n2. bar'
      if ((jumps || rwhitespace.test(line)) && rbulletafter.test(all.substr(p))) {
        return again(); // started with '', '\n', or '  ' and continued with '123. '
      }
      if (rbulletleft.test(line) && rbulletright.test(next)) {
        return again(); // started with '  123' and ended in '4' or '.'
      }
      if (rbulletline.test(line) && next === ' ') {
        return again(); // started with '  123.' and ended with ' '
      }

      if (ranchored.test(next) && jumps) {
        return again();
      }
      if (ranchored.test(prev) && next === ' ') {
        return again();
      }
      if (next === prev && rboundary.test(next)) {
        return again();
      }
      if (rhr.test(line) && next === '\n') {
        return again();
      }
      if (all.substr(p - 3, 3) === '```' && offset === 1) { // handles '```javascript\ncode\n```'
        while (all[p - 1] && all[p - 1] !== '\n') {
          p++;
        }
      }
      return p;

      function again (override) {
        var diff = override || offset;
        return move(p + diff, diff > 0 ? 1 : -1);
      }
      function backtrace (p, edge) {
        var last = all[p];
        var text = '';
        while (last && last !== edge) {
          text = last + text;
          last = all[--p];
        }
        return text;
      }
    }
  }

  function updateHTMLChunks (chunks) {
    var all = chunks.before + chunks.selection + chunks.after;
    var selectionStart = chunks.before.length;
    var selectionEnd = selectionStart + chunks.selection.length;
    var leftClose = chunks.before.lastIndexOf('>');
    var leftOpen = chunks.before.lastIndexOf('<');
    var rightClose = chunks.after.indexOf('>');
    var rightOpen = chunks.after.indexOf('<');
    var prevOpen;
    var nextClose;
    var balanceTags;

    // <fo[o]>bar</foo> into <foo>[]bar</foo>, <fo[o>ba]r</foo> into <foo>[ba]r</foo>
    if (leftOpen > leftClose) {
      nextClose = all.indexOf('>', leftClose + 1);
      if (nextClose !== -1) {
        selectionStart = nextClose + 1;
        balanceTags = true;
      }
    }

    // <foo>bar</[fo]o> into <foo>bar[]</foo>, <foo>b[ar</f]oo> into <foo>b[ar]</foo>
    if (rightOpen === -1 || rightOpen > rightClose) {
      prevOpen = all.substr(0, chunks.before.length + chunks.selection.length + rightClose).lastIndexOf('<');
      if (prevOpen !== -1) {
        selectionEnd = prevOpen;
        selectionStart = Math.min(selectionStart, selectionEnd);
        balanceTags = true;
      }
    }

    updateSelection(chunks, all, selectionStart, selectionEnd, balanceTags);
  }

  function updateSelection (chunks, all, selectionStart, selectionEnd, balanceTags) {
    if (selectionEnd < selectionStart) {
      selectionEnd = selectionStart;
    }
    if (balanceTags) {
      chunks.before = all.substr(0, selectionStart);
      chunks.selection = all.substr(selectionStart, selectionEnd - selectionStart);
      chunks.after = all.substr(selectionEnd);
    }
    chunks.selection = open + chunks.selection + close;
  }
}

module.exports = rememberSelection;
