/**
 * @fileOverview Utility functions for webgl rendering.
 *
 * @author Andrei Kashcha (aka anvaka) / https://github.com/anvaka
 */

module.exports = webgl;

function webgl(gl) {

  return {
    createProgram: createProgram,
    extendArray: extendArray,
    copyArrayPart: copyArrayPart,
    swapArrayPart: swapArrayPart,
    getLocations: getLocations,
    context: gl
  };

  function createShader(shaderText, type) {
    var shader = gl.createShader(type);
    gl.shaderSource(shader, shaderText);
    gl.compileShader(shader);

    if (!gl.getShaderParameter(shader, gl.COMPILE_STATUS)) {
      var msg = gl.getShaderInfoLog(shader);
      window.alert(msg);
      throw msg;
    }

    return shader;
  }

  function createProgram(vertexShaderSrc, fragmentShaderSrc) {
    var program = gl.createProgram();
    var vs = createShader(vertexShaderSrc, gl.VERTEX_SHADER);
    var fs = createShader(fragmentShaderSrc, gl.FRAGMENT_SHADER);

    gl.attachShader(program, vs);
    gl.attachShader(program, fs);
    gl.linkProgram(program);

    if (!gl.getProgramParameter(program, gl.LINK_STATUS)) {
      var msg = gl.getShaderInfoLog(program);
      window.alert(msg);
      throw msg;
    }

    return program;
  }

  function extendArray(buffer, itemsInBuffer, elementsPerItem) {
    if ((itemsInBuffer + 1) * elementsPerItem > buffer.length) {
      // Every time we run out of space create new array twice bigger.
      // TODO: it seems buffer size is limited. Consider using multiple arrays for huge graphs
      var extendedArray = new Float32Array(buffer.length * elementsPerItem * 2);
      extendedArray.set(buffer);

      return extendedArray;
    }

    return buffer;
  }

  function getLocations(program, uniformOrAttributeNames) {
    var foundLocations = {};
    for (var i = 0; i < uniformOrAttributeNames.length; ++i) {
      var name = uniformOrAttributeNames[i];
      var location = -1;
      if (name[0] === 'a' && name[1] === '_') {
        location = gl.getAttribLocation(program, name);
        if (location === -1) {
          throw new Error("Program doesn't have required attribute: " + name);
        }

        foundLocations[name.slice(2)] = location;
      } else if (name[0] === 'u' && name[1] === '_') {
        location = gl.getUniformLocation(program, name);
        if (location === null) {
          throw new Error("Program doesn't have required uniform: " + name);
        }

        foundLocations[name.slice(2)] = location;
      } else {
        throw new Error("Couldn't figure out your intent. All uniforms should start with 'u_' prefix, and attributes with 'a_'");
      }
    }

    return foundLocations;
  }
}

function copyArrayPart(array, to, from, elementsCount) {
  for (var i = 0; i < elementsCount; ++i) {
    array[to + i] = array[from + i];
  }
}

function swapArrayPart(array, from, to, elementsCount) {
  for (var i = 0; i < elementsCount; ++i) {
    var tmp = array[from + i];
    array[from + i] = array[to + i];
    array[to + i] = tmp;
  }
}
