var chai = require('chai');
var assert = chai.assert;
var coder = require('../packages/web3-eth-abi');

// TODO check line 108 again!

describe('lib/solidity/coder', function () {
    describe('decodeParam', function () {
        var test = function (t) {
            it('should turn ' + t.type +': ' + t.value + ' to ' + t.expected, function () {
                if (t.type === 'tuple') {
                    console.log(t);
                }
                assert.deepEqual(coder.decodeParameter(t.type, t.value), t.expected);
            });
        };

        test({ type: 'address', expected: '0x407D73d8a49eeb85D32Cf465507dd71d507100c1',
                                                            value: '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1'});
        test({ type: 'address[2]', expected: ['0x407D73d8a49eeb85D32Cf465507dd71d507100c1', '0x407D73d8A49eEB85D32Cf465507Dd71d507100c3'],
                                                            value: '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1' +
                                                                   '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3' });
        test({ type: 'address[]', expected: ['0x407D73d8a49eeb85D32Cf465507dd71d507100c1', '0x407D73d8A49eEB85D32Cf465507Dd71d507100c3'],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                   '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1' +
                                                                   '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3' });
        test({ type: 'address[][2]', expected: [['0x407D73d8a49eeb85D32Cf465507dd71d507100c1', '0x407d73d8a49EEB85d32Cf465507dD71D507100c2'],
                                               ['0x407D73d8A49eEB85D32Cf465507Dd71d507100c3', '0x407D73d8a49eeb85D32CF465507dd71d507100C4']],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                   '00000000000000000000000000000000000000000000000000000000000000a0' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                   '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1' +
                                                                   '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c2' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                   '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3' +
                                                                   '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c4'});
        test({ type: 'address[2][]', expected: [['0x407D73d8a49eeb85D32Cf465507dd71d507100c1', '0x407d73d8a49EEB85d32Cf465507dD71D507100c2'],
                                               ['0x407D73d8A49eEB85D32Cf465507Dd71d507100c3', '0x407D73d8a49eeb85D32CF465507dd71d507100C4']],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002' + /* 20 */
                                                                   '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1' +
                                                                   '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c2' +
                                                                   '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3' +
                                                                   '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c4' });
        // test({ type: 'address[][]', expected: [['0x407D73d8a49eeb85D32Cf465507dd71d507100c1', '0x407d73d8a49EEB85d32Cf465507dD71D507100c2'],
        //                                        ['0x407D73d8A49eEB85D32Cf465507Dd71d507100c3', '0x407D73d8a49eeb85D32CF465507dd71d507100C4']],
        //                                                     value: '0000000000000000000000000000000000000000000000000000000000000020' +
        //                                                            '0000000000000000000000000000000000000000000000000000000000000002' + /* 20 */
        //                                                            '0000000000000000000000000000000000000000000000000000000000000080' +
        //                                                            '00000000000000000000000000000000000000000000000000000000000000e0' +
        //                                                            '0000000000000000000000000000000000000000000000000000000000000002' + /* 80 */
        //                                                            '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1' + /* a0 */
        //                                                            '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c2' +
        //                                                            '0000000000000000000000000000000000000000000000000000000000000002' + /* e0 */
        //                                                            '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3' +
        //                                                            '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c4' });
        test({ type: 'bool', expected: true,                value: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ type: 'bool', expected: false,               value: '0000000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'bool[2]', expected: [true, false],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'bool[]', expected: [true, true, false],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000000'});

        test({ type: 'int', expected: '1',            value: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ type: 'int', expected: '1',            value: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ type: 'int', expected: '16',           value: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'int', expected: '-1',           value: 'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'});
        test({ type: 'int256', expected: '1',         value: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ type: 'int256', expected: '16',        value: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'int256', expected: '-1',        value: 'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'});
        test({ type: 'int8', expected: '16',          value: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'int8[2]', expected: ['16', '2'],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000010' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002'});
        test({ type: 'int32', expected: '16',         value: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'int64', expected: '16',         value: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'int128', expected: '16',        value: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'int[]', expected: [],                 value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'int[]', expected: ['3'],        value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ type: 'int256[]', expected: ['3'],     value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ type: 'int[]', expected: ['1', '2', '3'],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ type: 'int[3][]', expected: [['1', '2', '3'], ['4', '5', '6']],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000004' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000006'});

        test({ type: 'uint', expected: '1',           value: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ type: 'uint', expected: '1',           value: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ type: 'uint', expected: '16',          value: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'uint', expected: '115792089237316195423570985008687907853269984665640564039457584007913129639935', // old 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff
                                                            value: 'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'});
        test({ type: 'uint256', expected: '1',        value: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ type: 'uint256', expected: '16',       value: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'uint8', expected: '16',         value: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'uint32', expected: '16',        value: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'uint64', expected: '16',        value: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'uint128', expected: '16',       value: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'uint[]', expected: [],                value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'uint[]', expected: ['3'],       value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ type: 'uint256[]', expected: ['3'],    value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ type: 'uint[]', expected: ['1', '2', '3'],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ type: 'uint[3][]', expected: [['1', '2', '3'], ['4', '5', '6']],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000004' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000006'});
        test({ type: 'bytes', expected: '0x6761766f66796f726b',
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                   '6761766f66796f726b0000000000000000000000000000000000000000000000'});
        test({ type: 'bytes', expected: '0x731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b',
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b'});
        test({ type: 'bytes', expected: '0x131a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                          '231a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                          '331a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b',
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000060' +
                                                                   '131a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                   '231a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                   '331a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b'});
        test({ type: 'bytes', expected: '0x731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                          '731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b',
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                   '731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                   '731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b'});
        test({ type: 'bytes[2]', expected: ['0x00000c8c18f9252830fb3c56471c51335a8262f16a6d70e276417a7c7d897f617fff',
                                            '0x21f9252830fb3c56471c51335a8262f16a6d70e276417a7c7d897f617fff'],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                   '00000000000000000000000000000000000000000000000000000000000000a0' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000022' +
                                                                   '00000c8c18f9252830fb3c56471c51335a8262f16a6d70e276417a7c7d897f61' +
                                                                   '7fff000000000000000000000000000000000000000000000000000000000000' +
                                                                   '000000000000000000000000000000000000000000000000000000000000001e' +
                                                                   '21f9252830fb3c56471c51335a8262f16a6d70e276417a7c7d897f617fff0000'});


        test({ type: 'bytes[][2]', expected: [['0x00000c8c18f9252830fb3c56471c51335a8262f16a6d70e276417a7c7d897f617fff',
                                               '0x21f9252830fb3c56471c51335a8262f16a6d70e276417a7c7d897f617fff'],
                                              ['0x00000c8c18f9252830fb3c56471c51335a8262f16a6d70e276417a7c7d897f617fff',
                                               '0x21f9252830fb3c56471c51335a8262f16a6d70e276417a7c7d897f617fff']],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020'+
                                                                    '0000000000000000000000000000000000000000000000000000000000000040'+
                                                                    '0000000000000000000000000000000000000000000000000000000000000140' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                    '00000000000000000000000000000000000000000000000000000000000000a0' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000022' +
                                                                    '00000c8c18f9252830fb3c56471c51335a8262f16a6d70e276417a7c7d897f61' +
                                                                    '7fff000000000000000000000000000000000000000000000000000000000000' +
                                                                    '000000000000000000000000000000000000000000000000000000000000001e' +
                                                                    '21f9252830fb3c56471c51335a8262f16a6d70e276417a7c7d897f617fff0000' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                    '00000000000000000000000000000000000000000000000000000000000000a0' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000022' +
                                                                    '00000c8c18f9252830fb3c56471c51335a8262f16a6d70e276417a7c7d897f61' +
                                                                    '7fff000000000000000000000000000000000000000000000000000000000000' +
                                                                    '000000000000000000000000000000000000000000000000000000000000001e' +
                                                                    '21f9252830fb3c56471c51335a8262f16a6d70e276417a7c7d897f617fff0000'});

        test({ type: 'bytes1', expected: '0xcf',
                                                            value: 'cf00000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'bytes1[4]', expected: ['0xcf', '0x68', '0x4d', '0xfb'],
                                                            value: 'cf00000000000000000000000000000000000000000000000000000000000000' +
                                                                   '6800000000000000000000000000000000000000000000000000000000000000' +
                                                                   '4d00000000000000000000000000000000000000000000000000000000000000' +
                                                                   'fb00000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'bytes32', expected: '0x6761766f66796f726b0000000000000000000000000000000000000000000000',
                                                            value: '6761766f66796f726b0000000000000000000000000000000000000000000000'});

        test({ type: 'string', expected: 'gavofyork',       value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                   '6761766f66796f726b0000000000000000000000000000000000000000000000'});
        test({ type: 'string', expected: 'Ã¤Ã¤',
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000008' +
                                                                   'c383c2a4c383c2a4000000000000000000000000000000000000000000000000'});
        test({ type: 'string', expected: 'ü',
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                   'c3bc000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'string', expected: '',
                                            value: '0x00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'string', expected: 'Ã',
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                   'c383000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'string', expected: 'Heeäööä👅D34ɝɣ24Єͽ-.,äü+#/',
                                                            value: '00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000026486565c3a4c3b6c3b6c3a4f09f9185443334c99dc9a33234d084cdbd2d2e2cc3a4c3bc2b232f0000000000000000000000000000000000000000000000000000'});

        test({ type: 'bytes', expected: '0xc3a40000c3a4',
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000006' +
                                                                   'c3a40000c3a40000000000000000000000000000000000000000000000000000'});
        test({ type: 'bytes32', expected: '0xc3a40000c3a40000000000000000000000000000000000000000000000000000',
                                                            value: 'c3a40000c3a40000000000000000000000000000000000000000000000000000'});
        test({ type: 'address', expected: '0x407D73d8a49eeb85D32Cf465507dd71d507100c1',
                                                            value: '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1'});
        test({ type: 'string', expected: 'welcome to ethereum. welcome to ethereum. welcome to ethereum.',
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                    '000000000000000000000000000000000000000000000000000000000000003e' +
                                                                    '77656c636f6d6520746f20657468657265756d2e2077656c636f6d6520746f20' +
                                                                    '657468657265756d2e2077656c636f6d6520746f20657468657265756d2e0000'});
        test({ type: 'bytes', expected: '0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                          'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                          'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                          'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                          'fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff1',
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '000000000000000000000000000000000000000000000000000000000000009f' +
                                                                   'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                                                   'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                                                   'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                                                   'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                                                   'fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff100'});
        test({ type: 'tuple(address)', expected: ['0xbBF289D846208c16EDc8474705C748aff07732dB'],
                                                            value: '000000000000000000000000bbf289d846208c16edc8474705c748aff07732db'});
        test({ type: 'tuple(address,address)', expected: ['0xbBF289D846208c16EDc8474705C748aff07732dB', '0xbBF289D846208c16EDc8474705C748aff07732dB'],
                                                            value: '000000000000000000000000bbf289d846208c16edc8474705c748aff07732db' +
                                                                   '000000000000000000000000bbf289d846208c16edc8474705c748aff07732db'})
        test({ type: 'tuple(uint256,uint256)', expected: ["5","5"],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000005'})
        test({ type: 'tuple(string,string)', expected: ["hello", "world"],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000080' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                   '68656c6c6f000000000000000000000000000000000000000000000000000000' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                   '776f726c64000000000000000000000000000000000000000000000000000000'})
        test({ type: 'tuple(bytes,bytes)', expected: ["0x01fe517acd15ff","0xabcdef12345678"],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000080' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000007' +
                                                                   '01fe517acd15ff00000000000000000000000000000000000000000000000000' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000007' +
                                                                   'abcdef1234567800000000000000000000000000000000000000000000000000'})
        test({ type: 'tuple(bool,bool)', expected: [false, true],
                                                            value: '0000000000000000000000000000000000000000000000000000000000000000' +
                                                                   '0000000000000000000000000000000000000000000000000000000000000001'})

        test({ type: 'tuple(uint256,string,bytes)', expected: ["4", "what what", "0xabcdef12345678"],
                                                             value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000004' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000060' +
                                                                    '00000000000000000000000000000000000000000000000000000000000000a0' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                    '7768617420776861740000000000000000000000000000000000000000000000' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000007' +
                                                                    'abcdef1234567800000000000000000000000000000000000000000000000000'})
        test({ type: 'tuple(uint128,string,bytes)', expected: ["666", "encode your kids", "0x656e636f646520796f75722077696665"],
                                                              value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                     '000000000000000000000000000000000000000000000000000000000000029a' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000060' +
                                                                     '00000000000000000000000000000000000000000000000000000000000000a0' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000010' +
                                                                     '656e636f646520796f7572206b69647300000000000000000000000000000000' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000010' +
                                                                     '656e636f646520796f7572207769666500000000000000000000000000000000' })
        test({ type: 'tuple(string,bytes32,uint256,bool)', expected: ["foo bar", "0xaabbccddeeff0000000000000000000000000000000000000000000000000000", "321", true],
                                                              value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000080' +
                                                                     'aabbccddeeff0000000000000000000000000000000000000000000000000000' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000141' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000007' +
                                                                     '666f6f2062617200000000000000000000000000000000000000000000000000'})
        test({ type: 'tuple(uint8,uint8,uint8,uint8,string,address,bool)', expected: ["1", "2", "3", "4", "five", "0x0000000000000000000000000000000000000006", true],
                                                              value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000004' +
                                                                     '00000000000000000000000000000000000000000000000000000000000000e0' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000006' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000004' +
                                                                     '6669766500000000000000000000000000000000000000000000000000000000' })
        test({ type: 'tuple(tuple(address,address),tuple(uint256,uint256))', expected: [["0x1234567890123456789012345678901234567890","0x1234567890123456789012345678901234567890"],["5", "6"]],
                                                              value: '0000000000000000000000001234567890123456789012345678901234567890' +
                                                                     '0000000000000000000000001234567890123456789012345678901234567890' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000006' })
        test({ type: 'tuple(tuple(address,address),tuple(uint256,uint256),tuple(string,string))', expected: [["0x1234567890123456789012345678901234567890","0x1234567890123456789012345678901234567890"],["5","6"],["a string", "another string"]],
                                                              value: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                     '0000000000000000000000001234567890123456789012345678901234567890' +
                                                                     '0000000000000000000000001234567890123456789012345678901234567890' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000006' +
                                                                     '00000000000000000000000000000000000000000000000000000000000000a0' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000080' +
                                                                     '0000000000000000000000000000000000000000000000000000000000000008' +
                                                                     '6120737472696e67000000000000000000000000000000000000000000000000' +
                                                                     '000000000000000000000000000000000000000000000000000000000000000e' +
                                                                     '616e6f7468657220737472696e67000000000000000000000000000000000000'
                                                                     })
    });
});


describe('lib/solidity/coder', function () {
    describe('decodeParams', function () {
        var test = function (t) {
            it('should turn ' + t.values + ' to ' + t.expected, function () {
                var result = coder.decodeParameters(t.types, t.values);

                var resultArray = [];
                Object.keys(result).map(key => {
                     if(isFinite(key)) resultArray.push(result[key])
              });




                assert.deepEqual(resultArray, t.expected);
            });
        };


        test({ types: ['address'], expected: ['0x407D73d8a49eeb85D32Cf465507dd71d507100c1'],
                                                            values: '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1'});
        test({ types: ['address', 'address'], expected: ['0x407D73d8a49eeb85D32Cf465507dd71d507100c1', '0x407D73d8A49eEB85D32Cf465507Dd71d507100c3'],
                                                            values: '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1' +
                                                                    '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3'});
        test({ types: ['bool[2]', 'bool[3]'], expected: [[true, false], [false, false, true]],
                                                            values: '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000000' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000000' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000000' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ types: ['int[2]', 'int256[3]'], expected: [['1', '2'], ['3', '4', '5']],
                                                            values: '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000004' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000005'});
        test({ types: ['int'], expected: ['1'],       values: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ types: ['uint[2]', 'uint256[3]'], expected: [['1', '2'], ['3', '4', '5']],
                                                            values: '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000004' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000005'});
        test({ types: ['uint'], expected: ['1'],      values: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ types: ['bytes1', 'bytes1'], expected: ['0xaa', '0xbb'],
                                                            values: 'aa00000000000000000000000000000000000000000000000000000000000000' +
                                                                    'bb00000000000000000000000000000000000000000000000000000000000000'});
        test({ types: ['bytes1[2]', 'bytes1'], expected: [['0xaa', '0xbb'], '0xcc'],
                                                            values: 'aa00000000000000000000000000000000000000000000000000000000000000' +
                                                                    'bb00000000000000000000000000000000000000000000000000000000000000' +
                                                                    'cc00000000000000000000000000000000000000000000000000000000000000'});
        test({ types: ['bytes', 'bytes'], expected: ['0x731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b',
                                                     '0x731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134c'],
                                                            values: '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000080' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                    '731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                    '731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134c'});
        test({ types: ['int', 'string', 'int'], expected: ['1', 'gavofyork', '5'],
                                                            values: '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000060' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                    '6761766f66796f726b0000000000000000000000000000000000000000000000'});
        test({ types: ['bytes32', 'int'], expected: ['0x6761766f66796f726b0000000000000000000000000000000000000000000000', '5'],
                                                            values: '6761766f66796f726b0000000000000000000000000000000000000000000000' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000005'});
        test({ types: ['int', 'bytes32'], expected: ['5', '0x6761766f66796f726b0000000000000000000000000000000000000000000000'],
                                                            values: '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                    '6761766f66796f726b0000000000000000000000000000000000000000000000'});
        test({ types: ['int', 'string', 'int', 'int', 'int', 'int[]'], expected: ['1', 'gavofyork', '2', '3', '4',
                                                                                 ['5', '6', '7']],
                                                            values: '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                    '00000000000000000000000000000000000000000000000000000000000000c0' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000004' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000100' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                    '6761766f66796f726b0000000000000000000000000000000000000000000000' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000006' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000007'});
        test({ types: ['int', 'bytes', 'int', 'bytes'], expected: [
                                                                    '5',
                                                                    '0x131a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                      '231a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b',
                                                                    '3',
                                                                    '0x331a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                      '431a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b',
                                                                    ],
                                                            values: '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000080' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                    '00000000000000000000000000000000000000000000000000000000000000e0' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                    '131a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                    '231a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                    '331a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                    '431a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b'});
        test({ types: ['address[2][1]', 'bool'], expected: [[['0x407D73d8a49eeb85D32Cf465507dd71d507100c1',
                                                              '0x407D73d8A49eEB85D32Cf465507Dd71d507100c3']], false],
                                                            values: '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1' +
                                                                    '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000000'});
        test({ types: ['bool[2][1]', 'bool'], expected: [[[true, false]], true],
                                                            values: '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000000' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ types: ['bytes1[2][1]', 'bool'], expected: [[['0xaa', '0xbb']], true],
                                                            values: 'aa00000000000000000000000000000000000000000000000000000000000000' +
                                                                    'bb00000000000000000000000000000000000000000000000000000000000000' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ types: ['int[2][1]', 'bool'], expected: [[['1', '2']], true],
                                                            values: '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ types: ['uint[2][1]', 'bool'], expected: [[['1', '2']], true],
                                                            values: '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                    '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ types: ['tuple(address,address)', 'tuple(string,string)', 'bool'],
               expected: [['0x1234567890123456789012345678901234567890', '0x1234567890123456789012345678901234567890'],["hello", "world"], false],
               values: '0000000000000000000000001234567890123456789012345678901234567890' +
                      '0000000000000000000000001234567890123456789012345678901234567890' +
                      '0000000000000000000000000000000000000000000000000000000000000080' +
                      '0000000000000000000000000000000000000000000000000000000000000000' +
                      '0000000000000000000000000000000000000000000000000000000000000040' +
                      '0000000000000000000000000000000000000000000000000000000000000080' +
                      '0000000000000000000000000000000000000000000000000000000000000005' +
                      '68656c6c6f000000000000000000000000000000000000000000000000000000' +
                      '0000000000000000000000000000000000000000000000000000000000000005' +
                      '776f726c64000000000000000000000000000000000000000000000000000000' })

        test({ types: ['string', 'tuple(uint256,string)', 'bool', 'tuple(bytes32,bytes)'],
               expected: ["the string", ["56","some string"], true, ["0x1234567890123456789012345678901234567890123456789012345678901234", "0x129581"]],
               values: '0000000000000000000000000000000000000000000000000000000000000080' +
                       '00000000000000000000000000000000000000000000000000000000000000c0' +
                       '0000000000000000000000000000000000000000000000000000000000000001' +
                       '0000000000000000000000000000000000000000000000000000000000000140' +
                       '000000000000000000000000000000000000000000000000000000000000000a' +
                       '74686520737472696e6700000000000000000000000000000000000000000000' +
                       '0000000000000000000000000000000000000000000000000000000000000038' +
                       '0000000000000000000000000000000000000000000000000000000000000040' +
                       '000000000000000000000000000000000000000000000000000000000000000b' +
                       '736f6d6520737472696e67000000000000000000000000000000000000000000' +
                       '1234567890123456789012345678901234567890123456789012345678901234' +
                       '0000000000000000000000000000000000000000000000000000000000000040' +
                       '0000000000000000000000000000000000000000000000000000000000000003' +
                       '1295810000000000000000000000000000000000000000000000000000000000'})
        test({ types: ['bool', 'tuple(bool,tuple(bool,tuple(bool,bool)))', 'tuple(uint256,tuple(uint256,uint256,tuple(uint256,string)))', 'string'],
               expected: [true, [false, [true, [false, true]]], ["256",["76","67",["1337","hello"]]], "last param"],
               values: '0000000000000000000000000000000000000000000000000000000000000001' +
                       '0000000000000000000000000000000000000000000000000000000000000000' +
                       '0000000000000000000000000000000000000000000000000000000000000001' +
                       '0000000000000000000000000000000000000000000000000000000000000000' +
                       '0000000000000000000000000000000000000000000000000000000000000001' +
                       '00000000000000000000000000000000000000000000000000000000000000e0' +
                       '0000000000000000000000000000000000000000000000000000000000000200' +
                       '0000000000000000000000000000000000000000000000000000000000000100' +
                       '0000000000000000000000000000000000000000000000000000000000000040' +
                       '000000000000000000000000000000000000000000000000000000000000004c' +
                       '0000000000000000000000000000000000000000000000000000000000000043' +
                       '0000000000000000000000000000000000000000000000000000000000000060' +
                       '0000000000000000000000000000000000000000000000000000000000000539' +
                       '0000000000000000000000000000000000000000000000000000000000000040' +
                       '0000000000000000000000000000000000000000000000000000000000000005' +
                       '68656c6c6f000000000000000000000000000000000000000000000000000000' +
                       '000000000000000000000000000000000000000000000000000000000000000a' +
                       '6c61737420706172616d00000000000000000000000000000000000000000000' })

        test({types: ['string', 'tuple(string,string,tuple(string,string))', 'bytes', 'tuple(bytes,tuple(bytes,string))'],
              expected: ["hello world", ["what","is",["even","happening"]], '0x696e', ['0x74686973', ['0x676f64666f7273616b656e', "test"]]],
              values: '0000000000000000000000000000000000000000000000000000000000000080' +
                      '00000000000000000000000000000000000000000000000000000000000000c0' +
                      '0000000000000000000000000000000000000000000000000000000000000260' +
                      '00000000000000000000000000000000000000000000000000000000000002a0' +
                      '000000000000000000000000000000000000000000000000000000000000000b' +
                      '68656c6c6f20776f726c64000000000000000000000000000000000000000000' +
                      '0000000000000000000000000000000000000000000000000000000000000060' +
                      '00000000000000000000000000000000000000000000000000000000000000a0' +
                      '00000000000000000000000000000000000000000000000000000000000000e0' +
                      '0000000000000000000000000000000000000000000000000000000000000004' +
                      '7768617400000000000000000000000000000000000000000000000000000000' +
                      '0000000000000000000000000000000000000000000000000000000000000002' +
                      '6973000000000000000000000000000000000000000000000000000000000000' +
                      '0000000000000000000000000000000000000000000000000000000000000040' +
                      '0000000000000000000000000000000000000000000000000000000000000080' +
                      '0000000000000000000000000000000000000000000000000000000000000004' +
                      '6576656e00000000000000000000000000000000000000000000000000000000' +
                      '0000000000000000000000000000000000000000000000000000000000000009' +
                      '68617070656e696e670000000000000000000000000000000000000000000000' +
                      '0000000000000000000000000000000000000000000000000000000000000002' +
                      '696e000000000000000000000000000000000000000000000000000000000000' +
                      '0000000000000000000000000000000000000000000000000000000000000040' +
                      '0000000000000000000000000000000000000000000000000000000000000080' +
                      '0000000000000000000000000000000000000000000000000000000000000004' +
                      '7468697300000000000000000000000000000000000000000000000000000000' +
                      '0000000000000000000000000000000000000000000000000000000000000040' +
                      '0000000000000000000000000000000000000000000000000000000000000080' +
                      '000000000000000000000000000000000000000000000000000000000000000b' +
                      '676f64666f7273616b656e000000000000000000000000000000000000000000' +
                      '0000000000000000000000000000000000000000000000000000000000000004' +
                      '7465737400000000000000000000000000000000000000000000000000000000' })
        test({ types: ["string", "bool", "tuple(string,tuple(tuple(bool,string,bool),tuple(string,bytes32,bytes)))", "bytes", "tuple(bool,tuple(bytes32,address),bytes)"],
               expected: ["this",true, ["is", [[true, "utter", true],["madness","0x1234567890123456789012345678901234567890123456789012345678901234", "0x6275742049206c6f7665206974"]]],"0x6265636175736520697420776f726b73", [true,["0x1234567890123456789012345678901234567890123456789012345678901234","0x1337133713371337133713371337133713371337"],"0x6265636175736520697420776f726b736265636175736520697420776f726b73"]],
               values: '00000000000000000000000000000000000000000000000000000000000000a0' +
                       '0000000000000000000000000000000000000000000000000000000000000001' +
                       '00000000000000000000000000000000000000000000000000000000000000e0' +
                       '0000000000000000000000000000000000000000000000000000000000000320' +
                       '0000000000000000000000000000000000000000000000000000000000000360' +
                       '0000000000000000000000000000000000000000000000000000000000000004' +
                       '7468697300000000000000000000000000000000000000000000000000000000' +
                       '0000000000000000000000000000000000000000000000000000000000000040' +
                       '0000000000000000000000000000000000000000000000000000000000000080' +
                       '0000000000000000000000000000000000000000000000000000000000000002' +
                       '6973000000000000000000000000000000000000000000000000000000000000' +
                       '0000000000000000000000000000000000000000000000000000000000000040' +
                       '00000000000000000000000000000000000000000000000000000000000000e0' +
                       '0000000000000000000000000000000000000000000000000000000000000001' +
                       '0000000000000000000000000000000000000000000000000000000000000060' +
                       '0000000000000000000000000000000000000000000000000000000000000001' +
                       '0000000000000000000000000000000000000000000000000000000000000005' +
                       '7574746572000000000000000000000000000000000000000000000000000000' +
                       '0000000000000000000000000000000000000000000000000000000000000060' +
                       '1234567890123456789012345678901234567890123456789012345678901234' +
                       '00000000000000000000000000000000000000000000000000000000000000a0' +
                       '0000000000000000000000000000000000000000000000000000000000000007' +
                       '6d61646e65737300000000000000000000000000000000000000000000000000' +
                       '000000000000000000000000000000000000000000000000000000000000000d' +
                       '6275742049206c6f766520697400000000000000000000000000000000000000' +
                       '0000000000000000000000000000000000000000000000000000000000000010' +
                       '6265636175736520697420776f726b7300000000000000000000000000000000' +
                       '0000000000000000000000000000000000000000000000000000000000000001' +
                       '1234567890123456789012345678901234567890123456789012345678901234' +
                       '0000000000000000000000001337133713371337133713371337133713371337' +
                       '0000000000000000000000000000000000000000000000000000000000000080' +
                       '0000000000000000000000000000000000000000000000000000000000000020' +
                       '6265636175736520697420776f726b736265636175736520697420776f726b73'})
        test( { types: ['tuple(string,tuple(bool,bool))','address','bytes','tuple(bytes,bytes,string,bool,address)','int256','tuple(int256,uint256,tuple(string,int256,address))'],
                expected: [ ["this is more reasonable", [true, false]],"0x1b3F5FE0Fd513E6cbdEE459F0b0e19095FE91958","0x6c6f6c6f6c6f6c6f6c",["0xabcdef12345678", "0x87654321fedcba", "bazbar", false, "0xd13b6e9058E58B8677233CEc2315e1D9e77C79C4"], "-6", ["-7", "5", ["foobar","-8","0xB1eeF147028E9f480DbC5ccaA3277D417D1b85F0"]]],
                values: '00000000000000000000000000000000000000000000000000000000000000c0' +
                        '0000000000000000000000001b3f5fe0fd513e6cbdee459f0b0e19095fe91958' +
                        '0000000000000000000000000000000000000000000000000000000000000160' +
                        '00000000000000000000000000000000000000000000000000000000000001a0' +
                        'fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffa' +
                        '0000000000000000000000000000000000000000000000000000000000000300' +
                        '0000000000000000000000000000000000000000000000000000000000000060' +
                        '0000000000000000000000000000000000000000000000000000000000000001' +
                        '0000000000000000000000000000000000000000000000000000000000000000' +
                        '0000000000000000000000000000000000000000000000000000000000000017' +
                        '74686973206973206d6f726520726561736f6e61626c65000000000000000000' +
                        '0000000000000000000000000000000000000000000000000000000000000009' +
                        '6c6f6c6f6c6f6c6f6c0000000000000000000000000000000000000000000000' +
                        '00000000000000000000000000000000000000000000000000000000000000a0' +
                        '00000000000000000000000000000000000000000000000000000000000000e0' +
                        '0000000000000000000000000000000000000000000000000000000000000120' +
                        '0000000000000000000000000000000000000000000000000000000000000000' +
                        '000000000000000000000000d13b6e9058e58b8677233cec2315e1d9e77c79c4' +
                        '0000000000000000000000000000000000000000000000000000000000000007' +
                        'abcdef1234567800000000000000000000000000000000000000000000000000' +
                        '0000000000000000000000000000000000000000000000000000000000000007' +
                        '87654321fedcba00000000000000000000000000000000000000000000000000' +
                        '0000000000000000000000000000000000000000000000000000000000000006' +
                        '62617a6261720000000000000000000000000000000000000000000000000000' +
                        'fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff9' +
                        '0000000000000000000000000000000000000000000000000000000000000005' +
                        '0000000000000000000000000000000000000000000000000000000000000060' +
                        '0000000000000000000000000000000000000000000000000000000000000060' +
                        'fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff8' +
                        '000000000000000000000000b1eef147028e9f480dbc5ccaa3277d417d1b85f0' +
                        '0000000000000000000000000000000000000000000000000000000000000006' +
                        '666f6f6261720000000000000000000000000000000000000000000000000000'})
        test( { types: ['tuple(bytes32,bool,bytes32)', 'address', 'tuple(bytes32,bytes32,string)', 'tuple(tuple(address,bool),tuple(address,bytes32),tuple(int256,uint256))'],
                expected: [["0xffffffffffffffffffffffffffffffffabdef123849181759adebfadecaefbae", true, "0xffffffffffffffffffffffffffffffffabdef123849181759adebfadecaefbae"], "0x1234567890123456789012345678901234567890" ,["0x0ab3e6dfa1594c15af0000000000000000000000000000000000000000000000","0xffffffffffffffffffffffffffffffffabdef123849181759adebfadecaefbae","string"], [["0x1234567890123456789012345678901234567890", true],["0x1234567890123456789012345678901234567890","0xffffffffffffffffffffffffffffffffabdef123849181759adebfadecaefbae"],["-6124612", "89000"]]],
                values: 'ffffffffffffffffffffffffffffffffabdef123849181759adebfadecaefbae' +
                        '0000000000000000000000000000000000000000000000000000000000000001' +
                        'ffffffffffffffffffffffffffffffffabdef123849181759adebfadecaefbae' +
                        '0000000000000000000000001234567890123456789012345678901234567890' +
                        '0000000000000000000000000000000000000000000000000000000000000160' +
                        '0000000000000000000000001234567890123456789012345678901234567890' +
                        '0000000000000000000000000000000000000000000000000000000000000001' +
                        '0000000000000000000000001234567890123456789012345678901234567890' +
                        'ffffffffffffffffffffffffffffffffabdef123849181759adebfadecaefbae' +
                        'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffa28bbc' +
                        '0000000000000000000000000000000000000000000000000000000000015ba8' +
                        '0ab3e6dfa1594c15af0000000000000000000000000000000000000000000000' +
                        'ffffffffffffffffffffffffffffffffabdef123849181759adebfadecaefbae' +
                        '0000000000000000000000000000000000000000000000000000000000000060' +
                        '0000000000000000000000000000000000000000000000000000000000000006' +
                        '737472696e670000000000000000000000000000000000000000000000000000'})
    });
});

describe('/lib/solidity/coder', function() {
    describe('decodeParam', function () {
        it('should not alter inputs', function () {
            const t = {
                type: "function",
                name: "f",
                internalType: "function () external"
            };
            const copyOfT = Object.assign({}, t);
            coder.decodeParameter(t, '063e4f349a9e91c6575aedab0e70087fab642ecac04062260000000000000000'); //must not alter t!
            assert.deepEqual(t, copyOfT);
        });
    });
});
