var chai = require('chai');
var assert = chai.assert;
var BN = require('bn.js');
var BigNumber = require('bignumber.js');
var coder = require('../packages/web3-eth-abi');


describe('lib/solidity/coder', function () {
    describe('encodeParameter', function () {
        var test = function (t) {
            it('should turn ' + t.value + ' to ' + t.expected + ' for '+ t.type, function () {
                assert.equal(coder.encodeParameter(t.type, t.value).replace('0x',''), t.expected);
            });
        };


        test({ type: 'address', value: '0x407d73d8a49eeb85d32cf465507dd71d507100c1',
                                                    expected: '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1'});
        test({ type: 'address[2]', value: ['0x407d73d8a49eeb85d32cf465507dd71d507100c1', '0x407d73d8a49eeb85d32cf465507dd71d507100c3'],
                                                    expected: '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1' +
                                                              '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3' });
        test({ type: 'address[]', value: ['0x407d73d8a49eeb85d32cf465507dd71d507100c1', '0x407d73d8a49eeb85d32cf465507dd71d507100c3'],
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000002' +
                                                              '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1' +
                                                              '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3' });
        test({ type: 'address[][2]', value: [['0x407d73d8a49eeb85d32cf465507dd71d507100c1', '0x407d73d8a49eeb85d32cf465507dd71d507100c2'],
                                               ['0x407d73d8a49eeb85d32cf465507dd71d507100c3', '0x407d73d8a49eeb85d32cf465507dd71d507100c4']],
                                                    expected:   '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                '00000000000000000000000000000000000000000000000000000000000000a0' +
                                                                '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1' +
                                                                '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c2' +
                                                                '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3' +
                                                                '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c4' });
        test({ type: 'address[2][]', value: [['0x407d73d8a49eeb85d32cf465507dd71d507100c1', '0x407d73d8a49eeb85d32cf465507dd71d507100c2'],
                                               ['0x407d73d8a49eeb85d32cf465507dd71d507100c3', '0x407d73d8a49eeb85d32cf465507dd71d507100c4']],
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000002' +
                                                              '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1' +
                                                              '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c2' +
                                                              '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3' +
                                                              '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c4' });
        //test({ type: 'address[][]', value: [['0x407d73d8a49eeb85d32cf465507dd71d507100c5'],
                                               //['0x407d73d8a49eeb85d32cf465507dd71d507100c3']],
                                                    //expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              //'0000000000000000000000000000000000000000000000000000000000000002' +
                                                              //'0000000000000000000000000000000000000000000000000000000000000080' +
                                                              //'00000000000000000000000000000000000000000000000000000000000000c0' +
                                                              //'0000000000000000000000000000000000000000000000000000000000000001' +
                                                              //'000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c5' +
                                                              //'0000000000000000000000000000000000000000000000000000000000000001' +
                                                              //'000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3' });
        //test({ type: 'address[][]', value: [['0x407d73d8a49eeb85d32cf465507dd71d507100cf', '0x407d73d8a49eeb85d32cf465507dd71d507100c2'],
                                               //['0x407d73d8a49eeb85d32cf465507dd71d507100c3', '0x407d73d8a49eeb85d32cf465507dd71d507100c4']],
                                                    //expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              //'0000000000000000000000000000000000000000000000000000000000000002' +
                                                              //'0000000000000000000000000000000000000000000000000000000000000080' +
                                                              //'00000000000000000000000000000000000000000000000000000000000000e0' +
                                                              //'0000000000000000000000000000000000000000000000000000000000000002' +
                                                              //'000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100cf' +
                                                              //'000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c2' +
                                                              //'0000000000000000000000000000000000000000000000000000000000000002' +
                                                              //'000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3' +
                                                              //'000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c4' });
        test({ type: 'bool', value: true,           expected: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ type: 'bool', value: false,          expected: '0000000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'bool[1][2]', value: [[false], [false]],
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000000' +
                                                              '0000000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'bool[2]', value: [true, false],
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000001' +
                                                              '0000000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'bool[]', value: [true, true, false],
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000003' +
                                                              '0000000000000000000000000000000000000000000000000000000000000001' +
                                                              '0000000000000000000000000000000000000000000000000000000000000001' +
                                                              '0000000000000000000000000000000000000000000000000000000000000000'});

        test({ type: 'int', value: 1,               expected: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ type: 'int', value: 16,              expected: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'int', value: -1,              expected: 'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'});
        // test({ type: 'int', value: 0.1,             expected: '0000000000000000000000000000000000000000000000000000000000000000'});
        // test({ type: 'int', value: 3.9,             expected: '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ type: 'int256', value: 1,            expected: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ type: 'int256', value: 16,           expected: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'int256', value: -1,           expected: 'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'});

        test({ type: 'uint', value: 1,               expected: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ type: 'uint', value: 16,              expected: '0000000000000000000000000000000000000000000000000000000000000010'});
        // test({ type: 'uint', value: 0.1,             expected: '0000000000000000000000000000000000000000000000000000000000000000'});
        // test({ type: 'uint', value: 3.9,             expected: '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ type: 'uint256', value: 1,            expected: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ type: 'uint256', value: 16,           expected: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ type: 'uint256', value: '0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff',
                                                    expected: 'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'});
        test({ type: 'bytes32', value: '0x6761766f66796f726b',
                                                    expected: '6761766f66796f726b0000000000000000000000000000000000000000000000'});
        test({ type: 'bytes32', value: '0x731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b',
                                                    expected: '731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b'});
        test({ type: 'bytes32', value: '0x02838654a83c213dae3698391eabbd54a5b6e1fb3452bc7fa4ea0dd5c8ce7e29',
                                                    expected: '02838654a83c213dae3698391eabbd54a5b6e1fb3452bc7fa4ea0dd5c8ce7e29'});
        test({ type: 'bytes', value: '0x6761766f66796f726b',
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000009' +
                                                              '6761766f66796f726b0000000000000000000000000000000000000000000000'});
        test({ type: 'bytes', value: '0x731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b',
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '731a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b'});
        test({ type: 'bytes', value: '0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                       'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                       'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                       'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                       'fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff1',
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '000000000000000000000000000000000000000000000000000000000000009f' +
                                                              'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                                              'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                                              'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                                              'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                                                              'fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff100'});
        test({ type: 'string', value: 'gavofyork',  expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000009' +
                                                              '6761766f66796f726b0000000000000000000000000000000000000000000000'});

        test({ type: 'string', value: 'Heeäööä👅D34ɝɣ24Єͽ-.,äü+#/',
            expected: '00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000026486565c3a4c3b6c3b6c3a4f09f9185443334c99dc9a33234d084cdbd2d2e2cc3a4c3bc2b232f0000000000000000000000000000000000000000000000000000'});

        test({ type: 'bytes', value: '0xc3a40000c3a4',
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000006' +
                                                              'c3a40000c3a40000000000000000000000000000000000000000000000000000'});
        test({ type: 'bytes32', value: '0xc3a40000c3a4',
                                                    expected: 'c3a40000c3a40000000000000000000000000000000000000000000000000000'});
        test({ type: 'string', value: 'Ã¤Ã¤',
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000008' +
                                                              'c383c2a4c383c2a4000000000000000000000000000000000000000000000000'});
        test({ type: 'string', value: 'ü',
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000002' +
                                                              'c3bc000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'string', value: 'Ã',
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000002' +
                                                              'c383000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'int[]', value: [],            expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000000'});
        test({ type: 'int[]', value: [3],           expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000001' +
                                                              '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ type: 'int256[]', value: [3],        expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000001' +
                                                              '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ type: 'int[]', value: [1,2,3],       expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000003' +
                                                              '0000000000000000000000000000000000000000000000000000000000000001' +
                                                              '0000000000000000000000000000000000000000000000000000000000000002' +
                                                              '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ type: 'bytes1[4]', value: ['0xcf', '0x68', '0x4d', '0xfb'],
                                                    expected: 'cf00000000000000000000000000000000000000000000000000000000000000' +
                                                              '6800000000000000000000000000000000000000000000000000000000000000' +
                                                              '4d00000000000000000000000000000000000000000000000000000000000000' +
                                                              'fb00000000000000000000000000000000000000000000000000000000000000'});



        test({ type: 'bytes', value: '0x131a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                       '231a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b',
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000040' +
                                                              '131a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                              '231a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b'});
        test({ type: 'bytes', value: '0x131a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                       '231a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                       '331a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b',
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '0000000000000000000000000000000000000000000000000000000000000060' +
                                                              '131a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                              '231a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                              '331a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b'});
        test({ type: 'string', value: 'welcome to ethereum. welcome to ethereum. welcome to ethereum.',
                                                    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '000000000000000000000000000000000000000000000000000000000000003e' +
                                                              '77656c636f6d6520746f20657468657265756d2e2077656c636f6d6520746f20' +
                                                              '657468657265756d2e2077656c636f6d6520746f20657468657265756d2e0000'});
        test({
            type: 'tuple(string,string)',
            value: ["welcome to ethereum.", "welcome to ethereum."],
            expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                      '0000000000000000000000000000000000000000000000000000000000000040' +
                      '0000000000000000000000000000000000000000000000000000000000000080' +
                      '0000000000000000000000000000000000000000000000000000000000000014' +
                      '77656c636f6d6520746f20657468657265756d2e000000000000000000000000' +
                      '0000000000000000000000000000000000000000000000000000000000000014' +
                      '77656c636f6d6520746f20657468657265756d2e000000000000000000000000' })
        test({
            type: 'tuple(bytes,bytes)',
            value: ["0x77656c636f6d6520746f20657468657265756d2e", "0x77656c636f6d6520746f20657468657265756d2e"],
            expected:
              '0000000000000000000000000000000000000000000000000000000000000020' +
              '0000000000000000000000000000000000000000000000000000000000000040' +
              '0000000000000000000000000000000000000000000000000000000000000080' +
              '0000000000000000000000000000000000000000000000000000000000000014' +
              '77656c636f6d6520746f20657468657265756d2e000000000000000000000000' +
              '0000000000000000000000000000000000000000000000000000000000000014' +
              '77656c636f6d6520746f20657468657265756d2e000000000000000000000000' })
        test({
            type: 'tuple(bytes,bool,uint256)',
            value: ["0x77656c636f6d6520746f20657468657265756d2e", true, 124515],
            expected:
              '0000000000000000000000000000000000000000000000000000000000000020' +
              '0000000000000000000000000000000000000000000000000000000000000060' +
              '0000000000000000000000000000000000000000000000000000000000000001' +
              '000000000000000000000000000000000000000000000000000000000001e663' +
              '0000000000000000000000000000000000000000000000000000000000000014' +
              '77656c636f6d6520746f20657468657265756d2e000000000000000000000000' })
        test({
            type: 'tuple(string,tuple(bool,int256),address)',
            value: ["hello", [true,-151], "0x0175010374017501037401750103740175010374"],
            expected:
              '0000000000000000000000000000000000000000000000000000000000000020' +
              '0000000000000000000000000000000000000000000000000000000000000080' +
              '0000000000000000000000000000000000000000000000000000000000000001' +
              'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff69' +
              '0000000000000000000000000175010374017501037401750103740175010374' +
              '0000000000000000000000000000000000000000000000000000000000000005' +
              '68656c6c6f000000000000000000000000000000000000000000000000000000' })
        test({
            type: 'tuple(tuple(bool,bool),tuple(address,address),tuple(string,string))',
            value: [[true, false],["0x81017589ab81017589ab81017589ab81017589ab", "0x81017589ab81017589ab81017589ab81017589ab"],["string One", "string Two"]],
            expected:
              '0000000000000000000000000000000000000000000000000000000000000020' +
              '0000000000000000000000000000000000000000000000000000000000000001' +
              '0000000000000000000000000000000000000000000000000000000000000000' +
              '00000000000000000000000081017589ab81017589ab81017589ab81017589ab' +
              '00000000000000000000000081017589ab81017589ab81017589ab81017589ab' +
              '00000000000000000000000000000000000000000000000000000000000000a0' +
              '0000000000000000000000000000000000000000000000000000000000000040' +
              '0000000000000000000000000000000000000000000000000000000000000080' +
              '000000000000000000000000000000000000000000000000000000000000000a' +
              '737472696e67204f6e6500000000000000000000000000000000000000000000' +
              '000000000000000000000000000000000000000000000000000000000000000a' +
              '737472696e672054776f00000000000000000000000000000000000000000000'})
        test({
            type: 'tuple(tuple(tuple(bool,bool),tuple(bytes,bytes),tuple(address,bool)),address)',
            value: [[[false, false],["0xab1394581edfa2ef9ca71", "0x15abe391df19aef19a4561"],["0xec2270c849236333c86834728e783cd2f789088e", true]], "0x81017589ab81017589ab81017589ab81017589ab"],
            expected:
              '0000000000000000000000000000000000000000000000000000000000000020' +
              '0000000000000000000000000000000000000000000000000000000000000040' +
              '00000000000000000000000081017589ab81017589ab81017589ab81017589ab' +
              '0000000000000000000000000000000000000000000000000000000000000000' +
              '0000000000000000000000000000000000000000000000000000000000000000' +
              '00000000000000000000000000000000000000000000000000000000000000a0' +
              '000000000000000000000000ec2270c849236333c86834728e783cd2f789088e' +
              '0000000000000000000000000000000000000000000000000000000000000001' +
              '0000000000000000000000000000000000000000000000000000000000000040' +
              '0000000000000000000000000000000000000000000000000000000000000080' +
              '000000000000000000000000000000000000000000000000000000000000000b' +
              '0ab1394581edfa2ef9ca71000000000000000000000000000000000000000000' +
              '000000000000000000000000000000000000000000000000000000000000000b' +
              '15abe391df19aef19a4561000000000000000000000000000000000000000000' })
        test({
            type: 'tuple(bool,string,bool,tuple(address,address))',
            value: [true, "testing", false, ["0x1981710abe1981710abe1981710abe1981710abe", "0x1981710abe1981710abe1981710abe1981710abe"]],
            expected:
              '0000000000000000000000000000000000000000000000000000000000000020' +
              '0000000000000000000000000000000000000000000000000000000000000001' +
              '00000000000000000000000000000000000000000000000000000000000000a0' +
              '0000000000000000000000000000000000000000000000000000000000000000' +
              '0000000000000000000000001981710abe1981710abe1981710abe1981710abe' +
              '0000000000000000000000001981710abe1981710abe1981710abe1981710abe' +
              '0000000000000000000000000000000000000000000000000000000000000007' +
              '74657374696e6700000000000000000000000000000000000000000000000000'  })
        test({
            type: 'tuple(address,address,tuple(string,tuple(int256,int256),string))',
            value: ["0x1981710abe1981710abe1981710abe1981710abe", "0x1981710abe1981710abe1981710abe1981710abe", ["structs are great", [-1951, 194018], "so many possibilities"]],
            expected:
              '0000000000000000000000000000000000000000000000000000000000000020' +
              '0000000000000000000000001981710abe1981710abe1981710abe1981710abe' +
              '0000000000000000000000001981710abe1981710abe1981710abe1981710abe' +
              '0000000000000000000000000000000000000000000000000000000000000060' +
              '0000000000000000000000000000000000000000000000000000000000000080' +
              'fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff861' +
              '000000000000000000000000000000000000000000000000000000000002f5e2' +
              '00000000000000000000000000000000000000000000000000000000000000c0' +
              '0000000000000000000000000000000000000000000000000000000000000011' +
              '7374727563747320617265206772656174000000000000000000000000000000' +
              '0000000000000000000000000000000000000000000000000000000000000015' +
              '736f206d616e7920706f73736962696c69746965730000000000000000000000' })
        test({
            type: 'tuple(bool,tuple(bytes32,int256,tuple(bytes24,bytes8)),tuple(bool,bool,bool),string)',
            value: [true, ["0xabdef18710a18a18abdef18710a18a18abdef18710a18a18abdef18710a18a18", -18291849, ["0xabdef18710a18a18abdef18710a18a18abdef18710a18a18","0xabdef18710a18a18"]], [false, true, false], "testing testing"],
            expected:
              '0000000000000000000000000000000000000000000000000000000000000020' +
              '0000000000000000000000000000000000000000000000000000000000000001' +
              'abdef18710a18a18abdef18710a18a18abdef18710a18a18abdef18710a18a18' +
              'fffffffffffffffffffffffffffffffffffffffffffffffffffffffffee8e377' +
              'abdef18710a18a18abdef18710a18a18abdef18710a18a180000000000000000' +
              'abdef18710a18a18000000000000000000000000000000000000000000000000' +
              '0000000000000000000000000000000000000000000000000000000000000000' +
              '0000000000000000000000000000000000000000000000000000000000000001' +
              '0000000000000000000000000000000000000000000000000000000000000000' +
              '0000000000000000000000000000000000000000000000000000000000000120' +
              '000000000000000000000000000000000000000000000000000000000000000f' +
              '74657374696e672074657374696e670000000000000000000000000000000000' });
        test({
            type: 'tuple(bool,tuple(bytes32,int256,tuple(bytes24,bytes8)),tuple(bool,bool,bool),string)',
            value: [true, ["0xabdef", -18291849, ["0xabdef18710a18a18abdef18710a18a18abdef18710a18a18","0xabdef18710a18a18"]], [false, true, false], "testing testing"],
            expected:
              '0000000000000000000000000000000000000000000000000000000000000020' +
              '0000000000000000000000000000000000000000000000000000000000000001' +
              'abdef00000000000000000000000000000000000000000000000000000000000' +
              'fffffffffffffffffffffffffffffffffffffffffffffffffffffffffee8e377' +
              'abdef18710a18a18abdef18710a18a18abdef18710a18a180000000000000000' +
              'abdef18710a18a18000000000000000000000000000000000000000000000000' +
              '0000000000000000000000000000000000000000000000000000000000000000' +
              '0000000000000000000000000000000000000000000000000000000000000001' +
              '0000000000000000000000000000000000000000000000000000000000000000' +
              '0000000000000000000000000000000000000000000000000000000000000120' +
              '000000000000000000000000000000000000000000000000000000000000000f' +
              '74657374696e672074657374696e670000000000000000000000000000000000' });
        test({
            type: 'uint256',
            value: new BN(42),
            expected: '000000000000000000000000000000000000000000000000000000000000002a'
        });
        test({
            type: 'uint256',
            value: new BigNumber(42),
            expected: '000000000000000000000000000000000000000000000000000000000000002a'
        });
        test({
            type: 'uint256[]',
            value: ['42'],
            expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                      '0000000000000000000000000000000000000000000000000000000000000001' +
                      '000000000000000000000000000000000000000000000000000000000000002a'
        });
        test({
            type: 'uint256[]',
            value: [new BN(42)],
            expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                      '0000000000000000000000000000000000000000000000000000000000000001' +
                      '000000000000000000000000000000000000000000000000000000000000002a'
        });
        test({
            type: 'uint256[]',
            value: [new BigNumber(42)],
            expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                      '0000000000000000000000000000000000000000000000000000000000000001' +
                      '000000000000000000000000000000000000000000000000000000000000002a'
        });
        test({
            type: 'uint256[]',
            value: [new BN(42), new BN(42)],
            expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                      '0000000000000000000000000000000000000000000000000000000000000002' +
                      '000000000000000000000000000000000000000000000000000000000000002a' +
                      '000000000000000000000000000000000000000000000000000000000000002a'
        });
    });
});


describe('lib/solidity/coder', function () {
    describe('encodeParameters', function () {
        var test = function (t) {
            it('should turn ' + t.values + ' to ' + t.expected, function () {
                assert.equal(coder.encodeParameters(t.types, t.values).replace('0x',''), t.expected);
            });
        };


        test({ types: ['address', 'address'], values: ['0x407d73d8a49eeb85d32cf465507dd71d507100c1', '0x407d73d8a49eeb85d32cf465507dd71d507100c3'],
                                                            expected: '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c1' +
                                                                      '000000000000000000000000407d73d8a49eeb85d32cf465507dd71d507100c3'});
        test({ types: ['bool[2]', 'bool[3]'], values: [[true, false], [false, false, true]],
                                                            expected: '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000000' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000000' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000000' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ types: ['int'], values: [1],                 expected: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ types: ['int'], values: [16],                expected: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ types: ['int'], values: [-1],                expected: 'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'});
        test({ types: ['int256'], values: [1],              expected: '0000000000000000000000000000000000000000000000000000000000000001'});
        test({ types: ['int256'], values: [16],             expected: '0000000000000000000000000000000000000000000000000000000000000010'});
        test({ types: ['int256'], values: [-1],             expected: 'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'});
        test({ types: ['int[]'], values: [[3]],             expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ types: ['int256[]'], values: [[3]],          expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ types: ['int256[]'], values: [[1,2,3]],      expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ types: ['int[]', 'int[]'], values: [[1,2], [3,4]],
                                                            expected: '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                      '00000000000000000000000000000000000000000000000000000000000000a0' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000004'});
        test({ types: ['int[]', 'int[]', 'int[]'], values: [[1,2], [3,4], [5,6,7]],
                                                            expected: '0000000000000000000000000000000000000000000000000000000000000060' +
                                                                      '00000000000000000000000000000000000000000000000000000000000000c0' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000120' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000004' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000006' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000007'});
        test({ types: ['bytes32'], values: ['0x6761766f66796f726b'],
                                                            expected: '6761766f66796f726b0000000000000000000000000000000000000000000000'});
        test({ types: ['string'], values: ['gavofyork'],    expected: '0000000000000000000000000000000000000000000000000000000000000020' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                      '6761766f66796f726b0000000000000000000000000000000000000000000000'});
        test({ types: ['string', 'string'], values: ['gavofyork', 'gavofyork'],
                                                            expected: '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000080' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                      '6761766f66796f726b0000000000000000000000000000000000000000000000' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                      '6761766f66796f726b0000000000000000000000000000000000000000000000'});


        test({ types: ['bytes32', 'int'], values: ['0x6761766f66796f726b', 5],
                                                            expected: '6761766f66796f726b0000000000000000000000000000000000000000000000' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000005'});
        test({ types: ['int', 'bytes32'], values: [5, '0x6761766f66796f726b'],
                                                            expected: '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                      '6761766f66796f726b0000000000000000000000000000000000000000000000'});
        test({ types: ['int', 'string'], values: [5, 'gavofyork'],
                                                            expected: '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                      '6761766f66796f726b0000000000000000000000000000000000000000000000'});
        test({ types: ['string', 'int'], values: ['gavofyork', 5],
                                                            expected: '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                      '6761766f66796f726b0000000000000000000000000000000000000000000000'});
        test({ types: ['string', 'bool', 'int[]'], values: ['gavofyork', true, [1, 2, 3]],
                                                            expected: '0000000000000000000000000000000000000000000000000000000000000060' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                      '00000000000000000000000000000000000000000000000000000000000000a0' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                      '6761766f66796f726b0000000000000000000000000000000000000000000000' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ types: ['string', 'int[]'], values: ['gavofyork', [1, 2, 3]],
                                                            expected: '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000080' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                      '6761766f66796f726b0000000000000000000000000000000000000000000000' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003'});
        test({ types: ['int', 'string'], values: [5, 'gavofyork'],
                                                            expected: '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                      '6761766f66796f726b0000000000000000000000000000000000000000000000'});
        test({ types: ['int', 'string', 'int', 'int', 'int', 'int[]'], values: [1, 'gavofyork', 2, 3, 4, [5, 6, 7]],
                                                            expected: '0000000000000000000000000000000000000000000000000000000000000001' +
                                                                      '00000000000000000000000000000000000000000000000000000000000000c0' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000002' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000004' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000100' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000009' +
                                                                      '6761766f66796f726b0000000000000000000000000000000000000000000000' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000006' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000007'});
        test({ types: ['int', 'bytes', 'int', 'bytes'], values: [
                                                                    5,
                                                                    '0x131a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                      '231a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b',
                                                                    3,
                                                                    '0x331a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                      '431a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b',
                                                                    ],
                                                            expected: '0000000000000000000000000000000000000000000000000000000000000005' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000080' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000003' +
                                                                      '00000000000000000000000000000000000000000000000000000000000000e0' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                      '131a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                      '231a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                      '0000000000000000000000000000000000000000000000000000000000000040' +
                                                                      '331a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b' +
                                                                      '431a3afc00d1b1e3461b955e53fc866dcf303b3eb9f4c16f89e388930f48134b'});
        test({ types: ['bytes3', 'bytes'], values: ['0xcf0011', '0x4d00000000000000000000000000000000000000000000000000000000000012'],
                                                    expected: 'cf00110000000000000000000000000000000000000000000000000000000000' +
                                                              '0000000000000000000000000000000000000000000000000000000000000040' +
                                                              '0000000000000000000000000000000000000000000000000000000000000020' +
                                                              '4d00000000000000000000000000000000000000000000000000000000000012'});
        test({
          types: ['string', 'tuple(string,string)'],
          values: ["what", ["what what", "what what"]],
          expected: '0000000000000000000000000000000000000000000000000000000000000040' +
                    '0000000000000000000000000000000000000000000000000000000000000080' +
                    '0000000000000000000000000000000000000000000000000000000000000004' +
                    '7768617400000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000040' +
                    '0000000000000000000000000000000000000000000000000000000000000080' +
                    '0000000000000000000000000000000000000000000000000000000000000009' +
                    '7768617420776861740000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000009' +
                    '7768617420776861740000000000000000000000000000000000000000000000'})
        test({
          types: ['tuple(bytes32,bool)', 'tuple(bool,address)'],
          values: [["0xabdef18710a18a18abdef18710a18a18abdef18710a18a18abdef18710a18a18", true], [true, "0x77656c636f6d6520746f20657468657265756d2e"]],
          expected: 'abdef18710a18a18abdef18710a18a18abdef18710a18a18abdef18710a18a18' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e'})
        test({
          types: ['tuple(bytes32,bool)', 'tuple(bool,address)'],
          values: [["0xabdef18710a18a18abdef18710a18a18abdef18710a18a18abdef18710a18a18", true], [true, "0x77656c636f6d6520746f20657468657265756d2e"]],
          expected: 'abdef18710a18a18abdef18710a18a18abdef18710a18a18abdef18710a18a18' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e'})
        test({
          types: ['tuple(address,uint256)', 'tuple(uint256,bool)', 'tuple(bytes32,bytes32)'],
          values: [["0x77656c636f6d6520746f20657468657265756d2e", "148"],["5910", true],["0xabdef18710a18a18abdef18710a18a18abdef18710a18a18abdef18710a18a18", "0xabdef18710a18a18abdef18710a18a18abdef18710a18a18abdef18710a18a18"]],
          expected: '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '0000000000000000000000000000000000000000000000000000000000000094' +
                    '0000000000000000000000000000000000000000000000000000000000001716' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    'abdef18710a18a18abdef18710a18a18abdef18710a18a18abdef18710a18a18' +
                    'abdef18710a18a18abdef18710a18a18abdef18710a18a18abdef18710a18a18' })

        test({
          types: ['tuple(tuple(address,address),tuple(bool,bool))', 'tuple(tuple(bool,bool),tuple(bytes,bytes),tuple(uint256,uint256))', 'address'],
          values: [[["0x77656c636f6d6520746f20657468657265756d2e", "0x77656c636f6d6520746f20657468657265756d2e"], [true, false]], [[false, true],["0xab1394581edfa2ef9ca71","0x15abe391df19aef19a4561"],["182", "1937"]], "0x77656c636f6d6520746f20657468657265756d2e"],
          expected: '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    '0000000000000000000000000000000000000000000000000000000000000000' +
                    '00000000000000000000000000000000000000000000000000000000000000c0' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '0000000000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    '00000000000000000000000000000000000000000000000000000000000000a0' +
                    '00000000000000000000000000000000000000000000000000000000000000b6' +
                    '0000000000000000000000000000000000000000000000000000000000000791' +
                    '0000000000000000000000000000000000000000000000000000000000000040' +
                    '0000000000000000000000000000000000000000000000000000000000000080' +
                    '000000000000000000000000000000000000000000000000000000000000000b' +
                    '0ab1394581edfa2ef9ca71000000000000000000000000000000000000000000' +
                    '000000000000000000000000000000000000000000000000000000000000000b' +
                    '15abe391df19aef19a4561000000000000000000000000000000000000000000' })

        test({
          types: ['tuple(tuple(uint256,bool),tuple(uint256,tuple(bytes,bytes)))', 'bytes', 'tuple(bool,tuple(bytes,address),bytes)'],
          values: [[["18320", true],["691", ["0xab1394581edfa2ef9ca71", "0x15abe391df19aef19a4561"]]], "0xab1394581edfa2ef9ca71", [false, ["0xfe", "0x77656c636f6d6520746f20657468657265756d2e"], "0xabdef18710a18a18abdef18710a18a18abdef18710a18a18abdef18710a18a18"]],
          expected: '0000000000000000000000000000000000000000000000000000000000000060' +
                    '00000000000000000000000000000000000000000000000000000000000001c0' +
                    '0000000000000000000000000000000000000000000000000000000000000200' +
                    '0000000000000000000000000000000000000000000000000000000000004790' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    '0000000000000000000000000000000000000000000000000000000000000060' +
                    '00000000000000000000000000000000000000000000000000000000000002b3' +
                    '0000000000000000000000000000000000000000000000000000000000000040' +
                    '0000000000000000000000000000000000000000000000000000000000000040' +
                    '0000000000000000000000000000000000000000000000000000000000000080' +
                    '000000000000000000000000000000000000000000000000000000000000000b' +
                    '0ab1394581edfa2ef9ca71000000000000000000000000000000000000000000' +
                    '000000000000000000000000000000000000000000000000000000000000000b' +
                    '15abe391df19aef19a4561000000000000000000000000000000000000000000' +
                    '000000000000000000000000000000000000000000000000000000000000000b' +
                    '0ab1394581edfa2ef9ca71000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000060' +
                    '00000000000000000000000000000000000000000000000000000000000000e0' +
                    '0000000000000000000000000000000000000000000000000000000000000040' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    'fe00000000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000020' +
                    'abdef18710a18a18abdef18710a18a18abdef18710a18a18abdef18710a18a18' })

        test({
          types: ['address', 'tuple(bool,tuple(address,address))', 'tuple(tuple(address,tuple(int256,int256)),bool)'],
          values: ["0x77656c636f6d6520746f20657468657265756d2e", [true,["0x77656c636f6d6520746f20657468657265756d2e", "0x77656c636f6d6520746f20657468657265756d2e"]], [["0x77656c636f6d6520746f20657468657265756d2e", ["-12451", "-12451018"]], false]],
          expected: '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffcf5d' +
                    'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffff420336' +
                    '0000000000000000000000000000000000000000000000000000000000000000' })

        test({
          types: ['bytes', 'tuple(tuple(bool,bool,bool),tuple(bytes,bytes,bytes),tuple(bytes,bool,address),tuple(uint256,int256,address))', 'tuple(bytes,tuple(tuple(int256,bool),tuple(uint256,bytes)))'],
          values: ["0xabef15", [[true, false, true], ["0xabef15", "0xcdef151", "0xabfe151"],["0x15abe391df19aef19a4561", true, "0x77656c636f6d6520746f20657468657265756d2e"],["1840181", "-819184919", "0x77656c636f6d6520746f20657468657265756d2e"]], ["0x77656c636f6d6520746f20657468657265756d2e", [["-18491",false],["1918491", "0xabdcf151dae"]]]],
          expected: '0000000000000000000000000000000000000000000000000000000000000060' +
                    '00000000000000000000000000000000000000000000000000000000000000a0' +
                    '0000000000000000000000000000000000000000000000000000000000000360' +
                    '0000000000000000000000000000000000000000000000000000000000000003' +
                    'abef150000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    '0000000000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    '0000000000000000000000000000000000000000000000000000000000000100' +
                    '0000000000000000000000000000000000000000000000000000000000000220' +
                    '00000000000000000000000000000000000000000000000000000000001c1435' +
                    'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffcf2c3ae9' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '0000000000000000000000000000000000000000000000000000000000000060' +
                    '00000000000000000000000000000000000000000000000000000000000000a0' +
                    '00000000000000000000000000000000000000000000000000000000000000e0' +
                    '0000000000000000000000000000000000000000000000000000000000000003' +
                    'abef150000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000004' +
                    '0cdef15100000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000004' +
                    '0abfe15100000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000060' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '000000000000000000000000000000000000000000000000000000000000000b' +
                    '15abe391df19aef19a4561000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000040' +
                    '0000000000000000000000000000000000000000000000000000000000000080' +
                    '0000000000000000000000000000000000000000000000000000000000000014' +
                    '77656c636f6d6520746f20657468657265756d2e000000000000000000000000' +
                    'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffb7c5' +
                    '0000000000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000060' +
                    '00000000000000000000000000000000000000000000000000000000001d461b' +
                    '0000000000000000000000000000000000000000000000000000000000000040' +
                    '0000000000000000000000000000000000000000000000000000000000000006' +
                    '0abdcf151dae0000000000000000000000000000000000000000000000000000' })

        test({
          types: ['tuple(bytes,tuple(address,address))', 'tuple(address,address,address,bytes,address)', 'tuple(address,tuple(address,tuple(address,bool)))'],
          values: [["0xabef15",["0x77656c636f6d6520746f20657468657265756d2e", "0x77656c636f6d6520746f20657468657265756d2e"]],["0x77656c636f6d6520746f20657468657265756d2e", "0x77656c636f6d6520746f20657468657265756d2e", "0x77656c636f6d6520746f20657468657265756d2e", "0xabef15", "0x77656c636f6d6520746f20657468657265756d2e"],["0x81017589ab81017589ab81017589ab81017589ab",["0x77656c636f6d6520746f20657468657265756d2e",["0x81017589ab81017589ab81017589ab81017589ab", false]]]],
          expected: '00000000000000000000000000000000000000000000000000000000000000c0' +
                    '0000000000000000000000000000000000000000000000000000000000000160' +
                    '00000000000000000000000081017589ab81017589ab81017589ab81017589ab' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '00000000000000000000000081017589ab81017589ab81017589ab81017589ab' +
                    '0000000000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000060' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '0000000000000000000000000000000000000000000000000000000000000003' +
                    'abef150000000000000000000000000000000000000000000000000000000000' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '00000000000000000000000000000000000000000000000000000000000000a0' +
                    '00000000000000000000000077656c636f6d6520746f20657468657265756d2e' +
                    '0000000000000000000000000000000000000000000000000000000000000003' +
                    'abef150000000000000000000000000000000000000000000000000000000000' })

        test({
          types: ['tuple(bytes,bytes)', 'bytes', 'tuple(tuple(bytes),tuple(bytes,bytes),tuple(bytes,bytes,bytes))'],
          values: [["0xabef15", "0xa"], "0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff", [["0xaf"],["0xaf", "0xbc"],["0xaf", "0xbc", "0xde"]]],
          expected: '0000000000000000000000000000000000000000000000000000000000000060' +
                    '0000000000000000000000000000000000000000000000000000000000000120' +
                    '0000000000000000000000000000000000000000000000000000000000000160' +
                    '0000000000000000000000000000000000000000000000000000000000000040' +
                    '0000000000000000000000000000000000000000000000000000000000000080' +
                    '0000000000000000000000000000000000000000000000000000000000000003' +
                    'abef150000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    '0a00000000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000020' +
                    'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff' +
                    '0000000000000000000000000000000000000000000000000000000000000060' +
                    '00000000000000000000000000000000000000000000000000000000000000c0' +
                    '0000000000000000000000000000000000000000000000000000000000000180' +
                    '0000000000000000000000000000000000000000000000000000000000000020' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    'af00000000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000040' +
                    '0000000000000000000000000000000000000000000000000000000000000080' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    'af00000000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    'bc00000000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000060' +
                    '00000000000000000000000000000000000000000000000000000000000000a0' +
                    '00000000000000000000000000000000000000000000000000000000000000e0' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    'af00000000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    'bc00000000000000000000000000000000000000000000000000000000000000' +
                    '0000000000000000000000000000000000000000000000000000000000000001' +
                    'de00000000000000000000000000000000000000000000000000000000000000' })
        test({
            types: ['uint256', 'uint256'],
            values: [new BN(42), new BN(42)],
            expected: '000000000000000000000000000000000000000000000000000000000000002a000000000000000000000000000000000000000000000000000000000000002a'
        });
        test({
            types: ['uint256', 'uint256'],
            values: [new BigNumber(42), new BigNumber(42)],
            expected: '000000000000000000000000000000000000000000000000000000000000002a000000000000000000000000000000000000000000000000000000000000002a'
        });
        test({
            types: ['bytes', 'bytes32'],
            values: [Buffer.from('cool'), Buffer.from('beans')],
            expected: '00000000000000000000000000000000000000000000000000000000000000406265616e730000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004636f6f6c00000000000000000000000000000000000000000000000000000000'
        });
        test({
            types: ['bytes6', 'bytes12'],
            values: [Buffer.from('super'), Buffer.from('man')],
            expected: '73757065720000000000000000000000000000000000000000000000000000006d616e0000000000000000000000000000000000000000000000000000000000'
        });
        test({
            types: ['bytes12[]'],
            values: [[Buffer.from('buff'), Buffer.from('man')]],
            expected: '0000000000000000000000000000000000000000000000000000000000000020000000000000000000000000000000000000000000000000000000000000000262756666000000000000000000000000000000000000000000000000000000006d616e0000000000000000000000000000000000000000000000000000000000'
        });
        test({
            types: ['uint16', 'uint24'],
            values: ['10000', '100000'],
            expected: '000000000000000000000000000000000000000000000000000000000000271000000000000000000000000000000000000000000000000000000000000186a0'
        });
        test({
            types: ['tuple(string,uint)'],
            values: [['item1', 2]],
            expected: '00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000040000000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000056974656d31000000000000000000000000000000000000000000000000000000'
        });
        test({
            types: ['tuple(string,uint)[]'],
            values: [[['item1', 2], ['item2', 3]]],
            expected: '00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000002000000000000000000000000000000000000000000000000000000000000004000000000000000000000000000000000000000000000000000000000000000c00000000000000000000000000000000000000000000000000000000000000040000000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000056974656d310000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000040000000000000000000000000000000000000000000000000000000000000000300000000000000000000000000000000000000000000000000000000000000056974656d32000000000000000000000000000000000000000000000000000000'
        });
        test({
            types: ['tuple(string,uint16)[]'],
            values: [[['item1', 2], ['item2', 3]]],
            expected: '00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000002000000000000000000000000000000000000000000000000000000000000004000000000000000000000000000000000000000000000000000000000000000c00000000000000000000000000000000000000000000000000000000000000040000000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000056974656d310000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000040000000000000000000000000000000000000000000000000000000000000000300000000000000000000000000000000000000000000000000000000000000056974656d32000000000000000000000000000000000000000000000000000000'
        });
    });
});


