# Globalize 1.3.0 announcement

On July 3rd, we released Globalize 1.3.0. It is a special release, because it includes some very useful feature enhancements to support advanced date, time, timezone manipulation, and other long due fixes. We wanted to share more details on these improvements.

## IANA/Olson Time Zone Support

> This change was contributed by Kandaswamy Manikandan @rajavelmani (PayPal) and Rafael Xavier @rxaviers in #687 and #701.

In previous versions, Globalize had some partial time zone support for a user's runtime time zone. However specific CLDR patterns (`z`, `v`, and `V`) that display strings such as `PDT`,  `Pacific Daylight Time`, `Pacific Time`, and `Los Angeles Time` could not be displayed. The challenge [we had](https://github.com/globalizejs/globalize/pull/202) to determine how costly a solution would be to provide full IANA/Olson time zone support due to the additional manipulation code and data (i.e., IANA database). Therefore, in the past, we encouraged users that needed to manipulate date in arbitrary time zones to use a separate library, like *moment-timezone*. Nevertheless, this solution never closed the gap between internationalization (i18n) implementations leveraging CLDR and having full maneuverability of time zones.

With the latest release 1.3.0, Globalize fully supports time zone. Simply put, by using Globalize 1.3.0, you now have full IANA support with the strength of CLDR for i18n!

```js
Globalize.locale("en");
let date = new Date();

Globalize.formatDate(date, {datetime: "short", timeZone: "America/Los_Angeles"});
// > '3/19/17, 3:19 PM'
Globalize.formatDate(date, {datetime: "short", timeZone: "America/New_York"});
// > '3/19/17, 6:19 PM'
Globalize.formatDate(date, {datetime: "short", timeZone: "America/Sao_Paulo"});
// > '3/19/17, 7:19 PM'
Globalize.formatDate(date, {datetime: "short", timeZone: "Europe/Berlin"});
// > '3/19/17, 11:19 PM'

Globalize.formatDate(date, {datetime: "full", timeZone: "America/Los_Angeles"});
// > 'Sunday, March 19, 2017 at 3:19:22 PM Pacific Daylight Time'
Globalize.formatDate(date, {datetime: "full", timeZone: "America/New_York"});
// > 'Sunday, March 19, 2017 at 6:19:22 PM Eastern Daylight Time'
Globalize.formatDate(date, {datetime: "full", timeZone: "America/Sao_Paulo"});
// > 'Sunday, March 19, 2017 at 7:19:22 PM Brasilia Standard Time'
Globalize.formatDate(date, {datetime: "full", timeZone: "Europe/Berlin"});
// > 'Sunday, March 19, 2017 at 11:19:22 PM Central European Standard Time'

Globalize("pt").formatDate(date, {datetime: "full", timeZone: "America/Sao_Paulo"});
// > 'domingo, 19 de março de 2017 19:19:22 Horário Padrão de Brasília'
Globalize("de").formatDate(date, {datetime: "full", timeZone: "Europe/Berlin"});
// > 'Sonntag, 19. März 2017 um 23:19:22 Mitteleuropäische Normalzeit'
Globalize("zh").formatDate(date, {datetime: "full", timeZone: "Asia/Shanghai"});
// > '2017年3月20日星期一 中国标准时间 上午6:19:22'
Globalize("ar").formatDate(date, {datetime: "full", timeZone: "Africa/Cairo"});
// > 'الاثنين، ٢٠ مارس، ٢٠١٧ ١٢:١٩:٢٢ ص توقيت شرق أوروبا الرسمي'
```

We have solved this in a low footprint, high performance implementation using [zoned-date-time](https://github.com/rxaviers/zoned-date-time) under the hoods, which is a 0.6KB library for the time zone manipulations. We have leveraged the Globalize Compiler for precompling the IANA data base for production. For example, let's say you are serving content in English (e.g. locale en-US) for America/Los_Angeles time using the following formatter:

```js
var dateWithTimeZoneFormatter = Globalize.dateFormatter({
  datetime: "full",
  timeZone: "America/Los_Angeles"
});
```

The final size (for production) of this code will be:

| filename                                 | minified+gzipped size |
| ---------------------------------------- | --------------------- |
| i18n/en.js (includes CLDR and IANA data) | 1.7KB                 |
| core, number, and date globalize runtime lib + zoned-date-time | 7.0KB                 |

See globalize [compiler example](https://github.com/globalizejs/globalize/tree/master/examples/globalize-compiler) or [app-npm-webpack example](https://github.com/globalizejs/globalize/tree/master/examples/app-npm-webpack) for details.

## Format Date To Parts

> This change was contributed by Reza Payami @rpayami (PayPal) and Rafael Xavier @rxaviers in #697 and #700.

Modern user interfaces often need to manipulate the date format output, which is impossible via the existing format function that returns an opaque string. Making any attempt to do this can break internationalization support. [Ecma-402](https://github.com/tc39/ecma402/) has recently added [`Intl.DateTimeFormat.prototype.formatToParts`](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DateTimeFormat/formatToParts) to fulfill that purpose, which at the time of this post, is at stage 4 and is implemented by latest Firefox and Chrome.

In Globalize, we introduced [`.dateToPartsFormatter`](https://github.com/globalizejs/globalize/blob/master/doc/api/date/date-to-parts-formatter.md) and [`.formatDateToParts`](https://github.com/globalizejs/globalize/blob/master/doc/api/date/date-to-parts-formatter.md).

```js
Globalize.locale( "en" );
Globalize.formatDateToParts(new Date(2010, 10, 30));
// > [
//   { "type": "month", "value": "11" },
//   { "type": "literal", "value": "/" },
//   { "type": "day", "value": "30" },
//   { "type": "literal", "value": "/" },
//   { "type": "year", "value": "2010" }
// ]
```

The data is available separately and it can be formatted and concatenated again in a customized way. For example by using [`Array.prototype.map()`](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/map), [arrow functions](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Functions/Arrow_functions), a [switch statement](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/switch), [template literals](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Template_literals), and [`Array.prototype.reduce()`](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/reduce).

```js
let formatter;

Globalize.locale( "en" );
formatter = Globalize.dateToPartsFormatter({datetime: "short"});

formatter( new Date( 2010, 10, 30, 17, 55 ) ).map(({type, value}) => {
  switch ( type ) {
    case "year": return `<strong>${value}</strong>`;
    default: return value;
  }
}).join( "" );
// > "11/30/<strong>10</strong>, 5:55 PM"
```

See [React Date Input](https://github.com/rxaviers/react-date-input) as a demo of a UI component for React optimized for i18n and a11y.

| Localized and smart date input           | Feb 28 in `en`, `es`, `pt`, `de`, `zh`, `ko`, and `ar` |
| ---------------------------------------- | ---------------------------------------- |
| ![en](https://media.giphy.com/media/xUA7aZAUNINGP2jI4M/giphy.gif) | ![en-es-pt-de-zh-ko-ar](https://media.giphy.com/media/3og0ILQu0KxLRewJnW/giphy.gif) |

## Dynamically Augmented Date Skeletons

> This change was contributed by Marat Dyatko @vectart and Artur Eshenbrener @Strate in #462 and #604.

The style used to display a date format often varies depending on the application. CLDR offers data for certain presets like short (e.g., short date `"7/1/17"`), medium (e.g., medium date `"Jul 1, 2017"`), long (e.g., long date `"July 1, 2017"`), and full (e.g., full date `"Saturday, July 1, 2017"`). Although, we may want something different such as `"Jul 1"`. For that CLDR offers data for individual [date fields](http://www.unicode.org/reports/tr35/tr35-dates.html#Date_Field_Symbol_Table) and their combinations, which are used by Globalize to synthesize an open-ended list of custom formats (called skeletons). But, what's interesting is that it would be prohibitively large if CLDR provided data for every single possible combination. So, there's an algorithm specified by [UTS#35](http://www.unicode.org/reports/tr35/tr35-dates.html#availableFormats_appendItems) to deduce missing data from the requested format.

For the `"Jul 1"` example, we should use `{skeleton: "MMMd"}`. Internally, Globalize finds a direct match in CLDR for the requested skeleton. This works fine in previous Globalize versions.

For that next example, let's assume we want `"July 1"`, i.e., `{skeleton: "MMMMd"}`. Internally, Globalize doesn't find a direct match in CLDR. For this skeleton, Globalize needs to use the data for `MMMd`, which maps to `"MMM d"` in the English case, and then it needs to replace `MMM` with `MMMM` dynamically generating `"MMMM d"`. This doesn't work in previous versions of Globalize, but it works now on latest v1.3.0.

If we wanted `"07/01"` instead, we should use `{skeleton: "MMdd"}`. Internally, Globalize doesn't find a direct match in CLDR for this skeleton and, therefore, it fais in globalize v1.2.3. Globalize needs to use the data for `Md`, which in the case of English maps to `"M/d"`, and then replace `M` wtih `MM` and `d` with `dd` dynamically generating `"MM/dd"`.

To make a long story short, the algorithm in globalize v1.3.0 has been significantly improved and it allows using virtually any skeletons.

```js
// A skeleton not directly found in CLDR and that needs to be deduced by globalize.
// In English, globalize needs to use the data for GyMMMEd, and adjust MMM with MMMM,
// and E with EEEE. Then, it needs to find the data for hms and glue them together
// using the appropriate format.
// On globalize v1.2.3, an error is thrown saying this skeleton wasn't found.
let skeleton = "GyMMMMEEEEdhms";
Globalize("en").formatDate(new Date(), {skeleton});
// > 'Saturday, July 1, 2017 AD at 4:58:27 PM'
Globalize("pt").formatDate(new Date(), {skeleton});
// > 'sábado, 1 de julho de 2017 d.C. 5:01:20 PM'
Globalize("de").formatDate(new Date(), {skeleton});
// > 'Samstag, 1. Juli 2017 n. Chr. um 5:01:33 nachm.'
Globalize("zh").formatDate(new Date(), {skeleton});
// > '公元2017年七月月1日星期六 下午5:01:35'
Globalize("ko").formatDate(new Date(), {skeleton});
// > 'AD 2017년 7월 1일 토요일 오후 5:01:38'
Globalize("ar").formatDate(new Date(), {skeleton});
// > 'السبت، ١ يوليو، ٢٠١٧ م ٥:٠١:٤٠ م'
Globalize("ar-MA").formatDate(new Date(), {skeleton});
// > 'السبت، 1 يوليوز، 2017 م 5:04:29 م'
Globalize("it").formatDate(new Date(), {skeleton});
// > 'sabato 1 luglio 2017 d.C. 5:01:52 PM'
```

Read our [getting started](https://github.com/globalizejs/globalize/#getting-started) and play with it yourself.

## Other Enhancements and Bug Fixes

🎉 Enhancements

- Date: Show timezone offset optional minutes for O pattern (e.g., GMT-6:30 note the :30) [#339](https://github.com/globalizejs/globalize/pull/339) (via PR [#729](https://github.com/globalizejs/globalize/pull/729)) (Rafael Xavier)
- Date: Show timezone offset optional minutes and seconds for x and X patterns (e.g., -06:30 note the :30) [#339](https://github.com/globalizejs/globalize/pull/339) (via PR [#729](https://github.com/globalizejs/globalize/pull/729)) (Rafael Xavier)
- Date: Assert options.skeleton (PR [#726](https://github.com/globalizejs/globalize/pull/726)) (Rafael Xavier)
- Date parser: Make runtime phase lighter [#735](https://github.com/globalizejs/globalize/pull/735) (Rafael Xavier)
- Date parser: Loose Matching PR [#730](https://github.com/globalizejs/globalize/pull/730) (Rafael Xavier)
  - Allows, among others, parsing arabic dates as user types them (i.e., without control characters)
- Number formatter: Amend integer and fraction formatter for small numbers like 1e-7 [#750](https://github.com/globalizejs/globalize/pull/750) (Rafael Xavier)
- Number parser: Lenient about trailing decimal separator [#744](https://github.com/globalizejs/globalize/pull/744) (Rafael Xavier)
- Runtime: Use strict [#676](https://github.com/globalizejs/globalize/pull/676) (Zack Birkenbuel)

🐛 Fixes

- Date parser: invalid output by mixing numbering systems [#696](https://github.com/globalizejs/globalize/pull/696) (via PR [#733](https://github.com/globalizejs/globalize/pull/733)) (Rafael Xavier)
- Date parser: fails on Turkish full datetime with Monday or Saturday [#690](https://github.com/globalizejs/globalize/pull/690) (via PR [#732](https://github.com/globalizejs/globalize/pull/732)) (Rafael Xavier)

⚙️ Others

- Compiler tests! [#721](https://github.com/globalizejs/globalize/pull/721) (via PR [#727](https://github.com/globalizejs/globalize/pull/727)) (Nikola Kovacs)
- Documentation style refactor [#737](https://github.com/globalizejs/globalize/pull/737) (Rafael Xavier)

## Last but not least

Special thanks to other PayPal internationalization team members including Daniel Bruhn, Lucas Welti, Alolita Sharma, Mike McKenna for testing and helping integrate Globalize for PayPal products.

Special thanks to James Bellenger and Nicolas Gallagher for the React and Webpack integration enhancements for Twitter products, which certainly deserves its own blog post.

Many thanks to all of you who participated in this release by testing, reporting bugs, or submitting patches, including Jörn Zaefferer, Frédéric Miserey, Nova Patch, and whole Globalize team.
