/*
 * Copyright (C) 2017-2019 HERE Europe B.V.
 * Licensed under Apache 2.0, see full license in LICENSE
 * SPDX-License-Identifier: Apache-2.0
 */

import { LineCaps } from "@here/harp-datasource-protocol";
import * as THREE from "three";
import {
    DisplacementFeature,
    DisplacementFeatureParameters,
    FadingFeature,
    FadingFeatureParameters
} from "./MapMeshMaterials";
import linesShaderChunk, { LineCapsModes } from "./ShaderChunks/LinesChunks";
import {
    enforceBlending,
    getShaderMaterialDefine,
    setShaderDefine,
    setShaderMaterialDefine
} from "./Utils";

const LineCapsDefinesMapping: { [key in LineCaps]: number } = {
    None: LineCapsModes.CAPS_NONE,
    Square: LineCapsModes.CAPS_SQUARE,
    Round: LineCapsModes.CAPS_ROUND,
    TriangleIn: LineCapsModes.CAPS_TRIANGLE_IN,
    TriangleOut: LineCapsModes.CAPS_TRIANGLE_OUT
};

const DefinesLineCapsMapping: { [key: number]: LineCaps } = Object.keys(
    LineCapsDefinesMapping
).reduce((r, lineCapsName) => {
    const defineKey = lineCapsName as keyof typeof LineCapsDefinesMapping;
    const defineValue: number = LineCapsDefinesMapping[defineKey];
    r[defineValue] = defineKey;
    return r;
}, ({} as any) as { [key: number]: LineCaps });

/**
 * The vLength contains the actual line length, it's needed for the creation of line caps by
 * detecting line ends. `vLength == vExtrusionCoord.x + lineWidth * 2`
 */
/**
 * The vExtrusionStrength relies on the edges of the lines. Represents how far the current point was
 * extruded on the edges because of the current angle. Needed for preventing line caps artifacts on
 * sharp line edges. For example, on sharp edges, some vertices can be extruded much further than
 * the full line length.
 */

const tmpColor = new THREE.Color();
const vertexSource: string = `
#define SEGMENT_OFFSET 0.1

attribute vec3 extrusionCoord;
attribute vec3 position;
attribute vec4 bitangent;
attribute vec3 tangent;
attribute vec2 uv;
attribute vec3 normal;

uniform mat4 modelViewMatrix;
uniform mat4 projectionMatrix;
uniform float lineWidth;
uniform float outlineWidth;
uniform vec2 drawRange;

#ifdef USE_DISPLACEMENTMAP
uniform sampler2D displacementMap;
#endif

varying vec3 vPosition;
varying vec3 vRange;
varying vec4 vCoords;
#ifdef USE_COLOR
attribute vec3 color;
varying vec3 vColor;
#endif

#ifdef USE_FADING
#include <fading_pars_vertex>
#endif

#include <fog_pars_vertex>

#include <extrude_line_vert_func>

void main() {
    // Calculate the segment.
    vec2 segment = abs(extrusionCoord.xy) - SEGMENT_OFFSET;
    float segmentPos = sign(extrusionCoord.x) / 2.0 + 0.5;

    // Calculate the vertex position inside the line (segment) and extrusion direction and factor.
    float linePos = mix(segment.x, segment.y, segmentPos);
    vec2 extrusionDir = sign(extrusionCoord.xy);
    float extrusionFactor = extrusionDir.y * tan(bitangent.w / 2.0);

    // Calculate the extruded vertex position (and scale the extrusion direction).
    vec3 pos = extrudeLine(
        position, linePos, lineWidth + outlineWidth, bitangent, tangent, extrusionDir);

    // Store the normalized extrusion coordinates in vCoords (with their ranges in vRange).
    vRange = vec3(extrusionCoord.z, lineWidth, extrusionFactor);
    vCoords = vec4(extrusionDir / vRange.xy, segment / vRange.x);

    // Adjust the segment to fit the drawRange.
    float capDist = (lineWidth + outlineWidth) / extrusionCoord.z;
    if ((vCoords.w + capDist) < drawRange.x || (vCoords.z - capDist) > drawRange.y) {
        vCoords.zw += 1.0;
    }
    if (vCoords.z < drawRange.x) {
        vCoords.zw += vec2(drawRange.x - vCoords.z, 0.0);
    }
    if (vCoords.w > drawRange.y) {
        vCoords.zw -= vec2(0.0, vCoords.w - drawRange.y);
    }

    // Transform position.
    #ifdef USE_DISPLACEMENTMAP
    pos += normalize( normal ) * texture2D( displacementMap, uv ).x;
    #endif
    vec4 mvPosition = modelViewMatrix * vec4(pos, 1.0);
    gl_Position = projectionMatrix * mvPosition;

    // Pass extruded position to fragment shader.
    vPosition = pos;

    #ifdef USE_COLOR
    // Pass vertex color to fragment shader.
    vColor = color;
    #endif

    #ifdef USE_FADING
    #include <fading_vertex>
    #endif

    #include <fog_vertex>
}`;

const fragmentSource: string = `
precision highp float;
precision highp int;

uniform vec3 diffuse;
uniform vec3 outlineColor;
uniform float opacity;
uniform float lineWidth;
uniform float outlineWidth;
uniform vec2 tileSize;
uniform vec2 drawRange;

#ifdef USE_DASHED_LINE
uniform float dashSize;
uniform float gapSize;
uniform vec3 dashColor;
#endif

varying vec3 vPosition;
varying vec3 vRange;
varying vec4 vCoords;
#ifdef USE_COLOR
varying vec3 vColor;
#endif

#include <round_edges_and_add_caps>
#include <tile_clip_func>

#ifdef USE_FADING
#include <fading_pars_fragment>
#endif

#include <fog_pars_fragment>

void main() {
    float alpha = opacity;
    vec3 outputDiffuse = diffuse;

    #ifdef USE_TILE_CLIP
    tileClip(vPosition.xy, tileSize);
    #endif

    // Calculate distance to center (0.0: lineCenter, 1.0: lineEdge).
    float distToCenter = roundEdgesAndAddCaps(vCoords, vRange);
    // Calculate distance to edge (-1.0: lineCenter, 0.0: lineEdge).
    float distToEdge = distToCenter - (lineWidth + outlineWidth) / lineWidth;

    // Decrease the line opacity by the distToEdge, making the transition steeper when the slope
    // of distToChange increases (i.e. the line is further away).
    float width = fwidth(distToEdge);
    alpha *= (1.0 - smoothstep(-width, width, distToEdge));

    #ifdef USE_DASHED_LINE
    // Compute the distance to the dash origin (0.0: dashOrigin, 1.0: dashEnd, (d+g)/d: gapEnd).
    float d = dashSize / vRange.x;
    float g = gapSize / vRange.x;
    float distToDashOrigin = mod(vCoords.x, d + g) / d;

    // Compute distance to dash edge (0.5: dashCenter, 0.0: dashEdge) and compute the
    // dashBlendFactor similarly on how we did it for the line opacity.
    float distToDashEdge = 0.5 - distance(distToDashOrigin, (d + g) / d * 0.5);
    float dashWidth = fwidth(distToDashEdge);
    float dashBlendFactor = 1.0 - smoothstep(-dashWidth, dashWidth, distToDashEdge);

    #ifdef USE_DASH_COLOR
    outputDiffuse = mix(diffuse, dashColor, dashBlendFactor);
    #endif
    #endif

    #ifdef USE_OUTLINE
    // Calculate distance to outline (0.0: lineEdge, outlineWidth/lineWidth: outlineEdge) and
    // compute the outlineBlendFactor (used to mix line and outline colors).
    float distToOutline = distToCenter - 1.0;
    float outlineWidth = fwidth(distToOutline);
    float outlineBlendFactor = smoothstep(-outlineWidth, outlineWidth, distToOutline);

    // Mix the colors using the different computed factors.
    #if defined(USE_DASHED_LINE) && !defined(USE_DASH_COLOR)
    float colorBlendFactor = smoothstep(-1.0, 1.0, dashBlendFactor - outlineBlendFactor);
    outputDiffuse = mix(
      mix(
        mix(outlineColor, diffuse, colorBlendFactor),
        outputDiffuse,
        dashBlendFactor
      ),
      outlineColor,
      outlineBlendFactor
    );
    #else
    outputDiffuse = mix(outputDiffuse, outlineColor, outlineBlendFactor);
    #endif
    #endif

    #if defined(USE_DASHED_LINE) && !defined(USE_DASH_COLOR)
    // Multiply the alpha by the dashBlendFactor.
    #if defined(USE_OUTLINE)
    alpha *= clamp(dashBlendFactor + outlineBlendFactor, 0.0, 1.0);
    #else
    alpha *= 1.0 - dashBlendFactor;
    #endif
    #endif

    #ifdef USE_COLOR
    gl_FragColor = vec4( outputDiffuse * vColor, alpha );
    #else
    gl_FragColor = vec4( outputDiffuse, alpha );
    #endif

    #include <fog_fragment>

    #ifdef USE_FADING
    #include <fading_fragment>
    #endif
}`;

/**
 * Parameters used when constructing a new [[SolidLineMaterial]].
 */
export interface SolidLineMaterialParameters
    extends FadingFeatureParameters,
        DisplacementFeatureParameters {
    /**
     * Line color.
     */
    color?: number | string;

    /**
     * Line outline color.
     */
    outlineColor?: number | string;

    /**
     * Enables/Disable depth test.
     */
    depthTest?: boolean;

    /**
     * Enables/Disable depth write.
     */
    depthWrite?: boolean;

    /**
     * `SolidLineMaterial` extends the ThreeJS `RawShaderMaterial` that does not update fog at
     * runtime, so instead of recompiling everything we pass it in the constructor.
     */
    fog?: boolean;

    /**
     * Line width.
     */
    lineWidth?: number;

    /**
     * Outline width.
     */
    outlineWidth?: number;

    /**
     * Line opacity.
     */
    opacity?: number;

    /**
     * Describes line caps type (`"None"`, `"Round"`, `"Square"`, `"TriangleOut"`, `"TriangleIn"`).
     * Default is `"Round"`.
     */
    caps?: LineCaps;

    /**
     * Describes the starting drawing position for the line (in the range [0...1]).
     * Default is `0.0`.
     */
    drawRangeStart?: number;

    /**
     * Describes the ending drawing position for the line (in the range [0...1]).
     * Default is `1.0`.
     */
    drawRangeEnd?: number;

    /**
     * Line dashes color.
     */
    dashColor?: number | string;

    /**
     * Size of the dashed segments.
     */
    dashSize?: number;

    /**
     * Size of the gaps between dashed segments.
     */
    gapSize?: number;
}

/**
 * Material designed to render solid variable-width lines.
 */
export class SolidLineMaterial extends THREE.RawShaderMaterial
    implements DisplacementFeature, FadingFeature {
    static DEFAULT_COLOR: number = 0xff0000;
    static DEFAULT_WIDTH: number = 1.0;
    static DEFAULT_OUTLINE_WIDTH: number = 0.0;
    static DEFAULT_OPACITY: number = 1.0;
    static DEFAULT_DRAW_RANGE_START: number = 0.0;
    static DEFAULT_DRAW_RANGE_END: number = 1.0;
    static DEFAULT_DASH_SIZE: number = 1.0;
    static DEFAULT_GAP_SIZE: number = 1.0;

    /**
     * @hidden
     * Material properties overrides.
     */
    private m_fog: boolean;
    private m_opacity: number;

    /**
     * Constructs a new `SolidLineMaterial`.
     *
     * @param params `SolidLineMaterial` parameters.
     */
    constructor(params?: SolidLineMaterialParameters) {
        Object.assign(THREE.ShaderChunk, linesShaderChunk);

        FadingFeature.patchGlobalShaderChunks();

        // Setup default defines.
        const defines: { [key: string]: any } = {
            CAPS_MODE: LineCapsModes.CAPS_ROUND
        };

        // Prepare defines based on params passed in, before super class c-tor, this ensures
        // proper set for shader compilation, without need to re-compile.
        let fogParam = true;
        let opacityParam = 1.0;
        let displacementMap;
        if (params !== undefined) {
            fogParam = params.fog === true;
            if (fogParam) {
                setShaderDefine(defines, "USE_FOG", true);
            }
            opacityParam = params.opacity !== undefined ? params.opacity : opacityParam;
            displacementMap = params.displacementMap;
            if (displacementMap !== undefined) {
                setShaderDefine(defines, "USE_DISPLACEMENTMAP", true);
            }
            const hasOutline = params.outlineWidth !== undefined && params.outlineWidth > 0;
            if (hasOutline) {
                setShaderDefine(defines, "USE_OUTLINE", true);
            }
        }

        const shaderParams: THREE.ShaderMaterialParameters = {
            name: "SolidLineMaterial",
            vertexShader: vertexSource,
            fragmentShader: fragmentSource,
            uniforms: THREE.UniformsUtils.merge([
                {
                    diffuse: new THREE.Uniform(new THREE.Color(SolidLineMaterial.DEFAULT_COLOR)),
                    dashColor: new THREE.Uniform(new THREE.Color(SolidLineMaterial.DEFAULT_COLOR)),
                    outlineColor: new THREE.Uniform(
                        new THREE.Color(SolidLineMaterial.DEFAULT_COLOR)
                    ),
                    lineWidth: new THREE.Uniform(SolidLineMaterial.DEFAULT_WIDTH),
                    outlineWidth: new THREE.Uniform(SolidLineMaterial.DEFAULT_OUTLINE_WIDTH),
                    opacity: new THREE.Uniform(SolidLineMaterial.DEFAULT_OPACITY),
                    tileSize: new THREE.Uniform(new THREE.Vector2()),
                    fadeNear: new THREE.Uniform(FadingFeature.DEFAULT_FADE_NEAR),
                    fadeFar: new THREE.Uniform(FadingFeature.DEFAULT_FADE_FAR),
                    displacementMap: new THREE.Uniform(
                        displacementMap !== undefined ? displacementMap : new THREE.Texture()
                    ),
                    drawRange: new THREE.Uniform(
                        new THREE.Vector2(
                            SolidLineMaterial.DEFAULT_DRAW_RANGE_START,
                            SolidLineMaterial.DEFAULT_DRAW_RANGE_END
                        )
                    ),
                    dashSize: new THREE.Uniform(SolidLineMaterial.DEFAULT_DASH_SIZE),
                    gapSize: new THREE.Uniform(SolidLineMaterial.DEFAULT_GAP_SIZE)
                },
                // We need the fog uniforms available when we use `fog` setter as the internal
                // recompilation cannot add or remove uniforms.
                THREE.UniformsLib.fog
            ]),
            defines,
            // No need to pass overridden `fog` and `opacity` params they will be set
            // after super c-tor call.
            fog: fogParam,
            opacity: opacityParam
        };
        super(shaderParams);
        // Required to satisfy compiler error if fields has no initializer or are not definitely
        // assigned in the constructor, this also mimics ShaderMaterial set of defaults
        // for overridden props.
        this.m_fog = fogParam;
        this.m_opacity = opacityParam;

        enforceBlending(this);
        this.extensions.derivatives = true;

        // Apply initial parameter values.
        if (params !== undefined) {
            if (params.color !== undefined) {
                tmpColor.set(params.color as any);
                this.color = tmpColor;
            }
            if (params.outlineColor !== undefined) {
                tmpColor.set(params.outlineColor as any);
                this.outlineColor = tmpColor;
            }
            if (params.lineWidth !== undefined) {
                this.lineWidth = params.lineWidth;
            }
            if (params.outlineWidth !== undefined) {
                this.outlineWidth = params.outlineWidth;
            }
            if (params.opacity !== undefined) {
                this.opacity = params.opacity;
            }
            if (params.depthTest !== undefined) {
                this.depthTest = params.depthTest;
            }
            if (params.depthWrite !== undefined) {
                this.depthWrite = params.depthWrite;
            }
            if (params.fadeNear !== undefined) {
                this.fadeNear = params.fadeNear;
            }
            if (params.fadeFar !== undefined) {
                this.fadeFar = params.fadeFar;
            }
            if (params.displacementMap !== undefined) {
                this.displacementMap = params.displacementMap;
            }
            if (params.caps !== undefined) {
                this.caps = params.caps;
            }
            if (params.drawRangeStart !== undefined) {
                this.drawRangeStart = params.drawRangeStart;
            }
            if (params.drawRangeEnd !== undefined) {
                this.drawRangeEnd = params.drawRangeEnd;
            }
            if (params.dashColor !== undefined) {
                tmpColor.set(params.dashColor as any);
                this.dashColor = tmpColor;
            }
            if (params.dashSize !== undefined) {
                this.dashSize = params.dashSize;
            }
            if (params.gapSize !== undefined) {
                this.gapSize = params.gapSize;
            }
            if (params.fog !== undefined) {
                this.fog = params.fog;
            }
        }
        // ShaderMaterial overrides requires invalidation cause super c-tor may set this
        // properties before related `defines` and `uniforms` were created.
        this.invalidateFog();
        this.invalidateOpacity();
    }

    /**
     * Overrides THREE.Material.fog flag to add support for custom shader.
     *
     * @param enable Whether we want to enable the fog.
     */
    set fog(enable: boolean) {
        this.m_fog = enable;
        // Function may be called from THREE.js cause we override setter,
        // in this case defines are not yet initalized and require late invalidation in
        // SolidLineMaterial c-tor.
        if (this.defines !== undefined) {
            setShaderMaterialDefine(this, "USE_FOG", enable);
        }
    }

    /**
     * Checks if fog is enabled.
     */
    get fog(): boolean {
        return this.m_fog && getShaderMaterialDefine(this, "USE_FOG") === true;
    }

    /**
     * The method to call to recompile a material to enable/disable outline effect
     *
     * @param enable Whether we want to use outline.
     */
    set outline(enable: boolean) {
        setShaderMaterialDefine(this, "USE_OUTLINE", enable);
    }

    /**
     * Checks if outline is enabled.
     */
    get outline(): boolean {
        return getShaderMaterialDefine(this, "USE_OUTLINE") === true;
    }

    /**
     * Line opacity.
     */
    get opacity(): number {
        return this.m_opacity;
    }
    set opacity(value: number) {
        this.m_opacity = value;
        // Setting opacity before uniform being created requires late invalidation,
        // call to invalidateOpacity() is done at the end of c-tor.
        if (this.uniforms !== undefined) {
            this.uniforms.opacity.value = value;
        }
    }

    /**
     * Line color.
     */
    get color(): THREE.Color {
        return this.uniforms.diffuse.value as THREE.Color;
    }
    set color(value: THREE.Color) {
        this.uniforms.diffuse.value.copy(value);
    }

    /**
     * Outline color.
     *
     * @note The width of outline ([[outlineWidth]]) need to be also set to enable outlining.
     */
    get outlineColor(): THREE.Color {
        return this.uniforms.outlineColor.value as THREE.Color;
    }
    set outlineColor(value: THREE.Color) {
        this.uniforms.outlineColor.value.copy(value);
    }

    /**
     * Dash color.
     *
     * @note The property [[gapSize]] need to be set to enable dashed line.
     */
    get dashColor(): THREE.Color {
        return this.uniforms.dashColor.value as THREE.Color;
    }
    set dashColor(value: THREE.Color) {
        this.uniforms.dashColor.value.copy(value);
        setShaderMaterialDefine(this, "USE_DASH_COLOR", true);
    }

    /**
     * Line width.
     */
    get lineWidth(): number {
        return this.uniforms.lineWidth.value as number;
    }
    set lineWidth(value: number) {
        this.uniforms.lineWidth.value = value;
    }

    /**
     * Outline width.
     */
    get outlineWidth(): number {
        return this.uniforms.outlineWidth.value as number;
    }
    set outlineWidth(value: number) {
        this.uniforms.outlineWidth.value = value;
        this.outline = value > 0.0;
    }

    /**
     * Size of the dashed segments.
     *
     * @note Ths [[gapSize]] need to be also set to enable dashed line.
     * @see gapSize.
     */
    get dashSize(): number {
        return this.uniforms.dashSize.value as number;
    }
    set dashSize(value: number) {
        this.uniforms.dashSize.value = value;
    }

    /**
     * Size of the gaps between dashed segments.
     *
     * @note You may also need to set [[dashSize]].
     * @see dashSize.
     */
    get gapSize(): number {
        return this.uniforms.gapSize.value as number;
    }
    set gapSize(value: number) {
        this.uniforms.gapSize.value = value;
        setShaderMaterialDefine(this, "USE_DASHED_LINE", value > 0.0);
    }

    /**
     * Caps mode.
     */
    get caps(): LineCaps {
        let result: LineCaps = "Round";
        const capsMode = getShaderMaterialDefine(this, "CAPS_MODE");
        // Sanity check if material define is numerical and has direct mapping to LineCaps type.
        if (typeof capsMode === "number" && DefinesLineCapsMapping.hasOwnProperty(capsMode)) {
            result = DefinesLineCapsMapping[capsMode];
        }
        return result;
    }
    set caps(value: LineCaps) {
        // Line caps mode may be set directly from theme, thus we need to check value
        // correctness and if we provide string to define mapping.
        if (LineCapsDefinesMapping.hasOwnProperty(value)) {
            setShaderMaterialDefine(this, "CAPS_MODE", LineCapsDefinesMapping[value]);
        }
    }

    get fadeNear(): number {
        return this.uniforms.fadeNear.value as number;
    }
    set fadeNear(value: number) {
        this.uniforms.fadeNear.value = value;
    }

    get fadeFar(): number {
        return this.uniforms.fadeFar.value as number;
    }
    set fadeFar(value: number) {
        this.uniforms.fadeFar.value = value;
        setShaderMaterialDefine(this, "USE_FADING", value > 0.0);
    }

    get displacementMap(): THREE.Texture | undefined {
        return this.uniforms.displacementMap.value;
    }
    set displacementMap(map: THREE.Texture | undefined) {
        if (this.uniforms.displacementMap.value === map) {
            return;
        }
        this.uniforms.displacementMap.value = map;
        const useDisplacementMap = map !== undefined;
        if (useDisplacementMap) {
            this.uniforms.displacementMap.value.needsUpdate = true;
        }
        setShaderMaterialDefine(this, "USE_DISPLACEMENTMAP", useDisplacementMap);
    }

    get drawRangeStart(): number {
        return this.uniforms.drawRange.value.x as number;
    }
    set drawRangeStart(value: number) {
        this.uniforms.drawRange.value.x = value;
    }

    get drawRangeEnd(): number {
        return this.uniforms.drawRange.value.y as number;
    }
    set drawRangeEnd(value: number) {
        this.uniforms.drawRange.value.y = value;
    }

    set clipTileSize(tileSize: THREE.Vector2) {
        this.uniforms.tileSize.value.copy(tileSize);
        const useTileClip = tileSize.x > 0 && tileSize.y > 0;
        setShaderMaterialDefine(this, "USE_TILE_CLIP", useTileClip);
    }
    get clipTileSize(): THREE.Vector2 {
        return this.uniforms.tileSize.value as THREE.Vector2;
    }

    private invalidateFog() {
        if (this.m_fog !== getShaderMaterialDefine(this, "USE_FOG")) {
            setShaderMaterialDefine(this, "USE_FOG", this.m_fog);
        }
    }

    private invalidateOpacity() {
        if (this.m_opacity !== this.uniforms.opacity.value) {
            this.uniforms.opacity.value = this.m_opacity;
        }
    }
}
