/*!
 * jsoneditor.js
 *
 * @brief
 * JSONEditor is a web-based tool to view, edit, format, and validate JSON.
 * It has various modes such as a tree editor, a code editor, and a plain text
 * editor.
 *
 * Supported browsers: Chrome, Firefox, Safari, Opera, Internet Explorer 8+
 *
 * @license
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy
 * of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 * Copyright (c) 2011-2017 Jos de Jong, http://jsoneditoronline.org
 *
 * @author  Jos de Jong, <wjosdejong@gmail.com>
 * @version 5.14.1
 * @date    2018-03-21
 */
(function webpackUniversalModuleDefinition(root, factory) {
	if(typeof exports === 'object' && typeof module === 'object')
		module.exports = factory();
	else if(typeof define === 'function' && define.amd)
		define([], factory);
	else if(typeof exports === 'object')
		exports["JSONEditor"] = factory();
	else
		root["JSONEditor"] = factory();
})(this, function() {
return /******/ (function(modules) { // webpackBootstrap
/******/ 	// The module cache
/******/ 	var installedModules = {};

/******/ 	// The require function
/******/ 	function __webpack_require__(moduleId) {

/******/ 		// Check if module is in cache
/******/ 		if(installedModules[moduleId])
/******/ 			return installedModules[moduleId].exports;

/******/ 		// Create a new module (and put it into the cache)
/******/ 		var module = installedModules[moduleId] = {
/******/ 			exports: {},
/******/ 			id: moduleId,
/******/ 			loaded: false
/******/ 		};

/******/ 		// Execute the module function
/******/ 		modules[moduleId].call(module.exports, module, module.exports, __webpack_require__);

/******/ 		// Flag the module as loaded
/******/ 		module.loaded = true;

/******/ 		// Return the exports of the module
/******/ 		return module.exports;
/******/ 	}


/******/ 	// expose the modules object (__webpack_modules__)
/******/ 	__webpack_require__.m = modules;

/******/ 	// expose the module cache
/******/ 	__webpack_require__.c = installedModules;

/******/ 	// __webpack_public_path__
/******/ 	__webpack_require__.p = "";

/******/ 	// Load entry module and return exports
/******/ 	return __webpack_require__(0);
/******/ })
/************************************************************************/
/******/ ([
/* 0 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var Ajv;
	try {
	  Ajv = __webpack_require__(1);
	}
	catch (err) {
	  // no problem... when we need Ajv we will throw a neat exception
	}

	var treemode = __webpack_require__(51);
	var textmode = __webpack_require__(65);
	var util = __webpack_require__(54);

	/**
	 * @constructor JSONEditor
	 * @param {Element} container    Container element
	 * @param {Object}  [options]    Object with options. available options:
	 *                               {String} mode        Editor mode. Available values:
	 *                                                    'tree' (default), 'view',
	 *                                                    'form', 'text', and 'code'.
	 *                               {function} onChange  Callback method, triggered
	 *                                                    on change of contents
	 *                               {function} onError   Callback method, triggered
	 *                                                    when an error occurs
	 *                               {Boolean} search     Enable search box.
	 *                                                    True by default
	 *                                                    Only applicable for modes
	 *                                                    'tree', 'view', and 'form'
	 *                               {Boolean} history    Enable history (undo/redo).
	 *                                                    True by default
	 *                                                    Only applicable for modes
	 *                                                    'tree', 'view', and 'form'
	 *                               {String} name        Field name for the root node.
	 *                                                    Only applicable for modes
	 *                                                    'tree', 'view', and 'form'
	 *                               {Number} indentation     Number of indentation
	 *                                                        spaces. 4 by default.
	 *                                                        Only applicable for
	 *                                                        modes 'text' and 'code'
	 *                               {boolean} escapeUnicode  If true, unicode
	 *                                                        characters are escaped.
	 *                                                        false by default.
	 *                               {boolean} sortObjectKeys If true, object keys are
	 *                                                        sorted before display.
	 *                                                        false by default.
	 * @param {Object | undefined} json JSON object
	 */
	function JSONEditor (container, options, json) {
	  if (!(this instanceof JSONEditor)) {
	    throw new Error('JSONEditor constructor called without "new".');
	  }

	  // check for unsupported browser (IE8 and older)
	  var ieVersion = util.getInternetExplorerVersion();
	  if (ieVersion != -1 && ieVersion < 9) {
	    throw new Error('Unsupported browser, IE9 or newer required. ' +
	        'Please install the newest version of your browser.');
	  }

	  if (options) {
	    // check for deprecated options
	    if (options.error) {
	      console.warn('Option "error" has been renamed to "onError"');
	      options.onError = options.error;
	      delete options.error;
	    }
	    if (options.change) {
	      console.warn('Option "change" has been renamed to "onChange"');
	      options.onChange = options.change;
	      delete options.change;
	    }
	    if (options.editable) {
	      console.warn('Option "editable" has been renamed to "onEditable"');
	      options.onEditable = options.editable;
	      delete options.editable;
	    }

	    // validate options
	    if (options) {
	      var VALID_OPTIONS = [
	        'ajv', 'schema', 'schemaRefs','templates',
	        'ace', 'theme','autocomplete',
	        'onChange', 'onEditable', 'onError', 'onModeChange',
	        'escapeUnicode', 'history', 'search', 'mode', 'modes', 'name', 'indentation', 
	        'sortObjectKeys', 'navigationBar', 'statusBar', 'languages', 'language'
	      ];

	      Object.keys(options).forEach(function (option) {
	        if (VALID_OPTIONS.indexOf(option) === -1) {
	          console.warn('Unknown option "' + option + '". This option will be ignored');
	        }
	      });
	    }
	  }

	  if (arguments.length) {
	    this._create(container, options, json);
	  }
	}

	/**
	 * Configuration for all registered modes. Example:
	 * {
	 *     tree: {
	 *         mixin: TreeEditor,
	 *         data: 'json'
	 *     },
	 *     text: {
	 *         mixin: TextEditor,
	 *         data: 'text'
	 *     }
	 * }
	 *
	 * @type { Object.<String, {mixin: Object, data: String} > }
	 */
	JSONEditor.modes = {};

	// debounce interval for JSON schema vaidation in milliseconds
	JSONEditor.prototype.DEBOUNCE_INTERVAL = 150;

	/**
	 * Create the JSONEditor
	 * @param {Element} container    Container element
	 * @param {Object}  [options]    See description in constructor
	 * @param {Object | undefined} json JSON object
	 * @private
	 */
	JSONEditor.prototype._create = function (container, options, json) {
	  this.container = container;
	  this.options = options || {};
	  this.json = json || {};

	  var mode = this.options.mode || (this.options.modes && this.options.modes[0]) || 'tree';
	  this.setMode(mode);
	};

	/**
	 * Destroy the editor. Clean up DOM, event listeners, and web workers.
	 */
	JSONEditor.prototype.destroy = function () {};

	/**
	 * Set JSON object in editor
	 * @param {Object | undefined} json      JSON data
	 */
	JSONEditor.prototype.set = function (json) {
	  this.json = json;
	};

	/**
	 * Get JSON from the editor
	 * @returns {Object} json
	 */
	JSONEditor.prototype.get = function () {
	  return this.json;
	};

	/**
	 * Set string containing JSON for the editor
	 * @param {String | undefined} jsonText
	 */
	JSONEditor.prototype.setText = function (jsonText) {
	  this.json = util.parse(jsonText);
	};

	/**
	 * Get stringified JSON contents from the editor
	 * @returns {String} jsonText
	 */
	JSONEditor.prototype.getText = function () {
	  return JSON.stringify(this.json);
	};

	/**
	 * Set a field name for the root node.
	 * @param {String | undefined} name
	 */
	JSONEditor.prototype.setName = function (name) {
	  if (!this.options) {
	    this.options = {};
	  }
	  this.options.name = name;
	};

	/**
	 * Get the field name for the root node.
	 * @return {String | undefined} name
	 */
	JSONEditor.prototype.getName = function () {
	  return this.options && this.options.name;
	};

	/**
	 * Change the mode of the editor.
	 * JSONEditor will be extended with all methods needed for the chosen mode.
	 * @param {String} mode     Available modes: 'tree' (default), 'view', 'form',
	 *                          'text', and 'code'.
	 */
	JSONEditor.prototype.setMode = function (mode) {
	  var container = this.container;
	  var options = util.extend({}, this.options);
	  var oldMode = options.mode;
	  var data;
	  var name;

	  options.mode = mode;
	  var config = JSONEditor.modes[mode];
	  if (config) {
	    try {
	      var asText = (config.data == 'text');
	      name = this.getName();
	      data = this[asText ? 'getText' : 'get'](); // get text or json

	      this.destroy();
	      util.clear(this);
	      util.extend(this, config.mixin);
	      this.create(container, options);

	      this.setName(name);
	      this[asText ? 'setText' : 'set'](data); // set text or json

	      if (typeof config.load === 'function') {
	        try {
	          config.load.call(this);
	        }
	        catch (err) {
	          console.error(err);
	        }
	      }

	      if (typeof options.onModeChange === 'function' && mode !== oldMode) {
	        try {
	          options.onModeChange(mode, oldMode);
	        }
	        catch (err) {
	          console.error(err);
	        }
	      }
	    }
	    catch (err) {
	      this._onError(err);
	    }
	  }
	  else {
	    throw new Error('Unknown mode "' + options.mode + '"');
	  }
	};

	/**
	 * Get the current mode
	 * @return {string}
	 */
	JSONEditor.prototype.getMode = function () {
	  return this.options.mode;
	};

	/**
	 * Throw an error. If an error callback is configured in options.error, this
	 * callback will be invoked. Else, a regular error is thrown.
	 * @param {Error} err
	 * @private
	 */
	JSONEditor.prototype._onError = function(err) {
	  if (this.options && typeof this.options.onError === 'function') {
	    this.options.onError(err);
	  }
	  else {
	    throw err;
	  }
	};

	/**
	 * Set a JSON schema for validation of the JSON object.
	 * To remove the schema, call JSONEditor.setSchema(null)
	 * @param {Object | null} schema
	 * @param {Object.<string, Object>=} schemaRefs Schemas that are referenced using the `$ref` property from the JSON schema that are set in the `schema` option,
	 +  the object structure in the form of `{reference_key: schemaObject}`
	 */
	JSONEditor.prototype.setSchema = function (schema, schemaRefs) {
	  // compile a JSON schema validator if a JSON schema is provided
	  if (schema) {
	    var ajv;
	    try {
	      // grab ajv from options if provided, else create a new instance
	      ajv = this.options.ajv || Ajv({ allErrors: true, verbose: true });

	    }
	    catch (err) {
	      console.warn('Failed to create an instance of Ajv, JSON Schema validation is not available. Please use a JSONEditor bundle including Ajv, or pass an instance of Ajv as via the configuration option `ajv`.');
	    }

	    if (ajv) {
	      if(schemaRefs) {
	        for (var ref in schemaRefs) {
	          ajv.removeSchema(ref);  // When updating a schema - old refs has to be removed first
	          if(schemaRefs[ref]) {
	            ajv.addSchema(schemaRefs[ref], ref);
	          }
	        }
	        this.options.schemaRefs = schemaRefs;
	      }
	      this.validateSchema = ajv.compile(schema);

	      // add schema to the options, so that when switching to an other mode,
	      // the set schema is not lost
	      this.options.schema = schema;

	      // validate now
	      this.validate();
	    }

	    this.refresh(); // update DOM
	  }
	  else {
	    // remove current schema
	    this.validateSchema = null;
	    this.options.schema = null;
	    this.options.schemaRefs = null;
	    this.validate(); // to clear current error messages
	    this.refresh();  // update DOM
	  }
	};

	/**
	 * Validate current JSON object against the configured JSON schema
	 * Throws an exception when no JSON schema is configured
	 */
	JSONEditor.prototype.validate = function () {
	  // must be implemented by treemode and textmode
	};

	/**
	 * Refresh the rendered contents
	 */
	JSONEditor.prototype.refresh = function () {
	  // can be implemented by treemode and textmode
	};

	/**
	 * Register a plugin with one ore multiple modes for the JSON Editor.
	 *
	 * A mode is described as an object with properties:
	 *
	 * - `mode: String`           The name of the mode.
	 * - `mixin: Object`          An object containing the mixin functions which
	 *                            will be added to the JSONEditor. Must contain functions
	 *                            create, get, getText, set, and setText. May have
	 *                            additional functions.
	 *                            When the JSONEditor switches to a mixin, all mixin
	 *                            functions are added to the JSONEditor, and then
	 *                            the function `create(container, options)` is executed.
	 * - `data: 'text' | 'json'`  The type of data that will be used to load the mixin.
	 * - `[load: function]`       An optional function called after the mixin
	 *                            has been loaded.
	 *
	 * @param {Object | Array} mode  A mode object or an array with multiple mode objects.
	 */
	JSONEditor.registerMode = function (mode) {
	  var i, prop;

	  if (util.isArray(mode)) {
	    // multiple modes
	    for (i = 0; i < mode.length; i++) {
	      JSONEditor.registerMode(mode[i]);
	    }
	  }
	  else {
	    // validate the new mode
	    if (!('mode' in mode)) throw new Error('Property "mode" missing');
	    if (!('mixin' in mode)) throw new Error('Property "mixin" missing');
	    if (!('data' in mode)) throw new Error('Property "data" missing');
	    var name = mode.mode;
	    if (name in JSONEditor.modes) {
	      throw new Error('Mode "' + name + '" already registered');
	    }

	    // validate the mixin
	    if (typeof mode.mixin.create !== 'function') {
	      throw new Error('Required function "create" missing on mixin');
	    }
	    var reserved = ['setMode', 'registerMode', 'modes'];
	    for (i = 0; i < reserved.length; i++) {
	      prop = reserved[i];
	      if (prop in mode.mixin) {
	        throw new Error('Reserved property "' + prop + '" not allowed in mixin');
	      }
	    }

	    JSONEditor.modes[name] = mode;
	  }
	};

	// register tree and text modes
	JSONEditor.registerMode(treemode);
	JSONEditor.registerMode(textmode);

	module.exports = JSONEditor;


/***/ },
/* 1 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var compileSchema = __webpack_require__(2)
	  , resolve = __webpack_require__(3)
	  , Cache = __webpack_require__(19)
	  , SchemaObject = __webpack_require__(13)
	  , stableStringify = __webpack_require__(16)
	  , formats = __webpack_require__(20)
	  , rules = __webpack_require__(21)
	  , $dataMetaSchema = __webpack_require__(44)
	  , patternGroups = __webpack_require__(45)
	  , util = __webpack_require__(11)
	  , co = __webpack_require__(18);

	module.exports = Ajv;

	Ajv.prototype.validate = validate;
	Ajv.prototype.compile = compile;
	Ajv.prototype.addSchema = addSchema;
	Ajv.prototype.addMetaSchema = addMetaSchema;
	Ajv.prototype.validateSchema = validateSchema;
	Ajv.prototype.getSchema = getSchema;
	Ajv.prototype.removeSchema = removeSchema;
	Ajv.prototype.addFormat = addFormat;
	Ajv.prototype.errorsText = errorsText;

	Ajv.prototype._addSchema = _addSchema;
	Ajv.prototype._compile = _compile;

	Ajv.prototype.compileAsync = __webpack_require__(46);
	var customKeyword = __webpack_require__(47);
	Ajv.prototype.addKeyword = customKeyword.add;
	Ajv.prototype.getKeyword = customKeyword.get;
	Ajv.prototype.removeKeyword = customKeyword.remove;

	var errorClasses = __webpack_require__(15);
	Ajv.ValidationError = errorClasses.Validation;
	Ajv.MissingRefError = errorClasses.MissingRef;
	Ajv.$dataMetaSchema = $dataMetaSchema;

	var META_SCHEMA_ID = 'http://json-schema.org/draft-06/schema';

	var META_IGNORE_OPTIONS = [ 'removeAdditional', 'useDefaults', 'coerceTypes' ];
	var META_SUPPORT_DATA = ['/properties'];

	/**
	 * Creates validator instance.
	 * Usage: `Ajv(opts)`
	 * @param {Object} opts optional options
	 * @return {Object} ajv instance
	 */
	function Ajv(opts) {
	  if (!(this instanceof Ajv)) return new Ajv(opts);
	  opts = this._opts = util.copy(opts) || {};
	  setLogger(this);
	  this._schemas = {};
	  this._refs = {};
	  this._fragments = {};
	  this._formats = formats(opts.format);
	  var schemaUriFormat = this._schemaUriFormat = this._formats['uri-reference'];
	  this._schemaUriFormatFunc = function (str) { return schemaUriFormat.test(str); };

	  this._cache = opts.cache || new Cache;
	  this._loadingSchemas = {};
	  this._compilations = [];
	  this.RULES = rules();
	  this._getId = chooseGetId(opts);

	  opts.loopRequired = opts.loopRequired || Infinity;
	  if (opts.errorDataPath == 'property') opts._errorDataPathProperty = true;
	  if (opts.serialize === undefined) opts.serialize = stableStringify;
	  this._metaOpts = getMetaSchemaOptions(this);

	  if (opts.formats) addInitialFormats(this);
	  addDraft6MetaSchema(this);
	  if (typeof opts.meta == 'object') this.addMetaSchema(opts.meta);
	  addInitialSchemas(this);
	  if (opts.patternGroups) patternGroups(this);
	}



	/**
	 * Validate data using schema
	 * Schema will be compiled and cached (using serialized JSON as key. [fast-json-stable-stringify](https://github.com/epoberezkin/fast-json-stable-stringify) is used to serialize.
	 * @this   Ajv
	 * @param  {String|Object} schemaKeyRef key, ref or schema object
	 * @param  {Any} data to be validated
	 * @return {Boolean} validation result. Errors from the last validation will be available in `ajv.errors` (and also in compiled schema: `schema.errors`).
	 */
	function validate(schemaKeyRef, data) {
	  var v;
	  if (typeof schemaKeyRef == 'string') {
	    v = this.getSchema(schemaKeyRef);
	    if (!v) throw new Error('no schema with key or ref "' + schemaKeyRef + '"');
	  } else {
	    var schemaObj = this._addSchema(schemaKeyRef);
	    v = schemaObj.validate || this._compile(schemaObj);
	  }

	  var valid = v(data);
	  if (v.$async === true)
	    return this._opts.async == '*' ? co(valid) : valid;
	  this.errors = v.errors;
	  return valid;
	}


	/**
	 * Create validating function for passed schema.
	 * @this   Ajv
	 * @param  {Object} schema schema object
	 * @param  {Boolean} _meta true if schema is a meta-schema. Used internally to compile meta schemas of custom keywords.
	 * @return {Function} validating function
	 */
	function compile(schema, _meta) {
	  var schemaObj = this._addSchema(schema, undefined, _meta);
	  return schemaObj.validate || this._compile(schemaObj);
	}


	/**
	 * Adds schema to the instance.
	 * @this   Ajv
	 * @param {Object|Array} schema schema or array of schemas. If array is passed, `key` and other parameters will be ignored.
	 * @param {String} key Optional schema key. Can be passed to `validate` method instead of schema object or id/ref. One schema per instance can have empty `id` and `key`.
	 * @param {Boolean} _skipValidation true to skip schema validation. Used internally, option validateSchema should be used instead.
	 * @param {Boolean} _meta true if schema is a meta-schema. Used internally, addMetaSchema should be used instead.
	 * @return {Ajv} this for method chaining
	 */
	function addSchema(schema, key, _skipValidation, _meta) {
	  if (Array.isArray(schema)){
	    for (var i=0; i<schema.length; i++) this.addSchema(schema[i], undefined, _skipValidation, _meta);
	    return this;
	  }
	  var id = this._getId(schema);
	  if (id !== undefined && typeof id != 'string')
	    throw new Error('schema id must be string');
	  key = resolve.normalizeId(key || id);
	  checkUnique(this, key);
	  this._schemas[key] = this._addSchema(schema, _skipValidation, _meta, true);
	  return this;
	}


	/**
	 * Add schema that will be used to validate other schemas
	 * options in META_IGNORE_OPTIONS are alway set to false
	 * @this   Ajv
	 * @param {Object} schema schema object
	 * @param {String} key optional schema key
	 * @param {Boolean} skipValidation true to skip schema validation, can be used to override validateSchema option for meta-schema
	 * @return {Ajv} this for method chaining
	 */
	function addMetaSchema(schema, key, skipValidation) {
	  this.addSchema(schema, key, skipValidation, true);
	  return this;
	}


	/**
	 * Validate schema
	 * @this   Ajv
	 * @param {Object} schema schema to validate
	 * @param {Boolean} throwOrLogError pass true to throw (or log) an error if invalid
	 * @return {Boolean} true if schema is valid
	 */
	function validateSchema(schema, throwOrLogError) {
	  var $schema = schema.$schema;
	  if ($schema !== undefined && typeof $schema != 'string')
	    throw new Error('$schema must be a string');
	  $schema = $schema || this._opts.defaultMeta || defaultMeta(this);
	  if (!$schema) {
	    this.logger.warn('meta-schema not available');
	    this.errors = null;
	    return true;
	  }
	  var currentUriFormat = this._formats.uri;
	  this._formats.uri = typeof currentUriFormat == 'function'
	                      ? this._schemaUriFormatFunc
	                      : this._schemaUriFormat;
	  var valid;
	  try { valid = this.validate($schema, schema); }
	  finally { this._formats.uri = currentUriFormat; }
	  if (!valid && throwOrLogError) {
	    var message = 'schema is invalid: ' + this.errorsText();
	    if (this._opts.validateSchema == 'log') this.logger.error(message);
	    else throw new Error(message);
	  }
	  return valid;
	}


	function defaultMeta(self) {
	  var meta = self._opts.meta;
	  self._opts.defaultMeta = typeof meta == 'object'
	                            ? self._getId(meta) || meta
	                            : self.getSchema(META_SCHEMA_ID)
	                              ? META_SCHEMA_ID
	                              : undefined;
	  return self._opts.defaultMeta;
	}


	/**
	 * Get compiled schema from the instance by `key` or `ref`.
	 * @this   Ajv
	 * @param  {String} keyRef `key` that was passed to `addSchema` or full schema reference (`schema.id` or resolved id).
	 * @return {Function} schema validating function (with property `schema`).
	 */
	function getSchema(keyRef) {
	  var schemaObj = _getSchemaObj(this, keyRef);
	  switch (typeof schemaObj) {
	    case 'object': return schemaObj.validate || this._compile(schemaObj);
	    case 'string': return this.getSchema(schemaObj);
	    case 'undefined': return _getSchemaFragment(this, keyRef);
	  }
	}


	function _getSchemaFragment(self, ref) {
	  var res = resolve.schema.call(self, { schema: {} }, ref);
	  if (res) {
	    var schema = res.schema
	      , root = res.root
	      , baseId = res.baseId;
	    var v = compileSchema.call(self, schema, root, undefined, baseId);
	    self._fragments[ref] = new SchemaObject({
	      ref: ref,
	      fragment: true,
	      schema: schema,
	      root: root,
	      baseId: baseId,
	      validate: v
	    });
	    return v;
	  }
	}


	function _getSchemaObj(self, keyRef) {
	  keyRef = resolve.normalizeId(keyRef);
	  return self._schemas[keyRef] || self._refs[keyRef] || self._fragments[keyRef];
	}


	/**
	 * Remove cached schema(s).
	 * If no parameter is passed all schemas but meta-schemas are removed.
	 * If RegExp is passed all schemas with key/id matching pattern but meta-schemas are removed.
	 * Even if schema is referenced by other schemas it still can be removed as other schemas have local references.
	 * @this   Ajv
	 * @param  {String|Object|RegExp} schemaKeyRef key, ref, pattern to match key/ref or schema object
	 * @return {Ajv} this for method chaining
	 */
	function removeSchema(schemaKeyRef) {
	  if (schemaKeyRef instanceof RegExp) {
	    _removeAllSchemas(this, this._schemas, schemaKeyRef);
	    _removeAllSchemas(this, this._refs, schemaKeyRef);
	    return this;
	  }
	  switch (typeof schemaKeyRef) {
	    case 'undefined':
	      _removeAllSchemas(this, this._schemas);
	      _removeAllSchemas(this, this._refs);
	      this._cache.clear();
	      return this;
	    case 'string':
	      var schemaObj = _getSchemaObj(this, schemaKeyRef);
	      if (schemaObj) this._cache.del(schemaObj.cacheKey);
	      delete this._schemas[schemaKeyRef];
	      delete this._refs[schemaKeyRef];
	      return this;
	    case 'object':
	      var serialize = this._opts.serialize;
	      var cacheKey = serialize ? serialize(schemaKeyRef) : schemaKeyRef;
	      this._cache.del(cacheKey);
	      var id = this._getId(schemaKeyRef);
	      if (id) {
	        id = resolve.normalizeId(id);
	        delete this._schemas[id];
	        delete this._refs[id];
	      }
	  }
	  return this;
	}


	function _removeAllSchemas(self, schemas, regex) {
	  for (var keyRef in schemas) {
	    var schemaObj = schemas[keyRef];
	    if (!schemaObj.meta && (!regex || regex.test(keyRef))) {
	      self._cache.del(schemaObj.cacheKey);
	      delete schemas[keyRef];
	    }
	  }
	}


	/* @this   Ajv */
	function _addSchema(schema, skipValidation, meta, shouldAddSchema) {
	  if (typeof schema != 'object' && typeof schema != 'boolean')
	    throw new Error('schema should be object or boolean');
	  var serialize = this._opts.serialize;
	  var cacheKey = serialize ? serialize(schema) : schema;
	  var cached = this._cache.get(cacheKey);
	  if (cached) return cached;

	  shouldAddSchema = shouldAddSchema || this._opts.addUsedSchema !== false;

	  var id = resolve.normalizeId(this._getId(schema));
	  if (id && shouldAddSchema) checkUnique(this, id);

	  var willValidate = this._opts.validateSchema !== false && !skipValidation;
	  var recursiveMeta;
	  if (willValidate && !(recursiveMeta = id && id == resolve.normalizeId(schema.$schema)))
	    this.validateSchema(schema, true);

	  var localRefs = resolve.ids.call(this, schema);

	  var schemaObj = new SchemaObject({
	    id: id,
	    schema: schema,
	    localRefs: localRefs,
	    cacheKey: cacheKey,
	    meta: meta
	  });

	  if (id[0] != '#' && shouldAddSchema) this._refs[id] = schemaObj;
	  this._cache.put(cacheKey, schemaObj);

	  if (willValidate && recursiveMeta) this.validateSchema(schema, true);

	  return schemaObj;
	}


	/* @this   Ajv */
	function _compile(schemaObj, root) {
	  if (schemaObj.compiling) {
	    schemaObj.validate = callValidate;
	    callValidate.schema = schemaObj.schema;
	    callValidate.errors = null;
	    callValidate.root = root ? root : callValidate;
	    if (schemaObj.schema.$async === true)
	      callValidate.$async = true;
	    return callValidate;
	  }
	  schemaObj.compiling = true;

	  var currentOpts;
	  if (schemaObj.meta) {
	    currentOpts = this._opts;
	    this._opts = this._metaOpts;
	  }

	  var v;
	  try { v = compileSchema.call(this, schemaObj.schema, root, schemaObj.localRefs); }
	  finally {
	    schemaObj.compiling = false;
	    if (schemaObj.meta) this._opts = currentOpts;
	  }

	  schemaObj.validate = v;
	  schemaObj.refs = v.refs;
	  schemaObj.refVal = v.refVal;
	  schemaObj.root = v.root;
	  return v;


	  function callValidate() {
	    var _validate = schemaObj.validate;
	    var result = _validate.apply(null, arguments);
	    callValidate.errors = _validate.errors;
	    return result;
	  }
	}


	function chooseGetId(opts) {
	  switch (opts.schemaId) {
	    case '$id': return _get$Id;
	    case 'id': return _getId;
	    default: return _get$IdOrId;
	  }
	}

	/* @this   Ajv */
	function _getId(schema) {
	  if (schema.$id) this.logger.warn('schema $id ignored', schema.$id);
	  return schema.id;
	}

	/* @this   Ajv */
	function _get$Id(schema) {
	  if (schema.id) this.logger.warn('schema id ignored', schema.id);
	  return schema.$id;
	}


	function _get$IdOrId(schema) {
	  if (schema.$id && schema.id && schema.$id != schema.id)
	    throw new Error('schema $id is different from id');
	  return schema.$id || schema.id;
	}


	/**
	 * Convert array of error message objects to string
	 * @this   Ajv
	 * @param  {Array<Object>} errors optional array of validation errors, if not passed errors from the instance are used.
	 * @param  {Object} options optional options with properties `separator` and `dataVar`.
	 * @return {String} human readable string with all errors descriptions
	 */
	function errorsText(errors, options) {
	  errors = errors || this.errors;
	  if (!errors) return 'No errors';
	  options = options || {};
	  var separator = options.separator === undefined ? ', ' : options.separator;
	  var dataVar = options.dataVar === undefined ? 'data' : options.dataVar;

	  var text = '';
	  for (var i=0; i<errors.length; i++) {
	    var e = errors[i];
	    if (e) text += dataVar + e.dataPath + ' ' + e.message + separator;
	  }
	  return text.slice(0, -separator.length);
	}


	/**
	 * Add custom format
	 * @this   Ajv
	 * @param {String} name format name
	 * @param {String|RegExp|Function} format string is converted to RegExp; function should return boolean (true when valid)
	 * @return {Ajv} this for method chaining
	 */
	function addFormat(name, format) {
	  if (typeof format == 'string') format = new RegExp(format);
	  this._formats[name] = format;
	  return this;
	}


	function addDraft6MetaSchema(self) {
	  var $dataSchema;
	  if (self._opts.$data) {
	    $dataSchema = __webpack_require__(49);
	    self.addMetaSchema($dataSchema, $dataSchema.$id, true);
	  }
	  if (self._opts.meta === false) return;
	  var metaSchema = __webpack_require__(50);
	  if (self._opts.$data) metaSchema = $dataMetaSchema(metaSchema, META_SUPPORT_DATA);
	  self.addMetaSchema(metaSchema, META_SCHEMA_ID, true);
	  self._refs['http://json-schema.org/schema'] = META_SCHEMA_ID;
	}


	function addInitialSchemas(self) {
	  var optsSchemas = self._opts.schemas;
	  if (!optsSchemas) return;
	  if (Array.isArray(optsSchemas)) self.addSchema(optsSchemas);
	  else for (var key in optsSchemas) self.addSchema(optsSchemas[key], key);
	}


	function addInitialFormats(self) {
	  for (var name in self._opts.formats) {
	    var format = self._opts.formats[name];
	    self.addFormat(name, format);
	  }
	}


	function checkUnique(self, id) {
	  if (self._schemas[id] || self._refs[id])
	    throw new Error('schema with key or id "' + id + '" already exists');
	}


	function getMetaSchemaOptions(self) {
	  var metaOpts = util.copy(self._opts);
	  for (var i=0; i<META_IGNORE_OPTIONS.length; i++)
	    delete metaOpts[META_IGNORE_OPTIONS[i]];
	  return metaOpts;
	}


	function setLogger(self) {
	  var logger = self._opts.logger;
	  if (logger === false) {
	    self.logger = {log: noop, warn: noop, error: noop};
	  } else {
	    if (logger === undefined) logger = console;
	    if (!(typeof logger == 'object' && logger.log && logger.warn && logger.error))
	      throw new Error('logger must implement log, warn and error methods');
	    self.logger = logger;
	  }
	}


	function noop() {}


/***/ },
/* 2 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var resolve = __webpack_require__(3)
	  , util = __webpack_require__(11)
	  , errorClasses = __webpack_require__(15)
	  , stableStringify = __webpack_require__(16);

	var validateGenerator = __webpack_require__(17);

	/**
	 * Functions below are used inside compiled validations function
	 */

	var co = __webpack_require__(18);
	var ucs2length = util.ucs2length;
	var equal = __webpack_require__(10);

	// this error is thrown by async schemas to return validation errors via exception
	var ValidationError = errorClasses.Validation;

	module.exports = compile;


	/**
	 * Compiles schema to validation function
	 * @this   Ajv
	 * @param  {Object} schema schema object
	 * @param  {Object} root object with information about the root schema for this schema
	 * @param  {Object} localRefs the hash of local references inside the schema (created by resolve.id), used for inline resolution
	 * @param  {String} baseId base ID for IDs in the schema
	 * @return {Function} validation function
	 */
	function compile(schema, root, localRefs, baseId) {
	  /* jshint validthis: true, evil: true */
	  /* eslint no-shadow: 0 */
	  var self = this
	    , opts = this._opts
	    , refVal = [ undefined ]
	    , refs = {}
	    , patterns = []
	    , patternsHash = {}
	    , defaults = []
	    , defaultsHash = {}
	    , customRules = [];

	  root = root || { schema: schema, refVal: refVal, refs: refs };

	  var c = checkCompiling.call(this, schema, root, baseId);
	  var compilation = this._compilations[c.index];
	  if (c.compiling) return (compilation.callValidate = callValidate);

	  var formats = this._formats;
	  var RULES = this.RULES;

	  try {
	    var v = localCompile(schema, root, localRefs, baseId);
	    compilation.validate = v;
	    var cv = compilation.callValidate;
	    if (cv) {
	      cv.schema = v.schema;
	      cv.errors = null;
	      cv.refs = v.refs;
	      cv.refVal = v.refVal;
	      cv.root = v.root;
	      cv.$async = v.$async;
	      if (opts.sourceCode) cv.source = v.source;
	    }
	    return v;
	  } finally {
	    endCompiling.call(this, schema, root, baseId);
	  }

	  function callValidate() {
	    var validate = compilation.validate;
	    var result = validate.apply(null, arguments);
	    callValidate.errors = validate.errors;
	    return result;
	  }

	  function localCompile(_schema, _root, localRefs, baseId) {
	    var isRoot = !_root || (_root && _root.schema == _schema);
	    if (_root.schema != root.schema)
	      return compile.call(self, _schema, _root, localRefs, baseId);

	    var $async = _schema.$async === true;

	    var sourceCode = validateGenerator({
	      isTop: true,
	      schema: _schema,
	      isRoot: isRoot,
	      baseId: baseId,
	      root: _root,
	      schemaPath: '',
	      errSchemaPath: '#',
	      errorPath: '""',
	      MissingRefError: errorClasses.MissingRef,
	      RULES: RULES,
	      validate: validateGenerator,
	      util: util,
	      resolve: resolve,
	      resolveRef: resolveRef,
	      usePattern: usePattern,
	      useDefault: useDefault,
	      useCustomRule: useCustomRule,
	      opts: opts,
	      formats: formats,
	      logger: self.logger,
	      self: self
	    });

	    sourceCode = vars(refVal, refValCode) + vars(patterns, patternCode)
	                   + vars(defaults, defaultCode) + vars(customRules, customRuleCode)
	                   + sourceCode;

	    if (opts.processCode) sourceCode = opts.processCode(sourceCode);
	    // console.log('\n\n\n *** \n', JSON.stringify(sourceCode));
	    var validate;
	    try {
	      var makeValidate = new Function(
	        'self',
	        'RULES',
	        'formats',
	        'root',
	        'refVal',
	        'defaults',
	        'customRules',
	        'co',
	        'equal',
	        'ucs2length',
	        'ValidationError',
	        sourceCode
	      );

	      validate = makeValidate(
	        self,
	        RULES,
	        formats,
	        root,
	        refVal,
	        defaults,
	        customRules,
	        co,
	        equal,
	        ucs2length,
	        ValidationError
	      );

	      refVal[0] = validate;
	    } catch(e) {
	      self.logger.error('Error compiling schema, function code:', sourceCode);
	      throw e;
	    }

	    validate.schema = _schema;
	    validate.errors = null;
	    validate.refs = refs;
	    validate.refVal = refVal;
	    validate.root = isRoot ? validate : _root;
	    if ($async) validate.$async = true;
	    if (opts.sourceCode === true) {
	      validate.source = {
	        code: sourceCode,
	        patterns: patterns,
	        defaults: defaults
	      };
	    }

	    return validate;
	  }

	  function resolveRef(baseId, ref, isRoot) {
	    ref = resolve.url(baseId, ref);
	    var refIndex = refs[ref];
	    var _refVal, refCode;
	    if (refIndex !== undefined) {
	      _refVal = refVal[refIndex];
	      refCode = 'refVal[' + refIndex + ']';
	      return resolvedRef(_refVal, refCode);
	    }
	    if (!isRoot && root.refs) {
	      var rootRefId = root.refs[ref];
	      if (rootRefId !== undefined) {
	        _refVal = root.refVal[rootRefId];
	        refCode = addLocalRef(ref, _refVal);
	        return resolvedRef(_refVal, refCode);
	      }
	    }

	    refCode = addLocalRef(ref);
	    var v = resolve.call(self, localCompile, root, ref);
	    if (v === undefined) {
	      var localSchema = localRefs && localRefs[ref];
	      if (localSchema) {
	        v = resolve.inlineRef(localSchema, opts.inlineRefs)
	            ? localSchema
	            : compile.call(self, localSchema, root, localRefs, baseId);
	      }
	    }

	    if (v === undefined) {
	      removeLocalRef(ref);
	    } else {
	      replaceLocalRef(ref, v);
	      return resolvedRef(v, refCode);
	    }
	  }

	  function addLocalRef(ref, v) {
	    var refId = refVal.length;
	    refVal[refId] = v;
	    refs[ref] = refId;
	    return 'refVal' + refId;
	  }

	  function removeLocalRef(ref) {
	    delete refs[ref];
	  }

	  function replaceLocalRef(ref, v) {
	    var refId = refs[ref];
	    refVal[refId] = v;
	  }

	  function resolvedRef(refVal, code) {
	    return typeof refVal == 'object' || typeof refVal == 'boolean'
	            ? { code: code, schema: refVal, inline: true }
	            : { code: code, $async: refVal && refVal.$async };
	  }

	  function usePattern(regexStr) {
	    var index = patternsHash[regexStr];
	    if (index === undefined) {
	      index = patternsHash[regexStr] = patterns.length;
	      patterns[index] = regexStr;
	    }
	    return 'pattern' + index;
	  }

	  function useDefault(value) {
	    switch (typeof value) {
	      case 'boolean':
	      case 'number':
	        return '' + value;
	      case 'string':
	        return util.toQuotedString(value);
	      case 'object':
	        if (value === null) return 'null';
	        var valueStr = stableStringify(value);
	        var index = defaultsHash[valueStr];
	        if (index === undefined) {
	          index = defaultsHash[valueStr] = defaults.length;
	          defaults[index] = value;
	        }
	        return 'default' + index;
	    }
	  }

	  function useCustomRule(rule, schema, parentSchema, it) {
	    var validateSchema = rule.definition.validateSchema;
	    if (validateSchema && self._opts.validateSchema !== false) {
	      var valid = validateSchema(schema);
	      if (!valid) {
	        var message = 'keyword schema is invalid: ' + self.errorsText(validateSchema.errors);
	        if (self._opts.validateSchema == 'log') self.logger.error(message);
	        else throw new Error(message);
	      }
	    }

	    var compile = rule.definition.compile
	      , inline = rule.definition.inline
	      , macro = rule.definition.macro;

	    var validate;
	    if (compile) {
	      validate = compile.call(self, schema, parentSchema, it);
	    } else if (macro) {
	      validate = macro.call(self, schema, parentSchema, it);
	      if (opts.validateSchema !== false) self.validateSchema(validate, true);
	    } else if (inline) {
	      validate = inline.call(self, it, rule.keyword, schema, parentSchema);
	    } else {
	      validate = rule.definition.validate;
	      if (!validate) return;
	    }

	    if (validate === undefined)
	      throw new Error('custom keyword "' + rule.keyword + '"failed to compile');

	    var index = customRules.length;
	    customRules[index] = validate;

	    return {
	      code: 'customRule' + index,
	      validate: validate
	    };
	  }
	}


	/**
	 * Checks if the schema is currently compiled
	 * @this   Ajv
	 * @param  {Object} schema schema to compile
	 * @param  {Object} root root object
	 * @param  {String} baseId base schema ID
	 * @return {Object} object with properties "index" (compilation index) and "compiling" (boolean)
	 */
	function checkCompiling(schema, root, baseId) {
	  /* jshint validthis: true */
	  var index = compIndex.call(this, schema, root, baseId);
	  if (index >= 0) return { index: index, compiling: true };
	  index = this._compilations.length;
	  this._compilations[index] = {
	    schema: schema,
	    root: root,
	    baseId: baseId
	  };
	  return { index: index, compiling: false };
	}


	/**
	 * Removes the schema from the currently compiled list
	 * @this   Ajv
	 * @param  {Object} schema schema to compile
	 * @param  {Object} root root object
	 * @param  {String} baseId base schema ID
	 */
	function endCompiling(schema, root, baseId) {
	  /* jshint validthis: true */
	  var i = compIndex.call(this, schema, root, baseId);
	  if (i >= 0) this._compilations.splice(i, 1);
	}


	/**
	 * Index of schema compilation in the currently compiled list
	 * @this   Ajv
	 * @param  {Object} schema schema to compile
	 * @param  {Object} root root object
	 * @param  {String} baseId base schema ID
	 * @return {Integer} compilation index
	 */
	function compIndex(schema, root, baseId) {
	  /* jshint validthis: true */
	  for (var i=0; i<this._compilations.length; i++) {
	    var c = this._compilations[i];
	    if (c.schema == schema && c.root == root && c.baseId == baseId) return i;
	  }
	  return -1;
	}


	function patternCode(i, patterns) {
	  return 'var pattern' + i + ' = new RegExp(' + util.toQuotedString(patterns[i]) + ');';
	}


	function defaultCode(i) {
	  return 'var default' + i + ' = defaults[' + i + '];';
	}


	function refValCode(i, refVal) {
	  return refVal[i] === undefined ? '' : 'var refVal' + i + ' = refVal[' + i + '];';
	}


	function customRuleCode(i) {
	  return 'var customRule' + i + ' = customRules[' + i + '];';
	}


	function vars(arr, statement) {
	  if (!arr.length) return '';
	  var code = '';
	  for (var i=0; i<arr.length; i++)
	    code += statement(i, arr);
	  return code;
	}


/***/ },
/* 3 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var url = __webpack_require__(4)
	  , equal = __webpack_require__(10)
	  , util = __webpack_require__(11)
	  , SchemaObject = __webpack_require__(13)
	  , traverse = __webpack_require__(14);

	module.exports = resolve;

	resolve.normalizeId = normalizeId;
	resolve.fullPath = getFullPath;
	resolve.url = resolveUrl;
	resolve.ids = resolveIds;
	resolve.inlineRef = inlineRef;
	resolve.schema = resolveSchema;

	/**
	 * [resolve and compile the references ($ref)]
	 * @this   Ajv
	 * @param  {Function} compile reference to schema compilation funciton (localCompile)
	 * @param  {Object} root object with information about the root schema for the current schema
	 * @param  {String} ref reference to resolve
	 * @return {Object|Function} schema object (if the schema can be inlined) or validation function
	 */
	function resolve(compile, root, ref) {
	  /* jshint validthis: true */
	  var refVal = this._refs[ref];
	  if (typeof refVal == 'string') {
	    if (this._refs[refVal]) refVal = this._refs[refVal];
	    else return resolve.call(this, compile, root, refVal);
	  }

	  refVal = refVal || this._schemas[ref];
	  if (refVal instanceof SchemaObject) {
	    return inlineRef(refVal.schema, this._opts.inlineRefs)
	            ? refVal.schema
	            : refVal.validate || this._compile(refVal);
	  }

	  var res = resolveSchema.call(this, root, ref);
	  var schema, v, baseId;
	  if (res) {
	    schema = res.schema;
	    root = res.root;
	    baseId = res.baseId;
	  }

	  if (schema instanceof SchemaObject) {
	    v = schema.validate || compile.call(this, schema.schema, root, undefined, baseId);
	  } else if (schema !== undefined) {
	    v = inlineRef(schema, this._opts.inlineRefs)
	        ? schema
	        : compile.call(this, schema, root, undefined, baseId);
	  }

	  return v;
	}


	/**
	 * Resolve schema, its root and baseId
	 * @this Ajv
	 * @param  {Object} root root object with properties schema, refVal, refs
	 * @param  {String} ref  reference to resolve
	 * @return {Object} object with properties schema, root, baseId
	 */
	function resolveSchema(root, ref) {
	  /* jshint validthis: true */
	  var p = url.parse(ref, false, true)
	    , refPath = _getFullPath(p)
	    , baseId = getFullPath(this._getId(root.schema));
	  if (refPath !== baseId) {
	    var id = normalizeId(refPath);
	    var refVal = this._refs[id];
	    if (typeof refVal == 'string') {
	      return resolveRecursive.call(this, root, refVal, p);
	    } else if (refVal instanceof SchemaObject) {
	      if (!refVal.validate) this._compile(refVal);
	      root = refVal;
	    } else {
	      refVal = this._schemas[id];
	      if (refVal instanceof SchemaObject) {
	        if (!refVal.validate) this._compile(refVal);
	        if (id == normalizeId(ref))
	          return { schema: refVal, root: root, baseId: baseId };
	        root = refVal;
	      } else {
	        return;
	      }
	    }
	    if (!root.schema) return;
	    baseId = getFullPath(this._getId(root.schema));
	  }
	  return getJsonPointer.call(this, p, baseId, root.schema, root);
	}


	/* @this Ajv */
	function resolveRecursive(root, ref, parsedRef) {
	  /* jshint validthis: true */
	  var res = resolveSchema.call(this, root, ref);
	  if (res) {
	    var schema = res.schema;
	    var baseId = res.baseId;
	    root = res.root;
	    var id = this._getId(schema);
	    if (id) baseId = resolveUrl(baseId, id);
	    return getJsonPointer.call(this, parsedRef, baseId, schema, root);
	  }
	}


	var PREVENT_SCOPE_CHANGE = util.toHash(['properties', 'patternProperties', 'enum', 'dependencies', 'definitions']);
	/* @this Ajv */
	function getJsonPointer(parsedRef, baseId, schema, root) {
	  /* jshint validthis: true */
	  parsedRef.hash = parsedRef.hash || '';
	  if (parsedRef.hash.slice(0,2) != '#/') return;
	  var parts = parsedRef.hash.split('/');

	  for (var i = 1; i < parts.length; i++) {
	    var part = parts[i];
	    if (part) {
	      part = util.unescapeFragment(part);
	      schema = schema[part];
	      if (schema === undefined) break;
	      var id;
	      if (!PREVENT_SCOPE_CHANGE[part]) {
	        id = this._getId(schema);
	        if (id) baseId = resolveUrl(baseId, id);
	        if (schema.$ref) {
	          var $ref = resolveUrl(baseId, schema.$ref);
	          var res = resolveSchema.call(this, root, $ref);
	          if (res) {
	            schema = res.schema;
	            root = res.root;
	            baseId = res.baseId;
	          }
	        }
	      }
	    }
	  }
	  if (schema !== undefined && schema !== root.schema)
	    return { schema: schema, root: root, baseId: baseId };
	}


	var SIMPLE_INLINED = util.toHash([
	  'type', 'format', 'pattern',
	  'maxLength', 'minLength',
	  'maxProperties', 'minProperties',
	  'maxItems', 'minItems',
	  'maximum', 'minimum',
	  'uniqueItems', 'multipleOf',
	  'required', 'enum'
	]);
	function inlineRef(schema, limit) {
	  if (limit === false) return false;
	  if (limit === undefined || limit === true) return checkNoRef(schema);
	  else if (limit) return countKeys(schema) <= limit;
	}


	function checkNoRef(schema) {
	  var item;
	  if (Array.isArray(schema)) {
	    for (var i=0; i<schema.length; i++) {
	      item = schema[i];
	      if (typeof item == 'object' && !checkNoRef(item)) return false;
	    }
	  } else {
	    for (var key in schema) {
	      if (key == '$ref') return false;
	      item = schema[key];
	      if (typeof item == 'object' && !checkNoRef(item)) return false;
	    }
	  }
	  return true;
	}


	function countKeys(schema) {
	  var count = 0, item;
	  if (Array.isArray(schema)) {
	    for (var i=0; i<schema.length; i++) {
	      item = schema[i];
	      if (typeof item == 'object') count += countKeys(item);
	      if (count == Infinity) return Infinity;
	    }
	  } else {
	    for (var key in schema) {
	      if (key == '$ref') return Infinity;
	      if (SIMPLE_INLINED[key]) {
	        count++;
	      } else {
	        item = schema[key];
	        if (typeof item == 'object') count += countKeys(item) + 1;
	        if (count == Infinity) return Infinity;
	      }
	    }
	  }
	  return count;
	}


	function getFullPath(id, normalize) {
	  if (normalize !== false) id = normalizeId(id);
	  var p = url.parse(id, false, true);
	  return _getFullPath(p);
	}


	function _getFullPath(p) {
	  var protocolSeparator = p.protocol || p.href.slice(0,2) == '//' ? '//' : '';
	  return (p.protocol||'') + protocolSeparator + (p.host||'') + (p.path||'')  + '#';
	}


	var TRAILING_SLASH_HASH = /#\/?$/;
	function normalizeId(id) {
	  return id ? id.replace(TRAILING_SLASH_HASH, '') : '';
	}


	function resolveUrl(baseId, id) {
	  id = normalizeId(id);
	  return url.resolve(baseId, id);
	}


	/* @this Ajv */
	function resolveIds(schema) {
	  var schemaId = normalizeId(this._getId(schema));
	  var baseIds = {'': schemaId};
	  var fullPaths = {'': getFullPath(schemaId, false)};
	  var localRefs = {};
	  var self = this;

	  traverse(schema, {allKeys: true}, function(sch, jsonPtr, rootSchema, parentJsonPtr, parentKeyword, parentSchema, keyIndex) {
	    if (jsonPtr === '') return;
	    var id = self._getId(sch);
	    var baseId = baseIds[parentJsonPtr];
	    var fullPath = fullPaths[parentJsonPtr] + '/' + parentKeyword;
	    if (keyIndex !== undefined)
	      fullPath += '/' + (typeof keyIndex == 'number' ? keyIndex : util.escapeFragment(keyIndex));

	    if (typeof id == 'string') {
	      id = baseId = normalizeId(baseId ? url.resolve(baseId, id) : id);

	      var refVal = self._refs[id];
	      if (typeof refVal == 'string') refVal = self._refs[refVal];
	      if (refVal && refVal.schema) {
	        if (!equal(sch, refVal.schema))
	          throw new Error('id "' + id + '" resolves to more than one schema');
	      } else if (id != normalizeId(fullPath)) {
	        if (id[0] == '#') {
	          if (localRefs[id] && !equal(sch, localRefs[id]))
	            throw new Error('id "' + id + '" resolves to more than one schema');
	          localRefs[id] = sch;
	        } else {
	          self._refs[id] = fullPath;
	        }
	      }
	    }
	    baseIds[jsonPtr] = baseId;
	    fullPaths[jsonPtr] = fullPath;
	  });

	  return localRefs;
	}


/***/ },
/* 4 */
/***/ function(module, exports, __webpack_require__) {

	// Copyright Joyent, Inc. and other Node contributors.
	//
	// Permission is hereby granted, free of charge, to any person obtaining a
	// copy of this software and associated documentation files (the
	// "Software"), to deal in the Software without restriction, including
	// without limitation the rights to use, copy, modify, merge, publish,
	// distribute, sublicense, and/or sell copies of the Software, and to permit
	// persons to whom the Software is furnished to do so, subject to the
	// following conditions:
	//
	// The above copyright notice and this permission notice shall be included
	// in all copies or substantial portions of the Software.
	//
	// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
	// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
	// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
	// NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
	// DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
	// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
	// USE OR OTHER DEALINGS IN THE SOFTWARE.

	var punycode = __webpack_require__(5);

	exports.parse = urlParse;
	exports.resolve = urlResolve;
	exports.resolveObject = urlResolveObject;
	exports.format = urlFormat;

	exports.Url = Url;

	function Url() {
	  this.protocol = null;
	  this.slashes = null;
	  this.auth = null;
	  this.host = null;
	  this.port = null;
	  this.hostname = null;
	  this.hash = null;
	  this.search = null;
	  this.query = null;
	  this.pathname = null;
	  this.path = null;
	  this.href = null;
	}

	// Reference: RFC 3986, RFC 1808, RFC 2396

	// define these here so at least they only have to be
	// compiled once on the first module load.
	var protocolPattern = /^([a-z0-9.+-]+:)/i,
	    portPattern = /:[0-9]*$/,

	    // RFC 2396: characters reserved for delimiting URLs.
	    // We actually just auto-escape these.
	    delims = ['<', '>', '"', '`', ' ', '\r', '\n', '\t'],

	    // RFC 2396: characters not allowed for various reasons.
	    unwise = ['{', '}', '|', '\\', '^', '`'].concat(delims),

	    // Allowed by RFCs, but cause of XSS attacks.  Always escape these.
	    autoEscape = ['\''].concat(unwise),
	    // Characters that are never ever allowed in a hostname.
	    // Note that any invalid chars are also handled, but these
	    // are the ones that are *expected* to be seen, so we fast-path
	    // them.
	    nonHostChars = ['%', '/', '?', ';', '#'].concat(autoEscape),
	    hostEndingChars = ['/', '?', '#'],
	    hostnameMaxLen = 255,
	    hostnamePartPattern = /^[a-z0-9A-Z_-]{0,63}$/,
	    hostnamePartStart = /^([a-z0-9A-Z_-]{0,63})(.*)$/,
	    // protocols that can allow "unsafe" and "unwise" chars.
	    unsafeProtocol = {
	      'javascript': true,
	      'javascript:': true
	    },
	    // protocols that never have a hostname.
	    hostlessProtocol = {
	      'javascript': true,
	      'javascript:': true
	    },
	    // protocols that always contain a // bit.
	    slashedProtocol = {
	      'http': true,
	      'https': true,
	      'ftp': true,
	      'gopher': true,
	      'file': true,
	      'http:': true,
	      'https:': true,
	      'ftp:': true,
	      'gopher:': true,
	      'file:': true
	    },
	    querystring = __webpack_require__(7);

	function urlParse(url, parseQueryString, slashesDenoteHost) {
	  if (url && isObject(url) && url instanceof Url) return url;

	  var u = new Url;
	  u.parse(url, parseQueryString, slashesDenoteHost);
	  return u;
	}

	Url.prototype.parse = function(url, parseQueryString, slashesDenoteHost) {
	  if (!isString(url)) {
	    throw new TypeError("Parameter 'url' must be a string, not " + typeof url);
	  }

	  var rest = url;

	  // trim before proceeding.
	  // This is to support parse stuff like "  http://foo.com  \n"
	  rest = rest.trim();

	  var proto = protocolPattern.exec(rest);
	  if (proto) {
	    proto = proto[0];
	    var lowerProto = proto.toLowerCase();
	    this.protocol = lowerProto;
	    rest = rest.substr(proto.length);
	  }

	  // figure out if it's got a host
	  // user@server is *always* interpreted as a hostname, and url
	  // resolution will treat //foo/bar as host=foo,path=bar because that's
	  // how the browser resolves relative URLs.
	  if (slashesDenoteHost || proto || rest.match(/^\/\/[^@\/]+@[^@\/]+/)) {
	    var slashes = rest.substr(0, 2) === '//';
	    if (slashes && !(proto && hostlessProtocol[proto])) {
	      rest = rest.substr(2);
	      this.slashes = true;
	    }
	  }

	  if (!hostlessProtocol[proto] &&
	      (slashes || (proto && !slashedProtocol[proto]))) {

	    // there's a hostname.
	    // the first instance of /, ?, ;, or # ends the host.
	    //
	    // If there is an @ in the hostname, then non-host chars *are* allowed
	    // to the left of the last @ sign, unless some host-ending character
	    // comes *before* the @-sign.
	    // URLs are obnoxious.
	    //
	    // ex:
	    // http://a@b@c/ => user:a@b host:c
	    // http://a@b?@c => user:a host:c path:/?@c

	    // v0.12 TODO(isaacs): This is not quite how Chrome does things.
	    // Review our test case against browsers more comprehensively.

	    // find the first instance of any hostEndingChars
	    var hostEnd = -1;
	    for (var i = 0; i < hostEndingChars.length; i++) {
	      var hec = rest.indexOf(hostEndingChars[i]);
	      if (hec !== -1 && (hostEnd === -1 || hec < hostEnd))
	        hostEnd = hec;
	    }

	    // at this point, either we have an explicit point where the
	    // auth portion cannot go past, or the last @ char is the decider.
	    var auth, atSign;
	    if (hostEnd === -1) {
	      // atSign can be anywhere.
	      atSign = rest.lastIndexOf('@');
	    } else {
	      // atSign must be in auth portion.
	      // http://a@b/c@d => host:b auth:a path:/c@d
	      atSign = rest.lastIndexOf('@', hostEnd);
	    }

	    // Now we have a portion which is definitely the auth.
	    // Pull that off.
	    if (atSign !== -1) {
	      auth = rest.slice(0, atSign);
	      rest = rest.slice(atSign + 1);
	      this.auth = decodeURIComponent(auth);
	    }

	    // the host is the remaining to the left of the first non-host char
	    hostEnd = -1;
	    for (var i = 0; i < nonHostChars.length; i++) {
	      var hec = rest.indexOf(nonHostChars[i]);
	      if (hec !== -1 && (hostEnd === -1 || hec < hostEnd))
	        hostEnd = hec;
	    }
	    // if we still have not hit it, then the entire thing is a host.
	    if (hostEnd === -1)
	      hostEnd = rest.length;

	    this.host = rest.slice(0, hostEnd);
	    rest = rest.slice(hostEnd);

	    // pull out port.
	    this.parseHost();

	    // we've indicated that there is a hostname,
	    // so even if it's empty, it has to be present.
	    this.hostname = this.hostname || '';

	    // if hostname begins with [ and ends with ]
	    // assume that it's an IPv6 address.
	    var ipv6Hostname = this.hostname[0] === '[' &&
	        this.hostname[this.hostname.length - 1] === ']';

	    // validate a little.
	    if (!ipv6Hostname) {
	      var hostparts = this.hostname.split(/\./);
	      for (var i = 0, l = hostparts.length; i < l; i++) {
	        var part = hostparts[i];
	        if (!part) continue;
	        if (!part.match(hostnamePartPattern)) {
	          var newpart = '';
	          for (var j = 0, k = part.length; j < k; j++) {
	            if (part.charCodeAt(j) > 127) {
	              // we replace non-ASCII char with a temporary placeholder
	              // we need this to make sure size of hostname is not
	              // broken by replacing non-ASCII by nothing
	              newpart += 'x';
	            } else {
	              newpart += part[j];
	            }
	          }
	          // we test again with ASCII char only
	          if (!newpart.match(hostnamePartPattern)) {
	            var validParts = hostparts.slice(0, i);
	            var notHost = hostparts.slice(i + 1);
	            var bit = part.match(hostnamePartStart);
	            if (bit) {
	              validParts.push(bit[1]);
	              notHost.unshift(bit[2]);
	            }
	            if (notHost.length) {
	              rest = '/' + notHost.join('.') + rest;
	            }
	            this.hostname = validParts.join('.');
	            break;
	          }
	        }
	      }
	    }

	    if (this.hostname.length > hostnameMaxLen) {
	      this.hostname = '';
	    } else {
	      // hostnames are always lower case.
	      this.hostname = this.hostname.toLowerCase();
	    }

	    if (!ipv6Hostname) {
	      // IDNA Support: Returns a puny coded representation of "domain".
	      // It only converts the part of the domain name that
	      // has non ASCII characters. I.e. it dosent matter if
	      // you call it with a domain that already is in ASCII.
	      var domainArray = this.hostname.split('.');
	      var newOut = [];
	      for (var i = 0; i < domainArray.length; ++i) {
	        var s = domainArray[i];
	        newOut.push(s.match(/[^A-Za-z0-9_-]/) ?
	            'xn--' + punycode.encode(s) : s);
	      }
	      this.hostname = newOut.join('.');
	    }

	    var p = this.port ? ':' + this.port : '';
	    var h = this.hostname || '';
	    this.host = h + p;
	    this.href += this.host;

	    // strip [ and ] from the hostname
	    // the host field still retains them, though
	    if (ipv6Hostname) {
	      this.hostname = this.hostname.substr(1, this.hostname.length - 2);
	      if (rest[0] !== '/') {
	        rest = '/' + rest;
	      }
	    }
	  }

	  // now rest is set to the post-host stuff.
	  // chop off any delim chars.
	  if (!unsafeProtocol[lowerProto]) {

	    // First, make 100% sure that any "autoEscape" chars get
	    // escaped, even if encodeURIComponent doesn't think they
	    // need to be.
	    for (var i = 0, l = autoEscape.length; i < l; i++) {
	      var ae = autoEscape[i];
	      var esc = encodeURIComponent(ae);
	      if (esc === ae) {
	        esc = escape(ae);
	      }
	      rest = rest.split(ae).join(esc);
	    }
	  }


	  // chop off from the tail first.
	  var hash = rest.indexOf('#');
	  if (hash !== -1) {
	    // got a fragment string.
	    this.hash = rest.substr(hash);
	    rest = rest.slice(0, hash);
	  }
	  var qm = rest.indexOf('?');
	  if (qm !== -1) {
	    this.search = rest.substr(qm);
	    this.query = rest.substr(qm + 1);
	    if (parseQueryString) {
	      this.query = querystring.parse(this.query);
	    }
	    rest = rest.slice(0, qm);
	  } else if (parseQueryString) {
	    // no query string, but parseQueryString still requested
	    this.search = '';
	    this.query = {};
	  }
	  if (rest) this.pathname = rest;
	  if (slashedProtocol[lowerProto] &&
	      this.hostname && !this.pathname) {
	    this.pathname = '/';
	  }

	  //to support http.request
	  if (this.pathname || this.search) {
	    var p = this.pathname || '';
	    var s = this.search || '';
	    this.path = p + s;
	  }

	  // finally, reconstruct the href based on what has been validated.
	  this.href = this.format();
	  return this;
	};

	// format a parsed object into a url string
	function urlFormat(obj) {
	  // ensure it's an object, and not a string url.
	  // If it's an obj, this is a no-op.
	  // this way, you can call url_format() on strings
	  // to clean up potentially wonky urls.
	  if (isString(obj)) obj = urlParse(obj);
	  if (!(obj instanceof Url)) return Url.prototype.format.call(obj);
	  return obj.format();
	}

	Url.prototype.format = function() {
	  var auth = this.auth || '';
	  if (auth) {
	    auth = encodeURIComponent(auth);
	    auth = auth.replace(/%3A/i, ':');
	    auth += '@';
	  }

	  var protocol = this.protocol || '',
	      pathname = this.pathname || '',
	      hash = this.hash || '',
	      host = false,
	      query = '';

	  if (this.host) {
	    host = auth + this.host;
	  } else if (this.hostname) {
	    host = auth + (this.hostname.indexOf(':') === -1 ?
	        this.hostname :
	        '[' + this.hostname + ']');
	    if (this.port) {
	      host += ':' + this.port;
	    }
	  }

	  if (this.query &&
	      isObject(this.query) &&
	      Object.keys(this.query).length) {
	    query = querystring.stringify(this.query);
	  }

	  var search = this.search || (query && ('?' + query)) || '';

	  if (protocol && protocol.substr(-1) !== ':') protocol += ':';

	  // only the slashedProtocols get the //.  Not mailto:, xmpp:, etc.
	  // unless they had them to begin with.
	  if (this.slashes ||
	      (!protocol || slashedProtocol[protocol]) && host !== false) {
	    host = '//' + (host || '');
	    if (pathname && pathname.charAt(0) !== '/') pathname = '/' + pathname;
	  } else if (!host) {
	    host = '';
	  }

	  if (hash && hash.charAt(0) !== '#') hash = '#' + hash;
	  if (search && search.charAt(0) !== '?') search = '?' + search;

	  pathname = pathname.replace(/[?#]/g, function(match) {
	    return encodeURIComponent(match);
	  });
	  search = search.replace('#', '%23');

	  return protocol + host + pathname + search + hash;
	};

	function urlResolve(source, relative) {
	  return urlParse(source, false, true).resolve(relative);
	}

	Url.prototype.resolve = function(relative) {
	  return this.resolveObject(urlParse(relative, false, true)).format();
	};

	function urlResolveObject(source, relative) {
	  if (!source) return relative;
	  return urlParse(source, false, true).resolveObject(relative);
	}

	Url.prototype.resolveObject = function(relative) {
	  if (isString(relative)) {
	    var rel = new Url();
	    rel.parse(relative, false, true);
	    relative = rel;
	  }

	  var result = new Url();
	  Object.keys(this).forEach(function(k) {
	    result[k] = this[k];
	  }, this);

	  // hash is always overridden, no matter what.
	  // even href="" will remove it.
	  result.hash = relative.hash;

	  // if the relative url is empty, then there's nothing left to do here.
	  if (relative.href === '') {
	    result.href = result.format();
	    return result;
	  }

	  // hrefs like //foo/bar always cut to the protocol.
	  if (relative.slashes && !relative.protocol) {
	    // take everything except the protocol from relative
	    Object.keys(relative).forEach(function(k) {
	      if (k !== 'protocol')
	        result[k] = relative[k];
	    });

	    //urlParse appends trailing / to urls like http://www.example.com
	    if (slashedProtocol[result.protocol] &&
	        result.hostname && !result.pathname) {
	      result.path = result.pathname = '/';
	    }

	    result.href = result.format();
	    return result;
	  }

	  if (relative.protocol && relative.protocol !== result.protocol) {
	    // if it's a known url protocol, then changing
	    // the protocol does weird things
	    // first, if it's not file:, then we MUST have a host,
	    // and if there was a path
	    // to begin with, then we MUST have a path.
	    // if it is file:, then the host is dropped,
	    // because that's known to be hostless.
	    // anything else is assumed to be absolute.
	    if (!slashedProtocol[relative.protocol]) {
	      Object.keys(relative).forEach(function(k) {
	        result[k] = relative[k];
	      });
	      result.href = result.format();
	      return result;
	    }

	    result.protocol = relative.protocol;
	    if (!relative.host && !hostlessProtocol[relative.protocol]) {
	      var relPath = (relative.pathname || '').split('/');
	      while (relPath.length && !(relative.host = relPath.shift()));
	      if (!relative.host) relative.host = '';
	      if (!relative.hostname) relative.hostname = '';
	      if (relPath[0] !== '') relPath.unshift('');
	      if (relPath.length < 2) relPath.unshift('');
	      result.pathname = relPath.join('/');
	    } else {
	      result.pathname = relative.pathname;
	    }
	    result.search = relative.search;
	    result.query = relative.query;
	    result.host = relative.host || '';
	    result.auth = relative.auth;
	    result.hostname = relative.hostname || relative.host;
	    result.port = relative.port;
	    // to support http.request
	    if (result.pathname || result.search) {
	      var p = result.pathname || '';
	      var s = result.search || '';
	      result.path = p + s;
	    }
	    result.slashes = result.slashes || relative.slashes;
	    result.href = result.format();
	    return result;
	  }

	  var isSourceAbs = (result.pathname && result.pathname.charAt(0) === '/'),
	      isRelAbs = (
	          relative.host ||
	          relative.pathname && relative.pathname.charAt(0) === '/'
	      ),
	      mustEndAbs = (isRelAbs || isSourceAbs ||
	                    (result.host && relative.pathname)),
	      removeAllDots = mustEndAbs,
	      srcPath = result.pathname && result.pathname.split('/') || [],
	      relPath = relative.pathname && relative.pathname.split('/') || [],
	      psychotic = result.protocol && !slashedProtocol[result.protocol];

	  // if the url is a non-slashed url, then relative
	  // links like ../.. should be able
	  // to crawl up to the hostname, as well.  This is strange.
	  // result.protocol has already been set by now.
	  // Later on, put the first path part into the host field.
	  if (psychotic) {
	    result.hostname = '';
	    result.port = null;
	    if (result.host) {
	      if (srcPath[0] === '') srcPath[0] = result.host;
	      else srcPath.unshift(result.host);
	    }
	    result.host = '';
	    if (relative.protocol) {
	      relative.hostname = null;
	      relative.port = null;
	      if (relative.host) {
	        if (relPath[0] === '') relPath[0] = relative.host;
	        else relPath.unshift(relative.host);
	      }
	      relative.host = null;
	    }
	    mustEndAbs = mustEndAbs && (relPath[0] === '' || srcPath[0] === '');
	  }

	  if (isRelAbs) {
	    // it's absolute.
	    result.host = (relative.host || relative.host === '') ?
	                  relative.host : result.host;
	    result.hostname = (relative.hostname || relative.hostname === '') ?
	                      relative.hostname : result.hostname;
	    result.search = relative.search;
	    result.query = relative.query;
	    srcPath = relPath;
	    // fall through to the dot-handling below.
	  } else if (relPath.length) {
	    // it's relative
	    // throw away the existing file, and take the new path instead.
	    if (!srcPath) srcPath = [];
	    srcPath.pop();
	    srcPath = srcPath.concat(relPath);
	    result.search = relative.search;
	    result.query = relative.query;
	  } else if (!isNullOrUndefined(relative.search)) {
	    // just pull out the search.
	    // like href='?foo'.
	    // Put this after the other two cases because it simplifies the booleans
	    if (psychotic) {
	      result.hostname = result.host = srcPath.shift();
	      //occationaly the auth can get stuck only in host
	      //this especialy happens in cases like
	      //url.resolveObject('mailto:local1@domain1', 'local2@domain2')
	      var authInHost = result.host && result.host.indexOf('@') > 0 ?
	                       result.host.split('@') : false;
	      if (authInHost) {
	        result.auth = authInHost.shift();
	        result.host = result.hostname = authInHost.shift();
	      }
	    }
	    result.search = relative.search;
	    result.query = relative.query;
	    //to support http.request
	    if (!isNull(result.pathname) || !isNull(result.search)) {
	      result.path = (result.pathname ? result.pathname : '') +
	                    (result.search ? result.search : '');
	    }
	    result.href = result.format();
	    return result;
	  }

	  if (!srcPath.length) {
	    // no path at all.  easy.
	    // we've already handled the other stuff above.
	    result.pathname = null;
	    //to support http.request
	    if (result.search) {
	      result.path = '/' + result.search;
	    } else {
	      result.path = null;
	    }
	    result.href = result.format();
	    return result;
	  }

	  // if a url ENDs in . or .., then it must get a trailing slash.
	  // however, if it ends in anything else non-slashy,
	  // then it must NOT get a trailing slash.
	  var last = srcPath.slice(-1)[0];
	  var hasTrailingSlash = (
	      (result.host || relative.host) && (last === '.' || last === '..') ||
	      last === '');

	  // strip single dots, resolve double dots to parent dir
	  // if the path tries to go above the root, `up` ends up > 0
	  var up = 0;
	  for (var i = srcPath.length; i >= 0; i--) {
	    last = srcPath[i];
	    if (last == '.') {
	      srcPath.splice(i, 1);
	    } else if (last === '..') {
	      srcPath.splice(i, 1);
	      up++;
	    } else if (up) {
	      srcPath.splice(i, 1);
	      up--;
	    }
	  }

	  // if the path is allowed to go above the root, restore leading ..s
	  if (!mustEndAbs && !removeAllDots) {
	    for (; up--; up) {
	      srcPath.unshift('..');
	    }
	  }

	  if (mustEndAbs && srcPath[0] !== '' &&
	      (!srcPath[0] || srcPath[0].charAt(0) !== '/')) {
	    srcPath.unshift('');
	  }

	  if (hasTrailingSlash && (srcPath.join('/').substr(-1) !== '/')) {
	    srcPath.push('');
	  }

	  var isAbsolute = srcPath[0] === '' ||
	      (srcPath[0] && srcPath[0].charAt(0) === '/');

	  // put the host back
	  if (psychotic) {
	    result.hostname = result.host = isAbsolute ? '' :
	                                    srcPath.length ? srcPath.shift() : '';
	    //occationaly the auth can get stuck only in host
	    //this especialy happens in cases like
	    //url.resolveObject('mailto:local1@domain1', 'local2@domain2')
	    var authInHost = result.host && result.host.indexOf('@') > 0 ?
	                     result.host.split('@') : false;
	    if (authInHost) {
	      result.auth = authInHost.shift();
	      result.host = result.hostname = authInHost.shift();
	    }
	  }

	  mustEndAbs = mustEndAbs || (result.host && srcPath.length);

	  if (mustEndAbs && !isAbsolute) {
	    srcPath.unshift('');
	  }

	  if (!srcPath.length) {
	    result.pathname = null;
	    result.path = null;
	  } else {
	    result.pathname = srcPath.join('/');
	  }

	  //to support request.http
	  if (!isNull(result.pathname) || !isNull(result.search)) {
	    result.path = (result.pathname ? result.pathname : '') +
	                  (result.search ? result.search : '');
	  }
	  result.auth = relative.auth || result.auth;
	  result.slashes = result.slashes || relative.slashes;
	  result.href = result.format();
	  return result;
	};

	Url.prototype.parseHost = function() {
	  var host = this.host;
	  var port = portPattern.exec(host);
	  if (port) {
	    port = port[0];
	    if (port !== ':') {
	      this.port = port.substr(1);
	    }
	    host = host.substr(0, host.length - port.length);
	  }
	  if (host) this.hostname = host;
	};

	function isString(arg) {
	  return typeof arg === "string";
	}

	function isObject(arg) {
	  return typeof arg === 'object' && arg !== null;
	}

	function isNull(arg) {
	  return arg === null;
	}
	function isNullOrUndefined(arg) {
	  return  arg == null;
	}


/***/ },
/* 5 */
/***/ function(module, exports, __webpack_require__) {

	var __WEBPACK_AMD_DEFINE_RESULT__;/* WEBPACK VAR INJECTION */(function(module, global) {/*! https://mths.be/punycode v1.3.2 by @mathias */
	;(function(root) {

		/** Detect free variables */
		var freeExports = typeof exports == 'object' && exports &&
			!exports.nodeType && exports;
		var freeModule = typeof module == 'object' && module &&
			!module.nodeType && module;
		var freeGlobal = typeof global == 'object' && global;
		if (
			freeGlobal.global === freeGlobal ||
			freeGlobal.window === freeGlobal ||
			freeGlobal.self === freeGlobal
		) {
			root = freeGlobal;
		}

		/**
		 * The `punycode` object.
		 * @name punycode
		 * @type Object
		 */
		var punycode,

		/** Highest positive signed 32-bit float value */
		maxInt = 2147483647, // aka. 0x7FFFFFFF or 2^31-1

		/** Bootstring parameters */
		base = 36,
		tMin = 1,
		tMax = 26,
		skew = 38,
		damp = 700,
		initialBias = 72,
		initialN = 128, // 0x80
		delimiter = '-', // '\x2D'

		/** Regular expressions */
		regexPunycode = /^xn--/,
		regexNonASCII = /[^\x20-\x7E]/, // unprintable ASCII chars + non-ASCII chars
		regexSeparators = /[\x2E\u3002\uFF0E\uFF61]/g, // RFC 3490 separators

		/** Error messages */
		errors = {
			'overflow': 'Overflow: input needs wider integers to process',
			'not-basic': 'Illegal input >= 0x80 (not a basic code point)',
			'invalid-input': 'Invalid input'
		},

		/** Convenience shortcuts */
		baseMinusTMin = base - tMin,
		floor = Math.floor,
		stringFromCharCode = String.fromCharCode,

		/** Temporary variable */
		key;

		/*--------------------------------------------------------------------------*/

		/**
		 * A generic error utility function.
		 * @private
		 * @param {String} type The error type.
		 * @returns {Error} Throws a `RangeError` with the applicable error message.
		 */
		function error(type) {
			throw RangeError(errors[type]);
		}

		/**
		 * A generic `Array#map` utility function.
		 * @private
		 * @param {Array} array The array to iterate over.
		 * @param {Function} callback The function that gets called for every array
		 * item.
		 * @returns {Array} A new array of values returned by the callback function.
		 */
		function map(array, fn) {
			var length = array.length;
			var result = [];
			while (length--) {
				result[length] = fn(array[length]);
			}
			return result;
		}

		/**
		 * A simple `Array#map`-like wrapper to work with domain name strings or email
		 * addresses.
		 * @private
		 * @param {String} domain The domain name or email address.
		 * @param {Function} callback The function that gets called for every
		 * character.
		 * @returns {Array} A new string of characters returned by the callback
		 * function.
		 */
		function mapDomain(string, fn) {
			var parts = string.split('@');
			var result = '';
			if (parts.length > 1) {
				// In email addresses, only the domain name should be punycoded. Leave
				// the local part (i.e. everything up to `@`) intact.
				result = parts[0] + '@';
				string = parts[1];
			}
			// Avoid `split(regex)` for IE8 compatibility. See #17.
			string = string.replace(regexSeparators, '\x2E');
			var labels = string.split('.');
			var encoded = map(labels, fn).join('.');
			return result + encoded;
		}

		/**
		 * Creates an array containing the numeric code points of each Unicode
		 * character in the string. While JavaScript uses UCS-2 internally,
		 * this function will convert a pair of surrogate halves (each of which
		 * UCS-2 exposes as separate characters) into a single code point,
		 * matching UTF-16.
		 * @see `punycode.ucs2.encode`
		 * @see <https://mathiasbynens.be/notes/javascript-encoding>
		 * @memberOf punycode.ucs2
		 * @name decode
		 * @param {String} string The Unicode input string (UCS-2).
		 * @returns {Array} The new array of code points.
		 */
		function ucs2decode(string) {
			var output = [],
			    counter = 0,
			    length = string.length,
			    value,
			    extra;
			while (counter < length) {
				value = string.charCodeAt(counter++);
				if (value >= 0xD800 && value <= 0xDBFF && counter < length) {
					// high surrogate, and there is a next character
					extra = string.charCodeAt(counter++);
					if ((extra & 0xFC00) == 0xDC00) { // low surrogate
						output.push(((value & 0x3FF) << 10) + (extra & 0x3FF) + 0x10000);
					} else {
						// unmatched surrogate; only append this code unit, in case the next
						// code unit is the high surrogate of a surrogate pair
						output.push(value);
						counter--;
					}
				} else {
					output.push(value);
				}
			}
			return output;
		}

		/**
		 * Creates a string based on an array of numeric code points.
		 * @see `punycode.ucs2.decode`
		 * @memberOf punycode.ucs2
		 * @name encode
		 * @param {Array} codePoints The array of numeric code points.
		 * @returns {String} The new Unicode string (UCS-2).
		 */
		function ucs2encode(array) {
			return map(array, function(value) {
				var output = '';
				if (value > 0xFFFF) {
					value -= 0x10000;
					output += stringFromCharCode(value >>> 10 & 0x3FF | 0xD800);
					value = 0xDC00 | value & 0x3FF;
				}
				output += stringFromCharCode(value);
				return output;
			}).join('');
		}

		/**
		 * Converts a basic code point into a digit/integer.
		 * @see `digitToBasic()`
		 * @private
		 * @param {Number} codePoint The basic numeric code point value.
		 * @returns {Number} The numeric value of a basic code point (for use in
		 * representing integers) in the range `0` to `base - 1`, or `base` if
		 * the code point does not represent a value.
		 */
		function basicToDigit(codePoint) {
			if (codePoint - 48 < 10) {
				return codePoint - 22;
			}
			if (codePoint - 65 < 26) {
				return codePoint - 65;
			}
			if (codePoint - 97 < 26) {
				return codePoint - 97;
			}
			return base;
		}

		/**
		 * Converts a digit/integer into a basic code point.
		 * @see `basicToDigit()`
		 * @private
		 * @param {Number} digit The numeric value of a basic code point.
		 * @returns {Number} The basic code point whose value (when used for
		 * representing integers) is `digit`, which needs to be in the range
		 * `0` to `base - 1`. If `flag` is non-zero, the uppercase form is
		 * used; else, the lowercase form is used. The behavior is undefined
		 * if `flag` is non-zero and `digit` has no uppercase form.
		 */
		function digitToBasic(digit, flag) {
			//  0..25 map to ASCII a..z or A..Z
			// 26..35 map to ASCII 0..9
			return digit + 22 + 75 * (digit < 26) - ((flag != 0) << 5);
		}

		/**
		 * Bias adaptation function as per section 3.4 of RFC 3492.
		 * http://tools.ietf.org/html/rfc3492#section-3.4
		 * @private
		 */
		function adapt(delta, numPoints, firstTime) {
			var k = 0;
			delta = firstTime ? floor(delta / damp) : delta >> 1;
			delta += floor(delta / numPoints);
			for (/* no initialization */; delta > baseMinusTMin * tMax >> 1; k += base) {
				delta = floor(delta / baseMinusTMin);
			}
			return floor(k + (baseMinusTMin + 1) * delta / (delta + skew));
		}

		/**
		 * Converts a Punycode string of ASCII-only symbols to a string of Unicode
		 * symbols.
		 * @memberOf punycode
		 * @param {String} input The Punycode string of ASCII-only symbols.
		 * @returns {String} The resulting string of Unicode symbols.
		 */
		function decode(input) {
			// Don't use UCS-2
			var output = [],
			    inputLength = input.length,
			    out,
			    i = 0,
			    n = initialN,
			    bias = initialBias,
			    basic,
			    j,
			    index,
			    oldi,
			    w,
			    k,
			    digit,
			    t,
			    /** Cached calculation results */
			    baseMinusT;

			// Handle the basic code points: let `basic` be the number of input code
			// points before the last delimiter, or `0` if there is none, then copy
			// the first basic code points to the output.

			basic = input.lastIndexOf(delimiter);
			if (basic < 0) {
				basic = 0;
			}

			for (j = 0; j < basic; ++j) {
				// if it's not a basic code point
				if (input.charCodeAt(j) >= 0x80) {
					error('not-basic');
				}
				output.push(input.charCodeAt(j));
			}

			// Main decoding loop: start just after the last delimiter if any basic code
			// points were copied; start at the beginning otherwise.

			for (index = basic > 0 ? basic + 1 : 0; index < inputLength; /* no final expression */) {

				// `index` is the index of the next character to be consumed.
				// Decode a generalized variable-length integer into `delta`,
				// which gets added to `i`. The overflow checking is easier
				// if we increase `i` as we go, then subtract off its starting
				// value at the end to obtain `delta`.
				for (oldi = i, w = 1, k = base; /* no condition */; k += base) {

					if (index >= inputLength) {
						error('invalid-input');
					}

					digit = basicToDigit(input.charCodeAt(index++));

					if (digit >= base || digit > floor((maxInt - i) / w)) {
						error('overflow');
					}

					i += digit * w;
					t = k <= bias ? tMin : (k >= bias + tMax ? tMax : k - bias);

					if (digit < t) {
						break;
					}

					baseMinusT = base - t;
					if (w > floor(maxInt / baseMinusT)) {
						error('overflow');
					}

					w *= baseMinusT;

				}

				out = output.length + 1;
				bias = adapt(i - oldi, out, oldi == 0);

				// `i` was supposed to wrap around from `out` to `0`,
				// incrementing `n` each time, so we'll fix that now:
				if (floor(i / out) > maxInt - n) {
					error('overflow');
				}

				n += floor(i / out);
				i %= out;

				// Insert `n` at position `i` of the output
				output.splice(i++, 0, n);

			}

			return ucs2encode(output);
		}

		/**
		 * Converts a string of Unicode symbols (e.g. a domain name label) to a
		 * Punycode string of ASCII-only symbols.
		 * @memberOf punycode
		 * @param {String} input The string of Unicode symbols.
		 * @returns {String} The resulting Punycode string of ASCII-only symbols.
		 */
		function encode(input) {
			var n,
			    delta,
			    handledCPCount,
			    basicLength,
			    bias,
			    j,
			    m,
			    q,
			    k,
			    t,
			    currentValue,
			    output = [],
			    /** `inputLength` will hold the number of code points in `input`. */
			    inputLength,
			    /** Cached calculation results */
			    handledCPCountPlusOne,
			    baseMinusT,
			    qMinusT;

			// Convert the input in UCS-2 to Unicode
			input = ucs2decode(input);

			// Cache the length
			inputLength = input.length;

			// Initialize the state
			n = initialN;
			delta = 0;
			bias = initialBias;

			// Handle the basic code points
			for (j = 0; j < inputLength; ++j) {
				currentValue = input[j];
				if (currentValue < 0x80) {
					output.push(stringFromCharCode(currentValue));
				}
			}

			handledCPCount = basicLength = output.length;

			// `handledCPCount` is the number of code points that have been handled;
			// `basicLength` is the number of basic code points.

			// Finish the basic string - if it is not empty - with a delimiter
			if (basicLength) {
				output.push(delimiter);
			}

			// Main encoding loop:
			while (handledCPCount < inputLength) {

				// All non-basic code points < n have been handled already. Find the next
				// larger one:
				for (m = maxInt, j = 0; j < inputLength; ++j) {
					currentValue = input[j];
					if (currentValue >= n && currentValue < m) {
						m = currentValue;
					}
				}

				// Increase `delta` enough to advance the decoder's <n,i> state to <m,0>,
				// but guard against overflow
				handledCPCountPlusOne = handledCPCount + 1;
				if (m - n > floor((maxInt - delta) / handledCPCountPlusOne)) {
					error('overflow');
				}

				delta += (m - n) * handledCPCountPlusOne;
				n = m;

				for (j = 0; j < inputLength; ++j) {
					currentValue = input[j];

					if (currentValue < n && ++delta > maxInt) {
						error('overflow');
					}

					if (currentValue == n) {
						// Represent delta as a generalized variable-length integer
						for (q = delta, k = base; /* no condition */; k += base) {
							t = k <= bias ? tMin : (k >= bias + tMax ? tMax : k - bias);
							if (q < t) {
								break;
							}
							qMinusT = q - t;
							baseMinusT = base - t;
							output.push(
								stringFromCharCode(digitToBasic(t + qMinusT % baseMinusT, 0))
							);
							q = floor(qMinusT / baseMinusT);
						}

						output.push(stringFromCharCode(digitToBasic(q, 0)));
						bias = adapt(delta, handledCPCountPlusOne, handledCPCount == basicLength);
						delta = 0;
						++handledCPCount;
					}
				}

				++delta;
				++n;

			}
			return output.join('');
		}

		/**
		 * Converts a Punycode string representing a domain name or an email address
		 * to Unicode. Only the Punycoded parts of the input will be converted, i.e.
		 * it doesn't matter if you call it on a string that has already been
		 * converted to Unicode.
		 * @memberOf punycode
		 * @param {String} input The Punycoded domain name or email address to
		 * convert to Unicode.
		 * @returns {String} The Unicode representation of the given Punycode
		 * string.
		 */
		function toUnicode(input) {
			return mapDomain(input, function(string) {
				return regexPunycode.test(string)
					? decode(string.slice(4).toLowerCase())
					: string;
			});
		}

		/**
		 * Converts a Unicode string representing a domain name or an email address to
		 * Punycode. Only the non-ASCII parts of the domain name will be converted,
		 * i.e. it doesn't matter if you call it with a domain that's already in
		 * ASCII.
		 * @memberOf punycode
		 * @param {String} input The domain name or email address to convert, as a
		 * Unicode string.
		 * @returns {String} The Punycode representation of the given domain name or
		 * email address.
		 */
		function toASCII(input) {
			return mapDomain(input, function(string) {
				return regexNonASCII.test(string)
					? 'xn--' + encode(string)
					: string;
			});
		}

		/*--------------------------------------------------------------------------*/

		/** Define the public API */
		punycode = {
			/**
			 * A string representing the current Punycode.js version number.
			 * @memberOf punycode
			 * @type String
			 */
			'version': '1.3.2',
			/**
			 * An object of methods to convert from JavaScript's internal character
			 * representation (UCS-2) to Unicode code points, and back.
			 * @see <https://mathiasbynens.be/notes/javascript-encoding>
			 * @memberOf punycode
			 * @type Object
			 */
			'ucs2': {
				'decode': ucs2decode,
				'encode': ucs2encode
			},
			'decode': decode,
			'encode': encode,
			'toASCII': toASCII,
			'toUnicode': toUnicode
		};

		/** Expose `punycode` */
		// Some AMD build optimizers, like r.js, check for specific condition patterns
		// like the following:
		if (
			true
		) {
			!(__WEBPACK_AMD_DEFINE_RESULT__ = function() {
				return punycode;
			}.call(exports, __webpack_require__, exports, module), __WEBPACK_AMD_DEFINE_RESULT__ !== undefined && (module.exports = __WEBPACK_AMD_DEFINE_RESULT__));
		} else if (freeExports && freeModule) {
			if (module.exports == freeExports) { // in Node.js or RingoJS v0.8.0+
				freeModule.exports = punycode;
			} else { // in Narwhal or RingoJS v0.7.0-
				for (key in punycode) {
					punycode.hasOwnProperty(key) && (freeExports[key] = punycode[key]);
				}
			}
		} else { // in Rhino or a web browser
			root.punycode = punycode;
		}

	}(this));

	/* WEBPACK VAR INJECTION */}.call(exports, __webpack_require__(6)(module), (function() { return this; }())))

/***/ },
/* 6 */
/***/ function(module, exports) {

	module.exports = function(module) {
		if(!module.webpackPolyfill) {
			module.deprecate = function() {};
			module.paths = [];
			// module.parent = undefined by default
			module.children = [];
			module.webpackPolyfill = 1;
		}
		return module;
	}


/***/ },
/* 7 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	exports.decode = exports.parse = __webpack_require__(8);
	exports.encode = exports.stringify = __webpack_require__(9);


/***/ },
/* 8 */
/***/ function(module, exports) {

	// Copyright Joyent, Inc. and other Node contributors.
	//
	// Permission is hereby granted, free of charge, to any person obtaining a
	// copy of this software and associated documentation files (the
	// "Software"), to deal in the Software without restriction, including
	// without limitation the rights to use, copy, modify, merge, publish,
	// distribute, sublicense, and/or sell copies of the Software, and to permit
	// persons to whom the Software is furnished to do so, subject to the
	// following conditions:
	//
	// The above copyright notice and this permission notice shall be included
	// in all copies or substantial portions of the Software.
	//
	// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
	// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
	// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
	// NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
	// DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
	// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
	// USE OR OTHER DEALINGS IN THE SOFTWARE.

	'use strict';

	// If obj.hasOwnProperty has been overridden, then calling
	// obj.hasOwnProperty(prop) will break.
	// See: https://github.com/joyent/node/issues/1707
	function hasOwnProperty(obj, prop) {
	  return Object.prototype.hasOwnProperty.call(obj, prop);
	}

	module.exports = function(qs, sep, eq, options) {
	  sep = sep || '&';
	  eq = eq || '=';
	  var obj = {};

	  if (typeof qs !== 'string' || qs.length === 0) {
	    return obj;
	  }

	  var regexp = /\+/g;
	  qs = qs.split(sep);

	  var maxKeys = 1000;
	  if (options && typeof options.maxKeys === 'number') {
	    maxKeys = options.maxKeys;
	  }

	  var len = qs.length;
	  // maxKeys <= 0 means that we should not limit keys count
	  if (maxKeys > 0 && len > maxKeys) {
	    len = maxKeys;
	  }

	  for (var i = 0; i < len; ++i) {
	    var x = qs[i].replace(regexp, '%20'),
	        idx = x.indexOf(eq),
	        kstr, vstr, k, v;

	    if (idx >= 0) {
	      kstr = x.substr(0, idx);
	      vstr = x.substr(idx + 1);
	    } else {
	      kstr = x;
	      vstr = '';
	    }

	    k = decodeURIComponent(kstr);
	    v = decodeURIComponent(vstr);

	    if (!hasOwnProperty(obj, k)) {
	      obj[k] = v;
	    } else if (Array.isArray(obj[k])) {
	      obj[k].push(v);
	    } else {
	      obj[k] = [obj[k], v];
	    }
	  }

	  return obj;
	};


/***/ },
/* 9 */
/***/ function(module, exports) {

	// Copyright Joyent, Inc. and other Node contributors.
	//
	// Permission is hereby granted, free of charge, to any person obtaining a
	// copy of this software and associated documentation files (the
	// "Software"), to deal in the Software without restriction, including
	// without limitation the rights to use, copy, modify, merge, publish,
	// distribute, sublicense, and/or sell copies of the Software, and to permit
	// persons to whom the Software is furnished to do so, subject to the
	// following conditions:
	//
	// The above copyright notice and this permission notice shall be included
	// in all copies or substantial portions of the Software.
	//
	// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
	// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
	// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
	// NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
	// DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
	// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
	// USE OR OTHER DEALINGS IN THE SOFTWARE.

	'use strict';

	var stringifyPrimitive = function(v) {
	  switch (typeof v) {
	    case 'string':
	      return v;

	    case 'boolean':
	      return v ? 'true' : 'false';

	    case 'number':
	      return isFinite(v) ? v : '';

	    default:
	      return '';
	  }
	};

	module.exports = function(obj, sep, eq, name) {
	  sep = sep || '&';
	  eq = eq || '=';
	  if (obj === null) {
	    obj = undefined;
	  }

	  if (typeof obj === 'object') {
	    return Object.keys(obj).map(function(k) {
	      var ks = encodeURIComponent(stringifyPrimitive(k)) + eq;
	      if (Array.isArray(obj[k])) {
	        return obj[k].map(function(v) {
	          return ks + encodeURIComponent(stringifyPrimitive(v));
	        }).join(sep);
	      } else {
	        return ks + encodeURIComponent(stringifyPrimitive(obj[k]));
	      }
	    }).join(sep);

	  }

	  if (!name) return '';
	  return encodeURIComponent(stringifyPrimitive(name)) + eq +
	         encodeURIComponent(stringifyPrimitive(obj));
	};


/***/ },
/* 10 */
/***/ function(module, exports) {

	'use strict';

	module.exports = function equal(a, b) {
	  if (a === b) return true;

	  var arrA = Array.isArray(a)
	    , arrB = Array.isArray(b)
	    , i;

	  if (arrA && arrB) {
	    if (a.length != b.length) return false;
	    for (i = 0; i < a.length; i++)
	      if (!equal(a[i], b[i])) return false;
	    return true;
	  }

	  if (arrA != arrB) return false;

	  if (a && b && typeof a === 'object' && typeof b === 'object') {
	    var keys = Object.keys(a);
	    if (keys.length !== Object.keys(b).length) return false;

	    var dateA = a instanceof Date
	      , dateB = b instanceof Date;
	    if (dateA && dateB) return a.getTime() == b.getTime();
	    if (dateA != dateB) return false;

	    var regexpA = a instanceof RegExp
	      , regexpB = b instanceof RegExp;
	    if (regexpA && regexpB) return a.toString() == b.toString();
	    if (regexpA != regexpB) return false;

	    for (i = 0; i < keys.length; i++)
	      if (!Object.prototype.hasOwnProperty.call(b, keys[i])) return false;

	    for (i = 0; i < keys.length; i++)
	      if(!equal(a[keys[i]], b[keys[i]])) return false;

	    return true;
	  }

	  return false;
	};


/***/ },
/* 11 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';


	module.exports = {
	  copy: copy,
	  checkDataType: checkDataType,
	  checkDataTypes: checkDataTypes,
	  coerceToTypes: coerceToTypes,
	  toHash: toHash,
	  getProperty: getProperty,
	  escapeQuotes: escapeQuotes,
	  equal: __webpack_require__(10),
	  ucs2length: __webpack_require__(12),
	  varOccurences: varOccurences,
	  varReplace: varReplace,
	  cleanUpCode: cleanUpCode,
	  finalCleanUpCode: finalCleanUpCode,
	  schemaHasRules: schemaHasRules,
	  schemaHasRulesExcept: schemaHasRulesExcept,
	  toQuotedString: toQuotedString,
	  getPathExpr: getPathExpr,
	  getPath: getPath,
	  getData: getData,
	  unescapeFragment: unescapeFragment,
	  unescapeJsonPointer: unescapeJsonPointer,
	  escapeFragment: escapeFragment,
	  escapeJsonPointer: escapeJsonPointer
	};


	function copy(o, to) {
	  to = to || {};
	  for (var key in o) to[key] = o[key];
	  return to;
	}


	function checkDataType(dataType, data, negate) {
	  var EQUAL = negate ? ' !== ' : ' === '
	    , AND = negate ? ' || ' : ' && '
	    , OK = negate ? '!' : ''
	    , NOT = negate ? '' : '!';
	  switch (dataType) {
	    case 'null': return data + EQUAL + 'null';
	    case 'array': return OK + 'Array.isArray(' + data + ')';
	    case 'object': return '(' + OK + data + AND +
	                          'typeof ' + data + EQUAL + '"object"' + AND +
	                          NOT + 'Array.isArray(' + data + '))';
	    case 'integer': return '(typeof ' + data + EQUAL + '"number"' + AND +
	                           NOT + '(' + data + ' % 1)' +
	                           AND + data + EQUAL + data + ')';
	    default: return 'typeof ' + data + EQUAL + '"' + dataType + '"';
	  }
	}


	function checkDataTypes(dataTypes, data) {
	  switch (dataTypes.length) {
	    case 1: return checkDataType(dataTypes[0], data, true);
	    default:
	      var code = '';
	      var types = toHash(dataTypes);
	      if (types.array && types.object) {
	        code = types.null ? '(': '(!' + data + ' || ';
	        code += 'typeof ' + data + ' !== "object")';
	        delete types.null;
	        delete types.array;
	        delete types.object;
	      }
	      if (types.number) delete types.integer;
	      for (var t in types)
	        code += (code ? ' && ' : '' ) + checkDataType(t, data, true);

	      return code;
	  }
	}


	var COERCE_TO_TYPES = toHash([ 'string', 'number', 'integer', 'boolean', 'null' ]);
	function coerceToTypes(optionCoerceTypes, dataTypes) {
	  if (Array.isArray(dataTypes)) {
	    var types = [];
	    for (var i=0; i<dataTypes.length; i++) {
	      var t = dataTypes[i];
	      if (COERCE_TO_TYPES[t]) types[types.length] = t;
	      else if (optionCoerceTypes === 'array' && t === 'array') types[types.length] = t;
	    }
	    if (types.length) return types;
	  } else if (COERCE_TO_TYPES[dataTypes]) {
	    return [dataTypes];
	  } else if (optionCoerceTypes === 'array' && dataTypes === 'array') {
	    return ['array'];
	  }
	}


	function toHash(arr) {
	  var hash = {};
	  for (var i=0; i<arr.length; i++) hash[arr[i]] = true;
	  return hash;
	}


	var IDENTIFIER = /^[a-z$_][a-z$_0-9]*$/i;
	var SINGLE_QUOTE = /'|\\/g;
	function getProperty(key) {
	  return typeof key == 'number'
	          ? '[' + key + ']'
	          : IDENTIFIER.test(key)
	            ? '.' + key
	            : "['" + escapeQuotes(key) + "']";
	}


	function escapeQuotes(str) {
	  return str.replace(SINGLE_QUOTE, '\\$&')
	            .replace(/\n/g, '\\n')
	            .replace(/\r/g, '\\r')
	            .replace(/\f/g, '\\f')
	            .replace(/\t/g, '\\t');
	}


	function varOccurences(str, dataVar) {
	  dataVar += '[^0-9]';
	  var matches = str.match(new RegExp(dataVar, 'g'));
	  return matches ? matches.length : 0;
	}


	function varReplace(str, dataVar, expr) {
	  dataVar += '([^0-9])';
	  expr = expr.replace(/\$/g, '$$$$');
	  return str.replace(new RegExp(dataVar, 'g'), expr + '$1');
	}


	var EMPTY_ELSE = /else\s*{\s*}/g
	  , EMPTY_IF_NO_ELSE = /if\s*\([^)]+\)\s*\{\s*\}(?!\s*else)/g
	  , EMPTY_IF_WITH_ELSE = /if\s*\(([^)]+)\)\s*\{\s*\}\s*else(?!\s*if)/g;
	function cleanUpCode(out) {
	  return out.replace(EMPTY_ELSE, '')
	            .replace(EMPTY_IF_NO_ELSE, '')
	            .replace(EMPTY_IF_WITH_ELSE, 'if (!($1))');
	}


	var ERRORS_REGEXP = /[^v.]errors/g
	  , REMOVE_ERRORS = /var errors = 0;|var vErrors = null;|validate.errors = vErrors;/g
	  , REMOVE_ERRORS_ASYNC = /var errors = 0;|var vErrors = null;/g
	  , RETURN_VALID = 'return errors === 0;'
	  , RETURN_TRUE = 'validate.errors = null; return true;'
	  , RETURN_ASYNC = /if \(errors === 0\) return data;\s*else throw new ValidationError\(vErrors\);/
	  , RETURN_DATA_ASYNC = 'return data;'
	  , ROOTDATA_REGEXP = /[^A-Za-z_$]rootData[^A-Za-z0-9_$]/g
	  , REMOVE_ROOTDATA = /if \(rootData === undefined\) rootData = data;/;

	function finalCleanUpCode(out, async) {
	  var matches = out.match(ERRORS_REGEXP);
	  if (matches && matches.length == 2) {
	    out = async
	          ? out.replace(REMOVE_ERRORS_ASYNC, '')
	               .replace(RETURN_ASYNC, RETURN_DATA_ASYNC)
	          : out.replace(REMOVE_ERRORS, '')
	               .replace(RETURN_VALID, RETURN_TRUE);
	  }

	  matches = out.match(ROOTDATA_REGEXP);
	  if (!matches || matches.length !== 3) return out;
	  return out.replace(REMOVE_ROOTDATA, '');
	}


	function schemaHasRules(schema, rules) {
	  if (typeof schema == 'boolean') return !schema;
	  for (var key in schema) if (rules[key]) return true;
	}


	function schemaHasRulesExcept(schema, rules, exceptKeyword) {
	  if (typeof schema == 'boolean') return !schema && exceptKeyword != 'not';
	  for (var key in schema) if (key != exceptKeyword && rules[key]) return true;
	}


	function toQuotedString(str) {
	  return '\'' + escapeQuotes(str) + '\'';
	}


	function getPathExpr(currentPath, expr, jsonPointers, isNumber) {
	  var path = jsonPointers // false by default
	              ? '\'/\' + ' + expr + (isNumber ? '' : '.replace(/~/g, \'~0\').replace(/\\//g, \'~1\')')
	              : (isNumber ? '\'[\' + ' + expr + ' + \']\'' : '\'[\\\'\' + ' + expr + ' + \'\\\']\'');
	  return joinPaths(currentPath, path);
	}


	function getPath(currentPath, prop, jsonPointers) {
	  var path = jsonPointers // false by default
	              ? toQuotedString('/' + escapeJsonPointer(prop))
	              : toQuotedString(getProperty(prop));
	  return joinPaths(currentPath, path);
	}


	var JSON_POINTER = /^\/(?:[^~]|~0|~1)*$/;
	var RELATIVE_JSON_POINTER = /^([0-9]+)(#|\/(?:[^~]|~0|~1)*)?$/;
	function getData($data, lvl, paths) {
	  var up, jsonPointer, data, matches;
	  if ($data === '') return 'rootData';
	  if ($data[0] == '/') {
	    if (!JSON_POINTER.test($data)) throw new Error('Invalid JSON-pointer: ' + $data);
	    jsonPointer = $data;
	    data = 'rootData';
	  } else {
	    matches = $data.match(RELATIVE_JSON_POINTER);
	    if (!matches) throw new Error('Invalid JSON-pointer: ' + $data);
	    up = +matches[1];
	    jsonPointer = matches[2];
	    if (jsonPointer == '#') {
	      if (up >= lvl) throw new Error('Cannot access property/index ' + up + ' levels up, current level is ' + lvl);
	      return paths[lvl - up];
	    }

	    if (up > lvl) throw new Error('Cannot access data ' + up + ' levels up, current level is ' + lvl);
	    data = 'data' + ((lvl - up) || '');
	    if (!jsonPointer) return data;
	  }

	  var expr = data;
	  var segments = jsonPointer.split('/');
	  for (var i=0; i<segments.length; i++) {
	    var segment = segments[i];
	    if (segment) {
	      data += getProperty(unescapeJsonPointer(segment));
	      expr += ' && ' + data;
	    }
	  }
	  return expr;
	}


	function joinPaths (a, b) {
	  if (a == '""') return b;
	  return (a + ' + ' + b).replace(/' \+ '/g, '');
	}


	function unescapeFragment(str) {
	  return unescapeJsonPointer(decodeURIComponent(str));
	}


	function escapeFragment(str) {
	  return encodeURIComponent(escapeJsonPointer(str));
	}


	function escapeJsonPointer(str) {
	  return str.replace(/~/g, '~0').replace(/\//g, '~1');
	}


	function unescapeJsonPointer(str) {
	  return str.replace(/~1/g, '/').replace(/~0/g, '~');
	}


/***/ },
/* 12 */
/***/ function(module, exports) {

	'use strict';

	// https://mathiasbynens.be/notes/javascript-encoding
	// https://github.com/bestiejs/punycode.js - punycode.ucs2.decode
	module.exports = function ucs2length(str) {
	  var length = 0
	    , len = str.length
	    , pos = 0
	    , value;
	  while (pos < len) {
	    length++;
	    value = str.charCodeAt(pos++);
	    if (value >= 0xD800 && value <= 0xDBFF && pos < len) {
	      // high surrogate, and there is a next character
	      value = str.charCodeAt(pos);
	      if ((value & 0xFC00) == 0xDC00) pos++; // low surrogate
	    }
	  }
	  return length;
	};


/***/ },
/* 13 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var util = __webpack_require__(11);

	module.exports = SchemaObject;

	function SchemaObject(obj) {
	  util.copy(obj, this);
	}


/***/ },
/* 14 */
/***/ function(module, exports) {

	'use strict';

	var traverse = module.exports = function (schema, opts, cb) {
	  if (typeof opts == 'function') {
	    cb = opts;
	    opts = {};
	  }
	  _traverse(opts, cb, schema, '', schema);
	};


	traverse.keywords = {
	  additionalItems: true,
	  items: true,
	  contains: true,
	  additionalProperties: true,
	  propertyNames: true,
	  not: true
	};

	traverse.arrayKeywords = {
	  items: true,
	  allOf: true,
	  anyOf: true,
	  oneOf: true
	};

	traverse.propsKeywords = {
	  definitions: true,
	  properties: true,
	  patternProperties: true,
	  dependencies: true
	};

	traverse.skipKeywords = {
	  enum: true,
	  const: true,
	  required: true,
	  maximum: true,
	  minimum: true,
	  exclusiveMaximum: true,
	  exclusiveMinimum: true,
	  multipleOf: true,
	  maxLength: true,
	  minLength: true,
	  pattern: true,
	  format: true,
	  maxItems: true,
	  minItems: true,
	  uniqueItems: true,
	  maxProperties: true,
	  minProperties: true
	};


	function _traverse(opts, cb, schema, jsonPtr, rootSchema, parentJsonPtr, parentKeyword, parentSchema, keyIndex) {
	  if (schema && typeof schema == 'object' && !Array.isArray(schema)) {
	    cb(schema, jsonPtr, rootSchema, parentJsonPtr, parentKeyword, parentSchema, keyIndex);
	    for (var key in schema) {
	      var sch = schema[key];
	      if (Array.isArray(sch)) {
	        if (key in traverse.arrayKeywords) {
	          for (var i=0; i<sch.length; i++)
	            _traverse(opts, cb, sch[i], jsonPtr + '/' + key + '/' + i, rootSchema, jsonPtr, key, schema, i);
	        }
	      } else if (key in traverse.propsKeywords) {
	        if (sch && typeof sch == 'object') {
	          for (var prop in sch)
	            _traverse(opts, cb, sch[prop], jsonPtr + '/' + key + '/' + escapeJsonPtr(prop), rootSchema, jsonPtr, key, schema, prop);
	        }
	      } else if (key in traverse.keywords || (opts.allKeys && !(key in traverse.skipKeywords))) {
	        _traverse(opts, cb, sch, jsonPtr + '/' + key, rootSchema, jsonPtr, key, schema);
	      }
	    }
	  }
	}


	function escapeJsonPtr(str) {
	  return str.replace(/~/g, '~0').replace(/\//g, '~1');
	}


/***/ },
/* 15 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var resolve = __webpack_require__(3);

	module.exports = {
	  Validation: errorSubclass(ValidationError),
	  MissingRef: errorSubclass(MissingRefError)
	};


	function ValidationError(errors) {
	  this.message = 'validation failed';
	  this.errors = errors;
	  this.ajv = this.validation = true;
	}


	MissingRefError.message = function (baseId, ref) {
	  return 'can\'t resolve reference ' + ref + ' from id ' + baseId;
	};


	function MissingRefError(baseId, ref, message) {
	  this.message = message || MissingRefError.message(baseId, ref);
	  this.missingRef = resolve.url(baseId, ref);
	  this.missingSchema = resolve.normalizeId(resolve.fullPath(this.missingRef));
	}


	function errorSubclass(Subclass) {
	  Subclass.prototype = Object.create(Error.prototype);
	  Subclass.prototype.constructor = Subclass;
	  return Subclass;
	}


/***/ },
/* 16 */
/***/ function(module, exports) {

	'use strict';

	module.exports = function (data, opts) {
	    if (!opts) opts = {};
	    if (typeof opts === 'function') opts = { cmp: opts };
	    var cycles = (typeof opts.cycles === 'boolean') ? opts.cycles : false;

	    var cmp = opts.cmp && (function (f) {
	        return function (node) {
	            return function (a, b) {
	                var aobj = { key: a, value: node[a] };
	                var bobj = { key: b, value: node[b] };
	                return f(aobj, bobj);
	            };
	        };
	    })(opts.cmp);

	    var seen = [];
	    return (function stringify (node) {
	        if (node && node.toJSON && typeof node.toJSON === 'function') {
	            node = node.toJSON();
	        }

	        if (node === undefined) return;
	        if (typeof node == 'number') return isFinite(node) ? '' + node : 'null';
	        if (typeof node !== 'object') return JSON.stringify(node);

	        var i, out;
	        if (Array.isArray(node)) {
	            out = '[';
	            for (i = 0; i < node.length; i++) {
	                if (i) out += ',';
	                out += stringify(node[i]) || 'null';
	            }
	            return out + ']';
	        }

	        if (node === null) return 'null';

	        if (seen.indexOf(node) !== -1) {
	            if (cycles) return JSON.stringify('__cycle__');
	            throw new TypeError('Converting circular structure to JSON');
	        }

	        var seenIndex = seen.push(node) - 1;
	        var keys = Object.keys(node).sort(cmp && cmp(node));
	        out = '';
	        for (i = 0; i < keys.length; i++) {
	            var key = keys[i];
	            var value = stringify(node[key]);

	            if (!value) continue;
	            if (out) out += ',';
	            out += JSON.stringify(key) + ':' + value;
	        }
	        seen.splice(seenIndex, 1);
	        return '{' + out + '}';
	    })(data);
	};


/***/ },
/* 17 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_validate(it, $keyword, $ruleType) {
	  var out = '';
	  var $async = it.schema.$async === true,
	    $refKeywords = it.util.schemaHasRulesExcept(it.schema, it.RULES.all, '$ref'),
	    $id = it.self._getId(it.schema);
	  if (it.isTop) {
	    if ($async) {
	      it.async = true;
	      var $es7 = it.opts.async == 'es7';
	      it.yieldAwait = $es7 ? 'await' : 'yield';
	    }
	    out += ' var validate = ';
	    if ($async) {
	      if ($es7) {
	        out += ' (async function ';
	      } else {
	        if (it.opts.async != '*') {
	          out += 'co.wrap';
	        }
	        out += '(function* ';
	      }
	    } else {
	      out += ' (function ';
	    }
	    out += ' (data, dataPath, parentData, parentDataProperty, rootData) { \'use strict\'; ';
	    if ($id && (it.opts.sourceCode || it.opts.processCode)) {
	      out += ' ' + ('/\*# sourceURL=' + $id + ' */') + ' ';
	    }
	  }
	  if (typeof it.schema == 'boolean' || !($refKeywords || it.schema.$ref)) {
	    var $keyword = 'false schema';
	    var $lvl = it.level;
	    var $dataLvl = it.dataLevel;
	    var $schema = it.schema[$keyword];
	    var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	    var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	    var $breakOnError = !it.opts.allErrors;
	    var $errorKeyword;
	    var $data = 'data' + ($dataLvl || '');
	    var $valid = 'valid' + $lvl;
	    if (it.schema === false) {
	      if (it.isTop) {
	        $breakOnError = true;
	      } else {
	        out += ' var ' + ($valid) + ' = false; ';
	      }
	      var $$outStack = $$outStack || [];
	      $$outStack.push(out);
	      out = ''; /* istanbul ignore else */
	      if (it.createErrors !== false) {
	        out += ' { keyword: \'' + ($errorKeyword || 'false schema') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: {} ';
	        if (it.opts.messages !== false) {
	          out += ' , message: \'boolean schema is false\' ';
	        }
	        if (it.opts.verbose) {
	          out += ' , schema: false , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	        }
	        out += ' } ';
	      } else {
	        out += ' {} ';
	      }
	      var __err = out;
	      out = $$outStack.pop();
	      if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	        if (it.async) {
	          out += ' throw new ValidationError([' + (__err) + ']); ';
	        } else {
	          out += ' validate.errors = [' + (__err) + ']; return false; ';
	        }
	      } else {
	        out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	      }
	    } else {
	      if (it.isTop) {
	        if ($async) {
	          out += ' return data; ';
	        } else {
	          out += ' validate.errors = null; return true; ';
	        }
	      } else {
	        out += ' var ' + ($valid) + ' = true; ';
	      }
	    }
	    if (it.isTop) {
	      out += ' }); return validate; ';
	    }
	    return out;
	  }
	  if (it.isTop) {
	    var $top = it.isTop,
	      $lvl = it.level = 0,
	      $dataLvl = it.dataLevel = 0,
	      $data = 'data';
	    it.rootId = it.resolve.fullPath(it.self._getId(it.root.schema));
	    it.baseId = it.baseId || it.rootId;
	    delete it.isTop;
	    it.dataPathArr = [undefined];
	    out += ' var vErrors = null; ';
	    out += ' var errors = 0;     ';
	    out += ' if (rootData === undefined) rootData = data; ';
	  } else {
	    var $lvl = it.level,
	      $dataLvl = it.dataLevel,
	      $data = 'data' + ($dataLvl || '');
	    if ($id) it.baseId = it.resolve.url(it.baseId, $id);
	    if ($async && !it.async) throw new Error('async schema in sync schema');
	    out += ' var errs_' + ($lvl) + ' = errors;';
	  }
	  var $valid = 'valid' + $lvl,
	    $breakOnError = !it.opts.allErrors,
	    $closingBraces1 = '',
	    $closingBraces2 = '';
	  var $errorKeyword;
	  var $typeSchema = it.schema.type,
	    $typeIsArray = Array.isArray($typeSchema);
	  if ($typeIsArray && $typeSchema.length == 1) {
	    $typeSchema = $typeSchema[0];
	    $typeIsArray = false;
	  }
	  if (it.schema.$ref && $refKeywords) {
	    if (it.opts.extendRefs == 'fail') {
	      throw new Error('$ref: validation keywords used in schema at path "' + it.errSchemaPath + '" (see option extendRefs)');
	    } else if (it.opts.extendRefs !== true) {
	      $refKeywords = false;
	      it.logger.warn('$ref: keywords ignored in schema at path "' + it.errSchemaPath + '"');
	    }
	  }
	  if ($typeSchema) {
	    if (it.opts.coerceTypes) {
	      var $coerceToTypes = it.util.coerceToTypes(it.opts.coerceTypes, $typeSchema);
	    }
	    var $rulesGroup = it.RULES.types[$typeSchema];
	    if ($coerceToTypes || $typeIsArray || $rulesGroup === true || ($rulesGroup && !$shouldUseGroup($rulesGroup))) {
	      var $schemaPath = it.schemaPath + '.type',
	        $errSchemaPath = it.errSchemaPath + '/type';
	      var $schemaPath = it.schemaPath + '.type',
	        $errSchemaPath = it.errSchemaPath + '/type',
	        $method = $typeIsArray ? 'checkDataTypes' : 'checkDataType';
	      out += ' if (' + (it.util[$method]($typeSchema, $data, true)) + ') { ';
	      if ($coerceToTypes) {
	        var $dataType = 'dataType' + $lvl,
	          $coerced = 'coerced' + $lvl;
	        out += ' var ' + ($dataType) + ' = typeof ' + ($data) + '; ';
	        if (it.opts.coerceTypes == 'array') {
	          out += ' if (' + ($dataType) + ' == \'object\' && Array.isArray(' + ($data) + ')) ' + ($dataType) + ' = \'array\'; ';
	        }
	        out += ' var ' + ($coerced) + ' = undefined; ';
	        var $bracesCoercion = '';
	        var arr1 = $coerceToTypes;
	        if (arr1) {
	          var $type, $i = -1,
	            l1 = arr1.length - 1;
	          while ($i < l1) {
	            $type = arr1[$i += 1];
	            if ($i) {
	              out += ' if (' + ($coerced) + ' === undefined) { ';
	              $bracesCoercion += '}';
	            }
	            if (it.opts.coerceTypes == 'array' && $type != 'array') {
	              out += ' if (' + ($dataType) + ' == \'array\' && ' + ($data) + '.length == 1) { ' + ($coerced) + ' = ' + ($data) + ' = ' + ($data) + '[0]; ' + ($dataType) + ' = typeof ' + ($data) + ';  } ';
	            }
	            if ($type == 'string') {
	              out += ' if (' + ($dataType) + ' == \'number\' || ' + ($dataType) + ' == \'boolean\') ' + ($coerced) + ' = \'\' + ' + ($data) + '; else if (' + ($data) + ' === null) ' + ($coerced) + ' = \'\'; ';
	            } else if ($type == 'number' || $type == 'integer') {
	              out += ' if (' + ($dataType) + ' == \'boolean\' || ' + ($data) + ' === null || (' + ($dataType) + ' == \'string\' && ' + ($data) + ' && ' + ($data) + ' == +' + ($data) + ' ';
	              if ($type == 'integer') {
	                out += ' && !(' + ($data) + ' % 1)';
	              }
	              out += ')) ' + ($coerced) + ' = +' + ($data) + '; ';
	            } else if ($type == 'boolean') {
	              out += ' if (' + ($data) + ' === \'false\' || ' + ($data) + ' === 0 || ' + ($data) + ' === null) ' + ($coerced) + ' = false; else if (' + ($data) + ' === \'true\' || ' + ($data) + ' === 1) ' + ($coerced) + ' = true; ';
	            } else if ($type == 'null') {
	              out += ' if (' + ($data) + ' === \'\' || ' + ($data) + ' === 0 || ' + ($data) + ' === false) ' + ($coerced) + ' = null; ';
	            } else if (it.opts.coerceTypes == 'array' && $type == 'array') {
	              out += ' if (' + ($dataType) + ' == \'string\' || ' + ($dataType) + ' == \'number\' || ' + ($dataType) + ' == \'boolean\' || ' + ($data) + ' == null) ' + ($coerced) + ' = [' + ($data) + ']; ';
	            }
	          }
	        }
	        out += ' ' + ($bracesCoercion) + ' if (' + ($coerced) + ' === undefined) {   ';
	        var $$outStack = $$outStack || [];
	        $$outStack.push(out);
	        out = ''; /* istanbul ignore else */
	        if (it.createErrors !== false) {
	          out += ' { keyword: \'' + ($errorKeyword || 'type') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { type: \'';
	          if ($typeIsArray) {
	            out += '' + ($typeSchema.join(","));
	          } else {
	            out += '' + ($typeSchema);
	          }
	          out += '\' } ';
	          if (it.opts.messages !== false) {
	            out += ' , message: \'should be ';
	            if ($typeIsArray) {
	              out += '' + ($typeSchema.join(","));
	            } else {
	              out += '' + ($typeSchema);
	            }
	            out += '\' ';
	          }
	          if (it.opts.verbose) {
	            out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	          }
	          out += ' } ';
	        } else {
	          out += ' {} ';
	        }
	        var __err = out;
	        out = $$outStack.pop();
	        if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	          if (it.async) {
	            out += ' throw new ValidationError([' + (__err) + ']); ';
	          } else {
	            out += ' validate.errors = [' + (__err) + ']; return false; ';
	          }
	        } else {
	          out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	        }
	        out += ' } else {  ';
	        var $parentData = $dataLvl ? 'data' + (($dataLvl - 1) || '') : 'parentData',
	          $parentDataProperty = $dataLvl ? it.dataPathArr[$dataLvl] : 'parentDataProperty';
	        out += ' ' + ($data) + ' = ' + ($coerced) + '; ';
	        if (!$dataLvl) {
	          out += 'if (' + ($parentData) + ' !== undefined)';
	        }
	        out += ' ' + ($parentData) + '[' + ($parentDataProperty) + '] = ' + ($coerced) + '; } ';
	      } else {
	        var $$outStack = $$outStack || [];
	        $$outStack.push(out);
	        out = ''; /* istanbul ignore else */
	        if (it.createErrors !== false) {
	          out += ' { keyword: \'' + ($errorKeyword || 'type') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { type: \'';
	          if ($typeIsArray) {
	            out += '' + ($typeSchema.join(","));
	          } else {
	            out += '' + ($typeSchema);
	          }
	          out += '\' } ';
	          if (it.opts.messages !== false) {
	            out += ' , message: \'should be ';
	            if ($typeIsArray) {
	              out += '' + ($typeSchema.join(","));
	            } else {
	              out += '' + ($typeSchema);
	            }
	            out += '\' ';
	          }
	          if (it.opts.verbose) {
	            out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	          }
	          out += ' } ';
	        } else {
	          out += ' {} ';
	        }
	        var __err = out;
	        out = $$outStack.pop();
	        if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	          if (it.async) {
	            out += ' throw new ValidationError([' + (__err) + ']); ';
	          } else {
	            out += ' validate.errors = [' + (__err) + ']; return false; ';
	          }
	        } else {
	          out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	        }
	      }
	      out += ' } ';
	    }
	  }
	  if (it.schema.$ref && !$refKeywords) {
	    out += ' ' + (it.RULES.all.$ref.code(it, '$ref')) + ' ';
	    if ($breakOnError) {
	      out += ' } if (errors === ';
	      if ($top) {
	        out += '0';
	      } else {
	        out += 'errs_' + ($lvl);
	      }
	      out += ') { ';
	      $closingBraces2 += '}';
	    }
	  } else {
	    if (it.opts.v5 && it.schema.patternGroups) {
	      it.logger.warn('keyword "patternGroups" is deprecated and disabled. Use option patternGroups: true to enable.');
	    }
	    var arr2 = it.RULES;
	    if (arr2) {
	      var $rulesGroup, i2 = -1,
	        l2 = arr2.length - 1;
	      while (i2 < l2) {
	        $rulesGroup = arr2[i2 += 1];
	        if ($shouldUseGroup($rulesGroup)) {
	          if ($rulesGroup.type) {
	            out += ' if (' + (it.util.checkDataType($rulesGroup.type, $data)) + ') { ';
	          }
	          if (it.opts.useDefaults && !it.compositeRule) {
	            if ($rulesGroup.type == 'object' && it.schema.properties) {
	              var $schema = it.schema.properties,
	                $schemaKeys = Object.keys($schema);
	              var arr3 = $schemaKeys;
	              if (arr3) {
	                var $propertyKey, i3 = -1,
	                  l3 = arr3.length - 1;
	                while (i3 < l3) {
	                  $propertyKey = arr3[i3 += 1];
	                  var $sch = $schema[$propertyKey];
	                  if ($sch.default !== undefined) {
	                    var $passData = $data + it.util.getProperty($propertyKey);
	                    out += '  if (' + ($passData) + ' === undefined) ' + ($passData) + ' = ';
	                    if (it.opts.useDefaults == 'shared') {
	                      out += ' ' + (it.useDefault($sch.default)) + ' ';
	                    } else {
	                      out += ' ' + (JSON.stringify($sch.default)) + ' ';
	                    }
	                    out += '; ';
	                  }
	                }
	              }
	            } else if ($rulesGroup.type == 'array' && Array.isArray(it.schema.items)) {
	              var arr4 = it.schema.items;
	              if (arr4) {
	                var $sch, $i = -1,
	                  l4 = arr4.length - 1;
	                while ($i < l4) {
	                  $sch = arr4[$i += 1];
	                  if ($sch.default !== undefined) {
	                    var $passData = $data + '[' + $i + ']';
	                    out += '  if (' + ($passData) + ' === undefined) ' + ($passData) + ' = ';
	                    if (it.opts.useDefaults == 'shared') {
	                      out += ' ' + (it.useDefault($sch.default)) + ' ';
	                    } else {
	                      out += ' ' + (JSON.stringify($sch.default)) + ' ';
	                    }
	                    out += '; ';
	                  }
	                }
	              }
	            }
	          }
	          var arr5 = $rulesGroup.rules;
	          if (arr5) {
	            var $rule, i5 = -1,
	              l5 = arr5.length - 1;
	            while (i5 < l5) {
	              $rule = arr5[i5 += 1];
	              if ($shouldUseRule($rule)) {
	                var $code = $rule.code(it, $rule.keyword, $rulesGroup.type);
	                if ($code) {
	                  out += ' ' + ($code) + ' ';
	                  if ($breakOnError) {
	                    $closingBraces1 += '}';
	                  }
	                }
	              }
	            }
	          }
	          if ($breakOnError) {
	            out += ' ' + ($closingBraces1) + ' ';
	            $closingBraces1 = '';
	          }
	          if ($rulesGroup.type) {
	            out += ' } ';
	            if ($typeSchema && $typeSchema === $rulesGroup.type && !$coerceToTypes) {
	              out += ' else { ';
	              var $schemaPath = it.schemaPath + '.type',
	                $errSchemaPath = it.errSchemaPath + '/type';
	              var $$outStack = $$outStack || [];
	              $$outStack.push(out);
	              out = ''; /* istanbul ignore else */
	              if (it.createErrors !== false) {
	                out += ' { keyword: \'' + ($errorKeyword || 'type') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { type: \'';
	                if ($typeIsArray) {
	                  out += '' + ($typeSchema.join(","));
	                } else {
	                  out += '' + ($typeSchema);
	                }
	                out += '\' } ';
	                if (it.opts.messages !== false) {
	                  out += ' , message: \'should be ';
	                  if ($typeIsArray) {
	                    out += '' + ($typeSchema.join(","));
	                  } else {
	                    out += '' + ($typeSchema);
	                  }
	                  out += '\' ';
	                }
	                if (it.opts.verbose) {
	                  out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	                }
	                out += ' } ';
	              } else {
	                out += ' {} ';
	              }
	              var __err = out;
	              out = $$outStack.pop();
	              if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	                if (it.async) {
	                  out += ' throw new ValidationError([' + (__err) + ']); ';
	                } else {
	                  out += ' validate.errors = [' + (__err) + ']; return false; ';
	                }
	              } else {
	                out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	              }
	              out += ' } ';
	            }
	          }
	          if ($breakOnError) {
	            out += ' if (errors === ';
	            if ($top) {
	              out += '0';
	            } else {
	              out += 'errs_' + ($lvl);
	            }
	            out += ') { ';
	            $closingBraces2 += '}';
	          }
	        }
	      }
	    }
	  }
	  if ($breakOnError) {
	    out += ' ' + ($closingBraces2) + ' ';
	  }
	  if ($top) {
	    if ($async) {
	      out += ' if (errors === 0) return data;           ';
	      out += ' else throw new ValidationError(vErrors); ';
	    } else {
	      out += ' validate.errors = vErrors; ';
	      out += ' return errors === 0;       ';
	    }
	    out += ' }); return validate;';
	  } else {
	    out += ' var ' + ($valid) + ' = errors === errs_' + ($lvl) + ';';
	  }
	  out = it.util.cleanUpCode(out);
	  if ($top) {
	    out = it.util.finalCleanUpCode(out, $async);
	  }

	  function $shouldUseGroup($rulesGroup) {
	    var rules = $rulesGroup.rules;
	    for (var i = 0; i < rules.length; i++)
	      if ($shouldUseRule(rules[i])) return true;
	  }

	  function $shouldUseRule($rule) {
	    return it.schema[$rule.keyword] !== undefined || ($rule.implements && $ruleImplementsSomeKeyword($rule));
	  }

	  function $ruleImplementsSomeKeyword($rule) {
	    var impl = $rule.implements;
	    for (var i = 0; i < impl.length; i++)
	      if (it.schema[impl[i]] !== undefined) return true;
	  }
	  return out;
	}


/***/ },
/* 18 */
/***/ function(module, exports) {

	
	/**
	 * slice() reference.
	 */

	var slice = Array.prototype.slice;

	/**
	 * Expose `co`.
	 */

	module.exports = co['default'] = co.co = co;

	/**
	 * Wrap the given generator `fn` into a
	 * function that returns a promise.
	 * This is a separate function so that
	 * every `co()` call doesn't create a new,
	 * unnecessary closure.
	 *
	 * @param {GeneratorFunction} fn
	 * @return {Function}
	 * @api public
	 */

	co.wrap = function (fn) {
	  createPromise.__generatorFunction__ = fn;
	  return createPromise;
	  function createPromise() {
	    return co.call(this, fn.apply(this, arguments));
	  }
	};

	/**
	 * Execute the generator function or a generator
	 * and return a promise.
	 *
	 * @param {Function} fn
	 * @return {Promise}
	 * @api public
	 */

	function co(gen) {
	  var ctx = this;
	  var args = slice.call(arguments, 1)

	  // we wrap everything in a promise to avoid promise chaining,
	  // which leads to memory leak errors.
	  // see https://github.com/tj/co/issues/180
	  return new Promise(function(resolve, reject) {
	    if (typeof gen === 'function') gen = gen.apply(ctx, args);
	    if (!gen || typeof gen.next !== 'function') return resolve(gen);

	    onFulfilled();

	    /**
	     * @param {Mixed} res
	     * @return {Promise}
	     * @api private
	     */

	    function onFulfilled(res) {
	      var ret;
	      try {
	        ret = gen.next(res);
	      } catch (e) {
	        return reject(e);
	      }
	      next(ret);
	    }

	    /**
	     * @param {Error} err
	     * @return {Promise}
	     * @api private
	     */

	    function onRejected(err) {
	      var ret;
	      try {
	        ret = gen.throw(err);
	      } catch (e) {
	        return reject(e);
	      }
	      next(ret);
	    }

	    /**
	     * Get the next value in the generator,
	     * return a promise.
	     *
	     * @param {Object} ret
	     * @return {Promise}
	     * @api private
	     */

	    function next(ret) {
	      if (ret.done) return resolve(ret.value);
	      var value = toPromise.call(ctx, ret.value);
	      if (value && isPromise(value)) return value.then(onFulfilled, onRejected);
	      return onRejected(new TypeError('You may only yield a function, promise, generator, array, or object, '
	        + 'but the following object was passed: "' + String(ret.value) + '"'));
	    }
	  });
	}

	/**
	 * Convert a `yield`ed value into a promise.
	 *
	 * @param {Mixed} obj
	 * @return {Promise}
	 * @api private
	 */

	function toPromise(obj) {
	  if (!obj) return obj;
	  if (isPromise(obj)) return obj;
	  if (isGeneratorFunction(obj) || isGenerator(obj)) return co.call(this, obj);
	  if ('function' == typeof obj) return thunkToPromise.call(this, obj);
	  if (Array.isArray(obj)) return arrayToPromise.call(this, obj);
	  if (isObject(obj)) return objectToPromise.call(this, obj);
	  return obj;
	}

	/**
	 * Convert a thunk to a promise.
	 *
	 * @param {Function}
	 * @return {Promise}
	 * @api private
	 */

	function thunkToPromise(fn) {
	  var ctx = this;
	  return new Promise(function (resolve, reject) {
	    fn.call(ctx, function (err, res) {
	      if (err) return reject(err);
	      if (arguments.length > 2) res = slice.call(arguments, 1);
	      resolve(res);
	    });
	  });
	}

	/**
	 * Convert an array of "yieldables" to a promise.
	 * Uses `Promise.all()` internally.
	 *
	 * @param {Array} obj
	 * @return {Promise}
	 * @api private
	 */

	function arrayToPromise(obj) {
	  return Promise.all(obj.map(toPromise, this));
	}

	/**
	 * Convert an object of "yieldables" to a promise.
	 * Uses `Promise.all()` internally.
	 *
	 * @param {Object} obj
	 * @return {Promise}
	 * @api private
	 */

	function objectToPromise(obj){
	  var results = new obj.constructor();
	  var keys = Object.keys(obj);
	  var promises = [];
	  for (var i = 0; i < keys.length; i++) {
	    var key = keys[i];
	    var promise = toPromise.call(this, obj[key]);
	    if (promise && isPromise(promise)) defer(promise, key);
	    else results[key] = obj[key];
	  }
	  return Promise.all(promises).then(function () {
	    return results;
	  });

	  function defer(promise, key) {
	    // predefine the key in the result
	    results[key] = undefined;
	    promises.push(promise.then(function (res) {
	      results[key] = res;
	    }));
	  }
	}

	/**
	 * Check if `obj` is a promise.
	 *
	 * @param {Object} obj
	 * @return {Boolean}
	 * @api private
	 */

	function isPromise(obj) {
	  return 'function' == typeof obj.then;
	}

	/**
	 * Check if `obj` is a generator.
	 *
	 * @param {Mixed} obj
	 * @return {Boolean}
	 * @api private
	 */

	function isGenerator(obj) {
	  return 'function' == typeof obj.next && 'function' == typeof obj.throw;
	}

	/**
	 * Check if `obj` is a generator function.
	 *
	 * @param {Mixed} obj
	 * @return {Boolean}
	 * @api private
	 */
	function isGeneratorFunction(obj) {
	  var constructor = obj.constructor;
	  if (!constructor) return false;
	  if ('GeneratorFunction' === constructor.name || 'GeneratorFunction' === constructor.displayName) return true;
	  return isGenerator(constructor.prototype);
	}

	/**
	 * Check for plain object.
	 *
	 * @param {Mixed} val
	 * @return {Boolean}
	 * @api private
	 */

	function isObject(val) {
	  return Object == val.constructor;
	}


/***/ },
/* 19 */
/***/ function(module, exports) {

	'use strict';


	var Cache = module.exports = function Cache() {
	  this._cache = {};
	};


	Cache.prototype.put = function Cache_put(key, value) {
	  this._cache[key] = value;
	};


	Cache.prototype.get = function Cache_get(key) {
	  return this._cache[key];
	};


	Cache.prototype.del = function Cache_del(key) {
	  delete this._cache[key];
	};


	Cache.prototype.clear = function Cache_clear() {
	  this._cache = {};
	};


/***/ },
/* 20 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var util = __webpack_require__(11);

	var DATE = /^\d\d\d\d-(\d\d)-(\d\d)$/;
	var DAYS = [0,31,29,31,30,31,30,31,31,30,31,30,31];
	var TIME = /^(\d\d):(\d\d):(\d\d)(\.\d+)?(z|[+-]\d\d:\d\d)?$/i;
	var HOSTNAME = /^[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?(?:\.[a-z0-9](?:[-0-9a-z]{0,61}[0-9a-z])?)*$/i;
	var URI = /^(?:[a-z][a-z0-9+\-.]*:)(?:\/?\/(?:(?:[a-z0-9\-._~!$&'()*+,;=:]|%[0-9a-f]{2})*@)?(?:\[(?:(?:(?:(?:[0-9a-f]{1,4}:){6}|::(?:[0-9a-f]{1,4}:){5}|(?:[0-9a-f]{1,4})?::(?:[0-9a-f]{1,4}:){4}|(?:(?:[0-9a-f]{1,4}:){0,1}[0-9a-f]{1,4})?::(?:[0-9a-f]{1,4}:){3}|(?:(?:[0-9a-f]{1,4}:){0,2}[0-9a-f]{1,4})?::(?:[0-9a-f]{1,4}:){2}|(?:(?:[0-9a-f]{1,4}:){0,3}[0-9a-f]{1,4})?::[0-9a-f]{1,4}:|(?:(?:[0-9a-f]{1,4}:){0,4}[0-9a-f]{1,4})?::)(?:[0-9a-f]{1,4}:[0-9a-f]{1,4}|(?:(?:25[0-5]|2[0-4]\d|[01]?\d\d?)\.){3}(?:25[0-5]|2[0-4]\d|[01]?\d\d?))|(?:(?:[0-9a-f]{1,4}:){0,5}[0-9a-f]{1,4})?::[0-9a-f]{1,4}|(?:(?:[0-9a-f]{1,4}:){0,6}[0-9a-f]{1,4})?::)|[Vv][0-9a-f]+\.[a-z0-9\-._~!$&'()*+,;=:]+)\]|(?:(?:25[0-5]|2[0-4]\d|[01]?\d\d?)\.){3}(?:25[0-5]|2[0-4]\d|[01]?\d\d?)|(?:[a-z0-9\-._~!$&'()*+,;=]|%[0-9a-f]{2})*)(?::\d*)?(?:\/(?:[a-z0-9\-._~!$&'()*+,;=:@]|%[0-9a-f]{2})*)*|\/(?:(?:[a-z0-9\-._~!$&'()*+,;=:@]|%[0-9a-f]{2})+(?:\/(?:[a-z0-9\-._~!$&'()*+,;=:@]|%[0-9a-f]{2})*)*)?|(?:[a-z0-9\-._~!$&'()*+,;=:@]|%[0-9a-f]{2})+(?:\/(?:[a-z0-9\-._~!$&'()*+,;=:@]|%[0-9a-f]{2})*)*)(?:\?(?:[a-z0-9\-._~!$&'()*+,;=:@/?]|%[0-9a-f]{2})*)?(?:#(?:[a-z0-9\-._~!$&'()*+,;=:@/?]|%[0-9a-f]{2})*)?$/i;
	var URIREF = /^(?:[a-z][a-z0-9+\-.]*:)?(?:\/?\/(?:(?:[a-z0-9\-._~!$&'()*+,;=:]|%[0-9a-f]{2})*@)?(?:\[(?:(?:(?:(?:[0-9a-f]{1,4}:){6}|::(?:[0-9a-f]{1,4}:){5}|(?:[0-9a-f]{1,4})?::(?:[0-9a-f]{1,4}:){4}|(?:(?:[0-9a-f]{1,4}:){0,1}[0-9a-f]{1,4})?::(?:[0-9a-f]{1,4}:){3}|(?:(?:[0-9a-f]{1,4}:){0,2}[0-9a-f]{1,4})?::(?:[0-9a-f]{1,4}:){2}|(?:(?:[0-9a-f]{1,4}:){0,3}[0-9a-f]{1,4})?::[0-9a-f]{1,4}:|(?:(?:[0-9a-f]{1,4}:){0,4}[0-9a-f]{1,4})?::)(?:[0-9a-f]{1,4}:[0-9a-f]{1,4}|(?:(?:25[0-5]|2[0-4]\d|[01]?\d\d?)\.){3}(?:25[0-5]|2[0-4]\d|[01]?\d\d?))|(?:(?:[0-9a-f]{1,4}:){0,5}[0-9a-f]{1,4})?::[0-9a-f]{1,4}|(?:(?:[0-9a-f]{1,4}:){0,6}[0-9a-f]{1,4})?::)|[Vv][0-9a-f]+\.[a-z0-9\-._~!$&'()*+,;=:]+)\]|(?:(?:25[0-5]|2[0-4]\d|[01]?\d\d?)\.){3}(?:25[0-5]|2[0-4]\d|[01]?\d\d?)|(?:[a-z0-9\-._~!$&'"()*+,;=]|%[0-9a-f]{2})*)(?::\d*)?(?:\/(?:[a-z0-9\-._~!$&'"()*+,;=:@]|%[0-9a-f]{2})*)*|\/(?:(?:[a-z0-9\-._~!$&'"()*+,;=:@]|%[0-9a-f]{2})+(?:\/(?:[a-z0-9\-._~!$&'"()*+,;=:@]|%[0-9a-f]{2})*)*)?|(?:[a-z0-9\-._~!$&'"()*+,;=:@]|%[0-9a-f]{2})+(?:\/(?:[a-z0-9\-._~!$&'"()*+,;=:@]|%[0-9a-f]{2})*)*)?(?:\?(?:[a-z0-9\-._~!$&'"()*+,;=:@/?]|%[0-9a-f]{2})*)?(?:#(?:[a-z0-9\-._~!$&'"()*+,;=:@/?]|%[0-9a-f]{2})*)?$/i;
	// uri-template: https://tools.ietf.org/html/rfc6570
	var URITEMPLATE = /^(?:(?:[^\x00-\x20"'<>%\\^`{|}]|%[0-9a-f]{2})|\{[+#./;?&=,!@|]?(?:[a-z0-9_]|%[0-9a-f]{2})+(?::[1-9][0-9]{0,3}|\*)?(?:,(?:[a-z0-9_]|%[0-9a-f]{2})+(?::[1-9][0-9]{0,3}|\*)?)*\})*$/i;
	// For the source: https://gist.github.com/dperini/729294
	// For test cases: https://mathiasbynens.be/demo/url-regex
	// @todo Delete current URL in favour of the commented out URL rule when this issue is fixed https://github.com/eslint/eslint/issues/7983.
	// var URL = /^(?:(?:https?|ftp):\/\/)(?:\S+(?::\S*)?@)?(?:(?!10(?:\.\d{1,3}){3})(?!127(?:\.\d{1,3}){3})(?!169\.254(?:\.\d{1,3}){2})(?!192\.168(?:\.\d{1,3}){2})(?!172\.(?:1[6-9]|2\d|3[0-1])(?:\.\d{1,3}){2})(?:[1-9]\d?|1\d\d|2[01]\d|22[0-3])(?:\.(?:1?\d{1,2}|2[0-4]\d|25[0-5])){2}(?:\.(?:[1-9]\d?|1\d\d|2[0-4]\d|25[0-4]))|(?:(?:[a-z\u{00a1}-\u{ffff}0-9]+-?)*[a-z\u{00a1}-\u{ffff}0-9]+)(?:\.(?:[a-z\u{00a1}-\u{ffff}0-9]+-?)*[a-z\u{00a1}-\u{ffff}0-9]+)*(?:\.(?:[a-z\u{00a1}-\u{ffff}]{2,})))(?::\d{2,5})?(?:\/[^\s]*)?$/iu;
	var URL = /^(?:(?:http[s\u017F]?|ftp):\/\/)(?:(?:[\0-\x08\x0E-\x1F!-\x9F\xA1-\u167F\u1681-\u1FFF\u200B-\u2027\u202A-\u202E\u2030-\u205E\u2060-\u2FFF\u3001-\uD7FF\uE000-\uFEFE\uFF00-\uFFFF]|[\uD800-\uDBFF][\uDC00-\uDFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF])+(?::(?:[\0-\x08\x0E-\x1F!-\x9F\xA1-\u167F\u1681-\u1FFF\u200B-\u2027\u202A-\u202E\u2030-\u205E\u2060-\u2FFF\u3001-\uD7FF\uE000-\uFEFE\uFF00-\uFFFF]|[\uD800-\uDBFF][\uDC00-\uDFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF])*)?@)?(?:(?!10(?:\.[0-9]{1,3}){3})(?!127(?:\.[0-9]{1,3}){3})(?!169\.254(?:\.[0-9]{1,3}){2})(?!192\.168(?:\.[0-9]{1,3}){2})(?!172\.(?:1[6-9]|2[0-9]|3[01])(?:\.[0-9]{1,3}){2})(?:[1-9][0-9]?|1[0-9][0-9]|2[01][0-9]|22[0-3])(?:\.(?:1?[0-9]{1,2}|2[0-4][0-9]|25[0-5])){2}(?:\.(?:[1-9][0-9]?|1[0-9][0-9]|2[0-4][0-9]|25[0-4]))|(?:(?:(?:[0-9KSa-z\xA1-\uD7FF\uE000-\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF])+-?)*(?:[0-9KSa-z\xA1-\uD7FF\uE000-\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF])+)(?:\.(?:(?:[0-9KSa-z\xA1-\uD7FF\uE000-\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF])+-?)*(?:[0-9KSa-z\xA1-\uD7FF\uE000-\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF])+)*(?:\.(?:(?:[KSa-z\xA1-\uD7FF\uE000-\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF]){2,})))(?::[0-9]{2,5})?(?:\/(?:[\0-\x08\x0E-\x1F!-\x9F\xA1-\u167F\u1681-\u1FFF\u200B-\u2027\u202A-\u202E\u2030-\u205E\u2060-\u2FFF\u3001-\uD7FF\uE000-\uFEFE\uFF00-\uFFFF]|[\uD800-\uDBFF][\uDC00-\uDFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF])*)?$/i;
	var UUID = /^(?:urn:uuid:)?[0-9a-f]{8}-(?:[0-9a-f]{4}-){3}[0-9a-f]{12}$/i;
	var JSON_POINTER = /^(?:\/(?:[^~/]|~0|~1)*)*$|^#(?:\/(?:[a-z0-9_\-.!$&'()*+,;:=@]|%[0-9a-f]{2}|~0|~1)*)*$/i;
	var RELATIVE_JSON_POINTER = /^(?:0|[1-9][0-9]*)(?:#|(?:\/(?:[^~/]|~0|~1)*)*)$/;


	module.exports = formats;

	function formats(mode) {
	  mode = mode == 'full' ? 'full' : 'fast';
	  return util.copy(formats[mode]);
	}


	formats.fast = {
	  // date: http://tools.ietf.org/html/rfc3339#section-5.6
	  date: /^\d\d\d\d-[0-1]\d-[0-3]\d$/,
	  // date-time: http://tools.ietf.org/html/rfc3339#section-5.6
	  time: /^[0-2]\d:[0-5]\d:[0-5]\d(?:\.\d+)?(?:z|[+-]\d\d:\d\d)?$/i,
	  'date-time': /^\d\d\d\d-[0-1]\d-[0-3]\d[t\s][0-2]\d:[0-5]\d:[0-5]\d(?:\.\d+)?(?:z|[+-]\d\d:\d\d)$/i,
	  // uri: https://github.com/mafintosh/is-my-json-valid/blob/master/formats.js
	  uri: /^(?:[a-z][a-z0-9+-.]*)(?::|\/)\/?[^\s]*$/i,
	  'uri-reference': /^(?:(?:[a-z][a-z0-9+-.]*:)?\/\/)?[^\s]*$/i,
	  'uri-template': URITEMPLATE,
	  url: URL,
	  // email (sources from jsen validator):
	  // http://stackoverflow.com/questions/201323/using-a-regular-expression-to-validate-an-email-address#answer-8829363
	  // http://www.w3.org/TR/html5/forms.html#valid-e-mail-address (search for 'willful violation')
	  email: /^[a-z0-9.!#$%&'*+/=?^_`{|}~-]+@[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?(?:\.[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?)*$/i,
	  hostname: HOSTNAME,
	  // optimized https://www.safaribooksonline.com/library/view/regular-expressions-cookbook/9780596802837/ch07s16.html
	  ipv4: /^(?:(?:25[0-5]|2[0-4]\d|[01]?\d\d?)\.){3}(?:25[0-5]|2[0-4]\d|[01]?\d\d?)$/,
	  // optimized http://stackoverflow.com/questions/53497/regular-expression-that-matches-valid-ipv6-addresses
	  ipv6: /^\s*(?:(?:(?:[0-9a-f]{1,4}:){7}(?:[0-9a-f]{1,4}|:))|(?:(?:[0-9a-f]{1,4}:){6}(?::[0-9a-f]{1,4}|(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3})|:))|(?:(?:[0-9a-f]{1,4}:){5}(?:(?:(?::[0-9a-f]{1,4}){1,2})|:(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3})|:))|(?:(?:[0-9a-f]{1,4}:){4}(?:(?:(?::[0-9a-f]{1,4}){1,3})|(?:(?::[0-9a-f]{1,4})?:(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(?:(?:[0-9a-f]{1,4}:){3}(?:(?:(?::[0-9a-f]{1,4}){1,4})|(?:(?::[0-9a-f]{1,4}){0,2}:(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(?:(?:[0-9a-f]{1,4}:){2}(?:(?:(?::[0-9a-f]{1,4}){1,5})|(?:(?::[0-9a-f]{1,4}){0,3}:(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(?:(?:[0-9a-f]{1,4}:){1}(?:(?:(?::[0-9a-f]{1,4}){1,6})|(?:(?::[0-9a-f]{1,4}){0,4}:(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(?::(?:(?:(?::[0-9a-f]{1,4}){1,7})|(?:(?::[0-9a-f]{1,4}){0,5}:(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:)))(?:%.+)?\s*$/i,
	  regex: regex,
	  // uuid: http://tools.ietf.org/html/rfc4122
	  uuid: UUID,
	  // JSON-pointer: https://tools.ietf.org/html/rfc6901
	  // uri fragment: https://tools.ietf.org/html/rfc3986#appendix-A
	  'json-pointer': JSON_POINTER,
	  // relative JSON-pointer: http://tools.ietf.org/html/draft-luff-relative-json-pointer-00
	  'relative-json-pointer': RELATIVE_JSON_POINTER
	};


	formats.full = {
	  date: date,
	  time: time,
	  'date-time': date_time,
	  uri: uri,
	  'uri-reference': URIREF,
	  'uri-template': URITEMPLATE,
	  url: URL,
	  email: /^[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&''*+/=?^_`{|}~-]+)*@(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?$/i,
	  hostname: hostname,
	  ipv4: /^(?:(?:25[0-5]|2[0-4]\d|[01]?\d\d?)\.){3}(?:25[0-5]|2[0-4]\d|[01]?\d\d?)$/,
	  ipv6: /^\s*(?:(?:(?:[0-9a-f]{1,4}:){7}(?:[0-9a-f]{1,4}|:))|(?:(?:[0-9a-f]{1,4}:){6}(?::[0-9a-f]{1,4}|(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3})|:))|(?:(?:[0-9a-f]{1,4}:){5}(?:(?:(?::[0-9a-f]{1,4}){1,2})|:(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3})|:))|(?:(?:[0-9a-f]{1,4}:){4}(?:(?:(?::[0-9a-f]{1,4}){1,3})|(?:(?::[0-9a-f]{1,4})?:(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(?:(?:[0-9a-f]{1,4}:){3}(?:(?:(?::[0-9a-f]{1,4}){1,4})|(?:(?::[0-9a-f]{1,4}){0,2}:(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(?:(?:[0-9a-f]{1,4}:){2}(?:(?:(?::[0-9a-f]{1,4}){1,5})|(?:(?::[0-9a-f]{1,4}){0,3}:(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(?:(?:[0-9a-f]{1,4}:){1}(?:(?:(?::[0-9a-f]{1,4}){1,6})|(?:(?::[0-9a-f]{1,4}){0,4}:(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(?::(?:(?:(?::[0-9a-f]{1,4}){1,7})|(?:(?::[0-9a-f]{1,4}){0,5}:(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:)))(?:%.+)?\s*$/i,
	  regex: regex,
	  uuid: UUID,
	  'json-pointer': JSON_POINTER,
	  'relative-json-pointer': RELATIVE_JSON_POINTER
	};


	function date(str) {
	  // full-date from http://tools.ietf.org/html/rfc3339#section-5.6
	  var matches = str.match(DATE);
	  if (!matches) return false;

	  var month = +matches[1];
	  var day = +matches[2];
	  return month >= 1 && month <= 12 && day >= 1 && day <= DAYS[month];
	}


	function time(str, full) {
	  var matches = str.match(TIME);
	  if (!matches) return false;

	  var hour = matches[1];
	  var minute = matches[2];
	  var second = matches[3];
	  var timeZone = matches[5];
	  return hour <= 23 && minute <= 59 && second <= 59 && (!full || timeZone);
	}


	var DATE_TIME_SEPARATOR = /t|\s/i;
	function date_time(str) {
	  // http://tools.ietf.org/html/rfc3339#section-5.6
	  var dateTime = str.split(DATE_TIME_SEPARATOR);
	  return dateTime.length == 2 && date(dateTime[0]) && time(dateTime[1], true);
	}


	function hostname(str) {
	  // https://tools.ietf.org/html/rfc1034#section-3.5
	  // https://tools.ietf.org/html/rfc1123#section-2
	  return str.length <= 255 && HOSTNAME.test(str);
	}


	var NOT_URI_FRAGMENT = /\/|:/;
	function uri(str) {
	  // http://jmrware.com/articles/2009/uri_regexp/URI_regex.html + optional protocol + required "."
	  return NOT_URI_FRAGMENT.test(str) && URI.test(str);
	}


	var Z_ANCHOR = /[^\\]\\Z/;
	function regex(str) {
	  if (Z_ANCHOR.test(str)) return false;
	  try {
	    new RegExp(str);
	    return true;
	  } catch(e) {
	    return false;
	  }
	}


/***/ },
/* 21 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var ruleModules = __webpack_require__(22)
	  , toHash = __webpack_require__(11).toHash;

	module.exports = function rules() {
	  var RULES = [
	    { type: 'number',
	      rules: [ { 'maximum': ['exclusiveMaximum'] },
	               { 'minimum': ['exclusiveMinimum'] }, 'multipleOf', 'format'] },
	    { type: 'string',
	      rules: [ 'maxLength', 'minLength', 'pattern', 'format' ] },
	    { type: 'array',
	      rules: [ 'maxItems', 'minItems', 'uniqueItems', 'contains', 'items' ] },
	    { type: 'object',
	      rules: [ 'maxProperties', 'minProperties', 'required', 'dependencies', 'propertyNames',
	               { 'properties': ['additionalProperties', 'patternProperties'] } ] },
	    { rules: [ '$ref', 'const', 'enum', 'not', 'anyOf', 'oneOf', 'allOf' ] }
	  ];

	  var ALL = [ 'type' ];
	  var KEYWORDS = [
	    'additionalItems', '$schema', '$id', 'id', 'title',
	    'description', 'default', 'definitions'
	  ];
	  var TYPES = [ 'number', 'integer', 'string', 'array', 'object', 'boolean', 'null' ];
	  RULES.all = toHash(ALL);
	  RULES.types = toHash(TYPES);

	  RULES.forEach(function (group) {
	    group.rules = group.rules.map(function (keyword) {
	      var implKeywords;
	      if (typeof keyword == 'object') {
	        var key = Object.keys(keyword)[0];
	        implKeywords = keyword[key];
	        keyword = key;
	        implKeywords.forEach(function (k) {
	          ALL.push(k);
	          RULES.all[k] = true;
	        });
	      }
	      ALL.push(keyword);
	      var rule = RULES.all[keyword] = {
	        keyword: keyword,
	        code: ruleModules[keyword],
	        implements: implKeywords
	      };
	      return rule;
	    });

	    if (group.type) RULES.types[group.type] = group;
	  });

	  RULES.keywords = toHash(ALL.concat(KEYWORDS));
	  RULES.custom = {};

	  return RULES;
	};


/***/ },
/* 22 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	//all requires must be explicit because browserify won't work with dynamic requires
	module.exports = {
	  '$ref': __webpack_require__(23),
	  allOf: __webpack_require__(24),
	  anyOf: __webpack_require__(25),
	  const: __webpack_require__(26),
	  contains: __webpack_require__(27),
	  dependencies: __webpack_require__(28),
	  'enum': __webpack_require__(29),
	  format: __webpack_require__(30),
	  items: __webpack_require__(31),
	  maximum: __webpack_require__(32),
	  minimum: __webpack_require__(32),
	  maxItems: __webpack_require__(33),
	  minItems: __webpack_require__(33),
	  maxLength: __webpack_require__(34),
	  minLength: __webpack_require__(34),
	  maxProperties: __webpack_require__(35),
	  minProperties: __webpack_require__(35),
	  multipleOf: __webpack_require__(36),
	  not: __webpack_require__(37),
	  oneOf: __webpack_require__(38),
	  pattern: __webpack_require__(39),
	  properties: __webpack_require__(40),
	  propertyNames: __webpack_require__(41),
	  required: __webpack_require__(42),
	  uniqueItems: __webpack_require__(43),
	  validate: __webpack_require__(17)
	};


/***/ },
/* 23 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_ref(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $valid = 'valid' + $lvl;
	  var $async, $refCode;
	  if ($schema == '#' || $schema == '#/') {
	    if (it.isRoot) {
	      $async = it.async;
	      $refCode = 'validate';
	    } else {
	      $async = it.root.schema.$async === true;
	      $refCode = 'root.refVal[0]';
	    }
	  } else {
	    var $refVal = it.resolveRef(it.baseId, $schema, it.isRoot);
	    if ($refVal === undefined) {
	      var $message = it.MissingRefError.message(it.baseId, $schema);
	      if (it.opts.missingRefs == 'fail') {
	        it.logger.error($message);
	        var $$outStack = $$outStack || [];
	        $$outStack.push(out);
	        out = ''; /* istanbul ignore else */
	        if (it.createErrors !== false) {
	          out += ' { keyword: \'' + ('$ref') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { ref: \'' + (it.util.escapeQuotes($schema)) + '\' } ';
	          if (it.opts.messages !== false) {
	            out += ' , message: \'can\\\'t resolve reference ' + (it.util.escapeQuotes($schema)) + '\' ';
	          }
	          if (it.opts.verbose) {
	            out += ' , schema: ' + (it.util.toQuotedString($schema)) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	          }
	          out += ' } ';
	        } else {
	          out += ' {} ';
	        }
	        var __err = out;
	        out = $$outStack.pop();
	        if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	          if (it.async) {
	            out += ' throw new ValidationError([' + (__err) + ']); ';
	          } else {
	            out += ' validate.errors = [' + (__err) + ']; return false; ';
	          }
	        } else {
	          out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	        }
	        if ($breakOnError) {
	          out += ' if (false) { ';
	        }
	      } else if (it.opts.missingRefs == 'ignore') {
	        it.logger.warn($message);
	        if ($breakOnError) {
	          out += ' if (true) { ';
	        }
	      } else {
	        throw new it.MissingRefError(it.baseId, $schema, $message);
	      }
	    } else if ($refVal.inline) {
	      var $it = it.util.copy(it);
	      $it.level++;
	      var $nextValid = 'valid' + $it.level;
	      $it.schema = $refVal.schema;
	      $it.schemaPath = '';
	      $it.errSchemaPath = $schema;
	      var $code = it.validate($it).replace(/validate\.schema/g, $refVal.code);
	      out += ' ' + ($code) + ' ';
	      if ($breakOnError) {
	        out += ' if (' + ($nextValid) + ') { ';
	      }
	    } else {
	      $async = $refVal.$async === true;
	      $refCode = $refVal.code;
	    }
	  }
	  if ($refCode) {
	    var $$outStack = $$outStack || [];
	    $$outStack.push(out);
	    out = '';
	    if (it.opts.passContext) {
	      out += ' ' + ($refCode) + '.call(this, ';
	    } else {
	      out += ' ' + ($refCode) + '( ';
	    }
	    out += ' ' + ($data) + ', (dataPath || \'\')';
	    if (it.errorPath != '""') {
	      out += ' + ' + (it.errorPath);
	    }
	    var $parentData = $dataLvl ? 'data' + (($dataLvl - 1) || '') : 'parentData',
	      $parentDataProperty = $dataLvl ? it.dataPathArr[$dataLvl] : 'parentDataProperty';
	    out += ' , ' + ($parentData) + ' , ' + ($parentDataProperty) + ', rootData)  ';
	    var __callValidate = out;
	    out = $$outStack.pop();
	    if ($async) {
	      if (!it.async) throw new Error('async schema referenced by sync schema');
	      if ($breakOnError) {
	        out += ' var ' + ($valid) + '; ';
	      }
	      out += ' try { ' + (it.yieldAwait) + ' ' + (__callValidate) + '; ';
	      if ($breakOnError) {
	        out += ' ' + ($valid) + ' = true; ';
	      }
	      out += ' } catch (e) { if (!(e instanceof ValidationError)) throw e; if (vErrors === null) vErrors = e.errors; else vErrors = vErrors.concat(e.errors); errors = vErrors.length; ';
	      if ($breakOnError) {
	        out += ' ' + ($valid) + ' = false; ';
	      }
	      out += ' } ';
	      if ($breakOnError) {
	        out += ' if (' + ($valid) + ') { ';
	      }
	    } else {
	      out += ' if (!' + (__callValidate) + ') { if (vErrors === null) vErrors = ' + ($refCode) + '.errors; else vErrors = vErrors.concat(' + ($refCode) + '.errors); errors = vErrors.length; } ';
	      if ($breakOnError) {
	        out += ' else { ';
	      }
	    }
	  }
	  return out;
	}


/***/ },
/* 24 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_allOf(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $it = it.util.copy(it);
	  var $closingBraces = '';
	  $it.level++;
	  var $nextValid = 'valid' + $it.level;
	  var $currentBaseId = $it.baseId,
	    $allSchemasEmpty = true;
	  var arr1 = $schema;
	  if (arr1) {
	    var $sch, $i = -1,
	      l1 = arr1.length - 1;
	    while ($i < l1) {
	      $sch = arr1[$i += 1];
	      if (it.util.schemaHasRules($sch, it.RULES.all)) {
	        $allSchemasEmpty = false;
	        $it.schema = $sch;
	        $it.schemaPath = $schemaPath + '[' + $i + ']';
	        $it.errSchemaPath = $errSchemaPath + '/' + $i;
	        out += '  ' + (it.validate($it)) + ' ';
	        $it.baseId = $currentBaseId;
	        if ($breakOnError) {
	          out += ' if (' + ($nextValid) + ') { ';
	          $closingBraces += '}';
	        }
	      }
	    }
	  }
	  if ($breakOnError) {
	    if ($allSchemasEmpty) {
	      out += ' if (true) { ';
	    } else {
	      out += ' ' + ($closingBraces.slice(0, -1)) + ' ';
	    }
	  }
	  out = it.util.cleanUpCode(out);
	  return out;
	}


/***/ },
/* 25 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_anyOf(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $valid = 'valid' + $lvl;
	  var $errs = 'errs__' + $lvl;
	  var $it = it.util.copy(it);
	  var $closingBraces = '';
	  $it.level++;
	  var $nextValid = 'valid' + $it.level;
	  var $noEmptySchema = $schema.every(function($sch) {
	    return it.util.schemaHasRules($sch, it.RULES.all);
	  });
	  if ($noEmptySchema) {
	    var $currentBaseId = $it.baseId;
	    out += ' var ' + ($errs) + ' = errors; var ' + ($valid) + ' = false;  ';
	    var $wasComposite = it.compositeRule;
	    it.compositeRule = $it.compositeRule = true;
	    var arr1 = $schema;
	    if (arr1) {
	      var $sch, $i = -1,
	        l1 = arr1.length - 1;
	      while ($i < l1) {
	        $sch = arr1[$i += 1];
	        $it.schema = $sch;
	        $it.schemaPath = $schemaPath + '[' + $i + ']';
	        $it.errSchemaPath = $errSchemaPath + '/' + $i;
	        out += '  ' + (it.validate($it)) + ' ';
	        $it.baseId = $currentBaseId;
	        out += ' ' + ($valid) + ' = ' + ($valid) + ' || ' + ($nextValid) + '; if (!' + ($valid) + ') { ';
	        $closingBraces += '}';
	      }
	    }
	    it.compositeRule = $it.compositeRule = $wasComposite;
	    out += ' ' + ($closingBraces) + ' if (!' + ($valid) + ') {   var err =   '; /* istanbul ignore else */
	    if (it.createErrors !== false) {
	      out += ' { keyword: \'' + ('anyOf') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: {} ';
	      if (it.opts.messages !== false) {
	        out += ' , message: \'should match some schema in anyOf\' ';
	      }
	      if (it.opts.verbose) {
	        out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	      }
	      out += ' } ';
	    } else {
	      out += ' {} ';
	    }
	    out += ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	    if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	      if (it.async) {
	        out += ' throw new ValidationError(vErrors); ';
	      } else {
	        out += ' validate.errors = vErrors; return false; ';
	      }
	    }
	    out += ' } else {  errors = ' + ($errs) + '; if (vErrors !== null) { if (' + ($errs) + ') vErrors.length = ' + ($errs) + '; else vErrors = null; } ';
	    if (it.opts.allErrors) {
	      out += ' } ';
	    }
	    out = it.util.cleanUpCode(out);
	  } else {
	    if ($breakOnError) {
	      out += ' if (true) { ';
	    }
	  }
	  return out;
	}


/***/ },
/* 26 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_const(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $valid = 'valid' + $lvl;
	  var $isData = it.opts.$data && $schema && $schema.$data,
	    $schemaValue;
	  if ($isData) {
	    out += ' var schema' + ($lvl) + ' = ' + (it.util.getData($schema.$data, $dataLvl, it.dataPathArr)) + '; ';
	    $schemaValue = 'schema' + $lvl;
	  } else {
	    $schemaValue = $schema;
	  }
	  if (!$isData) {
	    out += ' var schema' + ($lvl) + ' = validate.schema' + ($schemaPath) + ';';
	  }
	  out += 'var ' + ($valid) + ' = equal(' + ($data) + ', schema' + ($lvl) + '); if (!' + ($valid) + ') {   ';
	  var $$outStack = $$outStack || [];
	  $$outStack.push(out);
	  out = ''; /* istanbul ignore else */
	  if (it.createErrors !== false) {
	    out += ' { keyword: \'' + ('const') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: {} ';
	    if (it.opts.messages !== false) {
	      out += ' , message: \'should be equal to constant\' ';
	    }
	    if (it.opts.verbose) {
	      out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	    }
	    out += ' } ';
	  } else {
	    out += ' {} ';
	  }
	  var __err = out;
	  out = $$outStack.pop();
	  if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	    if (it.async) {
	      out += ' throw new ValidationError([' + (__err) + ']); ';
	    } else {
	      out += ' validate.errors = [' + (__err) + ']; return false; ';
	    }
	  } else {
	    out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	  }
	  out += ' }';
	  if ($breakOnError) {
	    out += ' else { ';
	  }
	  return out;
	}


/***/ },
/* 27 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_contains(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $valid = 'valid' + $lvl;
	  var $errs = 'errs__' + $lvl;
	  var $it = it.util.copy(it);
	  var $closingBraces = '';
	  $it.level++;
	  var $nextValid = 'valid' + $it.level;
	  var $idx = 'i' + $lvl,
	    $dataNxt = $it.dataLevel = it.dataLevel + 1,
	    $nextData = 'data' + $dataNxt,
	    $currentBaseId = it.baseId,
	    $nonEmptySchema = it.util.schemaHasRules($schema, it.RULES.all);
	  out += 'var ' + ($errs) + ' = errors;var ' + ($valid) + ';';
	  if ($nonEmptySchema) {
	    var $wasComposite = it.compositeRule;
	    it.compositeRule = $it.compositeRule = true;
	    $it.schema = $schema;
	    $it.schemaPath = $schemaPath;
	    $it.errSchemaPath = $errSchemaPath;
	    out += ' var ' + ($nextValid) + ' = false; for (var ' + ($idx) + ' = 0; ' + ($idx) + ' < ' + ($data) + '.length; ' + ($idx) + '++) { ';
	    $it.errorPath = it.util.getPathExpr(it.errorPath, $idx, it.opts.jsonPointers, true);
	    var $passData = $data + '[' + $idx + ']';
	    $it.dataPathArr[$dataNxt] = $idx;
	    var $code = it.validate($it);
	    $it.baseId = $currentBaseId;
	    if (it.util.varOccurences($code, $nextData) < 2) {
	      out += ' ' + (it.util.varReplace($code, $nextData, $passData)) + ' ';
	    } else {
	      out += ' var ' + ($nextData) + ' = ' + ($passData) + '; ' + ($code) + ' ';
	    }
	    out += ' if (' + ($nextValid) + ') break; }  ';
	    it.compositeRule = $it.compositeRule = $wasComposite;
	    out += ' ' + ($closingBraces) + ' if (!' + ($nextValid) + ') {';
	  } else {
	    out += ' if (' + ($data) + '.length == 0) {';
	  }
	  var $$outStack = $$outStack || [];
	  $$outStack.push(out);
	  out = ''; /* istanbul ignore else */
	  if (it.createErrors !== false) {
	    out += ' { keyword: \'' + ('contains') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: {} ';
	    if (it.opts.messages !== false) {
	      out += ' , message: \'should contain a valid item\' ';
	    }
	    if (it.opts.verbose) {
	      out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	    }
	    out += ' } ';
	  } else {
	    out += ' {} ';
	  }
	  var __err = out;
	  out = $$outStack.pop();
	  if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	    if (it.async) {
	      out += ' throw new ValidationError([' + (__err) + ']); ';
	    } else {
	      out += ' validate.errors = [' + (__err) + ']; return false; ';
	    }
	  } else {
	    out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	  }
	  out += ' } else { ';
	  if ($nonEmptySchema) {
	    out += '  errors = ' + ($errs) + '; if (vErrors !== null) { if (' + ($errs) + ') vErrors.length = ' + ($errs) + '; else vErrors = null; } ';
	  }
	  if (it.opts.allErrors) {
	    out += ' } ';
	  }
	  out = it.util.cleanUpCode(out);
	  return out;
	}


/***/ },
/* 28 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_dependencies(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $errs = 'errs__' + $lvl;
	  var $it = it.util.copy(it);
	  var $closingBraces = '';
	  $it.level++;
	  var $nextValid = 'valid' + $it.level;
	  var $schemaDeps = {},
	    $propertyDeps = {},
	    $ownProperties = it.opts.ownProperties;
	  for ($property in $schema) {
	    var $sch = $schema[$property];
	    var $deps = Array.isArray($sch) ? $propertyDeps : $schemaDeps;
	    $deps[$property] = $sch;
	  }
	  out += 'var ' + ($errs) + ' = errors;';
	  var $currentErrorPath = it.errorPath;
	  out += 'var missing' + ($lvl) + ';';
	  for (var $property in $propertyDeps) {
	    $deps = $propertyDeps[$property];
	    if ($deps.length) {
	      out += ' if ( ' + ($data) + (it.util.getProperty($property)) + ' !== undefined ';
	      if ($ownProperties) {
	        out += ' && Object.prototype.hasOwnProperty.call(' + ($data) + ', \'' + (it.util.escapeQuotes($property)) + '\') ';
	      }
	      if ($breakOnError) {
	        out += ' && ( ';
	        var arr1 = $deps;
	        if (arr1) {
	          var $propertyKey, $i = -1,
	            l1 = arr1.length - 1;
	          while ($i < l1) {
	            $propertyKey = arr1[$i += 1];
	            if ($i) {
	              out += ' || ';
	            }
	            var $prop = it.util.getProperty($propertyKey),
	              $useData = $data + $prop;
	            out += ' ( ( ' + ($useData) + ' === undefined ';
	            if ($ownProperties) {
	              out += ' || ! Object.prototype.hasOwnProperty.call(' + ($data) + ', \'' + (it.util.escapeQuotes($propertyKey)) + '\') ';
	            }
	            out += ') && (missing' + ($lvl) + ' = ' + (it.util.toQuotedString(it.opts.jsonPointers ? $propertyKey : $prop)) + ') ) ';
	          }
	        }
	        out += ')) {  ';
	        var $propertyPath = 'missing' + $lvl,
	          $missingProperty = '\' + ' + $propertyPath + ' + \'';
	        if (it.opts._errorDataPathProperty) {
	          it.errorPath = it.opts.jsonPointers ? it.util.getPathExpr($currentErrorPath, $propertyPath, true) : $currentErrorPath + ' + ' + $propertyPath;
	        }
	        var $$outStack = $$outStack || [];
	        $$outStack.push(out);
	        out = ''; /* istanbul ignore else */
	        if (it.createErrors !== false) {
	          out += ' { keyword: \'' + ('dependencies') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { property: \'' + (it.util.escapeQuotes($property)) + '\', missingProperty: \'' + ($missingProperty) + '\', depsCount: ' + ($deps.length) + ', deps: \'' + (it.util.escapeQuotes($deps.length == 1 ? $deps[0] : $deps.join(", "))) + '\' } ';
	          if (it.opts.messages !== false) {
	            out += ' , message: \'should have ';
	            if ($deps.length == 1) {
	              out += 'property ' + (it.util.escapeQuotes($deps[0]));
	            } else {
	              out += 'properties ' + (it.util.escapeQuotes($deps.join(", ")));
	            }
	            out += ' when property ' + (it.util.escapeQuotes($property)) + ' is present\' ';
	          }
	          if (it.opts.verbose) {
	            out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	          }
	          out += ' } ';
	        } else {
	          out += ' {} ';
	        }
	        var __err = out;
	        out = $$outStack.pop();
	        if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	          if (it.async) {
	            out += ' throw new ValidationError([' + (__err) + ']); ';
	          } else {
	            out += ' validate.errors = [' + (__err) + ']; return false; ';
	          }
	        } else {
	          out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	        }
	      } else {
	        out += ' ) { ';
	        var arr2 = $deps;
	        if (arr2) {
	          var $propertyKey, i2 = -1,
	            l2 = arr2.length - 1;
	          while (i2 < l2) {
	            $propertyKey = arr2[i2 += 1];
	            var $prop = it.util.getProperty($propertyKey),
	              $missingProperty = it.util.escapeQuotes($propertyKey),
	              $useData = $data + $prop;
	            if (it.opts._errorDataPathProperty) {
	              it.errorPath = it.util.getPath($currentErrorPath, $propertyKey, it.opts.jsonPointers);
	            }
	            out += ' if ( ' + ($useData) + ' === undefined ';
	            if ($ownProperties) {
	              out += ' || ! Object.prototype.hasOwnProperty.call(' + ($data) + ', \'' + (it.util.escapeQuotes($propertyKey)) + '\') ';
	            }
	            out += ') {  var err =   '; /* istanbul ignore else */
	            if (it.createErrors !== false) {
	              out += ' { keyword: \'' + ('dependencies') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { property: \'' + (it.util.escapeQuotes($property)) + '\', missingProperty: \'' + ($missingProperty) + '\', depsCount: ' + ($deps.length) + ', deps: \'' + (it.util.escapeQuotes($deps.length == 1 ? $deps[0] : $deps.join(", "))) + '\' } ';
	              if (it.opts.messages !== false) {
	                out += ' , message: \'should have ';
	                if ($deps.length == 1) {
	                  out += 'property ' + (it.util.escapeQuotes($deps[0]));
	                } else {
	                  out += 'properties ' + (it.util.escapeQuotes($deps.join(", ")));
	                }
	                out += ' when property ' + (it.util.escapeQuotes($property)) + ' is present\' ';
	              }
	              if (it.opts.verbose) {
	                out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	              }
	              out += ' } ';
	            } else {
	              out += ' {} ';
	            }
	            out += ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; } ';
	          }
	        }
	      }
	      out += ' }   ';
	      if ($breakOnError) {
	        $closingBraces += '}';
	        out += ' else { ';
	      }
	    }
	  }
	  it.errorPath = $currentErrorPath;
	  var $currentBaseId = $it.baseId;
	  for (var $property in $schemaDeps) {
	    var $sch = $schemaDeps[$property];
	    if (it.util.schemaHasRules($sch, it.RULES.all)) {
	      out += ' ' + ($nextValid) + ' = true; if ( ' + ($data) + (it.util.getProperty($property)) + ' !== undefined ';
	      if ($ownProperties) {
	        out += ' && Object.prototype.hasOwnProperty.call(' + ($data) + ', \'' + (it.util.escapeQuotes($property)) + '\') ';
	      }
	      out += ') { ';
	      $it.schema = $sch;
	      $it.schemaPath = $schemaPath + it.util.getProperty($property);
	      $it.errSchemaPath = $errSchemaPath + '/' + it.util.escapeFragment($property);
	      out += '  ' + (it.validate($it)) + ' ';
	      $it.baseId = $currentBaseId;
	      out += ' }  ';
	      if ($breakOnError) {
	        out += ' if (' + ($nextValid) + ') { ';
	        $closingBraces += '}';
	      }
	    }
	  }
	  if ($breakOnError) {
	    out += '   ' + ($closingBraces) + ' if (' + ($errs) + ' == errors) {';
	  }
	  out = it.util.cleanUpCode(out);
	  return out;
	}


/***/ },
/* 29 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_enum(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $valid = 'valid' + $lvl;
	  var $isData = it.opts.$data && $schema && $schema.$data,
	    $schemaValue;
	  if ($isData) {
	    out += ' var schema' + ($lvl) + ' = ' + (it.util.getData($schema.$data, $dataLvl, it.dataPathArr)) + '; ';
	    $schemaValue = 'schema' + $lvl;
	  } else {
	    $schemaValue = $schema;
	  }
	  var $i = 'i' + $lvl,
	    $vSchema = 'schema' + $lvl;
	  if (!$isData) {
	    out += ' var ' + ($vSchema) + ' = validate.schema' + ($schemaPath) + ';';
	  }
	  out += 'var ' + ($valid) + ';';
	  if ($isData) {
	    out += ' if (schema' + ($lvl) + ' === undefined) ' + ($valid) + ' = true; else if (!Array.isArray(schema' + ($lvl) + ')) ' + ($valid) + ' = false; else {';
	  }
	  out += '' + ($valid) + ' = false;for (var ' + ($i) + '=0; ' + ($i) + '<' + ($vSchema) + '.length; ' + ($i) + '++) if (equal(' + ($data) + ', ' + ($vSchema) + '[' + ($i) + '])) { ' + ($valid) + ' = true; break; }';
	  if ($isData) {
	    out += '  }  ';
	  }
	  out += ' if (!' + ($valid) + ') {   ';
	  var $$outStack = $$outStack || [];
	  $$outStack.push(out);
	  out = ''; /* istanbul ignore else */
	  if (it.createErrors !== false) {
	    out += ' { keyword: \'' + ('enum') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { allowedValues: schema' + ($lvl) + ' } ';
	    if (it.opts.messages !== false) {
	      out += ' , message: \'should be equal to one of the allowed values\' ';
	    }
	    if (it.opts.verbose) {
	      out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	    }
	    out += ' } ';
	  } else {
	    out += ' {} ';
	  }
	  var __err = out;
	  out = $$outStack.pop();
	  if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	    if (it.async) {
	      out += ' throw new ValidationError([' + (__err) + ']); ';
	    } else {
	      out += ' validate.errors = [' + (__err) + ']; return false; ';
	    }
	  } else {
	    out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	  }
	  out += ' }';
	  if ($breakOnError) {
	    out += ' else { ';
	  }
	  return out;
	}


/***/ },
/* 30 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_format(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  if (it.opts.format === false) {
	    if ($breakOnError) {
	      out += ' if (true) { ';
	    }
	    return out;
	  }
	  var $isData = it.opts.$data && $schema && $schema.$data,
	    $schemaValue;
	  if ($isData) {
	    out += ' var schema' + ($lvl) + ' = ' + (it.util.getData($schema.$data, $dataLvl, it.dataPathArr)) + '; ';
	    $schemaValue = 'schema' + $lvl;
	  } else {
	    $schemaValue = $schema;
	  }
	  var $unknownFormats = it.opts.unknownFormats,
	    $allowUnknown = Array.isArray($unknownFormats);
	  if ($isData) {
	    var $format = 'format' + $lvl,
	      $isObject = 'isObject' + $lvl,
	      $formatType = 'formatType' + $lvl;
	    out += ' var ' + ($format) + ' = formats[' + ($schemaValue) + ']; var ' + ($isObject) + ' = typeof ' + ($format) + ' == \'object\' && !(' + ($format) + ' instanceof RegExp) && ' + ($format) + '.validate; var ' + ($formatType) + ' = ' + ($isObject) + ' && ' + ($format) + '.type || \'string\'; if (' + ($isObject) + ') { ';
	    if (it.async) {
	      out += ' var async' + ($lvl) + ' = ' + ($format) + '.async; ';
	    }
	    out += ' ' + ($format) + ' = ' + ($format) + '.validate; } if (  ';
	    if ($isData) {
	      out += ' (' + ($schemaValue) + ' !== undefined && typeof ' + ($schemaValue) + ' != \'string\') || ';
	    }
	    out += ' (';
	    if ($unknownFormats != 'ignore') {
	      out += ' (' + ($schemaValue) + ' && !' + ($format) + ' ';
	      if ($allowUnknown) {
	        out += ' && self._opts.unknownFormats.indexOf(' + ($schemaValue) + ') == -1 ';
	      }
	      out += ') || ';
	    }
	    out += ' (' + ($format) + ' && ' + ($formatType) + ' == \'' + ($ruleType) + '\' && !(typeof ' + ($format) + ' == \'function\' ? ';
	    if (it.async) {
	      out += ' (async' + ($lvl) + ' ? ' + (it.yieldAwait) + ' ' + ($format) + '(' + ($data) + ') : ' + ($format) + '(' + ($data) + ')) ';
	    } else {
	      out += ' ' + ($format) + '(' + ($data) + ') ';
	    }
	    out += ' : ' + ($format) + '.test(' + ($data) + '))))) {';
	  } else {
	    var $format = it.formats[$schema];
	    if (!$format) {
	      if ($unknownFormats == 'ignore') {
	        it.logger.warn('unknown format "' + $schema + '" ignored in schema at path "' + it.errSchemaPath + '"');
	        if ($breakOnError) {
	          out += ' if (true) { ';
	        }
	        return out;
	      } else if ($allowUnknown && $unknownFormats.indexOf($schema) >= 0) {
	        if ($breakOnError) {
	          out += ' if (true) { ';
	        }
	        return out;
	      } else {
	        throw new Error('unknown format "' + $schema + '" is used in schema at path "' + it.errSchemaPath + '"');
	      }
	    }
	    var $isObject = typeof $format == 'object' && !($format instanceof RegExp) && $format.validate;
	    var $formatType = $isObject && $format.type || 'string';
	    if ($isObject) {
	      var $async = $format.async === true;
	      $format = $format.validate;
	    }
	    if ($formatType != $ruleType) {
	      if ($breakOnError) {
	        out += ' if (true) { ';
	      }
	      return out;
	    }
	    if ($async) {
	      if (!it.async) throw new Error('async format in sync schema');
	      var $formatRef = 'formats' + it.util.getProperty($schema) + '.validate';
	      out += ' if (!(' + (it.yieldAwait) + ' ' + ($formatRef) + '(' + ($data) + '))) { ';
	    } else {
	      out += ' if (! ';
	      var $formatRef = 'formats' + it.util.getProperty($schema);
	      if ($isObject) $formatRef += '.validate';
	      if (typeof $format == 'function') {
	        out += ' ' + ($formatRef) + '(' + ($data) + ') ';
	      } else {
	        out += ' ' + ($formatRef) + '.test(' + ($data) + ') ';
	      }
	      out += ') { ';
	    }
	  }
	  var $$outStack = $$outStack || [];
	  $$outStack.push(out);
	  out = ''; /* istanbul ignore else */
	  if (it.createErrors !== false) {
	    out += ' { keyword: \'' + ('format') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { format:  ';
	    if ($isData) {
	      out += '' + ($schemaValue);
	    } else {
	      out += '' + (it.util.toQuotedString($schema));
	    }
	    out += '  } ';
	    if (it.opts.messages !== false) {
	      out += ' , message: \'should match format "';
	      if ($isData) {
	        out += '\' + ' + ($schemaValue) + ' + \'';
	      } else {
	        out += '' + (it.util.escapeQuotes($schema));
	      }
	      out += '"\' ';
	    }
	    if (it.opts.verbose) {
	      out += ' , schema:  ';
	      if ($isData) {
	        out += 'validate.schema' + ($schemaPath);
	      } else {
	        out += '' + (it.util.toQuotedString($schema));
	      }
	      out += '         , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	    }
	    out += ' } ';
	  } else {
	    out += ' {} ';
	  }
	  var __err = out;
	  out = $$outStack.pop();
	  if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	    if (it.async) {
	      out += ' throw new ValidationError([' + (__err) + ']); ';
	    } else {
	      out += ' validate.errors = [' + (__err) + ']; return false; ';
	    }
	  } else {
	    out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	  }
	  out += ' } ';
	  if ($breakOnError) {
	    out += ' else { ';
	  }
	  return out;
	}


/***/ },
/* 31 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_items(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $valid = 'valid' + $lvl;
	  var $errs = 'errs__' + $lvl;
	  var $it = it.util.copy(it);
	  var $closingBraces = '';
	  $it.level++;
	  var $nextValid = 'valid' + $it.level;
	  var $idx = 'i' + $lvl,
	    $dataNxt = $it.dataLevel = it.dataLevel + 1,
	    $nextData = 'data' + $dataNxt,
	    $currentBaseId = it.baseId;
	  out += 'var ' + ($errs) + ' = errors;var ' + ($valid) + ';';
	  if (Array.isArray($schema)) {
	    var $additionalItems = it.schema.additionalItems;
	    if ($additionalItems === false) {
	      out += ' ' + ($valid) + ' = ' + ($data) + '.length <= ' + ($schema.length) + '; ';
	      var $currErrSchemaPath = $errSchemaPath;
	      $errSchemaPath = it.errSchemaPath + '/additionalItems';
	      out += '  if (!' + ($valid) + ') {   ';
	      var $$outStack = $$outStack || [];
	      $$outStack.push(out);
	      out = ''; /* istanbul ignore else */
	      if (it.createErrors !== false) {
	        out += ' { keyword: \'' + ('additionalItems') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { limit: ' + ($schema.length) + ' } ';
	        if (it.opts.messages !== false) {
	          out += ' , message: \'should NOT have more than ' + ($schema.length) + ' items\' ';
	        }
	        if (it.opts.verbose) {
	          out += ' , schema: false , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	        }
	        out += ' } ';
	      } else {
	        out += ' {} ';
	      }
	      var __err = out;
	      out = $$outStack.pop();
	      if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	        if (it.async) {
	          out += ' throw new ValidationError([' + (__err) + ']); ';
	        } else {
	          out += ' validate.errors = [' + (__err) + ']; return false; ';
	        }
	      } else {
	        out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	      }
	      out += ' } ';
	      $errSchemaPath = $currErrSchemaPath;
	      if ($breakOnError) {
	        $closingBraces += '}';
	        out += ' else { ';
	      }
	    }
	    var arr1 = $schema;
	    if (arr1) {
	      var $sch, $i = -1,
	        l1 = arr1.length - 1;
	      while ($i < l1) {
	        $sch = arr1[$i += 1];
	        if (it.util.schemaHasRules($sch, it.RULES.all)) {
	          out += ' ' + ($nextValid) + ' = true; if (' + ($data) + '.length > ' + ($i) + ') { ';
	          var $passData = $data + '[' + $i + ']';
	          $it.schema = $sch;
	          $it.schemaPath = $schemaPath + '[' + $i + ']';
	          $it.errSchemaPath = $errSchemaPath + '/' + $i;
	          $it.errorPath = it.util.getPathExpr(it.errorPath, $i, it.opts.jsonPointers, true);
	          $it.dataPathArr[$dataNxt] = $i;
	          var $code = it.validate($it);
	          $it.baseId = $currentBaseId;
	          if (it.util.varOccurences($code, $nextData) < 2) {
	            out += ' ' + (it.util.varReplace($code, $nextData, $passData)) + ' ';
	          } else {
	            out += ' var ' + ($nextData) + ' = ' + ($passData) + '; ' + ($code) + ' ';
	          }
	          out += ' }  ';
	          if ($breakOnError) {
	            out += ' if (' + ($nextValid) + ') { ';
	            $closingBraces += '}';
	          }
	        }
	      }
	    }
	    if (typeof $additionalItems == 'object' && it.util.schemaHasRules($additionalItems, it.RULES.all)) {
	      $it.schema = $additionalItems;
	      $it.schemaPath = it.schemaPath + '.additionalItems';
	      $it.errSchemaPath = it.errSchemaPath + '/additionalItems';
	      out += ' ' + ($nextValid) + ' = true; if (' + ($data) + '.length > ' + ($schema.length) + ') {  for (var ' + ($idx) + ' = ' + ($schema.length) + '; ' + ($idx) + ' < ' + ($data) + '.length; ' + ($idx) + '++) { ';
	      $it.errorPath = it.util.getPathExpr(it.errorPath, $idx, it.opts.jsonPointers, true);
	      var $passData = $data + '[' + $idx + ']';
	      $it.dataPathArr[$dataNxt] = $idx;
	      var $code = it.validate($it);
	      $it.baseId = $currentBaseId;
	      if (it.util.varOccurences($code, $nextData) < 2) {
	        out += ' ' + (it.util.varReplace($code, $nextData, $passData)) + ' ';
	      } else {
	        out += ' var ' + ($nextData) + ' = ' + ($passData) + '; ' + ($code) + ' ';
	      }
	      if ($breakOnError) {
	        out += ' if (!' + ($nextValid) + ') break; ';
	      }
	      out += ' } }  ';
	      if ($breakOnError) {
	        out += ' if (' + ($nextValid) + ') { ';
	        $closingBraces += '}';
	      }
	    }
	  } else if (it.util.schemaHasRules($schema, it.RULES.all)) {
	    $it.schema = $schema;
	    $it.schemaPath = $schemaPath;
	    $it.errSchemaPath = $errSchemaPath;
	    out += '  for (var ' + ($idx) + ' = ' + (0) + '; ' + ($idx) + ' < ' + ($data) + '.length; ' + ($idx) + '++) { ';
	    $it.errorPath = it.util.getPathExpr(it.errorPath, $idx, it.opts.jsonPointers, true);
	    var $passData = $data + '[' + $idx + ']';
	    $it.dataPathArr[$dataNxt] = $idx;
	    var $code = it.validate($it);
	    $it.baseId = $currentBaseId;
	    if (it.util.varOccurences($code, $nextData) < 2) {
	      out += ' ' + (it.util.varReplace($code, $nextData, $passData)) + ' ';
	    } else {
	      out += ' var ' + ($nextData) + ' = ' + ($passData) + '; ' + ($code) + ' ';
	    }
	    if ($breakOnError) {
	      out += ' if (!' + ($nextValid) + ') break; ';
	    }
	    out += ' }';
	  }
	  if ($breakOnError) {
	    out += ' ' + ($closingBraces) + ' if (' + ($errs) + ' == errors) {';
	  }
	  out = it.util.cleanUpCode(out);
	  return out;
	}


/***/ },
/* 32 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate__limit(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $errorKeyword;
	  var $data = 'data' + ($dataLvl || '');
	  var $isData = it.opts.$data && $schema && $schema.$data,
	    $schemaValue;
	  if ($isData) {
	    out += ' var schema' + ($lvl) + ' = ' + (it.util.getData($schema.$data, $dataLvl, it.dataPathArr)) + '; ';
	    $schemaValue = 'schema' + $lvl;
	  } else {
	    $schemaValue = $schema;
	  }
	  var $isMax = $keyword == 'maximum',
	    $exclusiveKeyword = $isMax ? 'exclusiveMaximum' : 'exclusiveMinimum',
	    $schemaExcl = it.schema[$exclusiveKeyword],
	    $isDataExcl = it.opts.$data && $schemaExcl && $schemaExcl.$data,
	    $op = $isMax ? '<' : '>',
	    $notOp = $isMax ? '>' : '<',
	    $errorKeyword = undefined;
	  if ($isDataExcl) {
	    var $schemaValueExcl = it.util.getData($schemaExcl.$data, $dataLvl, it.dataPathArr),
	      $exclusive = 'exclusive' + $lvl,
	      $exclType = 'exclType' + $lvl,
	      $exclIsNumber = 'exclIsNumber' + $lvl,
	      $opExpr = 'op' + $lvl,
	      $opStr = '\' + ' + $opExpr + ' + \'';
	    out += ' var schemaExcl' + ($lvl) + ' = ' + ($schemaValueExcl) + '; ';
	    $schemaValueExcl = 'schemaExcl' + $lvl;
	    out += ' var ' + ($exclusive) + '; var ' + ($exclType) + ' = typeof ' + ($schemaValueExcl) + '; if (' + ($exclType) + ' != \'boolean\' && ' + ($exclType) + ' != \'undefined\' && ' + ($exclType) + ' != \'number\') { ';
	    var $errorKeyword = $exclusiveKeyword;
	    var $$outStack = $$outStack || [];
	    $$outStack.push(out);
	    out = ''; /* istanbul ignore else */
	    if (it.createErrors !== false) {
	      out += ' { keyword: \'' + ($errorKeyword || '_exclusiveLimit') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: {} ';
	      if (it.opts.messages !== false) {
	        out += ' , message: \'' + ($exclusiveKeyword) + ' should be boolean\' ';
	      }
	      if (it.opts.verbose) {
	        out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	      }
	      out += ' } ';
	    } else {
	      out += ' {} ';
	    }
	    var __err = out;
	    out = $$outStack.pop();
	    if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	      if (it.async) {
	        out += ' throw new ValidationError([' + (__err) + ']); ';
	      } else {
	        out += ' validate.errors = [' + (__err) + ']; return false; ';
	      }
	    } else {
	      out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	    }
	    out += ' } else if ( ';
	    if ($isData) {
	      out += ' (' + ($schemaValue) + ' !== undefined && typeof ' + ($schemaValue) + ' != \'number\') || ';
	    }
	    out += ' ' + ($exclType) + ' == \'number\' ? ( (' + ($exclusive) + ' = ' + ($schemaValue) + ' === undefined || ' + ($schemaValueExcl) + ' ' + ($op) + '= ' + ($schemaValue) + ') ? ' + ($data) + ' ' + ($notOp) + '= ' + ($schemaValueExcl) + ' : ' + ($data) + ' ' + ($notOp) + ' ' + ($schemaValue) + ' ) : ( (' + ($exclusive) + ' = ' + ($schemaValueExcl) + ' === true) ? ' + ($data) + ' ' + ($notOp) + '= ' + ($schemaValue) + ' : ' + ($data) + ' ' + ($notOp) + ' ' + ($schemaValue) + ' ) || ' + ($data) + ' !== ' + ($data) + ') { var op' + ($lvl) + ' = ' + ($exclusive) + ' ? \'' + ($op) + '\' : \'' + ($op) + '=\';';
	  } else {
	    var $exclIsNumber = typeof $schemaExcl == 'number',
	      $opStr = $op;
	    if ($exclIsNumber && $isData) {
	      var $opExpr = '\'' + $opStr + '\'';
	      out += ' if ( ';
	      if ($isData) {
	        out += ' (' + ($schemaValue) + ' !== undefined && typeof ' + ($schemaValue) + ' != \'number\') || ';
	      }
	      out += ' ( ' + ($schemaValue) + ' === undefined || ' + ($schemaExcl) + ' ' + ($op) + '= ' + ($schemaValue) + ' ? ' + ($data) + ' ' + ($notOp) + '= ' + ($schemaExcl) + ' : ' + ($data) + ' ' + ($notOp) + ' ' + ($schemaValue) + ' ) || ' + ($data) + ' !== ' + ($data) + ') { ';
	    } else {
	      if ($exclIsNumber && $schema === undefined) {
	        $exclusive = true;
	        $errorKeyword = $exclusiveKeyword;
	        $errSchemaPath = it.errSchemaPath + '/' + $exclusiveKeyword;
	        $schemaValue = $schemaExcl;
	        $notOp += '=';
	      } else {
	        if ($exclIsNumber) $schemaValue = Math[$isMax ? 'min' : 'max']($schemaExcl, $schema);
	        if ($schemaExcl === ($exclIsNumber ? $schemaValue : true)) {
	          $exclusive = true;
	          $errorKeyword = $exclusiveKeyword;
	          $errSchemaPath = it.errSchemaPath + '/' + $exclusiveKeyword;
	          $notOp += '=';
	        } else {
	          $exclusive = false;
	          $opStr += '=';
	        }
	      }
	      var $opExpr = '\'' + $opStr + '\'';
	      out += ' if ( ';
	      if ($isData) {
	        out += ' (' + ($schemaValue) + ' !== undefined && typeof ' + ($schemaValue) + ' != \'number\') || ';
	      }
	      out += ' ' + ($data) + ' ' + ($notOp) + ' ' + ($schemaValue) + ' || ' + ($data) + ' !== ' + ($data) + ') { ';
	    }
	  }
	  $errorKeyword = $errorKeyword || $keyword;
	  var $$outStack = $$outStack || [];
	  $$outStack.push(out);
	  out = ''; /* istanbul ignore else */
	  if (it.createErrors !== false) {
	    out += ' { keyword: \'' + ($errorKeyword || '_limit') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { comparison: ' + ($opExpr) + ', limit: ' + ($schemaValue) + ', exclusive: ' + ($exclusive) + ' } ';
	    if (it.opts.messages !== false) {
	      out += ' , message: \'should be ' + ($opStr) + ' ';
	      if ($isData) {
	        out += '\' + ' + ($schemaValue);
	      } else {
	        out += '' + ($schemaValue) + '\'';
	      }
	    }
	    if (it.opts.verbose) {
	      out += ' , schema:  ';
	      if ($isData) {
	        out += 'validate.schema' + ($schemaPath);
	      } else {
	        out += '' + ($schema);
	      }
	      out += '         , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	    }
	    out += ' } ';
	  } else {
	    out += ' {} ';
	  }
	  var __err = out;
	  out = $$outStack.pop();
	  if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	    if (it.async) {
	      out += ' throw new ValidationError([' + (__err) + ']); ';
	    } else {
	      out += ' validate.errors = [' + (__err) + ']; return false; ';
	    }
	  } else {
	    out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	  }
	  out += ' } ';
	  if ($breakOnError) {
	    out += ' else { ';
	  }
	  return out;
	}


/***/ },
/* 33 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate__limitItems(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $errorKeyword;
	  var $data = 'data' + ($dataLvl || '');
	  var $isData = it.opts.$data && $schema && $schema.$data,
	    $schemaValue;
	  if ($isData) {
	    out += ' var schema' + ($lvl) + ' = ' + (it.util.getData($schema.$data, $dataLvl, it.dataPathArr)) + '; ';
	    $schemaValue = 'schema' + $lvl;
	  } else {
	    $schemaValue = $schema;
	  }
	  var $op = $keyword == 'maxItems' ? '>' : '<';
	  out += 'if ( ';
	  if ($isData) {
	    out += ' (' + ($schemaValue) + ' !== undefined && typeof ' + ($schemaValue) + ' != \'number\') || ';
	  }
	  out += ' ' + ($data) + '.length ' + ($op) + ' ' + ($schemaValue) + ') { ';
	  var $errorKeyword = $keyword;
	  var $$outStack = $$outStack || [];
	  $$outStack.push(out);
	  out = ''; /* istanbul ignore else */
	  if (it.createErrors !== false) {
	    out += ' { keyword: \'' + ($errorKeyword || '_limitItems') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { limit: ' + ($schemaValue) + ' } ';
	    if (it.opts.messages !== false) {
	      out += ' , message: \'should NOT have ';
	      if ($keyword == 'maxItems') {
	        out += 'more';
	      } else {
	        out += 'less';
	      }
	      out += ' than ';
	      if ($isData) {
	        out += '\' + ' + ($schemaValue) + ' + \'';
	      } else {
	        out += '' + ($schema);
	      }
	      out += ' items\' ';
	    }
	    if (it.opts.verbose) {
	      out += ' , schema:  ';
	      if ($isData) {
	        out += 'validate.schema' + ($schemaPath);
	      } else {
	        out += '' + ($schema);
	      }
	      out += '         , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	    }
	    out += ' } ';
	  } else {
	    out += ' {} ';
	  }
	  var __err = out;
	  out = $$outStack.pop();
	  if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	    if (it.async) {
	      out += ' throw new ValidationError([' + (__err) + ']); ';
	    } else {
	      out += ' validate.errors = [' + (__err) + ']; return false; ';
	    }
	  } else {
	    out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	  }
	  out += '} ';
	  if ($breakOnError) {
	    out += ' else { ';
	  }
	  return out;
	}


/***/ },
/* 34 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate__limitLength(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $errorKeyword;
	  var $data = 'data' + ($dataLvl || '');
	  var $isData = it.opts.$data && $schema && $schema.$data,
	    $schemaValue;
	  if ($isData) {
	    out += ' var schema' + ($lvl) + ' = ' + (it.util.getData($schema.$data, $dataLvl, it.dataPathArr)) + '; ';
	    $schemaValue = 'schema' + $lvl;
	  } else {
	    $schemaValue = $schema;
	  }
	  var $op = $keyword == 'maxLength' ? '>' : '<';
	  out += 'if ( ';
	  if ($isData) {
	    out += ' (' + ($schemaValue) + ' !== undefined && typeof ' + ($schemaValue) + ' != \'number\') || ';
	  }
	  if (it.opts.unicode === false) {
	    out += ' ' + ($data) + '.length ';
	  } else {
	    out += ' ucs2length(' + ($data) + ') ';
	  }
	  out += ' ' + ($op) + ' ' + ($schemaValue) + ') { ';
	  var $errorKeyword = $keyword;
	  var $$outStack = $$outStack || [];
	  $$outStack.push(out);
	  out = ''; /* istanbul ignore else */
	  if (it.createErrors !== false) {
	    out += ' { keyword: \'' + ($errorKeyword || '_limitLength') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { limit: ' + ($schemaValue) + ' } ';
	    if (it.opts.messages !== false) {
	      out += ' , message: \'should NOT be ';
	      if ($keyword == 'maxLength') {
	        out += 'longer';
	      } else {
	        out += 'shorter';
	      }
	      out += ' than ';
	      if ($isData) {
	        out += '\' + ' + ($schemaValue) + ' + \'';
	      } else {
	        out += '' + ($schema);
	      }
	      out += ' characters\' ';
	    }
	    if (it.opts.verbose) {
	      out += ' , schema:  ';
	      if ($isData) {
	        out += 'validate.schema' + ($schemaPath);
	      } else {
	        out += '' + ($schema);
	      }
	      out += '         , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	    }
	    out += ' } ';
	  } else {
	    out += ' {} ';
	  }
	  var __err = out;
	  out = $$outStack.pop();
	  if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	    if (it.async) {
	      out += ' throw new ValidationError([' + (__err) + ']); ';
	    } else {
	      out += ' validate.errors = [' + (__err) + ']; return false; ';
	    }
	  } else {
	    out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	  }
	  out += '} ';
	  if ($breakOnError) {
	    out += ' else { ';
	  }
	  return out;
	}


/***/ },
/* 35 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate__limitProperties(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $errorKeyword;
	  var $data = 'data' + ($dataLvl || '');
	  var $isData = it.opts.$data && $schema && $schema.$data,
	    $schemaValue;
	  if ($isData) {
	    out += ' var schema' + ($lvl) + ' = ' + (it.util.getData($schema.$data, $dataLvl, it.dataPathArr)) + '; ';
	    $schemaValue = 'schema' + $lvl;
	  } else {
	    $schemaValue = $schema;
	  }
	  var $op = $keyword == 'maxProperties' ? '>' : '<';
	  out += 'if ( ';
	  if ($isData) {
	    out += ' (' + ($schemaValue) + ' !== undefined && typeof ' + ($schemaValue) + ' != \'number\') || ';
	  }
	  out += ' Object.keys(' + ($data) + ').length ' + ($op) + ' ' + ($schemaValue) + ') { ';
	  var $errorKeyword = $keyword;
	  var $$outStack = $$outStack || [];
	  $$outStack.push(out);
	  out = ''; /* istanbul ignore else */
	  if (it.createErrors !== false) {
	    out += ' { keyword: \'' + ($errorKeyword || '_limitProperties') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { limit: ' + ($schemaValue) + ' } ';
	    if (it.opts.messages !== false) {
	      out += ' , message: \'should NOT have ';
	      if ($keyword == 'maxProperties') {
	        out += 'more';
	      } else {
	        out += 'less';
	      }
	      out += ' than ';
	      if ($isData) {
	        out += '\' + ' + ($schemaValue) + ' + \'';
	      } else {
	        out += '' + ($schema);
	      }
	      out += ' properties\' ';
	    }
	    if (it.opts.verbose) {
	      out += ' , schema:  ';
	      if ($isData) {
	        out += 'validate.schema' + ($schemaPath);
	      } else {
	        out += '' + ($schema);
	      }
	      out += '         , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	    }
	    out += ' } ';
	  } else {
	    out += ' {} ';
	  }
	  var __err = out;
	  out = $$outStack.pop();
	  if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	    if (it.async) {
	      out += ' throw new ValidationError([' + (__err) + ']); ';
	    } else {
	      out += ' validate.errors = [' + (__err) + ']; return false; ';
	    }
	  } else {
	    out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	  }
	  out += '} ';
	  if ($breakOnError) {
	    out += ' else { ';
	  }
	  return out;
	}


/***/ },
/* 36 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_multipleOf(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $isData = it.opts.$data && $schema && $schema.$data,
	    $schemaValue;
	  if ($isData) {
	    out += ' var schema' + ($lvl) + ' = ' + (it.util.getData($schema.$data, $dataLvl, it.dataPathArr)) + '; ';
	    $schemaValue = 'schema' + $lvl;
	  } else {
	    $schemaValue = $schema;
	  }
	  out += 'var division' + ($lvl) + ';if (';
	  if ($isData) {
	    out += ' ' + ($schemaValue) + ' !== undefined && ( typeof ' + ($schemaValue) + ' != \'number\' || ';
	  }
	  out += ' (division' + ($lvl) + ' = ' + ($data) + ' / ' + ($schemaValue) + ', ';
	  if (it.opts.multipleOfPrecision) {
	    out += ' Math.abs(Math.round(division' + ($lvl) + ') - division' + ($lvl) + ') > 1e-' + (it.opts.multipleOfPrecision) + ' ';
	  } else {
	    out += ' division' + ($lvl) + ' !== parseInt(division' + ($lvl) + ') ';
	  }
	  out += ' ) ';
	  if ($isData) {
	    out += '  )  ';
	  }
	  out += ' ) {   ';
	  var $$outStack = $$outStack || [];
	  $$outStack.push(out);
	  out = ''; /* istanbul ignore else */
	  if (it.createErrors !== false) {
	    out += ' { keyword: \'' + ('multipleOf') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { multipleOf: ' + ($schemaValue) + ' } ';
	    if (it.opts.messages !== false) {
	      out += ' , message: \'should be multiple of ';
	      if ($isData) {
	        out += '\' + ' + ($schemaValue);
	      } else {
	        out += '' + ($schemaValue) + '\'';
	      }
	    }
	    if (it.opts.verbose) {
	      out += ' , schema:  ';
	      if ($isData) {
	        out += 'validate.schema' + ($schemaPath);
	      } else {
	        out += '' + ($schema);
	      }
	      out += '         , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	    }
	    out += ' } ';
	  } else {
	    out += ' {} ';
	  }
	  var __err = out;
	  out = $$outStack.pop();
	  if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	    if (it.async) {
	      out += ' throw new ValidationError([' + (__err) + ']); ';
	    } else {
	      out += ' validate.errors = [' + (__err) + ']; return false; ';
	    }
	  } else {
	    out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	  }
	  out += '} ';
	  if ($breakOnError) {
	    out += ' else { ';
	  }
	  return out;
	}


/***/ },
/* 37 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_not(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $errs = 'errs__' + $lvl;
	  var $it = it.util.copy(it);
	  $it.level++;
	  var $nextValid = 'valid' + $it.level;
	  if (it.util.schemaHasRules($schema, it.RULES.all)) {
	    $it.schema = $schema;
	    $it.schemaPath = $schemaPath;
	    $it.errSchemaPath = $errSchemaPath;
	    out += ' var ' + ($errs) + ' = errors;  ';
	    var $wasComposite = it.compositeRule;
	    it.compositeRule = $it.compositeRule = true;
	    $it.createErrors = false;
	    var $allErrorsOption;
	    if ($it.opts.allErrors) {
	      $allErrorsOption = $it.opts.allErrors;
	      $it.opts.allErrors = false;
	    }
	    out += ' ' + (it.validate($it)) + ' ';
	    $it.createErrors = true;
	    if ($allErrorsOption) $it.opts.allErrors = $allErrorsOption;
	    it.compositeRule = $it.compositeRule = $wasComposite;
	    out += ' if (' + ($nextValid) + ') {   ';
	    var $$outStack = $$outStack || [];
	    $$outStack.push(out);
	    out = ''; /* istanbul ignore else */
	    if (it.createErrors !== false) {
	      out += ' { keyword: \'' + ('not') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: {} ';
	      if (it.opts.messages !== false) {
	        out += ' , message: \'should NOT be valid\' ';
	      }
	      if (it.opts.verbose) {
	        out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	      }
	      out += ' } ';
	    } else {
	      out += ' {} ';
	    }
	    var __err = out;
	    out = $$outStack.pop();
	    if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	      if (it.async) {
	        out += ' throw new ValidationError([' + (__err) + ']); ';
	      } else {
	        out += ' validate.errors = [' + (__err) + ']; return false; ';
	      }
	    } else {
	      out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	    }
	    out += ' } else {  errors = ' + ($errs) + '; if (vErrors !== null) { if (' + ($errs) + ') vErrors.length = ' + ($errs) + '; else vErrors = null; } ';
	    if (it.opts.allErrors) {
	      out += ' } ';
	    }
	  } else {
	    out += '  var err =   '; /* istanbul ignore else */
	    if (it.createErrors !== false) {
	      out += ' { keyword: \'' + ('not') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: {} ';
	      if (it.opts.messages !== false) {
	        out += ' , message: \'should NOT be valid\' ';
	      }
	      if (it.opts.verbose) {
	        out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	      }
	      out += ' } ';
	    } else {
	      out += ' {} ';
	    }
	    out += ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	    if ($breakOnError) {
	      out += ' if (false) { ';
	    }
	  }
	  return out;
	}


/***/ },
/* 38 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_oneOf(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $valid = 'valid' + $lvl;
	  var $errs = 'errs__' + $lvl;
	  var $it = it.util.copy(it);
	  var $closingBraces = '';
	  $it.level++;
	  var $nextValid = 'valid' + $it.level;
	  out += 'var ' + ($errs) + ' = errors;var prevValid' + ($lvl) + ' = false;var ' + ($valid) + ' = false;';
	  var $currentBaseId = $it.baseId;
	  var $wasComposite = it.compositeRule;
	  it.compositeRule = $it.compositeRule = true;
	  var arr1 = $schema;
	  if (arr1) {
	    var $sch, $i = -1,
	      l1 = arr1.length - 1;
	    while ($i < l1) {
	      $sch = arr1[$i += 1];
	      if (it.util.schemaHasRules($sch, it.RULES.all)) {
	        $it.schema = $sch;
	        $it.schemaPath = $schemaPath + '[' + $i + ']';
	        $it.errSchemaPath = $errSchemaPath + '/' + $i;
	        out += '  ' + (it.validate($it)) + ' ';
	        $it.baseId = $currentBaseId;
	      } else {
	        out += ' var ' + ($nextValid) + ' = true; ';
	      }
	      if ($i) {
	        out += ' if (' + ($nextValid) + ' && prevValid' + ($lvl) + ') ' + ($valid) + ' = false; else { ';
	        $closingBraces += '}';
	      }
	      out += ' if (' + ($nextValid) + ') ' + ($valid) + ' = prevValid' + ($lvl) + ' = true;';
	    }
	  }
	  it.compositeRule = $it.compositeRule = $wasComposite;
	  out += '' + ($closingBraces) + 'if (!' + ($valid) + ') {   var err =   '; /* istanbul ignore else */
	  if (it.createErrors !== false) {
	    out += ' { keyword: \'' + ('oneOf') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: {} ';
	    if (it.opts.messages !== false) {
	      out += ' , message: \'should match exactly one schema in oneOf\' ';
	    }
	    if (it.opts.verbose) {
	      out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	    }
	    out += ' } ';
	  } else {
	    out += ' {} ';
	  }
	  out += ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	  if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	    if (it.async) {
	      out += ' throw new ValidationError(vErrors); ';
	    } else {
	      out += ' validate.errors = vErrors; return false; ';
	    }
	  }
	  out += '} else {  errors = ' + ($errs) + '; if (vErrors !== null) { if (' + ($errs) + ') vErrors.length = ' + ($errs) + '; else vErrors = null; }';
	  if (it.opts.allErrors) {
	    out += ' } ';
	  }
	  return out;
	}


/***/ },
/* 39 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_pattern(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $isData = it.opts.$data && $schema && $schema.$data,
	    $schemaValue;
	  if ($isData) {
	    out += ' var schema' + ($lvl) + ' = ' + (it.util.getData($schema.$data, $dataLvl, it.dataPathArr)) + '; ';
	    $schemaValue = 'schema' + $lvl;
	  } else {
	    $schemaValue = $schema;
	  }
	  var $regexp = $isData ? '(new RegExp(' + $schemaValue + '))' : it.usePattern($schema);
	  out += 'if ( ';
	  if ($isData) {
	    out += ' (' + ($schemaValue) + ' !== undefined && typeof ' + ($schemaValue) + ' != \'string\') || ';
	  }
	  out += ' !' + ($regexp) + '.test(' + ($data) + ') ) {   ';
	  var $$outStack = $$outStack || [];
	  $$outStack.push(out);
	  out = ''; /* istanbul ignore else */
	  if (it.createErrors !== false) {
	    out += ' { keyword: \'' + ('pattern') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { pattern:  ';
	    if ($isData) {
	      out += '' + ($schemaValue);
	    } else {
	      out += '' + (it.util.toQuotedString($schema));
	    }
	    out += '  } ';
	    if (it.opts.messages !== false) {
	      out += ' , message: \'should match pattern "';
	      if ($isData) {
	        out += '\' + ' + ($schemaValue) + ' + \'';
	      } else {
	        out += '' + (it.util.escapeQuotes($schema));
	      }
	      out += '"\' ';
	    }
	    if (it.opts.verbose) {
	      out += ' , schema:  ';
	      if ($isData) {
	        out += 'validate.schema' + ($schemaPath);
	      } else {
	        out += '' + (it.util.toQuotedString($schema));
	      }
	      out += '         , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	    }
	    out += ' } ';
	  } else {
	    out += ' {} ';
	  }
	  var __err = out;
	  out = $$outStack.pop();
	  if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	    if (it.async) {
	      out += ' throw new ValidationError([' + (__err) + ']); ';
	    } else {
	      out += ' validate.errors = [' + (__err) + ']; return false; ';
	    }
	  } else {
	    out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	  }
	  out += '} ';
	  if ($breakOnError) {
	    out += ' else { ';
	  }
	  return out;
	}


/***/ },
/* 40 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_properties(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $valid = 'valid' + $lvl;
	  var $errs = 'errs__' + $lvl;
	  var $it = it.util.copy(it);
	  var $closingBraces = '';
	  $it.level++;
	  var $nextValid = 'valid' + $it.level;
	  var $key = 'key' + $lvl,
	    $idx = 'idx' + $lvl,
	    $dataNxt = $it.dataLevel = it.dataLevel + 1,
	    $nextData = 'data' + $dataNxt,
	    $dataProperties = 'dataProperties' + $lvl;
	  var $schemaKeys = Object.keys($schema || {}),
	    $pProperties = it.schema.patternProperties || {},
	    $pPropertyKeys = Object.keys($pProperties),
	    $aProperties = it.schema.additionalProperties,
	    $someProperties = $schemaKeys.length || $pPropertyKeys.length,
	    $noAdditional = $aProperties === false,
	    $additionalIsSchema = typeof $aProperties == 'object' && Object.keys($aProperties).length,
	    $removeAdditional = it.opts.removeAdditional,
	    $checkAdditional = $noAdditional || $additionalIsSchema || $removeAdditional,
	    $ownProperties = it.opts.ownProperties,
	    $currentBaseId = it.baseId;
	  var $required = it.schema.required;
	  if ($required && !(it.opts.v5 && $required.$data) && $required.length < it.opts.loopRequired) var $requiredHash = it.util.toHash($required);
	  if (it.opts.patternGroups) {
	    var $pgProperties = it.schema.patternGroups || {},
	      $pgPropertyKeys = Object.keys($pgProperties);
	  }
	  out += 'var ' + ($errs) + ' = errors;var ' + ($nextValid) + ' = true;';
	  if ($ownProperties) {
	    out += ' var ' + ($dataProperties) + ' = undefined;';
	  }
	  if ($checkAdditional) {
	    if ($ownProperties) {
	      out += ' ' + ($dataProperties) + ' = ' + ($dataProperties) + ' || Object.keys(' + ($data) + '); for (var ' + ($idx) + '=0; ' + ($idx) + '<' + ($dataProperties) + '.length; ' + ($idx) + '++) { var ' + ($key) + ' = ' + ($dataProperties) + '[' + ($idx) + ']; ';
	    } else {
	      out += ' for (var ' + ($key) + ' in ' + ($data) + ') { ';
	    }
	    if ($someProperties) {
	      out += ' var isAdditional' + ($lvl) + ' = !(false ';
	      if ($schemaKeys.length) {
	        if ($schemaKeys.length > 5) {
	          out += ' || validate.schema' + ($schemaPath) + '[' + ($key) + '] ';
	        } else {
	          var arr1 = $schemaKeys;
	          if (arr1) {
	            var $propertyKey, i1 = -1,
	              l1 = arr1.length - 1;
	            while (i1 < l1) {
	              $propertyKey = arr1[i1 += 1];
	              out += ' || ' + ($key) + ' == ' + (it.util.toQuotedString($propertyKey)) + ' ';
	            }
	          }
	        }
	      }
	      if ($pPropertyKeys.length) {
	        var arr2 = $pPropertyKeys;
	        if (arr2) {
	          var $pProperty, $i = -1,
	            l2 = arr2.length - 1;
	          while ($i < l2) {
	            $pProperty = arr2[$i += 1];
	            out += ' || ' + (it.usePattern($pProperty)) + '.test(' + ($key) + ') ';
	          }
	        }
	      }
	      if (it.opts.patternGroups && $pgPropertyKeys.length) {
	        var arr3 = $pgPropertyKeys;
	        if (arr3) {
	          var $pgProperty, $i = -1,
	            l3 = arr3.length - 1;
	          while ($i < l3) {
	            $pgProperty = arr3[$i += 1];
	            out += ' || ' + (it.usePattern($pgProperty)) + '.test(' + ($key) + ') ';
	          }
	        }
	      }
	      out += ' ); if (isAdditional' + ($lvl) + ') { ';
	    }
	    if ($removeAdditional == 'all') {
	      out += ' delete ' + ($data) + '[' + ($key) + ']; ';
	    } else {
	      var $currentErrorPath = it.errorPath;
	      var $additionalProperty = '\' + ' + $key + ' + \'';
	      if (it.opts._errorDataPathProperty) {
	        it.errorPath = it.util.getPathExpr(it.errorPath, $key, it.opts.jsonPointers);
	      }
	      if ($noAdditional) {
	        if ($removeAdditional) {
	          out += ' delete ' + ($data) + '[' + ($key) + ']; ';
	        } else {
	          out += ' ' + ($nextValid) + ' = false; ';
	          var $currErrSchemaPath = $errSchemaPath;
	          $errSchemaPath = it.errSchemaPath + '/additionalProperties';
	          var $$outStack = $$outStack || [];
	          $$outStack.push(out);
	          out = ''; /* istanbul ignore else */
	          if (it.createErrors !== false) {
	            out += ' { keyword: \'' + ('additionalProperties') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { additionalProperty: \'' + ($additionalProperty) + '\' } ';
	            if (it.opts.messages !== false) {
	              out += ' , message: \'should NOT have additional properties\' ';
	            }
	            if (it.opts.verbose) {
	              out += ' , schema: false , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	            }
	            out += ' } ';
	          } else {
	            out += ' {} ';
	          }
	          var __err = out;
	          out = $$outStack.pop();
	          if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	            if (it.async) {
	              out += ' throw new ValidationError([' + (__err) + ']); ';
	            } else {
	              out += ' validate.errors = [' + (__err) + ']; return false; ';
	            }
	          } else {
	            out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	          }
	          $errSchemaPath = $currErrSchemaPath;
	          if ($breakOnError) {
	            out += ' break; ';
	          }
	        }
	      } else if ($additionalIsSchema) {
	        if ($removeAdditional == 'failing') {
	          out += ' var ' + ($errs) + ' = errors;  ';
	          var $wasComposite = it.compositeRule;
	          it.compositeRule = $it.compositeRule = true;
	          $it.schema = $aProperties;
	          $it.schemaPath = it.schemaPath + '.additionalProperties';
	          $it.errSchemaPath = it.errSchemaPath + '/additionalProperties';
	          $it.errorPath = it.opts._errorDataPathProperty ? it.errorPath : it.util.getPathExpr(it.errorPath, $key, it.opts.jsonPointers);
	          var $passData = $data + '[' + $key + ']';
	          $it.dataPathArr[$dataNxt] = $key;
	          var $code = it.validate($it);
	          $it.baseId = $currentBaseId;
	          if (it.util.varOccurences($code, $nextData) < 2) {
	            out += ' ' + (it.util.varReplace($code, $nextData, $passData)) + ' ';
	          } else {
	            out += ' var ' + ($nextData) + ' = ' + ($passData) + '; ' + ($code) + ' ';
	          }
	          out += ' if (!' + ($nextValid) + ') { errors = ' + ($errs) + '; if (validate.errors !== null) { if (errors) validate.errors.length = errors; else validate.errors = null; } delete ' + ($data) + '[' + ($key) + ']; }  ';
	          it.compositeRule = $it.compositeRule = $wasComposite;
	        } else {
	          $it.schema = $aProperties;
	          $it.schemaPath = it.schemaPath + '.additionalProperties';
	          $it.errSchemaPath = it.errSchemaPath + '/additionalProperties';
	          $it.errorPath = it.opts._errorDataPathProperty ? it.errorPath : it.util.getPathExpr(it.errorPath, $key, it.opts.jsonPointers);
	          var $passData = $data + '[' + $key + ']';
	          $it.dataPathArr[$dataNxt] = $key;
	          var $code = it.validate($it);
	          $it.baseId = $currentBaseId;
	          if (it.util.varOccurences($code, $nextData) < 2) {
	            out += ' ' + (it.util.varReplace($code, $nextData, $passData)) + ' ';
	          } else {
	            out += ' var ' + ($nextData) + ' = ' + ($passData) + '; ' + ($code) + ' ';
	          }
	          if ($breakOnError) {
	            out += ' if (!' + ($nextValid) + ') break; ';
	          }
	        }
	      }
	      it.errorPath = $currentErrorPath;
	    }
	    if ($someProperties) {
	      out += ' } ';
	    }
	    out += ' }  ';
	    if ($breakOnError) {
	      out += ' if (' + ($nextValid) + ') { ';
	      $closingBraces += '}';
	    }
	  }
	  var $useDefaults = it.opts.useDefaults && !it.compositeRule;
	  if ($schemaKeys.length) {
	    var arr4 = $schemaKeys;
	    if (arr4) {
	      var $propertyKey, i4 = -1,
	        l4 = arr4.length - 1;
	      while (i4 < l4) {
	        $propertyKey = arr4[i4 += 1];
	        var $sch = $schema[$propertyKey];
	        if (it.util.schemaHasRules($sch, it.RULES.all)) {
	          var $prop = it.util.getProperty($propertyKey),
	            $passData = $data + $prop,
	            $hasDefault = $useDefaults && $sch.default !== undefined;
	          $it.schema = $sch;
	          $it.schemaPath = $schemaPath + $prop;
	          $it.errSchemaPath = $errSchemaPath + '/' + it.util.escapeFragment($propertyKey);
	          $it.errorPath = it.util.getPath(it.errorPath, $propertyKey, it.opts.jsonPointers);
	          $it.dataPathArr[$dataNxt] = it.util.toQuotedString($propertyKey);
	          var $code = it.validate($it);
	          $it.baseId = $currentBaseId;
	          if (it.util.varOccurences($code, $nextData) < 2) {
	            $code = it.util.varReplace($code, $nextData, $passData);
	            var $useData = $passData;
	          } else {
	            var $useData = $nextData;
	            out += ' var ' + ($nextData) + ' = ' + ($passData) + '; ';
	          }
	          if ($hasDefault) {
	            out += ' ' + ($code) + ' ';
	          } else {
	            if ($requiredHash && $requiredHash[$propertyKey]) {
	              out += ' if ( ' + ($useData) + ' === undefined ';
	              if ($ownProperties) {
	                out += ' || ! Object.prototype.hasOwnProperty.call(' + ($data) + ', \'' + (it.util.escapeQuotes($propertyKey)) + '\') ';
	              }
	              out += ') { ' + ($nextValid) + ' = false; ';
	              var $currentErrorPath = it.errorPath,
	                $currErrSchemaPath = $errSchemaPath,
	                $missingProperty = it.util.escapeQuotes($propertyKey);
	              if (it.opts._errorDataPathProperty) {
	                it.errorPath = it.util.getPath($currentErrorPath, $propertyKey, it.opts.jsonPointers);
	              }
	              $errSchemaPath = it.errSchemaPath + '/required';
	              var $$outStack = $$outStack || [];
	              $$outStack.push(out);
	              out = ''; /* istanbul ignore else */
	              if (it.createErrors !== false) {
	                out += ' { keyword: \'' + ('required') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { missingProperty: \'' + ($missingProperty) + '\' } ';
	                if (it.opts.messages !== false) {
	                  out += ' , message: \'';
	                  if (it.opts._errorDataPathProperty) {
	                    out += 'is a required property';
	                  } else {
	                    out += 'should have required property \\\'' + ($missingProperty) + '\\\'';
	                  }
	                  out += '\' ';
	                }
	                if (it.opts.verbose) {
	                  out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	                }
	                out += ' } ';
	              } else {
	                out += ' {} ';
	              }
	              var __err = out;
	              out = $$outStack.pop();
	              if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	                if (it.async) {
	                  out += ' throw new ValidationError([' + (__err) + ']); ';
	                } else {
	                  out += ' validate.errors = [' + (__err) + ']; return false; ';
	                }
	              } else {
	                out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	              }
	              $errSchemaPath = $currErrSchemaPath;
	              it.errorPath = $currentErrorPath;
	              out += ' } else { ';
	            } else {
	              if ($breakOnError) {
	                out += ' if ( ' + ($useData) + ' === undefined ';
	                if ($ownProperties) {
	                  out += ' || ! Object.prototype.hasOwnProperty.call(' + ($data) + ', \'' + (it.util.escapeQuotes($propertyKey)) + '\') ';
	                }
	                out += ') { ' + ($nextValid) + ' = true; } else { ';
	              } else {
	                out += ' if (' + ($useData) + ' !== undefined ';
	                if ($ownProperties) {
	                  out += ' &&   Object.prototype.hasOwnProperty.call(' + ($data) + ', \'' + (it.util.escapeQuotes($propertyKey)) + '\') ';
	                }
	                out += ' ) { ';
	              }
	            }
	            out += ' ' + ($code) + ' } ';
	          }
	        }
	        if ($breakOnError) {
	          out += ' if (' + ($nextValid) + ') { ';
	          $closingBraces += '}';
	        }
	      }
	    }
	  }
	  if ($pPropertyKeys.length) {
	    var arr5 = $pPropertyKeys;
	    if (arr5) {
	      var $pProperty, i5 = -1,
	        l5 = arr5.length - 1;
	      while (i5 < l5) {
	        $pProperty = arr5[i5 += 1];
	        var $sch = $pProperties[$pProperty];
	        if (it.util.schemaHasRules($sch, it.RULES.all)) {
	          $it.schema = $sch;
	          $it.schemaPath = it.schemaPath + '.patternProperties' + it.util.getProperty($pProperty);
	          $it.errSchemaPath = it.errSchemaPath + '/patternProperties/' + it.util.escapeFragment($pProperty);
	          if ($ownProperties) {
	            out += ' ' + ($dataProperties) + ' = ' + ($dataProperties) + ' || Object.keys(' + ($data) + '); for (var ' + ($idx) + '=0; ' + ($idx) + '<' + ($dataProperties) + '.length; ' + ($idx) + '++) { var ' + ($key) + ' = ' + ($dataProperties) + '[' + ($idx) + ']; ';
	          } else {
	            out += ' for (var ' + ($key) + ' in ' + ($data) + ') { ';
	          }
	          out += ' if (' + (it.usePattern($pProperty)) + '.test(' + ($key) + ')) { ';
	          $it.errorPath = it.util.getPathExpr(it.errorPath, $key, it.opts.jsonPointers);
	          var $passData = $data + '[' + $key + ']';
	          $it.dataPathArr[$dataNxt] = $key;
	          var $code = it.validate($it);
	          $it.baseId = $currentBaseId;
	          if (it.util.varOccurences($code, $nextData) < 2) {
	            out += ' ' + (it.util.varReplace($code, $nextData, $passData)) + ' ';
	          } else {
	            out += ' var ' + ($nextData) + ' = ' + ($passData) + '; ' + ($code) + ' ';
	          }
	          if ($breakOnError) {
	            out += ' if (!' + ($nextValid) + ') break; ';
	          }
	          out += ' } ';
	          if ($breakOnError) {
	            out += ' else ' + ($nextValid) + ' = true; ';
	          }
	          out += ' }  ';
	          if ($breakOnError) {
	            out += ' if (' + ($nextValid) + ') { ';
	            $closingBraces += '}';
	          }
	        }
	      }
	    }
	  }
	  if (it.opts.patternGroups && $pgPropertyKeys.length) {
	    var arr6 = $pgPropertyKeys;
	    if (arr6) {
	      var $pgProperty, i6 = -1,
	        l6 = arr6.length - 1;
	      while (i6 < l6) {
	        $pgProperty = arr6[i6 += 1];
	        var $pgSchema = $pgProperties[$pgProperty],
	          $sch = $pgSchema.schema;
	        if (it.util.schemaHasRules($sch, it.RULES.all)) {
	          $it.schema = $sch;
	          $it.schemaPath = it.schemaPath + '.patternGroups' + it.util.getProperty($pgProperty) + '.schema';
	          $it.errSchemaPath = it.errSchemaPath + '/patternGroups/' + it.util.escapeFragment($pgProperty) + '/schema';
	          out += ' var pgPropCount' + ($lvl) + ' = 0;  ';
	          if ($ownProperties) {
	            out += ' ' + ($dataProperties) + ' = ' + ($dataProperties) + ' || Object.keys(' + ($data) + '); for (var ' + ($idx) + '=0; ' + ($idx) + '<' + ($dataProperties) + '.length; ' + ($idx) + '++) { var ' + ($key) + ' = ' + ($dataProperties) + '[' + ($idx) + ']; ';
	          } else {
	            out += ' for (var ' + ($key) + ' in ' + ($data) + ') { ';
	          }
	          out += ' if (' + (it.usePattern($pgProperty)) + '.test(' + ($key) + ')) { pgPropCount' + ($lvl) + '++; ';
	          $it.errorPath = it.util.getPathExpr(it.errorPath, $key, it.opts.jsonPointers);
	          var $passData = $data + '[' + $key + ']';
	          $it.dataPathArr[$dataNxt] = $key;
	          var $code = it.validate($it);
	          $it.baseId = $currentBaseId;
	          if (it.util.varOccurences($code, $nextData) < 2) {
	            out += ' ' + (it.util.varReplace($code, $nextData, $passData)) + ' ';
	          } else {
	            out += ' var ' + ($nextData) + ' = ' + ($passData) + '; ' + ($code) + ' ';
	          }
	          if ($breakOnError) {
	            out += ' if (!' + ($nextValid) + ') break; ';
	          }
	          out += ' } ';
	          if ($breakOnError) {
	            out += ' else ' + ($nextValid) + ' = true; ';
	          }
	          out += ' }  ';
	          if ($breakOnError) {
	            out += ' if (' + ($nextValid) + ') { ';
	            $closingBraces += '}';
	          }
	          var $pgMin = $pgSchema.minimum,
	            $pgMax = $pgSchema.maximum;
	          if ($pgMin !== undefined || $pgMax !== undefined) {
	            out += ' var ' + ($valid) + ' = true; ';
	            var $currErrSchemaPath = $errSchemaPath;
	            if ($pgMin !== undefined) {
	              var $limit = $pgMin,
	                $reason = 'minimum',
	                $moreOrLess = 'less';
	              out += ' ' + ($valid) + ' = pgPropCount' + ($lvl) + ' >= ' + ($pgMin) + '; ';
	              $errSchemaPath = it.errSchemaPath + '/patternGroups/minimum';
	              out += '  if (!' + ($valid) + ') {   ';
	              var $$outStack = $$outStack || [];
	              $$outStack.push(out);
	              out = ''; /* istanbul ignore else */
	              if (it.createErrors !== false) {
	                out += ' { keyword: \'' + ('patternGroups') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { reason: \'' + ($reason) + '\', limit: ' + ($limit) + ', pattern: \'' + (it.util.escapeQuotes($pgProperty)) + '\' } ';
	                if (it.opts.messages !== false) {
	                  out += ' , message: \'should NOT have ' + ($moreOrLess) + ' than ' + ($limit) + ' properties matching pattern "' + (it.util.escapeQuotes($pgProperty)) + '"\' ';
	                }
	                if (it.opts.verbose) {
	                  out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	                }
	                out += ' } ';
	              } else {
	                out += ' {} ';
	              }
	              var __err = out;
	              out = $$outStack.pop();
	              if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	                if (it.async) {
	                  out += ' throw new ValidationError([' + (__err) + ']); ';
	                } else {
	                  out += ' validate.errors = [' + (__err) + ']; return false; ';
	                }
	              } else {
	                out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	              }
	              out += ' } ';
	              if ($pgMax !== undefined) {
	                out += ' else ';
	              }
	            }
	            if ($pgMax !== undefined) {
	              var $limit = $pgMax,
	                $reason = 'maximum',
	                $moreOrLess = 'more';
	              out += ' ' + ($valid) + ' = pgPropCount' + ($lvl) + ' <= ' + ($pgMax) + '; ';
	              $errSchemaPath = it.errSchemaPath + '/patternGroups/maximum';
	              out += '  if (!' + ($valid) + ') {   ';
	              var $$outStack = $$outStack || [];
	              $$outStack.push(out);
	              out = ''; /* istanbul ignore else */
	              if (it.createErrors !== false) {
	                out += ' { keyword: \'' + ('patternGroups') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { reason: \'' + ($reason) + '\', limit: ' + ($limit) + ', pattern: \'' + (it.util.escapeQuotes($pgProperty)) + '\' } ';
	                if (it.opts.messages !== false) {
	                  out += ' , message: \'should NOT have ' + ($moreOrLess) + ' than ' + ($limit) + ' properties matching pattern "' + (it.util.escapeQuotes($pgProperty)) + '"\' ';
	                }
	                if (it.opts.verbose) {
	                  out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	                }
	                out += ' } ';
	              } else {
	                out += ' {} ';
	              }
	              var __err = out;
	              out = $$outStack.pop();
	              if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	                if (it.async) {
	                  out += ' throw new ValidationError([' + (__err) + ']); ';
	                } else {
	                  out += ' validate.errors = [' + (__err) + ']; return false; ';
	                }
	              } else {
	                out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	              }
	              out += ' } ';
	            }
	            $errSchemaPath = $currErrSchemaPath;
	            if ($breakOnError) {
	              out += ' if (' + ($valid) + ') { ';
	              $closingBraces += '}';
	            }
	          }
	        }
	      }
	    }
	  }
	  if ($breakOnError) {
	    out += ' ' + ($closingBraces) + ' if (' + ($errs) + ' == errors) {';
	  }
	  out = it.util.cleanUpCode(out);
	  return out;
	}


/***/ },
/* 41 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_propertyNames(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $errs = 'errs__' + $lvl;
	  var $it = it.util.copy(it);
	  var $closingBraces = '';
	  $it.level++;
	  var $nextValid = 'valid' + $it.level;
	  if (it.util.schemaHasRules($schema, it.RULES.all)) {
	    $it.schema = $schema;
	    $it.schemaPath = $schemaPath;
	    $it.errSchemaPath = $errSchemaPath;
	    var $key = 'key' + $lvl,
	      $idx = 'idx' + $lvl,
	      $i = 'i' + $lvl,
	      $invalidName = '\' + ' + $key + ' + \'',
	      $dataNxt = $it.dataLevel = it.dataLevel + 1,
	      $nextData = 'data' + $dataNxt,
	      $dataProperties = 'dataProperties' + $lvl,
	      $ownProperties = it.opts.ownProperties,
	      $currentBaseId = it.baseId;
	    out += ' var ' + ($errs) + ' = errors; ';
	    if ($ownProperties) {
	      out += ' var ' + ($dataProperties) + ' = undefined; ';
	    }
	    if ($ownProperties) {
	      out += ' ' + ($dataProperties) + ' = ' + ($dataProperties) + ' || Object.keys(' + ($data) + '); for (var ' + ($idx) + '=0; ' + ($idx) + '<' + ($dataProperties) + '.length; ' + ($idx) + '++) { var ' + ($key) + ' = ' + ($dataProperties) + '[' + ($idx) + ']; ';
	    } else {
	      out += ' for (var ' + ($key) + ' in ' + ($data) + ') { ';
	    }
	    out += ' var startErrs' + ($lvl) + ' = errors; ';
	    var $passData = $key;
	    var $wasComposite = it.compositeRule;
	    it.compositeRule = $it.compositeRule = true;
	    var $code = it.validate($it);
	    $it.baseId = $currentBaseId;
	    if (it.util.varOccurences($code, $nextData) < 2) {
	      out += ' ' + (it.util.varReplace($code, $nextData, $passData)) + ' ';
	    } else {
	      out += ' var ' + ($nextData) + ' = ' + ($passData) + '; ' + ($code) + ' ';
	    }
	    it.compositeRule = $it.compositeRule = $wasComposite;
	    out += ' if (!' + ($nextValid) + ') { for (var ' + ($i) + '=startErrs' + ($lvl) + '; ' + ($i) + '<errors; ' + ($i) + '++) { vErrors[' + ($i) + '].propertyName = ' + ($key) + '; }   var err =   '; /* istanbul ignore else */
	    if (it.createErrors !== false) {
	      out += ' { keyword: \'' + ('propertyNames') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { propertyName: \'' + ($invalidName) + '\' } ';
	      if (it.opts.messages !== false) {
	        out += ' , message: \'property name \\\'' + ($invalidName) + '\\\' is invalid\' ';
	      }
	      if (it.opts.verbose) {
	        out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	      }
	      out += ' } ';
	    } else {
	      out += ' {} ';
	    }
	    out += ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	    if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	      if (it.async) {
	        out += ' throw new ValidationError(vErrors); ';
	      } else {
	        out += ' validate.errors = vErrors; return false; ';
	      }
	    }
	    if ($breakOnError) {
	      out += ' break; ';
	    }
	    out += ' } }';
	  }
	  if ($breakOnError) {
	    out += ' ' + ($closingBraces) + ' if (' + ($errs) + ' == errors) {';
	  }
	  out = it.util.cleanUpCode(out);
	  return out;
	}


/***/ },
/* 42 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_required(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $valid = 'valid' + $lvl;
	  var $isData = it.opts.$data && $schema && $schema.$data,
	    $schemaValue;
	  if ($isData) {
	    out += ' var schema' + ($lvl) + ' = ' + (it.util.getData($schema.$data, $dataLvl, it.dataPathArr)) + '; ';
	    $schemaValue = 'schema' + $lvl;
	  } else {
	    $schemaValue = $schema;
	  }
	  var $vSchema = 'schema' + $lvl;
	  if (!$isData) {
	    if ($schema.length < it.opts.loopRequired && it.schema.properties && Object.keys(it.schema.properties).length) {
	      var $required = [];
	      var arr1 = $schema;
	      if (arr1) {
	        var $property, i1 = -1,
	          l1 = arr1.length - 1;
	        while (i1 < l1) {
	          $property = arr1[i1 += 1];
	          var $propertySch = it.schema.properties[$property];
	          if (!($propertySch && it.util.schemaHasRules($propertySch, it.RULES.all))) {
	            $required[$required.length] = $property;
	          }
	        }
	      }
	    } else {
	      var $required = $schema;
	    }
	  }
	  if ($isData || $required.length) {
	    var $currentErrorPath = it.errorPath,
	      $loopRequired = $isData || $required.length >= it.opts.loopRequired,
	      $ownProperties = it.opts.ownProperties;
	    if ($breakOnError) {
	      out += ' var missing' + ($lvl) + '; ';
	      if ($loopRequired) {
	        if (!$isData) {
	          out += ' var ' + ($vSchema) + ' = validate.schema' + ($schemaPath) + '; ';
	        }
	        var $i = 'i' + $lvl,
	          $propertyPath = 'schema' + $lvl + '[' + $i + ']',
	          $missingProperty = '\' + ' + $propertyPath + ' + \'';
	        if (it.opts._errorDataPathProperty) {
	          it.errorPath = it.util.getPathExpr($currentErrorPath, $propertyPath, it.opts.jsonPointers);
	        }
	        out += ' var ' + ($valid) + ' = true; ';
	        if ($isData) {
	          out += ' if (schema' + ($lvl) + ' === undefined) ' + ($valid) + ' = true; else if (!Array.isArray(schema' + ($lvl) + ')) ' + ($valid) + ' = false; else {';
	        }
	        out += ' for (var ' + ($i) + ' = 0; ' + ($i) + ' < ' + ($vSchema) + '.length; ' + ($i) + '++) { ' + ($valid) + ' = ' + ($data) + '[' + ($vSchema) + '[' + ($i) + ']] !== undefined ';
	        if ($ownProperties) {
	          out += ' &&   Object.prototype.hasOwnProperty.call(' + ($data) + ', ' + ($vSchema) + '[' + ($i) + ']) ';
	        }
	        out += '; if (!' + ($valid) + ') break; } ';
	        if ($isData) {
	          out += '  }  ';
	        }
	        out += '  if (!' + ($valid) + ') {   ';
	        var $$outStack = $$outStack || [];
	        $$outStack.push(out);
	        out = ''; /* istanbul ignore else */
	        if (it.createErrors !== false) {
	          out += ' { keyword: \'' + ('required') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { missingProperty: \'' + ($missingProperty) + '\' } ';
	          if (it.opts.messages !== false) {
	            out += ' , message: \'';
	            if (it.opts._errorDataPathProperty) {
	              out += 'is a required property';
	            } else {
	              out += 'should have required property \\\'' + ($missingProperty) + '\\\'';
	            }
	            out += '\' ';
	          }
	          if (it.opts.verbose) {
	            out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	          }
	          out += ' } ';
	        } else {
	          out += ' {} ';
	        }
	        var __err = out;
	        out = $$outStack.pop();
	        if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	          if (it.async) {
	            out += ' throw new ValidationError([' + (__err) + ']); ';
	          } else {
	            out += ' validate.errors = [' + (__err) + ']; return false; ';
	          }
	        } else {
	          out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	        }
	        out += ' } else { ';
	      } else {
	        out += ' if ( ';
	        var arr2 = $required;
	        if (arr2) {
	          var $propertyKey, $i = -1,
	            l2 = arr2.length - 1;
	          while ($i < l2) {
	            $propertyKey = arr2[$i += 1];
	            if ($i) {
	              out += ' || ';
	            }
	            var $prop = it.util.getProperty($propertyKey),
	              $useData = $data + $prop;
	            out += ' ( ( ' + ($useData) + ' === undefined ';
	            if ($ownProperties) {
	              out += ' || ! Object.prototype.hasOwnProperty.call(' + ($data) + ', \'' + (it.util.escapeQuotes($propertyKey)) + '\') ';
	            }
	            out += ') && (missing' + ($lvl) + ' = ' + (it.util.toQuotedString(it.opts.jsonPointers ? $propertyKey : $prop)) + ') ) ';
	          }
	        }
	        out += ') {  ';
	        var $propertyPath = 'missing' + $lvl,
	          $missingProperty = '\' + ' + $propertyPath + ' + \'';
	        if (it.opts._errorDataPathProperty) {
	          it.errorPath = it.opts.jsonPointers ? it.util.getPathExpr($currentErrorPath, $propertyPath, true) : $currentErrorPath + ' + ' + $propertyPath;
	        }
	        var $$outStack = $$outStack || [];
	        $$outStack.push(out);
	        out = ''; /* istanbul ignore else */
	        if (it.createErrors !== false) {
	          out += ' { keyword: \'' + ('required') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { missingProperty: \'' + ($missingProperty) + '\' } ';
	          if (it.opts.messages !== false) {
	            out += ' , message: \'';
	            if (it.opts._errorDataPathProperty) {
	              out += 'is a required property';
	            } else {
	              out += 'should have required property \\\'' + ($missingProperty) + '\\\'';
	            }
	            out += '\' ';
	          }
	          if (it.opts.verbose) {
	            out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	          }
	          out += ' } ';
	        } else {
	          out += ' {} ';
	        }
	        var __err = out;
	        out = $$outStack.pop();
	        if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	          if (it.async) {
	            out += ' throw new ValidationError([' + (__err) + ']); ';
	          } else {
	            out += ' validate.errors = [' + (__err) + ']; return false; ';
	          }
	        } else {
	          out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	        }
	        out += ' } else { ';
	      }
	    } else {
	      if ($loopRequired) {
	        if (!$isData) {
	          out += ' var ' + ($vSchema) + ' = validate.schema' + ($schemaPath) + '; ';
	        }
	        var $i = 'i' + $lvl,
	          $propertyPath = 'schema' + $lvl + '[' + $i + ']',
	          $missingProperty = '\' + ' + $propertyPath + ' + \'';
	        if (it.opts._errorDataPathProperty) {
	          it.errorPath = it.util.getPathExpr($currentErrorPath, $propertyPath, it.opts.jsonPointers);
	        }
	        if ($isData) {
	          out += ' if (' + ($vSchema) + ' && !Array.isArray(' + ($vSchema) + ')) {  var err =   '; /* istanbul ignore else */
	          if (it.createErrors !== false) {
	            out += ' { keyword: \'' + ('required') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { missingProperty: \'' + ($missingProperty) + '\' } ';
	            if (it.opts.messages !== false) {
	              out += ' , message: \'';
	              if (it.opts._errorDataPathProperty) {
	                out += 'is a required property';
	              } else {
	                out += 'should have required property \\\'' + ($missingProperty) + '\\\'';
	              }
	              out += '\' ';
	            }
	            if (it.opts.verbose) {
	              out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	            }
	            out += ' } ';
	          } else {
	            out += ' {} ';
	          }
	          out += ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; } else if (' + ($vSchema) + ' !== undefined) { ';
	        }
	        out += ' for (var ' + ($i) + ' = 0; ' + ($i) + ' < ' + ($vSchema) + '.length; ' + ($i) + '++) { if (' + ($data) + '[' + ($vSchema) + '[' + ($i) + ']] === undefined ';
	        if ($ownProperties) {
	          out += ' || ! Object.prototype.hasOwnProperty.call(' + ($data) + ', ' + ($vSchema) + '[' + ($i) + ']) ';
	        }
	        out += ') {  var err =   '; /* istanbul ignore else */
	        if (it.createErrors !== false) {
	          out += ' { keyword: \'' + ('required') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { missingProperty: \'' + ($missingProperty) + '\' } ';
	          if (it.opts.messages !== false) {
	            out += ' , message: \'';
	            if (it.opts._errorDataPathProperty) {
	              out += 'is a required property';
	            } else {
	              out += 'should have required property \\\'' + ($missingProperty) + '\\\'';
	            }
	            out += '\' ';
	          }
	          if (it.opts.verbose) {
	            out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	          }
	          out += ' } ';
	        } else {
	          out += ' {} ';
	        }
	        out += ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; } } ';
	        if ($isData) {
	          out += '  }  ';
	        }
	      } else {
	        var arr3 = $required;
	        if (arr3) {
	          var $propertyKey, i3 = -1,
	            l3 = arr3.length - 1;
	          while (i3 < l3) {
	            $propertyKey = arr3[i3 += 1];
	            var $prop = it.util.getProperty($propertyKey),
	              $missingProperty = it.util.escapeQuotes($propertyKey),
	              $useData = $data + $prop;
	            if (it.opts._errorDataPathProperty) {
	              it.errorPath = it.util.getPath($currentErrorPath, $propertyKey, it.opts.jsonPointers);
	            }
	            out += ' if ( ' + ($useData) + ' === undefined ';
	            if ($ownProperties) {
	              out += ' || ! Object.prototype.hasOwnProperty.call(' + ($data) + ', \'' + (it.util.escapeQuotes($propertyKey)) + '\') ';
	            }
	            out += ') {  var err =   '; /* istanbul ignore else */
	            if (it.createErrors !== false) {
	              out += ' { keyword: \'' + ('required') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { missingProperty: \'' + ($missingProperty) + '\' } ';
	              if (it.opts.messages !== false) {
	                out += ' , message: \'';
	                if (it.opts._errorDataPathProperty) {
	                  out += 'is a required property';
	                } else {
	                  out += 'should have required property \\\'' + ($missingProperty) + '\\\'';
	                }
	                out += '\' ';
	              }
	              if (it.opts.verbose) {
	                out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	              }
	              out += ' } ';
	            } else {
	              out += ' {} ';
	            }
	            out += ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; } ';
	          }
	        }
	      }
	    }
	    it.errorPath = $currentErrorPath;
	  } else if ($breakOnError) {
	    out += ' if (true) {';
	  }
	  return out;
	}


/***/ },
/* 43 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_uniqueItems(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $data = 'data' + ($dataLvl || '');
	  var $valid = 'valid' + $lvl;
	  var $isData = it.opts.$data && $schema && $schema.$data,
	    $schemaValue;
	  if ($isData) {
	    out += ' var schema' + ($lvl) + ' = ' + (it.util.getData($schema.$data, $dataLvl, it.dataPathArr)) + '; ';
	    $schemaValue = 'schema' + $lvl;
	  } else {
	    $schemaValue = $schema;
	  }
	  if (($schema || $isData) && it.opts.uniqueItems !== false) {
	    if ($isData) {
	      out += ' var ' + ($valid) + '; if (' + ($schemaValue) + ' === false || ' + ($schemaValue) + ' === undefined) ' + ($valid) + ' = true; else if (typeof ' + ($schemaValue) + ' != \'boolean\') ' + ($valid) + ' = false; else { ';
	    }
	    out += ' var ' + ($valid) + ' = true; if (' + ($data) + '.length > 1) { var i = ' + ($data) + '.length, j; outer: for (;i--;) { for (j = i; j--;) { if (equal(' + ($data) + '[i], ' + ($data) + '[j])) { ' + ($valid) + ' = false; break outer; } } } } ';
	    if ($isData) {
	      out += '  }  ';
	    }
	    out += ' if (!' + ($valid) + ') {   ';
	    var $$outStack = $$outStack || [];
	    $$outStack.push(out);
	    out = ''; /* istanbul ignore else */
	    if (it.createErrors !== false) {
	      out += ' { keyword: \'' + ('uniqueItems') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { i: i, j: j } ';
	      if (it.opts.messages !== false) {
	        out += ' , message: \'should NOT have duplicate items (items ## \' + j + \' and \' + i + \' are identical)\' ';
	      }
	      if (it.opts.verbose) {
	        out += ' , schema:  ';
	        if ($isData) {
	          out += 'validate.schema' + ($schemaPath);
	        } else {
	          out += '' + ($schema);
	        }
	        out += '         , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	      }
	      out += ' } ';
	    } else {
	      out += ' {} ';
	    }
	    var __err = out;
	    out = $$outStack.pop();
	    if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	      if (it.async) {
	        out += ' throw new ValidationError([' + (__err) + ']); ';
	      } else {
	        out += ' validate.errors = [' + (__err) + ']; return false; ';
	      }
	    } else {
	      out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	    }
	    out += ' } ';
	    if ($breakOnError) {
	      out += ' else { ';
	    }
	  } else {
	    if ($breakOnError) {
	      out += ' if (true) { ';
	    }
	  }
	  return out;
	}


/***/ },
/* 44 */
/***/ function(module, exports) {

	'use strict';

	var KEYWORDS = [
	  'multipleOf',
	  'maximum',
	  'exclusiveMaximum',
	  'minimum',
	  'exclusiveMinimum',
	  'maxLength',
	  'minLength',
	  'pattern',
	  'additionalItems',
	  'maxItems',
	  'minItems',
	  'uniqueItems',
	  'maxProperties',
	  'minProperties',
	  'required',
	  'additionalProperties',
	  'enum',
	  'format',
	  'const'
	];

	module.exports = function (metaSchema, keywordsJsonPointers) {
	  for (var i=0; i<keywordsJsonPointers.length; i++) {
	    metaSchema = JSON.parse(JSON.stringify(metaSchema));
	    var segments = keywordsJsonPointers[i].split('/');
	    var keywords = metaSchema;
	    var j;
	    for (j=1; j<segments.length; j++)
	      keywords = keywords[segments[j]];

	    for (j=0; j<KEYWORDS.length; j++) {
	      var key = KEYWORDS[j];
	      var schema = keywords[key];
	      if (schema) {
	        keywords[key] = {
	          anyOf: [
	            schema,
	            { $ref: 'https://raw.githubusercontent.com/epoberezkin/ajv/master/lib/refs/$data.json#' }
	          ]
	        };
	      }
	    }
	  }

	  return metaSchema;
	};


/***/ },
/* 45 */
/***/ function(module, exports) {

	'use strict';

	var META_SCHEMA_ID = 'http://json-schema.org/draft-06/schema';

	module.exports = function (ajv) {
	  var defaultMeta = ajv._opts.defaultMeta;
	  var metaSchemaRef = typeof defaultMeta == 'string'
	                      ? { $ref: defaultMeta }
	                      : ajv.getSchema(META_SCHEMA_ID)
	                        ? { $ref: META_SCHEMA_ID }
	                        : {};

	  ajv.addKeyword('patternGroups', {
	    // implemented in properties.jst
	    metaSchema: {
	      type: 'object',
	      additionalProperties: {
	        type: 'object',
	        required: [ 'schema' ],
	        properties: {
	          maximum: {
	            type: 'integer',
	            minimum: 0
	          },
	          minimum: {
	            type: 'integer',
	            minimum: 0
	          },
	          schema: metaSchemaRef
	        },
	        additionalProperties: false
	      }
	    }
	  });
	  ajv.RULES.all.properties.implements.push('patternGroups');
	};


/***/ },
/* 46 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var MissingRefError = __webpack_require__(15).MissingRef;

	module.exports = compileAsync;


	/**
	 * Creates validating function for passed schema with asynchronous loading of missing schemas.
	 * `loadSchema` option should be a function that accepts schema uri and returns promise that resolves with the schema.
	 * @this  Ajv
	 * @param {Object}   schema schema object
	 * @param {Boolean}  meta optional true to compile meta-schema; this parameter can be skipped
	 * @param {Function} callback an optional node-style callback, it is called with 2 parameters: error (or null) and validating function.
	 * @return {Promise} promise that resolves with a validating function.
	 */
	function compileAsync(schema, meta, callback) {
	  /* eslint no-shadow: 0 */
	  /* global Promise */
	  /* jshint validthis: true */
	  var self = this;
	  if (typeof this._opts.loadSchema != 'function')
	    throw new Error('options.loadSchema should be a function');

	  if (typeof meta == 'function') {
	    callback = meta;
	    meta = undefined;
	  }

	  var p = loadMetaSchemaOf(schema).then(function () {
	    var schemaObj = self._addSchema(schema, undefined, meta);
	    return schemaObj.validate || _compileAsync(schemaObj);
	  });

	  if (callback) {
	    p.then(
	      function(v) { callback(null, v); },
	      callback
	    );
	  }

	  return p;


	  function loadMetaSchemaOf(sch) {
	    var $schema = sch.$schema;
	    return $schema && !self.getSchema($schema)
	            ? compileAsync.call(self, { $ref: $schema }, true)
	            : Promise.resolve();
	  }


	  function _compileAsync(schemaObj) {
	    try { return self._compile(schemaObj); }
	    catch(e) {
	      if (e instanceof MissingRefError) return loadMissingSchema(e);
	      throw e;
	    }


	    function loadMissingSchema(e) {
	      var ref = e.missingSchema;
	      if (added(ref)) throw new Error('Schema ' + ref + ' is loaded but ' + e.missingRef + ' cannot be resolved');

	      var schemaPromise = self._loadingSchemas[ref];
	      if (!schemaPromise) {
	        schemaPromise = self._loadingSchemas[ref] = self._opts.loadSchema(ref);
	        schemaPromise.then(removePromise, removePromise);
	      }

	      return schemaPromise.then(function (sch) {
	        if (!added(ref)) {
	          return loadMetaSchemaOf(sch).then(function () {
	            if (!added(ref)) self.addSchema(sch, ref, undefined, meta);
	          });
	        }
	      }).then(function() {
	        return _compileAsync(schemaObj);
	      });

	      function removePromise() {
	        delete self._loadingSchemas[ref];
	      }

	      function added(ref) {
	        return self._refs[ref] || self._schemas[ref];
	      }
	    }
	  }
	}


/***/ },
/* 47 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var IDENTIFIER = /^[a-z_$][a-z0-9_$-]*$/i;
	var customRuleCode = __webpack_require__(48);

	module.exports = {
	  add: addKeyword,
	  get: getKeyword,
	  remove: removeKeyword
	};

	/**
	 * Define custom keyword
	 * @this  Ajv
	 * @param {String} keyword custom keyword, should be unique (including different from all standard, custom and macro keywords).
	 * @param {Object} definition keyword definition object with properties `type` (type(s) which the keyword applies to), `validate` or `compile`.
	 * @return {Ajv} this for method chaining
	 */
	function addKeyword(keyword, definition) {
	  /* jshint validthis: true */
	  /* eslint no-shadow: 0 */
	  var RULES = this.RULES;

	  if (RULES.keywords[keyword])
	    throw new Error('Keyword ' + keyword + ' is already defined');

	  if (!IDENTIFIER.test(keyword))
	    throw new Error('Keyword ' + keyword + ' is not a valid identifier');

	  if (definition) {
	    if (definition.macro && definition.valid !== undefined)
	      throw new Error('"valid" option cannot be used with macro keywords');

	    var dataType = definition.type;
	    if (Array.isArray(dataType)) {
	      var i, len = dataType.length;
	      for (i=0; i<len; i++) checkDataType(dataType[i]);
	      for (i=0; i<len; i++) _addRule(keyword, dataType[i], definition);
	    } else {
	      if (dataType) checkDataType(dataType);
	      _addRule(keyword, dataType, definition);
	    }

	    var $data = definition.$data === true && this._opts.$data;
	    if ($data && !definition.validate)
	      throw new Error('$data support: "validate" function is not defined');

	    var metaSchema = definition.metaSchema;
	    if (metaSchema) {
	      if ($data) {
	        metaSchema = {
	          anyOf: [
	            metaSchema,
	            { '$ref': 'https://raw.githubusercontent.com/epoberezkin/ajv/master/lib/refs/$data.json#' }
	          ]
	        };
	      }
	      definition.validateSchema = this.compile(metaSchema, true);
	    }
	  }

	  RULES.keywords[keyword] = RULES.all[keyword] = true;


	  function _addRule(keyword, dataType, definition) {
	    var ruleGroup;
	    for (var i=0; i<RULES.length; i++) {
	      var rg = RULES[i];
	      if (rg.type == dataType) {
	        ruleGroup = rg;
	        break;
	      }
	    }

	    if (!ruleGroup) {
	      ruleGroup = { type: dataType, rules: [] };
	      RULES.push(ruleGroup);
	    }

	    var rule = {
	      keyword: keyword,
	      definition: definition,
	      custom: true,
	      code: customRuleCode,
	      implements: definition.implements
	    };
	    ruleGroup.rules.push(rule);
	    RULES.custom[keyword] = rule;
	  }


	  function checkDataType(dataType) {
	    if (!RULES.types[dataType]) throw new Error('Unknown type ' + dataType);
	  }

	  return this;
	}


	/**
	 * Get keyword
	 * @this  Ajv
	 * @param {String} keyword pre-defined or custom keyword.
	 * @return {Object|Boolean} custom keyword definition, `true` if it is a predefined keyword, `false` otherwise.
	 */
	function getKeyword(keyword) {
	  /* jshint validthis: true */
	  var rule = this.RULES.custom[keyword];
	  return rule ? rule.definition : this.RULES.keywords[keyword] || false;
	}


	/**
	 * Remove keyword
	 * @this  Ajv
	 * @param {String} keyword pre-defined or custom keyword.
	 * @return {Ajv} this for method chaining
	 */
	function removeKeyword(keyword) {
	  /* jshint validthis: true */
	  var RULES = this.RULES;
	  delete RULES.keywords[keyword];
	  delete RULES.all[keyword];
	  delete RULES.custom[keyword];
	  for (var i=0; i<RULES.length; i++) {
	    var rules = RULES[i].rules;
	    for (var j=0; j<rules.length; j++) {
	      if (rules[j].keyword == keyword) {
	        rules.splice(j, 1);
	        break;
	      }
	    }
	  }
	  return this;
	}


/***/ },
/* 48 */
/***/ function(module, exports) {

	'use strict';
	module.exports = function generate_custom(it, $keyword, $ruleType) {
	  var out = ' ';
	  var $lvl = it.level;
	  var $dataLvl = it.dataLevel;
	  var $schema = it.schema[$keyword];
	  var $schemaPath = it.schemaPath + it.util.getProperty($keyword);
	  var $errSchemaPath = it.errSchemaPath + '/' + $keyword;
	  var $breakOnError = !it.opts.allErrors;
	  var $errorKeyword;
	  var $data = 'data' + ($dataLvl || '');
	  var $valid = 'valid' + $lvl;
	  var $errs = 'errs__' + $lvl;
	  var $isData = it.opts.$data && $schema && $schema.$data,
	    $schemaValue;
	  if ($isData) {
	    out += ' var schema' + ($lvl) + ' = ' + (it.util.getData($schema.$data, $dataLvl, it.dataPathArr)) + '; ';
	    $schemaValue = 'schema' + $lvl;
	  } else {
	    $schemaValue = $schema;
	  }
	  var $rule = this,
	    $definition = 'definition' + $lvl,
	    $rDef = $rule.definition,
	    $closingBraces = '';
	  var $compile, $inline, $macro, $ruleValidate, $validateCode;
	  if ($isData && $rDef.$data) {
	    $validateCode = 'keywordValidate' + $lvl;
	    var $validateSchema = $rDef.validateSchema;
	    out += ' var ' + ($definition) + ' = RULES.custom[\'' + ($keyword) + '\'].definition; var ' + ($validateCode) + ' = ' + ($definition) + '.validate;';
	  } else {
	    $ruleValidate = it.useCustomRule($rule, $schema, it.schema, it);
	    if (!$ruleValidate) return;
	    $schemaValue = 'validate.schema' + $schemaPath;
	    $validateCode = $ruleValidate.code;
	    $compile = $rDef.compile;
	    $inline = $rDef.inline;
	    $macro = $rDef.macro;
	  }
	  var $ruleErrs = $validateCode + '.errors',
	    $i = 'i' + $lvl,
	    $ruleErr = 'ruleErr' + $lvl,
	    $asyncKeyword = $rDef.async;
	  if ($asyncKeyword && !it.async) throw new Error('async keyword in sync schema');
	  if (!($inline || $macro)) {
	    out += '' + ($ruleErrs) + ' = null;';
	  }
	  out += 'var ' + ($errs) + ' = errors;var ' + ($valid) + ';';
	  if ($isData && $rDef.$data) {
	    $closingBraces += '}';
	    out += ' if (' + ($schemaValue) + ' === undefined) { ' + ($valid) + ' = true; } else { ';
	    if ($validateSchema) {
	      $closingBraces += '}';
	      out += ' ' + ($valid) + ' = ' + ($definition) + '.validateSchema(' + ($schemaValue) + '); if (' + ($valid) + ') { ';
	    }
	  }
	  if ($inline) {
	    if ($rDef.statements) {
	      out += ' ' + ($ruleValidate.validate) + ' ';
	    } else {
	      out += ' ' + ($valid) + ' = ' + ($ruleValidate.validate) + '; ';
	    }
	  } else if ($macro) {
	    var $it = it.util.copy(it);
	    var $closingBraces = '';
	    $it.level++;
	    var $nextValid = 'valid' + $it.level;
	    $it.schema = $ruleValidate.validate;
	    $it.schemaPath = '';
	    var $wasComposite = it.compositeRule;
	    it.compositeRule = $it.compositeRule = true;
	    var $code = it.validate($it).replace(/validate\.schema/g, $validateCode);
	    it.compositeRule = $it.compositeRule = $wasComposite;
	    out += ' ' + ($code);
	  } else {
	    var $$outStack = $$outStack || [];
	    $$outStack.push(out);
	    out = '';
	    out += '  ' + ($validateCode) + '.call( ';
	    if (it.opts.passContext) {
	      out += 'this';
	    } else {
	      out += 'self';
	    }
	    if ($compile || $rDef.schema === false) {
	      out += ' , ' + ($data) + ' ';
	    } else {
	      out += ' , ' + ($schemaValue) + ' , ' + ($data) + ' , validate.schema' + (it.schemaPath) + ' ';
	    }
	    out += ' , (dataPath || \'\')';
	    if (it.errorPath != '""') {
	      out += ' + ' + (it.errorPath);
	    }
	    var $parentData = $dataLvl ? 'data' + (($dataLvl - 1) || '') : 'parentData',
	      $parentDataProperty = $dataLvl ? it.dataPathArr[$dataLvl] : 'parentDataProperty';
	    out += ' , ' + ($parentData) + ' , ' + ($parentDataProperty) + ' , rootData )  ';
	    var def_callRuleValidate = out;
	    out = $$outStack.pop();
	    if ($rDef.errors === false) {
	      out += ' ' + ($valid) + ' = ';
	      if ($asyncKeyword) {
	        out += '' + (it.yieldAwait);
	      }
	      out += '' + (def_callRuleValidate) + '; ';
	    } else {
	      if ($asyncKeyword) {
	        $ruleErrs = 'customErrors' + $lvl;
	        out += ' var ' + ($ruleErrs) + ' = null; try { ' + ($valid) + ' = ' + (it.yieldAwait) + (def_callRuleValidate) + '; } catch (e) { ' + ($valid) + ' = false; if (e instanceof ValidationError) ' + ($ruleErrs) + ' = e.errors; else throw e; } ';
	      } else {
	        out += ' ' + ($ruleErrs) + ' = null; ' + ($valid) + ' = ' + (def_callRuleValidate) + '; ';
	      }
	    }
	  }
	  if ($rDef.modifying) {
	    out += ' if (' + ($parentData) + ') ' + ($data) + ' = ' + ($parentData) + '[' + ($parentDataProperty) + '];';
	  }
	  out += '' + ($closingBraces);
	  if ($rDef.valid) {
	    if ($breakOnError) {
	      out += ' if (true) { ';
	    }
	  } else {
	    out += ' if ( ';
	    if ($rDef.valid === undefined) {
	      out += ' !';
	      if ($macro) {
	        out += '' + ($nextValid);
	      } else {
	        out += '' + ($valid);
	      }
	    } else {
	      out += ' ' + (!$rDef.valid) + ' ';
	    }
	    out += ') { ';
	    $errorKeyword = $rule.keyword;
	    var $$outStack = $$outStack || [];
	    $$outStack.push(out);
	    out = '';
	    var $$outStack = $$outStack || [];
	    $$outStack.push(out);
	    out = ''; /* istanbul ignore else */
	    if (it.createErrors !== false) {
	      out += ' { keyword: \'' + ($errorKeyword || 'custom') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { keyword: \'' + ($rule.keyword) + '\' } ';
	      if (it.opts.messages !== false) {
	        out += ' , message: \'should pass "' + ($rule.keyword) + '" keyword validation\' ';
	      }
	      if (it.opts.verbose) {
	        out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	      }
	      out += ' } ';
	    } else {
	      out += ' {} ';
	    }
	    var __err = out;
	    out = $$outStack.pop();
	    if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	      if (it.async) {
	        out += ' throw new ValidationError([' + (__err) + ']); ';
	      } else {
	        out += ' validate.errors = [' + (__err) + ']; return false; ';
	      }
	    } else {
	      out += ' var err = ' + (__err) + ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	    }
	    var def_customError = out;
	    out = $$outStack.pop();
	    if ($inline) {
	      if ($rDef.errors) {
	        if ($rDef.errors != 'full') {
	          out += '  for (var ' + ($i) + '=' + ($errs) + '; ' + ($i) + '<errors; ' + ($i) + '++) { var ' + ($ruleErr) + ' = vErrors[' + ($i) + ']; if (' + ($ruleErr) + '.dataPath === undefined) ' + ($ruleErr) + '.dataPath = (dataPath || \'\') + ' + (it.errorPath) + '; if (' + ($ruleErr) + '.schemaPath === undefined) { ' + ($ruleErr) + '.schemaPath = "' + ($errSchemaPath) + '"; } ';
	          if (it.opts.verbose) {
	            out += ' ' + ($ruleErr) + '.schema = ' + ($schemaValue) + '; ' + ($ruleErr) + '.data = ' + ($data) + '; ';
	          }
	          out += ' } ';
	        }
	      } else {
	        if ($rDef.errors === false) {
	          out += ' ' + (def_customError) + ' ';
	        } else {
	          out += ' if (' + ($errs) + ' == errors) { ' + (def_customError) + ' } else {  for (var ' + ($i) + '=' + ($errs) + '; ' + ($i) + '<errors; ' + ($i) + '++) { var ' + ($ruleErr) + ' = vErrors[' + ($i) + ']; if (' + ($ruleErr) + '.dataPath === undefined) ' + ($ruleErr) + '.dataPath = (dataPath || \'\') + ' + (it.errorPath) + '; if (' + ($ruleErr) + '.schemaPath === undefined) { ' + ($ruleErr) + '.schemaPath = "' + ($errSchemaPath) + '"; } ';
	          if (it.opts.verbose) {
	            out += ' ' + ($ruleErr) + '.schema = ' + ($schemaValue) + '; ' + ($ruleErr) + '.data = ' + ($data) + '; ';
	          }
	          out += ' } } ';
	        }
	      }
	    } else if ($macro) {
	      out += '   var err =   '; /* istanbul ignore else */
	      if (it.createErrors !== false) {
	        out += ' { keyword: \'' + ($errorKeyword || 'custom') + '\' , dataPath: (dataPath || \'\') + ' + (it.errorPath) + ' , schemaPath: ' + (it.util.toQuotedString($errSchemaPath)) + ' , params: { keyword: \'' + ($rule.keyword) + '\' } ';
	        if (it.opts.messages !== false) {
	          out += ' , message: \'should pass "' + ($rule.keyword) + '" keyword validation\' ';
	        }
	        if (it.opts.verbose) {
	          out += ' , schema: validate.schema' + ($schemaPath) + ' , parentSchema: validate.schema' + (it.schemaPath) + ' , data: ' + ($data) + ' ';
	        }
	        out += ' } ';
	      } else {
	        out += ' {} ';
	      }
	      out += ';  if (vErrors === null) vErrors = [err]; else vErrors.push(err); errors++; ';
	      if (!it.compositeRule && $breakOnError) { /* istanbul ignore if */
	        if (it.async) {
	          out += ' throw new ValidationError(vErrors); ';
	        } else {
	          out += ' validate.errors = vErrors; return false; ';
	        }
	      }
	    } else {
	      if ($rDef.errors === false) {
	        out += ' ' + (def_customError) + ' ';
	      } else {
	        out += ' if (Array.isArray(' + ($ruleErrs) + ')) { if (vErrors === null) vErrors = ' + ($ruleErrs) + '; else vErrors = vErrors.concat(' + ($ruleErrs) + '); errors = vErrors.length;  for (var ' + ($i) + '=' + ($errs) + '; ' + ($i) + '<errors; ' + ($i) + '++) { var ' + ($ruleErr) + ' = vErrors[' + ($i) + ']; if (' + ($ruleErr) + '.dataPath === undefined) ' + ($ruleErr) + '.dataPath = (dataPath || \'\') + ' + (it.errorPath) + ';  ' + ($ruleErr) + '.schemaPath = "' + ($errSchemaPath) + '";  ';
	        if (it.opts.verbose) {
	          out += ' ' + ($ruleErr) + '.schema = ' + ($schemaValue) + '; ' + ($ruleErr) + '.data = ' + ($data) + '; ';
	        }
	        out += ' } } else { ' + (def_customError) + ' } ';
	      }
	    }
	    out += ' } ';
	    if ($breakOnError) {
	      out += ' else { ';
	    }
	  }
	  return out;
	}


/***/ },
/* 49 */
/***/ function(module, exports) {

	module.exports = {
		"$schema": "http://json-schema.org/draft-06/schema#",
		"$id": "https://raw.githubusercontent.com/epoberezkin/ajv/master/lib/refs/$data.json#",
		"description": "Meta-schema for $data reference (JSON-schema extension proposal)",
		"type": "object",
		"required": [
			"$data"
		],
		"properties": {
			"$data": {
				"type": "string",
				"anyOf": [
					{
						"format": "relative-json-pointer"
					},
					{
						"format": "json-pointer"
					}
				]
			}
		},
		"additionalProperties": false
	};

/***/ },
/* 50 */
/***/ function(module, exports) {

	module.exports = {
		"$schema": "http://json-schema.org/draft-06/schema#",
		"$id": "http://json-schema.org/draft-06/schema#",
		"title": "Core schema meta-schema",
		"definitions": {
			"schemaArray": {
				"type": "array",
				"minItems": 1,
				"items": {
					"$ref": "#"
				}
			},
			"nonNegativeInteger": {
				"type": "integer",
				"minimum": 0
			},
			"nonNegativeIntegerDefault0": {
				"allOf": [
					{
						"$ref": "#/definitions/nonNegativeInteger"
					},
					{
						"default": 0
					}
				]
			},
			"simpleTypes": {
				"enum": [
					"array",
					"boolean",
					"integer",
					"null",
					"number",
					"object",
					"string"
				]
			},
			"stringArray": {
				"type": "array",
				"items": {
					"type": "string"
				},
				"uniqueItems": true,
				"default": []
			}
		},
		"type": [
			"object",
			"boolean"
		],
		"properties": {
			"$id": {
				"type": "string",
				"format": "uri-reference"
			},
			"$schema": {
				"type": "string",
				"format": "uri"
			},
			"$ref": {
				"type": "string",
				"format": "uri-reference"
			},
			"title": {
				"type": "string"
			},
			"description": {
				"type": "string"
			},
			"default": {},
			"examples": {
				"type": "array",
				"items": {}
			},
			"multipleOf": {
				"type": "number",
				"exclusiveMinimum": 0
			},
			"maximum": {
				"type": "number"
			},
			"exclusiveMaximum": {
				"type": "number"
			},
			"minimum": {
				"type": "number"
			},
			"exclusiveMinimum": {
				"type": "number"
			},
			"maxLength": {
				"$ref": "#/definitions/nonNegativeInteger"
			},
			"minLength": {
				"$ref": "#/definitions/nonNegativeIntegerDefault0"
			},
			"pattern": {
				"type": "string",
				"format": "regex"
			},
			"additionalItems": {
				"$ref": "#"
			},
			"items": {
				"anyOf": [
					{
						"$ref": "#"
					},
					{
						"$ref": "#/definitions/schemaArray"
					}
				],
				"default": {}
			},
			"maxItems": {
				"$ref": "#/definitions/nonNegativeInteger"
			},
			"minItems": {
				"$ref": "#/definitions/nonNegativeIntegerDefault0"
			},
			"uniqueItems": {
				"type": "boolean",
				"default": false
			},
			"contains": {
				"$ref": "#"
			},
			"maxProperties": {
				"$ref": "#/definitions/nonNegativeInteger"
			},
			"minProperties": {
				"$ref": "#/definitions/nonNegativeIntegerDefault0"
			},
			"required": {
				"$ref": "#/definitions/stringArray"
			},
			"additionalProperties": {
				"$ref": "#"
			},
			"definitions": {
				"type": "object",
				"additionalProperties": {
					"$ref": "#"
				},
				"default": {}
			},
			"properties": {
				"type": "object",
				"additionalProperties": {
					"$ref": "#"
				},
				"default": {}
			},
			"patternProperties": {
				"type": "object",
				"additionalProperties": {
					"$ref": "#"
				},
				"default": {}
			},
			"dependencies": {
				"type": "object",
				"additionalProperties": {
					"anyOf": [
						{
							"$ref": "#"
						},
						{
							"$ref": "#/definitions/stringArray"
						}
					]
				}
			},
			"propertyNames": {
				"$ref": "#"
			},
			"const": {},
			"enum": {
				"type": "array",
				"minItems": 1,
				"uniqueItems": true
			},
			"type": {
				"anyOf": [
					{
						"$ref": "#/definitions/simpleTypes"
					},
					{
						"type": "array",
						"items": {
							"$ref": "#/definitions/simpleTypes"
						},
						"minItems": 1,
						"uniqueItems": true
					}
				]
			},
			"format": {
				"type": "string"
			},
			"allOf": {
				"$ref": "#/definitions/schemaArray"
			},
			"anyOf": {
				"$ref": "#/definitions/schemaArray"
			},
			"oneOf": {
				"$ref": "#/definitions/schemaArray"
			},
			"not": {
				"$ref": "#"
			}
		},
		"default": {}
	};

/***/ },
/* 51 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';


	var Highlighter = __webpack_require__(52);
	var History = __webpack_require__(53);
	var SearchBox = __webpack_require__(56);
	var ContextMenu = __webpack_require__(57);
	var TreePath = __webpack_require__(59);
	var Node = __webpack_require__(60);
	var ModeSwitcher = __webpack_require__(63);
	var util = __webpack_require__(54);
	var autocomplete = __webpack_require__(64);
	var translate = __webpack_require__(58).translate;
	var setLanguages = __webpack_require__(58).setLanguages;
	var setLanguage = __webpack_require__(58).setLanguage;

	// create a mixin with the functions for tree mode
	var treemode = {};

	/**
	 * Create a tree editor
	 * @param {Element} container    Container element
	 * @param {Object}  [options]    Object with options. available options:
	 *                               {String} mode            Editor mode. Available values:
	 *                                                        'tree' (default), 'view',
	 *                                                        and 'form'.
	 *                               {Boolean} search         Enable search box.
	 *                                                        True by default
	 *                               {Boolean} history        Enable history (undo/redo).
	 *                                                        True by default
	 *                               {function} onChange      Callback method, triggered
	 *                                                        on change of contents
	 *                               {String} name            Field name for the root node.
	 *                               {boolean} escapeUnicode  If true, unicode
	 *                                                        characters are escaped.
	 *                                                        false by default.
	 *                               {Object} schema          A JSON Schema for validation
	 * @private
	 */
	treemode.create = function (container, options) {
	  if (!container) {
	    throw new Error('No container element provided.');
	  }
	  this.container = container;
	  this.dom = {};
	  this.highlighter = new Highlighter();
	  this.selection = undefined; // will hold the last input selection
	  this.multiselection = {
	    nodes: []
	  };
	  this.validateSchema = null; // will be set in .setSchema(schema)
	  this.errorNodes = [];

	  this.node = null;
	  this.focusTarget = null;

	  this._setOptions(options);

	  if (options.autocomplete)
	      this.autocomplete = new autocomplete(options.autocomplete);

	  if (this.options.history && this.options.mode !== 'view') {
	    this.history = new History(this);
	  }

	  this._createFrame();
	  this._createTable();
	};

	/**
	 * Destroy the editor. Clean up DOM, event listeners, and web workers.
	 */
	treemode.destroy = function () {
	  if (this.frame && this.container && this.frame.parentNode == this.container) {
	    this.container.removeChild(this.frame);
	    this.frame = null;
	  }
	  this.container = null;

	  this.dom = null;

	  this.clear();
	  this.node = null;
	  this.focusTarget = null;
	  this.selection = null;
	  this.multiselection = null;
	  this.errorNodes = null;
	  this.validateSchema = null;
	  this._debouncedValidate = null;

	  if (this.history) {
	    this.history.destroy();
	    this.history = null;
	  }

	  if (this.searchBox) {
	    this.searchBox.destroy();
	    this.searchBox = null;
	  }

	  if (this.modeSwitcher) {
	    this.modeSwitcher.destroy();
	    this.modeSwitcher = null;
	  }
	};

	/**
	 * Initialize and set default options
	 * @param {Object}  [options]    See description in constructor
	 * @private
	 */
	treemode._setOptions = function (options) {
	  this.options = {
	    search: true,
	    history: true,
	    mode: 'tree',
	    name: undefined,   // field name of root node
	    schema: null,
	    schemaRefs: null,
	    autocomplete: null,
	    navigationBar : true
	  };

	  // copy all options
	  if (options) {
	    for (var prop in options) {
	      if (options.hasOwnProperty(prop)) {
	        this.options[prop] = options[prop];
	      }
	    }
	  }

	  // compile a JSON schema validator if a JSON schema is provided
	  this.setSchema(this.options.schema, this.options.schemaRefs);

	  // create a debounced validate function
	  this._debouncedValidate = util.debounce(this.validate.bind(this), this.DEBOUNCE_INTERVAL);

	  setLanguages(this.options.languages);
	  setLanguage(this.options.language)
	};

	/**
	 * Set JSON object in editor
	 * @param {Object | undefined} json      JSON data
	 * @param {String}             [name]    Optional field name for the root node.
	 *                                       Can also be set using setName(name).
	 */
	treemode.set = function (json, name) {
	  // adjust field name for root node
	  if (name) {
	    // TODO: deprecated since version 2.2.0. Cleanup some day.
	    console.warn('Second parameter "name" is deprecated. Use setName(name) instead.');
	    this.options.name = name;
	  }

	  // verify if json is valid JSON, ignore when a function
	  if (json instanceof Function || (json === undefined)) {
	    this.clear();
	  }
	  else {
	    this.content.removeChild(this.table);  // Take the table offline

	    // replace the root node
	    var params = {
	      field: this.options.name,
	      value: json
	    };
	    var node = new Node(this, params);
	    this._setRoot(node);

	    // validate JSON schema (if configured)
	    this.validate();

	    // expand
	    var recurse = false;
	    this.node.expand(recurse);

	    this.content.appendChild(this.table);  // Put the table online again
	  }

	  // TODO: maintain history, store last state and previous document
	  if (this.history) {
	    this.history.clear();
	  }

	  // clear search
	  if (this.searchBox) {
	    this.searchBox.clear();
	  }
	};

	/**
	 * Get JSON object from editor
	 * @return {Object | undefined} json
	 */
	treemode.get = function () {
	  // remove focus from currently edited node
	  if (this.focusTarget) {
	    var node = Node.getNodeFromTarget(this.focusTarget);
	    if (node) {
	      node.blur();
	    }
	  }

	  if (this.node) {
	    return this.node.getValue();
	  }
	  else {
	    return undefined;
	  }
	};

	/**
	 * Get the text contents of the editor
	 * @return {String} jsonText
	 */
	treemode.getText = function() {
	  return JSON.stringify(this.get());
	};

	/**
	 * Set the text contents of the editor
	 * @param {String} jsonText
	 */
	treemode.setText = function(jsonText) {
	  try {
	    this.set(util.parse(jsonText)); // this can throw an error
	  }
	  catch (err) {
	    // try to sanitize json, replace JavaScript notation with JSON notation
	    var sanitizedJsonText = util.sanitize(jsonText);

	    // try to parse again
	    this.set(util.parse(sanitizedJsonText)); // this can throw an error
	  }
	};

	/**
	 * Set a field name for the root node.
	 * @param {String | undefined} name
	 */
	treemode.setName = function (name) {
	  this.options.name = name;
	  if (this.node) {
	    this.node.updateField(this.options.name);
	  }
	};

	/**
	 * Get the field name for the root node.
	 * @return {String | undefined} name
	 */
	treemode.getName = function () {
	  return this.options.name;
	};

	/**
	 * Set focus to the editor. Focus will be set to:
	 * - the first editable field or value, or else
	 * - to the expand button of the root node, or else
	 * - to the context menu button of the root node, or else
	 * - to the first button in the top menu
	 */
	treemode.focus = function () {
	  var input = this.content.querySelector('[contenteditable=true]');
	  if (input) {
	    input.focus();
	  }
	  else if (this.node.dom.expand) {
	    this.node.dom.expand.focus();
	  }
	  else if (this.node.dom.menu) {
	    this.node.dom.menu.focus();
	  }
	  else {
	    // focus to the first button in the menu
	    input = this.frame.querySelector('button');
	    if (input) {
	      input.focus();
	    }
	  }
	};

	/**
	 * Remove the root node from the editor
	 */
	treemode.clear = function () {
	  if (this.node) {
	    this.node.collapse();
	    this.tbody.removeChild(this.node.getDom());
	    delete this.node;
	  }
	};

	/**
	 * Set the root node for the json editor
	 * @param {Node} node
	 * @private
	 */
	treemode._setRoot = function (node) {
	  this.clear();

	  this.node = node;

	  // append to the dom
	  this.tbody.appendChild(node.getDom());
	};

	/**
	 * Search text in all nodes
	 * The nodes will be expanded when the text is found one of its childs,
	 * else it will be collapsed. Searches are case insensitive.
	 * @param {String} text
	 * @return {Object[]} results  Array with nodes containing the search results
	 *                             The result objects contains fields:
	 *                             - {Node} node,
	 *                             - {String} elem  the dom element name where
	 *                                              the result is found ('field' or
	 *                                              'value')
	 */
	treemode.search = function (text) {
	  var results;
	  if (this.node) {
	    this.content.removeChild(this.table);  // Take the table offline
	    results = this.node.search(text);
	    this.content.appendChild(this.table);  // Put the table online again
	  }
	  else {
	    results = [];
	  }

	  return results;
	};

	/**
	 * Expand all nodes
	 */
	treemode.expandAll = function () {
	  if (this.node) {
	    this.content.removeChild(this.table);  // Take the table offline
	    this.node.expand();
	    this.content.appendChild(this.table);  // Put the table online again
	  }
	};

	/**
	 * Collapse all nodes
	 */
	treemode.collapseAll = function () {
	  if (this.node) {
	    this.content.removeChild(this.table);  // Take the table offline
	    this.node.collapse();
	    this.content.appendChild(this.table);  // Put the table online again
	  }
	};

	/**
	 * The method onChange is called whenever a field or value is changed, created,
	 * deleted, duplicated, etc.
	 * @param {String} action  Change action. Available values: "editField",
	 *                         "editValue", "changeType", "appendNode",
	 *                         "removeNode", "duplicateNode", "moveNode", "expand",
	 *                         "collapse".
	 * @param {Object} params  Object containing parameters describing the change.
	 *                         The parameters in params depend on the action (for
	 *                         example for "editValue" the Node, old value, and new
	 *                         value are provided). params contains all information
	 *                         needed to undo or redo the action.
	 * @private
	 */
	treemode._onAction = function (action, params) {
	  // add an action to the history
	  if (this.history) {
	    this.history.add(action, params);
	  }

	  this._onChange();
	};

	/**
	 * Handle a change:
	 * - Validate JSON schema
	 * - Send a callback to the onChange listener if provided
	 * @private
	 */
	treemode._onChange = function () {
	  // validate JSON schema (if configured)
	  this._debouncedValidate();

	  // trigger the onChange callback
	  if (this.options.onChange) {
	    try {
	      this.options.onChange();
	    }
	    catch (err) {
	      console.error('Error in onChange callback: ', err);
	    }
	  }
	};

	/**
	 * Validate current JSON object against the configured JSON schema
	 * Throws an exception when no JSON schema is configured
	 */
	treemode.validate = function () {
	  // clear all current errors
	  if (this.errorNodes) {
	    this.errorNodes.forEach(function (node) {
	      node.setError(null);
	    });
	  }

	  var root = this.node;
	  if (!root) { // TODO: this should be redundant but is needed on mode switch
	    return;
	  }

	  // check for duplicate keys
	  var duplicateErrors = root.validate();

	  // validate the JSON
	  var schemaErrors = [];
	  if (this.validateSchema) {
	    var valid = this.validateSchema(root.getValue());
	    if (!valid) {
	      // apply all new errors
	      schemaErrors = this.validateSchema.errors
	          .map(function (error) {
	            return util.improveSchemaError(error);
	          })
	          .map(function findNode (error) {
	            return {
	              node: root.findNode(error.dataPath),
	              error: error
	            }
	          })
	          .filter(function hasNode (entry) {
	            return entry.node != null
	          });
	    }
	  }

	  var errorNodes = duplicateErrors.concat(schemaErrors);
	  var parentPairs = errorNodes
	      .reduce(function (all, entry) {
	          return entry.node
	              .findParents()
	              .filter(function (parent) {
	                  return !all.some(function (pair) {
	                    return pair[0] === parent;
	                  });
	              })
	              .map(function (parent) {
	                  return [parent, entry.node];
	              })
	              .concat(all);
	      }, []);

	  this.errorNodes = parentPairs
	      .map(function (pair) {
	          return {
	            node: pair[0],
	            child: pair[1],
	            error: {
	              message: pair[0].type === 'object'
	                  ? 'Contains invalid properties' // object
	                  : 'Contains invalid items'      // array
	            }
	          };
	      })
	      .concat(errorNodes)
	      .map(function setError (entry) {
	        entry.node.setError(entry.error, entry.child);
	        return entry.node;
	      });
	};

	/**
	 * Refresh the rendered contents
	 */
	treemode.refresh = function () {
	  if (this.node) {
	    this.node.updateDom({recurse: true});
	  }
	};

	/**
	 * Start autoscrolling when given mouse position is above the top of the
	 * editor contents, or below the bottom.
	 * @param {Number} mouseY  Absolute mouse position in pixels
	 */
	treemode.startAutoScroll = function (mouseY) {
	  var me = this;
	  var content = this.content;
	  var top = util.getAbsoluteTop(content);
	  var height = content.clientHeight;
	  var bottom = top + height;
	  var margin = 24;
	  var interval = 50; // ms

	  if ((mouseY < top + margin) && content.scrollTop > 0) {
	    this.autoScrollStep = ((top + margin) - mouseY) / 3;
	  }
	  else if (mouseY > bottom - margin &&
	      height + content.scrollTop < content.scrollHeight) {
	    this.autoScrollStep = ((bottom - margin) - mouseY) / 3;
	  }
	  else {
	    this.autoScrollStep = undefined;
	  }

	  if (this.autoScrollStep) {
	    if (!this.autoScrollTimer) {
	      this.autoScrollTimer = setInterval(function () {
	        if (me.autoScrollStep) {
	          content.scrollTop -= me.autoScrollStep;
	        }
	        else {
	          me.stopAutoScroll();
	        }
	      }, interval);
	    }
	  }
	  else {
	    this.stopAutoScroll();
	  }
	};

	/**
	 * Stop auto scrolling. Only applicable when scrolling
	 */
	treemode.stopAutoScroll = function () {
	  if (this.autoScrollTimer) {
	    clearTimeout(this.autoScrollTimer);
	    delete this.autoScrollTimer;
	  }
	  if (this.autoScrollStep) {
	    delete this.autoScrollStep;
	  }
	};


	/**
	 * Set the focus to an element in the editor, set text selection, and
	 * set scroll position.
	 * @param {Object} selection  An object containing fields:
	 *                            {Element | undefined} dom     The dom element
	 *                                                          which has focus
	 *                            {Range | TextRange} range     A text selection
	 *                            {Node[]} nodes                Nodes in case of multi selection
	 *                            {Number} scrollTop            Scroll position
	 */
	treemode.setSelection = function (selection) {
	  if (!selection) {
	    return;
	  }

	  if ('scrollTop' in selection && this.content) {
	    // TODO: animated scroll
	    this.content.scrollTop = selection.scrollTop;
	  }
	  if (selection.nodes) {
	    // multi-select
	    this.select(selection.nodes);
	  }
	  if (selection.range) {
	    util.setSelectionOffset(selection.range);
	  }
	  if (selection.dom) {
	    selection.dom.focus();
	  }
	};

	/**
	 * Get the current focus
	 * @return {Object} selection An object containing fields:
	 *                            {Element | undefined} dom     The dom element
	 *                                                          which has focus
	 *                            {Range | TextRange} range     A text selection
	 *                            {Node[]} nodes                Nodes in case of multi selection
	 *                            {Number} scrollTop            Scroll position
	 */
	treemode.getSelection = function () {
	  var range = util.getSelectionOffset();
	  if (range && range.container.nodeName !== 'DIV') { // filter on (editable) divs)
	    range = null;
	  }

	  return {
	    dom: this.focusTarget,
	    range: range,
	    nodes: this.multiselection.nodes.slice(0),
	    scrollTop: this.content ? this.content.scrollTop : 0
	  };
	};

	/**
	 * Adjust the scroll position such that given top position is shown at 1/4
	 * of the window height.
	 * @param {Number} top
	 * @param {function(boolean)} [callback]   Callback, executed when animation is
	 *                                         finished. The callback returns true
	 *                                         when animation is finished, or false
	 *                                         when not.
	 */
	treemode.scrollTo = function (top, callback) {
	  var content = this.content;
	  if (content) {
	    var editor = this;
	    // cancel any running animation
	    if (editor.animateTimeout) {
	      clearTimeout(editor.animateTimeout);
	      delete editor.animateTimeout;
	    }
	    if (editor.animateCallback) {
	      editor.animateCallback(false);
	      delete editor.animateCallback;
	    }

	    // calculate final scroll position
	    var height = content.clientHeight;
	    var bottom = content.scrollHeight - height;
	    var finalScrollTop = Math.min(Math.max(top - height / 4, 0), bottom);

	    // animate towards the new scroll position
	    var animate = function () {
	      var scrollTop = content.scrollTop;
	      var diff = (finalScrollTop - scrollTop);
	      if (Math.abs(diff) > 3) {
	        content.scrollTop += diff / 3;
	        editor.animateCallback = callback;
	        editor.animateTimeout = setTimeout(animate, 50);
	      }
	      else {
	        // finished
	        if (callback) {
	          callback(true);
	        }
	        content.scrollTop = finalScrollTop;
	        delete editor.animateTimeout;
	        delete editor.animateCallback;
	      }
	    };
	    animate();
	  }
	  else {
	    if (callback) {
	      callback(false);
	    }
	  }
	};

	/**
	 * Create main frame
	 * @private
	 */
	treemode._createFrame = function () {
	  // create the frame
	  this.frame = document.createElement('div');
	  this.frame.className = 'jsoneditor jsoneditor-mode-' + this.options.mode;
	  this.container.appendChild(this.frame);

	  // create one global event listener to handle all events from all nodes
	  var editor = this;
	  function onEvent(event) {
	    // when switching to mode "code" or "text" via the menu, some events
	    // are still fired whilst the _onEvent methods is already removed.
	    if (editor._onEvent) {
	      editor._onEvent(event);
	    }
	  }
	  this.frame.onclick = function (event) {
	    var target = event.target;// || event.srcElement;

	    onEvent(event);

	    // prevent default submit action of buttons when editor is located
	    // inside a form
	    if (target.nodeName == 'BUTTON') {
	      event.preventDefault();
	    }
	  };
	  this.frame.oninput = onEvent;
	  this.frame.onchange = onEvent;
	  this.frame.onkeydown = onEvent;
	  this.frame.onkeyup = onEvent;
	  this.frame.oncut = onEvent;
	  this.frame.onpaste = onEvent;
	  this.frame.onmousedown = onEvent;
	  this.frame.onmouseup = onEvent;
	  this.frame.onmouseover = onEvent;
	  this.frame.onmouseout = onEvent;
	  // Note: focus and blur events do not propagate, therefore they defined
	  // using an eventListener with useCapture=true
	  // see http://www.quirksmode.org/blog/archives/2008/04/delegating_the.html
	  util.addEventListener(this.frame, 'focus', onEvent, true);
	  util.addEventListener(this.frame, 'blur', onEvent, true);
	  this.frame.onfocusin = onEvent;  // for IE
	  this.frame.onfocusout = onEvent; // for IE

	  // create menu
	  this.menu = document.createElement('div');
	  this.menu.className = 'jsoneditor-menu';
	  this.frame.appendChild(this.menu);

	  // create expand all button
	  var expandAll = document.createElement('button');
	  expandAll.type = 'button';
	  expandAll.className = 'jsoneditor-expand-all';
	  expandAll.title = translate('expandAll');
	  expandAll.onclick = function () {
	    editor.expandAll();
	  };
	  this.menu.appendChild(expandAll);

	  // create collapse all button
	  var collapseAll = document.createElement('button');
	  collapseAll.type = 'button';
	  collapseAll.title = translate('collapseAll');
	  collapseAll.className = 'jsoneditor-collapse-all';
	  collapseAll.onclick = function () {
	    editor.collapseAll();
	  };
	  this.menu.appendChild(collapseAll);

	  // create undo/redo buttons
	  if (this.history) {
	    // create undo button
	    var undo = document.createElement('button');
	    undo.type = 'button';
	    undo.className = 'jsoneditor-undo jsoneditor-separator';
	    undo.title = translate('undo');
	    undo.onclick = function () {
	      editor._onUndo();
	    };
	    this.menu.appendChild(undo);
	    this.dom.undo = undo;

	    // create redo button
	    var redo = document.createElement('button');
	    redo.type = 'button';
	    redo.className = 'jsoneditor-redo';
	    redo.title = translate('redo');
	    redo.onclick = function () {
	      editor._onRedo();
	    };
	    this.menu.appendChild(redo);
	    this.dom.redo = redo;

	    // register handler for onchange of history
	    this.history.onChange = function () {
	      undo.disabled = !editor.history.canUndo();
	      redo.disabled = !editor.history.canRedo();
	    };
	    this.history.onChange();
	  }

	  // create mode box
	  if (this.options && this.options.modes && this.options.modes.length) {
	    var me = this;
	    this.modeSwitcher = new ModeSwitcher(this.menu, this.options.modes, this.options.mode, function onSwitch(mode) {
	      me.modeSwitcher.destroy();

	      // switch mode and restore focus
	      me.setMode(mode);
	      me.modeSwitcher.focus();
	    });
	  }

	  // create search box
	  if (this.options.search) {
	    this.searchBox = new SearchBox(this, this.menu);
	  }

	  if(this.options.navigationBar) {
	    // create second menu row for treepath
	    this.navBar = document.createElement('div');
	    this.navBar.className = 'jsoneditor-navigation-bar nav-bar-empty';
	    this.frame.appendChild(this.navBar);

	    this.treePath = new TreePath(this.navBar);
	    this.treePath.onSectionSelected(this._onTreePathSectionSelected.bind(this));
	    this.treePath.onContextMenuItemSelected(this._onTreePathMenuItemSelected.bind(this));
	  }
	};

	/**
	 * Perform an undo action
	 * @private
	 */
	treemode._onUndo = function () {
	  if (this.history) {
	    // undo last action
	    this.history.undo();

	    // fire change event
	    this._onChange();
	  }
	};

	/**
	 * Perform a redo action
	 * @private
	 */
	treemode._onRedo = function () {
	  if (this.history) {
	    // redo last action
	    this.history.redo();

	    // fire change event
	    this._onChange();
	  }
	};

	/**
	 * Event handler
	 * @param event
	 * @private
	 */
	treemode._onEvent = function (event) {
	  if (event.type === 'keydown') {
	    this._onKeyDown(event);
	  }

	  if (event.type === 'focus') {
	    this.focusTarget = event.target;
	  }

	  if (event.type === 'mousedown') {
	    this._startDragDistance(event);
	  }
	  if (event.type === 'mousemove' || event.type === 'mouseup' || event.type === 'click') {
	    this._updateDragDistance(event);
	  }

	  var node = Node.getNodeFromTarget(event.target);

	  if (node && this.options && this.options.navigationBar && node && (event.type === 'keydown' || event.type === 'mousedown')) {
	    // apply on next tick, right after the new key press is applied
	    var me = this;
	    setTimeout(function () {
	      me._updateTreePath(node.getNodePath());
	    })
	  }

	  if (node && node.selected) {
	    if (event.type === 'click') {
	      if (event.target === node.dom.menu) {
	        this.showContextMenu(event.target);

	        // stop propagation (else we will open the context menu of a single node)
	        return;
	      }

	      // deselect a multi selection
	      if (!event.hasMoved) {
	        this.deselect();
	      }
	    }

	    if (event.type === 'mousedown') {
	      // drag multiple nodes
	      Node.onDragStart(this.multiselection.nodes, event);
	    }
	  }
	  else {
	    if (event.type === 'mousedown') {
	      this.deselect();

	      if (node && event.target === node.dom.drag) {
	        // drag a singe node
	        Node.onDragStart(node, event);
	      }
	      else if (!node || (event.target !== node.dom.field && event.target !== node.dom.value && event.target !== node.dom.select)) {
	        // select multiple nodes
	        this._onMultiSelectStart(event);
	      }
	    }
	  }

	  if (node) {
	    node.onEvent(event);
	  }
	};

	/**
	 * Update TreePath components
	 * @param {Array<Node>} pathNodes list of nodes in path from root to selection 
	 * @private
	 */
	treemode._updateTreePath = function (pathNodes) {
	  if (pathNodes && pathNodes.length) {
	    util.removeClassName(this.navBar, 'nav-bar-empty');
	    
	    var pathObjs = [];
	    pathNodes.forEach(function (node) {
	      var pathObj = {
	        name: getName(node),
	        node: node,
	        children: []
	      }
	      if (node.childs && node.childs.length) {
	        node.childs.forEach(function (childNode) {
	          pathObj.children.push({
	            name: getName(childNode),
	            node: childNode
	          });
	        });
	      }
	      pathObjs.push(pathObj);
	    });
	    this.treePath.setPath(pathObjs);
	  } else {
	    util.addClassName(this.navBar, 'nav-bar-empty');
	  }

	  function getName(node) {
	    return node.field !== undefined
	        ? node._escapeHTML(node.field)
	        : (isNaN(node.index) ? node.type : node.index);
	  }
	};

	/**
	 * Callback for tree path section selection - focus the selected node in the tree
	 * @param {Object} pathObj path object that was represents the selected section node
	 * @private
	 */
	treemode._onTreePathSectionSelected = function (pathObj) {
	  if(pathObj && pathObj.node) {
	    pathObj.node.expandTo();
	    pathObj.node.focus();
	  }
	};

	/**
	 * Callback for tree path menu item selection - rebuild the path accrding to the new selection and focus the selected node in the tree
	 * @param {Object} pathObj path object that was represents the parent section node
	 * @param {String} selection selected section child
	 * @private
	 */
	treemode._onTreePathMenuItemSelected = function (pathObj, selection) {
	  if(pathObj && pathObj.children.length) {
	    var selectionObj = pathObj.children.find(function (obj) {
	      return obj.name === selection;
	    });
	    if(selectionObj && selectionObj.node) {
	      this._updateTreePath(selectionObj.node.getNodePath());
	      selectionObj.node.expandTo();
	      selectionObj.node.focus();
	    }
	  }
	};

	treemode._startDragDistance = function (event) {
	  this.dragDistanceEvent = {
	    initialTarget: event.target,
	    initialPageX: event.pageX,
	    initialPageY: event.pageY,
	    dragDistance: 0,
	    hasMoved: false
	  };
	};

	treemode._updateDragDistance = function (event) {
	  if (!this.dragDistanceEvent) {
	    this._startDragDistance(event);
	  }

	  var diffX = event.pageX - this.dragDistanceEvent.initialPageX;
	  var diffY = event.pageY - this.dragDistanceEvent.initialPageY;

	  this.dragDistanceEvent.dragDistance = Math.sqrt(diffX * diffX + diffY * diffY);
	  this.dragDistanceEvent.hasMoved =
	      this.dragDistanceEvent.hasMoved || this.dragDistanceEvent.dragDistance > 10;

	  event.dragDistance = this.dragDistanceEvent.dragDistance;
	  event.hasMoved = this.dragDistanceEvent.hasMoved;

	  return event.dragDistance;
	};

	/**
	 * Start multi selection of nodes by dragging the mouse
	 * @param event
	 * @private
	 */
	treemode._onMultiSelectStart = function (event) {
	  var node = Node.getNodeFromTarget(event.target);

	  if (this.options.mode !== 'tree' || this.options.onEditable !== undefined) {
	    // dragging not allowed in modes 'view' and 'form'
	    // TODO: allow multiselection of items when option onEditable is specified
	    return;
	  }

	  this.multiselection = {
	    start: node || null,
	    end: null,
	    nodes: []
	  };

	  this._startDragDistance(event);

	  var editor = this;
	  if (!this.mousemove) {
	    this.mousemove = util.addEventListener(window, 'mousemove', function (event) {
	      editor._onMultiSelect(event);
	    });
	  }
	  if (!this.mouseup) {
	    this.mouseup = util.addEventListener(window, 'mouseup', function (event ) {
	      editor._onMultiSelectEnd(event);
	    });
	  }

	};

	/**
	 * Multiselect nodes by dragging
	 * @param event
	 * @private
	 */
	treemode._onMultiSelect = function (event) {
	  event.preventDefault();

	  this._updateDragDistance(event);
	  if (!event.hasMoved) {
	    return;
	  }

	  var node = Node.getNodeFromTarget(event.target);

	  if (node) {
	    if (this.multiselection.start == null) {
	      this.multiselection.start = node;
	    }
	    this.multiselection.end = node;
	  }

	  // deselect previous selection
	  this.deselect();

	  // find the selected nodes in the range from first to last
	  var start = this.multiselection.start;
	  var end = this.multiselection.end || this.multiselection.start;
	  if (start && end) {
	    // find the top level childs, all having the same parent
	    this.multiselection.nodes = this._findTopLevelNodes(start, end);
	    this.select(this.multiselection.nodes);
	  }
	};

	/**
	 * End of multiselect nodes by dragging
	 * @param event
	 * @private
	 */
	treemode._onMultiSelectEnd = function (event) {
	  // set focus to the context menu button of the first node
	  if (this.multiselection.nodes[0]) {
	    this.multiselection.nodes[0].dom.menu.focus();
	  }

	  this.multiselection.start = null;
	  this.multiselection.end = null;

	  // cleanup global event listeners
	  if (this.mousemove) {
	    util.removeEventListener(window, 'mousemove', this.mousemove);
	    delete this.mousemove;
	  }
	  if (this.mouseup) {
	    util.removeEventListener(window, 'mouseup', this.mouseup);
	    delete this.mouseup;
	  }
	};

	/**
	 * deselect currently selected nodes
	 * @param {boolean} [clearStartAndEnd=false]  If true, the `start` and `end`
	 *                                            state is cleared too.
	 */
	treemode.deselect = function (clearStartAndEnd) {
	  this.multiselection.nodes.forEach(function (node) {
	    node.setSelected(false);
	  });
	  this.multiselection.nodes = [];

	  if (clearStartAndEnd) {
	    this.multiselection.start = null;
	    this.multiselection.end = null;
	  }
	};

	/**
	 * select nodes
	 * @param {Node[] | Node} nodes
	 */
	treemode.select = function (nodes) {
	  if (!Array.isArray(nodes)) {
	    return this.select([nodes]);
	  }

	  if (nodes) {
	    this.deselect();

	    this.multiselection.nodes = nodes.slice(0);

	    var first = nodes[0];
	    nodes.forEach(function (node) {
	      node.setSelected(true, node === first);
	    });
	  }
	};

	/**
	 * From two arbitrary selected nodes, find their shared parent node.
	 * From that parent node, select the two child nodes in the brances going to
	 * nodes `start` and `end`, and select all childs in between.
	 * @param {Node} start
	 * @param {Node} end
	 * @return {Array.<Node>} Returns an ordered list with child nodes
	 * @private
	 */
	treemode._findTopLevelNodes = function (start, end) {
	  var startPath = start.getNodePath();
	  var endPath = end.getNodePath();
	  var i = 0;
	  while (i < startPath.length && startPath[i] === endPath[i]) {
	    i++;
	  }
	  var root = startPath[i - 1];
	  var startChild = startPath[i];
	  var endChild = endPath[i];

	  if (!startChild || !endChild) {
	    if (root.parent) {
	      // startChild is a parent of endChild or vice versa
	      startChild = root;
	      endChild = root;
	      root = root.parent
	    }
	    else {
	      // we have selected the root node (which doesn't have a parent)
	      startChild = root.childs[0];
	      endChild = root.childs[root.childs.length - 1];
	    }
	  }

	  if (root && startChild && endChild) {
	    var startIndex = root.childs.indexOf(startChild);
	    var endIndex = root.childs.indexOf(endChild);
	    var firstIndex = Math.min(startIndex, endIndex);
	    var lastIndex = Math.max(startIndex, endIndex);

	    return root.childs.slice(firstIndex, lastIndex + 1);
	  }
	  else {
	    return [];
	  }
	};

	/**
	 * Event handler for keydown. Handles shortcut keys
	 * @param {Event} event
	 * @private
	 */
	treemode._onKeyDown = function (event) {
	  var keynum = event.which || event.keyCode;
	  var altKey = event.altKey;
	  var ctrlKey = event.ctrlKey;
	  var metaKey = event.metaKey;
	  var shiftKey = event.shiftKey;
	  var handled = false;

	  if (keynum == 9) { // Tab or Shift+Tab
	    var me = this;
	    setTimeout(function () {
	      // select all text when moving focus to an editable div
	      util.selectContentEditable(me.focusTarget);
	    }, 0);
	  }

	  if (this.searchBox) {
	    if (ctrlKey && keynum == 70) { // Ctrl+F
	      this.searchBox.dom.search.focus();
	      this.searchBox.dom.search.select();
	      handled = true;
	    }
	    else if (keynum == 114 || (ctrlKey && keynum == 71)) { // F3 or Ctrl+G
	      var focus = true;
	      if (!shiftKey) {
	        // select next search result (F3 or Ctrl+G)
	        this.searchBox.next(focus);
	      }
	      else {
	        // select previous search result (Shift+F3 or Ctrl+Shift+G)
	        this.searchBox.previous(focus);
	      }

	      handled = true;
	    }
	  }

	  if (this.history) {
	    if (ctrlKey && !shiftKey && keynum == 90) { // Ctrl+Z
	      // undo
	      this._onUndo();
	      handled = true;
	    }
	    else if (ctrlKey && shiftKey && keynum == 90) { // Ctrl+Shift+Z
	      // redo
	      this._onRedo();
	      handled = true;
	    }
	  }

	  if ((this.options.autocomplete) && (!handled)) {
	      if (!ctrlKey && !altKey && !metaKey && (event.key.length == 1 || keynum == 8 || keynum == 46)) {
	          handled = false;
	          var jsonElementType = "";
	          if (event.target.className.indexOf("jsoneditor-value") >= 0) jsonElementType = "value";
	          if (event.target.className.indexOf("jsoneditor-field") >= 0) jsonElementType = "field";

	          var node = Node.getNodeFromTarget(event.target);
	          // Activate autocomplete
	          setTimeout(function (hnode, element) {
	              if (element.innerText.length > 0) {
	                  var result = this.options.autocomplete.getOptions(element.innerText, hnode.getPath(), jsonElementType, hnode.editor);
	                  if (result === null) {
	                      this.autocomplete.hideDropDown();
	                  } else if (typeof result.then === 'function') {
	                      // probably a promise
	                      if (result.then(function (obj) {
	                          if (obj === null) {
	                              this.autocomplete.hideDropDown();
	                          } else if (obj.options) {
	                              this.autocomplete.show(element, obj.startFrom, obj.options);
	                          } else {
	                              this.autocomplete.show(element, 0, obj);
	                          }
	                      }.bind(this)));
	                  } else {
	                      // definitely not a promise
	                      if (result.options)
	                          this.autocomplete.show(element, result.startFrom, result.options);
	                      else
	                          this.autocomplete.show(element, 0, result);
	                  }
	              }
	              else
	                  this.autocomplete.hideDropDown();

	          }.bind(this, node, event.target), 50);
	      } 
	  }

	  if (handled) {
	    event.preventDefault();
	    event.stopPropagation();
	  }
	};

	/**
	 * Create main table
	 * @private
	 */
	treemode._createTable = function () {
	  var contentOuter = document.createElement('div');
	  contentOuter.className = 'jsoneditor-outer';
	  if(this.options.navigationBar) {
	    util.addClassName(contentOuter, 'has-nav-bar');
	  }
	  this.contentOuter = contentOuter;

	  this.content = document.createElement('div');
	  this.content.className = 'jsoneditor-tree';
	  contentOuter.appendChild(this.content);

	  this.table = document.createElement('table');
	  this.table.className = 'jsoneditor-tree';
	  this.content.appendChild(this.table);

	  // create colgroup where the first two columns don't have a fixed
	  // width, and the edit columns do have a fixed width
	  var col;
	  this.colgroupContent = document.createElement('colgroup');
	  if (this.options.mode === 'tree') {
	    col = document.createElement('col');
	    col.width = "24px";
	    this.colgroupContent.appendChild(col);
	  }
	  col = document.createElement('col');
	  col.width = "24px";
	  this.colgroupContent.appendChild(col);
	  col = document.createElement('col');
	  this.colgroupContent.appendChild(col);
	  this.table.appendChild(this.colgroupContent);

	  this.tbody = document.createElement('tbody');
	  this.table.appendChild(this.tbody);

	  this.frame.appendChild(contentOuter);
	};

	/**
	 * Show a contextmenu for this node.
	 * Used for multiselection
	 * @param {HTMLElement} anchor   Anchor element to attach the context menu to.
	 * @param {function} [onClose]   Callback method called when the context menu
	 *                               is being closed.
	 */
	treemode.showContextMenu = function (anchor, onClose) {
	  var items = [];
	  var editor = this;

	  // create duplicate button
	  items.push({
	    text: translate('duplicateText'),
	    title: translate('duplicateTitle'),
	    className: 'jsoneditor-duplicate',
	    click: function () {
	      Node.onDuplicate(editor.multiselection.nodes);
	    }
	  });

	  // create remove button
	  items.push({
	    text: translate('remove'),
	    title: translate('removeTitle'),
	    className: 'jsoneditor-remove',
	    click: function () {
	      Node.onRemove(editor.multiselection.nodes);
	    }
	  });

	  var menu = new ContextMenu(items, {close: onClose});
	  menu.show(anchor, this.content);
	};


	// define modes
	module.exports = [
	  {
	    mode: 'tree',
	    mixin: treemode,
	    data: 'json'
	  },
	  {
	    mode: 'view',
	    mixin: treemode,
	    data: 'json'
	  },
	  {
	    mode: 'form',
	    mixin: treemode,
	    data: 'json'
	  }
	];


/***/ },
/* 52 */
/***/ function(module, exports) {

	'use strict';

	/**
	 * The highlighter can highlight/unhighlight a node, and
	 * animate the visibility of a context menu.
	 * @constructor Highlighter
	 */
	function Highlighter () {
	  this.locked = false;
	}

	/**
	 * Hightlight given node and its childs
	 * @param {Node} node
	 */
	Highlighter.prototype.highlight = function (node) {
	  if (this.locked) {
	    return;
	  }

	  if (this.node != node) {
	    // unhighlight current node
	    if (this.node) {
	      this.node.setHighlight(false);
	    }

	    // highlight new node
	    this.node = node;
	    this.node.setHighlight(true);
	  }

	  // cancel any current timeout
	  this._cancelUnhighlight();
	};

	/**
	 * Unhighlight currently highlighted node.
	 * Will be done after a delay
	 */
	Highlighter.prototype.unhighlight = function () {
	  if (this.locked) {
	    return;
	  }

	  var me = this;
	  if (this.node) {
	    this._cancelUnhighlight();

	    // do the unhighlighting after a small delay, to prevent re-highlighting
	    // the same node when moving from the drag-icon to the contextmenu-icon
	    // or vice versa.
	    this.unhighlightTimer = setTimeout(function () {
	      me.node.setHighlight(false);
	      me.node = undefined;
	      me.unhighlightTimer = undefined;
	    }, 0);
	  }
	};

	/**
	 * Cancel an unhighlight action (if before the timeout of the unhighlight action)
	 * @private
	 */
	Highlighter.prototype._cancelUnhighlight = function () {
	  if (this.unhighlightTimer) {
	    clearTimeout(this.unhighlightTimer);
	    this.unhighlightTimer = undefined;
	  }
	};

	/**
	 * Lock highlighting or unhighlighting nodes.
	 * methods highlight and unhighlight do not work while locked.
	 */
	Highlighter.prototype.lock = function () {
	  this.locked = true;
	};

	/**
	 * Unlock highlighting or unhighlighting nodes
	 */
	Highlighter.prototype.unlock = function () {
	  this.locked = false;
	};

	module.exports = Highlighter;


/***/ },
/* 53 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var util = __webpack_require__(54);

	/**
	 * @constructor History
	 * Store action history, enables undo and redo
	 * @param {JSONEditor} editor
	 */
	function History (editor) {
	  this.editor = editor;
	  this.history = [];
	  this.index = -1;

	  this.clear();

	  // map with all supported actions
	  this.actions = {
	    'editField': {
	      'undo': function (params) {
	        params.node.updateField(params.oldValue);
	      },
	      'redo': function (params) {
	        params.node.updateField(params.newValue);
	      }
	    },
	    'editValue': {
	      'undo': function (params) {
	        params.node.updateValue(params.oldValue);
	      },
	      'redo': function (params) {
	        params.node.updateValue(params.newValue);
	      }
	    },
	    'changeType': {
	      'undo': function (params) {
	        params.node.changeType(params.oldType);
	      },
	      'redo': function (params) {
	        params.node.changeType(params.newType);
	      }
	    },

	    'appendNodes': {
	      'undo': function (params) {
	        params.nodes.forEach(function (node) {
	          params.parent.removeChild(node);
	        });
	      },
	      'redo': function (params) {
	        params.nodes.forEach(function (node) {
	          params.parent.appendChild(node);
	        });
	      }
	    },
	    'insertBeforeNodes': {
	      'undo': function (params) {
	        params.nodes.forEach(function (node) {
	          params.parent.removeChild(node);
	        });
	      },
	      'redo': function (params) {
	        params.nodes.forEach(function (node) {
	          params.parent.insertBefore(node, params.beforeNode);
	        });
	      }
	    },
	    'insertAfterNodes': {
	      'undo': function (params) {
	        params.nodes.forEach(function (node) {
	          params.parent.removeChild(node);
	        });
	      },
	      'redo': function (params) {
	        var afterNode = params.afterNode;
	        params.nodes.forEach(function (node) {
	          params.parent.insertAfter(params.node, afterNode);
	          afterNode = node;
	        });
	      }
	    },
	    'removeNodes': {
	      'undo': function (params) {
	        var parent = params.parent;
	        var beforeNode = parent.childs[params.index] || parent.append;
	        params.nodes.forEach(function (node) {
	          parent.insertBefore(node, beforeNode);
	        });
	      },
	      'redo': function (params) {
	        params.nodes.forEach(function (node) {
	          params.parent.removeChild(node);
	        });
	      }
	    },
	    'duplicateNodes': {
	      'undo': function (params) {
	        params.nodes.forEach(function (node) {
	          params.parent.removeChild(node);
	        });
	      },
	      'redo': function (params) {
	        var afterNode = params.afterNode;
	        params.nodes.forEach(function (node) {
	          params.parent.insertAfter(node, afterNode);
	          afterNode = node;
	        });
	      }
	    },
	    'moveNodes': {
	      'undo': function (params) {
	        params.nodes.forEach(function (node) {
	          params.oldBeforeNode.parent.moveBefore(node, params.oldBeforeNode);
	        });
	      },
	      'redo': function (params) {
	        params.nodes.forEach(function (node) {
	          params.newBeforeNode.parent.moveBefore(node, params.newBeforeNode);
	        });
	      }
	    },

	    'sort': {
	      'undo': function (params) {
	        var node = params.node;
	        node.hideChilds();
	        node.sort = params.oldSort;
	        node.childs = params.oldChilds;
	        node.showChilds();
	      },
	      'redo': function (params) {
	        var node = params.node;
	        node.hideChilds();
	        node.sort = params.newSort;
	        node.childs = params.newChilds;
	        node.showChilds();
	      }
	    }

	    // TODO: restore the original caret position and selection with each undo
	    // TODO: implement history for actions "expand", "collapse", "scroll", "setDocument"
	  };
	}

	/**
	 * The method onChange is executed when the History is changed, and can
	 * be overloaded.
	 */
	History.prototype.onChange = function () {};

	/**
	 * Add a new action to the history
	 * @param {String} action  The executed action. Available actions: "editField",
	 *                         "editValue", "changeType", "appendNode",
	 *                         "removeNode", "duplicateNode", "moveNode"
	 * @param {Object} params  Object containing parameters describing the change.
	 *                         The parameters in params depend on the action (for
	 *                         example for "editValue" the Node, old value, and new
	 *                         value are provided). params contains all information
	 *                         needed to undo or redo the action.
	 */
	History.prototype.add = function (action, params) {
	  this.index++;
	  this.history[this.index] = {
	    'action': action,
	    'params': params,
	    'timestamp': new Date()
	  };

	  // remove redo actions which are invalid now
	  if (this.index < this.history.length - 1) {
	    this.history.splice(this.index + 1, this.history.length - this.index - 1);
	  }

	  // fire onchange event
	  this.onChange();
	};

	/**
	 * Clear history
	 */
	History.prototype.clear = function () {
	  this.history = [];
	  this.index = -1;

	  // fire onchange event
	  this.onChange();
	};

	/**
	 * Check if there is an action available for undo
	 * @return {Boolean} canUndo
	 */
	History.prototype.canUndo = function () {
	  return (this.index >= 0);
	};

	/**
	 * Check if there is an action available for redo
	 * @return {Boolean} canRedo
	 */
	History.prototype.canRedo = function () {
	  return (this.index < this.history.length - 1);
	};

	/**
	 * Undo the last action
	 */
	History.prototype.undo = function () {
	  if (this.canUndo()) {
	    var obj = this.history[this.index];
	    if (obj) {
	      var action = this.actions[obj.action];
	      if (action && action.undo) {
	        action.undo(obj.params);
	        if (obj.params.oldSelection) {
	          this.editor.setSelection(obj.params.oldSelection);
	        }
	      }
	      else {
	        console.error(new Error('unknown action "' + obj.action + '"'));
	      }
	    }
	    this.index--;

	    // fire onchange event
	    this.onChange();
	  }
	};

	/**
	 * Redo the last action
	 */
	History.prototype.redo = function () {
	  if (this.canRedo()) {
	    this.index++;

	    var obj = this.history[this.index];
	    if (obj) {
	      var action = this.actions[obj.action];
	      if (action && action.redo) {
	        action.redo(obj.params);
	        if (obj.params.newSelection) {
	          this.editor.setSelection(obj.params.newSelection);
	        }
	      }
	      else {
	        console.error(new Error('unknown action "' + obj.action + '"'));
	      }
	    }

	    // fire onchange event
	    this.onChange();
	  }
	};

	/**
	 * Destroy history
	 */
	History.prototype.destroy = function () {
	  this.editor = null;

	  this.history = [];
	  this.index = -1;
	};

	module.exports = History;


/***/ },
/* 54 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var jsonlint = __webpack_require__(55);

	/**
	 * Parse JSON using the parser built-in in the browser.
	 * On exception, the jsonString is validated and a detailed error is thrown.
	 * @param {String} jsonString
	 * @return {JSON} json
	 */
	exports.parse = function parse(jsonString) {
	  try {
	    return JSON.parse(jsonString);
	  }
	  catch (err) {
	    // try to throw a more detailed error message using validate
	    exports.validate(jsonString);

	    // rethrow the original error
	    throw err;
	  }
	};

	/**
	 * Sanitize a JSON-like string containing. For example changes JavaScript
	 * notation into JSON notation.
	 * This function for example changes a string like "{a: 2, 'b': {c: 'd'}"
	 * into '{"a": 2, "b": {"c": "d"}'
	 * @param {string} jsString
	 * @returns {string} json
	 */
	exports.sanitize = function (jsString) {
	  // escape all single and double quotes inside strings
	  var chars = [];
	  var i = 0;

	  //If JSON starts with a function (characters/digits/"_-"), remove this function.
	  //This is useful for "stripping" JSONP objects to become JSON
	  //For example: /* some comment */ function_12321321 ( [{"a":"b"}] ); => [{"a":"b"}]
	  var match = jsString.match(/^\s*(\/\*(.|[\r\n])*?\*\/)?\s*[\da-zA-Z_$]+\s*\(([\s\S]*)\)\s*;?\s*$/);
	  if (match) {
	    jsString = match[3];
	  }

	  var controlChars = {
	    '\b': '\\b',
	    '\f': '\\f',
	    '\n': '\\n',
	    '\r': '\\r',
	    '\t': '\\t'
	  };

	  var quote = '\'';
	  var quoteDbl = '"';
	  var quoteLeft = '\u2018';
	  var quoteRight = '\u2019';
	  var quoteDblLeft = '\u201C';
	  var quoteDblRight = '\u201D';
	  var graveAccent = '\u0060';
	  var acuteAccent = '\u00B4';

	  // helper functions to get the current/prev/next character
	  function curr () { return jsString.charAt(i);     }
	  function next()  { return jsString.charAt(i + 1); }
	  function prev()  { return jsString.charAt(i - 1); }

	  // get the last parsed non-whitespace character
	  function lastNonWhitespace () {
	    var p = chars.length - 1;

	    while (p >= 0) {
	      var pp = chars[p];
	      if (pp !== ' ' && pp !== '\n' && pp !== '\r' && pp !== '\t') { // non whitespace
	        return pp;
	      }
	      p--;
	    }

	    return '';
	  }

	  // skip a block comment '/* ... */'
	  function skipBlockComment () {
	    i += 2;
	    while (i < jsString.length && (curr() !== '*' || next() !== '/')) {
	      i++;
	    }
	    i += 2;
	  }

	  // skip a comment '// ...'
	  function skipComment () {
	    i += 2;
	    while (i < jsString.length && (curr() !== '\n')) {
	      i++;
	    }
	  }

	  // parse single or double quoted string
	  function parseString(endQuote) {
	    chars.push('"');
	    i++;
	    var c = curr();
	    while (i < jsString.length && c !== endQuote) {
	      if (c === '"' && prev() !== '\\') {
	        // unescaped double quote, escape it
	        chars.push('\\"');
	      }
	      else if (controlChars.hasOwnProperty(c)) {
	        // replace unescaped control characters with escaped ones
	        chars.push(controlChars[c])
	      }
	      else if (c === '\\') {
	        // remove the escape character when followed by a single quote ', not needed
	        i++;
	        c = curr();
	        if (c !== '\'') {
	          chars.push('\\');
	        }
	        chars.push(c);
	      }
	      else {
	        // regular character
	        chars.push(c);
	      }

	      i++;
	      c = curr();
	    }
	    if (c === endQuote) {
	      chars.push('"');
	      i++;
	    }
	  }

	  // parse an unquoted key
	  function parseKey() {
	    var specialValues = ['null', 'true', 'false'];
	    var key = '';
	    var c = curr();

	    var regexp = /[a-zA-Z_$\d]/; // letter, number, underscore, dollar character
	    while (regexp.test(c)) {
	      key += c;
	      i++;
	      c = curr();
	    }

	    if (specialValues.indexOf(key) === -1) {
	      chars.push('"' + key + '"');
	    }
	    else {
	      chars.push(key);
	    }
	  }

	  while(i < jsString.length) {
	    var c = curr();

	    if (c === '/' && next() === '*') {
	      skipBlockComment();
	    }
	    else if (c === '/' && next() === '/') {
	      skipComment();
	    }
	    else if (c === '\u00A0' || (c >= '\u2000' && c <= '\u200A') || c === '\u202F' || c === '\u205F' || c === '\u3000') {
	      // special white spaces (like non breaking space)
	      chars.push(' ')
	      i++
	    }
	    else if (c === quote) {
	      parseString(quote);
	    }
	    else if (c === quoteDbl) {
	      parseString(quoteDbl);
	    }
	    else if (c === graveAccent) {
	      parseString(acuteAccent);
	    }
	    else if (c === quoteLeft) {
	      parseString(quoteRight);
	    }
	    else if (c === quoteDblLeft) {
	      parseString(quoteDblRight);
	    }
	    else if (/[a-zA-Z_$]/.test(c) && ['{', ','].indexOf(lastNonWhitespace()) !== -1) {
	      // an unquoted object key (like a in '{a:2}')
	      parseKey();
	    }
	    else {
	      chars.push(c);
	      i++;
	    }
	  }

	  return chars.join('');
	};

	/**
	 * Escape unicode characters.
	 * For example input '\u2661' (length 1) will output '\\u2661' (length 5).
	 * @param {string} text
	 * @return {string}
	 */
	exports.escapeUnicodeChars = function (text) {
	  // see https://www.wikiwand.com/en/UTF-16
	  // note: we leave surrogate pairs as two individual chars,
	  // as JSON doesn't interpret them as a single unicode char.
	  return text.replace(/[\u007F-\uFFFF]/g, function(c) {
	    return '\\u'+('0000' + c.charCodeAt(0).toString(16)).slice(-4);
	  })
	};

	/**
	 * Validate a string containing a JSON object
	 * This method uses JSONLint to validate the String. If JSONLint is not
	 * available, the built-in JSON parser of the browser is used.
	 * @param {String} jsonString   String with an (invalid) JSON object
	 * @throws Error
	 */
	exports.validate = function validate(jsonString) {
	  if (typeof(jsonlint) != 'undefined') {
	    jsonlint.parse(jsonString);
	  }
	  else {
	    JSON.parse(jsonString);
	  }
	};

	/**
	 * Extend object a with the properties of object b
	 * @param {Object} a
	 * @param {Object} b
	 * @return {Object} a
	 */
	exports.extend = function extend(a, b) {
	  for (var prop in b) {
	    if (b.hasOwnProperty(prop)) {
	      a[prop] = b[prop];
	    }
	  }
	  return a;
	};

	/**
	 * Remove all properties from object a
	 * @param {Object} a
	 * @return {Object} a
	 */
	exports.clear = function clear (a) {
	  for (var prop in a) {
	    if (a.hasOwnProperty(prop)) {
	      delete a[prop];
	    }
	  }
	  return a;
	};

	/**
	 * Get the type of an object
	 * @param {*} object
	 * @return {String} type
	 */
	exports.type = function type (object) {
	  if (object === null) {
	    return 'null';
	  }
	  if (object === undefined) {
	    return 'undefined';
	  }
	  if ((object instanceof Number) || (typeof object === 'number')) {
	    return 'number';
	  }
	  if ((object instanceof String) || (typeof object === 'string')) {
	    return 'string';
	  }
	  if ((object instanceof Boolean) || (typeof object === 'boolean')) {
	    return 'boolean';
	  }
	  if ((object instanceof RegExp) || (typeof object === 'regexp')) {
	    return 'regexp';
	  }
	  if (exports.isArray(object)) {
	    return 'array';
	  }

	  return 'object';
	};

	/**
	 * Test whether a text contains a url (matches when a string starts
	 * with 'http://*' or 'https://*' and has no whitespace characters)
	 * @param {String} text
	 */
	var isUrlRegex = /^https?:\/\/\S+$/;
	exports.isUrl = function isUrl (text) {
	  return (typeof text == 'string' || text instanceof String) &&
	      isUrlRegex.test(text);
	};

	/**
	 * Tes whether given object is an Array
	 * @param {*} obj
	 * @returns {boolean} returns true when obj is an array
	 */
	exports.isArray = function (obj) {
	  return Object.prototype.toString.call(obj) === '[object Array]';
	};

	/**
	 * Retrieve the absolute left value of a DOM element
	 * @param {Element} elem    A dom element, for example a div
	 * @return {Number} left    The absolute left position of this element
	 *                          in the browser page.
	 */
	exports.getAbsoluteLeft = function getAbsoluteLeft(elem) {
	  var rect = elem.getBoundingClientRect();
	  return rect.left + window.pageXOffset || document.scrollLeft || 0;
	};

	/**
	 * Retrieve the absolute top value of a DOM element
	 * @param {Element} elem    A dom element, for example a div
	 * @return {Number} top     The absolute top position of this element
	 *                          in the browser page.
	 */
	exports.getAbsoluteTop = function getAbsoluteTop(elem) {
	  var rect = elem.getBoundingClientRect();
	  return rect.top + window.pageYOffset || document.scrollTop || 0;
	};

	/**
	 * add a className to the given elements style
	 * @param {Element} elem
	 * @param {String} className
	 */
	exports.addClassName = function addClassName(elem, className) {
	  var classes = elem.className.split(' ');
	  if (classes.indexOf(className) == -1) {
	    classes.push(className); // add the class to the array
	    elem.className = classes.join(' ');
	  }
	};

	/**
	 * add a className to the given elements style
	 * @param {Element} elem
	 * @param {String} className
	 */
	exports.removeClassName = function removeClassName(elem, className) {
	  var classes = elem.className.split(' ');
	  var index = classes.indexOf(className);
	  if (index != -1) {
	    classes.splice(index, 1); // remove the class from the array
	    elem.className = classes.join(' ');
	  }
	};

	/**
	 * Strip the formatting from the contents of a div
	 * the formatting from the div itself is not stripped, only from its childs.
	 * @param {Element} divElement
	 */
	exports.stripFormatting = function stripFormatting(divElement) {
	  var childs = divElement.childNodes;
	  for (var i = 0, iMax = childs.length; i < iMax; i++) {
	    var child = childs[i];

	    // remove the style
	    if (child.style) {
	      // TODO: test if child.attributes does contain style
	      child.removeAttribute('style');
	    }

	    // remove all attributes
	    var attributes = child.attributes;
	    if (attributes) {
	      for (var j = attributes.length - 1; j >= 0; j--) {
	        var attribute = attributes[j];
	        if (attribute.specified === true) {
	          child.removeAttribute(attribute.name);
	        }
	      }
	    }

	    // recursively strip childs
	    exports.stripFormatting(child);
	  }
	};

	/**
	 * Set focus to the end of an editable div
	 * code from Nico Burns
	 * http://stackoverflow.com/users/140293/nico-burns
	 * http://stackoverflow.com/questions/1125292/how-to-move-cursor-to-end-of-contenteditable-entity
	 * @param {Element} contentEditableElement   A content editable div
	 */
	exports.setEndOfContentEditable = function setEndOfContentEditable(contentEditableElement) {
	  var range, selection;
	  if(document.createRange) {
	    range = document.createRange();//Create a range (a range is a like the selection but invisible)
	    range.selectNodeContents(contentEditableElement);//Select the entire contents of the element with the range
	    range.collapse(false);//collapse the range to the end point. false means collapse to end rather than the start
	    selection = window.getSelection();//get the selection object (allows you to change selection)
	    selection.removeAllRanges();//remove any selections already made
	    selection.addRange(range);//make the range you have just created the visible selection
	  }
	};

	/**
	 * Select all text of a content editable div.
	 * http://stackoverflow.com/a/3806004/1262753
	 * @param {Element} contentEditableElement   A content editable div
	 */
	exports.selectContentEditable = function selectContentEditable(contentEditableElement) {
	  if (!contentEditableElement || contentEditableElement.nodeName != 'DIV') {
	    return;
	  }

	  var sel, range;
	  if (window.getSelection && document.createRange) {
	    range = document.createRange();
	    range.selectNodeContents(contentEditableElement);
	    sel = window.getSelection();
	    sel.removeAllRanges();
	    sel.addRange(range);
	  }
	};

	/**
	 * Get text selection
	 * http://stackoverflow.com/questions/4687808/contenteditable-selected-text-save-and-restore
	 * @return {Range | TextRange | null} range
	 */
	exports.getSelection = function getSelection() {
	  if (window.getSelection) {
	    var sel = window.getSelection();
	    if (sel.getRangeAt && sel.rangeCount) {
	      return sel.getRangeAt(0);
	    }
	  }
	  return null;
	};

	/**
	 * Set text selection
	 * http://stackoverflow.com/questions/4687808/contenteditable-selected-text-save-and-restore
	 * @param {Range | TextRange | null} range
	 */
	exports.setSelection = function setSelection(range) {
	  if (range) {
	    if (window.getSelection) {
	      var sel = window.getSelection();
	      sel.removeAllRanges();
	      sel.addRange(range);
	    }
	  }
	};

	/**
	 * Get selected text range
	 * @return {Object} params  object containing parameters:
	 *                              {Number}  startOffset
	 *                              {Number}  endOffset
	 *                              {Element} container  HTML element holding the
	 *                                                   selected text element
	 *                          Returns null if no text selection is found
	 */
	exports.getSelectionOffset = function getSelectionOffset() {
	  var range = exports.getSelection();

	  if (range && 'startOffset' in range && 'endOffset' in range &&
	      range.startContainer && (range.startContainer == range.endContainer)) {
	    return {
	      startOffset: range.startOffset,
	      endOffset: range.endOffset,
	      container: range.startContainer.parentNode
	    };
	  }

	  return null;
	};

	/**
	 * Set selected text range in given element
	 * @param {Object} params   An object containing:
	 *                              {Element} container
	 *                              {Number} startOffset
	 *                              {Number} endOffset
	 */
	exports.setSelectionOffset = function setSelectionOffset(params) {
	  if (document.createRange && window.getSelection) {
	    var selection = window.getSelection();
	    if(selection) {
	      var range = document.createRange();

	      if (!params.container.firstChild) {
	        params.container.appendChild(document.createTextNode(''));
	      }

	      // TODO: do not suppose that the first child of the container is a textnode,
	      //       but recursively find the textnodes
	      range.setStart(params.container.firstChild, params.startOffset);
	      range.setEnd(params.container.firstChild, params.endOffset);

	      exports.setSelection(range);
	    }
	  }
	};

	/**
	 * Get the inner text of an HTML element (for example a div element)
	 * @param {Element} element
	 * @param {Object} [buffer]
	 * @return {String} innerText
	 */
	exports.getInnerText = function getInnerText(element, buffer) {
	  var first = (buffer == undefined);
	  if (first) {
	    buffer = {
	      'text': '',
	      'flush': function () {
	        var text = this.text;
	        this.text = '';
	        return text;
	      },
	      'set': function (text) {
	        this.text = text;
	      }
	    };
	  }

	  // text node
	  if (element.nodeValue) {
	    return buffer.flush() + element.nodeValue;
	  }

	  // divs or other HTML elements
	  if (element.hasChildNodes()) {
	    var childNodes = element.childNodes;
	    var innerText = '';

	    for (var i = 0, iMax = childNodes.length; i < iMax; i++) {
	      var child = childNodes[i];

	      if (child.nodeName == 'DIV' || child.nodeName == 'P') {
	        var prevChild = childNodes[i - 1];
	        var prevName = prevChild ? prevChild.nodeName : undefined;
	        if (prevName && prevName != 'DIV' && prevName != 'P' && prevName != 'BR') {
	          innerText += '\n';
	          buffer.flush();
	        }
	        innerText += exports.getInnerText(child, buffer);
	        buffer.set('\n');
	      }
	      else if (child.nodeName == 'BR') {
	        innerText += buffer.flush();
	        buffer.set('\n');
	      }
	      else {
	        innerText += exports.getInnerText(child, buffer);
	      }
	    }

	    return innerText;
	  }
	  else {
	    if (element.nodeName == 'P' && exports.getInternetExplorerVersion() != -1) {
	      // On Internet Explorer, a <p> with hasChildNodes()==false is
	      // rendered with a new line. Note that a <p> with
	      // hasChildNodes()==true is rendered without a new line
	      // Other browsers always ensure there is a <br> inside the <p>,
	      // and if not, the <p> does not render a new line
	      return buffer.flush();
	    }
	  }

	  // br or unknown
	  return '';
	};

	/**
	 * Returns the version of Internet Explorer or a -1
	 * (indicating the use of another browser).
	 * Source: http://msdn.microsoft.com/en-us/library/ms537509(v=vs.85).aspx
	 * @return {Number} Internet Explorer version, or -1 in case of an other browser
	 */
	exports.getInternetExplorerVersion = function getInternetExplorerVersion() {
	  if (_ieVersion == -1) {
	    var rv = -1; // Return value assumes failure.
	    if (navigator.appName == 'Microsoft Internet Explorer')
	    {
	      var ua = navigator.userAgent;
	      var re  = new RegExp("MSIE ([0-9]{1,}[\.0-9]{0,})");
	      if (re.exec(ua) != null) {
	        rv = parseFloat( RegExp.$1 );
	      }
	    }

	    _ieVersion = rv;
	  }

	  return _ieVersion;
	};

	/**
	 * Test whether the current browser is Firefox
	 * @returns {boolean} isFirefox
	 */
	exports.isFirefox = function isFirefox () {
	  return (navigator.userAgent.indexOf("Firefox") != -1);
	};

	/**
	 * cached internet explorer version
	 * @type {Number}
	 * @private
	 */
	var _ieVersion = -1;

	/**
	 * Add and event listener. Works for all browsers
	 * @param {Element}     element    An html element
	 * @param {string}      action     The action, for example "click",
	 *                                 without the prefix "on"
	 * @param {function}    listener   The callback function to be executed
	 * @param {boolean}     [useCapture] false by default
	 * @return {function}   the created event listener
	 */
	exports.addEventListener = function addEventListener(element, action, listener, useCapture) {
	  if (element.addEventListener) {
	    if (useCapture === undefined)
	      useCapture = false;

	    if (action === "mousewheel" && exports.isFirefox()) {
	      action = "DOMMouseScroll";  // For Firefox
	    }

	    element.addEventListener(action, listener, useCapture);
	    return listener;
	  } else if (element.attachEvent) {
	    // Old IE browsers
	    var f = function () {
	      return listener.call(element, window.event);
	    };
	    element.attachEvent("on" + action, f);
	    return f;
	  }
	};

	/**
	 * Remove an event listener from an element
	 * @param {Element}  element   An html dom element
	 * @param {string}   action    The name of the event, for example "mousedown"
	 * @param {function} listener  The listener function
	 * @param {boolean}  [useCapture]   false by default
	 */
	exports.removeEventListener = function removeEventListener(element, action, listener, useCapture) {
	  if (element.removeEventListener) {
	    if (useCapture === undefined)
	      useCapture = false;

	    if (action === "mousewheel" && exports.isFirefox()) {
	      action = "DOMMouseScroll";  // For Firefox
	    }

	    element.removeEventListener(action, listener, useCapture);
	  } else if (element.detachEvent) {
	    // Old IE browsers
	    element.detachEvent("on" + action, listener);
	  }
	};

	/**
	 * Parse a JSON path like '.items[3].name' into an array
	 * @param {string} jsonPath
	 * @return {Array}
	 */
	exports.parsePath = function parsePath(jsonPath) {
	  var prop, remainder;

	  if (jsonPath.length === 0) {
	    return [];
	  }

	  // find a match like '.prop'
	  var match = jsonPath.match(/^\.(\w+)/);
	  if (match) {
	    prop = match[1];
	    remainder = jsonPath.substr(prop.length + 1);
	  }
	  else if (jsonPath[0] === '[') {
	    // find a match like
	    var end = jsonPath.indexOf(']');
	    if (end === -1) {
	      throw new SyntaxError('Character ] expected in path');
	    }
	    if (end === 1) {
	      throw new SyntaxError('Index expected after [');
	    }

	    var value = jsonPath.substring(1, end);
	    if (value[0] === '\'') {
	      // ajv produces string prop names with single quotes, so we need
	      // to reformat them into valid double-quoted JSON strings
	      value = '\"' + value.substring(1, value.length - 1) + '\"';
	    }

	    prop = value === '*' ? value : JSON.parse(value); // parse string and number
	    remainder = jsonPath.substr(end + 1);
	  }
	  else {
	    throw new SyntaxError('Failed to parse path');
	  }

	  return [prop].concat(parsePath(remainder))
	};

	/**
	 * Improve the error message of a JSON schema error
	 * @param {Object} error
	 * @return {Object} The error
	 */
	exports.improveSchemaError = function (error) {
	  if (error.keyword === 'enum' && Array.isArray(error.schema)) {
	    var enums = error.schema;
	    if (enums) {
	      enums = enums.map(function (value) {
	        return JSON.stringify(value);
	      });

	      if (enums.length > 5) {
	        var more = ['(' + (enums.length - 5) + ' more...)'];
	        enums = enums.slice(0, 5);
	        enums.push(more);
	      }
	      error.message = 'should be equal to one of: ' + enums.join(', ');
	    }
	  }

	  if (error.keyword === 'additionalProperties') {
	    error.message = 'should NOT have additional property: ' + error.params.additionalProperty;
	  }

	  return error;
	};

	/**
	 * Test whether the child rect fits completely inside the parent rect.
	 * @param {ClientRect} parent
	 * @param {ClientRect} child
	 * @param {number} margin
	 */
	exports.insideRect = function (parent, child, margin) {
	  var _margin = margin !== undefined ? margin : 0;
	  return child.left   - _margin >= parent.left
	      && child.right  + _margin <= parent.right
	      && child.top    - _margin >= parent.top
	      && child.bottom + _margin <= parent.bottom;
	};

	/**
	 * Returns a function, that, as long as it continues to be invoked, will not
	 * be triggered. The function will be called after it stops being called for
	 * N milliseconds.
	 *
	 * Source: https://davidwalsh.name/javascript-debounce-function
	 *
	 * @param {function} func
	 * @param {number} wait                 Number in milliseconds
	 * @param {boolean} [immediate=false]   If `immediate` is passed, trigger the
	 *                                      function on the leading edge, instead
	 *                                      of the trailing.
	 * @return {function} Return the debounced function
	 */
	exports.debounce = function debounce(func, wait, immediate) {
	  var timeout;
	  return function() {
	    var context = this, args = arguments;
	    var later = function() {
	      timeout = null;
	      if (!immediate) func.apply(context, args);
	    };
	    var callNow = immediate && !timeout;
	    clearTimeout(timeout);
	    timeout = setTimeout(later, wait);
	    if (callNow) func.apply(context, args);
	  };
	};

	/**
	 * Determines the difference between two texts.
	 * Can only detect one removed or inserted block of characters.
	 * @param {string} oldText
	 * @param {string} newText
	 * @return {{start: number, end: number}} Returns the start and end
	 *                                        of the changed part in newText.
	 */
	exports.textDiff = function textDiff(oldText, newText) {
	  var len = newText.length;
	  var start = 0;
	  var oldEnd = oldText.length;
	  var newEnd = newText.length;

	  while (newText.charAt(start) === oldText.charAt(start)
	  && start < len) {
	    start++;
	  }

	  while (newText.charAt(newEnd - 1) === oldText.charAt(oldEnd - 1)
	  && newEnd > start && oldEnd > 0) {
	    newEnd--;
	    oldEnd--;
	  }

	  return {start: start, end: newEnd};
	};


	/**
	 * Return an object with the selection range or cursor position (if both have the same value)
	 * Support also old browsers (IE8-)
	 * Source: http://ourcodeworld.com/articles/read/282/how-to-get-the-current-cursor-position-and-selection-within-a-text-input-or-textarea-in-javascript
	 * @param {DOMElement} el A dom element of a textarea or input text.
	 * @return {Object} reference Object with 2 properties (start and end) with the identifier of the location of the cursor and selected text.
	 **/
	exports.getInputSelection = function(el) {
	  var start = 0, end = 0, normalizedValue, range, textInputRange, len, endRange;

	  if (typeof el.selectionStart == "number" && typeof el.selectionEnd == "number") {
	      start = el.selectionStart;
	      end = el.selectionEnd;
	  } else {
	      range = document.selection.createRange();

	      if (range && range.parentElement() == el) {
	          len = el.value.length;
	          normalizedValue = el.value.replace(/\r\n/g, "\n");

	          // Create a working TextRange that lives only in the input
	          textInputRange = el.createTextRange();
	          textInputRange.moveToBookmark(range.getBookmark());

	          // Check if the start and end of the selection are at the very end
	          // of the input, since moveStart/moveEnd doesn't return what we want
	          // in those cases
	          endRange = el.createTextRange();
	          endRange.collapse(false);

	          if (textInputRange.compareEndPoints("StartToEnd", endRange) > -1) {
	              start = end = len;
	          } else {
	              start = -textInputRange.moveStart("character", -len);
	              start += normalizedValue.slice(0, start).split("\n").length - 1;

	              if (textInputRange.compareEndPoints("EndToEnd", endRange) > -1) {
	                  end = len;
	              } else {
	                  end = -textInputRange.moveEnd("character", -len);
	                  end += normalizedValue.slice(0, end).split("\n").length - 1;
	              }
	          }
	      }
	  }

	  var textTillCaret = el.value.substring(0,end);
	  var row = (textTillCaret.match(/\n/g) || []).length + 1;
	  var col = textTillCaret.length - textTillCaret.lastIndexOf("\n");

	  return {
	      start: start,
	      end: end,
	      col: col,
	      row: row
	  };
	}


	if (typeof Element !== 'undefined') {
	  // Polyfill for array remove
	  (function () {
	    function polyfill (item) {
	      if (item.hasOwnProperty('remove')) {
	        return;
	      }
	      Object.defineProperty(item, 'remove', {
	        configurable: true,
	        enumerable: true,
	        writable: true,
	        value: function remove() {
	          if (this.parentNode != null)
	            this.parentNode.removeChild(this);
	        }
	      });
	    }

	    if (typeof Element !== 'undefined')       { polyfill(Element.prototype); }
	    if (typeof CharacterData !== 'undefined') { polyfill(CharacterData.prototype); }
	    if (typeof DocumentType !== 'undefined')  { polyfill(DocumentType.prototype); }
	  })();
	}


	// Polyfill for startsWith
	if (!String.prototype.startsWith) {
	    String.prototype.startsWith = function (searchString, position) {
	        position = position || 0;
	        return this.substr(position, searchString.length) === searchString;
	    };
	}

	// Polyfill for Array.find
	if (!Array.prototype.find) {
	  Array.prototype.find = function(callback) {    
	    for (var i = 0; i < this.length; i++) {
	      var element = this[i];
	      if ( callback.call(this, element, i, this) ) {
	        return element;
	      }
	    }
	  }
	}

/***/ },
/* 55 */
/***/ function(module, exports, __webpack_require__) {

	/* Jison generated parser */
	var jsonlint = (function(){
	var parser = {trace: function trace() { },
	yy: {},
	symbols_: {"error":2,"JSONString":3,"STRING":4,"JSONNumber":5,"NUMBER":6,"JSONNullLiteral":7,"NULL":8,"JSONBooleanLiteral":9,"TRUE":10,"FALSE":11,"JSONText":12,"JSONValue":13,"EOF":14,"JSONObject":15,"JSONArray":16,"{":17,"}":18,"JSONMemberList":19,"JSONMember":20,":":21,",":22,"[":23,"]":24,"JSONElementList":25,"$accept":0,"$end":1},
	terminals_: {2:"error",4:"STRING",6:"NUMBER",8:"NULL",10:"TRUE",11:"FALSE",14:"EOF",17:"{",18:"}",21:":",22:",",23:"[",24:"]"},
	productions_: [0,[3,1],[5,1],[7,1],[9,1],[9,1],[12,2],[13,1],[13,1],[13,1],[13,1],[13,1],[13,1],[15,2],[15,3],[20,3],[19,1],[19,3],[16,2],[16,3],[25,1],[25,3]],
	performAction: function anonymous(yytext,yyleng,yylineno,yy,yystate,$$,_$) {

	var $0 = $$.length - 1;
	switch (yystate) {
	case 1: // replace escaped characters with actual character
	          this.$ = yytext.replace(/\\(\\|")/g, "$"+"1")
	                     .replace(/\\n/g,'\n')
	                     .replace(/\\r/g,'\r')
	                     .replace(/\\t/g,'\t')
	                     .replace(/\\v/g,'\v')
	                     .replace(/\\f/g,'\f')
	                     .replace(/\\b/g,'\b');
	        
	break;
	case 2:this.$ = Number(yytext);
	break;
	case 3:this.$ = null;
	break;
	case 4:this.$ = true;
	break;
	case 5:this.$ = false;
	break;
	case 6:return this.$ = $$[$0-1];
	break;
	case 13:this.$ = {};
	break;
	case 14:this.$ = $$[$0-1];
	break;
	case 15:this.$ = [$$[$0-2], $$[$0]];
	break;
	case 16:this.$ = {}; this.$[$$[$0][0]] = $$[$0][1];
	break;
	case 17:this.$ = $$[$0-2]; $$[$0-2][$$[$0][0]] = $$[$0][1];
	break;
	case 18:this.$ = [];
	break;
	case 19:this.$ = $$[$0-1];
	break;
	case 20:this.$ = [$$[$0]];
	break;
	case 21:this.$ = $$[$0-2]; $$[$0-2].push($$[$0]);
	break;
	}
	},
	table: [{3:5,4:[1,12],5:6,6:[1,13],7:3,8:[1,9],9:4,10:[1,10],11:[1,11],12:1,13:2,15:7,16:8,17:[1,14],23:[1,15]},{1:[3]},{14:[1,16]},{14:[2,7],18:[2,7],22:[2,7],24:[2,7]},{14:[2,8],18:[2,8],22:[2,8],24:[2,8]},{14:[2,9],18:[2,9],22:[2,9],24:[2,9]},{14:[2,10],18:[2,10],22:[2,10],24:[2,10]},{14:[2,11],18:[2,11],22:[2,11],24:[2,11]},{14:[2,12],18:[2,12],22:[2,12],24:[2,12]},{14:[2,3],18:[2,3],22:[2,3],24:[2,3]},{14:[2,4],18:[2,4],22:[2,4],24:[2,4]},{14:[2,5],18:[2,5],22:[2,5],24:[2,5]},{14:[2,1],18:[2,1],21:[2,1],22:[2,1],24:[2,1]},{14:[2,2],18:[2,2],22:[2,2],24:[2,2]},{3:20,4:[1,12],18:[1,17],19:18,20:19},{3:5,4:[1,12],5:6,6:[1,13],7:3,8:[1,9],9:4,10:[1,10],11:[1,11],13:23,15:7,16:8,17:[1,14],23:[1,15],24:[1,21],25:22},{1:[2,6]},{14:[2,13],18:[2,13],22:[2,13],24:[2,13]},{18:[1,24],22:[1,25]},{18:[2,16],22:[2,16]},{21:[1,26]},{14:[2,18],18:[2,18],22:[2,18],24:[2,18]},{22:[1,28],24:[1,27]},{22:[2,20],24:[2,20]},{14:[2,14],18:[2,14],22:[2,14],24:[2,14]},{3:20,4:[1,12],20:29},{3:5,4:[1,12],5:6,6:[1,13],7:3,8:[1,9],9:4,10:[1,10],11:[1,11],13:30,15:7,16:8,17:[1,14],23:[1,15]},{14:[2,19],18:[2,19],22:[2,19],24:[2,19]},{3:5,4:[1,12],5:6,6:[1,13],7:3,8:[1,9],9:4,10:[1,10],11:[1,11],13:31,15:7,16:8,17:[1,14],23:[1,15]},{18:[2,17],22:[2,17]},{18:[2,15],22:[2,15]},{22:[2,21],24:[2,21]}],
	defaultActions: {16:[2,6]},
	parseError: function parseError(str, hash) {
	    throw new Error(str);
	},
	parse: function parse(input) {
	    var self = this,
	        stack = [0],
	        vstack = [null], // semantic value stack
	        lstack = [], // location stack
	        table = this.table,
	        yytext = '',
	        yylineno = 0,
	        yyleng = 0,
	        recovering = 0,
	        TERROR = 2,
	        EOF = 1;

	    //this.reductionCount = this.shiftCount = 0;

	    this.lexer.setInput(input);
	    this.lexer.yy = this.yy;
	    this.yy.lexer = this.lexer;
	    if (typeof this.lexer.yylloc == 'undefined')
	        this.lexer.yylloc = {};
	    var yyloc = this.lexer.yylloc;
	    lstack.push(yyloc);

	    if (typeof this.yy.parseError === 'function')
	        this.parseError = this.yy.parseError;

	    function popStack (n) {
	        stack.length = stack.length - 2*n;
	        vstack.length = vstack.length - n;
	        lstack.length = lstack.length - n;
	    }

	    function lex() {
	        var token;
	        token = self.lexer.lex() || 1; // $end = 1
	        // if token isn't its numeric value, convert
	        if (typeof token !== 'number') {
	            token = self.symbols_[token] || token;
	        }
	        return token;
	    }

	    var symbol, preErrorSymbol, state, action, a, r, yyval={},p,len,newState, expected;
	    while (true) {
	        // retreive state number from top of stack
	        state = stack[stack.length-1];

	        // use default actions if available
	        if (this.defaultActions[state]) {
	            action = this.defaultActions[state];
	        } else {
	            if (symbol == null)
	                symbol = lex();
	            // read action for current state and first input
	            action = table[state] && table[state][symbol];
	        }

	        // handle parse error
	        _handle_error:
	        if (typeof action === 'undefined' || !action.length || !action[0]) {

	            if (!recovering) {
	                // Report error
	                expected = [];
	                for (p in table[state]) if (this.terminals_[p] && p > 2) {
	                    expected.push("'"+this.terminals_[p]+"'");
	                }
	                var errStr = '';
	                if (this.lexer.showPosition) {
	                    errStr = 'Parse error on line '+(yylineno+1)+":\n"+this.lexer.showPosition()+"\nExpecting "+expected.join(', ') + ", got '" + this.terminals_[symbol]+ "'";
	                } else {
	                    errStr = 'Parse error on line '+(yylineno+1)+": Unexpected " +
	                                  (symbol == 1 /*EOF*/ ? "end of input" :
	                                              ("'"+(this.terminals_[symbol] || symbol)+"'"));
	                }
	                this.parseError(errStr,
	                    {text: this.lexer.match, token: this.terminals_[symbol] || symbol, line: this.lexer.yylineno, loc: yyloc, expected: expected});
	            }

	            // just recovered from another error
	            if (recovering == 3) {
	                if (symbol == EOF) {
	                    throw new Error(errStr || 'Parsing halted.');
	                }

	                // discard current lookahead and grab another
	                yyleng = this.lexer.yyleng;
	                yytext = this.lexer.yytext;
	                yylineno = this.lexer.yylineno;
	                yyloc = this.lexer.yylloc;
	                symbol = lex();
	            }

	            // try to recover from error
	            while (1) {
	                // check for error recovery rule in this state
	                if ((TERROR.toString()) in table[state]) {
	                    break;
	                }
	                if (state == 0) {
	                    throw new Error(errStr || 'Parsing halted.');
	                }
	                popStack(1);
	                state = stack[stack.length-1];
	            }

	            preErrorSymbol = symbol; // save the lookahead token
	            symbol = TERROR;         // insert generic error symbol as new lookahead
	            state = stack[stack.length-1];
	            action = table[state] && table[state][TERROR];
	            recovering = 3; // allow 3 real symbols to be shifted before reporting a new error
	        }

	        // this shouldn't happen, unless resolve defaults are off
	        if (action[0] instanceof Array && action.length > 1) {
	            throw new Error('Parse Error: multiple actions possible at state: '+state+', token: '+symbol);
	        }

	        switch (action[0]) {

	            case 1: // shift
	                //this.shiftCount++;

	                stack.push(symbol);
	                vstack.push(this.lexer.yytext);
	                lstack.push(this.lexer.yylloc);
	                stack.push(action[1]); // push state
	                symbol = null;
	                if (!preErrorSymbol) { // normal execution/no error
	                    yyleng = this.lexer.yyleng;
	                    yytext = this.lexer.yytext;
	                    yylineno = this.lexer.yylineno;
	                    yyloc = this.lexer.yylloc;
	                    if (recovering > 0)
	                        recovering--;
	                } else { // error just occurred, resume old lookahead f/ before error
	                    symbol = preErrorSymbol;
	                    preErrorSymbol = null;
	                }
	                break;

	            case 2: // reduce
	                //this.reductionCount++;

	                len = this.productions_[action[1]][1];

	                // perform semantic action
	                yyval.$ = vstack[vstack.length-len]; // default to $$ = $1
	                // default location, uses first token for firsts, last for lasts
	                yyval._$ = {
	                    first_line: lstack[lstack.length-(len||1)].first_line,
	                    last_line: lstack[lstack.length-1].last_line,
	                    first_column: lstack[lstack.length-(len||1)].first_column,
	                    last_column: lstack[lstack.length-1].last_column
	                };
	                r = this.performAction.call(yyval, yytext, yyleng, yylineno, this.yy, action[1], vstack, lstack);

	                if (typeof r !== 'undefined') {
	                    return r;
	                }

	                // pop off stack
	                if (len) {
	                    stack = stack.slice(0,-1*len*2);
	                    vstack = vstack.slice(0, -1*len);
	                    lstack = lstack.slice(0, -1*len);
	                }

	                stack.push(this.productions_[action[1]][0]);    // push nonterminal (reduce)
	                vstack.push(yyval.$);
	                lstack.push(yyval._$);
	                // goto new state = table[STATE][NONTERMINAL]
	                newState = table[stack[stack.length-2]][stack[stack.length-1]];
	                stack.push(newState);
	                break;

	            case 3: // accept
	                return true;
	        }

	    }

	    return true;
	}};
	/* Jison generated lexer */
	var lexer = (function(){
	var lexer = ({EOF:1,
	parseError:function parseError(str, hash) {
	        if (this.yy.parseError) {
	            this.yy.parseError(str, hash);
	        } else {
	            throw new Error(str);
	        }
	    },
	setInput:function (input) {
	        this._input = input;
	        this._more = this._less = this.done = false;
	        this.yylineno = this.yyleng = 0;
	        this.yytext = this.matched = this.match = '';
	        this.conditionStack = ['INITIAL'];
	        this.yylloc = {first_line:1,first_column:0,last_line:1,last_column:0};
	        return this;
	    },
	input:function () {
	        var ch = this._input[0];
	        this.yytext+=ch;
	        this.yyleng++;
	        this.match+=ch;
	        this.matched+=ch;
	        var lines = ch.match(/\n/);
	        if (lines) this.yylineno++;
	        this._input = this._input.slice(1);
	        return ch;
	    },
	unput:function (ch) {
	        this._input = ch + this._input;
	        return this;
	    },
	more:function () {
	        this._more = true;
	        return this;
	    },
	less:function (n) {
	        this._input = this.match.slice(n) + this._input;
	    },
	pastInput:function () {
	        var past = this.matched.substr(0, this.matched.length - this.match.length);
	        return (past.length > 20 ? '...':'') + past.substr(-20).replace(/\n/g, "");
	    },
	upcomingInput:function () {
	        var next = this.match;
	        if (next.length < 20) {
	            next += this._input.substr(0, 20-next.length);
	        }
	        return (next.substr(0,20)+(next.length > 20 ? '...':'')).replace(/\n/g, "");
	    },
	showPosition:function () {
	        var pre = this.pastInput();
	        var c = new Array(pre.length + 1).join("-");
	        return pre + this.upcomingInput() + "\n" + c+"^";
	    },
	next:function () {
	        if (this.done) {
	            return this.EOF;
	        }
	        if (!this._input) this.done = true;

	        var token,
	            match,
	            tempMatch,
	            index,
	            col,
	            lines;
	        if (!this._more) {
	            this.yytext = '';
	            this.match = '';
	        }
	        var rules = this._currentRules();
	        for (var i=0;i < rules.length; i++) {
	            tempMatch = this._input.match(this.rules[rules[i]]);
	            if (tempMatch && (!match || tempMatch[0].length > match[0].length)) {
	                match = tempMatch;
	                index = i;
	                if (!this.options.flex) break;
	            }
	        }
	        if (match) {
	            lines = match[0].match(/\n.*/g);
	            if (lines) this.yylineno += lines.length;
	            this.yylloc = {first_line: this.yylloc.last_line,
	                           last_line: this.yylineno+1,
	                           first_column: this.yylloc.last_column,
	                           last_column: lines ? lines[lines.length-1].length-1 : this.yylloc.last_column + match[0].length}
	            this.yytext += match[0];
	            this.match += match[0];
	            this.yyleng = this.yytext.length;
	            this._more = false;
	            this._input = this._input.slice(match[0].length);
	            this.matched += match[0];
	            token = this.performAction.call(this, this.yy, this, rules[index],this.conditionStack[this.conditionStack.length-1]);
	            if (this.done && this._input) this.done = false;
	            if (token) return token;
	            else return;
	        }
	        if (this._input === "") {
	            return this.EOF;
	        } else {
	            this.parseError('Lexical error on line '+(this.yylineno+1)+'. Unrecognized text.\n'+this.showPosition(), 
	                    {text: "", token: null, line: this.yylineno});
	        }
	    },
	lex:function lex() {
	        var r = this.next();
	        if (typeof r !== 'undefined') {
	            return r;
	        } else {
	            return this.lex();
	        }
	    },
	begin:function begin(condition) {
	        this.conditionStack.push(condition);
	    },
	popState:function popState() {
	        return this.conditionStack.pop();
	    },
	_currentRules:function _currentRules() {
	        return this.conditions[this.conditionStack[this.conditionStack.length-1]].rules;
	    },
	topState:function () {
	        return this.conditionStack[this.conditionStack.length-2];
	    },
	pushState:function begin(condition) {
	        this.begin(condition);
	    }});
	lexer.options = {};
	lexer.performAction = function anonymous(yy,yy_,$avoiding_name_collisions,YY_START) {

	var YYSTATE=YY_START
	switch($avoiding_name_collisions) {
	case 0:/* skip whitespace */
	break;
	case 1:return 6
	break;
	case 2:yy_.yytext = yy_.yytext.substr(1,yy_.yyleng-2); return 4
	break;
	case 3:return 17
	break;
	case 4:return 18
	break;
	case 5:return 23
	break;
	case 6:return 24
	break;
	case 7:return 22
	break;
	case 8:return 21
	break;
	case 9:return 10
	break;
	case 10:return 11
	break;
	case 11:return 8
	break;
	case 12:return 14
	break;
	case 13:return 'INVALID'
	break;
	}
	};
	lexer.rules = [/^(?:\s+)/,/^(?:(-?([0-9]|[1-9][0-9]+))(\.[0-9]+)?([eE][-+]?[0-9]+)?\b)/,/^(?:"(?:\\[\\"bfnrt/]|\\u[a-fA-F0-9]{4}|[^\\\0-\x09\x0a-\x1f"])*")/,/^(?:\{)/,/^(?:\})/,/^(?:\[)/,/^(?:\])/,/^(?:,)/,/^(?::)/,/^(?:true\b)/,/^(?:false\b)/,/^(?:null\b)/,/^(?:$)/,/^(?:.)/];
	lexer.conditions = {"INITIAL":{"rules":[0,1,2,3,4,5,6,7,8,9,10,11,12,13],"inclusive":true}};


	;
	return lexer;})()
	parser.lexer = lexer;
	return parser;
	})();
	if (true) {
	  exports.parser = jsonlint;
	  exports.parse = jsonlint.parse.bind(jsonlint);
	}

/***/ },
/* 56 */
/***/ function(module, exports) {

	'use strict';

	/**
	 * @constructor SearchBox
	 * Create a search box in given HTML container
	 * @param {JSONEditor} editor    The JSON Editor to attach to
	 * @param {Element} container               HTML container element of where to
	 *                                          create the search box
	 */
	function SearchBox (editor, container) {
	  var searchBox = this;

	  this.editor = editor;
	  this.timeout = undefined;
	  this.delay = 200; // ms
	  this.lastText = undefined;

	  this.dom = {};
	  this.dom.container = container;

	  var table = document.createElement('table');
	  this.dom.table = table;
	  table.className = 'jsoneditor-search';
	  container.appendChild(table);
	  var tbody = document.createElement('tbody');
	  this.dom.tbody = tbody;
	  table.appendChild(tbody);
	  var tr = document.createElement('tr');
	  tbody.appendChild(tr);

	  var td = document.createElement('td');
	  tr.appendChild(td);
	  var results = document.createElement('div');
	  this.dom.results = results;
	  results.className = 'jsoneditor-results';
	  td.appendChild(results);

	  td = document.createElement('td');
	  tr.appendChild(td);
	  var divInput = document.createElement('div');
	  this.dom.input = divInput;
	  divInput.className = 'jsoneditor-frame';
	  divInput.title = 'Search fields and values';
	  td.appendChild(divInput);

	  // table to contain the text input and search button
	  var tableInput = document.createElement('table');
	  divInput.appendChild(tableInput);
	  var tbodySearch = document.createElement('tbody');
	  tableInput.appendChild(tbodySearch);
	  tr = document.createElement('tr');
	  tbodySearch.appendChild(tr);

	  var refreshSearch = document.createElement('button');
	  refreshSearch.type = 'button';
	  refreshSearch.className = 'jsoneditor-refresh';
	  td = document.createElement('td');
	  td.appendChild(refreshSearch);
	  tr.appendChild(td);

	  var search = document.createElement('input');
	  // search.type = 'button';
	  this.dom.search = search;
	  search.oninput = function (event) {
	    searchBox._onDelayedSearch(event);
	  };
	  search.onchange = function (event) { // For IE 9
	    searchBox._onSearch();
	  };
	  search.onkeydown = function (event) {
	    searchBox._onKeyDown(event);
	  };
	  search.onkeyup = function (event) {
	    searchBox._onKeyUp(event);
	  };
	  refreshSearch.onclick = function (event) {
	    search.select();
	  };

	  // TODO: ESC in FF restores the last input, is a FF bug, https://bugzilla.mozilla.org/show_bug.cgi?id=598819
	  td = document.createElement('td');
	  td.appendChild(search);
	  tr.appendChild(td);

	  var searchNext = document.createElement('button');
	  searchNext.type = 'button';
	  searchNext.title = 'Next result (Enter)';
	  searchNext.className = 'jsoneditor-next';
	  searchNext.onclick = function () {
	    searchBox.next();
	  };
	  td = document.createElement('td');
	  td.appendChild(searchNext);
	  tr.appendChild(td);

	  var searchPrevious = document.createElement('button');
	  searchPrevious.type = 'button';
	  searchPrevious.title = 'Previous result (Shift+Enter)';
	  searchPrevious.className = 'jsoneditor-previous';
	  searchPrevious.onclick = function () {
	    searchBox.previous();
	  };
	  td = document.createElement('td');
	  td.appendChild(searchPrevious);
	  tr.appendChild(td);
	}

	/**
	 * Go to the next search result
	 * @param {boolean} [focus]   If true, focus will be set to the next result
	 *                            focus is false by default.
	 */
	SearchBox.prototype.next = function(focus) {
	  if (this.results != undefined) {
	    var index = (this.resultIndex != undefined) ? this.resultIndex + 1 : 0;
	    if (index > this.results.length - 1) {
	      index = 0;
	    }
	    this._setActiveResult(index, focus);
	  }
	};

	/**
	 * Go to the prevous search result
	 * @param {boolean} [focus]   If true, focus will be set to the next result
	 *                            focus is false by default.
	 */
	SearchBox.prototype.previous = function(focus) {
	  if (this.results != undefined) {
	    var max = this.results.length - 1;
	    var index = (this.resultIndex != undefined) ? this.resultIndex - 1 : max;
	    if (index < 0) {
	      index = max;
	    }
	    this._setActiveResult(index, focus);
	  }
	};

	/**
	 * Set new value for the current active result
	 * @param {Number} index
	 * @param {boolean} [focus]   If true, focus will be set to the next result.
	 *                            focus is false by default.
	 * @private
	 */
	SearchBox.prototype._setActiveResult = function(index, focus) {
	  // de-activate current active result
	  if (this.activeResult) {
	    var prevNode = this.activeResult.node;
	    var prevElem = this.activeResult.elem;
	    if (prevElem == 'field') {
	      delete prevNode.searchFieldActive;
	    }
	    else {
	      delete prevNode.searchValueActive;
	    }
	    prevNode.updateDom();
	  }

	  if (!this.results || !this.results[index]) {
	    // out of range, set to undefined
	    this.resultIndex = undefined;
	    this.activeResult = undefined;
	    return;
	  }

	  this.resultIndex = index;

	  // set new node active
	  var node = this.results[this.resultIndex].node;
	  var elem = this.results[this.resultIndex].elem;
	  if (elem == 'field') {
	    node.searchFieldActive = true;
	  }
	  else {
	    node.searchValueActive = true;
	  }
	  this.activeResult = this.results[this.resultIndex];
	  node.updateDom();

	  // TODO: not so nice that the focus is only set after the animation is finished
	  node.scrollTo(function () {
	    if (focus) {
	      node.focus(elem);
	    }
	  });
	};

	/**
	 * Cancel any running onDelayedSearch.
	 * @private
	 */
	SearchBox.prototype._clearDelay = function() {
	  if (this.timeout != undefined) {
	    clearTimeout(this.timeout);
	    delete this.timeout;
	  }
	};

	/**
	 * Start a timer to execute a search after a short delay.
	 * Used for reducing the number of searches while typing.
	 * @param {Event} event
	 * @private
	 */
	SearchBox.prototype._onDelayedSearch = function (event) {
	  // execute the search after a short delay (reduces the number of
	  // search actions while typing in the search text box)
	  this._clearDelay();
	  var searchBox = this;
	  this.timeout = setTimeout(function (event) {
	    searchBox._onSearch();
	  },
	  this.delay);
	};

	/**
	 * Handle onSearch event
	 * @param {boolean} [forceSearch]  If true, search will be executed again even
	 *                                 when the search text is not changed.
	 *                                 Default is false.
	 * @private
	 */
	SearchBox.prototype._onSearch = function (forceSearch) {
	  this._clearDelay();

	  var value = this.dom.search.value;
	  var text = (value.length > 0) ? value : undefined;
	  if (text != this.lastText || forceSearch) {
	    // only search again when changed
	    this.lastText = text;
	    this.results = this.editor.search(text);
	    this._setActiveResult(undefined);

	    // display search results
	    if (text != undefined) {
	      var resultCount = this.results.length;
	      switch (resultCount) {
	        case 0: this.dom.results.innerHTML = 'no&nbsp;results'; break;
	        case 1: this.dom.results.innerHTML = '1&nbsp;result'; break;
	        default: this.dom.results.innerHTML = resultCount + '&nbsp;results'; break;
	      }
	    }
	    else {
	      this.dom.results.innerHTML = '';
	    }
	  }
	};

	/**
	 * Handle onKeyDown event in the input box
	 * @param {Event} event
	 * @private
	 */
	SearchBox.prototype._onKeyDown = function (event) {
	  var keynum = event.which;
	  if (keynum == 27) { // ESC
	    this.dom.search.value = '';  // clear search
	    this._onSearch();
	    event.preventDefault();
	    event.stopPropagation();
	  }
	  else if (keynum == 13) { // Enter
	    if (event.ctrlKey) {
	      // force to search again
	      this._onSearch(true);
	    }
	    else if (event.shiftKey) {
	      // move to the previous search result
	      this.previous();
	    }
	    else {
	      // move to the next search result
	      this.next();
	    }
	    event.preventDefault();
	    event.stopPropagation();
	  }
	};

	/**
	 * Handle onKeyUp event in the input box
	 * @param {Event} event
	 * @private
	 */
	SearchBox.prototype._onKeyUp = function (event) {
	  var keynum = event.keyCode;
	  if (keynum != 27 && keynum != 13) { // !show and !Enter
	    this._onDelayedSearch(event);   // For IE 9
	  }
	};

	/**
	 * Clear the search results
	 */
	SearchBox.prototype.clear = function () {
	  this.dom.search.value = '';
	  this._onSearch();
	};

	/**
	 * Destroy the search box
	 */
	SearchBox.prototype.destroy = function () {
	  this.editor = null;
	  this.dom.container.removeChild(this.dom.table);
	  this.dom = null;

	  this.results = null;
	  this.activeResult = null;

	  this._clearDelay();

	};

	module.exports = SearchBox;


/***/ },
/* 57 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var util = __webpack_require__(54);
	var translate = __webpack_require__(58).translate;

	/**
	 * Node.getRootNode shim
	 * @param  {Node} node node to check
	 * @return {Node}      node's rootNode or `window` if there is ShadowDOM is not supported.
	 */
	function getRootNode(node){
	    return node.getRootNode && node.getRootNode() || window;
	}

	/**
	 * A context menu
	 * @param {Object[]} items    Array containing the menu structure
	 *                            TODO: describe structure
	 * @param {Object} [options]  Object with options. Available options:
	 *                            {function} close    Callback called when the
	 *                                                context menu is being closed.
	 * @constructor
	 */
	function ContextMenu (items, options) {
	  this.dom = {};

	  var me = this;
	  var dom = this.dom;
	  this.anchor = undefined;
	  this.items = items;
	  this.eventListeners = {};
	  this.selection = undefined; // holds the selection before the menu was opened
	  this.onClose = options ? options.close : undefined;

	  // create root element
	  var root = document.createElement('div');
	  root.className = 'jsoneditor-contextmenu-root';
	  dom.root = root;

	  // create a container element
	  var menu = document.createElement('div');
	  menu.className = 'jsoneditor-contextmenu';
	  dom.menu = menu;
	  root.appendChild(menu);

	  // create a list to hold the menu items
	  var list = document.createElement('ul');
	  list.className = 'jsoneditor-menu';
	  menu.appendChild(list);
	  dom.list = list;
	  dom.items = []; // list with all buttons

	  // create a (non-visible) button to set the focus to the menu
	  var focusButton = document.createElement('button');
	  focusButton.type = 'button';
	  dom.focusButton = focusButton;
	  var li = document.createElement('li');
	  li.style.overflow = 'hidden';
	  li.style.height = '0';
	  li.appendChild(focusButton);
	  list.appendChild(li);

	  function createMenuItems (list, domItems, items) {
	    items.forEach(function (item) {
	      if (item.type == 'separator') {
	        // create a separator
	        var separator = document.createElement('div');
	        separator.className = 'jsoneditor-separator';
	        li = document.createElement('li');
	        li.appendChild(separator);
	        list.appendChild(li);
	      }
	      else {
	        var domItem = {};

	        // create a menu item
	        var li = document.createElement('li');
	        list.appendChild(li);

	        // create a button in the menu item
	        var button = document.createElement('button');
	        button.type = 'button';
	        button.className = item.className;
	        domItem.button = button;
	        if (item.title) {
	          button.title = item.title;
	        }
	        if (item.click) {
	          button.onclick = function (event) {
	            event.preventDefault();
	            me.hide();
	            item.click();
	          };
	        }
	        li.appendChild(button);

	        // create the contents of the button
	        if (item.submenu) {
	          // add the icon to the button
	          var divIcon = document.createElement('div');
	          divIcon.className = 'jsoneditor-icon';
	          button.appendChild(divIcon);
	          var divText = document.createElement('div');
	          divText.className = 'jsoneditor-text' +
	              (item.click ? '' : ' jsoneditor-right-margin');
	          divText.appendChild(document.createTextNode(item.text));
	          button.appendChild(divText);

	          var buttonSubmenu;
	          if (item.click) {
	            // submenu and a button with a click handler
	            button.className += ' jsoneditor-default';

	            var buttonExpand = document.createElement('button');
	            buttonExpand.type = 'button';
	            domItem.buttonExpand = buttonExpand;
	            buttonExpand.className = 'jsoneditor-expand';
	            buttonExpand.innerHTML = '<div class="jsoneditor-expand"></div>';
	            li.appendChild(buttonExpand);
	            if (item.submenuTitle) {
	              buttonExpand.title = item.submenuTitle;
	            }

	            buttonSubmenu = buttonExpand;
	          }
	          else {
	            // submenu and a button without a click handler
	            var divExpand = document.createElement('div');
	            divExpand.className = 'jsoneditor-expand';
	            button.appendChild(divExpand);

	            buttonSubmenu = button;
	          }

	          // attach a handler to expand/collapse the submenu
	          buttonSubmenu.onclick = function (event) {
	            event.preventDefault();
	            me._onExpandItem(domItem);
	            buttonSubmenu.focus();
	          };

	          // create the submenu
	          var domSubItems = [];
	          domItem.subItems = domSubItems;
	          var ul = document.createElement('ul');
	          domItem.ul = ul;
	          ul.className = 'jsoneditor-menu';
	          ul.style.height = '0';
	          li.appendChild(ul);
	          createMenuItems(ul, domSubItems, item.submenu);
	        }
	        else {
	          // no submenu, just a button with clickhandler
	          button.innerHTML = '<div class="jsoneditor-icon"></div>' +
	              '<div class="jsoneditor-text">' + translate(item.text) + '</div>';
	        }

	        domItems.push(domItem);
	      }
	    });
	  }
	  createMenuItems(list, this.dom.items, items);

	  // TODO: when the editor is small, show the submenu on the right instead of inline?

	  // calculate the max height of the menu with one submenu expanded
	  this.maxHeight = 0; // height in pixels
	  items.forEach(function (item) {
	    var height = (items.length + (item.submenu ? item.submenu.length : 0)) * 24;
	    me.maxHeight = Math.max(me.maxHeight, height);
	  });
	}

	/**
	 * Get the currently visible buttons
	 * @return {Array.<HTMLElement>} buttons
	 * @private
	 */
	ContextMenu.prototype._getVisibleButtons = function () {
	  var buttons = [];
	  var me = this;
	  this.dom.items.forEach(function (item) {
	    buttons.push(item.button);
	    if (item.buttonExpand) {
	      buttons.push(item.buttonExpand);
	    }
	    if (item.subItems && item == me.expandedItem) {
	      item.subItems.forEach(function (subItem) {
	        buttons.push(subItem.button);
	        if (subItem.buttonExpand) {
	          buttons.push(subItem.buttonExpand);
	        }
	        // TODO: change to fully recursive method
	      });
	    }
	  });

	  return buttons;
	};

	// currently displayed context menu, a singleton. We may only have one visible context menu
	ContextMenu.visibleMenu = undefined;

	/**
	 * Attach the menu to an anchor
	 * @param {HTMLElement} anchor          Anchor where the menu will be attached
	 *                                      as sibling.
	 * @param {HTMLElement} [contentWindow] The DIV with with the (scrollable) contents
	 */
	ContextMenu.prototype.show = function (anchor, contentWindow) {
	  this.hide();

	  // determine whether to display the menu below or above the anchor
	  var showBelow = true;
	  var parent = anchor.parentNode;
	  var anchorRect = anchor.getBoundingClientRect();
	  var parentRect = parent.getBoundingClientRect()

	  if (contentWindow) {
	    
	    var contentRect = contentWindow.getBoundingClientRect();

	    if (anchorRect.bottom + this.maxHeight < contentRect.bottom) {
	      // fits below -> show below
	    }
	    else if (anchorRect.top - this.maxHeight > contentRect.top) {
	      // fits above -> show above
	      showBelow = false;
	    }
	    else {
	      // doesn't fit above nor below -> show below
	    }
	  }

	  var leftGap = anchorRect.left - parentRect.left;
	  var topGap = anchorRect.top - parentRect.top;

	  // position the menu
	  if (showBelow) {
	    // display the menu below the anchor
	    var anchorHeight = anchor.offsetHeight;
	    this.dom.menu.style.left = leftGap + 'px';
	    this.dom.menu.style.top = topGap + anchorHeight + 'px';
	    this.dom.menu.style.bottom = '';
	  }
	  else {
	    // display the menu above the anchor
	    this.dom.menu.style.left = leftGap + 'px';
	    this.dom.menu.style.top = topGap + 'px';
	    this.dom.menu.style.bottom = '0px';
	  }

	  // find the root node of the page (window, or a shadow dom root element)
	  this.rootNode = getRootNode(anchor);

	  // attach the menu to the parent of the anchor
	  parent.insertBefore(this.dom.root, parent.firstChild);

	  // create and attach event listeners
	  var me = this;
	  var list = this.dom.list;
	  this.eventListeners.mousedown = util.addEventListener(this.rootNode, 'mousedown', function (event) {
	    // hide menu on click outside of the menu
	    var target = event.target;
	    if ((target != list) && !me._isChildOf(target, list)) {
	      me.hide();
	      event.stopPropagation();
	      event.preventDefault();
	    }
	  });
	  this.eventListeners.keydown = util.addEventListener(this.rootNode, 'keydown', function (event) {
	    me._onKeyDown(event);
	  });

	  // move focus to the first button in the context menu
	  this.selection = util.getSelection();
	  this.anchor = anchor;
	  setTimeout(function () {
	    me.dom.focusButton.focus();
	  }, 0);

	  if (ContextMenu.visibleMenu) {
	    ContextMenu.visibleMenu.hide();
	  }
	  ContextMenu.visibleMenu = this;
	};

	/**
	 * Hide the context menu if visible
	 */
	ContextMenu.prototype.hide = function () {
	  // remove the menu from the DOM
	  if (this.dom.root.parentNode) {
	    this.dom.root.parentNode.removeChild(this.dom.root);
	    if (this.onClose) {
	      this.onClose();
	    }
	  }

	  // remove all event listeners
	  // all event listeners are supposed to be attached to document.
	  for (var name in this.eventListeners) {
	    if (this.eventListeners.hasOwnProperty(name)) {
	      var fn = this.eventListeners[name];
	      if (fn) {
	        util.removeEventListener(this.rootNode, name, fn);
	      }
	      delete this.eventListeners[name];
	    }
	  }

	  if (ContextMenu.visibleMenu == this) {
	    ContextMenu.visibleMenu = undefined;
	  }
	};

	/**
	 * Expand a submenu
	 * Any currently expanded submenu will be hided.
	 * @param {Object} domItem
	 * @private
	 */
	ContextMenu.prototype._onExpandItem = function (domItem) {
	  var me = this;
	  var alreadyVisible = (domItem == this.expandedItem);

	  // hide the currently visible submenu
	  var expandedItem = this.expandedItem;
	  if (expandedItem) {
	    //var ul = expandedItem.ul;
	    expandedItem.ul.style.height = '0';
	    expandedItem.ul.style.padding = '';
	    setTimeout(function () {
	      if (me.expandedItem != expandedItem) {
	        expandedItem.ul.style.display = '';
	        util.removeClassName(expandedItem.ul.parentNode, 'jsoneditor-selected');
	      }
	    }, 300); // timeout duration must match the css transition duration
	    this.expandedItem = undefined;
	  }

	  if (!alreadyVisible) {
	    var ul = domItem.ul;
	    ul.style.display = 'block';
	    var height = ul.clientHeight; // force a reflow in Firefox
	    setTimeout(function () {
	      if (me.expandedItem == domItem) {
	        var childsHeight = 0;
	        for (var i = 0; i < ul.childNodes.length; i++) {
	          childsHeight += ul.childNodes[i].clientHeight;
	        }
	        ul.style.height = childsHeight + 'px';
	        ul.style.padding = '5px 10px';
	      }
	    }, 0);
	    util.addClassName(ul.parentNode, 'jsoneditor-selected');
	    this.expandedItem = domItem;
	  }
	};

	/**
	 * Handle onkeydown event
	 * @param {Event} event
	 * @private
	 */
	ContextMenu.prototype._onKeyDown = function (event) {
	  var target = event.target;
	  var keynum = event.which;
	  var handled = false;
	  var buttons, targetIndex, prevButton, nextButton;

	  if (keynum == 27) { // ESC
	    // hide the menu on ESC key

	    // restore previous selection and focus
	    if (this.selection) {
	      util.setSelection(this.selection);
	    }
	    if (this.anchor) {
	      this.anchor.focus();
	    }

	    this.hide();

	    handled = true;
	  }
	  else if (keynum == 9) { // Tab
	    if (!event.shiftKey) { // Tab
	      buttons = this._getVisibleButtons();
	      targetIndex = buttons.indexOf(target);
	      if (targetIndex == buttons.length - 1) {
	        // move to first button
	        buttons[0].focus();
	        handled = true;
	      }
	    }
	    else { // Shift+Tab
	      buttons = this._getVisibleButtons();
	      targetIndex = buttons.indexOf(target);
	      if (targetIndex == 0) {
	        // move to last button
	        buttons[buttons.length - 1].focus();
	        handled = true;
	      }
	    }
	  }
	  else if (keynum == 37) { // Arrow Left
	    if (target.className == 'jsoneditor-expand') {
	      buttons = this._getVisibleButtons();
	      targetIndex = buttons.indexOf(target);
	      prevButton = buttons[targetIndex - 1];
	      if (prevButton) {
	        prevButton.focus();
	      }
	    }
	    handled = true;
	  }
	  else if (keynum == 38) { // Arrow Up
	    buttons = this._getVisibleButtons();
	    targetIndex = buttons.indexOf(target);
	    prevButton = buttons[targetIndex - 1];
	    if (prevButton && prevButton.className == 'jsoneditor-expand') {
	      // skip expand button
	      prevButton = buttons[targetIndex - 2];
	    }
	    if (!prevButton) {
	      // move to last button
	      prevButton = buttons[buttons.length - 1];
	    }
	    if (prevButton) {
	      prevButton.focus();
	    }
	    handled = true;
	  }
	  else if (keynum == 39) { // Arrow Right
	    buttons = this._getVisibleButtons();
	    targetIndex = buttons.indexOf(target);
	    nextButton = buttons[targetIndex + 1];
	    if (nextButton && nextButton.className == 'jsoneditor-expand') {
	      nextButton.focus();
	    }
	    handled = true;
	  }
	  else if (keynum == 40) { // Arrow Down
	    buttons = this._getVisibleButtons();
	    targetIndex = buttons.indexOf(target);
	    nextButton = buttons[targetIndex + 1];
	    if (nextButton && nextButton.className == 'jsoneditor-expand') {
	      // skip expand button
	      nextButton = buttons[targetIndex + 2];
	    }
	    if (!nextButton) {
	      // move to first button
	      nextButton = buttons[0];
	    }
	    if (nextButton) {
	      nextButton.focus();
	      handled = true;
	    }
	    handled = true;
	  }
	  // TODO: arrow left and right

	  if (handled) {
	    event.stopPropagation();
	    event.preventDefault();
	  }
	};

	/**
	 * Test if an element is a child of a parent element.
	 * @param {Element} child
	 * @param {Element} parent
	 * @return {boolean} isChild
	 */
	ContextMenu.prototype._isChildOf = function (child, parent) {
	  var e = child.parentNode;
	  while (e) {
	    if (e == parent) {
	      return true;
	    }
	    e = e.parentNode;
	  }

	  return false;
	};

	module.exports = ContextMenu;


/***/ },
/* 58 */
/***/ function(module, exports) {

	'use strict';

	var _locales = ['en', 'pt-BR'];
	var _defs = {
	    en: {
	        'array': 'Array',
	        'auto': 'Auto',
	        'appendText': 'Append',
	        'appendTitle': 'Append a new field with type \'auto\' after this field (Ctrl+Shift+Ins)',
	        'appendSubmenuTitle': 'Select the type of the field to be appended',
	        'appendTitleAuto': 'Append a new field with type \'auto\' (Ctrl+Shift+Ins)',
	        'ascending': 'Ascending',
	        'ascendingTitle': 'Sort the childs of this ${type} in ascending order',
	        'actionsMenu': 'Click to open the actions menu (Ctrl+M)',
	        'collapseAll': 'Collapse all fields',
	        'descending': 'Descending',
	        'descendingTitle': 'Sort the childs of this ${type} in descending order',
	        'drag': 'Drag to move this field (Alt+Shift+Arrows)',
	        'duplicateKey': 'duplicate key',
	        'duplicateText': 'Duplicate',
	        'duplicateTitle': 'Duplicate selected fields (Ctrl+D)',
	        'duplicateField': 'Duplicate this field (Ctrl+D)',
	        'empty': 'empty',
	        'expandAll': 'Expand all fields',
	        'expandTitle': 'Click to expand/collapse this field (Ctrl+E). \n' +
	            'Ctrl+Click to expand/collapse including all childs.',
	        'insert': 'Insert',
	        'insertTitle': 'Insert a new field with type \'auto\' before this field (Ctrl+Ins)',
	        'insertSub': 'Select the type of the field to be inserted',
	        'object': 'Object',
	        'redo': 'Redo (Ctrl+Shift+Z)',
	        'removeText': 'Remove',
	        'removeTitle': 'Remove selected fields (Ctrl+Del)',
	        'removeField': 'Remove this field (Ctrl+Del)',
	        'sort': 'Sort',
	        'sortTitle': 'Sort the childs of this ',
	        'string': 'String',
	        'type': 'Type',
	        'typeTitle': 'Change the type of this field',
	        'openUrl': 'Ctrl+Click or Ctrl+Enter to open url in new window',
	        'undo': 'Undo last action (Ctrl+Z)',
	        'validationCannotMove': 'Cannot move a field into a child of itself',
	        'autoType': 'Field type "auto". ' +
	            'The field type is automatically determined from the value ' +
	            'and can be a string, number, boolean, or null.',
	        'objectType': 'Field type "object". ' +
	            'An object contains an unordered set of key/value pairs.',
	        'arrayType': 'Field type "array". ' +
	            'An array contains an ordered collection of values.',
	        'stringType': 'Field type "string". ' +
	            'Field type is not determined from the value, ' +
	            'but always returned as string.'
	    },
	    'pt-BR': {
	        'array': 'Lista',
	        'auto': 'Automatico',
	        'appendText': 'Adicionar',
	        'appendTitle': 'Adicionar novo campo com tipo \'auto\' depois deste campo (Ctrl+Shift+Ins)',
	        'appendSubmenuTitle': 'Selecione o tipo do campo a ser adicionado',
	        'appendTitleAuto': 'Adicionar novo campo com tipo \'auto\' (Ctrl+Shift+Ins)',
	        'ascending': 'Ascendente',
	        'ascendingTitle': 'Organizar filhor do tipo ${type} em crescente',
	        'actionsMenu': 'Clique para abrir o menu de ações (Ctrl+M)',
	        'collapseAll': 'Fechar todos campos',
	        'descending': 'Descendente',
	        'descendingTitle': 'Organizar o filhos do tipo ${type} em decrescente',
	        'duplicateKey': 'chave duplicada',
	        'drag': 'Arraste para mover este campo (Alt+Shift+Arrows)',
	        'duplicateText': 'Duplicar',
	        'duplicateTitle': 'Duplicar campos selecionados (Ctrl+D)',
	        'duplicateField': 'Duplicar este campo (Ctrl+D)',
	        'empty': 'vazio',
	        'expandAll': 'Expandir todos campos',
	        'expandTitle': 'Clique para expandir/encolher este campo (Ctrl+E). \n' +
	            'Ctrl+Click para expandir/encolher incluindo todos os filhos.',
	        'insert': 'Inserir',
	        'insertTitle': 'Inserir um novo campo do tipo \'auto\' antes deste campo (Ctrl+Ins)',
	        'insertSub': 'Selecionar o tipo de campo a ser inserido',
	        'object': 'Objeto',
	        'redo': 'Refazer (Ctrl+Shift+Z)',
	        'removeText': 'Remover',
	        'removeTitle': 'Remover campos selecionados (Ctrl+Del)',
	        'removeField': 'Remover este campo (Ctrl+Del)',
	        'sort': 'Organizar',
	        'sortTitle': 'Organizar os filhos deste ',
	        'string': 'Texto',
	        'type': 'Tipo',
	        'typeTitle': 'Mudar o tipo deste campo',
	        'openUrl': 'Ctrl+Click ou Ctrl+Enter para abrir link em nova janela',
	        'undo': 'Desfazer último ação (Ctrl+Z)',
	        'validationCannotMove': 'Não pode mover um campo como filho dele mesmo',
	        'autoType': 'Campo do tipo "auto". ' +
	            'O tipo do campo é determinao automaticamente a partir do seu valor ' +
	            'e pode ser texto, número, verdade/falso ou nulo.',
	        'objectType': 'Campo do tipo "objeto". ' +
	            'Um objeto contém uma lista de pares com chave e valor.',
	        'arrayType': 'Campo do tipo "lista". ' +
	            'Uma lista contem uma coleção de valores ordenados.',
	        'stringType': 'Campo do tipo "string". ' +
	            'Campo do tipo nao é determinado através do seu valor, ' +
	            'mas sempre retornara um texto.'
	    }
	};

	var _defaultLang = 'en';
	var _lang;
	var userLang = navigator.language || navigator.userLanguage;
	_lang = _locales.find(function (l) {
	    return l === userLang;
	});
	if (!_lang) {
	    _lang = _defaultLang;
	}

	module.exports = {
	    // supported locales
	    _locales: _locales,
	    _defs: _defs,
	    _lang: _lang,
	    setLanguage: function (lang) {
	        if (!lang) {
	            return;
	        }
	        var langFound = _locales.find(function (l) {
	            return l === lang;
	        });
	        if (langFound) {
	            _lang = langFound;
	        } else {
	            console.error('Language not found');
	        }
	    },
	    setLanguages: function (languages) {
	        if (!languages) {
	            return;
	        }
	        for (var key in languages) {
	            var langFound = _locales.find(function (l) {
	                return l === key;
	            });
	            if (!langFound) {
	                _locales.push(key);
	            }
	            _defs[key] = Object.assign({}, _defs[_defaultLang], _defs[key], languages[key]);
	        }
	    },
	    translate: function (key, data, lang) {
	        if (!lang) {
	            lang = _lang;
	        }
	        var text = _defs[lang][key];
	        if (data) {
	            for (key in data) {
	                text = text.replace('${' + key + '}', data[key]);
	            }
	        }
	        return text || key;
	    }
	};

/***/ },
/* 59 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var ContextMenu = __webpack_require__(57);

	/**
	 * Creates a component that visualize path selection in tree based editors
	 * @param {HTMLElement} container 
	 * @constructor
	 */
	function TreePath(container) {
	  if (container) {
	    this.path = document.createElement('div');
	    this.path.className = 'jsoneditor-treepath';
	    container.appendChild(this.path);
	    this.reset();
	  }
	};

	/**
	 * Reset component to initial status
	 */
	TreePath.prototype.reset = function () {
	  this.path.innerHTML = '';
	}

	/**
	 * Renders the component UI according to a given path objects
	 * @param {Array<name: String, childs: Array>} pathObjs a list of path objects
	 * 
	 */
	TreePath.prototype.setPath = function (pathObjs) {
	  var me = this;
	  this.reset();
	  if (pathObjs && pathObjs.length) {
	    pathObjs.forEach(function (pathObj, idx) {
	      var pathEl = document.createElement('span');
	      var sepEl;
	      pathEl.className = 'jsoneditor-treepath-element';
	      pathEl.innerText = pathObj.name;
	      pathEl.onclick = _onSegmentClick.bind(me, pathObj);
	  
	      me.path.appendChild(pathEl);

	      if (pathObj.children.length) {
	        sepEl = document.createElement('span');
	        sepEl.className = 'jsoneditor-treepath-seperator';
	        sepEl.innerHTML = '&#9658;';

	        sepEl.onclick = function () {
	          var items = [];
	          pathObj.children.forEach(function (child) {
	            items.push({
	              'text': child.name,
	              'className': 'jsoneditor-type-modes' + (pathObjs[idx + 1] + 1 && pathObjs[idx + 1].name === child.name ? ' jsoneditor-selected' : ''),
	              'click': _onContextMenuItemClick.bind(me, pathObj, child.name)
	            });
	          });
	          var menu = new ContextMenu(items);
	          menu.show(sepEl);
	        };

	        me.path.appendChild(sepEl, me.container);
	      }

	      if(idx === pathObjs.length - 1) {
	        var leftRectPos = (sepEl || pathEl).getBoundingClientRect().left;
	        if(me.path.offsetWidth < leftRectPos) {
	          me.path.scrollLeft = leftRectPos;
	        }
	      }
	    });
	  }

	  function _onSegmentClick(pathObj) {
	    if (this.selectionCallback) {
	      this.selectionCallback(pathObj);
	    }
	  };

	  function _onContextMenuItemClick(pathObj, selection) {
	    if (this.contextMenuCallback) {
	      this.contextMenuCallback(pathObj, selection);
	    }
	  };
	};

	/**
	 * set a callback function for selection of path section
	 * @param {Function} callback function to invoke when section is selected
	 */
	TreePath.prototype.onSectionSelected = function (callback) {
	  if (typeof callback === 'function') {
	    this.selectionCallback = callback;      
	  }
	};

	/**
	 * set a callback function for selection of path section
	 * @param {Function} callback function to invoke when section is selected
	 */
	TreePath.prototype.onContextMenuItemSelected = function (callback) {
	  if (typeof callback === 'function') {
	    this.contextMenuCallback = callback;
	  }
	};

	module.exports = TreePath;

/***/ },
/* 60 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var naturalSort = __webpack_require__(61);
	var ContextMenu = __webpack_require__(57);
	var appendNodeFactory = __webpack_require__(62);
	var util = __webpack_require__(54);
	var translate = __webpack_require__(58).translate;

	/**
	 * @constructor Node
	 * Create a new Node
	 * @param {./treemode} editor
	 * @param {Object} [params] Can contain parameters:
	 *                          {string}  field
	 *                          {boolean} fieldEditable
	 *                          {*}       value
	 *                          {String}  type  Can have values 'auto', 'array',
	 *                                          'object', or 'string'.
	 */
	function Node (editor, params) {
	  /** @type {./treemode} */
	  this.editor = editor;
	  this.dom = {};
	  this.expanded = false;

	  if(params && (params instanceof Object)) {
	    this.setField(params.field, params.fieldEditable);
	    this.setValue(params.value, params.type);
	  }
	  else {
	    this.setField('');
	    this.setValue(null);
	  }

	  this._debouncedOnChangeValue = util.debounce(this._onChangeValue.bind(this), Node.prototype.DEBOUNCE_INTERVAL);
	  this._debouncedOnChangeField = util.debounce(this._onChangeField.bind(this), Node.prototype.DEBOUNCE_INTERVAL);
	}

	// debounce interval for keyboard input in milliseconds
	Node.prototype.DEBOUNCE_INTERVAL = 150;

	/**
	 * Determine whether the field and/or value of this node are editable
	 * @private
	 */
	Node.prototype._updateEditability = function () {
	  this.editable = {
	    field: true,
	    value: true
	  };

	  if (this.editor) {
	    this.editable.field = this.editor.options.mode === 'tree';
	    this.editable.value = this.editor.options.mode !== 'view';

	    if ((this.editor.options.mode === 'tree' || this.editor.options.mode === 'form') &&
	        (typeof this.editor.options.onEditable === 'function')) {
	      var editable = this.editor.options.onEditable({
	        field: this.field,
	        value: this.value,
	        path: this.getPath()
	      });

	      if (typeof editable === 'boolean') {
	        this.editable.field = editable;
	        this.editable.value = editable;
	      }
	      else {
	        if (typeof editable.field === 'boolean') this.editable.field = editable.field;
	        if (typeof editable.value === 'boolean') this.editable.value = editable.value;
	      }
	    }
	  }
	};

	/**
	 * Get the path of this node
	 * @return {String[]} Array containing the path to this node
	 */
	Node.prototype.getPath = function () {
	  var node = this;
	  var path = [];
	  while (node) {
	    var field = !node.parent
	        ? undefined  // do not add an (optional) field name of the root node
	        :  (node.parent.type != 'array')
	            ? node.field
	            : node.index;

	    if (field !== undefined) {
	      path.unshift(field);
	    }
	    node = node.parent;
	  }
	  return path;
	};

	/**
	 * Find a Node from a JSON path like '.items[3].name'
	 * @param {string} jsonPath
	 * @return {Node | null} Returns the Node when found, returns null if not found
	 */
	Node.prototype.findNode = function (jsonPath) {
	  var path = util.parsePath(jsonPath);
	  var node = this;
	  while (node && path.length > 0) {
	    var prop = path.shift();
	    if (typeof prop === 'number') {
	      if (node.type !== 'array') {
	        throw new Error('Cannot get child node at index ' + prop + ': node is no array');
	      }
	      node = node.childs[prop];
	    }
	    else { // string
	      if (node.type !== 'object') {
	        throw new Error('Cannot get child node ' + prop + ': node is no object');
	      }
	      node = node.childs.filter(function (child) {
	        return child.field === prop;
	      })[0];
	    }
	  }

	  return node;
	};

	/**
	 * Find all parents of this node. The parents are ordered from root node towards
	 * the original node.
	 * @return {Array.<Node>}
	 */
	Node.prototype.findParents = function () {
	  var parents = [];
	  var parent = this.parent;
	  while (parent) {
	    parents.unshift(parent);
	    parent = parent.parent;
	  }
	  return parents;
	};

	/**
	 *
	 * @param {{dataPath: string, keyword: string, message: string, params: Object, schemaPath: string} | null} error
	 * @param {Node} [child]  When this is the error of a parent node, pointing
	 *                        to an invalid child node, the child node itself
	 *                        can be provided. If provided, clicking the error
	 *                        icon will set focus to the invalid child node.
	 */
	Node.prototype.setError = function (error, child) {
	  // ensure the dom exists
	  this.getDom();

	  this.error = error;
	  var tdError = this.dom.tdError;
	  if (error) {
	    if (!tdError) {
	      tdError = document.createElement('td');
	      this.dom.tdError = tdError;
	      this.dom.tdValue.parentNode.appendChild(tdError);
	    }

	    var popover = document.createElement('div');
	    popover.className = 'jsoneditor-popover jsoneditor-right';
	    popover.appendChild(document.createTextNode(error.message));

	    var button = document.createElement('button');
	    button.type = 'button';
	    button.className = 'jsoneditor-schema-error';
	    button.appendChild(popover);

	    // update the direction of the popover
	    button.onmouseover = button.onfocus = function updateDirection() {
	      var directions = ['right', 'above', 'below', 'left'];
	      for (var i = 0; i < directions.length; i++) {
	        var direction = directions[i];
	        popover.className = 'jsoneditor-popover jsoneditor-' + direction;

	        var contentRect = this.editor.content.getBoundingClientRect();
	        var popoverRect = popover.getBoundingClientRect();
	        var margin = 20; // account for a scroll bar
	        var fit = util.insideRect(contentRect, popoverRect, margin);

	        if (fit) {
	          break;
	        }
	      }
	    }.bind(this);

	    // when clicking the error icon, expand all nodes towards the invalid
	    // child node, and set focus to the child node
	    if (child) {
	      button.onclick = function showInvalidNode() {
	        child.findParents().forEach(function (parent) {
	          parent.expand(false);
	        });

	        child.scrollTo(function () {
	          child.focus();
	        });
	      };
	    }

	    // apply the error message to the node
	    while (tdError.firstChild) {
	      tdError.removeChild(tdError.firstChild);
	    }
	    tdError.appendChild(button);
	  }
	  else {
	    if (tdError) {
	      this.dom.tdError.parentNode.removeChild(this.dom.tdError);
	      delete this.dom.tdError;
	    }
	  }
	};

	/**
	 * Get the index of this node: the index in the list of childs where this
	 * node is part of
	 * @return {number} Returns the index, or -1 if this is the root node
	 */
	Node.prototype.getIndex = function () {
	  return this.parent ? this.parent.childs.indexOf(this) : -1;
	};

	/**
	 * Set parent node
	 * @param {Node} parent
	 */
	Node.prototype.setParent = function(parent) {
	  this.parent = parent;
	};

	/**
	 * Set field
	 * @param {String}  field
	 * @param {boolean} [fieldEditable]
	 */
	Node.prototype.setField = function(field, fieldEditable) {
	  this.field = field;
	  this.previousField = field;
	  this.fieldEditable = (fieldEditable === true);
	};

	/**
	 * Get field
	 * @return {String}
	 */
	Node.prototype.getField = function() {
	  if (this.field === undefined) {
	    this._getDomField();
	  }

	  return this.field;
	};

	/**
	 * Set value. Value is a JSON structure or an element String, Boolean, etc.
	 * @param {*} value
	 * @param {String} [type]  Specify the type of the value. Can be 'auto',
	 *                         'array', 'object', or 'string'
	 */
	Node.prototype.setValue = function(value, type) {
	  var childValue, child;

	  // first clear all current childs (if any)
	  var childs = this.childs;
	  if (childs) {
	    while (childs.length) {
	      this.removeChild(childs[0]);
	    }
	  }

	  // TODO: remove the DOM of this Node

	  this.type = this._getType(value);

	  // check if type corresponds with the provided type
	  if (type && type != this.type) {
	    if (type == 'string' && this.type == 'auto') {
	      this.type = type;
	    }
	    else {
	      throw new Error('Type mismatch: ' +
	          'cannot cast value of type "' + this.type +
	          ' to the specified type "' + type + '"');
	    }
	  }

	  if (this.type == 'array') {
	    // array
	    this.childs = [];
	    for (var i = 0, iMax = value.length; i < iMax; i++) {
	      childValue = value[i];
	      if (childValue !== undefined && !(childValue instanceof Function)) {
	        // ignore undefined and functions
	        child = new Node(this.editor, {
	          value: childValue
	        });
	        this.appendChild(child);
	      }
	    }
	    this.value = '';
	  }
	  else if (this.type == 'object') {
	    // object
	    this.childs = [];
	    for (var childField in value) {
	      if (value.hasOwnProperty(childField)) {
	        childValue = value[childField];
	        if (childValue !== undefined && !(childValue instanceof Function)) {
	          // ignore undefined and functions
	          child = new Node(this.editor, {
	            field: childField,
	            value: childValue
	          });
	          this.appendChild(child);
	        }
	      }
	    }
	    this.value = '';

	    // sort object keys
	    if (this.editor.options.sortObjectKeys === true) {
	      this.sort('asc');
	    }
	  }
	  else {
	    // value
	    this.childs = undefined;
	    this.value = value;
	  }

	  this.previousValue = this.value;
	};

	/**
	 * Get value. Value is a JSON structure
	 * @return {*} value
	 */
	Node.prototype.getValue = function() {
	  //var childs, i, iMax;

	  if (this.type == 'array') {
	    var arr = [];
	    this.childs.forEach (function (child) {
	      arr.push(child.getValue());
	    });
	    return arr;
	  }
	  else if (this.type == 'object') {
	    var obj = {};
	    this.childs.forEach (function (child) {
	      obj[child.getField()] = child.getValue();
	    });
	    return obj;
	  }
	  else {
	    if (this.value === undefined) {
	      this._getDomValue();
	    }

	    return this.value;
	  }
	};

	/**
	 * Get the nesting level of this node
	 * @return {Number} level
	 */
	Node.prototype.getLevel = function() {
	  return (this.parent ? this.parent.getLevel() + 1 : 0);
	};

	/**
	 * Get jsonpath of the current node
	 * @return {Node[]} Returns an array with nodes
	 */
	Node.prototype.getNodePath = function () {
	  var path = this.parent ? this.parent.getNodePath() : [];
	  path.push(this);
	  return path;
	};

	/**
	 * Create a clone of a node
	 * The complete state of a clone is copied, including whether it is expanded or
	 * not. The DOM elements are not cloned.
	 * @return {Node} clone
	 */
	Node.prototype.clone = function() {
	  var clone = new Node(this.editor);
	  clone.type = this.type;
	  clone.field = this.field;
	  clone.fieldInnerText = this.fieldInnerText;
	  clone.fieldEditable = this.fieldEditable;
	  clone.value = this.value;
	  clone.valueInnerText = this.valueInnerText;
	  clone.expanded = this.expanded;

	  if (this.childs) {
	    // an object or array
	    var cloneChilds = [];
	    this.childs.forEach(function (child) {
	      var childClone = child.clone();
	      childClone.setParent(clone);
	      cloneChilds.push(childClone);
	    });
	    clone.childs = cloneChilds;
	  }
	  else {
	    // a value
	    clone.childs = undefined;
	  }

	  return clone;
	};

	/**
	 * Expand this node and optionally its childs.
	 * @param {boolean} [recurse] Optional recursion, true by default. When
	 *                            true, all childs will be expanded recursively
	 */
	Node.prototype.expand = function(recurse) {
	  if (!this.childs) {
	    return;
	  }

	  // set this node expanded
	  this.expanded = true;
	  if (this.dom.expand) {
	    this.dom.expand.className = 'jsoneditor-expanded';
	  }

	  this.showChilds();

	  if (recurse !== false) {
	    this.childs.forEach(function (child) {
	      child.expand(recurse);
	    });
	  }
	};

	/**
	 * Collapse this node and optionally its childs.
	 * @param {boolean} [recurse] Optional recursion, true by default. When
	 *                            true, all childs will be collapsed recursively
	 */
	Node.prototype.collapse = function(recurse) {
	  if (!this.childs) {
	    return;
	  }

	  this.hideChilds();

	  // collapse childs in case of recurse
	  if (recurse !== false) {
	    this.childs.forEach(function (child) {
	      child.collapse(recurse);
	    });

	  }

	  // make this node collapsed
	  if (this.dom.expand) {
	    this.dom.expand.className = 'jsoneditor-collapsed';
	  }
	  this.expanded = false;
	};

	/**
	 * Recursively show all childs when they are expanded
	 */
	Node.prototype.showChilds = function() {
	  var childs = this.childs;
	  if (!childs) {
	    return;
	  }
	  if (!this.expanded) {
	    return;
	  }

	  var tr = this.dom.tr;
	  var table = tr ? tr.parentNode : undefined;
	  if (table) {
	    // show row with append button
	    var append = this.getAppend();
	    var nextTr = tr.nextSibling;
	    if (nextTr) {
	      table.insertBefore(append, nextTr);
	    }
	    else {
	      table.appendChild(append);
	    }

	    // show childs
	    this.childs.forEach(function (child) {
	      table.insertBefore(child.getDom(), append);
	      child.showChilds();
	    });
	  }
	};

	/**
	 * Hide the node with all its childs
	 */
	Node.prototype.hide = function() {
	  var tr = this.dom.tr;
	  var table = tr ? tr.parentNode : undefined;
	  if (table) {
	    table.removeChild(tr);
	  }
	  this.hideChilds();
	};


	/**
	 * Recursively hide all childs
	 */
	Node.prototype.hideChilds = function() {
	  var childs = this.childs;
	  if (!childs) {
	    return;
	  }
	  if (!this.expanded) {
	    return;
	  }

	  // hide append row
	  var append = this.getAppend();
	  if (append.parentNode) {
	    append.parentNode.removeChild(append);
	  }

	  // hide childs
	  this.childs.forEach(function (child) {
	    child.hide();
	  });
	};


	/**
	 * Goes through the path from the node to the root and ensures that it is expanded
	 */
	Node.prototype.expandTo = function() {
	  var currentNode = this.parent;
	  while (currentNode) {
	    if (!currentNode.expanded) {
	      currentNode.expand();
	    }
	    currentNode = currentNode.parent;
	  }
	};


	/**
	 * Add a new child to the node.
	 * Only applicable when Node value is of type array or object
	 * @param {Node} node
	 */
	Node.prototype.appendChild = function(node) {
	  if (this._hasChilds()) {
	    // adjust the link to the parent
	    node.setParent(this);
	    node.fieldEditable = (this.type == 'object');
	    if (this.type == 'array') {
	      node.index = this.childs.length;
	    }
	    this.childs.push(node);

	    if (this.expanded) {
	      // insert into the DOM, before the appendRow
	      var newTr = node.getDom();
	      var appendTr = this.getAppend();
	      var table = appendTr ? appendTr.parentNode : undefined;
	      if (appendTr && table) {
	        table.insertBefore(newTr, appendTr);
	      }

	      node.showChilds();
	    }

	    this.updateDom({'updateIndexes': true});
	    node.updateDom({'recurse': true});
	  }
	};


	/**
	 * Move a node from its current parent to this node
	 * Only applicable when Node value is of type array or object
	 * @param {Node} node
	 * @param {Node} beforeNode
	 */
	Node.prototype.moveBefore = function(node, beforeNode) {
	  if (this._hasChilds()) {
	    // create a temporary row, to prevent the scroll position from jumping
	    // when removing the node
	    var tbody = (this.dom.tr) ? this.dom.tr.parentNode : undefined;
	    if (tbody) {
	      var trTemp = document.createElement('tr');
	      trTemp.style.height = tbody.clientHeight + 'px';
	      tbody.appendChild(trTemp);
	    }

	    if (node.parent) {
	      node.parent.removeChild(node);
	    }

	    if (beforeNode instanceof AppendNode) {
	      this.appendChild(node);
	    }
	    else {
	      this.insertBefore(node, beforeNode);
	    }

	    if (tbody) {
	      tbody.removeChild(trTemp);
	    }
	  }
	};

	/**
	 * Move a node from its current parent to this node
	 * Only applicable when Node value is of type array or object.
	 * If index is out of range, the node will be appended to the end
	 * @param {Node} node
	 * @param {Number} index
	 */
	Node.prototype.moveTo = function (node, index) {
	  if (node.parent == this) {
	    // same parent
	    var currentIndex = this.childs.indexOf(node);
	    if (currentIndex < index) {
	      // compensate the index for removal of the node itself
	      index++;
	    }
	  }

	  var beforeNode = this.childs[index] || this.append;
	  this.moveBefore(node, beforeNode);
	};

	/**
	 * Insert a new child before a given node
	 * Only applicable when Node value is of type array or object
	 * @param {Node} node
	 * @param {Node} beforeNode
	 */
	Node.prototype.insertBefore = function(node, beforeNode) {
	  if (this._hasChilds()) {
	    if (beforeNode == this.append) {
	      // append to the child nodes

	      // adjust the link to the parent
	      node.setParent(this);
	      node.fieldEditable = (this.type == 'object');
	      this.childs.push(node);
	    }
	    else {
	      // insert before a child node
	      var index = this.childs.indexOf(beforeNode);
	      if (index == -1) {
	        throw new Error('Node not found');
	      }

	      // adjust the link to the parent
	      node.setParent(this);
	      node.fieldEditable = (this.type == 'object');
	      this.childs.splice(index, 0, node);
	    }

	    if (this.expanded) {
	      // insert into the DOM
	      var newTr = node.getDom();
	      var nextTr = beforeNode.getDom();
	      var table = nextTr ? nextTr.parentNode : undefined;
	      if (nextTr && table) {
	        table.insertBefore(newTr, nextTr);
	      }

	      node.showChilds();
	    }

	    this.updateDom({'updateIndexes': true});
	    node.updateDom({'recurse': true});
	  }
	};

	/**
	 * Insert a new child before a given node
	 * Only applicable when Node value is of type array or object
	 * @param {Node} node
	 * @param {Node} afterNode
	 */
	Node.prototype.insertAfter = function(node, afterNode) {
	  if (this._hasChilds()) {
	    var index = this.childs.indexOf(afterNode);
	    var beforeNode = this.childs[index + 1];
	    if (beforeNode) {
	      this.insertBefore(node, beforeNode);
	    }
	    else {
	      this.appendChild(node);
	    }
	  }
	};

	/**
	 * Search in this node
	 * The node will be expanded when the text is found one of its childs, else
	 * it will be collapsed. Searches are case insensitive.
	 * @param {String} text
	 * @return {Node[]} results  Array with nodes containing the search text
	 */
	Node.prototype.search = function(text) {
	  var results = [];
	  var index;
	  var search = text ? text.toLowerCase() : undefined;

	  // delete old search data
	  delete this.searchField;
	  delete this.searchValue;

	  // search in field
	  if (this.field != undefined) {
	    var field = String(this.field).toLowerCase();
	    index = field.indexOf(search);
	    if (index != -1) {
	      this.searchField = true;
	      results.push({
	        'node': this,
	        'elem': 'field'
	      });
	    }

	    // update dom
	    this._updateDomField();
	  }

	  // search in value
	  if (this._hasChilds()) {
	    // array, object

	    // search the nodes childs
	    if (this.childs) {
	      var childResults = [];
	      this.childs.forEach(function (child) {
	        childResults = childResults.concat(child.search(text));
	      });
	      results = results.concat(childResults);
	    }

	    // update dom
	    if (search != undefined) {
	      var recurse = false;
	      if (childResults.length == 0) {
	        this.collapse(recurse);
	      }
	      else {
	        this.expand(recurse);
	      }
	    }
	  }
	  else {
	    // string, auto
	    if (this.value != undefined ) {
	      var value = String(this.value).toLowerCase();
	      index = value.indexOf(search);
	      if (index != -1) {
	        this.searchValue = true;
	        results.push({
	          'node': this,
	          'elem': 'value'
	        });
	      }
	    }

	    // update dom
	    this._updateDomValue();
	  }

	  return results;
	};

	/**
	 * Move the scroll position such that this node is in the visible area.
	 * The node will not get the focus
	 * @param {function(boolean)} [callback]
	 */
	Node.prototype.scrollTo = function(callback) {
	  if (!this.dom.tr || !this.dom.tr.parentNode) {
	    // if the node is not visible, expand its parents
	    var parent = this.parent;
	    var recurse = false;
	    while (parent) {
	      parent.expand(recurse);
	      parent = parent.parent;
	    }
	  }

	  if (this.dom.tr && this.dom.tr.parentNode) {
	    this.editor.scrollTo(this.dom.tr.offsetTop, callback);
	  }
	};


	// stores the element name currently having the focus
	Node.focusElement = undefined;

	/**
	 * Set focus to this node
	 * @param {String} [elementName]  The field name of the element to get the
	 *                                focus available values: 'drag', 'menu',
	 *                                'expand', 'field', 'value' (default)
	 */
	Node.prototype.focus = function(elementName) {
	  Node.focusElement = elementName;

	  if (this.dom.tr && this.dom.tr.parentNode) {
	    var dom = this.dom;

	    switch (elementName) {
	      case 'drag':
	        if (dom.drag) {
	          dom.drag.focus();
	        }
	        else {
	          dom.menu.focus();
	        }
	        break;

	      case 'menu':
	        dom.menu.focus();
	        break;

	      case 'expand':
	        if (this._hasChilds()) {
	          dom.expand.focus();
	        }
	        else if (dom.field && this.fieldEditable) {
	          dom.field.focus();
	          util.selectContentEditable(dom.field);
	        }
	        else if (dom.value && !this._hasChilds()) {
	          dom.value.focus();
	          util.selectContentEditable(dom.value);
	        }
	        else {
	          dom.menu.focus();
	        }
	        break;

	      case 'field':
	        if (dom.field && this.fieldEditable) {
	          dom.field.focus();
	          util.selectContentEditable(dom.field);
	        }
	        else if (dom.value && !this._hasChilds()) {
	          dom.value.focus();
	          util.selectContentEditable(dom.value);
	        }
	        else if (this._hasChilds()) {
	          dom.expand.focus();
	        }
	        else {
	          dom.menu.focus();
	        }
	        break;

	      case 'value':
	      default:
	        if (dom.select) {
	          // enum select box
	          dom.select.focus();
	        }
	        else if (dom.value && !this._hasChilds()) {
	          dom.value.focus();
	          util.selectContentEditable(dom.value);
	        }
	        else if (dom.field && this.fieldEditable) {
	          dom.field.focus();
	          util.selectContentEditable(dom.field);
	        }
	        else if (this._hasChilds()) {
	          dom.expand.focus();
	        }
	        else {
	          dom.menu.focus();
	        }
	        break;
	    }
	  }
	};

	/**
	 * Select all text in an editable div after a delay of 0 ms
	 * @param {Element} editableDiv
	 */
	Node.select = function(editableDiv) {
	  setTimeout(function () {
	    util.selectContentEditable(editableDiv);
	  }, 0);
	};

	/**
	 * Update the values from the DOM field and value of this node
	 */
	Node.prototype.blur = function() {
	  // retrieve the actual field and value from the DOM.
	  this._getDomValue(false);
	  this._getDomField(false);
	};

	/**
	 * Check if given node is a child. The method will check recursively to find
	 * this node.
	 * @param {Node} node
	 * @return {boolean} containsNode
	 */
	Node.prototype.containsNode = function(node) {
	  if (this == node) {
	    return true;
	  }

	  var childs = this.childs;
	  if (childs) {
	    // TODO: use the js5 Array.some() here?
	    for (var i = 0, iMax = childs.length; i < iMax; i++) {
	      if (childs[i].containsNode(node)) {
	        return true;
	      }
	    }
	  }

	  return false;
	};

	/**
	 * Move given node into this node
	 * @param {Node} node           the childNode to be moved
	 * @param {Node} beforeNode     node will be inserted before given
	 *                                         node. If no beforeNode is given,
	 *                                         the node is appended at the end
	 * @private
	 */
	Node.prototype._move = function(node, beforeNode) {
	  if (node == beforeNode) {
	    // nothing to do...
	    return;
	  }

	  // check if this node is not a child of the node to be moved here
	  if (node.containsNode(this)) {
	    throw new Error(translate('validationCannotMove'));
	  }

	  // remove the original node
	  if (node.parent) {
	    node.parent.removeChild(node);
	  }

	  // create a clone of the node
	  var clone = node.clone();
	  node.clearDom();

	  // insert or append the node
	  if (beforeNode) {
	    this.insertBefore(clone, beforeNode);
	  }
	  else {
	    this.appendChild(clone);
	  }

	  /* TODO: adjust the field name (to prevent equal field names)
	   if (this.type == 'object') {
	   }
	   */
	};

	/**
	 * Remove a child from the node.
	 * Only applicable when Node value is of type array or object
	 * @param {Node} node   The child node to be removed;
	 * @return {Node | undefined} node  The removed node on success,
	 *                                             else undefined
	 */
	Node.prototype.removeChild = function(node) {
	  if (this.childs) {
	    var index = this.childs.indexOf(node);

	    if (index != -1) {
	      node.hide();

	      // delete old search results
	      delete node.searchField;
	      delete node.searchValue;

	      var removedNode = this.childs.splice(index, 1)[0];
	      removedNode.parent = null;

	      this.updateDom({'updateIndexes': true});

	      return removedNode;
	    }
	  }

	  return undefined;
	};

	/**
	 * Remove a child node node from this node
	 * This method is equal to Node.removeChild, except that _remove fire an
	 * onChange event.
	 * @param {Node} node
	 * @private
	 */
	Node.prototype._remove = function (node) {
	  this.removeChild(node);
	};

	/**
	 * Change the type of the value of this Node
	 * @param {String} newType
	 */
	Node.prototype.changeType = function (newType) {
	  var oldType = this.type;

	  if (oldType == newType) {
	    // type is not changed
	    return;
	  }

	  if ((newType == 'string' || newType == 'auto') &&
	      (oldType == 'string' || oldType == 'auto')) {
	    // this is an easy change
	    this.type = newType;
	  }
	  else {
	    // change from array to object, or from string/auto to object/array
	    var table = this.dom.tr ? this.dom.tr.parentNode : undefined;
	    var lastTr;
	    if (this.expanded) {
	      lastTr = this.getAppend();
	    }
	    else {
	      lastTr = this.getDom();
	    }
	    var nextTr = (lastTr && lastTr.parentNode) ? lastTr.nextSibling : undefined;

	    // hide current field and all its childs
	    this.hide();
	    this.clearDom();

	    // adjust the field and the value
	    this.type = newType;

	    // adjust childs
	    if (newType == 'object') {
	      if (!this.childs) {
	        this.childs = [];
	      }

	      this.childs.forEach(function (child, index) {
	        child.clearDom();
	        delete child.index;
	        child.fieldEditable = true;
	        if (child.field == undefined) {
	          child.field = '';
	        }
	      });

	      if (oldType == 'string' || oldType == 'auto') {
	        this.expanded = true;
	      }
	    }
	    else if (newType == 'array') {
	      if (!this.childs) {
	        this.childs = [];
	      }

	      this.childs.forEach(function (child, index) {
	        child.clearDom();
	        child.fieldEditable = false;
	        child.index = index;
	      });

	      if (oldType == 'string' || oldType == 'auto') {
	        this.expanded = true;
	      }
	    }
	    else {
	      this.expanded = false;
	    }

	    // create new DOM
	    if (table) {
	      if (nextTr) {
	        table.insertBefore(this.getDom(), nextTr);
	      }
	      else {
	        table.appendChild(this.getDom());
	      }
	    }
	    this.showChilds();
	  }

	  if (newType == 'auto' || newType == 'string') {
	    // cast value to the correct type
	    if (newType == 'string') {
	      this.value = String(this.value);
	    }
	    else {
	      this.value = this._stringCast(String(this.value));
	    }

	    this.focus();
	  }

	  this.updateDom({'updateIndexes': true});
	};

	/**
	 * Retrieve value from DOM
	 * @param {boolean} [silent]  If true (default), no errors will be thrown in
	 *                            case of invalid data
	 * @private
	 */
	Node.prototype._getDomValue = function(silent) {
	  if (this.dom.value && this.type != 'array' && this.type != 'object') {
	    this.valueInnerText = util.getInnerText(this.dom.value);
	  }

	  if (this.valueInnerText != undefined) {
	    try {
	      // retrieve the value
	      var value;
	      if (this.type == 'string') {
	        value = this._unescapeHTML(this.valueInnerText);
	      }
	      else {
	        var str = this._unescapeHTML(this.valueInnerText);
	        value = this._stringCast(str);
	      }
	      if (value !== this.value) {
	        this.value = value;
	        this._debouncedOnChangeValue();
	      }
	    }
	    catch (err) {
	      this.value = undefined;
	      // TODO: sent an action with the new, invalid value?
	      if (silent !== true) {
	        throw err;
	      }
	    }
	  }
	};

	/**
	 * Handle a changed value
	 * @private
	 */
	Node.prototype._onChangeValue = function () {
	  // get current selection, then override the range such that we can select
	  // the added/removed text on undo/redo
	  var oldSelection = this.editor.getSelection();
	  if (oldSelection.range) {
	    var undoDiff = util.textDiff(String(this.value), String(this.previousValue));
	    oldSelection.range.startOffset = undoDiff.start;
	    oldSelection.range.endOffset = undoDiff.end;
	  }
	  var newSelection = this.editor.getSelection();
	  if (newSelection.range) {
	    var redoDiff = util.textDiff(String(this.previousValue), String(this.value));
	    newSelection.range.startOffset = redoDiff.start;
	    newSelection.range.endOffset = redoDiff.end;
	  }

	  this.editor._onAction('editValue', {
	    node: this,
	    oldValue: this.previousValue,
	    newValue: this.value,
	    oldSelection: oldSelection,
	    newSelection: newSelection
	  });

	  this.previousValue = this.value;
	};

	/**
	 * Handle a changed field
	 * @private
	 */
	Node.prototype._onChangeField = function () {
	  // get current selection, then override the range such that we can select
	  // the added/removed text on undo/redo
	  var oldSelection = this.editor.getSelection();
	  var previous = this.previousField || '';
	  if (oldSelection.range) {
	    var undoDiff = util.textDiff(this.field, previous);
	    oldSelection.range.startOffset = undoDiff.start;
	    oldSelection.range.endOffset = undoDiff.end;
	  }
	  var newSelection = this.editor.getSelection();
	  if (newSelection.range) {
	    var redoDiff = util.textDiff(previous, this.field);
	    newSelection.range.startOffset = redoDiff.start;
	    newSelection.range.endOffset = redoDiff.end;
	  }

	  this.editor._onAction('editField', {
	    node: this,
	    oldValue: this.previousField,
	    newValue: this.field,
	    oldSelection: oldSelection,
	    newSelection: newSelection
	  });

	  this.previousField = this.field;
	};

	/**
	 * Update dom value:
	 * - the text color of the value, depending on the type of the value
	 * - the height of the field, depending on the width
	 * - background color in case it is empty
	 * @private
	 */
	Node.prototype._updateDomValue = function () {
	  var domValue = this.dom.value;
	  if (domValue) {
	    var classNames = ['jsoneditor-value'];


	    // set text color depending on value type
	    var value = this.value;
	    var type = (this.type == 'auto') ? util.type(value) : this.type;
	    var isUrl = type == 'string' && util.isUrl(value);
	    classNames.push('jsoneditor-' + type);
	    if (isUrl) {
	      classNames.push('jsoneditor-url');
	    }

	    // visual styling when empty
	    var isEmpty = (String(this.value) == '' && this.type != 'array' && this.type != 'object');
	    if (isEmpty) {
	      classNames.push('jsoneditor-empty');
	    }

	    // highlight when there is a search result
	    if (this.searchValueActive) {
	      classNames.push('jsoneditor-highlight-active');
	    }
	    if (this.searchValue) {
	      classNames.push('jsoneditor-highlight');
	    }

	    domValue.className = classNames.join(' ');

	    // update title
	    if (type == 'array' || type == 'object') {
	      var count = this.childs ? this.childs.length : 0;
	      domValue.title = this.type + ' containing ' + count + ' items';
	    }
	    else if (isUrl && this.editable.value) {
	      domValue.title = translate('openUrl');
	    }
	    else {
	      domValue.title = '';
	    }

	    // show checkbox when the value is a boolean
	    if (type === 'boolean' && this.editable.value) {
	      if (!this.dom.checkbox) {
	        this.dom.checkbox = document.createElement('input');
	        this.dom.checkbox.type = 'checkbox';
	        this.dom.tdCheckbox = document.createElement('td');
	        this.dom.tdCheckbox.className = 'jsoneditor-tree';
	        this.dom.tdCheckbox.appendChild(this.dom.checkbox);

	        this.dom.tdValue.parentNode.insertBefore(this.dom.tdCheckbox, this.dom.tdValue);
	      }

	      this.dom.checkbox.checked = this.value;
	    }
	    else {
	      // cleanup checkbox when displayed
	      if (this.dom.tdCheckbox) {
	        this.dom.tdCheckbox.parentNode.removeChild(this.dom.tdCheckbox);
	        delete this.dom.tdCheckbox;
	        delete this.dom.checkbox;
	      }
	    }

	    if (this.enum && this.editable.value) {
	      // create select box when this node has an enum object
	      if (!this.dom.select) {
	        this.dom.select = document.createElement('select');
	        this.id = this.field + "_" + new Date().getUTCMilliseconds();
	        this.dom.select.id = this.id;
	        this.dom.select.name = this.dom.select.id;

	        //Create the default empty option
	        this.dom.select.option = document.createElement('option');
	        this.dom.select.option.value = '';
	        this.dom.select.option.innerHTML = '--';
	        this.dom.select.appendChild(this.dom.select.option);

	        //Iterate all enum values and add them as options
	        for(var i = 0; i < this.enum.length; i++) {
	          this.dom.select.option = document.createElement('option');
	          this.dom.select.option.value = this.enum[i];
	          this.dom.select.option.innerHTML = this.enum[i];
	          if(this.dom.select.option.value == this.value){
	            this.dom.select.option.selected = true;
	          }
	          this.dom.select.appendChild(this.dom.select.option);
	        }

	        this.dom.tdSelect = document.createElement('td');
	        this.dom.tdSelect.className = 'jsoneditor-tree';
	        this.dom.tdSelect.appendChild(this.dom.select);
	        this.dom.tdValue.parentNode.insertBefore(this.dom.tdSelect, this.dom.tdValue);
	      }

	      // If the enum is inside a composite type display
	      // both the simple input and the dropdown field
	      if(this.schema && (
	          !this.schema.hasOwnProperty("oneOf") &&
	          !this.schema.hasOwnProperty("anyOf") &&
	          !this.schema.hasOwnProperty("allOf"))
	      ) {
	        this.valueFieldHTML = this.dom.tdValue.innerHTML;
	        this.dom.tdValue.style.visibility = 'hidden';
	        this.dom.tdValue.innerHTML = '';
	      } else {
	        delete this.valueFieldHTML;
	      }
	    }
	    else {
	      // cleanup select box when displayed
	      if (this.dom.tdSelect) {
	        this.dom.tdSelect.parentNode.removeChild(this.dom.tdSelect);
	        delete this.dom.tdSelect;
	        delete this.dom.select;
	        this.dom.tdValue.innerHTML = this.valueFieldHTML;
	        this.dom.tdValue.style.visibility = '';
	        delete this.valueFieldHTML;
	      }
	    }

	    // strip formatting from the contents of the editable div
	    util.stripFormatting(domValue);
	  }
	};

	/**
	 * Update dom field:
	 * - the text color of the field, depending on the text
	 * - the height of the field, depending on the width
	 * - background color in case it is empty
	 * @private
	 */
	Node.prototype._updateDomField = function () {
	  var domField = this.dom.field;
	  if (domField) {
	    // make backgound color lightgray when empty
	    var isEmpty = (String(this.field) == '' && this.parent.type != 'array');
	    if (isEmpty) {
	      util.addClassName(domField, 'jsoneditor-empty');
	    }
	    else {
	      util.removeClassName(domField, 'jsoneditor-empty');
	    }

	    // highlight when there is a search result
	    if (this.searchFieldActive) {
	      util.addClassName(domField, 'jsoneditor-highlight-active');
	    }
	    else {
	      util.removeClassName(domField, 'jsoneditor-highlight-active');
	    }
	    if (this.searchField) {
	      util.addClassName(domField, 'jsoneditor-highlight');
	    }
	    else {
	      util.removeClassName(domField, 'jsoneditor-highlight');
	    }

	    // strip formatting from the contents of the editable div
	    util.stripFormatting(domField);
	  }
	};

	/**
	 * Retrieve field from DOM
	 * @param {boolean} [silent]  If true (default), no errors will be thrown in
	 *                            case of invalid data
	 * @private
	 */
	Node.prototype._getDomField = function(silent) {
	  if (this.dom.field && this.fieldEditable) {
	    this.fieldInnerText = util.getInnerText(this.dom.field);
	  }

	  if (this.fieldInnerText != undefined) {
	    try {
	      var field = this._unescapeHTML(this.fieldInnerText);

	      if (field !== this.field) {
	        this.field = field;
	        this._debouncedOnChangeField();
	      }
	    }
	    catch (err) {
	      this.field = undefined;
	      // TODO: sent an action here, with the new, invalid value?
	      if (silent !== true) {
	        throw err;
	      }
	    }
	  }
	};

	/**
	 * Validate this node and all it's childs
	 * @return {Array.<{node: Node, error: {message: string}}>} Returns a list with duplicates
	 */
	Node.prototype.validate = function () {
	  var errors = [];

	  // find duplicate keys
	  if (this.type === 'object') {
	    var keys = {};
	    var duplicateKeys = [];
	    for (var i = 0; i < this.childs.length; i++) {
	      var child = this.childs[i];
	      if (keys.hasOwnProperty(child.field)) {
	        duplicateKeys.push(child.field);
	      }
	      keys[child.field] = true;
	    }

	    if (duplicateKeys.length > 0) {
	      errors = this.childs
	          .filter(function (node) {
	            return duplicateKeys.indexOf(node.field) !== -1;
	          })
	          .map(function (node) {
	            return {
	              node: node,
	              error: {
	                message: translate('duplicateKey') + ' "' + node.field + '"'
	              }
	            }
	          });
	    }
	  }

	  // recurse over the childs
	  if (this.childs) {
	    for (var i = 0; i < this.childs.length; i++) {
	      var e = this.childs[i].validate();
	      if (e.length > 0) {
	        errors = errors.concat(e);
	      }
	    }
	  }

	  return errors;
	};

	/**
	 * Clear the dom of the node
	 */
	Node.prototype.clearDom = function() {
	  // TODO: hide the node first?
	  //this.hide();
	  // TODO: recursively clear dom?

	  this.dom = {};
	};

	/**
	 * Get the HTML DOM TR element of the node.
	 * The dom will be generated when not yet created
	 * @return {Element} tr    HTML DOM TR Element
	 */
	Node.prototype.getDom = function() {
	  var dom = this.dom;
	  if (dom.tr) {
	    return dom.tr;
	  }

	  this._updateEditability();

	  // create row
	  dom.tr = document.createElement('tr');
	  dom.tr.node = this;

	  if (this.editor.options.mode === 'tree') { // note: we take here the global setting
	    var tdDrag = document.createElement('td');
	    if (this.editable.field) {
	      // create draggable area
	      if (this.parent) {
	        var domDrag = document.createElement('button');
	        domDrag.type = 'button';
	        dom.drag = domDrag;
	        domDrag.className = 'jsoneditor-dragarea';
	        domDrag.title = translate('drag');
	        tdDrag.appendChild(domDrag);
	      }
	    }
	    dom.tr.appendChild(tdDrag);

	    // create context menu
	    var tdMenu = document.createElement('td');
	    var menu = document.createElement('button');
	    menu.type = 'button';
	    dom.menu = menu;
	    menu.className = 'jsoneditor-contextmenu';
	    menu.title = translate('actionsMenu');
	    tdMenu.appendChild(dom.menu);
	    dom.tr.appendChild(tdMenu);
	  }

	  // create tree and field
	  var tdField = document.createElement('td');
	  dom.tr.appendChild(tdField);
	  dom.tree = this._createDomTree();
	  tdField.appendChild(dom.tree);

	  this.updateDom({'updateIndexes': true});

	  return dom.tr;
	};

	/**
	 * DragStart event, fired on mousedown on the dragarea at the left side of a Node
	 * @param {Node[] | Node} nodes
	 * @param {Event} event
	 */
	Node.onDragStart = function (nodes, event) {
	  if (!Array.isArray(nodes)) {
	    return Node.onDragStart([nodes], event);
	  }
	  if (nodes.length === 0) {
	    return;
	  }

	  var firstNode = nodes[0];
	  var lastNode = nodes[nodes.length - 1];
	  var draggedNode = Node.getNodeFromTarget(event.target);
	  var beforeNode = lastNode._nextSibling();
	  var editor = firstNode.editor;

	  // in case of multiple selected nodes, offsetY prevents the selection from
	  // jumping when you start dragging one of the lower down nodes in the selection
	  var offsetY = util.getAbsoluteTop(draggedNode.dom.tr) - util.getAbsoluteTop(firstNode.dom.tr);

	  if (!editor.mousemove) {
	    editor.mousemove = util.addEventListener(window, 'mousemove', function (event) {
	      Node.onDrag(nodes, event);
	    });
	  }

	  if (!editor.mouseup) {
	    editor.mouseup = util.addEventListener(window, 'mouseup',function (event ) {
	      Node.onDragEnd(nodes, event);
	    });
	  }

	  editor.highlighter.lock();
	  editor.drag = {
	    oldCursor: document.body.style.cursor,
	    oldSelection: editor.getSelection(),
	    oldBeforeNode: beforeNode,
	    mouseX: event.pageX,
	    offsetY: offsetY,
	    level: firstNode.getLevel()
	  };
	  document.body.style.cursor = 'move';

	  event.preventDefault();
	};

	/**
	 * Drag event, fired when moving the mouse while dragging a Node
	 * @param {Node[] | Node} nodes
	 * @param {Event} event
	 */
	Node.onDrag = function (nodes, event) {
	  if (!Array.isArray(nodes)) {
	    return Node.onDrag([nodes], event);
	  }
	  if (nodes.length === 0) {
	    return;
	  }

	  // TODO: this method has grown too large. Split it in a number of methods
	  var editor = nodes[0].editor;
	  var mouseY = event.pageY - editor.drag.offsetY;
	  var mouseX = event.pageX;
	  var trThis, trPrev, trNext, trFirst, trLast, trRoot;
	  var nodePrev, nodeNext;
	  var topThis, topPrev, topFirst, heightThis, bottomNext, heightNext;
	  var moved = false;

	  // TODO: add an ESC option, which resets to the original position

	  // move up/down
	  var firstNode = nodes[0];
	  trThis = firstNode.dom.tr;
	  topThis = util.getAbsoluteTop(trThis);
	  heightThis = trThis.offsetHeight;
	  if (mouseY < topThis) {
	    // move up
	    trPrev = trThis;
	    do {
	      trPrev = trPrev.previousSibling;
	      nodePrev = Node.getNodeFromTarget(trPrev);
	      topPrev = trPrev ? util.getAbsoluteTop(trPrev) : 0;
	    }
	    while (trPrev && mouseY < topPrev);

	    if (nodePrev && !nodePrev.parent) {
	      nodePrev = undefined;
	    }

	    if (!nodePrev) {
	      // move to the first node
	      trRoot = trThis.parentNode.firstChild;
	      trPrev = trRoot ? trRoot.nextSibling : undefined;
	      nodePrev = Node.getNodeFromTarget(trPrev);
	      if (nodePrev == firstNode) {
	        nodePrev = undefined;
	      }
	    }

	    if (nodePrev) {
	      // check if mouseY is really inside the found node
	      trPrev = nodePrev.dom.tr;
	      topPrev = trPrev ? util.getAbsoluteTop(trPrev) : 0;
	      if (mouseY > topPrev + heightThis) {
	        nodePrev = undefined;
	      }
	    }

	    if (nodePrev) {
	      nodes.forEach(function (node) {
	        nodePrev.parent.moveBefore(node, nodePrev);
	      });
	      moved = true;
	    }
	  }
	  else {
	    // move down
	    var lastNode = nodes[nodes.length - 1];
	    trLast = (lastNode.expanded && lastNode.append) ? lastNode.append.getDom() : lastNode.dom.tr;
	    trFirst = trLast ? trLast.nextSibling : undefined;
	    if (trFirst) {
	      topFirst = util.getAbsoluteTop(trFirst);
	      trNext = trFirst;
	      do {
	        nodeNext = Node.getNodeFromTarget(trNext);
	        if (trNext) {
	          bottomNext = trNext.nextSibling ?
	              util.getAbsoluteTop(trNext.nextSibling) : 0;
	          heightNext = trNext ? (bottomNext - topFirst) : 0;

	          if (nodeNext.parent.childs.length == nodes.length &&
	              nodeNext.parent.childs[nodes.length - 1] == lastNode) {
	            // We are about to remove the last child of this parent,
	            // which will make the parents appendNode visible.
	            topThis += 27;
	            // TODO: dangerous to suppose the height of the appendNode a constant of 27 px.
	          }
	        }

	        trNext = trNext.nextSibling;
	      }
	      while (trNext && mouseY > topThis + heightNext);

	      if (nodeNext && nodeNext.parent) {
	        // calculate the desired level
	        var diffX = (mouseX - editor.drag.mouseX);
	        var diffLevel = Math.round(diffX / 24 / 2);
	        var level = editor.drag.level + diffLevel; // desired level
	        var levelNext = nodeNext.getLevel();     // level to be

	        // find the best fitting level (move upwards over the append nodes)
	        trPrev = nodeNext.dom.tr.previousSibling;
	        while (levelNext < level && trPrev) {
	          nodePrev = Node.getNodeFromTarget(trPrev);

	          var isDraggedNode = nodes.some(function (node) {
	            return node === nodePrev || nodePrev._isChildOf(node);
	          });

	          if (isDraggedNode) {
	            // neglect the dragged nodes themselves and their childs
	          }
	          else if (nodePrev instanceof AppendNode) {
	            var childs = nodePrev.parent.childs;
	            if (childs.length != nodes.length || childs[nodes.length - 1] != lastNode) {
	              // non-visible append node of a list of childs
	              // consisting of not only this node (else the
	              // append node will change into a visible "empty"
	              // text when removing this node).
	              nodeNext = Node.getNodeFromTarget(trPrev);
	              levelNext = nodeNext.getLevel();
	            }
	            else {
	              break;
	            }
	          }
	          else {
	            break;
	          }

	          trPrev = trPrev.previousSibling;
	        }

	        // move the node when its position is changed
	        if (trLast.nextSibling != nodeNext.dom.tr) {
	          nodes.forEach(function (node) {
	            nodeNext.parent.moveBefore(node, nodeNext);
	          });
	          moved = true;
	        }
	      }
	    }
	  }

	  if (moved) {
	    // update the dragging parameters when moved
	    editor.drag.mouseX = mouseX;
	    editor.drag.level = firstNode.getLevel();
	  }

	  // auto scroll when hovering around the top of the editor
	  editor.startAutoScroll(mouseY);

	  event.preventDefault();
	};

	/**
	 * Drag event, fired on mouseup after having dragged a node
	 * @param {Node[] | Node} nodes
	 * @param {Event} event
	 */
	Node.onDragEnd = function (nodes, event) {
	  if (!Array.isArray(nodes)) {
	    return Node.onDrag([nodes], event);
	  }
	  if (nodes.length === 0) {
	    return;
	  }

	  var firstNode = nodes[0];
	  var editor = firstNode.editor;
	  var parent = firstNode.parent;
	  var firstIndex = parent.childs.indexOf(firstNode);
	  var beforeNode = parent.childs[firstIndex + nodes.length] || parent.append;

	  // set focus to the context menu button of the first node
	  if (nodes[0]) {
	    nodes[0].dom.menu.focus();
	  }

	  var params = {
	    nodes: nodes,
	    oldSelection: editor.drag.oldSelection,
	    newSelection: editor.getSelection(),
	    oldBeforeNode: editor.drag.oldBeforeNode,
	    newBeforeNode: beforeNode
	  };

	  if (params.oldBeforeNode != params.newBeforeNode) {
	    // only register this action if the node is actually moved to another place
	    editor._onAction('moveNodes', params);
	  }

	  document.body.style.cursor = editor.drag.oldCursor;
	  editor.highlighter.unlock();
	  nodes.forEach(function (node) {
	    if (event.target !== node.dom.drag && event.target !== node.dom.menu) {
	      editor.highlighter.unhighlight();
	    }
	  });
	  delete editor.drag;

	  if (editor.mousemove) {
	    util.removeEventListener(window, 'mousemove', editor.mousemove);
	    delete editor.mousemove;
	  }
	  if (editor.mouseup) {
	    util.removeEventListener(window, 'mouseup', editor.mouseup);
	    delete editor.mouseup;
	  }

	  // Stop any running auto scroll
	  editor.stopAutoScroll();

	  event.preventDefault();
	};

	/**
	 * Test if this node is a child of an other node
	 * @param {Node} node
	 * @return {boolean} isChild
	 * @private
	 */
	Node.prototype._isChildOf = function (node) {
	  var n = this.parent;
	  while (n) {
	    if (n == node) {
	      return true;
	    }
	    n = n.parent;
	  }

	  return false;
	};

	/**
	 * Create an editable field
	 * @return {Element} domField
	 * @private
	 */
	Node.prototype._createDomField = function () {
	  return document.createElement('div');
	};

	/**
	 * Set highlighting for this node and all its childs.
	 * Only applied to the currently visible (expanded childs)
	 * @param {boolean} highlight
	 */
	Node.prototype.setHighlight = function (highlight) {
	  if (this.dom.tr) {
	    if (highlight) {
	      util.addClassName(this.dom.tr, 'jsoneditor-highlight');
	    }
	    else {
	      util.removeClassName(this.dom.tr, 'jsoneditor-highlight');
	    }

	    if (this.append) {
	      this.append.setHighlight(highlight);
	    }

	    if (this.childs) {
	      this.childs.forEach(function (child) {
	        child.setHighlight(highlight);
	      });
	    }
	  }
	};

	/**
	 * Select or deselect a node
	 * @param {boolean} selected
	 * @param {boolean} [isFirst]
	 */
	Node.prototype.setSelected = function (selected, isFirst) {
	  this.selected = selected;

	  if (this.dom.tr) {
	    if (selected) {
	      util.addClassName(this.dom.tr, 'jsoneditor-selected');
	    }
	    else {
	      util.removeClassName(this.dom.tr, 'jsoneditor-selected');
	    }

	    if (isFirst) {
	      util.addClassName(this.dom.tr, 'jsoneditor-first');
	    }
	    else {
	      util.removeClassName(this.dom.tr, 'jsoneditor-first');
	    }

	    if (this.append) {
	      this.append.setSelected(selected);
	    }

	    if (this.childs) {
	      this.childs.forEach(function (child) {
	        child.setSelected(selected);
	      });
	    }
	  }
	};

	/**
	 * Update the value of the node. Only primitive types are allowed, no Object
	 * or Array is allowed.
	 * @param {String | Number | Boolean | null} value
	 */
	Node.prototype.updateValue = function (value) {
	  this.value = value;
	  this.updateDom();
	};

	/**
	 * Update the field of the node.
	 * @param {String} field
	 */
	Node.prototype.updateField = function (field) {
	  this.field = field;
	  this.updateDom();
	};

	/**
	 * Update the HTML DOM, optionally recursing through the childs
	 * @param {Object} [options] Available parameters:
	 *                          {boolean} [recurse]         If true, the
	 *                          DOM of the childs will be updated recursively.
	 *                          False by default.
	 *                          {boolean} [updateIndexes]   If true, the childs
	 *                          indexes of the node will be updated too. False by
	 *                          default.
	 */
	Node.prototype.updateDom = function (options) {
	  // update level indentation
	  var domTree = this.dom.tree;
	  if (domTree) {
	    domTree.style.marginLeft = this.getLevel() * 24 + 'px';
	  }

	  // apply field to DOM
	  var domField = this.dom.field;
	  if (domField) {
	    if (this.fieldEditable) {
	      // parent is an object
	      domField.contentEditable = this.editable.field;
	      domField.spellcheck = false;
	      domField.className = 'jsoneditor-field';
	    }
	    else {
	      // parent is an array this is the root node
	      domField.className = 'jsoneditor-readonly';
	    }

	    var fieldText;
	    if (this.index != undefined) {
	      fieldText = this.index;
	    }
	    else if (this.field != undefined) {
	      fieldText = this.field;
	    }
	    else if (this._hasChilds()) {
	      fieldText = this.type;
	    }
	    else {
	      fieldText = '';
	    }
	    domField.innerHTML = this._escapeHTML(fieldText);

	    this._updateSchema();
	  }

	  // apply value to DOM
	  var domValue = this.dom.value;
	  if (domValue) {
	    var count = this.childs ? this.childs.length : 0;
	    if (this.type == 'array') {
	      domValue.innerHTML = '[' + count + ']';
	      util.addClassName(this.dom.tr, 'jsoneditor-expandable');
	    }
	    else if (this.type == 'object') {
	      domValue.innerHTML = '{' + count + '}';
	      util.addClassName(this.dom.tr, 'jsoneditor-expandable');
	    }
	    else {
	      domValue.innerHTML = this._escapeHTML(this.value);
	      util.removeClassName(this.dom.tr, 'jsoneditor-expandable');
	    }
	  }

	  // update field and value
	  this._updateDomField();
	  this._updateDomValue();

	  // update childs indexes
	  if (options && options.updateIndexes === true) {
	    // updateIndexes is true or undefined
	    this._updateDomIndexes();
	  }

	  if (options && options.recurse === true) {
	    // recurse is true or undefined. update childs recursively
	    if (this.childs) {
	      this.childs.forEach(function (child) {
	        child.updateDom(options);
	      });
	    }
	  }

	  // update row with append button
	  if (this.append) {
	    this.append.updateDom();
	  }
	};

	/**
	 * Locate the JSON schema of the node and check for any enum type
	 * @private
	 */
	Node.prototype._updateSchema = function () {
	  //Locating the schema of the node and checking for any enum type
	  if(this.editor && this.editor.options) {
	    // find the part of the json schema matching this nodes path
	    this.schema = this.editor.options.schema 
	        ? Node._findSchema(this.editor.options.schema, this.getPath())
	        : null;
	    if (this.schema) {
	      this.enum = Node._findEnum(this.schema);
	    }
	    else {
	      delete this.enum;
	    }
	  }
	};

	/**
	 * find an enum definition in a JSON schema, as property `enum` or inside
	 * one of the schemas composites (`oneOf`, `anyOf`, `allOf`)
	 * @param  {Object} schema
	 * @return {Array | null} Returns the enum when found, null otherwise.
	 * @private
	 */
	Node._findEnum = function (schema) {
	  if (schema.enum) {
	    return schema.enum;
	  }

	  var composite = schema.oneOf || schema.anyOf || schema.allOf;
	  if (composite) {
	    var match = composite.filter(function (entry) {return entry.enum});
	    if (match.length > 0) {
	      return match[0].enum;
	    }
	  }

	  return null
	};

	/**
	 * Return the part of a JSON schema matching given path.
	 * @param {Object} schema
	 * @param {Array.<string | number>} path
	 * @return {Object | null}
	 * @private
	 */
	Node._findSchema = function (schema, path) {
	  var childSchema = schema;
	  var foundSchema = childSchema;

	  var allSchemas = schema.oneOf || schema.anyOf || schema.allOf;
	  if (!allSchemas) {
	    allSchemas = [schema];
	  }

	  for (var j = 0; j < allSchemas.length; j++) {
	    childSchema = allSchemas[j];

	    for (var i = 0; i < path.length && childSchema; i++) {
	      var key = path[i];

	      if (typeof key === 'string' && childSchema.patternProperties && i == path.length - 1) {
	        for (var prop in childSchema.patternProperties) {
	          foundSchema = Node._findSchema(childSchema.patternProperties[prop], path.slice(i, path.length));
	        }
	      }
	      else if (childSchema.items && childSchema.items.properties) {
	        childSchema = childSchema.items.properties[key];
	        if (childSchema) {
	          foundSchema = Node._findSchema(childSchema, path.slice(i, path.length));
	        }
	      }
	      else if (typeof key === 'string' && childSchema.properties) {
	        childSchema = childSchema.properties[key] || null;
	        if (childSchema) {
	          foundSchema = Node._findSchema(childSchema, path.slice(i, path.length));
	        }
	      }
	      else if (typeof key === 'number' && childSchema.items) {
	        childSchema = childSchema.items;
	        if (childSchema) {
	          foundSchema = Node._findSchema(childSchema, path.slice(i, path.length));
	        }
	      }
	    }

	  }
	  return foundSchema
	};

	/**
	 * Update the DOM of the childs of a node: update indexes and undefined field
	 * names.
	 * Only applicable when structure is an array or object
	 * @private
	 */
	Node.prototype._updateDomIndexes = function () {
	  var domValue = this.dom.value;
	  var childs = this.childs;
	  if (domValue && childs) {
	    if (this.type == 'array') {
	      childs.forEach(function (child, index) {
	        child.index = index;
	        var childField = child.dom.field;
	        if (childField) {
	          childField.innerHTML = index;
	        }
	      });
	    }
	    else if (this.type == 'object') {
	      childs.forEach(function (child) {
	        if (child.index != undefined) {
	          delete child.index;

	          if (child.field == undefined) {
	            child.field = '';
	          }
	        }
	      });
	    }
	  }
	};

	/**
	 * Create an editable value
	 * @private
	 */
	Node.prototype._createDomValue = function () {
	  var domValue;

	  if (this.type == 'array') {
	    domValue = document.createElement('div');
	    domValue.innerHTML = '[...]';
	  }
	  else if (this.type == 'object') {
	    domValue = document.createElement('div');
	    domValue.innerHTML = '{...}';
	  }
	  else {
	    if (!this.editable.value && util.isUrl(this.value)) {
	      // create a link in case of read-only editor and value containing an url
	      domValue = document.createElement('a');
	      domValue.href = this.value;
	      domValue.innerHTML = this._escapeHTML(this.value);
	    }
	    else {
	      // create an editable or read-only div
	      domValue = document.createElement('div');
	      domValue.contentEditable = this.editable.value;
	      domValue.spellcheck = false;
	      domValue.innerHTML = this._escapeHTML(this.value);
	    }
	  }

	  return domValue;
	};

	/**
	 * Create an expand/collapse button
	 * @return {Element} expand
	 * @private
	 */
	Node.prototype._createDomExpandButton = function () {
	  // create expand button
	  var expand = document.createElement('button');
	  expand.type = 'button';
	  if (this._hasChilds()) {
	    expand.className = this.expanded ? 'jsoneditor-expanded' : 'jsoneditor-collapsed';
	    expand.title = translate('expandTitle');
	  }
	  else {
	    expand.className = 'jsoneditor-invisible';
	    expand.title = '';
	  }

	  return expand;
	};


	/**
	 * Create a DOM tree element, containing the expand/collapse button
	 * @return {Element} domTree
	 * @private
	 */
	Node.prototype._createDomTree = function () {
	  var dom = this.dom;
	  var domTree = document.createElement('table');
	  var tbody = document.createElement('tbody');
	  domTree.style.borderCollapse = 'collapse'; // TODO: put in css
	  domTree.className = 'jsoneditor-values';
	  domTree.appendChild(tbody);
	  var tr = document.createElement('tr');
	  tbody.appendChild(tr);

	  // create expand button
	  var tdExpand = document.createElement('td');
	  tdExpand.className = 'jsoneditor-tree';
	  tr.appendChild(tdExpand);
	  dom.expand = this._createDomExpandButton();
	  tdExpand.appendChild(dom.expand);
	  dom.tdExpand = tdExpand;

	  // create the field
	  var tdField = document.createElement('td');
	  tdField.className = 'jsoneditor-tree';
	  tr.appendChild(tdField);
	  dom.field = this._createDomField();
	  tdField.appendChild(dom.field);
	  dom.tdField = tdField;

	  // create a separator
	  var tdSeparator = document.createElement('td');
	  tdSeparator.className = 'jsoneditor-tree';
	  tr.appendChild(tdSeparator);
	  if (this.type != 'object' && this.type != 'array') {
	    tdSeparator.appendChild(document.createTextNode(':'));
	    tdSeparator.className = 'jsoneditor-separator';
	  }
	  dom.tdSeparator = tdSeparator;

	  // create the value
	  var tdValue = document.createElement('td');
	  tdValue.className = 'jsoneditor-tree';
	  tr.appendChild(tdValue);
	  dom.value = this._createDomValue();
	  tdValue.appendChild(dom.value);
	  dom.tdValue = tdValue;

	  return domTree;
	};

	/**
	 * Handle an event. The event is caught centrally by the editor
	 * @param {Event} event
	 */
	Node.prototype.onEvent = function (event) {
	  var type = event.type,
	      target = event.target || event.srcElement,
	      dom = this.dom,
	      node = this,
	      expandable = this._hasChilds();

	  // check if mouse is on menu or on dragarea.
	  // If so, highlight current row and its childs
	  if (target == dom.drag || target == dom.menu) {
	    if (type == 'mouseover') {
	      this.editor.highlighter.highlight(this);
	    }
	    else if (type == 'mouseout') {
	      this.editor.highlighter.unhighlight();
	    }
	  }

	  // context menu events
	  if (type == 'click' && target == dom.menu) {
	    var highlighter = node.editor.highlighter;
	    highlighter.highlight(node);
	    highlighter.lock();
	    util.addClassName(dom.menu, 'jsoneditor-selected');
	    this.showContextMenu(dom.menu, function () {
	      util.removeClassName(dom.menu, 'jsoneditor-selected');
	      highlighter.unlock();
	      highlighter.unhighlight();
	    });
	  }

	  // expand events
	  if (type == 'click') {
	    if (target == dom.expand ||
	        ((node.editor.options.mode === 'view' || node.editor.options.mode === 'form') && target.nodeName === 'DIV')) {
	      if (expandable) {
	        var recurse = event.ctrlKey; // with ctrl-key, expand/collapse all
	        this._onExpand(recurse);
	      }
	    }
	  }

	  // swap the value of a boolean when the checkbox displayed left is clicked
	  if (type == 'change' && target == dom.checkbox) {
	    this.dom.value.innerHTML = !this.value;
	    this._getDomValue();
	  }

	  // update the value of the node based on the selected option
	  if (type == 'change' && target == dom.select) {
	    this.dom.value.innerHTML = dom.select.value;
	    this._getDomValue();
	    this._updateDomValue();
	  }

	  // value events
	  var domValue = dom.value;
	  if (target == domValue) {
	    //noinspection FallthroughInSwitchStatementJS
	    switch (type) {
	      case 'blur':
	      case 'change':
	        this._getDomValue(true);
	        this._updateDomValue();
	        if (this.value) {
	          domValue.innerHTML = this._escapeHTML(this.value);
	        }
	        break;

	      case 'input':
	        //this._debouncedGetDomValue(true); // TODO
	        this._getDomValue(true);
	        this._updateDomValue();
	        break;

	      case 'keydown':
	      case 'mousedown':
	          // TODO: cleanup
	        this.editor.selection = this.editor.getSelection();
	        break;

	      case 'click':
	        if (event.ctrlKey && this.editable.value) {
	          // if read-only, we use the regular click behavior of an anchor
	          if (util.isUrl(this.value)) {
	            event.preventDefault();
	            window.open(this.value, '_blank');
	          }
	        }
	        break;

	      case 'keyup':
	        //this._debouncedGetDomValue(true); // TODO
	        this._getDomValue(true);
	        this._updateDomValue();
	        break;

	      case 'cut':
	      case 'paste':
	        setTimeout(function () {
	          node._getDomValue(true);
	          node._updateDomValue();
	        }, 1);
	        break;
	    }
	  }

	  // field events
	  var domField = dom.field;
	  if (target == domField) {
	    switch (type) {
	      case 'blur':
	      case 'change':
	        this._getDomField(true);
	        this._updateDomField();
	        if (this.field) {
	          domField.innerHTML = this._escapeHTML(this.field);
	        }
	        break;

	      case 'input':
	        this._getDomField(true);
	        this._updateSchema();
	        this._updateDomField();
	        this._updateDomValue();
	        break;

	      case 'keydown':
	      case 'mousedown':
	        this.editor.selection = this.editor.getSelection();
	        break;

	      case 'keyup':
	        this._getDomField(true);
	        this._updateDomField();
	        break;

	      case 'cut':
	      case 'paste':
	        setTimeout(function () {
	          node._getDomField(true);
	          node._updateDomField();
	        }, 1);
	        break;
	    }
	  }

	  // focus
	  // when clicked in whitespace left or right from the field or value, set focus
	  var domTree = dom.tree;
	  if (target == domTree.parentNode && type == 'click' && !event.hasMoved) {
	    var left = (event.offsetX != undefined) ?
	        (event.offsetX < (this.getLevel() + 1) * 24) :
	        (event.pageX < util.getAbsoluteLeft(dom.tdSeparator));// for FF
	    if (left || expandable) {
	      // node is expandable when it is an object or array
	      if (domField) {
	        util.setEndOfContentEditable(domField);
	        domField.focus();
	      }
	    }
	    else {
	      if (domValue && !this.enum) {
	        util.setEndOfContentEditable(domValue);
	        domValue.focus();
	      }
	    }
	  }
	  if (((target == dom.tdExpand && !expandable) || target == dom.tdField || target == dom.tdSeparator) &&
	      (type == 'click' && !event.hasMoved)) {
	    if (domField) {
	      util.setEndOfContentEditable(domField);
	      domField.focus();
	    }
	  }

	  if (type == 'keydown') {
	    this.onKeyDown(event);
	  }
	};

	/**
	 * Key down event handler
	 * @param {Event} event
	 */
	Node.prototype.onKeyDown = function (event) {
	  var keynum = event.which || event.keyCode;
	  var target = event.target || event.srcElement;
	  var ctrlKey = event.ctrlKey;
	  var shiftKey = event.shiftKey;
	  var altKey = event.altKey;
	  var handled = false;
	  var prevNode, nextNode, nextDom, nextDom2;
	  var editable = this.editor.options.mode === 'tree';
	  var oldSelection;
	  var oldBeforeNode;
	  var nodes;
	  var multiselection;
	  var selectedNodes = this.editor.multiselection.nodes.length > 0
	      ? this.editor.multiselection.nodes
	      : [this];
	  var firstNode = selectedNodes[0];
	  var lastNode = selectedNodes[selectedNodes.length - 1];

	  // console.log(ctrlKey, keynum, event.charCode); // TODO: cleanup
	  if (keynum == 13) { // Enter
	    if (target == this.dom.value) {
	      if (!this.editable.value || event.ctrlKey) {
	        if (util.isUrl(this.value)) {
	          window.open(this.value, '_blank');
	          handled = true;
	        }
	      }
	    }
	    else if (target == this.dom.expand) {
	      var expandable = this._hasChilds();
	      if (expandable) {
	        var recurse = event.ctrlKey; // with ctrl-key, expand/collapse all
	        this._onExpand(recurse);
	        target.focus();
	        handled = true;
	      }
	    }
	  }
	  else if (keynum == 68) {  // D
	    if (ctrlKey && editable) {   // Ctrl+D
	      Node.onDuplicate(selectedNodes);
	      handled = true;
	    }
	  }
	  else if (keynum == 69) { // E
	    if (ctrlKey) {       // Ctrl+E and Ctrl+Shift+E
	      this._onExpand(shiftKey);  // recurse = shiftKey
	      target.focus(); // TODO: should restore focus in case of recursing expand (which takes DOM offline)
	      handled = true;
	    }
	  }
	  else if (keynum == 77 && editable) { // M
	    if (ctrlKey) { // Ctrl+M
	      this.showContextMenu(target);
	      handled = true;
	    }
	  }
	  else if (keynum == 46 && editable) { // Del
	    if (ctrlKey) {       // Ctrl+Del
	      Node.onRemove(selectedNodes);
	      handled = true;
	    }
	  }
	  else if (keynum == 45 && editable) { // Ins
	    if (ctrlKey && !shiftKey) {       // Ctrl+Ins
	      this._onInsertBefore();
	      handled = true;
	    }
	    else if (ctrlKey && shiftKey) {   // Ctrl+Shift+Ins
	      this._onInsertAfter();
	      handled = true;
	    }
	  }
	  else if (keynum == 35) { // End
	    if (altKey) { // Alt+End
	      // find the last node
	      var endNode = this._lastNode();
	      if (endNode) {
	        endNode.focus(Node.focusElement || this._getElementName(target));
	      }
	      handled = true;
	    }
	  }
	  else if (keynum == 36) { // Home
	    if (altKey) { // Alt+Home
	      // find the first node
	      var homeNode = this._firstNode();
	      if (homeNode) {
	        homeNode.focus(Node.focusElement || this._getElementName(target));
	      }
	      handled = true;
	    }
	  }
	  else if (keynum == 37) {        // Arrow Left
	    if (altKey && !shiftKey) {  // Alt + Arrow Left
	      // move to left element
	      var prevElement = this._previousElement(target);
	      if (prevElement) {
	        this.focus(this._getElementName(prevElement));
	      }
	      handled = true;
	    }
	    else if (altKey && shiftKey && editable) { // Alt + Shift + Arrow left
	      if (lastNode.expanded) {
	        var appendDom = lastNode.getAppend();
	        nextDom = appendDom ? appendDom.nextSibling : undefined;
	      }
	      else {
	        var dom = lastNode.getDom();
	        nextDom = dom.nextSibling;
	      }
	      if (nextDom) {
	        nextNode = Node.getNodeFromTarget(nextDom);
	        nextDom2 = nextDom.nextSibling;
	        nextNode2 = Node.getNodeFromTarget(nextDom2);
	        if (nextNode && nextNode instanceof AppendNode &&
	            !(lastNode.parent.childs.length == 1) &&
	            nextNode2 && nextNode2.parent) {
	          oldSelection = this.editor.getSelection();
	          oldBeforeNode = lastNode._nextSibling();

	          selectedNodes.forEach(function (node) {
	            nextNode2.parent.moveBefore(node, nextNode2);
	          });
	          this.focus(Node.focusElement || this._getElementName(target));

	          this.editor._onAction('moveNodes', {
	            nodes: selectedNodes,
	            oldBeforeNode: oldBeforeNode,
	            newBeforeNode: nextNode2,
	            oldSelection: oldSelection,
	            newSelection: this.editor.getSelection()
	          });
	        }
	      }
	    }
	  }
	  else if (keynum == 38) {        // Arrow Up
	    if (altKey && !shiftKey) {  // Alt + Arrow Up
	      // find the previous node
	      prevNode = this._previousNode();
	      if (prevNode) {
	        this.editor.deselect(true);
	        prevNode.focus(Node.focusElement || this._getElementName(target));
	      }
	      handled = true;
	    }
	    else if (!altKey && ctrlKey && shiftKey && editable) { // Ctrl + Shift + Arrow Up
	      // select multiple nodes
	      prevNode = this._previousNode();
	      if (prevNode) {
	        multiselection = this.editor.multiselection;
	        multiselection.start = multiselection.start || this;
	        multiselection.end = prevNode;
	        nodes = this.editor._findTopLevelNodes(multiselection.start, multiselection.end);

	        this.editor.select(nodes);
	        prevNode.focus('field'); // select field as we know this always exists
	      }
	      handled = true;
	    }
	    else if (altKey && shiftKey && editable) { // Alt + Shift + Arrow Up
	      // find the previous node
	      prevNode = firstNode._previousNode();
	      if (prevNode && prevNode.parent) {
	        oldSelection = this.editor.getSelection();
	        oldBeforeNode = lastNode._nextSibling();

	        selectedNodes.forEach(function (node) {
	          prevNode.parent.moveBefore(node, prevNode);
	        });
	        this.focus(Node.focusElement || this._getElementName(target));

	        this.editor._onAction('moveNodes', {
	          nodes: selectedNodes,
	          oldBeforeNode: oldBeforeNode,
	          newBeforeNode: prevNode,
	          oldSelection: oldSelection,
	          newSelection: this.editor.getSelection()
	        });
	      }
	      handled = true;
	    }
	  }
	  else if (keynum == 39) {        // Arrow Right
	    if (altKey && !shiftKey) {  // Alt + Arrow Right
	      // move to right element
	      var nextElement = this._nextElement(target);
	      if (nextElement) {
	        this.focus(this._getElementName(nextElement));
	      }
	      handled = true;
	    }
	    else if (altKey && shiftKey && editable) { // Alt + Shift + Arrow Right
	      dom = firstNode.getDom();
	      var prevDom = dom.previousSibling;
	      if (prevDom) {
	        prevNode = Node.getNodeFromTarget(prevDom);
	        if (prevNode && prevNode.parent &&
	            (prevNode instanceof AppendNode)
	            && !prevNode.isVisible()) {
	          oldSelection = this.editor.getSelection();
	          oldBeforeNode = lastNode._nextSibling();

	          selectedNodes.forEach(function (node) {
	            prevNode.parent.moveBefore(node, prevNode);
	          });
	          this.focus(Node.focusElement || this._getElementName(target));

	          this.editor._onAction('moveNodes', {
	            nodes: selectedNodes,
	            oldBeforeNode: oldBeforeNode,
	            newBeforeNode: prevNode,
	            oldSelection: oldSelection,
	            newSelection: this.editor.getSelection()
	          });
	        }
	      }
	    }
	  }
	  else if (keynum == 40) {        // Arrow Down
	    if (altKey && !shiftKey) {  // Alt + Arrow Down
	      // find the next node
	      nextNode = this._nextNode();
	      if (nextNode) {
	        this.editor.deselect(true);
	        nextNode.focus(Node.focusElement || this._getElementName(target));
	      }
	      handled = true;
	    }
	    else if (!altKey && ctrlKey && shiftKey && editable) { // Ctrl + Shift + Arrow Down
	      // select multiple nodes
	      nextNode = this._nextNode();
	      if (nextNode) {
	        multiselection = this.editor.multiselection;
	        multiselection.start = multiselection.start || this;
	        multiselection.end = nextNode;
	        nodes = this.editor._findTopLevelNodes(multiselection.start, multiselection.end);

	        this.editor.select(nodes);
	        nextNode.focus('field'); // select field as we know this always exists
	      }
	      handled = true;
	    }
	    else if (altKey && shiftKey && editable) { // Alt + Shift + Arrow Down
	      // find the 2nd next node and move before that one
	      if (lastNode.expanded) {
	        nextNode = lastNode.append ? lastNode.append._nextNode() : undefined;
	      }
	      else {
	        nextNode = lastNode._nextNode();
	      }
	      var nextNode2 = nextNode && (nextNode._nextNode() || nextNode.parent.append);
	      if (nextNode2 && nextNode2.parent) {
	        oldSelection = this.editor.getSelection();
	        oldBeforeNode = lastNode._nextSibling();

	        selectedNodes.forEach(function (node) {
	          nextNode2.parent.moveBefore(node, nextNode2);
	        });
	        this.focus(Node.focusElement || this._getElementName(target));

	        this.editor._onAction('moveNodes', {
	          nodes: selectedNodes,
	          oldBeforeNode: oldBeforeNode,
	          newBeforeNode: nextNode2,
	          oldSelection: oldSelection,
	          newSelection: this.editor.getSelection()
	        });
	      }
	      handled = true;
	    }
	  }

	  if (handled) {
	    event.preventDefault();
	    event.stopPropagation();
	  }
	};

	/**
	 * Handle the expand event, when clicked on the expand button
	 * @param {boolean} recurse   If true, child nodes will be expanded too
	 * @private
	 */
	Node.prototype._onExpand = function (recurse) {
	  if (recurse) {
	    // Take the table offline
	    var table = this.dom.tr.parentNode; // TODO: not nice to access the main table like this
	    var frame = table.parentNode;
	    var scrollTop = frame.scrollTop;
	    frame.removeChild(table);
	  }

	  if (this.expanded) {
	    this.collapse(recurse);
	  }
	  else {
	    this.expand(recurse);
	  }

	  if (recurse) {
	    // Put the table online again
	    frame.appendChild(table);
	    frame.scrollTop = scrollTop;
	  }
	};

	/**
	 * Remove nodes
	 * @param {Node[] | Node} nodes
	 */
	Node.onRemove = function(nodes) {
	  if (!Array.isArray(nodes)) {
	    return Node.onRemove([nodes]);
	  }

	  if (nodes && nodes.length > 0) {
	    var firstNode = nodes[0];
	    var parent = firstNode.parent;
	    var editor = firstNode.editor;
	    var firstIndex = firstNode.getIndex();
	    editor.highlighter.unhighlight();

	    // adjust the focus
	    var oldSelection = editor.getSelection();
	    Node.blurNodes(nodes);
	    var newSelection = editor.getSelection();

	    // remove the nodes
	    nodes.forEach(function (node) {
	      node.parent._remove(node);
	    });

	    // store history action
	    editor._onAction('removeNodes', {
	      nodes: nodes.slice(0), // store a copy of the array!
	      parent: parent,
	      index: firstIndex,
	      oldSelection: oldSelection,
	      newSelection: newSelection
	    });
	  }
	};


	/**
	 * Duplicate nodes
	 * duplicated nodes will be added right after the original nodes
	 * @param {Node[] | Node} nodes
	 */
	Node.onDuplicate = function(nodes) {
	  if (!Array.isArray(nodes)) {
	    return Node.onDuplicate([nodes]);
	  }

	  if (nodes && nodes.length > 0) {
	    var lastNode = nodes[nodes.length - 1];
	    var parent = lastNode.parent;
	    var editor = lastNode.editor;

	    editor.deselect(editor.multiselection.nodes);

	    // duplicate the nodes
	    var oldSelection = editor.getSelection();
	    var afterNode = lastNode;
	    var clones = nodes.map(function (node) {
	      var clone = node.clone();
	      parent.insertAfter(clone, afterNode);
	      afterNode = clone;
	      return clone;
	    });

	    // set selection to the duplicated nodes
	    if (nodes.length === 1) {
	      clones[0].focus();
	    }
	    else {
	      editor.select(clones);
	    }
	    var newSelection = editor.getSelection();

	    editor._onAction('duplicateNodes', {
	      afterNode: lastNode,
	      nodes: clones,
	      parent: parent,
	      oldSelection: oldSelection,
	      newSelection: newSelection
	    });
	  }
	};

	/**
	 * Handle insert before event
	 * @param {String} [field]
	 * @param {*} [value]
	 * @param {String} [type]   Can be 'auto', 'array', 'object', or 'string'
	 * @private
	 */
	Node.prototype._onInsertBefore = function (field, value, type) {
	  var oldSelection = this.editor.getSelection();

	  var newNode = new Node(this.editor, {
	    field: (field != undefined) ? field : '',
	    value: (value != undefined) ? value : '',
	    type: type
	  });
	  newNode.expand(true);
	  this.parent.insertBefore(newNode, this);
	  this.editor.highlighter.unhighlight();
	  newNode.focus('field');
	  var newSelection = this.editor.getSelection();

	  this.editor._onAction('insertBeforeNodes', {
	    nodes: [newNode],
	    beforeNode: this,
	    parent: this.parent,
	    oldSelection: oldSelection,
	    newSelection: newSelection
	  });
	};

	/**
	 * Handle insert after event
	 * @param {String} [field]
	 * @param {*} [value]
	 * @param {String} [type]   Can be 'auto', 'array', 'object', or 'string'
	 * @private
	 */
	Node.prototype._onInsertAfter = function (field, value, type) {
	  var oldSelection = this.editor.getSelection();

	  var newNode = new Node(this.editor, {
	    field: (field != undefined) ? field : '',
	    value: (value != undefined) ? value : '',
	    type: type
	  });
	  newNode.expand(true);
	  this.parent.insertAfter(newNode, this);
	  this.editor.highlighter.unhighlight();
	  newNode.focus('field');
	  var newSelection = this.editor.getSelection();

	  this.editor._onAction('insertAfterNodes', {
	    nodes: [newNode],
	    afterNode: this,
	    parent: this.parent,
	    oldSelection: oldSelection,
	    newSelection: newSelection
	  });
	};

	/**
	 * Handle append event
	 * @param {String} [field]
	 * @param {*} [value]
	 * @param {String} [type]   Can be 'auto', 'array', 'object', or 'string'
	 * @private
	 */
	Node.prototype._onAppend = function (field, value, type) {
	  var oldSelection = this.editor.getSelection();

	  var newNode = new Node(this.editor, {
	    field: (field != undefined) ? field : '',
	    value: (value != undefined) ? value : '',
	    type: type
	  });
	  newNode.expand(true);
	  this.parent.appendChild(newNode);
	  this.editor.highlighter.unhighlight();
	  newNode.focus('field');
	  var newSelection = this.editor.getSelection();

	  this.editor._onAction('appendNodes', {
	    nodes: [newNode],
	    parent: this.parent,
	    oldSelection: oldSelection,
	    newSelection: newSelection
	  });
	};

	/**
	 * Change the type of the node's value
	 * @param {String} newType
	 * @private
	 */
	Node.prototype._onChangeType = function (newType) {
	  var oldType = this.type;
	  if (newType != oldType) {
	    var oldSelection = this.editor.getSelection();
	    this.changeType(newType);
	    var newSelection = this.editor.getSelection();

	    this.editor._onAction('changeType', {
	      node: this,
	      oldType: oldType,
	      newType: newType,
	      oldSelection: oldSelection,
	      newSelection: newSelection
	    });
	  }
	};

	/**
	 * Sort the child's of the node. Only applicable when the node has type 'object'
	 * or 'array'.
	 * @param {String} direction   Sorting direction. Available values: "asc", "desc"
	 * @private
	 */
	Node.prototype.sort = function (direction) {
	  if (!this._hasChilds()) {
	    return;
	  }

	  var order = (direction == 'desc') ? -1 : 1;
	  var prop = (this.type == 'array') ? 'value': 'field';
	  this.hideChilds();

	  var oldChilds = this.childs;
	  var oldSortOrder = this.sortOrder;

	  // copy the array (the old one will be kept for an undo action
	  this.childs = this.childs.concat();

	  // sort the arrays
	  this.childs.sort(function (a, b) {
	    return order * naturalSort(a[prop], b[prop]);
	  });
	  this.sortOrder = (order == 1) ? 'asc' : 'desc';

	  this.editor._onAction('sort', {
	    node: this,
	    oldChilds: oldChilds,
	    oldSort: oldSortOrder,
	    newChilds: this.childs,
	    newSort: this.sortOrder
	  });

	  this.showChilds();
	};

	/**
	 * Create a table row with an append button.
	 * @return {HTMLElement | undefined} buttonAppend or undefined when inapplicable
	 */
	Node.prototype.getAppend = function () {
	  if (!this.append) {
	    this.append = new AppendNode(this.editor);
	    this.append.setParent(this);
	  }
	  return this.append.getDom();
	};

	/**
	 * Find the node from an event target
	 * @param {Node} target
	 * @return {Node | undefined} node  or undefined when not found
	 * @static
	 */
	Node.getNodeFromTarget = function (target) {
	  while (target) {
	    if (target.node) {
	      return target.node;
	    }
	    target = target.parentNode;
	  }

	  return undefined;
	};

	/**
	 * Remove the focus of given nodes, and move the focus to the (a) node before,
	 * (b) the node after, or (c) the parent node.
	 * @param {Array.<Node> | Node} nodes
	 */
	Node.blurNodes = function (nodes) {
	  if (!Array.isArray(nodes)) {
	    Node.blurNodes([nodes]);
	    return;
	  }

	  var firstNode = nodes[0];
	  var parent = firstNode.parent;
	  var firstIndex = firstNode.getIndex();

	  if (parent.childs[firstIndex + nodes.length]) {
	    parent.childs[firstIndex + nodes.length].focus();
	  }
	  else if (parent.childs[firstIndex - 1]) {
	    parent.childs[firstIndex - 1].focus();
	  }
	  else {
	    parent.focus();
	  }
	};

	/**
	 * Get the next sibling of current node
	 * @return {Node} nextSibling
	 * @private
	 */
	Node.prototype._nextSibling = function () {
	  var index = this.parent.childs.indexOf(this);
	  return this.parent.childs[index + 1] || this.parent.append;
	};

	/**
	 * Get the previously rendered node
	 * @return {Node | null} previousNode
	 * @private
	 */
	Node.prototype._previousNode = function () {
	  var prevNode = null;
	  var dom = this.getDom();
	  if (dom && dom.parentNode) {
	    // find the previous field
	    var prevDom = dom;
	    do {
	      prevDom = prevDom.previousSibling;
	      prevNode = Node.getNodeFromTarget(prevDom);
	    }
	    while (prevDom && (prevNode instanceof AppendNode && !prevNode.isVisible()));
	  }
	  return prevNode;
	};

	/**
	 * Get the next rendered node
	 * @return {Node | null} nextNode
	 * @private
	 */
	Node.prototype._nextNode = function () {
	  var nextNode = null;
	  var dom = this.getDom();
	  if (dom && dom.parentNode) {
	    // find the previous field
	    var nextDom = dom;
	    do {
	      nextDom = nextDom.nextSibling;
	      nextNode = Node.getNodeFromTarget(nextDom);
	    }
	    while (nextDom && (nextNode instanceof AppendNode && !nextNode.isVisible()));
	  }

	  return nextNode;
	};

	/**
	 * Get the first rendered node
	 * @return {Node | null} firstNode
	 * @private
	 */
	Node.prototype._firstNode = function () {
	  var firstNode = null;
	  var dom = this.getDom();
	  if (dom && dom.parentNode) {
	    var firstDom = dom.parentNode.firstChild;
	    firstNode = Node.getNodeFromTarget(firstDom);
	  }

	  return firstNode;
	};

	/**
	 * Get the last rendered node
	 * @return {Node | null} lastNode
	 * @private
	 */
	Node.prototype._lastNode = function () {
	  var lastNode = null;
	  var dom = this.getDom();
	  if (dom && dom.parentNode) {
	    var lastDom = dom.parentNode.lastChild;
	    lastNode =  Node.getNodeFromTarget(lastDom);
	    while (lastDom && (lastNode instanceof AppendNode && !lastNode.isVisible())) {
	      lastDom = lastDom.previousSibling;
	      lastNode =  Node.getNodeFromTarget(lastDom);
	    }
	  }
	  return lastNode;
	};

	/**
	 * Get the next element which can have focus.
	 * @param {Element} elem
	 * @return {Element | null} nextElem
	 * @private
	 */
	Node.prototype._previousElement = function (elem) {
	  var dom = this.dom;
	  // noinspection FallthroughInSwitchStatementJS
	  switch (elem) {
	    case dom.value:
	      if (this.fieldEditable) {
	        return dom.field;
	      }
	    // intentional fall through
	    case dom.field:
	      if (this._hasChilds()) {
	        return dom.expand;
	      }
	    // intentional fall through
	    case dom.expand:
	      return dom.menu;
	    case dom.menu:
	      if (dom.drag) {
	        return dom.drag;
	      }
	    // intentional fall through
	    default:
	      return null;
	  }
	};

	/**
	 * Get the next element which can have focus.
	 * @param {Element} elem
	 * @return {Element | null} nextElem
	 * @private
	 */
	Node.prototype._nextElement = function (elem) {
	  var dom = this.dom;
	  // noinspection FallthroughInSwitchStatementJS
	  switch (elem) {
	    case dom.drag:
	      return dom.menu;
	    case dom.menu:
	      if (this._hasChilds()) {
	        return dom.expand;
	      }
	    // intentional fall through
	    case dom.expand:
	      if (this.fieldEditable) {
	        return dom.field;
	      }
	    // intentional fall through
	    case dom.field:
	      if (!this._hasChilds()) {
	        return dom.value;
	      }
	    default:
	      return null;
	  }
	};

	/**
	 * Get the dom name of given element. returns null if not found.
	 * For example when element == dom.field, "field" is returned.
	 * @param {Element} element
	 * @return {String | null} elementName  Available elements with name: 'drag',
	 *                                      'menu', 'expand', 'field', 'value'
	 * @private
	 */
	Node.prototype._getElementName = function (element) {
	  var dom = this.dom;
	  for (var name in dom) {
	    if (dom.hasOwnProperty(name)) {
	      if (dom[name] == element) {
	        return name;
	      }
	    }
	  }
	  return null;
	};

	/**
	 * Test if this node has childs. This is the case when the node is an object
	 * or array.
	 * @return {boolean} hasChilds
	 * @private
	 */
	Node.prototype._hasChilds = function () {
	  return this.type == 'array' || this.type == 'object';
	};

	// titles with explanation for the different types
	Node.TYPE_TITLES = {
	  'auto': translate('autoType'),
	  'object': translate('objectType'),
	  'array': translate('arrayType'),
	  'string': translate('stringType')
	};

	Node.prototype.addTemplates = function (menu, append) {
	    var node = this;
	    var templates = node.editor.options.templates;
	    if (templates == null) return;
	    if (templates.length) {
	        // create a separator
	        menu.push({
	            'type': 'separator'
	        });
	    }
	    var appendData = function (name, data) {
	        node._onAppend(name, data);
	    };
	    var insertData = function (name, data) {
	        node._onInsertBefore(name, data);
	    };
	    templates.forEach(function (template) {
	        menu.push({
	            text: template.text,
	            className: (template.className || 'jsoneditor-type-object'),
	            title: template.title,
	            click: (append ? appendData.bind(this, template.field, template.value) : insertData.bind(this, template.field, template.value))
	        });
	    });
	};

	/**
	 * Show a contextmenu for this node
	 * @param {HTMLElement} anchor   Anchor element to attach the context menu to
	 *                               as sibling.
	 * @param {function} [onClose]   Callback method called when the context menu
	 *                               is being closed.
	 */
	Node.prototype.showContextMenu = function (anchor, onClose) {
	  var node = this;
	  var titles = Node.TYPE_TITLES;
	  var items = [];

	  if (this.editable.value) {
	    items.push({
	      text: translate('type'),
	      title: translate('typeTitle'),
	      className: 'jsoneditor-type-' + this.type,
	      submenu: [
	        {
	          text: translate('auto'),
	          className: 'jsoneditor-type-auto' +
	              (this.type == 'auto' ? ' jsoneditor-selected' : ''),
	          title: titles.auto,
	          click: function () {
	            node._onChangeType('auto');
	          }
	        },
	        {
	          text: translate('array'),
	          className: 'jsoneditor-type-array' +
	              (this.type == 'array' ? ' jsoneditor-selected' : ''),
	          title: titles.array,
	          click: function () {
	            node._onChangeType('array');
	          }
	        },
	        {
	          text: translate('object'),
	          className: 'jsoneditor-type-object' +
	              (this.type == 'object' ? ' jsoneditor-selected' : ''),
	          title: titles.object,
	          click: function () {
	            node._onChangeType('object');
	          }
	        },
	        {
	          text: translate('string'),
	          className: 'jsoneditor-type-string' +
	              (this.type == 'string' ? ' jsoneditor-selected' : ''),
	          title: titles.string,
	          click: function () {
	            node._onChangeType('string');
	          }
	        }
	      ]
	    });
	  }

	  if (this._hasChilds()) {
	    var direction = ((this.sortOrder == 'asc') ? 'desc': 'asc');
	    items.push({
	      text: translate('sort'),
	      title: translate('sortTitle') + this.type,
	      className: 'jsoneditor-sort-' + direction,
	      click: function () {
	        node.sort(direction);
	      },
	      submenu: [
	        {
	          text: translate('ascending'),
	          className: 'jsoneditor-sort-asc',
	          title: translate('ascendingTitle' , {type: this.type}),
	          click: function () {
	            node.sort('asc');
	          }
	        },
	        {
	          text: translate('descending'),
	          className: 'jsoneditor-sort-desc',
	          title: translate('descendingTitle' , {type: this.type}),
	          click: function () {
	            node.sort('desc');
	          }
	        }
	      ]
	    });
	  }

	  if (this.parent && this.parent._hasChilds()) {
	    if (items.length) {
	      // create a separator
	      items.push({
	        'type': 'separator'
	      });
	    }

	    // create append button (for last child node only)
	    var childs = node.parent.childs;
	    if (node == childs[childs.length - 1]) {
	        var appendSubmenu = [
	            {
	                text: translate('auto'),
	                className: 'jsoneditor-type-auto',
	                title: titles.auto,
	                click: function () {
	                    node._onAppend('', '', 'auto');
	                }
	            },
	            {
	                text: translate('array'),
	                className: 'jsoneditor-type-array',
	                title: titles.array,
	                click: function () {
	                    node._onAppend('', []);
	                }
	            },
	            {
	                text: translate('object'),
	                className: 'jsoneditor-type-object',
	                title: titles.object,
	                click: function () {
	                    node._onAppend('', {});
	                }
	            },
	            {
	                text: translate('string'),
	                className: 'jsoneditor-type-string',
	                title: titles.string,
	                click: function () {
	                    node._onAppend('', '', 'string');
	                }
	            }
	        ];
	        node.addTemplates(appendSubmenu, true);
	        items.push({
	            text: translate('appendText'),
	            title: translate('appendTitle'),
	            submenuTitle: translate('appendSubmenuTitle'),
	            className: 'jsoneditor-append',
	            click: function () {
	                node._onAppend('', '', 'auto');
	            },
	            submenu: appendSubmenu
	        });
	    }



	    // create insert button
	    var insertSubmenu = [
	        {
	            text: translate('auto'),
	            className: 'jsoneditor-type-auto',
	            title: titles.auto,
	            click: function () {
	                node._onInsertBefore('', '', 'auto');
	            }
	        },
	        {
	            text: translate('array'),
	            className: 'jsoneditor-type-array',
	            title: titles.array,
	            click: function () {
	                node._onInsertBefore('', []);
	            }
	        },
	        {
	            text: translate('object'),
	            className: 'jsoneditor-type-object',
	            title: titles.object,
	            click: function () {
	                node._onInsertBefore('', {});
	            }
	        },
	        {
	            text: translate('string'),
	            className: 'jsoneditor-type-string',
	            title: titles.string,
	            click: function () {
	                node._onInsertBefore('', '', 'string');
	            }
	        }
	    ];
	    node.addTemplates(insertSubmenu, false);
	    items.push({
	      text: translate('insert'),
	      title: translate('insertTitle'),
	      submenuTitle: translate('insertSub'),
	      className: 'jsoneditor-insert',
	      click: function () {
	        node._onInsertBefore('', '', 'auto');
	      },
	      submenu: insertSubmenu
	    });

	    if (this.editable.field) {
	      // create duplicate button
	      items.push({
	        text: translate('duplicateText'),
	        title: translate('duplicateField'),
	        className: 'jsoneditor-duplicate',
	        click: function () {
	          Node.onDuplicate(node);
	        }
	      });

	      // create remove button
	      items.push({
	        text: translate('removeText'),
	        title: translate('removeField'),
	        className: 'jsoneditor-remove',
	        click: function () {
	          Node.onRemove(node);
	        }
	      });
	    }
	  }

	  var menu = new ContextMenu(items, {close: onClose});
	  menu.show(anchor, this.editor.content);
	};

	/**
	 * get the type of a value
	 * @param {*} value
	 * @return {String} type   Can be 'object', 'array', 'string', 'auto'
	 * @private
	 */
	Node.prototype._getType = function(value) {
	  if (value instanceof Array) {
	    return 'array';
	  }
	  if (value instanceof Object) {
	    return 'object';
	  }
	  if (typeof(value) == 'string' && typeof(this._stringCast(value)) != 'string') {
	    return 'string';
	  }

	  return 'auto';
	};

	/**
	 * cast contents of a string to the correct type. This can be a string,
	 * a number, a boolean, etc
	 * @param {String} str
	 * @return {*} castedStr
	 * @private
	 */
	Node.prototype._stringCast = function(str) {
	  var lower = str.toLowerCase(),
	      num = Number(str),          // will nicely fail with '123ab'
	      numFloat = parseFloat(str); // will nicely fail with '  '

	  if (str == '') {
	    return '';
	  }
	  else if (lower == 'null') {
	    return null;
	  }
	  else if (lower == 'true') {
	    return true;
	  }
	  else if (lower == 'false') {
	    return false;
	  }
	  else if (!isNaN(num) && !isNaN(numFloat)) {
	    return num;
	  }
	  else {
	    return str;
	  }
	};

	/**
	 * escape a text, such that it can be displayed safely in an HTML element
	 * @param {String} text
	 * @return {String} escapedText
	 * @private
	 */
	Node.prototype._escapeHTML = function (text) {
	  if (typeof text !== 'string') {
	    return String(text);
	  }
	  else {
	    var htmlEscaped = String(text)
	        .replace(/&/g, '&amp;')    // must be replaced first!
	        .replace(/</g, '&lt;')
	        .replace(/>/g, '&gt;')
	        .replace(/  /g, ' &nbsp;') // replace double space with an nbsp and space
	        .replace(/^ /, '&nbsp;')   // space at start
	        .replace(/ $/, '&nbsp;');  // space at end

	    var json = JSON.stringify(htmlEscaped);
	    var html = json.substring(1, json.length - 1);
	    if (this.editor.options.escapeUnicode === true) {
	      html = util.escapeUnicodeChars(html);
	    }
	    return html;
	  }
	};

	/**
	 * unescape a string.
	 * @param {String} escapedText
	 * @return {String} text
	 * @private
	 */
	Node.prototype._unescapeHTML = function (escapedText) {
	  var json = '"' + this._escapeJSON(escapedText) + '"';
	  var htmlEscaped = util.parse(json);

	  return htmlEscaped
	      .replace(/&lt;/g, '<')
	      .replace(/&gt;/g, '>')
	      .replace(/&nbsp;|\u00A0/g, ' ')
	      .replace(/&amp;/g, '&');   // must be replaced last
	};

	/**
	 * escape a text to make it a valid JSON string. The method will:
	 *   - replace unescaped double quotes with '\"'
	 *   - replace unescaped backslash with '\\'
	 *   - replace returns with '\n'
	 * @param {String} text
	 * @return {String} escapedText
	 * @private
	 */
	Node.prototype._escapeJSON = function (text) {
	  // TODO: replace with some smart regex (only when a new solution is faster!)
	  var escaped = '';
	  var i = 0;
	  while (i < text.length) {
	    var c = text.charAt(i);
	    if (c == '\n') {
	      escaped += '\\n';
	    }
	    else if (c == '\\') {
	      escaped += c;
	      i++;

	      c = text.charAt(i);
	      if (c === '' || '"\\/bfnrtu'.indexOf(c) == -1) {
	        escaped += '\\';  // no valid escape character
	      }
	      escaped += c;
	    }
	    else if (c == '"') {
	      escaped += '\\"';
	    }
	    else {
	      escaped += c;
	    }
	    i++;
	  }

	  return escaped;
	};

	// TODO: find a nicer solution to resolve this circular dependency between Node and AppendNode
	var AppendNode = appendNodeFactory(Node);

	module.exports = Node;


/***/ },
/* 61 */
/***/ function(module, exports) {

	/*
	 * Natural Sort algorithm for Javascript - Version 0.7 - Released under MIT license
	 * Author: Jim Palmer (based on chunking idea from Dave Koelle)
	 */
	/*jshint unused:false */
	module.exports = function naturalSort (a, b) {
		"use strict";
		var re = /(^([+\-]?(?:0|[1-9]\d*)(?:\.\d*)?(?:[eE][+\-]?\d+)?)?$|^0x[0-9a-f]+$|\d+)/gi,
			sre = /(^[ ]*|[ ]*$)/g,
			dre = /(^([\w ]+,?[\w ]+)?[\w ]+,?[\w ]+\d+:\d+(:\d+)?[\w ]?|^\d{1,4}[\/\-]\d{1,4}[\/\-]\d{1,4}|^\w+, \w+ \d+, \d{4})/,
			hre = /^0x[0-9a-f]+$/i,
			ore = /^0/,
			i = function(s) { return naturalSort.insensitive && ('' + s).toLowerCase() || '' + s; },
			// convert all to strings strip whitespace
			x = i(a).replace(sre, '') || '',
			y = i(b).replace(sre, '') || '',
			// chunk/tokenize
			xN = x.replace(re, '\0$1\0').replace(/\0$/,'').replace(/^\0/,'').split('\0'),
			yN = y.replace(re, '\0$1\0').replace(/\0$/,'').replace(/^\0/,'').split('\0'),
			// numeric, hex or date detection
			xD = parseInt(x.match(hre), 16) || (xN.length !== 1 && x.match(dre) && Date.parse(x)),
			yD = parseInt(y.match(hre), 16) || xD && y.match(dre) && Date.parse(y) || null,
			oFxNcL, oFyNcL;
		// first try and sort Hex codes or Dates
		if (yD) {
			if ( xD < yD ) { return -1; }
			else if ( xD > yD ) { return 1; }
		}
		// natural sorting through split numeric strings and default strings
		for(var cLoc=0, numS=Math.max(xN.length, yN.length); cLoc < numS; cLoc++) {
			// find floats not starting with '0', string or 0 if not defined (Clint Priest)
			oFxNcL = !(xN[cLoc] || '').match(ore) && parseFloat(xN[cLoc]) || xN[cLoc] || 0;
			oFyNcL = !(yN[cLoc] || '').match(ore) && parseFloat(yN[cLoc]) || yN[cLoc] || 0;
			// handle numeric vs string comparison - number < string - (Kyle Adams)
			if (isNaN(oFxNcL) !== isNaN(oFyNcL)) { return (isNaN(oFxNcL)) ? 1 : -1; }
			// rely on string comparison if different types - i.e. '02' < 2 != '02' < '2'
			else if (typeof oFxNcL !== typeof oFyNcL) {
				oFxNcL += '';
				oFyNcL += '';
			}
			if (oFxNcL < oFyNcL) { return -1; }
			if (oFxNcL > oFyNcL) { return 1; }
		}
		return 0;
	};


/***/ },
/* 62 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var util = __webpack_require__(54);
	var ContextMenu = __webpack_require__(57);
	var translate = __webpack_require__(58).translate;

	/**
	 * A factory function to create an AppendNode, which depends on a Node
	 * @param {Node} Node
	 */
	function appendNodeFactory(Node) {
	  /**
	   * @constructor AppendNode
	   * @extends Node
	   * @param {TreeEditor} editor
	   * Create a new AppendNode. This is a special node which is created at the
	   * end of the list with childs for an object or array
	   */
	  function AppendNode (editor) {
	    /** @type {TreeEditor} */
	    this.editor = editor;
	    this.dom = {};
	  }

	  AppendNode.prototype = new Node();

	  /**
	   * Return a table row with an append button.
	   * @return {Element} dom   TR element
	   */
	  AppendNode.prototype.getDom = function () {
	    // TODO: implement a new solution for the append node
	    var dom = this.dom;

	    if (dom.tr) {
	      return dom.tr;
	    }

	    this._updateEditability();

	    // a row for the append button
	    var trAppend = document.createElement('tr');
	    trAppend.node = this;
	    dom.tr = trAppend;

	    // TODO: consistent naming

	    if (this.editor.options.mode === 'tree') {
	      // a cell for the dragarea column
	      dom.tdDrag = document.createElement('td');

	      // create context menu
	      var tdMenu = document.createElement('td');
	      dom.tdMenu = tdMenu;
	      var menu = document.createElement('button');
	      menu.type = 'button';
	      menu.className = 'jsoneditor-contextmenu';
	      menu.title = 'Click to open the actions menu (Ctrl+M)';
	      dom.menu = menu;
	      tdMenu.appendChild(dom.menu);
	    }

	    // a cell for the contents (showing text 'empty')
	    var tdAppend = document.createElement('td');
	    var domText = document.createElement('div');
	    domText.innerHTML = '(' + translate('empty') + ')';
	    domText.className = 'jsoneditor-readonly';
	    tdAppend.appendChild(domText);
	    dom.td = tdAppend;
	    dom.text = domText;

	    this.updateDom();

	    return trAppend;
	  };

	  /**
	   * Update the HTML dom of the Node
	   */
	  AppendNode.prototype.updateDom = function () {
	    var dom = this.dom;
	    var tdAppend = dom.td;
	    if (tdAppend) {
	      tdAppend.style.paddingLeft = (this.getLevel() * 24 + 26) + 'px';
	      // TODO: not so nice hard coded offset
	    }

	    var domText = dom.text;
	    if (domText) {
	      domText.innerHTML = '(' + translate('empty') + ' ' + this.parent.type + ')';
	    }

	    // attach or detach the contents of the append node:
	    // hide when the parent has childs, show when the parent has no childs
	    var trAppend = dom.tr;
	    if (!this.isVisible()) {
	      if (dom.tr.firstChild) {
	        if (dom.tdDrag) {
	          trAppend.removeChild(dom.tdDrag);
	        }
	        if (dom.tdMenu) {
	          trAppend.removeChild(dom.tdMenu);
	        }
	        trAppend.removeChild(tdAppend);
	      }
	    }
	    else {
	      if (!dom.tr.firstChild) {
	        if (dom.tdDrag) {
	          trAppend.appendChild(dom.tdDrag);
	        }
	        if (dom.tdMenu) {
	          trAppend.appendChild(dom.tdMenu);
	        }
	        trAppend.appendChild(tdAppend);
	      }
	    }
	  };

	  /**
	   * Check whether the AppendNode is currently visible.
	   * the AppendNode is visible when its parent has no childs (i.e. is empty).
	   * @return {boolean} isVisible
	   */
	  AppendNode.prototype.isVisible = function () {
	    return (this.parent.childs.length == 0);
	  };

	  /**
	   * Show a contextmenu for this node
	   * @param {HTMLElement} anchor   The element to attach the menu to.
	   * @param {function} [onClose]   Callback method called when the context menu
	   *                               is being closed.
	   */
	  AppendNode.prototype.showContextMenu = function (anchor, onClose) {
	    var node = this;
	    var titles = Node.TYPE_TITLES;
	    var appendSubmenu = [
	        {
	            text: translate('auto'),
	            className: 'jsoneditor-type-auto',
	            title: titles.auto,
	            click: function () {
	                node._onAppend('', '', 'auto');
	            }
	        },
	        {
	            text: translate('array'),
	            className: 'jsoneditor-type-array',
	            title: titles.array,
	            click: function () {
	                node._onAppend('', []);
	            }
	        },
	        {
	            text: translate('object'),
	            className: 'jsoneditor-type-object',
	            title: titles.object,
	            click: function () {
	                node._onAppend('', {});
	            }
	        },
	        {
	            text: translate('string'),
	            className: 'jsoneditor-type-string',
	            title: titles.string,
	            click: function () {
	                node._onAppend('', '', 'string');
	            }
	        }
	    ];
	    node.addTemplates(appendSubmenu, true);
	    var items = [
	      // create append button
	      {
	        'text': translate('appendText'),
	        'title': translate('appendTitleAuto'),
	        'submenuTitle': translate('appendSubmenuTitle'),
	        'className': 'jsoneditor-insert',
	        'click': function () {
	          node._onAppend('', '', 'auto');
	        },
	        'submenu': appendSubmenu
	      }
	    ];

	    var menu = new ContextMenu(items, {close: onClose});
	    menu.show(anchor, this.editor.content);
	  };

	  /**
	   * Handle an event. The event is catched centrally by the editor
	   * @param {Event} event
	   */
	  AppendNode.prototype.onEvent = function (event) {
	    var type = event.type;
	    var target = event.target || event.srcElement;
	    var dom = this.dom;

	    // highlight the append nodes parent
	    var menu = dom.menu;
	    if (target == menu) {
	      if (type == 'mouseover') {
	        this.editor.highlighter.highlight(this.parent);
	      }
	      else if (type == 'mouseout') {
	        this.editor.highlighter.unhighlight();
	      }
	    }

	    // context menu events
	    if (type == 'click' && target == dom.menu) {
	      var highlighter = this.editor.highlighter;
	      highlighter.highlight(this.parent);
	      highlighter.lock();
	      util.addClassName(dom.menu, 'jsoneditor-selected');
	      this.showContextMenu(dom.menu, function () {
	        util.removeClassName(dom.menu, 'jsoneditor-selected');
	        highlighter.unlock();
	        highlighter.unhighlight();
	      });
	    }

	    if (type == 'keydown') {
	      this.onKeyDown(event);
	    }
	  };

	  return AppendNode;
	}

	module.exports = appendNodeFactory;


/***/ },
/* 63 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var ContextMenu = __webpack_require__(57);

	/**
	 * Create a select box to be used in the editor menu's, which allows to switch mode
	 * @param {HTMLElement} container
	 * @param {String[]} modes  Available modes: 'code', 'form', 'text', 'tree', 'view'
	 * @param {String} current  Available modes: 'code', 'form', 'text', 'tree', 'view'
	 * @param {function(mode: string)} onSwitch  Callback invoked on switch
	 * @constructor
	 */
	function ModeSwitcher(container, modes, current, onSwitch) {
	  // available modes
	  var availableModes = {
	    code: {
	      'text': 'Code',
	      'title': 'Switch to code highlighter',
	      'click': function () {
	        onSwitch('code')
	      }
	    },
	    form: {
	      'text': 'Form',
	      'title': 'Switch to form editor',
	      'click': function () {
	        onSwitch('form');
	      }
	    },
	    text: {
	      'text': 'Text',
	      'title': 'Switch to plain text editor',
	      'click': function () {
	        onSwitch('text');
	      }
	    },
	    tree: {
	      'text': 'Tree',
	      'title': 'Switch to tree editor',
	      'click': function () {
	        onSwitch('tree');
	      }
	    },
	    view: {
	      'text': 'View',
	      'title': 'Switch to tree view',
	      'click': function () {
	        onSwitch('view');
	      }
	    }
	  };

	  // list the selected modes
	  var items = [];
	  for (var i = 0; i < modes.length; i++) {
	    var mode = modes[i];
	    var item = availableModes[mode];
	    if (!item) {
	      throw new Error('Unknown mode "' + mode + '"');
	    }

	    item.className = 'jsoneditor-type-modes' + ((current == mode) ? ' jsoneditor-selected' : '');
	    items.push(item);
	  }

	  // retrieve the title of current mode
	  var currentMode = availableModes[current];
	  if (!currentMode) {
	    throw new Error('Unknown mode "' + current + '"');
	  }
	  var currentTitle = currentMode.text;

	  // create the html element
	  var box = document.createElement('button');
	  box.type = 'button';
	  box.className = 'jsoneditor-modes jsoneditor-separator';
	  box.innerHTML = currentTitle + ' &#x25BE;';
	  box.title = 'Switch editor mode';
	  box.onclick = function () {
	    var menu = new ContextMenu(items);
	    menu.show(box);
	  };

	  var frame = document.createElement('div');
	  frame.className = 'jsoneditor-modes';
	  frame.style.position = 'relative';
	  frame.appendChild(box);

	  container.appendChild(frame);

	  this.dom = {
	    container: container,
	    box: box,
	    frame: frame
	  };
	}

	/**
	 * Set focus to switcher
	 */
	ModeSwitcher.prototype.focus = function () {
	  this.dom.box.focus();
	};

	/**
	 * Destroy the ModeSwitcher, remove from DOM
	 */
	ModeSwitcher.prototype.destroy = function () {
	  if (this.dom && this.dom.frame && this.dom.frame.parentNode) {
	    this.dom.frame.parentNode.removeChild(this.dom.frame);
	  }
	  this.dom = null;
	};

	module.exports = ModeSwitcher;


/***/ },
/* 64 */
/***/ function(module, exports) {

	'use strict';

	function completely(config) {
	    config = config || {};
	    config.confirmKeys = config.confirmKeys || [39, 35, 9] // right, end, tab 
	    config.caseSensitive = config.caseSensitive || false    // autocomplete case sensitive

	    var fontSize = '';
	    var fontFamily = '';    

	    var wrapper = document.createElement('div');
	    wrapper.style.position = 'relative';
	    wrapper.style.outline = '0';
	    wrapper.style.border = '0';
	    wrapper.style.margin = '0';
	    wrapper.style.padding = '0';

	    var dropDown = document.createElement('div');
	    dropDown.className = 'autocomplete dropdown';
	    dropDown.style.position = 'absolute';
	    dropDown.style.visibility = 'hidden';

	    var spacer;
	    var leftSide; // <-- it will contain the leftSide part of the textfield (the bit that was already autocompleted)
	    var createDropDownController = function (elem, rs) {
	        var rows = [];
	        var ix = 0;
	        var oldIndex = -1;

	        var onMouseOver = function () { this.style.outline = '1px solid #ddd'; }
	        var onMouseOut = function () { this.style.outline = '0'; }
	        var onMouseDown = function () { p.hide(); p.onmouseselection(this.__hint, p.rs); }

	        var p = {
	            rs: rs,
	            hide: function () {
	                elem.style.visibility = 'hidden';
	                //rs.hideDropDown();
	            },
	            refresh: function (token, array) {
	                elem.style.visibility = 'hidden';
	                ix = 0;
	                elem.innerHTML = '';
	                var vph = (window.innerHeight || document.documentElement.clientHeight);
	                var rect = elem.parentNode.getBoundingClientRect();
	                var distanceToTop = rect.top - 6;                        // heuristic give 6px 
	                var distanceToBottom = vph - rect.bottom - 6;  // distance from the browser border.

	                rows = [];
	                for (var i = 0; i < array.length; i++) {

	                    if (  (config.caseSensitive && array[i].indexOf(token) !== 0)
	                        ||(!config.caseSensitive && array[i].toLowerCase().indexOf(token.toLowerCase()) !== 0)) { continue; }

	                    var divRow = document.createElement('div');
	                    divRow.className = 'item';
	                    //divRow.style.color = config.color;
	                    divRow.onmouseover = onMouseOver;
	                    divRow.onmouseout = onMouseOut;
	                    divRow.onmousedown = onMouseDown;
	                    divRow.__hint = array[i];
	                    divRow.innerHTML = array[i].substring(0, token.length) + '<b>' + array[i].substring(token.length) + '</b>';
	                    rows.push(divRow);
	                    elem.appendChild(divRow);
	                }
	                if (rows.length === 0) {
	                    return; // nothing to show.
	                }
	                if (rows.length === 1 && (   (token.toLowerCase() === rows[0].__hint.toLowerCase() && !config.caseSensitive) 
	                                           ||(token === rows[0].__hint && config.caseSensitive))){
	                    return; // do not show the dropDown if it has only one element which matches what we have just displayed.
	                }

	                if (rows.length < 2) return;
	                p.highlight(0);

	                if (distanceToTop > distanceToBottom * 3) {        // Heuristic (only when the distance to the to top is 4 times more than distance to the bottom
	                    elem.style.maxHeight = distanceToTop + 'px';  // we display the dropDown on the top of the input text
	                    elem.style.top = '';
	                    elem.style.bottom = '100%';
	                } else {
	                    elem.style.top = '100%';
	                    elem.style.bottom = '';
	                    elem.style.maxHeight = distanceToBottom + 'px';
	                }
	                elem.style.visibility = 'visible';
	            },
	            highlight: function (index) {
	                if (oldIndex != -1 && rows[oldIndex]) {
	                    rows[oldIndex].className = "item";
	                }
	                rows[index].className = "item hover"; 
	                oldIndex = index;
	            },
	            move: function (step) { // moves the selection either up or down (unless it's not possible) step is either +1 or -1.
	                if (elem.style.visibility === 'hidden') return ''; // nothing to move if there is no dropDown. (this happens if the user hits escape and then down or up)
	                if (ix + step === -1 || ix + step === rows.length) return rows[ix].__hint; // NO CIRCULAR SCROLLING. 
	                ix += step;
	                p.highlight(ix);
	                return rows[ix].__hint;//txtShadow.value = uRows[uIndex].__hint ;
	            },
	            onmouseselection: function () { } // it will be overwritten. 
	        };
	        return p;
	    }

	    function setEndOfContenteditable(contentEditableElement) {
	        var range, selection;
	        if (document.createRange)//Firefox, Chrome, Opera, Safari, IE 9+
	        {
	            range = document.createRange();//Create a range (a range is a like the selection but invisible)
	            range.selectNodeContents(contentEditableElement);//Select the entire contents of the element with the range
	            range.collapse(false);//collapse the range to the end point. false means collapse to end rather than the start
	            selection = window.getSelection();//get the selection object (allows you to change selection)
	            selection.removeAllRanges();//remove any selections already made
	            selection.addRange(range);//make the range you have just created the visible selection
	        }
	        else if (document.selection)//IE 8 and lower
	        {
	            range = document.body.createTextRange();//Create a range (a range is a like the selection but invisible)
	            range.moveToElementText(contentEditableElement);//Select the entire contents of the element with the range
	            range.collapse(false);//collapse the range to the end point. false means collapse to end rather than the start
	            range.select();//Select the range (make it the visible selection
	        }
	    }

	    function calculateWidthForText(text) {
	        if (spacer === undefined) { // on first call only.
	            spacer = document.createElement('span');
	            spacer.style.visibility = 'hidden';
	            spacer.style.position = 'fixed';
	            spacer.style.outline = '0';
	            spacer.style.margin = '0';
	            spacer.style.padding = '0';
	            spacer.style.border = '0';
	            spacer.style.left = '0';
	            spacer.style.whiteSpace = 'pre';
	            spacer.style.fontSize = fontSize;
	            spacer.style.fontFamily = fontFamily;
	            spacer.style.fontWeight = 'normal';
	            document.body.appendChild(spacer);
	        }

	        // Used to encode an HTML string into a plain text.
	        // taken from http://stackoverflow.com/questions/1219860/javascript-jquery-html-encoding
	        spacer.innerHTML = String(text).replace(/&/g, '&amp;')
	            .replace(/"/g, '&quot;')
	            .replace(/'/g, '&#39;')
	            .replace(/</g, '&lt;')
	            .replace(/>/g, '&gt;');
	        return spacer.getBoundingClientRect().right;
	    }

	    var rs = {
	        onArrowDown: function () { }, // defaults to no action.
	        onArrowUp: function () { },   // defaults to no action.
	        onEnter: function () { },     // defaults to no action.
	        onTab: function () { },       // defaults to no action.
	        startFrom: 0,
	        options: [],
	        element: null,
	        elementHint: null,
	        elementStyle: null,
	        wrapper: wrapper,      // Only to allow  easy access to the HTML elements to the final user (possibly for minor customizations)
	        show: function (element, startPos, options) {
	            this.startFrom = startPos;
	            this.wrapper.remove();
	            if (this.elementHint) {
	                this.elementHint.remove();
	                this.elementHint = null;
	            }
	            
	            if (fontSize == '') {
	                fontSize = window.getComputedStyle(element).getPropertyValue('font-size');
	            }
	            if (fontFamily == '') {
	                fontFamily = window.getComputedStyle(element).getPropertyValue('font-family');
	            }
	            
	            var w = element.getBoundingClientRect().right - element.getBoundingClientRect().left;
	            dropDown.style.marginLeft = '0';
	            dropDown.style.marginTop = element.getBoundingClientRect().height + 'px';
	            this.options = options;

	            if (this.element != element) {
	                this.element = element;
	                this.elementStyle = {
	                    zIndex: this.element.style.zIndex,
	                    position: this.element.style.position,
	                    backgroundColor: this.element.style.backgroundColor,
	                    borderColor: this.element.style.borderColor
	                }
	            }

	            this.element.style.zIndex = 3;
	            this.element.style.position = 'relative';
	            this.element.style.backgroundColor = 'transparent';
	            this.element.style.borderColor = 'transparent';

	            this.elementHint = element.cloneNode();
	            this.elementHint.className = 'autocomplete hint';
	            this.elementHint.style.zIndex = 2;
	            this.elementHint.style.position = 'absolute';
	            this.elementHint.onfocus = function () { this.element.focus(); }.bind(this);



	            if (this.element.addEventListener) {
	                this.element.removeEventListener("keydown", keyDownHandler);
	                this.element.addEventListener("keydown", keyDownHandler, false);
	                this.element.removeEventListener("blur", onBlurHandler);
	                this.element.addEventListener("blur", onBlurHandler, false);                
	            } 

	            wrapper.appendChild(this.elementHint);
	            wrapper.appendChild(dropDown);
	            element.parentElement.appendChild(wrapper);


	            this.repaint(element);
	        },
	        setText: function (text) {
	            this.element.innerText = text;
	        },
	        getText: function () {
	            return this.element.innerText;
	        },
	        hideDropDown: function () {
	            this.wrapper.remove();
	            if (this.elementHint) {
	                this.elementHint.remove();
	                this.elementHint = null;
	                dropDownController.hide();
	                this.element.style.zIndex = this.elementStyle.zIndex;
	                this.element.style.position = this.elementStyle.position;
	                this.element.style.backgroundColor = this.elementStyle.backgroundColor;
	                this.element.style.borderColor = this.elementStyle.borderColor;
	            }
	            
	        },
	        repaint: function (element) {
	            var text = element.innerText;
	            text = text.replace('\n', '');

	            var startFrom = this.startFrom;
	            var options = this.options;
	            var optionsLength = this.options.length;

	            // breaking text in leftSide and token.
	            
	            var token = text.substring(this.startFrom);
	            leftSide = text.substring(0, this.startFrom);
	            
	            for (var i = 0; i < optionsLength; i++) {
	                var opt = this.options[i];
	                if (   (!config.caseSensitive && opt.toLowerCase().indexOf(token.toLowerCase()) === 0)
	                    || (config.caseSensitive && opt.indexOf(token) === 0)) {   // <-- how about upperCase vs. lowercase
	                    this.elementHint.innerText = leftSide + token + opt.substring(token.length);
	                    this.elementHint.realInnerText = leftSide + opt;
	                    break;
	                }
	            }
	            // moving the dropDown and refreshing it.
	            dropDown.style.left = calculateWidthForText(leftSide) + 'px';
	            dropDownController.refresh(token, this.options);
	            this.elementHint.style.width = calculateWidthForText(this.elementHint.innerText) + 10 + 'px'
	            var wasDropDownHidden = (dropDown.style.visibility == 'hidden');
	            if (!wasDropDownHidden)
	                this.elementHint.style.width = calculateWidthForText(this.elementHint.innerText) + dropDown.clientWidth + 'px';
	        }
	    };

	    var dropDownController = createDropDownController(dropDown, rs);

	    var keyDownHandler = function (e) {
	        //console.log("Keydown:" + e.keyCode);
	        e = e || window.event;
	        var keyCode = e.keyCode;

	        if (this.elementHint == null) return;

	        if (keyCode == 33) { return; } // page up (do nothing)
	        if (keyCode == 34) { return; } // page down (do nothing);

	        if (keyCode == 27) { //escape
	            rs.hideDropDown();
	            rs.element.focus();
	            e.preventDefault();
	            e.stopPropagation();
	            return;
	        }

	        var text = this.element.innerText;
	        text = text.replace('\n', '');
	        var startFrom = this.startFrom;

	        if (config.confirmKeys.indexOf(keyCode) >= 0) { //  (autocomplete triggered)
	            if (keyCode == 9) {                 
	                if (this.elementHint.innerText.length == 0) {
	                    rs.onTab(); 
	                }
	            }
	            if (this.elementHint.innerText.length > 0) { // if there is a hint               
	                if (this.element.innerText != this.elementHint.realInnerText) {
	                    this.element.innerText = this.elementHint.realInnerText;
	                    rs.hideDropDown();
	                    setEndOfContenteditable(this.element);
	                    if (keyCode == 9) {                
	                        rs.element.focus();
	                        e.preventDefault();
	                        e.stopPropagation();
	                    }
	                }                
	            }
	            return;
	        }

	        if (keyCode == 13) {       // enter  (autocomplete triggered)
	            if (this.elementHint.innerText.length == 0) { // if there is a hint
	                rs.onEnter();
	            } else {
	                var wasDropDownHidden = (dropDown.style.visibility == 'hidden');
	                dropDownController.hide();

	                if (wasDropDownHidden) {
	                    rs.hideDropDown();
	                    rs.element.focus();
	                    rs.onEnter();
	                    return;
	                }

	                this.element.innerText = this.elementHint.realInnerText;
	                rs.hideDropDown();
	                setEndOfContenteditable(this.element);
	                e.preventDefault();
	                e.stopPropagation();
	            }
	            return;
	        }

	        if (keyCode == 40) {     // down
	            var token = text.substring(this.startFrom);
	            var m = dropDownController.move(+1);
	            if (m == '') { rs.onArrowDown(); }
	            this.elementHint.innerText = leftSide + token + m.substring(token.length);
	            this.elementHint.realInnerText = leftSide + m;
	            e.preventDefault();
	            e.stopPropagation();
	            return;
	        }

	        if (keyCode == 38) {    // up
	            var token = text.substring(this.startFrom);
	            var m = dropDownController.move(-1);
	            if (m == '') { rs.onArrowUp(); }
	            this.elementHint.innerText = leftSide + token + m.substring(token.length);
	            this.elementHint.realInnerText = leftSide + m;
	            e.preventDefault();
	            e.stopPropagation();
	            return;
	        }

	    }.bind(rs);

	    var onBlurHandler = function (e) {
	        rs.hideDropDown();
	        //console.log("Lost focus.");
	    }.bind(rs);

	    dropDownController.onmouseselection = function (text, rs) {
	        rs.element.innerText = rs.elementHint.innerText = leftSide + text;        
	        rs.hideDropDown();   
	        window.setTimeout(function () {
	            rs.element.focus();
	            setEndOfContenteditable(rs.element);  
	        }, 1);              
	    };

	    return rs;
	}

	module.exports = completely;

/***/ },
/* 65 */
/***/ function(module, exports, __webpack_require__) {

	'use strict';

	var ace = __webpack_require__(66);
	var ModeSwitcher = __webpack_require__(63);
	var util = __webpack_require__(54);

	// create a mixin with the functions for text mode
	var textmode = {};

	var MAX_ERRORS = 3; // maximum number of displayed errors at the bottom

	var DEFAULT_THEME = 'ace/theme/jsoneditor';

	/**
	 * Create a text editor
	 * @param {Element} container
	 * @param {Object} [options]   Object with options. available options:
	 *                             {String} mode             Available values:
	 *                                                       "text" (default)
	 *                                                       or "code".
	 *                             {Number} indentation      Number of indentation
	 *                                                       spaces. 2 by default.
	 *                             {function} onChange       Callback method
	 *                                                       triggered on change
	 *                             {function} onModeChange   Callback method
	 *                                                       triggered after setMode
	 *                             {function} onEditable     Determine if textarea is readOnly
	 *                                                       readOnly defaults true
	 *                             {Object} ace              A custom instance of
	 *                                                       Ace editor.
	 *                             {boolean} escapeUnicode   If true, unicode
	 *                                                       characters are escaped.
	 *                                                       false by default.
	 * @private
	 */
	textmode.create = function (container, options) {
	  // read options
	  options = options || {};
	  
	  if(typeof options.statusBar === 'undefined') {
	    options.statusBar = true;
	  }

	  this.options = options;

	  // indentation
	  if (options.indentation) {
	    this.indentation = Number(options.indentation);
	  }
	  else {
	    this.indentation = 2; // number of spaces
	  }

	  // grab ace from options if provided
	  var _ace = options.ace ? options.ace : ace;
	  // TODO: make the option options.ace deprecated, it's not needed anymore (see #309)

	  // determine mode
	  this.mode = (options.mode == 'code') ? 'code' : 'text';
	  if (this.mode == 'code') {
	    // verify whether Ace editor is available and supported
	    if (typeof _ace === 'undefined') {
	      this.mode = 'text';
	      console.warn('Failed to load Ace editor, falling back to plain text mode. Please use a JSONEditor bundle including Ace, or pass Ace as via the configuration option `ace`.');
	    }
	  }

	  // determine theme
	  this.theme = options.theme || DEFAULT_THEME;
	  if (this.theme === DEFAULT_THEME && _ace) {
	    try {
	      __webpack_require__(72);
	    }
	    catch (err) {
	      console.error(err);
	    }
	  }

	  var me = this;
	  this.container = container;
	  this.dom = {};
	  this.aceEditor = undefined;  // ace code editor
	  this.textarea = undefined;  // plain text editor (fallback when Ace is not available)
	  this.validateSchema = null;

	  // create a debounced validate function
	  this._debouncedValidate = util.debounce(this.validate.bind(this), this.DEBOUNCE_INTERVAL);

	  this.width = container.clientWidth;
	  this.height = container.clientHeight;

	  this.frame = document.createElement('div');
	  this.frame.className = 'jsoneditor jsoneditor-mode-' + this.options.mode;
	  this.frame.onclick = function (event) {
	    // prevent default submit action when the editor is located inside a form
	    event.preventDefault();
	  };
	  this.frame.onkeydown = function (event) {
	    me._onKeyDown(event);
	  };
	  
	  // create menu
	  this.menu = document.createElement('div');
	  this.menu.className = 'jsoneditor-menu';
	  this.frame.appendChild(this.menu);

	  // create format button
	  var buttonFormat = document.createElement('button');
	  buttonFormat.type = 'button';
	  buttonFormat.className = 'jsoneditor-format';
	  buttonFormat.title = 'Format JSON data, with proper indentation and line feeds (Ctrl+\\)';
	  this.menu.appendChild(buttonFormat);
	  buttonFormat.onclick = function () {
	    try {
	      me.format();
	      me._onChange();
	    }
	    catch (err) {
	      me._onError(err);
	    }
	  };

	  // create compact button
	  var buttonCompact = document.createElement('button');
	  buttonCompact.type = 'button';
	  buttonCompact.className = 'jsoneditor-compact';
	  buttonCompact.title = 'Compact JSON data, remove all whitespaces (Ctrl+Shift+\\)';
	  this.menu.appendChild(buttonCompact);
	  buttonCompact.onclick = function () {
	    try {
	      me.compact();
	      me._onChange();
	    }
	    catch (err) {
	      me._onError(err);
	    }
	  };

	  // create repair button
	  var buttonRepair = document.createElement('button');
	  buttonRepair.type = 'button';
	  buttonRepair.className = 'jsoneditor-repair';
	  buttonRepair.title = 'Repair JSON: fix quotes and escape characters, remove comments and JSONP notation, turn JavaScript objects into JSON.';
	  this.menu.appendChild(buttonRepair);
	  buttonRepair.onclick = function () {
	    try {
	      me.repair();
	      me._onChange();
	    }
	    catch (err) {
	      me._onError(err);
	    }
	  };

	  // create mode box
	  if (this.options && this.options.modes && this.options.modes.length) {
	    this.modeSwitcher = new ModeSwitcher(this.menu, this.options.modes, this.options.mode, function onSwitch(mode) {
	      // switch mode and restore focus
	      me.setMode(mode);
	      me.modeSwitcher.focus();
	    });
	  }

	  var emptyNode = {};
	  var isReadOnly = (this.options.onEditable
	  && typeof(this.options.onEditable === 'function')
	  && !this.options.onEditable(emptyNode));

	  this.content = document.createElement('div');
	  this.content.className = 'jsoneditor-outer';
	  this.frame.appendChild(this.content);

	  this.container.appendChild(this.frame);

	  if (this.mode == 'code') {
	    this.editorDom = document.createElement('div');
	    this.editorDom.style.height = '100%'; // TODO: move to css
	    this.editorDom.style.width = '100%'; // TODO: move to css
	    this.content.appendChild(this.editorDom);

	    var aceEditor = _ace.edit(this.editorDom);
	    aceEditor.$blockScrolling = Infinity;
	    aceEditor.setTheme(this.theme);
	    aceEditor.setOptions({ readOnly: isReadOnly });
	    aceEditor.setShowPrintMargin(false);
	    aceEditor.setFontSize(13);
	    aceEditor.getSession().setMode('ace/mode/json');
	    aceEditor.getSession().setTabSize(this.indentation);
	    aceEditor.getSession().setUseSoftTabs(true);
	    aceEditor.getSession().setUseWrapMode(true);
	    aceEditor.commands.bindKey('Ctrl-L', null);    // disable Ctrl+L (is used by the browser to select the address bar)
	    aceEditor.commands.bindKey('Command-L', null); // disable Ctrl+L (is used by the browser to select the address bar)
	    this.aceEditor = aceEditor;

	    // TODO: deprecated since v5.0.0. Cleanup backward compatibility some day
	    if (!this.hasOwnProperty('editor')) {
	      Object.defineProperty(this, 'editor', {
	        get: function () {
	          console.warn('Property "editor" has been renamed to "aceEditor".');
	          return me.aceEditor;
	        },
	        set: function (aceEditor) {
	          console.warn('Property "editor" has been renamed to "aceEditor".');
	          me.aceEditor = aceEditor;
	        }
	      });
	    }

	    var poweredBy = document.createElement('a');
	    poweredBy.appendChild(document.createTextNode('powered by ace'));
	    poweredBy.href = 'http://ace.ajax.org';
	    poweredBy.target = '_blank';
	    poweredBy.className = 'jsoneditor-poweredBy';
	    poweredBy.onclick = function () {
	      // TODO: this anchor falls below the margin of the content,
	      // therefore the normal a.href does not work. We use a click event
	      // for now, but this should be fixed.
	      window.open(poweredBy.href, poweredBy.target);
	    };
	    this.menu.appendChild(poweredBy);

	    // register onchange event
	    aceEditor.on('change', this._onChange.bind(this));
	    aceEditor.on('changeSelection', this._onSelect.bind(this));
	  }
	  else {
	    // load a plain text textarea
	    var textarea = document.createElement('textarea');
	    textarea.className = 'jsoneditor-text';
	    textarea.spellcheck = false;
	    this.content.appendChild(textarea);
	    this.textarea = textarea;
	    this.textarea.readOnly = isReadOnly;

	    // register onchange event
	    if (this.textarea.oninput === null) {
	      this.textarea.oninput = this._onChange.bind(this);
	    }
	    else {
	      // oninput is undefined. For IE8-
	      this.textarea.onchange = this._onChange.bind(this);
	    }

	    textarea.onselect = this._onSelect.bind(this);
	    textarea.onmousedown = this._onMouseDown.bind(this);
	    textarea.onblur = this._onBlur.bind(this);
	  }

	  var validationErrorsContainer = document.createElement('div');
	  validationErrorsContainer.className = 'validation-errors-container';
	  this.dom.validationErrorsContainer = validationErrorsContainer;
	  this.frame.appendChild(validationErrorsContainer);

	  if (options.statusBar) {
	    util.addClassName(this.content, 'has-status-bar');

	    this.curserInfoElements = {};
	    var statusBar = document.createElement('div');
	    this.dom.statusBar = statusBar;
	    statusBar.className = 'jsoneditor-statusbar';
	    this.frame.appendChild(statusBar);

	    var lnLabel = document.createElement('span');
	    lnLabel.className = 'jsoneditor-curserinfo-label';
	    lnLabel.innerText = 'Ln:';

	    var lnVal = document.createElement('span');
	    lnVal.className = 'jsoneditor-curserinfo-val';
	    lnVal.innerText = '1';

	    statusBar.appendChild(lnLabel);
	    statusBar.appendChild(lnVal);

	    var colLabel = document.createElement('span');
	    colLabel.className = 'jsoneditor-curserinfo-label';
	    colLabel.innerText = 'Col:';

	    var colVal = document.createElement('span');
	    colVal.className = 'jsoneditor-curserinfo-val';
	    colVal.innerText = '1';

	    statusBar.appendChild(colLabel);
	    statusBar.appendChild(colVal);

	    this.curserInfoElements.colVal = colVal;
	    this.curserInfoElements.lnVal = lnVal;

	    var countLabel = document.createElement('span');
	    countLabel.className = 'jsoneditor-curserinfo-label';
	    countLabel.innerText = 'characters selected';
	    countLabel.style.display = 'none';

	    var countVal = document.createElement('span');
	    countVal.className = 'jsoneditor-curserinfo-count';
	    countVal.innerText = '0';
	    countVal.style.display = 'none';

	    this.curserInfoElements.countLabel = countLabel;
	    this.curserInfoElements.countVal = countVal;

	    statusBar.appendChild(countVal);
	    statusBar.appendChild(countLabel);
	  }

	  this.setSchema(this.options.schema, this.options.schemaRefs);  
	};

	/**
	 * Handle a change:
	 * - Validate JSON schema
	 * - Send a callback to the onChange listener if provided
	 * @private
	 */
	textmode._onChange = function () {
	  // validate JSON schema (if configured)
	  this._debouncedValidate();

	  // trigger the onChange callback
	  if (this.options.onChange) {
	    try {
	      this.options.onChange();
	    }
	    catch (err) {
	      console.error('Error in onChange callback: ', err);
	    }
	  }
	};

	/**
	 * Handle text selection
	 * Calculates the cursor position and selection range and updates menu
	 * @private
	 */
	textmode._onSelect = function () {
	  if(this.options.statusBar) {
	    this._updateCursorInfoDisplay();
	  }
	};

	/**
	 * Event handler for keydown. Handles shortcut keys
	 * @param {Event} event
	 * @private
	 */
	textmode._onKeyDown = function (event) {
	  var keynum = event.which || event.keyCode;
	  var handled = false;

	  if (keynum == 220 && event.ctrlKey) {
	    if (event.shiftKey) { // Ctrl+Shift+\
	      this.compact();
	      this._onChange();
	    }
	    else { // Ctrl+\
	      this.format();
	      this._onChange();
	    }
	    handled = true;
	  }

	  if (handled) {
	    event.preventDefault();
	    event.stopPropagation();
	  }

	  this._updateCursorInfoDisplay();
	};

	/**
	 * Event handler for mousedown.
	 * @param {Event} event
	 * @private
	 */
	textmode._onMouseDown = function (event) {
	  this._updateCursorInfoDisplay();
	};

	/**
	 * Event handler for blur.
	 * @param {Event} event
	 * @private
	 */
	textmode._onBlur = function (event) {
	  this._updateCursorInfoDisplay();
	};

	/**
	 * Update the status bar cursor info
	 */
	textmode._updateCursorInfoDisplay = function () {
	  var me = this;
	  var line, col, count;

	  if(this.options.statusBar) {
	    if (this.textarea) {
	      setTimeout(function() { //this to verify we get the most updated textarea cursor selection
	        var selectionRange = util.getInputSelection(me.textarea);      
	        line = selectionRange.row;
	        col = selectionRange.col;
	        if (selectionRange.start !== selectionRange.end) {
	          count = selectionRange.end - selectionRange.start;
	        }
	        updateDisplay();
	      },0);
	      
	    } else if (this.aceEditor && this.curserInfoElements) {
	      var curserPos = this.aceEditor.getCursorPosition();
	      var selectedText = this.aceEditor.getSelectedText();

	      line = curserPos.row + 1;
	      col = curserPos.column + 1;
	      count = selectedText.length;
	      updateDisplay();
	    }
	  }

	  function updateDisplay() {

	    if (me.curserInfoElements.countVal.innerText !== count) {
	      me.curserInfoElements.countVal.innerText = count;
	      me.curserInfoElements.countVal.style.display = count ? 'inline' : 'none';
	      me.curserInfoElements.countLabel.style.display = count ? 'inline' : 'none';
	    }
	    me.curserInfoElements.lnVal.innerText = line;
	    me.curserInfoElements.colVal.innerText = col;
	  }
	};

	/**
	 * Destroy the editor. Clean up DOM, event listeners, and web workers.
	 */
	textmode.destroy = function () {
	  // remove old ace editor
	  if (this.aceEditor) {
	    this.aceEditor.destroy();
	    this.aceEditor = null;
	  }

	  if (this.frame && this.container && this.frame.parentNode == this.container) {
	    this.container.removeChild(this.frame);
	  }

	  if (this.modeSwitcher) {
	    this.modeSwitcher.destroy();
	    this.modeSwitcher = null;
	  }

	  this.textarea = null;
	  
	  this._debouncedValidate = null;
	};

	/**
	 * Compact the code in the text editor
	 */
	textmode.compact = function () {
	  var json = this.get();
	  var text = JSON.stringify(json);
	  this.setText(text);
	};

	/**
	 * Format the code in the text editor
	 */
	textmode.format = function () {
	  var json = this.get();
	  var text = JSON.stringify(json, null, this.indentation);
	  this.setText(text);
	};

	/**
	 * Repair the code in the text editor
	 */
	textmode.repair = function () {
	  var text = this.getText();
	  var sanitizedText = util.sanitize(text);
	  this.setText(sanitizedText);
	};

	/**
	 * Set focus to the formatter
	 */
	textmode.focus = function () {
	  if (this.textarea) {
	    this.textarea.focus();
	  }
	  if (this.aceEditor) {
	    this.aceEditor.focus();
	  }
	};

	/**
	 * Resize the formatter
	 */
	textmode.resize = function () {
	  if (this.aceEditor) {
	    var force = false;
	    this.aceEditor.resize(force);
	  }
	};

	/**
	 * Set json data in the formatter
	 * @param {Object} json
	 */
	textmode.set = function(json) {
	  this.setText(JSON.stringify(json, null, this.indentation));
	};

	/**
	 * Get json data from the formatter
	 * @return {Object} json
	 */
	textmode.get = function() {
	  var text = this.getText();
	  var json;

	  try {
	    json = util.parse(text); // this can throw an error
	  }
	  catch (err) {
	    // try to sanitize json, replace JavaScript notation with JSON notation
	    text = util.sanitize(text);

	    // try to parse again
	    json = util.parse(text); // this can throw an error
	  }

	  return json;
	};

	/**
	 * Get the text contents of the editor
	 * @return {String} jsonText
	 */
	textmode.getText = function() {
	  if (this.textarea) {
	    return this.textarea.value;
	  }
	  if (this.aceEditor) {
	    return this.aceEditor.getValue();
	  }
	  return '';
	};

	/**
	 * Set the text contents of the editor
	 * @param {String} jsonText
	 */
	textmode.setText = function(jsonText) {
	  var text;

	  if (this.options.escapeUnicode === true) {
	    text = util.escapeUnicodeChars(jsonText);
	  }
	  else {
	    text = jsonText;
	  }

	  if (this.textarea) {
	    this.textarea.value = text;
	  }
	  if (this.aceEditor) {
	    // prevent emitting onChange events while setting new text
	    var originalOnChange = this.options.onChange;
	    this.options.onChange = null;

	    this.aceEditor.setValue(text, -1);

	    this.options.onChange = originalOnChange;
	  }
	  // validate JSON schema
	  this.validate();
	};

	/**
	 * Validate current JSON object against the configured JSON schema
	 * Throws an exception when no JSON schema is configured
	 */
	textmode.validate = function () {
	  // clear all current errors
	  if (this.dom.validationErrors) {
	    this.dom.validationErrors.parentNode.removeChild(this.dom.validationErrors);
	    this.dom.validationErrors = null;

	    this.content.style.marginBottom = '';
	    this.content.style.paddingBottom = '';
	  }

	  var doValidate = false;
	  var errors = [];
	  var json;
	  try {
	    json = this.get(); // this can fail when there is no valid json
	    doValidate = true;
	  }
	  catch (err) {
	    // no valid JSON, don't validate
	  }

	  // only validate the JSON when parsing the JSON succeeded
	  if (doValidate && this.validateSchema) {
	    var valid = this.validateSchema(json);
	    if (!valid) {
	      errors = this.validateSchema.errors.map(function (error) {
	        return util.improveSchemaError(error);
	      });
	    }
	  }

	  if (errors.length > 0) {  
	    // limit the number of displayed errors
	    var limit = errors.length > MAX_ERRORS;
	    if (limit) {
	      errors = errors.slice(0, MAX_ERRORS);
	      var hidden = this.validateSchema.errors.length - MAX_ERRORS;
	      errors.push('(' + hidden + ' more errors...)')
	    }

	    var validationErrors = document.createElement('div');
	    validationErrors.innerHTML = '<table class="jsoneditor-text-errors">' +
	        '<tbody>' +
	        errors.map(function (error) {
	          var message;
	          if (typeof error === 'string') {
	            message = '<td colspan="2"><pre>' + error + '</pre></td>';
	          }
	          else {
	            message = '<td>' + error.dataPath + '</td>' +
	                '<td>' + error.message + '</td>';
	          }

	          return '<tr><td><button class="jsoneditor-schema-error"></button></td>' + message + '</tr>'
	        }).join('') +
	        '</tbody>' +
	        '</table>';

	    this.dom.validationErrors = validationErrors;
	    this.dom.validationErrorsContainer.appendChild(validationErrors);

	    var height = validationErrors.clientHeight +
	        (this.dom.statusBar ? this.dom.statusBar.clientHeight : 0);
	    this.content.style.marginBottom = (-height) + 'px';
	    this.content.style.paddingBottom = height + 'px';
	  }

	  // update the height of the ace editor
	  if (this.aceEditor) {
	    var force = false;
	    this.aceEditor.resize(force);
	  }
	};

	// define modes
	module.exports = [
	  {
	    mode: 'text',
	    mixin: textmode,
	    data: 'text',
	    load: textmode.format
	  },
	  {
	    mode: 'code',
	    mixin: textmode,
	    data: 'text',
	    load: textmode.format
	  }
	];


/***/ },
/* 66 */
/***/ function(module, exports, __webpack_require__) {

	var ace
	if (window.ace) {
	  // use the already loaded instance of Ace
	  ace = window.ace
	}
	else {
	  try {
	    // load brace
	    ace = __webpack_require__(67);

	    // load required Ace plugins
	    __webpack_require__(69);
	    __webpack_require__(71);
	  }
	  catch (err) {
	    // failed to load brace (can be minimalist bundle).
	    // No worries, the editor will fall back to plain text if needed.
	  }
	}

	module.exports = ace;


/***/ },
/* 67 */
/***/ function(module, exports, __webpack_require__) {

	/* ***** BEGIN LICENSE BLOCK *****
	 * Distributed under the BSD license:
	 *
	 * Copyright (c) 2010, Ajax.org B.V.
	 * All rights reserved.
	 *
	 * Redistribution and use in source and binary forms, with or without
	 * modification, are permitted provided that the following conditions are met:
	 *     * Redistributions of source code must retain the above copyright
	 *       notice, this list of conditions and the following disclaimer.
	 *     * Redistributions in binary form must reproduce the above copyright
	 *       notice, this list of conditions and the following disclaimer in the
	 *       documentation and/or other materials provided with the distribution.
	 *     * Neither the name of Ajax.org B.V. nor the
	 *       names of its contributors may be used to endorse or promote products
	 *       derived from this software without specific prior written permission.
	 *
	 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
	 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
	 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
	 * DISCLAIMED. IN NO EVENT SHALL AJAX.ORG B.V. BE LIABLE FOR ANY
	 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
	 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
	 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
	 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
	 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
	 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
	 *
	 * ***** END LICENSE BLOCK ***** */

	/**
	 * Define a module along with a payload
	 * @param module a name for the payload
	 * @param payload a function to call with (acequire, exports, module) params
	 */

	(function() {

	var ACE_NAMESPACE = "ace";

	var global = (function() { return this; })();
	if (!global && typeof window != "undefined") global = window; // strict mode


	if (!ACE_NAMESPACE && typeof acequirejs !== "undefined")
	    return;


	var define = function(module, deps, payload) {
	    if (typeof module !== "string") {
	        if (define.original)
	            define.original.apply(this, arguments);
	        else {
	            console.error("dropping module because define wasn\'t a string.");
	            console.trace();
	        }
	        return;
	    }
	    if (arguments.length == 2)
	        payload = deps;
	    if (!define.modules[module]) {
	        define.payloads[module] = payload;
	        define.modules[module] = null;
	    }
	};

	define.modules = {};
	define.payloads = {};

	/**
	 * Get at functionality define()ed using the function above
	 */
	var _acequire = function(parentId, module, callback) {
	    if (typeof module === "string") {
	        var payload = lookup(parentId, module);
	        if (payload != undefined) {
	            callback && callback();
	            return payload;
	        }
	    } else if (Object.prototype.toString.call(module) === "[object Array]") {
	        var params = [];
	        for (var i = 0, l = module.length; i < l; ++i) {
	            var dep = lookup(parentId, module[i]);
	            if (dep == undefined && acequire.original)
	                return;
	            params.push(dep);
	        }
	        return callback && callback.apply(null, params) || true;
	    }
	};

	var acequire = function(module, callback) {
	    var packagedModule = _acequire("", module, callback);
	    if (packagedModule == undefined && acequire.original)
	        return acequire.original.apply(this, arguments);
	    return packagedModule;
	};

	var normalizeModule = function(parentId, moduleName) {
	    // normalize plugin acequires
	    if (moduleName.indexOf("!") !== -1) {
	        var chunks = moduleName.split("!");
	        return normalizeModule(parentId, chunks[0]) + "!" + normalizeModule(parentId, chunks[1]);
	    }
	    // normalize relative acequires
	    if (moduleName.charAt(0) == ".") {
	        var base = parentId.split("/").slice(0, -1).join("/");
	        moduleName = base + "/" + moduleName;

	        while(moduleName.indexOf(".") !== -1 && previous != moduleName) {
	            var previous = moduleName;
	            moduleName = moduleName.replace(/\/\.\//, "/").replace(/[^\/]+\/\.\.\//, "");
	        }
	    }
	    return moduleName;
	};

	/**
	 * Internal function to lookup moduleNames and resolve them by calling the
	 * definition function if needed.
	 */
	var lookup = function(parentId, moduleName) {
	    moduleName = normalizeModule(parentId, moduleName);

	    var module = define.modules[moduleName];
	    if (!module) {
	        module = define.payloads[moduleName];
	        if (typeof module === 'function') {
	            var exports = {};
	            var mod = {
	                id: moduleName,
	                uri: '',
	                exports: exports,
	                packaged: true
	            };

	            var req = function(module, callback) {
	                return _acequire(moduleName, module, callback);
	            };

	            var returnValue = module(req, exports, mod);
	            exports = returnValue || mod.exports;
	            define.modules[moduleName] = exports;
	            delete define.payloads[moduleName];
	        }
	        module = define.modules[moduleName] = exports || module;
	    }
	    return module;
	};

	function exportAce(ns) {
	    var root = global;
	    if (ns) {
	        if (!global[ns])
	            global[ns] = {};
	        root = global[ns];
	    }

	    if (!root.define || !root.define.packaged) {
	        define.original = root.define;
	        root.define = define;
	        root.define.packaged = true;
	    }

	    if (!root.acequire || !root.acequire.packaged) {
	        acequire.original = root.acequire;
	        root.acequire = acequire;
	        root.acequire.packaged = true;
	    }
	}

	exportAce(ACE_NAMESPACE);

	})();

	ace.define("ace/lib/regexp",["require","exports","module"], function(acequire, exports, module) {
	"use strict";

	    var real = {
	            exec: RegExp.prototype.exec,
	            test: RegExp.prototype.test,
	            match: String.prototype.match,
	            replace: String.prototype.replace,
	            split: String.prototype.split
	        },
	        compliantExecNpcg = real.exec.call(/()??/, "")[1] === undefined, // check `exec` handling of nonparticipating capturing groups
	        compliantLastIndexIncrement = function () {
	            var x = /^/g;
	            real.test.call(x, "");
	            return !x.lastIndex;
	        }();

	    if (compliantLastIndexIncrement && compliantExecNpcg)
	        return;
	    RegExp.prototype.exec = function (str) {
	        var match = real.exec.apply(this, arguments),
	            name, r2;
	        if ( typeof(str) == 'string' && match) {
	            if (!compliantExecNpcg && match.length > 1 && indexOf(match, "") > -1) {
	                r2 = RegExp(this.source, real.replace.call(getNativeFlags(this), "g", ""));
	                real.replace.call(str.slice(match.index), r2, function () {
	                    for (var i = 1; i < arguments.length - 2; i++) {
	                        if (arguments[i] === undefined)
	                            match[i] = undefined;
	                    }
	                });
	            }
	            if (this._xregexp && this._xregexp.captureNames) {
	                for (var i = 1; i < match.length; i++) {
	                    name = this._xregexp.captureNames[i - 1];
	                    if (name)
	                       match[name] = match[i];
	                }
	            }
	            if (!compliantLastIndexIncrement && this.global && !match[0].length && (this.lastIndex > match.index))
	                this.lastIndex--;
	        }
	        return match;
	    };
	    if (!compliantLastIndexIncrement) {
	        RegExp.prototype.test = function (str) {
	            var match = real.exec.call(this, str);
	            if (match && this.global && !match[0].length && (this.lastIndex > match.index))
	                this.lastIndex--;
	            return !!match;
	        };
	    }

	    function getNativeFlags (regex) {
	        return (regex.global     ? "g" : "") +
	               (regex.ignoreCase ? "i" : "") +
	               (regex.multiline  ? "m" : "") +
	               (regex.extended   ? "x" : "") + // Proposed for ES4; included in AS3
	               (regex.sticky     ? "y" : "");
	    }

	    function indexOf (array, item, from) {
	        if (Array.prototype.indexOf) // Use the native array method if available
	            return array.indexOf(item, from);
	        for (var i = from || 0; i < array.length; i++) {
	            if (array[i] === item)
	                return i;
	        }
	        return -1;
	    }

	});

	ace.define("ace/lib/es5-shim",["require","exports","module"], function(acequire, exports, module) {

	function Empty() {}

	if (!Function.prototype.bind) {
	    Function.prototype.bind = function bind(that) { // .length is 1
	        var target = this;
	        if (typeof target != "function") {
	            throw new TypeError("Function.prototype.bind called on incompatible " + target);
	        }
	        var args = slice.call(arguments, 1); // for normal call
	        var bound = function () {

	            if (this instanceof bound) {

	                var result = target.apply(
	                    this,
	                    args.concat(slice.call(arguments))
	                );
	                if (Object(result) === result) {
	                    return result;
	                }
	                return this;

	            } else {
	                return target.apply(
	                    that,
	                    args.concat(slice.call(arguments))
	                );

	            }

	        };
	        if(target.prototype) {
	            Empty.prototype = target.prototype;
	            bound.prototype = new Empty();
	            Empty.prototype = null;
	        }
	        return bound;
	    };
	}
	var call = Function.prototype.call;
	var prototypeOfArray = Array.prototype;
	var prototypeOfObject = Object.prototype;
	var slice = prototypeOfArray.slice;
	var _toString = call.bind(prototypeOfObject.toString);
	var owns = call.bind(prototypeOfObject.hasOwnProperty);
	var defineGetter;
	var defineSetter;
	var lookupGetter;
	var lookupSetter;
	var supportsAccessors;
	if ((supportsAccessors = owns(prototypeOfObject, "__defineGetter__"))) {
	    defineGetter = call.bind(prototypeOfObject.__defineGetter__);
	    defineSetter = call.bind(prototypeOfObject.__defineSetter__);
	    lookupGetter = call.bind(prototypeOfObject.__lookupGetter__);
	    lookupSetter = call.bind(prototypeOfObject.__lookupSetter__);
	}
	if ([1,2].splice(0).length != 2) {
	    if(function() { // test IE < 9 to splice bug - see issue #138
	        function makeArray(l) {
	            var a = new Array(l+2);
	            a[0] = a[1] = 0;
	            return a;
	        }
	        var array = [], lengthBefore;
	        
	        array.splice.apply(array, makeArray(20));
	        array.splice.apply(array, makeArray(26));

	        lengthBefore = array.length; //46
	        array.splice(5, 0, "XXX"); // add one element

	        lengthBefore + 1 == array.length

	        if (lengthBefore + 1 == array.length) {
	            return true;// has right splice implementation without bugs
	        }
	    }()) {//IE 6/7
	        var array_splice = Array.prototype.splice;
	        Array.prototype.splice = function(start, deleteCount) {
	            if (!arguments.length) {
	                return [];
	            } else {
	                return array_splice.apply(this, [
	                    start === void 0 ? 0 : start,
	                    deleteCount === void 0 ? (this.length - start) : deleteCount
	                ].concat(slice.call(arguments, 2)))
	            }
	        };
	    } else {//IE8
	        Array.prototype.splice = function(pos, removeCount){
	            var length = this.length;
	            if (pos > 0) {
	                if (pos > length)
	                    pos = length;
	            } else if (pos == void 0) {
	                pos = 0;
	            } else if (pos < 0) {
	                pos = Math.max(length + pos, 0);
	            }

	            if (!(pos+removeCount < length))
	                removeCount = length - pos;

	            var removed = this.slice(pos, pos+removeCount);
	            var insert = slice.call(arguments, 2);
	            var add = insert.length;            
	            if (pos === length) {
	                if (add) {
	                    this.push.apply(this, insert);
	                }
	            } else {
	                var remove = Math.min(removeCount, length - pos);
	                var tailOldPos = pos + remove;
	                var tailNewPos = tailOldPos + add - remove;
	                var tailCount = length - tailOldPos;
	                var lengthAfterRemove = length - remove;

	                if (tailNewPos < tailOldPos) { // case A
	                    for (var i = 0; i < tailCount; ++i) {
	                        this[tailNewPos+i] = this[tailOldPos+i];
	                    }
	                } else if (tailNewPos > tailOldPos) { // case B
	                    for (i = tailCount; i--; ) {
	                        this[tailNewPos+i] = this[tailOldPos+i];
	                    }
	                } // else, add == remove (nothing to do)

	                if (add && pos === lengthAfterRemove) {
	                    this.length = lengthAfterRemove; // truncate array
	                    this.push.apply(this, insert);
	                } else {
	                    this.length = lengthAfterRemove + add; // reserves space
	                    for (i = 0; i < add; ++i) {
	                        this[pos+i] = insert[i];
	                    }
	                }
	            }
	            return removed;
	        };
	    }
	}
	if (!Array.isArray) {
	    Array.isArray = function isArray(obj) {
	        return _toString(obj) == "[object Array]";
	    };
	}
	var boxedString = Object("a"),
	    splitString = boxedString[0] != "a" || !(0 in boxedString);

	if (!Array.prototype.forEach) {
	    Array.prototype.forEach = function forEach(fun /*, thisp*/) {
	        var object = toObject(this),
	            self = splitString && _toString(this) == "[object String]" ?
	                this.split("") :
	                object,
	            thisp = arguments[1],
	            i = -1,
	            length = self.length >>> 0;
	        if (_toString(fun) != "[object Function]") {
	            throw new TypeError(); // TODO message
	        }

	        while (++i < length) {
	            if (i in self) {
	                fun.call(thisp, self[i], i, object);
	            }
	        }
	    };
	}
	if (!Array.prototype.map) {
	    Array.prototype.map = function map(fun /*, thisp*/) {
	        var object = toObject(this),
	            self = splitString && _toString(this) == "[object String]" ?
	                this.split("") :
	                object,
	            length = self.length >>> 0,
	            result = Array(length),
	            thisp = arguments[1];
	        if (_toString(fun) != "[object Function]") {
	            throw new TypeError(fun + " is not a function");
	        }

	        for (var i = 0; i < length; i++) {
	            if (i in self)
	                result[i] = fun.call(thisp, self[i], i, object);
	        }
	        return result;
	    };
	}
	if (!Array.prototype.filter) {
	    Array.prototype.filter = function filter(fun /*, thisp */) {
	        var object = toObject(this),
	            self = splitString && _toString(this) == "[object String]" ?
	                this.split("") :
	                    object,
	            length = self.length >>> 0,
	            result = [],
	            value,
	            thisp = arguments[1];
	        if (_toString(fun) != "[object Function]") {
	            throw new TypeError(fun + " is not a function");
	        }

	        for (var i = 0; i < length; i++) {
	            if (i in self) {
	                value = self[i];
	                if (fun.call(thisp, value, i, object)) {
	                    result.push(value);
	                }
	            }
	        }
	        return result;
	    };
	}
	if (!Array.prototype.every) {
	    Array.prototype.every = function every(fun /*, thisp */) {
	        var object = toObject(this),
	            self = splitString && _toString(this) == "[object String]" ?
	                this.split("") :
	                object,
	            length = self.length >>> 0,
	            thisp = arguments[1];
	        if (_toString(fun) != "[object Function]") {
	            throw new TypeError(fun + " is not a function");
	        }

	        for (var i = 0; i < length; i++) {
	            if (i in self && !fun.call(thisp, self[i], i, object)) {
	                return false;
	            }
	        }
	        return true;
	    };
	}
	if (!Array.prototype.some) {
	    Array.prototype.some = function some(fun /*, thisp */) {
	        var object = toObject(this),
	            self = splitString && _toString(this) == "[object String]" ?
	                this.split("") :
	                object,
	            length = self.length >>> 0,
	            thisp = arguments[1];
	        if (_toString(fun) != "[object Function]") {
	            throw new TypeError(fun + " is not a function");
	        }

	        for (var i = 0; i < length; i++) {
	            if (i in self && fun.call(thisp, self[i], i, object)) {
	                return true;
	            }
	        }
	        return false;
	    };
	}
	if (!Array.prototype.reduce) {
	    Array.prototype.reduce = function reduce(fun /*, initial*/) {
	        var object = toObject(this),
	            self = splitString && _toString(this) == "[object String]" ?
	                this.split("") :
	                object,
	            length = self.length >>> 0;
	        if (_toString(fun) != "[object Function]") {
	            throw new TypeError(fun + " is not a function");
	        }
	        if (!length && arguments.length == 1) {
	            throw new TypeError("reduce of empty array with no initial value");
	        }

	        var i = 0;
	        var result;
	        if (arguments.length >= 2) {
	            result = arguments[1];
	        } else {
	            do {
	                if (i in self) {
	                    result = self[i++];
	                    break;
	                }
	                if (++i >= length) {
	                    throw new TypeError("reduce of empty array with no initial value");
	                }
	            } while (true);
	        }

	        for (; i < length; i++) {
	            if (i in self) {
	                result = fun.call(void 0, result, self[i], i, object);
	            }
	        }

	        return result;
	    };
	}
	if (!Array.prototype.reduceRight) {
	    Array.prototype.reduceRight = function reduceRight(fun /*, initial*/) {
	        var object = toObject(this),
	            self = splitString && _toString(this) == "[object String]" ?
	                this.split("") :
	                object,
	            length = self.length >>> 0;
	        if (_toString(fun) != "[object Function]") {
	            throw new TypeError(fun + " is not a function");
	        }
	        if (!length && arguments.length == 1) {
	            throw new TypeError("reduceRight of empty array with no initial value");
	        }

	        var result, i = length - 1;
	        if (arguments.length >= 2) {
	            result = arguments[1];
	        } else {
	            do {
	                if (i in self) {
	                    result = self[i--];
	                    break;
	                }
	                if (--i < 0) {
	                    throw new TypeError("reduceRight of empty array with no initial value");
	                }
	            } while (true);
	        }

	        do {
	            if (i in this) {
	                result = fun.call(void 0, result, self[i], i, object);
	            }
	        } while (i--);

	        return result;
	    };
	}
	if (!Array.prototype.indexOf || ([0, 1].indexOf(1, 2) != -1)) {
	    Array.prototype.indexOf = function indexOf(sought /*, fromIndex */ ) {
	        var self = splitString && _toString(this) == "[object String]" ?
	                this.split("") :
	                toObject(this),
	            length = self.length >>> 0;

	        if (!length) {
	            return -1;
	        }

	        var i = 0;
	        if (arguments.length > 1) {
	            i = toInteger(arguments[1]);
	        }
	        i = i >= 0 ? i : Math.max(0, length + i);
	        for (; i < length; i++) {
	            if (i in self && self[i] === sought) {
	                return i;
	            }
	        }
	        return -1;
	    };
	}
	if (!Array.prototype.lastIndexOf || ([0, 1].lastIndexOf(0, -3) != -1)) {
	    Array.prototype.lastIndexOf = function lastIndexOf(sought /*, fromIndex */) {
	        var self = splitString && _toString(this) == "[object String]" ?
	                this.split("") :
	                toObject(this),
	            length = self.length >>> 0;

	        if (!length) {
	            return -1;
	        }
	        var i = length - 1;
	        if (arguments.length > 1) {
	            i = Math.min(i, toInteger(arguments[1]));
	        }
	        i = i >= 0 ? i : length - Math.abs(i);
	        for (; i >= 0; i--) {
	            if (i in self && sought === self[i]) {
	                return i;
	            }
	        }
	        return -1;
	    };
	}
	if (!Object.getPrototypeOf) {
	    Object.getPrototypeOf = function getPrototypeOf(object) {
	        return object.__proto__ || (
	            object.constructor ?
	            object.constructor.prototype :
	            prototypeOfObject
	        );
	    };
	}
	if (!Object.getOwnPropertyDescriptor) {
	    var ERR_NON_OBJECT = "Object.getOwnPropertyDescriptor called on a " +
	                         "non-object: ";
	    Object.getOwnPropertyDescriptor = function getOwnPropertyDescriptor(object, property) {
	        if ((typeof object != "object" && typeof object != "function") || object === null)
	            throw new TypeError(ERR_NON_OBJECT + object);
	        if (!owns(object, property))
	            return;

	        var descriptor, getter, setter;
	        descriptor =  { enumerable: true, configurable: true };
	        if (supportsAccessors) {
	            var prototype = object.__proto__;
	            object.__proto__ = prototypeOfObject;

	            var getter = lookupGetter(object, property);
	            var setter = lookupSetter(object, property);
	            object.__proto__ = prototype;

	            if (getter || setter) {
	                if (getter) descriptor.get = getter;
	                if (setter) descriptor.set = setter;
	                return descriptor;
	            }
	        }
	        descriptor.value = object[property];
	        return descriptor;
	    };
	}
	if (!Object.getOwnPropertyNames) {
	    Object.getOwnPropertyNames = function getOwnPropertyNames(object) {
	        return Object.keys(object);
	    };
	}
	if (!Object.create) {
	    var createEmpty;
	    if (Object.prototype.__proto__ === null) {
	        createEmpty = function () {
	            return { "__proto__": null };
	        };
	    } else {
	        createEmpty = function () {
	            var empty = {};
	            for (var i in empty)
	                empty[i] = null;
	            empty.constructor =
	            empty.hasOwnProperty =
	            empty.propertyIsEnumerable =
	            empty.isPrototypeOf =
	            empty.toLocaleString =
	            empty.toString =
	            empty.valueOf =
	            empty.__proto__ = null;
	            return empty;
	        }
	    }

	    Object.create = function create(prototype, properties) {
	        var object;
	        if (prototype === null) {
	            object = createEmpty();
	        } else {
	            if (typeof prototype != "object")
	                throw new TypeError("typeof prototype["+(typeof prototype)+"] != 'object'");
	            var Type = function () {};
	            Type.prototype = prototype;
	            object = new Type();
	            object.__proto__ = prototype;
	        }
	        if (properties !== void 0)
	            Object.defineProperties(object, properties);
	        return object;
	    };
	}

	function doesDefinePropertyWork(object) {
	    try {
	        Object.defineProperty(object, "sentinel", {});
	        return "sentinel" in object;
	    } catch (exception) {
	    }
	}
	if (Object.defineProperty) {
	    var definePropertyWorksOnObject = doesDefinePropertyWork({});
	    var definePropertyWorksOnDom = typeof document == "undefined" ||
	        doesDefinePropertyWork(document.createElement("div"));
	    if (!definePropertyWorksOnObject || !definePropertyWorksOnDom) {
	        var definePropertyFallback = Object.defineProperty;
	    }
	}

	if (!Object.defineProperty || definePropertyFallback) {
	    var ERR_NON_OBJECT_DESCRIPTOR = "Property description must be an object: ";
	    var ERR_NON_OBJECT_TARGET = "Object.defineProperty called on non-object: "
	    var ERR_ACCESSORS_NOT_SUPPORTED = "getters & setters can not be defined " +
	                                      "on this javascript engine";

	    Object.defineProperty = function defineProperty(object, property, descriptor) {
	        if ((typeof object != "object" && typeof object != "function") || object === null)
	            throw new TypeError(ERR_NON_OBJECT_TARGET + object);
	        if ((typeof descriptor != "object" && typeof descriptor != "function") || descriptor === null)
	            throw new TypeError(ERR_NON_OBJECT_DESCRIPTOR + descriptor);
	        if (definePropertyFallback) {
	            try {
	                return definePropertyFallback.call(Object, object, property, descriptor);
	            } catch (exception) {
	            }
	        }
	        if (owns(descriptor, "value")) {

	            if (supportsAccessors && (lookupGetter(object, property) ||
	                                      lookupSetter(object, property)))
	            {
	                var prototype = object.__proto__;
	                object.__proto__ = prototypeOfObject;
	                delete object[property];
	                object[property] = descriptor.value;
	                object.__proto__ = prototype;
	            } else {
	                object[property] = descriptor.value;
	            }
	        } else {
	            if (!supportsAccessors)
	                throw new TypeError(ERR_ACCESSORS_NOT_SUPPORTED);
	            if (owns(descriptor, "get"))
	                defineGetter(object, property, descriptor.get);
	            if (owns(descriptor, "set"))
	                defineSetter(object, property, descriptor.set);
	        }

	        return object;
	    };
	}
	if (!Object.defineProperties) {
	    Object.defineProperties = function defineProperties(object, properties) {
	        for (var property in properties) {
	            if (owns(properties, property))
	                Object.defineProperty(object, property, properties[property]);
	        }
	        return object;
	    };
	}
	if (!Object.seal) {
	    Object.seal = function seal(object) {
	        return object;
	    };
	}
	if (!Object.freeze) {
	    Object.freeze = function freeze(object) {
	        return object;
	    };
	}
	try {
	    Object.freeze(function () {});
	} catch (exception) {
	    Object.freeze = (function freeze(freezeObject) {
	        return function freeze(object) {
	            if (typeof object == "function") {
	                return object;
	            } else {
	                return freezeObject(object);
	            }
	        };
	    })(Object.freeze);
	}
	if (!Object.preventExtensions) {
	    Object.preventExtensions = function preventExtensions(object) {
	        return object;
	    };
	}
	if (!Object.isSealed) {
	    Object.isSealed = function isSealed(object) {
	        return false;
	    };
	}
	if (!Object.isFrozen) {
	    Object.isFrozen = function isFrozen(object) {
	        return false;
	    };
	}
	if (!Object.isExtensible) {
	    Object.isExtensible = function isExtensible(object) {
	        if (Object(object) === object) {
	            throw new TypeError(); // TODO message
	        }
	        var name = '';
	        while (owns(object, name)) {
	            name += '?';
	        }
	        object[name] = true;
	        var returnValue = owns(object, name);
	        delete object[name];
	        return returnValue;
	    };
	}
	if (!Object.keys) {
	    var hasDontEnumBug = true,
	        dontEnums = [
	            "toString",
	            "toLocaleString",
	            "valueOf",
	            "hasOwnProperty",
	            "isPrototypeOf",
	            "propertyIsEnumerable",
	            "constructor"
	        ],
	        dontEnumsLength = dontEnums.length;

	    for (var key in {"toString": null}) {
	        hasDontEnumBug = false;
	    }

	    Object.keys = function keys(object) {

	        if (
	            (typeof object != "object" && typeof object != "function") ||
	            object === null
	        ) {
	            throw new TypeError("Object.keys called on a non-object");
	        }

	        var keys = [];
	        for (var name in object) {
	            if (owns(object, name)) {
	                keys.push(name);
	            }
	        }

	        if (hasDontEnumBug) {
	            for (var i = 0, ii = dontEnumsLength; i < ii; i++) {
	                var dontEnum = dontEnums[i];
	                if (owns(object, dontEnum)) {
	                    keys.push(dontEnum);
	                }
	            }
	        }
	        return keys;
	    };

	}
	if (!Date.now) {
	    Date.now = function now() {
	        return new Date().getTime();
	    };
	}
	var ws = "\x09\x0A\x0B\x0C\x0D\x20\xA0\u1680\u180E\u2000\u2001\u2002\u2003" +
	    "\u2004\u2005\u2006\u2007\u2008\u2009\u200A\u202F\u205F\u3000\u2028" +
	    "\u2029\uFEFF";
	if (!String.prototype.trim || ws.trim()) {
	    ws = "[" + ws + "]";
	    var trimBeginRegexp = new RegExp("^" + ws + ws + "*"),
	        trimEndRegexp = new RegExp(ws + ws + "*$");
	    String.prototype.trim = function trim() {
	        return String(this).replace(trimBeginRegexp, "").replace(trimEndRegexp, "");
	    };
	}

	function toInteger(n) {
	    n = +n;
	    if (n !== n) { // isNaN
	        n = 0;
	    } else if (n !== 0 && n !== (1/0) && n !== -(1/0)) {
	        n = (n > 0 || -1) * Math.floor(Math.abs(n));
	    }
	    return n;
	}

	function isPrimitive(input) {
	    var type = typeof input;
	    return (
	        input === null ||
	        type === "undefined" ||
	        type === "boolean" ||
	        type === "number" ||
	        type === "string"
	    );
	}

	function toPrimitive(input) {
	    var val, valueOf, toString;
	    if (isPrimitive(input)) {
	        return input;
	    }
	    valueOf = input.valueOf;
	    if (typeof valueOf === "function") {
	        val = valueOf.call(input);
	        if (isPrimitive(val)) {
	            return val;
	        }
	    }
	    toString = input.toString;
	    if (typeof toString === "function") {
	        val = toString.call(input);
	        if (isPrimitive(val)) {
	            return val;
	        }
	    }
	    throw new TypeError();
	}
	var toObject = function (o) {
	    if (o == null) { // this matches both null and undefined
	        throw new TypeError("can't convert "+o+" to object");
	    }
	    return Object(o);
	};

	});

	ace.define("ace/lib/fixoldbrowsers",["require","exports","module","ace/lib/regexp","ace/lib/es5-shim"], function(acequire, exports, module) {
	"use strict";

	acequire("./regexp");
	acequire("./es5-shim");

	});

	ace.define("ace/lib/dom",["require","exports","module"], function(acequire, exports, module) {
	"use strict";

	var XHTML_NS = "http://www.w3.org/1999/xhtml";

	exports.getDocumentHead = function(doc) {
	    if (!doc)
	        doc = document;
	    return doc.head || doc.getElementsByTagName("head")[0] || doc.documentElement;
	};

	exports.createElement = function(tag, ns) {
	    return document.createElementNS ?
	           document.createElementNS(ns || XHTML_NS, tag) :
	           document.createElement(tag);
	};

	exports.hasCssClass = function(el, name) {
	    var classes = (el.className + "").split(/\s+/g);
	    return classes.indexOf(name) !== -1;
	};
	exports.addCssClass = function(el, name) {
	    if (!exports.hasCssClass(el, name)) {
	        el.className += " " + name;
	    }
	};
	exports.removeCssClass = function(el, name) {
	    var classes = el.className.split(/\s+/g);
	    while (true) {
	        var index = classes.indexOf(name);
	        if (index == -1) {
	            break;
	        }
	        classes.splice(index, 1);
	    }
	    el.className = classes.join(" ");
	};

	exports.toggleCssClass = function(el, name) {
	    var classes = el.className.split(/\s+/g), add = true;
	    while (true) {
	        var index = classes.indexOf(name);
	        if (index == -1) {
	            break;
	        }
	        add = false;
	        classes.splice(index, 1);
	    }
	    if (add)
	        classes.push(name);

	    el.className = classes.join(" ");
	    return add;
	};
	exports.setCssClass = function(node, className, include) {
	    if (include) {
	        exports.addCssClass(node, className);
	    } else {
	        exports.removeCssClass(node, className);
	    }
	};

	exports.hasCssString = function(id, doc) {
	    var index = 0, sheets;
	    doc = doc || document;

	    if (doc.createStyleSheet && (sheets = doc.styleSheets)) {
	        while (index < sheets.length)
	            if (sheets[index++].owningElement.id === id) return true;
	    } else if ((sheets = doc.getElementsByTagName("style"))) {
	        while (index < sheets.length)
	            if (sheets[index++].id === id) return true;
	    }

	    return false;
	};

	exports.importCssString = function importCssString(cssText, id, doc) {
	    doc = doc || document;
	    if (id && exports.hasCssString(id, doc))
	        return null;
	    
	    var style;
	    
	    if (id)
	        cssText += "\n/*# sourceURL=ace/css/" + id + " */";
	    
	    if (doc.createStyleSheet) {
	        style = doc.createStyleSheet();
	        style.cssText = cssText;
	        if (id)
	            style.owningElement.id = id;
	    } else {
	        style = exports.createElement("style");
	        style.appendChild(doc.createTextNode(cssText));
	        if (id)
	            style.id = id;

	        exports.getDocumentHead(doc).appendChild(style);
	    }
	};

	exports.importCssStylsheet = function(uri, doc) {
	    if (doc.createStyleSheet) {
	        doc.createStyleSheet(uri);
	    } else {
	        var link = exports.createElement('link');
	        link.rel = 'stylesheet';
	        link.href = uri;

	        exports.getDocumentHead(doc).appendChild(link);
	    }
	};

	exports.getInnerWidth = function(element) {
	    return (
	        parseInt(exports.computedStyle(element, "paddingLeft"), 10) +
	        parseInt(exports.computedStyle(element, "paddingRight"), 10) + 
	        element.clientWidth
	    );
	};

	exports.getInnerHeight = function(element) {
	    return (
	        parseInt(exports.computedStyle(element, "paddingTop"), 10) +
	        parseInt(exports.computedStyle(element, "paddingBottom"), 10) +
	        element.clientHeight
	    );
	};

	exports.scrollbarWidth = function(document) {
	    var inner = exports.createElement("ace_inner");
	    inner.style.width = "100%";
	    inner.style.minWidth = "0px";
	    inner.style.height = "200px";
	    inner.style.display = "block";

	    var outer = exports.createElement("ace_outer");
	    var style = outer.style;

	    style.position = "absolute";
	    style.left = "-10000px";
	    style.overflow = "hidden";
	    style.width = "200px";
	    style.minWidth = "0px";
	    style.height = "150px";
	    style.display = "block";

	    outer.appendChild(inner);

	    var body = document.documentElement;
	    body.appendChild(outer);

	    var noScrollbar = inner.offsetWidth;

	    style.overflow = "scroll";
	    var withScrollbar = inner.offsetWidth;

	    if (noScrollbar == withScrollbar) {
	        withScrollbar = outer.clientWidth;
	    }

	    body.removeChild(outer);

	    return noScrollbar-withScrollbar;
	};

	if (typeof document == "undefined") {
	    exports.importCssString = function() {};
	    return;
	}

	if (window.pageYOffset !== undefined) {
	    exports.getPageScrollTop = function() {
	        return window.pageYOffset;
	    };

	    exports.getPageScrollLeft = function() {
	        return window.pageXOffset;
	    };
	}
	else {
	    exports.getPageScrollTop = function() {
	        return document.body.scrollTop;
	    };

	    exports.getPageScrollLeft = function() {
	        return document.body.scrollLeft;
	    };
	}

	if (window.getComputedStyle)
	    exports.computedStyle = function(element, style) {
	        if (style)
	            return (window.getComputedStyle(element, "") || {})[style] || "";
	        return window.getComputedStyle(element, "") || {};
	    };
	else
	    exports.computedStyle = function(element, style) {
	        if (style)
	            return element.currentStyle[style];
	        return element.currentStyle;
	    };
	exports.setInnerHtml = function(el, innerHtml) {
	    var element = el.cloneNode(false);//document.createElement("div");
	    element.innerHTML = innerHtml;
	    el.parentNode.replaceChild(element, el);
	    return element;
	};

	if ("textContent" in document.documentElement) {
	    exports.setInnerText = function(el, innerText) {
	        el.textContent = innerText;
	    };

	    exports.getInnerText = function(el) {
	        return el.textContent;
	    };
	}
	else {
	    exports.setInnerText = function(el, innerText) {
	        el.innerText = innerText;
	    };

	    exports.getInnerText = function(el) {
	        return el.innerText;
	    };
	}

	exports.getParentWindow = function(document) {
	    return document.defaultView || document.parentWindow;
	};

	});

	ace.define("ace/lib/oop",["require","exports","module"], function(acequire, exports, module) {
	"use strict";

	exports.inherits = function(ctor, superCtor) {
	    ctor.super_ = superCtor;
	    ctor.prototype = Object.create(superCtor.prototype, {
	        constructor: {
	            value: ctor,
	            enumerable: false,
	            writable: true,
	            configurable: true
	        }
	    });
	};

	exports.mixin = function(obj, mixin) {
	    for (var key in mixin) {
	        obj[key] = mixin[key];
	    }
	    return obj;
	};

	exports.implement = function(proto, mixin) {
	    exports.mixin(proto, mixin);
	};

	});

	ace.define("ace/lib/keys",["require","exports","module","ace/lib/fixoldbrowsers","ace/lib/oop"], function(acequire, exports, module) {
	"use strict";

	acequire("./fixoldbrowsers");

	var oop = acequire("./oop");
	var Keys = (function() {
	    var ret = {
	        MODIFIER_KEYS: {
	            16: 'Shift', 17: 'Ctrl', 18: 'Alt', 224: 'Meta'
	        },

	        KEY_MODS: {
	            "ctrl": 1, "alt": 2, "option" : 2, "shift": 4,
	            "super": 8, "meta": 8, "command": 8, "cmd": 8
	        },

	        FUNCTION_KEYS : {
	            8  : "Backspace",
	            9  : "Tab",
	            13 : "Return",
	            19 : "Pause",
	            27 : "Esc",
	            32 : "Space",
	            33 : "PageUp",
	            34 : "PageDown",
	            35 : "End",
	            36 : "Home",
	            37 : "Left",
	            38 : "Up",
	            39 : "Right",
	            40 : "Down",
	            44 : "Print",
	            45 : "Insert",
	            46 : "Delete",
	            96 : "Numpad0",
	            97 : "Numpad1",
	            98 : "Numpad2",
	            99 : "Numpad3",
	            100: "Numpad4",
	            101: "Numpad5",
	            102: "Numpad6",
	            103: "Numpad7",
	            104: "Numpad8",
	            105: "Numpad9",
	            '-13': "NumpadEnter",
	            112: "F1",
	            113: "F2",
	            114: "F3",
	            115: "F4",
	            116: "F5",
	            117: "F6",
	            118: "F7",
	            119: "F8",
	            120: "F9",
	            121: "F10",
	            122: "F11",
	            123: "F12",
	            144: "Numlock",
	            145: "Scrolllock"
	        },

	        PRINTABLE_KEYS: {
	           32: ' ',  48: '0',  49: '1',  50: '2',  51: '3',  52: '4', 53:  '5',
	           54: '6',  55: '7',  56: '8',  57: '9',  59: ';',  61: '=', 65:  'a',
	           66: 'b',  67: 'c',  68: 'd',  69: 'e',  70: 'f',  71: 'g', 72:  'h',
	           73: 'i',  74: 'j',  75: 'k',  76: 'l',  77: 'm',  78: 'n', 79:  'o',
	           80: 'p',  81: 'q',  82: 'r',  83: 's',  84: 't',  85: 'u', 86:  'v',
	           87: 'w',  88: 'x',  89: 'y',  90: 'z', 107: '+', 109: '-', 110: '.',
	          186: ';', 187: '=', 188: ',', 189: '-', 190: '.', 191: '/', 192: '`',
	          219: '[', 220: '\\',221: ']', 222: "'", 111: '/', 106: '*'
	        }
	    };
	    var name, i;
	    for (i in ret.FUNCTION_KEYS) {
	        name = ret.FUNCTION_KEYS[i].toLowerCase();
	        ret[name] = parseInt(i, 10);
	    }
	    for (i in ret.PRINTABLE_KEYS) {
	        name = ret.PRINTABLE_KEYS[i].toLowerCase();
	        ret[name] = parseInt(i, 10);
	    }
	    oop.mixin(ret, ret.MODIFIER_KEYS);
	    oop.mixin(ret, ret.PRINTABLE_KEYS);
	    oop.mixin(ret, ret.FUNCTION_KEYS);
	    ret.enter = ret["return"];
	    ret.escape = ret.esc;
	    ret.del = ret["delete"];
	    ret[173] = '-';
	    
	    (function() {
	        var mods = ["cmd", "ctrl", "alt", "shift"];
	        for (var i = Math.pow(2, mods.length); i--;) {            
	            ret.KEY_MODS[i] = mods.filter(function(x) {
	                return i & ret.KEY_MODS[x];
	            }).join("-") + "-";
	        }
	    })();

	    ret.KEY_MODS[0] = "";
	    ret.KEY_MODS[-1] = "input-";

	    return ret;
	})();
	oop.mixin(exports, Keys);

	exports.keyCodeToString = function(keyCode) {
	    var keyString = Keys[keyCode];
	    if (typeof keyString != "string")
	        keyString = String.fromCharCode(keyCode);
	    return keyString.toLowerCase();
	};

	});

	ace.define("ace/lib/useragent",["require","exports","module"], function(acequire, exports, module) {
	"use strict";
	exports.OS = {
	    LINUX: "LINUX",
	    MAC: "MAC",
	    WINDOWS: "WINDOWS"
	};
	exports.getOS = function() {
	    if (exports.isMac) {
	        return exports.OS.MAC;
	    } else if (exports.isLinux) {
	        return exports.OS.LINUX;
	    } else {
	        return exports.OS.WINDOWS;
	    }
	};
	if (typeof navigator != "object")
	    return;

	var os = (navigator.platform.match(/mac|win|linux/i) || ["other"])[0].toLowerCase();
	var ua = navigator.userAgent;
	exports.isWin = (os == "win");
	exports.isMac = (os == "mac");
	exports.isLinux = (os == "linux");
	exports.isIE = 
	    (navigator.appName == "Microsoft Internet Explorer" || navigator.appName.indexOf("MSAppHost") >= 0)
	    ? parseFloat((ua.match(/(?:MSIE |Trident\/[0-9]+[\.0-9]+;.*rv:)([0-9]+[\.0-9]+)/)||[])[1])
	    : parseFloat((ua.match(/(?:Trident\/[0-9]+[\.0-9]+;.*rv:)([0-9]+[\.0-9]+)/)||[])[1]); // for ie
	    
	exports.isOldIE = exports.isIE && exports.isIE < 9;
	exports.isGecko = exports.isMozilla = (window.Controllers || window.controllers) && window.navigator.product === "Gecko";
	exports.isOldGecko = exports.isGecko && parseInt((ua.match(/rv:(\d+)/)||[])[1], 10) < 4;
	exports.isOpera = window.opera && Object.prototype.toString.call(window.opera) == "[object Opera]";
	exports.isWebKit = parseFloat(ua.split("WebKit/")[1]) || undefined;

	exports.isChrome = parseFloat(ua.split(" Chrome/")[1]) || undefined;

	exports.isAIR = ua.indexOf("AdobeAIR") >= 0;

	exports.isIPad = ua.indexOf("iPad") >= 0;

	exports.isChromeOS = ua.indexOf(" CrOS ") >= 0;

	exports.isIOS = /iPad|iPhone|iPod/.test(ua) && !window.MSStream;

	if (exports.isIOS) exports.isMac = true;

	});

	ace.define("ace/lib/event",["require","exports","module","ace/lib/keys","ace/lib/useragent"], function(acequire, exports, module) {
	"use strict";

	var keys = acequire("./keys");
	var useragent = acequire("./useragent");

	var pressedKeys = null;
	var ts = 0;

	exports.addListener = function(elem, type, callback) {
	    if (elem.addEventListener) {
	        return elem.addEventListener(type, callback, false);
	    }
	    if (elem.attachEvent) {
	        var wrapper = function() {
	            callback.call(elem, window.event);
	        };
	        callback._wrapper = wrapper;
	        elem.attachEvent("on" + type, wrapper);
	    }
	};

	exports.removeListener = function(elem, type, callback) {
	    if (elem.removeEventListener) {
	        return elem.removeEventListener(type, callback, false);
	    }
	    if (elem.detachEvent) {
	        elem.detachEvent("on" + type, callback._wrapper || callback);
	    }
	};
	exports.stopEvent = function(e) {
	    exports.stopPropagation(e);
	    exports.preventDefault(e);
	    return false;
	};

	exports.stopPropagation = function(e) {
	    if (e.stopPropagation)
	        e.stopPropagation();
	    else
	        e.cancelBubble = true;
	};

	exports.preventDefault = function(e) {
	    if (e.preventDefault)
	        e.preventDefault();
	    else
	        e.returnValue = false;
	};
	exports.getButton = function(e) {
	    if (e.type == "dblclick")
	        return 0;
	    if (e.type == "contextmenu" || (useragent.isMac && (e.ctrlKey && !e.altKey && !e.shiftKey)))
	        return 2;
	    if (e.preventDefault) {
	        return e.button;
	    }
	    else {
	        return {1:0, 2:2, 4:1}[e.button];
	    }
	};

	exports.capture = function(el, eventHandler, releaseCaptureHandler) {
	    function onMouseUp(e) {
	        eventHandler && eventHandler(e);
	        releaseCaptureHandler && releaseCaptureHandler(e);

	        exports.removeListener(document, "mousemove", eventHandler, true);
	        exports.removeListener(document, "mouseup", onMouseUp, true);
	        exports.removeListener(document, "dragstart", onMouseUp, true);
	    }

	    exports.addListener(document, "mousemove", eventHandler, true);
	    exports.addListener(document, "mouseup", onMouseUp, true);
	    exports.addListener(document, "dragstart", onMouseUp, true);
	    
	    return onMouseUp;
	};

	exports.addTouchMoveListener = function (el, callback) {
	    var startx, starty;
	    exports.addListener(el, "touchstart", function (e) {
	        var touches = e.touches;
	        var touchObj = touches[0];
	        startx = touchObj.clientX;
	        starty = touchObj.clientY;
	    });
	    exports.addListener(el, "touchmove", function (e) {
	        var touches = e.touches;
	        if (touches.length > 1) return;

	        var touchObj = touches[0];

	        e.wheelX = startx - touchObj.clientX;
	        e.wheelY = starty - touchObj.clientY;

	        startx = touchObj.clientX;
	        starty = touchObj.clientY;

	        callback(e);
	    });
	};

	exports.addMouseWheelListener = function(el, callback) {
	    if ("onmousewheel" in el) {
	        exports.addListener(el, "mousewheel", function(e) {
	            var factor = 8;
	            if (e.wheelDeltaX !== undefined) {
	                e.wheelX = -e.wheelDeltaX / factor;
	                e.wheelY = -e.wheelDeltaY / factor;
	            } else {
	                e.wheelX = 0;
	                e.wheelY = -e.wheelDelta / factor;
	            }
	            callback(e);
	        });
	    } else if ("onwheel" in el) {
	        exports.addListener(el, "wheel",  function(e) {
	            var factor = 0.35;
	            switch (e.deltaMode) {
	                case e.DOM_DELTA_PIXEL:
	                    e.wheelX = e.deltaX * factor || 0;
	                    e.wheelY = e.deltaY * factor || 0;
	                    break;
	                case e.DOM_DELTA_LINE:
	                case e.DOM_DELTA_PAGE:
	                    e.wheelX = (e.deltaX || 0) * 5;
	                    e.wheelY = (e.deltaY || 0) * 5;
	                    break;
	            }
	            
	            callback(e);
	        });
	    } else {
	        exports.addListener(el, "DOMMouseScroll", function(e) {
	            if (e.axis && e.axis == e.HORIZONTAL_AXIS) {
	                e.wheelX = (e.detail || 0) * 5;
	                e.wheelY = 0;
	            } else {
	                e.wheelX = 0;
	                e.wheelY = (e.detail || 0) * 5;
	            }
	            callback(e);
	        });
	    }
	};

	exports.addMultiMouseDownListener = function(elements, timeouts, eventHandler, callbackName) {
	    var clicks = 0;
	    var startX, startY, timer; 
	    var eventNames = {
	        2: "dblclick",
	        3: "tripleclick",
	        4: "quadclick"
	    };

	    function onMousedown(e) {
	        if (exports.getButton(e) !== 0) {
	            clicks = 0;
	        } else if (e.detail > 1) {
	            clicks++;
	            if (clicks > 4)
	                clicks = 1;
	        } else {
	            clicks = 1;
	        }
	        if (useragent.isIE) {
	            var isNewClick = Math.abs(e.clientX - startX) > 5 || Math.abs(e.clientY - startY) > 5;
	            if (!timer || isNewClick)
	                clicks = 1;
	            if (timer)
	                clearTimeout(timer);
	            timer = setTimeout(function() {timer = null;}, timeouts[clicks - 1] || 600);

	            if (clicks == 1) {
	                startX = e.clientX;
	                startY = e.clientY;
	            }
	        }
	        
	        e._clicks = clicks;

	        eventHandler[callbackName]("mousedown", e);

	        if (clicks > 4)
	            clicks = 0;
	        else if (clicks > 1)
	            return eventHandler[callbackName](eventNames[clicks], e);
	    }
	    function onDblclick(e) {
	        clicks = 2;
	        if (timer)
	            clearTimeout(timer);
	        timer = setTimeout(function() {timer = null;}, timeouts[clicks - 1] || 600);
	        eventHandler[callbackName]("mousedown", e);
	        eventHandler[callbackName](eventNames[clicks], e);
	    }
	    if (!Array.isArray(elements))
	        elements = [elements];
	    elements.forEach(function(el) {
	        exports.addListener(el, "mousedown", onMousedown);
	        if (useragent.isOldIE)
	            exports.addListener(el, "dblclick", onDblclick);
	    });
	};

	var getModifierHash = useragent.isMac && useragent.isOpera && !("KeyboardEvent" in window)
	    ? function(e) {
	        return 0 | (e.metaKey ? 1 : 0) | (e.altKey ? 2 : 0) | (e.shiftKey ? 4 : 0) | (e.ctrlKey ? 8 : 0);
	    }
	    : function(e) {
	        return 0 | (e.ctrlKey ? 1 : 0) | (e.altKey ? 2 : 0) | (e.shiftKey ? 4 : 0) | (e.metaKey ? 8 : 0);
	    };

	exports.getModifierString = function(e) {
	    return keys.KEY_MODS[getModifierHash(e)];
	};

	function normalizeCommandKeys(callback, e, keyCode) {
	    var hashId = getModifierHash(e);

	    if (!useragent.isMac && pressedKeys) {
	        if (e.getModifierState && (e.getModifierState("OS") || e.getModifierState("Win")))
	            hashId |= 8;
	        if (pressedKeys.altGr) {
	            if ((3 & hashId) != 3)
	                pressedKeys.altGr = 0;
	            else
	                return;
	        }
	        if (keyCode === 18 || keyCode === 17) {
	            var location = "location" in e ? e.location : e.keyLocation;
	            if (keyCode === 17 && location === 1) {
	                if (pressedKeys[keyCode] == 1)
	                    ts = e.timeStamp;
	            } else if (keyCode === 18 && hashId === 3 && location === 2) {
	                var dt = e.timeStamp - ts;
	                if (dt < 50)
	                    pressedKeys.altGr = true;
	            }
	        }
	    }
	    
	    if (keyCode in keys.MODIFIER_KEYS) {
	        keyCode = -1;
	    }
	    if (hashId & 8 && (keyCode >= 91 && keyCode <= 93)) {
	        keyCode = -1;
	    }
	    
	    if (!hashId && keyCode === 13) {
	        var location = "location" in e ? e.location : e.keyLocation;
	        if (location === 3) {
	            callback(e, hashId, -keyCode);
	            if (e.defaultPrevented)
	                return;
	        }
	    }
	    
	    if (useragent.isChromeOS && hashId & 8) {
	        callback(e, hashId, keyCode);
	        if (e.defaultPrevented)
	            return;
	        else
	            hashId &= ~8;
	    }
	    if (!hashId && !(keyCode in keys.FUNCTION_KEYS) && !(keyCode in keys.PRINTABLE_KEYS)) {
	        return false;
	    }
	    
	    return callback(e, hashId, keyCode);
	}


	exports.addCommandKeyListener = function(el, callback) {
	    var addListener = exports.addListener;
	    if (useragent.isOldGecko || (useragent.isOpera && !("KeyboardEvent" in window))) {
	        var lastKeyDownKeyCode = null;
	        addListener(el, "keydown", function(e) {
	            lastKeyDownKeyCode = e.keyCode;
	        });
	        addListener(el, "keypress", function(e) {
	            return normalizeCommandKeys(callback, e, lastKeyDownKeyCode);
	        });
	    } else {
	        var lastDefaultPrevented = null;

	        addListener(el, "keydown", function(e) {
	            pressedKeys[e.keyCode] = (pressedKeys[e.keyCode] || 0) + 1;
	            var result = normalizeCommandKeys(callback, e, e.keyCode);
	            lastDefaultPrevented = e.defaultPrevented;
	            return result;
	        });

	        addListener(el, "keypress", function(e) {
	            if (lastDefaultPrevented && (e.ctrlKey || e.altKey || e.shiftKey || e.metaKey)) {
	                exports.stopEvent(e);
	                lastDefaultPrevented = null;
	            }
	        });

	        addListener(el, "keyup", function(e) {
	            pressedKeys[e.keyCode] = null;
	        });

	        if (!pressedKeys) {
	            resetPressedKeys();
	            addListener(window, "focus", resetPressedKeys);
	        }
	    }
	};
	function resetPressedKeys() {
	    pressedKeys = Object.create(null);
	}

	if (typeof window == "object" && window.postMessage && !useragent.isOldIE) {
	    var postMessageId = 1;
	    exports.nextTick = function(callback, win) {
	        win = win || window;
	        var messageName = "zero-timeout-message-" + postMessageId;
	        exports.addListener(win, "message", function listener(e) {
	            if (e.data == messageName) {
	                exports.stopPropagation(e);
	                exports.removeListener(win, "message", listener);
	                callback();
	            }
	        });
	        win.postMessage(messageName, "*");
	    };
	}


	exports.nextFrame = typeof window == "object" && (window.requestAnimationFrame
	    || window.mozRequestAnimationFrame
	    || window.webkitRequestAnimationFrame
	    || window.msRequestAnimationFrame
	    || window.oRequestAnimationFrame);

	if (exports.nextFrame)
	    exports.nextFrame = exports.nextFrame.bind(window);
	else
	    exports.nextFrame = function(callback) {
	        setTimeout(callback, 17);
	    };
	});

	ace.define("ace/lib/lang",["require","exports","module"], function(acequire, exports, module) {
	"use strict";

	exports.last = function(a) {
	    return a[a.length - 1];
	};

	exports.stringReverse = function(string) {
	    return string.split("").reverse().join("");
	};

	exports.stringRepeat = function (string, count) {
	    var result = '';
	    while (count > 0) {
	        if (count & 1)
	            result += string;

	        if (count >>= 1)
	            string += string;
	    }
	    return result;
	};

	var trimBeginRegexp = /^\s\s*/;
	var trimEndRegexp = /\s\s*$/;

	exports.stringTrimLeft = function (string) {
	    return string.replace(trimBeginRegexp, '');
	};

	exports.stringTrimRight = function (string) {
	    return string.replace(trimEndRegexp, '');
	};

	exports.copyObject = function(obj) {
	    var copy = {};
	    for (var key in obj) {
	        copy[key] = obj[key];
	    }
	    return copy;
	};

	exports.copyArray = function(array){
	    var copy = [];
	    for (var i=0, l=array.length; i<l; i++) {
	        if (array[i] && typeof array[i] == "object")
	            copy[i] = this.copyObject(array[i]);
	        else 
	            copy[i] = array[i];
	    }
	    return copy;
	};

	exports.deepCopy = function deepCopy(obj) {
	    if (typeof obj !== "object" || !obj)
	        return obj;
	    var copy;
	    if (Array.isArray(obj)) {
	        copy = [];
	        for (var key = 0; key < obj.length; key++) {
	            copy[key] = deepCopy(obj[key]);
	        }
	        return copy;
	    }
	    if (Object.prototype.toString.call(obj) !== "[object Object]")
	        return obj;
	    
	    copy = {};
	    for (var key in obj)
	        copy[key] = deepCopy(obj[key]);
	    return copy;
	};

	exports.arrayToMap = function(arr) {
	    var map = {};
	    for (var i=0; i<arr.length; i++) {
	        map[arr[i]] = 1;
	    }
	    return map;

	};

	exports.createMap = function(props) {
	    var map = Object.create(null);
	    for (var i in props) {
	        map[i] = props[i];
	    }
	    return map;
	};
	exports.arrayRemove = function(array, value) {
	  for (var i = 0; i <= array.length; i++) {
	    if (value === array[i]) {
	      array.splice(i, 1);
	    }
	  }
	};

	exports.escapeRegExp = function(str) {
	    return str.replace(/([.*+?^${}()|[\]\/\\])/g, '\\$1');
	};

	exports.escapeHTML = function(str) {
	    return str.replace(/&/g, "&#38;").replace(/"/g, "&#34;").replace(/'/g, "&#39;").replace(/</g, "&#60;");
	};

	exports.getMatchOffsets = function(string, regExp) {
	    var matches = [];

	    string.replace(regExp, function(str) {
	        matches.push({
	            offset: arguments[arguments.length-2],
	            length: str.length
	        });
	    });

	    return matches;
	};
	exports.deferredCall = function(fcn) {
	    var timer = null;
	    var callback = function() {
	        timer = null;
	        fcn();
	    };

	    var deferred = function(timeout) {
	        deferred.cancel();
	        timer = setTimeout(callback, timeout || 0);
	        return deferred;
	    };

	    deferred.schedule = deferred;

	    deferred.call = function() {
	        this.cancel();
	        fcn();
	        return deferred;
	    };

	    deferred.cancel = function() {
	        clearTimeout(timer);
	        timer = null;
	        return deferred;
	    };
	    
	    deferred.isPending = function() {
	        return timer;
	    };

	    return deferred;
	};


	exports.delayedCall = function(fcn, defaultTimeout) {
	    var timer = null;
	    var callback = function() {
	        timer = null;
	        fcn();
	    };

	    var _self = function(timeout) {
	        if (timer == null)
	            timer = setTimeout(callback, timeout || defaultTimeout);
	    };

	    _self.delay = function(timeout) {
	        timer && clearTimeout(timer);
	        timer = setTimeout(callback, timeout || defaultTimeout);
	    };
	    _self.schedule = _self;

	    _self.call = function() {
	        this.cancel();
	        fcn();
	    };

	    _self.cancel = function() {
	        timer && clearTimeout(timer);
	        timer = null;
	    };

	    _self.isPending = function() {
	        return timer;
	    };

	    return _self;
	};
	});

	ace.define("ace/keyboard/textinput_ios",["require","exports","module","ace/lib/event","ace/lib/useragent","ace/lib/dom","ace/lib/lang","ace/lib/keys"], function(acequire, exports, module) {
	"use strict";

	var event = acequire("../lib/event");
	var useragent = acequire("../lib/useragent");
	var dom = acequire("../lib/dom");
	var lang = acequire("../lib/lang");
	var KEYS = acequire("../lib/keys");
	var MODS = KEYS.KEY_MODS;
	var BROKEN_SETDATA = useragent.isChrome < 18;
	var USE_IE_MIME_TYPE =  useragent.isIE;

	var TextInput = function(parentNode, host) {
	    var self = this;
	    var text = dom.createElement("textarea");
	    text.className = useragent.isIOS ? "ace_text-input ace_text-input-ios" : "ace_text-input";

	    if (useragent.isTouchPad)
	        text.setAttribute("x-palm-disable-auto-cap", true);

	    text.setAttribute("wrap", "off");
	    text.setAttribute("autocorrect", "off");
	    text.setAttribute("autocapitalize", "off");
	    text.setAttribute("spellcheck", false);

	    text.style.opacity = "0";
	    parentNode.insertBefore(text, parentNode.firstChild);

	    var PLACEHOLDER = "\n aaaa a\n";

	    var copied = false;
	    var cut = false;
	    var pasted = false;
	    var inComposition = false;
	    var tempStyle = '';
	    var isSelectionEmpty = true;
	    try { var isFocused = document.activeElement === text; } catch(e) {}

	    event.addListener(text, "blur", function(e) {
	        host.onBlur(e);
	        isFocused = false;
	    });
	    event.addListener(text, "focus", function(e) {
	        isFocused = true;
	        host.onFocus(e);
	        resetSelection();
	    });
	    this.focus = function() {
	        if (tempStyle) return text.focus();
	        text.style.position = "fixed";
	        text.focus();
	    };
	    this.blur = function() {
	        text.blur();
	    };
	    this.isFocused = function() {
	        return isFocused;
	    };
	    var syncSelection = lang.delayedCall(function() {
	        isFocused && resetSelection(isSelectionEmpty);
	    });
	    var syncValue = lang.delayedCall(function() {
	         if (!inComposition) {
	            text.value = PLACEHOLDER;
	            isFocused && resetSelection();
	         }
	    });

	    function resetSelection(isEmpty) {
	        if (inComposition)
	            return;
	        inComposition = true;

	        if (inputHandler) {
	            selectionStart = 0;
	            selectionEnd = isEmpty ? 0 : text.value.length - 1;
	        } else {
	            var selectionStart = 4;
	            var selectionEnd = 5;
	        }
	        try {
	            text.setSelectionRange(selectionStart, selectionEnd);
	        } catch(e) {}

	        inComposition = false;
	    }

	    function resetValue() {
	        if (inComposition)
	            return;
	        text.value = PLACEHOLDER;
	        if (useragent.isWebKit)
	            syncValue.schedule();
	    }

	    useragent.isWebKit || host.addEventListener('changeSelection', function() {
	        if (host.selection.isEmpty() != isSelectionEmpty) {
	            isSelectionEmpty = !isSelectionEmpty;
	            syncSelection.schedule();
	        }
	    });

	    resetValue();
	    if (isFocused)
	        host.onFocus();


	    var isAllSelected = function(text) {
	        return text.selectionStart === 0 && text.selectionEnd === text.value.length;
	    };

	    var onSelect = function(e) {
	        if (isAllSelected(text)) {
	            host.selectAll();
	            resetSelection();
	        } else if (inputHandler) {
	            resetSelection(host.selection.isEmpty());
	        }
	    };

	    var inputHandler = null;
	    this.setInputHandler = function(cb) {inputHandler = cb;};
	    this.getInputHandler = function() {return inputHandler;};
	    var afterContextMenu = false;

	    var sendText = function(data) {
	        if (text.selectionStart === 4 && text.selectionEnd === 5) {
	          return;
	        }
	        if (inputHandler) {
	            data = inputHandler(data);
	            inputHandler = null;
	        }
	        if (pasted) {
	            resetSelection();
	            if (data)
	                host.onPaste(data);
	            pasted = false;
	        } else if (data == PLACEHOLDER.substr(0) && text.selectionStart === 4) {
	            if (afterContextMenu)
	                host.execCommand("del", {source: "ace"});
	            else // some versions of android do not fire keydown when pressing backspace
	                host.execCommand("backspace", {source: "ace"});
	        } else if (!copied) {
	            if (data.substring(0, 9) == PLACEHOLDER && data.length > PLACEHOLDER.length)
	                data = data.substr(9);
	            else if (data.substr(0, 4) == PLACEHOLDER.substr(0, 4))
	                data = data.substr(4, data.length - PLACEHOLDER.length + 1);
	            else if (data.charAt(data.length - 1) == PLACEHOLDER.charAt(0))
	                data = data.slice(0, -1);
	            if (data == PLACEHOLDER.charAt(0)) {
	            } else if (data.charAt(data.length - 1) == PLACEHOLDER.charAt(0))
	                data = data.slice(0, -1);

	            if (data)
	                host.onTextInput(data);
	        }
	        if (copied) {
	          copied = false;
	        }
	        if (afterContextMenu)
	            afterContextMenu = false;
	    };
	    var onInput = function(e) {
	        if (inComposition)
	            return;
	        var data = text.value;
	        sendText(data);
	        resetValue();
	    };

	    var handleClipboardData = function(e, data, forceIEMime) {
	        var clipboardData = e.clipboardData || window.clipboardData;
	        if (!clipboardData || BROKEN_SETDATA)
	            return;
	        var mime = USE_IE_MIME_TYPE || forceIEMime ? "Text" : "text/plain";
	        try {
	            if (data) {
	                return clipboardData.setData(mime, data) !== false;
	            } else {
	                return clipboardData.getData(mime);
	            }
	        } catch(e) {
	            if (!forceIEMime)
	                return handleClipboardData(e, data, true);
	        }
	    };

	    var doCopy = function(e, isCut) {
	        var data = host.getCopyText();
	        if (!data)
	            return event.preventDefault(e);

	        if (handleClipboardData(e, data)) {
	            if (useragent.isIOS) {
	                cut = isCut;
	                text.value = "\n aa" + data + "a a\n";
	                text.setSelectionRange(4, 4 + data.length);
	                copied = {
	                    value: data
	                };
	            }
	            isCut ? host.onCut() : host.onCopy();
	            if (!useragent.isIOS) event.preventDefault(e);
	        } else {
	            copied = true;
	            text.value = data;
	            text.select();
	            setTimeout(function(){
	                copied = false;
	                resetValue();
	                resetSelection();
	                isCut ? host.onCut() : host.onCopy();
	            });
	        }
	    };

	    var onCut = function(e) {
	        doCopy(e, true);
	    };

	    var onCopy = function(e) {
	        doCopy(e, false);
	    };

	    var onPaste = function(e) {
	        var data = handleClipboardData(e);
	        if (typeof data == "string") {
	            if (data)
	                host.onPaste(data, e);
	            if (useragent.isIE)
	                setTimeout(resetSelection);
	            event.preventDefault(e);
	        }
	        else {
	            text.value = "";
	            pasted = true;
	        }
	    };

	    event.addCommandKeyListener(text, host.onCommandKey.bind(host));

	    event.addListener(text, "select", onSelect);

	    event.addListener(text, "input", onInput);

	    event.addListener(text, "cut", onCut);
	    event.addListener(text, "copy", onCopy);
	    event.addListener(text, "paste", onPaste);
	    var onCompositionStart = function(e) {
	        if (inComposition || !host.onCompositionStart || host.$readOnly)
	            return;
	        inComposition = {};
	        inComposition.canUndo = host.session.$undoManager;
	        host.onCompositionStart();
	        setTimeout(onCompositionUpdate, 0);
	        host.on("mousedown", onCompositionEnd);
	        if (inComposition.canUndo && !host.selection.isEmpty()) {
	            host.insert("");
	            host.session.markUndoGroup();
	            host.selection.clearSelection();
	        }
	        host.session.markUndoGroup();
	    };

	    var onCompositionUpdate = function() {
	        if (!inComposition || !host.onCompositionUpdate || host.$readOnly)
	            return;
	        var val = text.value.replace(/\x01/g, "");
	        if (inComposition.lastValue === val) return;

	        host.onCompositionUpdate(val);
	        if (inComposition.lastValue)
	            host.undo();
	        if (inComposition.canUndo)
	            inComposition.lastValue = val;
	        if (inComposition.lastValue) {
	            var r = host.selection.getRange();
	            host.insert(inComposition.lastValue);
	            host.session.markUndoGroup();
	            inComposition.range = host.selection.getRange();
	            host.selection.setRange(r);
	            host.selection.clearSelection();
	        }
	    };

	    var onCompositionEnd = function(e) {
	        if (!host.onCompositionEnd || host.$readOnly) return;
	        var c = inComposition;
	        inComposition = false;
	        var timer = setTimeout(function() {
	            timer = null;
	            var str = text.value.replace(/\x01/g, "");
	            if (inComposition)
	                return;
	            else if (str == c.lastValue)
	                resetValue();
	            else if (!c.lastValue && str) {
	                resetValue();
	                sendText(str);
	            }
	        });
	        inputHandler = function compositionInputHandler(str) {
	            if (timer)
	                clearTimeout(timer);
	            str = str.replace(/\x01/g, "");
	            if (str == c.lastValue)
	                return "";
	            if (c.lastValue && timer)
	                host.undo();
	            return str;
	        };
	        host.onCompositionEnd();
	        host.removeListener("mousedown", onCompositionEnd);
	        if (e.type == "compositionend" && c.range) {
	            host.selection.setRange(c.range);
	        }
	        var needsOnInput =
	            (!!useragent.isChrome && useragent.isChrome >= 53) ||
	            (!!useragent.isWebKit && useragent.isWebKit >= 603);

	        if (needsOnInput) {
	          onInput();
	        }
	    };



	    var syncComposition = lang.delayedCall(onCompositionUpdate, 50);

	    event.addListener(text, "compositionstart", onCompositionStart);
	    if (useragent.isGecko) {
	        event.addListener(text, "text", function(){syncComposition.schedule();});
	    } else {
	        event.addListener(text, "keyup", function(){syncComposition.schedule();});
	        event.addListener(text, "keydown", function(){syncComposition.schedule();});
	    }
	    event.addListener(text, "compositionend", onCompositionEnd);

	    this.getElement = function() {
	        return text;
	    };

	    this.setReadOnly = function(readOnly) {
	       text.readOnly = readOnly;
	    };

	    this.onContextMenu = function(e) {
	        afterContextMenu = true;
	        resetSelection(host.selection.isEmpty());
	        host._emit("nativecontextmenu", {target: host, domEvent: e});
	        this.moveToMouse(e, true);
	    };

	    this.moveToMouse = function(e, bringToFront) {
	        if (!tempStyle)
	            tempStyle = text.style.cssText;
	        text.style.cssText = (bringToFront ? "z-index:100000;" : "")
	            + "height:" + text.style.height + ";"
	            + (useragent.isIE ? "opacity:0.1;" : "");

	        var rect = host.container.getBoundingClientRect();
	        var style = dom.computedStyle(host.container);
	        var top = rect.top + (parseInt(style.borderTopWidth) || 0);
	        var left = rect.left + (parseInt(rect.borderLeftWidth) || 0);
	        var maxTop = rect.bottom - top - text.clientHeight -2;
	        var move = function(e) {
	            text.style.left = e.clientX - left - 2 + "px";
	            text.style.top = Math.min(e.clientY - top - 2, maxTop) + "px";
	        };
	        move(e);

	        if (e.type != "mousedown")
	            return;

	        if (host.renderer.$keepTextAreaAtCursor)
	            host.renderer.$keepTextAreaAtCursor = null;

	        clearTimeout(closeTimeout);
	        if (useragent.isWin)
	            event.capture(host.container, move, onContextMenuClose);
	    };

	    this.onContextMenuClose = onContextMenuClose;
	    var closeTimeout;
	    function onContextMenuClose() {
	        clearTimeout(closeTimeout);
	        closeTimeout = setTimeout(function () {
	            if (tempStyle) {
	                text.style.cssText = tempStyle;
	                tempStyle = '';
	            }
	            if (host.renderer.$keepTextAreaAtCursor == null) {
	                host.renderer.$keepTextAreaAtCursor = true;
	                host.renderer.$moveTextAreaToCursor();
	            }
	        }, 0);
	    }

	    var onContextMenu = function(e) {
	        host.textInput.onContextMenu(e);
	        onContextMenuClose();
	    };
	    event.addListener(text, "mouseup", onContextMenu);
	    event.addListener(text, "mousedown", function(e) {
	        e.preventDefault();
	        onContextMenuClose();
	    });
	    event.addListener(host.renderer.scroller, "contextmenu", onContextMenu);
	    event.addListener(text, "contextmenu", onContextMenu);

	    if (useragent.isIOS) {
	        var typingResetTimeout = null;
	        var typing = false;

	        parentNode.addEventListener("keydown", function (e) {
	            if (typingResetTimeout) clearTimeout(typingResetTimeout);
	            typing = true;
	        });

	        parentNode.addEventListener("keyup", function (e) {
	            typingResetTimeout = setTimeout(function () {
	                typing = false;
	            }, 100);
	        });
	        var detectArrowKeys = function(e) {
	            if (document.activeElement !== text) return;
	            if (typing) return;

	            if (cut) {
	                return setTimeout(function () {
	                    cut = false;
	                }, 100);
	            }
	            var selectionStart = text.selectionStart;
	            var selectionEnd = text.selectionEnd;
	            text.setSelectionRange(4, 5);
	            if (selectionStart == selectionEnd) {
	                switch (selectionStart) {
	                    case 0: host.onCommandKey(null, 0, KEYS.up); break;
	                    case 1: host.onCommandKey(null, 0, KEYS.home); break;
	                    case 2: host.onCommandKey(null, MODS.option, KEYS.left); break;
	                    case 4: host.onCommandKey(null, 0, KEYS.left); break;
	                    case 5: host.onCommandKey(null, 0, KEYS.right); break;
	                    case 7: host.onCommandKey(null, MODS.option, KEYS.right); break;
	                    case 8: host.onCommandKey(null, 0, KEYS.end); break;
	                    case 9: host.onCommandKey(null, 0, KEYS.down); break;
	                }
	            } else {
	                switch (selectionEnd) {
	                    case 6: host.onCommandKey(null, MODS.shift, KEYS.right); break;
	                    case 7: host.onCommandKey(null, MODS.shift | MODS.option, KEYS.right); break;
	                    case 8: host.onCommandKey(null, MODS.shift, KEYS.end); break;
	                    case 9: host.onCommandKey(null, MODS.shift, KEYS.down); break;
	                }
	                switch (selectionStart) {
	                    case 0: host.onCommandKey(null, MODS.shift, KEYS.up); break;
	                    case 1: host.onCommandKey(null, MODS.shift, KEYS.home); break;
	                    case 2: host.onCommandKey(null, MODS.shift | MODS.option, KEYS.left); break;
	                    case 3: host.onCommandKey(null, MODS.shift, KEYS.left); break;
	                }
	            }
	        };
	        document.addEventListener("selectionchange", detectArrowKeys);
	        host.on("destroy", function() {
	            document.removeEventListener("selectionchange", detectArrowKeys);
	        });
	    }
	};

	exports.TextInput = TextInput;
	});

	ace.define("ace/keyboard/textinput",["require","exports","module","ace/lib/event","ace/lib/useragent","ace/lib/dom","ace/lib/lang","ace/keyboard/textinput_ios"], function(acequire, exports, module) {
	"use strict";

	var event = acequire("../lib/event");
	var useragent = acequire("../lib/useragent");
	var dom = acequire("../lib/dom");
	var lang = acequire("../lib/lang");
	var BROKEN_SETDATA = useragent.isChrome < 18;
	var USE_IE_MIME_TYPE =  useragent.isIE;

	var TextInputIOS = acequire("./textinput_ios").TextInput;
	var TextInput = function(parentNode, host) {
	    if (useragent.isIOS)
	        return TextInputIOS.call(this, parentNode, host);

	    var text = dom.createElement("textarea");
	    text.className = "ace_text-input";

	    text.setAttribute("wrap", "off");
	    text.setAttribute("autocorrect", "off");
	    text.setAttribute("autocapitalize", "off");
	    text.setAttribute("spellcheck", false);

	    text.style.opacity = "0";
	    parentNode.insertBefore(text, parentNode.firstChild);

	    var PLACEHOLDER = "\u2028\u2028";

	    var copied = false;
	    var pasted = false;
	    var inComposition = false;
	    var tempStyle = '';
	    var isSelectionEmpty = true;
	    try { var isFocused = document.activeElement === text; } catch(e) {}
	    
	    event.addListener(text, "blur", function(e) {
	        host.onBlur(e);
	        isFocused = false;
	    });
	    event.addListener(text, "focus", function(e) {
	        isFocused = true;
	        host.onFocus(e);
	        resetSelection();
	    });
	    this.focus = function() {
	        if (tempStyle) return text.focus();
	        var top = text.style.top;
	        text.style.position = "fixed";
	        text.style.top = "0px";
	        text.focus();
	        setTimeout(function() {
	            text.style.position = "";
	            if (text.style.top == "0px")
	                text.style.top = top;
	        }, 0);
	    };
	    this.blur = function() {
	        text.blur();
	    };
	    this.isFocused = function() {
	        return isFocused;
	    };
	    var syncSelection = lang.delayedCall(function() {
	        isFocused && resetSelection(isSelectionEmpty);
	    });
	    var syncValue = lang.delayedCall(function() {
	         if (!inComposition) {
	            text.value = PLACEHOLDER;
	            isFocused && resetSelection();
	         }
	    });

	    function resetSelection(isEmpty) {
	        if (inComposition)
	            return;
	        inComposition = true;
	        
	        if (inputHandler) {
	            var selectionStart = 0;
	            var selectionEnd = isEmpty ? 0 : text.value.length - 1;
	        } else {
	            var selectionStart = isEmpty ? 2 : 1;
	            var selectionEnd = 2;
	        }
	        try {
	            text.setSelectionRange(selectionStart, selectionEnd);
	        } catch(e){}
	        
	        inComposition = false;
	    }

	    function resetValue() {
	        if (inComposition)
	            return;
	        text.value = PLACEHOLDER;
	        if (useragent.isWebKit)
	            syncValue.schedule();
	    }

	    useragent.isWebKit || host.addEventListener('changeSelection', function() {
	        if (host.selection.isEmpty() != isSelectionEmpty) {
	            isSelectionEmpty = !isSelectionEmpty;
	            syncSelection.schedule();
	        }
	    });

	    resetValue();
	    if (isFocused)
	        host.onFocus();


	    var isAllSelected = function(text) {
	        return text.selectionStart === 0 && text.selectionEnd === text.value.length;
	    };

	    var onSelect = function(e) {
	        if (copied) {
	            copied = false;
	        } else if (isAllSelected(text)) {
	            host.selectAll();
	            resetSelection();
	        } else if (inputHandler) {
	            resetSelection(host.selection.isEmpty());
	        }
	    };

	    var inputHandler = null;
	    this.setInputHandler = function(cb) {inputHandler = cb;};
	    this.getInputHandler = function() {return inputHandler;};
	    var afterContextMenu = false;
	    
	    var sendText = function(data) {
	        if (inputHandler) {
	            data = inputHandler(data);
	            inputHandler = null;
	        }
	        if (pasted) {
	            resetSelection();
	            if (data)
	                host.onPaste(data);
	            pasted = false;
	        } else if (data == PLACEHOLDER.charAt(0)) {
	            if (afterContextMenu)
	                host.execCommand("del", {source: "ace"});
	            else // some versions of android do not fire keydown when pressing backspace
	                host.execCommand("backspace", {source: "ace"});
	        } else {
	            if (data.substring(0, 2) == PLACEHOLDER)
	                data = data.substr(2);
	            else if (data.charAt(0) == PLACEHOLDER.charAt(0))
	                data = data.substr(1);
	            else if (data.charAt(data.length - 1) == PLACEHOLDER.charAt(0))
	                data = data.slice(0, -1);
	            if (data.charAt(data.length - 1) == PLACEHOLDER.charAt(0))
	                data = data.slice(0, -1);
	            
	            if (data)
	                host.onTextInput(data);
	        }
	        if (afterContextMenu)
	            afterContextMenu = false;
	    };
	    var onInput = function(e) {
	        if (inComposition)
	            return;
	        var data = text.value;
	        sendText(data);
	        resetValue();
	    };
	    
	    var handleClipboardData = function(e, data, forceIEMime) {
	        var clipboardData = e.clipboardData || window.clipboardData;
	        if (!clipboardData || BROKEN_SETDATA)
	            return;
	        var mime = USE_IE_MIME_TYPE || forceIEMime ? "Text" : "text/plain";
	        try {
	            if (data) {
	                return clipboardData.setData(mime, data) !== false;
	            } else {
	                return clipboardData.getData(mime);
	            }
	        } catch(e) {
	            if (!forceIEMime)
	                return handleClipboardData(e, data, true);
	        }
	    };

	    var doCopy = function(e, isCut) {
	        var data = host.getCopyText();
	        if (!data)
	            return event.preventDefault(e);

	        if (handleClipboardData(e, data)) {
	            isCut ? host.onCut() : host.onCopy();
	            event.preventDefault(e);
	        } else {
	            copied = true;
	            text.value = data;
	            text.select();
	            setTimeout(function(){
	                copied = false;
	                resetValue();
	                resetSelection();
	                isCut ? host.onCut() : host.onCopy();
	            });
	        }
	    };
	    
	    var onCut = function(e) {
	        doCopy(e, true);
	    };
	    
	    var onCopy = function(e) {
	        doCopy(e, false);
	    };
	    
	    var onPaste = function(e) {
	        var data = handleClipboardData(e);
	        if (typeof data == "string") {
	            if (data)
	                host.onPaste(data, e);
	            if (useragent.isIE)
	                setTimeout(resetSelection);
	            event.preventDefault(e);
	        }
	        else {
	            text.value = "";
	            pasted = true;
	        }
	    };

	    event.addCommandKeyListener(text, host.onCommandKey.bind(host));

	    event.addListener(text, "select", onSelect);

	    event.addListener(text, "input", onInput);

	    event.addListener(text, "cut", onCut);
	    event.addListener(text, "copy", onCopy);
	    event.addListener(text, "paste", onPaste);
	    if (!('oncut' in text) || !('oncopy' in text) || !('onpaste' in text)) {
	        event.addListener(parentNode, "keydown", function(e) {
	            if ((useragent.isMac && !e.metaKey) || !e.ctrlKey)
	                return;

	            switch (e.keyCode) {
	                case 67:
	                    onCopy(e);
	                    break;
	                case 86:
	                    onPaste(e);
	                    break;
	                case 88:
	                    onCut(e);
	                    break;
	            }
	        });
	    }
	    var onCompositionStart = function(e) {
	        if (inComposition || !host.onCompositionStart || host.$readOnly) 
	            return;
	        inComposition = {};
	        inComposition.canUndo = host.session.$undoManager;
	        host.onCompositionStart();
	        setTimeout(onCompositionUpdate, 0);
	        host.on("mousedown", onCompositionEnd);
	        if (inComposition.canUndo && !host.selection.isEmpty()) {
	            host.insert("");
	            host.session.markUndoGroup();
	            host.selection.clearSelection();
	        }
	        host.session.markUndoGroup();
	    };

	    var onCompositionUpdate = function() {
	        if (!inComposition || !host.onCompositionUpdate || host.$readOnly)
	            return;
	        var val = text.value.replace(/\u2028/g, "");
	        if (inComposition.lastValue === val) return;
	        
	        host.onCompositionUpdate(val);
	        if (inComposition.lastValue)
	            host.undo();
	        if (inComposition.canUndo)
	            inComposition.lastValue = val;
	        if (inComposition.lastValue) {
	            var r = host.selection.getRange();
	            host.insert(inComposition.lastValue);
	            host.session.markUndoGroup();
	            inComposition.range = host.selection.getRange();
	            host.selection.setRange(r);
	            host.selection.clearSelection();
	        }
	    };

	    var onCompositionEnd = function(e) {
	        if (!host.onCompositionEnd || host.$readOnly) return;
	        var c = inComposition;
	        inComposition = false;
	        var timer = setTimeout(function() {
	            timer = null;
	            var str = text.value.replace(/\u2028/g, "");
	            if (inComposition)
	                return;
	            else if (str == c.lastValue)
	                resetValue();
	            else if (!c.lastValue && str) {
	                resetValue();
	                sendText(str);
	            }
	        });
	        inputHandler = function compositionInputHandler(str) {
	            if (timer)
	                clearTimeout(timer);
	            str = str.replace(/\u2028/g, "");
	            if (str == c.lastValue)
	                return "";
	            if (c.lastValue && timer)
	                host.undo();
	            return str;
	        };
	        host.onCompositionEnd();
	        host.removeListener("mousedown", onCompositionEnd);
	        if (e.type == "compositionend" && c.range) {
	            host.selection.setRange(c.range);
	        }
	        var needsOnInput =
	            (!!useragent.isChrome && useragent.isChrome >= 53) ||
	            (!!useragent.isWebKit && useragent.isWebKit >= 603);

	        if (needsOnInput) {
	          onInput();
	        }
	    };
	    
	    

	    var syncComposition = lang.delayedCall(onCompositionUpdate, 50);

	    event.addListener(text, "compositionstart", onCompositionStart);
	    if (useragent.isGecko) {
	        event.addListener(text, "text", function(){syncComposition.schedule();});
	    } else {
	        event.addListener(text, "keyup", function(){syncComposition.schedule();});
	        event.addListener(text, "keydown", function(){syncComposition.schedule();});
	    }
	    event.addListener(text, "compositionend", onCompositionEnd);

	    this.getElement = function() {
	        return text;
	    };

	    this.setReadOnly = function(readOnly) {
	       text.readOnly = readOnly;
	    };

	    this.onContextMenu = function(e) {
	        afterContextMenu = true;
	        resetSelection(host.selection.isEmpty());
	        host._emit("nativecontextmenu", {target: host, domEvent: e});
	        this.moveToMouse(e, true);
	    };
	    
	    this.moveToMouse = function(e, bringToFront) {
	        if (!tempStyle)
	            tempStyle = text.style.cssText;
	        text.style.cssText = (bringToFront ? "z-index:100000;" : "")
	            + "height:" + text.style.height + ";"
	            + (useragent.isIE ? "opacity:0.1;" : "");

	        var rect = host.container.getBoundingClientRect();
	        var style = dom.computedStyle(host.container);
	        var top = rect.top + (parseInt(style.borderTopWidth) || 0);
	        var left = rect.left + (parseInt(rect.borderLeftWidth) || 0);
	        var maxTop = rect.bottom - top - text.clientHeight -2;
	        var move = function(e) {
	            text.style.left = e.clientX - left - 2 + "px";
	            text.style.top = Math.min(e.clientY - top - 2, maxTop) + "px";
	        }; 
	        move(e);

	        if (e.type != "mousedown")
	            return;

	        if (host.renderer.$keepTextAreaAtCursor)
	            host.renderer.$keepTextAreaAtCursor = null;

	        clearTimeout(closeTimeout);
	        if (useragent.isWin)
	            event.capture(host.container, move, onContextMenuClose);
	    };

	    this.onContextMenuClose = onContextMenuClose;
	    var closeTimeout;
	    function onContextMenuClose() {
	        clearTimeout(closeTimeout);
	        closeTimeout = setTimeout(function () {
	            if (tempStyle) {
	                text.style.cssText = tempStyle;
	                tempStyle = '';
	            }
	            if (host.renderer.$keepTextAreaAtCursor == null) {
	                host.renderer.$keepTextAreaAtCursor = true;
	                host.renderer.$moveTextAreaToCursor();
	            }
	        }, 0);
	    }

	    var onContextMenu = function(e) {
	        host.textInput.onContextMenu(e);
	        onContextMenuClose();
	    };
	    event.addListener(text, "mouseup", onContextMenu);
	    event.addListener(text, "mousedown", function(e) {
	        e.preventDefault();
	        onContextMenuClose();
	    });
	    event.addListener(host.renderer.scroller, "contextmenu", onContextMenu);
	    event.addListener(text, "contextmenu", onContextMenu);
	};

	exports.TextInput = TextInput;
	});

	ace.define("ace/mouse/default_handlers",["require","exports","module","ace/lib/dom","ace/lib/event","ace/lib/useragent"], function(acequire, exports, module) {
	"use strict";

	var dom = acequire("../lib/dom");
	var event = acequire("../lib/event");
	var useragent = acequire("../lib/useragent");

	var DRAG_OFFSET = 0; // pixels
	var SCROLL_COOLDOWN_T = 250; // milliseconds

	function DefaultHandlers(mouseHandler) {
	    mouseHandler.$clickSelection = null;

	    var editor = mouseHandler.editor;
	    editor.setDefaultHandler("mousedown", this.onMouseDown.bind(mouseHandler));
	    editor.setDefaultHandler("dblclick", this.onDoubleClick.bind(mouseHandler));
	    editor.setDefaultHandler("tripleclick", this.onTripleClick.bind(mouseHandler));
	    editor.setDefaultHandler("quadclick", this.onQuadClick.bind(mouseHandler));
	    editor.setDefaultHandler("mousewheel", this.onMouseWheel.bind(mouseHandler));
	    editor.setDefaultHandler("touchmove", this.onTouchMove.bind(mouseHandler));

	    var exports = ["select", "startSelect", "selectEnd", "selectAllEnd", "selectByWordsEnd",
	        "selectByLinesEnd", "dragWait", "dragWaitEnd", "focusWait"];

	    exports.forEach(function(x) {
	        mouseHandler[x] = this[x];
	    }, this);

	    mouseHandler.selectByLines = this.extendSelectionBy.bind(mouseHandler, "getLineRange");
	    mouseHandler.selectByWords = this.extendSelectionBy.bind(mouseHandler, "getWordRange");
	}

	(function() {

	    this.onMouseDown = function(ev) {
	        var inSelection = ev.inSelection();
	        var pos = ev.getDocumentPosition();
	        this.mousedownEvent = ev;
	        var editor = this.editor;

	        var button = ev.getButton();
	        if (button !== 0) {
	            var selectionRange = editor.getSelectionRange();
	            var selectionEmpty = selectionRange.isEmpty();
	            editor.$blockScrolling++;
	            if (selectionEmpty || button == 1)
	                editor.selection.moveToPosition(pos);
	            editor.$blockScrolling--;
	            if (button == 2) {
	                editor.textInput.onContextMenu(ev.domEvent);
	                if (!useragent.isMozilla)
	                    ev.preventDefault();
	            }
	            return;
	        }

	        this.mousedownEvent.time = Date.now();
	        if (inSelection && !editor.isFocused()) {
	            editor.focus();
	            if (this.$focusTimout && !this.$clickSelection && !editor.inMultiSelectMode) {
	                this.setState("focusWait");
	                this.captureMouse(ev);
	                return;
	            }
	        }

	        this.captureMouse(ev);
	        this.startSelect(pos, ev.domEvent._clicks > 1);
	        return ev.preventDefault();
	    };

	    this.startSelect = function(pos, waitForClickSelection) {
	        pos = pos || this.editor.renderer.screenToTextCoordinates(this.x, this.y);
	        var editor = this.editor;
	        editor.$blockScrolling++;
	        if (this.mousedownEvent.getShiftKey())
	            editor.selection.selectToPosition(pos);
	        else if (!waitForClickSelection)
	            editor.selection.moveToPosition(pos);
	        if (!waitForClickSelection)
	            this.select();
	        if (editor.renderer.scroller.setCapture) {
	            editor.renderer.scroller.setCapture();
	        }
	        editor.setStyle("ace_selecting");
	        this.setState("select");
	        editor.$blockScrolling--;
	    };

	    this.select = function() {
	        var anchor, editor = this.editor;
	        var cursor = editor.renderer.screenToTextCoordinates(this.x, this.y);
	        editor.$blockScrolling++;
	        if (this.$clickSelection) {
	            var cmp = this.$clickSelection.comparePoint(cursor);

	            if (cmp == -1) {
	                anchor = this.$clickSelection.end;
	            } else if (cmp == 1) {
	                anchor = this.$clickSelection.start;
	            } else {
	                var orientedRange = calcRangeOrientation(this.$clickSelection, cursor);
	                cursor = orientedRange.cursor;
	                anchor = orientedRange.anchor;
	            }
	            editor.selection.setSelectionAnchor(anchor.row, anchor.column);
	        }
	        editor.selection.selectToPosition(cursor);
	        editor.$blockScrolling--;
	        editor.renderer.scrollCursorIntoView();
	    };

	    this.extendSelectionBy = function(unitName) {
	        var anchor, editor = this.editor;
	        var cursor = editor.renderer.screenToTextCoordinates(this.x, this.y);
	        var range = editor.selection[unitName](cursor.row, cursor.column);
	        editor.$blockScrolling++;
	        if (this.$clickSelection) {
	            var cmpStart = this.$clickSelection.comparePoint(range.start);
	            var cmpEnd = this.$clickSelection.comparePoint(range.end);

	            if (cmpStart == -1 && cmpEnd <= 0) {
	                anchor = this.$clickSelection.end;
	                if (range.end.row != cursor.row || range.end.column != cursor.column)
	                    cursor = range.start;
	            } else if (cmpEnd == 1 && cmpStart >= 0) {
	                anchor = this.$clickSelection.start;
	                if (range.start.row != cursor.row || range.start.column != cursor.column)
	                    cursor = range.end;
	            } else if (cmpStart == -1 && cmpEnd == 1) {
	                cursor = range.end;
	                anchor = range.start;
	            } else {
	                var orientedRange = calcRangeOrientation(this.$clickSelection, cursor);
	                cursor = orientedRange.cursor;
	                anchor = orientedRange.anchor;
	            }
	            editor.selection.setSelectionAnchor(anchor.row, anchor.column);
	        }
	        editor.selection.selectToPosition(cursor);
	        editor.$blockScrolling--;
	        editor.renderer.scrollCursorIntoView();
	    };

	    this.selectEnd =
	    this.selectAllEnd =
	    this.selectByWordsEnd =
	    this.selectByLinesEnd = function() {
	        this.$clickSelection = null;
	        this.editor.unsetStyle("ace_selecting");
	        if (this.editor.renderer.scroller.releaseCapture) {
	            this.editor.renderer.scroller.releaseCapture();
	        }
	    };

	    this.focusWait = function() {
	        var distance = calcDistance(this.mousedownEvent.x, this.mousedownEvent.y, this.x, this.y);
	        var time = Date.now();

	        if (distance > DRAG_OFFSET || time - this.mousedownEvent.time > this.$focusTimout)
	            this.startSelect(this.mousedownEvent.getDocumentPosition());
	    };

	    this.onDoubleClick = function(ev) {
	        var pos = ev.getDocumentPosition();
	        var editor = this.editor;
	        var session = editor.session;

	        var range = session.getBracketRange(pos);
	        if (range) {
	            if (range.isEmpty()) {
	                range.start.column--;
	                range.end.column++;
	            }
	            this.setState("select");
	        } else {
	            range = editor.selection.getWordRange(pos.row, pos.column);
	            this.setState("selectByWords");
	        }
	        this.$clickSelection = range;
	        this.select();
	    };

	    this.onTripleClick = function(ev) {
	        var pos = ev.getDocumentPosition();
	        var editor = this.editor;

	        this.setState("selectByLines");
	        var range = editor.getSelectionRange();
	        if (range.isMultiLine() && range.contains(pos.row, pos.column)) {
	            this.$clickSelection = editor.selection.getLineRange(range.start.row);
	            this.$clickSelection.end = editor.selection.getLineRange(range.end.row).end;
	        } else {
	            this.$clickSelection = editor.selection.getLineRange(pos.row);
	        }
	        this.select();
	    };

	    this.onQuadClick = function(ev) {
	        var editor = this.editor;

	        editor.selectAll();
	        this.$clickSelection = editor.getSelectionRange();
	        this.setState("selectAll");
	    };

	    this.onMouseWheel = function(ev) {
	        if (ev.getAccelKey())
	            return;
	        if (ev.getShiftKey() && ev.wheelY && !ev.wheelX) {
	            ev.wheelX = ev.wheelY;
	            ev.wheelY = 0;
	        }
	        
	        var editor = this.editor;

	        if (!this.$lastScroll)
	            this.$lastScroll = { t: 0, vx: 0, vy: 0, allowed: 0 };

	        var prevScroll = this.$lastScroll;
	        var t = ev.domEvent.timeStamp;
	        var dt = t - prevScroll.t;
	        var vx = ev.wheelX / dt;
	        var vy = ev.wheelY / dt;
	        if (dt < SCROLL_COOLDOWN_T) {
	            vx = (vx + prevScroll.vx) / 2;
	            vy = (vy + prevScroll.vy) / 2;
	        }

	        var direction = Math.abs(vx / vy);

	        var canScroll = false;
	        if (direction >= 1 && editor.renderer.isScrollableBy(ev.wheelX * ev.speed, 0))
	            canScroll = true;
	        if (direction <= 1 && editor.renderer.isScrollableBy(0, ev.wheelY * ev.speed))
	            canScroll = true;

	        if (canScroll) {
	            prevScroll.allowed = t;
	        } else if (t - prevScroll.allowed < SCROLL_COOLDOWN_T) {
	            var isSlower = Math.abs(vx) <= 1.1 * Math.abs(prevScroll.vx)
	                && Math.abs(vy) <= 1.1 * Math.abs(prevScroll.vy);
	            if (isSlower) {
	                canScroll = true;
	                prevScroll.allowed = t;
	            }
	            else {
	                prevScroll.allowed = 0;
	            }
	        }

	        prevScroll.t = t;
	        prevScroll.vx = vx;
	        prevScroll.vy = vy;

	        if (canScroll) {
	            editor.renderer.scrollBy(ev.wheelX * ev.speed, ev.wheelY * ev.speed);
	            return ev.stop();
	        }
	    };

	    this.onTouchMove = function(ev) {
	        this.editor._emit("mousewheel", ev);
	    };

	}).call(DefaultHandlers.prototype);

	exports.DefaultHandlers = DefaultHandlers;

	function calcDistance(ax, ay, bx, by) {
	    return Math.sqrt(Math.pow(bx - ax, 2) + Math.pow(by - ay, 2));
	}

	function calcRangeOrientation(range, cursor) {
	    if (range.start.row == range.end.row)
	        var cmp = 2 * cursor.column - range.start.column - range.end.column;
	    else if (range.start.row == range.end.row - 1 && !range.start.column && !range.end.column)
	        var cmp = cursor.column - 4;
	    else
	        var cmp = 2 * cursor.row - range.start.row - range.end.row;

	    if (cmp < 0)
	        return {cursor: range.start, anchor: range.end};
	    else
	        return {cursor: range.end, anchor: range.start};
	}

	});

	ace.define("ace/tooltip",["require","exports","module","ace/lib/oop","ace/lib/dom"], function(acequire, exports, module) {
	"use strict";

	var oop = acequire("./lib/oop");
	var dom = acequire("./lib/dom");
	function Tooltip (parentNode) {
	    this.isOpen = false;
	    this.$element = null;
	    this.$parentNode = parentNode;
	}

	(function() {
	    this.$init = function() {
	        this.$element = dom.createElement("div");
	        this.$element.className = "ace_tooltip";
	        this.$element.style.display = "none";
	        this.$parentNode.appendChild(this.$element);
	        return this.$element;
	    };
	    this.getElement = function() {
	        return this.$element || this.$init();
	    };
	    this.setText = function(text) {
	        dom.setInnerText(this.getElement(), text);
	    };
	    this.setHtml = function(html) {
	        this.getElement().innerHTML = html;
	    };
	    this.setPosition = function(x, y) {
	        this.getElement().style.left = x + "px";
	        this.getElement().style.top = y + "px";
	    };
	    this.setClassName = function(className) {
	        dom.addCssClass(this.getElement(), className);
	    };
	    this.show = function(text, x, y) {
	        if (text != null)
	            this.setText(text);
	        if (x != null && y != null)
	            this.setPosition(x, y);
	        if (!this.isOpen) {
	            this.getElement().style.display = "block";
	            this.isOpen = true;
	        }
	    };

	    this.hide = function() {
	        if (this.isOpen) {
	            this.getElement().style.display = "none";
	            this.isOpen = false;
	        }
	    };
	    this.getHeight = function() {
	        return this.getElement().offsetHeight;
	    };
	    this.getWidth = function() {
	        return this.getElement().offsetWidth;
	    };

	    this.destroy = function() {
	        this.isOpen = false;
	        if (this.$element && this.$element.parentNode) {
	            this.$element.parentNode.removeChild(this.$element);
	        }
	    };

	}).call(Tooltip.prototype);

	exports.Tooltip = Tooltip;
	});

	ace.define("ace/mouse/default_gutter_handler",["require","exports","module","ace/lib/dom","ace/lib/oop","ace/lib/event","ace/tooltip"], function(acequire, exports, module) {
	"use strict";
	var dom = acequire("../lib/dom");
	var oop = acequire("../lib/oop");
	var event = acequire("../lib/event");
	var Tooltip = acequire("../tooltip").Tooltip;

	function GutterHandler(mouseHandler) {
	    var editor = mouseHandler.editor;
	    var gutter = editor.renderer.$gutterLayer;
	    var tooltip = new GutterTooltip(editor.container);

	    mouseHandler.editor.setDefaultHandler("guttermousedown", function(e) {
	        if (!editor.isFocused() || e.getButton() != 0)
	            return;
	        var gutterRegion = gutter.getRegion(e);

	        if (gutterRegion == "foldWidgets")
	            return;

	        var row = e.getDocumentPosition().row;
	        var selection = editor.session.selection;

	        if (e.getShiftKey())
	            selection.selectTo(row, 0);
	        else {
	            if (e.domEvent.detail == 2) {
	                editor.selectAll();
	                return e.preventDefault();
	            }
	            mouseHandler.$clickSelection = editor.selection.getLineRange(row);
	        }
	        mouseHandler.setState("selectByLines");
	        mouseHandler.captureMouse(e);
	        return e.preventDefault();
	    });


	    var tooltipTimeout, mouseEvent, tooltipAnnotation;

	    function showTooltip() {
	        var row = mouseEvent.getDocumentPosition().row;
	        var annotation = gutter.$annotations[row];
	        if (!annotation)
	            return hideTooltip();

	        var maxRow = editor.session.getLength();
	        if (row == maxRow) {
	            var screenRow = editor.renderer.pixelToScreenCoordinates(0, mouseEvent.y).row;
	            var pos = mouseEvent.$pos;
	            if (screenRow > editor.session.documentToScreenRow(pos.row, pos.column))
	                return hideTooltip();
	        }

	        if (tooltipAnnotation == annotation)
	            return;
	        tooltipAnnotation = annotation.text.join("<br/>");

	        tooltip.setHtml(tooltipAnnotation);
	        tooltip.show();
	        editor._signal("showGutterTooltip", tooltip);
	        editor.on("mousewheel", hideTooltip);

	        if (mouseHandler.$tooltipFollowsMouse) {
	            moveTooltip(mouseEvent);
	        } else {
	            var gutterElement = mouseEvent.domEvent.target;
	            var rect = gutterElement.getBoundingClientRect();
	            var style = tooltip.getElement().style;
	            style.left = rect.right + "px";
	            style.top = rect.bottom + "px";
	        }
	    }

	    function hideTooltip() {
	        if (tooltipTimeout)
	            tooltipTimeout = clearTimeout(tooltipTimeout);
	        if (tooltipAnnotation) {
	            tooltip.hide();
	            tooltipAnnotation = null;
	            editor._signal("hideGutterTooltip", tooltip);
	            editor.removeEventListener("mousewheel", hideTooltip);
	        }
	    }

	    function moveTooltip(e) {
	        tooltip.setPosition(e.x, e.y);
	    }

	    mouseHandler.editor.setDefaultHandler("guttermousemove", function(e) {
	        var target = e.domEvent.target || e.domEvent.srcElement;
	        if (dom.hasCssClass(target, "ace_fold-widget"))
	            return hideTooltip();

	        if (tooltipAnnotation && mouseHandler.$tooltipFollowsMouse)
	            moveTooltip(e);

	        mouseEvent = e;
	        if (tooltipTimeout)
	            return;
	        tooltipTimeout = setTimeout(function() {
	            tooltipTimeout = null;
	            if (mouseEvent && !mouseHandler.isMousePressed)
	                showTooltip();
	            else
	                hideTooltip();
	        }, 50);
	    });

	    event.addListener(editor.renderer.$gutter, "mouseout", function(e) {
	        mouseEvent = null;
	        if (!tooltipAnnotation || tooltipTimeout)
	            return;

	        tooltipTimeout = setTimeout(function() {
	            tooltipTimeout = null;
	            hideTooltip();
	        }, 50);
	    });
	    
	    editor.on("changeSession", hideTooltip);
	}

	function GutterTooltip(parentNode) {
	    Tooltip.call(this, parentNode);
	}

	oop.inherits(GutterTooltip, Tooltip);

	(function(){
	    this.setPosition = function(x, y) {
	        var windowWidth = window.innerWidth || document.documentElement.clientWidth;
	        var windowHeight = window.innerHeight || document.documentElement.clientHeight;
	        var width = this.getWidth();
	        var height = this.getHeight();
	        x += 15;
	        y += 15;
	        if (x + width > windowWidth) {
	            x -= (x + width) - windowWidth;
	        }
	        if (y + height > windowHeight) {
	            y -= 20 + height;
	        }
	        Tooltip.prototype.setPosition.call(this, x, y);
	    };

	}).call(GutterTooltip.prototype);



	exports.GutterHandler = GutterHandler;

	});

	ace.define("ace/mouse/mouse_event",["require","exports","module","ace/lib/event","ace/lib/useragent"], function(acequire, exports, module) {
	"use strict";

	var event = acequire("../lib/event");
	var useragent = acequire("../lib/useragent");
	var MouseEvent = exports.MouseEvent = function(domEvent, editor) {
	    this.domEvent = domEvent;
	    this.editor = editor;
	    
	    this.x = this.clientX = domEvent.clientX;
	    this.y = this.clientY = domEvent.clientY;

	    this.$pos = null;
	    this.$inSelection = null;
	    
	    this.propagationStopped = false;
	    this.defaultPrevented = false;
	};

	(function() {  
	    
	    this.stopPropagation = function() {
	        event.stopPropagation(this.domEvent);
	        this.propagationStopped = true;
	    };
	    
	    this.preventDefault = function() {
	        event.preventDefault(this.domEvent);
	        this.defaultPrevented = true;
	    };
	    
	    this.stop = function() {
	        this.stopPropagation();
	        this.preventDefault();
	    };
	    this.getDocumentPosition = function() {
	        if (this.$pos)
	            return this.$pos;
	        
	        this.$pos = this.editor.renderer.screenToTextCoordinates(this.clientX, this.clientY);
	        return this.$pos;
	    };
	    this.inSelection = function() {
	        if (this.$inSelection !== null)
	            return this.$inSelection;
	            
	        var editor = this.editor;
	        

	        var selectionRange = editor.getSelectionRange();
	        if (selectionRange.isEmpty())
	            this.$inSelection = false;
	        else {
	            var pos = this.getDocumentPosition();
	            this.$inSelection = selectionRange.contains(pos.row, pos.column);
	        }

	        return this.$inSelection;
	    };
	    this.getButton = function() {
	        return event.getButton(this.domEvent);
	    };
	    this.getShiftKey = function() {
	        return this.domEvent.shiftKey;
	    };
	    
	    this.getAccelKey = useragent.isMac
	        ? function() { return this.domEvent.metaKey; }
	        : function() { return this.domEvent.ctrlKey; };
	    
	}).call(MouseEvent.prototype);

	});

	ace.define("ace/mouse/dragdrop_handler",["require","exports","module","ace/lib/dom","ace/lib/event","ace/lib/useragent"], function(acequire, exports, module) {
	"use strict";

	var dom = acequire("../lib/dom");
	var event = acequire("../lib/event");
	var useragent = acequire("../lib/useragent");

	var AUTOSCROLL_DELAY = 200;
	var SCROLL_CURSOR_DELAY = 200;
	var SCROLL_CURSOR_HYSTERESIS = 5;

	function DragdropHandler(mouseHandler) {

	    var editor = mouseHandler.editor;

	    var blankImage = dom.createElement("img");
	    blankImage.src = "data:image/gif;base64,R0lGODlhAQABAAAAACH5BAEKAAEALAAAAAABAAEAAAICTAEAOw==";
	    if (useragent.isOpera)
	        blankImage.style.cssText = "width:1px;height:1px;position:fixed;top:0;left:0;z-index:2147483647;opacity:0;";

	    var exports = ["dragWait", "dragWaitEnd", "startDrag", "dragReadyEnd", "onMouseDrag"];

	     exports.forEach(function(x) {
	         mouseHandler[x] = this[x];
	    }, this);
	    editor.addEventListener("mousedown", this.onMouseDown.bind(mouseHandler));


	    var mouseTarget = editor.container;
	    var dragSelectionMarker, x, y;
	    var timerId, range;
	    var dragCursor, counter = 0;
	    var dragOperation;
	    var isInternal;
	    var autoScrollStartTime;
	    var cursorMovedTime;
	    var cursorPointOnCaretMoved;

	    this.onDragStart = function(e) {
	        if (this.cancelDrag || !mouseTarget.draggable) {
	            var self = this;
	            setTimeout(function(){
	                self.startSelect();
	                self.captureMouse(e);
	            }, 0);
	            return e.preventDefault();
	        }
	        range = editor.getSelectionRange();

	        var dataTransfer = e.dataTransfer;
	        dataTransfer.effectAllowed = editor.getReadOnly() ? "copy" : "copyMove";
	        if (useragent.isOpera) {
	            editor.container.appendChild(blankImage);
	            blankImage.scrollTop = 0;
	        }
	        dataTransfer.setDragImage && dataTransfer.setDragImage(blankImage, 0, 0);
	        if (useragent.isOpera) {
	            editor.container.removeChild(blankImage);
	        }
	        dataTransfer.clearData();
	        dataTransfer.setData("Text", editor.session.getTextRange());

	        isInternal = true;
	        this.setState("drag");
	    };

	    this.onDragEnd = function(e) {
	        mouseTarget.draggable = false;
	        isInternal = false;
	        this.setState(null);
	        if (!editor.getReadOnly()) {
	            var dropEffect = e.dataTransfer.dropEffect;
	            if (!dragOperation && dropEffect == "move")
	                editor.session.remove(editor.getSelectionRange());
	            editor.renderer.$cursorLayer.setBlinking(true);
	        }
	        this.editor.unsetStyle("ace_dragging");
	        this.editor.renderer.setCursorStyle("");
	    };

	    this.onDragEnter = function(e) {
	        if (editor.getReadOnly() || !canAccept(e.dataTransfer))
	            return;
	        x = e.clientX;
	        y = e.clientY;
	        if (!dragSelectionMarker)
	            addDragMarker();
	        counter++;
	        e.dataTransfer.dropEffect = dragOperation = getDropEffect(e);
	        return event.preventDefault(e);
	    };

	    this.onDragOver = function(e) {
	        if (editor.getReadOnly() || !canAccept(e.dataTransfer))
	            return;
	        x = e.clientX;
	        y = e.clientY;
	        if (!dragSelectionMarker) {
	            addDragMarker();
	            counter++;
	        }
	        if (onMouseMoveTimer !== null)
	            onMouseMoveTimer = null;

	        e.dataTransfer.dropEffect = dragOperation = getDropEffect(e);
	        return event.preventDefault(e);
	    };

	    this.onDragLeave = function(e) {
	        counter--;
	        if (counter <= 0 && dragSelectionMarker) {
	            clearDragMarker();
	            dragOperation = null;
	            return event.preventDefault(e);
	        }
	    };

	    this.onDrop = function(e) {
	        if (!dragCursor)
	            return;
	        var dataTransfer = e.dataTransfer;
	        if (isInternal) {
	            switch (dragOperation) {
	                case "move":
	                    if (range.contains(dragCursor.row, dragCursor.column)) {
	                        range = {
	                            start: dragCursor,
	                            end: dragCursor
	                        };
	                    } else {
	                        range = editor.moveText(range, dragCursor);
	                    }
	                    break;
	                case "copy":
	                    range = editor.moveText(range, dragCursor, true);
	                    break;
	            }
	        } else {
	            var dropData = dataTransfer.getData('Text');
	            range = {
	                start: dragCursor,
	                end: editor.session.insert(dragCursor, dropData)
	            };
	            editor.focus();
	            dragOperation = null;
	        }
	        clearDragMarker();
	        return event.preventDefault(e);
	    };

	    event.addListener(mouseTarget, "dragstart", this.onDragStart.bind(mouseHandler));
	    event.addListener(mouseTarget, "dragend", this.onDragEnd.bind(mouseHandler));
	    event.addListener(mouseTarget, "dragenter", this.onDragEnter.bind(mouseHandler));
	    event.addListener(mouseTarget, "dragover", this.onDragOver.bind(mouseHandler));
	    event.addListener(mouseTarget, "dragleave", this.onDragLeave.bind(mouseHandler));
	    event.addListener(mouseTarget, "drop", this.onDrop.bind(mouseHandler));

	    function scrollCursorIntoView(cursor, prevCursor) {
	        var now = Date.now();
	        var vMovement = !prevCursor || cursor.row != prevCursor.row;
	        var hMovement = !prevCursor || cursor.column != prevCursor.column;
	        if (!cursorMovedTime || vMovement || hMovement) {
	            editor.$blockScrolling += 1;
	            editor.moveCursorToPosition(cursor);
	            editor.$blockScrolling -= 1;
	            cursorMovedTime = now;
	            cursorPointOnCaretMoved = {x: x, y: y};
	        } else {
	            var distance = calcDistance(cursorPointOnCaretMoved.x, cursorPointOnCaretMoved.y, x, y);
	            if (distance > SCROLL_CURSOR_HYSTERESIS) {
	                cursorMovedTime = null;
	            } else if (now - cursorMovedTime >= SCROLL_CURSOR_DELAY) {
	                editor.renderer.scrollCursorIntoView();
	                cursorMovedTime = null;
	            }
	        }
	    }

	    function autoScroll(cursor, prevCursor) {
	        var now = Date.now();
	        var lineHeight = editor.renderer.layerConfig.lineHeight;
	        var characterWidth = editor.renderer.layerConfig.characterWidth;
	        var editorRect = editor.renderer.scroller.getBoundingClientRect();
	        var offsets = {
	           x: {
	               left: x - editorRect.left,
	               right: editorRect.right - x
	           },
	           y: {
	               top: y - editorRect.top,
	               bottom: editorRect.bottom - y
	           }
	        };
	        var nearestXOffset = Math.min(offsets.x.left, offsets.x.right);
	        var nearestYOffset = Math.min(offsets.y.top, offsets.y.bottom);
	        var scrollCursor = {row: cursor.row, column: cursor.column};
	        if (nearestXOffset / characterWidth <= 2) {
	            scrollCursor.column += (offsets.x.left < offsets.x.right ? -3 : +2);
	        }
	        if (nearestYOffset / lineHeight <= 1) {
	            scrollCursor.row += (offsets.y.top < offsets.y.bottom ? -1 : +1);
	        }
	        var vScroll = cursor.row != scrollCursor.row;
	        var hScroll = cursor.column != scrollCursor.column;
	        var vMovement = !prevCursor || cursor.row != prevCursor.row;
	        if (vScroll || (hScroll && !vMovement)) {
	            if (!autoScrollStartTime)
	                autoScrollStartTime = now;
	            else if (now - autoScrollStartTime >= AUTOSCROLL_DELAY)
	                editor.renderer.scrollCursorIntoView(scrollCursor);
	        } else {
	            autoScrollStartTime = null;
	        }
	    }

	    function onDragInterval() {
	        var prevCursor = dragCursor;
	        dragCursor = editor.renderer.screenToTextCoordinates(x, y);
	        scrollCursorIntoView(dragCursor, prevCursor);
	        autoScroll(dragCursor, prevCursor);
	    }

	    function addDragMarker() {
	        range = editor.selection.toOrientedRange();
	        dragSelectionMarker = editor.session.addMarker(range, "ace_selection", editor.getSelectionStyle());
	        editor.clearSelection();
	        if (editor.isFocused())
	            editor.renderer.$cursorLayer.setBlinking(false);
	        clearInterval(timerId);
	        onDragInterval();
	        timerId = setInterval(onDragInterval, 20);
	        counter = 0;
	        event.addListener(document, "mousemove", onMouseMove);
	    }

	    function clearDragMarker() {
	        clearInterval(timerId);
	        editor.session.removeMarker(dragSelectionMarker);
	        dragSelectionMarker = null;
	        editor.$blockScrolling += 1;
	        editor.selection.fromOrientedRange(range);
	        editor.$blockScrolling -= 1;
	        if (editor.isFocused() && !isInternal)
	            editor.renderer.$cursorLayer.setBlinking(!editor.getReadOnly());
	        range = null;
	        dragCursor = null;
	        counter = 0;
	        autoScrollStartTime = null;
	        cursorMovedTime = null;
	        event.removeListener(document, "mousemove", onMouseMove);
	    }
	    var onMouseMoveTimer = null;
	    function onMouseMove() {
	        if (onMouseMoveTimer == null) {
	            onMouseMoveTimer = setTimeout(function() {
	                if (onMouseMoveTimer != null && dragSelectionMarker)
	                    clearDragMarker();
	            }, 20);
	        }
	    }

	    function canAccept(dataTransfer) {
	        var types = dataTransfer.types;
	        return !types || Array.prototype.some.call(types, function(type) {
	            return type == 'text/plain' || type == 'Text';
	        });
	    }

	    function getDropEffect(e) {
	        var copyAllowed = ['copy', 'copymove', 'all', 'uninitialized'];
	        var moveAllowed = ['move', 'copymove', 'linkmove', 'all', 'uninitialized'];

	        var copyModifierState = useragent.isMac ? e.altKey : e.ctrlKey;
	        var effectAllowed = "uninitialized";
	        try {
	            effectAllowed = e.dataTransfer.effectAllowed.toLowerCase();
	        } catch (e) {}
	        var dropEffect = "none";

	        if (copyModifierState && copyAllowed.indexOf(effectAllowed) >= 0)
	            dropEffect = "copy";
	        else if (moveAllowed.indexOf(effectAllowed) >= 0)
	            dropEffect = "move";
	        else if (copyAllowed.indexOf(effectAllowed) >= 0)
	            dropEffect = "copy";

	        return dropEffect;
	    }
	}

	(function() {

	    this.dragWait = function() {
	        var interval = Date.now() - this.mousedownEvent.time;
	        if (interval > this.editor.getDragDelay())
	            this.startDrag();
	    };

	    this.dragWaitEnd = function() {
	        var target = this.editor.container;
	        target.draggable = false;
	        this.startSelect(this.mousedownEvent.getDocumentPosition());
	        this.selectEnd();
	    };

	    this.dragReadyEnd = function(e) {
	        this.editor.renderer.$cursorLayer.setBlinking(!this.editor.getReadOnly());
	        this.editor.unsetStyle("ace_dragging");
	        this.editor.renderer.setCursorStyle("");
	        this.dragWaitEnd();
	    };

	    this.startDrag = function(){
	        this.cancelDrag = false;
	        var editor = this.editor;
	        var target = editor.container;
	        target.draggable = true;
	        editor.renderer.$cursorLayer.setBlinking(false);
	        editor.setStyle("ace_dragging");
	        var cursorStyle = useragent.isWin ? "default" : "move";
	        editor.renderer.setCursorStyle(cursorStyle);
	        this.setState("dragReady");
	    };

	    this.onMouseDrag = function(e) {
	        var target = this.editor.container;
	        if (useragent.isIE && this.state == "dragReady") {
	            var distance = calcDistance(this.mousedownEvent.x, this.mousedownEvent.y, this.x, this.y);
	            if (distance > 3)
	                target.dragDrop();
	        }
	        if (this.state === "dragWait") {
	            var distance = calcDistance(this.mousedownEvent.x, this.mousedownEvent.y, this.x, this.y);
	            if (distance > 0) {
	                target.draggable = false;
	                this.startSelect(this.mousedownEvent.getDocumentPosition());
	            }
	        }
	    };

	    this.onMouseDown = function(e) {
	        if (!this.$dragEnabled)
	            return;
	        this.mousedownEvent = e;
	        var editor = this.editor;

	        var inSelection = e.inSelection();
	        var button = e.getButton();
	        var clickCount = e.domEvent.detail || 1;
	        if (clickCount === 1 && button === 0 && inSelection) {
	            if (e.editor.inMultiSelectMode && (e.getAccelKey() || e.getShiftKey()))
	                return;
	            this.mousedownEvent.time = Date.now();
	            var eventTarget = e.domEvent.target || e.domEvent.srcElement;
	            if ("unselectable" in eventTarget)
	                eventTarget.unselectable = "on";
	            if (editor.getDragDelay()) {
	                if (useragent.isWebKit) {
	                    this.cancelDrag = true;
	                    var mouseTarget = editor.container;
	                    mouseTarget.draggable = true;
	                }
	                this.setState("dragWait");
	            } else {
	                this.startDrag();
	            }
	            this.captureMouse(e, this.onMouseDrag.bind(this));
	            e.defaultPrevented = true;
	        }
	    };

	}).call(DragdropHandler.prototype);


	function calcDistance(ax, ay, bx, by) {
	    return Math.sqrt(Math.pow(bx - ax, 2) + Math.pow(by - ay, 2));
	}

	exports.DragdropHandler = DragdropHandler;

	});

	ace.define("ace/lib/net",["require","exports","module","ace/lib/dom"], function(acequire, exports, module) {
	"use strict";
	var dom = acequire("./dom");

	exports.get = function (url, callback) {
	    var xhr = new XMLHttpRequest();
	    xhr.open('GET', url, true);
	    xhr.onreadystatechange = function () {
	        if (xhr.readyState === 4) {
	            callback(xhr.responseText);
	        }
	    };
	    xhr.send(null);
	};

	exports.loadScript = function(path, callback) {
	    var head = dom.getDocumentHead();
	    var s = document.createElement('script');

	    s.src = path;
	    head.appendChild(s);

	    s.onload = s.onreadystatechange = function(_, isAbort) {
	        if (isAbort || !s.readyState || s.readyState == "loaded" || s.readyState == "complete") {
	            s = s.onload = s.onreadystatechange = null;
	            if (!isAbort)
	                callback();
	        }
	    };
	};
	exports.qualifyURL = function(url) {
	    var a = document.createElement('a');
	    a.href = url;
	    return a.href;
	};

	});

	ace.define("ace/lib/event_emitter",["require","exports","module"], function(acequire, exports, module) {
	"use strict";

	var EventEmitter = {};
	var stopPropagation = function() { this.propagationStopped = true; };
	var preventDefault = function() { this.defaultPrevented = true; };

	EventEmitter._emit =
	EventEmitter._dispatchEvent = function(eventName, e) {
	    this._eventRegistry || (this._eventRegistry = {});
	    this._defaultHandlers || (this._defaultHandlers = {});

	    var listeners = this._eventRegistry[eventName] || [];
	    var defaultHandler = this._defaultHandlers[eventName];
	    if (!listeners.length && !defaultHandler)
	        return;

	    if (typeof e != "object" || !e)
	        e = {};

	    if (!e.type)
	        e.type = eventName;
	    if (!e.stopPropagation)
	        e.stopPropagation = stopPropagation;
	    if (!e.preventDefault)
	        e.preventDefault = preventDefault;

	    listeners = listeners.slice();
	    for (var i=0; i<listeners.length; i++) {
	        listeners[i](e, this);
	        if (e.propagationStopped)
	            break;
	    }
	    
	    if (defaultHandler && !e.defaultPrevented)
	        return defaultHandler(e, this);
	};


	EventEmitter._signal = function(eventName, e) {
	    var listeners = (this._eventRegistry || {})[eventName];
	    if (!listeners)
	        return;
	    listeners = listeners.slice();
	    for (var i=0; i<listeners.length; i++)
	        listeners[i](e, this);
	};

	EventEmitter.once = function(eventName, callback) {
	    var _self = this;
	    callback && this.addEventListener(eventName, function newCallback() {
	        _self.removeEventListener(eventName, newCallback);
	        callback.apply(null, arguments);
	    });
	};


	EventEmitter.setDefaultHandler = function(eventName, callback) {
	    var handlers = this._defaultHandlers;
	    if (!handlers)
	        handlers = this._defaultHandlers = {_disabled_: {}};
	    
	    if (handlers[eventName]) {
	        var old = handlers[eventName];
	        var disabled = handlers._disabled_[eventName];
	        if (!disabled)
	            handlers._disabled_[eventName] = disabled = [];
	        disabled.push(old);
	        var i = disabled.indexOf(callback);
	        if (i != -1) 
	            disabled.splice(i, 1);
	    }
	    handlers[eventName] = callback;
	};
	EventEmitter.removeDefaultHandler = function(eventName, callback) {
	    var handlers = this._defaultHandlers;
	    if (!handlers)
	        return;
	    var disabled = handlers._disabled_[eventName];
	    
	    if (handlers[eventName] == callback) {
	        var old = handlers[eventName];
	        if (disabled)
	            this.setDefaultHandler(eventName, disabled.pop());
	    } else if (disabled) {
	        var i = disabled.indexOf(callback);
	        if (i != -1)
	            disabled.splice(i, 1);
	    }
	};

	EventEmitter.on =
	EventEmitter.addEventListener = function(eventName, callback, capturing) {
	    this._eventRegistry = this._eventRegistry || {};

	    var listeners = this._eventRegistry[eventName];
	    if (!listeners)
	        listeners = this._eventRegistry[eventName] = [];

	    if (listeners.indexOf(callback) == -1)
	        listeners[capturing ? "unshift" : "push"](callback);
	    return callback;
	};

	EventEmitter.off =
	EventEmitter.removeListener =
	EventEmitter.removeEventListener = function(eventName, callback) {
	    this._eventRegistry = this._eventRegistry || {};

	    var listeners = this._eventRegistry[eventName];
	    if (!listeners)
	        return;

	    var index = listeners.indexOf(callback);
	    if (index !== -1)
	        listeners.splice(index, 1);
	};

	EventEmitter.removeAllListeners = function(eventName) {
	    if (this._eventRegistry) this._eventRegistry[eventName] = [];
	};

	exports.EventEmitter = EventEmitter;

	});

	ace.define("ace/lib/app_config",["require","exports","module","ace/lib/oop","ace/lib/event_emitter"], function(acequire, exports, module) {
	"no use strict";

	var oop = acequire("./oop");
	var EventEmitter = acequire("./event_emitter").EventEmitter;

	var optionsProvider = {
	    setOptions: function(optList) {
	        Object.keys(optList).forEach(function(key) {
	            this.setOption(key, optList[key]);
	        }, this);
	    },
	    getOptions: function(optionNames) {
	        var result = {};
	        if (!optionNames) {
	            optionNames = Object.keys(this.$options);
	        } else if (!Array.isArray(optionNames)) {
	            result = optionNames;
	            optionNames = Object.keys(result);
	        }
	        optionNames.forEach(function(key) {
	            result[key] = this.getOption(key);
	        }, this);
	        return result;
	    },
	    setOption: function(name, value) {
	        if (this["$" + name] === value)
	            return;
	        var opt = this.$options[name];
	        if (!opt) {
	            return warn('misspelled option "' + name + '"');
	        }
	        if (opt.forwardTo)
	            return this[opt.forwardTo] && this[opt.forwardTo].setOption(name, value);

	        if (!opt.handlesSet)
	            this["$" + name] = value;
	        if (opt && opt.set)
	            opt.set.call(this, value);
	    },
	    getOption: function(name) {
	        var opt = this.$options[name];
	        if (!opt) {
	            return warn('misspelled option "' + name + '"');
	        }
	        if (opt.forwardTo)
	            return this[opt.forwardTo] && this[opt.forwardTo].getOption(name);
	        return opt && opt.get ? opt.get.call(this) : this["$" + name];
	    }
	};

	function warn(message) {
	    if (typeof console != "undefined" && console.warn)
	        console.warn.apply(console, arguments);
	}

	function reportError(msg, data) {
	    var e = new Error(msg);
	    e.data = data;
	    if (typeof console == "object" && console.error)
	        console.error(e);
	    setTimeout(function() { throw e; });
	}

	var AppConfig = function() {
	    this.$defaultOptions = {};
	};

	(function() {
	    oop.implement(this, EventEmitter);
	    this.defineOptions = function(obj, path, options) {
	        if (!obj.$options)
	            this.$defaultOptions[path] = obj.$options = {};

	        Object.keys(options).forEach(function(key) {
	            var opt = options[key];
	            if (typeof opt == "string")
	                opt = {forwardTo: opt};

	            opt.name || (opt.name = key);
	            obj.$options[opt.name] = opt;
	            if ("initialValue" in opt)
	                obj["$" + opt.name] = opt.initialValue;
	        });
	        oop.implement(obj, optionsProvider);

	        return this;
	    };

	    this.resetOptions = function(obj) {
	        Object.keys(obj.$options).forEach(function(key) {
	            var opt = obj.$options[key];
	            if ("value" in opt)
	                obj.setOption(key, opt.value);
	        });
	    };

	    this.setDefaultValue = function(path, name, value) {
	        var opts = this.$defaultOptions[path] || (this.$defaultOptions[path] = {});
	        if (opts[name]) {
	            if (opts.forwardTo)
	                this.setDefaultValue(opts.forwardTo, name, value);
	            else
	                opts[name].value = value;
	        }
	    };

	    this.setDefaultValues = function(path, optionHash) {
	        Object.keys(optionHash).forEach(function(key) {
	            this.setDefaultValue(path, key, optionHash[key]);
	        }, this);
	    };
	    
	    this.warn = warn;
	    this.reportError = reportError;
	    
	}).call(AppConfig.prototype);

	exports.AppConfig = AppConfig;

	});

	ace.define("ace/config",["require","exports","module","ace/lib/lang","ace/lib/oop","ace/lib/net","ace/lib/app_config"], function(acequire, exports, module) {
	"no use strict";

	var lang = acequire("./lib/lang");
	var oop = acequire("./lib/oop");
	var net = acequire("./lib/net");
	var AppConfig = acequire("./lib/app_config").AppConfig;

	module.exports = exports = new AppConfig();

	var global = (function() {
	    return this || typeof window != "undefined" && window;
	})();

	var options = {
	    packaged: false,
	    workerPath: null,
	    modePath: null,
	    themePath: null,
	    basePath: "",
	    suffix: ".js",
	    $moduleUrls: {}
	};

	exports.get = function(key) {
	    if (!options.hasOwnProperty(key))
	        throw new Error("Unknown config key: " + key);

	    return options[key];
	};

	exports.set = function(key, value) {
	    if (!options.hasOwnProperty(key))
	        throw new Error("Unknown config key: " + key);

	    options[key] = value;
	};

	exports.all = function() {
	    return lang.copyObject(options);
	};
	exports.moduleUrl = function(name, component) {
	    if (options.$moduleUrls[name])
	        return options.$moduleUrls[name];

	    var parts = name.split("/");
	    component = component || parts[parts.length - 2] || "";
	    var sep = component == "snippets" ? "/" : "-";
	    var base = parts[parts.length - 1];
	    if (component == "worker" && sep == "-") {
	        var re = new RegExp("^" + component + "[\\-_]|[\\-_]" + component + "$", "g");
	        base = base.replace(re, "");
	    }

	    if ((!base || base == component) && parts.length > 1)
	        base = parts[parts.length - 2];
	    var path = options[component + "Path"];
	    if (path == null) {
	        path = options.basePath;
	    } else if (sep == "/") {
	        component = sep = "";
	    }
	    if (path && path.slice(-1) != "/")
	        path += "/";
	    return path + component + sep + base + this.get("suffix");
	};

	exports.setModuleUrl = function(name, subst) {
	    return options.$moduleUrls[name] = subst;
	};

	exports.$loading = {};
	exports.loadModule = function(moduleName, onLoad) {
	    var module, moduleType;
	    if (Array.isArray(moduleName)) {
	        moduleType = moduleName[0];
	        moduleName = moduleName[1];
	    }

	    try {
	        module = acequire(moduleName);
	    } catch (e) {}
	    if (module && !exports.$loading[moduleName])
	        return onLoad && onLoad(module);

	    if (!exports.$loading[moduleName])
	        exports.$loading[moduleName] = [];

	    exports.$loading[moduleName].push(onLoad);

	    if (exports.$loading[moduleName].length > 1)
	        return;

	    var afterLoad = function() {
	        acequire([moduleName], function(module) {
	            exports._emit("load.module", {name: moduleName, module: module});
	            var listeners = exports.$loading[moduleName];
	            exports.$loading[moduleName] = null;
	            listeners.forEach(function(onLoad) {
	                onLoad && onLoad(module);
	            });
	        });
	    };

	    if (!exports.get("packaged"))
	        return afterLoad();
	    net.loadScript(exports.moduleUrl(moduleName, moduleType), afterLoad);
	};
	init(true);function init(packaged) {

	    if (!global || !global.document)
	        return;
	    
	    options.packaged = packaged || acequire.packaged || module.packaged || (global.define && __webpack_require__(68).packaged);

	    var scriptOptions = {};
	    var scriptUrl = "";
	    var currentScript = (document.currentScript || document._currentScript ); // native or polyfill
	    var currentDocument = currentScript && currentScript.ownerDocument || document;
	    
	    var scripts = currentDocument.getElementsByTagName("script");
	    for (var i=0; i<scripts.length; i++) {
	        var script = scripts[i];

	        var src = script.src || script.getAttribute("src");
	        if (!src)
	            continue;

	        var attributes = script.attributes;
	        for (var j=0, l=attributes.length; j < l; j++) {
	            var attr = attributes[j];
	            if (attr.name.indexOf("data-ace-") === 0) {
	                scriptOptions[deHyphenate(attr.name.replace(/^data-ace-/, ""))] = attr.value;
	            }
	        }

	        var m = src.match(/^(.*)\/ace(\-\w+)?\.js(\?|$)/);
	        if (m)
	            scriptUrl = m[1];
	    }

	    if (scriptUrl) {
	        scriptOptions.base = scriptOptions.base || scriptUrl;
	        scriptOptions.packaged = true;
	    }

	    scriptOptions.basePath = scriptOptions.base;
	    scriptOptions.workerPath = scriptOptions.workerPath || scriptOptions.base;
	    scriptOptions.modePath = scriptOptions.modePath || scriptOptions.base;
	    scriptOptions.themePath = scriptOptions.themePath || scriptOptions.base;
	    delete scriptOptions.base;

	    for (var key in scriptOptions)
	        if (typeof scriptOptions[key] !== "undefined")
	            exports.set(key, scriptOptions[key]);
	}

	exports.init = init;

	function deHyphenate(str) {
	    return str.replace(/-(.)/g, function(m, m1) { return m1.toUpperCase(); });
	}

	});

	ace.define("ace/mouse/mouse_handler",["require","exports","module","ace/lib/event","ace/lib/useragent","ace/mouse/default_handlers","ace/mouse/default_gutter_handler","ace/mouse/mouse_event","ace/mouse/dragdrop_handler","ace/config"], function(acequire, exports, module) {
	"use strict";

	var event = acequire("../lib/event");
	var useragent = acequire("../lib/useragent");
	var DefaultHandlers = acequire("./default_handlers").DefaultHandlers;
	var DefaultGutterHandler = acequire("./default_gutter_handler").GutterHandler;
	var MouseEvent = acequire("./mouse_event").MouseEvent;
	var DragdropHandler = acequire("./dragdrop_handler").DragdropHandler;
	var config = acequire("../config");

	var MouseHandler = function(editor) {
	    var _self = this;
	    this.editor = editor;

	    new DefaultHandlers(this);
	    new DefaultGutterHandler(this);
	    new DragdropHandler(this);

	    var focusEditor = function(e) {
	        var windowBlurred = !document.hasFocus || !document.hasFocus()
	            || !editor.isFocused() && document.activeElement == (editor.textInput && editor.textInput.getElement());
	        if (windowBlurred)
	            window.focus();
	        editor.focus();
	    };

	    var mouseTarget = editor.renderer.getMouseEventTarget();
	    event.addListener(mouseTarget, "click", this.onMouseEvent.bind(this, "click"));
	    event.addListener(mouseTarget, "mousemove", this.onMouseMove.bind(this, "mousemove"));
	    event.addMultiMouseDownListener([
	        mouseTarget,
	        editor.renderer.scrollBarV && editor.renderer.scrollBarV.inner,
	        editor.renderer.scrollBarH && editor.renderer.scrollBarH.inner,
	        editor.textInput && editor.textInput.getElement()
	    ].filter(Boolean), [400, 300, 250], this, "onMouseEvent");
	    event.addMouseWheelListener(editor.container, this.onMouseWheel.bind(this, "mousewheel"));
	    event.addTouchMoveListener(editor.container, this.onTouchMove.bind(this, "touchmove"));

	    var gutterEl = editor.renderer.$gutter;
	    event.addListener(gutterEl, "mousedown", this.onMouseEvent.bind(this, "guttermousedown"));
	    event.addListener(gutterEl, "click", this.onMouseEvent.bind(this, "gutterclick"));
	    event.addListener(gutterEl, "dblclick", this.onMouseEvent.bind(this, "gutterdblclick"));
	    event.addListener(gutterEl, "mousemove", this.onMouseEvent.bind(this, "guttermousemove"));

	    event.addListener(mouseTarget, "mousedown", focusEditor);
	    event.addListener(gutterEl, "mousedown", focusEditor);
	    if (useragent.isIE && editor.renderer.scrollBarV) {
	        event.addListener(editor.renderer.scrollBarV.element, "mousedown", focusEditor);
	        event.addListener(editor.renderer.scrollBarH.element, "mousedown", focusEditor);
	    }

	    editor.on("mousemove", function(e){
	        if (_self.state || _self.$dragDelay || !_self.$dragEnabled)
	            return;

	        var character = editor.renderer.screenToTextCoordinates(e.x, e.y);
	        var range = editor.session.selection.getRange();
	        var renderer = editor.renderer;

	        if (!range.isEmpty() && range.insideStart(character.row, character.column)) {
	            renderer.setCursorStyle("default");
	        } else {
	            renderer.setCursorStyle("");
	        }
	    });
	};

	(function() {
	    this.onMouseEvent = function(name, e) {
	        this.editor._emit(name, new MouseEvent(e, this.editor));
	    };

	    this.onMouseMove = function(name, e) {
	        var listeners = this.editor._eventRegistry && this.editor._eventRegistry.mousemove;
	        if (!listeners || !listeners.length)
	            return;

	        this.editor._emit(name, new MouseEvent(e, this.editor));
	    };

	    this.onMouseWheel = function(name, e) {
	        var mouseEvent = new MouseEvent(e, this.editor);
	        mouseEvent.speed = this.$scrollSpeed * 2;
	        mouseEvent.wheelX = e.wheelX;
	        mouseEvent.wheelY = e.wheelY;

	        this.editor._emit(name, mouseEvent);
	    };
	    
	    this.onTouchMove = function (name, e) {
	        var mouseEvent = new MouseEvent(e, this.editor);
	        mouseEvent.speed = 1;//this.$scrollSpeed * 2;
	        mouseEvent.wheelX = e.wheelX;
	        mouseEvent.wheelY = e.wheelY;
	        this.editor._emit(name, mouseEvent);
	    };

	    this.setState = function(state) {
	        this.state = state;
	    };

	    this.captureMouse = function(ev, mouseMoveHandler) {
	        this.x = ev.x;
	        this.y = ev.y;

	        this.isMousePressed = true;
	        var renderer = this.editor.renderer;
	        if (renderer.$keepTextAreaAtCursor)
	            renderer.$keepTextAreaAtCursor = null;

	        var self = this;
	        var onMouseMove = function(e) {
	            if (!e) return;
	            if (useragent.isWebKit && !e.which && self.releaseMouse)
	                return self.releaseMouse();

	            self.x = e.clientX;
	            self.y = e.clientY;
	            mouseMoveHandler && mouseMoveHandler(e);
	            self.mouseEvent = new MouseEvent(e, self.editor);
	            self.$mouseMoved = true;
	        };

	        var onCaptureEnd = function(e) {
	            clearInterval(timerId);
	            onCaptureInterval();
	            self[self.state + "End"] && self[self.state + "End"](e);
	            self.state = "";
	            if (renderer.$keepTextAreaAtCursor == null) {
	                renderer.$keepTextAreaAtCursor = true;
	                renderer.$moveTextAreaToCursor();
	            }
	            self.isMousePressed = false;
	            self.$onCaptureMouseMove = self.releaseMouse = null;
	            e && self.onMouseEvent("mouseup", e);
	        };

	        var onCaptureInterval = function() {
	            self[self.state] && self[self.state]();
	            self.$mouseMoved = false;
	        };

	        if (useragent.isOldIE && ev.domEvent.type == "dblclick") {
	            return setTimeout(function() {onCaptureEnd(ev);});
	        }

	        self.$onCaptureMouseMove = onMouseMove;
	        self.releaseMouse = event.capture(this.editor.container, onMouseMove, onCaptureEnd);
	        var timerId = setInterval(onCaptureInterval, 20);
	    };
	    this.releaseMouse = null;
	    this.cancelContextMenu = function() {
	        var stop = function(e) {
	            if (e && e.domEvent && e.domEvent.type != "contextmenu")
	                return;
	            this.editor.off("nativecontextmenu", stop);
	            if (e && e.domEvent)
	                event.stopEvent(e.domEvent);
	        }.bind(this);
	        setTimeout(stop, 10);
	        this.editor.on("nativecontextmenu", stop);
	    };
	}).call(MouseHandler.prototype);

	config.defineOptions(MouseHandler.prototype, "mouseHandler", {
	    scrollSpeed: {initialValue: 2},
	    dragDelay: {initialValue: (useragent.isMac ? 150 : 0)},
	    dragEnabled: {initialValue: true},
	    focusTimout: {initialValue: 0},
	    tooltipFollowsMouse: {initialValue: true}
	});


	exports.MouseHandler = MouseHandler;
	});

	ace.define("ace/mouse/fold_handler",["require","exports","module"], function(acequire, exports, module) {
	"use strict";

	function FoldHandler(editor) {

	    editor.on("click", function(e) {
	        var position = e.getDocumentPosition();
	        var session = editor.session;
	        var fold = session.getFoldAt(position.row, position.column, 1);
	        if (fold) {
	            if (e.getAccelKey())
	                session.removeFold(fold);
	            else
	                session.expandFold(fold);

	            e.stop();
	        }
	    });

	    editor.on("gutterclick", function(e) {
	        var gutterRegion = editor.renderer.$gutterLayer.getRegion(e);

	        if (gutterRegion == "foldWidgets") {
	            var row = e.getDocumentPosition().row;
	            var session = editor.session;
	            if (session.foldWidgets && session.foldWidgets[row])
	                editor.session.onFoldWidgetClick(row, e);
	            if (!editor.isFocused())
	                editor.focus();
	            e.stop();
	        }
	    });

	    editor.on("gutterdblclick", function(e) {
	        var gutterRegion = editor.renderer.$gutterLayer.getRegion(e);

	        if (gutterRegion == "foldWidgets") {
	            var row = e.getDocumentPosition().row;
	            var session = editor.session;
	            var data = session.getParentFoldRangeData(row, true);
	            var range = data.range || data.firstRange;

	            if (range) {
	                row = range.start.row;
	                var fold = session.getFoldAt(row, session.getLine(row).length, 1);

	                if (fold) {
	                    session.removeFold(fold);
	                } else {
	                    session.addFold("...", range);
	                    editor.renderer.scrollCursorIntoView({row: range.start.row, column: 0});
	                }
	            }
	            e.stop();
	        }
	    });
	}

	exports.FoldHandler = FoldHandler;

	});

	ace.define("ace/keyboard/keybinding",["require","exports","module","ace/lib/keys","ace/lib/event"], function(acequire, exports, module) {
	"use strict";

	var keyUtil  = acequire("../lib/keys");
	var event = acequire("../lib/event");

	var KeyBinding = function(editor) {
	    this.$editor = editor;
	    this.$data = {editor: editor};
	    this.$handlers = [];
	    this.setDefaultHandler(editor.commands);
	};

	(function() {
	    this.setDefaultHandler = function(kb) {
	        this.removeKeyboardHandler(this.$defaultHandler);
	        this.$defaultHandler = kb;
	        this.addKeyboardHandler(kb, 0);
	    };

	    this.setKeyboardHandler = function(kb) {
	        var h = this.$handlers;
	        if (h[h.length - 1] == kb)
	            return;

	        while (h[h.length - 1] && h[h.length - 1] != this.$defaultHandler)
	            this.removeKeyboardHandler(h[h.length - 1]);

	        this.addKeyboardHandler(kb, 1);
	    };

	    this.addKeyboardHandler = function(kb, pos) {
	        if (!kb)
	            return;
	        if (typeof kb == "function" && !kb.handleKeyboard)
	            kb.handleKeyboard = kb;
	        var i = this.$handlers.indexOf(kb);
	        if (i != -1)
	            this.$handlers.splice(i, 1);

	        if (pos == undefined)
	            this.$handlers.push(kb);
	        else
	            this.$handlers.splice(pos, 0, kb);

	        if (i == -1 && kb.attach)
	            kb.attach(this.$editor);
	    };

	    this.removeKeyboardHandler = function(kb) {
	        var i = this.$handlers.indexOf(kb);
	        if (i == -1)
	            return false;
	        this.$handlers.splice(i, 1);
	        kb.detach && kb.detach(this.$editor);
	        return true;
	    };

	    this.getKeyboardHandler = function() {
	        return this.$handlers[this.$handlers.length - 1];
	    };
	    
	    this.getStatusText = function() {
	        var data = this.$data;
	        var editor = data.editor;
	        return this.$handlers.map(function(h) {
	            return h.getStatusText && h.getStatusText(editor, data) || "";
	        }).filter(Boolean).join(" ");
	    };

	    this.$callKeyboardHandlers = function(hashId, keyString, keyCode, e) {
	        var toExecute;
	        var success = false;
	        var commands = this.$editor.commands;

	        for (var i = this.$handlers.length; i--;) {
	            toExecute = this.$handlers[i].handleKeyboard(
	                this.$data, hashId, keyString, keyCode, e
	            );
	            if (!toExecute || !toExecute.command)
	                continue;
	            if (toExecute.command == "null") {
	                success = true;
	            } else {
	                success = commands.exec(toExecute.command, this.$editor, toExecute.args, e);
	            }
	            if (success && e && hashId != -1 && 
	                toExecute.passEvent != true && toExecute.command.passEvent != true
	            ) {
	                event.stopEvent(e);
	            }
	            if (success)
	                break;
	        }
	        
	        if (!success && hashId == -1) {
	            toExecute = {command: "insertstring"};
	            success = commands.exec("insertstring", this.$editor, keyString);
	        }
	        
	        if (success && this.$editor._signal)
	            this.$editor._signal("keyboardActivity", toExecute);
	        
	        return success;
	    };

	    this.onCommandKey = function(e, hashId, keyCode) {
	        var keyString = keyUtil.keyCodeToString(keyCode);
	        this.$callKeyboardHandlers(hashId, keyString, keyCode, e);
	    };

	    this.onTextInput = function(text) {
	        this.$callKeyboardHandlers(-1, text);
	    };

	}).call(KeyBinding.prototype);

	exports.KeyBinding = KeyBinding;
	});

	ace.define("ace/lib/bidiutil",["require","exports","module"], function(acequire, exports, module) {
	"use strict";

	var ArabicAlefBetIntervalsBegine = ['\u0621', '\u0641'];
	var ArabicAlefBetIntervalsEnd = ['\u063A', '\u064a'];
	var dir = 0, hiLevel = 0;
	var lastArabic = false, hasUBAT_AL = false,  hasUBAT_B = false,  hasUBAT_S = false, hasBlockSep = false, hasSegSep = false;

	var impTab_LTR = [	[	0,		3,		0,		1,		0,		0,		0	],	[	0,		3,		0,		1,		2,		2,		0	],	[	0,		3,		0,		0x11,		2,		0,		1	],	[	0,		3,		5,		5,		4,		1,		0	],	[	0,		3,		0x15,		0x15,		4,		0,		1	],	[	0,		3,		5,		5,		4,		2,		0	]
	];

	var impTab_RTL = [	[	2,		0,		1,		1,		0,		1,		0	],	[	2,		0,		1,		1,		0,		2,		0	],	[	2,		0,		2,		1,		3,		2,		0	],	[	2,		0,		2,		0x21,		3,		1,		1	]
	];

	var LTR = 0, RTL = 1;

	var L = 0;
	var R = 1;
	var EN = 2;
	var AN = 3;
	var ON = 4;
	var B = 5;
	var S = 6;
	var AL = 7;
	var WS = 8;
	var CS = 9;
	var ES = 10;
	var ET = 11;
	var NSM = 12;
	var LRE = 13;
	var RLE = 14;
	var PDF = 15;
	var LRO = 16;
	var RLO = 17;
	var BN = 18;

	var UnicodeTBL00 = [
	BN,BN,BN,BN,BN,BN,BN,BN,BN,S,B,S,WS,B,BN,BN,
	BN,BN,BN,BN,BN,BN,BN,BN,BN,BN,BN,BN,B,B,B,S,
	WS,ON,ON,ET,ET,ET,ON,ON,ON,ON,ON,ES,CS,ES,CS,CS,
	EN,EN,EN,EN,EN,EN,EN,EN,EN,EN,CS,ON,ON,ON,ON,ON,
	ON,L,L,L,L,L,L,L,L,L,L,L,L,L,L,L,
	L,L,L,L,L,L,L,L,L,L,L,ON,ON,ON,ON,ON,
	ON,L,L,L,L,L,L,L,L,L,L,L,L,L,L,L,
	L,L,L,L,L,L,L,L,L,L,L,ON,ON,ON,ON,BN,
	BN,BN,BN,BN,BN,B,BN,BN,BN,BN,BN,BN,BN,BN,BN,BN,
	BN,BN,BN,BN,BN,BN,BN,BN,BN,BN,BN,BN,BN,BN,BN,BN,
	CS,ON,ET,ET,ET,ET,ON,ON,ON,ON,L,ON,ON,BN,ON,ON,
	ET,ET,EN,EN,ON,L,ON,ON,ON,EN,L,ON,ON,ON,ON,ON
	];

	var UnicodeTBL20 = [
	WS,WS,WS,WS,WS,WS,WS,WS,WS,WS,WS,BN,BN,BN,L,R	,
	ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,
	ON,ON,ON,ON,ON,ON,ON,ON,WS,B,LRE,RLE,PDF,LRO,RLO,CS,
	ET,ET,ET,ET,ET,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,
	ON,ON,ON,ON,CS,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,
	ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,ON,WS
	];

	function _computeLevels(chars, levels, len, charTypes) {
		var impTab = dir ? impTab_RTL : impTab_LTR
			, prevState = null, newClass = null, newLevel = null, newState = 0
			, action = null, cond = null, condPos = -1, i = null, ix = null, classes = [];

		if (!charTypes) {
			for (i = 0, charTypes = []; i < len; i++) {
				charTypes[i] = _getCharacterType(chars[i]);
			}
		}
		hiLevel = dir;
		lastArabic = false;
		hasUBAT_AL = false;
		hasUBAT_B = false;
		hasUBAT_S = false;
		for (ix = 0; ix < len; ix++){
			prevState = newState;
			classes[ix] = newClass = _getCharClass(chars, charTypes, classes, ix);
			newState = impTab[prevState][newClass];
			action = newState & 0xF0;
			newState &= 0x0F;
			levels[ix] = newLevel = impTab[newState][5];
			if (action > 0){
				if (action == 0x10){
					for(i = condPos; i < ix; i++){
						levels[i] = 1;
					}
					condPos = -1;
				} else {
					condPos = -1;
				}
			}
			cond = impTab[newState][6];
			if (cond){
				if(condPos == -1){
					condPos = ix;
				}
			}else{
				if (condPos > -1){
					for(i = condPos; i < ix; i++){
						levels[i] = newLevel;
					}
					condPos = -1;
				}
			}
			if (charTypes[ix] == B){
				levels[ix] = 0;
			}
			hiLevel |= newLevel;
		}
		if (hasUBAT_S){
			for(i = 0; i < len; i++){
				if(charTypes[i] == S){
					levels[i] = dir;
					for(var j = i - 1; j >= 0; j--){
						if(charTypes[j] == WS){
							levels[j] = dir;
						}else{
							break;
						}
					}
				}
			}
		}
	}

	function _invertLevel(lev, levels, _array) {
		if (hiLevel < lev){
			return;
		}
		if (lev == 1 && dir == RTL && !hasUBAT_B){
			_array.reverse();
			return;
		}
		var len = _array.length, start = 0, end, lo, hi, tmp;
		while(start < len){
			if (levels[start] >= lev){
				end = start + 1;
			while(end < len && levels[end] >= lev){
				end++;
			}
			for(lo = start, hi = end - 1 ; lo < hi; lo++, hi--){
				tmp = _array[lo];
				_array[lo] = _array[hi];
				_array[hi] = tmp;
			}
			start = end;
		}
		start++;
		}
	}

	function _getCharClass(chars, types, classes, ix) {
		var cType = types[ix], wType, nType, len, i;
		switch(cType){
			case L:
			case R:
				lastArabic = false;
			case ON:
			case AN:
				return cType;
			case EN:
				return lastArabic ? AN : EN;
			case AL:
				lastArabic = true;
				hasUBAT_AL = true;
				return R;
			case WS:
				return ON;
			case CS:
				if (ix < 1 || (ix + 1) >= types.length ||
					((wType = classes[ix - 1]) != EN && wType != AN) ||
					((nType = types[ix + 1]) != EN && nType != AN)){
					return ON;
				}
				if (lastArabic){nType = AN;}
				return nType == wType ? nType : ON;
			case ES:
				wType = ix > 0 ? classes[ix - 1] : B;
				if (wType == EN && (ix + 1) < types.length && types[ix + 1] == EN){
					return EN;
				}
				return ON;
			case ET:
				if (ix > 0 && classes[ix - 1] == EN){
					return EN;
				}
				if (lastArabic){
					return ON;
				}
				i = ix + 1;
				len = types.length;
				while (i < len && types[i] == ET){
					i++;
				}
				if (i < len && types[i] == EN){
					return EN;
				}
				return ON;
			case NSM:
				len = types.length;
				i = ix + 1;
				while (i < len && types[i] == NSM){
					i++;
				}
				if (i < len){
					var c = chars[ix], rtlCandidate = (c >= 0x0591 && c <= 0x08FF) || c == 0xFB1E;

					wType = types[i];
					if (rtlCandidate && (wType == R || wType == AL)){
						return R;
					}
				}

				if (ix < 1 || (wType = types[ix - 1]) == B){
					return ON;
				}
				return classes[ix - 1];
			case B:
				lastArabic = false;
				hasUBAT_B = true;
				return dir;
			case S:
				hasUBAT_S = true;
				return ON;
			case LRE:
			case RLE:
			case LRO:
			case RLO:
			case PDF:
				lastArabic = false;
			case BN:
				return ON;
		}
	}

	function _getCharacterType( ch ) {
		var uc = ch.charCodeAt(0), hi = uc >> 8;

		if (hi == 0) {
			return ((uc > 0x00BF) ? L : UnicodeTBL00[uc]);
		} else if (hi == 5) {
			return (/[\u0591-\u05f4]/.test(ch) ? R : L);
		} else if (hi == 6) {
			if (/[\u0610-\u061a\u064b-\u065f\u06d6-\u06e4\u06e7-\u06ed]/.test(ch))
				return NSM;
			else if (/[\u0660-\u0669\u066b-\u066c]/.test(ch))
				return AN;
			else if (uc == 0x066A)
				return ET;
			else if (/[\u06f0-\u06f9]/.test(ch))
				return EN;
			else
				return AL;
		} else if (hi == 0x20 && uc <= 0x205F) {
			return UnicodeTBL20[uc & 0xFF];
		} else if (hi == 0xFE) {
			return (uc >= 0xFE70 ? AL : ON);
		}
		return ON;
	}

	function _isArabicDiacritics( ch ) {
		return (ch >= '\u064b' && ch <= '\u0655');
	}
	exports.L = L;
	exports.R = R;
	exports.EN = EN;
	exports.ON_R = 3;
	exports.AN = 4;
	exports.R_H = 5;
	exports.B = 6;

	exports.DOT = "\xB7";
	exports.doBidiReorder = function(text, textCharTypes, isRtl) {
		if (text.length < 2)
			return {};

		var chars = text.split(""), logicalFromVisual = new Array(chars.length),
			bidiLevels = new Array(chars.length), levels = [];

		dir = isRtl ? RTL : LTR;

		_computeLevels(chars, levels, chars.length, textCharTypes);

		for (var i = 0; i < logicalFromVisual.length; logicalFromVisual[i] = i, i++);

		_invertLevel(2, levels, logicalFromVisual);
		_invertLevel(1, levels, logicalFromVisual);

		for (var i = 0; i < logicalFromVisual.length - 1; i++) { //fix levels to reflect character width
			if (textCharTypes[i] === AN) {
				levels[i] = exports.AN;
			} else if (levels[i] === R && ((textCharTypes[i] > AL && textCharTypes[i] < LRE)
				|| textCharTypes[i] === ON || textCharTypes[i] === BN)) {
				levels[i] = exports.ON_R;
			} else if ((i > 0 && chars[i - 1] === '\u0644') && /\u0622|\u0623|\u0625|\u0627/.test(chars[i])) {
				levels[i - 1] = levels[i] = exports.R_H;
				i++;
			}
		}
		if (chars[chars.length - 1] === exports.DOT)
			levels[chars.length - 1] = exports.B;

		for (var i = 0; i < logicalFromVisual.length; i++) {
			bidiLevels[i] = levels[logicalFromVisual[i]];
		}

		return {'logicalFromVisual': logicalFromVisual, 'bidiLevels': bidiLevels};
	};
	exports.hasBidiCharacters = function(text, textCharTypes){
		var ret = false;
		for (var i = 0; i < text.length; i++){
			textCharTypes[i] = _getCharacterType(text.charAt(i));
			if (!ret && (textCharTypes[i] == R || textCharTypes[i] == AL))
				ret = true;
		}
		return ret;
	};
	exports.getVisualFromLogicalIdx = function(logIdx, rowMap) {
		for (var i = 0; i < rowMap.logicalFromVisual.length; i++) {
			if (rowMap.logicalFromVisual[i] == logIdx)
				return i;
		}
		return 0;
	};

	});

	ace.define("ace/bidihandler",["require","exports","module","ace/lib/bidiutil","ace/lib/lang","ace/lib/useragent"], function(acequire, exports, module) {
	"use strict";

	var bidiUtil = acequire("./lib/bidiutil");
	var lang = acequire("./lib/lang");
	var useragent = acequire("./lib/useragent");
	var bidiRE = /[\u0590-\u05f4\u0600-\u06ff\u0700-\u08ac]/;
	var BidiHandler = function(session) {
	    this.session = session;
	    this.bidiMap = {};
	    this.currentRow = null;
	    this.bidiUtil = bidiUtil;
	    this.charWidths = [];
	    this.EOL = "\xAC";
	    this.showInvisibles = true;
	    this.isRtlDir = false;
	    this.line = "";
	    this.wrapIndent = 0;
	    this.isLastRow = false;
	    this.EOF = "\xB6";
	    this.seenBidi = false;
	};

	(function() {
	    this.isBidiRow = function(screenRow, docRow, splitIndex) {
	        if (!this.seenBidi)
	            return false;
	        if (screenRow !== this.currentRow) {
	            this.currentRow = screenRow;
	            this.updateRowLine(docRow, splitIndex);
	            this.updateBidiMap();
	        }
	        return this.bidiMap.bidiLevels;
	    };

	    this.onChange = function(delta) {
	        if (!this.seenBidi) {
	            if (delta.action == "insert" && bidiRE.test(delta.lines.join("\n"))) {
	                this.seenBidi = true;
	                this.currentRow = null;
	            }
	        }
	        else {
	            this.currentRow = null;
	        }
	    };

	    this.getDocumentRow = function() {
	        var docRow = 0;
	        var rowCache = this.session.$screenRowCache;
	        if (rowCache.length) {
	            var index = this.session.$getRowCacheIndex(rowCache, this.currentRow);
	            if (index >= 0)
	                docRow = this.session.$docRowCache[index];
	        }

	        return docRow;
	    };

	    this.getSplitIndex = function() {
	        var splitIndex = 0;
	        var rowCache = this.session.$screenRowCache;
	        if (rowCache.length) {
	            var currentIndex, prevIndex = this.session.$getRowCacheIndex(rowCache, this.currentRow);
	            while (this.currentRow - splitIndex > 0) {
	                currentIndex = this.session.$getRowCacheIndex(rowCache, this.currentRow - splitIndex - 1);
	                if (currentIndex !== prevIndex)
	                    break;

	                prevIndex = currentIndex;
	                splitIndex++;
	            }
	        }

	        return splitIndex;
	    };

	    this.updateRowLine = function(docRow, splitIndex) {
	        if (docRow === undefined)
	            docRow = this.getDocumentRow();

	        this.wrapIndent = 0;
	        this.isLastRow = (docRow === this.session.getLength() - 1);
	        this.line = this.session.getLine(docRow);
	        if (this.session.$useWrapMode) {
	            var splits = this.session.$wrapData[docRow];
	            if (splits) {
	                if (splitIndex === undefined)
	                    splitIndex = this.getSplitIndex();

	                if(splitIndex > 0 && splits.length) {
	                    this.wrapIndent = splits.indent;
	                    this.line = (splitIndex < splits.length) ?
	                        this.line.substring(splits[splitIndex - 1], splits[splits.length - 1]) :
	                            this.line.substring(splits[splits.length - 1]);
	                } else {
	                    this.line = this.line.substring(0, splits[splitIndex]);
	                }
	            }
	        }
	        var session = this.session, shift = 0, size;
	        this.line = this.line.replace(/\t|[\u1100-\u2029, \u202F-\uFFE6]/g, function(ch, i){
	            if (ch === '\t' || session.isFullWidth(ch.charCodeAt(0))) {
	                size = (ch === '\t') ? session.getScreenTabSize(i + shift) : 2;
	                shift += size - 1;
	                return lang.stringRepeat(bidiUtil.DOT, size);
	            }
	            return ch;
	        });
	    };

	    this.updateBidiMap = function() {
	        var textCharTypes = [], endOfLine = this.isLastRow ? this.EOF : this.EOL;
	        var line = this.line + (this.showInvisibles ? endOfLine : bidiUtil.DOT);
	        if (bidiUtil.hasBidiCharacters(line, textCharTypes)) {
	            this.bidiMap = bidiUtil.doBidiReorder(line, textCharTypes, this.isRtlDir);
	        } else {
	            this.bidiMap = {};
	        }
	    };
	    this.markAsDirty = function() {
	        this.currentRow = null;
	    };
	    this.updateCharacterWidths = function(fontMetrics) {
	        if (!this.seenBidi)
	            return;
	        if (this.characterWidth === fontMetrics.$characterSize.width)
	            return;

	        var characterWidth = this.characterWidth = fontMetrics.$characterSize.width;
	        var bidiCharWidth = fontMetrics.$measureCharWidth("\u05d4");

	        this.charWidths[bidiUtil.L] = this.charWidths[bidiUtil.EN] = this.charWidths[bidiUtil.ON_R] = characterWidth;
	        this.charWidths[bidiUtil.R] = this.charWidths[bidiUtil.AN] = bidiCharWidth;
	        this.charWidths[bidiUtil.R_H] = useragent.isChrome ? bidiCharWidth : bidiCharWidth * 0.45;
	        this.charWidths[bidiUtil.B] = 0;

	        this.currentRow = null;
	    };

	    this.getShowInvisibles = function() {
	        return this.showInvisibles;
	    };

	    this.setShowInvisibles = function(showInvisibles) {
	        this.showInvisibles = showInvisibles;
	        this.currentRow = null;
	    };

	    this.setEolChar = function(eolChar) {
	        this.EOL = eolChar;
	    };

	    this.setTextDir = function(isRtlDir) {
	        this.isRtlDir = isRtlDir;
	    };
	    this.getPosLeft = function(col) {
	        col -= this.wrapIndent;
	        var visualIdx = bidiUtil.getVisualFromLogicalIdx(col > 0 ? col - 1 : 0, this.bidiMap),
	            levels = this.bidiMap.bidiLevels, left = 0;

	        if (col === 0 && levels[visualIdx] % 2 !== 0)
	            visualIdx++;

	        for (var i = 0; i < visualIdx; i++) {
	            left += this.charWidths[levels[i]];
	        }

	        if (col !== 0 && levels[visualIdx] % 2 === 0)
	            left += this.charWidths[levels[visualIdx]];

	        if (this.wrapIndent)
	            left += this.wrapIndent * this.charWidths[bidiUtil.L];

	        return left;
	    };
	    this.getSelections = function(startCol, endCol) {
	        var map = this.bidiMap, levels = map.bidiLevels, level, offset = this.wrapIndent * this.charWidths[bidiUtil.L], selections = [],
	            selColMin = Math.min(startCol, endCol) - this.wrapIndent, selColMax = Math.max(startCol, endCol) - this.wrapIndent,
	                isSelected = false, isSelectedPrev = false, selectionStart = 0;

	        for (var logIdx, visIdx = 0; visIdx < levels.length; visIdx++) {
	            logIdx = map.logicalFromVisual[visIdx];
	            level = levels[visIdx];
	            isSelected = (logIdx >= selColMin) && (logIdx < selColMax);
	            if (isSelected && !isSelectedPrev) {
	                selectionStart = offset;
	            } else if (!isSelected && isSelectedPrev) {
	                selections.push({left: selectionStart, width: offset - selectionStart});
	            }
	            offset += this.charWidths[level];
	            isSelectedPrev = isSelected;
	        }

	        if (isSelected && (visIdx === levels.length)) {
	            selections.push({left: selectionStart, width: offset - selectionStart});
	        }

	        return selections;
	    };
	    this.offsetToCol = function(posX) {
	        var logicalIdx = 0, posX = Math.max(posX, 0),
	            offset = 0, visualIdx = 0, levels = this.bidiMap.bidiLevels,
	                charWidth = this.charWidths[levels[visualIdx]];

	        if (this.wrapIndent) {
	            posX -= this.wrapIndent * this.charWidths[bidiUtil.L];
	        }

	        while(posX > offset + charWidth/2) {
	            offset += charWidth;
	            if(visualIdx === levels.length - 1) {
	                charWidth = 0;
	                break;
	            }
	            charWidth = this.charWidths[levels[++visualIdx]];
	        }

	        if (visualIdx > 0 && (levels[visualIdx - 1] % 2 !== 0) && (levels[visualIdx] % 2 === 0)){
	            if(posX < offset)
	                visualIdx--;
	            logicalIdx = this.bidiMap.logicalFromVisual[visualIdx];

	        } else if (visualIdx > 0 && (levels[visualIdx - 1] % 2 === 0) && (levels[visualIdx] % 2 !== 0)){
	            logicalIdx = 1 + ((posX > offset) ? this.bidiMap.logicalFromVisual[visualIdx]
	                    : this.bidiMap.logicalFromVisual[visualIdx - 1]);

	        } else if ((this.isRtlDir && visualIdx === levels.length - 1 && charWidth === 0 && (levels[visualIdx - 1] % 2 === 0))
	                || (!this.isRtlDir && visualIdx === 0 && (levels[visualIdx] % 2 !== 0))){
	            logicalIdx = 1 + this.bidiMap.logicalFromVisual[visualIdx];
	        } else {
	            if (visualIdx > 0 && (levels[visualIdx - 1] % 2 !== 0) && charWidth !== 0)
	                visualIdx--;
	            logicalIdx = this.bidiMap.logicalFromVisual[visualIdx];
	        }

	        return (logicalIdx + this.wrapIndent);
	    };

	}).call(BidiHandler.prototype);

	exports.BidiHandler = BidiHandler;
	});

	ace.define("ace/range",["require","exports","module"], function(acequire, exports, module) {
	"use strict";
	var comparePoints = function(p1, p2) {
	    return p1.row - p2.row || p1.column - p2.column;
	};
	var Range = function(startRow, startColumn, endRow, endColumn) {
	    this.start = {
	        row: startRow,
	        column: startColumn
	    };

	    this.end = {
	        row: endRow,
	        column: endColumn
	    };
	};

	(function() {
	    this.isEqual = function(range) {
	        return this.start.row === range.start.row &&
	            this.end.row === range.end.row &&
	            this.start.column === range.start.column &&
	            this.end.column === range.end.column;
	    };
	    this.toString = function() {
	        return ("Range: [" + this.start.row + "/" + this.start.column +
	            "] -> [" + this.end.row + "/" + this.end.column + "]");
	    };

	    this.contains = function(row, column) {
	        return this.compare(row, column) == 0;
	    };
	    this.compareRange = function(range) {
	        var cmp,
	            end = range.end,
	            start = range.start;

	        cmp = this.compare(end.row, end.column);
	        if (cmp == 1) {
	            cmp = this.compare(start.row, start.column);
	            if (cmp == 1) {
	                return 2;
	            } else if (cmp == 0) {
	                return 1;
	            } else {
	                return 0;
	            }
	        } else if (cmp == -1) {
	            return -2;
	        } else {
	            cmp = this.compare(start.row, start.column);
	            if (cmp == -1) {
	                return -1;
	            } else if (cmp == 1) {
	                return 42;
	            } else {
	                return 0;
	            }
	        }
	    };
	    this.comparePoint = function(p) {
	        return this.compare(p.row, p.column);
	    };
	    this.containsRange = function(range) {
	        return this.comparePoint(range.start) == 0 && this.comparePoint(range.end) == 0;
	    };
	    this.intersects = function(range) {
	        var cmp = this.compareRange(range);
	        return (cmp == -1 || cmp == 0 || cmp == 1);
	    };
	    this.isEnd = function(row, column) {
	        return this.end.row == row && this.end.column == column;
	    };
	    this.isStart = function(row, column) {
	        return this.start.row == row && this.start.column == column;
	    };
	    this.setStart = function(row, column) {
	        if (typeof row == "object") {
	            this.start.column = row.column;
	            this.start.row = row.row;
	        } else {
	            this.start.row = row;
	            this.start.column = column;
	        }
	    };
	    this.setEnd = function(row, column) {
	        if (typeof row == "object") {
	            this.end.column = row.column;
	            this.end.row = row.row;
	        } else {
	            this.end.row = row;
	            this.end.column = column;
	        }
	    };
	    this.inside = function(row, column) {
	        if (this.compare(row, column) == 0) {
	            if (this.isEnd(row, column) || this.isStart(row, column)) {
	                return false;
	            } else {
	                return true;
	            }
	        }
	        return false;
	    };
	    this.insideStart = function(row, column) {
	        if (this.compare(row, column) == 0) {
	            if (this.isEnd(row, column)) {
	                return false;
	            } else {
	                return true;
	            }
	        }
	        return false;
	    };
	    this.insideEnd = function(row, column) {
	        if (this.compare(row, column) == 0) {
	            if (this.isStart(row, column)) {
	                return false;
	            } else {
	                return true;
	            }
	        }
	        return false;
	    };
	    this.compare = function(row, column) {
	        if (!this.isMultiLine()) {
	            if (row === this.start.row) {
	                return column < this.start.column ? -1 : (column > this.end.column ? 1 : 0);
	            }
	        }

	        if (row < this.start.row)
	            return -1;

	        if (row > this.end.row)
	            return 1;

	        if (this.start.row === row)
	            return column >= this.start.column ? 0 : -1;

	        if (this.end.row === row)
	            return column <= this.end.column ? 0 : 1;

	        return 0;
	    };
	    this.compareStart = function(row, column) {
	        if (this.start.row == row && this.start.column == column) {
	            return -1;
	        } else {
	            return this.compare(row, column);
	        }
	    };
	    this.compareEnd = function(row, column) {
	        if (this.end.row == row && this.end.column == column) {
	            return 1;
	        } else {
	            return this.compare(row, column);
	        }
	    };
	    this.compareInside = function(row, column) {
	        if (this.end.row == row && this.end.column == column) {
	            return 1;
	        } else if (this.start.row == row && this.start.column == column) {
	            return -1;
	        } else {
	            return this.compare(row, column);
	        }
	    };
	    this.clipRows = function(firstRow, lastRow) {
	        if (this.end.row > lastRow)
	            var end = {row: lastRow + 1, column: 0};
	        else if (this.end.row < firstRow)
	            var end = {row: firstRow, column: 0};

	        if (this.start.row > lastRow)
	            var start = {row: lastRow + 1, column: 0};
	        else if (this.start.row < firstRow)
	            var start = {row: firstRow, column: 0};

	        return Range.fromPoints(start || this.start, end || this.end);
	    };
	    this.extend = function(row, column) {
	        var cmp = this.compare(row, column);

	        if (cmp == 0)
	            return this;
	        else if (cmp == -1)
	            var start = {row: row, column: column};
	        else
	            var end = {row: row, column: column};

	        return Range.fromPoints(start || this.start, end || this.end);
	    };

	    this.isEmpty = function() {
	        return (this.start.row === this.end.row && this.start.column === this.end.column);
	    };
	    this.isMultiLine = function() {
	        return (this.start.row !== this.end.row);
	    };
	    this.clone = function() {
	        return Range.fromPoints(this.start, this.end);
	    };
	    this.collapseRows = function() {
	        if (this.end.column == 0)
	            return new Range(this.start.row, 0, Math.max(this.start.row, this.end.row-1), 0);
	        else
	            return new Range(this.start.row, 0, this.end.row, 0);
	    };
	    this.toScreenRange = function(session) {
	        var screenPosStart = session.documentToScreenPosition(this.start);
	        var screenPosEnd = session.documentToScreenPosition(this.end);

	        return new Range(
	            screenPosStart.row, screenPosStart.column,
	            screenPosEnd.row, screenPosEnd.column
	        );
	    };
	    this.moveBy = function(row, column) {
	        this.start.row += row;
	        this.start.column += column;
	        this.end.row += row;
	        this.end.column += column;
	    };

	}).call(Range.prototype);
	Range.fromPoints = function(start, end) {
	    return new Range(start.row, start.column, end.row, end.column);
	};
	Range.comparePoints = comparePoints;

	Range.comparePoints = function(p1, p2) {
	    return p1.row - p2.row || p1.column - p2.column;
	};


	exports.Range = Range;
	});

	ace.define("ace/selection",["require","exports","module","ace/lib/oop","ace/lib/lang","ace/lib/event_emitter","ace/range"], function(acequire, exports, module) {
	"use strict";

	var oop = acequire("./lib/oop");
	var lang = acequire("./lib/lang");
	var EventEmitter = acequire("./lib/event_emitter").EventEmitter;
	var Range = acequire("./range").Range;
	var Selection = function(session) {
	    this.session = session;
	    this.doc = session.getDocument();

	    this.clearSelection();
	    this.lead = this.selectionLead = this.doc.createAnchor(0, 0);
	    this.anchor = this.selectionAnchor = this.doc.createAnchor(0, 0);

	    var self = this;
	    this.lead.on("change", function(e) {
	        self._emit("changeCursor");
	        if (!self.$isEmpty)
	            self._emit("changeSelection");
	        if (!self.$keepDesiredColumnOnChange && e.old.column != e.value.column)
	            self.$desiredColumn = null;
	    });

	    this.selectionAnchor.on("change", function() {
	        if (!self.$isEmpty)
	            self._emit("changeSelection");
	    });
	};

	(function() {

	    oop.implement(this, EventEmitter);
	    this.isEmpty = function() {
	        return (this.$isEmpty || (
	            this.anchor.row == this.lead.row &&
	            this.anchor.column == this.lead.column
	        ));
	    };
	    this.isMultiLine = function() {
	        if (this.isEmpty()) {
	            return false;
	        }

	        return this.getRange().isMultiLine();
	    };
	    this.getCursor = function() {
	        return this.lead.getPosition();
	    };
	    this.setSelectionAnchor = function(row, column) {
	        this.anchor.setPosition(row, column);

	        if (this.$isEmpty) {
	            this.$isEmpty = false;
	            this._emit("changeSelection");
	        }
	    };
	    this.getSelectionAnchor = function() {
	        if (this.$isEmpty)
	            return this.getSelectionLead();
	        else
	            return this.anchor.getPosition();
	    };
	    this.getSelectionLead = function() {
	        return this.lead.getPosition();
	    };
	    this.shiftSelection = function(columns) {
	        if (this.$isEmpty) {
	            this.moveCursorTo(this.lead.row, this.lead.column + columns);
	            return;
	        }

	        var anchor = this.getSelectionAnchor();
	        var lead = this.getSelectionLead();

	        var isBackwards = this.isBackwards();

	        if (!isBackwards || anchor.column !== 0)
	            this.setSelectionAnchor(anchor.row, anchor.column + columns);

	        if (isBackwards || lead.column !== 0) {
	            this.$moveSelection(function() {
	                this.moveCursorTo(lead.row, lead.column + columns);
	            });
	        }
	    };
	    this.isBackwards = function() {
	        var anchor = this.anchor;
	        var lead = this.lead;
	        return (anchor.row > lead.row || (anchor.row == lead.row && anchor.column > lead.column));
	    };
	    this.getRange = function() {
	        var anchor = this.anchor;
	        var lead = this.lead;

	        if (this.isEmpty())
	            return Range.fromPoints(lead, lead);

	        if (this.isBackwards()) {
	            return Range.fromPoints(lead, anchor);
	        }
	        else {
	            return Range.fromPoints(anchor, lead);
	        }
	    };
	    this.clearSelection = function() {
	        if (!this.$isEmpty) {
	            this.$isEmpty = true;
	            this._emit("changeSelection");
	        }
	    };
	    this.selectAll = function() {
	        var lastRow = this.doc.getLength() - 1;
	        this.setSelectionAnchor(0, 0);
	        this.moveCursorTo(lastRow, this.doc.getLine(lastRow).length);
	    };
	    this.setRange =
	    this.setSelectionRange = function(range, reverse) {
	        if (reverse) {
	            this.setSelectionAnchor(range.end.row, range.end.column);
	            this.selectTo(range.start.row, range.start.column);
	        } else {
	            this.setSelectionAnchor(range.start.row, range.start.column);
	            this.selectTo(range.end.row, range.end.column);
	        }
	        if (this.getRange().isEmpty())
	            this.$isEmpty = true;
	        this.$desiredColumn = null;
	    };

	    this.$moveSelection = function(mover) {
	        var lead = this.lead;
	        if (this.$isEmpty)
	            this.setSelectionAnchor(lead.row, lead.column);

	        mover.call(this);
	    };
	    this.selectTo = function(row, column) {
	        this.$moveSelection(function() {
	            this.moveCursorTo(row, column);
	        });
	    };
	    this.selectToPosition = function(pos) {
	        this.$moveSelection(function() {
	            this.moveCursorToPosition(pos);
	        });
	    };
	    this.moveTo = function(row, column) {
	        this.clearSelection();
	        this.moveCursorTo(row, column);
	    };
	    this.moveToPosition = function(pos) {
	        this.clearSelection();
	        this.moveCursorToPosition(pos);
	    };
	    this.selectUp = function() {
	        this.$moveSelection(this.moveCursorUp);
	    };
	    this.selectDown = function() {
	        this.$moveSelection(this.moveCursorDown);
	    };
	    this.selectRight = function() {
	        this.$moveSelection(this.moveCursorRight);
	    };
	    this.selectLeft = function() {
	        this.$moveSelection(this.moveCursorLeft);
	    };
	    this.selectLineStart = function() {
	        this.$moveSelection(this.moveCursorLineStart);
	    };
	    this.selectLineEnd = function() {
	        this.$moveSelection(this.moveCursorLineEnd);
	    };
	    this.selectFileEnd = function() {
	        this.$moveSelection(this.moveCursorFileEnd);
	    };
	    this.selectFileStart = function() {
	        this.$moveSelection(this.moveCursorFileStart);
	    };
	    this.selectWordRight = function() {
	        this.$moveSelection(this.moveCursorWordRight);
	    };
	    this.selectWordLeft = function() {
	        this.$moveSelection(this.moveCursorWordLeft);
	    };
	    this.getWordRange = function(row, column) {
	        if (typeof column == "undefined") {
	            var cursor = row || this.lead;
	            row = cursor.row;
	            column = cursor.column;
	        }
	        return this.session.getWordRange(row, column);
	    };
	    this.selectWord = function() {
	        this.setSelectionRange(this.getWordRange());
	    };
	    this.selectAWord = function() {
	        var cursor = this.getCursor();
	        var range = this.session.getAWordRange(cursor.row, cursor.column);
	        this.setSelectionRange(range);
	    };

	    this.getLineRange = function(row, excludeLastChar) {
	        var rowStart = typeof row == "number" ? row : this.lead.row;
	        var rowEnd;

	        var foldLine = this.session.getFoldLine(rowStart);
	        if (foldLine) {
	            rowStart = foldLine.start.row;
	            rowEnd = foldLine.end.row;
	        } else {
	            rowEnd = rowStart;
	        }
	        if (excludeLastChar === true)
	            return new Range(rowStart, 0, rowEnd, this.session.getLine(rowEnd).length);
	        else
	            return new Range(rowStart, 0, rowEnd + 1, 0);
	    };
	    this.selectLine = function() {
	        this.setSelectionRange(this.getLineRange());
	    };
	    this.moveCursorUp = function() {
	        this.moveCursorBy(-1, 0);
	    };
	    this.moveCursorDown = function() {
	        this.moveCursorBy(1, 0);
	    };
	    this.wouldMoveIntoSoftTab = function(cursor, tabSize, direction) {
	        var start = cursor.column;
	        var end = cursor.column + tabSize;

	        if (direction < 0) {
	            start = cursor.column - tabSize;
	            end = cursor.column;
	        }
	        return this.session.isTabStop(cursor) && this.doc.getLine(cursor.row).slice(start, end).split(" ").length-1 == tabSize;
	    };
	    this.moveCursorLeft = function() {
	        var cursor = this.lead.getPosition(),
	            fold;

	        if (fold = this.session.getFoldAt(cursor.row, cursor.column, -1)) {
	            this.moveCursorTo(fold.start.row, fold.start.column);
	        } else if (cursor.column === 0) {
	            if (cursor.row > 0) {
	                this.moveCursorTo(cursor.row - 1, this.doc.getLine(cursor.row - 1).length);
	            }
	        }
	        else {
	            var tabSize = this.session.getTabSize();
	            if (this.wouldMoveIntoSoftTab(cursor, tabSize, -1) && !this.session.getNavigateWithinSoftTabs()) {
	                this.moveCursorBy(0, -tabSize);
	            } else {
	                this.moveCursorBy(0, -1);
	            }
	        }
	    };
	    this.moveCursorRight = function() {
	        var cursor = this.lead.getPosition(),
	            fold;
	        if (fold = this.session.getFoldAt(cursor.row, cursor.column, 1)) {
	            this.moveCursorTo(fold.end.row, fold.end.column);
	        }
	        else if (this.lead.column == this.doc.getLine(this.lead.row).length) {
	            if (this.lead.row < this.doc.getLength() - 1) {
	                this.moveCursorTo(this.lead.row + 1, 0);
	            }
	        }
	        else {
	            var tabSize = this.session.getTabSize();
	            var cursor = this.lead;
	            if (this.wouldMoveIntoSoftTab(cursor, tabSize, 1) && !this.session.getNavigateWithinSoftTabs()) {
	                this.moveCursorBy(0, tabSize);
	            } else {
	                this.moveCursorBy(0, 1);
	            }
	        }
	    };
	    this.moveCursorLineStart = function() {
	        var row = this.lead.row;
	        var column = this.lead.column;
	        var screenRow = this.session.documentToScreenRow(row, column);
	        var firstColumnPosition = this.session.screenToDocumentPosition(screenRow, 0);
	        var beforeCursor = this.session.getDisplayLine(
	            row, null, firstColumnPosition.row,
	            firstColumnPosition.column
	        );

	        var leadingSpace = beforeCursor.match(/^\s*/);
	        if (leadingSpace[0].length != column && !this.session.$useEmacsStyleLineStart)
	            firstColumnPosition.column += leadingSpace[0].length;
	        this.moveCursorToPosition(firstColumnPosition);
	    };
	    this.moveCursorLineEnd = function() {
	        var lead = this.lead;
	        var lineEnd = this.session.getDocumentLastRowColumnPosition(lead.row, lead.column);
	        if (this.lead.column == lineEnd.column) {
	            var line = this.session.getLine(lineEnd.row);
	            if (lineEnd.column == line.length) {
	                var textEnd = line.search(/\s+$/);
	                if (textEnd > 0)
	                    lineEnd.column = textEnd;
	            }
	        }

	        this.moveCursorTo(lineEnd.row, lineEnd.column);
	    };
	    this.moveCursorFileEnd = function() {
	        var row = this.doc.getLength() - 1;
	        var column = this.doc.getLine(row).length;
	        this.moveCursorTo(row, column);
	    };
	    this.moveCursorFileStart = function() {
	        this.moveCursorTo(0, 0);
	    };
	    this.moveCursorLongWordRight = function() {
	        var row = this.lead.row;
	        var column = this.lead.column;
	        var line = this.doc.getLine(row);
	        var rightOfCursor = line.substring(column);

	        var match;
	        this.session.nonTokenRe.lastIndex = 0;
	        this.session.tokenRe.lastIndex = 0;
	        var fold = this.session.getFoldAt(row, column, 1);
	        if (fold) {
	            this.moveCursorTo(fold.end.row, fold.end.column);
	            return;
	        }
	        if (match = this.session.nonTokenRe.exec(rightOfCursor)) {
	            column += this.session.nonTokenRe.lastIndex;
	            this.session.nonTokenRe.lastIndex = 0;
	            rightOfCursor = line.substring(column);
	        }
	        if (column >= line.length) {
	            this.moveCursorTo(row, line.length);
	            this.moveCursorRight();
	            if (row < this.doc.getLength() - 1)
	                this.moveCursorWordRight();
	            return;
	        }
	        if (match = this.session.tokenRe.exec(rightOfCursor)) {
	            column += this.session.tokenRe.lastIndex;
	            this.session.tokenRe.lastIndex = 0;
	        }

	        this.moveCursorTo(row, column);
	    };
	    this.moveCursorLongWordLeft = function() {
	        var row = this.lead.row;
	        var column = this.lead.column;
	        var fold;
	        if (fold = this.session.getFoldAt(row, column, -1)) {
	            this.moveCursorTo(fold.start.row, fold.start.column);
	            return;
	        }

	        var str = this.session.getFoldStringAt(row, column, -1);
	        if (str == null) {
	            str = this.doc.getLine(row).substring(0, column);
	        }

	        var leftOfCursor = lang.stringReverse(str);
	        var match;
	        this.session.nonTokenRe.lastIndex = 0;
	        this.session.tokenRe.lastIndex = 0;
	        if (match = this.session.nonTokenRe.exec(leftOfCursor)) {
	            column -= this.session.nonTokenRe.lastIndex;
	            leftOfCursor = leftOfCursor.slice(this.session.nonTokenRe.lastIndex);
	            this.session.nonTokenRe.lastIndex = 0;
	        }
	        if (column <= 0) {
	            this.moveCursorTo(row, 0);
	            this.moveCursorLeft();
	            if (row > 0)
	                this.moveCursorWordLeft();
	            return;
	        }
	        if (match = this.session.tokenRe.exec(leftOfCursor)) {
	            column -= this.session.tokenRe.lastIndex;
	            this.session.tokenRe.lastIndex = 0;
	        }

	        this.moveCursorTo(row, column);
	    };

	    this.$shortWordEndIndex = function(rightOfCursor) {
	        var match, index = 0, ch;
	        var whitespaceRe = /\s/;
	        var tokenRe = this.session.tokenRe;

	        tokenRe.lastIndex = 0;
	        if (match = this.session.tokenRe.exec(rightOfCursor)) {
	            index = this.session.tokenRe.lastIndex;
	        } else {
	            while ((ch = rightOfCursor[index]) && whitespaceRe.test(ch))
	                index ++;

	            if (index < 1) {
	                tokenRe.lastIndex = 0;
	                 while ((ch = rightOfCursor[index]) && !tokenRe.test(ch)) {
	                    tokenRe.lastIndex = 0;
	                    index ++;
	                    if (whitespaceRe.test(ch)) {
	                        if (index > 2) {
	                            index--;
	                            break;
	                        } else {
	                            while ((ch = rightOfCursor[index]) && whitespaceRe.test(ch))
	                                index ++;
	                            if (index > 2)
	                                break;
	                        }
	                    }
	                }
	            }
	        }
	        tokenRe.lastIndex = 0;

	        return index;
	    };

	    this.moveCursorShortWordRight = function() {
	        var row = this.lead.row;
	        var column = this.lead.column;
	        var line = this.doc.getLine(row);
	        var rightOfCursor = line.substring(column);

	        var fold = this.session.getFoldAt(row, column, 1);
	        if (fold)
	            return this.moveCursorTo(fold.end.row, fold.end.column);

	        if (column == line.length) {
	            var l = this.doc.getLength();
	            do {
	                row++;
	                rightOfCursor = this.doc.getLine(row);
	            } while (row < l && /^\s*$/.test(rightOfCursor));

	            if (!/^\s+/.test(rightOfCursor))
	                rightOfCursor = "";
	            column = 0;
	        }

	        var index = this.$shortWordEndIndex(rightOfCursor);

	        this.moveCursorTo(row, column + index);
	    };

	    this.moveCursorShortWordLeft = function() {
	        var row = this.lead.row;
	        var column = this.lead.column;

	        var fold;
	        if (fold = this.session.getFoldAt(row, column, -1))
	            return this.moveCursorTo(fold.start.row, fold.start.column);

	        var line = this.session.getLine(row).substring(0, column);
	        if (column === 0) {
	            do {
	                row--;
	                line = this.doc.getLine(row);
	            } while (row > 0 && /^\s*$/.test(line));

	            column = line.length;
	            if (!/\s+$/.test(line))
	                line = "";
	        }

	        var leftOfCursor = lang.stringReverse(line);
	        var index = this.$shortWordEndIndex(leftOfCursor);

	        return this.moveCursorTo(row, column - index);
	    };

	    this.moveCursorWordRight = function() {
	        if (this.session.$selectLongWords)
	            this.moveCursorLongWordRight();
	        else
	            this.moveCursorShortWordRight();
	    };

	    this.moveCursorWordLeft = function() {
	        if (this.session.$selectLongWords)
	            this.moveCursorLongWordLeft();
	        else
	            this.moveCursorShortWordLeft();
	    };
	    this.moveCursorBy = function(rows, chars) {
	        var screenPos = this.session.documentToScreenPosition(
	            this.lead.row,
	            this.lead.column
	        );

	        var offsetX;

	        if (chars === 0) {
	            if (rows !== 0) {
	                if (this.session.$bidiHandler.isBidiRow(screenPos.row, this.lead.row)) {
	                    offsetX = this.session.$bidiHandler.getPosLeft(screenPos.column);
	                    screenPos.column = Math.round(offsetX / this.session.$bidiHandler.charWidths[0]);
	                } else {
	                    offsetX = screenPos.column * this.session.$bidiHandler.charWidths[0];
	                }
	            }

	            if (this.$desiredColumn)
	                screenPos.column = this.$desiredColumn;
	            else
	                this.$desiredColumn = screenPos.column;
	        }

	        var docPos = this.session.screenToDocumentPosition(screenPos.row + rows, screenPos.column, offsetX);
	        
	        if (rows !== 0 && chars === 0 && docPos.row === this.lead.row && docPos.column === this.lead.column) {
	            if (this.session.lineWidgets && this.session.lineWidgets[docPos.row]) {
	                if (docPos.row > 0 || rows > 0)
	                    docPos.row++;
	            }
	        }
	        this.moveCursorTo(docPos.row, docPos.column + chars, chars === 0);
	    };
	    this.moveCursorToPosition = function(position) {
	        this.moveCursorTo(position.row, position.column);
	    };
	    this.moveCursorTo = function(row, column, keepDesiredColumn) {
	        var fold = this.session.getFoldAt(row, column, 1);
	        if (fold) {
	            row = fold.start.row;
	            column = fold.start.column;
	        }

	        this.$keepDesiredColumnOnChange = true;
	        var line = this.session.getLine(row);
	        if (/[\uDC00-\uDFFF]/.test(line.charAt(column)) && line.charAt(column - 1)) {
	            if (this.lead.row == row && this.lead.column == column + 1)
	                column = column - 1;
	            else
	                column = column + 1;
	        }
	        this.lead.setPosition(row, column);
	        this.$keepDesiredColumnOnChange = false;

	        if (!keepDesiredColumn)
	            this.$desiredColumn = null;
	    };
	    this.moveCursorToScreen = function(row, column, keepDesiredColumn) {
	        var pos = this.session.screenToDocumentPosition(row, column);
	        this.moveCursorTo(pos.row, pos.column, keepDesiredColumn);
	    };
	    this.detach = function() {
	        this.lead.detach();
	        this.anchor.detach();
	        this.session = this.doc = null;
	    };

	    this.fromOrientedRange = function(range) {
	        this.setSelectionRange(range, range.cursor == range.start);
	        this.$desiredColumn = range.desiredColumn || this.$desiredColumn;
	    };

	    this.toOrientedRange = function(range) {
	        var r = this.getRange();
	        if (range) {
	            range.start.column = r.start.column;
	            range.start.row = r.start.row;
	            range.end.column = r.end.column;
	            range.end.row = r.end.row;
	        } else {
	            range = r;
	        }

	        range.cursor = this.isBackwards() ? range.start : range.end;
	        range.desiredColumn = this.$desiredColumn;
	        return range;
	    };
	    this.getRangeOfMovements = function(func) {
	        var start = this.getCursor();
	        try {
	            func(this);
	            var end = this.getCursor();
	            return Range.fromPoints(start,end);
	        } catch(e) {
	            return Range.fromPoints(start,start);
	        } finally {
	            this.moveCursorToPosition(start);
	        }
	    };

	    this.toJSON = function() {
	        if (this.rangeCount) {
	            var data = this.ranges.map(function(r) {
	                var r1 = r.clone();
	                r1.isBackwards = r.cursor == r.start;
	                return r1;
	            });
	        } else {
	            var data = this.getRange();
	            data.isBackwards = this.isBackwards();
	        }
	        return data;
	    };

	    this.fromJSON = function(data) {
	        if (data.start == undefined) {
	            if (this.rangeList) {
	                this.toSingleRange(data[0]);
	                for (var i = data.length; i--; ) {
	                    var r = Range.fromPoints(data[i].start, data[i].end);
	                    if (data[i].isBackwards)
	                        r.cursor = r.start;
	                    this.addRange(r, true);
	                }
	                return;
	            } else
	                data = data[0];
	        }
	        if (this.rangeList)
	            this.toSingleRange(data);
	        this.setSelectionRange(data, data.isBackwards);
	    };

	    this.isEqual = function(data) {
	        if ((data.length || this.rangeCount) && data.length != this.rangeCount)
	            return false;
	        if (!data.length || !this.ranges)
	            return this.getRange().isEqual(data);

	        for (var i = this.ranges.length; i--; ) {
	            if (!this.ranges[i].isEqual(data[i]))
	                return false;
	        }
	        return true;
	    };

	}).call(Selection.prototype);

	exports.Selection = Selection;
	});

	ace.define("ace/tokenizer",["require","exports","module","ace/config"], function(acequire, exports, module) {
	"use strict";

	var config = acequire("./config");
	var MAX_TOKEN_COUNT = 2000;
	var Tokenizer = function(rules) {
	    this.states = rules;

	    this.regExps = {};
	    this.matchMappings = {};
	    for (var key in this.states) {
	        var state = this.states[key];
	        var ruleRegExps = [];
	        var matchTotal = 0;
	        var mapping = this.matchMappings[key] = {defaultToken: "text"};
	        var flag = "g";

	        var splitterRurles = [];
	        for (var i = 0; i < state.length; i++) {
	            var rule = state[i];
	            if (rule.defaultToken)
	                mapping.defaultToken = rule.defaultToken;
	            if (rule.caseInsensitive)
	                flag = "gi";
	            if (rule.regex == null)
	                continue;

	            if (rule.regex instanceof RegExp)
	                rule.regex = rule.regex.toString().slice(1, -1);
	            var adjustedregex = rule.regex;
	            var matchcount = new RegExp("(?:(" + adjustedregex + ")|(.))").exec("a").length - 2;
	            if (Array.isArray(rule.token)) {
	                if (rule.token.length == 1 || matchcount == 1) {
	                    rule.token = rule.token[0];
	                } else if (matchcount - 1 != rule.token.length) {
	                    this.reportError("number of classes and regexp groups doesn't match", { 
	                        rule: rule,
	                        groupCount: matchcount - 1
	                    });
	                    rule.token = rule.token[0];
	                } else {
	                    rule.tokenArray = rule.token;
	                    rule.token = null;
	                    rule.onMatch = this.$arrayTokens;
	                }
	            } else if (typeof rule.token == "function" && !rule.onMatch) {
	                if (matchcount > 1)
	                    rule.onMatch = this.$applyToken;
	                else
	                    rule.onMatch = rule.token;
	            }

	            if (matchcount > 1) {
	                if (/\\\d/.test(rule.regex)) {
	                    adjustedregex = rule.regex.replace(/\\([0-9]+)/g, function(match, digit) {
	                        return "\\" + (parseInt(digit, 10) + matchTotal + 1);
	                    });
	                } else {
	                    matchcount = 1;
	                    adjustedregex = this.removeCapturingGroups(rule.regex);
	                }
	                if (!rule.splitRegex && typeof rule.token != "string")
	                    splitterRurles.push(rule); // flag will be known only at the very end
	            }

	            mapping[matchTotal] = i;
	            matchTotal += matchcount;

	            ruleRegExps.push(adjustedregex);
	            if (!rule.onMatch)
	                rule.onMatch = null;
	        }
	        
	        if (!ruleRegExps.length) {
	            mapping[0] = 0;
	            ruleRegExps.push("$");
	        }
	        
	        splitterRurles.forEach(function(rule) {
	            rule.splitRegex = this.createSplitterRegexp(rule.regex, flag);
	        }, this);

	        this.regExps[key] = new RegExp("(" + ruleRegExps.join(")|(") + ")|($)", flag);
	    }
	};

	(function() {
	    this.$setMaxTokenCount = function(m) {
	        MAX_TOKEN_COUNT = m | 0;
	    };
	    
	    this.$applyToken = function(str) {
	        var values = this.splitRegex.exec(str).slice(1);
	        var types = this.token.apply(this, values);
	        if (typeof types === "string")
	            return [{type: types, value: str}];

	        var tokens = [];
	        for (var i = 0, l = types.length; i < l; i++) {
	            if (values[i])
	                tokens[tokens.length] = {
	                    type: types[i],
	                    value: values[i]
	                };
	        }
	        return tokens;
	    };

	    this.$arrayTokens = function(str) {
	        if (!str)
	            return [];
	        var values = this.splitRegex.exec(str);
	        if (!values)
	            return "text";
	        var tokens = [];
	        var types = this.tokenArray;
	        for (var i = 0, l = types.length; i < l; i++) {
	            if (values[i + 1])
	                tokens[tokens.length] = {
	                    type: types[i],
	                    value: values[i + 1]
	                };
	        }
	        return tokens;
	    };

	    this.removeCapturingGroups = function(src) {
	        var r = src.replace(
	            /\[(?:\\.|[^\]])*?\]|\\.|\(\?[:=!]|(\()/g,
	            function(x, y) {return y ? "(?:" : x;}
	        );
	        return r;
	    };

	    this.createSplitterRegexp = function(src, flag) {
	        if (src.indexOf("(?=") != -1) {
	            var stack = 0;
	            var inChClass = false;
	            var lastCapture = {};
	            src.replace(/(\\.)|(\((?:\?[=!])?)|(\))|([\[\]])/g, function(
	                m, esc, parenOpen, parenClose, square, index
	            ) {
	                if (inChClass) {
	                    inChClass = square != "]";
	                } else if (square) {
	                    inChClass = true;
	                } else if (parenClose) {
	                    if (stack == lastCapture.stack) {
	                        lastCapture.end = index+1;
	                        lastCapture.stack = -1;
	                    }
	                    stack--;
	                } else if (parenOpen) {
	                    stack++;
	                    if (parenOpen.length != 1) {
	                        lastCapture.stack = stack;
	                        lastCapture.start = index;
	                    }
	                }
	                return m;
	            });

	            if (lastCapture.end != null && /^\)*$/.test(src.substr(lastCapture.end)))
	                src = src.substring(0, lastCapture.start) + src.substr(lastCapture.end);
	        }
	        if (src.charAt(0) != "^") src = "^" + src;
	        if (src.charAt(src.length - 1) != "$") src += "$";
	        
	        return new RegExp(src, (flag||"").replace("g", ""));
	    };
	    this.getLineTokens = function(line, startState) {
	        if (startState && typeof startState != "string") {
	            var stack = startState.slice(0);
	            startState = stack[0];
	            if (startState === "#tmp") {
	                stack.shift();
	                startState = stack.shift();
	            }
	        } else
	            var stack = [];

	        var currentState = startState || "start";
	        var state = this.states[currentState];
	        if (!state) {
	            currentState = "start";
	            state = this.states[currentState];
	        }
	        var mapping = this.matchMappings[currentState];
	        var re = this.regExps[currentState];
	        re.lastIndex = 0;

	        var match, tokens = [];
	        var lastIndex = 0;
	        var matchAttempts = 0;

	        var token = {type: null, value: ""};

	        while (match = re.exec(line)) {
	            var type = mapping.defaultToken;
	            var rule = null;
	            var value = match[0];
	            var index = re.lastIndex;

	            if (index - value.length > lastIndex) {
	                var skipped = line.substring(lastIndex, index - value.length);
	                if (token.type == type) {
	                    token.value += skipped;
	                } else {
	                    if (token.type)
	                        tokens.push(token);
	                    token = {type: type, value: skipped};
	                }
	            }

	            for (var i = 0; i < match.length-2; i++) {
	                if (match[i + 1] === undefined)
	                    continue;

	                rule = state[mapping[i]];

	                if (rule.onMatch)
	                    type = rule.onMatch(value, currentState, stack, line);
	                else
	                    type = rule.token;

	                if (rule.next) {
	                    if (typeof rule.next == "string") {
	                        currentState = rule.next;
	                    } else {
	                        currentState = rule.next(currentState, stack);
	                    }
	                    
	                    state = this.states[currentState];
	                    if (!state) {
	                        this.reportError("state doesn't exist", currentState);
	                        currentState = "start";
	                        state = this.states[currentState];
	                    }
	                    mapping = this.matchMappings[currentState];
	                    lastIndex = index;
	                    re = this.regExps[currentState];
	                    re.lastIndex = index;
	                }
	                if (rule.consumeLineEnd)
	                    lastIndex = index;
	                break;
	            }

	            if (value) {
	                if (typeof type === "string") {
	                    if ((!rule || rule.merge !== false) && token.type === type) {
	                        token.value += value;
	                    } else {
	                        if (token.type)
	                            tokens.push(token);
	                        token = {type: type, value: value};
	                    }
	                } else if (type) {
	                    if (token.type)
	                        tokens.push(token);
	                    token = {type: null, value: ""};
	                    for (var i = 0; i < type.length; i++)
	                        tokens.push(type[i]);
	                }
	            }

	            if (lastIndex == line.length)
	                break;

	            lastIndex = index;

	            if (matchAttempts++ > MAX_TOKEN_COUNT) {
	                if (matchAttempts > 2 * line.length) {
	                    this.reportError("infinite loop with in ace tokenizer", {
	                        startState: startState,
	                        line: line
	                    });
	                }
	                while (lastIndex < line.length) {
	                    if (token.type)
	                        tokens.push(token);
	                    token = {
	                        value: line.substring(lastIndex, lastIndex += 2000),
	                        type: "overflow"
	                    };
	                }
	                currentState = "start";
	                stack = [];
	                break;
	            }
	        }

	        if (token.type)
	            tokens.push(token);
	        
	        if (stack.length > 1) {
	            if (stack[0] !== currentState)
	                stack.unshift("#tmp", currentState);
	        }
	        return {
	            tokens : tokens,
	            state : stack.length ? stack : currentState
	        };
	    };
	    
	    this.reportError = config.reportError;
	    
	}).call(Tokenizer.prototype);

	exports.Tokenizer = Tokenizer;
	});

	ace.define("ace/mode/text_highlight_rules",["require","exports","module","ace/lib/lang"], function(acequire, exports, module) {
	"use strict";

	var lang = acequire("../lib/lang");

	var TextHighlightRules = function() {

	    this.$rules = {
	        "start" : [{
	            token : "empty_line",
	            regex : '^$'
	        }, {
	            defaultToken : "text"
	        }]
	    };
	};

	(function() {

	    this.addRules = function(rules, prefix) {
	        if (!prefix) {
	            for (var key in rules)
	                this.$rules[key] = rules[key];
	            return;
	        }
	        for (var key in rules) {
	            var state = rules[key];
	            for (var i = 0; i < state.length; i++) {
	                var rule = state[i];
	                if (rule.next || rule.onMatch) {
	                    if (typeof rule.next == "string") {
	                        if (rule.next.indexOf(prefix) !== 0)
	                            rule.next = prefix + rule.next;
	                    }
	                    if (rule.nextState && rule.nextState.indexOf(prefix) !== 0)
	                        rule.nextState = prefix + rule.nextState;
	                }
	            }
	            this.$rules[prefix + key] = state;
	        }
	    };

	    this.getRules = function() {
	        return this.$rules;
	    };

	    this.embedRules = function (HighlightRules, prefix, escapeRules, states, append) {
	        var embedRules = typeof HighlightRules == "function"
	            ? new HighlightRules().getRules()
	            : HighlightRules;
	        if (states) {
	            for (var i = 0; i < states.length; i++)
	                states[i] = prefix + states[i];
	        } else {
	            states = [];
	            for (var key in embedRules)
	                states.push(prefix + key);
	        }

	        this.addRules(embedRules, prefix);

	        if (escapeRules) {
	            var addRules = Array.prototype[append ? "push" : "unshift"];
	            for (var i = 0; i < states.length; i++)
	                addRules.apply(this.$rules[states[i]], lang.deepCopy(escapeRules));
	        }

	        if (!this.$embeds)
	            this.$embeds = [];
	        this.$embeds.push(prefix);
	    };

	    this.getEmbeds = function() {
	        return this.$embeds;
	    };

	    var pushState = function(currentState, stack) {
	        if (currentState != "start" || stack.length)
	            stack.unshift(this.nextState, currentState);
	        return this.nextState;
	    };
	    var popState = function(currentState, stack) {
	        stack.shift();
	        return stack.shift() || "start";
	    };

	    this.normalizeRules = function() {
	        var id = 0;
	        var rules = this.$rules;
	        function processState(key) {
	            var state = rules[key];
	            state.processed = true;
	            for (var i = 0; i < state.length; i++) {
	                var rule = state[i];
	                var toInsert = null;
	                if (Array.isArray(rule)) {
	                    toInsert = rule;
	                    rule = {};
	                }
	                if (!rule.regex && rule.start) {
	                    rule.regex = rule.start;
	                    if (!rule.next)
	                        rule.next = [];
	                    rule.next.push({
	                        defaultToken: rule.token
	                    }, {
	                        token: rule.token + ".end",
	                        regex: rule.end || rule.start,
	                        next: "pop"
	                    });
	                    rule.token = rule.token + ".start";
	                    rule.push = true;
	                }
	                var next = rule.next || rule.push;
	                if (next && Array.isArray(next)) {
	                    var stateName = rule.stateName;
	                    if (!stateName)  {
	                        stateName = rule.token;
	                        if (typeof stateName != "string")
	                            stateName = stateName[0] || "";
	                        if (rules[stateName])
	                            stateName += id++;
	                    }
	                    rules[stateName] = next;
	                    rule.next = stateName;
	                    processState(stateName);
	                } else if (next == "pop") {
	                    rule.next = popState;
	                }

	                if (rule.push) {
	                    rule.nextState = rule.next || rule.push;
	                    rule.next = pushState;
	                    delete rule.push;
	                }

	                if (rule.rules) {
	                    for (var r in rule.rules) {
	                        if (rules[r]) {
	                            if (rules[r].push)
	                                rules[r].push.apply(rules[r], rule.rules[r]);
	                        } else {
	                            rules[r] = rule.rules[r];
	                        }
	                    }
	                }
	                var includeName = typeof rule == "string" ? rule : rule.include;
	                if (includeName) {
	                    if (Array.isArray(includeName))
	                        toInsert = includeName.map(function(x) { return rules[x]; });
	                    else
	                        toInsert = rules[includeName];
	                }

	                if (toInsert) {
	                    var args = [i, 1].concat(toInsert);
	                    if (rule.noEscape)
	                        args = args.filter(function(x) {return !x.next;});
	                    state.splice.apply(state, args);
	                    i--;
	                }
	                
	                if (rule.keywordMap) {
	                    rule.token = this.createKeywordMapper(
	                        rule.keywordMap, rule.defaultToken || "text", rule.caseInsensitive
	                    );
	                    delete rule.defaultToken;
	                }
	            }
	        }
	        Object.keys(rules).forEach(processState, this);
	    };

	    this.createKeywordMapper = function(map, defaultToken, ignoreCase, splitChar) {
	        var keywords = Object.create(null);
	        Object.keys(map).forEach(function(className) {
	            var a = map[className];
	            if (ignoreCase)
	                a = a.toLowerCase();
	            var list = a.split(splitChar || "|");
	            for (var i = list.length; i--; )
	                keywords[list[i]] = className;
	        });
	        if (Object.getPrototypeOf(keywords)) {
	            keywords.__proto__ = null;
	        }
	        this.$keywordList = Object.keys(keywords);
	        map = null;
	        return ignoreCase
	            ? function(value) {return keywords[value.toLowerCase()] || defaultToken; }
	            : function(value) {return keywords[value] || defaultToken; };
	    };

	    this.getKeywords = function() {
	        return this.$keywords;
	    };

	}).call(TextHighlightRules.prototype);

	exports.TextHighlightRules = TextHighlightRules;
	});

	ace.define("ace/mode/behaviour",["require","exports","module"], function(acequire, exports, module) {
	"use strict";

	var Behaviour = function() {
	   this.$behaviours = {};
	};

	(function () {

	    this.add = function (name, action, callback) {
	        switch (undefined) {
	          case this.$behaviours:
	              this.$behaviours = {};
	          case this.$behaviours[name]:
	              this.$behaviours[name] = {};
	        }
	        this.$behaviours[name][action] = callback;
	    };
	    
	    this.addBehaviours = function (behaviours) {
	        for (var key in behaviours) {
	            for (var action in behaviours[key]) {
	                this.add(key, action, behaviours[key][action]);
	            }
	        }
	    };
	    
	    this.remove = function (name) {
	        if (this.$behaviours && this.$behaviours[name]) {
	            delete this.$behaviours[name];
	        }
	    };
	    
	    this.inherit = function (mode, filter) {
	        if (typeof mode === "function") {
	            var behaviours = new mode().getBehaviours(filter);
	        } else {
	            var behaviours = mode.getBehaviours(filter);
	        }
	        this.addBehaviours(behaviours);
	    };
	    
	    this.getBehaviours = function (filter) {
	        if (!filter) {
	            return this.$behaviours;
	        } else {
	            var ret = {};
	            for (var i = 0; i < filter.length; i++) {
	                if (this.$behaviours[filter[i]]) {
	                    ret[filter[i]] = this.$behaviours[filter[i]];
	                }
	            }
	            return ret;
	        }
	    };

	}).call(Behaviour.prototype);

	exports.Behaviour = Behaviour;
	});

	ace.define("ace/token_iterator",["require","exports","module","ace/range"], function(acequire, exports, module) {
	"use strict";

	var Range = acequire("./range").Range;
	var TokenIterator = function(session, initialRow, initialColumn) {
	    this.$session = session;
	    this.$row = initialRow;
	    this.$rowTokens = session.getTokens(initialRow);

	    var token = session.getTokenAt(initialRow, initialColumn);
	    this.$tokenIndex = token ? token.index : -1;
	};

	(function() { 
	    this.stepBackward = function() {
	        this.$tokenIndex -= 1;
	        
	        while (this.$tokenIndex < 0) {
	            this.$row -= 1;
	            if (this.$row < 0) {
	                this.$row = 0;
	                return null;
	            }
	                
	            this.$rowTokens = this.$session.getTokens(this.$row);
	            this.$tokenIndex = this.$rowTokens.length - 1;
	        }
	            
	        return this.$rowTokens[this.$tokenIndex];
	    };   
	    this.stepForward = function() {
	        this.$tokenIndex += 1;
	        var rowCount;
	        while (this.$tokenIndex >= this.$rowTokens.length) {
	            this.$row += 1;
	            if (!rowCount)
	                rowCount = this.$session.getLength();
	            if (this.$row >= rowCount) {
	                this.$row = rowCount - 1;
	                return null;
	            }

	            this.$rowTokens = this.$session.getTokens(this.$row);
	            this.$tokenIndex = 0;
	        }
	            
	        return this.$rowTokens[this.$tokenIndex];
	    };      
	    this.getCurrentToken = function () {
	        return this.$rowTokens[this.$tokenIndex];
	    };      
	    this.getCurrentTokenRow = function () {
	        return this.$row;
	    };     
	    this.getCurrentTokenColumn = function() {
	        var rowTokens = this.$rowTokens;
	        var tokenIndex = this.$tokenIndex;
	        var column = rowTokens[tokenIndex].start;
	        if (column !== undefined)
	            return column;
	            
	        column = 0;
	        while (tokenIndex > 0) {
	            tokenIndex -= 1;
	            column += rowTokens[tokenIndex].value.length;
	        }
	        
	        return column;  
	    };
	    this.getCurrentTokenPosition = function() {
	        return {row: this.$row, column: this.getCurrentTokenColumn()};
	    };
	    this.getCurrentTokenRange = function() {
	        var token = this.$rowTokens[this.$tokenIndex];
	        var column = this.getCurrentTokenColumn();
	        return new Range(this.$row, column, this.$row, column + token.value.length);
	    };

	}).call(TokenIterator.prototype);

	exports.TokenIterator = TokenIterator;
	});

	ace.define("ace/mode/behaviour/cstyle",["require","exports","module","ace/lib/oop","ace/mode/behaviour","ace/token_iterator","ace/lib/lang"], function(acequire, exports, module) {
	"use strict";

	var oop = acequire("../../lib/oop");
	var Behaviour = acequire("../behaviour").Behaviour;
	var TokenIterator = acequire("../../token_iterator").TokenIterator;
	var lang = acequire("../../lib/lang");

	var SAFE_INSERT_IN_TOKENS =
	    ["text", "paren.rparen", "punctuation.operator"];
	var SAFE_INSERT_BEFORE_TOKENS =
	    ["text", "paren.rparen", "punctuation.operator", "comment"];

	var context;
	var contextCache = {};
	var defaultQuotes = {'"' : '"', "'" : "'"};

	var initContext = function(editor) {
	    var id = -1;
	    if (editor.multiSelect) {
	        id = editor.selection.index;
	        if (contextCache.rangeCount != editor.multiSelect.rangeCount)
	            contextCache = {rangeCount: editor.multiSelect.rangeCount};
	    }
	    if (contextCache[id])
	        return context = contextCache[id];
	    context = contextCache[id] = {
	        autoInsertedBrackets: 0,
	        autoInsertedRow: -1,
	        autoInsertedLineEnd: "",
	        maybeInsertedBrackets: 0,
	        maybeInsertedRow: -1,
	        maybeInsertedLineStart: "",
	        maybeInsertedLineEnd: ""
	    };
	};

	var getWrapped = function(selection, selected, opening, closing) {
	    var rowDiff = selection.end.row - selection.start.row;
	    return {
	        text: opening + selected + closing,
	        selection: [
	                0,
	                selection.start.column + 1,
	                rowDiff,
	                selection.end.column + (rowDiff ? 0 : 1)
	            ]
	    };
	};

	var CstyleBehaviour = function(options) {
	    this.add("braces", "insertion", function(state, action, editor, session, text) {
	        var cursor = editor.getCursorPosition();
	        var line = session.doc.getLine(cursor.row);
	        if (text == '{') {
	            initContext(editor);
	            var selection = editor.getSelectionRange();
	            var selected = session.doc.getTextRange(selection);
	            if (selected !== "" && selected !== "{" && editor.getWrapBehavioursEnabled()) {
	                return getWrapped(selection, selected, '{', '}');
	            } else if (CstyleBehaviour.isSaneInsertion(editor, session)) {
	                if (/[\]\}\)]/.test(line[cursor.column]) || editor.inMultiSelectMode || options && options.braces) {
	                    CstyleBehaviour.recordAutoInsert(editor, session, "}");
	                    return {
	                        text: '{}',
	                        selection: [1, 1]
	                    };
	                } else {
	                    CstyleBehaviour.recordMaybeInsert(editor, session, "{");
	                    return {
	                        text: '{',
	                        selection: [1, 1]
	                    };
	                }
	            }
	        } else if (text == '}') {
	            initContext(editor);
	            var rightChar = line.substring(cursor.column, cursor.column + 1);
	            if (rightChar == '}') {
	                var matching = session.$findOpeningBracket('}', {column: cursor.column + 1, row: cursor.row});
	                if (matching !== null && CstyleBehaviour.isAutoInsertedClosing(cursor, line, text)) {
	                    CstyleBehaviour.popAutoInsertedClosing();
	                    return {
	                        text: '',
	                        selection: [1, 1]
	                    };
	                }
	            }
	        } else if (text == "\n" || text == "\r\n") {
	            initContext(editor);
	            var closing = "";
	            if (CstyleBehaviour.isMaybeInsertedClosing(cursor, line)) {
	                closing = lang.stringRepeat("}", context.maybeInsertedBrackets);
	                CstyleBehaviour.clearMaybeInsertedClosing();
	            }
	            var rightChar = line.substring(cursor.column, cursor.column + 1);
	            if (rightChar === '}') {
	                var openBracePos = session.findMatchingBracket({row: cursor.row, column: cursor.column+1}, '}');
	                if (!openBracePos)
	                     return null;
	                var next_indent = this.$getIndent(session.getLine(openBracePos.row));
	            } else if (closing) {
	                var next_indent = this.$getIndent(line);
	            } else {
	                CstyleBehaviour.clearMaybeInsertedClosing();
	                return;
	            }
	            var indent = next_indent + session.getTabString();

	            return {
	                text: '\n' + indent + '\n' + next_indent + closing,
	                selection: [1, indent.length, 1, indent.length]
	            };
	        } else {
	            CstyleBehaviour.clearMaybeInsertedClosing();
	        }
	    });

	    this.add("braces", "deletion", function(state, action, editor, session, range) {
	        var selected = session.doc.getTextRange(range);
	        if (!range.isMultiLine() && selected == '{') {
	            initContext(editor);
	            var line = session.doc.getLine(range.start.row);
	            var rightChar = line.substring(range.end.column, range.end.column + 1);
	            if (rightChar == '}') {
	                range.end.column++;
	                return range;
	            } else {
	                context.maybeInsertedBrackets--;
	            }
	        }
	    });

	    this.add("parens", "insertion", function(state, action, editor, session, text) {
	        if (text == '(') {
	            initContext(editor);
	            var selection = editor.getSelectionRange();
	            var selected = session.doc.getTextRange(selection);
	            if (selected !== "" && editor.getWrapBehavioursEnabled()) {
	                return getWrapped(selection, selected, '(', ')');
	            } else if (CstyleBehaviour.isSaneInsertion(editor, session)) {
	                CstyleBehaviour.recordAutoInsert(editor, session, ")");
	                return {
	                    text: '()',
	                    selection: [1, 1]
	                };
	            }
	        } else if (text == ')') {
	            initContext(editor);
	            var cursor = editor.getCursorPosition();
	            var line = session.doc.getLine(cursor.row);
	            var rightChar = line.substring(cursor.column, cursor.column + 1);
	            if (rightChar == ')') {
	                var matching = session.$findOpeningBracket(')', {column: cursor.column + 1, row: cursor.row});
	                if (matching !== null && CstyleBehaviour.isAutoInsertedClosing(cursor, line, text)) {
	                    CstyleBehaviour.popAutoInsertedClosing();
	                    return {
	                        text: '',
	                        selection: [1, 1]
	                    };
	                }
	            }
	        }
	    });

	    this.add("parens", "deletion", function(state, action, editor, session, range) {
	        var selected = session.doc.getTextRange(range);
	        if (!range.isMultiLine() && selected == '(') {
	            initContext(editor);
	            var line = session.doc.getLine(range.start.row);
	            var rightChar = line.substring(range.start.column + 1, range.start.column + 2);
	            if (rightChar == ')') {
	                range.end.column++;
	                return range;
	            }
	        }
	    });

	    this.add("brackets", "insertion", function(state, action, editor, session, text) {
	        if (text == '[') {
	            initContext(editor);
	            var selection = editor.getSelectionRange();
	            var selected = session.doc.getTextRange(selection);
	            if (selected !== "" && editor.getWrapBehavioursEnabled()) {
	                return getWrapped(selection, selected, '[', ']');
	            } else if (CstyleBehaviour.isSaneInsertion(editor, session)) {
	                CstyleBehaviour.recordAutoInsert(editor, session, "]");
	                return {
	                    text: '[]',
	                    selection: [1, 1]
	                };
	            }
	        } else if (text == ']') {
	            initContext(editor);
	            var cursor = editor.getCursorPosition();
	            var line = session.doc.getLine(cursor.row);
	            var rightChar = line.substring(cursor.column, cursor.column + 1);
	            if (rightChar == ']') {
	                var matching = session.$findOpeningBracket(']', {column: cursor.column + 1, row: cursor.row});
	                if (matching !== null && CstyleBehaviour.isAutoInsertedClosing(cursor, line, text)) {
	                    CstyleBehaviour.popAutoInsertedClosing();
	                    return {
	                        text: '',
	                        selection: [1, 1]
	                    };
	                }
	            }
	        }
	    });

	    this.add("brackets", "deletion", function(state, action, editor, session, range) {
	        var selected = session.doc.getTextRange(range);
	        if (!range.isMultiLine() && selected == '[') {
	            initContext(editor);
	            var line = session.doc.getLine(range.start.row);
	            var rightChar = line.substring(range.start.column + 1, range.start.column + 2);
	            if (rightChar == ']') {
	                range.end.column++;
	                return range;
	            }
	        }
	    });

	    this.add("string_dquotes", "insertion", function(state, action, editor, session, text) {
	        var quotes = session.$mode.$quotes || defaultQuotes;
	        if (text.length == 1 && quotes[text]) {
	            if (this.lineCommentStart && this.lineCommentStart.indexOf(text) != -1)
	                return;
	            initContext(editor);
	            var quote = text;
	            var selection = editor.getSelectionRange();
	            var selected = session.doc.getTextRange(selection);
	            if (selected !== "" && (selected.length != 1 || !quotes[selected]) && editor.getWrapBehavioursEnabled()) {
	                return getWrapped(selection, selected, quote, quote);
	            } else if (!selected) {
	                var cursor = editor.getCursorPosition();
	                var line = session.doc.getLine(cursor.row);
	                var leftChar = line.substring(cursor.column-1, cursor.column);
	                var rightChar = line.substring(cursor.column, cursor.column + 1);

	                var token = session.getTokenAt(cursor.row, cursor.column);
	                var rightToken = session.getTokenAt(cursor.row, cursor.column + 1);
	                if (leftChar == "\\" && token && /escape/.test(token.type))
	                    return null;

	                var stringBefore = token && /string|escape/.test(token.type);
	                var stringAfter = !rightToken || /string|escape/.test(rightToken.type);

	                var pair;
	                if (rightChar == quote) {
	                    pair = stringBefore !== stringAfter;
	                    if (pair && /string\.end/.test(rightToken.type))
	                        pair = false;
	                } else {
	                    if (stringBefore && !stringAfter)
	                        return null; // wrap string with different quote
	                    if (stringBefore && stringAfter)
	                        return null; // do not pair quotes inside strings
	                    var wordRe = session.$mode.tokenRe;
	                    wordRe.lastIndex = 0;
	                    var isWordBefore = wordRe.test(leftChar);
	                    wordRe.lastIndex = 0;
	                    var isWordAfter = wordRe.test(leftChar);
	                    if (isWordBefore || isWordAfter)
	                        return null; // before or after alphanumeric
	                    if (rightChar && !/[\s;,.})\]\\]/.test(rightChar))
	                        return null; // there is rightChar and it isn't closing
	                    pair = true;
	                }
	                return {
	                    text: pair ? quote + quote : "",
	                    selection: [1,1]
	                };
	            }
	        }
	    });

	    this.add("string_dquotes", "deletion", function(state, action, editor, session, range) {
	        var selected = session.doc.getTextRange(range);
	        if (!range.isMultiLine() && (selected == '"' || selected == "'")) {
	            initContext(editor);
	            var line = session.doc.getLine(range.start.row);
	            var rightChar = line.substring(range.start.column + 1, range.start.column + 2);
	            if (rightChar == selected) {
	                range.end.column++;
	                return range;
	            }
	        }
	    });

	};


	CstyleBehaviour.isSaneInsertion = function(editor, session) {
	    var cursor = editor.getCursorPosition();
	    var iterator = new TokenIterator(session, cursor.row, cursor.column);
	    if (!this.$matchTokenType(iterator.getCurrentToken() || "text", SAFE_INSERT_IN_TOKENS)) {
	        var iterator2 = new TokenIterator(session, cursor.row, cursor.column + 1);
	        if (!this.$matchTokenType(iterator2.getCurrentToken() || "text", SAFE_INSERT_IN_TOKENS))
	            return false;
	    }
	    iterator.stepForward();
	    return iterator.getCurrentTokenRow() !== cursor.row ||
	        this.$matchTokenType(iterator.getCurrentToken() || "text", SAFE_INSERT_BEFORE_TOKENS);
	};

	CstyleBehaviour.$matchTokenType = function(token, types) {
	    return types.indexOf(token.type || token) > -1;
	};

	CstyleBehaviour.recordAutoInsert = function(editor, session, bracket) {
	    var cursor = editor.getCursorPosition();
	    var line = session.doc.getLine(cursor.row);
	    if (!this.isAutoInsertedClosing(cursor, line, context.autoInsertedLineEnd[0]))
	        context.autoInsertedBrackets = 0;
	    context.autoInsertedRow = cursor.row;
	    context.autoInsertedLineEnd = bracket + line.substr(cursor.column);
	    context.autoInsertedBrackets++;
	};

	CstyleBehaviour.recordMaybeInsert = function(editor, session, bracket) {
	    var cursor = editor.getCursorPosition();
	    var line = session.doc.getLine(cursor.row);
	    if (!this.isMaybeInsertedClosing(cursor, line))
	        context.maybeInsertedBrackets = 0;
	    context.maybeInsertedRow = cursor.row;
	    context.maybeInsertedLineStart = line.substr(0, cursor.column) + bracket;
	    context.maybeInsertedLineEnd = line.substr(cursor.column);
	    context.maybeInsertedBrackets++;
	};

	CstyleBehaviour.isAutoInsertedClosing = function(cursor, line, bracket) {
	    return context.autoInsertedBrackets > 0 &&
	        cursor.row === context.autoInsertedRow &&
	        bracket === context.autoInsertedLineEnd[0] &&
	        line.substr(cursor.column) === context.autoInsertedLineEnd;
	};

	CstyleBehaviour.isMaybeInsertedClosing = function(cursor, line) {
	    return context.maybeInsertedBrackets > 0 &&
	        cursor.row === context.maybeInsertedRow &&
	        line.substr(cursor.column) === context.maybeInsertedLineEnd &&
	        line.substr(0, cursor.column) == context.maybeInsertedLineStart;
	};

	CstyleBehaviour.popAutoInsertedClosing = function() {
	    context.autoInsertedLineEnd = context.autoInsertedLineEnd.substr(1);
	    context.autoInsertedBrackets--;
	};

	CstyleBehaviour.clearMaybeInsertedClosing = function() {
	    if (context) {
	        context.maybeInsertedBrackets = 0;
	        context.maybeInsertedRow = -1;
	    }
	};



	oop.inherits(CstyleBehaviour, Behaviour);

	exports.CstyleBehaviour = CstyleBehaviour;
	});

	ace.define("ace/unicode",["require","exports","module"], function(acequire, exports, module) {
	"use strict";
	exports.packages = {};

	addUnicodePackage({
	    L:  "0041-005A0061-007A00AA00B500BA00C0-00D600D8-00F600F8-02C102C6-02D102E0-02E402EC02EE0370-037403760377037A-037D03860388-038A038C038E-03A103A3-03F503F7-0481048A-05250531-055605590561-058705D0-05EA05F0-05F20621-064A066E066F0671-06D306D506E506E606EE06EF06FA-06FC06FF07100712-072F074D-07A507B107CA-07EA07F407F507FA0800-0815081A082408280904-0939093D09500958-0961097109720979-097F0985-098C098F09900993-09A809AA-09B009B209B6-09B909BD09CE09DC09DD09DF-09E109F009F10A05-0A0A0A0F0A100A13-0A280A2A-0A300A320A330A350A360A380A390A59-0A5C0A5E0A72-0A740A85-0A8D0A8F-0A910A93-0AA80AAA-0AB00AB20AB30AB5-0AB90ABD0AD00AE00AE10B05-0B0C0B0F0B100B13-0B280B2A-0B300B320B330B35-0B390B3D0B5C0B5D0B5F-0B610B710B830B85-0B8A0B8E-0B900B92-0B950B990B9A0B9C0B9E0B9F0BA30BA40BA8-0BAA0BAE-0BB90BD00C05-0C0C0C0E-0C100C12-0C280C2A-0C330C35-0C390C3D0C580C590C600C610C85-0C8C0C8E-0C900C92-0CA80CAA-0CB30CB5-0CB90CBD0CDE0CE00CE10D05-0D0C0D0E-0D100D12-0D280D2A-0D390D3D0D600D610D7A-0D7F0D85-0D960D9A-0DB10DB3-0DBB0DBD0DC0-0DC60E01-0E300E320E330E40-0E460E810E820E840E870E880E8A0E8D0E94-0E970E99-0E9F0EA1-0EA30EA50EA70EAA0EAB0EAD-0EB00EB20EB30EBD0EC0-0EC40EC60EDC0EDD0F000F40-0F470F49-0F6C0F88-0F8B1000-102A103F1050-1055105A-105D106110651066106E-10701075-1081108E10A0-10C510D0-10FA10FC1100-1248124A-124D1250-12561258125A-125D1260-1288128A-128D1290-12B012B2-12B512B8-12BE12C012C2-12C512C8-12D612D8-13101312-13151318-135A1380-138F13A0-13F41401-166C166F-167F1681-169A16A0-16EA1700-170C170E-17111720-17311740-17511760-176C176E-17701780-17B317D717DC1820-18771880-18A818AA18B0-18F51900-191C1950-196D1970-19741980-19AB19C1-19C71A00-1A161A20-1A541AA71B05-1B331B45-1B4B1B83-1BA01BAE1BAF1C00-1C231C4D-1C4F1C5A-1C7D1CE9-1CEC1CEE-1CF11D00-1DBF1E00-1F151F18-1F1D1F20-1F451F48-1F4D1F50-1F571F591F5B1F5D1F5F-1F7D1F80-1FB41FB6-1FBC1FBE1FC2-1FC41FC6-1FCC1FD0-1FD31FD6-1FDB1FE0-1FEC1FF2-1FF41FF6-1FFC2071207F2090-209421022107210A-211321152119-211D212421262128212A-212D212F-2139213C-213F2145-2149214E218321842C00-2C2E2C30-2C5E2C60-2CE42CEB-2CEE2D00-2D252D30-2D652D6F2D80-2D962DA0-2DA62DA8-2DAE2DB0-2DB62DB8-2DBE2DC0-2DC62DC8-2DCE2DD0-2DD62DD8-2DDE2E2F300530063031-3035303B303C3041-3096309D-309F30A1-30FA30FC-30FF3105-312D3131-318E31A0-31B731F0-31FF3400-4DB54E00-9FCBA000-A48CA4D0-A4FDA500-A60CA610-A61FA62AA62BA640-A65FA662-A66EA67F-A697A6A0-A6E5A717-A71FA722-A788A78BA78CA7FB-A801A803-A805A807-A80AA80C-A822A840-A873A882-A8B3A8F2-A8F7A8FBA90A-A925A930-A946A960-A97CA984-A9B2A9CFAA00-AA28AA40-AA42AA44-AA4BAA60-AA76AA7AAA80-AAAFAAB1AAB5AAB6AAB9-AABDAAC0AAC2AADB-AADDABC0-ABE2AC00-D7A3D7B0-D7C6D7CB-D7FBF900-FA2DFA30-FA6DFA70-FAD9FB00-FB06FB13-FB17FB1DFB1F-FB28FB2A-FB36FB38-FB3CFB3EFB40FB41FB43FB44FB46-FBB1FBD3-FD3DFD50-FD8FFD92-FDC7FDF0-FDFBFE70-FE74FE76-FEFCFF21-FF3AFF41-FF5AFF66-FFBEFFC2-FFC7FFCA-FFCFFFD2-FFD7FFDA-FFDC",
	    Ll: "0061-007A00AA00B500BA00DF-00F600F8-00FF01010103010501070109010B010D010F01110113011501170119011B011D011F01210123012501270129012B012D012F01310133013501370138013A013C013E014001420144014601480149014B014D014F01510153015501570159015B015D015F01610163016501670169016B016D016F0171017301750177017A017C017E-0180018301850188018C018D019201950199-019B019E01A101A301A501A801AA01AB01AD01B001B401B601B901BA01BD-01BF01C601C901CC01CE01D001D201D401D601D801DA01DC01DD01DF01E101E301E501E701E901EB01ED01EF01F001F301F501F901FB01FD01FF02010203020502070209020B020D020F02110213021502170219021B021D021F02210223022502270229022B022D022F02310233-0239023C023F0240024202470249024B024D024F-02930295-02AF037103730377037B-037D039003AC-03CE03D003D103D5-03D703D903DB03DD03DF03E103E303E503E703E903EB03ED03EF-03F303F503F803FB03FC0430-045F04610463046504670469046B046D046F04710473047504770479047B047D047F0481048B048D048F04910493049504970499049B049D049F04A104A304A504A704A904AB04AD04AF04B104B304B504B704B904BB04BD04BF04C204C404C604C804CA04CC04CE04CF04D104D304D504D704D904DB04DD04DF04E104E304E504E704E904EB04ED04EF04F104F304F504F704F904FB04FD04FF05010503050505070509050B050D050F05110513051505170519051B051D051F0521052305250561-05871D00-1D2B1D62-1D771D79-1D9A1E011E031E051E071E091E0B1E0D1E0F1E111E131E151E171E191E1B1E1D1E1F1E211E231E251E271E291E2B1E2D1E2F1E311E331E351E371E391E3B1E3D1E3F1E411E431E451E471E491E4B1E4D1E4F1E511E531E551E571E591E5B1E5D1E5F1E611E631E651E671E691E6B1E6D1E6F1E711E731E751E771E791E7B1E7D1E7F1E811E831E851E871E891E8B1E8D1E8F1E911E931E95-1E9D1E9F1EA11EA31EA51EA71EA91EAB1EAD1EAF1EB11EB31EB51EB71EB91EBB1EBD1EBF1EC11EC31EC51EC71EC91ECB1ECD1ECF1ED11ED31ED51ED71ED91EDB1EDD1EDF1EE11EE31EE51EE71EE91EEB1EED1EEF1EF11EF31EF51EF71EF91EFB1EFD1EFF-1F071F10-1F151F20-1F271F30-1F371F40-1F451F50-1F571F60-1F671F70-1F7D1F80-1F871F90-1F971FA0-1FA71FB0-1FB41FB61FB71FBE1FC2-1FC41FC61FC71FD0-1FD31FD61FD71FE0-1FE71FF2-1FF41FF61FF7210A210E210F2113212F21342139213C213D2146-2149214E21842C30-2C5E2C612C652C662C682C6A2C6C2C712C732C742C76-2C7C2C812C832C852C872C892C8B2C8D2C8F2C912C932C952C972C992C9B2C9D2C9F2CA12CA32CA52CA72CA92CAB2CAD2CAF2CB12CB32CB52CB72CB92CBB2CBD2CBF2CC12CC32CC52CC72CC92CCB2CCD2CCF2CD12CD32CD52CD72CD92CDB2CDD2CDF2CE12CE32CE42CEC2CEE2D00-2D25A641A643A645A647A649A64BA64DA64FA651A653A655A657A659A65BA65DA65FA663A665A667A669A66BA66DA681A683A685A687A689A68BA68DA68FA691A693A695A697A723A725A727A729A72BA72DA72F-A731A733A735A737A739A73BA73DA73FA741A743A745A747A749A74BA74DA74FA751A753A755A757A759A75BA75DA75FA761A763A765A767A769A76BA76DA76FA771-A778A77AA77CA77FA781A783A785A787A78CFB00-FB06FB13-FB17FF41-FF5A",
	    Lu: "0041-005A00C0-00D600D8-00DE01000102010401060108010A010C010E01100112011401160118011A011C011E01200122012401260128012A012C012E01300132013401360139013B013D013F0141014301450147014A014C014E01500152015401560158015A015C015E01600162016401660168016A016C016E017001720174017601780179017B017D018101820184018601870189-018B018E-0191019301940196-0198019C019D019F01A001A201A401A601A701A901AC01AE01AF01B1-01B301B501B701B801BC01C401C701CA01CD01CF01D101D301D501D701D901DB01DE01E001E201E401E601E801EA01EC01EE01F101F401F6-01F801FA01FC01FE02000202020402060208020A020C020E02100212021402160218021A021C021E02200222022402260228022A022C022E02300232023A023B023D023E02410243-02460248024A024C024E03700372037603860388-038A038C038E038F0391-03A103A3-03AB03CF03D2-03D403D803DA03DC03DE03E003E203E403E603E803EA03EC03EE03F403F703F903FA03FD-042F04600462046404660468046A046C046E04700472047404760478047A047C047E0480048A048C048E04900492049404960498049A049C049E04A004A204A404A604A804AA04AC04AE04B004B204B404B604B804BA04BC04BE04C004C104C304C504C704C904CB04CD04D004D204D404D604D804DA04DC04DE04E004E204E404E604E804EA04EC04EE04F004F204F404F604F804FA04FC04FE05000502050405060508050A050C050E05100512051405160518051A051C051E0520052205240531-055610A0-10C51E001E021E041E061E081E0A1E0C1E0E1E101E121E141E161E181E1A1E1C1E1E1E201E221E241E261E281E2A1E2C1E2E1E301E321E341E361E381E3A1E3C1E3E1E401E421E441E461E481E4A1E4C1E4E1E501E521E541E561E581E5A1E5C1E5E1E601E621E641E661E681E6A1E6C1E6E1E701E721E741E761E781E7A1E7C1E7E1E801E821E841E861E881E8A1E8C1E8E1E901E921E941E9E1EA01EA21EA41EA61EA81EAA1EAC1EAE1EB01EB21EB41EB61EB81EBA1EBC1EBE1EC01EC21EC41EC61EC81ECA1ECC1ECE1ED01ED21ED41ED61ED81EDA1EDC1EDE1EE01EE21EE41EE61EE81EEA1EEC1EEE1EF01EF21EF41EF61EF81EFA1EFC1EFE1F08-1F0F1F18-1F1D1F28-1F2F1F38-1F3F1F48-1F4D1F591F5B1F5D1F5F1F68-1F6F1FB8-1FBB1FC8-1FCB1FD8-1FDB1FE8-1FEC1FF8-1FFB21022107210B-210D2110-211221152119-211D212421262128212A-212D2130-2133213E213F214521832C00-2C2E2C602C62-2C642C672C692C6B2C6D-2C702C722C752C7E-2C802C822C842C862C882C8A2C8C2C8E2C902C922C942C962C982C9A2C9C2C9E2CA02CA22CA42CA62CA82CAA2CAC2CAE2CB02CB22CB42CB62CB82CBA2CBC2CBE2CC02CC22CC42CC62CC82CCA2CCC2CCE2CD02CD22CD42CD62CD82CDA2CDC2CDE2CE02CE22CEB2CEDA640A642A644A646A648A64AA64CA64EA650A652A654A656A658A65AA65CA65EA662A664A666A668A66AA66CA680A682A684A686A688A68AA68CA68EA690A692A694A696A722A724A726A728A72AA72CA72EA732A734A736A738A73AA73CA73EA740A742A744A746A748A74AA74CA74EA750A752A754A756A758A75AA75CA75EA760A762A764A766A768A76AA76CA76EA779A77BA77DA77EA780A782A784A786A78BFF21-FF3A",
	    Lt: "01C501C801CB01F21F88-1F8F1F98-1F9F1FA8-1FAF1FBC1FCC1FFC",
	    Lm: "02B0-02C102C6-02D102E0-02E402EC02EE0374037A0559064006E506E607F407F507FA081A0824082809710E460EC610FC17D718431AA71C78-1C7D1D2C-1D611D781D9B-1DBF2071207F2090-20942C7D2D6F2E2F30053031-3035303B309D309E30FC-30FEA015A4F8-A4FDA60CA67FA717-A71FA770A788A9CFAA70AADDFF70FF9EFF9F",
	    Lo: "01BB01C0-01C3029405D0-05EA05F0-05F20621-063F0641-064A066E066F0671-06D306D506EE06EF06FA-06FC06FF07100712-072F074D-07A507B107CA-07EA0800-08150904-0939093D09500958-096109720979-097F0985-098C098F09900993-09A809AA-09B009B209B6-09B909BD09CE09DC09DD09DF-09E109F009F10A05-0A0A0A0F0A100A13-0A280A2A-0A300A320A330A350A360A380A390A59-0A5C0A5E0A72-0A740A85-0A8D0A8F-0A910A93-0AA80AAA-0AB00AB20AB30AB5-0AB90ABD0AD00AE00AE10B05-0B0C0B0F0B100B13-0B280B2A-0B300B320B330B35-0B390B3D0B5C0B5D0B5F-0B610B710B830B85-0B8A0B8E-0B900B92-0B950B990B9A0B9C0B9E0B9F0BA30BA40BA8-0BAA0BAE-0BB90BD00C05-0C0C0C0E-0C100C12-0C280C2A-0C330C35-0C390C3D0C580C590C600C610C85-0C8C0C8E-0C900C92-0CA80CAA-0CB30CB5-0CB90CBD0CDE0CE00CE10D05-0D0C0D0E-0D100D12-0D280D2A-0D390D3D0D600D610D7A-0D7F0D85-0D960D9A-0DB10DB3-0DBB0DBD0DC0-0DC60E01-0E300E320E330E40-0E450E810E820E840E870E880E8A0E8D0E94-0E970E99-0E9F0EA1-0EA30EA50EA70EAA0EAB0EAD-0EB00EB20EB30EBD0EC0-0EC40EDC0EDD0F000F40-0F470F49-0F6C0F88-0F8B1000-102A103F1050-1055105A-105D106110651066106E-10701075-1081108E10D0-10FA1100-1248124A-124D1250-12561258125A-125D1260-1288128A-128D1290-12B012B2-12B512B8-12BE12C012C2-12C512C8-12D612D8-13101312-13151318-135A1380-138F13A0-13F41401-166C166F-167F1681-169A16A0-16EA1700-170C170E-17111720-17311740-17511760-176C176E-17701780-17B317DC1820-18421844-18771880-18A818AA18B0-18F51900-191C1950-196D1970-19741980-19AB19C1-19C71A00-1A161A20-1A541B05-1B331B45-1B4B1B83-1BA01BAE1BAF1C00-1C231C4D-1C4F1C5A-1C771CE9-1CEC1CEE-1CF12135-21382D30-2D652D80-2D962DA0-2DA62DA8-2DAE2DB0-2DB62DB8-2DBE2DC0-2DC62DC8-2DCE2DD0-2DD62DD8-2DDE3006303C3041-3096309F30A1-30FA30FF3105-312D3131-318E31A0-31B731F0-31FF3400-4DB54E00-9FCBA000-A014A016-A48CA4D0-A4F7A500-A60BA610-A61FA62AA62BA66EA6A0-A6E5A7FB-A801A803-A805A807-A80AA80C-A822A840-A873A882-A8B3A8F2-A8F7A8FBA90A-A925A930-A946A960-A97CA984-A9B2AA00-AA28AA40-AA42AA44-AA4BAA60-AA6FAA71-AA76AA7AAA80-AAAFAAB1AAB5AAB6AAB9-AABDAAC0AAC2AADBAADCABC0-ABE2AC00-D7A3D7B0-D7C6D7CB-D7FBF900-FA2DFA30-FA6DFA70-FAD9FB1DFB1F-FB28FB2A-FB36FB38-FB3CFB3EFB40FB41FB43FB44FB46-FBB1FBD3-FD3DFD50-FD8FFD92-FDC7FDF0-FDFBFE70-FE74FE76-FEFCFF66-FF6FFF71-FF9DFFA0-FFBEFFC2-FFC7FFCA-FFCFFFD2-FFD7FFDA-FFDC",
	    M:  "0300-036F0483-04890591-05BD05BF05C105C205C405C505C70610-061A064B-065E067006D6-06DC06DE-06E406E706E806EA-06ED07110730-074A07A6-07B007EB-07F30816-0819081B-08230825-08270829-082D0900-0903093C093E-094E0951-0955096209630981-098309BC09BE-09C409C709C809CB-09CD09D709E209E30A01-0A030A3C0A3E-0A420A470A480A4B-0A4D0A510A700A710A750A81-0A830ABC0ABE-0AC50AC7-0AC90ACB-0ACD0AE20AE30B01-0B030B3C0B3E-0B440B470B480B4B-0B4D0B560B570B620B630B820BBE-0BC20BC6-0BC80BCA-0BCD0BD70C01-0C030C3E-0C440C46-0C480C4A-0C4D0C550C560C620C630C820C830CBC0CBE-0CC40CC6-0CC80CCA-0CCD0CD50CD60CE20CE30D020D030D3E-0D440D46-0D480D4A-0D4D0D570D620D630D820D830DCA0DCF-0DD40DD60DD8-0DDF0DF20DF30E310E34-0E3A0E47-0E4E0EB10EB4-0EB90EBB0EBC0EC8-0ECD0F180F190F350F370F390F3E0F3F0F71-0F840F860F870F90-0F970F99-0FBC0FC6102B-103E1056-1059105E-10601062-10641067-106D1071-10741082-108D108F109A-109D135F1712-17141732-1734175217531772177317B6-17D317DD180B-180D18A91920-192B1930-193B19B0-19C019C819C91A17-1A1B1A55-1A5E1A60-1A7C1A7F1B00-1B041B34-1B441B6B-1B731B80-1B821BA1-1BAA1C24-1C371CD0-1CD21CD4-1CE81CED1CF21DC0-1DE61DFD-1DFF20D0-20F02CEF-2CF12DE0-2DFF302A-302F3099309AA66F-A672A67CA67DA6F0A6F1A802A806A80BA823-A827A880A881A8B4-A8C4A8E0-A8F1A926-A92DA947-A953A980-A983A9B3-A9C0AA29-AA36AA43AA4CAA4DAA7BAAB0AAB2-AAB4AAB7AAB8AABEAABFAAC1ABE3-ABEAABECABEDFB1EFE00-FE0FFE20-FE26",
	    Mn: "0300-036F0483-04870591-05BD05BF05C105C205C405C505C70610-061A064B-065E067006D6-06DC06DF-06E406E706E806EA-06ED07110730-074A07A6-07B007EB-07F30816-0819081B-08230825-08270829-082D0900-0902093C0941-0948094D0951-095509620963098109BC09C1-09C409CD09E209E30A010A020A3C0A410A420A470A480A4B-0A4D0A510A700A710A750A810A820ABC0AC1-0AC50AC70AC80ACD0AE20AE30B010B3C0B3F0B41-0B440B4D0B560B620B630B820BC00BCD0C3E-0C400C46-0C480C4A-0C4D0C550C560C620C630CBC0CBF0CC60CCC0CCD0CE20CE30D41-0D440D4D0D620D630DCA0DD2-0DD40DD60E310E34-0E3A0E47-0E4E0EB10EB4-0EB90EBB0EBC0EC8-0ECD0F180F190F350F370F390F71-0F7E0F80-0F840F860F870F90-0F970F99-0FBC0FC6102D-10301032-10371039103A103D103E10581059105E-10601071-1074108210851086108D109D135F1712-17141732-1734175217531772177317B7-17BD17C617C9-17D317DD180B-180D18A91920-19221927192819321939-193B1A171A181A561A58-1A5E1A601A621A65-1A6C1A73-1A7C1A7F1B00-1B031B341B36-1B3A1B3C1B421B6B-1B731B801B811BA2-1BA51BA81BA91C2C-1C331C361C371CD0-1CD21CD4-1CE01CE2-1CE81CED1DC0-1DE61DFD-1DFF20D0-20DC20E120E5-20F02CEF-2CF12DE0-2DFF302A-302F3099309AA66FA67CA67DA6F0A6F1A802A806A80BA825A826A8C4A8E0-A8F1A926-A92DA947-A951A980-A982A9B3A9B6-A9B9A9BCAA29-AA2EAA31AA32AA35AA36AA43AA4CAAB0AAB2-AAB4AAB7AAB8AABEAABFAAC1ABE5ABE8ABEDFB1EFE00-FE0FFE20-FE26",
	    Mc: "0903093E-09400949-094C094E0982098309BE-09C009C709C809CB09CC09D70A030A3E-0A400A830ABE-0AC00AC90ACB0ACC0B020B030B3E0B400B470B480B4B0B4C0B570BBE0BBF0BC10BC20BC6-0BC80BCA-0BCC0BD70C01-0C030C41-0C440C820C830CBE0CC0-0CC40CC70CC80CCA0CCB0CD50CD60D020D030D3E-0D400D46-0D480D4A-0D4C0D570D820D830DCF-0DD10DD8-0DDF0DF20DF30F3E0F3F0F7F102B102C10311038103B103C105610571062-10641067-106D108310841087-108C108F109A-109C17B617BE-17C517C717C81923-19261929-192B193019311933-193819B0-19C019C819C91A19-1A1B1A551A571A611A631A641A6D-1A721B041B351B3B1B3D-1B411B431B441B821BA11BA61BA71BAA1C24-1C2B1C341C351CE11CF2A823A824A827A880A881A8B4-A8C3A952A953A983A9B4A9B5A9BAA9BBA9BD-A9C0AA2FAA30AA33AA34AA4DAA7BABE3ABE4ABE6ABE7ABE9ABEAABEC",
	    Me: "0488048906DE20DD-20E020E2-20E4A670-A672",
	    N:  "0030-003900B200B300B900BC-00BE0660-066906F0-06F907C0-07C90966-096F09E6-09EF09F4-09F90A66-0A6F0AE6-0AEF0B66-0B6F0BE6-0BF20C66-0C6F0C78-0C7E0CE6-0CEF0D66-0D750E50-0E590ED0-0ED90F20-0F331040-10491090-10991369-137C16EE-16F017E0-17E917F0-17F91810-18191946-194F19D0-19DA1A80-1A891A90-1A991B50-1B591BB0-1BB91C40-1C491C50-1C5920702074-20792080-20892150-21822185-21892460-249B24EA-24FF2776-27932CFD30073021-30293038-303A3192-31953220-32293251-325F3280-328932B1-32BFA620-A629A6E6-A6EFA830-A835A8D0-A8D9A900-A909A9D0-A9D9AA50-AA59ABF0-ABF9FF10-FF19",
	    Nd: "0030-00390660-066906F0-06F907C0-07C90966-096F09E6-09EF0A66-0A6F0AE6-0AEF0B66-0B6F0BE6-0BEF0C66-0C6F0CE6-0CEF0D66-0D6F0E50-0E590ED0-0ED90F20-0F291040-10491090-109917E0-17E91810-18191946-194F19D0-19DA1A80-1A891A90-1A991B50-1B591BB0-1BB91C40-1C491C50-1C59A620-A629A8D0-A8D9A900-A909A9D0-A9D9AA50-AA59ABF0-ABF9FF10-FF19",
	    Nl: "16EE-16F02160-21822185-218830073021-30293038-303AA6E6-A6EF",
	    No: "00B200B300B900BC-00BE09F4-09F90BF0-0BF20C78-0C7E0D70-0D750F2A-0F331369-137C17F0-17F920702074-20792080-20892150-215F21892460-249B24EA-24FF2776-27932CFD3192-31953220-32293251-325F3280-328932B1-32BFA830-A835",
	    P:  "0021-00230025-002A002C-002F003A003B003F0040005B-005D005F007B007D00A100AB00B700BB00BF037E0387055A-055F0589058A05BE05C005C305C605F305F40609060A060C060D061B061E061F066A-066D06D40700-070D07F7-07F90830-083E0964096509700DF40E4F0E5A0E5B0F04-0F120F3A-0F3D0F850FD0-0FD4104A-104F10FB1361-13681400166D166E169B169C16EB-16ED1735173617D4-17D617D8-17DA1800-180A1944194519DE19DF1A1E1A1F1AA0-1AA61AA8-1AAD1B5A-1B601C3B-1C3F1C7E1C7F1CD32010-20272030-20432045-20512053-205E207D207E208D208E2329232A2768-277527C527C627E6-27EF2983-299829D8-29DB29FC29FD2CF9-2CFC2CFE2CFF2E00-2E2E2E302E313001-30033008-30113014-301F3030303D30A030FBA4FEA4FFA60D-A60FA673A67EA6F2-A6F7A874-A877A8CEA8CFA8F8-A8FAA92EA92FA95FA9C1-A9CDA9DEA9DFAA5C-AA5FAADEAADFABEBFD3EFD3FFE10-FE19FE30-FE52FE54-FE61FE63FE68FE6AFE6BFF01-FF03FF05-FF0AFF0C-FF0FFF1AFF1BFF1FFF20FF3B-FF3DFF3FFF5BFF5DFF5F-FF65",
	    Pd: "002D058A05BE140018062010-20152E172E1A301C303030A0FE31FE32FE58FE63FF0D",
	    Ps: "0028005B007B0F3A0F3C169B201A201E2045207D208D23292768276A276C276E27702772277427C527E627E827EA27EC27EE2983298529872989298B298D298F299129932995299729D829DA29FC2E222E242E262E283008300A300C300E3010301430163018301A301DFD3EFE17FE35FE37FE39FE3BFE3DFE3FFE41FE43FE47FE59FE5BFE5DFF08FF3BFF5BFF5FFF62",
	    Pe: "0029005D007D0F3B0F3D169C2046207E208E232A2769276B276D276F27712773277527C627E727E927EB27ED27EF298429862988298A298C298E2990299229942996299829D929DB29FD2E232E252E272E293009300B300D300F3011301530173019301B301E301FFD3FFE18FE36FE38FE3AFE3CFE3EFE40FE42FE44FE48FE5AFE5CFE5EFF09FF3DFF5DFF60FF63",
	    Pi: "00AB2018201B201C201F20392E022E042E092E0C2E1C2E20",
	    Pf: "00BB2019201D203A2E032E052E0A2E0D2E1D2E21",
	    Pc: "005F203F20402054FE33FE34FE4D-FE4FFF3F",
	    Po: "0021-00230025-0027002A002C002E002F003A003B003F0040005C00A100B700BF037E0387055A-055F058905C005C305C605F305F40609060A060C060D061B061E061F066A-066D06D40700-070D07F7-07F90830-083E0964096509700DF40E4F0E5A0E5B0F04-0F120F850FD0-0FD4104A-104F10FB1361-1368166D166E16EB-16ED1735173617D4-17D617D8-17DA1800-18051807-180A1944194519DE19DF1A1E1A1F1AA0-1AA61AA8-1AAD1B5A-1B601C3B-1C3F1C7E1C7F1CD3201620172020-20272030-2038203B-203E2041-20432047-205120532055-205E2CF9-2CFC2CFE2CFF2E002E012E06-2E082E0B2E0E-2E162E182E192E1B2E1E2E1F2E2A-2E2E2E302E313001-3003303D30FBA4FEA4FFA60D-A60FA673A67EA6F2-A6F7A874-A877A8CEA8CFA8F8-A8FAA92EA92FA95FA9C1-A9CDA9DEA9DFAA5C-AA5FAADEAADFABEBFE10-FE16FE19FE30FE45FE46FE49-FE4CFE50-FE52FE54-FE57FE5F-FE61FE68FE6AFE6BFF01-FF03FF05-FF07FF0AFF0CFF0EFF0FFF1AFF1BFF1FFF20FF3CFF61FF64FF65",
	    S:  "0024002B003C-003E005E0060007C007E00A2-00A900AC00AE-00B100B400B600B800D700F702C2-02C502D2-02DF02E5-02EB02ED02EF-02FF03750384038503F604820606-0608060B060E060F06E906FD06FE07F609F209F309FA09FB0AF10B700BF3-0BFA0C7F0CF10CF20D790E3F0F01-0F030F13-0F170F1A-0F1F0F340F360F380FBE-0FC50FC7-0FCC0FCE0FCF0FD5-0FD8109E109F13601390-139917DB194019E0-19FF1B61-1B6A1B74-1B7C1FBD1FBF-1FC11FCD-1FCF1FDD-1FDF1FED-1FEF1FFD1FFE20442052207A-207C208A-208C20A0-20B8210021012103-21062108210921142116-2118211E-2123212521272129212E213A213B2140-2144214A-214D214F2190-2328232B-23E82400-24262440-244A249C-24E92500-26CD26CF-26E126E326E8-26FF2701-27042706-2709270C-27272729-274B274D274F-27522756-275E2761-276727942798-27AF27B1-27BE27C0-27C427C7-27CA27CC27D0-27E527F0-29822999-29D729DC-29FB29FE-2B4C2B50-2B592CE5-2CEA2E80-2E992E9B-2EF32F00-2FD52FF0-2FFB300430123013302030363037303E303F309B309C319031913196-319F31C0-31E33200-321E322A-32503260-327F328A-32B032C0-32FE3300-33FF4DC0-4DFFA490-A4C6A700-A716A720A721A789A78AA828-A82BA836-A839AA77-AA79FB29FDFCFDFDFE62FE64-FE66FE69FF04FF0BFF1C-FF1EFF3EFF40FF5CFF5EFFE0-FFE6FFE8-FFEEFFFCFFFD",
	    Sm: "002B003C-003E007C007E00AC00B100D700F703F60606-060820442052207A-207C208A-208C2140-2144214B2190-2194219A219B21A021A321A621AE21CE21CF21D221D421F4-22FF2308-230B23202321237C239B-23B323DC-23E125B725C125F8-25FF266F27C0-27C427C7-27CA27CC27D0-27E527F0-27FF2900-29822999-29D729DC-29FB29FE-2AFF2B30-2B442B47-2B4CFB29FE62FE64-FE66FF0BFF1C-FF1EFF5CFF5EFFE2FFE9-FFEC",
	    Sc: "002400A2-00A5060B09F209F309FB0AF10BF90E3F17DB20A0-20B8A838FDFCFE69FF04FFE0FFE1FFE5FFE6",
	    Sk: "005E006000A800AF00B400B802C2-02C502D2-02DF02E5-02EB02ED02EF-02FF0375038403851FBD1FBF-1FC11FCD-1FCF1FDD-1FDF1FED-1FEF1FFD1FFE309B309CA700-A716A720A721A789A78AFF3EFF40FFE3",
	    So: "00A600A700A900AE00B000B60482060E060F06E906FD06FE07F609FA0B700BF3-0BF80BFA0C7F0CF10CF20D790F01-0F030F13-0F170F1A-0F1F0F340F360F380FBE-0FC50FC7-0FCC0FCE0FCF0FD5-0FD8109E109F13601390-1399194019E0-19FF1B61-1B6A1B74-1B7C210021012103-21062108210921142116-2118211E-2123212521272129212E213A213B214A214C214D214F2195-2199219C-219F21A121A221A421A521A7-21AD21AF-21CD21D021D121D321D5-21F32300-2307230C-231F2322-2328232B-237B237D-239A23B4-23DB23E2-23E82400-24262440-244A249C-24E92500-25B625B8-25C025C2-25F72600-266E2670-26CD26CF-26E126E326E8-26FF2701-27042706-2709270C-27272729-274B274D274F-27522756-275E2761-276727942798-27AF27B1-27BE2800-28FF2B00-2B2F2B452B462B50-2B592CE5-2CEA2E80-2E992E9B-2EF32F00-2FD52FF0-2FFB300430123013302030363037303E303F319031913196-319F31C0-31E33200-321E322A-32503260-327F328A-32B032C0-32FE3300-33FF4DC0-4DFFA490-A4C6A828-A82BA836A837A839AA77-AA79FDFDFFE4FFE8FFEDFFEEFFFCFFFD",
	    Z:  "002000A01680180E2000-200A20282029202F205F3000",
	    Zs: "002000A01680180E2000-200A202F205F3000",
	    Zl: "2028",
	    Zp: "2029",
	    C:  "0000-001F007F-009F00AD03780379037F-0383038B038D03A20526-05300557055805600588058B-059005C8-05CF05EB-05EF05F5-0605061C061D0620065F06DD070E070F074B074C07B2-07BF07FB-07FF082E082F083F-08FF093A093B094F095609570973-097809800984098D098E0991099209A909B109B3-09B509BA09BB09C509C609C909CA09CF-09D609D8-09DB09DE09E409E509FC-0A000A040A0B-0A0E0A110A120A290A310A340A370A3A0A3B0A3D0A43-0A460A490A4A0A4E-0A500A52-0A580A5D0A5F-0A650A76-0A800A840A8E0A920AA90AB10AB40ABA0ABB0AC60ACA0ACE0ACF0AD1-0ADF0AE40AE50AF00AF2-0B000B040B0D0B0E0B110B120B290B310B340B3A0B3B0B450B460B490B4A0B4E-0B550B58-0B5B0B5E0B640B650B72-0B810B840B8B-0B8D0B910B96-0B980B9B0B9D0BA0-0BA20BA5-0BA70BAB-0BAD0BBA-0BBD0BC3-0BC50BC90BCE0BCF0BD1-0BD60BD8-0BE50BFB-0C000C040C0D0C110C290C340C3A-0C3C0C450C490C4E-0C540C570C5A-0C5F0C640C650C70-0C770C800C810C840C8D0C910CA90CB40CBA0CBB0CC50CC90CCE-0CD40CD7-0CDD0CDF0CE40CE50CF00CF3-0D010D040D0D0D110D290D3A-0D3C0D450D490D4E-0D560D58-0D5F0D640D650D76-0D780D800D810D840D97-0D990DB20DBC0DBE0DBF0DC7-0DC90DCB-0DCE0DD50DD70DE0-0DF10DF5-0E000E3B-0E3E0E5C-0E800E830E850E860E890E8B0E8C0E8E-0E930E980EA00EA40EA60EA80EA90EAC0EBA0EBE0EBF0EC50EC70ECE0ECF0EDA0EDB0EDE-0EFF0F480F6D-0F700F8C-0F8F0F980FBD0FCD0FD9-0FFF10C6-10CF10FD-10FF1249124E124F12571259125E125F1289128E128F12B112B612B712BF12C112C612C712D7131113161317135B-135E137D-137F139A-139F13F5-13FF169D-169F16F1-16FF170D1715-171F1737-173F1754-175F176D17711774-177F17B417B517DE17DF17EA-17EF17FA-17FF180F181A-181F1878-187F18AB-18AF18F6-18FF191D-191F192C-192F193C-193F1941-1943196E196F1975-197F19AC-19AF19CA-19CF19DB-19DD1A1C1A1D1A5F1A7D1A7E1A8A-1A8F1A9A-1A9F1AAE-1AFF1B4C-1B4F1B7D-1B7F1BAB-1BAD1BBA-1BFF1C38-1C3A1C4A-1C4C1C80-1CCF1CF3-1CFF1DE7-1DFC1F161F171F1E1F1F1F461F471F4E1F4F1F581F5A1F5C1F5E1F7E1F7F1FB51FC51FD41FD51FDC1FF01FF11FF51FFF200B-200F202A-202E2060-206F20722073208F2095-209F20B9-20CF20F1-20FF218A-218F23E9-23FF2427-243F244B-245F26CE26E226E4-26E727002705270A270B2728274C274E2753-2755275F27602795-279727B027BF27CB27CD-27CF2B4D-2B4F2B5A-2BFF2C2F2C5F2CF2-2CF82D26-2D2F2D66-2D6E2D70-2D7F2D97-2D9F2DA72DAF2DB72DBF2DC72DCF2DD72DDF2E32-2E7F2E9A2EF4-2EFF2FD6-2FEF2FFC-2FFF3040309730983100-3104312E-3130318F31B8-31BF31E4-31EF321F32FF4DB6-4DBF9FCC-9FFFA48D-A48FA4C7-A4CFA62C-A63FA660A661A674-A67BA698-A69FA6F8-A6FFA78D-A7FAA82C-A82FA83A-A83FA878-A87FA8C5-A8CDA8DA-A8DFA8FC-A8FFA954-A95EA97D-A97FA9CEA9DA-A9DDA9E0-A9FFAA37-AA3FAA4EAA4FAA5AAA5BAA7C-AA7FAAC3-AADAAAE0-ABBFABEEABEFABFA-ABFFD7A4-D7AFD7C7-D7CAD7FC-F8FFFA2EFA2FFA6EFA6FFADA-FAFFFB07-FB12FB18-FB1CFB37FB3DFB3FFB42FB45FBB2-FBD2FD40-FD4FFD90FD91FDC8-FDEFFDFEFDFFFE1A-FE1FFE27-FE2FFE53FE67FE6C-FE6FFE75FEFD-FF00FFBF-FFC1FFC8FFC9FFD0FFD1FFD8FFD9FFDD-FFDFFFE7FFEF-FFFBFFFEFFFF",
	    Cc: "0000-001F007F-009F",
	    Cf: "00AD0600-060306DD070F17B417B5200B-200F202A-202E2060-2064206A-206FFEFFFFF9-FFFB",
	    Co: "E000-F8FF",
	    Cs: "D800-DFFF",
	    Cn: "03780379037F-0383038B038D03A20526-05300557055805600588058B-059005C8-05CF05EB-05EF05F5-05FF06040605061C061D0620065F070E074B074C07B2-07BF07FB-07FF082E082F083F-08FF093A093B094F095609570973-097809800984098D098E0991099209A909B109B3-09B509BA09BB09C509C609C909CA09CF-09D609D8-09DB09DE09E409E509FC-0A000A040A0B-0A0E0A110A120A290A310A340A370A3A0A3B0A3D0A43-0A460A490A4A0A4E-0A500A52-0A580A5D0A5F-0A650A76-0A800A840A8E0A920AA90AB10AB40ABA0ABB0AC60ACA0ACE0ACF0AD1-0ADF0AE40AE50AF00AF2-0B000B040B0D0B0E0B110B120B290B310B340B3A0B3B0B450B460B490B4A0B4E-0B550B58-0B5B0B5E0B640B650B72-0B810B840B8B-0B8D0B910B96-0B980B9B0B9D0BA0-0BA20BA5-0BA70BAB-0BAD0BBA-0BBD0BC3-0BC50BC90BCE0BCF0BD1-0BD60BD8-0BE50BFB-0C000C040C0D0C110C290C340C3A-0C3C0C450C490C4E-0C540C570C5A-0C5F0C640C650C70-0C770C800C810C840C8D0C910CA90CB40CBA0CBB0CC50CC90CCE-0CD40CD7-0CDD0CDF0CE40CE50CF00CF3-0D010D040D0D0D110D290D3A-0D3C0D450D490D4E-0D560D58-0D5F0D640D650D76-0D780D800D810D840D97-0D990DB20DBC0DBE0DBF0DC7-0DC90DCB-0DCE0DD50DD70DE0-0DF10DF5-0E000E3B-0E3E0E5C-0E800E830E850E860E890E8B0E8C0E8E-0E930E980EA00EA40EA60EA80EA90EAC0EBA0EBE0EBF0EC50EC70ECE0ECF0EDA0EDB0EDE-0EFF0F480F6D-0F700F8C-0F8F0F980FBD0FCD0FD9-0FFF10C6-10CF10FD-10FF1249124E124F12571259125E125F1289128E128F12B112B612B712BF12C112C612C712D7131113161317135B-135E137D-137F139A-139F13F5-13FF169D-169F16F1-16FF170D1715-171F1737-173F1754-175F176D17711774-177F17DE17DF17EA-17EF17FA-17FF180F181A-181F1878-187F18AB-18AF18F6-18FF191D-191F192C-192F193C-193F1941-1943196E196F1975-197F19AC-19AF19CA-19CF19DB-19DD1A1C1A1D1A5F1A7D1A7E1A8A-1A8F1A9A-1A9F1AAE-1AFF1B4C-1B4F1B7D-1B7F1BAB-1BAD1BBA-1BFF1C38-1C3A1C4A-1C4C1C80-1CCF1CF3-1CFF1DE7-1DFC1F161F171F1E1F1F1F461F471F4E1F4F1F581F5A1F5C1F5E1F7E1F7F1FB51FC51FD41FD51FDC1FF01FF11FF51FFF2065-206920722073208F2095-209F20B9-20CF20F1-20FF218A-218F23E9-23FF2427-243F244B-245F26CE26E226E4-26E727002705270A270B2728274C274E2753-2755275F27602795-279727B027BF27CB27CD-27CF2B4D-2B4F2B5A-2BFF2C2F2C5F2CF2-2CF82D26-2D2F2D66-2D6E2D70-2D7F2D97-2D9F2DA72DAF2DB72DBF2DC72DCF2DD72DDF2E32-2E7F2E9A2EF4-2EFF2FD6-2FEF2FFC-2FFF3040309730983100-3104312E-3130318F31B8-31BF31E4-31EF321F32FF4DB6-4DBF9FCC-9FFFA48D-A48FA4C7-A4CFA62C-A63FA660A661A674-A67BA698-A69FA6F8-A6FFA78D-A7FAA82C-A82FA83A-A83FA878-A87FA8C5-A8CDA8DA-A8DFA8FC-A8FFA954-A95EA97D-A97FA9CEA9DA-A9DDA9E0-A9FFAA37-AA3FAA4EAA4FAA5AAA5BAA7C-AA7FAAC3-AADAAAE0-ABBFABEEABEFABFA-ABFFD7A4-D7AFD7C7-D7CAD7FC-D7FFFA2EFA2FFA6EFA6FFADA-FAFFFB07-FB12FB18-FB1CFB37FB3DFB3FFB42FB45FBB2-FBD2FD40-FD4FFD90FD91FDC8-FDEFFDFEFDFFFE1A-FE1FFE27-FE2FFE53FE67FE6C-FE6FFE75FEFDFEFEFF00FFBF-FFC1FFC8FFC9FFD0FFD1FFD8FFD9FFDD-FFDFFFE7FFEF-FFF8FFFEFFFF"
	});

	function addUnicodePackage (pack) {
	    var codePoint = /\w{4}/g;
	    for (var name in pack)
	        exports.packages[name] = pack[name].replace(codePoint, "\\u$&");
	}

	});

	ace.define("ace/mode/text",["require","exports","module","ace/tokenizer","ace/mode/text_highlight_rules","ace/mode/behaviour/cstyle","ace/unicode","ace/lib/lang","ace/token_iterator","ace/range"], function(acequire, exports, module) {
	"use strict";

	var Tokenizer = acequire("../tokenizer").Tokenizer;
	var TextHighlightRules = acequire("./text_highlight_rules").TextHighlightRules;
	var CstyleBehaviour = acequire("./behaviour/cstyle").CstyleBehaviour;
	var unicode = acequire("../unicode");
	var lang = acequire("../lib/lang");
	var TokenIterator = acequire("../token_iterator").TokenIterator;
	var Range = acequire("../range").Range;

	var Mode = function() {
	    this.HighlightRules = TextHighlightRules;
	};

	(function() {
	    this.$defaultBehaviour = new CstyleBehaviour();

	    this.tokenRe = new RegExp("^["
	        + unicode.packages.L
	        + unicode.packages.Mn + unicode.packages.Mc
	        + unicode.packages.Nd
	        + unicode.packages.Pc + "\\$_]+", "g"
	    );

	    this.nonTokenRe = new RegExp("^(?:[^"
	        + unicode.packages.L
	        + unicode.packages.Mn + unicode.packages.Mc
	        + unicode.packages.Nd
	        + unicode.packages.Pc + "\\$_]|\\s])+", "g"
	    );

	    this.getTokenizer = function() {
	        if (!this.$tokenizer) {
	            this.$highlightRules = this.$highlightRules || new this.HighlightRules(this.$highlightRuleConfig);
	            this.$tokenizer = new Tokenizer(this.$highlightRules.getRules());
	        }
	        return this.$tokenizer;
	    };

	    this.lineCommentStart = "";
	    this.blockComment = "";

	    this.toggleCommentLines = function(state, session, startRow, endRow) {
	        var doc = session.doc;

	        var ignoreBlankLines = true;
	        var shouldRemove = true;
	        var minIndent = Infinity;
	        var tabSize = session.getTabSize();
	        var insertAtTabStop = false;

	        if (!this.lineCommentStart) {
	            if (!this.blockComment)
	                return false;
	            var lineCommentStart = this.blockComment.start;
	            var lineCommentEnd = this.blockComment.end;
	            var regexpStart = new RegExp("^(\\s*)(?:" + lang.escapeRegExp(lineCommentStart) + ")");
	            var regexpEnd = new RegExp("(?:" + lang.escapeRegExp(lineCommentEnd) + ")\\s*$");

	            var comment = function(line, i) {
	                if (testRemove(line, i))
	                    return;
	                if (!ignoreBlankLines || /\S/.test(line)) {
	                    doc.insertInLine({row: i, column: line.length}, lineCommentEnd);
	                    doc.insertInLine({row: i, column: minIndent}, lineCommentStart);
	                }
	            };

	            var uncomment = function(line, i) {
	                var m;
	                if (m = line.match(regexpEnd))
	                    doc.removeInLine(i, line.length - m[0].length, line.length);
	                if (m = line.match(regexpStart))
	                    doc.removeInLine(i, m[1].length, m[0].length);
	            };

	            var testRemove = function(line, row) {
	                if (regexpStart.test(line))
	                    return true;
	                var tokens = session.getTokens(row);
	                for (var i = 0; i < tokens.length; i++) {
	                    if (tokens[i].type === "comment")
	                        return true;
	                }
	            };
	        } else {
	            if (Array.isArray(this.lineCommentStart)) {
	                var regexpStart = this.lineCommentStart.map(lang.escapeRegExp).join("|");
	                var lineCommentStart = this.lineCommentStart[0];
	            } else {
	                var regexpStart = lang.escapeRegExp(this.lineCommentStart);
	                var lineCommentStart = this.lineCommentStart;
	            }
	            regexpStart = new RegExp("^(\\s*)(?:" + regexpStart + ") ?");
	            
	            insertAtTabStop = session.getUseSoftTabs();

	            var uncomment = function(line, i) {
	                var m = line.match(regexpStart);
	                if (!m) return;
	                var start = m[1].length, end = m[0].length;
	                if (!shouldInsertSpace(line, start, end) && m[0][end - 1] == " ")
	                    end--;
	                doc.removeInLine(i, start, end);
	            };
	            var commentWithSpace = lineCommentStart + " ";
	            var comment = function(line, i) {
	                if (!ignoreBlankLines || /\S/.test(line)) {
	                    if (shouldInsertSpace(line, minIndent, minIndent))
	                        doc.insertInLine({row: i, column: minIndent}, commentWithSpace);
	                    else
	                        doc.insertInLine({row: i, column: minIndent}, lineCommentStart);
	                }
	            };
	            var testRemove = function(line, i) {
	                return regexpStart.test(line);
	            };
	            
	            var shouldInsertSpace = function(line, before, after) {
	                var spaces = 0;
	                while (before-- && line.charAt(before) == " ")
	                    spaces++;
	                if (spaces % tabSize != 0)
	                    return false;
	                var spaces = 0;
	                while (line.charAt(after++) == " ")
	                    spaces++;
	                if (tabSize > 2)
	                    return spaces % tabSize != tabSize - 1;
	                else
	                    return spaces % tabSize == 0;
	                return true;
	            };
	        }

	        function iter(fun) {
	            for (var i = startRow; i <= endRow; i++)
	                fun(doc.getLine(i), i);
	        }


	        var minEmptyLength = Infinity;
	        iter(function(line, i) {
	            var indent = line.search(/\S/);
	            if (indent !== -1) {
	                if (indent < minIndent)
	                    minIndent = indent;
	                if (shouldRemove && !testRemove(line, i))
	                    shouldRemove = false;
	            } else if (minEmptyLength > line.length) {
	                minEmptyLength = line.length;
	            }
	        });

	        if (minIndent == Infinity) {
	            minIndent = minEmptyLength;
	            ignoreBlankLines = false;
	            shouldRemove = false;
	        }

	        if (insertAtTabStop && minIndent % tabSize != 0)
	            minIndent = Math.floor(minIndent / tabSize) * tabSize;

	        iter(shouldRemove ? uncomment : comment);
	    };

	    this.toggleBlockComment = function(state, session, range, cursor) {
	        var comment = this.blockComment;
	        if (!comment)
	            return;
	        if (!comment.start && comment[0])
	            comment = comment[0];

	        var iterator = new TokenIterator(session, cursor.row, cursor.column);
	        var token = iterator.getCurrentToken();

	        var sel = session.selection;
	        var initialRange = session.selection.toOrientedRange();
	        var startRow, colDiff;

	        if (token && /comment/.test(token.type)) {
	            var startRange, endRange;
	            while (token && /comment/.test(token.type)) {
	                var i = token.value.indexOf(comment.start);
	                if (i != -1) {
	                    var row = iterator.getCurrentTokenRow();
	                    var column = iterator.getCurrentTokenColumn() + i;
	                    startRange = new Range(row, column, row, column + comment.start.length);
	                    break;
	                }
	                token = iterator.stepBackward();
	            }

	            var iterator = new TokenIterator(session, cursor.row, cursor.column);
	            var token = iterator.getCurrentToken();
	            while (token && /comment/.test(token.type)) {
	                var i = token.value.indexOf(comment.end);
	                if (i != -1) {
	                    var row = iterator.getCurrentTokenRow();
	                    var column = iterator.getCurrentTokenColumn() + i;
	                    endRange = new Range(row, column, row, column + comment.end.length);
	                    break;
	                }
	                token = iterator.stepForward();
	            }
	            if (endRange)
	                session.remove(endRange);
	            if (startRange) {
	                session.remove(startRange);
	                startRow = startRange.start.row;
	                colDiff = -comment.start.length;
	            }
	        } else {
	            colDiff = comment.start.length;
	            startRow = range.start.row;
	            session.insert(range.end, comment.end);
	            session.insert(range.start, comment.start);
	        }
	        if (initialRange.start.row == startRow)
	            initialRange.start.column += colDiff;
	        if (initialRange.end.row == startRow)
	            initialRange.end.column += colDiff;
	        session.selection.fromOrientedRange(initialRange);
	    };

	    this.getNextLineIndent = function(state, line, tab) {
	        return this.$getIndent(line);
	    };

	    this.checkOutdent = function(state, line, input) {
	        return false;
	    };

	    this.autoOutdent = function(state, doc, row) {
	    };

	    this.$getIndent = function(line) {
	        return line.match(/^\s*/)[0];
	    };

	    this.createWorker = function(session) {
	        return null;
	    };

	    this.createModeDelegates = function (mapping) {
	        this.$embeds = [];
	        this.$modes = {};
	        for (var i in mapping) {
	            if (mapping[i]) {
	                this.$embeds.push(i);
	                this.$modes[i] = new mapping[i]();
	            }
	        }

	        var delegations = ["toggleBlockComment", "toggleCommentLines", "getNextLineIndent", 
	            "checkOutdent", "autoOutdent", "transformAction", "getCompletions"];

	        for (var i = 0; i < delegations.length; i++) {
	            (function(scope) {
	              var functionName = delegations[i];
	              var defaultHandler = scope[functionName];
	              scope[delegations[i]] = function() {
	                  return this.$delegator(functionName, arguments, defaultHandler);
	              };
	            }(this));
	        }
	    };

	    this.$delegator = function(method, args, defaultHandler) {
	        var state = args[0];
	        if (typeof state != "string")
	            state = state[0];
	        for (var i = 0; i < this.$embeds.length; i++) {
	            if (!this.$modes[this.$embeds[i]]) continue;

	            var split = state.split(this.$embeds[i]);
	            if (!split[0] && split[1]) {
	                args[0] = split[1];
	                var mode = this.$modes[this.$embeds[i]];
	                return mode[method].apply(mode, args);
	            }
	        }
	        var ret = defaultHandler.apply(this, args);
	        return defaultHandler ? ret : undefined;
	    };

	    this.transformAction = function(state, action, editor, session, param) {
	        if (this.$behaviour) {
	            var behaviours = this.$behaviour.getBehaviours();
	            for (var key in behaviours) {
	                if (behaviours[key][action]) {
	                    var ret = behaviours[key][action].apply(this, arguments);
	                    if (ret) {
	                        return ret;
	                    }
	                }
	            }
	        }
	    };
	    
	    this.getKeywords = function(append) {
	        if (!this.completionKeywords) {
	            var rules = this.$tokenizer.rules;
	            var completionKeywords = [];
	            for (var rule in rules) {
	                var ruleItr = rules[rule];
	                for (var r = 0, l = ruleItr.length; r < l; r++) {
	                    if (typeof ruleItr[r].token === "string") {
	                        if (/keyword|support|storage/.test(ruleItr[r].token))
	                            completionKeywords.push(ruleItr[r].regex);
	                    }
	                    else if (typeof ruleItr[r].token === "object") {
	                        for (var a = 0, aLength = ruleItr[r].token.length; a < aLength; a++) {    
	                            if (/keyword|support|storage/.test(ruleItr[r].token[a])) {
	                                var rule = ruleItr[r].regex.match(/\(.+?\)/g)[a];
	                                completionKeywords.push(rule.substr(1, rule.length - 2));
	                            }
	                        }
	                    }
	                }
	            }
	            this.completionKeywords = completionKeywords;
	        }
	        if (!append)
	            return this.$keywordList;
	        return completionKeywords.concat(this.$keywordList || []);
	    };
	    
	    this.$createKeywordList = function() {
	        if (!this.$highlightRules)
	            this.getTokenizer();
	        return this.$keywordList = this.$highlightRules.$keywordList || [];
	    };

	    this.getCompletions = function(state, session, pos, prefix) {
	        var keywords = this.$keywordList || this.$createKeywordList();
	        return keywords.map(function(word) {
	            return {
	                name: word,
	                value: word,
	                score: 0,
	                meta: "keyword"
	            };
	        });
	    };

	    this.$id = "ace/mode/text";
	}).call(Mode.prototype);

	exports.Mode = Mode;
	});

	ace.define("ace/apply_delta",["require","exports","module"], function(acequire, exports, module) {
	"use strict";

	function throwDeltaError(delta, errorText){
	    console.log("Invalid Delta:", delta);
	    throw "Invalid Delta: " + errorText;
	}

	function positionInDocument(docLines, position) {
	    return position.row    >= 0 && position.row    <  docLines.length &&
	           position.column >= 0 && position.column <= docLines[position.row].length;
	}

	function validateDelta(docLines, delta) {
	    if (delta.action != "insert" && delta.action != "remove")
	        throwDeltaError(delta, "delta.action must be 'insert' or 'remove'");
	    if (!(delta.lines instanceof Array))
	        throwDeltaError(delta, "delta.lines must be an Array");
	    if (!delta.start || !delta.end)
	       throwDeltaError(delta, "delta.start/end must be an present");
	    var start = delta.start;
	    if (!positionInDocument(docLines, delta.start))
	        throwDeltaError(delta, "delta.start must be contained in document");
	    var end = delta.end;
	    if (delta.action == "remove" && !positionInDocument(docLines, end))
	        throwDeltaError(delta, "delta.end must contained in document for 'remove' actions");
	    var numRangeRows = end.row - start.row;
	    var numRangeLastLineChars = (end.column - (numRangeRows == 0 ? start.column : 0));
	    if (numRangeRows != delta.lines.length - 1 || delta.lines[numRangeRows].length != numRangeLastLineChars)
	        throwDeltaError(delta, "delta.range must match delta lines");
	}

	exports.applyDelta = function(docLines, delta, doNotValidate) {
	    
	    var row = delta.start.row;
	    var startColumn = delta.start.column;
	    var line = docLines[row] || "";
	    switch (delta.action) {
	        case "insert":
	            var lines = delta.lines;
	            if (lines.length === 1) {
	                docLines[row] = line.substring(0, startColumn) + delta.lines[0] + line.substring(startColumn);
	            } else {
	                var args = [row, 1].concat(delta.lines);
	                docLines.splice.apply(docLines, args);
	                docLines[row] = line.substring(0, startColumn) + docLines[row];
	                docLines[row + delta.lines.length - 1] += line.substring(startColumn);
	            }
	            break;
	        case "remove":
	            var endColumn = delta.end.column;
	            var endRow = delta.end.row;
	            if (row === endRow) {
	                docLines[row] = line.substring(0, startColumn) + line.substring(endColumn);
	            } else {
	                docLines.splice(
	                    row, endRow - row + 1,
	                    line.substring(0, startColumn) + docLines[endRow].substring(endColumn)
	                );
	            }
	            break;
	    }
	};
	});

	ace.define("ace/anchor",["require","exports","module","ace/lib/oop","ace/lib/event_emitter"], function(acequire, exports, module) {
	"use strict";

	var oop = acequire("./lib/oop");
	var EventEmitter = acequire("./lib/event_emitter").EventEmitter;

	var Anchor = exports.Anchor = function(doc, row, column) {
	    this.$onChange = this.onChange.bind(this);
	    this.attach(doc);
	    
	    if (typeof column == "undefined")
	        this.setPosition(row.row, row.column);
	    else
	        this.setPosition(row, column);
	};

	(function() {

	    oop.implement(this, EventEmitter);
	    this.getPosition = function() {
	        return this.$clipPositionToDocument(this.row, this.column);
	    };
	    this.getDocument = function() {
	        return this.document;
	    };
	    this.$insertRight = false;
	    this.onChange = function(delta) {
	        if (delta.start.row == delta.end.row && delta.start.row != this.row)
	            return;

	        if (delta.start.row > this.row)
	            return;
	            
	        var point = $getTransformedPoint(delta, {row: this.row, column: this.column}, this.$insertRight);
	        this.setPosition(point.row, point.column, true);
	    };
	    
	    function $pointsInOrder(point1, point2, equalPointsInOrder) {
	        var bColIsAfter = equalPointsInOrder ? point1.column <= point2.column : point1.column < point2.column;
	        return (point1.row < point2.row) || (point1.row == point2.row && bColIsAfter);
	    }
	            
	    function $getTransformedPoint(delta, point, moveIfEqual) {
	        var deltaIsInsert = delta.action == "insert";
	        var deltaRowShift = (deltaIsInsert ? 1 : -1) * (delta.end.row    - delta.start.row);
	        var deltaColShift = (deltaIsInsert ? 1 : -1) * (delta.end.column - delta.start.column);
	        var deltaStart = delta.start;
	        var deltaEnd = deltaIsInsert ? deltaStart : delta.end; // Collapse insert range.
	        if ($pointsInOrder(point, deltaStart, moveIfEqual)) {
	            return {
	                row: point.row,
	                column: point.column
	            };
	        }
	        if ($pointsInOrder(deltaEnd, point, !moveIfEqual)) {
	            return {
	                row: point.row + deltaRowShift,
	                column: point.column + (point.row == deltaEnd.row ? deltaColShift : 0)
	            };
	        }
	        
	        return {
	            row: deltaStart.row,
	            column: deltaStart.column
	        };
	    }
	    this.setPosition = function(row, column, noClip) {
	        var pos;
	        if (noClip) {
	            pos = {
	                row: row,
	                column: column
	            };
	        } else {
	            pos = this.$clipPositionToDocument(row, column);
	        }

	        if (this.row == pos.row && this.column == pos.column)
	            return;

	        var old = {
	            row: this.row,
	            column: this.column
	        };

	        this.row = pos.row;
	        this.column = pos.column;
	        this._signal("change", {
	            old: old,
	            value: pos
	        });
	    };
	    this.detach = function() {
	        this.document.removeEventListener("change", this.$onChange);
	    };
	    this.attach = function(doc) {
	        this.document = doc || this.document;
	        this.document.on("change", this.$onChange);
	    };
	    this.$clipPositionToDocument = function(row, column) {
	        var pos = {};

	        if (row >= this.document.getLength()) {
	            pos.row = Math.max(0, this.document.getLength() - 1);
	            pos.column = this.document.getLine(pos.row).length;
	        }
	        else if (row < 0) {
	            pos.row = 0;
	            pos.column = 0;
	        }
	        else {
	            pos.row = row;
	            pos.column = Math.min(this.document.getLine(pos.row).length, Math.max(0, column));
	        }

	        if (column < 0)
	            pos.column = 0;

	        return pos;
	    };

	}).call(Anchor.prototype);

	});

	ace.define("ace/document",["require","exports","module","ace/lib/oop","ace/apply_delta","ace/lib/event_emitter","ace/range","ace/anchor"], function(acequire, exports, module) {
	"use strict";

	var oop = acequire("./lib/oop");
	var applyDelta = acequire("./apply_delta").applyDelta;
	var EventEmitter = acequire("./lib/event_emitter").EventEmitter;
	var Range = acequire("./range").Range;
	var Anchor = acequire("./anchor").Anchor;

	var Document = function(textOrLines) {
	    this.$lines = [""];
	    if (textOrLines.length === 0) {
	        this.$lines = [""];
	    } else if (Array.isArray(textOrLines)) {
	        this.insertMergedLines({row: 0, column: 0}, textOrLines);
	    } else {
	        this.insert({row: 0, column:0}, textOrLines);
	    }
	};

	(function() {

	    oop.implement(this, EventEmitter);
	    this.setValue = function(text) {
	        var len = this.getLength() - 1;
	        this.remove(new Range(0, 0, len, this.getLine(len).length));
	        this.insert({row: 0, column: 0}, text);
	    };
	    this.getValue = function() {
	        return this.getAllLines().join(this.getNewLineCharacter());
	    };
	    this.createAnchor = function(row, column) {
	        return new Anchor(this, row, column);
	    };
	    if ("aaa".split(/a/).length === 0) {
	        this.$split = function(text) {
	            return text.replace(/\r\n|\r/g, "\n").split("\n");
	        };
	    } else {
	        this.$split = function(text) {
	            return text.split(/\r\n|\r|\n/);
	        };
	    }


	    this.$detectNewLine = function(text) {
	        var match = text.match(/^.*?(\r\n|\r|\n)/m);
	        this.$autoNewLine = match ? match[1] : "\n";
	        this._signal("changeNewLineMode");
	    };
	    this.getNewLineCharacter = function() {
	        switch (this.$newLineMode) {
	          case "windows":
	            return "\r\n";
	          case "unix":
	            return "\n";
	          default:
	            return this.$autoNewLine || "\n";
	        }
	    };

	    this.$autoNewLine = "";
	    this.$newLineMode = "auto";
	    this.setNewLineMode = function(newLineMode) {
	        if (this.$newLineMode === newLineMode)
	            return;

	        this.$newLineMode = newLineMode;
	        this._signal("changeNewLineMode");
	    };
	    this.getNewLineMode = function() {
	        return this.$newLineMode;
	    };
	    this.isNewLine = function(text) {
	        return (text == "\r\n" || text == "\r" || text == "\n");
	    };
	    this.getLine = function(row) {
	        return this.$lines[row] || "";
	    };
	    this.getLines = function(firstRow, lastRow) {
	        return this.$lines.slice(firstRow, lastRow + 1);
	    };
	    this.getAllLines = function() {
	        return this.getLines(0, this.getLength());
	    };
	    this.getLength = function() {
	        return this.$lines.length;
	    };
	    this.getTextRange = function(range) {
	        return this.getLinesForRange(range).join(this.getNewLineCharacter());
	    };
	    this.getLinesForRange = function(range) {
	        var lines;
	        if (range.start.row === range.end.row) {
	            lines = [this.getLine(range.start.row).substring(range.start.column, range.end.column)];
	        } else {
	            lines = this.getLines(range.start.row, range.end.row);
	            lines[0] = (lines[0] || "").substring(range.start.column);
	            var l = lines.length - 1;
	            if (range.end.row - range.start.row == l)
	                lines[l] = lines[l].substring(0, range.end.column);
	        }
	        return lines;
	    };
	    this.insertLines = function(row, lines) {
	        console.warn("Use of document.insertLines is deprecated. Use the insertFullLines method instead.");
	        return this.insertFullLines(row, lines);
	    };
	    this.removeLines = function(firstRow, lastRow) {
	        console.warn("Use of document.removeLines is deprecated. Use the removeFullLines method instead.");
	        return this.removeFullLines(firstRow, lastRow);
	    };
	    this.insertNewLine = function(position) {
	        console.warn("Use of document.insertNewLine is deprecated. Use insertMergedLines(position, ['', '']) instead.");
	        return this.insertMergedLines(position, ["", ""]);
	    };
	    this.insert = function(position, text) {
	        if (this.getLength() <= 1)
	            this.$detectNewLine(text);
	        
	        return this.insertMergedLines(position, this.$split(text));
	    };
	    this.insertInLine = function(position, text) {
	        var start = this.clippedPos(position.row, position.column);
	        var end = this.pos(position.row, position.column + text.length);
	        
	        this.applyDelta({
	            start: start,
	            end: end,
	            action: "insert",
	            lines: [text]
	        }, true);
	        
	        return this.clonePos(end);
	    };
	    
	    this.clippedPos = function(row, column) {
	        var length = this.getLength();
	        if (row === undefined) {
	            row = length;
	        } else if (row < 0) {
	            row = 0;
	        } else if (row >= length) {
	            row = length - 1;
	            column = undefined;
	        }
	        var line = this.getLine(row);
	        if (column == undefined)
	            column = line.length;
	        column = Math.min(Math.max(column, 0), line.length);
	        return {row: row, column: column};
	    };
	    
	    this.clonePos = function(pos) {
	        return {row: pos.row, column: pos.column};
	    };
	    
	    this.pos = function(row, column) {
	        return {row: row, column: column};
	    };
	    
	    this.$clipPosition = function(position) {
	        var length = this.getLength();
	        if (position.row >= length) {
	            position.row = Math.max(0, length - 1);
	            position.column = this.getLine(length - 1).length;
	        } else {
	            position.row = Math.max(0, position.row);
	            position.column = Math.min(Math.max(position.column, 0), this.getLine(position.row).length);
	        }
	        return position;
	    };
	    this.insertFullLines = function(row, lines) {
	        row = Math.min(Math.max(row, 0), this.getLength());
	        var column = 0;
	        if (row < this.getLength()) {
	            lines = lines.concat([""]);
	            column = 0;
	        } else {
	            lines = [""].concat(lines);
	            row--;
	            column = this.$lines[row].length;
	        }
	        this.insertMergedLines({row: row, column: column}, lines);
	    };    
	    this.insertMergedLines = function(position, lines) {
	        var start = this.clippedPos(position.row, position.column);
	        var end = {
	            row: start.row + lines.length - 1,
	            column: (lines.length == 1 ? start.column : 0) + lines[lines.length - 1].length
	        };
	        
	        this.applyDelta({
	            start: start,
	            end: end,
	            action: "insert",
	            lines: lines
	        });
	        
	        return this.clonePos(end);
	    };
	    this.remove = function(range) {
	        var start = this.clippedPos(range.start.row, range.start.column);
	        var end = this.clippedPos(range.end.row, range.end.column);
	        this.applyDelta({
	            start: start,
	            end: end,
	            action: "remove",
	            lines: this.getLinesForRange({start: start, end: end})
	        });
	        return this.clonePos(start);
	    };
	    this.removeInLine = function(row, startColumn, endColumn) {
	        var start = this.clippedPos(row, startColumn);
	        var end = this.clippedPos(row, endColumn);
	        
	        this.applyDelta({
	            start: start,
	            end: end,
	            action: "remove",
	            lines: this.getLinesForRange({start: start, end: end})
	        }, true);
	        
	        return this.clonePos(start);
	    };
	    this.removeFullLines = function(firstRow, lastRow) {
	        firstRow = Math.min(Math.max(0, firstRow), this.getLength() - 1);
	        lastRow  = Math.min(Math.max(0, lastRow ), this.getLength() - 1);
	        var deleteFirstNewLine = lastRow == this.getLength() - 1 && firstRow > 0;
	        var deleteLastNewLine  = lastRow  < this.getLength() - 1;
	        var startRow = ( deleteFirstNewLine ? firstRow - 1                  : firstRow                    );
	        var startCol = ( deleteFirstNewLine ? this.getLine(startRow).length : 0                           );
	        var endRow   = ( deleteLastNewLine  ? lastRow + 1                   : lastRow                     );
	        var endCol   = ( deleteLastNewLine  ? 0                             : this.getLine(endRow).length ); 
	        var range = new Range(startRow, startCol, endRow, endCol);
	        var deletedLines = this.$lines.slice(firstRow, lastRow + 1);
	        
	        this.applyDelta({
	            start: range.start,
	            end: range.end,
	            action: "remove",
	            lines: this.getLinesForRange(range)
	        });
	        return deletedLines;
	    };
	    this.removeNewLine = function(row) {
	        if (row < this.getLength() - 1 && row >= 0) {
	            this.applyDelta({
	                start: this.pos(row, this.getLine(row).length),
	                end: this.pos(row + 1, 0),
	                action: "remove",
	                lines: ["", ""]
	            });
	        }
	    };
	    this.replace = function(range, text) {
	        if (!(range instanceof Range))
	            range = Range.fromPoints(range.start, range.end);
	        if (text.length === 0 && range.isEmpty())
	            return range.start;
	        if (text == this.getTextRange(range))
	            return range.end;

	        this.remove(range);
	        var end;
	        if (text) {
	            end = this.insert(range.start, text);
	        }
	        else {
	            end = range.start;
	        }
	        
	        return end;
	    };
	    this.applyDeltas = function(deltas) {
	        for (var i=0; i<deltas.length; i++) {
	            this.applyDelta(deltas[i]);
	        }
	    };
	    this.revertDeltas = function(deltas) {
	        for (var i=deltas.length-1; i>=0; i--) {
	            this.revertDelta(deltas[i]);
	        }
	    };
	    this.applyDelta = function(delta, doNotValidate) {
	        var isInsert = delta.action == "insert";
	        if (isInsert ? delta.lines.length <= 1 && !delta.lines[0]
	            : !Range.comparePoints(delta.start, delta.end)) {
	            return;
	        }
	        
	        if (isInsert && delta.lines.length > 20000)
	            this.$splitAndapplyLargeDelta(delta, 20000);
	        applyDelta(this.$lines, delta, doNotValidate);
	        this._signal("change", delta);
	    };
	    
	    this.$splitAndapplyLargeDelta = function(delta, MAX) {
	        var lines = delta.lines;
	        var l = lines.length;
	        var row = delta.start.row; 
	        var column = delta.start.column;
	        var from = 0, to = 0;
	        do {
	            from = to;
	            to += MAX - 1;
	            var chunk = lines.slice(from, to);
	            if (to > l) {
	                delta.lines = chunk;
	                delta.start.row = row + from;
	                delta.start.column = column;
	                break;
	            }
	            chunk.push("");
	            this.applyDelta({
	                start: this.pos(row + from, column),
	                end: this.pos(row + to, column = 0),
	                action: delta.action,
	                lines: chunk
	            }, true);
	        } while(true);
	    };
	    this.revertDelta = function(delta) {
	        this.applyDelta({
	            start: this.clonePos(delta.start),
	            end: this.clonePos(delta.end),
	            action: (delta.action == "insert" ? "remove" : "insert"),
	            lines: delta.lines.slice()
	        });
	    };
	    this.indexToPosition = function(index, startRow) {
	        var lines = this.$lines || this.getAllLines();
	        var newlineLength = this.getNewLineCharacter().length;
	        for (var i = startRow || 0, l = lines.length; i < l; i++) {
	            index -= lines[i].length + newlineLength;
	            if (index < 0)
	                return {row: i, column: index + lines[i].length + newlineLength};
	        }
	        return {row: l-1, column: lines[l-1].length};
	    };
	    this.positionToIndex = function(pos, startRow) {
	        var lines = this.$lines || this.getAllLines();
	        var newlineLength = this.getNewLineCharacter().length;
	        var index = 0;
	        var row = Math.min(pos.row, lines.length);
	        for (var i = startRow || 0; i < row; ++i)
	            index += lines[i].length + newlineLength;

	        return index + pos.column;
	    };

	}).call(Document.prototype);

	exports.Document = Document;
	});

	ace.define("ace/background_tokenizer",["require","exports","module","ace/lib/oop","ace/lib/event_emitter"], function(acequire, exports, module) {
	"use strict";

	var oop = acequire("./lib/oop");
	var EventEmitter = acequire("./lib/event_emitter").EventEmitter;

	var BackgroundTokenizer = function(tokenizer, editor) {
	    this.running = false;
	    this.lines = [];
	    this.states = [];
	    this.currentLine = 0;
	    this.tokenizer = tokenizer;

	    var self = this;

	    this.$worker = function() {
	        if (!self.running) { return; }

	        var workerStart = new Date();
	        var currentLine = self.currentLine;
	        var endLine = -1;
	        var doc = self.doc;

	        var startLine = currentLine;
	        while (self.lines[currentLine])
	            currentLine++;
	        
	        var len = doc.getLength();
	        var processedLines = 0;
	        self.running = false;
	        while (currentLine < len) {
	            self.$tokenizeRow(currentLine);
	            endLine = currentLine;
	            do {
	                currentLine++;
	            } while (self.lines[currentLine]);
	            processedLines ++;
	            if ((processedLines % 5 === 0) && (new Date() - workerStart) > 20) {                
	                self.running = setTimeout(self.$worker, 20);
	                break;
	            }
	        }
	        self.currentLine = currentLine;
	        
	        if (endLine == -1)
	            endLine = currentLine;

	        if (startLine <= endLine)
	            self.fireUpdateEvent(startLine, endLine);
	    };
	};

	(function(){

	    oop.implement(this, EventEmitter);
	    this.setTokenizer = function(tokenizer) {
	        this.tokenizer = tokenizer;
	        this.lines = [];
	        this.states = [];

	        this.start(0);
	    };
	    this.setDocument = function(doc) {
	        this.doc = doc;
	        this.lines = [];
	        this.states = [];

	        this.stop();
	    };
	    this.fireUpdateEvent = function(firstRow, lastRow) {
	        var data = {
	            first: firstRow,
	            last: lastRow
	        };
	        this._signal("update", {data: data});
	    };
	    this.start = function(startRow) {
	        this.currentLine = Math.min(startRow || 0, this.currentLine, this.doc.getLength());
	        this.lines.splice(this.currentLine, this.lines.length);
	        this.states.splice(this.currentLine, this.states.length);

	        this.stop();
	        this.running = setTimeout(this.$worker, 700);
	    };
	    
	    this.scheduleStart = function() {
	        if (!this.running)
	            this.running = setTimeout(this.$worker, 700);
	    };

	    this.$updateOnChange = function(delta) {
	        var startRow = delta.start.row;
	        var len = delta.end.row - startRow;

	        if (len === 0) {
	            this.lines[startRow] = null;
	        } else if (delta.action == "remove") {
	            this.lines.splice(startRow, len + 1, null);
	            this.states.splice(startRow, len + 1, null);
	        } else {
	            var args = Array(len + 1);
	            args.unshift(startRow, 1);
	            this.lines.splice.apply(this.lines, args);
	            this.states.splice.apply(this.states, args);
	        }

	        this.currentLine = Math.min(startRow, this.currentLine, this.doc.getLength());

	        this.stop();
	    };
	    this.stop = function() {
	        if (this.running)
	            clearTimeout(this.running);
	        this.running = false;
	    };
	    this.getTokens = function(row) {
	        return this.lines[row] || this.$tokenizeRow(row);
	    };
	    this.getState = function(row) {
	        if (this.currentLine == row)
	            this.$tokenizeRow(row);
	        return this.states[row] || "start";
	    };

	    this.$tokenizeRow = function(row) {
	        var line = this.doc.getLine(row);
	        var state = this.states[row - 1];

	        var data = this.tokenizer.getLineTokens(line, state, row);

	        if (this.states[row] + "" !== data.state + "") {
	            this.states[row] = data.state;
	            this.lines[row + 1] = null;
	            if (this.currentLine > row + 1)
	                this.currentLine = row + 1;
	        } else if (this.currentLine == row) {
	            this.currentLine = row + 1;
	        }

	        return this.lines[row] = data.tokens;
	    };

	}).call(BackgroundTokenizer.prototype);

	exports.BackgroundTokenizer = BackgroundTokenizer;
	});

	ace.define("ace/search_highlight",["require","exports","module","ace/lib/lang","ace/lib/oop","ace/range"], function(acequire, exports, module) {
	"use strict";

	var lang = acequire("./lib/lang");
	var oop = acequire("./lib/oop");
	var Range = acequire("./range").Range;

	var SearchHighlight = function(regExp, clazz, type) {
	    this.setRegexp(regExp);
	    this.clazz = clazz;
	    this.type = type || "text";
	};

	(function() {
	    this.MAX_RANGES = 500;
	    
	    this.setRegexp = function(regExp) {
	        if (this.regExp+"" == regExp+"")
	            return;
	        this.regExp = regExp;
	        this.cache = [];
	    };

	    this.update = function(html, markerLayer, session, config) {
	        if (!this.regExp)
	            return;
	        var start = config.firstRow, end = config.lastRow;

	        for (var i = start; i <= end; i++) {
	            var ranges = this.cache[i];
	            if (ranges == null) {
	                ranges = lang.getMatchOffsets(session.getLine(i), this.regExp);
	                if (ranges.length > this.MAX_RANGES)
	                    ranges = ranges.slice(0, this.MAX_RANGES);
	                ranges = ranges.map(function(match) {
	                    return new Range(i, match.offset, i, match.offset + match.length);
	                });
	                this.cache[i] = ranges.length ? ranges : "";
	            }

	            for (var j = ranges.length; j --; ) {
	                markerLayer.drawSingleLineMarker(
	                    html, ranges[j].toScreenRange(session), this.clazz, config);
	            }
	        }
	    };

	}).call(SearchHighlight.prototype);

	exports.SearchHighlight = SearchHighlight;
	});

	ace.define("ace/edit_session/fold_line",["require","exports","module","ace/range"], function(acequire, exports, module) {
	"use strict";

	var Range = acequire("../range").Range;
	function FoldLine(foldData, folds) {
	    this.foldData = foldData;
	    if (Array.isArray(folds)) {
	        this.folds = folds;
	    } else {
	        folds = this.folds = [ folds ];
	    }

	    var last = folds[folds.length - 1];
	    this.range = new Range(folds[0].start.row, folds[0].start.column,
	                           last.end.row, last.end.column);
	    this.start = this.range.start;
	    this.end   = this.range.end;

	    this.folds.forEach(function(fold) {
	        fold.setFoldLine(this);
	    }, this);
	}

	(function() {
	    this.shiftRow = function(shift) {
	        this.start.row += shift;
	        this.end.row += shift;
	        this.folds.forEach(function(fold) {
	            fold.start.row += shift;
	            fold.end.row += shift;
	        });
	    };

	    this.addFold = function(fold) {
	        if (fold.sameRow) {
	            if (fold.start.row < this.startRow || fold.endRow > this.endRow) {
	                throw new Error("Can't add a fold to this FoldLine as it has no connection");
	            }
	            this.folds.push(fold);
	            this.folds.sort(function(a, b) {
	                return -a.range.compareEnd(b.start.row, b.start.column);
	            });
	            if (this.range.compareEnd(fold.start.row, fold.start.column) > 0) {
	                this.end.row = fold.end.row;
	                this.end.column =  fold.end.column;
	            } else if (this.range.compareStart(fold.end.row, fold.end.column) < 0) {
	                this.start.row = fold.start.row;
	                this.start.column = fold.start.column;
	            }
	        } else if (fold.start.row == this.end.row) {
	            this.folds.push(fold);
	            this.end.row = fold.end.row;
	            this.end.column = fold.end.column;
	        } else if (fold.end.row == this.start.row) {
	            this.folds.unshift(fold);
	            this.start.row = fold.start.row;
	            this.start.column = fold.start.column;
	        } else {
	            throw new Error("Trying to add fold to FoldRow that doesn't have a matching row");
	        }
	        fold.foldLine = this;
	    };

	    this.containsRow = function(row) {
	        return row >= this.start.row && row <= this.end.row;
	    };

	    this.walk = function(callback, endRow, endColumn) {
	        var lastEnd = 0,
	            folds = this.folds,
	            fold,
	            cmp, stop, isNewRow = true;

	        if (endRow == null) {
	            endRow = this.end.row;
	            endColumn = this.end.column;
	        }

	        for (var i = 0; i < folds.length; i++) {
	            fold = folds[i];

	            cmp = fold.range.compareStart(endRow, endColumn);
	            if (cmp == -1) {
	                callback(null, endRow, endColumn, lastEnd, isNewRow);
	                return;
	            }

	            stop = callback(null, fold.start.row, fold.start.column, lastEnd, isNewRow);
	            stop = !stop && callback(fold.placeholder, fold.start.row, fold.start.column, lastEnd);
	            if (stop || cmp === 0) {
	                return;
	            }
	            isNewRow = !fold.sameRow;
	            lastEnd = fold.end.column;
	        }
	        callback(null, endRow, endColumn, lastEnd, isNewRow);
	    };

	    this.getNextFoldTo = function(row, column) {
	        var fold, cmp;
	        for (var i = 0; i < this.folds.length; i++) {
	            fold = this.folds[i];
	            cmp = fold.range.compareEnd(row, column);
	            if (cmp == -1) {
	                return {
	                    fold: fold,
	                    kind: "after"
	                };
	            } else if (cmp === 0) {
	                return {
	                    fold: fold,
	                    kind: "inside"
	                };
	            }
	        }
	        return null;
	    };

	    this.addRemoveChars = function(row, column, len) {
	        var ret = this.getNextFoldTo(row, column),
	            fold, folds;
	        if (ret) {
	            fold = ret.fold;
	            if (ret.kind == "inside"
	                && fold.start.column != column
	                && fold.start.row != row)
	            {
	                window.console && window.console.log(row, column, fold);
	            } else if (fold.start.row == row) {
	                folds = this.folds;
	                var i = folds.indexOf(fold);
	                if (i === 0) {
	                    this.start.column += len;
	                }
	                for (i; i < folds.length; i++) {
	                    fold = folds[i];
	                    fold.start.column += len;
	                    if (!fold.sameRow) {
	                        return;
	                    }
	                    fold.end.column += len;
	                }
	                this.end.column += len;
	            }
	        }
	    };

	    this.split = function(row, column) {
	        var pos = this.getNextFoldTo(row, column);
	        
	        if (!pos || pos.kind == "inside")
	            return null;
	            
	        var fold = pos.fold;
	        var folds = this.folds;
	        var foldData = this.foldData;
	        
	        var i = folds.indexOf(fold);
	        var foldBefore = folds[i - 1];
	        this.end.row = foldBefore.end.row;
	        this.end.column = foldBefore.end.column;
	        folds = folds.splice(i, folds.length - i);

	        var newFoldLine = new FoldLine(foldData, folds);
	        foldData.splice(foldData.indexOf(this) + 1, 0, newFoldLine);
	        return newFoldLine;
	    };

	    this.merge = function(foldLineNext) {
	        var folds = foldLineNext.folds;
	        for (var i = 0; i < folds.length; i++) {
	            this.addFold(folds[i]);
	        }
	        var foldData = this.foldData;
	        foldData.splice(foldData.indexOf(foldLineNext), 1);
	    };

	    this.toString = function() {
	        var ret = [this.range.toString() + ": [" ];

	        this.folds.forEach(function(fold) {
	            ret.push("  " + fold.toString());
	        });
	        ret.push("]");
	        return ret.join("\n");
	    };

	    this.idxToPosition = function(idx) {
	        var lastFoldEndColumn = 0;

	        for (var i = 0; i < this.folds.length; i++) {
	            var fold = this.folds[i];

	            idx -= fold.start.column - lastFoldEndColumn;
	            if (idx < 0) {
	                return {
	                    row: fold.start.row,
	                    column: fold.start.column + idx
	                };
	            }

	            idx -= fold.placeholder.length;
	            if (idx < 0) {
	                return fold.start;
	            }

	            lastFoldEndColumn = fold.end.column;
	        }

	        return {
	            row: this.end.row,
	            column: this.end.column + idx
	        };
	    };
	}).call(FoldLine.prototype);

	exports.FoldLine = FoldLine;
	});

	ace.define("ace/range_list",["require","exports","module","ace/range"], function(acequire, exports, module) {
	"use strict";
	var Range = acequire("./range").Range;
	var comparePoints = Range.comparePoints;

	var RangeList = function() {
	    this.ranges = [];
	};

	(function() {
	    this.comparePoints = comparePoints;

	    this.pointIndex = function(pos, excludeEdges, startIndex) {
	        var list = this.ranges;

	        for (var i = startIndex || 0; i < list.length; i++) {
	            var range = list[i];
	            var cmpEnd = comparePoints(pos, range.end);
	            if (cmpEnd > 0)
	                continue;
	            var cmpStart = comparePoints(pos, range.start);
	            if (cmpEnd === 0)
	                return excludeEdges && cmpStart !== 0 ? -i-2 : i;
	            if (cmpStart > 0 || (cmpStart === 0 && !excludeEdges))
	                return i;

	            return -i-1;
	        }
	        return -i - 1;
	    };

	    this.add = function(range) {
	        var excludeEdges = !range.isEmpty();
	        var startIndex = this.pointIndex(range.start, excludeEdges);
	        if (startIndex < 0)
	            startIndex = -startIndex - 1;

	        var endIndex = this.pointIndex(range.end, excludeEdges, startIndex);

	        if (endIndex < 0)
	            endIndex = -endIndex - 1;
	        else
	            endIndex++;
	        return this.ranges.splice(startIndex, endIndex - startIndex, range);
	    };

	    this.addList = function(list) {
	        var removed = [];
	        for (var i = list.length; i--; ) {
	            removed.push.apply(removed, this.add(list[i]));
	        }
	        return removed;
	    };

	    this.substractPoint = function(pos) {
	        var i = this.pointIndex(pos);

	        if (i >= 0)
	            return this.ranges.splice(i, 1);
	    };
	    this.merge = function() {
	        var removed = [];
	        var list = this.ranges;
	        
	        list = list.sort(function(a, b) {
	            return comparePoints(a.start, b.start);
	        });
	        
	        var next = list[0], range;
	        for (var i = 1; i < list.length; i++) {
	            range = next;
	            next = list[i];
	            var cmp = comparePoints(range.end, next.start);
	            if (cmp < 0)
	                continue;

	            if (cmp == 0 && !range.isEmpty() && !next.isEmpty())
	                continue;

	            if (comparePoints(range.end, next.end) < 0) {
	                range.end.row = next.end.row;
	                range.end.column = next.end.column;
	            }

	            list.splice(i, 1);
	            removed.push(next);
	            next = range;
	            i--;
	        }
	        
	        this.ranges = list;

	        return removed;
	    };

	    this.contains = function(row, column) {
	        return this.pointIndex({row: row, column: column}) >= 0;
	    };

	    this.containsPoint = function(pos) {
	        return this.pointIndex(pos) >= 0;
	    };

	    this.rangeAtPoint = function(pos) {
	        var i = this.pointIndex(pos);
	        if (i >= 0)
	            return this.ranges[i];
	    };


	    this.clipRows = function(startRow, endRow) {
	        var list = this.ranges;
	        if (list[0].start.row > endRow || list[list.length - 1].start.row < startRow)
	            return [];

	        var startIndex = this.pointIndex({row: startRow, column: 0});
	        if (startIndex < 0)
	            startIndex = -startIndex - 1;
	        var endIndex = this.pointIndex({row: endRow, column: 0}, startIndex);
	        if (endIndex < 0)
	            endIndex = -endIndex - 1;

	        var clipped = [];
	        for (var i = startIndex; i < endIndex; i++) {
	            clipped.push(list[i]);
	        }
	        return clipped;
	    };

	    this.removeAll = function() {
	        return this.ranges.splice(0, this.ranges.length);
	    };

	    this.attach = function(session) {
	        if (this.session)
	            this.detach();

	        this.session = session;
	        this.onChange = this.$onChange.bind(this);

	        this.session.on('change', this.onChange);
	    };

	    this.detach = function() {
	        if (!this.session)
	            return;
	        this.session.removeListener('change', this.onChange);
	        this.session = null;
	    };

	    this.$onChange = function(delta) {
	        if (delta.action == "insert"){
	            var start = delta.start;
	            var end = delta.end;
	        } else {
	            var end = delta.start;
	            var start = delta.end;
	        }
	        var startRow = start.row;
	        var endRow = end.row;
	        var lineDif = endRow - startRow;

	        var colDiff = -start.column + end.column;
	        var ranges = this.ranges;

	        for (var i = 0, n = ranges.length; i < n; i++) {
	            var r = ranges[i];
	            if (r.end.row < startRow)
	                continue;
	            if (r.start.row > startRow)
	                break;

	            if (r.start.row == startRow && r.start.column >= start.column ) {
	                if (r.start.column == start.column && this.$insertRight) {
	                } else {
	                    r.start.column += colDiff;
	                    r.start.row += lineDif;
	                }
	            }
	            if (r.end.row == startRow && r.end.column >= start.column) {
	                if (r.end.column == start.column && this.$insertRight) {
	                    continue;
	                }
	                if (r.end.column == start.column && colDiff > 0 && i < n - 1) {                
	                    if (r.end.column > r.start.column && r.end.column == ranges[i+1].start.column)
	                        r.end.column -= colDiff;
	                }
	                r.end.column += colDiff;
	                r.end.row += lineDif;
	            }
	        }

	        if (lineDif != 0 && i < n) {
	            for (; i < n; i++) {
	                var r = ranges[i];
	                r.start.row += lineDif;
	                r.end.row += lineDif;
	            }
	        }
	    };

	}).call(RangeList.prototype);

	exports.RangeList = RangeList;
	});

	ace.define("ace/edit_session/fold",["require","exports","module","ace/range","ace/range_list","ace/lib/oop"], function(acequire, exports, module) {
	"use strict";

	var Range = acequire("../range").Range;
	var RangeList = acequire("../range_list").RangeList;
	var oop = acequire("../lib/oop");
	var Fold = exports.Fold = function(range, placeholder) {
	    this.foldLine = null;
	    this.placeholder = placeholder;
	    this.range = range;
	    this.start = range.start;
	    this.end = range.end;

	    this.sameRow = range.start.row == range.end.row;
	    this.subFolds = this.ranges = [];
	};

	oop.inherits(Fold, RangeList);

	(function() {

	    this.toString = function() {
	        return '"' + this.placeholder + '" ' + this.range.toString();
	    };

	    this.setFoldLine = function(foldLine) {
	        this.foldLine = foldLine;
	        this.subFolds.forEach(function(fold) {
	            fold.setFoldLine(foldLine);
	        });
	    };

	    this.clone = function() {
	        var range = this.range.clone();
	        var fold = new Fold(range, this.placeholder);
	        this.subFolds.forEach(function(subFold) {
	            fold.subFolds.push(subFold.clone());
	        });
	        fold.collapseChildren = this.collapseChildren;
	        return fold;
	    };

	    this.addSubFold = function(fold) {
	        if (this.range.isEqual(fold))
	            return;

	        if (!this.range.containsRange(fold))
	            throw new Error("A fold can't intersect already existing fold" + fold.range + this.range);
	        consumeRange(fold, this.start);

	        var row = fold.start.row, column = fold.start.column;
	        for (var i = 0, cmp = -1; i < this.subFolds.length; i++) {
	            cmp = this.subFolds[i].range.compare(row, column);
	            if (cmp != 1)
	                break;
	        }
	        var afterStart = this.subFolds[i];

	        if (cmp == 0)
	            return afterStart.addSubFold(fold);
	        var row = fold.range.end.row, column = fold.range.end.column;
	        for (var j = i, cmp = -1; j < this.subFolds.length; j++) {
	            cmp = this.subFolds[j].range.compare(row, column);
	            if (cmp != 1)
	                break;
	        }
	        var afterEnd = this.subFolds[j];

	        if (cmp == 0)
	            throw new Error("A fold can't intersect already existing fold" + fold.range + this.range);

	        var consumedFolds = this.subFolds.splice(i, j - i, fold);
	        fold.setFoldLine(this.foldLine);

	        return fold;
	    };
	    
	    this.restoreRange = function(range) {
	        return restoreRange(range, this.start);
	    };

	}).call(Fold.prototype);

	function consumePoint(point, anchor) {
	    point.row -= anchor.row;
	    if (point.row == 0)
	        point.column -= anchor.column;
	}
	function consumeRange(range, anchor) {
	    consumePoint(range.start, anchor);
	    consumePoint(range.end, anchor);
	}
	function restorePoint(point, anchor) {
	    if (point.row == 0)
	        point.column += anchor.column;
	    point.row += anchor.row;
	}
	function restoreRange(range, anchor) {
	    restorePoint(range.start, anchor);
	    restorePoint(range.end, anchor);
	}

	});

	ace.define("ace/edit_session/folding",["require","exports","module","ace/range","ace/edit_session/fold_line","ace/edit_session/fold","ace/token_iterator"], function(acequire, exports, module) {
	"use strict";

	var Range = acequire("../range").Range;
	var FoldLine = acequire("./fold_line").FoldLine;
	var Fold = acequire("./fold").Fold;
	var TokenIterator = acequire("../token_iterator").TokenIterator;

	function Folding() {
	    this.getFoldAt = function(row, column, side) {
	        var foldLine = this.getFoldLine(row);
	        if (!foldLine)
	            return null;

	        var folds = foldLine.folds;
	        for (var i = 0; i < folds.length; i++) {
	            var fold = folds[i];
	            if (fold.range.contains(row, column)) {
	                if (side == 1 && fold.range.isEnd(row, column)) {
	                    continue;
	                } else if (side == -1 && fold.range.isStart(row, column)) {
	                    continue;
	                }
	                return fold;
	            }
	        }
	    };
	    this.getFoldsInRange = function(range) {
	        var start = range.start;
	        var end = range.end;
	        var foldLines = this.$foldData;
	        var foundFolds = [];

	        start.column += 1;
	        end.column -= 1;

	        for (var i = 0; i < foldLines.length; i++) {
	            var cmp = foldLines[i].range.compareRange(range);
	            if (cmp == 2) {
	                continue;
	            }
	            else if (cmp == -2) {
	                break;
	            }

	            var folds = foldLines[i].folds;
	            for (var j = 0; j < folds.length; j++) {
	                var fold = folds[j];
	                cmp = fold.range.compareRange(range);
	                if (cmp == -2) {
	                    break;
	                } else if (cmp == 2) {
	                    continue;
	                } else
	                if (cmp == 42) {
	                    break;
	                }
	                foundFolds.push(fold);
	            }
	        }
	        start.column -= 1;
	        end.column += 1;

	        return foundFolds;
	    };

	    this.getFoldsInRangeList = function(ranges) {
	        if (Array.isArray(ranges)) {
	            var folds = [];
	            ranges.forEach(function(range) {
	                folds = folds.concat(this.getFoldsInRange(range));
	            }, this);
	        } else {
	            var folds = this.getFoldsInRange(ranges);
	        }
	        return folds;
	    };
	    this.getAllFolds = function() {
	        var folds = [];
	        var foldLines = this.$foldData;
	        
	        for (var i = 0; i < foldLines.length; i++)
	            for (var j = 0; j < foldLines[i].folds.length; j++)
	                folds.push(foldLines[i].folds[j]);

	        return folds;
	    };
	    this.getFoldStringAt = function(row, column, trim, foldLine) {
	        foldLine = foldLine || this.getFoldLine(row);
	        if (!foldLine)
	            return null;

	        var lastFold = {
	            end: { column: 0 }
	        };
	        var str, fold;
	        for (var i = 0; i < foldLine.folds.length; i++) {
	            fold = foldLine.folds[i];
	            var cmp = fold.range.compareEnd(row, column);
	            if (cmp == -1) {
	                str = this
	                    .getLine(fold.start.row)
	                    .substring(lastFold.end.column, fold.start.column);
	                break;
	            }
	            else if (cmp === 0) {
	                return null;
	            }
	            lastFold = fold;
	        }
	        if (!str)
	            str = this.getLine(fold.start.row).substring(lastFold.end.column);

	        if (trim == -1)
	            return str.substring(0, column - lastFold.end.column);
	        else if (trim == 1)
	            return str.substring(column - lastFold.end.column);
	        else
	            return str;
	    };

	    this.getFoldLine = function(docRow, startFoldLine) {
	        var foldData = this.$foldData;
	        var i = 0;
	        if (startFoldLine)
	            i = foldData.indexOf(startFoldLine);
	        if (i == -1)
	            i = 0;
	        for (i; i < foldData.length; i++) {
	            var foldLine = foldData[i];
	            if (foldLine.start.row <= docRow && foldLine.end.row >= docRow) {
	                return foldLine;
	            } else if (foldLine.end.row > docRow) {
	                return null;
	            }
	        }
	        return null;
	    };
	    this.getNextFoldLine = function(docRow, startFoldLine) {
	        var foldData = this.$foldData;
	        var i = 0;
	        if (startFoldLine)
	            i = foldData.indexOf(startFoldLine);
	        if (i == -1)
	            i = 0;
	        for (i; i < foldData.length; i++) {
	            var foldLine = foldData[i];
	            if (foldLine.end.row >= docRow) {
	                return foldLine;
	            }
	        }
	        return null;
	    };

	    this.getFoldedRowCount = function(first, last) {
	        var foldData = this.$foldData, rowCount = last-first+1;
	        for (var i = 0; i < foldData.length; i++) {
	            var foldLine = foldData[i],
	                end = foldLine.end.row,
	                start = foldLine.start.row;
	            if (end >= last) {
	                if (start < last) {
	                    if (start >= first)
	                        rowCount -= last-start;
	                    else
	                        rowCount = 0; // in one fold
	                }
	                break;
	            } else if (end >= first){
	                if (start >= first) // fold inside range
	                    rowCount -=  end-start;
	                else
	                    rowCount -=  end-first+1;
	            }
	        }
	        return rowCount;
	    };

	    this.$addFoldLine = function(foldLine) {
	        this.$foldData.push(foldLine);
	        this.$foldData.sort(function(a, b) {
	            return a.start.row - b.start.row;
	        });
	        return foldLine;
	    };
	    this.addFold = function(placeholder, range) {
	        var foldData = this.$foldData;
	        var added = false;
	        var fold;
	        
	        if (placeholder instanceof Fold)
	            fold = placeholder;
	        else {
	            fold = new Fold(range, placeholder);
	            fold.collapseChildren = range.collapseChildren;
	        }
	        this.$clipRangeToDocument(fold.range);

	        var startRow = fold.start.row;
	        var startColumn = fold.start.column;
	        var endRow = fold.end.row;
	        var endColumn = fold.end.column;
	        if (!(startRow < endRow || 
	            startRow == endRow && startColumn <= endColumn - 2))
	            throw new Error("The range has to be at least 2 characters width");

	        var startFold = this.getFoldAt(startRow, startColumn, 1);
	        var endFold = this.getFoldAt(endRow, endColumn, -1);
	        if (startFold && endFold == startFold)
	            return startFold.addSubFold(fold);

	        if (startFold && !startFold.range.isStart(startRow, startColumn))
	            this.removeFold(startFold);
	        
	        if (endFold && !endFold.range.isEnd(endRow, endColumn))
	            this.removeFold(endFold);
	        var folds = this.getFoldsInRange(fold.range);
	        if (folds.length > 0) {
	            this.removeFolds(folds);
	            folds.forEach(function(subFold) {
	                fold.addSubFold(subFold);
	            });
	        }

	        for (var i = 0; i < foldData.length; i++) {
	            var foldLine = foldData[i];
	            if (endRow == foldLine.start.row) {
	                foldLine.addFold(fold);
	                added = true;
	                break;
	            } else if (startRow == foldLine.end.row) {
	                foldLine.addFold(fold);
	                added = true;
	                if (!fold.sameRow) {
	                    var foldLineNext = foldData[i + 1];
	                    if (foldLineNext && foldLineNext.start.row == endRow) {
	                        foldLine.merge(foldLineNext);
	                        break;
	                    }
	                }
	                break;
	            } else if (endRow <= foldLine.start.row) {
	                break;
	            }
	        }

	        if (!added)
	            foldLine = this.$addFoldLine(new FoldLine(this.$foldData, fold));

	        if (this.$useWrapMode)
	            this.$updateWrapData(foldLine.start.row, foldLine.start.row);
	        else
	            this.$updateRowLengthCache(foldLine.start.row, foldLine.start.row);
	        this.$modified = true;
	        this._signal("changeFold", { data: fold, action: "add" });

	        return fold;
	    };

	    this.addFolds = function(folds) {
	        folds.forEach(function(fold) {
	            this.addFold(fold);
	        }, this);
	    };

	    this.removeFold = function(fold) {
	        var foldLine = fold.foldLine;
	        var startRow = foldLine.start.row;
	        var endRow = foldLine.end.row;

	        var foldLines = this.$foldData;
	        var folds = foldLine.folds;
	        if (folds.length == 1) {
	            foldLines.splice(foldLines.indexOf(foldLine), 1);
	        } else
	        if (foldLine.range.isEnd(fold.end.row, fold.end.column)) {
	            folds.pop();
	            foldLine.end.row = folds[folds.length - 1].end.row;
	            foldLine.end.column = folds[folds.length - 1].end.column;
	        } else
	        if (foldLine.range.isStart(fold.start.row, fold.start.column)) {
	            folds.shift();
	            foldLine.start.row = folds[0].start.row;
	            foldLine.start.column = folds[0].start.column;
	        } else
	        if (fold.sameRow) {
	            folds.splice(folds.indexOf(fold), 1);
	        } else
	        {
	            var newFoldLine = foldLine.split(fold.start.row, fold.start.column);
	            folds = newFoldLine.folds;
	            folds.shift();
	            newFoldLine.start.row = folds[0].start.row;
	            newFoldLine.start.column = folds[0].start.column;
	        }

	        if (!this.$updating) {
	            if (this.$useWrapMode)
	                this.$updateWrapData(startRow, endRow);
	            else
	                this.$updateRowLengthCache(startRow, endRow);
	        }
	        this.$modified = true;
	        this._signal("changeFold", { data: fold, action: "remove" });
	    };

	    this.removeFolds = function(folds) {
	        var cloneFolds = [];
	        for (var i = 0; i < folds.length; i++) {
	            cloneFolds.push(folds[i]);
	        }

	        cloneFolds.forEach(function(fold) {
	            this.removeFold(fold);
	        }, this);
	        this.$modified = true;
	    };

	    this.expandFold = function(fold) {
	        this.removeFold(fold);
	        fold.subFolds.forEach(function(subFold) {
	            fold.restoreRange(subFold);
	            this.addFold(subFold);
	        }, this);
	        if (fold.collapseChildren > 0) {
	            this.foldAll(fold.start.row+1, fold.end.row, fold.collapseChildren-1);
	        }
	        fold.subFolds = [];
	    };

	    this.expandFolds = function(folds) {
	        folds.forEach(function(fold) {
	            this.expandFold(fold);
	        }, this);
	    };

	    this.unfold = function(location, expandInner) {
	        var range, folds;
	        if (location == null) {
	            range = new Range(0, 0, this.getLength(), 0);
	            expandInner = true;
	        } else if (typeof location == "number")
	            range = new Range(location, 0, location, this.getLine(location).length);
	        else if ("row" in location)
	            range = Range.fromPoints(location, location);
	        else
	            range = location;
	        
	        folds = this.getFoldsInRangeList(range);
	        if (expandInner) {
	            this.removeFolds(folds);
	        } else {
	            var subFolds = folds;
	            while (subFolds.length) {
	                this.expandFolds(subFolds);
	                subFolds = this.getFoldsInRangeList(range);
	            }
	        }
	        if (folds.length)
	            return folds;
	    };
	    this.isRowFolded = function(docRow, startFoldRow) {
	        return !!this.getFoldLine(docRow, startFoldRow);
	    };

	    this.getRowFoldEnd = function(docRow, startFoldRow) {
	        var foldLine = this.getFoldLine(docRow, startFoldRow);
	        return foldLine ? foldLine.end.row : docRow;
	    };

	    this.getRowFoldStart = function(docRow, startFoldRow) {
	        var foldLine = this.getFoldLine(docRow, startFoldRow);
	        return foldLine ? foldLine.start.row : docRow;
	    };

	    this.getFoldDisplayLine = function(foldLine, endRow, endColumn, startRow, startColumn) {
	        if (startRow == null)
	            startRow = foldLine.start.row;
	        if (startColumn == null)
	            startColumn = 0;
	        if (endRow == null)
	            endRow = foldLine.end.row;
	        if (endColumn == null)
	            endColumn = this.getLine(endRow).length;
	        var doc = this.doc;
	        var textLine = "";

	        foldLine.walk(function(placeholder, row, column, lastColumn) {
	            if (row < startRow)
	                return;
	            if (row == startRow) {
	                if (column < startColumn)
	                    return;
	                lastColumn = Math.max(startColumn, lastColumn);
	            }

	            if (placeholder != null) {
	                textLine += placeholder;
	            } else {
	                textLine += doc.getLine(row).substring(lastColumn, column);
	            }
	        }, endRow, endColumn);
	        return textLine;
	    };

	    this.getDisplayLine = function(row, endColumn, startRow, startColumn) {
	        var foldLine = this.getFoldLine(row);

	        if (!foldLine) {
	            var line;
	            line = this.doc.getLine(row);
	            return line.substring(startColumn || 0, endColumn || line.length);
	        } else {
	            return this.getFoldDisplayLine(
	                foldLine, row, endColumn, startRow, startColumn);
	        }
	    };

	    this.$cloneFoldData = function() {
	        var fd = [];
	        fd = this.$foldData.map(function(foldLine) {
	            var folds = foldLine.folds.map(function(fold) {
	                return fold.clone();
	            });
	            return new FoldLine(fd, folds);
	        });

	        return fd;
	    };

	    this.toggleFold = function(tryToUnfold) {
	        var selection = this.selection;
	        var range = selection.getRange();
	        var fold;
	        var bracketPos;

	        if (range.isEmpty()) {
	            var cursor = range.start;
	            fold = this.getFoldAt(cursor.row, cursor.column);

	            if (fold) {
	                this.expandFold(fold);
	                return;
	            } else if (bracketPos = this.findMatchingBracket(cursor)) {
	                if (range.comparePoint(bracketPos) == 1) {
	                    range.end = bracketPos;
	                } else {
	                    range.start = bracketPos;
	                    range.start.column++;
	                    range.end.column--;
	                }
	            } else if (bracketPos = this.findMatchingBracket({row: cursor.row, column: cursor.column + 1})) {
	                if (range.comparePoint(bracketPos) == 1)
	                    range.end = bracketPos;
	                else
	                    range.start = bracketPos;

	                range.start.column++;
	            } else {
	                range = this.getCommentFoldRange(cursor.row, cursor.column) || range;
	            }
	        } else {
	            var folds = this.getFoldsInRange(range);
	            if (tryToUnfold && folds.length) {
	                this.expandFolds(folds);
	                return;
	            } else if (folds.length == 1 ) {
	                fold = folds[0];
	            }
	        }

	        if (!fold)
	            fold = this.getFoldAt(range.start.row, range.start.column);

	        if (fold && fold.range.toString() == range.toString()) {
	            this.expandFold(fold);
	            return;
	        }

	        var placeholder = "...";
	        if (!range.isMultiLine()) {
	            placeholder = this.getTextRange(range);
	            if (placeholder.length < 4)
	                return;
	            placeholder = placeholder.trim().substring(0, 2) + "..";
	        }

	        this.addFold(placeholder, range);
	    };

	    this.getCommentFoldRange = function(row, column, dir) {
	        var iterator = new TokenIterator(this, row, column);
	        var token = iterator.getCurrentToken();
	        var type = token.type;
	        if (token && /^comment|string/.test(type)) {
	            type = type.match(/comment|string/)[0];
	            if (type == "comment")
	                type += "|doc-start";
	            var re = new RegExp(type);
	            var range = new Range();
	            if (dir != 1) {
	                do {
	                    token = iterator.stepBackward();
	                } while (token && re.test(token.type));
	                iterator.stepForward();
	            }
	            
	            range.start.row = iterator.getCurrentTokenRow();
	            range.start.column = iterator.getCurrentTokenColumn() + 2;

	            iterator = new TokenIterator(this, row, column);
	            
	            if (dir != -1) {
	                var lastRow = -1;
	                do {
	                    token = iterator.stepForward();
	                    if (lastRow == -1) {
	                        var state = this.getState(iterator.$row);
	                        if (!re.test(state))
	                            lastRow = iterator.$row;
	                    } else if (iterator.$row > lastRow) {
	                        break;
	                    }
	                } while (token && re.test(token.type));
	                token = iterator.stepBackward();
	            } else
	                token = iterator.getCurrentToken();

	            range.end.row = iterator.getCurrentTokenRow();
	            range.end.column = iterator.getCurrentTokenColumn() + token.value.length - 2;
	            return range;
	        }
	    };

	    this.foldAll = function(startRow, endRow, depth) {
	        if (depth == undefined)
	            depth = 100000; // JSON.stringify doesn't hanle Infinity
	        var foldWidgets = this.foldWidgets;
	        if (!foldWidgets)
	            return; // mode doesn't support folding
	        endRow = endRow || this.getLength();
	        startRow = startRow || 0;
	        for (var row = startRow; row < endRow; row++) {
	            if (foldWidgets[row] == null)
	                foldWidgets[row] = this.getFoldWidget(row);
	            if (foldWidgets[row] != "start")
	                continue;

	            var range = this.getFoldWidgetRange(row);
	            if (range && range.isMultiLine()
	                && range.end.row <= endRow
	                && range.start.row >= startRow
	            ) {
	                row = range.end.row;
	                try {
	                    var fold = this.addFold("...", range);
	                    if (fold)
	                        fold.collapseChildren = depth;
	                } catch(e) {}
	            }
	        }
	    };
	    this.$foldStyles = {
	        "manual": 1,
	        "markbegin": 1,
	        "markbeginend": 1
	    };
	    this.$foldStyle = "markbegin";
	    this.setFoldStyle = function(style) {
	        if (!this.$foldStyles[style])
	            throw new Error("invalid fold style: " + style + "[" + Object.keys(this.$foldStyles).join(", ") + "]");
	        
	        if (this.$foldStyle == style)
	            return;

	        this.$foldStyle = style;
	        
	        if (style == "manual")
	            this.unfold();
	        var mode = this.$foldMode;
	        this.$setFolding(null);
	        this.$setFolding(mode);
	    };

	    this.$setFolding = function(foldMode) {
	        if (this.$foldMode == foldMode)
	            return;
	            
	        this.$foldMode = foldMode;
	        
	        this.off('change', this.$updateFoldWidgets);
	        this.off('tokenizerUpdate', this.$tokenizerUpdateFoldWidgets);
	        this._signal("changeAnnotation");
	        
	        if (!foldMode || this.$foldStyle == "manual") {
	            this.foldWidgets = null;
	            return;
	        }
	        
	        this.foldWidgets = [];
	        this.getFoldWidget = foldMode.getFoldWidget.bind(foldMode, this, this.$foldStyle);
	        this.getFoldWidgetRange = foldMode.getFoldWidgetRange.bind(foldMode, this, this.$foldStyle);
	        
	        this.$updateFoldWidgets = this.updateFoldWidgets.bind(this);
	        this.$tokenizerUpdateFoldWidgets = this.tokenizerUpdateFoldWidgets.bind(this);
	        this.on('change', this.$updateFoldWidgets);
	        this.on('tokenizerUpdate', this.$tokenizerUpdateFoldWidgets);
	    };

	    this.getParentFoldRangeData = function (row, ignoreCurrent) {
	        var fw = this.foldWidgets;
	        if (!fw || (ignoreCurrent && fw[row]))
	            return {};

	        var i = row - 1, firstRange;
	        while (i >= 0) {
	            var c = fw[i];
	            if (c == null)
	                c = fw[i] = this.getFoldWidget(i);

	            if (c == "start") {
	                var range = this.getFoldWidgetRange(i);
	                if (!firstRange)
	                    firstRange = range;
	                if (range && range.end.row >= row)
	                    break;
	            }
	            i--;
	        }

	        return {
	            range: i !== -1 && range,
	            firstRange: firstRange
	        };
	    };

	    this.onFoldWidgetClick = function(row, e) {
	        e = e.domEvent;
	        var options = {
	            children: e.shiftKey,
	            all: e.ctrlKey || e.metaKey,
	            siblings: e.altKey
	        };
	        
	        var range = this.$toggleFoldWidget(row, options);
	        if (!range) {
	            var el = (e.target || e.srcElement);
	            if (el && /ace_fold-widget/.test(el.className))
	                el.className += " ace_invalid";
	        }
	    };
	    
	    this.$toggleFoldWidget = function(row, options) {
	        if (!this.getFoldWidget)
	            return;
	        var type = this.getFoldWidget(row);
	        var line = this.getLine(row);

	        var dir = type === "end" ? -1 : 1;
	        var fold = this.getFoldAt(row, dir === -1 ? 0 : line.length, dir);

	        if (fold) {
	            if (options.children || options.all)
	                this.removeFold(fold);
	            else
	                this.expandFold(fold);
	            return fold;
	        }

	        var range = this.getFoldWidgetRange(row, true);
	        if (range && !range.isMultiLine()) {
	            fold = this.getFoldAt(range.start.row, range.start.column, 1);
	            if (fold && range.isEqual(fold.range)) {
	                this.removeFold(fold);
	                return fold;
	            }
	        }
	        
	        if (options.siblings) {
	            var data = this.getParentFoldRangeData(row);
	            if (data.range) {
	                var startRow = data.range.start.row + 1;
	                var endRow = data.range.end.row;
	            }
	            this.foldAll(startRow, endRow, options.all ? 10000 : 0);
	        } else if (options.children) {
	            endRow = range ? range.end.row : this.getLength();
	            this.foldAll(row + 1, endRow, options.all ? 10000 : 0);
	        } else if (range) {
	            if (options.all) 
	                range.collapseChildren = 10000;
	            this.addFold("...", range);
	        }
	        
	        return range;
	    };
	    
	    
	    
	    this.toggleFoldWidget = function(toggleParent) {
	        var row = this.selection.getCursor().row;
	        row = this.getRowFoldStart(row);
	        var range = this.$toggleFoldWidget(row, {});
	        
	        if (range)
	            return;
	        var data = this.getParentFoldRangeData(row, true);
	        range = data.range || data.firstRange;
	        
	        if (range) {
	            row = range.start.row;
	            var fold = this.getFoldAt(row, this.getLine(row).length, 1);

	            if (fold) {
	                this.removeFold(fold);
	            } else {
	                this.addFold("...", range);
	            }
	        }
	    };

	    this.updateFoldWidgets = function(delta) {
	        var firstRow = delta.start.row;
	        var len = delta.end.row - firstRow;

	        if (len === 0) {
	            this.foldWidgets[firstRow] = null;
	        } else if (delta.action == 'remove') {
	            this.foldWidgets.splice(firstRow, len + 1, null);
	        } else {
	            var args = Array(len + 1);
	            args.unshift(firstRow, 1);
	            this.foldWidgets.splice.apply(this.foldWidgets, args);
	        }
	    };
	    this.tokenizerUpdateFoldWidgets = function(e) {
	        var rows = e.data;
	        if (rows.first != rows.last) {
	            if (this.foldWidgets.length > rows.first)
	                this.foldWidgets.splice(rows.first, this.foldWidgets.length);
	        }
	    };
	}

	exports.Folding = Folding;

	});

	ace.define("ace/edit_session/bracket_match",["require","exports","module","ace/token_iterator","ace/range"], function(acequire, exports, module) {
	"use strict";

	var TokenIterator = acequire("../token_iterator").TokenIterator;
	var Range = acequire("../range").Range;


	function BracketMatch() {

	    this.findMatchingBracket = function(position, chr) {
	        if (position.column == 0) return null;

	        var charBeforeCursor = chr || this.getLine(position.row).charAt(position.column-1);
	        if (charBeforeCursor == "") return null;

	        var match = charBeforeCursor.match(/([\(\[\{])|([\)\]\}])/);
	        if (!match)
	            return null;

	        if (match[1])
	            return this.$findClosingBracket(match[1], position);
	        else
	            return this.$findOpeningBracket(match[2], position);
	    };
	    
	    this.getBracketRange = function(pos) {
	        var line = this.getLine(pos.row);
	        var before = true, range;

	        var chr = line.charAt(pos.column-1);
	        var match = chr && chr.match(/([\(\[\{])|([\)\]\}])/);
	        if (!match) {
	            chr = line.charAt(pos.column);
	            pos = {row: pos.row, column: pos.column + 1};
	            match = chr && chr.match(/([\(\[\{])|([\)\]\}])/);
	            before = false;
	        }
	        if (!match)
	            return null;

	        if (match[1]) {
	            var bracketPos = this.$findClosingBracket(match[1], pos);
	            if (!bracketPos)
	                return null;
	            range = Range.fromPoints(pos, bracketPos);
	            if (!before) {
	                range.end.column++;
	                range.start.column--;
	            }
	            range.cursor = range.end;
	        } else {
	            var bracketPos = this.$findOpeningBracket(match[2], pos);
	            if (!bracketPos)
	                return null;
	            range = Range.fromPoints(bracketPos, pos);
	            if (!before) {
	                range.start.column++;
	                range.end.column--;
	            }
	            range.cursor = range.start;
	        }
	        
	        return range;
	    };

	    this.$brackets = {
	        ")": "(",
	        "(": ")",
	        "]": "[",
	        "[": "]",
	        "{": "}",
	        "}": "{"
	    };

	    this.$findOpeningBracket = function(bracket, position, typeRe) {
	        var openBracket = this.$brackets[bracket];
	        var depth = 1;

	        var iterator = new TokenIterator(this, position.row, position.column);
	        var token = iterator.getCurrentToken();
	        if (!token)
	            token = iterator.stepForward();
	        if (!token)
	            return;
	        
	         if (!typeRe){
	            typeRe = new RegExp(
	                "(\\.?" +
	                token.type.replace(".", "\\.").replace("rparen", ".paren")
	                    .replace(/\b(?:end)\b/, "(?:start|begin|end)")
	                + ")+"
	            );
	        }
	        var valueIndex = position.column - iterator.getCurrentTokenColumn() - 2;
	        var value = token.value;
	        
	        while (true) {
	        
	            while (valueIndex >= 0) {
	                var chr = value.charAt(valueIndex);
	                if (chr == openBracket) {
	                    depth -= 1;
	                    if (depth == 0) {
	                        return {row: iterator.getCurrentTokenRow(),
	                            column: valueIndex + iterator.getCurrentTokenColumn()};
	                    }
	                }
	                else if (chr == bracket) {
	                    depth += 1;
	                }
	                valueIndex -= 1;
	            }
	            do {
	                token = iterator.stepBackward();
	            } while (token && !typeRe.test(token.type));

	            if (token == null)
	                break;
	                
	            value = token.value;
	            valueIndex = value.length - 1;
	        }
	        
	        return null;
	    };

	    this.$findClosingBracket = function(bracket, position, typeRe) {
	        var closingBracket = this.$brackets[bracket];
	        var depth = 1;

	        var iterator = new TokenIterator(this, position.row, position.column);
	        var token = iterator.getCurrentToken();
	        if (!token)
	            token = iterator.stepForward();
	        if (!token)
	            return;

	        if (!typeRe){
	            typeRe = new RegExp(
	                "(\\.?" +
	                token.type.replace(".", "\\.").replace("lparen", ".paren")
	                    .replace(/\b(?:start|begin)\b/, "(?:start|begin|end)")
	                + ")+"
	            );
	        }
	        var valueIndex = position.column - iterator.getCurrentTokenColumn();

	        while (true) {

	            var value = token.value;
	            var valueLength = value.length;
	            while (valueIndex < valueLength) {
	                var chr = value.charAt(valueIndex);
	                if (chr == closingBracket) {
	                    depth -= 1;
	                    if (depth == 0) {
	                        return {row: iterator.getCurrentTokenRow(),
	                            column: valueIndex + iterator.getCurrentTokenColumn()};
	                    }
	                }
	                else if (chr == bracket) {
	                    depth += 1;
	                }
	                valueIndex += 1;
	            }
	            do {
	                token = iterator.stepForward();
	            } while (token && !typeRe.test(token.type));

	            if (token == null)
	                break;

	            valueIndex = 0;
	        }
	        
	        return null;
	    };
	}
	exports.BracketMatch = BracketMatch;

	});

	ace.define("ace/edit_session",["require","exports","module","ace/lib/oop","ace/lib/lang","ace/bidihandler","ace/config","ace/lib/event_emitter","ace/selection","ace/mode/text","ace/range","ace/document","ace/background_tokenizer","ace/search_highlight","ace/edit_session/folding","ace/edit_session/bracket_match"], function(acequire, exports, module) {
	"use strict";

	var oop = acequire("./lib/oop");
	var lang = acequire("./lib/lang");
	var BidiHandler = acequire("./bidihandler").BidiHandler;
	var config = acequire("./config");
	var EventEmitter = acequire("./lib/event_emitter").EventEmitter;
	var Selection = acequire("./selection").Selection;
	var TextMode = acequire("./mode/text").Mode;
	var Range = acequire("./range").Range;
	var Document = acequire("./document").Document;
	var BackgroundTokenizer = acequire("./background_tokenizer").BackgroundTokenizer;
	var SearchHighlight = acequire("./search_highlight").SearchHighlight;

	var EditSession = function(text, mode) {
	    this.$breakpoints = [];
	    this.$decorations = [];
	    this.$frontMarkers = {};
	    this.$backMarkers = {};
	    this.$markerId = 1;
	    this.$undoSelect = true;

	    this.$foldData = [];
	    this.id = "session" + (++EditSession.$uid);
	    this.$foldData.toString = function() {
	        return this.join("\n");
	    };
	    this.on("changeFold", this.onChangeFold.bind(this));
	    this.$onChange = this.onChange.bind(this);

	    if (typeof text != "object" || !text.getLine)
	        text = new Document(text);

	    this.$bidiHandler = new BidiHandler(this);
	    this.setDocument(text);
	    this.selection = new Selection(this);

	    config.resetOptions(this);
	    this.setMode(mode);
	    config._signal("session", this);
	};


	EditSession.$uid = 0;

	(function() {

	    oop.implement(this, EventEmitter);
	    this.setDocument = function(doc) {
	        if (this.doc)
	            this.doc.removeListener("change", this.$onChange);

	        this.doc = doc;
	        doc.on("change", this.$onChange);

	        if (this.bgTokenizer)
	            this.bgTokenizer.setDocument(this.getDocument());

	        this.resetCaches();
	    };
	    this.getDocument = function() {
	        return this.doc;
	    };
	    this.$resetRowCache = function(docRow) {
	        if (!docRow) {
	            this.$docRowCache = [];
	            this.$screenRowCache = [];
	            return;
	        }
	        var l = this.$docRowCache.length;
	        var i = this.$getRowCacheIndex(this.$docRowCache, docRow) + 1;
	        if (l > i) {
	            this.$docRowCache.splice(i, l);
	            this.$screenRowCache.splice(i, l);
	        }
	    };

	    this.$getRowCacheIndex = function(cacheArray, val) {
	        var low = 0;
	        var hi = cacheArray.length - 1;

	        while (low <= hi) {
	            var mid = (low + hi) >> 1;
	            var c = cacheArray[mid];

	            if (val > c)
	                low = mid + 1;
	            else if (val < c)
	                hi = mid - 1;
	            else
	                return mid;
	        }

	        return low -1;
	    };

	    this.resetCaches = function() {
	        this.$modified = true;
	        this.$wrapData = [];
	        this.$rowLengthCache = [];
	        this.$resetRowCache(0);
	        if (this.bgTokenizer)
	            this.bgTokenizer.start(0);
	    };

	    this.onChangeFold = function(e) {
	        var fold = e.data;
	        this.$resetRowCache(fold.start.row);
	    };

	    this.onChange = function(delta) {
	        this.$modified = true;
	        this.$bidiHandler.onChange(delta);
	        this.$resetRowCache(delta.start.row);

	        var removedFolds = this.$updateInternalDataOnChange(delta);
	        if (!this.$fromUndo && this.$undoManager && !delta.ignore) {
	            this.$deltasDoc.push(delta);
	            if (removedFolds && removedFolds.length != 0) {
	                this.$deltasFold.push({
	                    action: "removeFolds",
	                    folds:  removedFolds
	                });
	            }

	            this.$informUndoManager.schedule();
	        }

	        this.bgTokenizer && this.bgTokenizer.$updateOnChange(delta);
	        this._signal("change", delta);
	    };
	    this.setValue = function(text) {
	        this.doc.setValue(text);
	        this.selection.moveTo(0, 0);

	        this.$resetRowCache(0);
	        this.$deltas = [];
	        this.$deltasDoc = [];
	        this.$deltasFold = [];
	        this.setUndoManager(this.$undoManager);
	        this.getUndoManager().reset();
	    };
	    this.getValue =
	    this.toString = function() {
	        return this.doc.getValue();
	    };
	    this.getSelection = function() {
	        return this.selection;
	    };
	    this.getState = function(row) {
	        return this.bgTokenizer.getState(row);
	    };
	    this.getTokens = function(row) {
	        return this.bgTokenizer.getTokens(row);
	    };
	    this.getTokenAt = function(row, column) {
	        var tokens = this.bgTokenizer.getTokens(row);
	        var token, c = 0;
	        if (column == null) {
	            var i = tokens.length - 1;
	            c = this.getLine(row).length;
	        } else {
	            for (var i = 0; i < tokens.length; i++) {
	                c += tokens[i].value.length;
	                if (c >= column)
	                    break;
	            }
	        }
	        token = tokens[i];
	        if (!token)
	            return null;
	        token.index = i;
	        token.start = c - token.value.length;
	        return token;
	    };
	    this.setUndoManager = function(undoManager) {
	        this.$undoManager = undoManager;
	        this.$deltas = [];
	        this.$deltasDoc = [];
	        this.$deltasFold = [];

	        if (this.$informUndoManager)
	            this.$informUndoManager.cancel();

	        if (undoManager) {
	            var self = this;

	            this.$syncInformUndoManager = function() {
	                self.$informUndoManager.cancel();

	                if (self.$deltasFold.length) {
	                    self.$deltas.push({
	                        group: "fold",
	                        deltas: self.$deltasFold
	                    });
	                    self.$deltasFold = [];
	                }

	                if (self.$deltasDoc.length) {
	                    self.$deltas.push({
	                        group: "doc",
	                        deltas: self.$deltasDoc
	                    });
	                    self.$deltasDoc = [];
	                }

	                if (self.$deltas.length > 0) {
	                    undoManager.execute({
	                        action: "aceupdate",
	                        args: [self.$deltas, self],
	                        merge: self.mergeUndoDeltas
	                    });
	                }
	                self.mergeUndoDeltas = false;
	                self.$deltas = [];
	            };
	            this.$informUndoManager = lang.delayedCall(this.$syncInformUndoManager);
	        }
	    };
	    this.markUndoGroup = function() {
	        if (this.$syncInformUndoManager)
	            this.$syncInformUndoManager();
	    };
	    
	    this.$defaultUndoManager = {
	        undo: function() {},
	        redo: function() {},
	        reset: function() {}
	    };
	    this.getUndoManager = function() {
	        return this.$undoManager || this.$defaultUndoManager;
	    };
	    this.getTabString = function() {
	        if (this.getUseSoftTabs()) {
	            return lang.stringRepeat(" ", this.getTabSize());
	        } else {
	            return "\t";
	        }
	    };
	    this.setUseSoftTabs = function(val) {
	        this.setOption("useSoftTabs", val);
	    };
	    this.getUseSoftTabs = function() {
	        return this.$useSoftTabs && !this.$mode.$indentWithTabs;
	    };
	    this.setTabSize = function(tabSize) {
	        this.setOption("tabSize", tabSize);
	    };
	    this.getTabSize = function() {
	        return this.$tabSize;
	    };
	    this.isTabStop = function(position) {
	        return this.$useSoftTabs && (position.column % this.$tabSize === 0);
	    };
	    this.setNavigateWithinSoftTabs = function (navigateWithinSoftTabs) {
	        this.setOption("navigateWithinSoftTabs", navigateWithinSoftTabs);
	    };
	    this.getNavigateWithinSoftTabs = function() {
	        return this.$navigateWithinSoftTabs;
	    };

	    this.$overwrite = false;
	    this.setOverwrite = function(overwrite) {
	        this.setOption("overwrite", overwrite);
	    };
	    this.getOverwrite = function() {
	        return this.$overwrite;
	    };
	    this.toggleOverwrite = function() {
	        this.setOverwrite(!this.$overwrite);
	    };
	    this.addGutterDecoration = function(row, className) {
	        if (!this.$decorations[row])
	            this.$decorations[row] = "";
	        this.$decorations[row] += " " + className;
	        this._signal("changeBreakpoint", {});
	    };
	    this.removeGutterDecoration = function(row, className) {
	        this.$decorations[row] = (this.$decorations[row] || "").replace(" " + className, "");
	        this._signal("changeBreakpoint", {});
	    };
	    this.getBreakpoints = function() {
	        return this.$breakpoints;
	    };
	    this.setBreakpoints = function(rows) {
	        this.$breakpoints = [];
	        for (var i=0; i<rows.length; i++) {
	            this.$breakpoints[rows[i]] = "ace_breakpoint";
	        }
	        this._signal("changeBreakpoint", {});
	    };
	    this.clearBreakpoints = function() {
	        this.$breakpoints = [];
	        this._signal("changeBreakpoint", {});
	    };
	    this.setBreakpoint = function(row, className) {
	        if (className === undefined)
	            className = "ace_breakpoint";
	        if (className)
	            this.$breakpoints[row] = className;
	        else
	            delete this.$breakpoints[row];
	        this._signal("changeBreakpoint", {});
	    };
	    this.clearBreakpoint = function(row) {
	        delete this.$breakpoints[row];
	        this._signal("changeBreakpoint", {});
	    };
	    this.addMarker = function(range, clazz, type, inFront) {
	        var id = this.$markerId++;

	        var marker = {
	            range : range,
	            type : type || "line",
	            renderer: typeof type == "function" ? type : null,
	            clazz : clazz,
	            inFront: !!inFront,
	            id: id
	        };

	        if (inFront) {
	            this.$frontMarkers[id] = marker;
	            this._signal("changeFrontMarker");
	        } else {
	            this.$backMarkers[id] = marker;
	            this._signal("changeBackMarker");
	        }

	        return id;
	    };
	    this.addDynamicMarker = function(marker, inFront) {
	        if (!marker.update)
	            return;
	        var id = this.$markerId++;
	        marker.id = id;
	        marker.inFront = !!inFront;

	        if (inFront) {
	            this.$frontMarkers[id] = marker;
	            this._signal("changeFrontMarker");
	        } else {
	            this.$backMarkers[id] = marker;
	            this._signal("changeBackMarker");
	        }

	        return marker;
	    };
	    this.removeMarker = function(markerId) {
	        var marker = this.$frontMarkers[markerId] || this.$backMarkers[markerId];
	        if (!marker)
	            return;

	        var markers = marker.inFront ? this.$frontMarkers : this.$backMarkers;
	        if (marker) {
	            delete (markers[markerId]);
	            this._signal(marker.inFront ? "changeFrontMarker" : "changeBackMarker");
	        }
	    };
	    this.getMarkers = function(inFront) {
	        return inFront ? this.$frontMarkers : this.$backMarkers;
	    };

	    this.highlight = function(re) {
	        if (!this.$searchHighlight) {
	            var highlight = new SearchHighlight(null, "ace_selected-word", "text");
	            this.$searchHighlight = this.addDynamicMarker(highlight);
	        }
	        this.$searchHighlight.setRegexp(re);
	    };
	    this.highlightLines = function(startRow, endRow, clazz, inFront) {
	        if (typeof endRow != "number") {
	            clazz = endRow;
	            endRow = startRow;
	        }
	        if (!clazz)
	            clazz = "ace_step";

	        var range = new Range(startRow, 0, endRow, Infinity);
	        range.id = this.addMarker(range, clazz, "fullLine", inFront);
	        return range;
	    };
	    this.setAnnotations = function(annotations) {
	        this.$annotations = annotations;
	        this._signal("changeAnnotation", {});
	    };
	    this.getAnnotations = function() {
	        return this.$annotations || [];
	    };
	    this.clearAnnotations = function() {
	        this.setAnnotations([]);
	    };
	    this.$detectNewLine = function(text) {
	        var match = text.match(/^.*?(\r?\n)/m);
	        if (match) {
	            this.$autoNewLine = match[1];
	        } else {
	            this.$autoNewLine = "\n";
	        }
	    };
	    this.getWordRange = function(row, column) {
	        var line = this.getLine(row);

	        var inToken = false;
	        if (column > 0)
	            inToken = !!line.charAt(column - 1).match(this.tokenRe);

	        if (!inToken)
	            inToken = !!line.charAt(column).match(this.tokenRe);

	        if (inToken)
	            var re = this.tokenRe;
	        else if (/^\s+$/.test(line.slice(column-1, column+1)))
	            var re = /\s/;
	        else
	            var re = this.nonTokenRe;

	        var start = column;
	        if (start > 0) {
	            do {
	                start--;
	            }
	            while (start >= 0 && line.charAt(start).match(re));
	            start++;
	        }

	        var end = column;
	        while (end < line.length && line.charAt(end).match(re)) {
	            end++;
	        }

	        return new Range(row, start, row, end);
	    };
	    this.getAWordRange = function(row, column) {
	        var wordRange = this.getWordRange(row, column);
	        var line = this.getLine(wordRange.end.row);

	        while (line.charAt(wordRange.end.column).match(/[ \t]/)) {
	            wordRange.end.column += 1;
	        }
	        return wordRange;
	    };
	    this.setNewLineMode = function(newLineMode) {
	        this.doc.setNewLineMode(newLineMode);
	    };
	    this.getNewLineMode = function() {
	        return this.doc.getNewLineMode();
	    };
	    this.setUseWorker = function(useWorker) { this.setOption("useWorker", useWorker); };
	    this.getUseWorker = function() { return this.$useWorker; };
	    this.onReloadTokenizer = function(e) {
	        var rows = e.data;
	        this.bgTokenizer.start(rows.first);
	        this._signal("tokenizerUpdate", e);
	    };

	    this.$modes = {};
	    this.$mode = null;
	    this.$modeId = null;
	    this.setMode = function(mode, cb) {
	        if (mode && typeof mode === "object") {
	            if (mode.getTokenizer)
	                return this.$onChangeMode(mode);
	            var options = mode;
	            var path = options.path;
	        } else {
	            path = mode || "ace/mode/text";
	        }
	        if (!this.$modes["ace/mode/text"])
	            this.$modes["ace/mode/text"] = new TextMode();

	        if (this.$modes[path] && !options) {
	            this.$onChangeMode(this.$modes[path]);
	            cb && cb();
	            return;
	        }
	        this.$modeId = path;
	        config.loadModule(["mode", path], function(m) {
	            if (this.$modeId !== path)
	                return cb && cb();
	            if (this.$modes[path] && !options) {
	                this.$onChangeMode(this.$modes[path]);
	            } else if (m && m.Mode) {
	                m = new m.Mode(options);
	                if (!options) {
	                    this.$modes[path] = m;
	                    m.$id = path;
	                }
	                this.$onChangeMode(m);
	            }
	            cb && cb();
	        }.bind(this));
	        if (!this.$mode)
	            this.$onChangeMode(this.$modes["ace/mode/text"], true);
	    };

	    this.$onChangeMode = function(mode, $isPlaceholder) {
	        if (!$isPlaceholder)
	            this.$modeId = mode.$id;
	        if (this.$mode === mode) 
	            return;

	        this.$mode = mode;

	        this.$stopWorker();

	        if (this.$useWorker)
	            this.$startWorker();

	        var tokenizer = mode.getTokenizer();

	        if(tokenizer.addEventListener !== undefined) {
	            var onReloadTokenizer = this.onReloadTokenizer.bind(this);
	            tokenizer.addEventListener("update", onReloadTokenizer);
	        }

	        if (!this.bgTokenizer) {
	            this.bgTokenizer = new BackgroundTokenizer(tokenizer);
	            var _self = this;
	            this.bgTokenizer.addEventListener("update", function(e) {
	                _self._signal("tokenizerUpdate", e);
	            });
	        } else {
	            this.bgTokenizer.setTokenizer(tokenizer);
	        }

	        this.bgTokenizer.setDocument(this.getDocument());

	        this.tokenRe = mode.tokenRe;
	        this.nonTokenRe = mode.nonTokenRe;

	        
	        if (!$isPlaceholder) {
	            if (mode.attachToSession)
	                mode.attachToSession(this);
	            this.$options.wrapMethod.set.call(this, this.$wrapMethod);
	            this.$setFolding(mode.foldingRules);
	            this.bgTokenizer.start(0);
	            this._emit("changeMode");
	        }
	    };

	    this.$stopWorker = function() {
	        if (this.$worker) {
	            this.$worker.terminate();
	            this.$worker = null;
	        }
	    };

	    this.$startWorker = function() {
	        try {
	            this.$worker = this.$mode.createWorker(this);
	        } catch (e) {
	            config.warn("Could not load worker", e);
	            this.$worker = null;
	        }
	    };
	    this.getMode = function() {
	        return this.$mode;
	    };

	    this.$scrollTop = 0;
	    this.setScrollTop = function(scrollTop) {
	        if (this.$scrollTop === scrollTop || isNaN(scrollTop))
	            return;

	        this.$scrollTop = scrollTop;
	        this._signal("changeScrollTop", scrollTop);
	    };
	    this.getScrollTop = function() {
	        return this.$scrollTop;
	    };

	    this.$scrollLeft = 0;
	    this.setScrollLeft = function(scrollLeft) {
	        if (this.$scrollLeft === scrollLeft || isNaN(scrollLeft))
	            return;

	        this.$scrollLeft = scrollLeft;
	        this._signal("changeScrollLeft", scrollLeft);
	    };
	    this.getScrollLeft = function() {
	        return this.$scrollLeft;
	    };
	    this.getScreenWidth = function() {
	        this.$computeWidth();
	        if (this.lineWidgets) 
	            return Math.max(this.getLineWidgetMaxWidth(), this.screenWidth);
	        return this.screenWidth;
	    };
	    
	    this.getLineWidgetMaxWidth = function() {
	        if (this.lineWidgetsWidth != null) return this.lineWidgetsWidth;
	        var width = 0;
	        this.lineWidgets.forEach(function(w) {
	            if (w && w.screenWidth > width)
	                width = w.screenWidth;
	        });
	        return this.lineWidgetWidth = width;
	    };

	    this.$computeWidth = function(force) {
	        if (this.$modified || force) {
	            this.$modified = false;

	            if (this.$useWrapMode)
	                return this.screenWidth = this.$wrapLimit;

	            var lines = this.doc.getAllLines();
	            var cache = this.$rowLengthCache;
	            var longestScreenLine = 0;
	            var foldIndex = 0;
	            var foldLine = this.$foldData[foldIndex];
	            var foldStart = foldLine ? foldLine.start.row : Infinity;
	            var len = lines.length;

	            for (var i = 0; i < len; i++) {
	                if (i > foldStart) {
	                    i = foldLine.end.row + 1;
	                    if (i >= len)
	                        break;
	                    foldLine = this.$foldData[foldIndex++];
	                    foldStart = foldLine ? foldLine.start.row : Infinity;
	                }

	                if (cache[i] == null)
	                    cache[i] = this.$getStringScreenWidth(lines[i])[0];

	                if (cache[i] > longestScreenLine)
	                    longestScreenLine = cache[i];
	            }
	            this.screenWidth = longestScreenLine;
	        }
	    };
	    this.getLine = function(row) {
	        return this.doc.getLine(row);
	    };
	    this.getLines = function(firstRow, lastRow) {
	        return this.doc.getLines(firstRow, lastRow);
	    };
	    this.getLength = function() {
	        return this.doc.getLength();
	    };
	    this.getTextRange = function(range) {
	        return this.doc.getTextRange(range || this.selection.getRange());
	    };
	    this.insert = function(position, text) {
	        return this.doc.insert(position, text);
	    };
	    this.remove = function(range) {
	        return this.doc.remove(range);
	    };
	    this.removeFullLines = function(firstRow, lastRow){
	        return this.doc.removeFullLines(firstRow, lastRow);
	    };
	    this.undoChanges = function(deltas, dontSelect) {
	        if (!deltas.length)
	            return;

	        this.$fromUndo = true;
	        var lastUndoRange = null;
	        for (var i = deltas.length - 1; i != -1; i--) {
	            var delta = deltas[i];
	            if (delta.group == "doc") {
	                this.doc.revertDeltas(delta.deltas);
	                lastUndoRange =
	                    this.$getUndoSelection(delta.deltas, true, lastUndoRange);
	            } else {
	                delta.deltas.forEach(function(foldDelta) {
	                    this.addFolds(foldDelta.folds);
	                }, this);
	            }
	        }
	        this.$fromUndo = false;
	        lastUndoRange &&
	            this.$undoSelect &&
	            !dontSelect &&
	            this.selection.setSelectionRange(lastUndoRange);
	        return lastUndoRange;
	    };
	    this.redoChanges = function(deltas, dontSelect) {
	        if (!deltas.length)
	            return;

	        this.$fromUndo = true;
	        var lastUndoRange = null;
	        for (var i = 0; i < deltas.length; i++) {
	            var delta = deltas[i];
	            if (delta.group == "doc") {
	                this.doc.applyDeltas(delta.deltas);
	                lastUndoRange =
	                    this.$getUndoSelection(delta.deltas, false, lastUndoRange);
	            }
	        }
	        this.$fromUndo = false;
	        lastUndoRange &&
	            this.$undoSelect &&
	            !dontSelect &&
	            this.selection.setSelectionRange(lastUndoRange);
	        return lastUndoRange;
	    };
	    this.setUndoSelect = function(enable) {
	        this.$undoSelect = enable;
	    };

	    this.$getUndoSelection = function(deltas, isUndo, lastUndoRange) {
	        function isInsert(delta) {
	            return isUndo ? delta.action !== "insert" : delta.action === "insert";
	        }

	        var delta = deltas[0];
	        var range, point;
	        var lastDeltaIsInsert = false;
	        if (isInsert(delta)) {
	            range = Range.fromPoints(delta.start, delta.end);
	            lastDeltaIsInsert = true;
	        } else {
	            range = Range.fromPoints(delta.start, delta.start);
	            lastDeltaIsInsert = false;
	        }

	        for (var i = 1; i < deltas.length; i++) {
	            delta = deltas[i];
	            if (isInsert(delta)) {
	                point = delta.start;
	                if (range.compare(point.row, point.column) == -1) {
	                    range.setStart(point);
	                }
	                point = delta.end;
	                if (range.compare(point.row, point.column) == 1) {
	                    range.setEnd(point);
	                }
	                lastDeltaIsInsert = true;
	            } else {
	                point = delta.start;
	                if (range.compare(point.row, point.column) == -1) {
	                    range = Range.fromPoints(delta.start, delta.start);
	                }
	                lastDeltaIsInsert = false;
	            }
	        }
	        if (lastUndoRange != null) {
	            if (Range.comparePoints(lastUndoRange.start, range.start) === 0) {
	                lastUndoRange.start.column += range.end.column - range.start.column;
	                lastUndoRange.end.column += range.end.column - range.start.column;
	            }

	            var cmp = lastUndoRange.compareRange(range);
	            if (cmp == 1) {
	                range.setStart(lastUndoRange.start);
	            } else if (cmp == -1) {
	                range.setEnd(lastUndoRange.end);
	            }
	        }

	        return range;
	    };
	    this.replace = function(range, text) {
	        return this.doc.replace(range, text);
	    };
	    this.moveText = function(fromRange, toPosition, copy) {
	        var text = this.getTextRange(fromRange);
	        var folds = this.getFoldsInRange(fromRange);

	        var toRange = Range.fromPoints(toPosition, toPosition);
	        if (!copy) {
	            this.remove(fromRange);
	            var rowDiff = fromRange.start.row - fromRange.end.row;
	            var collDiff = rowDiff ? -fromRange.end.column : fromRange.start.column - fromRange.end.column;
	            if (collDiff) {
	                if (toRange.start.row == fromRange.end.row && toRange.start.column > fromRange.end.column)
	                    toRange.start.column += collDiff;
	                if (toRange.end.row == fromRange.end.row && toRange.end.column > fromRange.end.column)
	                    toRange.end.column += collDiff;
	            }
	            if (rowDiff && toRange.start.row >= fromRange.end.row) {
	                toRange.start.row += rowDiff;
	                toRange.end.row += rowDiff;
	            }
	        }

	        toRange.end = this.insert(toRange.start, text);
	        if (folds.length) {
	            var oldStart = fromRange.start;
	            var newStart = toRange.start;
	            var rowDiff = newStart.row - oldStart.row;
	            var collDiff = newStart.column - oldStart.column;
	            this.addFolds(folds.map(function(x) {
	                x = x.clone();
	                if (x.start.row == oldStart.row)
	                    x.start.column += collDiff;
	                if (x.end.row == oldStart.row)
	                    x.end.column += collDiff;
	                x.start.row += rowDiff;
	                x.end.row += rowDiff;
	                return x;
	            }));
	        }

	        return toRange;
	    };
	    this.indentRows = function(startRow, endRow, indentString) {
	        indentString = indentString.replace(/\t/g, this.getTabString());
	        for (var row=startRow; row<=endRow; row++)
	            this.doc.insertInLine({row: row, column: 0}, indentString);
	    };
	    this.outdentRows = function (range) {
	        var rowRange = range.collapseRows();
	        var deleteRange = new Range(0, 0, 0, 0);
	        var size = this.getTabSize();

	        for (var i = rowRange.start.row; i <= rowRange.end.row; ++i) {
	            var line = this.getLine(i);

	            deleteRange.start.row = i;
	            deleteRange.end.row = i;
	            for (var j = 0; j < size; ++j)
	                if (line.charAt(j) != ' ')
	                    break;
	            if (j < size && line.charAt(j) == '\t') {
	                deleteRange.start.column = j;
	                deleteRange.end.column = j + 1;
	            } else {
	                deleteRange.start.column = 0;
	                deleteRange.end.column = j;
	            }
	            this.remove(deleteRange);
	        }
	    };

	    this.$moveLines = function(firstRow, lastRow, dir) {
	        firstRow = this.getRowFoldStart(firstRow);
	        lastRow = this.getRowFoldEnd(lastRow);
	        if (dir < 0) {
	            var row = this.getRowFoldStart(firstRow + dir);
	            if (row < 0) return 0;
	            var diff = row-firstRow;
	        } else if (dir > 0) {
	            var row = this.getRowFoldEnd(lastRow + dir);
	            if (row > this.doc.getLength()-1) return 0;
	            var diff = row-lastRow;
	        } else {
	            firstRow = this.$clipRowToDocument(firstRow);
	            lastRow = this.$clipRowToDocument(lastRow);
	            var diff = lastRow - firstRow + 1;
	        }

	        var range = new Range(firstRow, 0, lastRow, Number.MAX_VALUE);
	        var folds = this.getFoldsInRange(range).map(function(x){
	            x = x.clone();
	            x.start.row += diff;
	            x.end.row += diff;
	            return x;
	        });
	        
	        var lines = dir == 0
	            ? this.doc.getLines(firstRow, lastRow)
	            : this.doc.removeFullLines(firstRow, lastRow);
	        this.doc.insertFullLines(firstRow+diff, lines);
	        folds.length && this.addFolds(folds);
	        return diff;
	    };
	    this.moveLinesUp = function(firstRow, lastRow) {
	        return this.$moveLines(firstRow, lastRow, -1);
	    };
	    this.moveLinesDown = function(firstRow, lastRow) {
	        return this.$moveLines(firstRow, lastRow, 1);
	    };
	    this.duplicateLines = function(firstRow, lastRow) {
	        return this.$moveLines(firstRow, lastRow, 0);
	    };


	    this.$clipRowToDocument = function(row) {
	        return Math.max(0, Math.min(row, this.doc.getLength()-1));
	    };

	    this.$clipColumnToRow = function(row, column) {
	        if (column < 0)
	            return 0;
	        return Math.min(this.doc.getLine(row).length, column);
	    };


	    this.$clipPositionToDocument = function(row, column) {
	        column = Math.max(0, column);

	        if (row < 0) {
	            row = 0;
	            column = 0;
	        } else {
	            var len = this.doc.getLength();
	            if (row >= len) {
	                row = len - 1;
	                column = this.doc.getLine(len-1).length;
	            } else {
	                column = Math.min(this.doc.getLine(row).length, column);
	            }
	        }

	        return {
	            row: row,
	            column: column
	        };
	    };

	    this.$clipRangeToDocument = function(range) {
	        if (range.start.row < 0) {
	            range.start.row = 0;
	            range.start.column = 0;
	        } else {
	            range.start.column = this.$clipColumnToRow(
	                range.start.row,
	                range.start.column
	            );
	        }

	        var len = this.doc.getLength() - 1;
	        if (range.end.row > len) {
	            range.end.row = len;
	            range.end.column = this.doc.getLine(len).length;
	        } else {
	            range.end.column = this.$clipColumnToRow(
	                range.end.row,
	                range.end.column
	            );
	        }
	        return range;
	    };
	    this.$wrapLimit = 80;
	    this.$useWrapMode = false;
	    this.$wrapLimitRange = {
	        min : null,
	        max : null
	    };
	    this.setUseWrapMode = function(useWrapMode) {
	        if (useWrapMode != this.$useWrapMode) {
	            this.$useWrapMode = useWrapMode;
	            this.$modified = true;
	            this.$resetRowCache(0);
	            if (useWrapMode) {
	                var len = this.getLength();
	                this.$wrapData = Array(len);
	                this.$updateWrapData(0, len - 1);
	            }

	            this._signal("changeWrapMode");
	        }
	    };
	    this.getUseWrapMode = function() {
	        return this.$useWrapMode;
	    };
	    this.setWrapLimitRange = function(min, max) {
	        if (this.$wrapLimitRange.min !== min || this.$wrapLimitRange.max !== max) {
	            this.$wrapLimitRange = { min: min, max: max };
	            this.$modified = true;
	            this.$bidiHandler.markAsDirty();
	            if (this.$useWrapMode)
	                this._signal("changeWrapMode");
	        }
	    };
	    this.adjustWrapLimit = function(desiredLimit, $printMargin) {
	        var limits = this.$wrapLimitRange;
	        if (limits.max < 0)
	            limits = {min: $printMargin, max: $printMargin};
	        var wrapLimit = this.$constrainWrapLimit(desiredLimit, limits.min, limits.max);
	        if (wrapLimit != this.$wrapLimit && wrapLimit > 1) {
	            this.$wrapLimit = wrapLimit;
	            this.$modified = true;
	            if (this.$useWrapMode) {
	                this.$updateWrapData(0, this.getLength() - 1);
	                this.$resetRowCache(0);
	                this._signal("changeWrapLimit");
	            }
	            return true;
	        }
	        return false;
	    };

	    this.$constrainWrapLimit = function(wrapLimit, min, max) {
	        if (min)
	            wrapLimit = Math.max(min, wrapLimit);

	        if (max)
	            wrapLimit = Math.min(max, wrapLimit);

	        return wrapLimit;
	    };
	    this.getWrapLimit = function() {
	        return this.$wrapLimit;
	    };
	    this.setWrapLimit = function (limit) {
	        this.setWrapLimitRange(limit, limit);
	    };
	    this.getWrapLimitRange = function() {
	        return {
	            min : this.$wrapLimitRange.min,
	            max : this.$wrapLimitRange.max
	        };
	    };

	    this.$updateInternalDataOnChange = function(delta) {
	        var useWrapMode = this.$useWrapMode;
	        var action = delta.action;
	        var start = delta.start;
	        var end = delta.end;
	        var firstRow = start.row;
	        var lastRow = end.row;
	        var len = lastRow - firstRow;
	        var removedFolds = null;
	        
	        this.$updating = true;
	        if (len != 0) {
	            if (action === "remove") {
	                this[useWrapMode ? "$wrapData" : "$rowLengthCache"].splice(firstRow, len);

	                var foldLines = this.$foldData;
	                removedFolds = this.getFoldsInRange(delta);
	                this.removeFolds(removedFolds);

	                var foldLine = this.getFoldLine(end.row);
	                var idx = 0;
	                if (foldLine) {
	                    foldLine.addRemoveChars(end.row, end.column, start.column - end.column);
	                    foldLine.shiftRow(-len);

	                    var foldLineBefore = this.getFoldLine(firstRow);
	                    if (foldLineBefore && foldLineBefore !== foldLine) {
	                        foldLineBefore.merge(foldLine);
	                        foldLine = foldLineBefore;
	                    }
	                    idx = foldLines.indexOf(foldLine) + 1;
	                }

	                for (idx; idx < foldLines.length; idx++) {
	                    var foldLine = foldLines[idx];
	                    if (foldLine.start.row >= end.row) {
	                        foldLine.shiftRow(-len);
	                    }
	                }

	                lastRow = firstRow;
	            } else {
	                var args = Array(len);
	                args.unshift(firstRow, 0);
	                var arr = useWrapMode ? this.$wrapData : this.$rowLengthCache;
	                arr.splice.apply(arr, args);
	                var foldLines = this.$foldData;
	                var foldLine = this.getFoldLine(firstRow);
	                var idx = 0;
	                if (foldLine) {
	                    var cmp = foldLine.range.compareInside(start.row, start.column);
	                    if (cmp == 0) {
	                        foldLine = foldLine.split(start.row, start.column);
	                        if (foldLine) {
	                            foldLine.shiftRow(len);
	                            foldLine.addRemoveChars(lastRow, 0, end.column - start.column);
	                        }
	                    } else
	                    if (cmp == -1) {
	                        foldLine.addRemoveChars(firstRow, 0, end.column - start.column);
	                        foldLine.shiftRow(len);
	                    }
	                    idx = foldLines.indexOf(foldLine) + 1;
	                }

	                for (idx; idx < foldLines.length; idx++) {
	                    var foldLine = foldLines[idx];
	                    if (foldLine.start.row >= firstRow) {
	                        foldLine.shiftRow(len);
	                    }
	                }
	            }
	        } else {
	            len = Math.abs(delta.start.column - delta.end.column);
	            if (action === "remove") {
	                removedFolds = this.getFoldsInRange(delta);
	                this.removeFolds(removedFolds);

	                len = -len;
	            }
	            var foldLine = this.getFoldLine(firstRow);
	            if (foldLine) {
	                foldLine.addRemoveChars(firstRow, start.column, len);
	            }
	        }

	        if (useWrapMode && this.$wrapData.length != this.doc.getLength()) {
	            console.error("doc.getLength() and $wrapData.length have to be the same!");
	        }
	        this.$updating = false;

	        if (useWrapMode)
	            this.$updateWrapData(firstRow, lastRow);
	        else
	            this.$updateRowLengthCache(firstRow, lastRow);

	        return removedFolds;
	    };

	    this.$updateRowLengthCache = function(firstRow, lastRow, b) {
	        this.$rowLengthCache[firstRow] = null;
	        this.$rowLengthCache[lastRow] = null;
	    };

	    this.$updateWrapData = function(firstRow, lastRow) {
	        var lines = this.doc.getAllLines();
	        var tabSize = this.getTabSize();
	        var wrapData = this.$wrapData;
	        var wrapLimit = this.$wrapLimit;
	        var tokens;
	        var foldLine;

	        var row = firstRow;
	        lastRow = Math.min(lastRow, lines.length - 1);
	        while (row <= lastRow) {
	            foldLine = this.getFoldLine(row, foldLine);
	            if (!foldLine) {
	                tokens = this.$getDisplayTokens(lines[row]);
	                wrapData[row] = this.$computeWrapSplits(tokens, wrapLimit, tabSize);
	                row ++;
	            } else {
	                tokens = [];
	                foldLine.walk(function(placeholder, row, column, lastColumn) {
	                        var walkTokens;
	                        if (placeholder != null) {
	                            walkTokens = this.$getDisplayTokens(
	                                            placeholder, tokens.length);
	                            walkTokens[0] = PLACEHOLDER_START;
	                            for (var i = 1; i < walkTokens.length; i++) {
	                                walkTokens[i] = PLACEHOLDER_BODY;
	                            }
	                        } else {
	                            walkTokens = this.$getDisplayTokens(
	                                lines[row].substring(lastColumn, column),
	                                tokens.length);
	                        }
	                        tokens = tokens.concat(walkTokens);
	                    }.bind(this),
	                    foldLine.end.row,
	                    lines[foldLine.end.row].length + 1
	                );

	                wrapData[foldLine.start.row] = this.$computeWrapSplits(tokens, wrapLimit, tabSize);
	                row = foldLine.end.row + 1;
	            }
	        }
	    };
	    var CHAR = 1,
	        CHAR_EXT = 2,
	        PLACEHOLDER_START = 3,
	        PLACEHOLDER_BODY =  4,
	        PUNCTUATION = 9,
	        SPACE = 10,
	        TAB = 11,
	        TAB_SPACE = 12;


	    this.$computeWrapSplits = function(tokens, wrapLimit, tabSize) {
	        if (tokens.length == 0) {
	            return [];
	        }

	        var splits = [];
	        var displayLength = tokens.length;
	        var lastSplit = 0, lastDocSplit = 0;

	        var isCode = this.$wrapAsCode;

	        var indentedSoftWrap = this.$indentedSoftWrap;
	        var maxIndent = wrapLimit <= Math.max(2 * tabSize, 8)
	            || indentedSoftWrap === false ? 0 : Math.floor(wrapLimit / 2);

	        function getWrapIndent() {
	            var indentation = 0;
	            if (maxIndent === 0)
	                return indentation;
	            if (indentedSoftWrap) {
	                for (var i = 0; i < tokens.length; i++) {
	                    var token = tokens[i];
	                    if (token == SPACE)
	                        indentation += 1;
	                    else if (token == TAB)
	                        indentation += tabSize;
	                    else if (token == TAB_SPACE)
	                        continue;
	                    else
	                        break;
	                }
	            }
	            if (isCode && indentedSoftWrap !== false)
	                indentation += tabSize;
	            return Math.min(indentation, maxIndent);
	        }
	        function addSplit(screenPos) {
	            var displayed = tokens.slice(lastSplit, screenPos);
	            var len = displayed.length;
	            displayed.join("")
	                .replace(/12/g, function() {
	                    len -= 1;
	                })
	                .replace(/2/g, function() {
	                    len -= 1;
	                });

	            if (!splits.length) {
	                indent = getWrapIndent();
	                splits.indent = indent;
	            }
	            lastDocSplit += len;
	            splits.push(lastDocSplit);
	            lastSplit = screenPos;
	        }
	        var indent = 0;
	        while (displayLength - lastSplit > wrapLimit - indent) {
	            var split = lastSplit + wrapLimit - indent;
	            if (tokens[split - 1] >= SPACE && tokens[split] >= SPACE) {
	                addSplit(split);
	                continue;
	            }
	            if (tokens[split] == PLACEHOLDER_START || tokens[split] == PLACEHOLDER_BODY) {
	                for (split; split != lastSplit - 1; split--) {
	                    if (tokens[split] == PLACEHOLDER_START) {
	                        break;
	                    }
	                }
	                if (split > lastSplit) {
	                    addSplit(split);
	                    continue;
	                }
	                split = lastSplit + wrapLimit;
	                for (split; split < tokens.length; split++) {
	                    if (tokens[split] != PLACEHOLDER_BODY) {
	                        break;
	                    }
	                }
	                if (split == tokens.length) {
	                    break;  // Breaks the while-loop.
	                }
	                addSplit(split);
	                continue;
	            }
	            var minSplit = Math.max(split - (wrapLimit -(wrapLimit>>2)), lastSplit - 1);
	            while (split > minSplit && tokens[split] < PLACEHOLDER_START) {
	                split --;
	            }
	            if (isCode) {
	                while (split > minSplit && tokens[split] < PLACEHOLDER_START) {
	                    split --;
	                }
	                while (split > minSplit && tokens[split] == PUNCTUATION) {
	                    split --;
	                }
	            } else {
	                while (split > minSplit && tokens[split] < SPACE) {
	                    split --;
	                }
	            }
	            if (split > minSplit) {
	                addSplit(++split);
	                continue;
	            }
	            split = lastSplit + wrapLimit;
	            if (tokens[split] == CHAR_EXT)
	                split--;
	            addSplit(split - indent);
	        }
	        return splits;
	    };
	    this.$getDisplayTokens = function(str, offset) {
	        var arr = [];
	        var tabSize;
	        offset = offset || 0;

	        for (var i = 0; i < str.length; i++) {
	            var c = str.charCodeAt(i);
	            if (c == 9) {
	                tabSize = this.getScreenTabSize(arr.length + offset);
	                arr.push(TAB);
	                for (var n = 1; n < tabSize; n++) {
	                    arr.push(TAB_SPACE);
	                }
	            }
	            else if (c == 32) {
	                arr.push(SPACE);
	            } else if((c > 39 && c < 48) || (c > 57 && c < 64)) {
	                arr.push(PUNCTUATION);
	            }
	            else if (c >= 0x1100 && isFullWidth(c)) {
	                arr.push(CHAR, CHAR_EXT);
	            } else {
	                arr.push(CHAR);
	            }
	        }
	        return arr;
	    };
	    this.$getStringScreenWidth = function(str, maxScreenColumn, screenColumn) {
	        if (maxScreenColumn == 0)
	            return [0, 0];
	        if (maxScreenColumn == null)
	            maxScreenColumn = Infinity;
	        screenColumn = screenColumn || 0;

	        var c, column;
	        for (column = 0; column < str.length; column++) {
	            c = str.charCodeAt(column);
	            if (c == 9) {
	                screenColumn += this.getScreenTabSize(screenColumn);
	            }
	            else if (c >= 0x1100 && isFullWidth(c)) {
	                screenColumn += 2;
	            } else {
	                screenColumn += 1;
	            }
	            if (screenColumn > maxScreenColumn) {
	                break;
	            }
	        }

	        return [screenColumn, column];
	    };

	    this.lineWidgets = null;
	    this.getRowLength = function(row) {
	        if (this.lineWidgets)
	            var h = this.lineWidgets[row] && this.lineWidgets[row].rowCount || 0;
	        else 
	            h = 0;
	        if (!this.$useWrapMode || !this.$wrapData[row]) {
	            return 1 + h;
	        } else {
	            return this.$wrapData[row].length + 1 + h;
	        }
	    };
	    this.getRowLineCount = function(row) {
	        if (!this.$useWrapMode || !this.$wrapData[row]) {
	            return 1;
	        } else {
	            return this.$wrapData[row].length + 1;
	        }
	    };

	    this.getRowWrapIndent = function(screenRow) {
	        if (this.$useWrapMode) {
	            var pos = this.screenToDocumentPosition(screenRow, Number.MAX_VALUE);
	            var splits = this.$wrapData[pos.row];
	            return splits.length && splits[0] < pos.column ? splits.indent : 0;
	        } else {
	            return 0;
	        }
	    };
	    this.getScreenLastRowColumn = function(screenRow) {
	        var pos = this.screenToDocumentPosition(screenRow, Number.MAX_VALUE);
	        return this.documentToScreenColumn(pos.row, pos.column);
	    };
	    this.getDocumentLastRowColumn = function(docRow, docColumn) {
	        var screenRow = this.documentToScreenRow(docRow, docColumn);
	        return this.getScreenLastRowColumn(screenRow);
	    };
	    this.getDocumentLastRowColumnPosition = function(docRow, docColumn) {
	        var screenRow = this.documentToScreenRow(docRow, docColumn);
	        return this.screenToDocumentPosition(screenRow, Number.MAX_VALUE / 10);
	    };
	    this.getRowSplitData = function(row) {
	        if (!this.$useWrapMode) {
	            return undefined;
	        } else {
	            return this.$wrapData[row];
	        }
	    };
	    this.getScreenTabSize = function(screenColumn) {
	        return this.$tabSize - screenColumn % this.$tabSize;
	    };


	    this.screenToDocumentRow = function(screenRow, screenColumn) {
	        return this.screenToDocumentPosition(screenRow, screenColumn).row;
	    };


	    this.screenToDocumentColumn = function(screenRow, screenColumn) {
	        return this.screenToDocumentPosition(screenRow, screenColumn).column;
	    };
	    this.screenToDocumentPosition = function(screenRow, screenColumn, offsetX) {
	        if (screenRow < 0)
	            return {row: 0, column: 0};

	        var line;
	        var docRow = 0;
	        var docColumn = 0;
	        var column;
	        var row = 0;
	        var rowLength = 0;

	        var rowCache = this.$screenRowCache;
	        var i = this.$getRowCacheIndex(rowCache, screenRow);
	        var l = rowCache.length;
	        if (l && i >= 0) {
	            var row = rowCache[i];
	            var docRow = this.$docRowCache[i];
	            var doCache = screenRow > rowCache[l - 1];
	        } else {
	            var doCache = !l;
	        }

	        var maxRow = this.getLength() - 1;
	        var foldLine = this.getNextFoldLine(docRow);
	        var foldStart = foldLine ? foldLine.start.row : Infinity;

	        while (row <= screenRow) {
	            rowLength = this.getRowLength(docRow);
	            if (row + rowLength > screenRow || docRow >= maxRow) {
	                break;
	            } else {
	                row += rowLength;
	                docRow++;
	                if (docRow > foldStart) {
	                    docRow = foldLine.end.row+1;
	                    foldLine = this.getNextFoldLine(docRow, foldLine);
	                    foldStart = foldLine ? foldLine.start.row : Infinity;
	                }
	            }

	            if (doCache) {
	                this.$docRowCache.push(docRow);
	                this.$screenRowCache.push(row);
	            }
	        }

	        if (foldLine && foldLine.start.row <= docRow) {
	            line = this.getFoldDisplayLine(foldLine);
	            docRow = foldLine.start.row;
	        } else if (row + rowLength <= screenRow || docRow > maxRow) {
	            return {
	                row: maxRow,
	                column: this.getLine(maxRow).length
	            };
	        } else {
	            line = this.getLine(docRow);
	            foldLine = null;
	        }
	        var wrapIndent = 0, splitIndex = Math.floor(screenRow - row);
	        if (this.$useWrapMode) {
	            var splits = this.$wrapData[docRow];
	            if (splits) {
	                column = splits[splitIndex];
	                if(splitIndex > 0 && splits.length) {
	                    wrapIndent = splits.indent;
	                    docColumn = splits[splitIndex - 1] || splits[splits.length - 1];
	                    line = line.substring(docColumn);
	                }
	            }
	        }

	        if (offsetX !== undefined && this.$bidiHandler.isBidiRow(row + splitIndex, docRow, splitIndex))
	            screenColumn = this.$bidiHandler.offsetToCol(offsetX);

	        docColumn += this.$getStringScreenWidth(line, screenColumn - wrapIndent)[1];
	        if (this.$useWrapMode && docColumn >= column)
	            docColumn = column - 1;

	        if (foldLine)
	            return foldLine.idxToPosition(docColumn);

	        return {row: docRow, column: docColumn};
	    };
	    this.documentToScreenPosition = function(docRow, docColumn) {
	        if (typeof docColumn === "undefined")
	            var pos = this.$clipPositionToDocument(docRow.row, docRow.column);
	        else
	            pos = this.$clipPositionToDocument(docRow, docColumn);

	        docRow = pos.row;
	        docColumn = pos.column;

	        var screenRow = 0;
	        var foldStartRow = null;
	        var fold = null;
	        fold = this.getFoldAt(docRow, docColumn, 1);
	        if (fold) {
	            docRow = fold.start.row;
	            docColumn = fold.start.column;
	        }

	        var rowEnd, row = 0;


	        var rowCache = this.$docRowCache;
	        var i = this.$getRowCacheIndex(rowCache, docRow);
	        var l = rowCache.length;
	        if (l && i >= 0) {
	            var row = rowCache[i];
	            var screenRow = this.$screenRowCache[i];
	            var doCache = docRow > rowCache[l - 1];
	        } else {
	            var doCache = !l;
	        }

	        var foldLine = this.getNextFoldLine(row);
	        var foldStart = foldLine ?foldLine.start.row :Infinity;

	        while (row < docRow) {
	            if (row >= foldStart) {
	                rowEnd = foldLine.end.row + 1;
	                if (rowEnd > docRow)
	                    break;
	                foldLine = this.getNextFoldLine(rowEnd, foldLine);
	                foldStart = foldLine ?foldLine.start.row :Infinity;
	            }
	            else {
	                rowEnd = row + 1;
	            }

	            screenRow += this.getRowLength(row);
	            row = rowEnd;

	            if (doCache) {
	                this.$docRowCache.push(row);
	                this.$screenRowCache.push(screenRow);
	            }
	        }
	        var textLine = "";
	        if (foldLine && row >= foldStart) {
	            textLine = this.getFoldDisplayLine(foldLine, docRow, docColumn);
	            foldStartRow = foldLine.start.row;
	        } else {
	            textLine = this.getLine(docRow).substring(0, docColumn);
	            foldStartRow = docRow;
	        }
	        var wrapIndent = 0;
	        if (this.$useWrapMode) {
	            var wrapRow = this.$wrapData[foldStartRow];
	            if (wrapRow) {
	                var screenRowOffset = 0;
	                while (textLine.length >= wrapRow[screenRowOffset]) {
	                    screenRow ++;
	                    screenRowOffset++;
	                }
	                textLine = textLine.substring(
	                    wrapRow[screenRowOffset - 1] || 0, textLine.length
	                );
	                wrapIndent = screenRowOffset > 0 ? wrapRow.indent : 0;
	            }
	        }

	        return {
	            row: screenRow,
	            column: wrapIndent + this.$getStringScreenWidth(textLine)[0]
	        };
	    };
	    this.documentToScreenColumn = function(row, docColumn) {
	        return this.documentToScreenPosition(row, docColumn).column;
	    };
	    this.documentToScreenRow = function(docRow, docColumn) {
	        return this.documentToScreenPosition(docRow, docColumn).row;
	    };
	    this.getScreenLength = function() {
	        var screenRows = 0;
	        var fold = null;
	        if (!this.$useWrapMode) {
	            screenRows = this.getLength();
	            var foldData = this.$foldData;
	            for (var i = 0; i < foldData.length; i++) {
	                fold = foldData[i];
	                screenRows -= fold.end.row - fold.start.row;
	            }
	        } else {
	            var lastRow = this.$wrapData.length;
	            var row = 0, i = 0;
	            var fold = this.$foldData[i++];
	            var foldStart = fold ? fold.start.row :Infinity;

	            while (row < lastRow) {
	                var splits = this.$wrapData[row];
	                screenRows += splits ? splits.length + 1 : 1;
	                row ++;
	                if (row > foldStart) {
	                    row = fold.end.row+1;
	                    fold = this.$foldData[i++];
	                    foldStart = fold ?fold.start.row :Infinity;
	                }
	            }
	        }
	        if (this.lineWidgets)
	            screenRows += this.$getWidgetScreenLength();

	        return screenRows;
	    };
	    this.$setFontMetrics = function(fm) {
	        if (!this.$enableVarChar) return;
	        this.$getStringScreenWidth = function(str, maxScreenColumn, screenColumn) {
	            if (maxScreenColumn === 0)
	                return [0, 0];
	            if (!maxScreenColumn)
	                maxScreenColumn = Infinity;
	            screenColumn = screenColumn || 0;
	            
	            var c, column;
	            for (column = 0; column < str.length; column++) {
	                c = str.charAt(column);
	                if (c === "\t") {
	                    screenColumn += this.getScreenTabSize(screenColumn);
	                } else {
	                    screenColumn += fm.getCharacterWidth(c);
	                }
	                if (screenColumn > maxScreenColumn) {
	                    break;
	                }
	            }
	            
	            return [screenColumn, column];
	        };
	    };
	    
	    this.destroy = function() {
	        if (this.bgTokenizer) {
	            this.bgTokenizer.setDocument(null);
	            this.bgTokenizer = null;
	        }
	        this.$stopWorker();
	    };

	    this.isFullWidth = isFullWidth;
	    function isFullWidth(c) {
	        if (c < 0x1100)
	            return false;
	        return c >= 0x1100 && c <= 0x115F ||
	               c >= 0x11A3 && c <= 0x11A7 ||
	               c >= 0x11FA && c <= 0x11FF ||
	               c >= 0x2329 && c <= 0x232A ||
	               c >= 0x2E80 && c <= 0x2E99 ||
	               c >= 0x2E9B && c <= 0x2EF3 ||
	               c >= 0x2F00 && c <= 0x2FD5 ||
	               c >= 0x2FF0 && c <= 0x2FFB ||
	               c >= 0x3000 && c <= 0x303E ||
	               c >= 0x3041 && c <= 0x3096 ||
	               c >= 0x3099 && c <= 0x30FF ||
	               c >= 0x3105 && c <= 0x312D ||
	               c >= 0x3131 && c <= 0x318E ||
	               c >= 0x3190 && c <= 0x31BA ||
	               c >= 0x31C0 && c <= 0x31E3 ||
	               c >= 0x31F0 && c <= 0x321E ||
	               c >= 0x3220 && c <= 0x3247 ||
	               c >= 0x3250 && c <= 0x32FE ||
	               c >= 0x3300 && c <= 0x4DBF ||
	               c >= 0x4E00 && c <= 0xA48C ||
	               c >= 0xA490 && c <= 0xA4C6 ||
	               c >= 0xA960 && c <= 0xA97C ||
	               c >= 0xAC00 && c <= 0xD7A3 ||
	               c >= 0xD7B0 && c <= 0xD7C6 ||
	               c >= 0xD7CB && c <= 0xD7FB ||
	               c >= 0xF900 && c <= 0xFAFF ||
	               c >= 0xFE10 && c <= 0xFE19 ||
	               c >= 0xFE30 && c <= 0xFE52 ||
	               c >= 0xFE54 && c <= 0xFE66 ||
	               c >= 0xFE68 && c <= 0xFE6B ||
	               c >= 0xFF01 && c <= 0xFF60 ||
	               c >= 0xFFE0 && c <= 0xFFE6;
	    }

	}).call(EditSession.prototype);

	acequire("./edit_session/folding").Folding.call(EditSession.prototype);
	acequire("./edit_session/bracket_match").BracketMatch.call(EditSession.prototype);


	config.defineOptions(EditSession.prototype, "session", {
	    wrap: {
	        set: function(value) {
	            if (!value || value == "off")
	                value = false;
	            else if (value == "free")
	                value = true;
	            else if (value == "printMargin")
	                value = -1;
	            else if (typeof value == "string")
	                value = parseInt(value, 10) || false;

	            if (this.$wrap == value)
	                return;
	            this.$wrap = value;
	            if (!value) {
	                this.setUseWrapMode(false);
	            } else {
	                var col = typeof value == "number" ? value : null;
	                this.setWrapLimitRange(col, col);
	                this.setUseWrapMode(true);
	            }
	        },
	        get: function() {
	            if (this.getUseWrapMode()) {
	                if (this.$wrap == -1)
	                    return "printMargin";
	                if (!this.getWrapLimitRange().min)
	                    return "free";
	                return this.$wrap;
	            }
	            return "off";
	        },
	        handlesSet: true
	    },    
	    wrapMethod: {
	        set: function(val) {
	            val = val == "auto"
	                ? this.$mode.type != "text"
	                : val != "text";
	            if (val != this.$wrapAsCode) {
	                this.$wrapAsCode = val;
	                if (this.$useWrapMode) {
	                    this.$modified = true;
	                    this.$resetRowCache(0);
	                    this.$updateWrapData(0, this.getLength() - 1);
	                }
	            }
	        },
	        initialValue: "auto"
	    },
	    indentedSoftWrap: { initialValue: true },
	    firstLineNumber: {
	        set: function() {this._signal("changeBreakpoint");},
	        initialValue: 1
	    },
	    useWorker: {
	        set: function(useWorker) {
	            this.$useWorker = useWorker;

	            this.$stopWorker();
	            if (useWorker)
	                this.$startWorker();
	        },
	        initialValue: true
	    },
	    useSoftTabs: {initialValue: true},
	    tabSize: {
	        set: function(tabSize) {
	            if (isNaN(tabSize) || this.$tabSize === tabSize) return;

	            this.$modified = true;
	            this.$rowLengthCache = [];
	            this.$tabSize = tabSize;
	            this._signal("changeTabSize");
	        },
	        initialValue: 4,
	        handlesSet: true
	    },
	    navigateWithinSoftTabs: {initialValue: false},
	    overwrite: {
	        set: function(val) {this._signal("changeOverwrite");},
	        initialValue: false
	    },
	    newLineMode: {
	        set: function(val) {this.doc.setNewLineMode(val);},
	        get: function() {return this.doc.getNewLineMode();},
	        handlesSet: true
	    },
	    mode: {
	        set: function(val) { this.setMode(val); },
	        get: function() { return this.$modeId; }
	    }
	});

	exports.EditSession = EditSession;
	});

	ace.define("ace/search",["require","exports","module","ace/lib/lang","ace/lib/oop","ace/range"], function(acequire, exports, module) {
	"use strict";

	var lang = acequire("./lib/lang");
	var oop = acequire("./lib/oop");
	var Range = acequire("./range").Range;

	var Search = function() {
	    this.$options = {};
	};

	(function() {
	    this.set = function(options) {
	        oop.mixin(this.$options, options);
	        return this;
	    };
	    this.getOptions = function() {
	        return lang.copyObject(this.$options);
	    };
	    this.setOptions = function(options) {
	        this.$options = options;
	    };
	    this.find = function(session) {
	        var options = this.$options;
	        var iterator = this.$matchIterator(session, options);
	        if (!iterator)
	            return false;

	        var firstRange = null;
	        iterator.forEach(function(sr, sc, er, ec) {
	            firstRange = new Range(sr, sc, er, ec);
	            if (sc == ec && options.start && options.start.start
	                && options.skipCurrent != false && firstRange.isEqual(options.start)
	            ) {
	                firstRange = null;
	                return false;
	            }

	            return true;
	        });

	        return firstRange;
	    };
	    this.findAll = function(session) {
	        var options = this.$options;
	        if (!options.needle)
	            return [];
	        this.$assembleRegExp(options);

	        var range = options.range;
	        var lines = range
	            ? session.getLines(range.start.row, range.end.row)
	            : session.doc.getAllLines();

	        var ranges = [];
	        var re = options.re;
	        if (options.$isMultiLine) {
	            var len = re.length;
	            var maxRow = lines.length - len;
	            var prevRange;
	            outer: for (var row = re.offset || 0; row <= maxRow; row++) {
	                for (var j = 0; j < len; j++)
	                    if (lines[row + j].search(re[j]) == -1)
	                        continue outer;
	                
	                var startLine = lines[row];
	                var line = lines[row + len - 1];
	                var startIndex = startLine.length - startLine.match(re[0])[0].length;
	                var endIndex = line.match(re[len - 1])[0].length;
	                
	                if (prevRange && prevRange.end.row === row &&
	                    prevRange.end.column > startIndex
	                ) {
	                    continue;
	                }
	                ranges.push(prevRange = new Range(
	                    row, startIndex, row + len - 1, endIndex
	                ));
	                if (len > 2)
	                    row = row + len - 2;
	            }
	        } else {
	            for (var i = 0; i < lines.length; i++) {
	                var matches = lang.getMatchOffsets(lines[i], re);
	                for (var j = 0; j < matches.length; j++) {
	                    var match = matches[j];
	                    ranges.push(new Range(i, match.offset, i, match.offset + match.length));
	                }
	            }
	        }

	        if (range) {
	            var startColumn = range.start.column;
	            var endColumn = range.start.column;
	            var i = 0, j = ranges.length - 1;
	            while (i < j && ranges[i].start.column < startColumn && ranges[i].start.row == range.start.row)
	                i++;

	            while (i < j && ranges[j].end.column > endColumn && ranges[j].end.row == range.end.row)
	                j--;
	            
	            ranges = ranges.slice(i, j + 1);
	            for (i = 0, j = ranges.length; i < j; i++) {
	                ranges[i].start.row += range.start.row;
	                ranges[i].end.row += range.start.row;
	            }
	        }

	        return ranges;
	    };
	    this.replace = function(input, replacement) {
	        var options = this.$options;

	        var re = this.$assembleRegExp(options);
	        if (options.$isMultiLine)
	            return replacement;

	        if (!re)
	            return;

	        var match = re.exec(input);
	        if (!match || match[0].length != input.length)
	            return null;
	        
	        replacement = input.replace(re, replacement);
	        if (options.preserveCase) {
	            replacement = replacement.split("");
	            for (var i = Math.min(input.length, input.length); i--; ) {
	                var ch = input[i];
	                if (ch && ch.toLowerCase() != ch)
	                    replacement[i] = replacement[i].toUpperCase();
	                else
	                    replacement[i] = replacement[i].toLowerCase();
	            }
	            replacement = replacement.join("");
	        }
	        
	        return replacement;
	    };

	    this.$assembleRegExp = function(options, $disableFakeMultiline) {
	        if (options.needle instanceof RegExp)
	            return options.re = options.needle;

	        var needle = options.needle;

	        if (!options.needle)
	            return options.re = false;

	        if (!options.regExp)
	            needle = lang.escapeRegExp(needle);

	        if (options.wholeWord)
	            needle = addWordBoundary(needle, options);

	        var modifier = options.caseSensitive ? "gm" : "gmi";

	        options.$isMultiLine = !$disableFakeMultiline && /[\n\r]/.test(needle);
	        if (options.$isMultiLine)
	            return options.re = this.$assembleMultilineRegExp(needle, modifier);

	        try {
	            var re = new RegExp(needle, modifier);
	        } catch(e) {
	            re = false;
	        }
	        return options.re = re;
	    };

	    this.$assembleMultilineRegExp = function(needle, modifier) {
	        var parts = needle.replace(/\r\n|\r|\n/g, "$\n^").split("\n");
	        var re = [];
	        for (var i = 0; i < parts.length; i++) try {
	            re.push(new RegExp(parts[i], modifier));
	        } catch(e) {
	            return false;
	        }
	        return re;
	    };

	    this.$matchIterator = function(session, options) {
	        var re = this.$assembleRegExp(options);
	        if (!re)
	            return false;
	        var backwards = options.backwards == true;
	        var skipCurrent = options.skipCurrent != false;

	        var range = options.range;
	        var start = options.start;
	        if (!start)
	            start = range ? range[backwards ? "end" : "start"] : session.selection.getRange();
	         
	        if (start.start)
	            start = start[skipCurrent != backwards ? "end" : "start"];

	        var firstRow = range ? range.start.row : 0;
	        var lastRow = range ? range.end.row : session.getLength() - 1;

	        if (backwards) {
	            var forEach = function(callback) {
	                var row = start.row;
	                if (forEachInLine(row, start.column, callback))
	                    return;
	                for (row--; row >= firstRow; row--)
	                    if (forEachInLine(row, Number.MAX_VALUE, callback))
	                        return;
	                if (options.wrap == false)
	                    return;
	                for (row = lastRow, firstRow = start.row; row >= firstRow; row--)
	                    if (forEachInLine(row, Number.MAX_VALUE, callback))
	                        return;
	            };
	        }
	        else {
	            var forEach = function(callback) {
	                var row = start.row;
	                if (forEachInLine(row, start.column, callback))
	                    return;
	                for (row = row + 1; row <= lastRow; row++)
	                    if (forEachInLine(row, 0, callback))
	                        return;
	                if (options.wrap == false)
	                    return;
	                for (row = firstRow, lastRow = start.row; row <= lastRow; row++)
	                    if (forEachInLine(row, 0, callback))
	                        return;
	            };
	        }
	        
	        if (options.$isMultiLine) {
	            var len = re.length;
	            var forEachInLine = function(row, offset, callback) {
	                var startRow = backwards ? row - len + 1 : row;
	                if (startRow < 0) return;
	                var line = session.getLine(startRow);
	                var startIndex = line.search(re[0]);
	                if (!backwards && startIndex < offset || startIndex === -1) return;
	                for (var i = 1; i < len; i++) {
	                    line = session.getLine(startRow + i);
	                    if (line.search(re[i]) == -1)
	                        return;
	                }
	                var endIndex = line.match(re[len - 1])[0].length;
	                if (backwards && endIndex > offset) return;
	                if (callback(startRow, startIndex, startRow + len - 1, endIndex))
	                    return true;
	            };
	        }
	        else if (backwards) {
	            var forEachInLine = function(row, endIndex, callback) {
	                var line = session.getLine(row);
	                var matches = [];
	                var m, last = 0;
	                re.lastIndex = 0;
	                while((m = re.exec(line))) {
	                    var length = m[0].length;
	                    last = m.index;
	                    if (!length) {
	                        if (last >= line.length) break;
	                        re.lastIndex = last += 1;
	                    }
	                    if (m.index + length > endIndex)
	                        break;
	                    matches.push(m.index, length);
	                }
	                for (var i = matches.length - 1; i >= 0; i -= 2) {
	                    var column = matches[i - 1];
	                    var length = matches[i];
	                    if (callback(row, column, row, column + length))
	                        return true;
	                }
	            };
	        }
	        else {
	            var forEachInLine = function(row, startIndex, callback) {
	                var line = session.getLine(row);
	                var m;
	                var last = startIndex;
	                re.lastIndex = startIndex;
	                while((m = re.exec(line))) {
	                    var length = m[0].length;
	                    last = m.index;
	                    if (callback(row, last, row,last + length))
	                        return true;
	                    if (!length) {
	                        re.lastIndex = last += 1;
	                        if (last >= line.length) return false;
	                    }
	                }
	            };
	        }
	        return {forEach: forEach};
	    };

	}).call(Search.prototype);

	function addWordBoundary(needle, options) {
	    function wordBoundary(c) {
	        if (/\w/.test(c) || options.regExp) return "\\b";
	        return "";
	    }
	    return wordBoundary(needle[0]) + needle
	        + wordBoundary(needle[needle.length - 1]);
	}

	exports.Search = Search;
	});

	ace.define("ace/keyboard/hash_handler",["require","exports","module","ace/lib/keys","ace/lib/useragent"], function(acequire, exports, module) {
	"use strict";

	var keyUtil = acequire("../lib/keys");
	var useragent = acequire("../lib/useragent");
	var KEY_MODS = keyUtil.KEY_MODS;

	function HashHandler(config, platform) {
	    this.platform = platform || (useragent.isMac ? "mac" : "win");
	    this.commands = {};
	    this.commandKeyBinding = {};
	    this.addCommands(config);
	    this.$singleCommand = true;
	}

	function MultiHashHandler(config, platform) {
	    HashHandler.call(this, config, platform);
	    this.$singleCommand = false;
	}

	MultiHashHandler.prototype = HashHandler.prototype;

	(function() {
	    

	    this.addCommand = function(command) {
	        if (this.commands[command.name])
	            this.removeCommand(command);

	        this.commands[command.name] = command;

	        if (command.bindKey)
	            this._buildKeyHash(command);
	    };

	    this.removeCommand = function(command, keepCommand) {
	        var name = command && (typeof command === 'string' ? command : command.name);
	        command = this.commands[name];
	        if (!keepCommand)
	            delete this.commands[name];
	        var ckb = this.commandKeyBinding;
	        for (var keyId in ckb) {
	            var cmdGroup = ckb[keyId];
	            if (cmdGroup == command) {
	                delete ckb[keyId];
	            } else if (Array.isArray(cmdGroup)) {
	                var i = cmdGroup.indexOf(command);
	                if (i != -1) {
	                    cmdGroup.splice(i, 1);
	                    if (cmdGroup.length == 1)
	                        ckb[keyId] = cmdGroup[0];
	                }
	            }
	        }
	    };

	    this.bindKey = function(key, command, position) {
	        if (typeof key == "object" && key) {
	            if (position == undefined)
	                position = key.position;
	            key = key[this.platform];
	        }
	        if (!key)
	            return;
	        if (typeof command == "function")
	            return this.addCommand({exec: command, bindKey: key, name: command.name || key});
	        
	        key.split("|").forEach(function(keyPart) {
	            var chain = "";
	            if (keyPart.indexOf(" ") != -1) {
	                var parts = keyPart.split(/\s+/);
	                keyPart = parts.pop();
	                parts.forEach(function(keyPart) {
	                    var binding = this.parseKeys(keyPart);
	                    var id = KEY_MODS[binding.hashId] + binding.key;
	                    chain += (chain ? " " : "") + id;
	                    this._addCommandToBinding(chain, "chainKeys");
	                }, this);
	                chain += " ";
	            }
	            var binding = this.parseKeys(keyPart);
	            var id = KEY_MODS[binding.hashId] + binding.key;
	            this._addCommandToBinding(chain + id, command, position);
	        }, this);
	    };
	    
	    function getPosition(command) {
	        return typeof command == "object" && command.bindKey
	            && command.bindKey.position
	            || (command.isDefault ? -100 : 0);
	    }
	    this._addCommandToBinding = function(keyId, command, position) {
	        var ckb = this.commandKeyBinding, i;
	        if (!command) {
	            delete ckb[keyId];
	        } else if (!ckb[keyId] || this.$singleCommand) {
	            ckb[keyId] = command;
	        } else {
	            if (!Array.isArray(ckb[keyId])) {
	                ckb[keyId] = [ckb[keyId]];
	            } else if ((i = ckb[keyId].indexOf(command)) != -1) {
	                ckb[keyId].splice(i, 1);
	            }

	            if (typeof position != "number") {
	                position = getPosition(command);
	            }

	            var commands = ckb[keyId];
	            for (i = 0; i < commands.length; i++) {
	                var other = commands[i];
	                var otherPos = getPosition(other);
	                if (otherPos > position)
	                    break;
	            }
	            commands.splice(i, 0, command);
	        }
	    };

	    this.addCommands = function(commands) {
	        commands && Object.keys(commands).forEach(function(name) {
	            var command = commands[name];
	            if (!command)
	                return;
	            
	            if (typeof command === "string")
	                return this.bindKey(command, name);

	            if (typeof command === "function")
	                command = { exec: command };

	            if (typeof command !== "object")
	                return;

	            if (!command.name)
	                command.name = name;

	            this.addCommand(command);
	        }, this);
	    };

	    this.removeCommands = function(commands) {
	        Object.keys(commands).forEach(function(name) {
	            this.removeCommand(commands[name]);
	        }, this);
	    };

	    this.bindKeys = function(keyList) {
	        Object.keys(keyList).forEach(function(key) {
	            this.bindKey(key, keyList[key]);
	        }, this);
	    };

	    this._buildKeyHash = function(command) {
	        this.bindKey(command.bindKey, command);
	    };
	    this.parseKeys = function(keys) {
	        var parts = keys.toLowerCase().split(/[\-\+]([\-\+])?/).filter(function(x){return x;});
	        var key = parts.pop();

	        var keyCode = keyUtil[key];
	        if (keyUtil.FUNCTION_KEYS[keyCode])
	            key = keyUtil.FUNCTION_KEYS[keyCode].toLowerCase();
	        else if (!parts.length)
	            return {key: key, hashId: -1};
	        else if (parts.length == 1 && parts[0] == "shift")
	            return {key: key.toUpperCase(), hashId: -1};

	        var hashId = 0;
	        for (var i = parts.length; i--;) {
	            var modifier = keyUtil.KEY_MODS[parts[i]];
	            if (modifier == null) {
	                if (typeof console != "undefined")
	                    console.error("invalid modifier " + parts[i] + " in " + keys);
	                return false;
	            }
	            hashId |= modifier;
	        }
	        return {key: key, hashId: hashId};
	    };

	    this.findKeyCommand = function findKeyCommand(hashId, keyString) {
	        var key = KEY_MODS[hashId] + keyString;
	        return this.commandKeyBinding[key];
	    };

	    this.handleKeyboard = function(data, hashId, keyString, keyCode) {
	        if (keyCode < 0) return;
	        var key = KEY_MODS[hashId] + keyString;
	        var command = this.commandKeyBinding[key];
	        if (data.$keyChain) {
	            data.$keyChain += " " + key;
	            command = this.commandKeyBinding[data.$keyChain] || command;
	        }
	        
	        if (command) {
	            if (command == "chainKeys" || command[command.length - 1] == "chainKeys") {
	                data.$keyChain = data.$keyChain || key;
	                return {command: "null"};
	            }
	        }
	        
	        if (data.$keyChain) {
	            if ((!hashId || hashId == 4) && keyString.length == 1)
	                data.$keyChain = data.$keyChain.slice(0, -key.length - 1); // wait for input
	            else if (hashId == -1 || keyCode > 0)
	                data.$keyChain = ""; // reset keyChain
	        }
	        return {command: command};
	    };
	    
	    this.getStatusText = function(editor, data) {
	        return data.$keyChain || "";
	    };

	}).call(HashHandler.prototype);

	exports.HashHandler = HashHandler;
	exports.MultiHashHandler = MultiHashHandler;
	});

	ace.define("ace/commands/command_manager",["require","exports","module","ace/lib/oop","ace/keyboard/hash_handler","ace/lib/event_emitter"], function(acequire, exports, module) {
	"use strict";

	var oop = acequire("../lib/oop");
	var MultiHashHandler = acequire("../keyboard/hash_handler").MultiHashHandler;
	var EventEmitter = acequire("../lib/event_emitter").EventEmitter;

	var CommandManager = function(platform, commands) {
	    MultiHashHandler.call(this, commands, platform);
	    this.byName = this.commands;
	    this.setDefaultHandler("exec", function(e) {
	        return e.command.exec(e.editor, e.args || {});
	    });
	};

	oop.inherits(CommandManager, MultiHashHandler);

	(function() {

	    oop.implement(this, EventEmitter);

	    this.exec = function(command, editor, args) {
	        if (Array.isArray(command)) {
	            for (var i = command.length; i--; ) {
	                if (this.exec(command[i], editor, args)) return true;
	            }
	            return false;
	        }

	        if (typeof command === "string")
	            command = this.commands[command];

	        if (!command)
	            return false;

	        if (editor && editor.$readOnly && !command.readOnly)
	            return false;

	        if (command.isAvailable && !command.isAvailable(editor))
	            return false;

	        var e = {editor: editor, command: command, args: args};
	        e.returnValue = this._emit("exec", e);
	        this._signal("afterExec", e);

	        return e.returnValue === false ? false : true;
	    };

	    this.toggleRecording = function(editor) {
	        if (this.$inReplay)
	            return;

	        editor && editor._emit("changeStatus");
	        if (this.recording) {
	            this.macro.pop();
	            this.removeEventListener("exec", this.$addCommandToMacro);

	            if (!this.macro.length)
	                this.macro = this.oldMacro;

	            return this.recording = false;
	        }
	        if (!this.$addCommandToMacro) {
	            this.$addCommandToMacro = function(e) {
	                this.macro.push([e.command, e.args]);
	            }.bind(this);
	        }

	        this.oldMacro = this.macro;
	        this.macro = [];
	        this.on("exec", this.$addCommandToMacro);
	        return this.recording = true;
	    };

	    this.replay = function(editor) {
	        if (this.$inReplay || !this.macro)
	            return;

	        if (this.recording)
	            return this.toggleRecording(editor);

	        try {
	            this.$inReplay = true;
	            this.macro.forEach(function(x) {
	                if (typeof x == "string")
	                    this.exec(x, editor);
	                else
	                    this.exec(x[0], editor, x[1]);
	            }, this);
	        } finally {
	            this.$inReplay = false;
	        }
	    };

	    this.trimMacro = function(m) {
	        return m.map(function(x){
	            if (typeof x[0] != "string")
	                x[0] = x[0].name;
	            if (!x[1])
	                x = x[0];
	            return x;
	        });
	    };

	}).call(CommandManager.prototype);

	exports.CommandManager = CommandManager;

	});

	ace.define("ace/commands/default_commands",["require","exports","module","ace/lib/lang","ace/config","ace/range"], function(acequire, exports, module) {
	"use strict";

	var lang = acequire("../lib/lang");
	var config = acequire("../config");
	var Range = acequire("../range").Range;

	function bindKey(win, mac) {
	    return {win: win, mac: mac};
	}
	exports.commands = [{
	    name: "showSettingsMenu",
	    bindKey: bindKey("Ctrl-,", "Command-,"),
	    exec: function(editor) {
	        config.loadModule("ace/ext/settings_menu", function(module) {
	            module.init(editor);
	            editor.showSettingsMenu();
	        });
	    },
	    readOnly: true
	}, {
	    name: "goToNextError",
	    bindKey: bindKey("Alt-E", "F4"),
	    exec: function(editor) {
	        config.loadModule("ace/ext/error_marker", function(module) {
	            module.showErrorMarker(editor, 1);
	        });
	    },
	    scrollIntoView: "animate",
	    readOnly: true
	}, {
	    name: "goToPreviousError",
	    bindKey: bindKey("Alt-Shift-E", "Shift-F4"),
	    exec: function(editor) {
	        config.loadModule("ace/ext/error_marker", function(module) {
	            module.showErrorMarker(editor, -1);
	        });
	    },
	    scrollIntoView: "animate",
	    readOnly: true
	}, {
	    name: "selectall",
	    bindKey: bindKey("Ctrl-A", "Command-A"),
	    exec: function(editor) { editor.selectAll(); },
	    readOnly: true
	}, {
	    name: "centerselection",
	    bindKey: bindKey(null, "Ctrl-L"),
	    exec: function(editor) { editor.centerSelection(); },
	    readOnly: true
	}, {
	    name: "gotoline",
	    bindKey: bindKey("Ctrl-L", "Command-L"),
	    exec: function(editor) {
	        var line = parseInt(prompt("Enter line number:"), 10);
	        if (!isNaN(line)) {
	            editor.gotoLine(line);
	        }
	    },
	    readOnly: true
	}, {
	    name: "fold",
	    bindKey: bindKey("Alt-L|Ctrl-F1", "Command-Alt-L|Command-F1"),
	    exec: function(editor) { editor.session.toggleFold(false); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "center",
	    readOnly: true
	}, {
	    name: "unfold",
	    bindKey: bindKey("Alt-Shift-L|Ctrl-Shift-F1", "Command-Alt-Shift-L|Command-Shift-F1"),
	    exec: function(editor) { editor.session.toggleFold(true); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "center",
	    readOnly: true
	}, {
	    name: "toggleFoldWidget",
	    bindKey: bindKey("F2", "F2"),
	    exec: function(editor) { editor.session.toggleFoldWidget(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "center",
	    readOnly: true
	}, {
	    name: "toggleParentFoldWidget",
	    bindKey: bindKey("Alt-F2", "Alt-F2"),
	    exec: function(editor) { editor.session.toggleFoldWidget(true); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "center",
	    readOnly: true
	}, {
	    name: "foldall",
	    bindKey: bindKey(null, "Ctrl-Command-Option-0"),
	    exec: function(editor) { editor.session.foldAll(); },
	    scrollIntoView: "center",
	    readOnly: true
	}, {
	    name: "foldOther",
	    bindKey: bindKey("Alt-0", "Command-Option-0"),
	    exec: function(editor) { 
	        editor.session.foldAll();
	        editor.session.unfold(editor.selection.getAllRanges());
	    },
	    scrollIntoView: "center",
	    readOnly: true
	}, {
	    name: "unfoldall",
	    bindKey: bindKey("Alt-Shift-0", "Command-Option-Shift-0"),
	    exec: function(editor) { editor.session.unfold(); },
	    scrollIntoView: "center",
	    readOnly: true
	}, {
	    name: "findnext",
	    bindKey: bindKey("Ctrl-K", "Command-G"),
	    exec: function(editor) { editor.findNext(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "center",
	    readOnly: true
	}, {
	    name: "findprevious",
	    bindKey: bindKey("Ctrl-Shift-K", "Command-Shift-G"),
	    exec: function(editor) { editor.findPrevious(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "center",
	    readOnly: true
	}, {
	    name: "selectOrFindNext",
	    bindKey: bindKey("Alt-K", "Ctrl-G"),
	    exec: function(editor) {
	        if (editor.selection.isEmpty())
	            editor.selection.selectWord();
	        else
	            editor.findNext(); 
	    },
	    readOnly: true
	}, {
	    name: "selectOrFindPrevious",
	    bindKey: bindKey("Alt-Shift-K", "Ctrl-Shift-G"),
	    exec: function(editor) { 
	        if (editor.selection.isEmpty())
	            editor.selection.selectWord();
	        else
	            editor.findPrevious();
	    },
	    readOnly: true
	}, {
	    name: "find",
	    bindKey: bindKey("Ctrl-F", "Command-F"),
	    exec: function(editor) {
	        config.loadModule("ace/ext/searchbox", function(e) {e.Search(editor);});
	    },
	    readOnly: true
	}, {
	    name: "overwrite",
	    bindKey: "Insert",
	    exec: function(editor) { editor.toggleOverwrite(); },
	    readOnly: true
	}, {
	    name: "selecttostart",
	    bindKey: bindKey("Ctrl-Shift-Home", "Command-Shift-Home|Command-Shift-Up"),
	    exec: function(editor) { editor.getSelection().selectFileStart(); },
	    multiSelectAction: "forEach",
	    readOnly: true,
	    scrollIntoView: "animate",
	    aceCommandGroup: "fileJump"
	}, {
	    name: "gotostart",
	    bindKey: bindKey("Ctrl-Home", "Command-Home|Command-Up"),
	    exec: function(editor) { editor.navigateFileStart(); },
	    multiSelectAction: "forEach",
	    readOnly: true,
	    scrollIntoView: "animate",
	    aceCommandGroup: "fileJump"
	}, {
	    name: "selectup",
	    bindKey: bindKey("Shift-Up", "Shift-Up|Ctrl-Shift-P"),
	    exec: function(editor) { editor.getSelection().selectUp(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "golineup",
	    bindKey: bindKey("Up", "Up|Ctrl-P"),
	    exec: function(editor, args) { editor.navigateUp(args.times); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "selecttoend",
	    bindKey: bindKey("Ctrl-Shift-End", "Command-Shift-End|Command-Shift-Down"),
	    exec: function(editor) { editor.getSelection().selectFileEnd(); },
	    multiSelectAction: "forEach",
	    readOnly: true,
	    scrollIntoView: "animate",
	    aceCommandGroup: "fileJump"
	}, {
	    name: "gotoend",
	    bindKey: bindKey("Ctrl-End", "Command-End|Command-Down"),
	    exec: function(editor) { editor.navigateFileEnd(); },
	    multiSelectAction: "forEach",
	    readOnly: true,
	    scrollIntoView: "animate",
	    aceCommandGroup: "fileJump"
	}, {
	    name: "selectdown",
	    bindKey: bindKey("Shift-Down", "Shift-Down|Ctrl-Shift-N"),
	    exec: function(editor) { editor.getSelection().selectDown(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "golinedown",
	    bindKey: bindKey("Down", "Down|Ctrl-N"),
	    exec: function(editor, args) { editor.navigateDown(args.times); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "selectwordleft",
	    bindKey: bindKey("Ctrl-Shift-Left", "Option-Shift-Left"),
	    exec: function(editor) { editor.getSelection().selectWordLeft(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "gotowordleft",
	    bindKey: bindKey("Ctrl-Left", "Option-Left"),
	    exec: function(editor) { editor.navigateWordLeft(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "selecttolinestart",
	    bindKey: bindKey("Alt-Shift-Left", "Command-Shift-Left|Ctrl-Shift-A"),
	    exec: function(editor) { editor.getSelection().selectLineStart(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "gotolinestart",
	    bindKey: bindKey("Alt-Left|Home", "Command-Left|Home|Ctrl-A"),
	    exec: function(editor) { editor.navigateLineStart(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "selectleft",
	    bindKey: bindKey("Shift-Left", "Shift-Left|Ctrl-Shift-B"),
	    exec: function(editor) { editor.getSelection().selectLeft(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "gotoleft",
	    bindKey: bindKey("Left", "Left|Ctrl-B"),
	    exec: function(editor, args) { editor.navigateLeft(args.times); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "selectwordright",
	    bindKey: bindKey("Ctrl-Shift-Right", "Option-Shift-Right"),
	    exec: function(editor) { editor.getSelection().selectWordRight(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "gotowordright",
	    bindKey: bindKey("Ctrl-Right", "Option-Right"),
	    exec: function(editor) { editor.navigateWordRight(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "selecttolineend",
	    bindKey: bindKey("Alt-Shift-Right", "Command-Shift-Right|Shift-End|Ctrl-Shift-E"),
	    exec: function(editor) { editor.getSelection().selectLineEnd(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "gotolineend",
	    bindKey: bindKey("Alt-Right|End", "Command-Right|End|Ctrl-E"),
	    exec: function(editor) { editor.navigateLineEnd(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "selectright",
	    bindKey: bindKey("Shift-Right", "Shift-Right"),
	    exec: function(editor) { editor.getSelection().selectRight(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "gotoright",
	    bindKey: bindKey("Right", "Right|Ctrl-F"),
	    exec: function(editor, args) { editor.navigateRight(args.times); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "selectpagedown",
	    bindKey: "Shift-PageDown",
	    exec: function(editor) { editor.selectPageDown(); },
	    readOnly: true
	}, {
	    name: "pagedown",
	    bindKey: bindKey(null, "Option-PageDown"),
	    exec: function(editor) { editor.scrollPageDown(); },
	    readOnly: true
	}, {
	    name: "gotopagedown",
	    bindKey: bindKey("PageDown", "PageDown|Ctrl-V"),
	    exec: function(editor) { editor.gotoPageDown(); },
	    readOnly: true
	}, {
	    name: "selectpageup",
	    bindKey: "Shift-PageUp",
	    exec: function(editor) { editor.selectPageUp(); },
	    readOnly: true
	}, {
	    name: "pageup",
	    bindKey: bindKey(null, "Option-PageUp"),
	    exec: function(editor) { editor.scrollPageUp(); },
	    readOnly: true
	}, {
	    name: "gotopageup",
	    bindKey: "PageUp",
	    exec: function(editor) { editor.gotoPageUp(); },
	    readOnly: true
	}, {
	    name: "scrollup",
	    bindKey: bindKey("Ctrl-Up", null),
	    exec: function(e) { e.renderer.scrollBy(0, -2 * e.renderer.layerConfig.lineHeight); },
	    readOnly: true
	}, {
	    name: "scrolldown",
	    bindKey: bindKey("Ctrl-Down", null),
	    exec: function(e) { e.renderer.scrollBy(0, 2 * e.renderer.layerConfig.lineHeight); },
	    readOnly: true
	}, {
	    name: "selectlinestart",
	    bindKey: "Shift-Home",
	    exec: function(editor) { editor.getSelection().selectLineStart(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "selectlineend",
	    bindKey: "Shift-End",
	    exec: function(editor) { editor.getSelection().selectLineEnd(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor",
	    readOnly: true
	}, {
	    name: "togglerecording",
	    bindKey: bindKey("Ctrl-Alt-E", "Command-Option-E"),
	    exec: function(editor) { editor.commands.toggleRecording(editor); },
	    readOnly: true
	}, {
	    name: "replaymacro",
	    bindKey: bindKey("Ctrl-Shift-E", "Command-Shift-E"),
	    exec: function(editor) { editor.commands.replay(editor); },
	    readOnly: true
	}, {
	    name: "jumptomatching",
	    bindKey: bindKey("Ctrl-P", "Ctrl-P"),
	    exec: function(editor) { editor.jumpToMatching(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "animate",
	    readOnly: true
	}, {
	    name: "selecttomatching",
	    bindKey: bindKey("Ctrl-Shift-P", "Ctrl-Shift-P"),
	    exec: function(editor) { editor.jumpToMatching(true); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "animate",
	    readOnly: true
	}, {
	    name: "expandToMatching",
	    bindKey: bindKey("Ctrl-Shift-M", "Ctrl-Shift-M"),
	    exec: function(editor) { editor.jumpToMatching(true, true); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "animate",
	    readOnly: true
	}, {
	    name: "passKeysToBrowser",
	    bindKey: bindKey(null, null),
	    exec: function() {},
	    passEvent: true,
	    readOnly: true
	}, {
	    name: "copy",
	    exec: function(editor) {
	    },
	    readOnly: true
	},
	{
	    name: "cut",
	    exec: function(editor) {
	        var range = editor.getSelectionRange();
	        editor._emit("cut", range);

	        if (!editor.selection.isEmpty()) {
	            editor.session.remove(range);
	            editor.clearSelection();
	        }
	    },
	    scrollIntoView: "cursor",
	    multiSelectAction: "forEach"
	}, {
	    name: "paste",
	    exec: function(editor, args) {
	        editor.$handlePaste(args);
	    },
	    scrollIntoView: "cursor"
	}, {
	    name: "removeline",
	    bindKey: bindKey("Ctrl-D", "Command-D"),
	    exec: function(editor) { editor.removeLines(); },
	    scrollIntoView: "cursor",
	    multiSelectAction: "forEachLine"
	}, {
	    name: "duplicateSelection",
	    bindKey: bindKey("Ctrl-Shift-D", "Command-Shift-D"),
	    exec: function(editor) { editor.duplicateSelection(); },
	    scrollIntoView: "cursor",
	    multiSelectAction: "forEach"
	}, {
	    name: "sortlines",
	    bindKey: bindKey("Ctrl-Alt-S", "Command-Alt-S"),
	    exec: function(editor) { editor.sortLines(); },
	    scrollIntoView: "selection",
	    multiSelectAction: "forEachLine"
	}, {
	    name: "togglecomment",
	    bindKey: bindKey("Ctrl-/", "Command-/"),
	    exec: function(editor) { editor.toggleCommentLines(); },
	    multiSelectAction: "forEachLine",
	    scrollIntoView: "selectionPart"
	}, {
	    name: "toggleBlockComment",
	    bindKey: bindKey("Ctrl-Shift-/", "Command-Shift-/"),
	    exec: function(editor) { editor.toggleBlockComment(); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "selectionPart"
	}, {
	    name: "modifyNumberUp",
	    bindKey: bindKey("Ctrl-Shift-Up", "Alt-Shift-Up"),
	    exec: function(editor) { editor.modifyNumber(1); },
	    scrollIntoView: "cursor",
	    multiSelectAction: "forEach"
	}, {
	    name: "modifyNumberDown",
	    bindKey: bindKey("Ctrl-Shift-Down", "Alt-Shift-Down"),
	    exec: function(editor) { editor.modifyNumber(-1); },
	    scrollIntoView: "cursor",
	    multiSelectAction: "forEach"
	}, {
	    name: "replace",
	    bindKey: bindKey("Ctrl-H", "Command-Option-F"),
	    exec: function(editor) {
	        config.loadModule("ace/ext/searchbox", function(e) {e.Search(editor, true);});
	    }
	}, {
	    name: "undo",
	    bindKey: bindKey("Ctrl-Z", "Command-Z"),
	    exec: function(editor) { editor.undo(); }
	}, {
	    name: "redo",
	    bindKey: bindKey("Ctrl-Shift-Z|Ctrl-Y", "Command-Shift-Z|Command-Y"),
	    exec: function(editor) { editor.redo(); }
	}, {
	    name: "copylinesup",
	    bindKey: bindKey("Alt-Shift-Up", "Command-Option-Up"),
	    exec: function(editor) { editor.copyLinesUp(); },
	    scrollIntoView: "cursor"
	}, {
	    name: "movelinesup",
	    bindKey: bindKey("Alt-Up", "Option-Up"),
	    exec: function(editor) { editor.moveLinesUp(); },
	    scrollIntoView: "cursor"
	}, {
	    name: "copylinesdown",
	    bindKey: bindKey("Alt-Shift-Down", "Command-Option-Down"),
	    exec: function(editor) { editor.copyLinesDown(); },
	    scrollIntoView: "cursor"
	}, {
	    name: "movelinesdown",
	    bindKey: bindKey("Alt-Down", "Option-Down"),
	    exec: function(editor) { editor.moveLinesDown(); },
	    scrollIntoView: "cursor"
	}, {
	    name: "del",
	    bindKey: bindKey("Delete", "Delete|Ctrl-D|Shift-Delete"),
	    exec: function(editor) { editor.remove("right"); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor"
	}, {
	    name: "backspace",
	    bindKey: bindKey(
	        "Shift-Backspace|Backspace",
	        "Ctrl-Backspace|Shift-Backspace|Backspace|Ctrl-H"
	    ),
	    exec: function(editor) { editor.remove("left"); },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor"
	}, {
	    name: "cut_or_delete",
	    bindKey: bindKey("Shift-Delete", null),
	    exec: function(editor) { 
	        if (editor.selection.isEmpty()) {
	            editor.remove("left");
	        } else {
	            return false;
	        }
	    },
	    multiSelectAction: "forEach",
	    scrollIntoView: "cursor"
	}, {
	    name: "removetolinestart",
	    bindKey: bindKey("Alt-Backspace", "Command-Backspace"),
	    exec: function(editor) {