<details>
  <summary><strong>Release history</strong></summary>

{%= changelog("changelog.md") %}

</details>


## What is nanomatch?

Nanomatch is a fast and accurate glob matcher with full support for standard Bash glob features, including the following "metacharacters": `*`, `**`, `?` and `[...]`. 

**Learn more**

- [Getting started](#getting-started): learn how to install and begin using nanomatch
- [Features](#features): jump to info about supported patterns, and a glob matching reference
- [API documentation](#api): jump to available options and methods
- [Unit tests](test): visit unit tests. there is no better way to learn a code library than spending time the unit tests. Nanomatch has 36,000 unit tests - go become a glob matching ninja!

<details>
  <summary><strong>How is this different?</strong></summary>

**Speed and accuracy**

Nanomatch uses [snapdragon][] for parsing and compiling globs, which results in:

- Granular control over the entire conversion process in a way that is easy to understand, reason about, and customize.
- Faster matching, from a combination of optimized glob patterns and (optional) caching.
- Much greater accuracy than minimatch. In fact, nanomatch passes _all of the spec tests_ from bash, including some that bash still fails. However, since there is no real specification for globs, if you encounter a pattern that yields unexpected match results [after researching previous issues](../../issues), [please let us know](../../issues/new).

**Basic globbing only**

Nanomatch supports [basic globbing only](#features), which is limited to `*`, `**`, `?` and regex-like brackets. 

If you need support for the other [bash "expansion" types](#bash-expansion-libs) (in addition to the wildcard matching provided by nanomatch), consider using [micromatch][] instead. _(micromatch >=3.0.0  uses the nanomatch parser and compiler for basic glob matching)_

</details>


## Getting started

### Installing nanomatch

**Install with [yarn](https://yarnpkg.com/)**

```sh
$ yarn add nanomatch
```

**Install with [npm](https://npmjs.com)**

```sh
$ npm install nanomatch
```

### Usage

Add nanomatch to your project using node's `require()` system:

```js
var nanomatch = require('{%= name %}');

// the main export is a function that takes an array of strings to match
// and a string or array of patterns to use for matching
nanomatch(list, patterns[, options]);
```

**Params**

- `list` **{String|Array}**: List of strings to perform matches against. This is often a list of file paths.
- `patterns` **{String|Array}**: One or more [glob paterns](#features) to use for matching.
- `options` **{Object}**: Any [supported options](#options) may be passed

**Examples**

```js
var nm = require('nanomatch');
console.log(nm(['a', 'b/b', 'c/c/c'], '*'));
//=> ['a']

console.log(nm(['a', 'b/b', 'c/c/c'], '*/*'));
//=> ['b/b']

console.log(nm(['a', 'b/b', 'c/c/c'], '**'));
//=> ['a', 'b/b', 'c/c/c']
```

See the [API documentation](#api) for available methods and [options][].

## Documentation

### Escaping

_Backslashes and quotes_ can be used to escape characters, forcing nanomatch to regard those characters as a literal characters. 

**Backslashes**

Use backslashes to escape single characters. For example, the following pattern would match `foo/*/bar` exactly:

```js
'foo/\*/bar'
```

The following pattern would match `foo/` followed by a literal `*`, followed by zero or more of any characters besides `/`, followed by `/bar`.

```js
'foo/\**/bar'
```

**Quoted strings**

Use single or double quotes to escape sequences of characters. For example, the following patterns would match `foo/**/bar` exactly:

```js
'foo/"**"/bar'
'foo/\'**\'/bar'
"foo/'**'/bar"
```

**Matching literal quotes**

If you need to match quotes literally, you can escape them as well. For example, the following will match `foo/"*"/bar`, `foo/"a"/bar`, `foo/"b"/bar`, or `foo/"c"/bar`:

```js
'foo/\\"*\\"/bar'
```

And the following will match `foo/'*'/bar`, `foo/'a'/bar`, `foo/'b'/bar`, or `foo/'c'/bar`:

```js
'foo/\\\'*\\\'/bar'
```

## API

{%= apidocs("index.js") %}

## Options

<details>
<summary><strong>basename</strong></summary>

### options.basename

Allow glob patterns without slashes to match a file path based on its basename. Same behavior as [minimatch][] option `matchBase`.

Type: `boolean`

Default: `false`

**Example**

```js
nm(['a/b.js', 'a/c.md'], '*.js');
//=> []

nm(['a/b.js', 'a/c.md'], '*.js', {matchBase: true});
//=> ['a/b.js']
```

</details>

<details>
<summary><strong>bash</strong></summary>

### options.bash

Enabled by default, this option enforces bash-like behavior with stars immediately following a bracket expression. Bash bracket expressions are similar to regex character classes, but unlike regex, a star following a bracket expression **does not repeat the bracketed characters**. Instead, the star is treated the same as an other star.

Type: `boolean`

Default: `true` 

**Example**

```js
var files = ['abc', 'ajz'];
console.log(nm(files, '[a-c]*'));
//=> ['abc', 'ajz']

console.log(nm(files, '[a-c]*', {bash: false}));
```

</details>

<details>
<summary><strong>cache</strong></summary>

### options.cache

Disable regex and function memoization. 

Type: `boolean`

Default: `undefined`


</details>

<details>
<summary><strong>dot</strong></summary>

### options.dot

Match dotfiles. Same behavior as [minimatch][] option `dot`.

Type: `boolean`

Default: `false`


</details>

<details>
<summary><strong>failglob</strong></summary>

### options.failglob

Similar to the `--failglob` behavior in Bash, throws an error when no matches are found.

Type: `boolean`

Default: `undefined`


</details>

<details>
<summary><strong>ignore</strong></summary>

### options.ignore

String or array of glob patterns to match files to ignore.

Type: `String|Array`

Default: `undefined`


</details>

<details>
<summary><strong>matchBase</strong></summary>

### options.matchBase

Alias for [options.basename](#options-basename).


</details>

<details>
<summary><strong>nocase</strong></summary>

### options.nocase

Use a case-insensitive regex for matching files. Same behavior as [minimatch][].

Type: `boolean`

Default: `undefined`


</details>

<details>
<summary><strong>nodupes</strong></summary>

### options.nodupes

Remove duplicate elements from the result array.

Type: `boolean`

Default: `true` (enabled by default)

**Example**

Example of using the `unescape` and `nodupes` options together:

```js
nm.match(['a/b/c', 'a/b/c'], '**');
//=> ['abc']

nm.match(['a/b/c', 'a/b/c'], '**', {nodupes: false});
//=> ['a/b/c', 'a/b/c']
```

</details>

<details>
<summary><strong>nonegate</strong></summary>

### options.noglobstar

Disable matching with globstars (`**`).

Type: `boolean`

Default: `undefined`

```js
nm(['a/b', 'a/b/c', 'a/b/c/d'], 'a/**');
//=> ['a/b', 'a/b/c', 'a/b/c/d']

nm(['a/b', 'a/b/c', 'a/b/c/d'], 'a/**', {noglobstar: true});
//=> ['a/b']
```

</details>

<details>
<summary><strong>nonegate</strong></summary>

### options.nonegate

Disallow negation (`!`) patterns, and treat leading `!` as a literal character to match.

Type: `boolean`

Default: `undefined`


</details>

<details>
<summary><strong>nonull</strong></summary>

### options.nonull

Alias for [options.nullglob](#options-nullglob).


</details>

<details>
<summary><strong>nullglob</strong></summary>

### options.nullglob

If `true`, when no matches are found the actual (arrayified) glob pattern is returned instead of an empty array. Same behavior as [minimatch][] option `nonull`.

Type: `boolean`

Default: `undefined`


</details>

<details>
<summary><strong><a name="slash">slash</a></strong></summary>

### options.slash

Customize the slash character(s) to use for matching. 

Type: `string|function`

Default: `[/\\]` (forward slash and backslash)


</details>

<details>
<summary><strong><a name="star">star</a></strong></summary>

### options.star

Customize the star character(s) to use for matching. It's not recommended that you modify this unless you have advanced knowledge of the compiler and matching rules.

Type: `string|function`

Default: `[^/\\]*?`


</details>

<details>
<summary><strong><a name="snapdragon">snapdragon</a></strong></summary>

### options.snapdragon

Pass your own instance of [snapdragon][] to customize parsers or compilers.

Type: `object`

Default: `undefined`


</details>

<details>
<summary><strong>snapdragon</strong></summary>

### options.sourcemap

Generate a source map by enabling the `sourcemap` option with the `.parse`, `.compile`, or `.create` methods.

**Examples**

```js
var nm = require('nanomatch');

var res = nm.create('abc/*.js', {sourcemap: true});
console.log(res.map);
// { version: 3,
//   sources: [ 'string' ],
//   names: [],
//   mappings: 'AAAA,GAAG,EAAC,iBAAC,EAAC,EAAE',
//   sourcesContent: [ 'abc/*.js' ] }

var ast = nm.parse('abc/**/*.js');
var res = nm.compile(ast, {sourcemap: true});
console.log(res.map);
// { version: 3,
//   sources: [ 'string' ],
//   names: [],
//   mappings: 'AAAA,GAAG,EAAC,2BAAE,EAAC,iBAAC,EAAC,EAAE',
//   sourcesContent: [ 'abc/**/*.js' ] }
```

</details>

<details>
<summary><strong>unescape</strong></summary>

### options.unescape

Remove backslashes from returned matches.

Type: `boolean`

Default: `undefined`

**Example**

In this example we want to match a literal `*`:

```js
nm.match(['abc', 'a\\*c'], 'a\\*c');
//=> ['a\\*c']

nm.match(['abc', 'a\\*c'], 'a\\*c', {unescape: true});
//=> ['a*c']
```

</details>

<details>
<summary><strong>unixify</strong></summary>

### options.unixify

Convert path separators on returned files to posix/unix-style forward slashes.

Type: `boolean`

Default: `true`

**Example**

```js
nm.match(['a\\b\\c'], 'a/**');
//=> ['a/b/c']

nm.match(['a\\b\\c'], {unixify: false});
//=> ['a\\b\\c']
```

</details>

## Features

Nanomatch has full support for standard Bash glob features, including the following "metacharacters": `*`, `**`, `?` and `[...]`.

Here are some examples of how they work:

| **Pattern** | **Description** |
| --- | --- |
| `*`         | Matches any string except for `/`, leading `.`, or `/.` inside a path  |
| `**`        | Matches any string including `/`, but not a leading `.` or `/.` inside a path. More than two stars (e.g. `***` is treated the same as one star, and `**` loses its special meaning | when it's not the only thing in a path segment, per Bash specifications) |
| `foo*`      | Matches any string beginning with `foo` |
| `*bar*`     | Matches any string containing `bar` (beginning, middle or end) |
| `*.min.js`  | Matches any string ending with `.min.js` |
| `[abc]*.js` | Matches any string beginning with `a`, `b`, or `c` and ending with `.js` |
| `abc?`      | Matches `abcd` or `abcz` but not `abcde` |

The exceptions noted for `*` apply to all patterns that contain a `*`.

**Not supported**

The following extended-globbing features are not supported:

- [brace expansion][braces] (e.g. `{a,b,c}`)
- [extglobs][extglob] (e.g. `@(a|!(c|d))`)
- [POSIX brackets][brackets] (e.g. `[[:alpha:][:digit:]]`)

If you need any of these features consider using [micromatch][] instead.

## Bash expansion libs

Nanomatch is part of a suite of libraries aimed at bringing the power and expressiveness of [Bash's][bash] matching and expansion capabilities to JavaScript, _and - as you can see by the [benchmarks](#benchmarks) - without sacrificing speed_.

| **Related library** | **Matching Type** | **Example** | **Description** |
| --- | --- | --- | --- |
| `nanomatch` (you are here) | Wildcards  | `*`     | [Filename expansion][bash-globs], also referred to as globbing and pathname expansion, allows the use of [wildcards](#features) for matching.  |
| [expand-tilde][]     | Tildes     | `~`           | [Tilde expansion][bash-tilde] converts the leading tilde in a file path to the user home directory. |
| [braces][]           | Braces     | `{a,b,c}`     | [Brace expansion][bash-braces] |
| [expand-brackets][]  | Brackets   | `[[:alpha:]]` | [POSIX character classes][bash-brackets] (also referred to as POSIX brackets, or POSIX character classes) |
| [extglob][]          | Parens     | `!(a\|b)`     | [Extglobs][bash-extglobs] |
| [micromatch][]       | All        | all           | Micromatch is built on top of the other libraries. |

There are many resources available on the web if you want to dive deeper into how these features work in Bash.

## Benchmarks

### Running benchmarks

Install dev dependencies:

```bash
npm i -d && node benchmark
```

### Nanomatch vs. Minimatch vs. Multimatch

```bash
{%= include("benchmark/stats.md") %}
```

[bash]: https://www.gnu.org/software/bash/
[bash-braces]: https://www.gnu.org/software/bash/manual/html_node/Brace-Expansion.html
[bash-brackets]: https://www.gnu.org/software/grep/manual/html_node/Character-Classes-and-Bracket-Expressions.html
[bash-extglobs]: https://www.gnu.org/software/bash/manual/html_node/Pattern-Matching.html#Pattern-Matching
[bash-globs]: https://www.gnu.org/software/bash/manual/html_node/Filename-Expansion.html#Filename-Expansion
[bash-tilde]: https://www.gnu.org/software/bash/manual/html_node/Tilde-Expansion.html#Tilde-Expansion
[regex]: http://www.regular-expressions.info/

[tilde]: https://github.com/jonschlinkert/expand-tilde
[brackets]: https://github.com/jonschlinkert/expand-brackets
[extglob]: https://github.com/jonschlinkert/extglob
[braces]: https://github.com/jonschlinkert/braces
