# ✈️ v4.0 Migration Guide
## Introduction

TOAST UI Chart (hereafter referred to as the 'Chart') version 4.0 was developed based on the `canvas` and provides light development environment with removed dependencies and new support for tree shaking. Carefully consider the contents of the migration guide as there are numerous changes to keep in mind when updating.

## Table of Contents

- [Changes](#changes)
  1. [Installation](#1-installation)
  2. [Creating the Chart](#2-creating-the-chart)
  3. [Themes](#3-themes)
  4. [Tooltip Option](#4-tooltip-option)
  5. [Axes Option](#5-axes-option)
  6. [Data Label Option](#6-data-label-option)
  7. [NestedPie Chart(Pie&Donut Combo Chart, previously)](#7-nestedpie-chartpiedonut-combo-chart-previously)
  8. [Name Changes](#8-name-changes)
  9. [Improved Live Update](#9-improved-live-update)
  10. [Added `'auto'` Type for `chart.width` and `chart.height` Options](#10-added-auto-type-for-chartwidth-and-chartheight-options)
  11. [Responsive Option](#11-responsive-option)
  12. [Layout Configuration](#12-layout-configuration)
  13. [Changed Pie Chart Series Option](#13-changed-pie-chart-series-option)
  14. [New Instance Method](#14-new-instance-method)
  15. [New Custom Event](#15-new-custom-event)
  16. [Map Chart](#16-Map-chart)
* [Removed Features](#removed-features)
  1. [Bower Support Dropped](#1-bower-support-dropped)
  2. [Removed APIs](#2-removed-apis-list)

  ## Changes
### 1. Installation

In this major version update, changes were made to the names and dependency information required to use the Chart. Because of the names of packages, namespaces, and file bundles, it is not enough to update the chart, and the user must also consider the following changes to proceed with the migration successfully.

#### Package Name Change

With v4.0, the Chart applied the [Scoped package](https://docs.npmjs.com/cli/v6/using-npm/scope) rules and changed its package name from `tui-chart` to `@toast-ui/chart`. The following is an example of installing the chart using the npm command.

**v3.x**

```sh
$ npm install tui-chart
$ npm install tui-chart@<version>
```

**v4.0**

```sh
$ npm install @toast-ui/chart
$ npm install @toast-ui/chart@<version>
```

Charts can only be used as `tui-chart` for v3.x and `@toast-ui/chart` for v4.0 and above, so users must be careful when installing a specific version. Furthermore, in order to prevent duplicate installation, the original `tui-chart` must be removed in order for the v4.0 to be installed. When the installation is complete, the dependency information found in `package.json` will be updated as below.

```diff
{
  "dependencies": {
-  "tui-chart": "^3.11.2",
+  "@toast-ui/chart": "^4.0.0"
  }
}
```

Therefore, the changed name must be used for import when including the module in the file.

**v3.x**

```js
const Chart = require('tui-chart'); /* CommonJS Syntax */
```

```js
import Chart from 'tui-chart'; /* ES6 Module Syntax */
```

**v4.0**

```js
const Chart = require('@toast-ui/chart'); /* CommonJS Syntax */
```

```js
import Chart from '@toast-ui/chart'; /* ES6 Module Syntax */
```

Furthermore, Chart 4.0 is made even lighter by allowing users to load the charts that are necessary. Importing only the necessary modules can be seen below.

```js
import { BarChart } from '@toast-ui/chart'; /* ES6 Module  Syntax */
```

#### Namespace Change

According to the new TOAST UI product policy, Chart, from v4.0, uses the `toastui` namespace instead of the original `tui`.

In order to use the chart in the browser environment as a namespace, users must do so by using the constructor function (`toastui.Chart`) associated with the new namespace.

**v3.x**

```js
const chart = tui.Chart.barChart(el, data, options);
```

**v4.0**

```js
const chart = toastui.Chart.barChart({el, data, options});
```

#### Bundle File Change

Chart v4.0 experienced a bundle file type change during the process of removing its dependencies and segmenting its features. The dependent bundles were removed as were the dependencies, and the bundle file name's prefix was also changed from `tui-` to `toastui-`.

**v3.x**

> refer to https://github.com/nhn/tui.chart/tree/v3.11.2/dist

| Bundle Type | File Name | Availability of a Minified Version (`*.min`) |
| :--- | :--- | :--- |
| Basic | `tui-chart.js` | Y |
| Includes All Dependent Modules | `tui-chart-all.js` | Y |
| Includes babel-polyfill | `tui-chart-polyfill.js` | Y |

**v4.0**

| Bundle Type | File Name | Availability of a Minified Version (`*.min`) |
| :--- | :--- | :--- |
| Basic | `toastui-chart.js` | Y |

<br>
CSS file that is required to apply the Charts' style was also changed.

**v3.x**

```js
import 'tui-chart/dist/tui-chart.css';
```

**v4.0**

```js
import '@toast-ui/chart/dist/toastui-chart.css';
```

The provided CSS file, `toastui-chart.css` provides a minified version in the same file. File types provided through npm and CDN are identical.

#### Dependency Information Change

Chart v4.0 no longer has any dependent modules. In order to use the CDN for development, outdated dependencies must be removed.

**v3.x**

- [List of Dependent Files Required for Browser Development Environment](https://github.com/nhn/tui.chart/blob/v3.11.2/docs/wiki/getting-started.md)

**v4.0**

```html
<head>
  ...
  <link
    rel="stylesheet"
    href="https://uicdn.toast.com/chart/latest/toastui-chart.min.css"
  />
  ...
</head>
<body>
  ...
  <script src="https://uicdn.toast.com/chart/latest/toastui-chart.min.js"></script>
  ...
</body>
```

### 2. Creating the Chart
In v3.x, `el` (the element on which the chart will be drawn), `data` (chart data), and options were provided to the static function sequentially to create the chart.

**v3.x**
```js
import Chart from 'tui-chart';

const chart = Chart.barChart(el, data, options);
```

In v4.0, users can use the constructor function as well as the static function to create the chart, and for its parameters, the constructor function is built to receive the element, data, and options **object**.

**v4.0**
```js
// 1. Using the static function
import Chart from '@toast-ui/chart';

const chart = Chart.barChart({el, data, options});

// 2. Using the constructor
import { BarChart } from '@toast-ui/chart';

const chart = new BarChart({el, data, options});
```

#### package.json File Change
From v4.0, the `module` and `exports` fields will be defined as well as the `main` field in package.json.

* Files defined in the `main` field are for UMD format.
* The `module` field is used by bundlers like Webpack and serves as file directories for ESM.
* When accessing the ESM file through the `exports` field, sub-paths are provided.


```js
  "main": "dist/toastui-chart.js",
  "module": "dist/esm/",
  "exports": {
    ".": {
      "import": "./dist/esm/index.js",
      "require": "./dist/toastui-chart.js"
    },
    "./line": {
      "import": "./dist/esm/charts/lineChart.js"
    },
    "./area": {
      "import": "./dist/esm/charts/areaChart.js"
    },
    "./lineArea": {
      "import": "./dist/esm/charts/lineAreaChart.js"
    },
    "./bar": {
      "import": "./dist/esm/charts/barChart.js"
    },
    "./column": {
      "import": "./dist/esm/charts/columnChart.js"
    },
    "./columnLine": {
      "import": "./dist/esm/charts/columnLineChart.js"
    },
    "./bullet": {
      "import": "./dist/esm/charts/bulletChart.js"
    },
    "./boxPlot": {
      "import": "./dist/esm/charts/boxPlotChart.js"
    },
    "./treemap": {
      "import": "./dist/esm/charts/treemapChart.js"
    },
    "./heatmap": {
      "import": "./dist/esm/charts/heatmapChart.js"
    },
    "./scatter": {
      "import": "./dist/esm/charts/scatterChart.js"
    },
    "./lineScatter": {
      "import": "./dist/esm/charts/lineScatterChart.js"
    },
    "./bubble": {
      "import": "./dist/esm/charts/bubbleChart.js"
    },
    "./pie": {
      "import": "./dist/esm/charts/pieChart.js"
    },
    "./nestedPie": {
      "import": "./dist/esm/charts/nestedPieChart.js"
    },
    "./radar": {
      "import": "./dist/esm/charts/radarChart.js"
    },
    "./": "./"
  },
```

[In Webpack 4, when importing package modules, the parts that are defined in the module field have higher priority than the parts defined in the main field](https://webpack.js.org/configuration/resolve/#resolvemainfields). To use the Webpack 4 with the `require` syntax to import `@toast-ui/chart`, the ESM file defined in the `module` field will be loaded, and the file will be transpiled to be compatible with the `require` syntax. In order to use the **bundle file for UMD**, the user must personally load and use the `@toast-ui/chart/dist/toastui-chart.js` or `@toast-ui/chart/dist/toastui-chart.min.js`.

```js
const Chart = require('@toast-ui/chart/dist/toastui-chart.min.js'); // loading the bundle file for UMD
```

Webpack 5 supports the `exports` field. The entry point can be determined by the `exports` field defined in the package. Furthermore, the necessary chart can be loaded through a sub-path, as presented below.

```js
const Chart = require('@toast-ui/chart');  // ./dist/toastui-chart.js

import { BarChart } from '@toast-ui/chart';  // ./dist/esm/index.js

import BarChart from '@toast-ui/chart/bar';
import ColumnChart from '@toast-ui/chart/column';
import LineChart from '@toast-ui/chart/line';
import AreaChart from '@toast-ui/chart/area';
import LineAreaChart from '@toast-ui/chart/lineArea';
import ColumnLineChart from '@toast-ui/chart/columnLine';
import BulletChart from '@toast-ui/chart/bullet';
import BoxPlotChart from '@toast-ui/chart/boxPlot';
import TreemapChart from '@toast-ui/chart/treemap';
import HeatmapChart from '@toast-ui/chart/heatmap';
import ScatterChart from '@toast-ui/chart/scatter';
import LineScatterChart from '@toast-ui/chart/lineScatter';
import BubbleChart from '@toast-ui/chart/bubble';
import PieChart from '@toast-ui/chart/pie';
import NestedPieChart from '@toast-ui/chart/nestedPie';
import RadarChart from '@toast-ui/chart/radar';
```

#### Chart Name Change
In v4.0, some names for the charts, including combination charts, were changed for when using the static function to create the charts. The following is a table that compares the constructors of the charts between v3.x and v4.0.

* General Charts

  | v3.x | v4.0 - static function | v4.0 - constructor function | Name Changed |
  | --- | --- | --- | --- |
  | barChart | barChart | BarChart |  |
  | columnChart | columnChart | ColumnChart |  |
  | bulletChart | bulletChart | BulletChart |  |
  | boxplotChart | boxPlotChart | BoxPlotChart | Y |
  | lineChart | lineChart | LineChart |  |
  | areaChart | areaChart | AreaChart |  |
  | heatmapChart | heatmapChart | HeatmapChart |  |
  | treemapChart | treemapChart | TreemapChart |  |
  | bubbleChart | bubbleChart | BubbleChart |  |
  | scatterChart | scatterChart | ScatterChart |  |
  | radialChart | radarChart | RadarChart | Y |
  | pieChart | pieChart | PieChart |  |

* Combination (Combo) Charts

  In v3.x, the static function `comboChart` was used to create the combination charts. In v4.0, the names were changed to represent the supported combination charts more directly.

  | v3.x | v4.0 - static function | v4.0 - constructor function | Name Changed |
  | --- | --- | --- | --- |
  | comboChart | nestedPieChart | NestedPieChart | Y |
  | comboChart | columnLineChart | ColumnLineChart | Y |
  | comboChart | lineAreaChart | LineAreaChart | Y |
  | comboChart | lineScatterChart | lineScatterChart | Y |

  All charts, except for the NestedPie chart, have the same data and available options as their 3.x versions. The NestedPie chart will be explained more in detail in the [NestedPie Chart(Pie&Donut Combo Chart, previously)](#7-nestedpie-chartpiedonut-combo-chart-previously) section.

### 3. Themes
Users could use static methods to register and modify the themes with the Chart v3.x. With v4.0, the themes are defined using the options and can be changed by [`setOptions API`]((https://github.com/nhn/tui.chart/blot/next/docs/ko/common-api.md#setoptions)). The themes can be applied to tooltips and export menus, and it can even be applied to data labels when working with series.

**v3.x**

```js
const theme = {
  chart: {/* */},
  title: {/* */},
  xAxis: {/* */},
  yAxis: {/* */},
  plot: {/* */},
  series: {/* */},
  legend: {/* */}
};

Chart.registerTheme('newTheme', theme);

const options = {
  ...
  theme: 'newTheme'
};
```

**v4.0**
```js
const options = {
  ...
  theme: {
    chart: {/* Global Chart Style */},
    title: {/* Chart Title Style */},
    xAxis: {/* X-Axis Style */},
    yAxis: {/* Y-Axis Style */},
    plot: {/* Plot Style */},
    series: {/* Series Style */},
    legend: {/* Legend Style */},
    tooltip: {/* Tooltip Style */},
    exportMenu: {/* Export Menu Style */}
  }
};

const chart = Chart.barChart({el, data, options});

chart.setOptions({
  ...
  theme: {
    series: {
      theme: {
        chart: {/* Global Chart Style */},
        title: {/* Chart Title Style */},
        xAxis: {/* X-Axis Style */},
        yAxis: {/* Y-Axis Style */},
        plot: {/* Plot Style */},
        series: {/* Series Style */},
        legend: {/* Legend Style */},
        tooltip: {/* Tooltip Style */},
        exportMenu: {/* Export Menu Style */}
      }
    }
  }
})
```

Themes related to global style, chart title, axes, plot, legend, tooltip, and export menu can be found in the [Common Component Themes](https://github.com/nhn/tui.chart/blob/main/docs/en/common-theme.md), and themes related to the series can be found in the respective guides.

* [Bar](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-bar.md#series-theme), [Column](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-column.md#series-theme), [Bullet](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-bullet.md#series-theme), [BoxPlot](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-boxplot.md#series-theme), [Line](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-line.md#series-theme), [Area](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-area.md#series-theme), [Heatmap](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-heatmap.md#series-theme), [Treemap](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-treemap.md#series-theme), [Bubble](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-bubble.md#series-theme), [Scatter](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-scatter.md#series-theme), [Radar](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-radar.md#series-theme), [Pie](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-pie.md#series-theme), [NestedPie](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-nestedPie.md#series-theme), [ColumnLine](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-columnLine.md#series-theme), [LineArea](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-lineArea.md#series-theme), [LineScatter](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-lineScatter.md#series-theme)

The style attributes that were provided as options for the `series` for v3.x have been included in the `theme` attribute in v4.0

| Chart Type | v3.x | v4.0 |
| --- | --- | --- |
| Area Chart | `series.areaOpacity` | `theme.series.areaOpacity` |
| Area, Line Chart | `series.pointWidth` | `theme.series.lineWidth` |
| Bar, Column Chart | `series.barWidth` | `theme.series.barWidth` |

### 4. Tooltip Option

#### Data Display Format Change

In v3.x, users could use `chart.format` option with the `tooltip.suffix` option to affect the data's display formatting. With v4.0, users can modify the display format more flexibly by using functions that can pass parameters.

**v3.x**

```js
const options = {
  chart: { format: '1,000' }
  tooltip: { suffix:  '℃' },
};
```


**v4.0**

```js
const options = {
  tooltip: {
    formatter: (value) => {
      const temp = Number(value.toFixed(2));
      let icon = '☀️';
      if (temp < 0) {
        icon = '❄️';
      } else if (temp > 25) {
        icon = '🔥';
      }

      return `${icon} ${value} ℃`;
    },
  },
};
```

#### Template Usage Change

With v3.x, templates for tooltips only accepted a very limited range of parameters. The v4.0 facilitate the use of the template by providing users with the ability to pass data model, internally defined default tooltip templates (`header`, `body`), and theme information.

**v3.x**
```js
const options = {
  tooltip: {
    template: (category, item, categoryTimestamp) => {
      const head = `<div>${category}</div>`;
      const body = `<div>${item.value}:${item.legend}</div>`;
      return `${head}${body}`;
    }
  }
};
```

**v4.0**
```js
const options = {
  tooltip: {
    template: (model, defaultTooltipTemplate, theme) => {
      const { body, header } = defaultTooltipTemplate;
      const { background } = theme;

      return `
        <div style="
          background: ${background};
          width: 140px;
          padding: 0 5px;
          text-align: center;
          color: white;
          ">
            <p>🎊 ${model.category} 🎊</p>
            ${body}
        </div>
      `;
    }
  }
};
```

#### Mouse Event Detection Methods

With Chart v3.x, the data had to be grouped using `tooltip.grouped` option in order to be detectable via the mouse. Chart v4.0 provides `series.eventDetectType` in order to offer more ways to detect events through the mouse. Refer to the following table for various options compatible with different charts.

| Chart Type | Detection Type | Default Value |
| --- | --- | --- |
| Line, Area Chart | `'near'`, `'nearest'`, `'grouped'`, `'point'` | `'nearest'` |
| Bar, Column, Bullet, BoxPlot | `'grouped'`, `'point'` | `'point'` |
| ColumnLine Chart  | `'grouped'`, `'point'` | `'grouped'` |

### 5. Axes Option

#### Axis Ticks and Label Interval Configurations

v3.x provided `tickInterval` and `labelInterval` options. The `tickInterval` only accepted `'auto'` and the `labelInterval` only allowed users to modify the label intervals. v4.0 allows users to control both tick and label intervals through `tick.interval` and `label.interval`. Additionally, it comes with a new `scale` option, and the `scale` option can be used with `scale.min`, `scale.max`, `scale.stepSize` to control the tick and label intervals more delicately.

**v3.x**
```js
const options = {
  xAxis: {
    min: 0,
    max: 9000,
    tickInterval: 'auto',
    labelInterval: 3
  }
}
```

**v4.0**
```js
const options = {
  xAxis: {
    tick: {
      interval: 3
    },
    label: {
      interval: 6
    },
    scale: {
      min: 0,
      max: 9000,
      stepSize: 1000 // or 'auto'
    }
  }
}
```

#### xAxis Label Date Format Change

With v3.x, users had to define `type: 'datetime'` and `dateFormat` in order to display labels with date format on the x-axis. In v4.0, the datetime option has been changed to `date` for dates.

**v3.x**
```js
const options = {
  xAxis: {
    type: 'datetime',
    dateFormat: 'YYYY-MM-DD'
  }
};
```

**v4.0**
```ts
const options = {
  xAxis: {
    date: { format: 'YYYY-MM-DD' } // or true
  }
};
```

### 6. Data Label Option

Displaying series values with Chart v3.x was very limited and did not allow for a style change. With v4.0, users can define the data label's position and the display format as well as style it in various ways through options.

**v3.x**
```js
const options = {
  series: {
    showLabel: true
  }
};
```

**v4.0**
```js
const options = {
  series: {
    dataLabels: {
      visible: true,
      offsetX: 0,
      offsetY: 0,
      formatter: (value) => `$${value}`
      ...
      /* Data Label Options for Each Chart Series */
    }
  },
  theme: {
    series: {
      dataLabels: {/* */}
    }
  }
};
```

`dataLabels` options for each chart series differ slightly from one another. Refer to the **dataLabels guide** for each chart.

The following is a list of charts that support data label feature.

* [Bar](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-bar.md#datalabels), [Column](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-column.md#datalabels), [Bullet](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-bullet.md#datalabels), [Line](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-line.md#datalabels), [Area](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-area.md#datalabels), [Heatmap](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-heatmap.md#datalabels), [Treemap](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-treemap.md#datalabels), [Pie](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-pie.md#datalabels), [NestedPie](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-nestedPie.md#datalabels), [ColumnLine](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-columnLine.md#datalabels-options), [LineArea](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-lineArea.md#datalabels), [LineScatter](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-lineScatter.md#datalabels-options)

### 7. NestedPie Chart(Pie&Donut Combo Chart, previously)

The Pie & Donut Combo Chart from v3.x was changed to NestedPie Chart in v4.0. In v3.x, the maximum number of charts that could be nested was two, and the series names were fixed to be `'pie1'` and `'pie2'`. In v4.0, in order to address these issues, NestedPie Chart now allows multiple nested charts, and the data can be grouped as necessary.

**v3.x**
```js
const data = {
  categories: ['Browser'],
  seriesAlias: {
    pie1: 'pie',
    pie2: 'pie'
  },
  series: {
    pie1: [/* */],
    pie2: [/* */]
  }
};

const options = {
  series: {
    pie1: {/* */},
    pie2: {/* */}
  }
};

Chart.combochart(el, data, options);
```
> [Guide to Creating v3.x Pie&Donut Combo Chart](https://github.com/nhn/tui.chart/blob/v3.11.2/docs/wiki/chart-types-pie-donut-combo.md)

**v4.0**

```js
const data = {
  series: [
    {
      name: 'browsers',
      data: [/* */],
    },
    {
      name: 'versions',
      data: [/* */],
    },
    {
      name: 'details',
      data: [/* */],
    }
  ],
};

const options = {
  series: {
    browsers: {/* */},
    versions: {/* */},
    details: {/* */}
  }
};

Chart.nestedPieChart({el, data, options});
```

Data can be grouped by setting the parent series by using `parentName`. For the value of the `parentName`, use the `name`(`series.data.name`) of the previous element in the series array.

```js
const data = {
  categories: ['A', 'B'],
  series: [
    {
      name: 'browsers',
      data: [
        {
          name: 'Chrome',
          data: 50,
        },
        ...
      ],
    },
    {
      name: 'versions',
      data: [
        {
          name: 'Chrome 64',
          parentName: 'Chrome',
          data: 40,
        },
        {
          name: 'Chrome 63',
          parentName: 'Chrome',
          data: 10,
        },
        ...
      ],
    },
    {
      name: 'details',
      data: [
        {
          name: 'Chrome 64 - 1',
          parentName: 'Chrome 64',
          data: 25,
        },
        {
          name: 'Chrome 64 - 2',
          parentName: 'Chrome 64',
          data: 15,
        },
        {
          name: 'Chrome 63 - 1',
          parentName: 'Chrome 63',
          data: 7,
        },
        {
          name: 'Chrome 63 - 2',
          parentName: 'Chrome 63',
          data: 3,
        },
        ...
      ],
    },
  ],
};
```

Refer to the [NestedPie Chart](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-nestedPie.md) for more information.

### 8. Name Changes
The following options, methods, and custom events had their names changed, without any change in functionality, from v3.x to v4.0.

* Options

  | v3.x | v4.0 | Details |
  | --- | --- | --- |
  | `xAxis.min` | `xAxis.scale.min` | x-axis minimum value |
  | `xAxis.max` | `xAxis.scale.max` | x-axis maximum value |
  | `yAxis.min` | `yAxis.scale.min` | y-axis minimum value |
  | `yAxis.max` | `yAxis.scale.max` | y-axis maximum value |
  | `series.allowSelect` | `series.selectable` | whether the series can be selected |
  | `series.shifting` | `series.shift` | whether the shift can be used |
  | `chartExportMenu` | `exportMenu` | export option |

* Methods

  | v3.x | v4.0 | Details |
  | --- | --- | --- |
  | `showSeriesLabel` | `showSeriesDataLabel` | show data label |
  | `hideSeriesLabel` | `hideSeriesDataLabel` | hide data label |

* Custom Events

  | v3.x | v4.0 | Details |
  | --- | --- | --- |
  | `'changeCheckedLegends'` | `clickLegendCheckbox` | triggered when legend check box is clicked |
  | `'selectLegend'` | `clickLegendLabel` | triggered when legend label is clicked |

### 9. Improved Live Update

In v3.x, the chart data could be updated using `series.shifting: true` and `addData` methods. These features only worked with v3.x Line and Area charts. In v4.0, the feature can now be used with Column, Heatmap, LineArea, ColumnLine charts as well as Line and Area charts. For using the feature, refer to the [Live Update Guide](https://github.com/nhn/tui.chart/blob/main/docs/en/common-liveUpdate-options.md).

### 10. Added `'auto'` Type for `chart.width` and `chart.height` Options


With v3.x, users were required to provide numeric values for `chart.width` and `chart.height` options. In v4.0, if the users do not provide values for these options, the chart will be rendered according to the size of the container. If the value `'auto'` is given, the `window.resize` event is registered internally, and the chart size will automatically be re-rendered according to the size of the changed container.

### 11. Responsive Option

In v3.x, it was difficult to render charts responsively according to the size changes of the charts due to the lack of addressable API. The Chart v4.0, in order to address this issue, now comes with `responsive` option. For using the option, refer to the [responsive Option](https://github.com/nhn/tui.chart/blob/main/docs/en/common-responsive-options.md).

### 12. Layout Configuration
v3.x only provided `yAxis.maxWidth`, `legend.maxWidth` options so that the users could only set maximum widths of y-axis and legends. v4.0 addresses this issue by allowing users to be able to adjust the sizes and widths of x-axis, y-axis, plot area, and legend. This feature can be used effectively when using multiple charts or having to fix the width of each component. For using the options, refer to the [layout configurations](https://github.com/nhn/tui.chart/blob/main/docs/en/common-layout-options.md).

### 13. Changed Pie Chart Series Option
The features and options for the Pie chart were improved and structured in v4.0. In v3.x, the `radiusRange` option only accepted a string type that includes the `%` character. In v4.0, the option can be given a numeric value and the radius will be calculated to be the absolute value. Furthermore, the `clockwise` option, the direction of the animation in which the chart is drawn, has been added. The `showLegend` and `labelAlign` options from v3.x were combined as the `dataLabels` option in v4.0.

**v3.x**

```js
const options = {
  series: {
    radiusRange: ['60%', '100%'],
    startAngle: -90,
    endAngle: 90,
    showLabel: true,
    showLegend: true,
    labelAlign: 'outer'
  }
};
```

**v4.0**
```js
const options = {
  series: {
    radiusRange: {
      inner: 120, // or '60%',
      outer: 200  // or '100%'
    },
    angleRange: {
      start: -90,
      end: -90
    },
    clockwise: true,
    dataLabels: {
      visible: true,
      pieSeriesName: {
        visible: true,
        anchor: 'outer',
      }
    }
  }
};
```
> [Guide to Creating the v4.0 Pie Charts](https://github.com/nhn/tui.chart/blob/main/docs/en/chart-pie.md)

### 14. New Instance Method

Chart v4.0 provides new instance methods for the users. The newly added methods are as follows.

```js
const chart = new LineChart({ el, data, options });

chart.addSeries(/* */);
chart.setOptions(/* */);
chart.updateOptions(/* */);
chart.getOptions();
chart.showTooltip(/* */);
chart.hideTooltip();
```

| Method Name | Details | Link |
| --- | --- | --- |
| `addSeries` | Adds a new series | [Guide](https://github.com/nhn/tui.chart/blob/main/docs/en/common-api.md#addseries) |
| `setOptions` | Changes all of chart's options | [Guide](https://github.com/nhn/tui.chart/blob/main/docs/en/common-api.md#setoptions) |
| `updateOptions` | Changes the chart's original options | [Guide](https://github.com/nhn/tui.chart/blob/main/docs/en/common-api.md#updateoptions) |
| `getOptions` | Returns the options that are applied to the chart | [Guide](https://github.com/nhn/tui.chart/blob/main/docs/en/common-api.md#getoptions) |
| `showTooltip` | Shows the series tooltip | [Guide](https://github.com/nhn/tui.chart/blob/main/docs/en/common-api.md#showtooltip) |
| `hideTooltip` | Hides the series tooltip | [Guide](https://github.com/nhn/tui.chart/blob/main/docs/en/common-api.md#hidetooltip) |

### 15. New Custom Event
Chart v4.0 provides new custom events for the users. The newly added events are as follows.

```js
const chart = Chart.lineChart({el, data, options});

chart.on('hoverSeries', (info) => {/* */});
chart.on('unhoverSeries', (info) => {/* */});
chart.on('zoom', (dataRange) => {/* */});
chart.on('resetZoom', () => {/* */});
```

| Custom Event Names | Details | Chart Types |
| --- | --- | --- |
| `'hoverSeries'` | Triggered when the mouse hovers over the series data | All Charts |
| `'unhoverSeries'` | Triggered when the mouse leaves the area after triggering `hoverSeries` event | All Charts |
| `'zoom'` | Triggered when zoomed | Line, Area, LineArea, Treemap Chart |
| `'resetZoom'` | Triggered when zoom resets | Line, Area, LineArea, Treemap Charts |

### 16. Map Chart

The v3.x included the Map Chart, but from v4.0, it will be provided as a separate SVG based library.

| v3.x | v4.0 |
| --- | --- |
| Included in TOAST UI Chart | To be provided as a separate SVG based library |

## Removed Features

### 1. Bower Support Dropped

From v4.0, the Chart will no longer support [Bower](https://bower.io/). Since dropping the Bower support, the Github's `production` branch has also been removed. Users who used the Chart from the original `dist` folder of the `production` branch must be extra cautious. Node development environment is encouraged, but Bower can still be used via the CDN.

### 2. Removed APIs List
Lastly, the following is a list of APIs that have been removed from the Chart v4.0.

#### Methods
| Type | Method Name |
| --- | --- |
| Static Methods | `registerPlugin`, `registerTheme`, `registerMap` |
| Instance Methods |  `setTooltipAlign`, `resetTooltipAlign`, `resetTooltipOffset`, `resetTooltipPosition` |

#### Custom Events
* `afterShowTooltip`, `beforeHideTooltip`, `beforeShowTooltip`
