# 🌏 Internationalization (i18n)

TOAST UI Editor provides the ability to set the text set in UI in various languages. There are language files provided by default, you can import each language file and set the code ​​for the language you want to use when you create an instance.

## Files Structure

### Soucre File (For Contributors)

If you want to contribute language files in addition to the [languages ​​supported by default](#supported-languages), add them to the path below. See the [Contributing](#contributing) section for a more detailed contributing process.

```
- tui.editor/apps/editor/src/
  - i18n/
    - en-us.ts
    - ko-kr.ts
    - ...
```

### Build (For Maintainers)

```
- tui.editor/apps/editor/dist/
  - i18n/
    - ko-kr.js
    - ...
```

### Files Distributed on npm

```
- node_modules/@toast-ui/editor/dist/
  - i18n/
    - ko-kr.js
    - ...
```

### Files Distributed on CDN

```
- uicdn.toast.com/editor/latest/
  - i18n/
    - ko-kr.js
    - ko-kr.min.js
    - ...
```

## Supported Languages

Below is a table of valid language codes for i18n files provided by the TOAST UI Editor. This language code is based on the [IETF language tag](https://en.wikipedia.org/wiki/IETF_language_tag). When you import a language file, the language code is registered and you can set the code as the option.

> Note : The default language is English. The production language file(`en-us.js`) for English is not provided and you don't need to import this file.

| Language Name              | i18n File | Registered Code |
| -------------------------- | --------- | --------------- |
| Arabic                     | ar.js     | `ar`            |
| Chinese (S)                | zh-cn.js  | `zh-CN`         |
| Chinese (T)                | zh-tw.js  | `zh-TW`         |
| Croatian (Croatia)         | hr-hr.js  | `hr` \| `hr-HR` |
| Czech (Czech Republic)     | cs-cz.js  | `cs` \| `cs-CZ` |
| Dutch (Netherlands)        | nl-nl.js  | `nl` \| `nl-NL` |
| English (United States)    | en-us.js  | `en` \| `en-US` |
| Finnish (Finland)          | fi-fi.js  | `fi` \| `fi-FI` |
| French (France)            | fr-fr.js  | `fr` \| `fr-FR` |
| Galician (Spain)           | gl-es.js  | `gl` \| `gl-ES` |
| German (Germany)           | de-de.js  | `de` \| `de-DE` |
| Italian (Italy)            | it-it.js  | `it` \| `it-IT` |
| Japanese (Japan)           | ja-jp.js  | `ja` \| `ja-JP` |
| Korean (Korea)             | ko-kr.js  | `ko` \| `ko-KR` |
| Norwegian Bokmål (Norway)  | nb-no.js  | `nb` \| `nb-NO` |
| Polish (Poland)            | pl-pl.js  | `pl` \| `pl-PL` |
| Portuguese (Brazil)        | pt-br.js  | `pt` \| `pt-BR` |
| Russian (Russia)           | ru-ru.js  | `ru` \| `ru-RU` |
| Spanish (Castilian, Spain) | es-es.js  | `es` \| `es-ES` |
| Swedish (Sweden)           | sv-se.js  | `sv` \| `sv-SE` |
| Turkish (Turkey)           | tr-tr.js  | `tr` \| `tr-TR` |
| Ukrainian (Ukraine)        | uk-ua.js  | `uk` \| `uk-UA` |

## Importing Language Files

You must register the language by importing the language file you want to use. The `${fileName}` is corresponding to the 'i18n File' column in [Supported Languages](#supported-languages) (Can be used without an extension).

### ES Modules

```js
import '@toast-ui/editor/dist/i18n/${fileName}';
```

### CommonJS

```js
require('@toast-ui/editor/dist/i18n/${fileName}');
```

### Usage CDN

You must register the language by including the language file you want to use. The `${fileName}` is corresponding to the 'i18n File' column in [Supported Languages](#supported-languages). It also provides a
minified version.

```html
<script src="https://uicdn.toast.com/editor/latest/i18n/${fileName}"></script>
```

## How to Use

> Note : The following examples are based on npm usage.

### Use Case 1 : Basic Usage

If you want to set a specific language, you can use the `language` option to create an instance. The value of this option corresponds to the 'Registered Code' column in [Supported Languages](#supported-languages). The default value is `en` and `en-US`.

```js
import Editor from '@toast-ui/editor';

// Step 1 : Import language file
import '@toast-ui/editor/dist/i18n/ko-kr';

// Step 2 : Set language each editor
const foo = new Editor({
  // Use default language in English
  // ...
});

const bar = new Editor({
  // Use other language
  // ...
  language: 'ko-KR',
});
```

### Use Case 2 : Some Value Overrides

Use the `setLanguage` static method to override the value for a specific language code. See [here](https://github.com/nhn/tui.editor/tree/master/apps/editor/src/i18n/en-us.ts) for default values.

```js
import Editor from '@toast-ui/editor';

// Step 1 : Import language file
import '@toast-ui/editor/dist/i18n/ko-kr';

// Step 2 : Override values of language
Editor.setLanguage('en-US', {
  'Add row': '[Add Row]', // Default value is 'Add row'
});

Editor.setLanguage('ko-KR', {
  'Add row': '[로우 추가]', // Default value is '행 추가'
});

// Step 3 : Set language each editor
const foo = new Editor({
  // Use default language in English
  // ...
});

const bar = new Editor({
  // Use other language
  // ...
  language: 'ko-KR',
});
```

### Use Case 3 : New Language Registration

If the language you want to use is not provided by default, you can register it with the `setLanguage` static method.

```js
import Editor from '@toast-ui/editor';

// Step 1 : Register new language
Editor.setLanguage('en-GB', {
  Markdown: '...',
  WYSIWYG: '...',
  // ...
});

// Step 2 : Set language with new registration code
const bar = new Editor({
  // ...
  language: 'en-GB',
});
```

## Contributing

If you want to contribute to providing a different language file, follow this process:

### Step 1

Fork the repository and add the language file to the path below. The name of the language file follows the `${languageCode}-${countryCode}.js` convention. `languageCode` and `countryCode` must be written in lowercase. (e.g. `en-gb.ts`)

> Reference : [Nominatim/Country Codes](https://wiki.openstreetmap.org/wiki/Nominatim/Country_Codes)

```
- tui.editor/apps/editor/src/
  - i18n/
    - en-us.ts
    - ko-kr.ts
    - ...
```

### Step 2

Refer to [this file](https://github.com/nhn/tui.editor/tree/master/src/i18n/en-us.ts) and write each parameter value used when calling the `setLanguage` method.

The first parameter is a code value that maps to the language file to register. Code values ​​follow the [`${languageCode}-${countryCode}` convention](https://en.wikipedia.org/wiki/IETF_language_tag). `languageCode` must be in lowercase and `countryCode` in uppercase.

```js
// th-th.js
// ...

Editor.setLanguage('th-TH', {
  Markdown: '...',
  WYSIWYG: '...',
  // ...
});
```

If the following conditions are satisfied, the language code except for the country code can be added.

> IETF Language Tag's Reference : Optional script and region subtags are preferred to be omitted when they add no distinguishing information to a language tag. For example, es is preferred over es-Latn, as Spanish is fully expected to be written in the Latin script; ja is preferred over ja-JP, as Japanese as used in Japan does not differ markedly from Japanese as used elsewhere.
>
> Not all linguistic regions can be represented with a valid region subtag: the subnational regional dialects of a primary language are registered as variant subtags. For example, the valencia variant subtag for the Valencian dialect of Catalan is registered in the Language Subtag Registry with the prefix ca. As this dialect is spoken almost exclusively in Spain, the region subtag ES can normally be omitted.

```js
// th-th.js
// ...

Editor.setLanguage(['th', 'th-TH'], {
  Markdown: '...',
  WYSIWYG: '...',
  // ...
});
```

## Example

You can see the example [here](https://nhn.github.io/tui.editor/latest/tutorial-example16-i18n).
