---
navigation.icon: uil:wrench
description: Nuxt is configured with sensible defaults. The config file can override or extend them.
---

# Configuration

By default, Nuxt is configured to cover most use cases. The [`nuxt.config.ts`](/docs/guide/directory-structure/nuxt.config) file can override or extend this default configuration.

## Nuxt Configuration

The `nuxt.config.ts` file is located at the root of a Nuxt project and can override or extend the application's behavior.

A minimal configuration file exports the `defineNuxtConfig` function containing an object with your configuration. The `defineNuxtConfig` helper is globally available without import.

```ts [nuxt.config.ts]
export default defineNuxtConfig({
  // My Nuxt config
})
```

This file will often be mentioned in the documentation, for example to add custom scripts, register modules or change rendering modes.

::alert{type=info}
Every configuration option is described in the [Configuration Reference](/docs/api/configuration/nuxt-config).
::

::alert{type=info}
You don't have to use TypeScript to build an application with Nuxt. However, it is strongly recommended to use the `.ts` extension for the `nuxt.config` file. This way you can benefit from hints in your IDE to avoid typos and mistakes while editing your configuration.
::

### Environment Variables and Private Tokens

The `runtimeConfig` API exposes values like environment variables to the rest of your application. By default, these keys are only available server-side. The keys within `runtimeConfig.public` are also available client-side.

Those values should be defined in `nuxt.config` and can be overridden using environment variables.

::code-group

```ts [nuxt.config.ts]
export default defineNuxtConfig({
  runtimeConfig: {
    // The private keys which are only available server-side
    apiSecret: '123',
    // Keys within public are also exposed client-side
    public: {
      apiBase: '/api'
    }
  }
})
```

```text [.env]
# This will override the value of apiSecret
NUXT_API_SECRET=api_secret_token
```

::

These variables are exposed to the rest of your application using the [`useRuntimeConfig`](/docs/api/composables/use-runtime-config) composable.

```vue [pages/index.vue]
<script setup>
const runtimeConfig = useRuntimeConfig()
</script>
```

:ReadMore{link="/guide/going-further/runtime-config"}

## App Configuration

The `app.config.ts` file, also located at the root of a Nuxt project, is used to expose public variables that can be determined at build time. Contrary to the `runtimeConfig` option, these can not be overriden using environment variables.

A minimal configuration file exports the `defineAppConfig` function containing an object with your configuration. The `defineAppConfig` helper is globally available without import.

```ts [app.config.ts]
export default defineAppConfig({
  title: 'Hello Nuxt',
  theme: {
    dark: true,
    colors: {
      primary: '#ff0000'
    }
  }
})
```

These variables are exposed to the rest of your application using the [`useAppConfig`](/docs/api/composables/use-app-config) composable.

```vue [pages/index.vue]
<script setup>
const appConfig = useAppConfig()
</script>
```

:ReadMore{link="/guide/directory-structure/app.config"}

## `runtimeConfig` vs `app.config`

As stated above, `runtimeConfig` and `app.config` are both used to expose variables to the rest of your application. To determine whether you should use one or the other, here are some guidelines:

- `runtimeConfig`: Private or public tokens that need to be specified after build using environment variables.
- `app.config` : Public tokens that are determined at build time, website configuration such as theme variant, title and any project config that are not sensitive.

Feature                        | `runtimeConfig`  | `app.config`
-------------------------------|------------------|-------------------
Client Side                    | Hydrated         | Bundled
Environment Variables          | ✅ Yes           | ❌ No
Reactive                       | ✅ Yes           | ✅ Yes
Types support                  | ✅ Partial       | ✅ Yes
Configuration per Request      | ❌ No            | ✅ Yes
Hot Module Replacement         | ❌ No            | ✅ Yes
Non primitive JS types         | ❌ No            | ✅ Yes
