---
navigation.icon: uil:database
description: Nuxt provides useState composable to create a reactive and SSR-friendly shared state.
---

# State Management

Nuxt provides useState composable to create a reactive and SSR-friendly shared state across components.

`useState` is an SSR-friendly [`ref`](https://vuejs.org/api/reactivity-core.html#ref) replacement. Its value will be preserved after server-side rendering (during client-side hydration) and shared across all components using a unique key.

::ReadMore{link="/docs/api/composables/use-state"}
::

::alert{icon=👉}
`useState` only works during `setup` or [`Lifecycle Hooks`](https://vuejs.org/api/composition-api-lifecycle.html#composition-api-lifecycle-hooks).
::
::alert{type=warning}
Because the data inside `useState` will be serialized to JSON, it is important that it does not contain anything that cannot be serialized, such as classes, functions or symbols.
::

## Best Practices

::alert{type=danger icon=🚨}
Never define `const state = ref()` outside of `<script setup>` or `setup()` function.<br>
Such state will be shared across all users visiting your website and can lead to memory leaks!
::
::alert{type=success icon=✅}
Instead use `const useX = () => useState('x')`
::

## Examples

### Basic Usage

In this example, we use a component-local counter state. Any other component that uses `useState('counter')` shares the same reactive state.

```vue [app.vue]
<script setup>
const counter = useState('counter', () => Math.round(Math.random() * 1000))
</script>

<template>
  <div>
    Counter: {{ counter }}
    <button @click="counter++">
      +
    </button>
    <button @click="counter--">
      -
    </button>
  </div>
</template>
```

::LinkExample{link="/docs/examples/composables/use-state"}
::

::ReadMore{link="/docs/api/composables/use-state"}
::

### Advanced

In this example, we use a composable that detects the user's default locale from the HTTP request headers and keeps it in a `locale` state.

::LinkExample{link="/docs/examples/other/locale"}
::

## Shared State

By using [auto-imported composables](/docs/guide/directory-structure/composables) we can define global type-safe states and import them across the app.

```ts [composables/states.ts]
export const useCounter = () => useState<number>('counter', () => 0)
export const useColor = () => useState<string>('color', () => 'pink')
```

```vue [app.vue]
<script setup>
const color = useColor() // Same as useState('color')
</script>

<template>
  <p>Current color: {{ color }}</p>
</template>
```
