---
description: "Nuxt auto-imports helper functions, composables and Vue APIs."
---

# Auto imports

Nuxt auto-imports helper functions, composables and Vue APIs to use across your application without explicitly importing them. Based on the directory structure, every Nuxt application can also use auto-imports for its own components, composables and plugins. Components, composables or plugins can use these functions.

Contrary to a classic global declaration, Nuxt preserves typings and IDEs completions and hints, and only includes what is actually used in your production code.

::alert{type=info icon=💡}
In the documentation, every function that is not explicitly imported is auto-imported by Nuxt and can be used as-is in your code.
You can find a reference for auto-imported [composables](/docs/api/composables/use-async-data) and [utilities](/docs/api/utils/dollarfetch) in the API section.
::

::alert{type=warning}
Auto imports don't currently work within the [server directory](/docs/guide/directory-structure/server).
::

## Nuxt Auto-imports

Nuxt auto-imports functions and composables to perform [data fetching](/docs/getting-started/data-fetching), get access to the [app context](/docs/api/composables/use-nuxt-app) and [runtime config](/docs/guide/going-further/runtime-config), manage [state](/docs/getting-started/state-management) or define components and plugins.

```vue
<script setup>
  /* useAsyncData() and $fetch() are auto-imported */
  const { data, refresh, pending } = await useAsyncData('/api/hello', () => $fetch('/api/hello'))
</script>
```

## Vue Auto-imports

Vue 3 exposes Reactivity APIs like `ref` or `computed`, as well as lifecycle hooks and helpers that are auto-imported by Nuxt.

```vue
<script setup>
  /* ref() and computed() are auto-imported */
  const count = ref(1)
  const double = computed(() => count.value * 2)
</script>
```

## Directory-based Auto-imports

Nuxt directly auto-imports files created in defined directories:

- `components/` for [Vue components](/docs/guide/directory-structure/components).
- `composables/` for [Vue composables](/docs/guide/directory-structure/composables).

## Explicit Imports

Nuxt exposes every auto-import with the `#imports` alias that can be used to make the import explicit if needed:

```vue
<script setup>
  import { ref, computed } from '#imports'

  const count = ref(1)
  const double = computed(() => count.value * 2)
</script>
```

## Disable Auto-imports

In case you want to disable auto-imports, you can set `imports.autoImport` to `false`.

```ts [nuxt.config.ts]
export default defineNuxtConfig({
  imports: {
    autoImport: false
  }
})
```

This will disable implicit auto imports completely but it's still possible to use [Explicit Imports](#explicit-imports).
