---
title: "<NuxtLink>"
description: "Nuxt provides <NuxtLink> component to handle any kind of links within your application."
---

# `<NuxtLink>`

Nuxt provides `<NuxtLink>` component to handle any kind of links within your application.

`<NuxtLink>` is a drop-in replacement for both Vue Router's `<RouterLink>` component and HTML's `<a>` tag. It intelligently determines whether the link is _internal_ or _external_ and renders it accordingly with available optimizations (prefetching, default attributes, etc.)

## Examples

### Basic Usage

In this example, we use `<NuxtLink>` component to link to a website.

```vue [app.vue]
<template>
  <NuxtLink to="https://nuxtjs.org">
    Nuxt website
  </NuxtLink>
  <!-- <a href="https://nuxtjs.org" rel="noopener noreferrer">...</a> -->
</template>
```

:button-link[Open on StackBlitz]{href="https://stackblitz.com/github/nuxt/framework/tree/main/examples/routing/nuxt-link?terminal=dev&file=/pages/index.vue" blank}

### Internal Routing

In this example, we use `<NuxtLink>` component to link to another page of the application.

```vue [pages/index.vue]
<template>
  <NuxtLink to="/about">
    About page
  </NuxtLink>
  <!-- <a href="/about">...</a> (+Vue Router & prefetching) -->
</template>
```

:button-link[Open on StackBlitz]{href="https://stackblitz.com/github/nuxt/framework/tree/main/examples/routing/nuxt-link?terminal=dev&file=/pages/index.vue" blank}

### `target` and `rel` Attributes

In this example, we use `<NuxtLink>` with `target`, `rel`, and `noRel` props.

```vue [app.vue]
<template>
  <NuxtLink to="https://twitter.com/nuxt_js" target="_blank">
    Nuxt Twitter
  </NuxtLink>
  <!-- <a href="https://twitter.com/nuxt_js" target="_blank" rel="noopener noreferrer">...</a> -->

  <NuxtLink to="https://discord.nuxtjs.org" target="_blank" rel="noopener">
    Nuxt Discord
  </NuxtLink>
  <!-- <a href="https://discord.nuxtjs.org" target="_blank" rel="noopener">...</a> -->

  <NuxtLink to="https://github.com/nuxt" no-rel>
    Nuxt GitHub
  </NuxtLink>
  <!-- <a href="https://github.com/nuxt">...</a> -->

  <NuxtLink to="/contact" target="_blank">
    Contact page opens in another tab
  </NuxtLink>
  <!-- <a href="/contact" target="_blank" rel="noopener noreferrer">...</a> -->
</template>
```

:button-link[Open on StackBlitz]{href="https://stackblitz.com/github/nuxt/framework/tree/main/examples/routing/nuxt-link?terminal=dev&file=/pages/index.vue" blank}

## Props

- **to**: Any URL or a [route location object](https://router.vuejs.org/api/#routelocationraw) from Vue Router
- **href**: An alias for `to`. If used with `to`, `href` will be ignored
- **target**: A `target` attribute value to apply on the link
- **rel**: A `rel` attribute value to apply on the link. Defaults to `"noopener noreferrer"` for external links.
- **noRel**: If set to `true`, no `rel` attribute will be added to the link
- **activeClass**: A class to apply on active links. Works the same as [Vue Router's `active-class` prop](https://router.vuejs.org/api/#active-class) on internal links. Defaults to Vue Router's default (`"router-link-active"`)
- **exactActiveClass**: A class to apply on exact active links. Works the same as [Vue Router's `exact-active-class` prop](https://router.vuejs.org/api/#exact-active-class) on internal links. Defaults to Vue Router's default `"router-link-exact-active"`)
- **replace**: Works the same as [Vue Router's `replace` prop](https://router.vuejs.org/api/#replace) on internal links
- **ariaCurrentValue**: An `aria-current` attribute value to apply on exact active links. Works the same as [Vue Router's `aria-current-value` prop](https://router.vuejs.org/api/#aria-current-value) on internal links
- **external**: Forces the link to be considered as external (`true`) or internal (`false`). This is helpful to handle edge-cases
- **prefetch** and **noPrefetch**: Whether to enable prefetching assets for links that enter the view port.
- **prefetchedClass**: A class to apply to links that have been prefetched.
- **custom**: Whether `<NuxtLink>` should wrap its content in an `<a>` element. It allows taking full control of how a link is rendered and how navigation works when it is clicked. Works the same as [Vue Router's `custom` prop](https://router.vuejs.org/api/#custom)

::alert{icon=👉}
Defaults can be overwritten, see [overwriting defaults](#overwriting-defaults) if you want to change them.
::

## Overwriting Defaults

You can overwrite `<NuxtLink>` defaults by creating your own link component using `defineNuxtLink`.

```js [components/MyNuxtLink.ts]
export default defineNuxtLink({
  componentName: 'MyNuxtLink',
  /* see signature below for more */
})
```

You can then use `<MyNuxtLink />` component as usual with your new defaults.

:button-link[Open on StackBlitz]{href="https://stackblitz.com/github/nuxt/framework/tree/main/examples/routing/nuxt-link?terminal=dev&file=/components/MyNuxtLink.ts" blank}

### `defineNuxtLink` Signature

```ts
defineNuxtLink({
  componentName?: string;
  externalRelAttribute?: string;
  activeClass?: string;
  exactActiveClass?: string;
  prefetchedClass?: string;
}) => Component
```

- **componentName**: A name for the defined `<NuxtLink>` component.
- **externalRelAttribute**: A default `rel` attribute value applied on external links. Defaults to `"noopener noreferrer"`. Set it to `""` to disable
- **activeClass**: A default class to apply on active links. Works the same as [Vue Router's `linkActiveClass` option](https://router.vuejs.org/api/#linkactiveclass). Defaults to Vue Router's default (`"router-link-active"`)
- **exactActiveClass**: A default class to apply on exact active links. Works the same as [Vue Router's `linkExactActiveClass` option](https://router.vuejs.org/api/#linkexactactiveclass). Defaults to Vue Router's default (`"router-link-exact-active"`)
- **prefetchedClass**: A default class to apply to links that have been prefetched.

::LinkExample{link="/docs/examples/routing/nuxt-link"}
::
