"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var Constants_1 = require("./Constants");
var Enums_1 = require("./Enums");
var Utils_1 = require("./Utils");
/**
 * Incoming SIP message sanity check.
 * @function
 * @param {SIP.IncomingMessage} message
 * @param {SIP.UA} ua
 * @param {SIP.Transport} transport
 * @returns {Boolean}
 */
var SanityCheck;
(function (SanityCheck) {
    // Reply
    function reply(statusCode, message, transport) {
        var response = Utils_1.Utils.buildStatusLine(statusCode);
        var vias = message.getHeaders("via");
        for (var _i = 0, vias_1 = vias; _i < vias_1.length; _i++) {
            var via = vias_1[_i];
            response += "Via: " + via + "\r\n";
        }
        var to = message.getHeader("To") || "";
        if (!message.toTag) {
            to += ";tag=" + Utils_1.Utils.newTag();
        }
        response += "To: " + to + "\r\n";
        response += "From: " + message.getHeader("From") + "\r\n";
        response += "Call-ID: " + message.callId + "\r\n";
        response += "CSeq: " + message.cseq + " " + message.method + "\r\n";
        response += "\r\n";
        transport.send(response);
    }
    SanityCheck.reply = reply;
    /*
    * Sanity Check for incoming Messages
    *
    * Requests:
    *  - _rfc3261_8_2_2_1_ Receive a Request with a non supported URI scheme
    *  - _rfc3261_16_3_4_ Receive a Request already sent by us
    *   Does not look at via sent-by but at sipjsId, which is inserted as
    *   a prefix in all initial requests generated by the ua
    *  - _rfc3261_18_3_request_ Body Content-Length
    *  - _rfc3261_8_2_2_2_ Merged Requests
    *
    * Responses:
    *  - _rfc3261_8_1_3_3_ Multiple Via headers
    *  - _rfc3261_18_1_2_ sent-by mismatch
    *  - _rfc3261_18_3_response_ Body Content-Length
    *
    * All:
    *  - Minimum headers in a SIP message
    */
    // Sanity Check functions for requests
    function rfc3261_8_2_2_1(message, ua, transport) {
        if (!message.ruri || message.ruri.scheme !== "sip") {
            reply(416, message, transport);
            return false;
        }
        return true;
    }
    SanityCheck.rfc3261_8_2_2_1 = rfc3261_8_2_2_1;
    function rfc3261_16_3_4(message, ua, transport) {
        if (!message.toTag) {
            if (message.callId.substr(0, 5) === ua.configuration.sipjsId) {
                reply(482, message, transport);
                return false;
            }
        }
        return true;
    }
    SanityCheck.rfc3261_16_3_4 = rfc3261_16_3_4;
    function rfc3261_18_3_request(message, ua, transport) {
        var len = Utils_1.Utils.str_utf8_length(message.body);
        var contentLength = message.getHeader("content-length");
        if (contentLength && len < Number(contentLength)) {
            reply(400, message, transport);
            return false;
        }
        return true;
    }
    SanityCheck.rfc3261_18_3_request = rfc3261_18_3_request;
    function rfc3261_8_2_2_2(message, ua, transport) {
        var fromTag = message.fromTag;
        var callId = message.callId;
        var cseq = message.cseq;
        if (!message.toTag) {
            if (message.method === Constants_1.C.INVITE) {
                if (ua.transactions.ist[message.viaBranch]) {
                    return true;
                }
                else {
                    for (var idx in ua.transactions.ist) {
                        if (ua.transactions.ist.hasOwnProperty(idx)) {
                            var tr = ua.transactions.ist[idx];
                            if (tr.request.fromTag === fromTag && tr.request.callId === callId && tr.request.cseq === cseq) {
                                reply(482, message, transport);
                                return false;
                            }
                        }
                    }
                }
            }
            else {
                if (ua.transactions.nist[message.viaBranch]) {
                    return true;
                }
                else {
                    for (var idx in ua.transactions.nist) {
                        if (ua.transactions.nist.hasOwnProperty(idx)) {
                            var tr = ua.transactions.nist[idx];
                            if (tr.request.fromTag === fromTag && tr.request.callId === callId && tr.request.cseq === cseq) {
                                reply(482, message, transport);
                                return false;
                            }
                        }
                    }
                }
            }
        }
        return true;
    }
    SanityCheck.rfc3261_8_2_2_2 = rfc3261_8_2_2_2;
    // Sanity Check functions for responses
    function rfc3261_8_1_3_3(message, ua) {
        if (message.getHeaders("via").length > 1) {
            ua.getLogger("sip.sanitycheck").warn("More than one Via header field present in the response." +
                " Dropping the response");
            return false;
        }
        return true;
    }
    SanityCheck.rfc3261_8_1_3_3 = rfc3261_8_1_3_3;
    function rfc3261_18_1_2(message, ua) {
        if (message.via.host !== ua.configuration.viaHost || message.via.port !== undefined) {
            ua.getLogger("sip.sanitycheck").warn("Via sent-by in the response does not match UA Via host value." +
                " Dropping the response");
            return false;
        }
        return true;
    }
    SanityCheck.rfc3261_18_1_2 = rfc3261_18_1_2;
    function rfc3261_18_3_response(message, ua) {
        var len = Utils_1.Utils.str_utf8_length(message.body);
        var contentLength = message.getHeader("content-length");
        if (contentLength && len < Number(contentLength)) {
            ua.getLogger("sip.sanitycheck").warn("Message body length is lower than the value in" +
                " Content-Length header field. Dropping the response");
            return false;
        }
        return true;
    }
    SanityCheck.rfc3261_18_3_response = rfc3261_18_3_response;
    // Sanity Check functions for requests and responses
    function minimumHeaders(message, ua) {
        var mandatoryHeaders = ["from", "to", "call_id", "cseq", "via"];
        for (var _i = 0, mandatoryHeaders_1 = mandatoryHeaders; _i < mandatoryHeaders_1.length; _i++) {
            var header = mandatoryHeaders_1[_i];
            if (!message.hasHeader(header)) {
                ua.getLogger("sip.sanitycheck").warn("Missing mandatory header field : " +
                    header + ". Dropping the response");
                return false;
            }
        }
        return true;
    }
    SanityCheck.minimumHeaders = minimumHeaders;
    function sanityCheck(message, ua, transport) {
        var requests = [
            rfc3261_8_2_2_1,
            rfc3261_16_3_4,
            rfc3261_18_3_request,
            rfc3261_8_2_2_2
        ];
        var responses = [
            rfc3261_8_1_3_3,
            rfc3261_18_1_2,
            rfc3261_18_3_response
        ];
        var all = [
            minimumHeaders
        ];
        for (var _i = 0, all_1 = all; _i < all_1.length; _i++) {
            var checkFunction = all_1[_i];
            if (!checkFunction(message, ua, transport)) {
                return false;
            }
        }
        if (message.type === Enums_1.TypeStrings.IncomingRequest) {
            for (var _a = 0, requests_1 = requests; _a < requests_1.length; _a++) {
                var checkFunction = requests_1[_a];
                if (!checkFunction(message, ua, transport)) {
                    return false;
                }
            }
        }
        else if (message.type === Enums_1.TypeStrings.IncomingResponse) {
            for (var _b = 0, responses_1 = responses; _b < responses_1.length; _b++) {
                var checkFunction = responses_1[_b];
                if (!checkFunction(message, ua, transport)) {
                    return false;
                }
            }
        }
        // Everything is OK
        return true;
    }
    SanityCheck.sanityCheck = sanityCheck;
})(SanityCheck = exports.SanityCheck || (exports.SanityCheck = {}));
