import { IncomingRequest, IncomingResponse } from "../types/sip-message";
import { InviteServerTransaction, NonInviteServerTransaction } from "../types/transactions";
import { Transport } from "../types/transport";
import { UA } from "../types/ua";

import { C } from "./Constants";
import { TypeStrings } from "./Enums";
import { Utils } from "./Utils";

/**
 * Incoming SIP message sanity check.
 * @function
 * @param {SIP.IncomingMessage} message
 * @param {SIP.UA} ua
 * @param {SIP.Transport} transport
 * @returns {Boolean}
 */

export namespace SanityCheck {
  // Reply
  export function reply(
    statusCode: number,
    message: IncomingRequest | IncomingResponse,
    transport: Transport
  ): void {
    let response: string = Utils.buildStatusLine(statusCode);
    const vias = message.getHeaders("via");

    for (const via of vias) {
      response += "Via: " + via + "\r\n";
    }

    let to: string = message.getHeader("To") || "";

    if (!message.toTag) {
      to += ";tag=" + Utils.newTag();
    }

    response += "To: " + to + "\r\n";
    response += "From: " + message.getHeader("From") + "\r\n";
    response += "Call-ID: " + message.callId + "\r\n";
    response += "CSeq: " + message.cseq + " " + message.method + "\r\n";
    response += "\r\n";

    transport.send(response);
  }

  /*
  * Sanity Check for incoming Messages
  *
  * Requests:
  *  - _rfc3261_8_2_2_1_ Receive a Request with a non supported URI scheme
  *  - _rfc3261_16_3_4_ Receive a Request already sent by us
  *   Does not look at via sent-by but at sipjsId, which is inserted as
  *   a prefix in all initial requests generated by the ua
  *  - _rfc3261_18_3_request_ Body Content-Length
  *  - _rfc3261_8_2_2_2_ Merged Requests
  *
  * Responses:
  *  - _rfc3261_8_1_3_3_ Multiple Via headers
  *  - _rfc3261_18_1_2_ sent-by mismatch
  *  - _rfc3261_18_3_response_ Body Content-Length
  *
  * All:
  *  - Minimum headers in a SIP message
  */

  // Sanity Check functions for requests
  export function rfc3261_8_2_2_1(message: IncomingRequest, ua: UA, transport: Transport): boolean {
    if (!message.ruri || message.ruri.scheme !== "sip") {
      reply(416, message, transport);
      return false;
    }
    return true;
  }

  export function rfc3261_16_3_4(message: IncomingRequest, ua: UA, transport: Transport): boolean {
    if (!message.toTag) {
      if (message.callId.substr(0, 5) === ua.configuration.sipjsId) {
        reply(482, message, transport);
        return false;
      }
    }
    return true;
  }

  export function rfc3261_18_3_request(
    message: IncomingRequest | IncomingResponse,
    ua: UA,
    transport: Transport
  ): boolean {
    const len: number = Utils.str_utf8_length(message.body);
    const contentLength: string | undefined = message.getHeader("content-length");

    if (contentLength && len < Number(contentLength)) {
      reply(400, message, transport);
      return false;
    }
    return true;
  }

  export function rfc3261_8_2_2_2(message: IncomingRequest, ua: UA, transport: Transport): boolean {
    const fromTag: string = message.fromTag;
    const callId: string = message.callId;
    const cseq: number = message.cseq;

    if (!message.toTag) {
      if (message.method === C.INVITE) {
        if (ua.transactions.ist[message.viaBranch]) {
          return true;
        } else {
          for (const idx in ua.transactions.ist) {
            if (ua.transactions.ist.hasOwnProperty(idx)) {
              const tr: InviteServerTransaction = ua.transactions.ist[idx];
              if (tr.request.fromTag === fromTag && tr.request.callId === callId && tr.request.cseq === cseq) {
                reply(482, message, transport);
                return false;
              }
            }
          }
        }
      } else {
        if (ua.transactions.nist[message.viaBranch]) {
          return true;
        } else {
          for (const idx in ua.transactions.nist) {
            if (ua.transactions.nist.hasOwnProperty(idx)) {
              const tr: NonInviteServerTransaction = ua.transactions.nist[idx];
              if (tr.request.fromTag === fromTag && tr.request.callId === callId && tr.request.cseq === cseq) {
                reply(482, message, transport);
                return false;
              }
            }
          }
        }
      }
    }
    return true;
  }

  // Sanity Check functions for responses
  export function rfc3261_8_1_3_3(message: IncomingResponse, ua: UA): boolean {
    if (message.getHeaders("via").length > 1) {
      ua.getLogger("sip.sanitycheck").warn("More than one Via header field present in the response." +
        " Dropping the response");
      return false;
    }
    return true;
  }

  export function rfc3261_18_1_2(message: IncomingResponse, ua: UA): boolean {
    if (message.via.host !== ua.configuration.viaHost || message.via.port !== undefined) {
      ua.getLogger("sip.sanitycheck").warn("Via sent-by in the response does not match UA Via host value." +
        " Dropping the response");
      return false;
    }
    return true;
  }

  export function rfc3261_18_3_response(message: IncomingResponse, ua: UA): boolean {
    const len: number = Utils.str_utf8_length(message.body);
    const contentLength: string | undefined = message.getHeader("content-length");

    if (contentLength && len < Number(contentLength)) {
      ua.getLogger("sip.sanitycheck").warn("Message body length is lower than the value in" +
        " Content-Length header field. Dropping the response");
      return false;
    }
    return true;
  }

  // Sanity Check functions for requests and responses
  export function minimumHeaders(message: IncomingRequest | IncomingResponse, ua: UA): boolean {
    const mandatoryHeaders: Array<string> = ["from", "to", "call_id", "cseq", "via"];

    for (const header of mandatoryHeaders) {
      if (!message.hasHeader(header)) {
        ua.getLogger("sip.sanitycheck").warn("Missing mandatory header field : " +
          header + ". Dropping the response");
        return false;
      }
    }
    return true;
  }

  export function sanityCheck(
    message: IncomingRequest | IncomingResponse,
    ua: UA,
    transport: Transport
  ): boolean {
    const requests: Array<any> = [
      rfc3261_8_2_2_1,
      rfc3261_16_3_4,
      rfc3261_18_3_request,
      rfc3261_8_2_2_2
    ];

    const responses: Array<any> = [
      rfc3261_8_1_3_3,
      rfc3261_18_1_2,
      rfc3261_18_3_response
    ];

    const all: Array<any> = [
      minimumHeaders
    ];

    for (const checkFunction of all) {
      if (!checkFunction(message, ua, transport)) {
        return false;
      }
    }

    if (message.type === TypeStrings.IncomingRequest) {
      for (const checkFunction of requests) {
        if (!checkFunction(message, ua, transport)) {
          return false;
        }
      }
    } else if (message.type === TypeStrings.IncomingResponse) {
      for (const checkFunction of responses) {
        if (!checkFunction(message, ua, transport)) {
          return false;
        }
      }
    }

    // Everything is OK
    return true;
  }
}
