README={}
README['str'] = 
"# Getting Codepages\n\nThe fields of the pages.csv manifest are `codepage,url,bytes` (SBCS=1, DBCS=2)\n\n```>pages.csv\n37,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/EBCDIC/CP037.TXT,1\n437,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP437.TXT,1\n500,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/EBCDIC/CP500.TXT,1\n737,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP737.TXT,1\n775,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP775.TXT,1\n850,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP850.TXT,1\n852,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP852.TXT,1\n855,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP855.TXT,1\n857,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP857.TXT,1\n860,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP860.TXT,1\n861,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP861.TXT,1\n862,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP862.TXT,1\n863,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP863.TXT,1\n864,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP864.TXT,1\n865,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP865.TXT,1\n866,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP866.TXT,1\n869,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/PC/CP869.TXT,1\n874,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP874.TXT,1\n875,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/EBCDIC/CP875.TXT,1\n932,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP932.TXT,2\n936,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP936.TXT,2\n949,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP949.TXT,2\n950,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP950.TXT,2\n1026,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/EBCDIC/CP1026.TXT,1\n1250,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP1250.TXT,1\n1251,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP1251.TXT,1\n1252,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP1252.TXT,1\n1253,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP1253.TXT,1\n1254,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP1254.TXT,1\n1255,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP1255.TXT,1\n1256,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP1256.TXT,1\n1257,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP1257.TXT,1\n1258,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/WINDOWS/CP1258.TXT,1\n```\n\nNote that the Windows rendering is used for the Mac code pages.  The primary\ndifference is the use of the private `0xF8FF` code (which renders as an Apple\nlogo on macs but as garbage on other operating systems).  It may be desirable\nto fall back to the behavior, in which case the files are under APPLE and not\nMICSFT.  Codepages are an absolute pain :/\n\n```>pages.csv\n10000,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/MAC/ROMAN.TXT,1\n10006,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/MAC/GREEK.TXT,1\n10007,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/MAC/CYRILLIC.TXT,1\n10029,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/MAC/LATIN2.TXT,1\n10079,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/MAC/ICELAND.TXT,1\n10081,http://www.unicode.org/Public/MAPPINGS/VENDORS/MICSFT/MAC/TURKISH.TXT,1\n```\n\nThe numbering scheme for the `ISO-8859-X` series is `28590 + X`:\n\n```>pages.csv\n28591,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-1.TXT,1\n28592,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-2.TXT,1\n28593,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-3.TXT,1\n28594,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-4.TXT,1\n28595,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-5.TXT,1\n28596,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-6.TXT,1\n28597,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-7.TXT,1\n28598,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-8.TXT,1\n28599,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-9.TXT,1\n28600,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-10.TXT,1\n28601,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-11.TXT,1\n28603,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-13.TXT,1\n28604,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-14.TXT,1\n28605,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-15.TXT,1\n28606,http://www.unicode.org/Public/MAPPINGS/ISO8859/8859-16.TXT,1\n```\n\n## Generated Codepages\n\nThe following codepages are available in .NET on Windows:\n\n- 708   Arabic (ASMO 708)\n- 720   Arabic (Transparent ASMO); Arabic (DOS)\n- 858   OEM Multilingual Latin 1 + Euro symbol\n- 870   IBM EBCDIC Multilingual/ROECE (Latin 2); IBM EBCDIC Multilingual Latin 2\n- 1047  IBM EBCDIC Latin 1/Open System\n- 1140  IBM EBCDIC US-Canada (037 + Euro symbol); IBM EBCDIC (US-Canada-Euro)\n- 1141  IBM EBCDIC Germany (20273 + Euro symbol); IBM EBCDIC (Germany-Euro)\n- 1142  IBM EBCDIC Denmark-Norway (20277 + Euro symbol); IBM EBCDIC (Denmark-Norway-Euro)\n- 1143  IBM EBCDIC Finland-Sweden (20278 + Euro symbol); IBM EBCDIC (Finland-Sweden-Euro)\n- 1144  IBM EBCDIC Italy (20280 + Euro symbol); IBM EBCDIC (Italy-Euro)\n- 1145  IBM EBCDIC Latin America-Spain (20284 + Euro symbol); IBM EBCDIC (Spain-Euro)\n- 1146  IBM EBCDIC United Kingdom (20285 + Euro symbol); IBM EBCDIC (UK-Euro)\n- 1147  IBM EBCDIC France (20297 + Euro symbol); IBM EBCDIC (France-Euro)\n- 1148  IBM EBCDIC International (500 + Euro symbol); IBM EBCDIC (International-Euro)\n- 1149  IBM EBCDIC Icelandic (20871 + Euro symbol); IBM EBCDIC (Icelandic-Euro)\n- 1361  Korean (Johab)\n- 10001 Japanese (Mac)\n- 10002 MAC Traditional Chinese (Big5); Chinese Traditional (Mac)\n- 10003 Korean (Mac)\n- 10004 Arabic (Mac)\n- 10005 Hebrew (Mac)\n- 10008 MAC Simplified Chinese (GB 2312); Chinese Simplified (Mac)\n- 10010 Romanian (Mac)\n- 10017 Ukrainian (Mac)\n- 10021 Thai (Mac)\n- 10082 Croatian (Mac)\n- 20000 CNS Taiwan; Chinese Traditional (CNS)\n- 20001 TCA Taiwan\n- 20002 Eten Taiwan; Chinese Traditional (Eten)\n- 20003 IBM5550 Taiwan\n- 20004 TeleText Taiwan\n- 20005 Wang Taiwan\n- 20105 IA5 (IRV International Alphabet No. 5, 7-bit); Western European (IA5)\n- 20106 IA5 German (7-bit)\n- 20107 IA5 Swedish (7-bit)\n- 20108 IA5 Norwegian (7-bit)\n- 20261 T.61\n- 20269 ISO 6937 Non-Spacing Accent\n- 20273 IBM EBCDIC Germany\n- 20277 IBM EBCDIC Denmark-Norway\n- 20278 IBM EBCDIC Finland-Sweden\n- 20280 IBM EBCDIC Italy\n- 20284 IBM EBCDIC Latin America-Spain\n- 20285 IBM EBCDIC United Kingdom\n- 20290 IBM EBCDIC Japanese Katakana Extended\n- 20297 IBM EBCDIC France\n- 20420 IBM EBCDIC Arabic\n- 20423 IBM EBCDIC Greek\n- 20424 IBM EBCDIC Hebrew\n- 20833 IBM EBCDIC Korean Extended\n- 20838 IBM EBCDIC Thai\n- 20866 Russian (KOI8-R); Cyrillic (KOI8-R)\n- 20871 IBM EBCDIC Icelandic\n- 20880 IBM EBCDIC Cyrillic Russian\n- 20905 IBM EBCDIC Turkish\n- 20924 IBM EBCDIC Latin 1/Open System (1047 + Euro symbol)\n- 20932 Japanese (JIS 0208-1990 and 0212-1990)\n- 20936 Simplified Chinese (GB2312); Chinese Simplified (GB2312-80)\n- 20949 Korean Wansung\n- 21025 IBM EBCDIC Cyrillic Serbian-Bulgarian\n- 21027 Extended/Ext Alpha Lowercase\n- 21866 Ukrainian (KOI8-U); Cyrillic (KOI8-U)\n- 29001 Europa 3\n- 38598 ISO 8859-8 Hebrew; Hebrew (ISO-Logical)\n- 50220 ISO 2022 Japanese with no halfwidth Katakana; Japanese (JIS)\n- 50221 ISO 2022 Japanese with halfwidth Katakana; Japanese (JIS-Allow 1 byte Kana)\n- 50222 ISO 2022 Japanese JIS X 0201-1989; Japanese (JIS-Allow 1 byte Kana - SO/SI)\n- 50225 ISO 2022 Korean\n- 50227 ISO 2022 Simplified Chinese; Chinese Simplified (ISO 2022)\n- 51932 EUC Japanese\n- 51936 EUC Simplified Chinese; Chinese Simplified (EUC)\n- 51949 EUC Korean\n- 52936 HZ-GB2312 Simplified Chinese; Chinese Simplified (HZ)\n- 54936 Windows XP and later: GB18030 Simplified Chinese (4 byte); Chinese Simplified (GB18030)\n- 57002 ISCII Devanagari\n- 57003 ISCII Bengali\n- 57004 ISCII Tamil\n- 57005 ISCII Telugu\n- 57006 ISCII Assamese\n- 57007 ISCII Oriya\n- 57008 ISCII Kannada\n- 57009 ISCII Malayalam\n- 57010 ISCII Gujarati\n- 57011 ISCII Punjabi\n\n```>pages.csv\n708,,1\n720,,1\n858,,1\n870,,1\n1047,,1\n1140,,1\n1141,,1\n1142,,1\n1143,,1\n1144,,1\n1145,,1\n1146,,1\n1147,,1\n1148,,1\n1149,,1\n1361,,2\n10001,,2\n10002,,2\n10003,,2\n10004,,1\n10005,,1\n10008,,2\n10010,,1\n10017,,1\n10021,,1\n10082,,1\n20000,,2\n20001,,2\n20002,,2\n20003,,2\n20004,,2\n20005,,2\n20105,,1\n20106,,1\n20107,,1\n20108,,1\n20261,,2\n20269,,1\n20273,,1\n20277,,1\n20278,,1\n20280,,1\n20284,,1\n20285,,1\n20290,,1\n20297,,1\n20420,,1\n20423,,1\n20424,,1\n20833,,1\n20838,,1\n20866,,1\n20871,,1\n20880,,1\n20905,,1\n20924,,1\n20932,,2\n20936,,2\n20949,,2\n21025,,1\n21027,,1\n21866,,1\n29001,,1\n38598,,1\n50220,,2\n50221,,2\n50222,,2\n50225,,2\n50227,,2\n51932,,2\n51936,,2\n51949,,2\n52936,,2\n54936,,2\n57002,,2\n57003,,2\n57004,,2\n57005,,2\n57006,,2\n57007,,2\n57008,,2\n57009,,2\n57010,,2\n57011,,2\n```\n\nThe following codepages are dependencies for Visual FoxPro:\n\n- 620 Mazovia (Polish) MS-DOS\n- 895 Kamenický (Czech) MS-DOS\n\n```>pages.csv\n620,,1\n895,,1\n```\n\nThe known missing codepages are enumerated in the README.\n\n## Building Notes\n\nThe script `make.sh` (described later) will get these files and massage the data\n(printing code-unicode pairs).  The eventual tables are dropped in the paths\n`./codepages/<CODEPAGE>.TBL`.  For example, the last 10 lines of `10000.TBL` are\n\n```>\n0xF6\t0x02C6\n0xF7\t0x02DC\n0xF8\t0x00AF\n0xF9\t0x02D8\n0xFA\t0x02D9\n0xFB\t0x02DA\n0xFC\t0x00B8\n0xFD\t0x02DD\n0xFE\t0x02DB\n0xFF\t0x02C7\n```\n\nwhich implies that code 0xF6 is `String.fromCharCode(0x02C6)` and vice versa.\n\n## Windows-dependent build step\n\nTo build the sources on windows, consult `dotnet/MakeEncoding.cs`.\n\nAfter saving the standard output to `out`, a simple awk script (`dotnet.sh`) takes care of the rest:\n\n```>dotnet.sh\n#!/bin/bash\nif [ ! -e dotnet/out ]; then exit; fi\n<dotnet/out tr -s ' ' '\\t' | awk 'NF>2 {if(outfile) close(outfile); outfile=\"codepages/\" $1 \".TBL\"} NF==2 {print > outfile}'\n```\n\n# Building the script\n\n`make.njs` takes a codepage argument, reads the corresponding table file and\ngenerates JS code for encoding and decoding:\n\n## Raw Codepages\n\n```>make.njs\n#!/usr/bin/env node\nvar argv = process.argv.slice(1), fs = require('fs');\nif(argv.length < 2) {\n    console.error(\"usage: make.njs <codepage_index> [variable]\");\n    process.exit(22); /* EINVAL */\n}\n\nvar cp = argv[1];\nvar jsvar = argv[2] || \"cptable\";\nvar x = fs.readFileSync(\"codepages/\" + cp + \".TBL\",\"utf8\");\nvar maxcp = 0;\n\nvar y = x.split(\"\\n\").map(function(z) {\n    var w = z.split(\"\\t\");\n    if(w.length < 2) return w;\n    return [Number(w[0]), Number(w[1])];\n}).filter(function(z) { return z.length > 1; });\n```\n\nThe DBCS and SBCS code generation strategies are different.  The maximum code is\nused to distinguish (max 0xFF for SBCS).\n\n```\nfor(var i = 0; i != y.length; ++i) if(y[i][0] > maxcp) maxcp = y[i][0];\n\nvar enc = {}, dec = (maxcp < 256 ? [] : {});\nfor(var i = 0; i != y.length; ++i) {\n    dec[y[i][0]] = String.fromCharCode(y[i][1]);\n    enc[String.fromCharCode(y[i][1])] = y[i][0];\n}\n\nvar odec, oenc, outstr;\nif(maxcp < 256) {\n```\n\nThe unicode character `0xFFFD` (REPLACEMENT CHARACTER) is used as a placeholder\nfor characters that are not specified in the map (for example, `0xF0` is not in\ncode page 10000).\n\nFor SBCS, the idea is to embed a raw string with the contents of the 256 codes.\nThe `dec` field is merely a split of the string, and `enc` is an eversion:\n\n```\n    for(var i = 0; i != 256; ++i) if(typeof dec[i] === \"undefined\") dec[i] = String.fromCharCode(0xFFFD);\n    odec = JSON.stringify(dec.join(\"\"));\n    outstr = '(function(){ var d = ' + odec + ', D = [], e = {}; for(var i=0;i!=d.length;++i) { if(d.charCodeAt(i) !== 0xFFFD) e[d[i]] = i; D[i] = d.charAt(i); } return {\"enc\": e, \"dec\": D }; })();';\n} else {\n```\n\nDBCS is similar, except that the space is sliced into 256-byte chunks (strings\nare only generated for those high-bytes represented in the codepage).\n\nThe strategy is to construct an array-of-arrays so that `dd[high][low]` is the\ncharacter associated with the code.  This array is combined at runtime to yield\nthe complete decoding object (and the encoding object is an eversion):\n\n```\n    var dd = [];\n    for(var i in dec) if(dec.hasOwnProperty(i)) {\n        if(typeof dd[i >> 8] === \"undefined\") dd[i >> 8] = [];\n        dd[i >> 8][i % 256] = dec[i];\n    }\n    outstr = '(function(){ var d = [], e = {}, D = [], j;\\n';\n    for(var i = 0; i != 256; ++i) if(dd[i]) {\n        for(var j = 0; j != 256; ++j) if(typeof dd[i][j] === \"undefined\") dd[i][j] = String.fromCharCode(0xFFFD);\n        outstr += 'D[' + i + '] = ' + JSON.stringify(dd[i].join(\"\")) + '.split(\"\");\\n';\n        outstr += 'for(j = 0; j != D[' + i + '].length; ++j) if(D[' + i + '][j].charCodeAt(0) !== 0xFFFD) { e[D[' + i + '][j]] = ' + (i*256) + ' + j; d[' + (i*256) + ' + j] = D[' + i + '][j];}\\n'\n    }\n    outstr += 'return {\"enc\": e, \"dec\": d }; })();';\n}\nconsole.log(jsvar + \"[\" + cp + \"] = \" + outstr);\n\n```\n\n`make.sh` generates the tables used by `make.njs`.  The raw unicode TXT files\nare columnar: `code unicode #comments`.  For example, the last 10 lines of the\ntext file ROMAN.TXT (for CP 10000) are:\n\n```>\n0xF6\t0x02C6\t#MODIFIER LETTER CIRCUMFLEX ACCENT\n0xF7\t0x02DC\t#SMALL TILDE\n0xF8\t0x00AF\t#MACRON\n0xF9\t0x02D8\t#BREVE\n0xFA\t0x02D9\t#DOT ABOVE\n0xFB\t0x02DA\t#RING ABOVE\n0xFC\t0x00B8\t#CEDILLA\n0xFD\t0x02DD\t#DOUBLE ACUTE ACCENT\n0xFE\t0x02DB\t#OGONEK\n0xFF\t0x02C7\t#CARON\n```\n\nIn processing the data, the comments (after the `#`) are stripped and undefined\nelements (like `0x7F` for CP 10000) are removed.\n\n```>make.sh\n#!/bin/bash\nINFILE=${1:-pages.csv}\nOUTFILE=${2:-cptable.js}\nJSVAR=${3:-cptable}\nVERSION=$(cat package.json | grep version | tr -dc [0-9.])\n\nmkdir -p codepages bits\nrm -f $OUTFILE $OUTFILE.tmp\necho \"/* $OUTFILE (C) 2013-2014 SheetJS -- http://sheetjs.com */\" > $OUTFILE.tmp\necho \"/*jshint -W100 */\" >> $OUTFILE.tmp\necho \"var $JSVAR = {version:\\\"$VERSION\\\"};\" >> $OUTFILE.tmp\nif [ -e dotnet.sh ]; then bash dotnet.sh; fi\nawk -F, '{print $1, $2, $3}' $INFILE | while read cp url cptype; do\n    echo $cp $url\n    if [ ! -e codepages/$cp.TBL ]; then\n        curl $url | sed 's/#.*//g' | awk 'NF==2' > codepages/$cp.TBL\n    fi\n    echo \"if(typeof $JSVAR === 'undefined') $JSVAR = {};\" > bits/$cp.js.tmp\n    node make.njs $cp $JSVAR | tee -a bits/$cp.js.tmp >> $OUTFILE.tmp\n    sed 's/\"\\([0-9]+\\)\":/\\1:/g' <bits/$cp.js.tmp >bits/$cp.js\n    rm -f bits/$cp.js.tmp\ndone\necho \"if (typeof module !== 'undefined' && module.exports) module.exports = $JSVAR;\" >> $OUTFILE.tmp\nsed 's/\"\\([0-9]+\\)\":/\\1:/g' <$OUTFILE.tmp >$OUTFILE\nrm -f $OUTFILE.tmp\n```\n\n## Utilities\n\nThe encode and decode functions are kept in a separate script (cputils.js).\n\nBoth encode and decode deal with data represented as:\n\n- String (encode expects JS string, decode interprets UCS2 chars as codes)\n- Array (encode expects array of JS String characters, decode expects numbers)\n- Buffer (encode expects UTF-8 string, decode expects codepoints/bytes).\n\nThe `ofmt` variable controls `encode` output (`str`, `arr` respectively)\nwhile the input format is automatically determined.\n\n# Tests\n\nThe tests include JS validity tests (requiring or eval'ing code):\n\n```>test.js\nvar fs = require('fs'), assert = require('assert'), vm = require('vm');\nvar cptable, sbcs;\ndescribe('source', function() {\n  it('should load node', function() { cptable = require('./'); });\n  it('should load sbcs', function() { sbcs = require('./sbcs'); });\n  it('should load excel', function() { excel = require('./cpexcel'); });\n  it('should process bits', function() {\n    var files = fs.readdirSync('bits').filter(function(x){return x.substr(-3)==\".js\";});\n    files.forEach(function(x) {\n      vm.runInThisContext(fs.readFileSync('./bits/' + x));\n    });\n  });\n});\n```\n\nThe README tests verify the snippets in the README:\n\n```>test.js\ndescribe('README', function() {\n  var readme = function() {\n    var unicode_cp10000_255 = cptable[10000].dec[255]; // ˇ\n    assert.equal(unicode_cp10000_255, \"ˇ\");\n\n    var cp10000_711 = cptable[10000].enc[String.fromCharCode(711)]; // 255\n    assert.equal(cp10000_711, 255);\n\n    var b1 = [0xbb,0xe3,0xd7,0xdc];\n    var 汇总 = cptable.utils.decode(936, b1);\n    var buf =  cptable.utils.encode(936,  汇总);\n    assert.equal(汇总,\"汇总\");\n    assert.equal(buf.length, 4);\n    for(var i = 0; i != 4; ++i) assert.equal(b1[i], buf[i]);\n\n    var b2 = [0xf0,0x9f,0x8d,0xa3];\n    var sushi= cptable.utils.decode(65001, b2);\n    var sbuf = cptable.utils.encode(65001, sushi);\n    assert.equal(sushi,\"🍣\");\n    assert.equal(sbuf.length, 4);\n    for(var i = 0; i != 4; ++i) assert.equal(b2[i], sbuf[i]);\n\n  };\n  it('should be correct', function() {\n    cptable.utils.cache.encache();\n    readme();\n    cptable.utils.cache.decache();\n    readme();\n  });\n});\n```\n\nThe consistency tests make sure that encoding and decoding are pseudo inverses:\n\n```>test.js\ndescribe('consistency', function() {\n  cptable = require('./');\n  U = cptable.utils;\n  var chk = function(cptable, cacheit) { return function(x) {\n    it('should consistently process CP ' + x, function() {\n      var cp = cptable[x], D = cp.dec, E = cp.enc;\n      if(cacheit) cptable.utils.cache.encache();\n      else cptable.utils.cache.decache();\n      Object.keys(D).forEach(function(d) {\n        if(E[D[d]] != d) {\n          if(typeof E[D[d]] !== \"undefined\") return;\n          if(D[d].charCodeAt(0) == 0xFFFD) return;\n          if(D[E[D[d]]] === D[d]) return;\n          throw new Error(x + \" e.d[\" + d + \"] = \" + E[D[d]] + \"; d[\" + d + \"]=\" + D[d] + \"; d.e.d[\" + d + \"] = \" + D[E[D[d]]]);\n        }\n      });\n      Object.keys(E).forEach(function(e) {\n        if(D[E[e]] != e) {\n          throw new Error(x + \" d.e[\" + e + \"] = \" + D[E[e]] + \"; e[\" + e + \"]=\" + E[e] + \"; e.d.e[\" + e + \"] = \" + E[D[E[e]]]);\n        }\n      });\n      var corpus = [\"foobar\"];\n      corpus.forEach(function(w){\n        assert.equal(U.decode(x,U.encode(x,w)),w);\n      });\n      cptable.utils.cache.encache();\n    });\n  }; };\n  describe('cached', function() {\n    Object.keys(cptable).filter(function(w) { return w == +w; }).forEach(chk(cptable, true));\n  });\n  describe('direct', function() {\n    Object.keys(cptable).filter(function(w) { return w == +w; }).forEach(chk(cptable, false));\n  });\n});\n```\n\nThe next tests look at possible entry conditions:\n\n```\ndescribe('entry conditions', function() {\n  it('should fail to load utils if cptable unavailable', function() {\n    var sandbox = {};\n    var ctx = vm.createContext(sandbox);\n    assert.throws(function() {\n      vm.runInContext(fs.readFileSync('cputils.js','utf8'),ctx);\n    });\n  });\n  it('should load utils if cptable is available', function() {\n    var sandbox = {};\n    var ctx = vm.createContext(sandbox);\n    vm.runInContext(fs.readFileSync('cpexcel.js','utf8'),ctx);\n    vm.runInContext(fs.readFileSync('cputils.js','utf8'),ctx);\n  });\n  var chken = function(cp, i) {\n    var c = function(cp, i, e) {\n      var str = cptable.utils.encode(cp,i,e);\n      var arr = cptable.utils.encode(cp,i.split(\"\"),e);\n      assert.deepEqual(str,arr);\n      if(typeof Buffer === 'undefined') return;\n      var buf = cptable.utils.encode(cp,new Buffer(i),e);\n      assert.deepEqual(str,buf);\n    };\n    cptable.utils.cache.encache();\n    c(cp,i);\n    c(cp,i,'buf');\n    c(cp,i,'arr');\n    c(cp,i,'str');\n    cptable.utils.cache.decache();\n    c(cp,i);\n    c(cp,i,'buf');\n    c(cp,i,'arr');\n    c(cp,i,'str');\n  };\n  describe('encode', function() {\n    it('CP  1252 : sbcs', function() { chken(1252,\"foobar\"); });\n    it('CP   708 : sbcs', function() { chken(708,\"ت and ث smiley faces\");});\n    it('CP   936 : dbcs', function() { chken(936, \"这是中文字符测试\");});\n  });\n  var chkde = function(cp, i) {\n    var c = function(cp, i) {\n      var s;\n      if(typeof Buffer !== 'undefined' && i instanceof Buffer) s = [].map.call(i, function(s){return String.fromCharCode(s); });\n      else s=(i.map) ? i.map(function(s){return String.fromCharCode(s); }) : i;\n      var str = cptable.utils.decode(cp,i);\n      var arr = cptable.utils.decode(cp,s.join?s.join(\"\"):s);\n      assert.deepEqual(str,arr);\n      if(typeof Buffer === 'undefined') return;\n      var buf = cptable.utils.decode(cp,new Buffer(i));\n      assert.deepEqual(str,buf);\n    };\n    cptable.utils.cache.encache();\n    c(cp,i);\n    cptable.utils.cache.decache();\n    c(cp,i);\n  };\n  describe('decode', function() {\n    it('CP  1252 : sbcs', function() { chkde(1252,[0x66, 0x6f, 0x6f, 0x62, 0x61, 0x72]); }); /* \"foobar\" */\n    if(typeof Buffer !== 'undefined') it('CP   708 : sbcs', function() { chkde(708, new Buffer([0xca, 0x20, 0x61, 0x6e, 0x64, 0x20, 0xcb, 0x20, 0x73, 0x6d, 0x69, 0x6c, 0x65, 0x79, 0x20, 0x66, 0x61, 0x63, 0x65, 0x73])); }); /* (\"ت and ث smiley faces\") */\n    it('CP   936 : dbcs', function() { chkde(936, [0xd5, 0xe2, 0xca, 0xc7, 0xd6, 0xd0, 0xce, 0xc4, 0xd7, 0xd6, 0xb7, 0xfb, 0xb2, 0xe2, 0xca, 0xd4]);}); /* \"这是中文字符测试\" */\n  });\n});\n```\n\nThe `testfile` helper function reads a file and compares to node's read facilities:\n\n```>test.js\nfunction testfile(f,cp,type,skip) {\n  var d = fs.readFileSync(f);\n  var x = fs.readFileSync(f, type);\n  var a = x.split(\"\");\n  var chk = function(cp) {\n    var y = cptable.utils.decode(cp, d);\n    assert.equal(x,y);\n    var z = cptable.utils.encode(cp, x);\n    if(z.length != d.length) throw new Error(f + \" \" + JSON.stringify(z) + \" != \" + JSON.stringify(d) + \" : \" + z.length + \" \" + d.length);\n    for(var i = 0; i != d.length; ++i) if(d[i] !== z[i]) throw new Error(\"\" + i + \" \" + d[i] + \"!=\" + z[i]);\n    if(skip) return;\n    z = cptable.utils.encode(cp, a);\n    if(z.length != d.length) throw new Error(f + \" \" + JSON.stringify(z) + \" != \" + JSON.stringify(d) + \" : \" + z.length + \" \" + d.length);\n    for(var i = 0; i != d.length; ++i) if(d[i] !== z[i]) throw new Error(\"\" + i + \" \" + d[i] + \"!=\" + z[i]);\n  }\n  cptable.utils.cache.encache();\n  chk(cp);\n  if(skip) return;\n  cptable.utils.cache.decache();\n  chk(cp);\n  cptable.utils.cache.encache();\n}\n```\n\nThe `utf8` tests verify utf8 encoding of the actual JS sources:\n\n```>test.js\ndescribe('node natives', function() {\n  var node = [[65001, 'utf8',1], [1200, 'utf16le',1], [20127, 'ascii',0]];\n  var unicodefiles = ['codepage.md','README.md','cptable.js'];\n  var asciifiles = ['cputils.js'];\n  node.forEach(function(w) {\n    describe(w[1], function() {\n      cptable = require('./');\n      asciifiles.forEach(function(f) {\n        it('should process ' + f, function() { testfile('./misc/'+f+'.'+w[1],w[0],w[1]); });\n      });\n      if(!w[2]) return;\n      unicodefiles.forEach(function(f) {\n        it('should process ' + f, function() { testfile('./misc/'+f+'.'+w[1],w[0],w[1]); });\n      });\n      if(w[1] === 'utf8') it('should process bits', function() {\n        var files = fs.readdirSync('bits').filter(function(x){return x.substr(-3)==\".js\";});\n        files.forEach(function(f) { testfile('./bits/' + f,w[0],w[1],true); });\n      });\n    });\n  });\n});\n```\n\nThe utf* and ascii tests attempt to test other magic formats:\n\n```>test.js\nvar m = cptable.utils.magic;\nfunction cmp(x,z) {\n  assert.equal(x.length, z.length);\n  for(var i = 0; i != z.length; ++i) assert.equal(i+\"/\"+x.length+\"\"+x[i], i+\"/\"+z.length+\"\"+z[i]);\n}\nObject.keys(m).forEach(function(t){if(t != 16969) describe(m[t], function() {\n  it(\"should process codepage.md.\" + m[t], fs.existsSync('./misc/codepage.md.' + m[t]) ?\n    function() {\n      var b = fs.readFileSync('./misc/codepage.md.utf8', \"utf8\");\n      if(m[t] === \"ascii\") b = b.replace(/[\\u0080-\\uffff]*/g,\"\");\n      var x = fs.readFileSync('./misc/codepage.md.' + m[t]);\n      var y, z;\n      cptable.utils.cache.encache();\n      y = cptable.utils.decode(t, x);\n      assert.equal(y,b);\n      z = cptable.utils.encode(t, y);\n      if(t != 65000) cmp(x,z);\n      else { assert.equal(y, cptable.utils.decode(t, z)); }\n      cptable.utils.cache.decache();\n      y = cptable.utils.decode(t, x);\n      assert.equal(y,b);\n      z = cptable.utils.encode(t, y);\n      if(t != 65000) cmp(x,z);\n      else { assert.equal(y, cptable.utils.decode(t, z)); }\n      cptable.utils.cache.encache();\n    }\n  : null);\n  it(\"should process README.md.\" + m[t], fs.existsSync('./misc/README.md.' + m[t]) ?\n    function() {\n      var b = fs.readFileSync('./misc/README.md.utf8', \"utf8\");\n      if(m[t] === \"ascii\") b = b.replace(/[\\u0080-\\uffff]*/g,\"\");\n      var x = fs.readFileSync('./misc/README.md.' + m[t]);\n      x = [].slice.call(x);\n      cptable.utils.cache.encache();\n      var y = cptable.utils.decode(t, x);\n      assert.equal(y,b);\n      cptable.utils.cache.decache();\n      var y = cptable.utils.decode(t, x);\n      assert.equal(y,b);\n      cptable.utils.cache.encache();\n    }\n  : null);\n});});\n```\n\nThe codepage `6969` is not defined, so operations should fail:\n\n```>test.js\ndescribe('failures', function() {\n  it('should fail to find CP 6969', function() {\n    assert.throws(function(){cptable[6969].dec});\n    assert.throws(function(){cptable[6969].enc});\n  });\n  it('should fail using utils', function() {\n    assert(!cptable.utils.hascp(6969));\n    assert.throws(function(){return cptable.utils.encode(6969, \"foobar\"); });\n    assert.throws(function(){return cptable.utils.decode(6969, [0x20]); });\n  });\n  it('should fail with black magic', function() {\n    assert(cptable.utils.hascp(16969));\n    assert.throws(function(){return cptable.utils.encode(16969, \"foobar\"); });\n    assert.throws(function(){return cptable.utils.decode(16969, [0x20]); });\n  });\n  it('should fail when presented with invalid char codes', function() {\n    assert.throws(function(){cptable.utils.cache.decache(); return cptable.utils.encode(20127, [String.fromCharCode(0xAA)]);});\n  });\n});\n```\n\n# Nitty Gritty\n\n```json>package.json\n{\n  \"name\": \"codepage\",\n  \"version\": \"1.3.8\",\n  \"author\": \"SheetJS\",\n  \"description\": \"pure-JS library to handle codepages\",\n  \"keywords\": [ \"codepage\", \"iconv\", \"convert\", \"strings\" ],\n  \"bin\": {\n    \"codepage\": \"./bin/codepage.njs\"\n  },\n  \"files\": [\n    \"LICENSE\",\n    \"README.md\",\n    \"bin\",\n    \"cptable.js\",\n    \"cputils.js\",\n    \"dist/cpexcel.full.js\"\n  ],\n  \"main\": \"cputils.js\",\n  \"dependencies\": {\n    \"voc\":\"\",\n    \"concat-stream\":\"\",\n    \"commander\":\"\"\n  },\n  \"devDependencies\": {\n    \"mocha\":\"\"\n  },\n  \"repository\": { \"type\":\"git\", \"url\":\"git://github.com/SheetJS/js-codepage.git\"},\n  \"scripts\": {\n    \"pretest\": \"git submodule init && git submodule update\",\n    \"test\": \"make test\",\n    \"build\": \"make js\"\n  },\n  \"config\": {\n    \"blanket\": {\n      \"pattern\": \"[cptable.js,cputils.js,cpexcel.js]\"\n    }\n  },\n  \"bugs\": { \"url\": \"https://github.com/SheetJS/js-codepage/issues\" },\n  \"license\": \"Apache-2.0\",\n  \"engines\": { \"node\": \">=0.8\" }\n}\n```\n\n```>.vocrc\n{ \"post\": \"make js\" }\n```\n\n```>.gitignore\n.gitignore\ncodepages/\n.vocrc\nnode_modules/\nmake.sh\nmake.njs\nmisc/coverage.html\n```\n"
;
README['ascii'] = 
[35,32,71,101,116,116,105,110,103,32,67,111,100,101,112,97,103,101,115,10,10,84,104,101,32,102,105,101,108,100,115,32,111,102,32,116,104,101,32,112,97,103,101,115,46,99,115,118,32,109,97,110,105,102,101,115,116,32,97,114,101,32,96,99,111,100,101,112,97,103,101,44,117,114,108,44,98,121,116,101,115,96,32,40,83,66,67,83,61,49,44,32,68,66,67,83,61,50,41,10,10,96,96,96,62,112,97,103,101,115,46,99,115,118,10,51,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,69,66,67,68,73,67,47,67,80,48,51,55,46,84,88,84,44,49,10,52,51,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,52,51,55,46,84,88,84,44,49,10,53,48,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,69,66,67,68,73,67,47,67,80,53,48,48,46,84,88,84,44,49,10,55,51,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,55,51,55,46,84,88,84,44,49,10,55,55,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,55,55,53,46,84,88,84,44,49,10,56,53,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,53,48,46,84,88,84,44,49,10,56,53,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,53,50,46,84,88,84,44,49,10,56,53,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,53,53,46,84,88,84,44,49,10,56,53,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,53,55,46,84,88,84,44,49,10,56,54,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,48,46,84,88,84,44,49,10,56,54,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,49,46,84,88,84,44,49,10,56,54,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,50,46,84,88,84,44,49,10,56,54,51,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,51,46,84,88,84,44,49,10,56,54,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,52,46,84,88,84,44,49,10,56,54,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,53,46,84,88,84,44,49,10,56,54,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,54,46,84,88,84,44,49,10,56,54,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,57,46,84,88,84,44,49,10,56,55,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,56,55,52,46,84,88,84,44,49,10,56,55,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,69,66,67,68,73,67,47,67,80,56,55,53,46,84,88,84,44,49,10,57,51,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,57,51,50,46,84,88,84,44,50,10,57,51,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,57,51,54,46,84,88,84,44,50,10,57,52,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,57,52,57,46,84,88,84,44,50,10,57,53,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,57,53,48,46,84,88,84,44,50,10,49,48,50,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,69,66,67,68,73,67,47,67,80,49,48,50,54,46,84,88,84,44,49,10,49,50,53,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,48,46,84,88,84,44,49,10,49,50,53,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,49,46,84,88,84,44,49,10,49,50,53,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,50,46,84,88,84,44,49,10,49,50,53,51,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,51,46,84,88,84,44,49,10,49,50,53,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,52,46,84,88,84,44,49,10,49,50,53,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,53,46,84,88,84,44,49,10,49,50,53,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,54,46,84,88,84,44,49,10,49,50,53,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,55,46,84,88,84,44,49,10,49,50,53,56,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,56,46,84,88,84,44,49,10,96,96,96,10,10,78,111,116,101,32,116,104,97,116,32,116,104,101,32,87,105,110,100,111,119,115,32,114,101,110,100,101,114,105,110,103,32,105,115,32,117,115,101,100,32,102,111,114,32,116,104,101,32,77,97,99,32,99,111,100,101,32,112,97,103,101,115,46,32,32,84,104,101,32,112,114,105,109,97,114,121,10,100,105,102,102,101,114,101,110,99,101,32,105,115,32,116,104,101,32,117,115,101,32,111,102,32,116,104,101,32,112,114,105,118,97,116,101,32,96,48,120,70,56,70,70,96,32,99,111,100,101,32,40,119,104,105,99,104,32,114,101,110,100,101,114,115,32,97,115,32,97,110,32,65,112,112,108,101,10,108,111,103,111,32,111,110,32,109,97,99,115,32,98,117,116,32,97,115,32,103,97,114,98,97,103,101,32,111,110,32,111,116,104,101,114,32,111,112,101,114,97,116,105,110,103,32,115,121,115,116,101,109,115,41,46,32,32,73,116,32,109,97,121,32,98,101,32,100,101,115,105,114,97,98,108,101,10,116,111,32,102,97,108,108,32,98,97,99,107,32,116,111,32,116,104,101,32,98,101,104,97,118,105,111,114,44,32,105,110,32,119,104,105,99,104,32,99,97,115,101,32,116,104,101,32,102,105,108,101,115,32,97,114,101,32,117,110,100,101,114,32,65,80,80,76,69,32,97,110,100,32,110,111,116,10,77,73,67,83,70,84,46,32,32,67,111,100,101,112,97,103,101,115,32,97,114,101,32,97,110,32,97,98,115,111,108,117,116,101,32,112,97,105,110,32,58,47,10,10,96,96,96,62,112,97,103,101,115,46,99,115,118,10,49,48,48,48,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,82,79,77,65,78,46,84,88,84,44,49,10,49,48,48,48,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,71,82,69,69,75,46,84,88,84,44,49,10,49,48,48,48,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,67,89,82,73,76,76,73,67,46,84,88,84,44,49,10,49,48,48,50,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,76,65,84,73,78,50,46,84,88,84,44,49,10,49,48,48,55,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,73,67,69,76,65,78,68,46,84,88,84,44,49,10,49,48,48,56,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,84,85,82,75,73,83,72,46,84,88,84,44,49,10,96,96,96,10,10,84,104,101,32,110,117,109,98,101,114,105,110,103,32,115,99,104,101,109,101,32,102,111,114,32,116,104,101,32,96,73,83,79,45,56,56,53,57,45,88,96,32,115,101,114,105,101,115,32,105,115,32,96,50,56,53,57,48,32,43,32,88,96,58,10,10,96,96,96,62,112,97,103,101,115,46,99,115,118,10,50,56,53,57,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,46,84,88,84,44,49,10,50,56,53,57,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,50,46,84,88,84,44,49,10,50,56,53,57,51,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,51,46,84,88,84,44,49,10,50,56,53,57,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,52,46,84,88,84,44,49,10,50,56,53,57,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,53,46,84,88,84,44,49,10,50,56,53,57,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,54,46,84,88,84,44,49,10,50,56,53,57,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,55,46,84,88,84,44,49,10,50,56,53,57,56,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,56,46,84,88,84,44,49,10,50,56,53,57,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,57,46,84,88,84,44,49,10,50,56,54,48,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,48,46,84,88,84,44,49,10,50,56,54,48,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,49,46,84,88,84,44,49,10,50,56,54,48,51,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,51,46,84,88,84,44,49,10,50,56,54,48,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,52,46,84,88,84,44,49,10,50,56,54,48,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,53,46,84,88,84,44,49,10,50,56,54,48,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,54,46,84,88,84,44,49,10,96,96,96,10,10,35,35,32,71,101,110,101,114,97,116,101,100,32,67,111,100,101,112,97,103,101,115,10,10,84,104,101,32,102,111,108,108,111,119,105,110,103,32,99,111,100,101,112,97,103,101,115,32,97,114,101,32,97,118,97,105,108,97,98,108,101,32,105,110,32,46,78,69,84,32,111,110,32,87,105,110,100,111,119,115,58,10,10,45,32,55,48,56,32,32,32,65,114,97,98,105,99,32,40,65,83,77,79,32,55,48,56,41,10,45,32,55,50,48,32,32,32,65,114,97,98,105,99,32,40,84,114,97,110,115,112,97,114,101,110,116,32,65,83,77,79,41,59,32,65,114,97,98,105,99,32,40,68,79,83,41,10,45,32,56,53,56,32,32,32,79,69,77,32,77,117,108,116,105,108,105,110,103,117,97,108,32,76,97,116,105,110,32,49,32,43,32,69,117,114,111,32,115,121,109,98,111,108,10,45,32,56,55,48,32,32,32,73,66,77,32,69,66,67,68,73,67,32,77,117,108,116,105,108,105,110,103,117,97,108,47,82,79,69,67,69,32,40,76,97,116,105,110,32,50,41,59,32,73,66,77,32,69,66,67,68,73,67,32,77,117,108,116,105,108,105,110,103,117,97,108,32,76,97,116,105,110,32,50,10,45,32,49,48,52,55,32,32,73,66,77,32,69,66,67,68,73,67,32,76,97,116,105,110,32,49,47,79,112,101,110,32,83,121,115,116,101,109,10,45,32,49,49,52,48,32,32,73,66,77,32,69,66,67,68,73,67,32,85,83,45,67,97,110,97,100,97,32,40,48,51,55,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,85,83,45,67,97,110,97,100,97,45,69,117,114,111,41,10,45,32,49,49,52,49,32,32,73,66,77,32,69,66,67,68,73,67,32,71,101,114,109,97,110,121,32,40,50,48,50,55,51,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,71,101,114,109,97,110,121,45,69,117,114,111,41,10,45,32,49,49,52,50,32,32,73,66,77,32,69,66,67,68,73,67,32,68,101,110,109,97,114,107,45,78,111,114,119,97,121,32,40,50,48,50,55,55,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,68,101,110,109,97,114,107,45,78,111,114,119,97,121,45,69,117,114,111,41,10,45,32,49,49,52,51,32,32,73,66,77,32,69,66,67,68,73,67,32,70,105,110,108,97,110,100,45,83,119,101,100,101,110,32,40,50,48,50,55,56,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,70,105,110,108,97,110,100,45,83,119,101,100,101,110,45,69,117,114,111,41,10,45,32,49,49,52,52,32,32,73,66,77,32,69,66,67,68,73,67,32,73,116,97,108,121,32,40,50,48,50,56,48,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,73,116,97,108,121,45,69,117,114,111,41,10,45,32,49,49,52,53,32,32,73,66,77,32,69,66,67,68,73,67,32,76,97,116,105,110,32,65,109,101,114,105,99,97,45,83,112,97,105,110,32,40,50,48,50,56,52,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,83,112,97,105,110,45,69,117,114,111,41,10,45,32,49,49,52,54,32,32,73,66,77,32,69,66,67,68,73,67,32,85,110,105,116,101,100,32,75,105,110,103,100,111,109,32,40,50,48,50,56,53,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,85,75,45,69,117,114,111,41,10,45,32,49,49,52,55,32,32,73,66,77,32,69,66,67,68,73,67,32,70,114,97,110,99,101,32,40,50,48,50,57,55,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,70,114,97,110,99,101,45,69,117,114,111,41,10,45,32,49,49,52,56,32,32,73,66,77,32,69,66,67,68,73,67,32,73,110,116,101,114,110,97,116,105,111,110,97,108,32,40,53,48,48,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,73,110,116,101,114,110,97,116,105,111,110,97,108,45,69,117,114,111,41,10,45,32,49,49,52,57,32,32,73,66,77,32,69,66,67,68,73,67,32,73,99,101,108,97,110,100,105,99,32,40,50,48,56,55,49,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,73,99,101,108,97,110,100,105,99,45,69,117,114,111,41,10,45,32,49,51,54,49,32,32,75,111,114,101,97,110,32,40,74,111,104,97,98,41,10,45,32,49,48,48,48,49,32,74,97,112,97,110,101,115,101,32,40,77,97,99,41,10,45,32,49,48,48,48,50,32,77,65,67,32,84,114,97,100,105,116,105,111,110,97,108,32,67,104,105,110,101,115,101,32,40,66,105,103,53,41,59,32,67,104,105,110,101,115,101,32,84,114,97,100,105,116,105,111,110,97,108,32,40,77,97,99,41,10,45,32,49,48,48,48,51,32,75,111,114,101,97,110,32,40,77,97,99,41,10,45,32,49,48,48,48,52,32,65,114,97,98,105,99,32,40,77,97,99,41,10,45,32,49,48,48,48,53,32,72,101,98,114,101,119,32,40,77,97,99,41,10,45,32,49,48,48,48,56,32,77,65,67,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,32,40,71,66,32,50,51,49,50,41,59,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,77,97,99,41,10,45,32,49,48,48,49,48,32,82,111,109,97,110,105,97,110,32,40,77,97,99,41,10,45,32,49,48,48,49,55,32,85,107,114,97,105,110,105,97,110,32,40,77,97,99,41,10,45,32,49,48,48,50,49,32,84,104,97,105,32,40,77,97,99,41,10,45,32,49,48,48,56,50,32,67,114,111,97,116,105,97,110,32,40,77,97,99,41,10,45,32,50,48,48,48,48,32,67,78,83,32,84,97,105,119,97,110,59,32,67,104,105,110,101,115,101,32,84,114,97,100,105,116,105,111,110,97,108,32,40,67,78,83,41,10,45,32,50,48,48,48,49,32,84,67,65,32,84,97,105,119,97,110,10,45,32,50,48,48,48,50,32,69,116,101,110,32,84,97,105,119,97,110,59,32,67,104,105,110,101,115,101,32,84,114,97,100,105,116,105,111,110,97,108,32,40,69,116,101,110,41,10,45,32,50,48,48,48,51,32,73,66,77,53,53,53,48,32,84,97,105,119,97,110,10,45,32,50,48,48,48,52,32,84,101,108,101,84,101,120,116,32,84,97,105,119,97,110,10,45,32,50,48,48,48,53,32,87,97,110,103,32,84,97,105,119,97,110,10,45,32,50,48,49,48,53,32,73,65,53,32,40,73,82,86,32,73,110,116,101,114,110,97,116,105,111,110,97,108,32,65,108,112,104,97,98,101,116,32,78,111,46,32,53,44,32,55,45,98,105,116,41,59,32,87,101,115,116,101,114,110,32,69,117,114,111,112,101,97,110,32,40,73,65,53,41,10,45,32,50,48,49,48,54,32,73,65,53,32,71,101,114,109,97,110,32,40,55,45,98,105,116,41,10,45,32,50,48,49,48,55,32,73,65,53,32,83,119,101,100,105,115,104,32,40,55,45,98,105,116,41,10,45,32,50,48,49,48,56,32,73,65,53,32,78,111,114,119,101,103,105,97,110,32,40,55,45,98,105,116,41,10,45,32,50,48,50,54,49,32,84,46,54,49,10,45,32,50,48,50,54,57,32,73,83,79,32,54,57,51,55,32,78,111,110,45,83,112,97,99,105,110,103,32,65,99,99,101,110,116,10,45,32,50,48,50,55,51,32,73,66,77,32,69,66,67,68,73,67,32,71,101,114,109,97,110,121,10,45,32,50,48,50,55,55,32,73,66,77,32,69,66,67,68,73,67,32,68,101,110,109,97,114,107,45,78,111,114,119,97,121,10,45,32,50,48,50,55,56,32,73,66,77,32,69,66,67,68,73,67,32,70,105,110,108,97,110,100,45,83,119,101,100,101,110,10,45,32,50,48,50,56,48,32,73,66,77,32,69,66,67,68,73,67,32,73,116,97,108,121,10,45,32,50,48,50,56,52,32,73,66,77,32,69,66,67,68,73,67,32,76,97,116,105,110,32,65,109,101,114,105,99,97,45,83,112,97,105,110,10,45,32,50,48,50,56,53,32,73,66,77,32,69,66,67,68,73,67,32,85,110,105,116,101,100,32,75,105,110,103,100,111,109,10,45,32,50,48,50,57,48,32,73,66,77,32,69,66,67,68,73,67,32,74,97,112,97,110,101,115,101,32,75,97,116,97,107,97,110,97,32,69,120,116,101,110,100,101,100,10,45,32,50,48,50,57,55,32,73,66,77,32,69,66,67,68,73,67,32,70,114,97,110,99,101,10,45,32,50,48,52,50,48,32,73,66,77,32,69,66,67,68,73,67,32,65,114,97,98,105,99,10,45,32,50,48,52,50,51,32,73,66,77,32,69,66,67,68,73,67,32,71,114,101,101,107,10,45,32,50,48,52,50,52,32,73,66,77,32,69,66,67,68,73,67,32,72,101,98,114,101,119,10,45,32,50,48,56,51,51,32,73,66,77,32,69,66,67,68,73,67,32,75,111,114,101,97,110,32,69,120,116,101,110,100,101,100,10,45,32,50,48,56,51,56,32,73,66,77,32,69,66,67,68,73,67,32,84,104,97,105,10,45,32,50,48,56,54,54,32,82,117,115,115,105,97,110,32,40,75,79,73,56,45,82,41,59,32,67,121,114,105,108,108,105,99,32,40,75,79,73,56,45,82,41,10,45,32,50,48,56,55,49,32,73,66,77,32,69,66,67,68,73,67,32,73,99,101,108,97,110,100,105,99,10,45,32,50,48,56,56,48,32,73,66,77,32,69,66,67,68,73,67,32,67,121,114,105,108,108,105,99,32,82,117,115,115,105,97,110,10,45,32,50,48,57,48,53,32,73,66,77,32,69,66,67,68,73,67,32,84,117,114,107,105,115,104,10,45,32,50,48,57,50,52,32,73,66,77,32,69,66,67,68,73,67,32,76,97,116,105,110,32,49,47,79,112,101,110,32,83,121,115,116,101,109,32,40,49,48,52,55,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,10,45,32,50,48,57,51,50,32,74,97,112,97,110,101,115,101,32,40,74,73,83,32,48,50,48,56,45,49,57,57,48,32,97,110,100,32,48,50,49,50,45,49,57,57,48,41,10,45,32,50,48,57,51,54,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,32,40,71,66,50,51,49,50,41,59,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,71,66,50,51,49,50,45,56,48,41,10,45,32,50,48,57,52,57,32,75,111,114,101,97,110,32,87,97,110,115,117,110,103,10,45,32,50,49,48,50,53,32,73,66,77,32,69,66,67,68,73,67,32,67,121,114,105,108,108,105,99,32,83,101,114,98,105,97,110,45,66,117,108,103,97,114,105,97,110,10,45,32,50,49,48,50,55,32,69,120,116,101,110,100,101,100,47,69,120,116,32,65,108,112,104,97,32,76,111,119,101,114,99,97,115,101,10,45,32,50,49,56,54,54,32,85,107,114,97,105,110,105,97,110,32,40,75,79,73,56,45,85,41,59,32,67,121,114,105,108,108,105,99,32,40,75,79,73,56,45,85,41,10,45,32,50,57,48,48,49,32,69,117,114,111,112,97,32,51,10,45,32,51,56,53,57,56,32,73,83,79,32,56,56,53,57,45,56,32,72,101,98,114,101,119,59,32,72,101,98,114,101,119,32,40,73,83,79,45,76,111,103,105,99,97,108,41,10,45,32,53,48,50,50,48,32,73,83,79,32,50,48,50,50,32,74,97,112,97,110,101,115,101,32,119,105,116,104,32,110,111,32,104,97,108,102,119,105,100,116,104,32,75,97,116,97,107,97,110,97,59,32,74,97,112,97,110,101,115,101,32,40,74,73,83,41,10,45,32,53,48,50,50,49,32,73,83,79,32,50,48,50,50,32,74,97,112,97,110,101,115,101,32,119,105,116,104,32,104,97,108,102,119,105,100,116,104,32,75,97,116,97,107,97,110,97,59,32,74,97,112,97,110,101,115,101,32,40,74,73,83,45,65,108,108,111,119,32,49,32,98,121,116,101,32,75,97,110,97,41,10,45,32,53,48,50,50,50,32,73,83,79,32,50,48,50,50,32,74,97,112,97,110,101,115,101,32,74,73,83,32,88,32,48,50,48,49,45,49,57,56,57,59,32,74,97,112,97,110,101,115,101,32,40,74,73,83,45,65,108,108,111,119,32,49,32,98,121,116,101,32,75,97,110,97,32,45,32,83,79,47,83,73,41,10,45,32,53,48,50,50,53,32,73,83,79,32,50,48,50,50,32,75,111,114,101,97,110,10,45,32,53,48,50,50,55,32,73,83,79,32,50,48,50,50,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,59,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,73,83,79,32,50,48,50,50,41,10,45,32,53,49,57,51,50,32,69,85,67,32,74,97,112,97,110,101,115,101,10,45,32,53,49,57,51,54,32,69,85,67,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,59,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,69,85,67,41,10,45,32,53,49,57,52,57,32,69,85,67,32,75,111,114,101,97,110,10,45,32,53,50,57,51,54,32,72,90,45,71,66,50,51,49,50,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,59,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,72,90,41,10,45,32,53,52,57,51,54,32,87,105,110,100,111,119,115,32,88,80,32,97,110,100,32,108,97,116,101,114,58,32,71,66,49,56,48,51,48,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,32,40,52,32,98,121,116,101,41,59,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,71,66,49,56,48,51,48,41,10,45,32,53,55,48,48,50,32,73,83,67,73,73,32,68,101,118,97,110,97,103,97,114,105,10,45,32,53,55,48,48,51,32,73,83,67,73,73,32,66,101,110,103,97,108,105,10,45,32,53,55,48,48,52,32,73,83,67,73,73,32,84,97,109,105,108,10,45,32,53,55,48,48,53,32,73,83,67,73,73,32,84,101,108,117,103,117,10,45,32,53,55,48,48,54,32,73,83,67,73,73,32,65,115,115,97,109,101,115,101,10,45,32,53,55,48,48,55,32,73,83,67,73,73,32,79,114,105,121,97,10,45,32,53,55,48,48,56,32,73,83,67,73,73,32,75,97,110,110,97,100,97,10,45,32,53,55,48,48,57,32,73,83,67,73,73,32,77,97,108,97,121,97,108,97,109,10,45,32,53,55,48,49,48,32,73,83,67,73,73,32,71,117,106,97,114,97,116,105,10,45,32,53,55,48,49,49,32,73,83,67,73,73,32,80,117,110,106,97,98,105,10,10,96,96,96,62,112,97,103,101,115,46,99,115,118,10,55,48,56,44,44,49,10,55,50,48,44,44,49,10,56,53,56,44,44,49,10,56,55,48,44,44,49,10,49,48,52,55,44,44,49,10,49,49,52,48,44,44,49,10,49,49,52,49,44,44,49,10,49,49,52,50,44,44,49,10,49,49,52,51,44,44,49,10,49,49,52,52,44,44,49,10,49,49,52,53,44,44,49,10,49,49,52,54,44,44,49,10,49,49,52,55,44,44,49,10,49,49,52,56,44,44,49,10,49,49,52,57,44,44,49,10,49,51,54,49,44,44,50,10,49,48,48,48,49,44,44,50,10,49,48,48,48,50,44,44,50,10,49,48,48,48,51,44,44,50,10,49,48,48,48,52,44,44,49,10,49,48,48,48,53,44,44,49,10,49,48,48,48,56,44,44,50,10,49,48,48,49,48,44,44,49,10,49,48,48,49,55,44,44,49,10,49,48,48,50,49,44,44,49,10,49,48,48,56,50,44,44,49,10,50,48,48,48,48,44,44,50,10,50,48,48,48,49,44,44,50,10,50,48,48,48,50,44,44,50,10,50,48,48,48,51,44,44,50,10,50,48,48,48,52,44,44,50,10,50,48,48,48,53,44,44,50,10,50,48,49,48,53,44,44,49,10,50,48,49,48,54,44,44,49,10,50,48,49,48,55,44,44,49,10,50,48,49,48,56,44,44,49,10,50,48,50,54,49,44,44,50,10,50,48,50,54,57,44,44,49,10,50,48,50,55,51,44,44,49,10,50,48,50,55,55,44,44,49,10,50,48,50,55,56,44,44,49,10,50,48,50,56,48,44,44,49,10,50,48,50,56,52,44,44,49,10,50,48,50,56,53,44,44,49,10,50,48,50,57,48,44,44,49,10,50,48,50,57,55,44,44,49,10,50,48,52,50,48,44,44,49,10,50,48,52,50,51,44,44,49,10,50,48,52,50,52,44,44,49,10,50,48,56,51,51,44,44,49,10,50,48,56,51,56,44,44,49,10,50,48,56,54,54,44,44,49,10,50,48,56,55,49,44,44,49,10,50,48,56,56,48,44,44,49,10,50,48,57,48,53,44,44,49,10,50,48,57,50,52,44,44,49,10,50,48,57,51,50,44,44,50,10,50,48,57,51,54,44,44,50,10,50,48,57,52,57,44,44,50,10,50,49,48,50,53,44,44,49,10,50,49,48,50,55,44,44,49,10,50,49,56,54,54,44,44,49,10,50,57,48,48,49,44,44,49,10,51,56,53,57,56,44,44,49,10,53,48,50,50,48,44,44,50,10,53,48,50,50,49,44,44,50,10,53,48,50,50,50,44,44,50,10,53,48,50,50,53,44,44,50,10,53,48,50,50,55,44,44,50,10,53,49,57,51,50,44,44,50,10,53,49,57,51,54,44,44,50,10,53,49,57,52,57,44,44,50,10,53,50,57,51,54,44,44,50,10,53,52,57,51,54,44,44,50,10,53,55,48,48,50,44,44,50,10,53,55,48,48,51,44,44,50,10,53,55,48,48,52,44,44,50,10,53,55,48,48,53,44,44,50,10,53,55,48,48,54,44,44,50,10,53,55,48,48,55,44,44,50,10,53,55,48,48,56,44,44,50,10,53,55,48,48,57,44,44,50,10,53,55,48,49,48,44,44,50,10,53,55,48,49,49,44,44,50,10,96,96,96,10,10,84,104,101,32,102,111,108,108,111,119,105,110,103,32,99,111,100,101,112,97,103,101,115,32,97,114,101,32,100,101,112,101,110,100,101,110,99,105,101,115,32,102,111,114,32,86,105,115,117,97,108,32,70,111,120,80,114,111,58,10,10,45,32,54,50,48,32,77,97,122,111,118,105,97,32,40,80,111,108,105,115,104,41,32,77,83,45,68,79,83,10,45,32,56,57,53,32,75,97,109,101,110,105,99,107,32,40,67,122,101,99,104,41,32,77,83,45,68,79,83,10,10,96,96,96,62,112,97,103,101,115,46,99,115,118,10,54,50,48,44,44,49,10,56,57,53,44,44,49,10,96,96,96,10,10,84,104,101,32,107,110,111,119,110,32,109,105,115,115,105,110,103,32,99,111,100,101,112,97,103,101,115,32,97,114,101,32,101,110,117,109,101,114,97,116,101,100,32,105,110,32,116,104,101,32,82,69,65,68,77,69,46,10,10,35,35,32,66,117,105,108,100,105,110,103,32,78,111,116,101,115,10,10,84,104,101,32,115,99,114,105,112,116,32,96,109,97,107,101,46,115,104,96,32,40,100,101,115,99,114,105,98,101,100,32,108,97,116,101,114,41,32,119,105,108,108,32,103,101,116,32,116,104,101,115,101,32,102,105,108,101,115,32,97,110,100,32,109,97,115,115,97,103,101,32,116,104,101,32,100,97,116,97,10,40,112,114,105,110,116,105,110,103,32,99,111,100,101,45,117,110,105,99,111,100,101,32,112,97,105,114,115,41,46,32,32,84,104,101,32,101,118,101,110,116,117,97,108,32,116,97,98,108,101,115,32,97,114,101,32,100,114,111,112,112,101,100,32,105,110,32,116,104,101,32,112,97,116,104,115,10,96,46,47,99,111,100,101,112,97,103,101,115,47,60,67,79,68,69,80,65,71,69,62,46,84,66,76,96,46,32,32,70,111,114,32,101,120,97,109,112,108,101,44,32,116,104,101,32,108,97,115,116,32,49,48,32,108,105,110,101,115,32,111,102,32,96,49,48,48,48,48,46,84,66,76,96,32,97,114,101,10,10,96,96,96,62,10,48,120,70,54,9,48,120,48,50,67,54,10,48,120,70,55,9,48,120,48,50,68,67,10,48,120,70,56,9,48,120,48,48,65,70,10,48,120,70,57,9,48,120,48,50,68,56,10,48,120,70,65,9,48,120,48,50,68,57,10,48,120,70,66,9,48,120,48,50,68,65,10,48,120,70,67,9,48,120,48,48,66,56,10,48,120,70,68,9,48,120,48,50,68,68,10,48,120,70,69,9,48,120,48,50,68,66,10,48,120,70,70,9,48,120,48,50,67,55,10,96,96,96,10,10,119,104,105,99,104,32,105,109,112,108,105,101,115,32,116,104,97,116,32,99,111,100,101,32,48,120,70,54,32,105,115,32,96,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,48,120,48,50,67,54,41,96,32,97,110,100,32,118,105,99,101,32,118,101,114,115,97,46,10,10,35,35,32,87,105,110,100,111,119,115,45,100,101,112,101,110,100,101,110,116,32,98,117,105,108,100,32,115,116,101,112,10,10,84,111,32,98,117,105,108,100,32,116,104,101,32,115,111,117,114,99,101,115,32,111,110,32,119,105,110,100,111,119,115,44,32,99,111,110,115,117,108,116,32,96,100,111,116,110,101,116,47,77,97,107,101,69,110,99,111,100,105,110,103,46,99,115,96,46,10,10,65,102,116,101,114,32,115,97,118,105,110,103,32,116,104,101,32,115,116,97,110,100,97,114,100,32,111,117,116,112,117,116,32,116,111,32,96,111,117,116,96,44,32,97,32,115,105,109,112,108,101,32,97,119,107,32,115,99,114,105,112,116,32,40,96,100,111,116,110,101,116,46,115,104,96,41,32,116,97,107,101,115,32,99,97,114,101,32,111,102,32,116,104,101,32,114,101,115,116,58,10,10,96,96,96,62,100,111,116,110,101,116,46,115,104,10,35,33,47,98,105,110,47,98,97,115,104,10,105,102,32,91,32,33,32,45,101,32,100,111,116,110,101,116,47,111,117,116,32,93,59,32,116,104,101,110,32,101,120,105,116,59,32,102,105,10,60,100,111,116,110,101,116,47,111,117,116,32,116,114,32,45,115,32,39,32,39,32,39,92,116,39,32,124,32,97,119,107,32,39,78,70,62,50,32,123,105,102,40,111,117,116,102,105,108,101,41,32,99,108,111,115,101,40,111,117,116,102,105,108,101,41,59,32,111,117,116,102,105,108,101,61,34,99,111,100,101,112,97,103,101,115,47,34,32,36,49,32,34,46,84,66,76,34,125,32,78,70,61,61,50,32,123,112,114,105,110,116,32,62,32,111,117,116,102,105,108,101,125,39,10,96,96,96,10,10,35,32,66,117,105,108,100,105,110,103,32,116,104,101,32,115,99,114,105,112,116,10,10,96,109,97,107,101,46,110,106,115,96,32,116,97,107,101,115,32,97,32,99,111,100,101,112,97,103,101,32,97,114,103,117,109,101,110,116,44,32,114,101,97,100,115,32,116,104,101,32,99,111,114,114,101,115,112,111,110,100,105,110,103,32,116,97,98,108,101,32,102,105,108,101,32,97,110,100,10,103,101,110,101,114,97,116,101,115,32,74,83,32,99,111,100,101,32,102,111,114,32,101,110,99,111,100,105,110,103,32,97,110,100,32,100,101,99,111,100,105,110,103,58,10,10,35,35,32,82,97,119,32,67,111,100,101,112,97,103,101,115,10,10,96,96,96,62,109,97,107,101,46,110,106,115,10,35,33,47,117,115,114,47,98,105,110,47,101,110,118,32,110,111,100,101,10,118,97,114,32,97,114,103,118,32,61,32,112,114,111,99,101,115,115,46,97,114,103,118,46,115,108,105,99,101,40,49,41,44,32,102,115,32,61,32,114,101,113,117,105,114,101,40,39,102,115,39,41,59,10,105,102,40,97,114,103,118,46,108,101,110,103,116,104,32,60,32,50,41,32,123,10,32,32,32,32,99,111,110,115,111,108,101,46,101,114,114,111,114,40,34,117,115,97,103,101,58,32,109,97,107,101,46,110,106,115,32,60,99,111,100,101,112,97,103,101,95,105,110,100,101,120,62,32,91,118,97,114,105,97,98,108,101,93,34,41,59,10,32,32,32,32,112,114,111,99,101,115,115,46,101,120,105,116,40,50,50,41,59,32,47,42,32,69,73,78,86,65,76,32,42,47,10,125,10,10,118,97,114,32,99,112,32,61,32,97,114,103,118,91,49,93,59,10,118,97,114,32,106,115,118,97,114,32,61,32,97,114,103,118,91,50,93,32,124,124,32,34,99,112,116,97,98,108,101,34,59,10,118,97,114,32,120,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,34,99,111,100,101,112,97,103,101,115,47,34,32,43,32,99,112,32,43,32,34,46,84,66,76,34,44,34,117,116,102,56,34,41,59,10,118,97,114,32,109,97,120,99,112,32,61,32,48,59,10,10,118,97,114,32,121,32,61,32,120,46,115,112,108,105,116,40,34,92,110,34,41,46,109,97,112,40,102,117,110,99,116,105,111,110,40,122,41,32,123,10,32,32,32,32,118,97,114,32,119,32,61,32,122,46,115,112,108,105,116,40,34,92,116,34,41,59,10,32,32,32,32,105,102,40,119,46,108,101,110,103,116,104,32,60,32,50,41,32,114,101,116,117,114,110,32,119,59,10,32,32,32,32,114,101,116,117,114,110,32,91,78,117,109,98,101,114,40,119,91,48,93,41,44,32,78,117,109,98,101,114,40,119,91,49,93,41,93,59,10,125,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,122,41,32,123,32,114,101,116,117,114,110,32,122,46,108,101,110,103,116,104,32,62,32,49,59,32,125,41,59,10,96,96,96,10,10,84,104,101,32,68,66,67,83,32,97,110,100,32,83,66,67,83,32,99,111,100,101,32,103,101,110,101,114,97,116,105,111,110,32,115,116,114,97,116,101,103,105,101,115,32,97,114,101,32,100,105,102,102,101,114,101,110,116,46,32,32,84,104,101,32,109,97,120,105,109,117,109,32,99,111,100,101,32,105,115,10,117,115,101,100,32,116,111,32,100,105,115,116,105,110,103,117,105,115,104,32,40,109,97,120,32,48,120,70,70,32,102,111,114,32,83,66,67,83,41,46,10,10,96,96,96,10,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,121,46,108,101,110,103,116,104,59,32,43,43,105,41,32,105,102,40,121,91,105,93,91,48,93,32,62,32,109,97,120,99,112,41,32,109,97,120,99,112,32,61,32,121,91,105,93,91,48,93,59,10,10,118,97,114,32,101,110,99,32,61,32,123,125,44,32,100,101,99,32,61,32,40,109,97,120,99,112,32,60,32,50,53,54,32,63,32,91,93,32,58,32,123,125,41,59,10,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,121,46,108,101,110,103,116,104,59,32,43,43,105,41,32,123,10,32,32,32,32,100,101,99,91,121,91,105,93,91,48,93,93,32,61,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,121,91,105,93,91,49,93,41,59,10,32,32,32,32,101,110,99,91,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,121,91,105,93,91,49,93,41,93,32,61,32,121,91,105,93,91,48,93,59,10,125,10,10,118,97,114,32,111,100,101,99,44,32,111,101,110,99,44,32,111,117,116,115,116,114,59,10,105,102,40,109,97,120,99,112,32,60,32,50,53,54,41,32,123,10,96,96,96,10,10,84,104,101,32,117,110,105,99,111,100,101,32,99,104,97,114,97,99,116,101,114,32,96,48,120,70,70,70,68,96,32,40,82,69,80,76,65,67,69,77,69,78,84,32,67,72,65,82,65,67,84,69,82,41,32,105,115,32,117,115,101,100,32,97,115,32,97,32,112,108,97,99,101,104,111,108,100,101,114,10,102,111,114,32,99,104,97,114,97,99,116,101,114,115,32,116,104,97,116,32,97,114,101,32,110,111,116,32,115,112,101,99,105,102,105,101,100,32,105,110,32,116,104,101,32,109,97,112,32,40,102,111,114,32,101,120,97,109,112,108,101,44,32,96,48,120,70,48,96,32,105,115,32,110,111,116,32,105,110,10,99,111,100,101,32,112,97,103,101,32,49,48,48,48,48,41,46,10,10,70,111,114,32,83,66,67,83,44,32,116,104,101,32,105,100,101,97,32,105,115,32,116,111,32,101,109,98,101,100,32,97,32,114,97,119,32,115,116,114,105,110,103,32,119,105,116,104,32,116,104,101,32,99,111,110,116,101,110,116,115,32,111,102,32,116,104,101,32,50,53,54,32,99,111,100,101,115,46,10,84,104,101,32,96,100,101,99,96,32,102,105,101,108,100,32,105,115,32,109,101,114,101,108,121,32,97,32,115,112,108,105,116,32,111,102,32,116,104,101,32,115,116,114,105,110,103,44,32,97,110,100,32,96,101,110,99,96,32,105,115,32,97,110,32,101,118,101,114,115,105,111,110,58,10,10,96,96,96,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,50,53,54,59,32,43,43,105,41,32,105,102,40,116,121,112,101,111,102,32,100,101,99,91,105,93,32,61,61,61,32,34,117,110,100,101,102,105,110,101,100,34,41,32,100,101,99,91,105,93,32,61,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,48,120,70,70,70,68,41,59,10,32,32,32,32,111,100,101,99,32,61,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,100,101,99,46,106,111,105,110,40,34,34,41,41,59,10,32,32,32,32,111,117,116,115,116,114,32,61,32,39,40,102,117,110,99,116,105,111,110,40,41,123,32,118,97,114,32,100,32,61,32,39,32,43,32,111,100,101,99,32,43,32,39,44,32,68,32,61,32,91,93,44,32,101,32,61,32,123,125,59,32,102,111,114,40,118,97,114,32,105,61,48,59,105,33,61,100,46,108,101,110,103,116,104,59,43,43,105,41,32,123,32,105,102,40,100,46,99,104,97,114,67,111,100,101,65,116,40,105,41,32,33,61,61,32,48,120,70,70,70,68,41,32,101,91,100,91,105,93,93,32,61,32,105,59,32,68,91,105,93,32,61,32,100,46,99,104,97,114,65,116,40,105,41,59,32,125,32,114,101,116,117,114,110,32,123,34,101,110,99,34,58,32,101,44,32,34,100,101,99,34,58,32,68,32,125,59,32,125,41,40,41,59,39,59,10,125,32,101,108,115,101,32,123,10,96,96,96,10,10,68,66,67,83,32,105,115,32,115,105,109,105,108,97,114,44,32,101,120,99,101,112,116,32,116,104,97,116,32,116,104,101,32,115,112,97,99,101,32,105,115,32,115,108,105,99,101,100,32,105,110,116,111,32,50,53,54,45,98,121,116,101,32,99,104,117,110,107,115,32,40,115,116,114,105,110,103,115,10,97,114,101,32,111,110,108,121,32,103,101,110,101,114,97,116,101,100,32,102,111,114,32,116,104,111,115,101,32,104,105,103,104,45,98,121,116,101,115,32,114,101,112,114,101,115,101,110,116,101,100,32,105,110,32,116,104,101,32,99,111,100,101,112,97,103,101,41,46,10,10,84,104,101,32,115,116,114,97,116,101,103,121,32,105,115,32,116,111,32,99,111,110,115,116,114,117,99,116,32,97,110,32,97,114,114,97,121,45,111,102,45,97,114,114,97,121,115,32,115,111,32,116,104,97,116,32,96,100,100,91,104,105,103,104,93,91,108,111,119,93,96,32,105,115,32,116,104,101,10,99,104,97,114,97,99,116,101,114,32,97,115,115,111,99,105,97,116,101,100,32,119,105,116,104,32,116,104,101,32,99,111,100,101,46,32,32,84,104,105,115,32,97,114,114,97,121,32,105,115,32,99,111,109,98,105,110,101,100,32,97,116,32,114,117,110,116,105,109,101,32,116,111,32,121,105,101,108,100,10,116,104,101,32,99,111,109,112,108,101,116,101,32,100,101,99,111,100,105,110,103,32,111,98,106,101,99,116,32,40,97,110,100,32,116,104,101,32,101,110,99,111,100,105,110,103,32,111,98,106,101,99,116,32,105,115,32,97,110,32,101,118,101,114,115,105,111,110,41,58,10,10,96,96,96,10,32,32,32,32,118,97,114,32,100,100,32,61,32,91,93,59,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,105,110,32,100,101,99,41,32,105,102,40,100,101,99,46,104,97,115,79,119,110,80,114,111,112,101,114,116,121,40,105,41,41,32,123,10,32,32,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,100,100,91,105,32,62,62,32,56,93,32,61,61,61,32,34,117,110,100,101,102,105,110,101,100,34,41,32,100,100,91,105,32,62,62,32,56,93,32,61,32,91,93,59,10,32,32,32,32,32,32,32,32,100,100,91,105,32,62,62,32,56,93,91,105,32,37,32,50,53,54,93,32,61,32,100,101,99,91,105,93,59,10,32,32,32,32,125,10,32,32,32,32,111,117,116,115,116,114,32,61,32,39,40,102,117,110,99,116,105,111,110,40,41,123,32,118,97,114,32,100,32,61,32,91,93,44,32,101,32,61,32,123,125,44,32,68,32,61,32,91,93,44,32,106,59,92,110,39,59,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,50,53,54,59,32,43,43,105,41,32,105,102,40,100,100,91,105,93,41,32,123,10,32,32,32,32,32,32,32,32,102,111,114,40,118,97,114,32,106,32,61,32,48,59,32,106,32,33,61,32,50,53,54,59,32,43,43,106,41,32,105,102,40,116,121,112,101,111,102,32,100,100,91,105,93,91,106,93,32,61,61,61,32,34,117,110,100,101,102,105,110,101,100,34,41,32,100,100,91,105,93,91,106,93,32,61,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,48,120,70,70,70,68,41,59,10,32,32,32,32,32,32,32,32,111,117,116,115,116,114,32,43,61,32,39,68,91,39,32,43,32,105,32,43,32,39,93,32,61,32,39,32,43,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,100,100,91,105,93,46,106,111,105,110,40,34,34,41,41,32,43,32,39,46,115,112,108,105,116,40,34,34,41,59,92,110,39,59,10,32,32,32,32,32,32,32,32,111,117,116,115,116,114,32,43,61,32,39,102,111,114,40,106,32,61,32,48,59,32,106,32,33,61,32,68,91,39,32,43,32,105,32,43,32,39,93,46,108,101,110,103,116,104,59,32,43,43,106,41,32,105,102,40,68,91,39,32,43,32,105,32,43,32,39,93,91,106,93,46,99,104,97,114,67,111,100,101,65,116,40,48,41,32,33,61,61,32,48,120,70,70,70,68,41,32,123,32,101,91,68,91,39,32,43,32,105,32,43,32,39,93,91,106,93,93,32,61,32,39,32,43,32,40,105,42,50,53,54,41,32,43,32,39,32,43,32,106,59,32,100,91,39,32,43,32,40,105,42,50,53,54,41,32,43,32,39,32,43,32,106,93,32,61,32,68,91,39,32,43,32,105,32,43,32,39,93,91,106,93,59,125,92,110,39,10,32,32,32,32,125,10,32,32,32,32,111,117,116,115,116,114,32,43,61,32,39,114,101,116,117,114,110,32,123,34,101,110,99,34,58,32,101,44,32,34,100,101,99,34,58,32,100,32,125,59,32,125,41,40,41,59,39,59,10,125,10,99,111,110,115,111,108,101,46,108,111,103,40,106,115,118,97,114,32,43,32,34,91,34,32,43,32,99,112,32,43,32,34,93,32,61,32,34,32,43,32,111,117,116,115,116,114,41,59,10,10,96,96,96,10,10,96,109,97,107,101,46,115,104,96,32,103,101,110,101,114,97,116,101,115,32,116,104,101,32,116,97,98,108,101,115,32,117,115,101,100,32,98,121,32,96,109,97,107,101,46,110,106,115,96,46,32,32,84,104,101,32,114,97,119,32,117,110,105,99,111,100,101,32,84,88,84,32,102,105,108,101,115,10,97,114,101,32,99,111,108,117,109,110,97,114,58,32,96,99,111,100,101,32,117,110,105,99,111,100,101,32,35,99,111,109,109,101,110,116,115,96,46,32,32,70,111,114,32,101,120,97,109,112,108,101,44,32,116,104,101,32,108,97,115,116,32,49,48,32,108,105,110,101,115,32,111,102,32,116,104,101,10,116,101,120,116,32,102,105,108,101,32,82,79,77,65,78,46,84,88,84,32,40,102,111,114,32,67,80,32,49,48,48,48,48,41,32,97,114,101,58,10,10,96,96,96,62,10,48,120,70,54,9,48,120,48,50,67,54,9,35,77,79,68,73,70,73,69,82,32,76,69,84,84,69,82,32,67,73,82,67,85,77,70,76,69,88,32,65,67,67,69,78,84,10,48,120,70,55,9,48,120,48,50,68,67,9,35,83,77,65,76,76,32,84,73,76,68,69,10,48,120,70,56,9,48,120,48,48,65,70,9,35,77,65,67,82,79,78,10,48,120,70,57,9,48,120,48,50,68,56,9,35,66,82,69,86,69,10,48,120,70,65,9,48,120,48,50,68,57,9,35,68,79,84,32,65,66,79,86,69,10,48,120,70,66,9,48,120,48,50,68,65,9,35,82,73,78,71,32,65,66,79,86,69,10,48,120,70,67,9,48,120,48,48,66,56,9,35,67,69,68,73,76,76,65,10,48,120,70,68,9,48,120,48,50,68,68,9,35,68,79,85,66,76,69,32,65,67,85,84,69,32,65,67,67,69,78,84,10,48,120,70,69,9,48,120,48,50,68,66,9,35,79,71,79,78,69,75,10,48,120,70,70,9,48,120,48,50,67,55,9,35,67,65,82,79,78,10,96,96,96,10,10,73,110,32,112,114,111,99,101,115,115,105,110,103,32,116,104,101,32,100,97,116,97,44,32,116,104,101,32,99,111,109,109,101,110,116,115,32,40,97,102,116,101,114,32,116,104,101,32,96,35,96,41,32,97,114,101,32,115,116,114,105,112,112,101,100,32,97,110,100,32,117,110,100,101,102,105,110,101,100,10,101,108,101,109,101,110,116,115,32,40,108,105,107,101,32,96,48,120,55,70,96,32,102,111,114,32,67,80,32,49,48,48,48,48,41,32,97,114,101,32,114,101,109,111,118,101,100,46,10,10,96,96,96,62,109,97,107,101,46,115,104,10,35,33,47,98,105,110,47,98,97,115,104,10,73,78,70,73,76,69,61,36,123,49,58,45,112,97,103,101,115,46,99,115,118,125,10,79,85,84,70,73,76,69,61,36,123,50,58,45,99,112,116,97,98,108,101,46,106,115,125,10,74,83,86,65,82,61,36,123,51,58,45,99,112,116,97,98,108,101,125,10,86,69,82,83,73,79,78,61,36,40,99,97,116,32,112,97,99,107,97,103,101,46,106,115,111,110,32,124,32,103,114,101,112,32,118,101,114,115,105,111,110,32,124,32,116,114,32,45,100,99,32,91,48,45,57,46,93,41,10,10,109,107,100,105,114,32,45,112,32,99,111,100,101,112,97,103,101,115,32,98,105,116,115,10,114,109,32,45,102,32,36,79,85,84,70,73,76,69,32,36,79,85,84,70,73,76,69,46,116,109,112,10,101,99,104,111,32,34,47,42,32,36,79,85,84,70,73,76,69,32,40,67,41,32,50,48,49,51,45,50,48,49,52,32,83,104,101,101,116,74,83,32,45,45,32,104,116,116,112,58,47,47,115,104,101,101,116,106,115,46,99,111,109,32,42,47,34,32,62,32,36,79,85,84,70,73,76,69,46,116,109,112,10,101,99,104,111,32,34,47,42,106,115,104,105,110,116,32,45,87,49,48,48,32,42,47,34,32,62,62,32,36,79,85,84,70,73,76,69,46,116,109,112,10,101,99,104,111,32,34,118,97,114,32,36,74,83,86,65,82,32,61,32,123,118,101,114,115,105,111,110,58,92,34,36,86,69,82,83,73,79,78,92,34,125,59,34,32,62,62,32,36,79,85,84,70,73,76,69,46,116,109,112,10,105,102,32,91,32,45,101,32,100,111,116,110,101,116,46,115,104,32,93,59,32,116,104,101,110,32,98,97,115,104,32,100,111,116,110,101,116,46,115,104,59,32,102,105,10,97,119,107,32,45,70,44,32,39,123,112,114,105,110,116,32,36,49,44,32,36,50,44,32,36,51,125,39,32,36,73,78,70,73,76,69,32,124,32,119,104,105,108,101,32,114,101,97,100,32,99,112,32,117,114,108,32,99,112,116,121,112,101,59,32,100,111,10,32,32,32,32,101,99,104,111,32,36,99,112,32,36,117,114,108,10,32,32,32,32,105,102,32,91,32,33,32,45,101,32,99,111,100,101,112,97,103,101,115,47,36,99,112,46,84,66,76,32,93,59,32,116,104,101,110,10,32,32,32,32,32,32,32,32,99,117,114,108,32,36,117,114,108,32,124,32,115,101,100,32,39,115,47,35,46,42,47,47,103,39,32,124,32,97,119,107,32,39,78,70,61,61,50,39,32,62,32,99,111,100,101,112,97,103,101,115,47,36,99,112,46,84,66,76,10,32,32,32,32,102,105,10,32,32,32,32,101,99,104,111,32,34,105,102,40,116,121,112,101,111,102,32,36,74,83,86,65,82,32,61,61,61,32,39,117,110,100,101,102,105,110,101,100,39,41,32,36,74,83,86,65,82,32,61,32,123,125,59,34,32,62,32,98,105,116,115,47,36,99,112,46,106,115,46,116,109,112,10,32,32,32,32,110,111,100,101,32,109,97,107,101,46,110,106,115,32,36,99,112,32,36,74,83,86,65,82,32,124,32,116,101,101,32,45,97,32,98,105,116,115,47,36,99,112,46,106,115,46,116,109,112,32,62,62,32,36,79,85,84,70,73,76,69,46,116,109,112,10,32,32,32,32,115,101,100,32,39,115,47,34,92,40,91,48,45,57,93,43,92,41,34,58,47,92,49,58,47,103,39,32,60,98,105,116,115,47,36,99,112,46,106,115,46,116,109,112,32,62,98,105,116,115,47,36,99,112,46,106,115,10,32,32,32,32,114,109,32,45,102,32,98,105,116,115,47,36,99,112,46,106,115,46,116,109,112,10,100,111,110,101,10,101,99,104,111,32,34,105,102,32,40,116,121,112,101,111,102,32,109,111,100,117,108,101,32,33,61,61,32,39,117,110,100,101,102,105,110,101,100,39,32,38,38,32,109,111,100,117,108,101,46,101,120,112,111,114,116,115,41,32,109,111,100,117,108,101,46,101,120,112,111,114,116,115,32,61,32,36,74,83,86,65,82,59,34,32,62,62,32,36,79,85,84,70,73,76,69,46,116,109,112,10,115,101,100,32,39,115,47,34,92,40,91,48,45,57,93,43,92,41,34,58,47,92,49,58,47,103,39,32,60,36,79,85,84,70,73,76,69,46,116,109,112,32,62,36,79,85,84,70,73,76,69,10,114,109,32,45,102,32,36,79,85,84,70,73,76,69,46,116,109,112,10,96,96,96,10,10,35,35,32,85,116,105,108,105,116,105,101,115,10,10,84,104,101,32,101,110,99,111,100,101,32,97,110,100,32,100,101,99,111,100,101,32,102,117,110,99,116,105,111,110,115,32,97,114,101,32,107,101,112,116,32,105,110,32,97,32,115,101,112,97,114,97,116,101,32,115,99,114,105,112,116,32,40,99,112,117,116,105,108,115,46,106,115,41,46,10,10,66,111,116,104,32,101,110,99,111,100,101,32,97,110,100,32,100,101,99,111,100,101,32,100,101,97,108,32,119,105,116,104,32,100,97,116,97,32,114,101,112,114,101,115,101,110,116,101,100,32,97,115,58,10,10,45,32,83,116,114,105,110,103,32,40,101,110,99,111,100,101,32,101,120,112,101,99,116,115,32,74,83,32,115,116,114,105,110,103,44,32,100,101,99,111,100,101,32,105,110,116,101,114,112,114,101,116,115,32,85,67,83,50,32,99,104,97,114,115,32,97,115,32,99,111,100,101,115,41,10,45,32,65,114,114,97,121,32,40,101,110,99,111,100,101,32,101,120,112,101,99,116,115,32,97,114,114,97,121,32,111,102,32,74,83,32,83,116,114,105,110,103,32,99,104,97,114,97,99,116,101,114,115,44,32,100,101,99,111,100,101,32,101,120,112,101,99,116,115,32,110,117,109,98,101,114,115,41,10,45,32,66,117,102,102,101,114,32,40,101,110,99,111,100,101,32,101,120,112,101,99,116,115,32,85,84,70,45,56,32,115,116,114,105,110,103,44,32,100,101,99,111,100,101,32,101,120,112,101,99,116,115,32,99,111,100,101,112,111,105,110,116,115,47,98,121,116,101,115,41,46,10,10,84,104,101,32,96,111,102,109,116,96,32,118,97,114,105,97,98,108,101,32,99,111,110,116,114,111,108,115,32,96,101,110,99,111,100,101,96,32,111,117,116,112,117,116,32,40,96,115,116,114,96,44,32,96,97,114,114,96,32,114,101,115,112,101,99,116,105,118,101,108,121,41,10,119,104,105,108,101,32,116,104,101,32,105,110,112,117,116,32,102,111,114,109,97,116,32,105,115,32,97,117,116,111,109,97,116,105,99,97,108,108,121,32,100,101,116,101,114,109,105,110,101,100,46,10,10,35,32,84,101,115,116,115,10,10,84,104,101,32,116,101,115,116,115,32,105,110,99,108,117,100,101,32,74,83,32,118,97,108,105,100,105,116,121,32,116,101,115,116,115,32,40,114,101,113,117,105,114,105,110,103,32,111,114,32,101,118,97,108,39,105,110,103,32,99,111,100,101,41,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,118,97,114,32,102,115,32,61,32,114,101,113,117,105,114,101,40,39,102,115,39,41,44,32,97,115,115,101,114,116,32,61,32,114,101,113,117,105,114,101,40,39,97,115,115,101,114,116,39,41,44,32,118,109,32,61,32,114,101,113,117,105,114,101,40,39,118,109,39,41,59,10,118,97,114,32,99,112,116,97,98,108,101,44,32,115,98,99,115,59,10,100,101,115,99,114,105,98,101,40,39,115,111,117,114,99,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,105,116,40,39,115,104,111,117,108,100,32,108,111,97,100,32,110,111,100,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,112,116,97,98,108,101,32,61,32,114,101,113,117,105,114,101,40,39,46,47,39,41,59,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,108,111,97,100,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,115,98,99,115,32,61,32,114,101,113,117,105,114,101,40,39,46,47,115,98,99,115,39,41,59,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,108,111,97,100,32,101,120,99,101,108,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,101,120,99,101,108,32,61,32,114,101,113,117,105,114,101,40,39,46,47,99,112,101,120,99,101,108,39,41,59,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,98,105,116,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,118,97,114,32,102,105,108,101,115,32,61,32,102,115,46,114,101,97,100,100,105,114,83,121,110,99,40,39,98,105,116,115,39,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,120,41,123,114,101,116,117,114,110,32,120,46,115,117,98,115,116,114,40,45,51,41,61,61,34,46,106,115,34,59,125,41,59,10,32,32,32,32,102,105,108,101,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,120,41,32,123,10,32,32,32,32,32,32,118,109,46,114,117,110,73,110,84,104,105,115,67,111,110,116,101,120,116,40,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,98,105,116,115,47,39,32,43,32,120,41,41,59,10,32,32,32,32,125,41,59,10,32,32,125,41,59,10,125,41,59,10,96,96,96,10,10,84,104,101,32,82,69,65,68,77,69,32,116,101,115,116,115,32,118,101,114,105,102,121,32,116,104,101,32,115,110,105,112,112,101,116,115,32,105,110,32,116,104,101,32,82,69,65,68,77,69,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,100,101,115,99,114,105,98,101,40,39,82,69,65,68,77,69,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,118,97,114,32,114,101,97,100,109,101,32,61,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,118,97,114,32,117,110,105,99,111,100,101,95,99,112,49,48,48,48,48,95,50,53,53,32,61,32,99,112,116,97,98,108,101,91,49,48,48,48,48,93,46,100,101,99,91,50,53,53,93,59,32,47,47,32,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,117,110,105,99,111,100,101,95,99,112,49,48,48,48,48,95,50,53,53,44,32,34,34,41,59,10,10,32,32,32,32,118,97,114,32,99,112,49,48,48,48,48,95,55,49,49,32,61,32,99,112,116,97,98,108,101,91,49,48,48,48,48,93,46,101,110,99,91,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,55,49,49,41,93,59,32,47,47,32,50,53,53,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,99,112,49,48,48,48,48,95,55,49,49,44,32,50,53,53,41,59,10,10,32,32,32,32,118,97,114,32,98,49,32,61,32,91,48,120,98,98,44,48,120,101,51,44,48,120,100,55,44,48,120,100,99,93,59,10,32,32,32,32,118,97,114,32,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,57,51,54,44,32,98,49,41,59,10,32,32,32,32,118,97,114,32,98,117,102,32,61,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,57,51,54,44,32,32,41,59,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,44,34,34,41,59,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,98,117,102,46,108,101,110,103,116,104,44,32,52,41,59,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,52,59,32,43,43,105,41,32,97,115,115,101,114,116,46,101,113,117,97,108,40,98,49,91,105,93,44,32,98,117,102,91,105,93,41,59,10,10,32,32,32,32,118,97,114,32,98,50,32,61,32,91,48,120,102,48,44,48,120,57,102,44,48,120,56,100,44,48,120,97,51,93,59,10,32,32,32,32,118,97,114,32,115,117,115,104,105,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,54,53,48,48,49,44,32,98,50,41,59,10,32,32,32,32,118,97,114,32,115,98,117,102,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,54,53,48,48,49,44,32,115,117,115,104,105,41,59,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,115,117,115,104,105,44,34,34,41,59,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,115,98,117,102,46,108,101,110,103,116,104,44,32,52,41,59,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,52,59,32,43,43,105,41,32,97,115,115,101,114,116,46,101,113,117,97,108,40,98,50,91,105,93,44,32,115,98,117,102,91,105,93,41,59,10,10,32,32,125,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,98,101,32,99,111,114,114,101,99,116,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,114,101,97,100,109,101,40,41,59,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,32,32,114,101,97,100,109,101,40,41,59,10,32,32,125,41,59,10,125,41,59,10,96,96,96,10,10,84,104,101,32,99,111,110,115,105,115,116,101,110,99,121,32,116,101,115,116,115,32,109,97,107,101,32,115,117,114,101,32,116,104,97,116,32,101,110,99,111,100,105,110,103,32,97,110,100,32,100,101,99,111,100,105,110,103,32,97,114,101,32,112,115,101,117,100,111,32,105,110,118,101,114,115,101,115,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,100,101,115,99,114,105,98,101,40,39,99,111,110,115,105,115,116,101,110,99,121,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,99,112,116,97,98,108,101,32,61,32,114,101,113,117,105,114,101,40,39,46,47,39,41,59,10,32,32,85,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,59,10,32,32,118,97,114,32,99,104,107,32,61,32,102,117,110,99,116,105,111,110,40,99,112,116,97,98,108,101,44,32,99,97,99,104,101,105,116,41,32,123,32,114,101,116,117,114,110,32,102,117,110,99,116,105,111,110,40,120,41,32,123,10,32,32,32,32,105,116,40,39,115,104,111,117,108,100,32,99,111,110,115,105,115,116,101,110,116,108,121,32,112,114,111,99,101,115,115,32,67,80,32,39,32,43,32,120,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,32,32,118,97,114,32,99,112,32,61,32,99,112,116,97,98,108,101,91,120,93,44,32,68,32,61,32,99,112,46,100,101,99,44,32,69,32,61,32,99,112,46,101,110,99,59,10,32,32,32,32,32,32,105,102,40,99,97,99,104,101,105,116,41,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,32,32,101,108,115,101,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,32,32,32,32,79,98,106,101,99,116,46,107,101,121,115,40,68,41,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,100,41,32,123,10,32,32,32,32,32,32,32,32,105,102,40,69,91,68,91,100,93,93,32,33,61,32,100,41,32,123,10,32,32,32,32,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,69,91,68,91,100,93,93,32,33,61,61,32,34,117,110,100,101,102,105,110,101,100,34,41,32,114,101,116,117,114,110,59,10,32,32,32,32,32,32,32,32,32,32,105,102,40,68,91,100,93,46,99,104,97,114,67,111,100,101,65,116,40,48,41,32,61,61,32,48,120,70,70,70,68,41,32,114,101,116,117,114,110,59,10,32,32,32,32,32,32,32,32,32,32,105,102,40,68,91,69,91,68,91,100,93,93,93,32,61,61,61,32,68,91,100,93,41,32,114,101,116,117,114,110,59,10,32,32,32,32,32,32,32,32,32,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,120,32,43,32,34,32,101,46,100,91,34,32,43,32,100,32,43,32,34,93,32,61,32,34,32,43,32,69,91,68,91,100,93,93,32,43,32,34,59,32,100,91,34,32,43,32,100,32,43,32,34,93,61,34,32,43,32,68,91,100,93,32,43,32,34,59,32,100,46,101,46,100,91,34,32,43,32,100,32,43,32,34,93,32,61,32,34,32,43,32,68,91,69,91,68,91,100,93,93,93,41,59,10,32,32,32,32,32,32,32,32,125,10,32,32,32,32,32,32,125,41,59,10,32,32,32,32,32,32,79,98,106,101,99,116,46,107,101,121,115,40,69,41,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,101,41,32,123,10,32,32,32,32,32,32,32,32,105,102,40,68,91,69,91,101,93,93,32,33,61,32,101,41,32,123,10,32,32,32,32,32,32,32,32,32,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,120,32,43,32,34,32,100,46,101,91,34,32,43,32,101,32,43,32,34,93,32,61,32,34,32,43,32,68,91,69,91,101,93,93,32,43,32,34,59,32,101,91,34,32,43,32,101,32,43,32,34,93,61,34,32,43,32,69,91,101,93,32,43,32,34,59,32,101,46,100,46,101,91,34,32,43,32,101,32,43,32,34,93,32,61,32,34,32,43,32,69,91,68,91,69,91,101,93,93,93,41,59,10,32,32,32,32,32,32,32,32,125,10,32,32,32,32,32,32,125,41,59,10,32,32,32,32,32,32,118,97,114,32,99,111,114,112,117,115,32,61,32,91,34,102,111,111,98,97,114,34,93,59,10,32,32,32,32,32,32,99,111,114,112,117,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,119,41,123,10,32,32,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,85,46,100,101,99,111,100,101,40,120,44,85,46,101,110,99,111,100,101,40,120,44,119,41,41,44,119,41,59,10,32,32,32,32,32,32,125,41,59,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,125,41,59,10,32,32,125,59,32,125,59,10,32,32,100,101,115,99,114,105,98,101,40,39,99,97,99,104,101,100,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,79,98,106,101,99,116,46,107,101,121,115,40,99,112,116,97,98,108,101,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,119,41,32,123,32,114,101,116,117,114,110,32,119,32,61,61,32,43,119,59,32,125,41,46,102,111,114,69,97,99,104,40,99,104,107,40,99,112,116,97,98,108,101,44,32,116,114,117,101,41,41,59,10,32,32,125,41,59,10,32,32,100,101,115,99,114,105,98,101,40,39,100,105,114,101,99,116,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,79,98,106,101,99,116,46,107,101,121,115,40,99,112,116,97,98,108,101,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,119,41,32,123,32,114,101,116,117,114,110,32,119,32,61,61,32,43,119,59,32,125,41,46,102,111,114,69,97,99,104,40,99,104,107,40,99,112,116,97,98,108,101,44,32,102,97,108,115,101,41,41,59,10,32,32,125,41,59,10,125,41,59,10,96,96,96,10,10,84,104,101,32,110,101,120,116,32,116,101,115,116,115,32,108,111,111,107,32,97,116,32,112,111,115,115,105,98,108,101,32,101,110,116,114,121,32,99,111,110,100,105,116,105,111,110,115,58,10,10,96,96,96,10,100,101,115,99,114,105,98,101,40,39,101,110,116,114,121,32,99,111,110,100,105,116,105,111,110,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,116,111,32,108,111,97,100,32,117,116,105,108,115,32,105,102,32,99,112,116,97,98,108,101,32,117,110,97,118,97,105,108,97,98,108,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,118,97,114,32,115,97,110,100,98,111,120,32,61,32,123,125,59,10,32,32,32,32,118,97,114,32,99,116,120,32,61,32,118,109,46,99,114,101,97,116,101,67,111,110,116,101,120,116,40,115,97,110,100,98,111,120,41,59,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,32,32,118,109,46,114,117,110,73,110,67,111,110,116,101,120,116,40,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,99,112,117,116,105,108,115,46,106,115,39,44,39,117,116,102,56,39,41,44,99,116,120,41,59,10,32,32,32,32,125,41,59,10,32,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,108,111,97,100,32,117,116,105,108,115,32,105,102,32,99,112,116,97,98,108,101,32,105,115,32,97,118,97,105,108,97,98,108,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,118,97,114,32,115,97,110,100,98,111,120,32,61,32,123,125,59,10,32,32,32,32,118,97,114,32,99,116,120,32,61,32,118,109,46,99,114,101,97,116,101,67,111,110,116,101,120,116,40,115,97,110,100,98,111,120,41,59,10,32,32,32,32,118,109,46,114,117,110,73,110,67,111,110,116,101,120,116,40,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,99,112,101,120,99,101,108,46,106,115,39,44,39,117,116,102,56,39,41,44,99,116,120,41,59,10,32,32,32,32,118,109,46,114,117,110,73,110,67,111,110,116,101,120,116,40,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,99,112,117,116,105,108,115,46,106,115,39,44,39,117,116,102,56,39,41,44,99,116,120,41,59,10,32,32,125,41,59,10,32,32,118,97,114,32,99,104,107,101,110,32,61,32,102,117,110,99,116,105,111,110,40,99,112,44,32,105,41,32,123,10,32,32,32,32,118,97,114,32,99,32,61,32,102,117,110,99,116,105,111,110,40,99,112,44,32,105,44,32,101,41,32,123,10,32,32,32,32,32,32,118,97,114,32,115,116,114,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,105,44,101,41,59,10,32,32,32,32,32,32,118,97,114,32,97,114,114,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,105,46,115,112,108,105,116,40,34,34,41,44,101,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,100,101,101,112,69,113,117,97,108,40,115,116,114,44,97,114,114,41,59,10,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,66,117,102,102,101,114,32,61,61,61,32,39,117,110,100,101,102,105,110,101,100,39,41,32,114,101,116,117,114,110,59,10,32,32,32,32,32,32,118,97,114,32,98,117,102,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,110,101,119,32,66,117,102,102,101,114,40,105,41,44,101,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,100,101,101,112,69,113,117,97,108,40,115,116,114,44,98,117,102,41,59,10,32,32,32,32,125,59,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,99,40,99,112,44,105,41,59,10,32,32,32,32,99,40,99,112,44,105,44,39,98,117,102,39,41,59,10,32,32,32,32,99,40,99,112,44,105,44,39,97,114,114,39,41,59,10,32,32,32,32,99,40,99,112,44,105,44,39,115,116,114,39,41,59,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,32,32,99,40,99,112,44,105,41,59,10,32,32,32,32,99,40,99,112,44,105,44,39,98,117,102,39,41,59,10,32,32,32,32,99,40,99,112,44,105,44,39,97,114,114,39,41,59,10,32,32,32,32,99,40,99,112,44,105,44,39,115,116,114,39,41,59,10,32,32,125,59,10,32,32,100,101,115,99,114,105,98,101,40,39,101,110,99,111,100,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,105,116,40,39,67,80,32,32,49,50,53,50,32,58,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,104,107,101,110,40,49,50,53,50,44,34,102,111,111,98,97,114,34,41,59,32,125,41,59,10,32,32,32,32,105,116,40,39,67,80,32,32,32,55,48,56,32,58,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,104,107,101,110,40,55,48,56,44,34,32,97,110,100,32,32,115,109,105,108,101,121,32,102,97,99,101,115,34,41,59,125,41,59,10,32,32,32,32,105,116,40,39,67,80,32,32,32,57,51,54,32,58,32,100,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,104,107,101,110,40,57,51,54,44,32,34,34,41,59,125,41,59,10,32,32,125,41,59,10,32,32,118,97,114,32,99,104,107,100,101,32,61,32,102,117,110,99,116,105,111,110,40,99,112,44,32,105,41,32,123,10,32,32,32,32,118,97,114,32,99,32,61,32,102,117,110,99,116,105,111,110,40,99,112,44,32,105,41,32,123,10,32,32,32,32,32,32,118,97,114,32,115,59,10,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,66,117,102,102,101,114,32,33,61,61,32,39,117,110,100,101,102,105,110,101,100,39,32,38,38,32,105,32,105,110,115,116,97,110,99,101,111,102,32,66,117,102,102,101,114,41,32,115,32,61,32,91,93,46,109,97,112,46,99,97,108,108,40,105,44,32,102,117,110,99,116,105,111,110,40,115,41,123,114,101,116,117,114,110,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,115,41,59,32,125,41,59,10,32,32,32,32,32,32,101,108,115,101,32,115,61,40,105,46,109,97,112,41,32,63,32,105,46,109,97,112,40,102,117,110,99,116,105,111,110,40,115,41,123,114,101,116,117,114,110,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,115,41,59,32,125,41,32,58,32,105,59,10,32,32,32,32,32,32,118,97,114,32,115,116,114,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,99,112,44,105,41,59,10,32,32,32,32,32,32,118,97,114,32,97,114,114,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,99,112,44,115,46,106,111,105,110,63,115,46,106,111,105,110,40,34,34,41,58,115,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,100,101,101,112,69,113,117,97,108,40,115,116,114,44,97,114,114,41,59,10,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,66,117,102,102,101,114,32,61,61,61,32,39,117,110,100,101,102,105,110,101,100,39,41,32,114,101,116,117,114,110,59,10,32,32,32,32,32,32,118,97,114,32,98,117,102,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,99,112,44,110,101,119,32,66,117,102,102,101,114,40,105,41,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,100,101,101,112,69,113,117,97,108,40,115,116,114,44,98,117,102,41,59,10,32,32,32,32,125,59,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,99,40,99,112,44,105,41,59,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,32,32,99,40,99,112,44,105,41,59,10,32,32,125,59,10,32,32,100,101,115,99,114,105,98,101,40,39,100,101,99,111,100,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,105,116,40,39,67,80,32,32,49,50,53,50,32,58,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,104,107,100,101,40,49,50,53,50,44,91,48,120,54,54,44,32,48,120,54,102,44,32,48,120,54,102,44,32,48,120,54,50,44,32,48,120,54,49,44,32,48,120,55,50,93,41,59,32,125,41,59,32,47,42,32,34,102,111,111,98,97,114,34,32,42,47,10,32,32,32,32,105,102,40,116,121,112,101,111,102,32,66,117,102,102,101,114,32,33,61,61,32,39,117,110,100,101,102,105,110,101,100,39,41,32,105,116,40,39,67,80,32,32,32,55,48,56,32,58,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,104,107,100,101,40,55,48,56,44,32,110,101,119,32,66,117,102,102,101,114,40,91,48,120,99,97,44,32,48,120,50,48,44,32,48,120,54,49,44,32,48,120,54,101,44,32,48,120,54,52,44,32,48,120,50,48,44,32,48,120,99,98,44,32,48,120,50,48,44,32,48,120,55,51,44,32,48,120,54,100,44,32,48,120,54,57,44,32,48,120,54,99,44,32,48,120,54,53,44,32,48,120,55,57,44,32,48,120,50,48,44,32,48,120,54,54,44,32,48,120,54,49,44,32,48,120,54,51,44,32,48,120,54,53,44,32,48,120,55,51,93,41,41,59,32,125,41,59,32,47,42,32,40,34,32,97,110,100,32,32,115,109,105,108,101,121,32,102,97,99,101,115,34,41,32,42,47,10,32,32,32,32,105,116,40,39,67,80,32,32,32,57,51,54,32,58,32,100,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,104,107,100,101,40,57,51,54,44,32,91,48,120,100,53,44,32,48,120,101,50,44,32,48,120,99,97,44,32,48,120,99,55,44,32,48,120,100,54,44,32,48,120,100,48,44,32,48,120,99,101,44,32,48,120,99,52,44,32,48,120,100,55,44,32,48,120,100,54,44,32,48,120,98,55,44,32,48,120,102,98,44,32,48,120,98,50,44,32,48,120,101,50,44,32,48,120,99,97,44,32,48,120,100,52,93,41,59,125,41,59,32,47,42,32,34,34,32,42,47,10,32,32,125,41,59,10,125,41,59,10,96,96,96,10,10,84,104,101,32,96,116,101,115,116,102,105,108,101,96,32,104,101,108,112,101,114,32,102,117,110,99,116,105,111,110,32,114,101,97,100,115,32,97,32,102,105,108,101,32,97,110,100,32,99,111,109,112,97,114,101,115,32,116,111,32,110,111,100,101,39,115,32,114,101,97,100,32,102,97,99,105,108,105,116,105,101,115,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,102,117,110,99,116,105,111,110,32,116,101,115,116,102,105,108,101,40,102,44,99,112,44,116,121,112,101,44,115,107,105,112,41,32,123,10,32,32,118,97,114,32,100,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,102,41,59,10,32,32,118,97,114,32,120,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,102,44,32,116,121,112,101,41,59,10,32,32,118,97,114,32,97,32,61,32,120,46,115,112,108,105,116,40,34,34,41,59,10,32,32,118,97,114,32,99,104,107,32,61,32,102,117,110,99,116,105,111,110,40,99,112,41,32,123,10,32,32,32,32,118,97,114,32,121,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,99,112,44,32,100,41,59,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,120,44,121,41,59,10,32,32,32,32,118,97,114,32,122,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,32,120,41,59,10,32,32,32,32,105,102,40,122,46,108,101,110,103,116,104,32,33,61,32,100,46,108,101,110,103,116,104,41,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,102,32,43,32,34,32,34,32,43,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,122,41,32,43,32,34,32,33,61,32,34,32,43,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,100,41,32,43,32,34,32,58,32,34,32,43,32,122,46,108,101,110,103,116,104,32,43,32,34,32,34,32,43,32,100,46,108,101,110,103,116,104,41,59,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,100,46,108,101,110,103,116,104,59,32,43,43,105,41,32,105,102,40,100,91,105,93,32,33,61,61,32,122,91,105,93,41,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,34,34,32,43,32,105,32,43,32,34,32,34,32,43,32,100,91,105,93,32,43,32,34,33,61,34,32,43,32,122,91,105,93,41,59,10,32,32,32,32,105,102,40,115,107,105,112,41,32,114,101,116,117,114,110,59,10,32,32,32,32,122,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,32,97,41,59,10,32,32,32,32,105,102,40,122,46,108,101,110,103,116,104,32,33,61,32,100,46,108,101,110,103,116,104,41,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,102,32,43,32,34,32,34,32,43,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,122,41,32,43,32,34,32,33,61,32,34,32,43,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,100,41,32,43,32,34,32,58,32,34,32,43,32,122,46,108,101,110,103,116,104,32,43,32,34,32,34,32,43,32,100,46,108,101,110,103,116,104,41,59,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,100,46,108,101,110,103,116,104,59,32,43,43,105,41,32,105,102,40,100,91,105,93,32,33,61,61,32,122,91,105,93,41,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,34,34,32,43,32,105,32,43,32,34,32,34,32,43,32,100,91,105,93,32,43,32,34,33,61,34,32,43,32,122,91,105,93,41,59,10,32,32,125,10,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,99,104,107,40,99,112,41,59,10,32,32,105,102,40,115,107,105,112,41,32,114,101,116,117,114,110,59,10,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,99,104,107,40,99,112,41,59,10,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,125,10,96,96,96,10,10,84,104,101,32,96,117,116,102,56,96,32,116,101,115,116,115,32,118,101,114,105,102,121,32,117,116,102,56,32,101,110,99,111,100,105,110,103,32,111,102,32,116,104,101,32,97,99,116,117,97,108,32,74,83,32,115,111,117,114,99,101,115,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,100,101,115,99,114,105,98,101,40,39,110,111,100,101,32,110,97,116,105,118,101,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,118,97,114,32,110,111,100,101,32,61,32,91,91,54,53,48,48,49,44,32,39,117,116,102,56,39,44,49,93,44,32,91,49,50,48,48,44,32,39,117,116,102,49,54,108,101,39,44,49,93,44,32,91,50,48,49,50,55,44,32,39,97,115,99,105,105,39,44,48,93,93,59,10,32,32,118,97,114,32,117,110,105,99,111,100,101,102,105,108,101,115,32,61,32,91,39,99,111,100,101,112,97,103,101,46,109,100,39,44,39,82,69,65,68,77,69,46,109,100,39,44,39,99,112,116,97,98,108,101,46,106,115,39,93,59,10,32,32,118,97,114,32,97,115,99,105,105,102,105,108,101,115,32,61,32,91,39,99,112,117,116,105,108,115,46,106,115,39,93,59,10,32,32,110,111,100,101,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,119,41,32,123,10,32,32,32,32,100,101,115,99,114,105,98,101,40,119,91,49,93,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,32,32,99,112,116,97,98,108,101,32,61,32,114,101,113,117,105,114,101,40,39,46,47,39,41,59,10,32,32,32,32,32,32,97,115,99,105,105,102,105,108,101,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,102,41,32,123,10,32,32,32,32,32,32,32,32,105,116,40,39,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,39,32,43,32,102,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,116,101,115,116,102,105,108,101,40,39,46,47,109,105,115,99,47,39,43,102,43,39,46,39,43,119,91,49,93,44,119,91,48,93,44,119,91,49,93,41,59,32,125,41,59,10,32,32,32,32,32,32,125,41,59,10,32,32,32,32,32,32,105,102,40,33,119,91,50,93,41,32,114,101,116,117,114,110,59,10,32,32,32,32,32,32,117,110,105,99,111,100,101,102,105,108,101,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,102,41,32,123,10,32,32,32,32,32,32,32,32,105,116,40,39,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,39,32,43,32,102,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,116,101,115,116,102,105,108,101,40,39,46,47,109,105,115,99,47,39,43,102,43,39,46,39,43,119,91,49,93,44,119,91,48,93,44,119,91,49,93,41,59,32,125,41,59,10,32,32,32,32,32,32,125,41,59,10,32,32,32,32,32,32,105,102,40,119,91,49,93,32,61,61,61,32,39,117,116,102,56,39,41,32,105,116,40,39,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,98,105,116,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,32,32,32,32,118,97,114,32,102,105,108,101,115,32,61,32,102,115,46,114,101,97,100,100,105,114,83,121,110,99,40,39,98,105,116,115,39,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,120,41,123,114,101,116,117,114,110,32,120,46,115,117,98,115,116,114,40,45,51,41,61,61,34,46,106,115,34,59,125,41,59,10,32,32,32,32,32,32,32,32,102,105,108,101,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,102,41,32,123,32,116,101,115,116,102,105,108,101,40,39,46,47,98,105,116,115,47,39,32,43,32,102,44,119,91,48,93,44,119,91,49,93,44,116,114,117,101,41,59,32,125,41,59,10,32,32,32,32,32,32,125,41,59,10,32,32,32,32,125,41,59,10,32,32,125,41,59,10,125,41,59,10,96,96,96,10,10,84,104,101,32,117,116,102,42,32,97,110,100,32,97,115,99,105,105,32,116,101,115,116,115,32,97,116,116,101,109,112,116,32,116,111,32,116,101,115,116,32,111,116,104,101,114,32,109,97,103,105,99,32,102,111,114,109,97,116,115,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,118,97,114,32,109,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,109,97,103,105,99,59,10,102,117,110,99,116,105,111,110,32,99,109,112,40,120,44,122,41,32,123,10,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,120,46,108,101,110,103,116,104,44,32,122,46,108,101,110,103,116,104,41,59,10,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,122,46,108,101,110,103,116,104,59,32,43,43,105,41,32,97,115,115,101,114,116,46,101,113,117,97,108,40,105,43,34,47,34,43,120,46,108,101,110,103,116,104,43,34,34,43,120,91,105,93,44,32,105,43,34,47,34,43,122,46,108,101,110,103,116,104,43,34,34,43,122,91,105,93,41,59,10,125,10,79,98,106,101,99,116,46,107,101,121,115,40,109,41,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,116,41,123,105,102,40,116,32,33,61,32,49,54,57,54,57,41,32,100,101,115,99,114,105,98,101,40,109,91,116,93,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,105,116,40,34,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,99,111,100,101,112,97,103,101,46,109,100,46,34,32,43,32,109,91,116,93,44,32,102,115,46,101,120,105,115,116,115,83,121,110,99,40,39,46,47,109,105,115,99,47,99,111,100,101,112,97,103,101,46,109,100,46,39,32,43,32,109,91,116,93,41,32,63,10,32,32,32,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,32,32,118,97,114,32,98,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,109,105,115,99,47,99,111,100,101,112,97,103,101,46,109,100,46,117,116,102,56,39,44,32,34,117,116,102,56,34,41,59,10,32,32,32,32,32,32,105,102,40,109,91,116,93,32,61,61,61,32,34,97,115,99,105,105,34,41,32,98,32,61,32,98,46,114,101,112,108,97,99,101,40,47,91,92,117,48,48,56,48,45,92,117,102,102,102,102,93,42,47,103,44,34,34,41,59,10,32,32,32,32,32,32,118,97,114,32,120,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,109,105,115,99,47,99,111,100,101,112,97,103,101,46,109,100,46,39,32,43,32,109,91,116,93,41,59,10,32,32,32,32,32,32,118,97,114,32,121,44,32,122,59,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,32,32,121,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,120,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,98,41,59,10,32,32,32,32,32,32,122,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,116,44,32,121,41,59,10,32,32,32,32,32,32,105,102,40,116,32,33,61,32,54,53,48,48,48,41,32,99,109,112,40,120,44,122,41,59,10,32,32,32,32,32,32,101,108,115,101,32,123,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,122,41,41,59,32,125,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,32,32,32,32,121,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,120,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,98,41,59,10,32,32,32,32,32,32,122,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,116,44,32,121,41,59,10,32,32,32,32,32,32,105,102,40,116,32,33,61,32,54,53,48,48,48,41,32,99,109,112,40,120,44,122,41,59,10,32,32,32,32,32,32,101,108,115,101,32,123,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,122,41,41,59,32,125,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,125,10,32,32,58,32,110,117,108,108,41,59,10,32,32,105,116,40,34,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,82,69,65,68,77,69,46,109,100,46,34,32,43,32,109,91,116,93,44,32,102,115,46,101,120,105,115,116,115,83,121,110,99,40,39,46,47,109,105,115,99,47,82,69,65,68,77,69,46,109,100,46,39,32,43,32,109,91,116,93,41,32,63,10,32,32,32,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,32,32,118,97,114,32,98,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,109,105,115,99,47,82,69,65,68,77,69,46,109,100,46,117,116,102,56,39,44,32,34,117,116,102,56,34,41,59,10,32,32,32,32,32,32,105,102,40,109,91,116,93,32,61,61,61,32,34,97,115,99,105,105,34,41,32,98,32,61,32,98,46,114,101,112,108,97,99,101,40,47,91,92,117,48,48,56,48,45,92,117,102,102,102,102,93,42,47,103,44,34,34,41,59,10,32,32,32,32,32,32,118,97,114,32,120,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,109,105,115,99,47,82,69,65,68,77,69,46,109,100,46,39,32,43,32,109,91,116,93,41,59,10,32,32,32,32,32,32,120,32,61,32,91,93,46,115,108,105,99,101,46,99,97,108,108,40,120,41,59,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,32,32,118,97,114,32,121,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,120,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,98,41,59,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,32,32,32,32,118,97,114,32,121,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,120,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,98,41,59,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,125,10,32,32,58,32,110,117,108,108,41,59,10,125,41,59,125,41,59,10,96,96,96,10,10,84,104,101,32,99,111,100,101,112,97,103,101,32,96,54,57,54,57,96,32,105,115,32,110,111,116,32,100,101,102,105,110,101,100,44,32,115,111,32,111,112,101,114,97,116,105,111,110,115,32,115,104,111,117,108,100,32,102,97,105,108,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,100,101,115,99,114,105,98,101,40,39,102,97,105,108,117,114,101,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,116,111,32,102,105,110,100,32,67,80,32,54,57,54,57,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,99,112,116,97,98,108,101,91,54,57,54,57,93,46,100,101,99,125,41,59,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,99,112,116,97,98,108,101,91,54,57,54,57,93,46,101,110,99,125,41,59,10,32,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,117,115,105,110,103,32,117,116,105,108,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,97,115,115,101,114,116,40,33,99,112,116,97,98,108,101,46,117,116,105,108,115,46,104,97,115,99,112,40,54,57,54,57,41,41,59,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,54,57,54,57,44,32,34,102,111,111,98,97,114,34,41,59,32,125,41,59,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,54,57,54,57,44,32,91,48,120,50,48,93,41,59,32,125,41,59,10,32,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,119,105,116,104,32,98,108,97,99,107,32,109,97,103,105,99,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,97,115,115,101,114,116,40,99,112,116,97,98,108,101,46,117,116,105,108,115,46,104,97,115,99,112,40,49,54,57,54,57,41,41,59,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,49,54,57,54,57,44,32,34,102,111,111,98,97,114,34,41,59,32,125,41,59,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,49,54,57,54,57,44,32,91,48,120,50,48,93,41,59,32,125,41,59,10,32,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,119,104,101,110,32,112,114,101,115,101,110,116,101,100,32,119,105,116,104,32,105,110,118,97,108,105,100,32,99,104,97,114,32,99,111,100,101,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,32,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,50,48,49,50,55,44,32,91,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,48,120,65,65,41,93,41,59,125,41,59,10,32,32,125,41,59,10,125,41,59,10,96,96,96,10,10,35,32,78,105,116,116,121,32,71,114,105,116,116,121,10,10,96,96,96,106,115,111,110,62,112,97,99,107,97,103,101,46,106,115,111,110,10,123,10,32,32,34,110,97,109,101,34,58,32,34,99,111,100,101,112,97,103,101,34,44,10,32,32,34,118,101,114,115,105,111,110,34,58,32,34,49,46,51,46,56,34,44,10,32,32,34,97,117,116,104,111,114,34,58,32,34,83,104,101,101,116,74,83,34,44,10,32,32,34,100,101,115,99,114,105,112,116,105,111,110,34,58,32,34,112,117,114,101,45,74,83,32,108,105,98,114,97,114,121,32,116,111,32,104,97,110,100,108,101,32,99,111,100,101,112,97,103,101,115,34,44,10,32,32,34,107,101,121,119,111,114,100,115,34,58,32,91,32,34,99,111,100,101,112,97,103,101,34,44,32,34,105,99,111,110,118,34,44,32,34,99,111,110,118,101,114,116,34,44,32,34,115,116,114,105,110,103,115,34,32,93,44,10,32,32,34,98,105,110,34,58,32,123,10,32,32,32,32,34,99,111,100,101,112,97,103,101,34,58,32,34,46,47,98,105,110,47,99,111,100,101,112,97,103,101,46,110,106,115,34,10,32,32,125,44,10,32,32,34,102,105,108,101,115,34,58,32,91,10,32,32,32,32,34,76,73,67,69,78,83,69,34,44,10,32,32,32,32,34,82,69,65,68,77,69,46,109,100,34,44,10,32,32,32,32,34,98,105,110,34,44,10,32,32,32,32,34,99,112,116,97,98,108,101,46,106,115,34,44,10,32,32,32,32,34,99,112,117,116,105,108,115,46,106,115,34,44,10,32,32,32,32,34,100,105,115,116,47,99,112,101,120,99,101,108,46,102,117,108,108,46,106,115,34,10,32,32,93,44,10,32,32,34,109,97,105,110,34,58,32,34,99,112,117,116,105,108,115,46,106,115,34,44,10,32,32,34,100,101,112,101,110,100,101,110,99,105,101,115,34,58,32,123,10,32,32,32,32,34,118,111,99,34,58,34,34,44,10,32,32,32,32,34,99,111,110,99,97,116,45,115,116,114,101,97,109,34,58,34,34,44,10,32,32,32,32,34,99,111,109,109,97,110,100,101,114,34,58,34,34,10,32,32,125,44,10,32,32,34,100,101,118,68,101,112,101,110,100,101,110,99,105,101,115,34,58,32,123,10,32,32,32,32,34,109,111,99,104,97,34,58,34,34,10,32,32,125,44,10,32,32,34,114,101,112,111,115,105,116,111,114,121,34,58,32,123,32,34,116,121,112,101,34,58,34,103,105,116,34,44,32,34,117,114,108,34,58,34,103,105,116,58,47,47,103,105,116,104,117,98,46,99,111,109,47,83,104,101,101,116,74,83,47,106,115,45,99,111,100,101,112,97,103,101,46,103,105,116,34,125,44,10,32,32,34,115,99,114,105,112,116,115,34,58,32,123,10,32,32,32,32,34,112,114,101,116,101,115,116,34,58,32,34,103,105,116,32,115,117,98,109,111,100,117,108,101,32,105,110,105,116,32,38,38,32,103,105,116,32,115,117,98,109,111,100,117,108,101,32,117,112,100,97,116,101,34,44,10,32,32,32,32,34,116,101,115,116,34,58,32,34,109,97,107,101,32,116,101,115,116,34,44,10,32,32,32,32,34,98,117,105,108,100,34,58,32,34,109,97,107,101,32,106,115,34,10,32,32,125,44,10,32,32,34,99,111,110,102,105,103,34,58,32,123,10,32,32,32,32,34,98,108,97,110,107,101,116,34,58,32,123,10,32,32,32,32,32,32,34,112,97,116,116,101,114,110,34,58,32,34,91,99,112,116,97,98,108,101,46,106,115,44,99,112,117,116,105,108,115,46,106,115,44,99,112,101,120,99,101,108,46,106,115,93,34,10,32,32,32,32,125,10,32,32,125,44,10,32,32,34,98,117,103,115,34,58,32,123,32,34,117,114,108,34,58,32,34,104,116,116,112,115,58,47,47,103,105,116,104,117,98,46,99,111,109,47,83,104,101,101,116,74,83,47,106,115,45,99,111,100,101,112,97,103,101,47,105,115,115,117,101,115,34,32,125,44,10,32,32,34,108,105,99,101,110,115,101,34,58,32,34,65,112,97,99,104,101,45,50,46,48,34,44,10,32,32,34,101,110,103,105,110,101,115,34,58,32,123,32,34,110,111,100,101,34,58,32,34,62,61,48,46,56,34,32,125,10,125,10,96,96,96,10,10,96,96,96,62,46,118,111,99,114,99,10,123,32,34,112,111,115,116,34,58,32,34,109,97,107,101,32,106,115,34,32,125,10,96,96,96,10,10,96,96,96,62,46,103,105,116,105,103,110,111,114,101,10,46,103,105,116,105,103,110,111,114,101,10,99,111,100,101,112,97,103,101,115,47,10,46,118,111,99,114,99,10,110,111,100,101,95,109,111,100,117,108,101,115,47,10,109,97,107,101,46,115,104,10,109,97,107,101,46,110,106,115,10,109,105,115,99,47,99,111,118,101,114,97,103,101,46,104,116,109,108,10,96,96,96,10]
;
README['utf7'] = 
[43,65,67,77,32,71,101,116,116,105,110,103,32,67,111,100,101,112,97,103,101,115,10,10,84,104,101,32,102,105,101,108,100,115,32,111,102,32,116,104,101,32,112,97,103,101,115,46,99,115,118,32,109,97,110,105,102,101,115,116,32,97,114,101,32,43,65,71,65,45,99,111,100,101,112,97,103,101,44,117,114,108,44,98,121,116,101,115,43,65,71,65,32,40,83,66,67,83,43,65,68,48,45,49,44,32,68,66,67,83,43,65,68,48,45,50,41,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,112,97,103,101,115,46,99,115,118,10,51,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,69,66,67,68,73,67,47,67,80,48,51,55,46,84,88,84,44,49,10,52,51,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,52,51,55,46,84,88,84,44,49,10,53,48,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,69,66,67,68,73,67,47,67,80,53,48,48,46,84,88,84,44,49,10,55,51,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,55,51,55,46,84,88,84,44,49,10,55,55,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,55,55,53,46,84,88,84,44,49,10,56,53,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,53,48,46,84,88,84,44,49,10,56,53,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,53,50,46,84,88,84,44,49,10,56,53,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,53,53,46,84,88,84,44,49,10,56,53,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,53,55,46,84,88,84,44,49,10,56,54,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,48,46,84,88,84,44,49,10,56,54,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,49,46,84,88,84,44,49,10,56,54,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,50,46,84,88,84,44,49,10,56,54,51,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,51,46,84,88,84,44,49,10,56,54,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,52,46,84,88,84,44,49,10,56,54,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,53,46,84,88,84,44,49,10,56,54,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,54,46,84,88,84,44,49,10,56,54,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,57,46,84,88,84,44,49,10,56,55,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,56,55,52,46,84,88,84,44,49,10,56,55,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,69,66,67,68,73,67,47,67,80,56,55,53,46,84,88,84,44,49,10,57,51,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,57,51,50,46,84,88,84,44,50,10,57,51,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,57,51,54,46,84,88,84,44,50,10,57,52,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,57,52,57,46,84,88,84,44,50,10,57,53,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,57,53,48,46,84,88,84,44,50,10,49,48,50,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,69,66,67,68,73,67,47,67,80,49,48,50,54,46,84,88,84,44,49,10,49,50,53,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,48,46,84,88,84,44,49,10,49,50,53,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,49,46,84,88,84,44,49,10,49,50,53,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,50,46,84,88,84,44,49,10,49,50,53,51,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,51,46,84,88,84,44,49,10,49,50,53,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,52,46,84,88,84,44,49,10,49,50,53,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,53,46,84,88,84,44,49,10,49,50,53,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,54,46,84,88,84,44,49,10,49,50,53,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,55,46,84,88,84,44,49,10,49,50,53,56,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,56,46,84,88,84,44,49,10,43,65,71,65,65,89,65,66,103,10,10,78,111,116,101,32,116,104,97,116,32,116,104,101,32,87,105,110,100,111,119,115,32,114,101,110,100,101,114,105,110,103,32,105,115,32,117,115,101,100,32,102,111,114,32,116,104,101,32,77,97,99,32,99,111,100,101,32,112,97,103,101,115,46,32,32,84,104,101,32,112,114,105,109,97,114,121,10,100,105,102,102,101,114,101,110,99,101,32,105,115,32,116,104,101,32,117,115,101,32,111,102,32,116,104,101,32,112,114,105,118,97,116,101,32,43,65,71,65,45,48,120,70,56,70,70,43,65,71,65,32,99,111,100,101,32,40,119,104,105,99,104,32,114,101,110,100,101,114,115,32,97,115,32,97,110,32,65,112,112,108,101,10,108,111,103,111,32,111,110,32,109,97,99,115,32,98,117,116,32,97,115,32,103,97,114,98,97,103,101,32,111,110,32,111,116,104,101,114,32,111,112,101,114,97,116,105,110,103,32,115,121,115,116,101,109,115,41,46,32,32,73,116,32,109,97,121,32,98,101,32,100,101,115,105,114,97,98,108,101,10,116,111,32,102,97,108,108,32,98,97,99,107,32,116,111,32,116,104,101,32,98,101,104,97,118,105,111,114,44,32,105,110,32,119,104,105,99,104,32,99,97,115,101,32,116,104,101,32,102,105,108,101,115,32,97,114,101,32,117,110,100,101,114,32,65,80,80,76,69,32,97,110,100,32,110,111,116,10,77,73,67,83,70,84,46,32,32,67,111,100,101,112,97,103,101,115,32,97,114,101,32,97,110,32,97,98,115,111,108,117,116,101,32,112,97,105,110,32,58,47,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,112,97,103,101,115,46,99,115,118,10,49,48,48,48,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,82,79,77,65,78,46,84,88,84,44,49,10,49,48,48,48,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,71,82,69,69,75,46,84,88,84,44,49,10,49,48,48,48,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,67,89,82,73,76,76,73,67,46,84,88,84,44,49,10,49,48,48,50,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,76,65,84,73,78,50,46,84,88,84,44,49,10,49,48,48,55,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,73,67,69,76,65,78,68,46,84,88,84,44,49,10,49,48,48,56,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,84,85,82,75,73,83,72,46,84,88,84,44,49,10,43,65,71,65,65,89,65,66,103,10,10,84,104,101,32,110,117,109,98,101,114,105,110,103,32,115,99,104,101,109,101,32,102,111,114,32,116,104,101,32,43,65,71,65,45,73,83,79,45,56,56,53,57,45,88,43,65,71,65,32,115,101,114,105,101,115,32,105,115,32,43,65,71,65,45,50,56,53,57,48,32,43,45,32,88,43,65,71,65,58,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,112,97,103,101,115,46,99,115,118,10,50,56,53,57,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,46,84,88,84,44,49,10,50,56,53,57,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,50,46,84,88,84,44,49,10,50,56,53,57,51,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,51,46,84,88,84,44,49,10,50,56,53,57,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,52,46,84,88,84,44,49,10,50,56,53,57,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,53,46,84,88,84,44,49,10,50,56,53,57,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,54,46,84,88,84,44,49,10,50,56,53,57,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,55,46,84,88,84,44,49,10,50,56,53,57,56,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,56,46,84,88,84,44,49,10,50,56,53,57,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,57,46,84,88,84,44,49,10,50,56,54,48,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,48,46,84,88,84,44,49,10,50,56,54,48,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,49,46,84,88,84,44,49,10,50,56,54,48,51,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,51,46,84,88,84,44,49,10,50,56,54,48,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,52,46,84,88,84,44,49,10,50,56,54,48,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,53,46,84,88,84,44,49,10,50,56,54,48,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,54,46,84,88,84,44,49,10,43,65,71,65,65,89,65,66,103,10,10,43,65,67,77,65,73,119,32,71,101,110,101,114,97,116,101,100,32,67,111,100,101,112,97,103,101,115,10,10,84,104,101,32,102,111,108,108,111,119,105,110,103,32,99,111,100,101,112,97,103,101,115,32,97,114,101,32,97,118,97,105,108,97,98,108,101,32,105,110,32,46,78,69,84,32,111,110,32,87,105,110,100,111,119,115,58,10,10,45,32,55,48,56,32,32,32,65,114,97,98,105,99,32,40,65,83,77,79,32,55,48,56,41,10,45,32,55,50,48,32,32,32,65,114,97,98,105,99,32,40,84,114,97,110,115,112,97,114,101,110,116,32,65,83,77,79,41,43,65,68,115,32,65,114,97,98,105,99,32,40,68,79,83,41,10,45,32,56,53,56,32,32,32,79,69,77,32,77,117,108,116,105,108,105,110,103,117,97,108,32,76,97,116,105,110,32,49,32,43,45,32,69,117,114,111,32,115,121,109,98,111,108,10,45,32,56,55,48,32,32,32,73,66,77,32,69,66,67,68,73,67,32,77,117,108,116,105,108,105,110,103,117,97,108,47,82,79,69,67,69,32,40,76,97,116,105,110,32,50,41,43,65,68,115,32,73,66,77,32,69,66,67,68,73,67,32,77,117,108,116,105,108,105,110,103,117,97,108,32,76,97,116,105,110,32,50,10,45,32,49,48,52,55,32,32,73,66,77,32,69,66,67,68,73,67,32,76,97,116,105,110,32,49,47,79,112,101,110,32,83,121,115,116,101,109,10,45,32,49,49,52,48,32,32,73,66,77,32,69,66,67,68,73,67,32,85,83,45,67,97,110,97,100,97,32,40,48,51,55,32,43,45,32,69,117,114,111,32,115,121,109,98,111,108,41,43,65,68,115,32,73,66,77,32,69,66,67,68,73,67,32,40,85,83,45,67,97,110,97,100,97,45,69,117,114,111,41,10,45,32,49,49,52,49,32,32,73,66,77,32,69,66,67,68,73,67,32,71,101,114,109,97,110,121,32,40,50,48,50,55,51,32,43,45,32,69,117,114,111,32,115,121,109,98,111,108,41,43,65,68,115,32,73,66,77,32,69,66,67,68,73,67,32,40,71,101,114,109,97,110,121,45,69,117,114,111,41,10,45,32,49,49,52,50,32,32,73,66,77,32,69,66,67,68,73,67,32,68,101,110,109,97,114,107,45,78,111,114,119,97,121,32,40,50,48,50,55,55,32,43,45,32,69,117,114,111,32,115,121,109,98,111,108,41,43,65,68,115,32,73,66,77,32,69,66,67,68,73,67,32,40,68,101,110,109,97,114,107,45,78,111,114,119,97,121,45,69,117,114,111,41,10,45,32,49,49,52,51,32,32,73,66,77,32,69,66,67,68,73,67,32,70,105,110,108,97,110,100,45,83,119,101,100,101,110,32,40,50,48,50,55,56,32,43,45,32,69,117,114,111,32,115,121,109,98,111,108,41,43,65,68,115,32,73,66,77,32,69,66,67,68,73,67,32,40,70,105,110,108,97,110,100,45,83,119,101,100,101,110,45,69,117,114,111,41,10,45,32,49,49,52,52,32,32,73,66,77,32,69,66,67,68,73,67,32,73,116,97,108,121,32,40,50,48,50,56,48,32,43,45,32,69,117,114,111,32,115,121,109,98,111,108,41,43,65,68,115,32,73,66,77,32,69,66,67,68,73,67,32,40,73,116,97,108,121,45,69,117,114,111,41,10,45,32,49,49,52,53,32,32,73,66,77,32,69,66,67,68,73,67,32,76,97,116,105,110,32,65,109,101,114,105,99,97,45,83,112,97,105,110,32,40,50,48,50,56,52,32,43,45,32,69,117,114,111,32,115,121,109,98,111,108,41,43,65,68,115,32,73,66,77,32,69,66,67,68,73,67,32,40,83,112,97,105,110,45,69,117,114,111,41,10,45,32,49,49,52,54,32,32,73,66,77,32,69,66,67,68,73,67,32,85,110,105,116,101,100,32,75,105,110,103,100,111,109,32,40,50,48,50,56,53,32,43,45,32,69,117,114,111,32,115,121,109,98,111,108,41,43,65,68,115,32,73,66,77,32,69,66,67,68,73,67,32,40,85,75,45,69,117,114,111,41,10,45,32,49,49,52,55,32,32,73,66,77,32,69,66,67,68,73,67,32,70,114,97,110,99,101,32,40,50,48,50,57,55,32,43,45,32,69,117,114,111,32,115,121,109,98,111,108,41,43,65,68,115,32,73,66,77,32,69,66,67,68,73,67,32,40,70,114,97,110,99,101,45,69,117,114,111,41,10,45,32,49,49,52,56,32,32,73,66,77,32,69,66,67,68,73,67,32,73,110,116,101,114,110,97,116,105,111,110,97,108,32,40,53,48,48,32,43,45,32,69,117,114,111,32,115,121,109,98,111,108,41,43,65,68,115,32,73,66,77,32,69,66,67,68,73,67,32,40,73,110,116,101,114,110,97,116,105,111,110,97,108,45,69,117,114,111,41,10,45,32,49,49,52,57,32,32,73,66,77,32,69,66,67,68,73,67,32,73,99,101,108,97,110,100,105,99,32,40,50,48,56,55,49,32,43,45,32,69,117,114,111,32,115,121,109,98,111,108,41,43,65,68,115,32,73,66,77,32,69,66,67,68,73,67,32,40,73,99,101,108,97,110,100,105,99,45,69,117,114,111,41,10,45,32,49,51,54,49,32,32,75,111,114,101,97,110,32,40,74,111,104,97,98,41,10,45,32,49,48,48,48,49,32,74,97,112,97,110,101,115,101,32,40,77,97,99,41,10,45,32,49,48,48,48,50,32,77,65,67,32,84,114,97,100,105,116,105,111,110,97,108,32,67,104,105,110,101,115,101,32,40,66,105,103,53,41,43,65,68,115,32,67,104,105,110,101,115,101,32,84,114,97,100,105,116,105,111,110,97,108,32,40,77,97,99,41,10,45,32,49,48,48,48,51,32,75,111,114,101,97,110,32,40,77,97,99,41,10,45,32,49,48,48,48,52,32,65,114,97,98,105,99,32,40,77,97,99,41,10,45,32,49,48,48,48,53,32,72,101,98,114,101,119,32,40,77,97,99,41,10,45,32,49,48,48,48,56,32,77,65,67,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,32,40,71,66,32,50,51,49,50,41,43,65,68,115,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,77,97,99,41,10,45,32,49,48,48,49,48,32,82,111,109,97,110,105,97,110,32,40,77,97,99,41,10,45,32,49,48,48,49,55,32,85,107,114,97,105,110,105,97,110,32,40,77,97,99,41,10,45,32,49,48,48,50,49,32,84,104,97,105,32,40,77,97,99,41,10,45,32,49,48,48,56,50,32,67,114,111,97,116,105,97,110,32,40,77,97,99,41,10,45,32,50,48,48,48,48,32,67,78,83,32,84,97,105,119,97,110,43,65,68,115,32,67,104,105,110,101,115,101,32,84,114,97,100,105,116,105,111,110,97,108,32,40,67,78,83,41,10,45,32,50,48,48,48,49,32,84,67,65,32,84,97,105,119,97,110,10,45,32,50,48,48,48,50,32,69,116,101,110,32,84,97,105,119,97,110,43,65,68,115,32,67,104,105,110,101,115,101,32,84,114,97,100,105,116,105,111,110,97,108,32,40,69,116,101,110,41,10,45,32,50,48,48,48,51,32,73,66,77,53,53,53,48,32,84,97,105,119,97,110,10,45,32,50,48,48,48,52,32,84,101,108,101,84,101,120,116,32,84,97,105,119,97,110,10,45,32,50,48,48,48,53,32,87,97,110,103,32,84,97,105,119,97,110,10,45,32,50,48,49,48,53,32,73,65,53,32,40,73,82,86,32,73,110,116,101,114,110,97,116,105,111,110,97,108,32,65,108,112,104,97,98,101,116,32,78,111,46,32,53,44,32,55,45,98,105,116,41,43,65,68,115,32,87,101,115,116,101,114,110,32,69,117,114,111,112,101,97,110,32,40,73,65,53,41,10,45,32,50,48,49,48,54,32,73,65,53,32,71,101,114,109,97,110,32,40,55,45,98,105,116,41,10,45,32,50,48,49,48,55,32,73,65,53,32,83,119,101,100,105,115,104,32,40,55,45,98,105,116,41,10,45,32,50,48,49,48,56,32,73,65,53,32,78,111,114,119,101,103,105,97,110,32,40,55,45,98,105,116,41,10,45,32,50,48,50,54,49,32,84,46,54,49,10,45,32,50,48,50,54,57,32,73,83,79,32,54,57,51,55,32,78,111,110,45,83,112,97,99,105,110,103,32,65,99,99,101,110,116,10,45,32,50,48,50,55,51,32,73,66,77,32,69,66,67,68,73,67,32,71,101,114,109,97,110,121,10,45,32,50,48,50,55,55,32,73,66,77,32,69,66,67,68,73,67,32,68,101,110,109,97,114,107,45,78,111,114,119,97,121,10,45,32,50,48,50,55,56,32,73,66,77,32,69,66,67,68,73,67,32,70,105,110,108,97,110,100,45,83,119,101,100,101,110,10,45,32,50,48,50,56,48,32,73,66,77,32,69,66,67,68,73,67,32,73,116,97,108,121,10,45,32,50,48,50,56,52,32,73,66,77,32,69,66,67,68,73,67,32,76,97,116,105,110,32,65,109,101,114,105,99,97,45,83,112,97,105,110,10,45,32,50,48,50,56,53,32,73,66,77,32,69,66,67,68,73,67,32,85,110,105,116,101,100,32,75,105,110,103,100,111,109,10,45,32,50,48,50,57,48,32,73,66,77,32,69,66,67,68,73,67,32,74,97,112,97,110,101,115,101,32,75,97,116,97,107,97,110,97,32,69,120,116,101,110,100,101,100,10,45,32,50,48,50,57,55,32,73,66,77,32,69,66,67,68,73,67,32,70,114,97,110,99,101,10,45,32,50,48,52,50,48,32,73,66,77,32,69,66,67,68,73,67,32,65,114,97,98,105,99,10,45,32,50,48,52,50,51,32,73,66,77,32,69,66,67,68,73,67,32,71,114,101,101,107,10,45,32,50,48,52,50,52,32,73,66,77,32,69,66,67,68,73,67,32,72,101,98,114,101,119,10,45,32,50,48,56,51,51,32,73,66,77,32,69,66,67,68,73,67,32,75,111,114,101,97,110,32,69,120,116,101,110,100,101,100,10,45,32,50,48,56,51,56,32,73,66,77,32,69,66,67,68,73,67,32,84,104,97,105,10,45,32,50,48,56,54,54,32,82,117,115,115,105,97,110,32,40,75,79,73,56,45,82,41,43,65,68,115,32,67,121,114,105,108,108,105,99,32,40,75,79,73,56,45,82,41,10,45,32,50,48,56,55,49,32,73,66,77,32,69,66,67,68,73,67,32,73,99,101,108,97,110,100,105,99,10,45,32,50,48,56,56,48,32,73,66,77,32,69,66,67,68,73,67,32,67,121,114,105,108,108,105,99,32,82,117,115,115,105,97,110,10,45,32,50,48,57,48,53,32,73,66,77,32,69,66,67,68,73,67,32,84,117,114,107,105,115,104,10,45,32,50,48,57,50,52,32,73,66,77,32,69,66,67,68,73,67,32,76,97,116,105,110,32,49,47,79,112,101,110,32,83,121,115,116,101,109,32,40,49,48,52,55,32,43,45,32,69,117,114,111,32,115,121,109,98,111,108,41,10,45,32,50,48,57,51,50,32,74,97,112,97,110,101,115,101,32,40,74,73,83,32,48,50,48,56,45,49,57,57,48,32,97,110,100,32,48,50,49,50,45,49,57,57,48,41,10,45,32,50,48,57,51,54,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,32,40,71,66,50,51,49,50,41,43,65,68,115,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,71,66,50,51,49,50,45,56,48,41,10,45,32,50,48,57,52,57,32,75,111,114,101,97,110,32,87,97,110,115,117,110,103,10,45,32,50,49,48,50,53,32,73,66,77,32,69,66,67,68,73,67,32,67,121,114,105,108,108,105,99,32,83,101,114,98,105,97,110,45,66,117,108,103,97,114,105,97,110,10,45,32,50,49,48,50,55,32,69,120,116,101,110,100,101,100,47,69,120,116,32,65,108,112,104,97,32,76,111,119,101,114,99,97,115,101,10,45,32,50,49,56,54,54,32,85,107,114,97,105,110,105,97,110,32,40,75,79,73,56,45,85,41,43,65,68,115,32,67,121,114,105,108,108,105,99,32,40,75,79,73,56,45,85,41,10,45,32,50,57,48,48,49,32,69,117,114,111,112,97,32,51,10,45,32,51,56,53,57,56,32,73,83,79,32,56,56,53,57,45,56,32,72,101,98,114,101,119,43,65,68,115,32,72,101,98,114,101,119,32,40,73,83,79,45,76,111,103,105,99,97,108,41,10,45,32,53,48,50,50,48,32,73,83,79,32,50,48,50,50,32,74,97,112,97,110,101,115,101,32,119,105,116,104,32,110,111,32,104,97,108,102,119,105,100,116,104,32,75,97,116,97,107,97,110,97,43,65,68,115,32,74,97,112,97,110,101,115,101,32,40,74,73,83,41,10,45,32,53,48,50,50,49,32,73,83,79,32,50,48,50,50,32,74,97,112,97,110,101,115,101,32,119,105,116,104,32,104,97,108,102,119,105,100,116,104,32,75,97,116,97,107,97,110,97,43,65,68,115,32,74,97,112,97,110,101,115,101,32,40,74,73,83,45,65,108,108,111,119,32,49,32,98,121,116,101,32,75,97,110,97,41,10,45,32,53,48,50,50,50,32,73,83,79,32,50,48,50,50,32,74,97,112,97,110,101,115,101,32,74,73,83,32,88,32,48,50,48,49,45,49,57,56,57,43,65,68,115,32,74,97,112,97,110,101,115,101,32,40,74,73,83,45,65,108,108,111,119,32,49,32,98,121,116,101,32,75,97,110,97,32,45,32,83,79,47,83,73,41,10,45,32,53,48,50,50,53,32,73,83,79,32,50,48,50,50,32,75,111,114,101,97,110,10,45,32,53,48,50,50,55,32,73,83,79,32,50,48,50,50,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,43,65,68,115,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,73,83,79,32,50,48,50,50,41,10,45,32,53,49,57,51,50,32,69,85,67,32,74,97,112,97,110,101,115,101,10,45,32,53,49,57,51,54,32,69,85,67,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,43,65,68,115,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,69,85,67,41,10,45,32,53,49,57,52,57,32,69,85,67,32,75,111,114,101,97,110,10,45,32,53,50,57,51,54,32,72,90,45,71,66,50,51,49,50,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,43,65,68,115,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,72,90,41,10,45,32,53,52,57,51,54,32,87,105,110,100,111,119,115,32,88,80,32,97,110,100,32,108,97,116,101,114,58,32,71,66,49,56,48,51,48,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,32,40,52,32,98,121,116,101,41,43,65,68,115,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,71,66,49,56,48,51,48,41,10,45,32,53,55,48,48,50,32,73,83,67,73,73,32,68,101,118,97,110,97,103,97,114,105,10,45,32,53,55,48,48,51,32,73,83,67,73,73,32,66,101,110,103,97,108,105,10,45,32,53,55,48,48,52,32,73,83,67,73,73,32,84,97,109,105,108,10,45,32,53,55,48,48,53,32,73,83,67,73,73,32,84,101,108,117,103,117,10,45,32,53,55,48,48,54,32,73,83,67,73,73,32,65,115,115,97,109,101,115,101,10,45,32,53,55,48,48,55,32,73,83,67,73,73,32,79,114,105,121,97,10,45,32,53,55,48,48,56,32,73,83,67,73,73,32,75,97,110,110,97,100,97,10,45,32,53,55,48,48,57,32,73,83,67,73,73,32,77,97,108,97,121,97,108,97,109,10,45,32,53,55,48,49,48,32,73,83,67,73,73,32,71,117,106,97,114,97,116,105,10,45,32,53,55,48,49,49,32,73,83,67,73,73,32,80,117,110,106,97,98,105,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,112,97,103,101,115,46,99,115,118,10,55,48,56,44,44,49,10,55,50,48,44,44,49,10,56,53,56,44,44,49,10,56,55,48,44,44,49,10,49,48,52,55,44,44,49,10,49,49,52,48,44,44,49,10,49,49,52,49,44,44,49,10,49,49,52,50,44,44,49,10,49,49,52,51,44,44,49,10,49,49,52,52,44,44,49,10,49,49,52,53,44,44,49,10,49,49,52,54,44,44,49,10,49,49,52,55,44,44,49,10,49,49,52,56,44,44,49,10,49,49,52,57,44,44,49,10,49,51,54,49,44,44,50,10,49,48,48,48,49,44,44,50,10,49,48,48,48,50,44,44,50,10,49,48,48,48,51,44,44,50,10,49,48,48,48,52,44,44,49,10,49,48,48,48,53,44,44,49,10,49,48,48,48,56,44,44,50,10,49,48,48,49,48,44,44,49,10,49,48,48,49,55,44,44,49,10,49,48,48,50,49,44,44,49,10,49,48,48,56,50,44,44,49,10,50,48,48,48,48,44,44,50,10,50,48,48,48,49,44,44,50,10,50,48,48,48,50,44,44,50,10,50,48,48,48,51,44,44,50,10,50,48,48,48,52,44,44,50,10,50,48,48,48,53,44,44,50,10,50,48,49,48,53,44,44,49,10,50,48,49,48,54,44,44,49,10,50,48,49,48,55,44,44,49,10,50,48,49,48,56,44,44,49,10,50,48,50,54,49,44,44,50,10,50,48,50,54,57,44,44,49,10,50,48,50,55,51,44,44,49,10,50,48,50,55,55,44,44,49,10,50,48,50,55,56,44,44,49,10,50,48,50,56,48,44,44,49,10,50,48,50,56,52,44,44,49,10,50,48,50,56,53,44,44,49,10,50,48,50,57,48,44,44,49,10,50,48,50,57,55,44,44,49,10,50,48,52,50,48,44,44,49,10,50,48,52,50,51,44,44,49,10,50,48,52,50,52,44,44,49,10,50,48,56,51,51,44,44,49,10,50,48,56,51,56,44,44,49,10,50,48,56,54,54,44,44,49,10,50,48,56,55,49,44,44,49,10,50,48,56,56,48,44,44,49,10,50,48,57,48,53,44,44,49,10,50,48,57,50,52,44,44,49,10,50,48,57,51,50,44,44,50,10,50,48,57,51,54,44,44,50,10,50,48,57,52,57,44,44,50,10,50,49,48,50,53,44,44,49,10,50,49,48,50,55,44,44,49,10,50,49,56,54,54,44,44,49,10,50,57,48,48,49,44,44,49,10,51,56,53,57,56,44,44,49,10,53,48,50,50,48,44,44,50,10,53,48,50,50,49,44,44,50,10,53,48,50,50,50,44,44,50,10,53,48,50,50,53,44,44,50,10,53,48,50,50,55,44,44,50,10,53,49,57,51,50,44,44,50,10,53,49,57,51,54,44,44,50,10,53,49,57,52,57,44,44,50,10,53,50,57,51,54,44,44,50,10,53,52,57,51,54,44,44,50,10,53,55,48,48,50,44,44,50,10,53,55,48,48,51,44,44,50,10,53,55,48,48,52,44,44,50,10,53,55,48,48,53,44,44,50,10,53,55,48,48,54,44,44,50,10,53,55,48,48,55,44,44,50,10,53,55,48,48,56,44,44,50,10,53,55,48,48,57,44,44,50,10,53,55,48,49,48,44,44,50,10,53,55,48,49,49,44,44,50,10,43,65,71,65,65,89,65,66,103,10,10,84,104,101,32,102,111,108,108,111,119,105,110,103,32,99,111,100,101,112,97,103,101,115,32,97,114,101,32,100,101,112,101,110,100,101,110,99,105,101,115,32,102,111,114,32,86,105,115,117,97,108,32,70,111,120,80,114,111,58,10,10,45,32,54,50,48,32,77,97,122,111,118,105,97,32,40,80,111,108,105,115,104,41,32,77,83,45,68,79,83,10,45,32,56,57,53,32,75,97,109,101,110,105,99,107,43,65,80,48,32,40,67,122,101,99,104,41,32,77,83,45,68,79,83,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,112,97,103,101,115,46,99,115,118,10,54,50,48,44,44,49,10,56,57,53,44,44,49,10,43,65,71,65,65,89,65,66,103,10,10,84,104,101,32,107,110,111,119,110,32,109,105,115,115,105,110,103,32,99,111,100,101,112,97,103,101,115,32,97,114,101,32,101,110,117,109,101,114,97,116,101,100,32,105,110,32,116,104,101,32,82,69,65,68,77,69,46,10,10,43,65,67,77,65,73,119,32,66,117,105,108,100,105,110,103,32,78,111,116,101,115,10,10,84,104,101,32,115,99,114,105,112,116,32,43,65,71,65,45,109,97,107,101,46,115,104,43,65,71,65,32,40,100,101,115,99,114,105,98,101,100,32,108,97,116,101,114,41,32,119,105,108,108,32,103,101,116,32,116,104,101,115,101,32,102,105,108,101,115,32,97,110,100,32,109,97,115,115,97,103,101,32,116,104,101,32,100,97,116,97,10,40,112,114,105,110,116,105,110,103,32,99,111,100,101,45,117,110,105,99,111,100,101,32,112,97,105,114,115,41,46,32,32,84,104,101,32,101,118,101,110,116,117,97,108,32,116,97,98,108,101,115,32,97,114,101,32,100,114,111,112,112,101,100,32,105,110,32,116,104,101,32,112,97,116,104,115,10,43,65,71,65,46,47,99,111,100,101,112,97,103,101,115,47,43,65,68,119,45,67,79,68,69,80,65,71,69,43,65,68,52,46,84,66,76,43,65,71,65,46,32,32,70,111,114,32,101,120,97,109,112,108,101,44,32,116,104,101,32,108,97,115,116,32,49,48,32,108,105,110,101,115,32,111,102,32,43,65,71,65,45,49,48,48,48,48,46,84,66,76,43,65,71,65,32,97,114,101,10,10,43,65,71,65,65,89,65,66,103,65,68,52,10,48,120,70,54,9,48,120,48,50,67,54,10,48,120,70,55,9,48,120,48,50,68,67,10,48,120,70,56,9,48,120,48,48,65,70,10,48,120,70,57,9,48,120,48,50,68,56,10,48,120,70,65,9,48,120,48,50,68,57,10,48,120,70,66,9,48,120,48,50,68,65,10,48,120,70,67,9,48,120,48,48,66,56,10,48,120,70,68,9,48,120,48,50,68,68,10,48,120,70,69,9,48,120,48,50,68,66,10,48,120,70,70,9,48,120,48,50,67,55,10,43,65,71,65,65,89,65,66,103,10,10,119,104,105,99,104,32,105,109,112,108,105,101,115,32,116,104,97,116,32,99,111,100,101,32,48,120,70,54,32,105,115,32,43,65,71,65,45,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,48,120,48,50,67,54,41,43,65,71,65,32,97,110,100,32,118,105,99,101,32,118,101,114,115,97,46,10,10,43,65,67,77,65,73,119,32,87,105,110,100,111,119,115,45,100,101,112,101,110,100,101,110,116,32,98,117,105,108,100,32,115,116,101,112,10,10,84,111,32,98,117,105,108,100,32,116,104,101,32,115,111,117,114,99,101,115,32,111,110,32,119,105,110,100,111,119,115,44,32,99,111,110,115,117,108,116,32,43,65,71,65,45,100,111,116,110,101,116,47,77,97,107,101,69,110,99,111,100,105,110,103,46,99,115,43,65,71,65,46,10,10,65,102,116,101,114,32,115,97,118,105,110,103,32,116,104,101,32,115,116,97,110,100,97,114,100,32,111,117,116,112,117,116,32,116,111,32,43,65,71,65,45,111,117,116,43,65,71,65,44,32,97,32,115,105,109,112,108,101,32,97,119,107,32,115,99,114,105,112,116,32,40,43,65,71,65,45,100,111,116,110,101,116,46,115,104,43,65,71,65,41,32,116,97,107,101,115,32,99,97,114,101,32,111,102,32,116,104,101,32,114,101,115,116,58,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,100,111,116,110,101,116,46,115,104,10,43,65,67,77,65,73,81,45,47,98,105,110,47,98,97,115,104,10,105,102,32,43,65,70,115,32,43,65,67,69,32,45,101,32,100,111,116,110,101,116,47,111,117,116,32,43,65,70,48,65,79,119,32,116,104,101,110,32,101,120,105,116,43,65,68,115,32,102,105,10,43,65,68,119,45,100,111,116,110,101,116,47,111,117,116,32,116,114,32,45,115,32,39,32,39,32,39,43,65,70,119,45,116,39,32,43,65,72,119,32,97,119,107,32,39,78,70,43,65,68,52,45,50,32,43,65,72,115,45,105,102,40,111,117,116,102,105,108,101,41,32,99,108,111,115,101,40,111,117,116,102,105,108,101,41,43,65,68,115,32,111,117,116,102,105,108,101,43,65,68,48,65,73,103,45,99,111,100,101,112,97,103,101,115,47,43,65,67,73,32,43,65,67,81,45,49,32,43,65,67,73,46,84,66,76,43,65,67,73,65,102,81,32,78,70,43,65,68,48,65,80,81,45,50,32,43,65,72,115,45,112,114,105,110,116,32,43,65,68,52,32,111,117,116,102,105,108,101,43,65,72,48,39,10,43,65,71,65,65,89,65,66,103,10,10,43,65,67,77,32,66,117,105,108,100,105,110,103,32,116,104,101,32,115,99,114,105,112,116,10,10,43,65,71,65,45,109,97,107,101,46,110,106,115,43,65,71,65,32,116,97,107,101,115,32,97,32,99,111,100,101,112,97,103,101,32,97,114,103,117,109,101,110,116,44,32,114,101,97,100,115,32,116,104,101,32,99,111,114,114,101,115,112,111,110,100,105,110,103,32,116,97,98,108,101,32,102,105,108,101,32,97,110,100,10,103,101,110,101,114,97,116,101,115,32,74,83,32,99,111,100,101,32,102,111,114,32,101,110,99,111,100,105,110,103,32,97,110,100,32,100,101,99,111,100,105,110,103,58,10,10,43,65,67,77,65,73,119,32,82,97,119,32,67,111,100,101,112,97,103,101,115,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,109,97,107,101,46,110,106,115,10,43,65,67,77,65,73,81,45,47,117,115,114,47,98,105,110,47,101,110,118,32,110,111,100,101,10,118,97,114,32,97,114,103,118,32,43,65,68,48,32,112,114,111,99,101,115,115,46,97,114,103,118,46,115,108,105,99,101,40,49,41,44,32,102,115,32,43,65,68,48,32,114,101,113,117,105,114,101,40,39,102,115,39,41,43,65,68,115,10,105,102,40,97,114,103,118,46,108,101,110,103,116,104,32,43,65,68,119,32,50,41,32,43,65,72,115,10,32,32,32,32,99,111,110,115,111,108,101,46,101,114,114,111,114,40,43,65,67,73,45,117,115,97,103,101,58,32,109,97,107,101,46,110,106,115,32,43,65,68,119,45,99,111,100,101,112,97,103,101,43,65,70,56,45,105,110,100,101,120,43,65,68,52,32,43,65,70,115,45,118,97,114,105,97,98,108,101,43,65,70,48,65,73,103,41,43,65,68,115,10,32,32,32,32,112,114,111,99,101,115,115,46,101,120,105,116,40,50,50,41,43,65,68,115,32,47,43,65,67,111,32,69,73,78,86,65,76,32,43,65,67,111,45,47,10,43,65,72,48,10,10,118,97,114,32,99,112,32,43,65,68,48,32,97,114,103,118,43,65,70,115,45,49,43,65,70,48,65,79,119,10,118,97,114,32,106,115,118,97,114,32,43,65,68,48,32,97,114,103,118,43,65,70,115,45,50,43,65,70,48,32,43,65,72,119,65,102,65,32,43,65,67,73,45,99,112,116,97,98,108,101,43,65,67,73,65,79,119,10,118,97,114,32,120,32,43,65,68,48,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,43,65,67,73,45,99,111,100,101,112,97,103,101,115,47,43,65,67,73,32,43,45,32,99,112,32,43,45,32,43,65,67,73,46,84,66,76,43,65,67,73,44,43,65,67,73,45,117,116,102,56,43,65,67,73,41,43,65,68,115,10,118,97,114,32,109,97,120,99,112,32,43,65,68,48,32,48,43,65,68,115,10,10,118,97,114,32,121,32,43,65,68,48,32,120,46,115,112,108,105,116,40,43,65,67,73,65,88,65,45,110,43,65,67,73,41,46,109,97,112,40,102,117,110,99,116,105,111,110,40,122,41,32,43,65,72,115,10,32,32,32,32,118,97,114,32,119,32,43,65,68,48,32,122,46,115,112,108,105,116,40,43,65,67,73,65,88,65,45,116,43,65,67,73,41,43,65,68,115,10,32,32,32,32,105,102,40,119,46,108,101,110,103,116,104,32,43,65,68,119,32,50,41,32,114,101,116,117,114,110,32,119,43,65,68,115,10,32,32,32,32,114,101,116,117,114,110,32,43,65,70,115,45,78,117,109,98,101,114,40,119,43,65,70,115,45,48,43,65,70,48,41,44,32,78,117,109,98,101,114,40,119,43,65,70,115,45,49,43,65,70,48,41,43,65,70,48,65,79,119,10,43,65,72,48,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,122,41,32,43,65,72,115,32,114,101,116,117,114,110,32,122,46,108,101,110,103,116,104,32,43,65,68,52,32,49,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,43,65,71,65,65,89,65,66,103,10,10,84,104,101,32,68,66,67,83,32,97,110,100,32,83,66,67,83,32,99,111,100,101,32,103,101,110,101,114,97,116,105,111,110,32,115,116,114,97,116,101,103,105,101,115,32,97,114,101,32,100,105,102,102,101,114,101,110,116,46,32,32,84,104,101,32,109,97,120,105,109,117,109,32,99,111,100,101,32,105,115,10,117,115,101,100,32,116,111,32,100,105,115,116,105,110,103,117,105,115,104,32,40,109,97,120,32,48,120,70,70,32,102,111,114,32,83,66,67,83,41,46,10,10,43,65,71,65,65,89,65,66,103,10,102,111,114,40,118,97,114,32,105,32,43,65,68,48,32,48,43,65,68,115,32,105,32,43,65,67,69,65,80,81,32,121,46,108,101,110,103,116,104,43,65,68,115,32,43,45,43,45,105,41,32,105,102,40,121,43,65,70,115,45,105,43,65,70,48,65,87,119,45,48,43,65,70,48,32,43,65,68,52,32,109,97,120,99,112,41,32,109,97,120,99,112,32,43,65,68,48,32,121,43,65,70,115,45,105,43,65,70,48,65,87,119,45,48,43,65,70,48,65,79,119,10,10,118,97,114,32,101,110,99,32,43,65,68,48,32,43,65,72,115,65,102,81,44,32,100,101,99,32,43,65,68,48,32,40,109,97,120,99,112,32,43,65,68,119,32,50,53,54,32,63,32,43,65,70,115,65,88,81,32,58,32,43,65,72,115,65,102,81,41,43,65,68,115,10,102,111,114,40,118,97,114,32,105,32,43,65,68,48,32,48,43,65,68,115,32,105,32,43,65,67,69,65,80,81,32,121,46,108,101,110,103,116,104,43,65,68,115,32,43,45,43,45,105,41,32,43,65,72,115,10,32,32,32,32,100,101,99,43,65,70,115,45,121,43,65,70,115,45,105,43,65,70,48,65,87,119,45,48,43,65,70,48,65,88,81,32,43,65,68,48,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,121,43,65,70,115,45,105,43,65,70,48,65,87,119,45,49,43,65,70,48,41,43,65,68,115,10,32,32,32,32,101,110,99,43,65,70,115,45,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,121,43,65,70,115,45,105,43,65,70,48,65,87,119,45,49,43,65,70,48,41,43,65,70,48,32,43,65,68,48,32,121,43,65,70,115,45,105,43,65,70,48,65,87,119,45,48,43,65,70,48,65,79,119,10,43,65,72,48,10,10,118,97,114,32,111,100,101,99,44,32,111,101,110,99,44,32,111,117,116,115,116,114,43,65,68,115,10,105,102,40,109,97,120,99,112,32,43,65,68,119,32,50,53,54,41,32,43,65,72,115,10,43,65,71,65,65,89,65,66,103,10,10,84,104,101,32,117,110,105,99,111,100,101,32,99,104,97,114,97,99,116,101,114,32,43,65,71,65,45,48,120,70,70,70,68,43,65,71,65,32,40,82,69,80,76,65,67,69,77,69,78,84,32,67,72,65,82,65,67,84,69,82,41,32,105,115,32,117,115,101,100,32,97,115,32,97,32,112,108,97,99,101,104,111,108,100,101,114,10,102,111,114,32,99,104,97,114,97,99,116,101,114,115,32,116,104,97,116,32,97,114,101,32,110,111,116,32,115,112,101,99,105,102,105,101,100,32,105,110,32,116,104,101,32,109,97,112,32,40,102,111,114,32,101,120,97,109,112,108,101,44,32,43,65,71,65,45,48,120,70,48,43,65,71,65,32,105,115,32,110,111,116,32,105,110,10,99,111,100,101,32,112,97,103,101,32,49,48,48,48,48,41,46,10,10,70,111,114,32,83,66,67,83,44,32,116,104,101,32,105,100,101,97,32,105,115,32,116,111,32,101,109,98,101,100,32,97,32,114,97,119,32,115,116,114,105,110,103,32,119,105,116,104,32,116,104,101,32,99,111,110,116,101,110,116,115,32,111,102,32,116,104,101,32,50,53,54,32,99,111,100,101,115,46,10,84,104,101,32,43,65,71,65,45,100,101,99,43,65,71,65,32,102,105,101,108,100,32,105,115,32,109,101,114,101,108,121,32,97,32,115,112,108,105,116,32,111,102,32,116,104,101,32,115,116,114,105,110,103,44,32,97,110,100,32,43,65,71,65,45,101,110,99,43,65,71,65,32,105,115,32,97,110,32,101,118,101,114,115,105,111,110,58,10,10,43,65,71,65,65,89,65,66,103,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,43,65,68,48,32,48,43,65,68,115,32,105,32,43,65,67,69,65,80,81,32,50,53,54,43,65,68,115,32,43,45,43,45,105,41,32,105,102,40,116,121,112,101,111,102,32,100,101,99,43,65,70,115,45,105,43,65,70,48,32,43,65,68,48,65,80,81,65,57,32,43,65,67,73,45,117,110,100,101,102,105,110,101,100,43,65,67,73,41,32,100,101,99,43,65,70,115,45,105,43,65,70,48,32,43,65,68,48,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,48,120,70,70,70,68,41,43,65,68,115,10,32,32,32,32,111,100,101,99,32,43,65,68,48,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,100,101,99,46,106,111,105,110,40,43,65,67,73,65,73,103,41,41,43,65,68,115,10,32,32,32,32,111,117,116,115,116,114,32,43,65,68,48,32,39,40,102,117,110,99,116,105,111,110,40,41,43,65,72,115,32,118,97,114,32,100,32,43,65,68,48,32,39,32,43,45,32,111,100,101,99,32,43,45,32,39,44,32,68,32,43,65,68,48,32,43,65,70,115,65,88,81,44,32,101,32,43,65,68,48,32,43,65,72,115,65,102,81,65,55,32,102,111,114,40,118,97,114,32,105,43,65,68,48,45,48,43,65,68,115,45,105,43,65,67,69,65,80,81,45,100,46,108,101,110,103,116,104,43,65,68,115,65,75,119,65,114,45,105,41,32,43,65,72,115,32,105,102,40,100,46,99,104,97,114,67,111,100,101,65,116,40,105,41,32,43,65,67,69,65,80,81,65,57,32,48,120,70,70,70,68,41,32,101,43,65,70,115,45,100,43,65,70,115,45,105,43,65,70,48,65,88,81,32,43,65,68,48,32,105,43,65,68,115,32,68,43,65,70,115,45,105,43,65,70,48,32,43,65,68,48,32,100,46,99,104,97,114,65,116,40,105,41,43,65,68,115,32,43,65,72,48,32,114,101,116,117,114,110,32,43,65,72,115,65,73,103,45,101,110,99,43,65,67,73,58,32,101,44,32,43,65,67,73,45,100,101,99,43,65,67,73,58,32,68,32,43,65,72,48,65,79,119,32,43,65,72,48,41,40,41,43,65,68,115,39,43,65,68,115,10,43,65,72,48,32,101,108,115,101,32,43,65,72,115,10,43,65,71,65,65,89,65,66,103,10,10,68,66,67,83,32,105,115,32,115,105,109,105,108,97,114,44,32,101,120,99,101,112,116,32,116,104,97,116,32,116,104,101,32,115,112,97,99,101,32,105,115,32,115,108,105,99,101,100,32,105,110,116,111,32,50,53,54,45,98,121,116,101,32,99,104,117,110,107,115,32,40,115,116,114,105,110,103,115,10,97,114,101,32,111,110,108,121,32,103,101,110,101,114,97,116,101,100,32,102,111,114,32,116,104,111,115,101,32,104,105,103,104,45,98,121,116,101,115,32,114,101,112,114,101,115,101,110,116,101,100,32,105,110,32,116,104,101,32,99,111,100,101,112,97,103,101,41,46,10,10,84,104,101,32,115,116,114,97,116,101,103,121,32,105,115,32,116,111,32,99,111,110,115,116,114,117,99,116,32,97,110,32,97,114,114,97,121,45,111,102,45,97,114,114,97,121,115,32,115,111,32,116,104,97,116,32,43,65,71,65,45,100,100,43,65,70,115,45,104,105,103,104,43,65,70,48,65,87,119,45,108,111,119,43,65,70,48,65,89,65,32,105,115,32,116,104,101,10,99,104,97,114,97,99,116,101,114,32,97,115,115,111,99,105,97,116,101,100,32,119,105,116,104,32,116,104,101,32,99,111,100,101,46,32,32,84,104,105,115,32,97,114,114,97,121,32,105,115,32,99,111,109,98,105,110,101,100,32,97,116,32,114,117,110,116,105,109,101,32,116,111,32,121,105,101,108,100,10,116,104,101,32,99,111,109,112,108,101,116,101,32,100,101,99,111,100,105,110,103,32,111,98,106,101,99,116,32,40,97,110,100,32,116,104,101,32,101,110,99,111,100,105,110,103,32,111,98,106,101,99,116,32,105,115,32,97,110,32,101,118,101,114,115,105,111,110,41,58,10,10,43,65,71,65,65,89,65,66,103,10,32,32,32,32,118,97,114,32,100,100,32,43,65,68,48,32,43,65,70,115,65,88,81,65,55,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,105,110,32,100,101,99,41,32,105,102,40,100,101,99,46,104,97,115,79,119,110,80,114,111,112,101,114,116,121,40,105,41,41,32,43,65,72,115,10,32,32,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,100,100,43,65,70,115,45,105,32,43,65,68,52,65,80,103,32,56,43,65,70,48,32,43,65,68,48,65,80,81,65,57,32,43,65,67,73,45,117,110,100,101,102,105,110,101,100,43,65,67,73,41,32,100,100,43,65,70,115,45,105,32,43,65,68,52,65,80,103,32,56,43,65,70,48,32,43,65,68,48,32,43,65,70,115,65,88,81,65,55,10,32,32,32,32,32,32,32,32,100,100,43,65,70,115,45,105,32,43,65,68,52,65,80,103,32,56,43,65,70,48,65,87,119,45,105,32,43,65,67,85,32,50,53,54,43,65,70,48,32,43,65,68,48,32,100,101,99,43,65,70,115,45,105,43,65,70,48,65,79,119,10,32,32,32,32,43,65,72,48,10,32,32,32,32,111,117,116,115,116,114,32,43,65,68,48,32,39,40,102,117,110,99,116,105,111,110,40,41,43,65,72,115,32,118,97,114,32,100,32,43,65,68,48,32,43,65,70,115,65,88,81,44,32,101,32,43,65,68,48,32,43,65,72,115,65,102,81,44,32,68,32,43,65,68,48,32,43,65,70,115,65,88,81,44,32,106,43,65,68,115,65,88,65,45,110,39,43,65,68,115,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,43,65,68,48,32,48,43,65,68,115,32,105,32,43,65,67,69,65,80,81,32,50,53,54,43,65,68,115,32,43,45,43,45,105,41,32,105,102,40,100,100,43,65,70,115,45,105,43,65,70,48,41,32,43,65,72,115,10,32,32,32,32,32,32,32,32,102,111,114,40,118,97,114,32,106,32,43,65,68,48,32,48,43,65,68,115,32,106,32,43,65,67,69,65,80,81,32,50,53,54,43,65,68,115,32,43,45,43,45,106,41,32,105,102,40,116,121,112,101,111,102,32,100,100,43,65,70,115,45,105,43,65,70,48,65,87,119,45,106,43,65,70,48,32,43,65,68,48,65,80,81,65,57,32,43,65,67,73,45,117,110,100,101,102,105,110,101,100,43,65,67,73,41,32,100,100,43,65,70,115,45,105,43,65,70,48,65,87,119,45,106,43,65,70,48,32,43,65,68,48,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,48,120,70,70,70,68,41,43,65,68,115,10,32,32,32,32,32,32,32,32,111,117,116,115,116,114,32,43,45,43,65,68,48,32,39,68,43,65,70,115,39,32,43,45,32,105,32,43,45,32,39,43,65,70,48,32,43,65,68,48,32,39,32,43,45,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,100,100,43,65,70,115,45,105,43,65,70,48,46,106,111,105,110,40,43,65,67,73,65,73,103,41,41,32,43,45,32,39,46,115,112,108,105,116,40,43,65,67,73,65,73,103,41,43,65,68,115,65,88,65,45,110,39,43,65,68,115,10,32,32,32,32,32,32,32,32,111,117,116,115,116,114,32,43,45,43,65,68,48,32,39,102,111,114,40,106,32,43,65,68,48,32,48,43,65,68,115,32,106,32,43,65,67,69,65,80,81,32,68,43,65,70,115,39,32,43,45,32,105,32,43,45,32,39,43,65,70,48,46,108,101,110,103,116,104,43,65,68,115,32,43,45,43,45,106,41,32,105,102,40,68,43,65,70,115,39,32,43,45,32,105,32,43,45,32,39,43,65,70,48,65,87,119,45,106,43,65,70,48,46,99,104,97,114,67,111,100,101,65,116,40,48,41,32,43,65,67,69,65,80,81,65,57,32,48,120,70,70,70,68,41,32,43,65,72,115,32,101,43,65,70,115,45,68,43,65,70,115,39,32,43,45,32,105,32,43,45,32,39,43,65,70,48,65,87,119,45,106,43,65,70,48,65,88,81,32,43,65,68,48,32,39,32,43,45,32,40,105,43,65,67,111,45,50,53,54,41,32,43,45,32,39,32,43,45,32,106,43,65,68,115,32,100,43,65,70,115,39,32,43,45,32,40,105,43,65,67,111,45,50,53,54,41,32,43,45,32,39,32,43,45,32,106,43,65,70,48,32,43,65,68,48,32,68,43,65,70,115,39,32,43,45,32,105,32,43,45,32,39,43,65,70,48,65,87,119,45,106,43,65,70,48,65,79,119,66,57,65,70,119,45,110,39,10,32,32,32,32,43,65,72,48,10,32,32,32,32,111,117,116,115,116,114,32,43,45,43,65,68,48,32,39,114,101,116,117,114,110,32,43,65,72,115,65,73,103,45,101,110,99,43,65,67,73,58,32,101,44,32,43,65,67,73,45,100,101,99,43,65,67,73,58,32,100,32,43,65,72,48,65,79,119,32,43,65,72,48,41,40,41,43,65,68,115,39,43,65,68,115,10,43,65,72,48,10,99,111,110,115,111,108,101,46,108,111,103,40,106,115,118,97,114,32,43,45,32,43,65,67,73,65,87,119,65,105,32,43,45,32,99,112,32,43,45,32,43,65,67,73,65,88,81,32,43,65,68,48,32,43,65,67,73,32,43,45,32,111,117,116,115,116,114,41,43,65,68,115,10,10,43,65,71,65,65,89,65,66,103,10,10,43,65,71,65,45,109,97,107,101,46,115,104,43,65,71,65,32,103,101,110,101,114,97,116,101,115,32,116,104,101,32,116,97,98,108,101,115,32,117,115,101,100,32,98,121,32,43,65,71,65,45,109,97,107,101,46,110,106,115,43,65,71,65,46,32,32,84,104,101,32,114,97,119,32,117,110,105,99,111,100,101,32,84,88,84,32,102,105,108,101,115,10,97,114,101,32,99,111,108,117,109,110,97,114,58,32,43,65,71,65,45,99,111,100,101,32,117,110,105,99,111,100,101,32,43,65,67,77,45,99,111,109,109,101,110,116,115,43,65,71,65,46,32,32,70,111,114,32,101,120,97,109,112,108,101,44,32,116,104,101,32,108,97,115,116,32,49,48,32,108,105,110,101,115,32,111,102,32,116,104,101,10,116,101,120,116,32,102,105,108,101,32,82,79,77,65,78,46,84,88,84,32,40,102,111,114,32,67,80,32,49,48,48,48,48,41,32,97,114,101,58,10,10,43,65,71,65,65,89,65,66,103,65,68,52,10,48,120,70,54,9,48,120,48,50,67,54,9,43,65,67,77,45,77,79,68,73,70,73,69,82,32,76,69,84,84,69,82,32,67,73,82,67,85,77,70,76,69,88,32,65,67,67,69,78,84,10,48,120,70,55,9,48,120,48,50,68,67,9,43,65,67,77,45,83,77,65,76,76,32,84,73,76,68,69,10,48,120,70,56,9,48,120,48,48,65,70,9,43,65,67,77,45,77,65,67,82,79,78,10,48,120,70,57,9,48,120,48,50,68,56,9,43,65,67,77,45,66,82,69,86,69,10,48,120,70,65,9,48,120,48,50,68,57,9,43,65,67,77,45,68,79,84,32,65,66,79,86,69,10,48,120,70,66,9,48,120,48,50,68,65,9,43,65,67,77,45,82,73,78,71,32,65,66,79,86,69,10,48,120,70,67,9,48,120,48,48,66,56,9,43,65,67,77,45,67,69,68,73,76,76,65,10,48,120,70,68,9,48,120,48,50,68,68,9,43,65,67,77,45,68,79,85,66,76,69,32,65,67,85,84,69,32,65,67,67,69,78,84,10,48,120,70,69,9,48,120,48,50,68,66,9,43,65,67,77,45,79,71,79,78,69,75,10,48,120,70,70,9,48,120,48,50,67,55,9,43,65,67,77,45,67,65,82,79,78,10,43,65,71,65,65,89,65,66,103,10,10,73,110,32,112,114,111,99,101,115,115,105,110,103,32,116,104,101,32,100,97,116,97,44,32,116,104,101,32,99,111,109,109,101,110,116,115,32,40,97,102,116,101,114,32,116,104,101,32,43,65,71,65,65,73,119,66,103,41,32,97,114,101,32,115,116,114,105,112,112,101,100,32,97,110,100,32,117,110,100,101,102,105,110,101,100,10,101,108,101,109,101,110,116,115,32,40,108,105,107,101,32,43,65,71,65,45,48,120,55,70,43,65,71,65,32,102,111,114,32,67,80,32,49,48,48,48,48,41,32,97,114,101,32,114,101,109,111,118,101,100,46,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,109,97,107,101,46,115,104,10,43,65,67,77,65,73,81,45,47,98,105,110,47,98,97,115,104,10,73,78,70,73,76,69,43,65,68,48,65,74,65,66,55,45,49,58,45,112,97,103,101,115,46,99,115,118,43,65,72,48,10,79,85,84,70,73,76,69,43,65,68,48,65,74,65,66,55,45,50,58,45,99,112,116,97,98,108,101,46,106,115,43,65,72,48,10,74,83,86,65,82,43,65,68,48,65,74,65,66,55,45,51,58,45,99,112,116,97,98,108,101,43,65,72,48,10,86,69,82,83,73,79,78,43,65,68,48,65,74,65,40,99,97,116,32,112,97,99,107,97,103,101,46,106,115,111,110,32,43,65,72,119,32,103,114,101,112,32,118,101,114,115,105,111,110,32,43,65,72,119,32,116,114,32,45,100,99,32,43,65,70,115,45,48,45,57,46,43,65,70,48,41,10,10,109,107,100,105,114,32,45,112,32,99,111,100,101,112,97,103,101,115,32,98,105,116,115,10,114,109,32,45,102,32,43,65,67,81,45,79,85,84,70,73,76,69,32,43,65,67,81,45,79,85,84,70,73,76,69,46,116,109,112,10,101,99,104,111,32,43,65,67,73,45,47,43,65,67,111,32,43,65,67,81,45,79,85,84,70,73,76,69,32,40,67,41,32,50,48,49,51,45,50,48,49,52,32,83,104,101,101,116,74,83,32,45,45,32,104,116,116,112,58,47,47,115,104,101,101,116,106,115,46,99,111,109,32,43,65,67,111,45,47,43,65,67,73,32,43,65,68,52,32,43,65,67,81,45,79,85,84,70,73,76,69,46,116,109,112,10,101,99,104,111,32,43,65,67,73,45,47,43,65,67,111,45,106,115,104,105,110,116,32,45,87,49,48,48,32,43,65,67,111,45,47,43,65,67,73,32,43,65,68,52,65,80,103,32,43,65,67,81,45,79,85,84,70,73,76,69,46,116,109,112,10,101,99,104,111,32,43,65,67,73,45,118,97,114,32,43,65,67,81,45,74,83,86,65,82,32,43,65,68,48,32,43,65,72,115,45,118,101,114,115,105,111,110,58,43,65,70,119,65,73,103,65,107,45,86,69,82,83,73,79,78,43,65,70,119,65,73,103,66,57,65,68,115,65,73,103,32,43,65,68,52,65,80,103,32,43,65,67,81,45,79,85,84,70,73,76,69,46,116,109,112,10,105,102,32,43,65,70,115,32,45,101,32,100,111,116,110,101,116,46,115,104,32,43,65,70,48,65,79,119,32,116,104,101,110,32,98,97,115,104,32,100,111,116,110,101,116,46,115,104,43,65,68,115,32,102,105,10,97,119,107,32,45,70,44,32,39,43,65,72,115,45,112,114,105,110,116,32,43,65,67,81,45,49,44,32,43,65,67,81,45,50,44,32,43,65,67,81,45,51,43,65,72,48,39,32,43,65,67,81,45,73,78,70,73,76,69,32,43,65,72,119,32,119,104,105,108,101,32,114,101,97,100,32,99,112,32,117,114,108,32,99,112,116,121,112,101,43,65,68,115,32,100,111,10,32,32,32,32,101,99,104,111,32,43,65,67,81,45,99,112,32,43,65,67,81,45,117,114,108,10,32,32,32,32,105,102,32,43,65,70,115,32,43,65,67,69,32,45,101,32,99,111,100,101,112,97,103,101,115,47,43,65,67,81,45,99,112,46,84,66,76,32,43,65,70,48,65,79,119,32,116,104,101,110,10,32,32,32,32,32,32,32,32,99,117,114,108,32,43,65,67,81,45,117,114,108,32,43,65,72,119,32,115,101,100,32,39,115,47,43,65,67,77,46,43,65,67,111,45,47,47,103,39,32,43,65,72,119,32,97,119,107,32,39,78,70,43,65,68,48,65,80,81,45,50,39,32,43,65,68,52,32,99,111,100,101,112,97,103,101,115,47,43,65,67,81,45,99,112,46,84,66,76,10,32,32,32,32,102,105,10,32,32,32,32,101,99,104,111,32,43,65,67,73,45,105,102,40,116,121,112,101,111,102,32,43,65,67,81,45,74,83,86,65,82,32,43,65,68,48,65,80,81,65,57,32,39,117,110,100,101,102,105,110,101,100,39,41,32,43,65,67,81,45,74,83,86,65,82,32,43,65,68,48,32,43,65,72,115,65,102,81,65,55,65,67,73,32,43,65,68,52,32,98,105,116,115,47,43,65,67,81,45,99,112,46,106,115,46,116,109,112,10,32,32,32,32,110,111,100,101,32,109,97,107,101,46,110,106,115,32,43,65,67,81,45,99,112,32,43,65,67,81,45,74,83,86,65,82,32,43,65,72,119,32,116,101,101,32,45,97,32,98,105,116,115,47,43,65,67,81,45,99,112,46,106,115,46,116,109,112,32,43,65,68,52,65,80,103,32,43,65,67,81,45,79,85,84,70,73,76,69,46,116,109,112,10,32,32,32,32,115,101,100,32,39,115,47,43,65,67,73,65,88,65,40,43,65,70,115,45,48,45,57,43,65,70,48,65,75,119,66,99,41,43,65,67,73,58,47,43,65,70,119,45,49,58,47,103,39,32,43,65,68,119,45,98,105,116,115,47,43,65,67,81,45,99,112,46,106,115,46,116,109,112,32,43,65,68,52,45,98,105,116,115,47,43,65,67,81,45,99,112,46,106,115,10,32,32,32,32,114,109,32,45,102,32,98,105,116,115,47,43,65,67,81,45,99,112,46,106,115,46,116,109,112,10,100,111,110,101,10,101,99,104,111,32,43,65,67,73,45,105,102,32,40,116,121,112,101,111,102,32,109,111,100,117,108,101,32,43,65,67,69,65,80,81,65,57,32,39,117,110,100,101,102,105,110,101,100,39,32,43,65,67,89,65,74,103,32,109,111,100,117,108,101,46,101,120,112,111,114,116,115,41,32,109,111,100,117,108,101,46,101,120,112,111,114,116,115,32,43,65,68,48,32,43,65,67,81,45,74,83,86,65,82,43,65,68,115,65,73,103,32,43,65,68,52,65,80,103,32,43,65,67,81,45,79,85,84,70,73,76,69,46,116,109,112,10,115,101,100,32,39,115,47,43,65,67,73,65,88,65,40,43,65,70,115,45,48,45,57,43,65,70,48,65,75,119,66,99,41,43,65,67,73,58,47,43,65,70,119,45,49,58,47,103,39,32,43,65,68,119,65,74,65,45,79,85,84,70,73,76,69,46,116,109,112,32,43,65,68,52,65,74,65,45,79,85,84,70,73,76,69,10,114,109,32,45,102,32,43,65,67,81,45,79,85,84,70,73,76,69,46,116,109,112,10,43,65,71,65,65,89,65,66,103,10,10,43,65,67,77,65,73,119,32,85,116,105,108,105,116,105,101,115,10,10,84,104,101,32,101,110,99,111,100,101,32,97,110,100,32,100,101,99,111,100,101,32,102,117,110,99,116,105,111,110,115,32,97,114,101,32,107,101,112,116,32,105,110,32,97,32,115,101,112,97,114,97,116,101,32,115,99,114,105,112,116,32,40,99,112,117,116,105,108,115,46,106,115,41,46,10,10,66,111,116,104,32,101,110,99,111,100,101,32,97,110,100,32,100,101,99,111,100,101,32,100,101,97,108,32,119,105,116,104,32,100,97,116,97,32,114,101,112,114,101,115,101,110,116,101,100,32,97,115,58,10,10,45,32,83,116,114,105,110,103,32,40,101,110,99,111,100,101,32,101,120,112,101,99,116,115,32,74,83,32,115,116,114,105,110,103,44,32,100,101,99,111,100,101,32,105,110,116,101,114,112,114,101,116,115,32,85,67,83,50,32,99,104,97,114,115,32,97,115,32,99,111,100,101,115,41,10,45,32,65,114,114,97,121,32,40,101,110,99,111,100,101,32,101,120,112,101,99,116,115,32,97,114,114,97,121,32,111,102,32,74,83,32,83,116,114,105,110,103,32,99,104,97,114,97,99,116,101,114,115,44,32,100,101,99,111,100,101,32,101,120,112,101,99,116,115,32,110,117,109,98,101,114,115,41,10,45,32,66,117,102,102,101,114,32,40,101,110,99,111,100,101,32,101,120,112,101,99,116,115,32,85,84,70,45,56,32,115,116,114,105,110,103,44,32,100,101,99,111,100,101,32,101,120,112,101,99,116,115,32,99,111,100,101,112,111,105,110,116,115,47,98,121,116,101,115,41,46,10,10,84,104,101,32,43,65,71,65,45,111,102,109,116,43,65,71,65,32,118,97,114,105,97,98,108,101,32,99,111,110,116,114,111,108,115,32,43,65,71,65,45,101,110,99,111,100,101,43,65,71,65,32,111,117,116,112,117,116,32,40,43,65,71,65,45,115,116,114,43,65,71,65,44,32,43,65,71,65,45,97,114,114,43,65,71,65,32,114,101,115,112,101,99,116,105,118,101,108,121,41,10,119,104,105,108,101,32,116,104,101,32,105,110,112,117,116,32,102,111,114,109,97,116,32,105,115,32,97,117,116,111,109,97,116,105,99,97,108,108,121,32,100,101,116,101,114,109,105,110,101,100,46,10,10,43,65,67,77,32,84,101,115,116,115,10,10,84,104,101,32,116,101,115,116,115,32,105,110,99,108,117,100,101,32,74,83,32,118,97,108,105,100,105,116,121,32,116,101,115,116,115,32,40,114,101,113,117,105,114,105,110,103,32,111,114,32,101,118,97,108,39,105,110,103,32,99,111,100,101,41,58,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,116,101,115,116,46,106,115,10,118,97,114,32,102,115,32,43,65,68,48,32,114,101,113,117,105,114,101,40,39,102,115,39,41,44,32,97,115,115,101,114,116,32,43,65,68,48,32,114,101,113,117,105,114,101,40,39,97,115,115,101,114,116,39,41,44,32,118,109,32,43,65,68,48,32,114,101,113,117,105,114,101,40,39,118,109,39,41,43,65,68,115,10,118,97,114,32,99,112,116,97,98,108,101,44,32,115,98,99,115,43,65,68,115,10,100,101,115,99,114,105,98,101,40,39,115,111,117,114,99,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,105,116,40,39,115,104,111,117,108,100,32,108,111,97,100,32,110,111,100,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,32,99,112,116,97,98,108,101,32,43,65,68,48,32,114,101,113,117,105,114,101,40,39,46,47,39,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,32,32,105,116,40,39,115,104,111,117,108,100,32,108,111,97,100,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,32,115,98,99,115,32,43,65,68,48,32,114,101,113,117,105,114,101,40,39,46,47,115,98,99,115,39,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,32,32,105,116,40,39,115,104,111,117,108,100,32,108,111,97,100,32,101,120,99,101,108,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,32,101,120,99,101,108,32,43,65,68,48,32,114,101,113,117,105,114,101,40,39,46,47,99,112,101,120,99,101,108,39,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,32,32,105,116,40,39,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,98,105,116,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,118,97,114,32,102,105,108,101,115,32,43,65,68,48,32,102,115,46,114,101,97,100,100,105,114,83,121,110,99,40,39,98,105,116,115,39,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,120,41,43,65,72,115,45,114,101,116,117,114,110,32,120,46,115,117,98,115,116,114,40,45,51,41,43,65,68,48,65,80,81,65,105,46,106,115,43,65,67,73,65,79,119,66,57,41,43,65,68,115,10,32,32,32,32,102,105,108,101,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,120,41,32,43,65,72,115,10,32,32,32,32,32,32,118,109,46,114,117,110,73,110,84,104,105,115,67,111,110,116,101,120,116,40,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,98,105,116,115,47,39,32,43,45,32,120,41,41,43,65,68,115,10,32,32,32,32,43,65,72,48,41,43,65,68,115,10,32,32,43,65,72,48,41,43,65,68,115,10,43,65,72,48,41,43,65,68,115,10,43,65,71,65,65,89,65,66,103,10,10,84,104,101,32,82,69,65,68,77,69,32,116,101,115,116,115,32,118,101,114,105,102,121,32,116,104,101,32,115,110,105,112,112,101,116,115,32,105,110,32,116,104,101,32,82,69,65,68,77,69,58,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,116,101,115,116,46,106,115,10,100,101,115,99,114,105,98,101,40,39,82,69,65,68,77,69,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,118,97,114,32,114,101,97,100,109,101,32,43,65,68,48,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,118,97,114,32,117,110,105,99,111,100,101,43,65,70,56,45,99,112,49,48,48,48,48,43,65,70,56,45,50,53,53,32,43,65,68,48,32,99,112,116,97,98,108,101,43,65,70,115,45,49,48,48,48,48,43,65,70,48,46,100,101,99,43,65,70,115,45,50,53,53,43,65,70,48,65,79,119,32,47,47,32,43,65,115,99,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,117,110,105,99,111,100,101,43,65,70,56,45,99,112,49,48,48,48,48,43,65,70,56,45,50,53,53,44,32,43,65,67,73,67,120,119,65,105,41,43,65,68,115,10,10,32,32,32,32,118,97,114,32,99,112,49,48,48,48,48,43,65,70,56,45,55,49,49,32,43,65,68,48,32,99,112,116,97,98,108,101,43,65,70,115,45,49,48,48,48,48,43,65,70,48,46,101,110,99,43,65,70,115,45,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,55,49,49,41,43,65,70,48,65,79,119,32,47,47,32,50,53,53,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,99,112,49,48,48,48,48,43,65,70,56,45,55,49,49,44,32,50,53,53,41,43,65,68,115,10,10,32,32,32,32,118,97,114,32,98,49,32,43,65,68,48,32,43,65,70,115,45,48,120,98,98,44,48,120,101,51,44,48,120,100,55,44,48,120,100,99,43,65,70,48,65,79,119,10,32,32,32,32,118,97,114,32,43,98,69,100,103,79,119,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,57,51,54,44,32,98,49,41,43,65,68,115,10,32,32,32,32,118,97,114,32,98,117,102,32,43,65,68,48,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,57,51,54,44,32,32,43,98,69,100,103,79,119,41,43,65,68,115,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,43,98,69,100,103,79,119,44,43,65,67,74,115,82,50,65,55,65,67,73,41,43,65,68,115,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,98,117,102,46,108,101,110,103,116,104,44,32,52,41,43,65,68,115,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,43,65,68,48,32,48,43,65,68,115,32,105,32,43,65,67,69,65,80,81,32,52,43,65,68,115,32,43,45,43,45,105,41,32,97,115,115,101,114,116,46,101,113,117,97,108,40,98,49,43,65,70,115,45,105,43,65,70,48,44,32,98,117,102,43,65,70,115,45,105,43,65,70,48,41,43,65,68,115,10,10,32,32,32,32,118,97,114,32,98,50,32,43,65,68,48,32,43,65,70,115,45,48,120,102,48,44,48,120,57,102,44,48,120,56,100,44,48,120,97,51,43,65,70,48,65,79,119,10,32,32,32,32,118,97,114,32,115,117,115,104,105,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,54,53,48,48,49,44,32,98,50,41,43,65,68,115,10,32,32,32,32,118,97,114,32,115,98,117,102,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,54,53,48,48,49,44,32,115,117,115,104,105,41,43,65,68,115,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,115,117,115,104,105,44,43,65,67,76,89,80,78,57,106,65,67,73,41,43,65,68,115,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,115,98,117,102,46,108,101,110,103,116,104,44,32,52,41,43,65,68,115,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,43,65,68,48,32,48,43,65,68,115,32,105,32,43,65,67,69,65,80,81,32,52,43,65,68,115,32,43,45,43,45,105,41,32,97,115,115,101,114,116,46,101,113,117,97,108,40,98,50,43,65,70,115,45,105,43,65,70,48,44,32,115,98,117,102,43,65,70,115,45,105,43,65,70,48,41,43,65,68,115,10,10,32,32,43,65,72,48,65,79,119,10,32,32,105,116,40,39,115,104,111,117,108,100,32,98,101,32,99,111,114,114,101,99,116,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,114,101,97,100,109,101,40,41,43,65,68,115,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,114,101,97,100,109,101,40,41,43,65,68,115,10,32,32,43,65,72,48,41,43,65,68,115,10,43,65,72,48,41,43,65,68,115,10,43,65,71,65,65,89,65,66,103,10,10,84,104,101,32,99,111,110,115,105,115,116,101,110,99,121,32,116,101,115,116,115,32,109,97,107,101,32,115,117,114,101,32,116,104,97,116,32,101,110,99,111,100,105,110,103,32,97,110,100,32,100,101,99,111,100,105,110,103,32,97,114,101,32,112,115,101,117,100,111,32,105,110,118,101,114,115,101,115,58,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,116,101,115,116,46,106,115,10,100,101,115,99,114,105,98,101,40,39,99,111,110,115,105,115,116,101,110,99,121,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,99,112,116,97,98,108,101,32,43,65,68,48,32,114,101,113,117,105,114,101,40,39,46,47,39,41,43,65,68,115,10,32,32,85,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,43,65,68,115,10,32,32,118,97,114,32,99,104,107,32,43,65,68,48,32,102,117,110,99,116,105,111,110,40,99,112,116,97,98,108,101,44,32,99,97,99,104,101,105,116,41,32,43,65,72,115,32,114,101,116,117,114,110,32,102,117,110,99,116,105,111,110,40,120,41,32,43,65,72,115,10,32,32,32,32,105,116,40,39,115,104,111,117,108,100,32,99,111,110,115,105,115,116,101,110,116,108,121,32,112,114,111,99,101,115,115,32,67,80,32,39,32,43,45,32,120,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,32,32,118,97,114,32,99,112,32,43,65,68,48,32,99,112,116,97,98,108,101,43,65,70,115,45,120,43,65,70,48,44,32,68,32,43,65,68,48,32,99,112,46,100,101,99,44,32,69,32,43,65,68,48,32,99,112,46,101,110,99,43,65,68,115,10,32,32,32,32,32,32,105,102,40,99,97,99,104,101,105,116,41,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,32,32,101,108,115,101,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,32,32,79,98,106,101,99,116,46,107,101,121,115,40,68,41,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,100,41,32,43,65,72,115,10,32,32,32,32,32,32,32,32,105,102,40,69,43,65,70,115,45,68,43,65,70,115,45,100,43,65,70,48,65,88,81,32,43,65,67,69,65,80,81,32,100,41,32,43,65,72,115,10,32,32,32,32,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,69,43,65,70,115,45,68,43,65,70,115,45,100,43,65,70,48,65,88,81,32,43,65,67,69,65,80,81,65,57,32,43,65,67,73,45,117,110,100,101,102,105,110,101,100,43,65,67,73,41,32,114,101,116,117,114,110,43,65,68,115,10,32,32,32,32,32,32,32,32,32,32,105,102,40,68,43,65,70,115,45,100,43,65,70,48,46,99,104,97,114,67,111,100,101,65,116,40,48,41,32,43,65,68,48,65,80,81,32,48,120,70,70,70,68,41,32,114,101,116,117,114,110,43,65,68,115,10,32,32,32,32,32,32,32,32,32,32,105,102,40,68,43,65,70,115,45,69,43,65,70,115,45,68,43,65,70,115,45,100,43,65,70,48,65,88,81,66,100,32,43,65,68,48,65,80,81,65,57,32,68,43,65,70,115,45,100,43,65,70,48,41,32,114,101,116,117,114,110,43,65,68,115,10,32,32,32,32,32,32,32,32,32,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,120,32,43,45,32,43,65,67,73,32,101,46,100,43,65,70,115,65,73,103,32,43,45,32,100,32,43,45,32,43,65,67,73,65,88,81,32,43,65,68,48,32,43,65,67,73,32,43,45,32,69,43,65,70,115,45,68,43,65,70,115,45,100,43,65,70,48,65,88,81,32,43,45,32,43,65,67,73,65,79,119,32,100,43,65,70,115,65,73,103,32,43,45,32,100,32,43,45,32,43,65,67,73,65,88,81,65,57,65,67,73,32,43,45,32,68,43,65,70,115,45,100,43,65,70,48,32,43,45,32,43,65,67,73,65,79,119,32,100,46,101,46,100,43,65,70,115,65,73,103,32,43,45,32,100,32,43,45,32,43,65,67,73,65,88,81,32,43,65,68,48,32,43,65,67,73,32,43,45,32,68,43,65,70,115,45,69,43,65,70,115,45,68,43,65,70,115,45,100,43,65,70,48,65,88,81,66,100,41,43,65,68,115,10,32,32,32,32,32,32,32,32,43,65,72,48,10,32,32,32,32,32,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,32,32,79,98,106,101,99,116,46,107,101,121,115,40,69,41,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,101,41,32,43,65,72,115,10,32,32,32,32,32,32,32,32,105,102,40,68,43,65,70,115,45,69,43,65,70,115,45,101,43,65,70,48,65,88,81,32,43,65,67,69,65,80,81,32,101,41,32,43,65,72,115,10,32,32,32,32,32,32,32,32,32,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,120,32,43,45,32,43,65,67,73,32,100,46,101,43,65,70,115,65,73,103,32,43,45,32,101,32,43,45,32,43,65,67,73,65,88,81,32,43,65,68,48,32,43,65,67,73,32,43,45,32,68,43,65,70,115,45,69,43,65,70,115,45,101,43,65,70,48,65,88,81,32,43,45,32,43,65,67,73,65,79,119,32,101,43,65,70,115,65,73,103,32,43,45,32,101,32,43,45,32,43,65,67,73,65,88,81,65,57,65,67,73,32,43,45,32,69,43,65,70,115,45,101,43,65,70,48,32,43,45,32,43,65,67,73,65,79,119,32,101,46,100,46,101,43,65,70,115,65,73,103,32,43,45,32,101,32,43,45,32,43,65,67,73,65,88,81,32,43,65,68,48,32,43,65,67,73,32,43,45,32,69,43,65,70,115,45,68,43,65,70,115,45,69,43,65,70,115,45,101,43,65,70,48,65,88,81,66,100,41,43,65,68,115,10,32,32,32,32,32,32,32,32,43,65,72,48,10,32,32,32,32,32,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,32,32,118,97,114,32,99,111,114,112,117,115,32,43,65,68,48,32,43,65,70,115,65,73,103,45,102,111,111,98,97,114,43,65,67,73,65,88,81,65,55,10,32,32,32,32,32,32,99,111,114,112,117,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,119,41,43,65,72,115,10,32,32,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,85,46,100,101,99,111,100,101,40,120,44,85,46,101,110,99,111,100,101,40,120,44,119,41,41,44,119,41,43,65,68,115,10,32,32,32,32,32,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,43,65,72,48,41,43,65,68,115,10,32,32,43,65,72,48,65,79,119,32,43,65,72,48,65,79,119,10,32,32,100,101,115,99,114,105,98,101,40,39,99,97,99,104,101,100,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,79,98,106,101,99,116,46,107,101,121,115,40,99,112,116,97,98,108,101,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,119,41,32,43,65,72,115,32,114,101,116,117,114,110,32,119,32,43,65,68,48,65,80,81,32,43,45,119,43,65,68,115,32,43,65,72,48,41,46,102,111,114,69,97,99,104,40,99,104,107,40,99,112,116,97,98,108,101,44,32,116,114,117,101,41,41,43,65,68,115,10,32,32,43,65,72,48,41,43,65,68,115,10,32,32,100,101,115,99,114,105,98,101,40,39,100,105,114,101,99,116,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,79,98,106,101,99,116,46,107,101,121,115,40,99,112,116,97,98,108,101,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,119,41,32,43,65,72,115,32,114,101,116,117,114,110,32,119,32,43,65,68,48,65,80,81,32,43,45,119,43,65,68,115,32,43,65,72,48,41,46,102,111,114,69,97,99,104,40,99,104,107,40,99,112,116,97,98,108,101,44,32,102,97,108,115,101,41,41,43,65,68,115,10,32,32,43,65,72,48,41,43,65,68,115,10,43,65,72,48,41,43,65,68,115,10,43,65,71,65,65,89,65,66,103,10,10,84,104,101,32,110,101,120,116,32,116,101,115,116,115,32,108,111,111,107,32,97,116,32,112,111,115,115,105,98,108,101,32,101,110,116,114,121,32,99,111,110,100,105,116,105,111,110,115,58,10,10,43,65,71,65,65,89,65,66,103,10,100,101,115,99,114,105,98,101,40,39,101,110,116,114,121,32,99,111,110,100,105,116,105,111,110,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,116,111,32,108,111,97,100,32,117,116,105,108,115,32,105,102,32,99,112,116,97,98,108,101,32,117,110,97,118,97,105,108,97,98,108,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,118,97,114,32,115,97,110,100,98,111,120,32,43,65,68,48,32,43,65,72,115,65,102,81,65,55,10,32,32,32,32,118,97,114,32,99,116,120,32,43,65,68,48,32,118,109,46,99,114,101,97,116,101,67,111,110,116,101,120,116,40,115,97,110,100,98,111,120,41,43,65,68,115,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,32,32,118,109,46,114,117,110,73,110,67,111,110,116,101,120,116,40,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,99,112,117,116,105,108,115,46,106,115,39,44,39,117,116,102,56,39,41,44,99,116,120,41,43,65,68,115,10,32,32,32,32,43,65,72,48,41,43,65,68,115,10,32,32,43,65,72,48,41,43,65,68,115,10,32,32,105,116,40,39,115,104,111,117,108,100,32,108,111,97,100,32,117,116,105,108,115,32,105,102,32,99,112,116,97,98,108,101,32,105,115,32,97,118,97,105,108,97,98,108,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,118,97,114,32,115,97,110,100,98,111,120,32,43,65,68,48,32,43,65,72,115,65,102,81,65,55,10,32,32,32,32,118,97,114,32,99,116,120,32,43,65,68,48,32,118,109,46,99,114,101,97,116,101,67,111,110,116,101,120,116,40,115,97,110,100,98,111,120,41,43,65,68,115,10,32,32,32,32,118,109,46,114,117,110,73,110,67,111,110,116,101,120,116,40,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,99,112,101,120,99,101,108,46,106,115,39,44,39,117,116,102,56,39,41,44,99,116,120,41,43,65,68,115,10,32,32,32,32,118,109,46,114,117,110,73,110,67,111,110,116,101,120,116,40,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,99,112,117,116,105,108,115,46,106,115,39,44,39,117,116,102,56,39,41,44,99,116,120,41,43,65,68,115,10,32,32,43,65,72,48,41,43,65,68,115,10,32,32,118,97,114,32,99,104,107,101,110,32,43,65,68,48,32,102,117,110,99,116,105,111,110,40,99,112,44,32,105,41,32,43,65,72,115,10,32,32,32,32,118,97,114,32,99,32,43,65,68,48,32,102,117,110,99,116,105,111,110,40,99,112,44,32,105,44,32,101,41,32,43,65,72,115,10,32,32,32,32,32,32,118,97,114,32,115,116,114,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,105,44,101,41,43,65,68,115,10,32,32,32,32,32,32,118,97,114,32,97,114,114,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,105,46,115,112,108,105,116,40,43,65,67,73,65,73,103,41,44,101,41,43,65,68,115,10,32,32,32,32,32,32,97,115,115,101,114,116,46,100,101,101,112,69,113,117,97,108,40,115,116,114,44,97,114,114,41,43,65,68,115,10,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,66,117,102,102,101,114,32,43,65,68,48,65,80,81,65,57,32,39,117,110,100,101,102,105,110,101,100,39,41,32,114,101,116,117,114,110,43,65,68,115,10,32,32,32,32,32,32,118,97,114,32,98,117,102,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,110,101,119,32,66,117,102,102,101,114,40,105,41,44,101,41,43,65,68,115,10,32,32,32,32,32,32,97,115,115,101,114,116,46,100,101,101,112,69,113,117,97,108,40,115,116,114,44,98,117,102,41,43,65,68,115,10,32,32,32,32,43,65,72,48,65,79,119,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,99,40,99,112,44,105,41,43,65,68,115,10,32,32,32,32,99,40,99,112,44,105,44,39,98,117,102,39,41,43,65,68,115,10,32,32,32,32,99,40,99,112,44,105,44,39,97,114,114,39,41,43,65,68,115,10,32,32,32,32,99,40,99,112,44,105,44,39,115,116,114,39,41,43,65,68,115,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,99,40,99,112,44,105,41,43,65,68,115,10,32,32,32,32,99,40,99,112,44,105,44,39,98,117,102,39,41,43,65,68,115,10,32,32,32,32,99,40,99,112,44,105,44,39,97,114,114,39,41,43,65,68,115,10,32,32,32,32,99,40,99,112,44,105,44,39,115,116,114,39,41,43,65,68,115,10,32,32,43,65,72,48,65,79,119,10,32,32,100,101,115,99,114,105,98,101,40,39,101,110,99,111,100,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,105,116,40,39,67,80,32,32,49,50,53,50,32,58,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,32,99,104,107,101,110,40,49,50,53,50,44,43,65,67,73,45,102,111,111,98,97,114,43,65,67,73,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,105,116,40,39,67,80,32,32,32,55,48,56,32,58,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,32,99,104,107,101,110,40,55,48,56,44,43,65,67,73,71,75,103,32,97,110,100,32,43,66,105,115,32,115,109,105,108,101,121,32,102,97,99,101,115,43,65,67,73,41,43,65,68,115,65,102,81,41,43,65,68,115,10,32,32,32,32,105,116,40,39,67,80,32,32,32,57,51,54,32,58,32,100,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,32,99,104,107,101,110,40,57,51,54,44,32,43,65,67,75,80,50,87,89,118,84,105,49,108,104,49,116,88,101,121,90,116,83,52,118,86,65,67,73,41,43,65,68,115,65,102,81,41,43,65,68,115,10,32,32,43,65,72,48,41,43,65,68,115,10,32,32,118,97,114,32,99,104,107,100,101,32,43,65,68,48,32,102,117,110,99,116,105,111,110,40,99,112,44,32,105,41,32,43,65,72,115,10,32,32,32,32,118,97,114,32,99,32,43,65,68,48,32,102,117,110,99,116,105,111,110,40,99,112,44,32,105,41,32,43,65,72,115,10,32,32,32,32,32,32,118,97,114,32,115,43,65,68,115,10,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,66,117,102,102,101,114,32,43,65,67,69,65,80,81,65,57,32,39,117,110,100,101,102,105,110,101,100,39,32,43,65,67,89,65,74,103,32,105,32,105,110,115,116,97,110,99,101,111,102,32,66,117,102,102,101,114,41,32,115,32,43,65,68,48,32,43,65,70,115,65,88,81,46,109,97,112,46,99,97,108,108,40,105,44,32,102,117,110,99,116,105,111,110,40,115,41,43,65,72,115,45,114,101,116,117,114,110,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,115,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,32,32,101,108,115,101,32,115,43,65,68,48,40,105,46,109,97,112,41,32,63,32,105,46,109,97,112,40,102,117,110,99,116,105,111,110,40,115,41,43,65,72,115,45,114,101,116,117,114,110,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,115,41,43,65,68,115,32,43,65,72,48,41,32,58,32,105,43,65,68,115,10,32,32,32,32,32,32,118,97,114,32,115,116,114,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,99,112,44,105,41,43,65,68,115,10,32,32,32,32,32,32,118,97,114,32,97,114,114,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,99,112,44,115,46,106,111,105,110,63,115,46,106,111,105,110,40,43,65,67,73,65,73,103,41,58,115,41,43,65,68,115,10,32,32,32,32,32,32,97,115,115,101,114,116,46,100,101,101,112,69,113,117,97,108,40,115,116,114,44,97,114,114,41,43,65,68,115,10,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,66,117,102,102,101,114,32,43,65,68,48,65,80,81,65,57,32,39,117,110,100,101,102,105,110,101,100,39,41,32,114,101,116,117,114,110,43,65,68,115,10,32,32,32,32,32,32,118,97,114,32,98,117,102,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,99,112,44,110,101,119,32,66,117,102,102,101,114,40,105,41,41,43,65,68,115,10,32,32,32,32,32,32,97,115,115,101,114,116,46,100,101,101,112,69,113,117,97,108,40,115,116,114,44,98,117,102,41,43,65,68,115,10,32,32,32,32,43,65,72,48,65,79,119,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,99,40,99,112,44,105,41,43,65,68,115,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,99,40,99,112,44,105,41,43,65,68,115,10,32,32,43,65,72,48,65,79,119,10,32,32,100,101,115,99,114,105,98,101,40,39,100,101,99,111,100,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,105,116,40,39,67,80,32,32,49,50,53,50,32,58,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,32,99,104,107,100,101,40,49,50,53,50,44,43,65,70,115,45,48,120,54,54,44,32,48,120,54,102,44,32,48,120,54,102,44,32,48,120,54,50,44,32,48,120,54,49,44,32,48,120,55,50,43,65,70,48,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,32,47,43,65,67,111,32,43,65,67,73,45,102,111,111,98,97,114,43,65,67,73,32,43,65,67,111,45,47,10,32,32,32,32,105,102,40,116,121,112,101,111,102,32,66,117,102,102,101,114,32,43,65,67,69,65,80,81,65,57,32,39,117,110,100,101,102,105,110,101,100,39,41,32,105,116,40,39,67,80,32,32,32,55,48,56,32,58,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,32,99,104,107,100,101,40,55,48,56,44,32,110,101,119,32,66,117,102,102,101,114,40,43,65,70,115,45,48,120,99,97,44,32,48,120,50,48,44,32,48,120,54,49,44,32,48,120,54,101,44,32,48,120,54,52,44,32,48,120,50,48,44,32,48,120,99,98,44,32,48,120,50,48,44,32,48,120,55,51,44,32,48,120,54,100,44,32,48,120,54,57,44,32,48,120,54,99,44,32,48,120,54,53,44,32,48,120,55,57,44,32,48,120,50,48,44,32,48,120,54,54,44,32,48,120,54,49,44,32,48,120,54,51,44,32,48,120,54,53,44,32,48,120,55,51,43,65,70,48,41,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,32,47,43,65,67,111,32,40,43,65,67,73,71,75,103,32,97,110,100,32,43,66,105,115,32,115,109,105,108,101,121,32,102,97,99,101,115,43,65,67,73,41,32,43,65,67,111,45,47,10,32,32,32,32,105,116,40,39,67,80,32,32,32,57,51,54,32,58,32,100,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,32,99,104,107,100,101,40,57,51,54,44,32,43,65,70,115,45,48,120,100,53,44,32,48,120,101,50,44,32,48,120,99,97,44,32,48,120,99,55,44,32,48,120,100,54,44,32,48,120,100,48,44,32,48,120,99,101,44,32,48,120,99,52,44,32,48,120,100,55,44,32,48,120,100,54,44,32,48,120,98,55,44,32,48,120,102,98,44,32,48,120,98,50,44,32,48,120,101,50,44,32,48,120,99,97,44,32,48,120,100,52,43,65,70,48,41,43,65,68,115,65,102,81,41,43,65,68,115,32,47,43,65,67,111,32,43,65,67,75,80,50,87,89,118,84,105,49,108,104,49,116,88,101,121,90,116,83,52,118,86,65,67,73,32,43,65,67,111,45,47,10,32,32,43,65,72,48,41,43,65,68,115,10,43,65,72,48,41,43,65,68,115,10,43,65,71,65,65,89,65,66,103,10,10,84,104,101,32,43,65,71,65,45,116,101,115,116,102,105,108,101,43,65,71,65,32,104,101,108,112,101,114,32,102,117,110,99,116,105,111,110,32,114,101,97,100,115,32,97,32,102,105,108,101,32,97,110,100,32,99,111,109,112,97,114,101,115,32,116,111,32,110,111,100,101,39,115,32,114,101,97,100,32,102,97,99,105,108,105,116,105,101,115,58,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,116,101,115,116,46,106,115,10,102,117,110,99,116,105,111,110,32,116,101,115,116,102,105,108,101,40,102,44,99,112,44,116,121,112,101,44,115,107,105,112,41,32,43,65,72,115,10,32,32,118,97,114,32,100,32,43,65,68,48,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,102,41,43,65,68,115,10,32,32,118,97,114,32,120,32,43,65,68,48,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,102,44,32,116,121,112,101,41,43,65,68,115,10,32,32,118,97,114,32,97,32,43,65,68,48,32,120,46,115,112,108,105,116,40,43,65,67,73,65,73,103,41,43,65,68,115,10,32,32,118,97,114,32,99,104,107,32,43,65,68,48,32,102,117,110,99,116,105,111,110,40,99,112,41,32,43,65,72,115,10,32,32,32,32,118,97,114,32,121,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,99,112,44,32,100,41,43,65,68,115,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,120,44,121,41,43,65,68,115,10,32,32,32,32,118,97,114,32,122,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,32,120,41,43,65,68,115,10,32,32,32,32,105,102,40,122,46,108,101,110,103,116,104,32,43,65,67,69,65,80,81,32,100,46,108,101,110,103,116,104,41,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,102,32,43,45,32,43,65,67,73,32,43,65,67,73,32,43,45,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,122,41,32,43,45,32,43,65,67,73,32,43,65,67,69,65,80,81,32,43,65,67,73,32,43,45,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,100,41,32,43,45,32,43,65,67,73,32,58,32,43,65,67,73,32,43,45,32,122,46,108,101,110,103,116,104,32,43,45,32,43,65,67,73,32,43,65,67,73,32,43,45,32,100,46,108,101,110,103,116,104,41,43,65,68,115,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,43,65,68,48,32,48,43,65,68,115,32,105,32,43,65,67,69,65,80,81,32,100,46,108,101,110,103,116,104,43,65,68,115,32,43,45,43,45,105,41,32,105,102,40,100,43,65,70,115,45,105,43,65,70,48,32,43,65,67,69,65,80,81,65,57,32,122,43,65,70,115,45,105,43,65,70,48,41,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,43,65,67,73,65,73,103,32,43,45,32,105,32,43,45,32,43,65,67,73,32,43,65,67,73,32,43,45,32,100,43,65,70,115,45,105,43,65,70,48,32,43,45,32,43,65,67,73,65,73,81,65,57,65,67,73,32,43,45,32,122,43,65,70,115,45,105,43,65,70,48,41,43,65,68,115,10,32,32,32,32,105,102,40,115,107,105,112,41,32,114,101,116,117,114,110,43,65,68,115,10,32,32,32,32,122,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,32,97,41,43,65,68,115,10,32,32,32,32,105,102,40,122,46,108,101,110,103,116,104,32,43,65,67,69,65,80,81,32,100,46,108,101,110,103,116,104,41,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,102,32,43,45,32,43,65,67,73,32,43,65,67,73,32,43,45,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,122,41,32,43,45,32,43,65,67,73,32,43,65,67,69,65,80,81,32,43,65,67,73,32,43,45,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,100,41,32,43,45,32,43,65,67,73,32,58,32,43,65,67,73,32,43,45,32,122,46,108,101,110,103,116,104,32,43,45,32,43,65,67,73,32,43,65,67,73,32,43,45,32,100,46,108,101,110,103,116,104,41,43,65,68,115,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,43,65,68,48,32,48,43,65,68,115,32,105,32,43,65,67,69,65,80,81,32,100,46,108,101,110,103,116,104,43,65,68,115,32,43,45,43,45,105,41,32,105,102,40,100,43,65,70,115,45,105,43,65,70,48,32,43,65,67,69,65,80,81,65,57,32,122,43,65,70,115,45,105,43,65,70,48,41,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,43,65,67,73,65,73,103,32,43,45,32,105,32,43,45,32,43,65,67,73,32,43,65,67,73,32,43,45,32,100,43,65,70,115,45,105,43,65,70,48,32,43,45,32,43,65,67,73,65,73,81,65,57,65,67,73,32,43,45,32,122,43,65,70,115,45,105,43,65,70,48,41,43,65,68,115,10,32,32,43,65,72,48,10,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,43,65,68,115,10,32,32,99,104,107,40,99,112,41,43,65,68,115,10,32,32,105,102,40,115,107,105,112,41,32,114,101,116,117,114,110,43,65,68,115,10,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,43,65,68,115,10,32,32,99,104,107,40,99,112,41,43,65,68,115,10,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,43,65,68,115,10,43,65,72,48,10,43,65,71,65,65,89,65,66,103,10,10,84,104,101,32,43,65,71,65,45,117,116,102,56,43,65,71,65,32,116,101,115,116,115,32,118,101,114,105,102,121,32,117,116,102,56,32,101,110,99,111,100,105,110,103,32,111,102,32,116,104,101,32,97,99,116,117,97,108,32,74,83,32,115,111,117,114,99,101,115,58,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,116,101,115,116,46,106,115,10,100,101,115,99,114,105,98,101,40,39,110,111,100,101,32,110,97,116,105,118,101,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,118,97,114,32,110,111,100,101,32,43,65,68,48,32,43,65,70,115,65,87,119,45,54,53,48,48,49,44,32,39,117,116,102,56,39,44,49,43,65,70,48,44,32,43,65,70,115,45,49,50,48,48,44,32,39,117,116,102,49,54,108,101,39,44,49,43,65,70,48,44,32,43,65,70,115,45,50,48,49,50,55,44,32,39,97,115,99,105,105,39,44,48,43,65,70,48,65,88,81,65,55,10,32,32,118,97,114,32,117,110,105,99,111,100,101,102,105,108,101,115,32,43,65,68,48,32,43,65,70,115,39,99,111,100,101,112,97,103,101,46,109,100,39,44,39,82,69,65,68,77,69,46,109,100,39,44,39,99,112,116,97,98,108,101,46,106,115,39,43,65,70,48,65,79,119,10,32,32,118,97,114,32,97,115,99,105,105,102,105,108,101,115,32,43,65,68,48,32,43,65,70,115,39,99,112,117,116,105,108,115,46,106,115,39,43,65,70,48,65,79,119,10,32,32,110,111,100,101,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,119,41,32,43,65,72,115,10,32,32,32,32,100,101,115,99,114,105,98,101,40,119,43,65,70,115,45,49,43,65,70,48,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,32,32,99,112,116,97,98,108,101,32,43,65,68,48,32,114,101,113,117,105,114,101,40,39,46,47,39,41,43,65,68,115,10,32,32,32,32,32,32,97,115,99,105,105,102,105,108,101,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,102,41,32,43,65,72,115,10,32,32,32,32,32,32,32,32,105,116,40,39,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,39,32,43,45,32,102,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,32,116,101,115,116,102,105,108,101,40,39,46,47,109,105,115,99,47,39,43,45,102,43,45,39,46,39,43,45,119,43,65,70,115,45,49,43,65,70,48,44,119,43,65,70,115,45,48,43,65,70,48,44,119,43,65,70,115,45,49,43,65,70,48,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,32,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,32,32,105,102,40,43,65,67,69,45,119,43,65,70,115,45,50,43,65,70,48,41,32,114,101,116,117,114,110,43,65,68,115,10,32,32,32,32,32,32,117,110,105,99,111,100,101,102,105,108,101,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,102,41,32,43,65,72,115,10,32,32,32,32,32,32,32,32,105,116,40,39,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,39,32,43,45,32,102,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,32,116,101,115,116,102,105,108,101,40,39,46,47,109,105,115,99,47,39,43,45,102,43,45,39,46,39,43,45,119,43,65,70,115,45,49,43,65,70,48,44,119,43,65,70,115,45,48,43,65,70,48,44,119,43,65,70,115,45,49,43,65,70,48,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,32,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,32,32,105,102,40,119,43,65,70,115,45,49,43,65,70,48,32,43,65,68,48,65,80,81,65,57,32,39,117,116,102,56,39,41,32,105,116,40,39,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,98,105,116,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,32,32,32,32,118,97,114,32,102,105,108,101,115,32,43,65,68,48,32,102,115,46,114,101,97,100,100,105,114,83,121,110,99,40,39,98,105,116,115,39,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,120,41,43,65,72,115,45,114,101,116,117,114,110,32,120,46,115,117,98,115,116,114,40,45,51,41,43,65,68,48,65,80,81,65,105,46,106,115,43,65,67,73,65,79,119,66,57,41,43,65,68,115,10,32,32,32,32,32,32,32,32,102,105,108,101,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,102,41,32,43,65,72,115,32,116,101,115,116,102,105,108,101,40,39,46,47,98,105,116,115,47,39,32,43,45,32,102,44,119,43,65,70,115,45,48,43,65,70,48,44,119,43,65,70,115,45,49,43,65,70,48,44,116,114,117,101,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,32,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,43,65,72,48,41,43,65,68,115,10,32,32,43,65,72,48,41,43,65,68,115,10,43,65,72,48,41,43,65,68,115,10,43,65,71,65,65,89,65,66,103,10,10,84,104,101,32,117,116,102,43,65,67,111,32,97,110,100,32,97,115,99,105,105,32,116,101,115,116,115,32,97,116,116,101,109,112,116,32,116,111,32,116,101,115,116,32,111,116,104,101,114,32,109,97,103,105,99,32,102,111,114,109,97,116,115,58,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,116,101,115,116,46,106,115,10,118,97,114,32,109,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,109,97,103,105,99,43,65,68,115,10,102,117,110,99,116,105,111,110,32,99,109,112,40,120,44,122,41,32,43,65,72,115,10,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,120,46,108,101,110,103,116,104,44,32,122,46,108,101,110,103,116,104,41,43,65,68,115,10,32,32,102,111,114,40,118,97,114,32,105,32,43,65,68,48,32,48,43,65,68,115,32,105,32,43,65,67,69,65,80,81,32,122,46,108,101,110,103,116,104,43,65,68,115,32,43,45,43,45,105,41,32,97,115,115,101,114,116,46,101,113,117,97,108,40,105,43,45,43,65,67,73,45,47,43,65,67,73,65,75,119,45,120,46,108,101,110,103,116,104,43,45,43,65,67,73,65,73,103,65,114,45,120,43,65,70,115,45,105,43,65,70,48,44,32,105,43,45,43,65,67,73,45,47,43,65,67,73,65,75,119,45,122,46,108,101,110,103,116,104,43,45,43,65,67,73,65,73,103,65,114,45,122,43,65,70,115,45,105,43,65,70,48,41,43,65,68,115,10,43,65,72,48,10,79,98,106,101,99,116,46,107,101,121,115,40,109,41,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,116,41,43,65,72,115,45,105,102,40,116,32,43,65,67,69,65,80,81,32,49,54,57,54,57,41,32,100,101,115,99,114,105,98,101,40,109,43,65,70,115,45,116,43,65,70,48,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,105,116,40,43,65,67,73,45,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,99,111,100,101,112,97,103,101,46,109,100,46,43,65,67,73,32,43,45,32,109,43,65,70,115,45,116,43,65,70,48,44,32,102,115,46,101,120,105,115,116,115,83,121,110,99,40,39,46,47,109,105,115,99,47,99,111,100,101,112,97,103,101,46,109,100,46,39,32,43,45,32,109,43,65,70,115,45,116,43,65,70,48,41,32,63,10,32,32,32,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,32,32,118,97,114,32,98,32,43,65,68,48,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,109,105,115,99,47,99,111,100,101,112,97,103,101,46,109,100,46,117,116,102,56,39,44,32,43,65,67,73,45,117,116,102,56,43,65,67,73,41,43,65,68,115,10,32,32,32,32,32,32,105,102,40,109,43,65,70,115,45,116,43,65,70,48,32,43,65,68,48,65,80,81,65,57,32,43,65,67,73,45,97,115,99,105,105,43,65,67,73,41,32,98,32,43,65,68,48,32,98,46,114,101,112,108,97,99,101,40,47,43,65,70,115,65,88,65,45,117,48,48,56,48,45,43,65,70,119,45,117,102,102,102,102,43,65,70,48,65,75,103,45,47,103,44,43,65,67,73,65,73,103,41,43,65,68,115,10,32,32,32,32,32,32,118,97,114,32,120,32,43,65,68,48,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,109,105,115,99,47,99,111,100,101,112,97,103,101,46,109,100,46,39,32,43,45,32,109,43,65,70,115,45,116,43,65,70,48,41,43,65,68,115,10,32,32,32,32,32,32,118,97,114,32,121,44,32,122,43,65,68,115,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,32,32,121,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,120,41,43,65,68,115,10,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,98,41,43,65,68,115,10,32,32,32,32,32,32,122,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,116,44,32,121,41,43,65,68,115,10,32,32,32,32,32,32,105,102,40,116,32,43,65,67,69,65,80,81,32,54,53,48,48,48,41,32,99,109,112,40,120,44,122,41,43,65,68,115,10,32,32,32,32,32,32,101,108,115,101,32,43,65,72,115,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,122,41,41,43,65,68,115,32,43,65,72,48,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,32,32,121,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,120,41,43,65,68,115,10,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,98,41,43,65,68,115,10,32,32,32,32,32,32,122,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,116,44,32,121,41,43,65,68,115,10,32,32,32,32,32,32,105,102,40,116,32,43,65,67,69,65,80,81,32,54,53,48,48,48,41,32,99,109,112,40,120,44,122,41,43,65,68,115,10,32,32,32,32,32,32,101,108,115,101,32,43,65,72,115,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,122,41,41,43,65,68,115,32,43,65,72,48,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,43,65,72,48,10,32,32,58,32,110,117,108,108,41,43,65,68,115,10,32,32,105,116,40,43,65,67,73,45,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,82,69,65,68,77,69,46,109,100,46,43,65,67,73,32,43,45,32,109,43,65,70,115,45,116,43,65,70,48,44,32,102,115,46,101,120,105,115,116,115,83,121,110,99,40,39,46,47,109,105,115,99,47,82,69,65,68,77,69,46,109,100,46,39,32,43,45,32,109,43,65,70,115,45,116,43,65,70,48,41,32,63,10,32,32,32,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,32,32,118,97,114,32,98,32,43,65,68,48,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,109,105,115,99,47,82,69,65,68,77,69,46,109,100,46,117,116,102,56,39,44,32,43,65,67,73,45,117,116,102,56,43,65,67,73,41,43,65,68,115,10,32,32,32,32,32,32,105,102,40,109,43,65,70,115,45,116,43,65,70,48,32,43,65,68,48,65,80,81,65,57,32,43,65,67,73,45,97,115,99,105,105,43,65,67,73,41,32,98,32,43,65,68,48,32,98,46,114,101,112,108,97,99,101,40,47,43,65,70,115,65,88,65,45,117,48,48,56,48,45,43,65,70,119,45,117,102,102,102,102,43,65,70,48,65,75,103,45,47,103,44,43,65,67,73,65,73,103,41,43,65,68,115,10,32,32,32,32,32,32,118,97,114,32,120,32,43,65,68,48,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,109,105,115,99,47,82,69,65,68,77,69,46,109,100,46,39,32,43,45,32,109,43,65,70,115,45,116,43,65,70,48,41,43,65,68,115,10,32,32,32,32,32,32,120,32,43,65,68,48,32,43,65,70,115,65,88,81,46,115,108,105,99,101,46,99,97,108,108,40,120,41,43,65,68,115,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,32,32,118,97,114,32,121,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,120,41,43,65,68,115,10,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,98,41,43,65,68,115,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,32,32,118,97,114,32,121,32,43,65,68,48,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,120,41,43,65,68,115,10,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,98,41,43,65,68,115,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,43,65,68,115,10,32,32,32,32,43,65,72,48,10,32,32,58,32,110,117,108,108,41,43,65,68,115,10,43,65,72,48,41,43,65,68,115,65,102,81,41,43,65,68,115,10,43,65,71,65,65,89,65,66,103,10,10,84,104,101,32,99,111,100,101,112,97,103,101,32,43,65,71,65,45,54,57,54,57,43,65,71,65,32,105,115,32,110,111,116,32,100,101,102,105,110,101,100,44,32,115,111,32,111,112,101,114,97,116,105,111,110,115,32,115,104,111,117,108,100,32,102,97,105,108,58,10,10,43,65,71,65,65,89,65,66,103,65,68,52,45,116,101,115,116,46,106,115,10,100,101,115,99,114,105,98,101,40,39,102,97,105,108,117,114,101,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,116,111,32,102,105,110,100,32,67,80,32,54,57,54,57,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,43,65,72,115,45,99,112,116,97,98,108,101,43,65,70,115,45,54,57,54,57,43,65,70,48,46,100,101,99,43,65,72,48,41,43,65,68,115,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,43,65,72,115,45,99,112,116,97,98,108,101,43,65,70,115,45,54,57,54,57,43,65,70,48,46,101,110,99,43,65,72,48,41,43,65,68,115,10,32,32,43,65,72,48,41,43,65,68,115,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,117,115,105,110,103,32,117,116,105,108,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,97,115,115,101,114,116,40,43,65,67,69,45,99,112,116,97,98,108,101,46,117,116,105,108,115,46,104,97,115,99,112,40,54,57,54,57,41,41,43,65,68,115,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,43,65,72,115,45,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,54,57,54,57,44,32,43,65,67,73,45,102,111,111,98,97,114,43,65,67,73,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,43,65,72,115,45,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,54,57,54,57,44,32,43,65,70,115,45,48,120,50,48,43,65,70,48,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,32,32,43,65,72,48,41,43,65,68,115,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,119,105,116,104,32,98,108,97,99,107,32,109,97,103,105,99,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,97,115,115,101,114,116,40,99,112,116,97,98,108,101,46,117,116,105,108,115,46,104,97,115,99,112,40,49,54,57,54,57,41,41,43,65,68,115,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,43,65,72,115,45,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,49,54,57,54,57,44,32,43,65,67,73,45,102,111,111,98,97,114,43,65,67,73,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,43,65,72,115,45,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,49,54,57,54,57,44,32,43,65,70,115,45,48,120,50,48,43,65,70,48,41,43,65,68,115,32,43,65,72,48,41,43,65,68,115,10,32,32,43,65,72,48,41,43,65,68,115,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,119,104,101,110,32,112,114,101,115,101,110,116,101,100,32,119,105,116,104,32,105,110,118,97,108,105,100,32,99,104,97,114,32,99,111,100,101,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,43,65,72,115,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,43,65,72,115,45,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,43,65,68,115,32,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,50,48,49,50,55,44,32,43,65,70,115,45,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,48,120,65,65,41,43,65,70,48,41,43,65,68,115,65,102,81,41,43,65,68,115,10,32,32,43,65,72,48,41,43,65,68,115,10,43,65,72,48,41,43,65,68,115,10,43,65,71,65,65,89,65,66,103,10,10,43,65,67,77,32,78,105,116,116,121,32,71,114,105,116,116,121,10,10,43,65,71,65,65,89,65,66,103,45,106,115,111,110,43,65,68,52,45,112,97,99,107,97,103,101,46,106,115,111,110,10,43,65,72,115,10,32,32,43,65,67,73,45,110,97,109,101,43,65,67,73,58,32,43,65,67,73,45,99,111,100,101,112,97,103,101,43,65,67,73,44,10,32,32,43,65,67,73,45,118,101,114,115,105,111,110,43,65,67,73,58,32,43,65,67,73,45,49,46,51,46,56,43,65,67,73,44,10,32,32,43,65,67,73,45,97,117,116,104,111,114,43,65,67,73,58,32,43,65,67,73,45,83,104,101,101,116,74,83,43,65,67,73,44,10,32,32,43,65,67,73,45,100,101,115,99,114,105,112,116,105,111,110,43,65,67,73,58,32,43,65,67,73,45,112,117,114,101,45,74,83,32,108,105,98,114,97,114,121,32,116,111,32,104,97,110,100,108,101,32,99,111,100,101,112,97,103,101,115,43,65,67,73,44,10,32,32,43,65,67,73,45,107,101,121,119,111,114,100,115,43,65,67,73,58,32,43,65,70,115,32,43,65,67,73,45,99,111,100,101,112,97,103,101,43,65,67,73,44,32,43,65,67,73,45,105,99,111,110,118,43,65,67,73,44,32,43,65,67,73,45,99,111,110,118,101,114,116,43,65,67,73,44,32,43,65,67,73,45,115,116,114,105,110,103,115,43,65,67,73,32,43,65,70,48,44,10,32,32,43,65,67,73,45,98,105,110,43,65,67,73,58,32,43,65,72,115,10,32,32,32,32,43,65,67,73,45,99,111,100,101,112,97,103,101,43,65,67,73,58,32,43,65,67,73,46,47,98,105,110,47,99,111,100,101,112,97,103,101,46,110,106,115,43,65,67,73,10,32,32,43,65,72,48,44,10,32,32,43,65,67,73,45,102,105,108,101,115,43,65,67,73,58,32,43,65,70,115,10,32,32,32,32,43,65,67,73,45,76,73,67,69,78,83,69,43,65,67,73,44,10,32,32,32,32,43,65,67,73,45,82,69,65,68,77,69,46,109,100,43,65,67,73,44,10,32,32,32,32,43,65,67,73,45,98,105,110,43,65,67,73,44,10,32,32,32,32,43,65,67,73,45,99,112,116,97,98,108,101,46,106,115,43,65,67,73,44,10,32,32,32,32,43,65,67,73,45,99,112,117,116,105,108,115,46,106,115,43,65,67,73,44,10,32,32,32,32,43,65,67,73,45,100,105,115,116,47,99,112,101,120,99,101,108,46,102,117,108,108,46,106,115,43,65,67,73,10,32,32,43,65,70,48,44,10,32,32,43,65,67,73,45,109,97,105,110,43,65,67,73,58,32,43,65,67,73,45,99,112,117,116,105,108,115,46,106,115,43,65,67,73,44,10,32,32,43,65,67,73,45,100,101,112,101,110,100,101,110,99,105,101,115,43,65,67,73,58,32,43,65,72,115,10,32,32,32,32,43,65,67,73,45,118,111,99,43,65,67,73,58,43,65,67,73,65,73,103,44,10,32,32,32,32,43,65,67,73,45,99,111,110,99,97,116,45,115,116,114,101,97,109,43,65,67,73,58,43,65,67,73,65,73,103,44,10,32,32,32,32,43,65,67,73,45,99,111,109,109,97,110,100,101,114,43,65,67,73,58,43,65,67,73,65,73,103,10,32,32,43,65,72,48,44,10,32,32,43,65,67,73,45,100,101,118,68,101,112,101,110,100,101,110,99,105,101,115,43,65,67,73,58,32,43,65,72,115,10,32,32,32,32,43,65,67,73,45,109,111,99,104,97,43,65,67,73,58,43,65,67,73,65,73,103,10,32,32,43,65,72,48,44,10,32,32,43,65,67,73,45,114,101,112,111,115,105,116,111,114,121,43,65,67,73,58,32,43,65,72,115,32,43,65,67,73,45,116,121,112,101,43,65,67,73,58,43,65,67,73,45,103,105,116,43,65,67,73,44,32,43,65,67,73,45,117,114,108,43,65,67,73,58,43,65,67,73,45,103,105,116,58,47,47,103,105,116,104,117,98,46,99,111,109,47,83,104,101,101,116,74,83,47,106,115,45,99,111,100,101,112,97,103,101,46,103,105,116,43,65,67,73,65,102,81,44,10,32,32,43,65,67,73,45,115,99,114,105,112,116,115,43,65,67,73,58,32,43,65,72,115,10,32,32,32,32,43,65,67,73,45,112,114,101,116,101,115,116,43,65,67,73,58,32,43,65,67,73,45,103,105,116,32,115,117,98,109,111,100,117,108,101,32,105,110,105,116,32,43,65,67,89,65,74,103,32,103,105,116,32,115,117,98,109,111,100,117,108,101,32,117,112,100,97,116,101,43,65,67,73,44,10,32,32,32,32,43,65,67,73,45,116,101,115,116,43,65,67,73,58,32,43,65,67,73,45,109,97,107,101,32,116,101,115,116,43,65,67,73,44,10,32,32,32,32,43,65,67,73,45,98,117,105,108,100,43,65,67,73,58,32,43,65,67,73,45,109,97,107,101,32,106,115,43,65,67,73,10,32,32,43,65,72,48,44,10,32,32,43,65,67,73,45,99,111,110,102,105,103,43,65,67,73,58,32,43,65,72,115,10,32,32,32,32,43,65,67,73,45,98,108,97,110,107,101,116,43,65,67,73,58,32,43,65,72,115,10,32,32,32,32,32,32,43,65,67,73,45,112,97,116,116,101,114,110,43,65,67,73,58,32,43,65,67,73,65,87,119,45,99,112,116,97,98,108,101,46,106,115,44,99,112,117,116,105,108,115,46,106,115,44,99,112,101,120,99,101,108,46,106,115,43,65,70,48,65,73,103,10,32,32,32,32,43,65,72,48,10,32,32,43,65,72,48,44,10,32,32,43,65,67,73,45,98,117,103,115,43,65,67,73,58,32,43,65,72,115,32,43,65,67,73,45,117,114,108,43,65,67,73,58,32,43,65,67,73,45,104,116,116,112,115,58,47,47,103,105,116,104,117,98,46,99,111,109,47,83,104,101,101,116,74,83,47,106,115,45,99,111,100,101,112,97,103,101,47,105,115,115,117,101,115,43,65,67,73,32,43,65,72,48,44,10,32,32,43,65,67,73,45,108,105,99,101,110,115,101,43,65,67,73,58,32,43,65,67,73,45,65,112,97,99,104,101,45,50,46,48,43,65,67,73,44,10,32,32,43,65,67,73,45,101,110,103,105,110,101,115,43,65,67,73,58,32,43,65,72,115,32,43,65,67,73,45,110,111,100,101,43,65,67,73,58,32,43,65,67,73,65,80,103,65,57,45,48,46,56,43,65,67,73,32,43,65,72,48,10,43,65,72,48,10,43,65,71,65,65,89,65,66,103,10,10,43,65,71,65,65,89,65,66,103,65,68,52,46,118,111,99,114,99,10,43,65,72,115,32,43,65,67,73,45,112,111,115,116,43,65,67,73,58,32,43,65,67,73,45,109,97,107,101,32,106,115,43,65,67,73,32,43,65,72,48,10,43,65,71,65,65,89,65,66,103,10,10,43,65,71,65,65,89,65,66,103,65,68,52,46,103,105,116,105,103,110,111,114,101,10,46,103,105,116,105,103,110,111,114,101,10,99,111,100,101,112,97,103,101,115,47,10,46,118,111,99,114,99,10,110,111,100,101,43,65,70,56,45,109,111,100,117,108,101,115,47,10,109,97,107,101,46,115,104,10,109,97,107,101,46,110,106,115,10,109,105,115,99,47,99,111,118,101,114,97,103,101,46,104,116,109,108,10,43,65,71,65,65,89,65,66,103,10]
;
README['utf8'] = 
[35,32,71,101,116,116,105,110,103,32,67,111,100,101,112,97,103,101,115,10,10,84,104,101,32,102,105,101,108,100,115,32,111,102,32,116,104,101,32,112,97,103,101,115,46,99,115,118,32,109,97,110,105,102,101,115,116,32,97,114,101,32,96,99,111,100,101,112,97,103,101,44,117,114,108,44,98,121,116,101,115,96,32,40,83,66,67,83,61,49,44,32,68,66,67,83,61,50,41,10,10,96,96,96,62,112,97,103,101,115,46,99,115,118,10,51,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,69,66,67,68,73,67,47,67,80,48,51,55,46,84,88,84,44,49,10,52,51,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,52,51,55,46,84,88,84,44,49,10,53,48,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,69,66,67,68,73,67,47,67,80,53,48,48,46,84,88,84,44,49,10,55,51,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,55,51,55,46,84,88,84,44,49,10,55,55,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,55,55,53,46,84,88,84,44,49,10,56,53,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,53,48,46,84,88,84,44,49,10,56,53,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,53,50,46,84,88,84,44,49,10,56,53,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,53,53,46,84,88,84,44,49,10,56,53,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,53,55,46,84,88,84,44,49,10,56,54,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,48,46,84,88,84,44,49,10,56,54,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,49,46,84,88,84,44,49,10,56,54,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,50,46,84,88,84,44,49,10,56,54,51,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,51,46,84,88,84,44,49,10,56,54,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,52,46,84,88,84,44,49,10,56,54,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,53,46,84,88,84,44,49,10,56,54,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,54,46,84,88,84,44,49,10,56,54,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,80,67,47,67,80,56,54,57,46,84,88,84,44,49,10,56,55,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,56,55,52,46,84,88,84,44,49,10,56,55,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,69,66,67,68,73,67,47,67,80,56,55,53,46,84,88,84,44,49,10,57,51,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,57,51,50,46,84,88,84,44,50,10,57,51,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,57,51,54,46,84,88,84,44,50,10,57,52,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,57,52,57,46,84,88,84,44,50,10,57,53,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,57,53,48,46,84,88,84,44,50,10,49,48,50,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,69,66,67,68,73,67,47,67,80,49,48,50,54,46,84,88,84,44,49,10,49,50,53,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,48,46,84,88,84,44,49,10,49,50,53,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,49,46,84,88,84,44,49,10,49,50,53,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,50,46,84,88,84,44,49,10,49,50,53,51,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,51,46,84,88,84,44,49,10,49,50,53,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,52,46,84,88,84,44,49,10,49,50,53,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,53,46,84,88,84,44,49,10,49,50,53,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,54,46,84,88,84,44,49,10,49,50,53,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,55,46,84,88,84,44,49,10,49,50,53,56,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,87,73,78,68,79,87,83,47,67,80,49,50,53,56,46,84,88,84,44,49,10,96,96,96,10,10,78,111,116,101,32,116,104,97,116,32,116,104,101,32,87,105,110,100,111,119,115,32,114,101,110,100,101,114,105,110,103,32,105,115,32,117,115,101,100,32,102,111,114,32,116,104,101,32,77,97,99,32,99,111,100,101,32,112,97,103,101,115,46,32,32,84,104,101,32,112,114,105,109,97,114,121,10,100,105,102,102,101,114,101,110,99,101,32,105,115,32,116,104,101,32,117,115,101,32,111,102,32,116,104,101,32,112,114,105,118,97,116,101,32,96,48,120,70,56,70,70,96,32,99,111,100,101,32,40,119,104,105,99,104,32,114,101,110,100,101,114,115,32,97,115,32,97,110,32,65,112,112,108,101,10,108,111,103,111,32,111,110,32,109,97,99,115,32,98,117,116,32,97,115,32,103,97,114,98,97,103,101,32,111,110,32,111,116,104,101,114,32,111,112,101,114,97,116,105,110,103,32,115,121,115,116,101,109,115,41,46,32,32,73,116,32,109,97,121,32,98,101,32,100,101,115,105,114,97,98,108,101,10,116,111,32,102,97,108,108,32,98,97,99,107,32,116,111,32,116,104,101,32,98,101,104,97,118,105,111,114,44,32,105,110,32,119,104,105,99,104,32,99,97,115,101,32,116,104,101,32,102,105,108,101,115,32,97,114,101,32,117,110,100,101,114,32,65,80,80,76,69,32,97,110,100,32,110,111,116,10,77,73,67,83,70,84,46,32,32,67,111,100,101,112,97,103,101,115,32,97,114,101,32,97,110,32,97,98,115,111,108,117,116,101,32,112,97,105,110,32,58,47,10,10,96,96,96,62,112,97,103,101,115,46,99,115,118,10,49,48,48,48,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,82,79,77,65,78,46,84,88,84,44,49,10,49,48,48,48,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,71,82,69,69,75,46,84,88,84,44,49,10,49,48,48,48,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,67,89,82,73,76,76,73,67,46,84,88,84,44,49,10,49,48,48,50,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,76,65,84,73,78,50,46,84,88,84,44,49,10,49,48,48,55,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,73,67,69,76,65,78,68,46,84,88,84,44,49,10,49,48,48,56,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,86,69,78,68,79,82,83,47,77,73,67,83,70,84,47,77,65,67,47,84,85,82,75,73,83,72,46,84,88,84,44,49,10,96,96,96,10,10,84,104,101,32,110,117,109,98,101,114,105,110,103,32,115,99,104,101,109,101,32,102,111,114,32,116,104,101,32,96,73,83,79,45,56,56,53,57,45,88,96,32,115,101,114,105,101,115,32,105,115,32,96,50,56,53,57,48,32,43,32,88,96,58,10,10,96,96,96,62,112,97,103,101,115,46,99,115,118,10,50,56,53,57,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,46,84,88,84,44,49,10,50,56,53,57,50,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,50,46,84,88,84,44,49,10,50,56,53,57,51,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,51,46,84,88,84,44,49,10,50,56,53,57,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,52,46,84,88,84,44,49,10,50,56,53,57,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,53,46,84,88,84,44,49,10,50,56,53,57,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,54,46,84,88,84,44,49,10,50,56,53,57,55,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,55,46,84,88,84,44,49,10,50,56,53,57,56,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,56,46,84,88,84,44,49,10,50,56,53,57,57,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,57,46,84,88,84,44,49,10,50,56,54,48,48,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,48,46,84,88,84,44,49,10,50,56,54,48,49,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,49,46,84,88,84,44,49,10,50,56,54,48,51,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,51,46,84,88,84,44,49,10,50,56,54,48,52,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,52,46,84,88,84,44,49,10,50,56,54,48,53,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,53,46,84,88,84,44,49,10,50,56,54,48,54,44,104,116,116,112,58,47,47,119,119,119,46,117,110,105,99,111,100,101,46,111,114,103,47,80,117,98,108,105,99,47,77,65,80,80,73,78,71,83,47,73,83,79,56,56,53,57,47,56,56,53,57,45,49,54,46,84,88,84,44,49,10,96,96,96,10,10,35,35,32,71,101,110,101,114,97,116,101,100,32,67,111,100,101,112,97,103,101,115,10,10,84,104,101,32,102,111,108,108,111,119,105,110,103,32,99,111,100,101,112,97,103,101,115,32,97,114,101,32,97,118,97,105,108,97,98,108,101,32,105,110,32,46,78,69,84,32,111,110,32,87,105,110,100,111,119,115,58,10,10,45,32,55,48,56,32,32,32,65,114,97,98,105,99,32,40,65,83,77,79,32,55,48,56,41,10,45,32,55,50,48,32,32,32,65,114,97,98,105,99,32,40,84,114,97,110,115,112,97,114,101,110,116,32,65,83,77,79,41,59,32,65,114,97,98,105,99,32,40,68,79,83,41,10,45,32,56,53,56,32,32,32,79,69,77,32,77,117,108,116,105,108,105,110,103,117,97,108,32,76,97,116,105,110,32,49,32,43,32,69,117,114,111,32,115,121,109,98,111,108,10,45,32,56,55,48,32,32,32,73,66,77,32,69,66,67,68,73,67,32,77,117,108,116,105,108,105,110,103,117,97,108,47,82,79,69,67,69,32,40,76,97,116,105,110,32,50,41,59,32,73,66,77,32,69,66,67,68,73,67,32,77,117,108,116,105,108,105,110,103,117,97,108,32,76,97,116,105,110,32,50,10,45,32,49,48,52,55,32,32,73,66,77,32,69,66,67,68,73,67,32,76,97,116,105,110,32,49,47,79,112,101,110,32,83,121,115,116,101,109,10,45,32,49,49,52,48,32,32,73,66,77,32,69,66,67,68,73,67,32,85,83,45,67,97,110,97,100,97,32,40,48,51,55,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,85,83,45,67,97,110,97,100,97,45,69,117,114,111,41,10,45,32,49,49,52,49,32,32,73,66,77,32,69,66,67,68,73,67,32,71,101,114,109,97,110,121,32,40,50,48,50,55,51,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,71,101,114,109,97,110,121,45,69,117,114,111,41,10,45,32,49,49,52,50,32,32,73,66,77,32,69,66,67,68,73,67,32,68,101,110,109,97,114,107,45,78,111,114,119,97,121,32,40,50,48,50,55,55,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,68,101,110,109,97,114,107,45,78,111,114,119,97,121,45,69,117,114,111,41,10,45,32,49,49,52,51,32,32,73,66,77,32,69,66,67,68,73,67,32,70,105,110,108,97,110,100,45,83,119,101,100,101,110,32,40,50,48,50,55,56,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,70,105,110,108,97,110,100,45,83,119,101,100,101,110,45,69,117,114,111,41,10,45,32,49,49,52,52,32,32,73,66,77,32,69,66,67,68,73,67,32,73,116,97,108,121,32,40,50,48,50,56,48,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,73,116,97,108,121,45,69,117,114,111,41,10,45,32,49,49,52,53,32,32,73,66,77,32,69,66,67,68,73,67,32,76,97,116,105,110,32,65,109,101,114,105,99,97,45,83,112,97,105,110,32,40,50,48,50,56,52,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,83,112,97,105,110,45,69,117,114,111,41,10,45,32,49,49,52,54,32,32,73,66,77,32,69,66,67,68,73,67,32,85,110,105,116,101,100,32,75,105,110,103,100,111,109,32,40,50,48,50,56,53,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,85,75,45,69,117,114,111,41,10,45,32,49,49,52,55,32,32,73,66,77,32,69,66,67,68,73,67,32,70,114,97,110,99,101,32,40,50,48,50,57,55,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,70,114,97,110,99,101,45,69,117,114,111,41,10,45,32,49,49,52,56,32,32,73,66,77,32,69,66,67,68,73,67,32,73,110,116,101,114,110,97,116,105,111,110,97,108,32,40,53,48,48,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,73,110,116,101,114,110,97,116,105,111,110,97,108,45,69,117,114,111,41,10,45,32,49,49,52,57,32,32,73,66,77,32,69,66,67,68,73,67,32,73,99,101,108,97,110,100,105,99,32,40,50,48,56,55,49,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,59,32,73,66,77,32,69,66,67,68,73,67,32,40,73,99,101,108,97,110,100,105,99,45,69,117,114,111,41,10,45,32,49,51,54,49,32,32,75,111,114,101,97,110,32,40,74,111,104,97,98,41,10,45,32,49,48,48,48,49,32,74,97,112,97,110,101,115,101,32,40,77,97,99,41,10,45,32,49,48,48,48,50,32,77,65,67,32,84,114,97,100,105,116,105,111,110,97,108,32,67,104,105,110,101,115,101,32,40,66,105,103,53,41,59,32,67,104,105,110,101,115,101,32,84,114,97,100,105,116,105,111,110,97,108,32,40,77,97,99,41,10,45,32,49,48,48,48,51,32,75,111,114,101,97,110,32,40,77,97,99,41,10,45,32,49,48,48,48,52,32,65,114,97,98,105,99,32,40,77,97,99,41,10,45,32,49,48,48,48,53,32,72,101,98,114,101,119,32,40,77,97,99,41,10,45,32,49,48,48,48,56,32,77,65,67,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,32,40,71,66,32,50,51,49,50,41,59,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,77,97,99,41,10,45,32,49,48,48,49,48,32,82,111,109,97,110,105,97,110,32,40,77,97,99,41,10,45,32,49,48,48,49,55,32,85,107,114,97,105,110,105,97,110,32,40,77,97,99,41,10,45,32,49,48,48,50,49,32,84,104,97,105,32,40,77,97,99,41,10,45,32,49,48,48,56,50,32,67,114,111,97,116,105,97,110,32,40,77,97,99,41,10,45,32,50,48,48,48,48,32,67,78,83,32,84,97,105,119,97,110,59,32,67,104,105,110,101,115,101,32,84,114,97,100,105,116,105,111,110,97,108,32,40,67,78,83,41,10,45,32,50,48,48,48,49,32,84,67,65,32,84,97,105,119,97,110,10,45,32,50,48,48,48,50,32,69,116,101,110,32,84,97,105,119,97,110,59,32,67,104,105,110,101,115,101,32,84,114,97,100,105,116,105,111,110,97,108,32,40,69,116,101,110,41,10,45,32,50,48,48,48,51,32,73,66,77,53,53,53,48,32,84,97,105,119,97,110,10,45,32,50,48,48,48,52,32,84,101,108,101,84,101,120,116,32,84,97,105,119,97,110,10,45,32,50,48,48,48,53,32,87,97,110,103,32,84,97,105,119,97,110,10,45,32,50,48,49,48,53,32,73,65,53,32,40,73,82,86,32,73,110,116,101,114,110,97,116,105,111,110,97,108,32,65,108,112,104,97,98,101,116,32,78,111,46,32,53,44,32,55,45,98,105,116,41,59,32,87,101,115,116,101,114,110,32,69,117,114,111,112,101,97,110,32,40,73,65,53,41,10,45,32,50,48,49,48,54,32,73,65,53,32,71,101,114,109,97,110,32,40,55,45,98,105,116,41,10,45,32,50,48,49,48,55,32,73,65,53,32,83,119,101,100,105,115,104,32,40,55,45,98,105,116,41,10,45,32,50,48,49,48,56,32,73,65,53,32,78,111,114,119,101,103,105,97,110,32,40,55,45,98,105,116,41,10,45,32,50,48,50,54,49,32,84,46,54,49,10,45,32,50,48,50,54,57,32,73,83,79,32,54,57,51,55,32,78,111,110,45,83,112,97,99,105,110,103,32,65,99,99,101,110,116,10,45,32,50,48,50,55,51,32,73,66,77,32,69,66,67,68,73,67,32,71,101,114,109,97,110,121,10,45,32,50,48,50,55,55,32,73,66,77,32,69,66,67,68,73,67,32,68,101,110,109,97,114,107,45,78,111,114,119,97,121,10,45,32,50,48,50,55,56,32,73,66,77,32,69,66,67,68,73,67,32,70,105,110,108,97,110,100,45,83,119,101,100,101,110,10,45,32,50,48,50,56,48,32,73,66,77,32,69,66,67,68,73,67,32,73,116,97,108,121,10,45,32,50,48,50,56,52,32,73,66,77,32,69,66,67,68,73,67,32,76,97,116,105,110,32,65,109,101,114,105,99,97,45,83,112,97,105,110,10,45,32,50,48,50,56,53,32,73,66,77,32,69,66,67,68,73,67,32,85,110,105,116,101,100,32,75,105,110,103,100,111,109,10,45,32,50,48,50,57,48,32,73,66,77,32,69,66,67,68,73,67,32,74,97,112,97,110,101,115,101,32,75,97,116,97,107,97,110,97,32,69,120,116,101,110,100,101,100,10,45,32,50,48,50,57,55,32,73,66,77,32,69,66,67,68,73,67,32,70,114,97,110,99,101,10,45,32,50,48,52,50,48,32,73,66,77,32,69,66,67,68,73,67,32,65,114,97,98,105,99,10,45,32,50,48,52,50,51,32,73,66,77,32,69,66,67,68,73,67,32,71,114,101,101,107,10,45,32,50,48,52,50,52,32,73,66,77,32,69,66,67,68,73,67,32,72,101,98,114,101,119,10,45,32,50,48,56,51,51,32,73,66,77,32,69,66,67,68,73,67,32,75,111,114,101,97,110,32,69,120,116,101,110,100,101,100,10,45,32,50,48,56,51,56,32,73,66,77,32,69,66,67,68,73,67,32,84,104,97,105,10,45,32,50,48,56,54,54,32,82,117,115,115,105,97,110,32,40,75,79,73,56,45,82,41,59,32,67,121,114,105,108,108,105,99,32,40,75,79,73,56,45,82,41,10,45,32,50,48,56,55,49,32,73,66,77,32,69,66,67,68,73,67,32,73,99,101,108,97,110,100,105,99,10,45,32,50,48,56,56,48,32,73,66,77,32,69,66,67,68,73,67,32,67,121,114,105,108,108,105,99,32,82,117,115,115,105,97,110,10,45,32,50,48,57,48,53,32,73,66,77,32,69,66,67,68,73,67,32,84,117,114,107,105,115,104,10,45,32,50,48,57,50,52,32,73,66,77,32,69,66,67,68,73,67,32,76,97,116,105,110,32,49,47,79,112,101,110,32,83,121,115,116,101,109,32,40,49,48,52,55,32,43,32,69,117,114,111,32,115,121,109,98,111,108,41,10,45,32,50,48,57,51,50,32,74,97,112,97,110,101,115,101,32,40,74,73,83,32,48,50,48,56,45,49,57,57,48,32,97,110,100,32,48,50,49,50,45,49,57,57,48,41,10,45,32,50,48,57,51,54,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,32,40,71,66,50,51,49,50,41,59,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,71,66,50,51,49,50,45,56,48,41,10,45,32,50,48,57,52,57,32,75,111,114,101,97,110,32,87,97,110,115,117,110,103,10,45,32,50,49,48,50,53,32,73,66,77,32,69,66,67,68,73,67,32,67,121,114,105,108,108,105,99,32,83,101,114,98,105,97,110,45,66,117,108,103,97,114,105,97,110,10,45,32,50,49,48,50,55,32,69,120,116,101,110,100,101,100,47,69,120,116,32,65,108,112,104,97,32,76,111,119,101,114,99,97,115,101,10,45,32,50,49,56,54,54,32,85,107,114,97,105,110,105,97,110,32,40,75,79,73,56,45,85,41,59,32,67,121,114,105,108,108,105,99,32,40,75,79,73,56,45,85,41,10,45,32,50,57,48,48,49,32,69,117,114,111,112,97,32,51,10,45,32,51,56,53,57,56,32,73,83,79,32,56,56,53,57,45,56,32,72,101,98,114,101,119,59,32,72,101,98,114,101,119,32,40,73,83,79,45,76,111,103,105,99,97,108,41,10,45,32,53,48,50,50,48,32,73,83,79,32,50,48,50,50,32,74,97,112,97,110,101,115,101,32,119,105,116,104,32,110,111,32,104,97,108,102,119,105,100,116,104,32,75,97,116,97,107,97,110,97,59,32,74,97,112,97,110,101,115,101,32,40,74,73,83,41,10,45,32,53,48,50,50,49,32,73,83,79,32,50,48,50,50,32,74,97,112,97,110,101,115,101,32,119,105,116,104,32,104,97,108,102,119,105,100,116,104,32,75,97,116,97,107,97,110,97,59,32,74,97,112,97,110,101,115,101,32,40,74,73,83,45,65,108,108,111,119,32,49,32,98,121,116,101,32,75,97,110,97,41,10,45,32,53,48,50,50,50,32,73,83,79,32,50,48,50,50,32,74,97,112,97,110,101,115,101,32,74,73,83,32,88,32,48,50,48,49,45,49,57,56,57,59,32,74,97,112,97,110,101,115,101,32,40,74,73,83,45,65,108,108,111,119,32,49,32,98,121,116,101,32,75,97,110,97,32,45,32,83,79,47,83,73,41,10,45,32,53,48,50,50,53,32,73,83,79,32,50,48,50,50,32,75,111,114,101,97,110,10,45,32,53,48,50,50,55,32,73,83,79,32,50,48,50,50,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,59,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,73,83,79,32,50,48,50,50,41,10,45,32,53,49,57,51,50,32,69,85,67,32,74,97,112,97,110,101,115,101,10,45,32,53,49,57,51,54,32,69,85,67,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,59,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,69,85,67,41,10,45,32,53,49,57,52,57,32,69,85,67,32,75,111,114,101,97,110,10,45,32,53,50,57,51,54,32,72,90,45,71,66,50,51,49,50,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,59,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,72,90,41,10,45,32,53,52,57,51,54,32,87,105,110,100,111,119,115,32,88,80,32,97,110,100,32,108,97,116,101,114,58,32,71,66,49,56,48,51,48,32,83,105,109,112,108,105,102,105,101,100,32,67,104,105,110,101,115,101,32,40,52,32,98,121,116,101,41,59,32,67,104,105,110,101,115,101,32,83,105,109,112,108,105,102,105,101,100,32,40,71,66,49,56,48,51,48,41,10,45,32,53,55,48,48,50,32,73,83,67,73,73,32,68,101,118,97,110,97,103,97,114,105,10,45,32,53,55,48,48,51,32,73,83,67,73,73,32,66,101,110,103,97,108,105,10,45,32,53,55,48,48,52,32,73,83,67,73,73,32,84,97,109,105,108,10,45,32,53,55,48,48,53,32,73,83,67,73,73,32,84,101,108,117,103,117,10,45,32,53,55,48,48,54,32,73,83,67,73,73,32,65,115,115,97,109,101,115,101,10,45,32,53,55,48,48,55,32,73,83,67,73,73,32,79,114,105,121,97,10,45,32,53,55,48,48,56,32,73,83,67,73,73,32,75,97,110,110,97,100,97,10,45,32,53,55,48,48,57,32,73,83,67,73,73,32,77,97,108,97,121,97,108,97,109,10,45,32,53,55,48,49,48,32,73,83,67,73,73,32,71,117,106,97,114,97,116,105,10,45,32,53,55,48,49,49,32,73,83,67,73,73,32,80,117,110,106,97,98,105,10,10,96,96,96,62,112,97,103,101,115,46,99,115,118,10,55,48,56,44,44,49,10,55,50,48,44,44,49,10,56,53,56,44,44,49,10,56,55,48,44,44,49,10,49,48,52,55,44,44,49,10,49,49,52,48,44,44,49,10,49,49,52,49,44,44,49,10,49,49,52,50,44,44,49,10,49,49,52,51,44,44,49,10,49,49,52,52,44,44,49,10,49,49,52,53,44,44,49,10,49,49,52,54,44,44,49,10,49,49,52,55,44,44,49,10,49,49,52,56,44,44,49,10,49,49,52,57,44,44,49,10,49,51,54,49,44,44,50,10,49,48,48,48,49,44,44,50,10,49,48,48,48,50,44,44,50,10,49,48,48,48,51,44,44,50,10,49,48,48,48,52,44,44,49,10,49,48,48,48,53,44,44,49,10,49,48,48,48,56,44,44,50,10,49,48,48,49,48,44,44,49,10,49,48,48,49,55,44,44,49,10,49,48,48,50,49,44,44,49,10,49,48,48,56,50,44,44,49,10,50,48,48,48,48,44,44,50,10,50,48,48,48,49,44,44,50,10,50,48,48,48,50,44,44,50,10,50,48,48,48,51,44,44,50,10,50,48,48,48,52,44,44,50,10,50,48,48,48,53,44,44,50,10,50,48,49,48,53,44,44,49,10,50,48,49,48,54,44,44,49,10,50,48,49,48,55,44,44,49,10,50,48,49,48,56,44,44,49,10,50,48,50,54,49,44,44,50,10,50,48,50,54,57,44,44,49,10,50,48,50,55,51,44,44,49,10,50,48,50,55,55,44,44,49,10,50,48,50,55,56,44,44,49,10,50,48,50,56,48,44,44,49,10,50,48,50,56,52,44,44,49,10,50,48,50,56,53,44,44,49,10,50,48,50,57,48,44,44,49,10,50,48,50,57,55,44,44,49,10,50,48,52,50,48,44,44,49,10,50,48,52,50,51,44,44,49,10,50,48,52,50,52,44,44,49,10,50,48,56,51,51,44,44,49,10,50,48,56,51,56,44,44,49,10,50,48,56,54,54,44,44,49,10,50,48,56,55,49,44,44,49,10,50,48,56,56,48,44,44,49,10,50,48,57,48,53,44,44,49,10,50,48,57,50,52,44,44,49,10,50,48,57,51,50,44,44,50,10,50,48,57,51,54,44,44,50,10,50,48,57,52,57,44,44,50,10,50,49,48,50,53,44,44,49,10,50,49,48,50,55,44,44,49,10,50,49,56,54,54,44,44,49,10,50,57,48,48,49,44,44,49,10,51,56,53,57,56,44,44,49,10,53,48,50,50,48,44,44,50,10,53,48,50,50,49,44,44,50,10,53,48,50,50,50,44,44,50,10,53,48,50,50,53,44,44,50,10,53,48,50,50,55,44,44,50,10,53,49,57,51,50,44,44,50,10,53,49,57,51,54,44,44,50,10,53,49,57,52,57,44,44,50,10,53,50,57,51,54,44,44,50,10,53,52,57,51,54,44,44,50,10,53,55,48,48,50,44,44,50,10,53,55,48,48,51,44,44,50,10,53,55,48,48,52,44,44,50,10,53,55,48,48,53,44,44,50,10,53,55,48,48,54,44,44,50,10,53,55,48,48,55,44,44,50,10,53,55,48,48,56,44,44,50,10,53,55,48,48,57,44,44,50,10,53,55,48,49,48,44,44,50,10,53,55,48,49,49,44,44,50,10,96,96,96,10,10,84,104,101,32,102,111,108,108,111,119,105,110,103,32,99,111,100,101,112,97,103,101,115,32,97,114,101,32,100,101,112,101,110,100,101,110,99,105,101,115,32,102,111,114,32,86,105,115,117,97,108,32,70,111,120,80,114,111,58,10,10,45,32,54,50,48,32,77,97,122,111,118,105,97,32,40,80,111,108,105,115,104,41,32,77,83,45,68,79,83,10,45,32,56,57,53,32,75,97,109,101,110,105,99,107,195,189,32,40,67,122,101,99,104,41,32,77,83,45,68,79,83,10,10,96,96,96,62,112,97,103,101,115,46,99,115,118,10,54,50,48,44,44,49,10,56,57,53,44,44,49,10,96,96,96,10,10,84,104,101,32,107,110,111,119,110,32,109,105,115,115,105,110,103,32,99,111,100,101,112,97,103,101,115,32,97,114,101,32,101,110,117,109,101,114,97,116,101,100,32,105,110,32,116,104,101,32,82,69,65,68,77,69,46,10,10,35,35,32,66,117,105,108,100,105,110,103,32,78,111,116,101,115,10,10,84,104,101,32,115,99,114,105,112,116,32,96,109,97,107,101,46,115,104,96,32,40,100,101,115,99,114,105,98,101,100,32,108,97,116,101,114,41,32,119,105,108,108,32,103,101,116,32,116,104,101,115,101,32,102,105,108,101,115,32,97,110,100,32,109,97,115,115,97,103,101,32,116,104,101,32,100,97,116,97,10,40,112,114,105,110,116,105,110,103,32,99,111,100,101,45,117,110,105,99,111,100,101,32,112,97,105,114,115,41,46,32,32,84,104,101,32,101,118,101,110,116,117,97,108,32,116,97,98,108,101,115,32,97,114,101,32,100,114,111,112,112,101,100,32,105,110,32,116,104,101,32,112,97,116,104,115,10,96,46,47,99,111,100,101,112,97,103,101,115,47,60,67,79,68,69,80,65,71,69,62,46,84,66,76,96,46,32,32,70,111,114,32,101,120,97,109,112,108,101,44,32,116,104,101,32,108,97,115,116,32,49,48,32,108,105,110,101,115,32,111,102,32,96,49,48,48,48,48,46,84,66,76,96,32,97,114,101,10,10,96,96,96,62,10,48,120,70,54,9,48,120,48,50,67,54,10,48,120,70,55,9,48,120,48,50,68,67,10,48,120,70,56,9,48,120,48,48,65,70,10,48,120,70,57,9,48,120,48,50,68,56,10,48,120,70,65,9,48,120,48,50,68,57,10,48,120,70,66,9,48,120,48,50,68,65,10,48,120,70,67,9,48,120,48,48,66,56,10,48,120,70,68,9,48,120,48,50,68,68,10,48,120,70,69,9,48,120,48,50,68,66,10,48,120,70,70,9,48,120,48,50,67,55,10,96,96,96,10,10,119,104,105,99,104,32,105,109,112,108,105,101,115,32,116,104,97,116,32,99,111,100,101,32,48,120,70,54,32,105,115,32,96,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,48,120,48,50,67,54,41,96,32,97,110,100,32,118,105,99,101,32,118,101,114,115,97,46,10,10,35,35,32,87,105,110,100,111,119,115,45,100,101,112,101,110,100,101,110,116,32,98,117,105,108,100,32,115,116,101,112,10,10,84,111,32,98,117,105,108,100,32,116,104,101,32,115,111,117,114,99,101,115,32,111,110,32,119,105,110,100,111,119,115,44,32,99,111,110,115,117,108,116,32,96,100,111,116,110,101,116,47,77,97,107,101,69,110,99,111,100,105,110,103,46,99,115,96,46,10,10,65,102,116,101,114,32,115,97,118,105,110,103,32,116,104,101,32,115,116,97,110,100,97,114,100,32,111,117,116,112,117,116,32,116,111,32,96,111,117,116,96,44,32,97,32,115,105,109,112,108,101,32,97,119,107,32,115,99,114,105,112,116,32,40,96,100,111,116,110,101,116,46,115,104,96,41,32,116,97,107,101,115,32,99,97,114,101,32,111,102,32,116,104,101,32,114,101,115,116,58,10,10,96,96,96,62,100,111,116,110,101,116,46,115,104,10,35,33,47,98,105,110,47,98,97,115,104,10,105,102,32,91,32,33,32,45,101,32,100,111,116,110,101,116,47,111,117,116,32,93,59,32,116,104,101,110,32,101,120,105,116,59,32,102,105,10,60,100,111,116,110,101,116,47,111,117,116,32,116,114,32,45,115,32,39,32,39,32,39,92,116,39,32,124,32,97,119,107,32,39,78,70,62,50,32,123,105,102,40,111,117,116,102,105,108,101,41,32,99,108,111,115,101,40,111,117,116,102,105,108,101,41,59,32,111,117,116,102,105,108,101,61,34,99,111,100,101,112,97,103,101,115,47,34,32,36,49,32,34,46,84,66,76,34,125,32,78,70,61,61,50,32,123,112,114,105,110,116,32,62,32,111,117,116,102,105,108,101,125,39,10,96,96,96,10,10,35,32,66,117,105,108,100,105,110,103,32,116,104,101,32,115,99,114,105,112,116,10,10,96,109,97,107,101,46,110,106,115,96,32,116,97,107,101,115,32,97,32,99,111,100,101,112,97,103,101,32,97,114,103,117,109,101,110,116,44,32,114,101,97,100,115,32,116,104,101,32,99,111,114,114,101,115,112,111,110,100,105,110,103,32,116,97,98,108,101,32,102,105,108,101,32,97,110,100,10,103,101,110,101,114,97,116,101,115,32,74,83,32,99,111,100,101,32,102,111,114,32,101,110,99,111,100,105,110,103,32,97,110,100,32,100,101,99,111,100,105,110,103,58,10,10,35,35,32,82,97,119,32,67,111,100,101,112,97,103,101,115,10,10,96,96,96,62,109,97,107,101,46,110,106,115,10,35,33,47,117,115,114,47,98,105,110,47,101,110,118,32,110,111,100,101,10,118,97,114,32,97,114,103,118,32,61,32,112,114,111,99,101,115,115,46,97,114,103,118,46,115,108,105,99,101,40,49,41,44,32,102,115,32,61,32,114,101,113,117,105,114,101,40,39,102,115,39,41,59,10,105,102,40,97,114,103,118,46,108,101,110,103,116,104,32,60,32,50,41,32,123,10,32,32,32,32,99,111,110,115,111,108,101,46,101,114,114,111,114,40,34,117,115,97,103,101,58,32,109,97,107,101,46,110,106,115,32,60,99,111,100,101,112,97,103,101,95,105,110,100,101,120,62,32,91,118,97,114,105,97,98,108,101,93,34,41,59,10,32,32,32,32,112,114,111,99,101,115,115,46,101,120,105,116,40,50,50,41,59,32,47,42,32,69,73,78,86,65,76,32,42,47,10,125,10,10,118,97,114,32,99,112,32,61,32,97,114,103,118,91,49,93,59,10,118,97,114,32,106,115,118,97,114,32,61,32,97,114,103,118,91,50,93,32,124,124,32,34,99,112,116,97,98,108,101,34,59,10,118,97,114,32,120,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,34,99,111,100,101,112,97,103,101,115,47,34,32,43,32,99,112,32,43,32,34,46,84,66,76,34,44,34,117,116,102,56,34,41,59,10,118,97,114,32,109,97,120,99,112,32,61,32,48,59,10,10,118,97,114,32,121,32,61,32,120,46,115,112,108,105,116,40,34,92,110,34,41,46,109,97,112,40,102,117,110,99,116,105,111,110,40,122,41,32,123,10,32,32,32,32,118,97,114,32,119,32,61,32,122,46,115,112,108,105,116,40,34,92,116,34,41,59,10,32,32,32,32,105,102,40,119,46,108,101,110,103,116,104,32,60,32,50,41,32,114,101,116,117,114,110,32,119,59,10,32,32,32,32,114,101,116,117,114,110,32,91,78,117,109,98,101,114,40,119,91,48,93,41,44,32,78,117,109,98,101,114,40,119,91,49,93,41,93,59,10,125,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,122,41,32,123,32,114,101,116,117,114,110,32,122,46,108,101,110,103,116,104,32,62,32,49,59,32,125,41,59,10,96,96,96,10,10,84,104,101,32,68,66,67,83,32,97,110,100,32,83,66,67,83,32,99,111,100,101,32,103,101,110,101,114,97,116,105,111,110,32,115,116,114,97,116,101,103,105,101,115,32,97,114,101,32,100,105,102,102,101,114,101,110,116,46,32,32,84,104,101,32,109,97,120,105,109,117,109,32,99,111,100,101,32,105,115,10,117,115,101,100,32,116,111,32,100,105,115,116,105,110,103,117,105,115,104,32,40,109,97,120,32,48,120,70,70,32,102,111,114,32,83,66,67,83,41,46,10,10,96,96,96,10,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,121,46,108,101,110,103,116,104,59,32,43,43,105,41,32,105,102,40,121,91,105,93,91,48,93,32,62,32,109,97,120,99,112,41,32,109,97,120,99,112,32,61,32,121,91,105,93,91,48,93,59,10,10,118,97,114,32,101,110,99,32,61,32,123,125,44,32,100,101,99,32,61,32,40,109,97,120,99,112,32,60,32,50,53,54,32,63,32,91,93,32,58,32,123,125,41,59,10,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,121,46,108,101,110,103,116,104,59,32,43,43,105,41,32,123,10,32,32,32,32,100,101,99,91,121,91,105,93,91,48,93,93,32,61,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,121,91,105,93,91,49,93,41,59,10,32,32,32,32,101,110,99,91,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,121,91,105,93,91,49,93,41,93,32,61,32,121,91,105,93,91,48,93,59,10,125,10,10,118,97,114,32,111,100,101,99,44,32,111,101,110,99,44,32,111,117,116,115,116,114,59,10,105,102,40,109,97,120,99,112,32,60,32,50,53,54,41,32,123,10,96,96,96,10,10,84,104,101,32,117,110,105,99,111,100,101,32,99,104,97,114,97,99,116,101,114,32,96,48,120,70,70,70,68,96,32,40,82,69,80,76,65,67,69,77,69,78,84,32,67,72,65,82,65,67,84,69,82,41,32,105,115,32,117,115,101,100,32,97,115,32,97,32,112,108,97,99,101,104,111,108,100,101,114,10,102,111,114,32,99,104,97,114,97,99,116,101,114,115,32,116,104,97,116,32,97,114,101,32,110,111,116,32,115,112,101,99,105,102,105,101,100,32,105,110,32,116,104,101,32,109,97,112,32,40,102,111,114,32,101,120,97,109,112,108,101,44,32,96,48,120,70,48,96,32,105,115,32,110,111,116,32,105,110,10,99,111,100,101,32,112,97,103,101,32,49,48,48,48,48,41,46,10,10,70,111,114,32,83,66,67,83,44,32,116,104,101,32,105,100,101,97,32,105,115,32,116,111,32,101,109,98,101,100,32,97,32,114,97,119,32,115,116,114,105,110,103,32,119,105,116,104,32,116,104,101,32,99,111,110,116,101,110,116,115,32,111,102,32,116,104,101,32,50,53,54,32,99,111,100,101,115,46,10,84,104,101,32,96,100,101,99,96,32,102,105,101,108,100,32,105,115,32,109,101,114,101,108,121,32,97,32,115,112,108,105,116,32,111,102,32,116,104,101,32,115,116,114,105,110,103,44,32,97,110,100,32,96,101,110,99,96,32,105,115,32,97,110,32,101,118,101,114,115,105,111,110,58,10,10,96,96,96,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,50,53,54,59,32,43,43,105,41,32,105,102,40,116,121,112,101,111,102,32,100,101,99,91,105,93,32,61,61,61,32,34,117,110,100,101,102,105,110,101,100,34,41,32,100,101,99,91,105,93,32,61,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,48,120,70,70,70,68,41,59,10,32,32,32,32,111,100,101,99,32,61,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,100,101,99,46,106,111,105,110,40,34,34,41,41,59,10,32,32,32,32,111,117,116,115,116,114,32,61,32,39,40,102,117,110,99,116,105,111,110,40,41,123,32,118,97,114,32,100,32,61,32,39,32,43,32,111,100,101,99,32,43,32,39,44,32,68,32,61,32,91,93,44,32,101,32,61,32,123,125,59,32,102,111,114,40,118,97,114,32,105,61,48,59,105,33,61,100,46,108,101,110,103,116,104,59,43,43,105,41,32,123,32,105,102,40,100,46,99,104,97,114,67,111,100,101,65,116,40,105,41,32,33,61,61,32,48,120,70,70,70,68,41,32,101,91,100,91,105,93,93,32,61,32,105,59,32,68,91,105,93,32,61,32,100,46,99,104,97,114,65,116,40,105,41,59,32,125,32,114,101,116,117,114,110,32,123,34,101,110,99,34,58,32,101,44,32,34,100,101,99,34,58,32,68,32,125,59,32,125,41,40,41,59,39,59,10,125,32,101,108,115,101,32,123,10,96,96,96,10,10,68,66,67,83,32,105,115,32,115,105,109,105,108,97,114,44,32,101,120,99,101,112,116,32,116,104,97,116,32,116,104,101,32,115,112,97,99,101,32,105,115,32,115,108,105,99,101,100,32,105,110,116,111,32,50,53,54,45,98,121,116,101,32,99,104,117,110,107,115,32,40,115,116,114,105,110,103,115,10,97,114,101,32,111,110,108,121,32,103,101,110,101,114,97,116,101,100,32,102,111,114,32,116,104,111,115,101,32,104,105,103,104,45,98,121,116,101,115,32,114,101,112,114,101,115,101,110,116,101,100,32,105,110,32,116,104,101,32,99,111,100,101,112,97,103,101,41,46,10,10,84,104,101,32,115,116,114,97,116,101,103,121,32,105,115,32,116,111,32,99,111,110,115,116,114,117,99,116,32,97,110,32,97,114,114,97,121,45,111,102,45,97,114,114,97,121,115,32,115,111,32,116,104,97,116,32,96,100,100,91,104,105,103,104,93,91,108,111,119,93,96,32,105,115,32,116,104,101,10,99,104,97,114,97,99,116,101,114,32,97,115,115,111,99,105,97,116,101,100,32,119,105,116,104,32,116,104,101,32,99,111,100,101,46,32,32,84,104,105,115,32,97,114,114,97,121,32,105,115,32,99,111,109,98,105,110,101,100,32,97,116,32,114,117,110,116,105,109,101,32,116,111,32,121,105,101,108,100,10,116,104,101,32,99,111,109,112,108,101,116,101,32,100,101,99,111,100,105,110,103,32,111,98,106,101,99,116,32,40,97,110,100,32,116,104,101,32,101,110,99,111,100,105,110,103,32,111,98,106,101,99,116,32,105,115,32,97,110,32,101,118,101,114,115,105,111,110,41,58,10,10,96,96,96,10,32,32,32,32,118,97,114,32,100,100,32,61,32,91,93,59,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,105,110,32,100,101,99,41,32,105,102,40,100,101,99,46,104,97,115,79,119,110,80,114,111,112,101,114,116,121,40,105,41,41,32,123,10,32,32,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,100,100,91,105,32,62,62,32,56,93,32,61,61,61,32,34,117,110,100,101,102,105,110,101,100,34,41,32,100,100,91,105,32,62,62,32,56,93,32,61,32,91,93,59,10,32,32,32,32,32,32,32,32,100,100,91,105,32,62,62,32,56,93,91,105,32,37,32,50,53,54,93,32,61,32,100,101,99,91,105,93,59,10,32,32,32,32,125,10,32,32,32,32,111,117,116,115,116,114,32,61,32,39,40,102,117,110,99,116,105,111,110,40,41,123,32,118,97,114,32,100,32,61,32,91,93,44,32,101,32,61,32,123,125,44,32,68,32,61,32,91,93,44,32,106,59,92,110,39,59,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,50,53,54,59,32,43,43,105,41,32,105,102,40,100,100,91,105,93,41,32,123,10,32,32,32,32,32,32,32,32,102,111,114,40,118,97,114,32,106,32,61,32,48,59,32,106,32,33,61,32,50,53,54,59,32,43,43,106,41,32,105,102,40,116,121,112,101,111,102,32,100,100,91,105,93,91,106,93,32,61,61,61,32,34,117,110,100,101,102,105,110,101,100,34,41,32,100,100,91,105,93,91,106,93,32,61,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,48,120,70,70,70,68,41,59,10,32,32,32,32,32,32,32,32,111,117,116,115,116,114,32,43,61,32,39,68,91,39,32,43,32,105,32,43,32,39,93,32,61,32,39,32,43,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,100,100,91,105,93,46,106,111,105,110,40,34,34,41,41,32,43,32,39,46,115,112,108,105,116,40,34,34,41,59,92,110,39,59,10,32,32,32,32,32,32,32,32,111,117,116,115,116,114,32,43,61,32,39,102,111,114,40,106,32,61,32,48,59,32,106,32,33,61,32,68,91,39,32,43,32,105,32,43,32,39,93,46,108,101,110,103,116,104,59,32,43,43,106,41,32,105,102,40,68,91,39,32,43,32,105,32,43,32,39,93,91,106,93,46,99,104,97,114,67,111,100,101,65,116,40,48,41,32,33,61,61,32,48,120,70,70,70,68,41,32,123,32,101,91,68,91,39,32,43,32,105,32,43,32,39,93,91,106,93,93,32,61,32,39,32,43,32,40,105,42,50,53,54,41,32,43,32,39,32,43,32,106,59,32,100,91,39,32,43,32,40,105,42,50,53,54,41,32,43,32,39,32,43,32,106,93,32,61,32,68,91,39,32,43,32,105,32,43,32,39,93,91,106,93,59,125,92,110,39,10,32,32,32,32,125,10,32,32,32,32,111,117,116,115,116,114,32,43,61,32,39,114,101,116,117,114,110,32,123,34,101,110,99,34,58,32,101,44,32,34,100,101,99,34,58,32,100,32,125,59,32,125,41,40,41,59,39,59,10,125,10,99,111,110,115,111,108,101,46,108,111,103,40,106,115,118,97,114,32,43,32,34,91,34,32,43,32,99,112,32,43,32,34,93,32,61,32,34,32,43,32,111,117,116,115,116,114,41,59,10,10,96,96,96,10,10,96,109,97,107,101,46,115,104,96,32,103,101,110,101,114,97,116,101,115,32,116,104,101,32,116,97,98,108,101,115,32,117,115,101,100,32,98,121,32,96,109,97,107,101,46,110,106,115,96,46,32,32,84,104,101,32,114,97,119,32,117,110,105,99,111,100,101,32,84,88,84,32,102,105,108,101,115,10,97,114,101,32,99,111,108,117,109,110,97,114,58,32,96,99,111,100,101,32,117,110,105,99,111,100,101,32,35,99,111,109,109,101,110,116,115,96,46,32,32,70,111,114,32,101,120,97,109,112,108,101,44,32,116,104,101,32,108,97,115,116,32,49,48,32,108,105,110,101,115,32,111,102,32,116,104,101,10,116,101,120,116,32,102,105,108,101,32,82,79,77,65,78,46,84,88,84,32,40,102,111,114,32,67,80,32,49,48,48,48,48,41,32,97,114,101,58,10,10,96,96,96,62,10,48,120,70,54,9,48,120,48,50,67,54,9,35,77,79,68,73,70,73,69,82,32,76,69,84,84,69,82,32,67,73,82,67,85,77,70,76,69,88,32,65,67,67,69,78,84,10,48,120,70,55,9,48,120,48,50,68,67,9,35,83,77,65,76,76,32,84,73,76,68,69,10,48,120,70,56,9,48,120,48,48,65,70,9,35,77,65,67,82,79,78,10,48,120,70,57,9,48,120,48,50,68,56,9,35,66,82,69,86,69,10,48,120,70,65,9,48,120,48,50,68,57,9,35,68,79,84,32,65,66,79,86,69,10,48,120,70,66,9,48,120,48,50,68,65,9,35,82,73,78,71,32,65,66,79,86,69,10,48,120,70,67,9,48,120,48,48,66,56,9,35,67,69,68,73,76,76,65,10,48,120,70,68,9,48,120,48,50,68,68,9,35,68,79,85,66,76,69,32,65,67,85,84,69,32,65,67,67,69,78,84,10,48,120,70,69,9,48,120,48,50,68,66,9,35,79,71,79,78,69,75,10,48,120,70,70,9,48,120,48,50,67,55,9,35,67,65,82,79,78,10,96,96,96,10,10,73,110,32,112,114,111,99,101,115,115,105,110,103,32,116,104,101,32,100,97,116,97,44,32,116,104,101,32,99,111,109,109,101,110,116,115,32,40,97,102,116,101,114,32,116,104,101,32,96,35,96,41,32,97,114,101,32,115,116,114,105,112,112,101,100,32,97,110,100,32,117,110,100,101,102,105,110,101,100,10,101,108,101,109,101,110,116,115,32,40,108,105,107,101,32,96,48,120,55,70,96,32,102,111,114,32,67,80,32,49,48,48,48,48,41,32,97,114,101,32,114,101,109,111,118,101,100,46,10,10,96,96,96,62,109,97,107,101,46,115,104,10,35,33,47,98,105,110,47,98,97,115,104,10,73,78,70,73,76,69,61,36,123,49,58,45,112,97,103,101,115,46,99,115,118,125,10,79,85,84,70,73,76,69,61,36,123,50,58,45,99,112,116,97,98,108,101,46,106,115,125,10,74,83,86,65,82,61,36,123,51,58,45,99,112,116,97,98,108,101,125,10,86,69,82,83,73,79,78,61,36,40,99,97,116,32,112,97,99,107,97,103,101,46,106,115,111,110,32,124,32,103,114,101,112,32,118,101,114,115,105,111,110,32,124,32,116,114,32,45,100,99,32,91,48,45,57,46,93,41,10,10,109,107,100,105,114,32,45,112,32,99,111,100,101,112,97,103,101,115,32,98,105,116,115,10,114,109,32,45,102,32,36,79,85,84,70,73,76,69,32,36,79,85,84,70,73,76,69,46,116,109,112,10,101,99,104,111,32,34,47,42,32,36,79,85,84,70,73,76,69,32,40,67,41,32,50,48,49,51,45,50,48,49,52,32,83,104,101,101,116,74,83,32,45,45,32,104,116,116,112,58,47,47,115,104,101,101,116,106,115,46,99,111,109,32,42,47,34,32,62,32,36,79,85,84,70,73,76,69,46,116,109,112,10,101,99,104,111,32,34,47,42,106,115,104,105,110,116,32,45,87,49,48,48,32,42,47,34,32,62,62,32,36,79,85,84,70,73,76,69,46,116,109,112,10,101,99,104,111,32,34,118,97,114,32,36,74,83,86,65,82,32,61,32,123,118,101,114,115,105,111,110,58,92,34,36,86,69,82,83,73,79,78,92,34,125,59,34,32,62,62,32,36,79,85,84,70,73,76,69,46,116,109,112,10,105,102,32,91,32,45,101,32,100,111,116,110,101,116,46,115,104,32,93,59,32,116,104,101,110,32,98,97,115,104,32,100,111,116,110,101,116,46,115,104,59,32,102,105,10,97,119,107,32,45,70,44,32,39,123,112,114,105,110,116,32,36,49,44,32,36,50,44,32,36,51,125,39,32,36,73,78,70,73,76,69,32,124,32,119,104,105,108,101,32,114,101,97,100,32,99,112,32,117,114,108,32,99,112,116,121,112,101,59,32,100,111,10,32,32,32,32,101,99,104,111,32,36,99,112,32,36,117,114,108,10,32,32,32,32,105,102,32,91,32,33,32,45,101,32,99,111,100,101,112,97,103,101,115,47,36,99,112,46,84,66,76,32,93,59,32,116,104,101,110,10,32,32,32,32,32,32,32,32,99,117,114,108,32,36,117,114,108,32,124,32,115,101,100,32,39,115,47,35,46,42,47,47,103,39,32,124,32,97,119,107,32,39,78,70,61,61,50,39,32,62,32,99,111,100,101,112,97,103,101,115,47,36,99,112,46,84,66,76,10,32,32,32,32,102,105,10,32,32,32,32,101,99,104,111,32,34,105,102,40,116,121,112,101,111,102,32,36,74,83,86,65,82,32,61,61,61,32,39,117,110,100,101,102,105,110,101,100,39,41,32,36,74,83,86,65,82,32,61,32,123,125,59,34,32,62,32,98,105,116,115,47,36,99,112,46,106,115,46,116,109,112,10,32,32,32,32,110,111,100,101,32,109,97,107,101,46,110,106,115,32,36,99,112,32,36,74,83,86,65,82,32,124,32,116,101,101,32,45,97,32,98,105,116,115,47,36,99,112,46,106,115,46,116,109,112,32,62,62,32,36,79,85,84,70,73,76,69,46,116,109,112,10,32,32,32,32,115,101,100,32,39,115,47,34,92,40,91,48,45,57,93,43,92,41,34,58,47,92,49,58,47,103,39,32,60,98,105,116,115,47,36,99,112,46,106,115,46,116,109,112,32,62,98,105,116,115,47,36,99,112,46,106,115,10,32,32,32,32,114,109,32,45,102,32,98,105,116,115,47,36,99,112,46,106,115,46,116,109,112,10,100,111,110,101,10,101,99,104,111,32,34,105,102,32,40,116,121,112,101,111,102,32,109,111,100,117,108,101,32,33,61,61,32,39,117,110,100,101,102,105,110,101,100,39,32,38,38,32,109,111,100,117,108,101,46,101,120,112,111,114,116,115,41,32,109,111,100,117,108,101,46,101,120,112,111,114,116,115,32,61,32,36,74,83,86,65,82,59,34,32,62,62,32,36,79,85,84,70,73,76,69,46,116,109,112,10,115,101,100,32,39,115,47,34,92,40,91,48,45,57,93,43,92,41,34,58,47,92,49,58,47,103,39,32,60,36,79,85,84,70,73,76,69,46,116,109,112,32,62,36,79,85,84,70,73,76,69,10,114,109,32,45,102,32,36,79,85,84,70,73,76,69,46,116,109,112,10,96,96,96,10,10,35,35,32,85,116,105,108,105,116,105,101,115,10,10,84,104,101,32,101,110,99,111,100,101,32,97,110,100,32,100,101,99,111,100,101,32,102,117,110,99,116,105,111,110,115,32,97,114,101,32,107,101,112,116,32,105,110,32,97,32,115,101,112,97,114,97,116,101,32,115,99,114,105,112,116,32,40,99,112,117,116,105,108,115,46,106,115,41,46,10,10,66,111,116,104,32,101,110,99,111,100,101,32,97,110,100,32,100,101,99,111,100,101,32,100,101,97,108,32,119,105,116,104,32,100,97,116,97,32,114,101,112,114,101,115,101,110,116,101,100,32,97,115,58,10,10,45,32,83,116,114,105,110,103,32,40,101,110,99,111,100,101,32,101,120,112,101,99,116,115,32,74,83,32,115,116,114,105,110,103,44,32,100,101,99,111,100,101,32,105,110,116,101,114,112,114,101,116,115,32,85,67,83,50,32,99,104,97,114,115,32,97,115,32,99,111,100,101,115,41,10,45,32,65,114,114,97,121,32,40,101,110,99,111,100,101,32,101,120,112,101,99,116,115,32,97,114,114,97,121,32,111,102,32,74,83,32,83,116,114,105,110,103,32,99,104,97,114,97,99,116,101,114,115,44,32,100,101,99,111,100,101,32,101,120,112,101,99,116,115,32,110,117,109,98,101,114,115,41,10,45,32,66,117,102,102,101,114,32,40,101,110,99,111,100,101,32,101,120,112,101,99,116,115,32,85,84,70,45,56,32,115,116,114,105,110,103,44,32,100,101,99,111,100,101,32,101,120,112,101,99,116,115,32,99,111,100,101,112,111,105,110,116,115,47,98,121,116,101,115,41,46,10,10,84,104,101,32,96,111,102,109,116,96,32,118,97,114,105,97,98,108,101,32,99,111,110,116,114,111,108,115,32,96,101,110,99,111,100,101,96,32,111,117,116,112,117,116,32,40,96,115,116,114,96,44,32,96,97,114,114,96,32,114,101,115,112,101,99,116,105,118,101,108,121,41,10,119,104,105,108,101,32,116,104,101,32,105,110,112,117,116,32,102,111,114,109,97,116,32,105,115,32,97,117,116,111,109,97,116,105,99,97,108,108,121,32,100,101,116,101,114,109,105,110,101,100,46,10,10,35,32,84,101,115,116,115,10,10,84,104,101,32,116,101,115,116,115,32,105,110,99,108,117,100,101,32,74,83,32,118,97,108,105,100,105,116,121,32,116,101,115,116,115,32,40,114,101,113,117,105,114,105,110,103,32,111,114,32,101,118,97,108,39,105,110,103,32,99,111,100,101,41,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,118,97,114,32,102,115,32,61,32,114,101,113,117,105,114,101,40,39,102,115,39,41,44,32,97,115,115,101,114,116,32,61,32,114,101,113,117,105,114,101,40,39,97,115,115,101,114,116,39,41,44,32,118,109,32,61,32,114,101,113,117,105,114,101,40,39,118,109,39,41,59,10,118,97,114,32,99,112,116,97,98,108,101,44,32,115,98,99,115,59,10,100,101,115,99,114,105,98,101,40,39,115,111,117,114,99,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,105,116,40,39,115,104,111,117,108,100,32,108,111,97,100,32,110,111,100,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,112,116,97,98,108,101,32,61,32,114,101,113,117,105,114,101,40,39,46,47,39,41,59,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,108,111,97,100,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,115,98,99,115,32,61,32,114,101,113,117,105,114,101,40,39,46,47,115,98,99,115,39,41,59,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,108,111,97,100,32,101,120,99,101,108,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,101,120,99,101,108,32,61,32,114,101,113,117,105,114,101,40,39,46,47,99,112,101,120,99,101,108,39,41,59,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,98,105,116,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,118,97,114,32,102,105,108,101,115,32,61,32,102,115,46,114,101,97,100,100,105,114,83,121,110,99,40,39,98,105,116,115,39,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,120,41,123,114,101,116,117,114,110,32,120,46,115,117,98,115,116,114,40,45,51,41,61,61,34,46,106,115,34,59,125,41,59,10,32,32,32,32,102,105,108,101,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,120,41,32,123,10,32,32,32,32,32,32,118,109,46,114,117,110,73,110,84,104,105,115,67,111,110,116,101,120,116,40,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,98,105,116,115,47,39,32,43,32,120,41,41,59,10,32,32,32,32,125,41,59,10,32,32,125,41,59,10,125,41,59,10,96,96,96,10,10,84,104,101,32,82,69,65,68,77,69,32,116,101,115,116,115,32,118,101,114,105,102,121,32,116,104,101,32,115,110,105,112,112,101,116,115,32,105,110,32,116,104,101,32,82,69,65,68,77,69,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,100,101,115,99,114,105,98,101,40,39,82,69,65,68,77,69,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,118,97,114,32,114,101,97,100,109,101,32,61,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,118,97,114,32,117,110,105,99,111,100,101,95,99,112,49,48,48,48,48,95,50,53,53,32,61,32,99,112,116,97,98,108,101,91,49,48,48,48,48,93,46,100,101,99,91,50,53,53,93,59,32,47,47,32,203,135,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,117,110,105,99,111,100,101,95,99,112,49,48,48,48,48,95,50,53,53,44,32,34,203,135,34,41,59,10,10,32,32,32,32,118,97,114,32,99,112,49,48,48,48,48,95,55,49,49,32,61,32,99,112,116,97,98,108,101,91,49,48,48,48,48,93,46,101,110,99,91,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,55,49,49,41,93,59,32,47,47,32,50,53,53,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,99,112,49,48,48,48,48,95,55,49,49,44,32,50,53,53,41,59,10,10,32,32,32,32,118,97,114,32,98,49,32,61,32,91,48,120,98,98,44,48,120,101,51,44,48,120,100,55,44,48,120,100,99,93,59,10,32,32,32,32,118,97,114,32,230,177,135,230,128,187,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,57,51,54,44,32,98,49,41,59,10,32,32,32,32,118,97,114,32,98,117,102,32,61,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,57,51,54,44,32,32,230,177,135,230,128,187,41,59,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,230,177,135,230,128,187,44,34,230,177,135,230,128,187,34,41,59,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,98,117,102,46,108,101,110,103,116,104,44,32,52,41,59,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,52,59,32,43,43,105,41,32,97,115,115,101,114,116,46,101,113,117,97,108,40,98,49,91,105,93,44,32,98,117,102,91,105,93,41,59,10,10,32,32,32,32,118,97,114,32,98,50,32,61,32,91,48,120,102,48,44,48,120,57,102,44,48,120,56,100,44,48,120,97,51,93,59,10,32,32,32,32,118,97,114,32,115,117,115,104,105,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,54,53,48,48,49,44,32,98,50,41,59,10,32,32,32,32,118,97,114,32,115,98,117,102,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,54,53,48,48,49,44,32,115,117,115,104,105,41,59,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,115,117,115,104,105,44,34,240,159,141,163,34,41,59,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,115,98,117,102,46,108,101,110,103,116,104,44,32,52,41,59,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,52,59,32,43,43,105,41,32,97,115,115,101,114,116,46,101,113,117,97,108,40,98,50,91,105,93,44,32,115,98,117,102,91,105,93,41,59,10,10,32,32,125,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,98,101,32,99,111,114,114,101,99,116,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,114,101,97,100,109,101,40,41,59,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,32,32,114,101,97,100,109,101,40,41,59,10,32,32,125,41,59,10,125,41,59,10,96,96,96,10,10,84,104,101,32,99,111,110,115,105,115,116,101,110,99,121,32,116,101,115,116,115,32,109,97,107,101,32,115,117,114,101,32,116,104,97,116,32,101,110,99,111,100,105,110,103,32,97,110,100,32,100,101,99,111,100,105,110,103,32,97,114,101,32,112,115,101,117,100,111,32,105,110,118,101,114,115,101,115,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,100,101,115,99,114,105,98,101,40,39,99,111,110,115,105,115,116,101,110,99,121,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,99,112,116,97,98,108,101,32,61,32,114,101,113,117,105,114,101,40,39,46,47,39,41,59,10,32,32,85,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,59,10,32,32,118,97,114,32,99,104,107,32,61,32,102,117,110,99,116,105,111,110,40,99,112,116,97,98,108,101,44,32,99,97,99,104,101,105,116,41,32,123,32,114,101,116,117,114,110,32,102,117,110,99,116,105,111,110,40,120,41,32,123,10,32,32,32,32,105,116,40,39,115,104,111,117,108,100,32,99,111,110,115,105,115,116,101,110,116,108,121,32,112,114,111,99,101,115,115,32,67,80,32,39,32,43,32,120,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,32,32,118,97,114,32,99,112,32,61,32,99,112,116,97,98,108,101,91,120,93,44,32,68,32,61,32,99,112,46,100,101,99,44,32,69,32,61,32,99,112,46,101,110,99,59,10,32,32,32,32,32,32,105,102,40,99,97,99,104,101,105,116,41,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,32,32,101,108,115,101,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,32,32,32,32,79,98,106,101,99,116,46,107,101,121,115,40,68,41,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,100,41,32,123,10,32,32,32,32,32,32,32,32,105,102,40,69,91,68,91,100,93,93,32,33,61,32,100,41,32,123,10,32,32,32,32,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,69,91,68,91,100,93,93,32,33,61,61,32,34,117,110,100,101,102,105,110,101,100,34,41,32,114,101,116,117,114,110,59,10,32,32,32,32,32,32,32,32,32,32,105,102,40,68,91,100,93,46,99,104,97,114,67,111,100,101,65,116,40,48,41,32,61,61,32,48,120,70,70,70,68,41,32,114,101,116,117,114,110,59,10,32,32,32,32,32,32,32,32,32,32,105,102,40,68,91,69,91,68,91,100,93,93,93,32,61,61,61,32,68,91,100,93,41,32,114,101,116,117,114,110,59,10,32,32,32,32,32,32,32,32,32,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,120,32,43,32,34,32,101,46,100,91,34,32,43,32,100,32,43,32,34,93,32,61,32,34,32,43,32,69,91,68,91,100,93,93,32,43,32,34,59,32,100,91,34,32,43,32,100,32,43,32,34,93,61,34,32,43,32,68,91,100,93,32,43,32,34,59,32,100,46,101,46,100,91,34,32,43,32,100,32,43,32,34,93,32,61,32,34,32,43,32,68,91,69,91,68,91,100,93,93,93,41,59,10,32,32,32,32,32,32,32,32,125,10,32,32,32,32,32,32,125,41,59,10,32,32,32,32,32,32,79,98,106,101,99,116,46,107,101,121,115,40,69,41,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,101,41,32,123,10,32,32,32,32,32,32,32,32,105,102,40,68,91,69,91,101,93,93,32,33,61,32,101,41,32,123,10,32,32,32,32,32,32,32,32,32,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,120,32,43,32,34,32,100,46,101,91,34,32,43,32,101,32,43,32,34,93,32,61,32,34,32,43,32,68,91,69,91,101,93,93,32,43,32,34,59,32,101,91,34,32,43,32,101,32,43,32,34,93,61,34,32,43,32,69,91,101,93,32,43,32,34,59,32,101,46,100,46,101,91,34,32,43,32,101,32,43,32,34,93,32,61,32,34,32,43,32,69,91,68,91,69,91,101,93,93,93,41,59,10,32,32,32,32,32,32,32,32,125,10,32,32,32,32,32,32,125,41,59,10,32,32,32,32,32,32,118,97,114,32,99,111,114,112,117,115,32,61,32,91,34,102,111,111,98,97,114,34,93,59,10,32,32,32,32,32,32,99,111,114,112,117,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,119,41,123,10,32,32,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,85,46,100,101,99,111,100,101,40,120,44,85,46,101,110,99,111,100,101,40,120,44,119,41,41,44,119,41,59,10,32,32,32,32,32,32,125,41,59,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,125,41,59,10,32,32,125,59,32,125,59,10,32,32,100,101,115,99,114,105,98,101,40,39,99,97,99,104,101,100,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,79,98,106,101,99,116,46,107,101,121,115,40,99,112,116,97,98,108,101,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,119,41,32,123,32,114,101,116,117,114,110,32,119,32,61,61,32,43,119,59,32,125,41,46,102,111,114,69,97,99,104,40,99,104,107,40,99,112,116,97,98,108,101,44,32,116,114,117,101,41,41,59,10,32,32,125,41,59,10,32,32,100,101,115,99,114,105,98,101,40,39,100,105,114,101,99,116,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,79,98,106,101,99,116,46,107,101,121,115,40,99,112,116,97,98,108,101,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,119,41,32,123,32,114,101,116,117,114,110,32,119,32,61,61,32,43,119,59,32,125,41,46,102,111,114,69,97,99,104,40,99,104,107,40,99,112,116,97,98,108,101,44,32,102,97,108,115,101,41,41,59,10,32,32,125,41,59,10,125,41,59,10,96,96,96,10,10,84,104,101,32,110,101,120,116,32,116,101,115,116,115,32,108,111,111,107,32,97,116,32,112,111,115,115,105,98,108,101,32,101,110,116,114,121,32,99,111,110,100,105,116,105,111,110,115,58,10,10,96,96,96,10,100,101,115,99,114,105,98,101,40,39,101,110,116,114,121,32,99,111,110,100,105,116,105,111,110,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,116,111,32,108,111,97,100,32,117,116,105,108,115,32,105,102,32,99,112,116,97,98,108,101,32,117,110,97,118,97,105,108,97,98,108,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,118,97,114,32,115,97,110,100,98,111,120,32,61,32,123,125,59,10,32,32,32,32,118,97,114,32,99,116,120,32,61,32,118,109,46,99,114,101,97,116,101,67,111,110,116,101,120,116,40,115,97,110,100,98,111,120,41,59,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,32,32,118,109,46,114,117,110,73,110,67,111,110,116,101,120,116,40,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,99,112,117,116,105,108,115,46,106,115,39,44,39,117,116,102,56,39,41,44,99,116,120,41,59,10,32,32,32,32,125,41,59,10,32,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,108,111,97,100,32,117,116,105,108,115,32,105,102,32,99,112,116,97,98,108,101,32,105,115,32,97,118,97,105,108,97,98,108,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,118,97,114,32,115,97,110,100,98,111,120,32,61,32,123,125,59,10,32,32,32,32,118,97,114,32,99,116,120,32,61,32,118,109,46,99,114,101,97,116,101,67,111,110,116,101,120,116,40,115,97,110,100,98,111,120,41,59,10,32,32,32,32,118,109,46,114,117,110,73,110,67,111,110,116,101,120,116,40,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,99,112,101,120,99,101,108,46,106,115,39,44,39,117,116,102,56,39,41,44,99,116,120,41,59,10,32,32,32,32,118,109,46,114,117,110,73,110,67,111,110,116,101,120,116,40,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,99,112,117,116,105,108,115,46,106,115,39,44,39,117,116,102,56,39,41,44,99,116,120,41,59,10,32,32,125,41,59,10,32,32,118,97,114,32,99,104,107,101,110,32,61,32,102,117,110,99,116,105,111,110,40,99,112,44,32,105,41,32,123,10,32,32,32,32,118,97,114,32,99,32,61,32,102,117,110,99,116,105,111,110,40,99,112,44,32,105,44,32,101,41,32,123,10,32,32,32,32,32,32,118,97,114,32,115,116,114,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,105,44,101,41,59,10,32,32,32,32,32,32,118,97,114,32,97,114,114,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,105,46,115,112,108,105,116,40,34,34,41,44,101,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,100,101,101,112,69,113,117,97,108,40,115,116,114,44,97,114,114,41,59,10,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,66,117,102,102,101,114,32,61,61,61,32,39,117,110,100,101,102,105,110,101,100,39,41,32,114,101,116,117,114,110,59,10,32,32,32,32,32,32,118,97,114,32,98,117,102,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,110,101,119,32,66,117,102,102,101,114,40,105,41,44,101,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,100,101,101,112,69,113,117,97,108,40,115,116,114,44,98,117,102,41,59,10,32,32,32,32,125,59,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,99,40,99,112,44,105,41,59,10,32,32,32,32,99,40,99,112,44,105,44,39,98,117,102,39,41,59,10,32,32,32,32,99,40,99,112,44,105,44,39,97,114,114,39,41,59,10,32,32,32,32,99,40,99,112,44,105,44,39,115,116,114,39,41,59,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,32,32,99,40,99,112,44,105,41,59,10,32,32,32,32,99,40,99,112,44,105,44,39,98,117,102,39,41,59,10,32,32,32,32,99,40,99,112,44,105,44,39,97,114,114,39,41,59,10,32,32,32,32,99,40,99,112,44,105,44,39,115,116,114,39,41,59,10,32,32,125,59,10,32,32,100,101,115,99,114,105,98,101,40,39,101,110,99,111,100,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,105,116,40,39,67,80,32,32,49,50,53,50,32,58,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,104,107,101,110,40,49,50,53,50,44,34,102,111,111,98,97,114,34,41,59,32,125,41,59,10,32,32,32,32,105,116,40,39,67,80,32,32,32,55,48,56,32,58,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,104,107,101,110,40,55,48,56,44,34,216,170,32,97,110,100,32,216,171,32,115,109,105,108,101,121,32,102,97,99,101,115,34,41,59,125,41,59,10,32,32,32,32,105,116,40,39,67,80,32,32,32,57,51,54,32,58,32,100,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,104,107,101,110,40,57,51,54,44,32,34,232,191,153,230,152,175,228,184,173,230,150,135,229,173,151,231,172,166,230,181,139,232,175,149,34,41,59,125,41,59,10,32,32,125,41,59,10,32,32,118,97,114,32,99,104,107,100,101,32,61,32,102,117,110,99,116,105,111,110,40,99,112,44,32,105,41,32,123,10,32,32,32,32,118,97,114,32,99,32,61,32,102,117,110,99,116,105,111,110,40,99,112,44,32,105,41,32,123,10,32,32,32,32,32,32,118,97,114,32,115,59,10,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,66,117,102,102,101,114,32,33,61,61,32,39,117,110,100,101,102,105,110,101,100,39,32,38,38,32,105,32,105,110,115,116,97,110,99,101,111,102,32,66,117,102,102,101,114,41,32,115,32,61,32,91,93,46,109,97,112,46,99,97,108,108,40,105,44,32,102,117,110,99,116,105,111,110,40,115,41,123,114,101,116,117,114,110,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,115,41,59,32,125,41,59,10,32,32,32,32,32,32,101,108,115,101,32,115,61,40,105,46,109,97,112,41,32,63,32,105,46,109,97,112,40,102,117,110,99,116,105,111,110,40,115,41,123,114,101,116,117,114,110,32,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,115,41,59,32,125,41,32,58,32,105,59,10,32,32,32,32,32,32,118,97,114,32,115,116,114,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,99,112,44,105,41,59,10,32,32,32,32,32,32,118,97,114,32,97,114,114,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,99,112,44,115,46,106,111,105,110,63,115,46,106,111,105,110,40,34,34,41,58,115,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,100,101,101,112,69,113,117,97,108,40,115,116,114,44,97,114,114,41,59,10,32,32,32,32,32,32,105,102,40,116,121,112,101,111,102,32,66,117,102,102,101,114,32,61,61,61,32,39,117,110,100,101,102,105,110,101,100,39,41,32,114,101,116,117,114,110,59,10,32,32,32,32,32,32,118,97,114,32,98,117,102,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,99,112,44,110,101,119,32,66,117,102,102,101,114,40,105,41,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,100,101,101,112,69,113,117,97,108,40,115,116,114,44,98,117,102,41,59,10,32,32,32,32,125,59,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,99,40,99,112,44,105,41,59,10,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,32,32,99,40,99,112,44,105,41,59,10,32,32,125,59,10,32,32,100,101,115,99,114,105,98,101,40,39,100,101,99,111,100,101,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,105,116,40,39,67,80,32,32,49,50,53,50,32,58,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,104,107,100,101,40,49,50,53,50,44,91,48,120,54,54,44,32,48,120,54,102,44,32,48,120,54,102,44,32,48,120,54,50,44,32,48,120,54,49,44,32,48,120,55,50,93,41,59,32,125,41,59,32,47,42,32,34,102,111,111,98,97,114,34,32,42,47,10,32,32,32,32,105,102,40,116,121,112,101,111,102,32,66,117,102,102,101,114,32,33,61,61,32,39,117,110,100,101,102,105,110,101,100,39,41,32,105,116,40,39,67,80,32,32,32,55,48,56,32,58,32,115,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,104,107,100,101,40,55,48,56,44,32,110,101,119,32,66,117,102,102,101,114,40,91,48,120,99,97,44,32,48,120,50,48,44,32,48,120,54,49,44,32,48,120,54,101,44,32,48,120,54,52,44,32,48,120,50,48,44,32,48,120,99,98,44,32,48,120,50,48,44,32,48,120,55,51,44,32,48,120,54,100,44,32,48,120,54,57,44,32,48,120,54,99,44,32,48,120,54,53,44,32,48,120,55,57,44,32,48,120,50,48,44,32,48,120,54,54,44,32,48,120,54,49,44,32,48,120,54,51,44,32,48,120,54,53,44,32,48,120,55,51,93,41,41,59,32,125,41,59,32,47,42,32,40,34,216,170,32,97,110,100,32,216,171,32,115,109,105,108,101,121,32,102,97,99,101,115,34,41,32,42,47,10,32,32,32,32,105,116,40,39,67,80,32,32,32,57,51,54,32,58,32,100,98,99,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,99,104,107,100,101,40,57,51,54,44,32,91,48,120,100,53,44,32,48,120,101,50,44,32,48,120,99,97,44,32,48,120,99,55,44,32,48,120,100,54,44,32,48,120,100,48,44,32,48,120,99,101,44,32,48,120,99,52,44,32,48,120,100,55,44,32,48,120,100,54,44,32,48,120,98,55,44,32,48,120,102,98,44,32,48,120,98,50,44,32,48,120,101,50,44,32,48,120,99,97,44,32,48,120,100,52,93,41,59,125,41,59,32,47,42,32,34,232,191,153,230,152,175,228,184,173,230,150,135,229,173,151,231,172,166,230,181,139,232,175,149,34,32,42,47,10,32,32,125,41,59,10,125,41,59,10,96,96,96,10,10,84,104,101,32,96,116,101,115,116,102,105,108,101,96,32,104,101,108,112,101,114,32,102,117,110,99,116,105,111,110,32,114,101,97,100,115,32,97,32,102,105,108,101,32,97,110,100,32,99,111,109,112,97,114,101,115,32,116,111,32,110,111,100,101,39,115,32,114,101,97,100,32,102,97,99,105,108,105,116,105,101,115,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,102,117,110,99,116,105,111,110,32,116,101,115,116,102,105,108,101,40,102,44,99,112,44,116,121,112,101,44,115,107,105,112,41,32,123,10,32,32,118,97,114,32,100,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,102,41,59,10,32,32,118,97,114,32,120,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,102,44,32,116,121,112,101,41,59,10,32,32,118,97,114,32,97,32,61,32,120,46,115,112,108,105,116,40,34,34,41,59,10,32,32,118,97,114,32,99,104,107,32,61,32,102,117,110,99,116,105,111,110,40,99,112,41,32,123,10,32,32,32,32,118,97,114,32,121,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,99,112,44,32,100,41,59,10,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,120,44,121,41,59,10,32,32,32,32,118,97,114,32,122,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,32,120,41,59,10,32,32,32,32,105,102,40,122,46,108,101,110,103,116,104,32,33,61,32,100,46,108,101,110,103,116,104,41,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,102,32,43,32,34,32,34,32,43,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,122,41,32,43,32,34,32,33,61,32,34,32,43,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,100,41,32,43,32,34,32,58,32,34,32,43,32,122,46,108,101,110,103,116,104,32,43,32,34,32,34,32,43,32,100,46,108,101,110,103,116,104,41,59,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,100,46,108,101,110,103,116,104,59,32,43,43,105,41,32,105,102,40,100,91,105,93,32,33,61,61,32,122,91,105,93,41,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,34,34,32,43,32,105,32,43,32,34,32,34,32,43,32,100,91,105,93,32,43,32,34,33,61,34,32,43,32,122,91,105,93,41,59,10,32,32,32,32,105,102,40,115,107,105,112,41,32,114,101,116,117,114,110,59,10,32,32,32,32,122,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,99,112,44,32,97,41,59,10,32,32,32,32,105,102,40,122,46,108,101,110,103,116,104,32,33,61,32,100,46,108,101,110,103,116,104,41,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,102,32,43,32,34,32,34,32,43,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,122,41,32,43,32,34,32,33,61,32,34,32,43,32,74,83,79,78,46,115,116,114,105,110,103,105,102,121,40,100,41,32,43,32,34,32,58,32,34,32,43,32,122,46,108,101,110,103,116,104,32,43,32,34,32,34,32,43,32,100,46,108,101,110,103,116,104,41,59,10,32,32,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,100,46,108,101,110,103,116,104,59,32,43,43,105,41,32,105,102,40,100,91,105,93,32,33,61,61,32,122,91,105,93,41,32,116,104,114,111,119,32,110,101,119,32,69,114,114,111,114,40,34,34,32,43,32,105,32,43,32,34,32,34,32,43,32,100,91,105,93,32,43,32,34,33,61,34,32,43,32,122,91,105,93,41,59,10,32,32,125,10,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,99,104,107,40,99,112,41,59,10,32,32,105,102,40,115,107,105,112,41,32,114,101,116,117,114,110,59,10,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,99,104,107,40,99,112,41,59,10,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,125,10,96,96,96,10,10,84,104,101,32,96,117,116,102,56,96,32,116,101,115,116,115,32,118,101,114,105,102,121,32,117,116,102,56,32,101,110,99,111,100,105,110,103,32,111,102,32,116,104,101,32,97,99,116,117,97,108,32,74,83,32,115,111,117,114,99,101,115,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,100,101,115,99,114,105,98,101,40,39,110,111,100,101,32,110,97,116,105,118,101,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,118,97,114,32,110,111,100,101,32,61,32,91,91,54,53,48,48,49,44,32,39,117,116,102,56,39,44,49,93,44,32,91,49,50,48,48,44,32,39,117,116,102,49,54,108,101,39,44,49,93,44,32,91,50,48,49,50,55,44,32,39,97,115,99,105,105,39,44,48,93,93,59,10,32,32,118,97,114,32,117,110,105,99,111,100,101,102,105,108,101,115,32,61,32,91,39,99,111,100,101,112,97,103,101,46,109,100,39,44,39,82,69,65,68,77,69,46,109,100,39,44,39,99,112,116,97,98,108,101,46,106,115,39,93,59,10,32,32,118,97,114,32,97,115,99,105,105,102,105,108,101,115,32,61,32,91,39,99,112,117,116,105,108,115,46,106,115,39,93,59,10,32,32,110,111,100,101,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,119,41,32,123,10,32,32,32,32,100,101,115,99,114,105,98,101,40,119,91,49,93,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,32,32,99,112,116,97,98,108,101,32,61,32,114,101,113,117,105,114,101,40,39,46,47,39,41,59,10,32,32,32,32,32,32,97,115,99,105,105,102,105,108,101,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,102,41,32,123,10,32,32,32,32,32,32,32,32,105,116,40,39,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,39,32,43,32,102,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,116,101,115,116,102,105,108,101,40,39,46,47,109,105,115,99,47,39,43,102,43,39,46,39,43,119,91,49,93,44,119,91,48,93,44,119,91,49,93,41,59,32,125,41,59,10,32,32,32,32,32,32,125,41,59,10,32,32,32,32,32,32,105,102,40,33,119,91,50,93,41,32,114,101,116,117,114,110,59,10,32,32,32,32,32,32,117,110,105,99,111,100,101,102,105,108,101,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,102,41,32,123,10,32,32,32,32,32,32,32,32,105,116,40,39,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,39,32,43,32,102,44,32,102,117,110,99,116,105,111,110,40,41,32,123,32,116,101,115,116,102,105,108,101,40,39,46,47,109,105,115,99,47,39,43,102,43,39,46,39,43,119,91,49,93,44,119,91,48,93,44,119,91,49,93,41,59,32,125,41,59,10,32,32,32,32,32,32,125,41,59,10,32,32,32,32,32,32,105,102,40,119,91,49,93,32,61,61,61,32,39,117,116,102,56,39,41,32,105,116,40,39,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,98,105,116,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,32,32,32,32,118,97,114,32,102,105,108,101,115,32,61,32,102,115,46,114,101,97,100,100,105,114,83,121,110,99,40,39,98,105,116,115,39,41,46,102,105,108,116,101,114,40,102,117,110,99,116,105,111,110,40,120,41,123,114,101,116,117,114,110,32,120,46,115,117,98,115,116,114,40,45,51,41,61,61,34,46,106,115,34,59,125,41,59,10,32,32,32,32,32,32,32,32,102,105,108,101,115,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,102,41,32,123,32,116,101,115,116,102,105,108,101,40,39,46,47,98,105,116,115,47,39,32,43,32,102,44,119,91,48,93,44,119,91,49,93,44,116,114,117,101,41,59,32,125,41,59,10,32,32,32,32,32,32,125,41,59,10,32,32,32,32,125,41,59,10,32,32,125,41,59,10,125,41,59,10,96,96,96,10,10,84,104,101,32,117,116,102,42,32,97,110,100,32,97,115,99,105,105,32,116,101,115,116,115,32,97,116,116,101,109,112,116,32,116,111,32,116,101,115,116,32,111,116,104,101,114,32,109,97,103,105,99,32,102,111,114,109,97,116,115,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,118,97,114,32,109,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,109,97,103,105,99,59,10,102,117,110,99,116,105,111,110,32,99,109,112,40,120,44,122,41,32,123,10,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,120,46,108,101,110,103,116,104,44,32,122,46,108,101,110,103,116,104,41,59,10,32,32,102,111,114,40,118,97,114,32,105,32,61,32,48,59,32,105,32,33,61,32,122,46,108,101,110,103,116,104,59,32,43,43,105,41,32,97,115,115,101,114,116,46,101,113,117,97,108,40,105,43,34,47,34,43,120,46,108,101,110,103,116,104,43,34,34,43,120,91,105,93,44,32,105,43,34,47,34,43,122,46,108,101,110,103,116,104,43,34,34,43,122,91,105,93,41,59,10,125,10,79,98,106,101,99,116,46,107,101,121,115,40,109,41,46,102,111,114,69,97,99,104,40,102,117,110,99,116,105,111,110,40,116,41,123,105,102,40,116,32,33,61,32,49,54,57,54,57,41,32,100,101,115,99,114,105,98,101,40,109,91,116,93,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,105,116,40,34,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,99,111,100,101,112,97,103,101,46,109,100,46,34,32,43,32,109,91,116,93,44,32,102,115,46,101,120,105,115,116,115,83,121,110,99,40,39,46,47,109,105,115,99,47,99,111,100,101,112,97,103,101,46,109,100,46,39,32,43,32,109,91,116,93,41,32,63,10,32,32,32,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,32,32,118,97,114,32,98,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,109,105,115,99,47,99,111,100,101,112,97,103,101,46,109,100,46,117,116,102,56,39,44,32,34,117,116,102,56,34,41,59,10,32,32,32,32,32,32,105,102,40,109,91,116,93,32,61,61,61,32,34,97,115,99,105,105,34,41,32,98,32,61,32,98,46,114,101,112,108,97,99,101,40,47,91,92,117,48,48,56,48,45,92,117,102,102,102,102,93,42,47,103,44,34,34,41,59,10,32,32,32,32,32,32,118,97,114,32,120,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,109,105,115,99,47,99,111,100,101,112,97,103,101,46,109,100,46,39,32,43,32,109,91,116,93,41,59,10,32,32,32,32,32,32,118,97,114,32,121,44,32,122,59,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,32,32,121,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,120,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,98,41,59,10,32,32,32,32,32,32,122,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,116,44,32,121,41,59,10,32,32,32,32,32,32,105,102,40,116,32,33,61,32,54,53,48,48,48,41,32,99,109,112,40,120,44,122,41,59,10,32,32,32,32,32,32,101,108,115,101,32,123,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,122,41,41,59,32,125,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,32,32,32,32,121,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,120,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,98,41,59,10,32,32,32,32,32,32,122,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,116,44,32,121,41,59,10,32,32,32,32,32,32,105,102,40,116,32,33,61,32,54,53,48,48,48,41,32,99,109,112,40,120,44,122,41,59,10,32,32,32,32,32,32,101,108,115,101,32,123,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,122,41,41,59,32,125,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,125,10,32,32,58,32,110,117,108,108,41,59,10,32,32,105,116,40,34,115,104,111,117,108,100,32,112,114,111,99,101,115,115,32,82,69,65,68,77,69,46,109,100,46,34,32,43,32,109,91,116,93,44,32,102,115,46,101,120,105,115,116,115,83,121,110,99,40,39,46,47,109,105,115,99,47,82,69,65,68,77,69,46,109,100,46,39,32,43,32,109,91,116,93,41,32,63,10,32,32,32,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,32,32,118,97,114,32,98,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,109,105,115,99,47,82,69,65,68,77,69,46,109,100,46,117,116,102,56,39,44,32,34,117,116,102,56,34,41,59,10,32,32,32,32,32,32,105,102,40,109,91,116,93,32,61,61,61,32,34,97,115,99,105,105,34,41,32,98,32,61,32,98,46,114,101,112,108,97,99,101,40,47,91,92,117,48,48,56,48,45,92,117,102,102,102,102,93,42,47,103,44,34,34,41,59,10,32,32,32,32,32,32,118,97,114,32,120,32,61,32,102,115,46,114,101,97,100,70,105,108,101,83,121,110,99,40,39,46,47,109,105,115,99,47,82,69,65,68,77,69,46,109,100,46,39,32,43,32,109,91,116,93,41,59,10,32,32,32,32,32,32,120,32,61,32,91,93,46,115,108,105,99,101,46,99,97,108,108,40,120,41,59,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,32,32,118,97,114,32,121,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,120,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,98,41,59,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,10,32,32,32,32,32,32,118,97,114,32,121,32,61,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,116,44,32,120,41,59,10,32,32,32,32,32,32,97,115,115,101,114,116,46,101,113,117,97,108,40,121,44,98,41,59,10,32,32,32,32,32,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,101,110,99,97,99,104,101,40,41,59,10,32,32,32,32,125,10,32,32,58,32,110,117,108,108,41,59,10,125,41,59,125,41,59,10,96,96,96,10,10,84,104,101,32,99,111,100,101,112,97,103,101,32,96,54,57,54,57,96,32,105,115,32,110,111,116,32,100,101,102,105,110,101,100,44,32,115,111,32,111,112,101,114,97,116,105,111,110,115,32,115,104,111,117,108,100,32,102,97,105,108,58,10,10,96,96,96,62,116,101,115,116,46,106,115,10,100,101,115,99,114,105,98,101,40,39,102,97,105,108,117,114,101,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,116,111,32,102,105,110,100,32,67,80,32,54,57,54,57,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,99,112,116,97,98,108,101,91,54,57,54,57,93,46,100,101,99,125,41,59,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,99,112,116,97,98,108,101,91,54,57,54,57,93,46,101,110,99,125,41,59,10,32,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,117,115,105,110,103,32,117,116,105,108,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,97,115,115,101,114,116,40,33,99,112,116,97,98,108,101,46,117,116,105,108,115,46,104,97,115,99,112,40,54,57,54,57,41,41,59,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,54,57,54,57,44,32,34,102,111,111,98,97,114,34,41,59,32,125,41,59,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,54,57,54,57,44,32,91,48,120,50,48,93,41,59,32,125,41,59,10,32,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,119,105,116,104,32,98,108,97,99,107,32,109,97,103,105,99,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,97,115,115,101,114,116,40,99,112,116,97,98,108,101,46,117,116,105,108,115,46,104,97,115,99,112,40,49,54,57,54,57,41,41,59,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,49,54,57,54,57,44,32,34,102,111,111,98,97,114,34,41,59,32,125,41,59,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,100,101,99,111,100,101,40,49,54,57,54,57,44,32,91,48,120,50,48,93,41,59,32,125,41,59,10,32,32,125,41,59,10,32,32,105,116,40,39,115,104,111,117,108,100,32,102,97,105,108,32,119,104,101,110,32,112,114,101,115,101,110,116,101,100,32,119,105,116,104,32,105,110,118,97,108,105,100,32,99,104,97,114,32,99,111,100,101,115,39,44,32,102,117,110,99,116,105,111,110,40,41,32,123,10,32,32,32,32,97,115,115,101,114,116,46,116,104,114,111,119,115,40,102,117,110,99,116,105,111,110,40,41,123,99,112,116,97,98,108,101,46,117,116,105,108,115,46,99,97,99,104,101,46,100,101,99,97,99,104,101,40,41,59,32,114,101,116,117,114,110,32,99,112,116,97,98,108,101,46,117,116,105,108,115,46,101,110,99,111,100,101,40,50,48,49,50,55,44,32,91,83,116,114,105,110,103,46,102,114,111,109,67,104,97,114,67,111,100,101,40,48,120,65,65,41,93,41,59,125,41,59,10,32,32,125,41,59,10,125,41,59,10,96,96,96,10,10,35,32,78,105,116,116,121,32,71,114,105,116,116,121,10,10,96,96,96,106,115,111,110,62,112,97,99,107,97,103,101,46,106,115,111,110,10,123,10,32,32,34,110,97,109,101,34,58,32,34,99,111,100,101,112,97,103,101,34,44,10,32,32,34,118,101,114,115,105,111,110,34,58,32,34,49,46,51,46,56,34,44,10,32,32,34,97,117,116,104,111,114,34,58,32,34,83,104,101,101,116,74,83,34,44,10,32,32,34,100,101,115,99,114,105,112,116,105,111,110,34,58,32,34,112,117,114,101,45,74,83,32,108,105,98,114,97,114,121,32,116,111,32,104,97,110,100,108,101,32,99,111,100,101,112,97,103,101,115,34,44,10,32,32,34,107,101,121,119,111,114,100,115,34,58,32,91,32,34,99,111,100,101,112,97,103,101,34,44,32,34,105,99,111,110,118,34,44,32,34,99,111,110,118,101,114,116,34,44,32,34,115,116,114,105,110,103,115,34,32,93,44,10,32,32,34,98,105,110,34,58,32,123,10,32,32,32,32,34,99,111,100,101,112,97,103,101,34,58,32,34,46,47,98,105,110,47,99,111,100,101,112,97,103,101,46,110,106,115,34,10,32,32,125,44,10,32,32,34,102,105,108,101,115,34,58,32,91,10,32,32,32,32,34,76,73,67,69,78,83,69,34,44,10,32,32,32,32,34,82,69,65,68,77,69,46,109,100,34,44,10,32,32,32,32,34,98,105,110,34,44,10,32,32,32,32,34,99,112,116,97,98,108,101,46,106,115,34,44,10,32,32,32,32,34,99,112,117,116,105,108,115,46,106,115,34,44,10,32,32,32,32,34,100,105,115,116,47,99,112,101,120,99,101,108,46,102,117,108,108,46,106,115,34,10,32,32,93,44,10,32,32,34,109,97,105,110,34,58,32,34,99,112,117,116,105,108,115,46,106,115,34,44,10,32,32,34,100,101,112,101,110,100,101,110,99,105,101,115,34,58,32,123,10,32,32,32,32,34,118,111,99,34,58,34,34,44,10,32,32,32,32,34,99,111,110,99,97,116,45,115,116,114,101,97,109,34,58,34,34,44,10,32,32,32,32,34,99,111,109,109,97,110,100,101,114,34,58,34,34,10,32,32,125,44,10,32,32,34,100,101,118,68,101,112,101,110,100,101,110,99,105,101,115,34,58,32,123,10,32,32,32,32,34,109,111,99,104,97,34,58,34,34,10,32,32,125,44,10,32,32,34,114,101,112,111,115,105,116,111,114,121,34,58,32,123,32,34,116,121,112,101,34,58,34,103,105,116,34,44,32,34,117,114,108,34,58,34,103,105,116,58,47,47,103,105,116,104,117,98,46,99,111,109,47,83,104,101,101,116,74,83,47,106,115,45,99,111,100,101,112,97,103,101,46,103,105,116,34,125,44,10,32,32,34,115,99,114,105,112,116,115,34,58,32,123,10,32,32,32,32,34,112,114,101,116,101,115,116,34,58,32,34,103,105,116,32,115,117,98,109,111,100,117,108,101,32,105,110,105,116,32,38,38,32,103,105,116,32,115,117,98,109,111,100,117,108,101,32,117,112,100,97,116,101,34,44,10,32,32,32,32,34,116,101,115,116,34,58,32,34,109,97,107,101,32,116,101,115,116,34,44,10,32,32,32,32,34,98,117,105,108,100,34,58,32,34,109,97,107,101,32,106,115,34,10,32,32,125,44,10,32,32,34,99,111,110,102,105,103,34,58,32,123,10,32,32,32,32,34,98,108,97,110,107,101,116,34,58,32,123,10,32,32,32,32,32,32,34,112,97,116,116,101,114,110,34,58,32,34,91,99,112,116,97,98,108,101,46,106,115,44,99,112,117,116,105,108,115,46,106,115,44,99,112,101,120,99,101,108,46,106,115,93,34,10,32,32,32,32,125,10,32,32,125,44,10,32,32,34,98,117,103,115,34,58,32,123,32,34,117,114,108,34,58,32,34,104,116,116,112,115,58,47,47,103,105,116,104,117,98,46,99,111,109,47,83,104,101,101,116,74,83,47,106,115,45,99,111,100,101,112,97,103,101,47,105,115,115,117,101,115,34,32,125,44,10,32,32,34,108,105,99,101,110,115,101,34,58,32,34,65,112,97,99,104,101,45,50,46,48,34,44,10,32,32,34,101,110,103,105,110,101,115,34,58,32,123,32,34,110,111,100,101,34,58,32,34,62,61,48,46,56,34,32,125,10,125,10,96,96,96,10,10,96,96,96,62,46,118,111,99,114,99,10,123,32,34,112,111,115,116,34,58,32,34,109,97,107,101,32,106,115,34,32,125,10,96,96,96,10,10,96,96,96,62,46,103,105,116,105,103,110,111,114,101,10,46,103,105,116,105,103,110,111,114,101,10,99,111,100,101,112,97,103,101,115,47,10,46,118,111,99,114,99,10,110,111,100,101,95,109,111,100,117,108,101,115,47,10,109,97,107,101,46,115,104,10,109,97,107,101,46,110,106,115,10,109,105,115,99,47,99,111,118,101,114,97,103,101,46,104,116,109,108,10,96,96,96,10]
;
README['utf16le'] = 
[35,0,32,0,71,0,101,0,116,0,116,0,105,0,110,0,103,0,32,0,67,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,10,0,10,0,84,0,104,0,101,0,32,0,102,0,105,0,101,0,108,0,100,0,115,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,32,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,32,0,109,0,97,0,110,0,105,0,102,0,101,0,115,0,116,0,32,0,97,0,114,0,101,0,32,0,96,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,44,0,117,0,114,0,108,0,44,0,98,0,121,0,116,0,101,0,115,0,96,0,32,0,40,0,83,0,66,0,67,0,83,0,61,0,49,0,44,0,32,0,68,0,66,0,67,0,83,0,61,0,50,0,41,0,10,0,10,0,96,0,96,0,96,0,62,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,10,0,51,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,69,0,66,0,67,0,68,0,73,0,67,0,47,0,67,0,80,0,48,0,51,0,55,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,52,0,51,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,52,0,51,0,55,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,53,0,48,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,69,0,66,0,67,0,68,0,73,0,67,0,47,0,67,0,80,0,53,0,48,0,48,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,55,0,51,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,55,0,51,0,55,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,55,0,55,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,55,0,55,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,53,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,53,0,48,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,53,0,50,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,53,0,50,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,53,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,53,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,53,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,53,0,55,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,48,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,49,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,49,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,50,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,50,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,51,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,51,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,52,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,52,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,54,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,57,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,57,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,55,0,52,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,56,0,55,0,52,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,55,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,69,0,66,0,67,0,68,0,73,0,67,0,47,0,67,0,80,0,56,0,55,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,57,0,51,0,50,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,57,0,51,0,50,0,46,0,84,0,88,0,84,0,44,0,50,0,10,0,57,0,51,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,57,0,51,0,54,0,46,0,84,0,88,0,84,0,44,0,50,0,10,0,57,0,52,0,57,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,57,0,52,0,57,0,46,0,84,0,88,0,84,0,44,0,50,0,10,0,57,0,53,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,57,0,53,0,48,0,46,0,84,0,88,0,84,0,44,0,50,0,10,0,49,0,48,0,50,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,69,0,66,0,67,0,68,0,73,0,67,0,47,0,67,0,80,0,49,0,48,0,50,0,54,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,48,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,49,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,49,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,50,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,50,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,51,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,51,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,52,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,52,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,54,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,55,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,56,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,56,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,96,0,96,0,96,0,10,0,10,0,78,0,111,0,116,0,101,0,32,0,116,0,104,0,97,0,116,0,32,0,116,0,104,0,101,0,32,0,87,0,105,0,110,0,100,0,111,0,119,0,115,0,32,0,114,0,101,0,110,0,100,0,101,0,114,0,105,0,110,0,103,0,32,0,105,0,115,0,32,0,117,0,115,0,101,0,100,0,32,0,102,0,111,0,114,0,32,0,116,0,104,0,101,0,32,0,77,0,97,0,99,0,32,0,99,0,111,0,100,0,101,0,32,0,112,0,97,0,103,0,101,0,115,0,46,0,32,0,32,0,84,0,104,0,101,0,32,0,112,0,114,0,105,0,109,0,97,0,114,0,121,0,10,0,100,0,105,0,102,0,102,0,101,0,114,0,101,0,110,0,99,0,101,0,32,0,105,0,115,0,32,0,116,0,104,0,101,0,32,0,117,0,115,0,101,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,32,0,112,0,114,0,105,0,118,0,97,0,116,0,101,0,32,0,96,0,48,0,120,0,70,0,56,0,70,0,70,0,96,0,32,0,99,0,111,0,100,0,101,0,32,0,40,0,119,0,104,0,105,0,99,0,104,0,32,0,114,0,101,0,110,0,100,0,101,0,114,0,115,0,32,0,97,0,115,0,32,0,97,0,110,0,32,0,65,0,112,0,112,0,108,0,101,0,10,0,108,0,111,0,103,0,111,0,32,0,111,0,110,0,32,0,109,0,97,0,99,0,115,0,32,0,98,0,117,0,116,0,32,0,97,0,115,0,32,0,103,0,97,0,114,0,98,0,97,0,103,0,101,0,32,0,111,0,110,0,32,0,111,0,116,0,104,0,101,0,114,0,32,0,111,0,112,0,101,0,114,0,97,0,116,0,105,0,110,0,103,0,32,0,115,0,121,0,115,0,116,0,101,0,109,0,115,0,41,0,46,0,32,0,32,0,73,0,116,0,32,0,109,0,97,0,121,0,32,0,98,0,101,0,32,0,100,0,101,0,115,0,105,0,114,0,97,0,98,0,108,0,101,0,10,0,116,0,111,0,32,0,102,0,97,0,108,0,108,0,32,0,98,0,97,0,99,0,107,0,32,0,116,0,111,0,32,0,116,0,104,0,101,0,32,0,98,0,101,0,104,0,97,0,118,0,105,0,111,0,114,0,44,0,32,0,105,0,110,0,32,0,119,0,104,0,105,0,99,0,104,0,32,0,99,0,97,0,115,0,101,0,32,0,116,0,104,0,101,0,32,0,102,0,105,0,108,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,117,0,110,0,100,0,101,0,114,0,32,0,65,0,80,0,80,0,76,0,69,0,32,0,97,0,110,0,100,0,32,0,110,0,111,0,116,0,10,0,77,0,73,0,67,0,83,0,70,0,84,0,46,0,32,0,32,0,67,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,97,0,110,0,32,0,97,0,98,0,115,0,111,0,108,0,117,0,116,0,101,0,32,0,112,0,97,0,105,0,110,0,32,0,58,0,47,0,10,0,10,0,96,0,96,0,96,0,62,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,10,0,49,0,48,0,48,0,48,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,77,0,65,0,67,0,47,0,82,0,79,0,77,0,65,0,78,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,48,0,48,0,48,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,77,0,65,0,67,0,47,0,71,0,82,0,69,0,69,0,75,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,48,0,48,0,48,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,77,0,65,0,67,0,47,0,67,0,89,0,82,0,73,0,76,0,76,0,73,0,67,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,48,0,48,0,50,0,57,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,77,0,65,0,67,0,47,0,76,0,65,0,84,0,73,0,78,0,50,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,48,0,48,0,55,0,57,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,77,0,65,0,67,0,47,0,73,0,67,0,69,0,76,0,65,0,78,0,68,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,48,0,48,0,56,0,49,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,77,0,65,0,67,0,47,0,84,0,85,0,82,0,75,0,73,0,83,0,72,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,110,0,117,0,109,0,98,0,101,0,114,0,105,0,110,0,103,0,32,0,115,0,99,0,104,0,101,0,109,0,101,0,32,0,102,0,111,0,114,0,32,0,116,0,104,0,101,0,32,0,96,0,73,0,83,0,79,0,45,0,56,0,56,0,53,0,57,0,45,0,88,0,96,0,32,0,115,0,101,0,114,0,105,0,101,0,115,0,32,0,105,0,115,0,32,0,96,0,50,0,56,0,53,0,57,0,48,0,32,0,43,0,32,0,88,0,96,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,10,0,50,0,56,0,53,0,57,0,49,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,50,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,50,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,51,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,51,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,52,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,52,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,54,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,55,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,56,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,56,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,57,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,57,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,54,0,48,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,48,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,54,0,48,0,49,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,49,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,54,0,48,0,51,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,51,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,54,0,48,0,52,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,52,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,54,0,48,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,54,0,48,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,54,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,96,0,96,0,96,0,10,0,10,0,35,0,35,0,32,0,71,0,101,0,110,0,101,0,114,0,97,0,116,0,101,0,100,0,32,0,67,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,10,0,10,0,84,0,104,0,101,0,32,0,102,0,111,0,108,0,108,0,111,0,119,0,105,0,110,0,103,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,97,0,118,0,97,0,105,0,108,0,97,0,98,0,108,0,101,0,32,0,105,0,110,0,32,0,46,0,78,0,69,0,84,0,32,0,111,0,110,0,32,0,87,0,105,0,110,0,100,0,111,0,119,0,115,0,58,0,10,0,10,0,45,0,32,0,55,0,48,0,56,0,32,0,32,0,32,0,65,0,114,0,97,0,98,0,105,0,99,0,32,0,40,0,65,0,83,0,77,0,79,0,32,0,55,0,48,0,56,0,41,0,10,0,45,0,32,0,55,0,50,0,48,0,32,0,32,0,32,0,65,0,114,0,97,0,98,0,105,0,99,0,32,0,40,0,84,0,114,0,97,0,110,0,115,0,112,0,97,0,114,0,101,0,110,0,116,0,32,0,65,0,83,0,77,0,79,0,41,0,59,0,32,0,65,0,114,0,97,0,98,0,105,0,99,0,32,0,40,0,68,0,79,0,83,0,41,0,10,0,45,0,32,0,56,0,53,0,56,0,32,0,32,0,32,0,79,0,69,0,77,0,32,0,77,0,117,0,108,0,116,0,105,0,108,0,105,0,110,0,103,0,117,0,97,0,108,0,32,0,76,0,97,0,116,0,105,0,110,0,32,0,49,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,10,0,45,0,32,0,56,0,55,0,48,0,32,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,77,0,117,0,108,0,116,0,105,0,108,0,105,0,110,0,103,0,117,0,97,0,108,0,47,0,82,0,79,0,69,0,67,0,69,0,32,0,40,0,76,0,97,0,116,0,105,0,110,0,32,0,50,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,77,0,117,0,108,0,116,0,105,0,108,0,105,0,110,0,103,0,117,0,97,0,108,0,32,0,76,0,97,0,116,0,105,0,110,0,32,0,50,0,10,0,45,0,32,0,49,0,48,0,52,0,55,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,76,0,97,0,116,0,105,0,110,0,32,0,49,0,47,0,79,0,112,0,101,0,110,0,32,0,83,0,121,0,115,0,116,0,101,0,109,0,10,0,45,0,32,0,49,0,49,0,52,0,48,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,85,0,83,0,45,0,67,0,97,0,110,0,97,0,100,0,97,0,32,0,40,0,48,0,51,0,55,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,85,0,83,0,45,0,67,0,97,0,110,0,97,0,100,0,97,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,49,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,71,0,101,0,114,0,109,0,97,0,110,0,121,0,32,0,40,0,50,0,48,0,50,0,55,0,51,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,71,0,101,0,114,0,109,0,97,0,110,0,121,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,50,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,68,0,101,0,110,0,109,0,97,0,114,0,107,0,45,0,78,0,111,0,114,0,119,0,97,0,121,0,32,0,40,0,50,0,48,0,50,0,55,0,55,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,68,0,101,0,110,0,109,0,97,0,114,0,107,0,45,0,78,0,111,0,114,0,119,0,97,0,121,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,51,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,70,0,105,0,110,0,108,0,97,0,110,0,100,0,45,0,83,0,119,0,101,0,100,0,101,0,110,0,32,0,40,0,50,0,48,0,50,0,55,0,56,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,70,0,105,0,110,0,108,0,97,0,110,0,100,0,45,0,83,0,119,0,101,0,100,0,101,0,110,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,52,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,73,0,116,0,97,0,108,0,121,0,32,0,40,0,50,0,48,0,50,0,56,0,48,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,73,0,116,0,97,0,108,0,121,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,53,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,76,0,97,0,116,0,105,0,110,0,32,0,65,0,109,0,101,0,114,0,105,0,99,0,97,0,45,0,83,0,112,0,97,0,105,0,110,0,32,0,40,0,50,0,48,0,50,0,56,0,52,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,83,0,112,0,97,0,105,0,110,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,54,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,85,0,110,0,105,0,116,0,101,0,100,0,32,0,75,0,105,0,110,0,103,0,100,0,111,0,109,0,32,0,40,0,50,0,48,0,50,0,56,0,53,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,85,0,75,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,55,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,70,0,114,0,97,0,110,0,99,0,101,0,32,0,40,0,50,0,48,0,50,0,57,0,55,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,70,0,114,0,97,0,110,0,99,0,101,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,56,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,73,0,110,0,116,0,101,0,114,0,110,0,97,0,116,0,105,0,111,0,110,0,97,0,108,0,32,0,40,0,53,0,48,0,48,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,73,0,110,0,116,0,101,0,114,0,110,0,97,0,116,0,105,0,111,0,110,0,97,0,108,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,57,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,73,0,99,0,101,0,108,0,97,0,110,0,100,0,105,0,99,0,32,0,40,0,50,0,48,0,56,0,55,0,49,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,73,0,99,0,101,0,108,0,97,0,110,0,100,0,105,0,99,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,51,0,54,0,49,0,32,0,32,0,75,0,111,0,114,0,101,0,97,0,110,0,32,0,40,0,74,0,111,0,104,0,97,0,98,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,48,0,49,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,48,0,50,0,32,0,77,0,65,0,67,0,32,0,84,0,114,0,97,0,100,0,105,0,116,0,105,0,111,0,110,0,97,0,108,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,40,0,66,0,105,0,103,0,53,0,41,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,84,0,114,0,97,0,100,0,105,0,116,0,105,0,111,0,110,0,97,0,108,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,48,0,51,0,32,0,75,0,111,0,114,0,101,0,97,0,110,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,48,0,52,0,32,0,65,0,114,0,97,0,98,0,105,0,99,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,48,0,53,0,32,0,72,0,101,0,98,0,114,0,101,0,119,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,48,0,56,0,32,0,77,0,65,0,67,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,40,0,71,0,66,0,32,0,50,0,51,0,49,0,50,0,41,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,49,0,48,0,32,0,82,0,111,0,109,0,97,0,110,0,105,0,97,0,110,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,49,0,55,0,32,0,85,0,107,0,114,0,97,0,105,0,110,0,105,0,97,0,110,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,50,0,49,0,32,0,84,0,104,0,97,0,105,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,56,0,50,0,32,0,67,0,114,0,111,0,97,0,116,0,105,0,97,0,110,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,50,0,48,0,48,0,48,0,48,0,32,0,67,0,78,0,83,0,32,0,84,0,97,0,105,0,119,0,97,0,110,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,84,0,114,0,97,0,100,0,105,0,116,0,105,0,111,0,110,0,97,0,108,0,32,0,40,0,67,0,78,0,83,0,41,0,10,0,45,0,32,0,50,0,48,0,48,0,48,0,49,0,32,0,84,0,67,0,65,0,32,0,84,0,97,0,105,0,119,0,97,0,110,0,10,0,45,0,32,0,50,0,48,0,48,0,48,0,50,0,32,0,69,0,116,0,101,0,110,0,32,0,84,0,97,0,105,0,119,0,97,0,110,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,84,0,114,0,97,0,100,0,105,0,116,0,105,0,111,0,110,0,97,0,108,0,32,0,40,0,69,0,116,0,101,0,110,0,41,0,10,0,45,0,32,0,50,0,48,0,48,0,48,0,51,0,32,0,73,0,66,0,77,0,53,0,53,0,53,0,48,0,32,0,84,0,97,0,105,0,119,0,97,0,110,0,10,0,45,0,32,0,50,0,48,0,48,0,48,0,52,0,32,0,84,0,101,0,108,0,101,0,84,0,101,0,120,0,116,0,32,0,84,0,97,0,105,0,119,0,97,0,110,0,10,0,45,0,32,0,50,0,48,0,48,0,48,0,53,0,32,0,87,0,97,0,110,0,103,0,32,0,84,0,97,0,105,0,119,0,97,0,110,0,10,0,45,0,32,0,50,0,48,0,49,0,48,0,53,0,32,0,73,0,65,0,53,0,32,0,40,0,73,0,82,0,86,0,32,0,73,0,110,0,116,0,101,0,114,0,110,0,97,0,116,0,105,0,111,0,110,0,97,0,108,0,32,0,65,0,108,0,112,0,104,0,97,0,98,0,101,0,116,0,32,0,78,0,111,0,46,0,32,0,53,0,44,0,32,0,55,0,45,0,98,0,105,0,116,0,41,0,59,0,32,0,87,0,101,0,115,0,116,0,101,0,114,0,110,0,32,0,69,0,117,0,114,0,111,0,112,0,101,0,97,0,110,0,32,0,40,0,73,0,65,0,53,0,41,0,10,0,45,0,32,0,50,0,48,0,49,0,48,0,54,0,32,0,73,0,65,0,53,0,32,0,71,0,101,0,114,0,109,0,97,0,110,0,32,0,40,0,55,0,45,0,98,0,105,0,116,0,41,0,10,0,45,0,32,0,50,0,48,0,49,0,48,0,55,0,32,0,73,0,65,0,53,0,32,0,83,0,119,0,101,0,100,0,105,0,115,0,104,0,32,0,40,0,55,0,45,0,98,0,105,0,116,0,41,0,10,0,45,0,32,0,50,0,48,0,49,0,48,0,56,0,32,0,73,0,65,0,53,0,32,0,78,0,111,0,114,0,119,0,101,0,103,0,105,0,97,0,110,0,32,0,40,0,55,0,45,0,98,0,105,0,116,0,41,0,10,0,45,0,32,0,50,0,48,0,50,0,54,0,49,0,32,0,84,0,46,0,54,0,49,0,10,0,45,0,32,0,50,0,48,0,50,0,54,0,57,0,32,0,73,0,83,0,79,0,32,0,54,0,57,0,51,0,55,0,32,0,78,0,111,0,110,0,45,0,83,0,112,0,97,0,99,0,105,0,110,0,103,0,32,0,65,0,99,0,99,0,101,0,110,0,116,0,10,0,45,0,32,0,50,0,48,0,50,0,55,0,51,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,71,0,101,0,114,0,109,0,97,0,110,0,121,0,10,0,45,0,32,0,50,0,48,0,50,0,55,0,55,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,68,0,101,0,110,0,109,0,97,0,114,0,107,0,45,0,78,0,111,0,114,0,119,0,97,0,121,0,10,0,45,0,32,0,50,0,48,0,50,0,55,0,56,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,70,0,105,0,110,0,108,0,97,0,110,0,100,0,45,0,83,0,119,0,101,0,100,0,101,0,110,0,10,0,45,0,32,0,50,0,48,0,50,0,56,0,48,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,73,0,116,0,97,0,108,0,121,0,10,0,45,0,32,0,50,0,48,0,50,0,56,0,52,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,76,0,97,0,116,0,105,0,110,0,32,0,65,0,109,0,101,0,114,0,105,0,99,0,97,0,45,0,83,0,112,0,97,0,105,0,110,0,10,0,45,0,32,0,50,0,48,0,50,0,56,0,53,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,85,0,110,0,105,0,116,0,101,0,100,0,32,0,75,0,105,0,110,0,103,0,100,0,111,0,109,0,10,0,45,0,32,0,50,0,48,0,50,0,57,0,48,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,75,0,97,0,116,0,97,0,107,0,97,0,110,0,97,0,32,0,69,0,120,0,116,0,101,0,110,0,100,0,101,0,100,0,10,0,45,0,32,0,50,0,48,0,50,0,57,0,55,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,70,0,114,0,97,0,110,0,99,0,101,0,10,0,45,0,32,0,50,0,48,0,52,0,50,0,48,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,65,0,114,0,97,0,98,0,105,0,99,0,10,0,45,0,32,0,50,0,48,0,52,0,50,0,51,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,71,0,114,0,101,0,101,0,107,0,10,0,45,0,32,0,50,0,48,0,52,0,50,0,52,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,72,0,101,0,98,0,114,0,101,0,119,0,10,0,45,0,32,0,50,0,48,0,56,0,51,0,51,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,75,0,111,0,114,0,101,0,97,0,110,0,32,0,69,0,120,0,116,0,101,0,110,0,100,0,101,0,100,0,10,0,45,0,32,0,50,0,48,0,56,0,51,0,56,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,84,0,104,0,97,0,105,0,10,0,45,0,32,0,50,0,48,0,56,0,54,0,54,0,32,0,82,0,117,0,115,0,115,0,105,0,97,0,110,0,32,0,40,0,75,0,79,0,73,0,56,0,45,0,82,0,41,0,59,0,32,0,67,0,121,0,114,0,105,0,108,0,108,0,105,0,99,0,32,0,40,0,75,0,79,0,73,0,56,0,45,0,82,0,41,0,10,0,45,0,32,0,50,0,48,0,56,0,55,0,49,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,73,0,99,0,101,0,108,0,97,0,110,0,100,0,105,0,99,0,10,0,45,0,32,0,50,0,48,0,56,0,56,0,48,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,67,0,121,0,114,0,105,0,108,0,108,0,105,0,99,0,32,0,82,0,117,0,115,0,115,0,105,0,97,0,110,0,10,0,45,0,32,0,50,0,48,0,57,0,48,0,53,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,84,0,117,0,114,0,107,0,105,0,115,0,104,0,10,0,45,0,32,0,50,0,48,0,57,0,50,0,52,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,76,0,97,0,116,0,105,0,110,0,32,0,49,0,47,0,79,0,112,0,101,0,110,0,32,0,83,0,121,0,115,0,116,0,101,0,109,0,32,0,40,0,49,0,48,0,52,0,55,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,10,0,45,0,32,0,50,0,48,0,57,0,51,0,50,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,40,0,74,0,73,0,83,0,32,0,48,0,50,0,48,0,56,0,45,0,49,0,57,0,57,0,48,0,32,0,97,0,110,0,100,0,32,0,48,0,50,0,49,0,50,0,45,0,49,0,57,0,57,0,48,0,41,0,10,0,45,0,32,0,50,0,48,0,57,0,51,0,54,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,40,0,71,0,66,0,50,0,51,0,49,0,50,0,41,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,40,0,71,0,66,0,50,0,51,0,49,0,50,0,45,0,56,0,48,0,41,0,10,0,45,0,32,0,50,0,48,0,57,0,52,0,57,0,32,0,75,0,111,0,114,0,101,0,97,0,110,0,32,0,87,0,97,0,110,0,115,0,117,0,110,0,103,0,10,0,45,0,32,0,50,0,49,0,48,0,50,0,53,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,67,0,121,0,114,0,105,0,108,0,108,0,105,0,99,0,32,0,83,0,101,0,114,0,98,0,105,0,97,0,110,0,45,0,66,0,117,0,108,0,103,0,97,0,114,0,105,0,97,0,110,0,10,0,45,0,32,0,50,0,49,0,48,0,50,0,55,0,32,0,69,0,120,0,116,0,101,0,110,0,100,0,101,0,100,0,47,0,69,0,120,0,116,0,32,0,65,0,108,0,112,0,104,0,97,0,32,0,76,0,111,0,119,0,101,0,114,0,99,0,97,0,115,0,101,0,10,0,45,0,32,0,50,0,49,0,56,0,54,0,54,0,32,0,85,0,107,0,114,0,97,0,105,0,110,0,105,0,97,0,110,0,32,0,40,0,75,0,79,0,73,0,56,0,45,0,85,0,41,0,59,0,32,0,67,0,121,0,114,0,105,0,108,0,108,0,105,0,99,0,32,0,40,0,75,0,79,0,73,0,56,0,45,0,85,0,41,0,10,0,45,0,32,0,50,0,57,0,48,0,48,0,49,0,32,0,69,0,117,0,114,0,111,0,112,0,97,0,32,0,51,0,10,0,45,0,32,0,51,0,56,0,53,0,57,0,56,0,32,0,73,0,83,0,79,0,32,0,56,0,56,0,53,0,57,0,45,0,56,0,32,0,72,0,101,0,98,0,114,0,101,0,119,0,59,0,32,0,72,0,101,0,98,0,114,0,101,0,119,0,32,0,40,0,73,0,83,0,79,0,45,0,76,0,111,0,103,0,105,0,99,0,97,0,108,0,41,0,10,0,45,0,32,0,53,0,48,0,50,0,50,0,48,0,32,0,73,0,83,0,79,0,32,0,50,0,48,0,50,0,50,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,119,0,105,0,116,0,104,0,32,0,110,0,111,0,32,0,104,0,97,0,108,0,102,0,119,0,105,0,100,0,116,0,104,0,32,0,75,0,97,0,116,0,97,0,107,0,97,0,110,0,97,0,59,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,40,0,74,0,73,0,83,0,41,0,10,0,45,0,32,0,53,0,48,0,50,0,50,0,49,0,32,0,73,0,83,0,79,0,32,0,50,0,48,0,50,0,50,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,119,0,105,0,116,0,104,0,32,0,104,0,97,0,108,0,102,0,119,0,105,0,100,0,116,0,104,0,32,0,75,0,97,0,116,0,97,0,107,0,97,0,110,0,97,0,59,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,40,0,74,0,73,0,83,0,45,0,65,0,108,0,108,0,111,0,119,0,32,0,49,0,32,0,98,0,121,0,116,0,101,0,32,0,75,0,97,0,110,0,97,0,41,0,10,0,45,0,32,0,53,0,48,0,50,0,50,0,50,0,32,0,73,0,83,0,79,0,32,0,50,0,48,0,50,0,50,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,74,0,73,0,83,0,32,0,88,0,32,0,48,0,50,0,48,0,49,0,45,0,49,0,57,0,56,0,57,0,59,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,40,0,74,0,73,0,83,0,45,0,65,0,108,0,108,0,111,0,119,0,32,0,49,0,32,0,98,0,121,0,116,0,101,0,32,0,75,0,97,0,110,0,97,0,32,0,45,0,32,0,83,0,79,0,47,0,83,0,73,0,41,0,10,0,45,0,32,0,53,0,48,0,50,0,50,0,53,0,32,0,73,0,83,0,79,0,32,0,50,0,48,0,50,0,50,0,32,0,75,0,111,0,114,0,101,0,97,0,110,0,10,0,45,0,32,0,53,0,48,0,50,0,50,0,55,0,32,0,73,0,83,0,79,0,32,0,50,0,48,0,50,0,50,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,40,0,73,0,83,0,79,0,32,0,50,0,48,0,50,0,50,0,41,0,10,0,45,0,32,0,53,0,49,0,57,0,51,0,50,0,32,0,69,0,85,0,67,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,10,0,45,0,32,0,53,0,49,0,57,0,51,0,54,0,32,0,69,0,85,0,67,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,40,0,69,0,85,0,67,0,41,0,10,0,45,0,32,0,53,0,49,0,57,0,52,0,57,0,32,0,69,0,85,0,67,0,32,0,75,0,111,0,114,0,101,0,97,0,110,0,10,0,45,0,32,0,53,0,50,0,57,0,51,0,54,0,32,0,72,0,90,0,45,0,71,0,66,0,50,0,51,0,49,0,50,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,40,0,72,0,90,0,41,0,10,0,45,0,32,0,53,0,52,0,57,0,51,0,54,0,32,0,87,0,105,0,110,0,100,0,111,0,119,0,115,0,32,0,88,0,80,0,32,0,97,0,110,0,100,0,32,0,108,0,97,0,116,0,101,0,114,0,58,0,32,0,71,0,66,0,49,0,56,0,48,0,51,0,48,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,40,0,52,0,32,0,98,0,121,0,116,0,101,0,41,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,40,0,71,0,66,0,49,0,56,0,48,0,51,0,48,0,41,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,50,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,68,0,101,0,118,0,97,0,110,0,97,0,103,0,97,0,114,0,105,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,51,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,66,0,101,0,110,0,103,0,97,0,108,0,105,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,52,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,84,0,97,0,109,0,105,0,108,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,53,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,84,0,101,0,108,0,117,0,103,0,117,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,54,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,65,0,115,0,115,0,97,0,109,0,101,0,115,0,101,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,55,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,79,0,114,0,105,0,121,0,97,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,56,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,75,0,97,0,110,0,110,0,97,0,100,0,97,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,57,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,77,0,97,0,108,0,97,0,121,0,97,0,108,0,97,0,109,0,10,0,45,0,32,0,53,0,55,0,48,0,49,0,48,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,71,0,117,0,106,0,97,0,114,0,97,0,116,0,105,0,10,0,45,0,32,0,53,0,55,0,48,0,49,0,49,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,80,0,117,0,110,0,106,0,97,0,98,0,105,0,10,0,10,0,96,0,96,0,96,0,62,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,10,0,55,0,48,0,56,0,44,0,44,0,49,0,10,0,55,0,50,0,48,0,44,0,44,0,49,0,10,0,56,0,53,0,56,0,44,0,44,0,49,0,10,0,56,0,55,0,48,0,44,0,44,0,49,0,10,0,49,0,48,0,52,0,55,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,48,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,49,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,50,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,51,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,52,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,53,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,54,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,55,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,56,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,57,0,44,0,44,0,49,0,10,0,49,0,51,0,54,0,49,0,44,0,44,0,50,0,10,0,49,0,48,0,48,0,48,0,49,0,44,0,44,0,50,0,10,0,49,0,48,0,48,0,48,0,50,0,44,0,44,0,50,0,10,0,49,0,48,0,48,0,48,0,51,0,44,0,44,0,50,0,10,0,49,0,48,0,48,0,48,0,52,0,44,0,44,0,49,0,10,0,49,0,48,0,48,0,48,0,53,0,44,0,44,0,49,0,10,0,49,0,48,0,48,0,48,0,56,0,44,0,44,0,50,0,10,0,49,0,48,0,48,0,49,0,48,0,44,0,44,0,49,0,10,0,49,0,48,0,48,0,49,0,55,0,44,0,44,0,49,0,10,0,49,0,48,0,48,0,50,0,49,0,44,0,44,0,49,0,10,0,49,0,48,0,48,0,56,0,50,0,44,0,44,0,49,0,10,0,50,0,48,0,48,0,48,0,48,0,44,0,44,0,50,0,10,0,50,0,48,0,48,0,48,0,49,0,44,0,44,0,50,0,10,0,50,0,48,0,48,0,48,0,50,0,44,0,44,0,50,0,10,0,50,0,48,0,48,0,48,0,51,0,44,0,44,0,50,0,10,0,50,0,48,0,48,0,48,0,52,0,44,0,44,0,50,0,10,0,50,0,48,0,48,0,48,0,53,0,44,0,44,0,50,0,10,0,50,0,48,0,49,0,48,0,53,0,44,0,44,0,49,0,10,0,50,0,48,0,49,0,48,0,54,0,44,0,44,0,49,0,10,0,50,0,48,0,49,0,48,0,55,0,44,0,44,0,49,0,10,0,50,0,48,0,49,0,48,0,56,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,54,0,49,0,44,0,44,0,50,0,10,0,50,0,48,0,50,0,54,0,57,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,55,0,51,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,55,0,55,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,55,0,56,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,56,0,48,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,56,0,52,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,56,0,53,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,57,0,48,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,57,0,55,0,44,0,44,0,49,0,10,0,50,0,48,0,52,0,50,0,48,0,44,0,44,0,49,0,10,0,50,0,48,0,52,0,50,0,51,0,44,0,44,0,49,0,10,0,50,0,48,0,52,0,50,0,52,0,44,0,44,0,49,0,10,0,50,0,48,0,56,0,51,0,51,0,44,0,44,0,49,0,10,0,50,0,48,0,56,0,51,0,56,0,44,0,44,0,49,0,10,0,50,0,48,0,56,0,54,0,54,0,44,0,44,0,49,0,10,0,50,0,48,0,56,0,55,0,49,0,44,0,44,0,49,0,10,0,50,0,48,0,56,0,56,0,48,0,44,0,44,0,49,0,10,0,50,0,48,0,57,0,48,0,53,0,44,0,44,0,49,0,10,0,50,0,48,0,57,0,50,0,52,0,44,0,44,0,49,0,10,0,50,0,48,0,57,0,51,0,50,0,44,0,44,0,50,0,10,0,50,0,48,0,57,0,51,0,54,0,44,0,44,0,50,0,10,0,50,0,48,0,57,0,52,0,57,0,44,0,44,0,50,0,10,0,50,0,49,0,48,0,50,0,53,0,44,0,44,0,49,0,10,0,50,0,49,0,48,0,50,0,55,0,44,0,44,0,49,0,10,0,50,0,49,0,56,0,54,0,54,0,44,0,44,0,49,0,10,0,50,0,57,0,48,0,48,0,49,0,44,0,44,0,49,0,10,0,51,0,56,0,53,0,57,0,56,0,44,0,44,0,49,0,10,0,53,0,48,0,50,0,50,0,48,0,44,0,44,0,50,0,10,0,53,0,48,0,50,0,50,0,49,0,44,0,44,0,50,0,10,0,53,0,48,0,50,0,50,0,50,0,44,0,44,0,50,0,10,0,53,0,48,0,50,0,50,0,53,0,44,0,44,0,50,0,10,0,53,0,48,0,50,0,50,0,55,0,44,0,44,0,50,0,10,0,53,0,49,0,57,0,51,0,50,0,44,0,44,0,50,0,10,0,53,0,49,0,57,0,51,0,54,0,44,0,44,0,50,0,10,0,53,0,49,0,57,0,52,0,57,0,44,0,44,0,50,0,10,0,53,0,50,0,57,0,51,0,54,0,44,0,44,0,50,0,10,0,53,0,52,0,57,0,51,0,54,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,50,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,51,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,52,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,53,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,54,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,55,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,56,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,57,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,49,0,48,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,49,0,49,0,44,0,44,0,50,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,102,0,111,0,108,0,108,0,111,0,119,0,105,0,110,0,103,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,100,0,101,0,112,0,101,0,110,0,100,0,101,0,110,0,99,0,105,0,101,0,115,0,32,0,102,0,111,0,114,0,32,0,86,0,105,0,115,0,117,0,97,0,108,0,32,0,70,0,111,0,120,0,80,0,114,0,111,0,58,0,10,0,10,0,45,0,32,0,54,0,50,0,48,0,32,0,77,0,97,0,122,0,111,0,118,0,105,0,97,0,32,0,40,0,80,0,111,0,108,0,105,0,115,0,104,0,41,0,32,0,77,0,83,0,45,0,68,0,79,0,83,0,10,0,45,0,32,0,56,0,57,0,53,0,32,0,75,0,97,0,109,0,101,0,110,0,105,0,99,0,107,0,253,0,32,0,40,0,67,0,122,0,101,0,99,0,104,0,41,0,32,0,77,0,83,0,45,0,68,0,79,0,83,0,10,0,10,0,96,0,96,0,96,0,62,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,10,0,54,0,50,0,48,0,44,0,44,0,49,0,10,0,56,0,57,0,53,0,44,0,44,0,49,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,107,0,110,0,111,0,119,0,110,0,32,0,109,0,105,0,115,0,115,0,105,0,110,0,103,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,101,0,110,0,117,0,109,0,101,0,114,0,97,0,116,0,101,0,100,0,32,0,105,0,110,0,32,0,116,0,104,0,101,0,32,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,10,0,10,0,35,0,35,0,32,0,66,0,117,0,105,0,108,0,100,0,105,0,110,0,103,0,32,0,78,0,111,0,116,0,101,0,115,0,10,0,10,0,84,0,104,0,101,0,32,0,115,0,99,0,114,0,105,0,112,0,116,0,32,0,96,0,109,0,97,0,107,0,101,0,46,0,115,0,104,0,96,0,32,0,40,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,100,0,32,0,108,0,97,0,116,0,101,0,114,0,41,0,32,0,119,0,105,0,108,0,108,0,32,0,103,0,101,0,116,0,32,0,116,0,104,0,101,0,115,0,101,0,32,0,102,0,105,0,108,0,101,0,115,0,32,0,97,0,110,0,100,0,32,0,109,0,97,0,115,0,115,0,97,0,103,0,101,0,32,0,116,0,104,0,101,0,32,0,100,0,97,0,116,0,97,0,10,0,40,0,112,0,114,0,105,0,110,0,116,0,105,0,110,0,103,0,32,0,99,0,111,0,100,0,101,0,45,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,32,0,112,0,97,0,105,0,114,0,115,0,41,0,46,0,32,0,32,0,84,0,104,0,101,0,32,0,101,0,118,0,101,0,110,0,116,0,117,0,97,0,108,0,32,0,116,0,97,0,98,0,108,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,100,0,114,0,111,0,112,0,112,0,101,0,100,0,32,0,105,0,110,0,32,0,116,0,104,0,101,0,32,0,112,0,97,0,116,0,104,0,115,0,10,0,96,0,46,0,47,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,47,0,60,0,67,0,79,0,68,0,69,0,80,0,65,0,71,0,69,0,62,0,46,0,84,0,66,0,76,0,96,0,46,0,32,0,32,0,70,0,111,0,114,0,32,0,101,0,120,0,97,0,109,0,112,0,108,0,101,0,44,0,32,0,116,0,104,0,101,0,32,0,108,0,97,0,115,0,116,0,32,0,49,0,48,0,32,0,108,0,105,0,110,0,101,0,115,0,32,0,111,0,102,0,32,0,96,0,49,0,48,0,48,0,48,0,48,0,46,0,84,0,66,0,76,0,96,0,32,0,97,0,114,0,101,0,10,0,10,0,96,0,96,0,96,0,62,0,10,0,48,0,120,0,70,0,54,0,9,0,48,0,120,0,48,0,50,0,67,0,54,0,10,0,48,0,120,0,70,0,55,0,9,0,48,0,120,0,48,0,50,0,68,0,67,0,10,0,48,0,120,0,70,0,56,0,9,0,48,0,120,0,48,0,48,0,65,0,70,0,10,0,48,0,120,0,70,0,57,0,9,0,48,0,120,0,48,0,50,0,68,0,56,0,10,0,48,0,120,0,70,0,65,0,9,0,48,0,120,0,48,0,50,0,68,0,57,0,10,0,48,0,120,0,70,0,66,0,9,0,48,0,120,0,48,0,50,0,68,0,65,0,10,0,48,0,120,0,70,0,67,0,9,0,48,0,120,0,48,0,48,0,66,0,56,0,10,0,48,0,120,0,70,0,68,0,9,0,48,0,120,0,48,0,50,0,68,0,68,0,10,0,48,0,120,0,70,0,69,0,9,0,48,0,120,0,48,0,50,0,68,0,66,0,10,0,48,0,120,0,70,0,70,0,9,0,48,0,120,0,48,0,50,0,67,0,55,0,10,0,96,0,96,0,96,0,10,0,10,0,119,0,104,0,105,0,99,0,104,0,32,0,105,0,109,0,112,0,108,0,105,0,101,0,115,0,32,0,116,0,104,0,97,0,116,0,32,0,99,0,111,0,100,0,101,0,32,0,48,0,120,0,70,0,54,0,32,0,105,0,115,0,32,0,96,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,48,0,120,0,48,0,50,0,67,0,54,0,41,0,96,0,32,0,97,0,110,0,100,0,32,0,118,0,105,0,99,0,101,0,32,0,118,0,101,0,114,0,115,0,97,0,46,0,10,0,10,0,35,0,35,0,32,0,87,0,105,0,110,0,100,0,111,0,119,0,115,0,45,0,100,0,101,0,112,0,101,0,110,0,100,0,101,0,110,0,116,0,32,0,98,0,117,0,105,0,108,0,100,0,32,0,115,0,116,0,101,0,112,0,10,0,10,0,84,0,111,0,32,0,98,0,117,0,105,0,108,0,100,0,32,0,116,0,104,0,101,0,32,0,115,0,111,0,117,0,114,0,99,0,101,0,115,0,32,0,111,0,110,0,32,0,119,0,105,0,110,0,100,0,111,0,119,0,115,0,44,0,32,0,99,0,111,0,110,0,115,0,117,0,108,0,116,0,32,0,96,0,100,0,111,0,116,0,110,0,101,0,116,0,47,0,77,0,97,0,107,0,101,0,69,0,110,0,99,0,111,0,100,0,105,0,110,0,103,0,46,0,99,0,115,0,96,0,46,0,10,0,10,0,65,0,102,0,116,0,101,0,114,0,32,0,115,0,97,0,118,0,105,0,110,0,103,0,32,0,116,0,104,0,101,0,32,0,115,0,116,0,97,0,110,0,100,0,97,0,114,0,100,0,32,0,111,0,117,0,116,0,112,0,117,0,116,0,32,0,116,0,111,0,32,0,96,0,111,0,117,0,116,0,96,0,44,0,32,0,97,0,32,0,115,0,105,0,109,0,112,0,108,0,101,0,32,0,97,0,119,0,107,0,32,0,115,0,99,0,114,0,105,0,112,0,116,0,32,0,40,0,96,0,100,0,111,0,116,0,110,0,101,0,116,0,46,0,115,0,104,0,96,0,41,0,32,0,116,0,97,0,107,0,101,0,115,0,32,0,99,0,97,0,114,0,101,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,32,0,114,0,101,0,115,0,116,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,100,0,111,0,116,0,110,0,101,0,116,0,46,0,115,0,104,0,10,0,35,0,33,0,47,0,98,0,105,0,110,0,47,0,98,0,97,0,115,0,104,0,10,0,105,0,102,0,32,0,91,0,32,0,33,0,32,0,45,0,101,0,32,0,100,0,111,0,116,0,110,0,101,0,116,0,47,0,111,0,117,0,116,0,32,0,93,0,59,0,32,0,116,0,104,0,101,0,110,0,32,0,101,0,120,0,105,0,116,0,59,0,32,0,102,0,105,0,10,0,60,0,100,0,111,0,116,0,110,0,101,0,116,0,47,0,111,0,117,0,116,0,32,0,116,0,114,0,32,0,45,0,115,0,32,0,39,0,32,0,39,0,32,0,39,0,92,0,116,0,39,0,32,0,124,0,32,0,97,0,119,0,107,0,32,0,39,0,78,0,70,0,62,0,50,0,32,0,123,0,105,0,102,0,40,0,111,0,117,0,116,0,102,0,105,0,108,0,101,0,41,0,32,0,99,0,108,0,111,0,115,0,101,0,40,0,111,0,117,0,116,0,102,0,105,0,108,0,101,0,41,0,59,0,32,0,111,0,117,0,116,0,102,0,105,0,108,0,101,0,61,0,34,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,47,0,34,0,32,0,36,0,49,0,32,0,34,0,46,0,84,0,66,0,76,0,34,0,125,0,32,0,78,0,70,0,61,0,61,0,50,0,32,0,123,0,112,0,114,0,105,0,110,0,116,0,32,0,62,0,32,0,111,0,117,0,116,0,102,0,105,0,108,0,101,0,125,0,39,0,10,0,96,0,96,0,96,0,10,0,10,0,35,0,32,0,66,0,117,0,105,0,108,0,100,0,105,0,110,0,103,0,32,0,116,0,104,0,101,0,32,0,115,0,99,0,114,0,105,0,112,0,116,0,10,0,10,0,96,0,109,0,97,0,107,0,101,0,46,0,110,0,106,0,115,0,96,0,32,0,116,0,97,0,107,0,101,0,115,0,32,0,97,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,32,0,97,0,114,0,103,0,117,0,109,0,101,0,110,0,116,0,44,0,32,0,114,0,101,0,97,0,100,0,115,0,32,0,116,0,104,0,101,0,32,0,99,0,111,0,114,0,114,0,101,0,115,0,112,0,111,0,110,0,100,0,105,0,110,0,103,0,32,0,116,0,97,0,98,0,108,0,101,0,32,0,102,0,105,0,108,0,101,0,32,0,97,0,110,0,100,0,10,0,103,0,101,0,110,0,101,0,114,0,97,0,116,0,101,0,115,0,32,0,74,0,83,0,32,0,99,0,111,0,100,0,101,0,32,0,102,0,111,0,114,0,32,0,101,0,110,0,99,0,111,0,100,0,105,0,110,0,103,0,32,0,97,0,110,0,100,0,32,0,100,0,101,0,99,0,111,0,100,0,105,0,110,0,103,0,58,0,10,0,10,0,35,0,35,0,32,0,82,0,97,0,119,0,32,0,67,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,10,0,10,0,96,0,96,0,96,0,62,0,109,0,97,0,107,0,101,0,46,0,110,0,106,0,115,0,10,0,35,0,33,0,47,0,117,0,115,0,114,0,47,0,98,0,105,0,110,0,47,0,101,0,110,0,118,0,32,0,110,0,111,0,100,0,101,0,10,0,118,0,97,0,114,0,32,0,97,0,114,0,103,0,118,0,32,0,61,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,46,0,97,0,114,0,103,0,118,0,46,0,115,0,108,0,105,0,99,0,101,0,40,0,49,0,41,0,44,0,32,0,102,0,115,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,102,0,115,0,39,0,41,0,59,0,10,0,105,0,102,0,40,0,97,0,114,0,103,0,118,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,60,0,32,0,50,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,99,0,111,0,110,0,115,0,111,0,108,0,101,0,46,0,101,0,114,0,114,0,111,0,114,0,40,0,34,0,117,0,115,0,97,0,103,0,101,0,58,0,32,0,109,0,97,0,107,0,101,0,46,0,110,0,106,0,115,0,32,0,60,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,95,0,105,0,110,0,100,0,101,0,120,0,62,0,32,0,91,0,118,0,97,0,114,0,105,0,97,0,98,0,108,0,101,0,93,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,46,0,101,0,120,0,105,0,116,0,40,0,50,0,50,0,41,0,59,0,32,0,47,0,42,0,32,0,69,0,73,0,78,0,86,0,65,0,76,0,32,0,42,0,47,0,10,0,125,0,10,0,10,0,118,0,97,0,114,0,32,0,99,0,112,0,32,0,61,0,32,0,97,0,114,0,103,0,118,0,91,0,49,0,93,0,59,0,10,0,118,0,97,0,114,0,32,0,106,0,115,0,118,0,97,0,114,0,32,0,61,0,32,0,97,0,114,0,103,0,118,0,91,0,50,0,93,0,32,0,124,0,124,0,32,0,34,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,34,0,59,0,10,0,118,0,97,0,114,0,32,0,120,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,34,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,47,0,34,0,32,0,43,0,32,0,99,0,112,0,32,0,43,0,32,0,34,0,46,0,84,0,66,0,76,0,34,0,44,0,34,0,117,0,116,0,102,0,56,0,34,0,41,0,59,0,10,0,118,0,97,0,114,0,32,0,109,0,97,0,120,0,99,0,112,0,32,0,61,0,32,0,48,0,59,0,10,0,10,0,118,0,97,0,114,0,32,0,121,0,32,0,61,0,32,0,120,0,46,0,115,0,112,0,108,0,105,0,116,0,40,0,34,0,92,0,110,0,34,0,41,0,46,0,109,0,97,0,112,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,122,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,119,0,32,0,61,0,32,0,122,0,46,0,115,0,112,0,108,0,105,0,116,0,40,0,34,0,92,0,116,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,119,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,60,0,32,0,50,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,119,0,59,0,10,0,32,0,32,0,32,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,91,0,78,0,117,0,109,0,98,0,101,0,114,0,40,0,119,0,91,0,48,0,93,0,41,0,44,0,32,0,78,0,117,0,109,0,98,0,101,0,114,0,40,0,119,0,91,0,49,0,93,0,41,0,93,0,59,0,10,0,125,0,41,0,46,0,102,0,105,0,108,0,116,0,101,0,114,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,122,0,41,0,32,0,123,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,62,0,32,0,49,0,59,0,32,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,68,0,66,0,67,0,83,0,32,0,97,0,110,0,100,0,32,0,83,0,66,0,67,0,83,0,32,0,99,0,111,0,100,0,101,0,32,0,103,0,101,0,110,0,101,0,114,0,97,0,116,0,105,0,111,0,110,0,32,0,115,0,116,0,114,0,97,0,116,0,101,0,103,0,105,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,100,0,105,0,102,0,102,0,101,0,114,0,101,0,110,0,116,0,46,0,32,0,32,0,84,0,104,0,101,0,32,0,109,0,97,0,120,0,105,0,109,0,117,0,109,0,32,0,99,0,111,0,100,0,101,0,32,0,105,0,115,0,10,0,117,0,115,0,101,0,100,0,32,0,116,0,111,0,32,0,100,0,105,0,115,0,116,0,105,0,110,0,103,0,117,0,105,0,115,0,104,0,32,0,40,0,109,0,97,0,120,0,32,0,48,0,120,0,70,0,70,0,32,0,102,0,111,0,114,0,32,0,83,0,66,0,67,0,83,0,41,0,46,0,10,0,10,0,96,0,96,0,96,0,10,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,121,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,105,0,102,0,40,0,121,0,91,0,105,0,93,0,91,0,48,0,93,0,32,0,62,0,32,0,109,0,97,0,120,0,99,0,112,0,41,0,32,0,109,0,97,0,120,0,99,0,112,0,32,0,61,0,32,0,121,0,91,0,105,0,93,0,91,0,48,0,93,0,59,0,10,0,10,0,118,0,97,0,114,0,32,0,101,0,110,0,99,0,32,0,61,0,32,0,123,0,125,0,44,0,32,0,100,0,101,0,99,0,32,0,61,0,32,0,40,0,109,0,97,0,120,0,99,0,112,0,32,0,60,0,32,0,50,0,53,0,54,0,32,0,63,0,32,0,91,0,93,0,32,0,58,0,32,0,123,0,125,0,41,0,59,0,10,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,121,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,100,0,101,0,99,0,91,0,121,0,91,0,105,0,93,0,91,0,48,0,93,0,93,0,32,0,61,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,121,0,91,0,105,0,93,0,91,0,49,0,93,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,101,0,110,0,99,0,91,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,121,0,91,0,105,0,93,0,91,0,49,0,93,0,41,0,93,0,32,0,61,0,32,0,121,0,91,0,105,0,93,0,91,0,48,0,93,0,59,0,10,0,125,0,10,0,10,0,118,0,97,0,114,0,32,0,111,0,100,0,101,0,99,0,44,0,32,0,111,0,101,0,110,0,99,0,44,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,59,0,10,0,105,0,102,0,40,0,109,0,97,0,120,0,99,0,112,0,32,0,60,0,32,0,50,0,53,0,54,0,41,0,32,0,123,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,32,0,99,0,104,0,97,0,114,0,97,0,99,0,116,0,101,0,114,0,32,0,96,0,48,0,120,0,70,0,70,0,70,0,68,0,96,0,32,0,40,0,82,0,69,0,80,0,76,0,65,0,67,0,69,0,77,0,69,0,78,0,84,0,32,0,67,0,72,0,65,0,82,0,65,0,67,0,84,0,69,0,82,0,41,0,32,0,105,0,115,0,32,0,117,0,115,0,101,0,100,0,32,0,97,0,115,0,32,0,97,0,32,0,112,0,108,0,97,0,99,0,101,0,104,0,111,0,108,0,100,0,101,0,114,0,10,0,102,0,111,0,114,0,32,0,99,0,104,0,97,0,114,0,97,0,99,0,116,0,101,0,114,0,115,0,32,0,116,0,104,0,97,0,116,0,32,0,97,0,114,0,101,0,32,0,110,0,111,0,116,0,32,0,115,0,112,0,101,0,99,0,105,0,102,0,105,0,101,0,100,0,32,0,105,0,110,0,32,0,116,0,104,0,101,0,32,0,109,0,97,0,112,0,32,0,40,0,102,0,111,0,114,0,32,0,101,0,120,0,97,0,109,0,112,0,108,0,101,0,44,0,32,0,96,0,48,0,120,0,70,0,48,0,96,0,32,0,105,0,115,0,32,0,110,0,111,0,116,0,32,0,105,0,110,0,10,0,99,0,111,0,100,0,101,0,32,0,112,0,97,0,103,0,101,0,32,0,49,0,48,0,48,0,48,0,48,0,41,0,46,0,10,0,10,0,70,0,111,0,114,0,32,0,83,0,66,0,67,0,83,0,44,0,32,0,116,0,104,0,101,0,32,0,105,0,100,0,101,0,97,0,32,0,105,0,115,0,32,0,116,0,111,0,32,0,101,0,109,0,98,0,101,0,100,0,32,0,97,0,32,0,114,0,97,0,119,0,32,0,115,0,116,0,114,0,105,0,110,0,103,0,32,0,119,0,105,0,116,0,104,0,32,0,116,0,104,0,101,0,32,0,99,0,111,0,110,0,116,0,101,0,110,0,116,0,115,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,32,0,50,0,53,0,54,0,32,0,99,0,111,0,100,0,101,0,115,0,46,0,10,0,84,0,104,0,101,0,32,0,96,0,100,0,101,0,99,0,96,0,32,0,102,0,105,0,101,0,108,0,100,0,32,0,105,0,115,0,32,0,109,0,101,0,114,0,101,0,108,0,121,0,32,0,97,0,32,0,115,0,112,0,108,0,105,0,116,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,32,0,115,0,116,0,114,0,105,0,110,0,103,0,44,0,32,0,97,0,110,0,100,0,32,0,96,0,101,0,110,0,99,0,96,0,32,0,105,0,115,0,32,0,97,0,110,0,32,0,101,0,118,0,101,0,114,0,115,0,105,0,111,0,110,0,58,0,10,0,10,0,96,0,96,0,96,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,50,0,53,0,54,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,100,0,101,0,99,0,91,0,105,0,93,0,32,0,61,0,61,0,61,0,32,0,34,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,34,0,41,0,32,0,100,0,101,0,99,0,91,0,105,0,93,0,32,0,61,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,48,0,120,0,70,0,70,0,70,0,68,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,111,0,100,0,101,0,99,0,32,0,61,0,32,0,74,0,83,0,79,0,78,0,46,0,115,0,116,0,114,0,105,0,110,0,103,0,105,0,102,0,121,0,40,0,100,0,101,0,99,0,46,0,106,0,111,0,105,0,110,0,40,0,34,0,34,0,41,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,32,0,61,0,32,0,39,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,32,0,118,0,97,0,114,0,32,0,100,0,32,0,61,0,32,0,39,0,32,0,43,0,32,0,111,0,100,0,101,0,99,0,32,0,43,0,32,0,39,0,44,0,32,0,68,0,32,0,61,0,32,0,91,0,93,0,44,0,32,0,101,0,32,0,61,0,32,0,123,0,125,0,59,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,61,0,48,0,59,0,105,0,33,0,61,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,43,0,43,0,105,0,41,0,32,0,123,0,32,0,105,0,102,0,40,0,100,0,46,0,99,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,65,0,116,0,40,0,105,0,41,0,32,0,33,0,61,0,61,0,32,0,48,0,120,0,70,0,70,0,70,0,68,0,41,0,32,0,101,0,91,0,100,0,91,0,105,0,93,0,93,0,32,0,61,0,32,0,105,0,59,0,32,0,68,0,91,0,105,0,93,0,32,0,61,0,32,0,100,0,46,0,99,0,104,0,97,0,114,0,65,0,116,0,40,0,105,0,41,0,59,0,32,0,125,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,123,0,34,0,101,0,110,0,99,0,34,0,58,0,32,0,101,0,44,0,32,0,34,0,100,0,101,0,99,0,34,0,58,0,32,0,68,0,32,0,125,0,59,0,32,0,125,0,41,0,40,0,41,0,59,0,39,0,59,0,10,0,125,0,32,0,101,0,108,0,115,0,101,0,32,0,123,0,10,0,96,0,96,0,96,0,10,0,10,0,68,0,66,0,67,0,83,0,32,0,105,0,115,0,32,0,115,0,105,0,109,0,105,0,108,0,97,0,114,0,44,0,32,0,101,0,120,0,99,0,101,0,112,0,116,0,32,0,116,0,104,0,97,0,116,0,32,0,116,0,104,0,101,0,32,0,115,0,112,0,97,0,99,0,101,0,32,0,105,0,115,0,32,0,115,0,108,0,105,0,99,0,101,0,100,0,32,0,105,0,110,0,116,0,111,0,32,0,50,0,53,0,54,0,45,0,98,0,121,0,116,0,101,0,32,0,99,0,104,0,117,0,110,0,107,0,115,0,32,0,40,0,115,0,116,0,114,0,105,0,110,0,103,0,115,0,10,0,97,0,114,0,101,0,32,0,111,0,110,0,108,0,121,0,32,0,103,0,101,0,110,0,101,0,114,0,97,0,116,0,101,0,100,0,32,0,102,0,111,0,114,0,32,0,116,0,104,0,111,0,115,0,101,0,32,0,104,0,105,0,103,0,104,0,45,0,98,0,121,0,116,0,101,0,115,0,32,0,114,0,101,0,112,0,114,0,101,0,115,0,101,0,110,0,116,0,101,0,100,0,32,0,105,0,110,0,32,0,116,0,104,0,101,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,41,0,46,0,10,0,10,0,84,0,104,0,101,0,32,0,115,0,116,0,114,0,97,0,116,0,101,0,103,0,121,0,32,0,105,0,115,0,32,0,116,0,111,0,32,0,99,0,111,0,110,0,115,0,116,0,114,0,117,0,99,0,116,0,32,0,97,0,110,0,32,0,97,0,114,0,114,0,97,0,121,0,45,0,111,0,102,0,45,0,97,0,114,0,114,0,97,0,121,0,115,0,32,0,115,0,111,0,32,0,116,0,104,0,97,0,116,0,32,0,96,0,100,0,100,0,91,0,104,0,105,0,103,0,104,0,93,0,91,0,108,0,111,0,119,0,93,0,96,0,32,0,105,0,115,0,32,0,116,0,104,0,101,0,10,0,99,0,104,0,97,0,114,0,97,0,99,0,116,0,101,0,114,0,32,0,97,0,115,0,115,0,111,0,99,0,105,0,97,0,116,0,101,0,100,0,32,0,119,0,105,0,116,0,104,0,32,0,116,0,104,0,101,0,32,0,99,0,111,0,100,0,101,0,46,0,32,0,32,0,84,0,104,0,105,0,115,0,32,0,97,0,114,0,114,0,97,0,121,0,32,0,105,0,115,0,32,0,99,0,111,0,109,0,98,0,105,0,110,0,101,0,100,0,32,0,97,0,116,0,32,0,114,0,117,0,110,0,116,0,105,0,109,0,101,0,32,0,116,0,111,0,32,0,121,0,105,0,101,0,108,0,100,0,10,0,116,0,104,0,101,0,32,0,99,0,111,0,109,0,112,0,108,0,101,0,116,0,101,0,32,0,100,0,101,0,99,0,111,0,100,0,105,0,110,0,103,0,32,0,111,0,98,0,106,0,101,0,99,0,116,0,32,0,40,0,97,0,110,0,100,0,32,0,116,0,104,0,101,0,32,0,101,0,110,0,99,0,111,0,100,0,105,0,110,0,103,0,32,0,111,0,98,0,106,0,101,0,99,0,116,0,32,0,105,0,115,0,32,0,97,0,110,0,32,0,101,0,118,0,101,0,114,0,115,0,105,0,111,0,110,0,41,0,58,0,10,0,10,0,96,0,96,0,96,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,100,0,100,0,32,0,61,0,32,0,91,0,93,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,105,0,110,0,32,0,100,0,101,0,99,0,41,0,32,0,105,0,102,0,40,0,100,0,101,0,99,0,46,0,104,0,97,0,115,0,79,0,119,0,110,0,80,0,114,0,111,0,112,0,101,0,114,0,116,0,121,0,40,0,105,0,41,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,100,0,100,0,91,0,105,0,32,0,62,0,62,0,32,0,56,0,93,0,32,0,61,0,61,0,61,0,32,0,34,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,34,0,41,0,32,0,100,0,100,0,91,0,105,0,32,0,62,0,62,0,32,0,56,0,93,0,32,0,61,0,32,0,91,0,93,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,100,0,100,0,91,0,105,0,32,0,62,0,62,0,32,0,56,0,93,0,91,0,105,0,32,0,37,0,32,0,50,0,53,0,54,0,93,0,32,0,61,0,32,0,100,0,101,0,99,0,91,0,105,0,93,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,32,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,32,0,61,0,32,0,39,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,32,0,118,0,97,0,114,0,32,0,100,0,32,0,61,0,32,0,91,0,93,0,44,0,32,0,101,0,32,0,61,0,32,0,123,0,125,0,44,0,32,0,68,0,32,0,61,0,32,0,91,0,93,0,44,0,32,0,106,0,59,0,92,0,110,0,39,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,50,0,53,0,54,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,105,0,102,0,40,0,100,0,100,0,91,0,105,0,93,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,106,0,32,0,61,0,32,0,48,0,59,0,32,0,106,0,32,0,33,0,61,0,32,0,50,0,53,0,54,0,59,0,32,0,43,0,43,0,106,0,41,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,100,0,100,0,91,0,105,0,93,0,91,0,106,0,93,0,32,0,61,0,61,0,61,0,32,0,34,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,34,0,41,0,32,0,100,0,100,0,91,0,105,0,93,0,91,0,106,0,93,0,32,0,61,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,48,0,120,0,70,0,70,0,70,0,68,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,32,0,43,0,61,0,32,0,39,0,68,0,91,0,39,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,39,0,93,0,32,0,61,0,32,0,39,0,32,0,43,0,32,0,74,0,83,0,79,0,78,0,46,0,115,0,116,0,114,0,105,0,110,0,103,0,105,0,102,0,121,0,40,0,100,0,100,0,91,0,105,0,93,0,46,0,106,0,111,0,105,0,110,0,40,0,34,0,34,0,41,0,41,0,32,0,43,0,32,0,39,0,46,0,115,0,112,0,108,0,105,0,116,0,40,0,34,0,34,0,41,0,59,0,92,0,110,0,39,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,32,0,43,0,61,0,32,0,39,0,102,0,111,0,114,0,40,0,106,0,32,0,61,0,32,0,48,0,59,0,32,0,106,0,32,0,33,0,61,0,32,0,68,0,91,0,39,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,39,0,93,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,32,0,43,0,43,0,106,0,41,0,32,0,105,0,102,0,40,0,68,0,91,0,39,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,39,0,93,0,91,0,106,0,93,0,46,0,99,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,65,0,116,0,40,0,48,0,41,0,32,0,33,0,61,0,61,0,32,0,48,0,120,0,70,0,70,0,70,0,68,0,41,0,32,0,123,0,32,0,101,0,91,0,68,0,91,0,39,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,39,0,93,0,91,0,106,0,93,0,93,0,32,0,61,0,32,0,39,0,32,0,43,0,32,0,40,0,105,0,42,0,50,0,53,0,54,0,41,0,32,0,43,0,32,0,39,0,32,0,43,0,32,0,106,0,59,0,32,0,100,0,91,0,39,0,32,0,43,0,32,0,40,0,105,0,42,0,50,0,53,0,54,0,41,0,32,0,43,0,32,0,39,0,32,0,43,0,32,0,106,0,93,0,32,0,61,0,32,0,68,0,91,0,39,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,39,0,93,0,91,0,106,0,93,0,59,0,125,0,92,0,110,0,39,0,10,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,32,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,32,0,43,0,61,0,32,0,39,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,123,0,34,0,101,0,110,0,99,0,34,0,58,0,32,0,101,0,44,0,32,0,34,0,100,0,101,0,99,0,34,0,58,0,32,0,100,0,32,0,125,0,59,0,32,0,125,0,41,0,40,0,41,0,59,0,39,0,59,0,10,0,125,0,10,0,99,0,111,0,110,0,115,0,111,0,108,0,101,0,46,0,108,0,111,0,103,0,40,0,106,0,115,0,118,0,97,0,114,0,32,0,43,0,32,0,34,0,91,0,34,0,32,0,43,0,32,0,99,0,112,0,32,0,43,0,32,0,34,0,93,0,32,0,61,0,32,0,34,0,32,0,43,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,41,0,59,0,10,0,10,0,96,0,96,0,96,0,10,0,10,0,96,0,109,0,97,0,107,0,101,0,46,0,115,0,104,0,96,0,32,0,103,0,101,0,110,0,101,0,114,0,97,0,116,0,101,0,115,0,32,0,116,0,104,0,101,0,32,0,116,0,97,0,98,0,108,0,101,0,115,0,32,0,117,0,115,0,101,0,100,0,32,0,98,0,121,0,32,0,96,0,109,0,97,0,107,0,101,0,46,0,110,0,106,0,115,0,96,0,46,0,32,0,32,0,84,0,104,0,101,0,32,0,114,0,97,0,119,0,32,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,32,0,84,0,88,0,84,0,32,0,102,0,105,0,108,0,101,0,115,0,10,0,97,0,114,0,101,0,32,0,99,0,111,0,108,0,117,0,109,0,110,0,97,0,114,0,58,0,32,0,96,0,99,0,111,0,100,0,101,0,32,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,32,0,35,0,99,0,111,0,109,0,109,0,101,0,110,0,116,0,115,0,96,0,46,0,32,0,32,0,70,0,111,0,114,0,32,0,101,0,120,0,97,0,109,0,112,0,108,0,101,0,44,0,32,0,116,0,104,0,101,0,32,0,108,0,97,0,115,0,116,0,32,0,49,0,48,0,32,0,108,0,105,0,110,0,101,0,115,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,10,0,116,0,101,0,120,0,116,0,32,0,102,0,105,0,108,0,101,0,32,0,82,0,79,0,77,0,65,0,78,0,46,0,84,0,88,0,84,0,32,0,40,0,102,0,111,0,114,0,32,0,67,0,80,0,32,0,49,0,48,0,48,0,48,0,48,0,41,0,32,0,97,0,114,0,101,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,10,0,48,0,120,0,70,0,54,0,9,0,48,0,120,0,48,0,50,0,67,0,54,0,9,0,35,0,77,0,79,0,68,0,73,0,70,0,73,0,69,0,82,0,32,0,76,0,69,0,84,0,84,0,69,0,82,0,32,0,67,0,73,0,82,0,67,0,85,0,77,0,70,0,76,0,69,0,88,0,32,0,65,0,67,0,67,0,69,0,78,0,84,0,10,0,48,0,120,0,70,0,55,0,9,0,48,0,120,0,48,0,50,0,68,0,67,0,9,0,35,0,83,0,77,0,65,0,76,0,76,0,32,0,84,0,73,0,76,0,68,0,69,0,10,0,48,0,120,0,70,0,56,0,9,0,48,0,120,0,48,0,48,0,65,0,70,0,9,0,35,0,77,0,65,0,67,0,82,0,79,0,78,0,10,0,48,0,120,0,70,0,57,0,9,0,48,0,120,0,48,0,50,0,68,0,56,0,9,0,35,0,66,0,82,0,69,0,86,0,69,0,10,0,48,0,120,0,70,0,65,0,9,0,48,0,120,0,48,0,50,0,68,0,57,0,9,0,35,0,68,0,79,0,84,0,32,0,65,0,66,0,79,0,86,0,69,0,10,0,48,0,120,0,70,0,66,0,9,0,48,0,120,0,48,0,50,0,68,0,65,0,9,0,35,0,82,0,73,0,78,0,71,0,32,0,65,0,66,0,79,0,86,0,69,0,10,0,48,0,120,0,70,0,67,0,9,0,48,0,120,0,48,0,48,0,66,0,56,0,9,0,35,0,67,0,69,0,68,0,73,0,76,0,76,0,65,0,10,0,48,0,120,0,70,0,68,0,9,0,48,0,120,0,48,0,50,0,68,0,68,0,9,0,35,0,68,0,79,0,85,0,66,0,76,0,69,0,32,0,65,0,67,0,85,0,84,0,69,0,32,0,65,0,67,0,67,0,69,0,78,0,84,0,10,0,48,0,120,0,70,0,69,0,9,0,48,0,120,0,48,0,50,0,68,0,66,0,9,0,35,0,79,0,71,0,79,0,78,0,69,0,75,0,10,0,48,0,120,0,70,0,70,0,9,0,48,0,120,0,48,0,50,0,67,0,55,0,9,0,35,0,67,0,65,0,82,0,79,0,78,0,10,0,96,0,96,0,96,0,10,0,10,0,73,0,110,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,105,0,110,0,103,0,32,0,116,0,104,0,101,0,32,0,100,0,97,0,116,0,97,0,44,0,32,0,116,0,104,0,101,0,32,0,99,0,111,0,109,0,109,0,101,0,110,0,116,0,115,0,32,0,40,0,97,0,102,0,116,0,101,0,114,0,32,0,116,0,104,0,101,0,32,0,96,0,35,0,96,0,41,0,32,0,97,0,114,0,101,0,32,0,115,0,116,0,114,0,105,0,112,0,112,0,101,0,100,0,32,0,97,0,110,0,100,0,32,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,10,0,101,0,108,0,101,0,109,0,101,0,110,0,116,0,115,0,32,0,40,0,108,0,105,0,107,0,101,0,32,0,96,0,48,0,120,0,55,0,70,0,96,0,32,0,102,0,111,0,114,0,32,0,67,0,80,0,32,0,49,0,48,0,48,0,48,0,48,0,41,0,32,0,97,0,114,0,101,0,32,0,114,0,101,0,109,0,111,0,118,0,101,0,100,0,46,0,10,0,10,0,96,0,96,0,96,0,62,0,109,0,97,0,107,0,101,0,46,0,115,0,104,0,10,0,35,0,33,0,47,0,98,0,105,0,110,0,47,0,98,0,97,0,115,0,104,0,10,0,73,0,78,0,70,0,73,0,76,0,69,0,61,0,36,0,123,0,49,0,58,0,45,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,125,0,10,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,61,0,36,0,123,0,50,0,58,0,45,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,106,0,115,0,125,0,10,0,74,0,83,0,86,0,65,0,82,0,61,0,36,0,123,0,51,0,58,0,45,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,125,0,10,0,86,0,69,0,82,0,83,0,73,0,79,0,78,0,61,0,36,0,40,0,99,0,97,0,116,0,32,0,112,0,97,0,99,0,107,0,97,0,103,0,101,0,46,0,106,0,115,0,111,0,110,0,32,0,124,0,32,0,103,0,114,0,101,0,112,0,32,0,118,0,101,0,114,0,115,0,105,0,111,0,110,0,32,0,124,0,32,0,116,0,114,0,32,0,45,0,100,0,99,0,32,0,91,0,48,0,45,0,57,0,46,0,93,0,41,0,10,0,10,0,109,0,107,0,100,0,105,0,114,0,32,0,45,0,112,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,32,0,98,0,105,0,116,0,115,0,10,0,114,0,109,0,32,0,45,0,102,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,101,0,99,0,104,0,111,0,32,0,34,0,47,0,42,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,32,0,40,0,67,0,41,0,32,0,50,0,48,0,49,0,51,0,45,0,50,0,48,0,49,0,52,0,32,0,83,0,104,0,101,0,101,0,116,0,74,0,83,0,32,0,45,0,45,0,32,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,115,0,104,0,101,0,101,0,116,0,106,0,115,0,46,0,99,0,111,0,109,0,32,0,42,0,47,0,34,0,32,0,62,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,101,0,99,0,104,0,111,0,32,0,34,0,47,0,42,0,106,0,115,0,104,0,105,0,110,0,116,0,32,0,45,0,87,0,49,0,48,0,48,0,32,0,42,0,47,0,34,0,32,0,62,0,62,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,101,0,99,0,104,0,111,0,32,0,34,0,118,0,97,0,114,0,32,0,36,0,74,0,83,0,86,0,65,0,82,0,32,0,61,0,32,0,123,0,118,0,101,0,114,0,115,0,105,0,111,0,110,0,58,0,92,0,34,0,36,0,86,0,69,0,82,0,83,0,73,0,79,0,78,0,92,0,34,0,125,0,59,0,34,0,32,0,62,0,62,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,105,0,102,0,32,0,91,0,32,0,45,0,101,0,32,0,100,0,111,0,116,0,110,0,101,0,116,0,46,0,115,0,104,0,32,0,93,0,59,0,32,0,116,0,104,0,101,0,110,0,32,0,98,0,97,0,115,0,104,0,32,0,100,0,111,0,116,0,110,0,101,0,116,0,46,0,115,0,104,0,59,0,32,0,102,0,105,0,10,0,97,0,119,0,107,0,32,0,45,0,70,0,44,0,32,0,39,0,123,0,112,0,114,0,105,0,110,0,116,0,32,0,36,0,49,0,44,0,32,0,36,0,50,0,44,0,32,0,36,0,51,0,125,0,39,0,32,0,36,0,73,0,78,0,70,0,73,0,76,0,69,0,32,0,124,0,32,0,119,0,104,0,105,0,108,0,101,0,32,0,114,0,101,0,97,0,100,0,32,0,99,0,112,0,32,0,117,0,114,0,108,0,32,0,99,0,112,0,116,0,121,0,112,0,101,0,59,0,32,0,100,0,111,0,10,0,32,0,32,0,32,0,32,0,101,0,99,0,104,0,111,0,32,0,36,0,99,0,112,0,32,0,36,0,117,0,114,0,108,0,10,0,32,0,32,0,32,0,32,0,105,0,102,0,32,0,91,0,32,0,33,0,32,0,45,0,101,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,47,0,36,0,99,0,112,0,46,0,84,0,66,0,76,0,32,0,93,0,59,0,32,0,116,0,104,0,101,0,110,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,117,0,114,0,108,0,32,0,36,0,117,0,114,0,108,0,32,0,124,0,32,0,115,0,101,0,100,0,32,0,39,0,115,0,47,0,35,0,46,0,42,0,47,0,47,0,103,0,39,0,32,0,124,0,32,0,97,0,119,0,107,0,32,0,39,0,78,0,70,0,61,0,61,0,50,0,39,0,32,0,62,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,47,0,36,0,99,0,112,0,46,0,84,0,66,0,76,0,10,0,32,0,32,0,32,0,32,0,102,0,105,0,10,0,32,0,32,0,32,0,32,0,101,0,99,0,104,0,111,0,32,0,34,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,36,0,74,0,83,0,86,0,65,0,82,0,32,0,61,0,61,0,61,0,32,0,39,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,39,0,41,0,32,0,36,0,74,0,83,0,86,0,65,0,82,0,32,0,61,0,32,0,123,0,125,0,59,0,34,0,32,0,62,0,32,0,98,0,105,0,116,0,115,0,47,0,36,0,99,0,112,0,46,0,106,0,115,0,46,0,116,0,109,0,112,0,10,0,32,0,32,0,32,0,32,0,110,0,111,0,100,0,101,0,32,0,109,0,97,0,107,0,101,0,46,0,110,0,106,0,115,0,32,0,36,0,99,0,112,0,32,0,36,0,74,0,83,0,86,0,65,0,82,0,32,0,124,0,32,0,116,0,101,0,101,0,32,0,45,0,97,0,32,0,98,0,105,0,116,0,115,0,47,0,36,0,99,0,112,0,46,0,106,0,115,0,46,0,116,0,109,0,112,0,32,0,62,0,62,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,32,0,32,0,32,0,32,0,115,0,101,0,100,0,32,0,39,0,115,0,47,0,34,0,92,0,40,0,91,0,48,0,45,0,57,0,93,0,43,0,92,0,41,0,34,0,58,0,47,0,92,0,49,0,58,0,47,0,103,0,39,0,32,0,60,0,98,0,105,0,116,0,115,0,47,0,36,0,99,0,112,0,46,0,106,0,115,0,46,0,116,0,109,0,112,0,32,0,62,0,98,0,105,0,116,0,115,0,47,0,36,0,99,0,112,0,46,0,106,0,115,0,10,0,32,0,32,0,32,0,32,0,114,0,109,0,32,0,45,0,102,0,32,0,98,0,105,0,116,0,115,0,47,0,36,0,99,0,112,0,46,0,106,0,115,0,46,0,116,0,109,0,112,0,10,0,100,0,111,0,110,0,101,0,10,0,101,0,99,0,104,0,111,0,32,0,34,0,105,0,102,0,32,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,109,0,111,0,100,0,117,0,108,0,101,0,32,0,33,0,61,0,61,0,32,0,39,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,39,0,32,0,38,0,38,0,32,0,109,0,111,0,100,0,117,0,108,0,101,0,46,0,101,0,120,0,112,0,111,0,114,0,116,0,115,0,41,0,32,0,109,0,111,0,100,0,117,0,108,0,101,0,46,0,101,0,120,0,112,0,111,0,114,0,116,0,115,0,32,0,61,0,32,0,36,0,74,0,83,0,86,0,65,0,82,0,59,0,34,0,32,0,62,0,62,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,115,0,101,0,100,0,32,0,39,0,115,0,47,0,34,0,92,0,40,0,91,0,48,0,45,0,57,0,93,0,43,0,92,0,41,0,34,0,58,0,47,0,92,0,49,0,58,0,47,0,103,0,39,0,32,0,60,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,32,0,62,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,10,0,114,0,109,0,32,0,45,0,102,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,96,0,96,0,96,0,10,0,10,0,35,0,35,0,32,0,85,0,116,0,105,0,108,0,105,0,116,0,105,0,101,0,115,0,10,0,10,0,84,0,104,0,101,0,32,0,101,0,110,0,99,0,111,0,100,0,101,0,32,0,97,0,110,0,100,0,32,0,100,0,101,0,99,0,111,0,100,0,101,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,115,0,32,0,97,0,114,0,101,0,32,0,107,0,101,0,112,0,116,0,32,0,105,0,110,0,32,0,97,0,32,0,115,0,101,0,112,0,97,0,114,0,97,0,116,0,101,0,32,0,115,0,99,0,114,0,105,0,112,0,116,0,32,0,40,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,41,0,46,0,10,0,10,0,66,0,111,0,116,0,104,0,32,0,101,0,110,0,99,0,111,0,100,0,101,0,32,0,97,0,110,0,100,0,32,0,100,0,101,0,99,0,111,0,100,0,101,0,32,0,100,0,101,0,97,0,108,0,32,0,119,0,105,0,116,0,104,0,32,0,100,0,97,0,116,0,97,0,32,0,114,0,101,0,112,0,114,0,101,0,115,0,101,0,110,0,116,0,101,0,100,0,32,0,97,0,115,0,58,0,10,0,10,0,45,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,32,0,40,0,101,0,110,0,99,0,111,0,100,0,101,0,32,0,101,0,120,0,112,0,101,0,99,0,116,0,115,0,32,0,74,0,83,0,32,0,115,0,116,0,114,0,105,0,110,0,103,0,44,0,32,0,100,0,101,0,99,0,111,0,100,0,101,0,32,0,105,0,110,0,116,0,101,0,114,0,112,0,114,0,101,0,116,0,115,0,32,0,85,0,67,0,83,0,50,0,32,0,99,0,104,0,97,0,114,0,115,0,32,0,97,0,115,0,32,0,99,0,111,0,100,0,101,0,115,0,41,0,10,0,45,0,32,0,65,0,114,0,114,0,97,0,121,0,32,0,40,0,101,0,110,0,99,0,111,0,100,0,101,0,32,0,101,0,120,0,112,0,101,0,99,0,116,0,115,0,32,0,97,0,114,0,114,0,97,0,121,0,32,0,111,0,102,0,32,0,74,0,83,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,32,0,99,0,104,0,97,0,114,0,97,0,99,0,116,0,101,0,114,0,115,0,44,0,32,0,100,0,101,0,99,0,111,0,100,0,101,0,32,0,101,0,120,0,112,0,101,0,99,0,116,0,115,0,32,0,110,0,117,0,109,0,98,0,101,0,114,0,115,0,41,0,10,0,45,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,32,0,40,0,101,0,110,0,99,0,111,0,100,0,101,0,32,0,101,0,120,0,112,0,101,0,99,0,116,0,115,0,32,0,85,0,84,0,70,0,45,0,56,0,32,0,115,0,116,0,114,0,105,0,110,0,103,0,44,0,32,0,100,0,101,0,99,0,111,0,100,0,101,0,32,0,101,0,120,0,112,0,101,0,99,0,116,0,115,0,32,0,99,0,111,0,100,0,101,0,112,0,111,0,105,0,110,0,116,0,115,0,47,0,98,0,121,0,116,0,101,0,115,0,41,0,46,0,10,0,10,0,84,0,104,0,101,0,32,0,96,0,111,0,102,0,109,0,116,0,96,0,32,0,118,0,97,0,114,0,105,0,97,0,98,0,108,0,101,0,32,0,99,0,111,0,110,0,116,0,114,0,111,0,108,0,115,0,32,0,96,0,101,0,110,0,99,0,111,0,100,0,101,0,96,0,32,0,111,0,117,0,116,0,112,0,117,0,116,0,32,0,40,0,96,0,115,0,116,0,114,0,96,0,44,0,32,0,96,0,97,0,114,0,114,0,96,0,32,0,114,0,101,0,115,0,112,0,101,0,99,0,116,0,105,0,118,0,101,0,108,0,121,0,41,0,10,0,119,0,104,0,105,0,108,0,101,0,32,0,116,0,104,0,101,0,32,0,105,0,110,0,112,0,117,0,116,0,32,0,102,0,111,0,114,0,109,0,97,0,116,0,32,0,105,0,115,0,32,0,97,0,117,0,116,0,111,0,109,0,97,0,116,0,105,0,99,0,97,0,108,0,108,0,121,0,32,0,100,0,101,0,116,0,101,0,114,0,109,0,105,0,110,0,101,0,100,0,46,0,10,0,10,0,35,0,32,0,84,0,101,0,115,0,116,0,115,0,10,0,10,0,84,0,104,0,101,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,105,0,110,0,99,0,108,0,117,0,100,0,101,0,32,0,74,0,83,0,32,0,118,0,97,0,108,0,105,0,100,0,105,0,116,0,121,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,40,0,114,0,101,0,113,0,117,0,105,0,114,0,105,0,110,0,103,0,32,0,111,0,114,0,32,0,101,0,118,0,97,0,108,0,39,0,105,0,110,0,103,0,32,0,99,0,111,0,100,0,101,0,41,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,118,0,97,0,114,0,32,0,102,0,115,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,102,0,115,0,39,0,41,0,44,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,97,0,115,0,115,0,101,0,114,0,116,0,39,0,41,0,44,0,32,0,118,0,109,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,118,0,109,0,39,0,41,0,59,0,10,0,118,0,97,0,114,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,44,0,32,0,115,0,98,0,99,0,115,0,59,0,10,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,115,0,111,0,117,0,114,0,99,0,101,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,108,0,111,0,97,0,100,0,32,0,110,0,111,0,100,0,101,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,46,0,47,0,39,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,108,0,111,0,97,0,100,0,32,0,115,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,115,0,98,0,99,0,115,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,46,0,47,0,115,0,98,0,99,0,115,0,39,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,108,0,111,0,97,0,100,0,32,0,101,0,120,0,99,0,101,0,108,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,101,0,120,0,99,0,101,0,108,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,46,0,47,0,99,0,112,0,101,0,120,0,99,0,101,0,108,0,39,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,98,0,105,0,116,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,102,0,105,0,108,0,101,0,115,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,100,0,105,0,114,0,83,0,121,0,110,0,99,0,40,0,39,0,98,0,105,0,116,0,115,0,39,0,41,0,46,0,102,0,105,0,108,0,116,0,101,0,114,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,120,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,120,0,46,0,115,0,117,0,98,0,115,0,116,0,114,0,40,0,45,0,51,0,41,0,61,0,61,0,34,0,46,0,106,0,115,0,34,0,59,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,105,0,108,0,101,0,115,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,120,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,109,0,46,0,114,0,117,0,110,0,73,0,110,0,84,0,104,0,105,0,115,0,67,0,111,0,110,0,116,0,101,0,120,0,116,0,40,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,98,0,105,0,116,0,115,0,47,0,39,0,32,0,43,0,32,0,120,0,41,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,82,0,69,0,65,0,68,0,77,0,69,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,118,0,101,0,114,0,105,0,102,0,121,0,32,0,116,0,104,0,101,0,32,0,115,0,110,0,105,0,112,0,112,0,101,0,116,0,115,0,32,0,105,0,110,0,32,0,116,0,104,0,101,0,32,0,82,0,69,0,65,0,68,0,77,0,69,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,82,0,69,0,65,0,68,0,77,0,69,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,114,0,101,0,97,0,100,0,109,0,101,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,95,0,99,0,112,0,49,0,48,0,48,0,48,0,48,0,95,0,50,0,53,0,53,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,91,0,49,0,48,0,48,0,48,0,48,0,93,0,46,0,100,0,101,0,99,0,91,0,50,0,53,0,53,0,93,0,59,0,32,0,47,0,47,0,32,0,199,2,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,95,0,99,0,112,0,49,0,48,0,48,0,48,0,48,0,95,0,50,0,53,0,53,0,44,0,32,0,34,0,199,2,34,0,41,0,59,0,10,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,112,0,49,0,48,0,48,0,48,0,48,0,95,0,55,0,49,0,49,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,91,0,49,0,48,0,48,0,48,0,48,0,93,0,46,0,101,0,110,0,99,0,91,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,55,0,49,0,49,0,41,0,93,0,59,0,32,0,47,0,47,0,32,0,50,0,53,0,53,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,99,0,112,0,49,0,48,0,48,0,48,0,48,0,95,0,55,0,49,0,49,0,44,0,32,0,50,0,53,0,53,0,41,0,59,0,10,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,49,0,32,0,61,0,32,0,91,0,48,0,120,0,98,0,98,0,44,0,48,0,120,0,101,0,51,0,44,0,48,0,120,0,100,0,55,0,44,0,48,0,120,0,100,0,99,0,93,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,71,108,59,96,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,57,0,51,0,54,0,44,0,32,0,98,0,49,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,117,0,102,0,32,0,61,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,57,0,51,0,54,0,44,0,32,0,32,0,71,108,59,96,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,71,108,59,96,44,0,34,0,71,108,59,96,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,98,0,117,0,102,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,44,0,32,0,52,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,52,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,98,0,49,0,91,0,105,0,93,0,44,0,32,0,98,0,117,0,102,0,91,0,105,0,93,0,41,0,59,0,10,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,50,0,32,0,61,0,32,0,91,0,48,0,120,0,102,0,48,0,44,0,48,0,120,0,57,0,102,0,44,0,48,0,120,0,56,0,100,0,44,0,48,0,120,0,97,0,51,0,93,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,117,0,115,0,104,0,105,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,54,0,53,0,48,0,48,0,49,0,44,0,32,0,98,0,50,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,98,0,117,0,102,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,54,0,53,0,48,0,48,0,49,0,44,0,32,0,115,0,117,0,115,0,104,0,105,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,115,0,117,0,115,0,104,0,105,0,44,0,34,0,60,216,99,223,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,115,0,98,0,117,0,102,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,44,0,32,0,52,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,52,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,98,0,50,0,91,0,105,0,93,0,44,0,32,0,115,0,98,0,117,0,102,0,91,0,105,0,93,0,41,0,59,0,10,0,10,0,32,0,32,0,125,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,98,0,101,0,32,0,99,0,111,0,114,0,114,0,101,0,99,0,116,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,114,0,101,0,97,0,100,0,109,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,114,0,101,0,97,0,100,0,109,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,99,0,111,0,110,0,115,0,105,0,115,0,116,0,101,0,110,0,99,0,121,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,109,0,97,0,107,0,101,0,32,0,115,0,117,0,114,0,101,0,32,0,116,0,104,0,97,0,116,0,32,0,101,0,110,0,99,0,111,0,100,0,105,0,110,0,103,0,32,0,97,0,110,0,100,0,32,0,100,0,101,0,99,0,111,0,100,0,105,0,110,0,103,0,32,0,97,0,114,0,101,0,32,0,112,0,115,0,101,0,117,0,100,0,111,0,32,0,105,0,110,0,118,0,101,0,114,0,115,0,101,0,115,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,99,0,111,0,110,0,115,0,105,0,115,0,116,0,101,0,110,0,99,0,121,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,46,0,47,0,39,0,41,0,59,0,10,0,32,0,32,0,85,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,104,0,107,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,44,0,32,0,99,0,97,0,99,0,104,0,101,0,105,0,116,0,41,0,32,0,123,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,120,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,99,0,111,0,110,0,115,0,105,0,115,0,116,0,101,0,110,0,116,0,108,0,121,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,67,0,80,0,32,0,39,0,32,0,43,0,32,0,120,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,112,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,91,0,120,0,93,0,44,0,32,0,68,0,32,0,61,0,32,0,99,0,112,0,46,0,100,0,101,0,99,0,44,0,32,0,69,0,32,0,61,0,32,0,99,0,112,0,46,0,101,0,110,0,99,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,99,0,97,0,99,0,104,0,101,0,105,0,116,0,41,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,101,0,108,0,115,0,101,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,79,0,98,0,106,0,101,0,99,0,116,0,46,0,107,0,101,0,121,0,115,0,40,0,68,0,41,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,100,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,69,0,91,0,68,0,91,0,100,0,93,0,93,0,32,0,33,0,61,0,32,0,100,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,69,0,91,0,68,0,91,0,100,0,93,0,93,0,32,0,33,0,61,0,61,0,32,0,34,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,34,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,68,0,91,0,100,0,93,0,46,0,99,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,65,0,116,0,40,0,48,0,41,0,32,0,61,0,61,0,32,0,48,0,120,0,70,0,70,0,70,0,68,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,68,0,91,0,69,0,91,0,68,0,91,0,100,0,93,0,93,0,93,0,32,0,61,0,61,0,61,0,32,0,68,0,91,0,100,0,93,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,116,0,104,0,114,0,111,0,119,0,32,0,110,0,101,0,119,0,32,0,69,0,114,0,114,0,111,0,114,0,40,0,120,0,32,0,43,0,32,0,34,0,32,0,101,0,46,0,100,0,91,0,34,0,32,0,43,0,32,0,100,0,32,0,43,0,32,0,34,0,93,0,32,0,61,0,32,0,34,0,32,0,43,0,32,0,69,0,91,0,68,0,91,0,100,0,93,0,93,0,32,0,43,0,32,0,34,0,59,0,32,0,100,0,91,0,34,0,32,0,43,0,32,0,100,0,32,0,43,0,32,0,34,0,93,0,61,0,34,0,32,0,43,0,32,0,68,0,91,0,100,0,93,0,32,0,43,0,32,0,34,0,59,0,32,0,100,0,46,0,101,0,46,0,100,0,91,0,34,0,32,0,43,0,32,0,100,0,32,0,43,0,32,0,34,0,93,0,32,0,61,0,32,0,34,0,32,0,43,0,32,0,68,0,91,0,69,0,91,0,68,0,91,0,100,0,93,0,93,0,93,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,79,0,98,0,106,0,101,0,99,0,116,0,46,0,107,0,101,0,121,0,115,0,40,0,69,0,41,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,101,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,68,0,91,0,69,0,91,0,101,0,93,0,93,0,32,0,33,0,61,0,32,0,101,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,116,0,104,0,114,0,111,0,119,0,32,0,110,0,101,0,119,0,32,0,69,0,114,0,114,0,111,0,114,0,40,0,120,0,32,0,43,0,32,0,34,0,32,0,100,0,46,0,101,0,91,0,34,0,32,0,43,0,32,0,101,0,32,0,43,0,32,0,34,0,93,0,32,0,61,0,32,0,34,0,32,0,43,0,32,0,68,0,91,0,69,0,91,0,101,0,93,0,93,0,32,0,43,0,32,0,34,0,59,0,32,0,101,0,91,0,34,0,32,0,43,0,32,0,101,0,32,0,43,0,32,0,34,0,93,0,61,0,34,0,32,0,43,0,32,0,69,0,91,0,101,0,93,0,32,0,43,0,32,0,34,0,59,0,32,0,101,0,46,0,100,0,46,0,101,0,91,0,34,0,32,0,43,0,32,0,101,0,32,0,43,0,32,0,34,0,93,0,32,0,61,0,32,0,34,0,32,0,43,0,32,0,69,0,91,0,68,0,91,0,69,0,91,0,101,0,93,0,93,0,93,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,111,0,114,0,112,0,117,0,115,0,32,0,61,0,32,0,91,0,34,0,102,0,111,0,111,0,98,0,97,0,114,0,34,0,93,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,111,0,114,0,112,0,117,0,115,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,119,0,41,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,85,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,120,0,44,0,85,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,120,0,44,0,119,0,41,0,41,0,44,0,119,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,59,0,32,0,125,0,59,0,10,0,32,0,32,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,99,0,97,0,99,0,104,0,101,0,100,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,79,0,98,0,106,0,101,0,99,0,116,0,46,0,107,0,101,0,121,0,115,0,40,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,41,0,46,0,102,0,105,0,108,0,116,0,101,0,114,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,119,0,41,0,32,0,123,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,119,0,32,0,61,0,61,0,32,0,43,0,119,0,59,0,32,0,125,0,41,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,99,0,104,0,107,0,40,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,44,0,32,0,116,0,114,0,117,0,101,0,41,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,100,0,105,0,114,0,101,0,99,0,116,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,79,0,98,0,106,0,101,0,99,0,116,0,46,0,107,0,101,0,121,0,115,0,40,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,41,0,46,0,102,0,105,0,108,0,116,0,101,0,114,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,119,0,41,0,32,0,123,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,119,0,32,0,61,0,61,0,32,0,43,0,119,0,59,0,32,0,125,0,41,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,99,0,104,0,107,0,40,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,44,0,32,0,102,0,97,0,108,0,115,0,101,0,41,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,110,0,101,0,120,0,116,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,108,0,111,0,111,0,107,0,32,0,97,0,116,0,32,0,112,0,111,0,115,0,115,0,105,0,98,0,108,0,101,0,32,0,101,0,110,0,116,0,114,0,121,0,32,0,99,0,111,0,110,0,100,0,105,0,116,0,105,0,111,0,110,0,115,0,58,0,10,0,10,0,96,0,96,0,96,0,10,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,101,0,110,0,116,0,114,0,121,0,32,0,99,0,111,0,110,0,100,0,105,0,116,0,105,0,111,0,110,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,102,0,97,0,105,0,108,0,32,0,116,0,111,0,32,0,108,0,111,0,97,0,100,0,32,0,117,0,116,0,105,0,108,0,115,0,32,0,105,0,102,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,32,0,117,0,110,0,97,0,118,0,97,0,105,0,108,0,97,0,98,0,108,0,101,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,97,0,110,0,100,0,98,0,111,0,120,0,32,0,61,0,32,0,123,0,125,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,116,0,120,0,32,0,61,0,32,0,118,0,109,0,46,0,99,0,114,0,101,0,97,0,116,0,101,0,67,0,111,0,110,0,116,0,101,0,120,0,116,0,40,0,115,0,97,0,110,0,100,0,98,0,111,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,109,0,46,0,114,0,117,0,110,0,73,0,110,0,67,0,111,0,110,0,116,0,101,0,120,0,116,0,40,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,39,0,44,0,39,0,117,0,116,0,102,0,56,0,39,0,41,0,44,0,99,0,116,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,108,0,111,0,97,0,100,0,32,0,117,0,116,0,105,0,108,0,115,0,32,0,105,0,102,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,32,0,105,0,115,0,32,0,97,0,118,0,97,0,105,0,108,0,97,0,98,0,108,0,101,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,97,0,110,0,100,0,98,0,111,0,120,0,32,0,61,0,32,0,123,0,125,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,116,0,120,0,32,0,61,0,32,0,118,0,109,0,46,0,99,0,114,0,101,0,97,0,116,0,101,0,67,0,111,0,110,0,116,0,101,0,120,0,116,0,40,0,115,0,97,0,110,0,100,0,98,0,111,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,109,0,46,0,114,0,117,0,110,0,73,0,110,0,67,0,111,0,110,0,116,0,101,0,120,0,116,0,40,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,99,0,112,0,101,0,120,0,99,0,101,0,108,0,46,0,106,0,115,0,39,0,44,0,39,0,117,0,116,0,102,0,56,0,39,0,41,0,44,0,99,0,116,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,109,0,46,0,114,0,117,0,110,0,73,0,110,0,67,0,111,0,110,0,116,0,101,0,120,0,116,0,40,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,39,0,44,0,39,0,117,0,116,0,102,0,56,0,39,0,41,0,44,0,99,0,116,0,120,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,104,0,107,0,101,0,110,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,99,0,112,0,44,0,32,0,105,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,99,0,112,0,44,0,32,0,105,0,44,0,32,0,101,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,116,0,114,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,105,0,44,0,101,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,97,0,114,0,114,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,105,0,46,0,115,0,112,0,108,0,105,0,116,0,40,0,34,0,34,0,41,0,44,0,101,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,100,0,101,0,101,0,112,0,69,0,113,0,117,0,97,0,108,0,40,0,115,0,116,0,114,0,44,0,97,0,114,0,114,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,32,0,61,0,61,0,61,0,32,0,39,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,39,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,117,0,102,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,110,0,101,0,119,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,40,0,105,0,41,0,44,0,101,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,100,0,101,0,101,0,112,0,69,0,113,0,117,0,97,0,108,0,40,0,115,0,116,0,114,0,44,0,98,0,117,0,102,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,44,0,39,0,98,0,117,0,102,0,39,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,44,0,39,0,97,0,114,0,114,0,39,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,44,0,39,0,115,0,116,0,114,0,39,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,44,0,39,0,98,0,117,0,102,0,39,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,44,0,39,0,97,0,114,0,114,0,39,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,44,0,39,0,115,0,116,0,114,0,39,0,41,0,59,0,10,0,32,0,32,0,125,0,59,0,10,0,32,0,32,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,101,0,110,0,99,0,111,0,100,0,101,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,67,0,80,0,32,0,32,0,49,0,50,0,53,0,50,0,32,0,58,0,32,0,115,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,104,0,107,0,101,0,110,0,40,0,49,0,50,0,53,0,50,0,44,0,34,0,102,0,111,0,111,0,98,0,97,0,114,0,34,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,67,0,80,0,32,0,32,0,32,0,55,0,48,0,56,0,32,0,58,0,32,0,115,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,104,0,107,0,101,0,110,0,40,0,55,0,48,0,56,0,44,0,34,0,42,6,32,0,97,0,110,0,100,0,32,0,43,6,32,0,115,0,109,0,105,0,108,0,101,0,121,0,32,0,102,0,97,0,99,0,101,0,115,0,34,0,41,0,59,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,67,0,80,0,32,0,32,0,32,0,57,0,51,0,54,0,32,0,58,0,32,0,100,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,104,0,107,0,101,0,110,0,40,0,57,0,51,0,54,0,44,0,32,0,34,0,217,143,47,102,45,78,135,101,87,91,38,123,75,109,213,139,34,0,41,0,59,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,104,0,107,0,100,0,101,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,99,0,112,0,44,0,32,0,105,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,99,0,112,0,44,0,32,0,105,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,32,0,33,0,61,0,61,0,32,0,39,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,39,0,32,0,38,0,38,0,32,0,105,0,32,0,105,0,110,0,115,0,116,0,97,0,110,0,99,0,101,0,111,0,102,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,41,0,32,0,115,0,32,0,61,0,32,0,91,0,93,0,46,0,109,0,97,0,112,0,46,0,99,0,97,0,108,0,108,0,40,0,105,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,115,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,115,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,101,0,108,0,115,0,101,0,32,0,115,0,61,0,40,0,105,0,46,0,109,0,97,0,112,0,41,0,32,0,63,0,32,0,105,0,46,0,109,0,97,0,112,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,115,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,115,0,41,0,59,0,32,0,125,0,41,0,32,0,58,0,32,0,105,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,116,0,114,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,105,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,97,0,114,0,114,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,115,0,46,0,106,0,111,0,105,0,110,0,63,0,115,0,46,0,106,0,111,0,105,0,110,0,40,0,34,0,34,0,41,0,58,0,115,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,100,0,101,0,101,0,112,0,69,0,113,0,117,0,97,0,108,0,40,0,115,0,116,0,114,0,44,0,97,0,114,0,114,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,32,0,61,0,61,0,61,0,32,0,39,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,39,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,117,0,102,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,110,0,101,0,119,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,40,0,105,0,41,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,100,0,101,0,101,0,112,0,69,0,113,0,117,0,97,0,108,0,40,0,115,0,116,0,114,0,44,0,98,0,117,0,102,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,41,0,59,0,10,0,32,0,32,0,125,0,59,0,10,0,32,0,32,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,100,0,101,0,99,0,111,0,100,0,101,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,67,0,80,0,32,0,32,0,49,0,50,0,53,0,50,0,32,0,58,0,32,0,115,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,104,0,107,0,100,0,101,0,40,0,49,0,50,0,53,0,50,0,44,0,91,0,48,0,120,0,54,0,54,0,44,0,32,0,48,0,120,0,54,0,102,0,44,0,32,0,48,0,120,0,54,0,102,0,44,0,32,0,48,0,120,0,54,0,50,0,44,0,32,0,48,0,120,0,54,0,49,0,44,0,32,0,48,0,120,0,55,0,50,0,93,0,41,0,59,0,32,0,125,0,41,0,59,0,32,0,47,0,42,0,32,0,34,0,102,0,111,0,111,0,98,0,97,0,114,0,34,0,32,0,42,0,47,0,10,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,32,0,33,0,61,0,61,0,32,0,39,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,39,0,41,0,32,0,105,0,116,0,40,0,39,0,67,0,80,0,32,0,32,0,32,0,55,0,48,0,56,0,32,0,58,0,32,0,115,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,104,0,107,0,100,0,101,0,40,0,55,0,48,0,56,0,44,0,32,0,110,0,101,0,119,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,40,0,91,0,48,0,120,0,99,0,97,0,44,0,32,0,48,0,120,0,50,0,48,0,44,0,32,0,48,0,120,0,54,0,49,0,44,0,32,0,48,0,120,0,54,0,101,0,44,0,32,0,48,0,120,0,54,0,52,0,44,0,32,0,48,0,120,0,50,0,48,0,44,0,32,0,48,0,120,0,99,0,98,0,44,0,32,0,48,0,120,0,50,0,48,0,44,0,32,0,48,0,120,0,55,0,51,0,44,0,32,0,48,0,120,0,54,0,100,0,44,0,32,0,48,0,120,0,54,0,57,0,44,0,32,0,48,0,120,0,54,0,99,0,44,0,32,0,48,0,120,0,54,0,53,0,44,0,32,0,48,0,120,0,55,0,57,0,44,0,32,0,48,0,120,0,50,0,48,0,44,0,32,0,48,0,120,0,54,0,54,0,44,0,32,0,48,0,120,0,54,0,49,0,44,0,32,0,48,0,120,0,54,0,51,0,44,0,32,0,48,0,120,0,54,0,53,0,44,0,32,0,48,0,120,0,55,0,51,0,93,0,41,0,41,0,59,0,32,0,125,0,41,0,59,0,32,0,47,0,42,0,32,0,40,0,34,0,42,6,32,0,97,0,110,0,100,0,32,0,43,6,32,0,115,0,109,0,105,0,108,0,101,0,121,0,32,0,102,0,97,0,99,0,101,0,115,0,34,0,41,0,32,0,42,0,47,0,10,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,67,0,80,0,32,0,32,0,32,0,57,0,51,0,54,0,32,0,58,0,32,0,100,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,104,0,107,0,100,0,101,0,40,0,57,0,51,0,54,0,44,0,32,0,91,0,48,0,120,0,100,0,53,0,44,0,32,0,48,0,120,0,101,0,50,0,44,0,32,0,48,0,120,0,99,0,97,0,44,0,32,0,48,0,120,0,99,0,55,0,44,0,32,0,48,0,120,0,100,0,54,0,44,0,32,0,48,0,120,0,100,0,48,0,44,0,32,0,48,0,120,0,99,0,101,0,44,0,32,0,48,0,120,0,99,0,52,0,44,0,32,0,48,0,120,0,100,0,55,0,44,0,32,0,48,0,120,0,100,0,54,0,44,0,32,0,48,0,120,0,98,0,55,0,44,0,32,0,48,0,120,0,102,0,98,0,44,0,32,0,48,0,120,0,98,0,50,0,44,0,32,0,48,0,120,0,101,0,50,0,44,0,32,0,48,0,120,0,99,0,97,0,44,0,32,0,48,0,120,0,100,0,52,0,93,0,41,0,59,0,125,0,41,0,59,0,32,0,47,0,42,0,32,0,34,0,217,143,47,102,45,78,135,101,87,91,38,123,75,109,213,139,34,0,32,0,42,0,47,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,96,0,116,0,101,0,115,0,116,0,102,0,105,0,108,0,101,0,96,0,32,0,104,0,101,0,108,0,112,0,101,0,114,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,32,0,114,0,101,0,97,0,100,0,115,0,32,0,97,0,32,0,102,0,105,0,108,0,101,0,32,0,97,0,110,0,100,0,32,0,99,0,111,0,109,0,112,0,97,0,114,0,101,0,115,0,32,0,116,0,111,0,32,0,110,0,111,0,100,0,101,0,39,0,115,0,32,0,114,0,101,0,97,0,100,0,32,0,102,0,97,0,99,0,105,0,108,0,105,0,116,0,105,0,101,0,115,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,32,0,116,0,101,0,115,0,116,0,102,0,105,0,108,0,101,0,40,0,102,0,44,0,99,0,112,0,44,0,116,0,121,0,112,0,101,0,44,0,115,0,107,0,105,0,112,0,41,0,32,0,123,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,100,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,102,0,41,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,120,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,102,0,44,0,32,0,116,0,121,0,112,0,101,0,41,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,97,0,32,0,61,0,32,0,120,0,46,0,115,0,112,0,108,0,105,0,116,0,40,0,34,0,34,0,41,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,104,0,107,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,99,0,112,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,121,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,32,0,100,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,120,0,44,0,121,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,122,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,32,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,33,0,61,0,32,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,41,0,32,0,116,0,104,0,114,0,111,0,119,0,32,0,110,0,101,0,119,0,32,0,69,0,114,0,114,0,111,0,114,0,40,0,102,0,32,0,43,0,32,0,34,0,32,0,34,0,32,0,43,0,32,0,74,0,83,0,79,0,78,0,46,0,115,0,116,0,114,0,105,0,110,0,103,0,105,0,102,0,121,0,40,0,122,0,41,0,32,0,43,0,32,0,34,0,32,0,33,0,61,0,32,0,34,0,32,0,43,0,32,0,74,0,83,0,79,0,78,0,46,0,115,0,116,0,114,0,105,0,110,0,103,0,105,0,102,0,121,0,40,0,100,0,41,0,32,0,43,0,32,0,34,0,32,0,58,0,32,0,34,0,32,0,43,0,32,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,43,0,32,0,34,0,32,0,34,0,32,0,43,0,32,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,105,0,102,0,40,0,100,0,91,0,105,0,93,0,32,0,33,0,61,0,61,0,32,0,122,0,91,0,105,0,93,0,41,0,32,0,116,0,104,0,114,0,111,0,119,0,32,0,110,0,101,0,119,0,32,0,69,0,114,0,114,0,111,0,114,0,40,0,34,0,34,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,34,0,32,0,34,0,32,0,43,0,32,0,100,0,91,0,105,0,93,0,32,0,43,0,32,0,34,0,33,0,61,0,34,0,32,0,43,0,32,0,122,0,91,0,105,0,93,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,115,0,107,0,105,0,112,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,122,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,32,0,97,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,33,0,61,0,32,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,41,0,32,0,116,0,104,0,114,0,111,0,119,0,32,0,110,0,101,0,119,0,32,0,69,0,114,0,114,0,111,0,114,0,40,0,102,0,32,0,43,0,32,0,34,0,32,0,34,0,32,0,43,0,32,0,74,0,83,0,79,0,78,0,46,0,115,0,116,0,114,0,105,0,110,0,103,0,105,0,102,0,121,0,40,0,122,0,41,0,32,0,43,0,32,0,34,0,32,0,33,0,61,0,32,0,34,0,32,0,43,0,32,0,74,0,83,0,79,0,78,0,46,0,115,0,116,0,114,0,105,0,110,0,103,0,105,0,102,0,121,0,40,0,100,0,41,0,32,0,43,0,32,0,34,0,32,0,58,0,32,0,34,0,32,0,43,0,32,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,43,0,32,0,34,0,32,0,34,0,32,0,43,0,32,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,105,0,102,0,40,0,100,0,91,0,105,0,93,0,32,0,33,0,61,0,61,0,32,0,122,0,91,0,105,0,93,0,41,0,32,0,116,0,104,0,114,0,111,0,119,0,32,0,110,0,101,0,119,0,32,0,69,0,114,0,114,0,111,0,114,0,40,0,34,0,34,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,34,0,32,0,34,0,32,0,43,0,32,0,100,0,91,0,105,0,93,0,32,0,43,0,32,0,34,0,33,0,61,0,34,0,32,0,43,0,32,0,122,0,91,0,105,0,93,0,41,0,59,0,10,0,32,0,32,0,125,0,10,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,99,0,104,0,107,0,40,0,99,0,112,0,41,0,59,0,10,0,32,0,32,0,105,0,102,0,40,0,115,0,107,0,105,0,112,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,99,0,104,0,107,0,40,0,99,0,112,0,41,0,59,0,10,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,125,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,96,0,117,0,116,0,102,0,56,0,96,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,118,0,101,0,114,0,105,0,102,0,121,0,32,0,117,0,116,0,102,0,56,0,32,0,101,0,110,0,99,0,111,0,100,0,105,0,110,0,103,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,32,0,97,0,99,0,116,0,117,0,97,0,108,0,32,0,74,0,83,0,32,0,115,0,111,0,117,0,114,0,99,0,101,0,115,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,110,0,111,0,100,0,101,0,32,0,110,0,97,0,116,0,105,0,118,0,101,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,110,0,111,0,100,0,101,0,32,0,61,0,32,0,91,0,91,0,54,0,53,0,48,0,48,0,49,0,44,0,32,0,39,0,117,0,116,0,102,0,56,0,39,0,44,0,49,0,93,0,44,0,32,0,91,0,49,0,50,0,48,0,48,0,44,0,32,0,39,0,117,0,116,0,102,0,49,0,54,0,108,0,101,0,39,0,44,0,49,0,93,0,44,0,32,0,91,0,50,0,48,0,49,0,50,0,55,0,44,0,32,0,39,0,97,0,115,0,99,0,105,0,105,0,39,0,44,0,48,0,93,0,93,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,102,0,105,0,108,0,101,0,115,0,32,0,61,0,32,0,91,0,39,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,109,0,100,0,39,0,44,0,39,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,109,0,100,0,39,0,44,0,39,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,106,0,115,0,39,0,93,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,97,0,115,0,99,0,105,0,105,0,102,0,105,0,108,0,101,0,115,0,32,0,61,0,32,0,91,0,39,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,39,0,93,0,59,0,10,0,32,0,32,0,110,0,111,0,100,0,101,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,119,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,119,0,91,0,49,0,93,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,46,0,47,0,39,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,99,0,105,0,105,0,102,0,105,0,108,0,101,0,115,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,102,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,39,0,32,0,43,0,32,0,102,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,116,0,101,0,115,0,116,0,102,0,105,0,108,0,101,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,39,0,43,0,102,0,43,0,39,0,46,0,39,0,43,0,119,0,91,0,49,0,93,0,44,0,119,0,91,0,48,0,93,0,44,0,119,0,91,0,49,0,93,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,33,0,119,0,91,0,50,0,93,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,102,0,105,0,108,0,101,0,115,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,102,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,39,0,32,0,43,0,32,0,102,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,116,0,101,0,115,0,116,0,102,0,105,0,108,0,101,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,39,0,43,0,102,0,43,0,39,0,46,0,39,0,43,0,119,0,91,0,49,0,93,0,44,0,119,0,91,0,48,0,93,0,44,0,119,0,91,0,49,0,93,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,119,0,91,0,49,0,93,0,32,0,61,0,61,0,61,0,32,0,39,0,117,0,116,0,102,0,56,0,39,0,41,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,98,0,105,0,116,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,102,0,105,0,108,0,101,0,115,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,100,0,105,0,114,0,83,0,121,0,110,0,99,0,40,0,39,0,98,0,105,0,116,0,115,0,39,0,41,0,46,0,102,0,105,0,108,0,116,0,101,0,114,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,120,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,120,0,46,0,115,0,117,0,98,0,115,0,116,0,114,0,40,0,45,0,51,0,41,0,61,0,61,0,34,0,46,0,106,0,115,0,34,0,59,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,102,0,105,0,108,0,101,0,115,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,102,0,41,0,32,0,123,0,32,0,116,0,101,0,115,0,116,0,102,0,105,0,108,0,101,0,40,0,39,0,46,0,47,0,98,0,105,0,116,0,115,0,47,0,39,0,32,0,43,0,32,0,102,0,44,0,119,0,91,0,48,0,93,0,44,0,119,0,91,0,49,0,93,0,44,0,116,0,114,0,117,0,101,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,117,0,116,0,102,0,42,0,32,0,97,0,110,0,100,0,32,0,97,0,115,0,99,0,105,0,105,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,97,0,116,0,116,0,101,0,109,0,112,0,116,0,32,0,116,0,111,0,32,0,116,0,101,0,115,0,116,0,32,0,111,0,116,0,104,0,101,0,114,0,32,0,109,0,97,0,103,0,105,0,99,0,32,0,102,0,111,0,114,0,109,0,97,0,116,0,115,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,118,0,97,0,114,0,32,0,109,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,109,0,97,0,103,0,105,0,99,0,59,0,10,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,32,0,99,0,109,0,112,0,40,0,120,0,44,0,122,0,41,0,32,0,123,0,10,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,120,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,44,0,32,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,41,0,59,0,10,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,105,0,43,0,34,0,47,0,34,0,43,0,120,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,43,0,34,0,34,0,43,0,120,0,91,0,105,0,93,0,44,0,32,0,105,0,43,0,34,0,47,0,34,0,43,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,43,0,34,0,34,0,43,0,122,0,91,0,105,0,93,0,41,0,59,0,10,0,125,0,10,0,79,0,98,0,106,0,101,0,99,0,116,0,46,0,107,0,101,0,121,0,115,0,40,0,109,0,41,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,116,0,41,0,123,0,105,0,102,0,40,0,116,0,32,0,33,0,61,0,32,0,49,0,54,0,57,0,54,0,57,0,41,0,32,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,109,0,91,0,116,0,93,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,105,0,116,0,40,0,34,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,109,0,100,0,46,0,34,0,32,0,43,0,32,0,109,0,91,0,116,0,93,0,44,0,32,0,102,0,115,0,46,0,101,0,120,0,105,0,115,0,116,0,115,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,109,0,100,0,46,0,39,0,32,0,43,0,32,0,109,0,91,0,116,0,93,0,41,0,32,0,63,0,10,0,32,0,32,0,32,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,109,0,100,0,46,0,117,0,116,0,102,0,56,0,39,0,44,0,32,0,34,0,117,0,116,0,102,0,56,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,109,0,91,0,116,0,93,0,32,0,61,0,61,0,61,0,32,0,34,0,97,0,115,0,99,0,105,0,105,0,34,0,41,0,32,0,98,0,32,0,61,0,32,0,98,0,46,0,114,0,101,0,112,0,108,0,97,0,99,0,101,0,40,0,47,0,91,0,92,0,117,0,48,0,48,0,56,0,48,0,45,0,92,0,117,0,102,0,102,0,102,0,102,0,93,0,42,0,47,0,103,0,44,0,34,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,120,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,109,0,100,0,46,0,39,0,32,0,43,0,32,0,109,0,91,0,116,0,93,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,121,0,44,0,32,0,122,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,121,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,121,0,44,0,98,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,122,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,121,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,32,0,33,0,61,0,32,0,54,0,53,0,48,0,48,0,48,0,41,0,32,0,99,0,109,0,112,0,40,0,120,0,44,0,122,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,101,0,108,0,115,0,101,0,32,0,123,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,121,0,44,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,122,0,41,0,41,0,59,0,32,0,125,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,121,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,121,0,44,0,98,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,122,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,121,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,32,0,33,0,61,0,32,0,54,0,53,0,48,0,48,0,48,0,41,0,32,0,99,0,109,0,112,0,40,0,120,0,44,0,122,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,101,0,108,0,115,0,101,0,32,0,123,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,121,0,44,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,122,0,41,0,41,0,59,0,32,0,125,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,58,0,32,0,110,0,117,0,108,0,108,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,34,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,109,0,100,0,46,0,34,0,32,0,43,0,32,0,109,0,91,0,116,0,93,0,44,0,32,0,102,0,115,0,46,0,101,0,120,0,105,0,115,0,116,0,115,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,109,0,100,0,46,0,39,0,32,0,43,0,32,0,109,0,91,0,116,0,93,0,41,0,32,0,63,0,10,0,32,0,32,0,32,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,109,0,100,0,46,0,117,0,116,0,102,0,56,0,39,0,44,0,32,0,34,0,117,0,116,0,102,0,56,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,109,0,91,0,116,0,93,0,32,0,61,0,61,0,61,0,32,0,34,0,97,0,115,0,99,0,105,0,105,0,34,0,41,0,32,0,98,0,32,0,61,0,32,0,98,0,46,0,114,0,101,0,112,0,108,0,97,0,99,0,101,0,40,0,47,0,91,0,92,0,117,0,48,0,48,0,56,0,48,0,45,0,92,0,117,0,102,0,102,0,102,0,102,0,93,0,42,0,47,0,103,0,44,0,34,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,120,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,109,0,100,0,46,0,39,0,32,0,43,0,32,0,109,0,91,0,116,0,93,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,120,0,32,0,61,0,32,0,91,0,93,0,46,0,115,0,108,0,105,0,99,0,101,0,46,0,99,0,97,0,108,0,108,0,40,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,121,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,121,0,44,0,98,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,121,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,121,0,44,0,98,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,58,0,32,0,110,0,117,0,108,0,108,0,41,0,59,0,10,0,125,0,41,0,59,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,32,0,96,0,54,0,57,0,54,0,57,0,96,0,32,0,105,0,115,0,32,0,110,0,111,0,116,0,32,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,44,0,32,0,115,0,111,0,32,0,111,0,112,0,101,0,114,0,97,0,116,0,105,0,111,0,110,0,115,0,32,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,102,0,97,0,105,0,108,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,102,0,97,0,105,0,108,0,117,0,114,0,101,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,102,0,97,0,105,0,108,0,32,0,116,0,111,0,32,0,102,0,105,0,110,0,100,0,32,0,67,0,80,0,32,0,54,0,57,0,54,0,57,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,91,0,54,0,57,0,54,0,57,0,93,0,46,0,100,0,101,0,99,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,91,0,54,0,57,0,54,0,57,0,93,0,46,0,101,0,110,0,99,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,102,0,97,0,105,0,108,0,32,0,117,0,115,0,105,0,110,0,103,0,32,0,117,0,116,0,105,0,108,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,40,0,33,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,104,0,97,0,115,0,99,0,112,0,40,0,54,0,57,0,54,0,57,0,41,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,54,0,57,0,54,0,57,0,44,0,32,0,34,0,102,0,111,0,111,0,98,0,97,0,114,0,34,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,54,0,57,0,54,0,57,0,44,0,32,0,91,0,48,0,120,0,50,0,48,0,93,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,102,0,97,0,105,0,108,0,32,0,119,0,105,0,116,0,104,0,32,0,98,0,108,0,97,0,99,0,107,0,32,0,109,0,97,0,103,0,105,0,99,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,40,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,104,0,97,0,115,0,99,0,112,0,40,0,49,0,54,0,57,0,54,0,57,0,41,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,49,0,54,0,57,0,54,0,57,0,44,0,32,0,34,0,102,0,111,0,111,0,98,0,97,0,114,0,34,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,49,0,54,0,57,0,54,0,57,0,44,0,32,0,91,0,48,0,120,0,50,0,48,0,93,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,102,0,97,0,105,0,108,0,32,0,119,0,104,0,101,0,110,0,32,0,112,0,114,0,101,0,115,0,101,0,110,0,116,0,101,0,100,0,32,0,119,0,105,0,116,0,104,0,32,0,105,0,110,0,118,0,97,0,108,0,105,0,100,0,32,0,99,0,104,0,97,0,114,0,32,0,99,0,111,0,100,0,101,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,50,0,48,0,49,0,50,0,55,0,44,0,32,0,91,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,48,0,120,0,65,0,65,0,41,0,93,0,41,0,59,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,35,0,32,0,78,0,105,0,116,0,116,0,121,0,32,0,71,0,114,0,105,0,116,0,116,0,121,0,10,0,10,0,96,0,96,0,96,0,106,0,115,0,111,0,110,0,62,0,112,0,97,0,99,0,107,0,97,0,103,0,101,0,46,0,106,0,115,0,111,0,110,0,10,0,123,0,10,0,32,0,32,0,34,0,110,0,97,0,109,0,101,0,34,0,58,0,32,0,34,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,34,0,44,0,10,0,32,0,32,0,34,0,118,0,101,0,114,0,115,0,105,0,111,0,110,0,34,0,58,0,32,0,34,0,49,0,46,0,51,0,46,0,56,0,34,0,44,0,10,0,32,0,32,0,34,0,97,0,117,0,116,0,104,0,111,0,114,0,34,0,58,0,32,0,34,0,83,0,104,0,101,0,101,0,116,0,74,0,83,0,34,0,44,0,10,0,32,0,32,0,34,0,100,0,101,0,115,0,99,0,114,0,105,0,112,0,116,0,105,0,111,0,110,0,34,0,58,0,32,0,34,0,112,0,117,0,114,0,101,0,45,0,74,0,83,0,32,0,108,0,105,0,98,0,114,0,97,0,114,0,121,0,32,0,116,0,111,0,32,0,104,0,97,0,110,0,100,0,108,0,101,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,34,0,44,0,10,0,32,0,32,0,34,0,107,0,101,0,121,0,119,0,111,0,114,0,100,0,115,0,34,0,58,0,32,0,91,0,32,0,34,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,34,0,44,0,32,0,34,0,105,0,99,0,111,0,110,0,118,0,34,0,44,0,32,0,34,0,99,0,111,0,110,0,118,0,101,0,114,0,116,0,34,0,44,0,32,0,34,0,115,0,116,0,114,0,105,0,110,0,103,0,115,0,34,0,32,0,93,0,44,0,10,0,32,0,32,0,34,0,98,0,105,0,110,0,34,0,58,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,34,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,34,0,58,0,32,0,34,0,46,0,47,0,98,0,105,0,110,0,47,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,110,0,106,0,115,0,34,0,10,0,32,0,32,0,125,0,44,0,10,0,32,0,32,0,34,0,102,0,105,0,108,0,101,0,115,0,34,0,58,0,32,0,91,0,10,0,32,0,32,0,32,0,32,0,34,0,76,0,73,0,67,0,69,0,78,0,83,0,69,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,109,0,100,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,98,0,105,0,110,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,106,0,115,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,100,0,105,0,115,0,116,0,47,0,99,0,112,0,101,0,120,0,99,0,101,0,108,0,46,0,102,0,117,0,108,0,108,0,46,0,106,0,115,0,34,0,10,0,32,0,32,0,93,0,44,0,10,0,32,0,32,0,34,0,109,0,97,0,105,0,110,0,34,0,58,0,32,0,34,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,34,0,44,0,10,0,32,0,32,0,34,0,100,0,101,0,112,0,101,0,110,0,100,0,101,0,110,0,99,0,105,0,101,0,115,0,34,0,58,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,34,0,118,0,111,0,99,0,34,0,58,0,34,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,99,0,111,0,110,0,99,0,97,0,116,0,45,0,115,0,116,0,114,0,101,0,97,0,109,0,34,0,58,0,34,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,99,0,111,0,109,0,109,0,97,0,110,0,100,0,101,0,114,0,34,0,58,0,34,0,34,0,10,0,32,0,32,0,125,0,44,0,10,0,32,0,32,0,34,0,100,0,101,0,118,0,68,0,101,0,112,0,101,0,110,0,100,0,101,0,110,0,99,0,105,0,101,0,115,0,34,0,58,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,34,0,109,0,111,0,99,0,104,0,97,0,34,0,58,0,34,0,34,0,10,0,32,0,32,0,125,0,44,0,10,0,32,0,32,0,34,0,114,0,101,0,112,0,111,0,115,0,105,0,116,0,111,0,114,0,121,0,34,0,58,0,32,0,123,0,32,0,34,0,116,0,121,0,112,0,101,0,34,0,58,0,34,0,103,0,105,0,116,0,34,0,44,0,32,0,34,0,117,0,114,0,108,0,34,0,58,0,34,0,103,0,105,0,116,0,58,0,47,0,47,0,103,0,105,0,116,0,104,0,117,0,98,0,46,0,99,0,111,0,109,0,47,0,83,0,104,0,101,0,101,0,116,0,74,0,83,0,47,0,106,0,115,0,45,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,103,0,105,0,116,0,34,0,125,0,44,0,10,0,32,0,32,0,34,0,115,0,99,0,114,0,105,0,112,0,116,0,115,0,34,0,58,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,34,0,112,0,114,0,101,0,116,0,101,0,115,0,116,0,34,0,58,0,32,0,34,0,103,0,105,0,116,0,32,0,115,0,117,0,98,0,109,0,111,0,100,0,117,0,108,0,101,0,32,0,105,0,110,0,105,0,116,0,32,0,38,0,38,0,32,0,103,0,105,0,116,0,32,0,115,0,117,0,98,0,109,0,111,0,100,0,117,0,108,0,101,0,32,0,117,0,112,0,100,0,97,0,116,0,101,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,116,0,101,0,115,0,116,0,34,0,58,0,32,0,34,0,109,0,97,0,107,0,101,0,32,0,116,0,101,0,115,0,116,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,98,0,117,0,105,0,108,0,100,0,34,0,58,0,32,0,34,0,109,0,97,0,107,0,101,0,32,0,106,0,115,0,34,0,10,0,32,0,32,0,125,0,44,0,10,0,32,0,32,0,34,0,99,0,111,0,110,0,102,0,105,0,103,0,34,0,58,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,34,0,98,0,108,0,97,0,110,0,107,0,101,0,116,0,34,0,58,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,34,0,112,0,97,0,116,0,116,0,101,0,114,0,110,0,34,0,58,0,32,0,34,0,91,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,106,0,115,0,44,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,44,0,99,0,112,0,101,0,120,0,99,0,101,0,108,0,46,0,106,0,115,0,93,0,34,0,10,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,125,0,44,0,10,0,32,0,32,0,34,0,98,0,117,0,103,0,115,0,34,0,58,0,32,0,123,0,32,0,34,0,117,0,114,0,108,0,34,0,58,0,32,0,34,0,104,0,116,0,116,0,112,0,115,0,58,0,47,0,47,0,103,0,105,0,116,0,104,0,117,0,98,0,46,0,99,0,111,0,109,0,47,0,83,0,104,0,101,0,101,0,116,0,74,0,83,0,47,0,106,0,115,0,45,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,47,0,105,0,115,0,115,0,117,0,101,0,115,0,34,0,32,0,125,0,44,0,10,0,32,0,32,0,34,0,108,0,105,0,99,0,101,0,110,0,115,0,101,0,34,0,58,0,32,0,34,0,65,0,112,0,97,0,99,0,104,0,101,0,45,0,50,0,46,0,48,0,34,0,44,0,10,0,32,0,32,0,34,0,101,0,110,0,103,0,105,0,110,0,101,0,115,0,34,0,58,0,32,0,123,0,32,0,34,0,110,0,111,0,100,0,101,0,34,0,58,0,32,0,34,0,62,0,61,0,48,0,46,0,56,0,34,0,32,0,125,0,10,0,125,0,10,0,96,0,96,0,96,0,10,0,10,0,96,0,96,0,96,0,62,0,46,0,118,0,111,0,99,0,114,0,99,0,10,0,123,0,32,0,34,0,112,0,111,0,115,0,116,0,34,0,58,0,32,0,34,0,109,0,97,0,107,0,101,0,32,0,106,0,115,0,34,0,32,0,125,0,10,0,96,0,96,0,96,0,10,0,10,0,96,0,96,0,96,0,62,0,46,0,103,0,105,0,116,0,105,0,103,0,110,0,111,0,114,0,101,0,10,0,46,0,103,0,105,0,116,0,105,0,103,0,110,0,111,0,114,0,101,0,10,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,47,0,10,0,46,0,118,0,111,0,99,0,114,0,99,0,10,0,110,0,111,0,100,0,101,0,95,0,109,0,111,0,100,0,117,0,108,0,101,0,115,0,47,0,10,0,109,0,97,0,107,0,101,0,46,0,115,0,104,0,10,0,109,0,97,0,107,0,101,0,46,0,110,0,106,0,115,0,10,0,109,0,105,0,115,0,99,0,47,0,99,0,111,0,118,0,101,0,114,0,97,0,103,0,101,0,46,0,104,0,116,0,109,0,108,0,10,0,96,0,96,0,96,0,10,0]
;
README['utf16be'] = 
[0,35,0,32,0,71,0,101,0,116,0,116,0,105,0,110,0,103,0,32,0,67,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,10,0,10,0,84,0,104,0,101,0,32,0,102,0,105,0,101,0,108,0,100,0,115,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,32,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,32,0,109,0,97,0,110,0,105,0,102,0,101,0,115,0,116,0,32,0,97,0,114,0,101,0,32,0,96,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,44,0,117,0,114,0,108,0,44,0,98,0,121,0,116,0,101,0,115,0,96,0,32,0,40,0,83,0,66,0,67,0,83,0,61,0,49,0,44,0,32,0,68,0,66,0,67,0,83,0,61,0,50,0,41,0,10,0,10,0,96,0,96,0,96,0,62,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,10,0,51,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,69,0,66,0,67,0,68,0,73,0,67,0,47,0,67,0,80,0,48,0,51,0,55,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,52,0,51,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,52,0,51,0,55,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,53,0,48,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,69,0,66,0,67,0,68,0,73,0,67,0,47,0,67,0,80,0,53,0,48,0,48,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,55,0,51,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,55,0,51,0,55,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,55,0,55,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,55,0,55,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,53,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,53,0,48,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,53,0,50,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,53,0,50,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,53,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,53,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,53,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,53,0,55,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,48,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,49,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,49,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,50,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,50,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,51,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,51,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,52,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,52,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,54,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,54,0,57,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,80,0,67,0,47,0,67,0,80,0,56,0,54,0,57,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,55,0,52,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,56,0,55,0,52,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,56,0,55,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,69,0,66,0,67,0,68,0,73,0,67,0,47,0,67,0,80,0,56,0,55,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,57,0,51,0,50,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,57,0,51,0,50,0,46,0,84,0,88,0,84,0,44,0,50,0,10,0,57,0,51,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,57,0,51,0,54,0,46,0,84,0,88,0,84,0,44,0,50,0,10,0,57,0,52,0,57,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,57,0,52,0,57,0,46,0,84,0,88,0,84,0,44,0,50,0,10,0,57,0,53,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,57,0,53,0,48,0,46,0,84,0,88,0,84,0,44,0,50,0,10,0,49,0,48,0,50,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,69,0,66,0,67,0,68,0,73,0,67,0,47,0,67,0,80,0,49,0,48,0,50,0,54,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,48,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,49,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,49,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,50,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,50,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,51,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,51,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,52,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,52,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,54,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,55,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,50,0,53,0,56,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,87,0,73,0,78,0,68,0,79,0,87,0,83,0,47,0,67,0,80,0,49,0,50,0,53,0,56,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,96,0,96,0,96,0,10,0,10,0,78,0,111,0,116,0,101,0,32,0,116,0,104,0,97,0,116,0,32,0,116,0,104,0,101,0,32,0,87,0,105,0,110,0,100,0,111,0,119,0,115,0,32,0,114,0,101,0,110,0,100,0,101,0,114,0,105,0,110,0,103,0,32,0,105,0,115,0,32,0,117,0,115,0,101,0,100,0,32,0,102,0,111,0,114,0,32,0,116,0,104,0,101,0,32,0,77,0,97,0,99,0,32,0,99,0,111,0,100,0,101,0,32,0,112,0,97,0,103,0,101,0,115,0,46,0,32,0,32,0,84,0,104,0,101,0,32,0,112,0,114,0,105,0,109,0,97,0,114,0,121,0,10,0,100,0,105,0,102,0,102,0,101,0,114,0,101,0,110,0,99,0,101,0,32,0,105,0,115,0,32,0,116,0,104,0,101,0,32,0,117,0,115,0,101,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,32,0,112,0,114,0,105,0,118,0,97,0,116,0,101,0,32,0,96,0,48,0,120,0,70,0,56,0,70,0,70,0,96,0,32,0,99,0,111,0,100,0,101,0,32,0,40,0,119,0,104,0,105,0,99,0,104,0,32,0,114,0,101,0,110,0,100,0,101,0,114,0,115,0,32,0,97,0,115,0,32,0,97,0,110,0,32,0,65,0,112,0,112,0,108,0,101,0,10,0,108,0,111,0,103,0,111,0,32,0,111,0,110,0,32,0,109,0,97,0,99,0,115,0,32,0,98,0,117,0,116,0,32,0,97,0,115,0,32,0,103,0,97,0,114,0,98,0,97,0,103,0,101,0,32,0,111,0,110,0,32,0,111,0,116,0,104,0,101,0,114,0,32,0,111,0,112,0,101,0,114,0,97,0,116,0,105,0,110,0,103,0,32,0,115,0,121,0,115,0,116,0,101,0,109,0,115,0,41,0,46,0,32,0,32,0,73,0,116,0,32,0,109,0,97,0,121,0,32,0,98,0,101,0,32,0,100,0,101,0,115,0,105,0,114,0,97,0,98,0,108,0,101,0,10,0,116,0,111,0,32,0,102,0,97,0,108,0,108,0,32,0,98,0,97,0,99,0,107,0,32,0,116,0,111,0,32,0,116,0,104,0,101,0,32,0,98,0,101,0,104,0,97,0,118,0,105,0,111,0,114,0,44,0,32,0,105,0,110,0,32,0,119,0,104,0,105,0,99,0,104,0,32,0,99,0,97,0,115,0,101,0,32,0,116,0,104,0,101,0,32,0,102,0,105,0,108,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,117,0,110,0,100,0,101,0,114,0,32,0,65,0,80,0,80,0,76,0,69,0,32,0,97,0,110,0,100,0,32,0,110,0,111,0,116,0,10,0,77,0,73,0,67,0,83,0,70,0,84,0,46,0,32,0,32,0,67,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,97,0,110,0,32,0,97,0,98,0,115,0,111,0,108,0,117,0,116,0,101,0,32,0,112,0,97,0,105,0,110,0,32,0,58,0,47,0,10,0,10,0,96,0,96,0,96,0,62,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,10,0,49,0,48,0,48,0,48,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,77,0,65,0,67,0,47,0,82,0,79,0,77,0,65,0,78,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,48,0,48,0,48,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,77,0,65,0,67,0,47,0,71,0,82,0,69,0,69,0,75,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,48,0,48,0,48,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,77,0,65,0,67,0,47,0,67,0,89,0,82,0,73,0,76,0,76,0,73,0,67,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,48,0,48,0,50,0,57,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,77,0,65,0,67,0,47,0,76,0,65,0,84,0,73,0,78,0,50,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,48,0,48,0,55,0,57,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,77,0,65,0,67,0,47,0,73,0,67,0,69,0,76,0,65,0,78,0,68,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,49,0,48,0,48,0,56,0,49,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,86,0,69,0,78,0,68,0,79,0,82,0,83,0,47,0,77,0,73,0,67,0,83,0,70,0,84,0,47,0,77,0,65,0,67,0,47,0,84,0,85,0,82,0,75,0,73,0,83,0,72,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,110,0,117,0,109,0,98,0,101,0,114,0,105,0,110,0,103,0,32,0,115,0,99,0,104,0,101,0,109,0,101,0,32,0,102,0,111,0,114,0,32,0,116,0,104,0,101,0,32,0,96,0,73,0,83,0,79,0,45,0,56,0,56,0,53,0,57,0,45,0,88,0,96,0,32,0,115,0,101,0,114,0,105,0,101,0,115,0,32,0,105,0,115,0,32,0,96,0,50,0,56,0,53,0,57,0,48,0,32,0,43,0,32,0,88,0,96,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,10,0,50,0,56,0,53,0,57,0,49,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,50,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,50,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,51,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,51,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,52,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,52,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,54,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,55,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,55,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,56,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,56,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,53,0,57,0,57,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,57,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,54,0,48,0,48,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,48,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,54,0,48,0,49,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,49,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,54,0,48,0,51,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,51,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,54,0,48,0,52,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,52,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,54,0,48,0,53,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,53,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,50,0,56,0,54,0,48,0,54,0,44,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,119,0,119,0,119,0,46,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,46,0,111,0,114,0,103,0,47,0,80,0,117,0,98,0,108,0,105,0,99,0,47,0,77,0,65,0,80,0,80,0,73,0,78,0,71,0,83,0,47,0,73,0,83,0,79,0,56,0,56,0,53,0,57,0,47,0,56,0,56,0,53,0,57,0,45,0,49,0,54,0,46,0,84,0,88,0,84,0,44,0,49,0,10,0,96,0,96,0,96,0,10,0,10,0,35,0,35,0,32,0,71,0,101,0,110,0,101,0,114,0,97,0,116,0,101,0,100,0,32,0,67,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,10,0,10,0,84,0,104,0,101,0,32,0,102,0,111,0,108,0,108,0,111,0,119,0,105,0,110,0,103,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,97,0,118,0,97,0,105,0,108,0,97,0,98,0,108,0,101,0,32,0,105,0,110,0,32,0,46,0,78,0,69,0,84,0,32,0,111,0,110,0,32,0,87,0,105,0,110,0,100,0,111,0,119,0,115,0,58,0,10,0,10,0,45,0,32,0,55,0,48,0,56,0,32,0,32,0,32,0,65,0,114,0,97,0,98,0,105,0,99,0,32,0,40,0,65,0,83,0,77,0,79,0,32,0,55,0,48,0,56,0,41,0,10,0,45,0,32,0,55,0,50,0,48,0,32,0,32,0,32,0,65,0,114,0,97,0,98,0,105,0,99,0,32,0,40,0,84,0,114,0,97,0,110,0,115,0,112,0,97,0,114,0,101,0,110,0,116,0,32,0,65,0,83,0,77,0,79,0,41,0,59,0,32,0,65,0,114,0,97,0,98,0,105,0,99,0,32,0,40,0,68,0,79,0,83,0,41,0,10,0,45,0,32,0,56,0,53,0,56,0,32,0,32,0,32,0,79,0,69,0,77,0,32,0,77,0,117,0,108,0,116,0,105,0,108,0,105,0,110,0,103,0,117,0,97,0,108,0,32,0,76,0,97,0,116,0,105,0,110,0,32,0,49,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,10,0,45,0,32,0,56,0,55,0,48,0,32,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,77,0,117,0,108,0,116,0,105,0,108,0,105,0,110,0,103,0,117,0,97,0,108,0,47,0,82,0,79,0,69,0,67,0,69,0,32,0,40,0,76,0,97,0,116,0,105,0,110,0,32,0,50,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,77,0,117,0,108,0,116,0,105,0,108,0,105,0,110,0,103,0,117,0,97,0,108,0,32,0,76,0,97,0,116,0,105,0,110,0,32,0,50,0,10,0,45,0,32,0,49,0,48,0,52,0,55,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,76,0,97,0,116,0,105,0,110,0,32,0,49,0,47,0,79,0,112,0,101,0,110,0,32,0,83,0,121,0,115,0,116,0,101,0,109,0,10,0,45,0,32,0,49,0,49,0,52,0,48,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,85,0,83,0,45,0,67,0,97,0,110,0,97,0,100,0,97,0,32,0,40,0,48,0,51,0,55,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,85,0,83,0,45,0,67,0,97,0,110,0,97,0,100,0,97,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,49,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,71,0,101,0,114,0,109,0,97,0,110,0,121,0,32,0,40,0,50,0,48,0,50,0,55,0,51,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,71,0,101,0,114,0,109,0,97,0,110,0,121,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,50,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,68,0,101,0,110,0,109,0,97,0,114,0,107,0,45,0,78,0,111,0,114,0,119,0,97,0,121,0,32,0,40,0,50,0,48,0,50,0,55,0,55,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,68,0,101,0,110,0,109,0,97,0,114,0,107,0,45,0,78,0,111,0,114,0,119,0,97,0,121,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,51,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,70,0,105,0,110,0,108,0,97,0,110,0,100,0,45,0,83,0,119,0,101,0,100,0,101,0,110,0,32,0,40,0,50,0,48,0,50,0,55,0,56,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,70,0,105,0,110,0,108,0,97,0,110,0,100,0,45,0,83,0,119,0,101,0,100,0,101,0,110,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,52,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,73,0,116,0,97,0,108,0,121,0,32,0,40,0,50,0,48,0,50,0,56,0,48,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,73,0,116,0,97,0,108,0,121,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,53,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,76,0,97,0,116,0,105,0,110,0,32,0,65,0,109,0,101,0,114,0,105,0,99,0,97,0,45,0,83,0,112,0,97,0,105,0,110,0,32,0,40,0,50,0,48,0,50,0,56,0,52,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,83,0,112,0,97,0,105,0,110,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,54,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,85,0,110,0,105,0,116,0,101,0,100,0,32,0,75,0,105,0,110,0,103,0,100,0,111,0,109,0,32,0,40,0,50,0,48,0,50,0,56,0,53,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,85,0,75,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,55,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,70,0,114,0,97,0,110,0,99,0,101,0,32,0,40,0,50,0,48,0,50,0,57,0,55,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,70,0,114,0,97,0,110,0,99,0,101,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,56,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,73,0,110,0,116,0,101,0,114,0,110,0,97,0,116,0,105,0,111,0,110,0,97,0,108,0,32,0,40,0,53,0,48,0,48,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,73,0,110,0,116,0,101,0,114,0,110,0,97,0,116,0,105,0,111,0,110,0,97,0,108,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,49,0,52,0,57,0,32,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,73,0,99,0,101,0,108,0,97,0,110,0,100,0,105,0,99,0,32,0,40,0,50,0,48,0,56,0,55,0,49,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,59,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,40,0,73,0,99,0,101,0,108,0,97,0,110,0,100,0,105,0,99,0,45,0,69,0,117,0,114,0,111,0,41,0,10,0,45,0,32,0,49,0,51,0,54,0,49,0,32,0,32,0,75,0,111,0,114,0,101,0,97,0,110,0,32,0,40,0,74,0,111,0,104,0,97,0,98,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,48,0,49,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,48,0,50,0,32,0,77,0,65,0,67,0,32,0,84,0,114,0,97,0,100,0,105,0,116,0,105,0,111,0,110,0,97,0,108,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,40,0,66,0,105,0,103,0,53,0,41,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,84,0,114,0,97,0,100,0,105,0,116,0,105,0,111,0,110,0,97,0,108,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,48,0,51,0,32,0,75,0,111,0,114,0,101,0,97,0,110,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,48,0,52,0,32,0,65,0,114,0,97,0,98,0,105,0,99,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,48,0,53,0,32,0,72,0,101,0,98,0,114,0,101,0,119,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,48,0,56,0,32,0,77,0,65,0,67,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,40,0,71,0,66,0,32,0,50,0,51,0,49,0,50,0,41,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,49,0,48,0,32,0,82,0,111,0,109,0,97,0,110,0,105,0,97,0,110,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,49,0,55,0,32,0,85,0,107,0,114,0,97,0,105,0,110,0,105,0,97,0,110,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,50,0,49,0,32,0,84,0,104,0,97,0,105,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,49,0,48,0,48,0,56,0,50,0,32,0,67,0,114,0,111,0,97,0,116,0,105,0,97,0,110,0,32,0,40,0,77,0,97,0,99,0,41,0,10,0,45,0,32,0,50,0,48,0,48,0,48,0,48,0,32,0,67,0,78,0,83,0,32,0,84,0,97,0,105,0,119,0,97,0,110,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,84,0,114,0,97,0,100,0,105,0,116,0,105,0,111,0,110,0,97,0,108,0,32,0,40,0,67,0,78,0,83,0,41,0,10,0,45,0,32,0,50,0,48,0,48,0,48,0,49,0,32,0,84,0,67,0,65,0,32,0,84,0,97,0,105,0,119,0,97,0,110,0,10,0,45,0,32,0,50,0,48,0,48,0,48,0,50,0,32,0,69,0,116,0,101,0,110,0,32,0,84,0,97,0,105,0,119,0,97,0,110,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,84,0,114,0,97,0,100,0,105,0,116,0,105,0,111,0,110,0,97,0,108,0,32,0,40,0,69,0,116,0,101,0,110,0,41,0,10,0,45,0,32,0,50,0,48,0,48,0,48,0,51,0,32,0,73,0,66,0,77,0,53,0,53,0,53,0,48,0,32,0,84,0,97,0,105,0,119,0,97,0,110,0,10,0,45,0,32,0,50,0,48,0,48,0,48,0,52,0,32,0,84,0,101,0,108,0,101,0,84,0,101,0,120,0,116,0,32,0,84,0,97,0,105,0,119,0,97,0,110,0,10,0,45,0,32,0,50,0,48,0,48,0,48,0,53,0,32,0,87,0,97,0,110,0,103,0,32,0,84,0,97,0,105,0,119,0,97,0,110,0,10,0,45,0,32,0,50,0,48,0,49,0,48,0,53,0,32,0,73,0,65,0,53,0,32,0,40,0,73,0,82,0,86,0,32,0,73,0,110,0,116,0,101,0,114,0,110,0,97,0,116,0,105,0,111,0,110,0,97,0,108,0,32,0,65,0,108,0,112,0,104,0,97,0,98,0,101,0,116,0,32,0,78,0,111,0,46,0,32,0,53,0,44,0,32,0,55,0,45,0,98,0,105,0,116,0,41,0,59,0,32,0,87,0,101,0,115,0,116,0,101,0,114,0,110,0,32,0,69,0,117,0,114,0,111,0,112,0,101,0,97,0,110,0,32,0,40,0,73,0,65,0,53,0,41,0,10,0,45,0,32,0,50,0,48,0,49,0,48,0,54,0,32,0,73,0,65,0,53,0,32,0,71,0,101,0,114,0,109,0,97,0,110,0,32,0,40,0,55,0,45,0,98,0,105,0,116,0,41,0,10,0,45,0,32,0,50,0,48,0,49,0,48,0,55,0,32,0,73,0,65,0,53,0,32,0,83,0,119,0,101,0,100,0,105,0,115,0,104,0,32,0,40,0,55,0,45,0,98,0,105,0,116,0,41,0,10,0,45,0,32,0,50,0,48,0,49,0,48,0,56,0,32,0,73,0,65,0,53,0,32,0,78,0,111,0,114,0,119,0,101,0,103,0,105,0,97,0,110,0,32,0,40,0,55,0,45,0,98,0,105,0,116,0,41,0,10,0,45,0,32,0,50,0,48,0,50,0,54,0,49,0,32,0,84,0,46,0,54,0,49,0,10,0,45,0,32,0,50,0,48,0,50,0,54,0,57,0,32,0,73,0,83,0,79,0,32,0,54,0,57,0,51,0,55,0,32,0,78,0,111,0,110,0,45,0,83,0,112,0,97,0,99,0,105,0,110,0,103,0,32,0,65,0,99,0,99,0,101,0,110,0,116,0,10,0,45,0,32,0,50,0,48,0,50,0,55,0,51,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,71,0,101,0,114,0,109,0,97,0,110,0,121,0,10,0,45,0,32,0,50,0,48,0,50,0,55,0,55,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,68,0,101,0,110,0,109,0,97,0,114,0,107,0,45,0,78,0,111,0,114,0,119,0,97,0,121,0,10,0,45,0,32,0,50,0,48,0,50,0,55,0,56,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,70,0,105,0,110,0,108,0,97,0,110,0,100,0,45,0,83,0,119,0,101,0,100,0,101,0,110,0,10,0,45,0,32,0,50,0,48,0,50,0,56,0,48,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,73,0,116,0,97,0,108,0,121,0,10,0,45,0,32,0,50,0,48,0,50,0,56,0,52,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,76,0,97,0,116,0,105,0,110,0,32,0,65,0,109,0,101,0,114,0,105,0,99,0,97,0,45,0,83,0,112,0,97,0,105,0,110,0,10,0,45,0,32,0,50,0,48,0,50,0,56,0,53,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,85,0,110,0,105,0,116,0,101,0,100,0,32,0,75,0,105,0,110,0,103,0,100,0,111,0,109,0,10,0,45,0,32,0,50,0,48,0,50,0,57,0,48,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,75,0,97,0,116,0,97,0,107,0,97,0,110,0,97,0,32,0,69,0,120,0,116,0,101,0,110,0,100,0,101,0,100,0,10,0,45,0,32,0,50,0,48,0,50,0,57,0,55,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,70,0,114,0,97,0,110,0,99,0,101,0,10,0,45,0,32,0,50,0,48,0,52,0,50,0,48,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,65,0,114,0,97,0,98,0,105,0,99,0,10,0,45,0,32,0,50,0,48,0,52,0,50,0,51,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,71,0,114,0,101,0,101,0,107,0,10,0,45,0,32,0,50,0,48,0,52,0,50,0,52,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,72,0,101,0,98,0,114,0,101,0,119,0,10,0,45,0,32,0,50,0,48,0,56,0,51,0,51,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,75,0,111,0,114,0,101,0,97,0,110,0,32,0,69,0,120,0,116,0,101,0,110,0,100,0,101,0,100,0,10,0,45,0,32,0,50,0,48,0,56,0,51,0,56,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,84,0,104,0,97,0,105,0,10,0,45,0,32,0,50,0,48,0,56,0,54,0,54,0,32,0,82,0,117,0,115,0,115,0,105,0,97,0,110,0,32,0,40,0,75,0,79,0,73,0,56,0,45,0,82,0,41,0,59,0,32,0,67,0,121,0,114,0,105,0,108,0,108,0,105,0,99,0,32,0,40,0,75,0,79,0,73,0,56,0,45,0,82,0,41,0,10,0,45,0,32,0,50,0,48,0,56,0,55,0,49,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,73,0,99,0,101,0,108,0,97,0,110,0,100,0,105,0,99,0,10,0,45,0,32,0,50,0,48,0,56,0,56,0,48,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,67,0,121,0,114,0,105,0,108,0,108,0,105,0,99,0,32,0,82,0,117,0,115,0,115,0,105,0,97,0,110,0,10,0,45,0,32,0,50,0,48,0,57,0,48,0,53,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,84,0,117,0,114,0,107,0,105,0,115,0,104,0,10,0,45,0,32,0,50,0,48,0,57,0,50,0,52,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,76,0,97,0,116,0,105,0,110,0,32,0,49,0,47,0,79,0,112,0,101,0,110,0,32,0,83,0,121,0,115,0,116,0,101,0,109,0,32,0,40,0,49,0,48,0,52,0,55,0,32,0,43,0,32,0,69,0,117,0,114,0,111,0,32,0,115,0,121,0,109,0,98,0,111,0,108,0,41,0,10,0,45,0,32,0,50,0,48,0,57,0,51,0,50,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,40,0,74,0,73,0,83,0,32,0,48,0,50,0,48,0,56,0,45,0,49,0,57,0,57,0,48,0,32,0,97,0,110,0,100,0,32,0,48,0,50,0,49,0,50,0,45,0,49,0,57,0,57,0,48,0,41,0,10,0,45,0,32,0,50,0,48,0,57,0,51,0,54,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,40,0,71,0,66,0,50,0,51,0,49,0,50,0,41,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,40,0,71,0,66,0,50,0,51,0,49,0,50,0,45,0,56,0,48,0,41,0,10,0,45,0,32,0,50,0,48,0,57,0,52,0,57,0,32,0,75,0,111,0,114,0,101,0,97,0,110,0,32,0,87,0,97,0,110,0,115,0,117,0,110,0,103,0,10,0,45,0,32,0,50,0,49,0,48,0,50,0,53,0,32,0,73,0,66,0,77,0,32,0,69,0,66,0,67,0,68,0,73,0,67,0,32,0,67,0,121,0,114,0,105,0,108,0,108,0,105,0,99,0,32,0,83,0,101,0,114,0,98,0,105,0,97,0,110,0,45,0,66,0,117,0,108,0,103,0,97,0,114,0,105,0,97,0,110,0,10,0,45,0,32,0,50,0,49,0,48,0,50,0,55,0,32,0,69,0,120,0,116,0,101,0,110,0,100,0,101,0,100,0,47,0,69,0,120,0,116,0,32,0,65,0,108,0,112,0,104,0,97,0,32,0,76,0,111,0,119,0,101,0,114,0,99,0,97,0,115,0,101,0,10,0,45,0,32,0,50,0,49,0,56,0,54,0,54,0,32,0,85,0,107,0,114,0,97,0,105,0,110,0,105,0,97,0,110,0,32,0,40,0,75,0,79,0,73,0,56,0,45,0,85,0,41,0,59,0,32,0,67,0,121,0,114,0,105,0,108,0,108,0,105,0,99,0,32,0,40,0,75,0,79,0,73,0,56,0,45,0,85,0,41,0,10,0,45,0,32,0,50,0,57,0,48,0,48,0,49,0,32,0,69,0,117,0,114,0,111,0,112,0,97,0,32,0,51,0,10,0,45,0,32,0,51,0,56,0,53,0,57,0,56,0,32,0,73,0,83,0,79,0,32,0,56,0,56,0,53,0,57,0,45,0,56,0,32,0,72,0,101,0,98,0,114,0,101,0,119,0,59,0,32,0,72,0,101,0,98,0,114,0,101,0,119,0,32,0,40,0,73,0,83,0,79,0,45,0,76,0,111,0,103,0,105,0,99,0,97,0,108,0,41,0,10,0,45,0,32,0,53,0,48,0,50,0,50,0,48,0,32,0,73,0,83,0,79,0,32,0,50,0,48,0,50,0,50,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,119,0,105,0,116,0,104,0,32,0,110,0,111,0,32,0,104,0,97,0,108,0,102,0,119,0,105,0,100,0,116,0,104,0,32,0,75,0,97,0,116,0,97,0,107,0,97,0,110,0,97,0,59,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,40,0,74,0,73,0,83,0,41,0,10,0,45,0,32,0,53,0,48,0,50,0,50,0,49,0,32,0,73,0,83,0,79,0,32,0,50,0,48,0,50,0,50,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,119,0,105,0,116,0,104,0,32,0,104,0,97,0,108,0,102,0,119,0,105,0,100,0,116,0,104,0,32,0,75,0,97,0,116,0,97,0,107,0,97,0,110,0,97,0,59,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,40,0,74,0,73,0,83,0,45,0,65,0,108,0,108,0,111,0,119,0,32,0,49,0,32,0,98,0,121,0,116,0,101,0,32,0,75,0,97,0,110,0,97,0,41,0,10,0,45,0,32,0,53,0,48,0,50,0,50,0,50,0,32,0,73,0,83,0,79,0,32,0,50,0,48,0,50,0,50,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,74,0,73,0,83,0,32,0,88,0,32,0,48,0,50,0,48,0,49,0,45,0,49,0,57,0,56,0,57,0,59,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,32,0,40,0,74,0,73,0,83,0,45,0,65,0,108,0,108,0,111,0,119,0,32,0,49,0,32,0,98,0,121,0,116,0,101,0,32,0,75,0,97,0,110,0,97,0,32,0,45,0,32,0,83,0,79,0,47,0,83,0,73,0,41,0,10,0,45,0,32,0,53,0,48,0,50,0,50,0,53,0,32,0,73,0,83,0,79,0,32,0,50,0,48,0,50,0,50,0,32,0,75,0,111,0,114,0,101,0,97,0,110,0,10,0,45,0,32,0,53,0,48,0,50,0,50,0,55,0,32,0,73,0,83,0,79,0,32,0,50,0,48,0,50,0,50,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,40,0,73,0,83,0,79,0,32,0,50,0,48,0,50,0,50,0,41,0,10,0,45,0,32,0,53,0,49,0,57,0,51,0,50,0,32,0,69,0,85,0,67,0,32,0,74,0,97,0,112,0,97,0,110,0,101,0,115,0,101,0,10,0,45,0,32,0,53,0,49,0,57,0,51,0,54,0,32,0,69,0,85,0,67,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,40,0,69,0,85,0,67,0,41,0,10,0,45,0,32,0,53,0,49,0,57,0,52,0,57,0,32,0,69,0,85,0,67,0,32,0,75,0,111,0,114,0,101,0,97,0,110,0,10,0,45,0,32,0,53,0,50,0,57,0,51,0,54,0,32,0,72,0,90,0,45,0,71,0,66,0,50,0,51,0,49,0,50,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,40,0,72,0,90,0,41,0,10,0,45,0,32,0,53,0,52,0,57,0,51,0,54,0,32,0,87,0,105,0,110,0,100,0,111,0,119,0,115,0,32,0,88,0,80,0,32,0,97,0,110,0,100,0,32,0,108,0,97,0,116,0,101,0,114,0,58,0,32,0,71,0,66,0,49,0,56,0,48,0,51,0,48,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,40,0,52,0,32,0,98,0,121,0,116,0,101,0,41,0,59,0,32,0,67,0,104,0,105,0,110,0,101,0,115,0,101,0,32,0,83,0,105,0,109,0,112,0,108,0,105,0,102,0,105,0,101,0,100,0,32,0,40,0,71,0,66,0,49,0,56,0,48,0,51,0,48,0,41,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,50,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,68,0,101,0,118,0,97,0,110,0,97,0,103,0,97,0,114,0,105,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,51,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,66,0,101,0,110,0,103,0,97,0,108,0,105,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,52,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,84,0,97,0,109,0,105,0,108,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,53,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,84,0,101,0,108,0,117,0,103,0,117,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,54,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,65,0,115,0,115,0,97,0,109,0,101,0,115,0,101,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,55,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,79,0,114,0,105,0,121,0,97,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,56,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,75,0,97,0,110,0,110,0,97,0,100,0,97,0,10,0,45,0,32,0,53,0,55,0,48,0,48,0,57,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,77,0,97,0,108,0,97,0,121,0,97,0,108,0,97,0,109,0,10,0,45,0,32,0,53,0,55,0,48,0,49,0,48,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,71,0,117,0,106,0,97,0,114,0,97,0,116,0,105,0,10,0,45,0,32,0,53,0,55,0,48,0,49,0,49,0,32,0,73,0,83,0,67,0,73,0,73,0,32,0,80,0,117,0,110,0,106,0,97,0,98,0,105,0,10,0,10,0,96,0,96,0,96,0,62,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,10,0,55,0,48,0,56,0,44,0,44,0,49,0,10,0,55,0,50,0,48,0,44,0,44,0,49,0,10,0,56,0,53,0,56,0,44,0,44,0,49,0,10,0,56,0,55,0,48,0,44,0,44,0,49,0,10,0,49,0,48,0,52,0,55,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,48,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,49,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,50,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,51,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,52,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,53,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,54,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,55,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,56,0,44,0,44,0,49,0,10,0,49,0,49,0,52,0,57,0,44,0,44,0,49,0,10,0,49,0,51,0,54,0,49,0,44,0,44,0,50,0,10,0,49,0,48,0,48,0,48,0,49,0,44,0,44,0,50,0,10,0,49,0,48,0,48,0,48,0,50,0,44,0,44,0,50,0,10,0,49,0,48,0,48,0,48,0,51,0,44,0,44,0,50,0,10,0,49,0,48,0,48,0,48,0,52,0,44,0,44,0,49,0,10,0,49,0,48,0,48,0,48,0,53,0,44,0,44,0,49,0,10,0,49,0,48,0,48,0,48,0,56,0,44,0,44,0,50,0,10,0,49,0,48,0,48,0,49,0,48,0,44,0,44,0,49,0,10,0,49,0,48,0,48,0,49,0,55,0,44,0,44,0,49,0,10,0,49,0,48,0,48,0,50,0,49,0,44,0,44,0,49,0,10,0,49,0,48,0,48,0,56,0,50,0,44,0,44,0,49,0,10,0,50,0,48,0,48,0,48,0,48,0,44,0,44,0,50,0,10,0,50,0,48,0,48,0,48,0,49,0,44,0,44,0,50,0,10,0,50,0,48,0,48,0,48,0,50,0,44,0,44,0,50,0,10,0,50,0,48,0,48,0,48,0,51,0,44,0,44,0,50,0,10,0,50,0,48,0,48,0,48,0,52,0,44,0,44,0,50,0,10,0,50,0,48,0,48,0,48,0,53,0,44,0,44,0,50,0,10,0,50,0,48,0,49,0,48,0,53,0,44,0,44,0,49,0,10,0,50,0,48,0,49,0,48,0,54,0,44,0,44,0,49,0,10,0,50,0,48,0,49,0,48,0,55,0,44,0,44,0,49,0,10,0,50,0,48,0,49,0,48,0,56,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,54,0,49,0,44,0,44,0,50,0,10,0,50,0,48,0,50,0,54,0,57,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,55,0,51,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,55,0,55,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,55,0,56,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,56,0,48,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,56,0,52,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,56,0,53,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,57,0,48,0,44,0,44,0,49,0,10,0,50,0,48,0,50,0,57,0,55,0,44,0,44,0,49,0,10,0,50,0,48,0,52,0,50,0,48,0,44,0,44,0,49,0,10,0,50,0,48,0,52,0,50,0,51,0,44,0,44,0,49,0,10,0,50,0,48,0,52,0,50,0,52,0,44,0,44,0,49,0,10,0,50,0,48,0,56,0,51,0,51,0,44,0,44,0,49,0,10,0,50,0,48,0,56,0,51,0,56,0,44,0,44,0,49,0,10,0,50,0,48,0,56,0,54,0,54,0,44,0,44,0,49,0,10,0,50,0,48,0,56,0,55,0,49,0,44,0,44,0,49,0,10,0,50,0,48,0,56,0,56,0,48,0,44,0,44,0,49,0,10,0,50,0,48,0,57,0,48,0,53,0,44,0,44,0,49,0,10,0,50,0,48,0,57,0,50,0,52,0,44,0,44,0,49,0,10,0,50,0,48,0,57,0,51,0,50,0,44,0,44,0,50,0,10,0,50,0,48,0,57,0,51,0,54,0,44,0,44,0,50,0,10,0,50,0,48,0,57,0,52,0,57,0,44,0,44,0,50,0,10,0,50,0,49,0,48,0,50,0,53,0,44,0,44,0,49,0,10,0,50,0,49,0,48,0,50,0,55,0,44,0,44,0,49,0,10,0,50,0,49,0,56,0,54,0,54,0,44,0,44,0,49,0,10,0,50,0,57,0,48,0,48,0,49,0,44,0,44,0,49,0,10,0,51,0,56,0,53,0,57,0,56,0,44,0,44,0,49,0,10,0,53,0,48,0,50,0,50,0,48,0,44,0,44,0,50,0,10,0,53,0,48,0,50,0,50,0,49,0,44,0,44,0,50,0,10,0,53,0,48,0,50,0,50,0,50,0,44,0,44,0,50,0,10,0,53,0,48,0,50,0,50,0,53,0,44,0,44,0,50,0,10,0,53,0,48,0,50,0,50,0,55,0,44,0,44,0,50,0,10,0,53,0,49,0,57,0,51,0,50,0,44,0,44,0,50,0,10,0,53,0,49,0,57,0,51,0,54,0,44,0,44,0,50,0,10,0,53,0,49,0,57,0,52,0,57,0,44,0,44,0,50,0,10,0,53,0,50,0,57,0,51,0,54,0,44,0,44,0,50,0,10,0,53,0,52,0,57,0,51,0,54,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,50,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,51,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,52,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,53,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,54,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,55,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,56,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,48,0,57,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,49,0,48,0,44,0,44,0,50,0,10,0,53,0,55,0,48,0,49,0,49,0,44,0,44,0,50,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,102,0,111,0,108,0,108,0,111,0,119,0,105,0,110,0,103,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,100,0,101,0,112,0,101,0,110,0,100,0,101,0,110,0,99,0,105,0,101,0,115,0,32,0,102,0,111,0,114,0,32,0,86,0,105,0,115,0,117,0,97,0,108,0,32,0,70,0,111,0,120,0,80,0,114,0,111,0,58,0,10,0,10,0,45,0,32,0,54,0,50,0,48,0,32,0,77,0,97,0,122,0,111,0,118,0,105,0,97,0,32,0,40,0,80,0,111,0,108,0,105,0,115,0,104,0,41,0,32,0,77,0,83,0,45,0,68,0,79,0,83,0,10,0,45,0,32,0,56,0,57,0,53,0,32,0,75,0,97,0,109,0,101,0,110,0,105,0,99,0,107,0,253,0,32,0,40,0,67,0,122,0,101,0,99,0,104,0,41,0,32,0,77,0,83,0,45,0,68,0,79,0,83,0,10,0,10,0,96,0,96,0,96,0,62,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,10,0,54,0,50,0,48,0,44,0,44,0,49,0,10,0,56,0,57,0,53,0,44,0,44,0,49,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,107,0,110,0,111,0,119,0,110,0,32,0,109,0,105,0,115,0,115,0,105,0,110,0,103,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,101,0,110,0,117,0,109,0,101,0,114,0,97,0,116,0,101,0,100,0,32,0,105,0,110,0,32,0,116,0,104,0,101,0,32,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,10,0,10,0,35,0,35,0,32,0,66,0,117,0,105,0,108,0,100,0,105,0,110,0,103,0,32,0,78,0,111,0,116,0,101,0,115,0,10,0,10,0,84,0,104,0,101,0,32,0,115,0,99,0,114,0,105,0,112,0,116,0,32,0,96,0,109,0,97,0,107,0,101,0,46,0,115,0,104,0,96,0,32,0,40,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,100,0,32,0,108,0,97,0,116,0,101,0,114,0,41,0,32,0,119,0,105,0,108,0,108,0,32,0,103,0,101,0,116,0,32,0,116,0,104,0,101,0,115,0,101,0,32,0,102,0,105,0,108,0,101,0,115,0,32,0,97,0,110,0,100,0,32,0,109,0,97,0,115,0,115,0,97,0,103,0,101,0,32,0,116,0,104,0,101,0,32,0,100,0,97,0,116,0,97,0,10,0,40,0,112,0,114,0,105,0,110,0,116,0,105,0,110,0,103,0,32,0,99,0,111,0,100,0,101,0,45,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,32,0,112,0,97,0,105,0,114,0,115,0,41,0,46,0,32,0,32,0,84,0,104,0,101,0,32,0,101,0,118,0,101,0,110,0,116,0,117,0,97,0,108,0,32,0,116,0,97,0,98,0,108,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,100,0,114,0,111,0,112,0,112,0,101,0,100,0,32,0,105,0,110,0,32,0,116,0,104,0,101,0,32,0,112,0,97,0,116,0,104,0,115,0,10,0,96,0,46,0,47,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,47,0,60,0,67,0,79,0,68,0,69,0,80,0,65,0,71,0,69,0,62,0,46,0,84,0,66,0,76,0,96,0,46,0,32,0,32,0,70,0,111,0,114,0,32,0,101,0,120,0,97,0,109,0,112,0,108,0,101,0,44,0,32,0,116,0,104,0,101,0,32,0,108,0,97,0,115,0,116,0,32,0,49,0,48,0,32,0,108,0,105,0,110,0,101,0,115,0,32,0,111,0,102,0,32,0,96,0,49,0,48,0,48,0,48,0,48,0,46,0,84,0,66,0,76,0,96,0,32,0,97,0,114,0,101,0,10,0,10,0,96,0,96,0,96,0,62,0,10,0,48,0,120,0,70,0,54,0,9,0,48,0,120,0,48,0,50,0,67,0,54,0,10,0,48,0,120,0,70,0,55,0,9,0,48,0,120,0,48,0,50,0,68,0,67,0,10,0,48,0,120,0,70,0,56,0,9,0,48,0,120,0,48,0,48,0,65,0,70,0,10,0,48,0,120,0,70,0,57,0,9,0,48,0,120,0,48,0,50,0,68,0,56,0,10,0,48,0,120,0,70,0,65,0,9,0,48,0,120,0,48,0,50,0,68,0,57,0,10,0,48,0,120,0,70,0,66,0,9,0,48,0,120,0,48,0,50,0,68,0,65,0,10,0,48,0,120,0,70,0,67,0,9,0,48,0,120,0,48,0,48,0,66,0,56,0,10,0,48,0,120,0,70,0,68,0,9,0,48,0,120,0,48,0,50,0,68,0,68,0,10,0,48,0,120,0,70,0,69,0,9,0,48,0,120,0,48,0,50,0,68,0,66,0,10,0,48,0,120,0,70,0,70,0,9,0,48,0,120,0,48,0,50,0,67,0,55,0,10,0,96,0,96,0,96,0,10,0,10,0,119,0,104,0,105,0,99,0,104,0,32,0,105,0,109,0,112,0,108,0,105,0,101,0,115,0,32,0,116,0,104,0,97,0,116,0,32,0,99,0,111,0,100,0,101,0,32,0,48,0,120,0,70,0,54,0,32,0,105,0,115,0,32,0,96,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,48,0,120,0,48,0,50,0,67,0,54,0,41,0,96,0,32,0,97,0,110,0,100,0,32,0,118,0,105,0,99,0,101,0,32,0,118,0,101,0,114,0,115,0,97,0,46,0,10,0,10,0,35,0,35,0,32,0,87,0,105,0,110,0,100,0,111,0,119,0,115,0,45,0,100,0,101,0,112,0,101,0,110,0,100,0,101,0,110,0,116,0,32,0,98,0,117,0,105,0,108,0,100,0,32,0,115,0,116,0,101,0,112,0,10,0,10,0,84,0,111,0,32,0,98,0,117,0,105,0,108,0,100,0,32,0,116,0,104,0,101,0,32,0,115,0,111,0,117,0,114,0,99,0,101,0,115,0,32,0,111,0,110,0,32,0,119,0,105,0,110,0,100,0,111,0,119,0,115,0,44,0,32,0,99,0,111,0,110,0,115,0,117,0,108,0,116,0,32,0,96,0,100,0,111,0,116,0,110,0,101,0,116,0,47,0,77,0,97,0,107,0,101,0,69,0,110,0,99,0,111,0,100,0,105,0,110,0,103,0,46,0,99,0,115,0,96,0,46,0,10,0,10,0,65,0,102,0,116,0,101,0,114,0,32,0,115,0,97,0,118,0,105,0,110,0,103,0,32,0,116,0,104,0,101,0,32,0,115,0,116,0,97,0,110,0,100,0,97,0,114,0,100,0,32,0,111,0,117,0,116,0,112,0,117,0,116,0,32,0,116,0,111,0,32,0,96,0,111,0,117,0,116,0,96,0,44,0,32,0,97,0,32,0,115,0,105,0,109,0,112,0,108,0,101,0,32,0,97,0,119,0,107,0,32,0,115,0,99,0,114,0,105,0,112,0,116,0,32,0,40,0,96,0,100,0,111,0,116,0,110,0,101,0,116,0,46,0,115,0,104,0,96,0,41,0,32,0,116,0,97,0,107,0,101,0,115,0,32,0,99,0,97,0,114,0,101,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,32,0,114,0,101,0,115,0,116,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,100,0,111,0,116,0,110,0,101,0,116,0,46,0,115,0,104,0,10,0,35,0,33,0,47,0,98,0,105,0,110,0,47,0,98,0,97,0,115,0,104,0,10,0,105,0,102,0,32,0,91,0,32,0,33,0,32,0,45,0,101,0,32,0,100,0,111,0,116,0,110,0,101,0,116,0,47,0,111,0,117,0,116,0,32,0,93,0,59,0,32,0,116,0,104,0,101,0,110,0,32,0,101,0,120,0,105,0,116,0,59,0,32,0,102,0,105,0,10,0,60,0,100,0,111,0,116,0,110,0,101,0,116,0,47,0,111,0,117,0,116,0,32,0,116,0,114,0,32,0,45,0,115,0,32,0,39,0,32,0,39,0,32,0,39,0,92,0,116,0,39,0,32,0,124,0,32,0,97,0,119,0,107,0,32,0,39,0,78,0,70,0,62,0,50,0,32,0,123,0,105,0,102,0,40,0,111,0,117,0,116,0,102,0,105,0,108,0,101,0,41,0,32,0,99,0,108,0,111,0,115,0,101,0,40,0,111,0,117,0,116,0,102,0,105,0,108,0,101,0,41,0,59,0,32,0,111,0,117,0,116,0,102,0,105,0,108,0,101,0,61,0,34,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,47,0,34,0,32,0,36,0,49,0,32,0,34,0,46,0,84,0,66,0,76,0,34,0,125,0,32,0,78,0,70,0,61,0,61,0,50,0,32,0,123,0,112,0,114,0,105,0,110,0,116,0,32,0,62,0,32,0,111,0,117,0,116,0,102,0,105,0,108,0,101,0,125,0,39,0,10,0,96,0,96,0,96,0,10,0,10,0,35,0,32,0,66,0,117,0,105,0,108,0,100,0,105,0,110,0,103,0,32,0,116,0,104,0,101,0,32,0,115,0,99,0,114,0,105,0,112,0,116,0,10,0,10,0,96,0,109,0,97,0,107,0,101,0,46,0,110,0,106,0,115,0,96,0,32,0,116,0,97,0,107,0,101,0,115,0,32,0,97,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,32,0,97,0,114,0,103,0,117,0,109,0,101,0,110,0,116,0,44,0,32,0,114,0,101,0,97,0,100,0,115,0,32,0,116,0,104,0,101,0,32,0,99,0,111,0,114,0,114,0,101,0,115,0,112,0,111,0,110,0,100,0,105,0,110,0,103,0,32,0,116,0,97,0,98,0,108,0,101,0,32,0,102,0,105,0,108,0,101,0,32,0,97,0,110,0,100,0,10,0,103,0,101,0,110,0,101,0,114,0,97,0,116,0,101,0,115,0,32,0,74,0,83,0,32,0,99,0,111,0,100,0,101,0,32,0,102,0,111,0,114,0,32,0,101,0,110,0,99,0,111,0,100,0,105,0,110,0,103,0,32,0,97,0,110,0,100,0,32,0,100,0,101,0,99,0,111,0,100,0,105,0,110,0,103,0,58,0,10,0,10,0,35,0,35,0,32,0,82,0,97,0,119,0,32,0,67,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,10,0,10,0,96,0,96,0,96,0,62,0,109,0,97,0,107,0,101,0,46,0,110,0,106,0,115,0,10,0,35,0,33,0,47,0,117,0,115,0,114,0,47,0,98,0,105,0,110,0,47,0,101,0,110,0,118,0,32,0,110,0,111,0,100,0,101,0,10,0,118,0,97,0,114,0,32,0,97,0,114,0,103,0,118,0,32,0,61,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,46,0,97,0,114,0,103,0,118,0,46,0,115,0,108,0,105,0,99,0,101,0,40,0,49,0,41,0,44,0,32,0,102,0,115,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,102,0,115,0,39,0,41,0,59,0,10,0,105,0,102,0,40,0,97,0,114,0,103,0,118,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,60,0,32,0,50,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,99,0,111,0,110,0,115,0,111,0,108,0,101,0,46,0,101,0,114,0,114,0,111,0,114,0,40,0,34,0,117,0,115,0,97,0,103,0,101,0,58,0,32,0,109,0,97,0,107,0,101,0,46,0,110,0,106,0,115,0,32,0,60,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,95,0,105,0,110,0,100,0,101,0,120,0,62,0,32,0,91,0,118,0,97,0,114,0,105,0,97,0,98,0,108,0,101,0,93,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,46,0,101,0,120,0,105,0,116,0,40,0,50,0,50,0,41,0,59,0,32,0,47,0,42,0,32,0,69,0,73,0,78,0,86,0,65,0,76,0,32,0,42,0,47,0,10,0,125,0,10,0,10,0,118,0,97,0,114,0,32,0,99,0,112,0,32,0,61,0,32,0,97,0,114,0,103,0,118,0,91,0,49,0,93,0,59,0,10,0,118,0,97,0,114,0,32,0,106,0,115,0,118,0,97,0,114,0,32,0,61,0,32,0,97,0,114,0,103,0,118,0,91,0,50,0,93,0,32,0,124,0,124,0,32,0,34,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,34,0,59,0,10,0,118,0,97,0,114,0,32,0,120,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,34,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,47,0,34,0,32,0,43,0,32,0,99,0,112,0,32,0,43,0,32,0,34,0,46,0,84,0,66,0,76,0,34,0,44,0,34,0,117,0,116,0,102,0,56,0,34,0,41,0,59,0,10,0,118,0,97,0,114,0,32,0,109,0,97,0,120,0,99,0,112,0,32,0,61,0,32,0,48,0,59,0,10,0,10,0,118,0,97,0,114,0,32,0,121,0,32,0,61,0,32,0,120,0,46,0,115,0,112,0,108,0,105,0,116,0,40,0,34,0,92,0,110,0,34,0,41,0,46,0,109,0,97,0,112,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,122,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,119,0,32,0,61,0,32,0,122,0,46,0,115,0,112,0,108,0,105,0,116,0,40,0,34,0,92,0,116,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,119,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,60,0,32,0,50,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,119,0,59,0,10,0,32,0,32,0,32,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,91,0,78,0,117,0,109,0,98,0,101,0,114,0,40,0,119,0,91,0,48,0,93,0,41,0,44,0,32,0,78,0,117,0,109,0,98,0,101,0,114,0,40,0,119,0,91,0,49,0,93,0,41,0,93,0,59,0,10,0,125,0,41,0,46,0,102,0,105,0,108,0,116,0,101,0,114,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,122,0,41,0,32,0,123,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,62,0,32,0,49,0,59,0,32,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,68,0,66,0,67,0,83,0,32,0,97,0,110,0,100,0,32,0,83,0,66,0,67,0,83,0,32,0,99,0,111,0,100,0,101,0,32,0,103,0,101,0,110,0,101,0,114,0,97,0,116,0,105,0,111,0,110,0,32,0,115,0,116,0,114,0,97,0,116,0,101,0,103,0,105,0,101,0,115,0,32,0,97,0,114,0,101,0,32,0,100,0,105,0,102,0,102,0,101,0,114,0,101,0,110,0,116,0,46,0,32,0,32,0,84,0,104,0,101,0,32,0,109,0,97,0,120,0,105,0,109,0,117,0,109,0,32,0,99,0,111,0,100,0,101,0,32,0,105,0,115,0,10,0,117,0,115,0,101,0,100,0,32,0,116,0,111,0,32,0,100,0,105,0,115,0,116,0,105,0,110,0,103,0,117,0,105,0,115,0,104,0,32,0,40,0,109,0,97,0,120,0,32,0,48,0,120,0,70,0,70,0,32,0,102,0,111,0,114,0,32,0,83,0,66,0,67,0,83,0,41,0,46,0,10,0,10,0,96,0,96,0,96,0,10,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,121,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,105,0,102,0,40,0,121,0,91,0,105,0,93,0,91,0,48,0,93,0,32,0,62,0,32,0,109,0,97,0,120,0,99,0,112,0,41,0,32,0,109,0,97,0,120,0,99,0,112,0,32,0,61,0,32,0,121,0,91,0,105,0,93,0,91,0,48,0,93,0,59,0,10,0,10,0,118,0,97,0,114,0,32,0,101,0,110,0,99,0,32,0,61,0,32,0,123,0,125,0,44,0,32,0,100,0,101,0,99,0,32,0,61,0,32,0,40,0,109,0,97,0,120,0,99,0,112,0,32,0,60,0,32,0,50,0,53,0,54,0,32,0,63,0,32,0,91,0,93,0,32,0,58,0,32,0,123,0,125,0,41,0,59,0,10,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,121,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,100,0,101,0,99,0,91,0,121,0,91,0,105,0,93,0,91,0,48,0,93,0,93,0,32,0,61,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,121,0,91,0,105,0,93,0,91,0,49,0,93,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,101,0,110,0,99,0,91,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,121,0,91,0,105,0,93,0,91,0,49,0,93,0,41,0,93,0,32,0,61,0,32,0,121,0,91,0,105,0,93,0,91,0,48,0,93,0,59,0,10,0,125,0,10,0,10,0,118,0,97,0,114,0,32,0,111,0,100,0,101,0,99,0,44,0,32,0,111,0,101,0,110,0,99,0,44,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,59,0,10,0,105,0,102,0,40,0,109,0,97,0,120,0,99,0,112,0,32,0,60,0,32,0,50,0,53,0,54,0,41,0,32,0,123,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,32,0,99,0,104,0,97,0,114,0,97,0,99,0,116,0,101,0,114,0,32,0,96,0,48,0,120,0,70,0,70,0,70,0,68,0,96,0,32,0,40,0,82,0,69,0,80,0,76,0,65,0,67,0,69,0,77,0,69,0,78,0,84,0,32,0,67,0,72,0,65,0,82,0,65,0,67,0,84,0,69,0,82,0,41,0,32,0,105,0,115,0,32,0,117,0,115,0,101,0,100,0,32,0,97,0,115,0,32,0,97,0,32,0,112,0,108,0,97,0,99,0,101,0,104,0,111,0,108,0,100,0,101,0,114,0,10,0,102,0,111,0,114,0,32,0,99,0,104,0,97,0,114,0,97,0,99,0,116,0,101,0,114,0,115,0,32,0,116,0,104,0,97,0,116,0,32,0,97,0,114,0,101,0,32,0,110,0,111,0,116,0,32,0,115,0,112,0,101,0,99,0,105,0,102,0,105,0,101,0,100,0,32,0,105,0,110,0,32,0,116,0,104,0,101,0,32,0,109,0,97,0,112,0,32,0,40,0,102,0,111,0,114,0,32,0,101,0,120,0,97,0,109,0,112,0,108,0,101,0,44,0,32,0,96,0,48,0,120,0,70,0,48,0,96,0,32,0,105,0,115,0,32,0,110,0,111,0,116,0,32,0,105,0,110,0,10,0,99,0,111,0,100,0,101,0,32,0,112,0,97,0,103,0,101,0,32,0,49,0,48,0,48,0,48,0,48,0,41,0,46,0,10,0,10,0,70,0,111,0,114,0,32,0,83,0,66,0,67,0,83,0,44,0,32,0,116,0,104,0,101,0,32,0,105,0,100,0,101,0,97,0,32,0,105,0,115,0,32,0,116,0,111,0,32,0,101,0,109,0,98,0,101,0,100,0,32,0,97,0,32,0,114,0,97,0,119,0,32,0,115,0,116,0,114,0,105,0,110,0,103,0,32,0,119,0,105,0,116,0,104,0,32,0,116,0,104,0,101,0,32,0,99,0,111,0,110,0,116,0,101,0,110,0,116,0,115,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,32,0,50,0,53,0,54,0,32,0,99,0,111,0,100,0,101,0,115,0,46,0,10,0,84,0,104,0,101,0,32,0,96,0,100,0,101,0,99,0,96,0,32,0,102,0,105,0,101,0,108,0,100,0,32,0,105,0,115,0,32,0,109,0,101,0,114,0,101,0,108,0,121,0,32,0,97,0,32,0,115,0,112,0,108,0,105,0,116,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,32,0,115,0,116,0,114,0,105,0,110,0,103,0,44,0,32,0,97,0,110,0,100,0,32,0,96,0,101,0,110,0,99,0,96,0,32,0,105,0,115,0,32,0,97,0,110,0,32,0,101,0,118,0,101,0,114,0,115,0,105,0,111,0,110,0,58,0,10,0,10,0,96,0,96,0,96,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,50,0,53,0,54,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,100,0,101,0,99,0,91,0,105,0,93,0,32,0,61,0,61,0,61,0,32,0,34,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,34,0,41,0,32,0,100,0,101,0,99,0,91,0,105,0,93,0,32,0,61,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,48,0,120,0,70,0,70,0,70,0,68,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,111,0,100,0,101,0,99,0,32,0,61,0,32,0,74,0,83,0,79,0,78,0,46,0,115,0,116,0,114,0,105,0,110,0,103,0,105,0,102,0,121,0,40,0,100,0,101,0,99,0,46,0,106,0,111,0,105,0,110,0,40,0,34,0,34,0,41,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,32,0,61,0,32,0,39,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,32,0,118,0,97,0,114,0,32,0,100,0,32,0,61,0,32,0,39,0,32,0,43,0,32,0,111,0,100,0,101,0,99,0,32,0,43,0,32,0,39,0,44,0,32,0,68,0,32,0,61,0,32,0,91,0,93,0,44,0,32,0,101,0,32,0,61,0,32,0,123,0,125,0,59,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,61,0,48,0,59,0,105,0,33,0,61,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,43,0,43,0,105,0,41,0,32,0,123,0,32,0,105,0,102,0,40,0,100,0,46,0,99,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,65,0,116,0,40,0,105,0,41,0,32,0,33,0,61,0,61,0,32,0,48,0,120,0,70,0,70,0,70,0,68,0,41,0,32,0,101,0,91,0,100,0,91,0,105,0,93,0,93,0,32,0,61,0,32,0,105,0,59,0,32,0,68,0,91,0,105,0,93,0,32,0,61,0,32,0,100,0,46,0,99,0,104,0,97,0,114,0,65,0,116,0,40,0,105,0,41,0,59,0,32,0,125,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,123,0,34,0,101,0,110,0,99,0,34,0,58,0,32,0,101,0,44,0,32,0,34,0,100,0,101,0,99,0,34,0,58,0,32,0,68,0,32,0,125,0,59,0,32,0,125,0,41,0,40,0,41,0,59,0,39,0,59,0,10,0,125,0,32,0,101,0,108,0,115,0,101,0,32,0,123,0,10,0,96,0,96,0,96,0,10,0,10,0,68,0,66,0,67,0,83,0,32,0,105,0,115,0,32,0,115,0,105,0,109,0,105,0,108,0,97,0,114,0,44,0,32,0,101,0,120,0,99,0,101,0,112,0,116,0,32,0,116,0,104,0,97,0,116,0,32,0,116,0,104,0,101,0,32,0,115,0,112,0,97,0,99,0,101,0,32,0,105,0,115,0,32,0,115,0,108,0,105,0,99,0,101,0,100,0,32,0,105,0,110,0,116,0,111,0,32,0,50,0,53,0,54,0,45,0,98,0,121,0,116,0,101,0,32,0,99,0,104,0,117,0,110,0,107,0,115,0,32,0,40,0,115,0,116,0,114,0,105,0,110,0,103,0,115,0,10,0,97,0,114,0,101,0,32,0,111,0,110,0,108,0,121,0,32,0,103,0,101,0,110,0,101,0,114,0,97,0,116,0,101,0,100,0,32,0,102,0,111,0,114,0,32,0,116,0,104,0,111,0,115,0,101,0,32,0,104,0,105,0,103,0,104,0,45,0,98,0,121,0,116,0,101,0,115,0,32,0,114,0,101,0,112,0,114,0,101,0,115,0,101,0,110,0,116,0,101,0,100,0,32,0,105,0,110,0,32,0,116,0,104,0,101,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,41,0,46,0,10,0,10,0,84,0,104,0,101,0,32,0,115,0,116,0,114,0,97,0,116,0,101,0,103,0,121,0,32,0,105,0,115,0,32,0,116,0,111,0,32,0,99,0,111,0,110,0,115,0,116,0,114,0,117,0,99,0,116,0,32,0,97,0,110,0,32,0,97,0,114,0,114,0,97,0,121,0,45,0,111,0,102,0,45,0,97,0,114,0,114,0,97,0,121,0,115,0,32,0,115,0,111,0,32,0,116,0,104,0,97,0,116,0,32,0,96,0,100,0,100,0,91,0,104,0,105,0,103,0,104,0,93,0,91,0,108,0,111,0,119,0,93,0,96,0,32,0,105,0,115,0,32,0,116,0,104,0,101,0,10,0,99,0,104,0,97,0,114,0,97,0,99,0,116,0,101,0,114,0,32,0,97,0,115,0,115,0,111,0,99,0,105,0,97,0,116,0,101,0,100,0,32,0,119,0,105,0,116,0,104,0,32,0,116,0,104,0,101,0,32,0,99,0,111,0,100,0,101,0,46,0,32,0,32,0,84,0,104,0,105,0,115,0,32,0,97,0,114,0,114,0,97,0,121,0,32,0,105,0,115,0,32,0,99,0,111,0,109,0,98,0,105,0,110,0,101,0,100,0,32,0,97,0,116,0,32,0,114,0,117,0,110,0,116,0,105,0,109,0,101,0,32,0,116,0,111,0,32,0,121,0,105,0,101,0,108,0,100,0,10,0,116,0,104,0,101,0,32,0,99,0,111,0,109,0,112,0,108,0,101,0,116,0,101,0,32,0,100,0,101,0,99,0,111,0,100,0,105,0,110,0,103,0,32,0,111,0,98,0,106,0,101,0,99,0,116,0,32,0,40,0,97,0,110,0,100,0,32,0,116,0,104,0,101,0,32,0,101,0,110,0,99,0,111,0,100,0,105,0,110,0,103,0,32,0,111,0,98,0,106,0,101,0,99,0,116,0,32,0,105,0,115,0,32,0,97,0,110,0,32,0,101,0,118,0,101,0,114,0,115,0,105,0,111,0,110,0,41,0,58,0,10,0,10,0,96,0,96,0,96,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,100,0,100,0,32,0,61,0,32,0,91,0,93,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,105,0,110,0,32,0,100,0,101,0,99,0,41,0,32,0,105,0,102,0,40,0,100,0,101,0,99,0,46,0,104,0,97,0,115,0,79,0,119,0,110,0,80,0,114,0,111,0,112,0,101,0,114,0,116,0,121,0,40,0,105,0,41,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,100,0,100,0,91,0,105,0,32,0,62,0,62,0,32,0,56,0,93,0,32,0,61,0,61,0,61,0,32,0,34,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,34,0,41,0,32,0,100,0,100,0,91,0,105,0,32,0,62,0,62,0,32,0,56,0,93,0,32,0,61,0,32,0,91,0,93,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,100,0,100,0,91,0,105,0,32,0,62,0,62,0,32,0,56,0,93,0,91,0,105,0,32,0,37,0,32,0,50,0,53,0,54,0,93,0,32,0,61,0,32,0,100,0,101,0,99,0,91,0,105,0,93,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,32,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,32,0,61,0,32,0,39,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,32,0,118,0,97,0,114,0,32,0,100,0,32,0,61,0,32,0,91,0,93,0,44,0,32,0,101,0,32,0,61,0,32,0,123,0,125,0,44,0,32,0,68,0,32,0,61,0,32,0,91,0,93,0,44,0,32,0,106,0,59,0,92,0,110,0,39,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,50,0,53,0,54,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,105,0,102,0,40,0,100,0,100,0,91,0,105,0,93,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,106,0,32,0,61,0,32,0,48,0,59,0,32,0,106,0,32,0,33,0,61,0,32,0,50,0,53,0,54,0,59,0,32,0,43,0,43,0,106,0,41,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,100,0,100,0,91,0,105,0,93,0,91,0,106,0,93,0,32,0,61,0,61,0,61,0,32,0,34,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,34,0,41,0,32,0,100,0,100,0,91,0,105,0,93,0,91,0,106,0,93,0,32,0,61,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,48,0,120,0,70,0,70,0,70,0,68,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,32,0,43,0,61,0,32,0,39,0,68,0,91,0,39,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,39,0,93,0,32,0,61,0,32,0,39,0,32,0,43,0,32,0,74,0,83,0,79,0,78,0,46,0,115,0,116,0,114,0,105,0,110,0,103,0,105,0,102,0,121,0,40,0,100,0,100,0,91,0,105,0,93,0,46,0,106,0,111,0,105,0,110,0,40,0,34,0,34,0,41,0,41,0,32,0,43,0,32,0,39,0,46,0,115,0,112,0,108,0,105,0,116,0,40,0,34,0,34,0,41,0,59,0,92,0,110,0,39,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,32,0,43,0,61,0,32,0,39,0,102,0,111,0,114,0,40,0,106,0,32,0,61,0,32,0,48,0,59,0,32,0,106,0,32,0,33,0,61,0,32,0,68,0,91,0,39,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,39,0,93,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,32,0,43,0,43,0,106,0,41,0,32,0,105,0,102,0,40,0,68,0,91,0,39,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,39,0,93,0,91,0,106,0,93,0,46,0,99,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,65,0,116,0,40,0,48,0,41,0,32,0,33,0,61,0,61,0,32,0,48,0,120,0,70,0,70,0,70,0,68,0,41,0,32,0,123,0,32,0,101,0,91,0,68,0,91,0,39,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,39,0,93,0,91,0,106,0,93,0,93,0,32,0,61,0,32,0,39,0,32,0,43,0,32,0,40,0,105,0,42,0,50,0,53,0,54,0,41,0,32,0,43,0,32,0,39,0,32,0,43,0,32,0,106,0,59,0,32,0,100,0,91,0,39,0,32,0,43,0,32,0,40,0,105,0,42,0,50,0,53,0,54,0,41,0,32,0,43,0,32,0,39,0,32,0,43,0,32,0,106,0,93,0,32,0,61,0,32,0,68,0,91,0,39,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,39,0,93,0,91,0,106,0,93,0,59,0,125,0,92,0,110,0,39,0,10,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,32,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,32,0,43,0,61,0,32,0,39,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,123,0,34,0,101,0,110,0,99,0,34,0,58,0,32,0,101,0,44,0,32,0,34,0,100,0,101,0,99,0,34,0,58,0,32,0,100,0,32,0,125,0,59,0,32,0,125,0,41,0,40,0,41,0,59,0,39,0,59,0,10,0,125,0,10,0,99,0,111,0,110,0,115,0,111,0,108,0,101,0,46,0,108,0,111,0,103,0,40,0,106,0,115,0,118,0,97,0,114,0,32,0,43,0,32,0,34,0,91,0,34,0,32,0,43,0,32,0,99,0,112,0,32,0,43,0,32,0,34,0,93,0,32,0,61,0,32,0,34,0,32,0,43,0,32,0,111,0,117,0,116,0,115,0,116,0,114,0,41,0,59,0,10,0,10,0,96,0,96,0,96,0,10,0,10,0,96,0,109,0,97,0,107,0,101,0,46,0,115,0,104,0,96,0,32,0,103,0,101,0,110,0,101,0,114,0,97,0,116,0,101,0,115,0,32,0,116,0,104,0,101,0,32,0,116,0,97,0,98,0,108,0,101,0,115,0,32,0,117,0,115,0,101,0,100,0,32,0,98,0,121,0,32,0,96,0,109,0,97,0,107,0,101,0,46,0,110,0,106,0,115,0,96,0,46,0,32,0,32,0,84,0,104,0,101,0,32,0,114,0,97,0,119,0,32,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,32,0,84,0,88,0,84,0,32,0,102,0,105,0,108,0,101,0,115,0,10,0,97,0,114,0,101,0,32,0,99,0,111,0,108,0,117,0,109,0,110,0,97,0,114,0,58,0,32,0,96,0,99,0,111,0,100,0,101,0,32,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,32,0,35,0,99,0,111,0,109,0,109,0,101,0,110,0,116,0,115,0,96,0,46,0,32,0,32,0,70,0,111,0,114,0,32,0,101,0,120,0,97,0,109,0,112,0,108,0,101,0,44,0,32,0,116,0,104,0,101,0,32,0,108,0,97,0,115,0,116,0,32,0,49,0,48,0,32,0,108,0,105,0,110,0,101,0,115,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,10,0,116,0,101,0,120,0,116,0,32,0,102,0,105,0,108,0,101,0,32,0,82,0,79,0,77,0,65,0,78,0,46,0,84,0,88,0,84,0,32,0,40,0,102,0,111,0,114,0,32,0,67,0,80,0,32,0,49,0,48,0,48,0,48,0,48,0,41,0,32,0,97,0,114,0,101,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,10,0,48,0,120,0,70,0,54,0,9,0,48,0,120,0,48,0,50,0,67,0,54,0,9,0,35,0,77,0,79,0,68,0,73,0,70,0,73,0,69,0,82,0,32,0,76,0,69,0,84,0,84,0,69,0,82,0,32,0,67,0,73,0,82,0,67,0,85,0,77,0,70,0,76,0,69,0,88,0,32,0,65,0,67,0,67,0,69,0,78,0,84,0,10,0,48,0,120,0,70,0,55,0,9,0,48,0,120,0,48,0,50,0,68,0,67,0,9,0,35,0,83,0,77,0,65,0,76,0,76,0,32,0,84,0,73,0,76,0,68,0,69,0,10,0,48,0,120,0,70,0,56,0,9,0,48,0,120,0,48,0,48,0,65,0,70,0,9,0,35,0,77,0,65,0,67,0,82,0,79,0,78,0,10,0,48,0,120,0,70,0,57,0,9,0,48,0,120,0,48,0,50,0,68,0,56,0,9,0,35,0,66,0,82,0,69,0,86,0,69,0,10,0,48,0,120,0,70,0,65,0,9,0,48,0,120,0,48,0,50,0,68,0,57,0,9,0,35,0,68,0,79,0,84,0,32,0,65,0,66,0,79,0,86,0,69,0,10,0,48,0,120,0,70,0,66,0,9,0,48,0,120,0,48,0,50,0,68,0,65,0,9,0,35,0,82,0,73,0,78,0,71,0,32,0,65,0,66,0,79,0,86,0,69,0,10,0,48,0,120,0,70,0,67,0,9,0,48,0,120,0,48,0,48,0,66,0,56,0,9,0,35,0,67,0,69,0,68,0,73,0,76,0,76,0,65,0,10,0,48,0,120,0,70,0,68,0,9,0,48,0,120,0,48,0,50,0,68,0,68,0,9,0,35,0,68,0,79,0,85,0,66,0,76,0,69,0,32,0,65,0,67,0,85,0,84,0,69,0,32,0,65,0,67,0,67,0,69,0,78,0,84,0,10,0,48,0,120,0,70,0,69,0,9,0,48,0,120,0,48,0,50,0,68,0,66,0,9,0,35,0,79,0,71,0,79,0,78,0,69,0,75,0,10,0,48,0,120,0,70,0,70,0,9,0,48,0,120,0,48,0,50,0,67,0,55,0,9,0,35,0,67,0,65,0,82,0,79,0,78,0,10,0,96,0,96,0,96,0,10,0,10,0,73,0,110,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,105,0,110,0,103,0,32,0,116,0,104,0,101,0,32,0,100,0,97,0,116,0,97,0,44,0,32,0,116,0,104,0,101,0,32,0,99,0,111,0,109,0,109,0,101,0,110,0,116,0,115,0,32,0,40,0,97,0,102,0,116,0,101,0,114,0,32,0,116,0,104,0,101,0,32,0,96,0,35,0,96,0,41,0,32,0,97,0,114,0,101,0,32,0,115,0,116,0,114,0,105,0,112,0,112,0,101,0,100,0,32,0,97,0,110,0,100,0,32,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,10,0,101,0,108,0,101,0,109,0,101,0,110,0,116,0,115,0,32,0,40,0,108,0,105,0,107,0,101,0,32,0,96,0,48,0,120,0,55,0,70,0,96,0,32,0,102,0,111,0,114,0,32,0,67,0,80,0,32,0,49,0,48,0,48,0,48,0,48,0,41,0,32,0,97,0,114,0,101,0,32,0,114,0,101,0,109,0,111,0,118,0,101,0,100,0,46,0,10,0,10,0,96,0,96,0,96,0,62,0,109,0,97,0,107,0,101,0,46,0,115,0,104,0,10,0,35,0,33,0,47,0,98,0,105,0,110,0,47,0,98,0,97,0,115,0,104,0,10,0,73,0,78,0,70,0,73,0,76,0,69,0,61,0,36,0,123,0,49,0,58,0,45,0,112,0,97,0,103,0,101,0,115,0,46,0,99,0,115,0,118,0,125,0,10,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,61,0,36,0,123,0,50,0,58,0,45,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,106,0,115,0,125,0,10,0,74,0,83,0,86,0,65,0,82,0,61,0,36,0,123,0,51,0,58,0,45,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,125,0,10,0,86,0,69,0,82,0,83,0,73,0,79,0,78,0,61,0,36,0,40,0,99,0,97,0,116,0,32,0,112,0,97,0,99,0,107,0,97,0,103,0,101,0,46,0,106,0,115,0,111,0,110,0,32,0,124,0,32,0,103,0,114,0,101,0,112,0,32,0,118,0,101,0,114,0,115,0,105,0,111,0,110,0,32,0,124,0,32,0,116,0,114,0,32,0,45,0,100,0,99,0,32,0,91,0,48,0,45,0,57,0,46,0,93,0,41,0,10,0,10,0,109,0,107,0,100,0,105,0,114,0,32,0,45,0,112,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,32,0,98,0,105,0,116,0,115,0,10,0,114,0,109,0,32,0,45,0,102,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,101,0,99,0,104,0,111,0,32,0,34,0,47,0,42,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,32,0,40,0,67,0,41,0,32,0,50,0,48,0,49,0,51,0,45,0,50,0,48,0,49,0,52,0,32,0,83,0,104,0,101,0,101,0,116,0,74,0,83,0,32,0,45,0,45,0,32,0,104,0,116,0,116,0,112,0,58,0,47,0,47,0,115,0,104,0,101,0,101,0,116,0,106,0,115,0,46,0,99,0,111,0,109,0,32,0,42,0,47,0,34,0,32,0,62,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,101,0,99,0,104,0,111,0,32,0,34,0,47,0,42,0,106,0,115,0,104,0,105,0,110,0,116,0,32,0,45,0,87,0,49,0,48,0,48,0,32,0,42,0,47,0,34,0,32,0,62,0,62,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,101,0,99,0,104,0,111,0,32,0,34,0,118,0,97,0,114,0,32,0,36,0,74,0,83,0,86,0,65,0,82,0,32,0,61,0,32,0,123,0,118,0,101,0,114,0,115,0,105,0,111,0,110,0,58,0,92,0,34,0,36,0,86,0,69,0,82,0,83,0,73,0,79,0,78,0,92,0,34,0,125,0,59,0,34,0,32,0,62,0,62,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,105,0,102,0,32,0,91,0,32,0,45,0,101,0,32,0,100,0,111,0,116,0,110,0,101,0,116,0,46,0,115,0,104,0,32,0,93,0,59,0,32,0,116,0,104,0,101,0,110,0,32,0,98,0,97,0,115,0,104,0,32,0,100,0,111,0,116,0,110,0,101,0,116,0,46,0,115,0,104,0,59,0,32,0,102,0,105,0,10,0,97,0,119,0,107,0,32,0,45,0,70,0,44,0,32,0,39,0,123,0,112,0,114,0,105,0,110,0,116,0,32,0,36,0,49,0,44,0,32,0,36,0,50,0,44,0,32,0,36,0,51,0,125,0,39,0,32,0,36,0,73,0,78,0,70,0,73,0,76,0,69,0,32,0,124,0,32,0,119,0,104,0,105,0,108,0,101,0,32,0,114,0,101,0,97,0,100,0,32,0,99,0,112,0,32,0,117,0,114,0,108,0,32,0,99,0,112,0,116,0,121,0,112,0,101,0,59,0,32,0,100,0,111,0,10,0,32,0,32,0,32,0,32,0,101,0,99,0,104,0,111,0,32,0,36,0,99,0,112,0,32,0,36,0,117,0,114,0,108,0,10,0,32,0,32,0,32,0,32,0,105,0,102,0,32,0,91,0,32,0,33,0,32,0,45,0,101,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,47,0,36,0,99,0,112,0,46,0,84,0,66,0,76,0,32,0,93,0,59,0,32,0,116,0,104,0,101,0,110,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,117,0,114,0,108,0,32,0,36,0,117,0,114,0,108,0,32,0,124,0,32,0,115,0,101,0,100,0,32,0,39,0,115,0,47,0,35,0,46,0,42,0,47,0,47,0,103,0,39,0,32,0,124,0,32,0,97,0,119,0,107,0,32,0,39,0,78,0,70,0,61,0,61,0,50,0,39,0,32,0,62,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,47,0,36,0,99,0,112,0,46,0,84,0,66,0,76,0,10,0,32,0,32,0,32,0,32,0,102,0,105,0,10,0,32,0,32,0,32,0,32,0,101,0,99,0,104,0,111,0,32,0,34,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,36,0,74,0,83,0,86,0,65,0,82,0,32,0,61,0,61,0,61,0,32,0,39,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,39,0,41,0,32,0,36,0,74,0,83,0,86,0,65,0,82,0,32,0,61,0,32,0,123,0,125,0,59,0,34,0,32,0,62,0,32,0,98,0,105,0,116,0,115,0,47,0,36,0,99,0,112,0,46,0,106,0,115,0,46,0,116,0,109,0,112,0,10,0,32,0,32,0,32,0,32,0,110,0,111,0,100,0,101,0,32,0,109,0,97,0,107,0,101,0,46,0,110,0,106,0,115,0,32,0,36,0,99,0,112,0,32,0,36,0,74,0,83,0,86,0,65,0,82,0,32,0,124,0,32,0,116,0,101,0,101,0,32,0,45,0,97,0,32,0,98,0,105,0,116,0,115,0,47,0,36,0,99,0,112,0,46,0,106,0,115,0,46,0,116,0,109,0,112,0,32,0,62,0,62,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,32,0,32,0,32,0,32,0,115,0,101,0,100,0,32,0,39,0,115,0,47,0,34,0,92,0,40,0,91,0,48,0,45,0,57,0,93,0,43,0,92,0,41,0,34,0,58,0,47,0,92,0,49,0,58,0,47,0,103,0,39,0,32,0,60,0,98,0,105,0,116,0,115,0,47,0,36,0,99,0,112,0,46,0,106,0,115,0,46,0,116,0,109,0,112,0,32,0,62,0,98,0,105,0,116,0,115,0,47,0,36,0,99,0,112,0,46,0,106,0,115,0,10,0,32,0,32,0,32,0,32,0,114,0,109,0,32,0,45,0,102,0,32,0,98,0,105,0,116,0,115,0,47,0,36,0,99,0,112,0,46,0,106,0,115,0,46,0,116,0,109,0,112,0,10,0,100,0,111,0,110,0,101,0,10,0,101,0,99,0,104,0,111,0,32,0,34,0,105,0,102,0,32,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,109,0,111,0,100,0,117,0,108,0,101,0,32,0,33,0,61,0,61,0,32,0,39,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,39,0,32,0,38,0,38,0,32,0,109,0,111,0,100,0,117,0,108,0,101,0,46,0,101,0,120,0,112,0,111,0,114,0,116,0,115,0,41,0,32,0,109,0,111,0,100,0,117,0,108,0,101,0,46,0,101,0,120,0,112,0,111,0,114,0,116,0,115,0,32,0,61,0,32,0,36,0,74,0,83,0,86,0,65,0,82,0,59,0,34,0,32,0,62,0,62,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,115,0,101,0,100,0,32,0,39,0,115,0,47,0,34,0,92,0,40,0,91,0,48,0,45,0,57,0,93,0,43,0,92,0,41,0,34,0,58,0,47,0,92,0,49,0,58,0,47,0,103,0,39,0,32,0,60,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,32,0,62,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,10,0,114,0,109,0,32,0,45,0,102,0,32,0,36,0,79,0,85,0,84,0,70,0,73,0,76,0,69,0,46,0,116,0,109,0,112,0,10,0,96,0,96,0,96,0,10,0,10,0,35,0,35,0,32,0,85,0,116,0,105,0,108,0,105,0,116,0,105,0,101,0,115,0,10,0,10,0,84,0,104,0,101,0,32,0,101,0,110,0,99,0,111,0,100,0,101,0,32,0,97,0,110,0,100,0,32,0,100,0,101,0,99,0,111,0,100,0,101,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,115,0,32,0,97,0,114,0,101,0,32,0,107,0,101,0,112,0,116,0,32,0,105,0,110,0,32,0,97,0,32,0,115,0,101,0,112,0,97,0,114,0,97,0,116,0,101,0,32,0,115,0,99,0,114,0,105,0,112,0,116,0,32,0,40,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,41,0,46,0,10,0,10,0,66,0,111,0,116,0,104,0,32,0,101,0,110,0,99,0,111,0,100,0,101,0,32,0,97,0,110,0,100,0,32,0,100,0,101,0,99,0,111,0,100,0,101,0,32,0,100,0,101,0,97,0,108,0,32,0,119,0,105,0,116,0,104,0,32,0,100,0,97,0,116,0,97,0,32,0,114,0,101,0,112,0,114,0,101,0,115,0,101,0,110,0,116,0,101,0,100,0,32,0,97,0,115,0,58,0,10,0,10,0,45,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,32,0,40,0,101,0,110,0,99,0,111,0,100,0,101,0,32,0,101,0,120,0,112,0,101,0,99,0,116,0,115,0,32,0,74,0,83,0,32,0,115,0,116,0,114,0,105,0,110,0,103,0,44,0,32,0,100,0,101,0,99,0,111,0,100,0,101,0,32,0,105,0,110,0,116,0,101,0,114,0,112,0,114,0,101,0,116,0,115,0,32,0,85,0,67,0,83,0,50,0,32,0,99,0,104,0,97,0,114,0,115,0,32,0,97,0,115,0,32,0,99,0,111,0,100,0,101,0,115,0,41,0,10,0,45,0,32,0,65,0,114,0,114,0,97,0,121,0,32,0,40,0,101,0,110,0,99,0,111,0,100,0,101,0,32,0,101,0,120,0,112,0,101,0,99,0,116,0,115,0,32,0,97,0,114,0,114,0,97,0,121,0,32,0,111,0,102,0,32,0,74,0,83,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,32,0,99,0,104,0,97,0,114,0,97,0,99,0,116,0,101,0,114,0,115,0,44,0,32,0,100,0,101,0,99,0,111,0,100,0,101,0,32,0,101,0,120,0,112,0,101,0,99,0,116,0,115,0,32,0,110,0,117,0,109,0,98,0,101,0,114,0,115,0,41,0,10,0,45,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,32,0,40,0,101,0,110,0,99,0,111,0,100,0,101,0,32,0,101,0,120,0,112,0,101,0,99,0,116,0,115,0,32,0,85,0,84,0,70,0,45,0,56,0,32,0,115,0,116,0,114,0,105,0,110,0,103,0,44,0,32,0,100,0,101,0,99,0,111,0,100,0,101,0,32,0,101,0,120,0,112,0,101,0,99,0,116,0,115,0,32,0,99,0,111,0,100,0,101,0,112,0,111,0,105,0,110,0,116,0,115,0,47,0,98,0,121,0,116,0,101,0,115,0,41,0,46,0,10,0,10,0,84,0,104,0,101,0,32,0,96,0,111,0,102,0,109,0,116,0,96,0,32,0,118,0,97,0,114,0,105,0,97,0,98,0,108,0,101,0,32,0,99,0,111,0,110,0,116,0,114,0,111,0,108,0,115,0,32,0,96,0,101,0,110,0,99,0,111,0,100,0,101,0,96,0,32,0,111,0,117,0,116,0,112,0,117,0,116,0,32,0,40,0,96,0,115,0,116,0,114,0,96,0,44,0,32,0,96,0,97,0,114,0,114,0,96,0,32,0,114,0,101,0,115,0,112,0,101,0,99,0,116,0,105,0,118,0,101,0,108,0,121,0,41,0,10,0,119,0,104,0,105,0,108,0,101,0,32,0,116,0,104,0,101,0,32,0,105,0,110,0,112,0,117,0,116,0,32,0,102,0,111,0,114,0,109,0,97,0,116,0,32,0,105,0,115,0,32,0,97,0,117,0,116,0,111,0,109,0,97,0,116,0,105,0,99,0,97,0,108,0,108,0,121,0,32,0,100,0,101,0,116,0,101,0,114,0,109,0,105,0,110,0,101,0,100,0,46,0,10,0,10,0,35,0,32,0,84,0,101,0,115,0,116,0,115,0,10,0,10,0,84,0,104,0,101,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,105,0,110,0,99,0,108,0,117,0,100,0,101,0,32,0,74,0,83,0,32,0,118,0,97,0,108,0,105,0,100,0,105,0,116,0,121,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,40,0,114,0,101,0,113,0,117,0,105,0,114,0,105,0,110,0,103,0,32,0,111,0,114,0,32,0,101,0,118,0,97,0,108,0,39,0,105,0,110,0,103,0,32,0,99,0,111,0,100,0,101,0,41,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,118,0,97,0,114,0,32,0,102,0,115,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,102,0,115,0,39,0,41,0,44,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,97,0,115,0,115,0,101,0,114,0,116,0,39,0,41,0,44,0,32,0,118,0,109,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,118,0,109,0,39,0,41,0,59,0,10,0,118,0,97,0,114,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,44,0,32,0,115,0,98,0,99,0,115,0,59,0,10,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,115,0,111,0,117,0,114,0,99,0,101,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,108,0,111,0,97,0,100,0,32,0,110,0,111,0,100,0,101,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,46,0,47,0,39,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,108,0,111,0,97,0,100,0,32,0,115,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,115,0,98,0,99,0,115,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,46,0,47,0,115,0,98,0,99,0,115,0,39,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,108,0,111,0,97,0,100,0,32,0,101,0,120,0,99,0,101,0,108,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,101,0,120,0,99,0,101,0,108,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,46,0,47,0,99,0,112,0,101,0,120,0,99,0,101,0,108,0,39,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,98,0,105,0,116,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,102,0,105,0,108,0,101,0,115,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,100,0,105,0,114,0,83,0,121,0,110,0,99,0,40,0,39,0,98,0,105,0,116,0,115,0,39,0,41,0,46,0,102,0,105,0,108,0,116,0,101,0,114,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,120,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,120,0,46,0,115,0,117,0,98,0,115,0,116,0,114,0,40,0,45,0,51,0,41,0,61,0,61,0,34,0,46,0,106,0,115,0,34,0,59,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,105,0,108,0,101,0,115,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,120,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,109,0,46,0,114,0,117,0,110,0,73,0,110,0,84,0,104,0,105,0,115,0,67,0,111,0,110,0,116,0,101,0,120,0,116,0,40,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,98,0,105,0,116,0,115,0,47,0,39,0,32,0,43,0,32,0,120,0,41,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,82,0,69,0,65,0,68,0,77,0,69,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,118,0,101,0,114,0,105,0,102,0,121,0,32,0,116,0,104,0,101,0,32,0,115,0,110,0,105,0,112,0,112,0,101,0,116,0,115,0,32,0,105,0,110,0,32,0,116,0,104,0,101,0,32,0,82,0,69,0,65,0,68,0,77,0,69,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,82,0,69,0,65,0,68,0,77,0,69,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,114,0,101,0,97,0,100,0,109,0,101,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,95,0,99,0,112,0,49,0,48,0,48,0,48,0,48,0,95,0,50,0,53,0,53,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,91,0,49,0,48,0,48,0,48,0,48,0,93,0,46,0,100,0,101,0,99,0,91,0,50,0,53,0,53,0,93,0,59,0,32,0,47,0,47,0,32,2,199,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,95,0,99,0,112,0,49,0,48,0,48,0,48,0,48,0,95,0,50,0,53,0,53,0,44,0,32,0,34,2,199,0,34,0,41,0,59,0,10,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,112,0,49,0,48,0,48,0,48,0,48,0,95,0,55,0,49,0,49,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,91,0,49,0,48,0,48,0,48,0,48,0,93,0,46,0,101,0,110,0,99,0,91,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,55,0,49,0,49,0,41,0,93,0,59,0,32,0,47,0,47,0,32,0,50,0,53,0,53,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,99,0,112,0,49,0,48,0,48,0,48,0,48,0,95,0,55,0,49,0,49,0,44,0,32,0,50,0,53,0,53,0,41,0,59,0,10,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,49,0,32,0,61,0,32,0,91,0,48,0,120,0,98,0,98,0,44,0,48,0,120,0,101,0,51,0,44,0,48,0,120,0,100,0,55,0,44,0,48,0,120,0,100,0,99,0,93,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,108,71,96,59,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,57,0,51,0,54,0,44,0,32,0,98,0,49,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,117,0,102,0,32,0,61,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,57,0,51,0,54,0,44,0,32,0,32,108,71,96,59,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,108,71,96,59,0,44,0,34,108,71,96,59,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,98,0,117,0,102,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,44,0,32,0,52,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,52,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,98,0,49,0,91,0,105,0,93,0,44,0,32,0,98,0,117,0,102,0,91,0,105,0,93,0,41,0,59,0,10,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,50,0,32,0,61,0,32,0,91,0,48,0,120,0,102,0,48,0,44,0,48,0,120,0,57,0,102,0,44,0,48,0,120,0,56,0,100,0,44,0,48,0,120,0,97,0,51,0,93,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,117,0,115,0,104,0,105,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,54,0,53,0,48,0,48,0,49,0,44,0,32,0,98,0,50,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,98,0,117,0,102,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,54,0,53,0,48,0,48,0,49,0,44,0,32,0,115,0,117,0,115,0,104,0,105,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,115,0,117,0,115,0,104,0,105,0,44,0,34,216,60,223,99,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,115,0,98,0,117,0,102,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,44,0,32,0,52,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,52,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,98,0,50,0,91,0,105,0,93,0,44,0,32,0,115,0,98,0,117,0,102,0,91,0,105,0,93,0,41,0,59,0,10,0,10,0,32,0,32,0,125,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,98,0,101,0,32,0,99,0,111,0,114,0,114,0,101,0,99,0,116,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,114,0,101,0,97,0,100,0,109,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,114,0,101,0,97,0,100,0,109,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,99,0,111,0,110,0,115,0,105,0,115,0,116,0,101,0,110,0,99,0,121,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,109,0,97,0,107,0,101,0,32,0,115,0,117,0,114,0,101,0,32,0,116,0,104,0,97,0,116,0,32,0,101,0,110,0,99,0,111,0,100,0,105,0,110,0,103,0,32,0,97,0,110,0,100,0,32,0,100,0,101,0,99,0,111,0,100,0,105,0,110,0,103,0,32,0,97,0,114,0,101,0,32,0,112,0,115,0,101,0,117,0,100,0,111,0,32,0,105,0,110,0,118,0,101,0,114,0,115,0,101,0,115,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,99,0,111,0,110,0,115,0,105,0,115,0,116,0,101,0,110,0,99,0,121,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,46,0,47,0,39,0,41,0,59,0,10,0,32,0,32,0,85,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,104,0,107,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,44,0,32,0,99,0,97,0,99,0,104,0,101,0,105,0,116,0,41,0,32,0,123,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,120,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,99,0,111,0,110,0,115,0,105,0,115,0,116,0,101,0,110,0,116,0,108,0,121,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,67,0,80,0,32,0,39,0,32,0,43,0,32,0,120,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,112,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,91,0,120,0,93,0,44,0,32,0,68,0,32,0,61,0,32,0,99,0,112,0,46,0,100,0,101,0,99,0,44,0,32,0,69,0,32,0,61,0,32,0,99,0,112,0,46,0,101,0,110,0,99,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,99,0,97,0,99,0,104,0,101,0,105,0,116,0,41,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,101,0,108,0,115,0,101,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,79,0,98,0,106,0,101,0,99,0,116,0,46,0,107,0,101,0,121,0,115,0,40,0,68,0,41,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,100,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,69,0,91,0,68,0,91,0,100,0,93,0,93,0,32,0,33,0,61,0,32,0,100,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,69,0,91,0,68,0,91,0,100,0,93,0,93,0,32,0,33,0,61,0,61,0,32,0,34,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,34,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,68,0,91,0,100,0,93,0,46,0,99,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,65,0,116,0,40,0,48,0,41,0,32,0,61,0,61,0,32,0,48,0,120,0,70,0,70,0,70,0,68,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,68,0,91,0,69,0,91,0,68,0,91,0,100,0,93,0,93,0,93,0,32,0,61,0,61,0,61,0,32,0,68,0,91,0,100,0,93,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,116,0,104,0,114,0,111,0,119,0,32,0,110,0,101,0,119,0,32,0,69,0,114,0,114,0,111,0,114,0,40,0,120,0,32,0,43,0,32,0,34,0,32,0,101,0,46,0,100,0,91,0,34,0,32,0,43,0,32,0,100,0,32,0,43,0,32,0,34,0,93,0,32,0,61,0,32,0,34,0,32,0,43,0,32,0,69,0,91,0,68,0,91,0,100,0,93,0,93,0,32,0,43,0,32,0,34,0,59,0,32,0,100,0,91,0,34,0,32,0,43,0,32,0,100,0,32,0,43,0,32,0,34,0,93,0,61,0,34,0,32,0,43,0,32,0,68,0,91,0,100,0,93,0,32,0,43,0,32,0,34,0,59,0,32,0,100,0,46,0,101,0,46,0,100,0,91,0,34,0,32,0,43,0,32,0,100,0,32,0,43,0,32,0,34,0,93,0,32,0,61,0,32,0,34,0,32,0,43,0,32,0,68,0,91,0,69,0,91,0,68,0,91,0,100,0,93,0,93,0,93,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,79,0,98,0,106,0,101,0,99,0,116,0,46,0,107,0,101,0,121,0,115,0,40,0,69,0,41,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,101,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,68,0,91,0,69,0,91,0,101,0,93,0,93,0,32,0,33,0,61,0,32,0,101,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,116,0,104,0,114,0,111,0,119,0,32,0,110,0,101,0,119,0,32,0,69,0,114,0,114,0,111,0,114,0,40,0,120,0,32,0,43,0,32,0,34,0,32,0,100,0,46,0,101,0,91,0,34,0,32,0,43,0,32,0,101,0,32,0,43,0,32,0,34,0,93,0,32,0,61,0,32,0,34,0,32,0,43,0,32,0,68,0,91,0,69,0,91,0,101,0,93,0,93,0,32,0,43,0,32,0,34,0,59,0,32,0,101,0,91,0,34,0,32,0,43,0,32,0,101,0,32,0,43,0,32,0,34,0,93,0,61,0,34,0,32,0,43,0,32,0,69,0,91,0,101,0,93,0,32,0,43,0,32,0,34,0,59,0,32,0,101,0,46,0,100,0,46,0,101,0,91,0,34,0,32,0,43,0,32,0,101,0,32,0,43,0,32,0,34,0,93,0,32,0,61,0,32,0,34,0,32,0,43,0,32,0,69,0,91,0,68,0,91,0,69,0,91,0,101,0,93,0,93,0,93,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,111,0,114,0,112,0,117,0,115,0,32,0,61,0,32,0,91,0,34,0,102,0,111,0,111,0,98,0,97,0,114,0,34,0,93,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,111,0,114,0,112,0,117,0,115,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,119,0,41,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,85,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,120,0,44,0,85,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,120,0,44,0,119,0,41,0,41,0,44,0,119,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,59,0,32,0,125,0,59,0,10,0,32,0,32,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,99,0,97,0,99,0,104,0,101,0,100,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,79,0,98,0,106,0,101,0,99,0,116,0,46,0,107,0,101,0,121,0,115,0,40,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,41,0,46,0,102,0,105,0,108,0,116,0,101,0,114,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,119,0,41,0,32,0,123,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,119,0,32,0,61,0,61,0,32,0,43,0,119,0,59,0,32,0,125,0,41,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,99,0,104,0,107,0,40,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,44,0,32,0,116,0,114,0,117,0,101,0,41,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,100,0,105,0,114,0,101,0,99,0,116,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,79,0,98,0,106,0,101,0,99,0,116,0,46,0,107,0,101,0,121,0,115,0,40,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,41,0,46,0,102,0,105,0,108,0,116,0,101,0,114,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,119,0,41,0,32,0,123,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,119,0,32,0,61,0,61,0,32,0,43,0,119,0,59,0,32,0,125,0,41,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,99,0,104,0,107,0,40,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,44,0,32,0,102,0,97,0,108,0,115,0,101,0,41,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,110,0,101,0,120,0,116,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,108,0,111,0,111,0,107,0,32,0,97,0,116,0,32,0,112,0,111,0,115,0,115,0,105,0,98,0,108,0,101,0,32,0,101,0,110,0,116,0,114,0,121,0,32,0,99,0,111,0,110,0,100,0,105,0,116,0,105,0,111,0,110,0,115,0,58,0,10,0,10,0,96,0,96,0,96,0,10,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,101,0,110,0,116,0,114,0,121,0,32,0,99,0,111,0,110,0,100,0,105,0,116,0,105,0,111,0,110,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,102,0,97,0,105,0,108,0,32,0,116,0,111,0,32,0,108,0,111,0,97,0,100,0,32,0,117,0,116,0,105,0,108,0,115,0,32,0,105,0,102,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,32,0,117,0,110,0,97,0,118,0,97,0,105,0,108,0,97,0,98,0,108,0,101,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,97,0,110,0,100,0,98,0,111,0,120,0,32,0,61,0,32,0,123,0,125,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,116,0,120,0,32,0,61,0,32,0,118,0,109,0,46,0,99,0,114,0,101,0,97,0,116,0,101,0,67,0,111,0,110,0,116,0,101,0,120,0,116,0,40,0,115,0,97,0,110,0,100,0,98,0,111,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,109,0,46,0,114,0,117,0,110,0,73,0,110,0,67,0,111,0,110,0,116,0,101,0,120,0,116,0,40,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,39,0,44,0,39,0,117,0,116,0,102,0,56,0,39,0,41,0,44,0,99,0,116,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,108,0,111,0,97,0,100,0,32,0,117,0,116,0,105,0,108,0,115,0,32,0,105,0,102,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,32,0,105,0,115,0,32,0,97,0,118,0,97,0,105,0,108,0,97,0,98,0,108,0,101,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,97,0,110,0,100,0,98,0,111,0,120,0,32,0,61,0,32,0,123,0,125,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,116,0,120,0,32,0,61,0,32,0,118,0,109,0,46,0,99,0,114,0,101,0,97,0,116,0,101,0,67,0,111,0,110,0,116,0,101,0,120,0,116,0,40,0,115,0,97,0,110,0,100,0,98,0,111,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,109,0,46,0,114,0,117,0,110,0,73,0,110,0,67,0,111,0,110,0,116,0,101,0,120,0,116,0,40,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,99,0,112,0,101,0,120,0,99,0,101,0,108,0,46,0,106,0,115,0,39,0,44,0,39,0,117,0,116,0,102,0,56,0,39,0,41,0,44,0,99,0,116,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,109,0,46,0,114,0,117,0,110,0,73,0,110,0,67,0,111,0,110,0,116,0,101,0,120,0,116,0,40,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,39,0,44,0,39,0,117,0,116,0,102,0,56,0,39,0,41,0,44,0,99,0,116,0,120,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,104,0,107,0,101,0,110,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,99,0,112,0,44,0,32,0,105,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,99,0,112,0,44,0,32,0,105,0,44,0,32,0,101,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,116,0,114,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,105,0,44,0,101,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,97,0,114,0,114,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,105,0,46,0,115,0,112,0,108,0,105,0,116,0,40,0,34,0,34,0,41,0,44,0,101,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,100,0,101,0,101,0,112,0,69,0,113,0,117,0,97,0,108,0,40,0,115,0,116,0,114,0,44,0,97,0,114,0,114,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,32,0,61,0,61,0,61,0,32,0,39,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,39,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,117,0,102,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,110,0,101,0,119,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,40,0,105,0,41,0,44,0,101,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,100,0,101,0,101,0,112,0,69,0,113,0,117,0,97,0,108,0,40,0,115,0,116,0,114,0,44,0,98,0,117,0,102,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,44,0,39,0,98,0,117,0,102,0,39,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,44,0,39,0,97,0,114,0,114,0,39,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,44,0,39,0,115,0,116,0,114,0,39,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,44,0,39,0,98,0,117,0,102,0,39,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,44,0,39,0,97,0,114,0,114,0,39,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,44,0,39,0,115,0,116,0,114,0,39,0,41,0,59,0,10,0,32,0,32,0,125,0,59,0,10,0,32,0,32,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,101,0,110,0,99,0,111,0,100,0,101,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,67,0,80,0,32,0,32,0,49,0,50,0,53,0,50,0,32,0,58,0,32,0,115,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,104,0,107,0,101,0,110,0,40,0,49,0,50,0,53,0,50,0,44,0,34,0,102,0,111,0,111,0,98,0,97,0,114,0,34,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,67,0,80,0,32,0,32,0,32,0,55,0,48,0,56,0,32,0,58,0,32,0,115,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,104,0,107,0,101,0,110,0,40,0,55,0,48,0,56,0,44,0,34,6,42,0,32,0,97,0,110,0,100,0,32,6,43,0,32,0,115,0,109,0,105,0,108,0,101,0,121,0,32,0,102,0,97,0,99,0,101,0,115,0,34,0,41,0,59,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,67,0,80,0,32,0,32,0,32,0,57,0,51,0,54,0,32,0,58,0,32,0,100,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,104,0,107,0,101,0,110,0,40,0,57,0,51,0,54,0,44,0,32,0,34,143,217,102,47,78,45,101,135,91,87,123,38,109,75,139,213,0,34,0,41,0,59,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,104,0,107,0,100,0,101,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,99,0,112,0,44,0,32,0,105,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,99,0,112,0,44,0,32,0,105,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,32,0,33,0,61,0,61,0,32,0,39,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,39,0,32,0,38,0,38,0,32,0,105,0,32,0,105,0,110,0,115,0,116,0,97,0,110,0,99,0,101,0,111,0,102,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,41,0,32,0,115,0,32,0,61,0,32,0,91,0,93,0,46,0,109,0,97,0,112,0,46,0,99,0,97,0,108,0,108,0,40,0,105,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,115,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,115,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,101,0,108,0,115,0,101,0,32,0,115,0,61,0,40,0,105,0,46,0,109,0,97,0,112,0,41,0,32,0,63,0,32,0,105,0,46,0,109,0,97,0,112,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,115,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,115,0,41,0,59,0,32,0,125,0,41,0,32,0,58,0,32,0,105,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,115,0,116,0,114,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,105,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,97,0,114,0,114,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,115,0,46,0,106,0,111,0,105,0,110,0,63,0,115,0,46,0,106,0,111,0,105,0,110,0,40,0,34,0,34,0,41,0,58,0,115,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,100,0,101,0,101,0,112,0,69,0,113,0,117,0,97,0,108,0,40,0,115,0,116,0,114,0,44,0,97,0,114,0,114,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,32,0,61,0,61,0,61,0,32,0,39,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,39,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,117,0,102,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,110,0,101,0,119,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,40,0,105,0,41,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,100,0,101,0,101,0,112,0,69,0,113,0,117,0,97,0,108,0,40,0,115,0,116,0,114,0,44,0,98,0,117,0,102,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,99,0,40,0,99,0,112,0,44,0,105,0,41,0,59,0,10,0,32,0,32,0,125,0,59,0,10,0,32,0,32,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,100,0,101,0,99,0,111,0,100,0,101,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,67,0,80,0,32,0,32,0,49,0,50,0,53,0,50,0,32,0,58,0,32,0,115,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,104,0,107,0,100,0,101,0,40,0,49,0,50,0,53,0,50,0,44,0,91,0,48,0,120,0,54,0,54,0,44,0,32,0,48,0,120,0,54,0,102,0,44,0,32,0,48,0,120,0,54,0,102,0,44,0,32,0,48,0,120,0,54,0,50,0,44,0,32,0,48,0,120,0,54,0,49,0,44,0,32,0,48,0,120,0,55,0,50,0,93,0,41,0,59,0,32,0,125,0,41,0,59,0,32,0,47,0,42,0,32,0,34,0,102,0,111,0,111,0,98,0,97,0,114,0,34,0,32,0,42,0,47,0,10,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,121,0,112,0,101,0,111,0,102,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,32,0,33,0,61,0,61,0,32,0,39,0,117,0,110,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,39,0,41,0,32,0,105,0,116,0,40,0,39,0,67,0,80,0,32,0,32,0,32,0,55,0,48,0,56,0,32,0,58,0,32,0,115,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,104,0,107,0,100,0,101,0,40,0,55,0,48,0,56,0,44,0,32,0,110,0,101,0,119,0,32,0,66,0,117,0,102,0,102,0,101,0,114,0,40,0,91,0,48,0,120,0,99,0,97,0,44,0,32,0,48,0,120,0,50,0,48,0,44,0,32,0,48,0,120,0,54,0,49,0,44,0,32,0,48,0,120,0,54,0,101,0,44,0,32,0,48,0,120,0,54,0,52,0,44,0,32,0,48,0,120,0,50,0,48,0,44,0,32,0,48,0,120,0,99,0,98,0,44,0,32,0,48,0,120,0,50,0,48,0,44,0,32,0,48,0,120,0,55,0,51,0,44,0,32,0,48,0,120,0,54,0,100,0,44,0,32,0,48,0,120,0,54,0,57,0,44,0,32,0,48,0,120,0,54,0,99,0,44,0,32,0,48,0,120,0,54,0,53,0,44,0,32,0,48,0,120,0,55,0,57,0,44,0,32,0,48,0,120,0,50,0,48,0,44,0,32,0,48,0,120,0,54,0,54,0,44,0,32,0,48,0,120,0,54,0,49,0,44,0,32,0,48,0,120,0,54,0,51,0,44,0,32,0,48,0,120,0,54,0,53,0,44,0,32,0,48,0,120,0,55,0,51,0,93,0,41,0,41,0,59,0,32,0,125,0,41,0,59,0,32,0,47,0,42,0,32,0,40,0,34,6,42,0,32,0,97,0,110,0,100,0,32,6,43,0,32,0,115,0,109,0,105,0,108,0,101,0,121,0,32,0,102,0,97,0,99,0,101,0,115,0,34,0,41,0,32,0,42,0,47,0,10,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,67,0,80,0,32,0,32,0,32,0,57,0,51,0,54,0,32,0,58,0,32,0,100,0,98,0,99,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,99,0,104,0,107,0,100,0,101,0,40,0,57,0,51,0,54,0,44,0,32,0,91,0,48,0,120,0,100,0,53,0,44,0,32,0,48,0,120,0,101,0,50,0,44,0,32,0,48,0,120,0,99,0,97,0,44,0,32,0,48,0,120,0,99,0,55,0,44,0,32,0,48,0,120,0,100,0,54,0,44,0,32,0,48,0,120,0,100,0,48,0,44,0,32,0,48,0,120,0,99,0,101,0,44,0,32,0,48,0,120,0,99,0,52,0,44,0,32,0,48,0,120,0,100,0,55,0,44,0,32,0,48,0,120,0,100,0,54,0,44,0,32,0,48,0,120,0,98,0,55,0,44,0,32,0,48,0,120,0,102,0,98,0,44,0,32,0,48,0,120,0,98,0,50,0,44,0,32,0,48,0,120,0,101,0,50,0,44,0,32,0,48,0,120,0,99,0,97,0,44,0,32,0,48,0,120,0,100,0,52,0,93,0,41,0,59,0,125,0,41,0,59,0,32,0,47,0,42,0,32,0,34,143,217,102,47,78,45,101,135,91,87,123,38,109,75,139,213,0,34,0,32,0,42,0,47,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,96,0,116,0,101,0,115,0,116,0,102,0,105,0,108,0,101,0,96,0,32,0,104,0,101,0,108,0,112,0,101,0,114,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,32,0,114,0,101,0,97,0,100,0,115,0,32,0,97,0,32,0,102,0,105,0,108,0,101,0,32,0,97,0,110,0,100,0,32,0,99,0,111,0,109,0,112,0,97,0,114,0,101,0,115,0,32,0,116,0,111,0,32,0,110,0,111,0,100,0,101,0,39,0,115,0,32,0,114,0,101,0,97,0,100,0,32,0,102,0,97,0,99,0,105,0,108,0,105,0,116,0,105,0,101,0,115,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,32,0,116,0,101,0,115,0,116,0,102,0,105,0,108,0,101,0,40,0,102,0,44,0,99,0,112,0,44,0,116,0,121,0,112,0,101,0,44,0,115,0,107,0,105,0,112,0,41,0,32,0,123,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,100,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,102,0,41,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,120,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,102,0,44,0,32,0,116,0,121,0,112,0,101,0,41,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,97,0,32,0,61,0,32,0,120,0,46,0,115,0,112,0,108,0,105,0,116,0,40,0,34,0,34,0,41,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,99,0,104,0,107,0,32,0,61,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,99,0,112,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,121,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,32,0,100,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,120,0,44,0,121,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,122,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,32,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,33,0,61,0,32,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,41,0,32,0,116,0,104,0,114,0,111,0,119,0,32,0,110,0,101,0,119,0,32,0,69,0,114,0,114,0,111,0,114,0,40,0,102,0,32,0,43,0,32,0,34,0,32,0,34,0,32,0,43,0,32,0,74,0,83,0,79,0,78,0,46,0,115,0,116,0,114,0,105,0,110,0,103,0,105,0,102,0,121,0,40,0,122,0,41,0,32,0,43,0,32,0,34,0,32,0,33,0,61,0,32,0,34,0,32,0,43,0,32,0,74,0,83,0,79,0,78,0,46,0,115,0,116,0,114,0,105,0,110,0,103,0,105,0,102,0,121,0,40,0,100,0,41,0,32,0,43,0,32,0,34,0,32,0,58,0,32,0,34,0,32,0,43,0,32,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,43,0,32,0,34,0,32,0,34,0,32,0,43,0,32,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,105,0,102,0,40,0,100,0,91,0,105,0,93,0,32,0,33,0,61,0,61,0,32,0,122,0,91,0,105,0,93,0,41,0,32,0,116,0,104,0,114,0,111,0,119,0,32,0,110,0,101,0,119,0,32,0,69,0,114,0,114,0,111,0,114,0,40,0,34,0,34,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,34,0,32,0,34,0,32,0,43,0,32,0,100,0,91,0,105,0,93,0,32,0,43,0,32,0,34,0,33,0,61,0,34,0,32,0,43,0,32,0,122,0,91,0,105,0,93,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,115,0,107,0,105,0,112,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,122,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,99,0,112,0,44,0,32,0,97,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,33,0,61,0,32,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,41,0,32,0,116,0,104,0,114,0,111,0,119,0,32,0,110,0,101,0,119,0,32,0,69,0,114,0,114,0,111,0,114,0,40,0,102,0,32,0,43,0,32,0,34,0,32,0,34,0,32,0,43,0,32,0,74,0,83,0,79,0,78,0,46,0,115,0,116,0,114,0,105,0,110,0,103,0,105,0,102,0,121,0,40,0,122,0,41,0,32,0,43,0,32,0,34,0,32,0,33,0,61,0,32,0,34,0,32,0,43,0,32,0,74,0,83,0,79,0,78,0,46,0,115,0,116,0,114,0,105,0,110,0,103,0,105,0,102,0,121,0,40,0,100,0,41,0,32,0,43,0,32,0,34,0,32,0,58,0,32,0,34,0,32,0,43,0,32,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,32,0,43,0,32,0,34,0,32,0,34,0,32,0,43,0,32,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,100,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,105,0,102,0,40,0,100,0,91,0,105,0,93,0,32,0,33,0,61,0,61,0,32,0,122,0,91,0,105,0,93,0,41,0,32,0,116,0,104,0,114,0,111,0,119,0,32,0,110,0,101,0,119,0,32,0,69,0,114,0,114,0,111,0,114,0,40,0,34,0,34,0,32,0,43,0,32,0,105,0,32,0,43,0,32,0,34,0,32,0,34,0,32,0,43,0,32,0,100,0,91,0,105,0,93,0,32,0,43,0,32,0,34,0,33,0,61,0,34,0,32,0,43,0,32,0,122,0,91,0,105,0,93,0,41,0,59,0,10,0,32,0,32,0,125,0,10,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,99,0,104,0,107,0,40,0,99,0,112,0,41,0,59,0,10,0,32,0,32,0,105,0,102,0,40,0,115,0,107,0,105,0,112,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,99,0,104,0,107,0,40,0,99,0,112,0,41,0,59,0,10,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,125,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,96,0,117,0,116,0,102,0,56,0,96,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,118,0,101,0,114,0,105,0,102,0,121,0,32,0,117,0,116,0,102,0,56,0,32,0,101,0,110,0,99,0,111,0,100,0,105,0,110,0,103,0,32,0,111,0,102,0,32,0,116,0,104,0,101,0,32,0,97,0,99,0,116,0,117,0,97,0,108,0,32,0,74,0,83,0,32,0,115,0,111,0,117,0,114,0,99,0,101,0,115,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,110,0,111,0,100,0,101,0,32,0,110,0,97,0,116,0,105,0,118,0,101,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,110,0,111,0,100,0,101,0,32,0,61,0,32,0,91,0,91,0,54,0,53,0,48,0,48,0,49,0,44,0,32,0,39,0,117,0,116,0,102,0,56,0,39,0,44,0,49,0,93,0,44,0,32,0,91,0,49,0,50,0,48,0,48,0,44,0,32,0,39,0,117,0,116,0,102,0,49,0,54,0,108,0,101,0,39,0,44,0,49,0,93,0,44,0,32,0,91,0,50,0,48,0,49,0,50,0,55,0,44,0,32,0,39,0,97,0,115,0,99,0,105,0,105,0,39,0,44,0,48,0,93,0,93,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,102,0,105,0,108,0,101,0,115,0,32,0,61,0,32,0,91,0,39,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,109,0,100,0,39,0,44,0,39,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,109,0,100,0,39,0,44,0,39,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,106,0,115,0,39,0,93,0,59,0,10,0,32,0,32,0,118,0,97,0,114,0,32,0,97,0,115,0,99,0,105,0,105,0,102,0,105,0,108,0,101,0,115,0,32,0,61,0,32,0,91,0,39,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,39,0,93,0,59,0,10,0,32,0,32,0,110,0,111,0,100,0,101,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,119,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,119,0,91,0,49,0,93,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,32,0,61,0,32,0,114,0,101,0,113,0,117,0,105,0,114,0,101,0,40,0,39,0,46,0,47,0,39,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,99,0,105,0,105,0,102,0,105,0,108,0,101,0,115,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,102,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,39,0,32,0,43,0,32,0,102,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,116,0,101,0,115,0,116,0,102,0,105,0,108,0,101,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,39,0,43,0,102,0,43,0,39,0,46,0,39,0,43,0,119,0,91,0,49,0,93,0,44,0,119,0,91,0,48,0,93,0,44,0,119,0,91,0,49,0,93,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,33,0,119,0,91,0,50,0,93,0,41,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,117,0,110,0,105,0,99,0,111,0,100,0,101,0,102,0,105,0,108,0,101,0,115,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,102,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,39,0,32,0,43,0,32,0,102,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,32,0,116,0,101,0,115,0,116,0,102,0,105,0,108,0,101,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,39,0,43,0,102,0,43,0,39,0,46,0,39,0,43,0,119,0,91,0,49,0,93,0,44,0,119,0,91,0,48,0,93,0,44,0,119,0,91,0,49,0,93,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,119,0,91,0,49,0,93,0,32,0,61,0,61,0,61,0,32,0,39,0,117,0,116,0,102,0,56,0,39,0,41,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,98,0,105,0,116,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,102,0,105,0,108,0,101,0,115,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,100,0,105,0,114,0,83,0,121,0,110,0,99,0,40,0,39,0,98,0,105,0,116,0,115,0,39,0,41,0,46,0,102,0,105,0,108,0,116,0,101,0,114,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,120,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,120,0,46,0,115,0,117,0,98,0,115,0,116,0,114,0,40,0,45,0,51,0,41,0,61,0,61,0,34,0,46,0,106,0,115,0,34,0,59,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,32,0,102,0,105,0,108,0,101,0,115,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,102,0,41,0,32,0,123,0,32,0,116,0,101,0,115,0,116,0,102,0,105,0,108,0,101,0,40,0,39,0,46,0,47,0,98,0,105,0,116,0,115,0,47,0,39,0,32,0,43,0,32,0,102,0,44,0,119,0,91,0,48,0,93,0,44,0,119,0,91,0,49,0,93,0,44,0,116,0,114,0,117,0,101,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,117,0,116,0,102,0,42,0,32,0,97,0,110,0,100,0,32,0,97,0,115,0,99,0,105,0,105,0,32,0,116,0,101,0,115,0,116,0,115,0,32,0,97,0,116,0,116,0,101,0,109,0,112,0,116,0,32,0,116,0,111,0,32,0,116,0,101,0,115,0,116,0,32,0,111,0,116,0,104,0,101,0,114,0,32,0,109,0,97,0,103,0,105,0,99,0,32,0,102,0,111,0,114,0,109,0,97,0,116,0,115,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,118,0,97,0,114,0,32,0,109,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,109,0,97,0,103,0,105,0,99,0,59,0,10,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,32,0,99,0,109,0,112,0,40,0,120,0,44,0,122,0,41,0,32,0,123,0,10,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,120,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,44,0,32,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,41,0,59,0,10,0,32,0,32,0,102,0,111,0,114,0,40,0,118,0,97,0,114,0,32,0,105,0,32,0,61,0,32,0,48,0,59,0,32,0,105,0,32,0,33,0,61,0,32,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,59,0,32,0,43,0,43,0,105,0,41,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,105,0,43,0,34,0,47,0,34,0,43,0,120,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,43,0,34,0,34,0,43,0,120,0,91,0,105,0,93,0,44,0,32,0,105,0,43,0,34,0,47,0,34,0,43,0,122,0,46,0,108,0,101,0,110,0,103,0,116,0,104,0,43,0,34,0,34,0,43,0,122,0,91,0,105,0,93,0,41,0,59,0,10,0,125,0,10,0,79,0,98,0,106,0,101,0,99,0,116,0,46,0,107,0,101,0,121,0,115,0,40,0,109,0,41,0,46,0,102,0,111,0,114,0,69,0,97,0,99,0,104,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,116,0,41,0,123,0,105,0,102,0,40,0,116,0,32,0,33,0,61,0,32,0,49,0,54,0,57,0,54,0,57,0,41,0,32,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,109,0,91,0,116,0,93,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,105,0,116,0,40,0,34,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,109,0,100,0,46,0,34,0,32,0,43,0,32,0,109,0,91,0,116,0,93,0,44,0,32,0,102,0,115,0,46,0,101,0,120,0,105,0,115,0,116,0,115,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,109,0,100,0,46,0,39,0,32,0,43,0,32,0,109,0,91,0,116,0,93,0,41,0,32,0,63,0,10,0,32,0,32,0,32,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,109,0,100,0,46,0,117,0,116,0,102,0,56,0,39,0,44,0,32,0,34,0,117,0,116,0,102,0,56,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,109,0,91,0,116,0,93,0,32,0,61,0,61,0,61,0,32,0,34,0,97,0,115,0,99,0,105,0,105,0,34,0,41,0,32,0,98,0,32,0,61,0,32,0,98,0,46,0,114,0,101,0,112,0,108,0,97,0,99,0,101,0,40,0,47,0,91,0,92,0,117,0,48,0,48,0,56,0,48,0,45,0,92,0,117,0,102,0,102,0,102,0,102,0,93,0,42,0,47,0,103,0,44,0,34,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,120,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,109,0,100,0,46,0,39,0,32,0,43,0,32,0,109,0,91,0,116,0,93,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,121,0,44,0,32,0,122,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,121,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,121,0,44,0,98,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,122,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,121,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,32,0,33,0,61,0,32,0,54,0,53,0,48,0,48,0,48,0,41,0,32,0,99,0,109,0,112,0,40,0,120,0,44,0,122,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,101,0,108,0,115,0,101,0,32,0,123,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,121,0,44,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,122,0,41,0,41,0,59,0,32,0,125,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,121,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,121,0,44,0,98,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,122,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,121,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,116,0,32,0,33,0,61,0,32,0,54,0,53,0,48,0,48,0,48,0,41,0,32,0,99,0,109,0,112,0,40,0,120,0,44,0,122,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,101,0,108,0,115,0,101,0,32,0,123,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,121,0,44,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,122,0,41,0,41,0,59,0,32,0,125,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,58,0,32,0,110,0,117,0,108,0,108,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,34,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,112,0,114,0,111,0,99,0,101,0,115,0,115,0,32,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,109,0,100,0,46,0,34,0,32,0,43,0,32,0,109,0,91,0,116,0,93,0,44,0,32,0,102,0,115,0,46,0,101,0,120,0,105,0,115,0,116,0,115,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,109,0,100,0,46,0,39,0,32,0,43,0,32,0,109,0,91,0,116,0,93,0,41,0,32,0,63,0,10,0,32,0,32,0,32,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,98,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,109,0,100,0,46,0,117,0,116,0,102,0,56,0,39,0,44,0,32,0,34,0,117,0,116,0,102,0,56,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,105,0,102,0,40,0,109,0,91,0,116,0,93,0,32,0,61,0,61,0,61,0,32,0,34,0,97,0,115,0,99,0,105,0,105,0,34,0,41,0,32,0,98,0,32,0,61,0,32,0,98,0,46,0,114,0,101,0,112,0,108,0,97,0,99,0,101,0,40,0,47,0,91,0,92,0,117,0,48,0,48,0,56,0,48,0,45,0,92,0,117,0,102,0,102,0,102,0,102,0,93,0,42,0,47,0,103,0,44,0,34,0,34,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,120,0,32,0,61,0,32,0,102,0,115,0,46,0,114,0,101,0,97,0,100,0,70,0,105,0,108,0,101,0,83,0,121,0,110,0,99,0,40,0,39,0,46,0,47,0,109,0,105,0,115,0,99,0,47,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,109,0,100,0,46,0,39,0,32,0,43,0,32,0,109,0,91,0,116,0,93,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,120,0,32,0,61,0,32,0,91,0,93,0,46,0,115,0,108,0,105,0,99,0,101,0,46,0,99,0,97,0,108,0,108,0,40,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,121,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,121,0,44,0,98,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,118,0,97,0,114,0,32,0,121,0,32,0,61,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,116,0,44,0,32,0,120,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,101,0,113,0,117,0,97,0,108,0,40,0,121,0,44,0,98,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,101,0,110,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,58,0,32,0,110,0,117,0,108,0,108,0,41,0,59,0,10,0,125,0,41,0,59,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,84,0,104,0,101,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,32,0,96,0,54,0,57,0,54,0,57,0,96,0,32,0,105,0,115,0,32,0,110,0,111,0,116,0,32,0,100,0,101,0,102,0,105,0,110,0,101,0,100,0,44,0,32,0,115,0,111,0,32,0,111,0,112,0,101,0,114,0,97,0,116,0,105,0,111,0,110,0,115,0,32,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,102,0,97,0,105,0,108,0,58,0,10,0,10,0,96,0,96,0,96,0,62,0,116,0,101,0,115,0,116,0,46,0,106,0,115,0,10,0,100,0,101,0,115,0,99,0,114,0,105,0,98,0,101,0,40,0,39,0,102,0,97,0,105,0,108,0,117,0,114,0,101,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,102,0,97,0,105,0,108,0,32,0,116,0,111,0,32,0,102,0,105,0,110,0,100,0,32,0,67,0,80,0,32,0,54,0,57,0,54,0,57,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,91,0,54,0,57,0,54,0,57,0,93,0,46,0,100,0,101,0,99,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,91,0,54,0,57,0,54,0,57,0,93,0,46,0,101,0,110,0,99,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,102,0,97,0,105,0,108,0,32,0,117,0,115,0,105,0,110,0,103,0,32,0,117,0,116,0,105,0,108,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,40,0,33,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,104,0,97,0,115,0,99,0,112,0,40,0,54,0,57,0,54,0,57,0,41,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,54,0,57,0,54,0,57,0,44,0,32,0,34,0,102,0,111,0,111,0,98,0,97,0,114,0,34,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,54,0,57,0,54,0,57,0,44,0,32,0,91,0,48,0,120,0,50,0,48,0,93,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,102,0,97,0,105,0,108,0,32,0,119,0,105,0,116,0,104,0,32,0,98,0,108,0,97,0,99,0,107,0,32,0,109,0,97,0,103,0,105,0,99,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,40,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,104,0,97,0,115,0,99,0,112,0,40,0,49,0,54,0,57,0,54,0,57,0,41,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,49,0,54,0,57,0,54,0,57,0,44,0,32,0,34,0,102,0,111,0,111,0,98,0,97,0,114,0,34,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,100,0,101,0,99,0,111,0,100,0,101,0,40,0,49,0,54,0,57,0,54,0,57,0,44,0,32,0,91,0,48,0,120,0,50,0,48,0,93,0,41,0,59,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,32,0,32,0,105,0,116,0,40,0,39,0,115,0,104,0,111,0,117,0,108,0,100,0,32,0,102,0,97,0,105,0,108,0,32,0,119,0,104,0,101,0,110,0,32,0,112,0,114,0,101,0,115,0,101,0,110,0,116,0,101,0,100,0,32,0,119,0,105,0,116,0,104,0,32,0,105,0,110,0,118,0,97,0,108,0,105,0,100,0,32,0,99,0,104,0,97,0,114,0,32,0,99,0,111,0,100,0,101,0,115,0,39,0,44,0,32,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,97,0,115,0,115,0,101,0,114,0,116,0,46,0,116,0,104,0,114,0,111,0,119,0,115,0,40,0,102,0,117,0,110,0,99,0,116,0,105,0,111,0,110,0,40,0,41,0,123,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,99,0,97,0,99,0,104,0,101,0,46,0,100,0,101,0,99,0,97,0,99,0,104,0,101,0,40,0,41,0,59,0,32,0,114,0,101,0,116,0,117,0,114,0,110,0,32,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,117,0,116,0,105,0,108,0,115,0,46,0,101,0,110,0,99,0,111,0,100,0,101,0,40,0,50,0,48,0,49,0,50,0,55,0,44,0,32,0,91,0,83,0,116,0,114,0,105,0,110,0,103,0,46,0,102,0,114,0,111,0,109,0,67,0,104,0,97,0,114,0,67,0,111,0,100,0,101,0,40,0,48,0,120,0,65,0,65,0,41,0,93,0,41,0,59,0,125,0,41,0,59,0,10,0,32,0,32,0,125,0,41,0,59,0,10,0,125,0,41,0,59,0,10,0,96,0,96,0,96,0,10,0,10,0,35,0,32,0,78,0,105,0,116,0,116,0,121,0,32,0,71,0,114,0,105,0,116,0,116,0,121,0,10,0,10,0,96,0,96,0,96,0,106,0,115,0,111,0,110,0,62,0,112,0,97,0,99,0,107,0,97,0,103,0,101,0,46,0,106,0,115,0,111,0,110,0,10,0,123,0,10,0,32,0,32,0,34,0,110,0,97,0,109,0,101,0,34,0,58,0,32,0,34,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,34,0,44,0,10,0,32,0,32,0,34,0,118,0,101,0,114,0,115,0,105,0,111,0,110,0,34,0,58,0,32,0,34,0,49,0,46,0,51,0,46,0,56,0,34,0,44,0,10,0,32,0,32,0,34,0,97,0,117,0,116,0,104,0,111,0,114,0,34,0,58,0,32,0,34,0,83,0,104,0,101,0,101,0,116,0,74,0,83,0,34,0,44,0,10,0,32,0,32,0,34,0,100,0,101,0,115,0,99,0,114,0,105,0,112,0,116,0,105,0,111,0,110,0,34,0,58,0,32,0,34,0,112,0,117,0,114,0,101,0,45,0,74,0,83,0,32,0,108,0,105,0,98,0,114,0,97,0,114,0,121,0,32,0,116,0,111,0,32,0,104,0,97,0,110,0,100,0,108,0,101,0,32,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,34,0,44,0,10,0,32,0,32,0,34,0,107,0,101,0,121,0,119,0,111,0,114,0,100,0,115,0,34,0,58,0,32,0,91,0,32,0,34,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,34,0,44,0,32,0,34,0,105,0,99,0,111,0,110,0,118,0,34,0,44,0,32,0,34,0,99,0,111,0,110,0,118,0,101,0,114,0,116,0,34,0,44,0,32,0,34,0,115,0,116,0,114,0,105,0,110,0,103,0,115,0,34,0,32,0,93,0,44,0,10,0,32,0,32,0,34,0,98,0,105,0,110,0,34,0,58,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,34,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,34,0,58,0,32,0,34,0,46,0,47,0,98,0,105,0,110,0,47,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,110,0,106,0,115,0,34,0,10,0,32,0,32,0,125,0,44,0,10,0,32,0,32,0,34,0,102,0,105,0,108,0,101,0,115,0,34,0,58,0,32,0,91,0,10,0,32,0,32,0,32,0,32,0,34,0,76,0,73,0,67,0,69,0,78,0,83,0,69,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,82,0,69,0,65,0,68,0,77,0,69,0,46,0,109,0,100,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,98,0,105,0,110,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,106,0,115,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,100,0,105,0,115,0,116,0,47,0,99,0,112,0,101,0,120,0,99,0,101,0,108,0,46,0,102,0,117,0,108,0,108,0,46,0,106,0,115,0,34,0,10,0,32,0,32,0,93,0,44,0,10,0,32,0,32,0,34,0,109,0,97,0,105,0,110,0,34,0,58,0,32,0,34,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,34,0,44,0,10,0,32,0,32,0,34,0,100,0,101,0,112,0,101,0,110,0,100,0,101,0,110,0,99,0,105,0,101,0,115,0,34,0,58,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,34,0,118,0,111,0,99,0,34,0,58,0,34,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,99,0,111,0,110,0,99,0,97,0,116,0,45,0,115,0,116,0,114,0,101,0,97,0,109,0,34,0,58,0,34,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,99,0,111,0,109,0,109,0,97,0,110,0,100,0,101,0,114,0,34,0,58,0,34,0,34,0,10,0,32,0,32,0,125,0,44,0,10,0,32,0,32,0,34,0,100,0,101,0,118,0,68,0,101,0,112,0,101,0,110,0,100,0,101,0,110,0,99,0,105,0,101,0,115,0,34,0,58,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,34,0,109,0,111,0,99,0,104,0,97,0,34,0,58,0,34,0,34,0,10,0,32,0,32,0,125,0,44,0,10,0,32,0,32,0,34,0,114,0,101,0,112,0,111,0,115,0,105,0,116,0,111,0,114,0,121,0,34,0,58,0,32,0,123,0,32,0,34,0,116,0,121,0,112,0,101,0,34,0,58,0,34,0,103,0,105,0,116,0,34,0,44,0,32,0,34,0,117,0,114,0,108,0,34,0,58,0,34,0,103,0,105,0,116,0,58,0,47,0,47,0,103,0,105,0,116,0,104,0,117,0,98,0,46,0,99,0,111,0,109,0,47,0,83,0,104,0,101,0,101,0,116,0,74,0,83,0,47,0,106,0,115,0,45,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,46,0,103,0,105,0,116,0,34,0,125,0,44,0,10,0,32,0,32,0,34,0,115,0,99,0,114,0,105,0,112,0,116,0,115,0,34,0,58,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,34,0,112,0,114,0,101,0,116,0,101,0,115,0,116,0,34,0,58,0,32,0,34,0,103,0,105,0,116,0,32,0,115,0,117,0,98,0,109,0,111,0,100,0,117,0,108,0,101,0,32,0,105,0,110,0,105,0,116,0,32,0,38,0,38,0,32,0,103,0,105,0,116,0,32,0,115,0,117,0,98,0,109,0,111,0,100,0,117,0,108,0,101,0,32,0,117,0,112,0,100,0,97,0,116,0,101,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,116,0,101,0,115,0,116,0,34,0,58,0,32,0,34,0,109,0,97,0,107,0,101,0,32,0,116,0,101,0,115,0,116,0,34,0,44,0,10,0,32,0,32,0,32,0,32,0,34,0,98,0,117,0,105,0,108,0,100,0,34,0,58,0,32,0,34,0,109,0,97,0,107,0,101,0,32,0,106,0,115,0,34,0,10,0,32,0,32,0,125,0,44,0,10,0,32,0,32,0,34,0,99,0,111,0,110,0,102,0,105,0,103,0,34,0,58,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,34,0,98,0,108,0,97,0,110,0,107,0,101,0,116,0,34,0,58,0,32,0,123,0,10,0,32,0,32,0,32,0,32,0,32,0,32,0,34,0,112,0,97,0,116,0,116,0,101,0,114,0,110,0,34,0,58,0,32,0,34,0,91,0,99,0,112,0,116,0,97,0,98,0,108,0,101,0,46,0,106,0,115,0,44,0,99,0,112,0,117,0,116,0,105,0,108,0,115,0,46,0,106,0,115,0,44,0,99,0,112,0,101,0,120,0,99,0,101,0,108,0,46,0,106,0,115,0,93,0,34,0,10,0,32,0,32,0,32,0,32,0,125,0,10,0,32,0,32,0,125,0,44,0,10,0,32,0,32,0,34,0,98,0,117,0,103,0,115,0,34,0,58,0,32,0,123,0,32,0,34,0,117,0,114,0,108,0,34,0,58,0,32,0,34,0,104,0,116,0,116,0,112,0,115,0,58,0,47,0,47,0,103,0,105,0,116,0,104,0,117,0,98,0,46,0,99,0,111,0,109,0,47,0,83,0,104,0,101,0,101,0,116,0,74,0,83,0,47,0,106,0,115,0,45,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,47,0,105,0,115,0,115,0,117,0,101,0,115,0,34,0,32,0,125,0,44,0,10,0,32,0,32,0,34,0,108,0,105,0,99,0,101,0,110,0,115,0,101,0,34,0,58,0,32,0,34,0,65,0,112,0,97,0,99,0,104,0,101,0,45,0,50,0,46,0,48,0,34,0,44,0,10,0,32,0,32,0,34,0,101,0,110,0,103,0,105,0,110,0,101,0,115,0,34,0,58,0,32,0,123,0,32,0,34,0,110,0,111,0,100,0,101,0,34,0,58,0,32,0,34,0,62,0,61,0,48,0,46,0,56,0,34,0,32,0,125,0,10,0,125,0,10,0,96,0,96,0,96,0,10,0,10,0,96,0,96,0,96,0,62,0,46,0,118,0,111,0,99,0,114,0,99,0,10,0,123,0,32,0,34,0,112,0,111,0,115,0,116,0,34,0,58,0,32,0,34,0,109,0,97,0,107,0,101,0,32,0,106,0,115,0,34,0,32,0,125,0,10,0,96,0,96,0,96,0,10,0,10,0,96,0,96,0,96,0,62,0,46,0,103,0,105,0,116,0,105,0,103,0,110,0,111,0,114,0,101,0,10,0,46,0,103,0,105,0,116,0,105,0,103,0,110,0,111,0,114,0,101,0,10,0,99,0,111,0,100,0,101,0,112,0,97,0,103,0,101,0,115,0,47,0,10,0,46,0,118,0,111,0,99,0,114,0,99,0,10,0,110,0,111,0,100,0,101,0,95,0,109,0,111,0,100,0,117,0,108,0,101,0,115,0,47,0,10,0,109,0,97,0,107,0,101,0,46,0,115,0,104,0,10,0,109,0,97,0,107,0,101,0,46,0,110,0,106,0,115,0,10,0,109,0,105,0,115,0,99,0,47,0,99,0,111,0,118,0,101,0,114,0,97,0,103,0,101,0,46,0,104,0,116,0,109,0,108,0,10,0,96,0,96,0,96,0,10]
;
README['utf32le'] = 
[35,0,0,0,32,0,0,0,71,0,0,0,101,0,0,0,116,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,108,0,0,0,100,0,0,0,115,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,105,0,0,0,102,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,44,0,0,0,117,0,0,0,114,0,0,0,108,0,0,0,44,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,96,0,0,0,32,0,0,0,40,0,0,0,83,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,61,0,0,0,49,0,0,0,44,0,0,0,32,0,0,0,68,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,61,0,0,0,50,0,0,0,41,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,10,0,0,0,51,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,48,0,0,0,51,0,0,0,55,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,52,0,0,0,51,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,52,0,0,0,51,0,0,0,55,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,53,0,0,0,48,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,53,0,0,0,48,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,55,0,0,0,51,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,55,0,0,0,51,0,0,0,55,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,55,0,0,0,55,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,55,0,0,0,55,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,53,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,53,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,53,0,0,0,50,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,53,0,0,0,50,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,53,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,53,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,53,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,53,0,0,0,55,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,49,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,49,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,50,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,50,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,51,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,51,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,52,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,52,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,54,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,57,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,57,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,55,0,0,0,52,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,55,0,0,0,52,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,55,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,55,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,57,0,0,0,51,0,0,0,50,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,57,0,0,0,51,0,0,0,50,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,57,0,0,0,52,0,0,0,57,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,57,0,0,0,52,0,0,0,57,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,57,0,0,0,53,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,57,0,0,0,53,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,50,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,48,0,0,0,50,0,0,0,54,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,49,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,49,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,50,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,50,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,51,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,51,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,52,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,52,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,55,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,56,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,56,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,78,0,0,0,111,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,87,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,117,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,105,0,0,0,109,0,0,0,97,0,0,0,114,0,0,0,121,0,0,0,10,0,0,0,100,0,0,0,105,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,117,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,105,0,0,0,118,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,56,0,0,0,70,0,0,0,70,0,0,0,96,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,119,0,0,0,104,0,0,0,105,0,0,0,99,0,0,0,104,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,65,0,0,0,112,0,0,0,112,0,0,0,108,0,0,0,101,0,0,0,10,0,0,0,108,0,0,0,111,0,0,0,103,0,0,0,111,0,0,0,32,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,99,0,0,0,115,0,0,0,32,0,0,0,98,0,0,0,117,0,0,0,116,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,32,0,0,0,103,0,0,0,97,0,0,0,114,0,0,0,98,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,32,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,111,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,111,0,0,0,112,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,109,0,0,0,115,0,0,0,41,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,116,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,121,0,0,0,32,0,0,0,98,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,105,0,0,0,114,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,10,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,102,0,0,0,97,0,0,0,108,0,0,0,108,0,0,0,32,0,0,0,98,0,0,0,97,0,0,0,99,0,0,0,107,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,98,0,0,0,101,0,0,0,104,0,0,0,97,0,0,0,118,0,0,0,105,0,0,0,111,0,0,0,114,0,0,0,44,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,119,0,0,0,104,0,0,0,105,0,0,0,99,0,0,0,104,0,0,0,32,0,0,0,99,0,0,0,97,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,76,0,0,0,69,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,116,0,0,0,10,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,97,0,0,0,98,0,0,0,115,0,0,0,111,0,0,0,108,0,0,0,117,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,58,0,0,0,47,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,47,0,0,0,82,0,0,0,79,0,0,0,77,0,0,0,65,0,0,0,78,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,47,0,0,0,71,0,0,0,82,0,0,0,69,0,0,0,69,0,0,0,75,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,89,0,0,0,82,0,0,0,73,0,0,0,76,0,0,0,76,0,0,0,73,0,0,0,67,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,57,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,47,0,0,0,76,0,0,0,65,0,0,0,84,0,0,0,73,0,0,0,78,0,0,0,50,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,55,0,0,0,57,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,47,0,0,0,73,0,0,0,67,0,0,0,69,0,0,0,76,0,0,0,65,0,0,0,78,0,0,0,68,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,49,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,47,0,0,0,84,0,0,0,85,0,0,0,82,0,0,0,75,0,0,0,73,0,0,0,83,0,0,0,72,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,110,0,0,0,117,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,115,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,109,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,45,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,88,0,0,0,96,0,0,0,32,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,96,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,48,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,88,0,0,0,96,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,49,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,50,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,50,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,51,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,51,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,52,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,52,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,54,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,55,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,56,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,56,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,57,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,57,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,49,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,49,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,51,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,51,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,52,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,52,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,54,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,35,0,0,0,32,0,0,0,71,0,0,0,101,0,0,0,110,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,108,0,0,0,108,0,0,0,111,0,0,0,119,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,118,0,0,0,97,0,0,0,105,0,0,0,108,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,46,0,0,0,78,0,0,0,69,0,0,0,84,0,0,0,32,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,87,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,55,0,0,0,48,0,0,0,56,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,65,0,0,0,114,0,0,0,97,0,0,0,98,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,65,0,0,0,83,0,0,0,77,0,0,0,79,0,0,0,32,0,0,0,55,0,0,0,48,0,0,0,56,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,55,0,0,0,50,0,0,0,48,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,65,0,0,0,114,0,0,0,97,0,0,0,98,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,84,0,0,0,114,0,0,0,97,0,0,0,110,0,0,0,115,0,0,0,112,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,32,0,0,0,65,0,0,0,83,0,0,0,77,0,0,0,79,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,65,0,0,0,114,0,0,0,97,0,0,0,98,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,68,0,0,0,79,0,0,0,83,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,56,0,0,0,53,0,0,0,56,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,79,0,0,0,69,0,0,0,77,0,0,0,32,0,0,0,77,0,0,0,117,0,0,0,108,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,49,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,56,0,0,0,55,0,0,0,48,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,77,0,0,0,117,0,0,0,108,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,47,0,0,0,82,0,0,0,79,0,0,0,69,0,0,0,67,0,0,0,69,0,0,0,32,0,0,0,40,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,50,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,77,0,0,0,117,0,0,0,108,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,50,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,52,0,0,0,55,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,49,0,0,0,47,0,0,0,79,0,0,0,112,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,83,0,0,0,121,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,109,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,48,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,85,0,0,0,83,0,0,0,45,0,0,0,67,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,100,0,0,0,97,0,0,0,32,0,0,0,40,0,0,0,48,0,0,0,51,0,0,0,55,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,85,0,0,0,83,0,0,0,45,0,0,0,67,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,100,0,0,0,97,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,49,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,71,0,0,0,101,0,0,0,114,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,121,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,51,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,71,0,0,0,101,0,0,0,114,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,121,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,50,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,68,0,0,0,101,0,0,0,110,0,0,0,109,0,0,0,97,0,0,0,114,0,0,0,107,0,0,0,45,0,0,0,78,0,0,0,111,0,0,0,114,0,0,0,119,0,0,0,97,0,0,0,121,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,55,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,68,0,0,0,101,0,0,0,110,0,0,0,109,0,0,0,97,0,0,0,114,0,0,0,107,0,0,0,45,0,0,0,78,0,0,0,111,0,0,0,114,0,0,0,119,0,0,0,97,0,0,0,121,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,51,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,70,0,0,0,105,0,0,0,110,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,45,0,0,0,83,0,0,0,119,0,0,0,101,0,0,0,100,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,56,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,70,0,0,0,105,0,0,0,110,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,45,0,0,0,83,0,0,0,119,0,0,0,101,0,0,0,100,0,0,0,101,0,0,0,110,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,52,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,73,0,0,0,116,0,0,0,97,0,0,0,108,0,0,0,121,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,48,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,116,0,0,0,97,0,0,0,108,0,0,0,121,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,53,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,65,0,0,0,109,0,0,0,101,0,0,0,114,0,0,0,105,0,0,0,99,0,0,0,97,0,0,0,45,0,0,0,83,0,0,0,112,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,52,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,83,0,0,0,112,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,54,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,85,0,0,0,110,0,0,0,105,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,75,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,100,0,0,0,111,0,0,0,109,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,85,0,0,0,75,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,55,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,70,0,0,0,114,0,0,0,97,0,0,0,110,0,0,0,99,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,57,0,0,0,55,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,70,0,0,0,114,0,0,0,97,0,0,0,110,0,0,0,99,0,0,0,101,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,56,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,73,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,110,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,40,0,0,0,53,0,0,0,48,0,0,0,48,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,110,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,57,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,73,0,0,0,99,0,0,0,101,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,55,0,0,0,49,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,99,0,0,0,101,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,105,0,0,0,99,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,51,0,0,0,54,0,0,0,49,0,0,0,32,0,0,0,32,0,0,0,75,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,74,0,0,0,111,0,0,0,104,0,0,0,97,0,0,0,98,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,32,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,32,0,0,0,84,0,0,0,114,0,0,0,97,0,0,0,100,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,66,0,0,0,105,0,0,0,103,0,0,0,53,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,84,0,0,0,114,0,0,0,97,0,0,0,100,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,51,0,0,0,32,0,0,0,75,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,52,0,0,0,32,0,0,0,65,0,0,0,114,0,0,0,97,0,0,0,98,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,53,0,0,0,32,0,0,0,72,0,0,0,101,0,0,0,98,0,0,0,114,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,32,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,71,0,0,0,66,0,0,0,32,0,0,0,50,0,0,0,51,0,0,0,49,0,0,0,50,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,32,0,0,0,82,0,0,0,111,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,55,0,0,0,32,0,0,0,85,0,0,0,107,0,0,0,114,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,49,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,97,0,0,0,105,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,50,0,0,0,32,0,0,0,67,0,0,0,114,0,0,0,111,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,32,0,0,0,67,0,0,0,78,0,0,0,83,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,105,0,0,0,119,0,0,0,97,0,0,0,110,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,84,0,0,0,114,0,0,0,97,0,0,0,100,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,40,0,0,0,67,0,0,0,78,0,0,0,83,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,32,0,0,0,84,0,0,0,67,0,0,0,65,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,105,0,0,0,119,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,32,0,0,0,69,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,105,0,0,0,119,0,0,0,97,0,0,0,110,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,84,0,0,0,114,0,0,0,97,0,0,0,100,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,40,0,0,0,69,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,51,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,53,0,0,0,53,0,0,0,53,0,0,0,48,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,105,0,0,0,119,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,52,0,0,0,32,0,0,0,84,0,0,0,101,0,0,0,108,0,0,0,101,0,0,0,84,0,0,0,101,0,0,0,120,0,0,0,116,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,105,0,0,0,119,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,53,0,0,0,32,0,0,0,87,0,0,0,97,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,105,0,0,0,119,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,53,0,0,0,32,0,0,0,73,0,0,0,65,0,0,0,53,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,82,0,0,0,86,0,0,0,32,0,0,0,73,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,110,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,65,0,0,0,108,0,0,0,112,0,0,0,104,0,0,0,97,0,0,0,98,0,0,0,101,0,0,0,116,0,0,0,32,0,0,0,78,0,0,0,111,0,0,0,46,0,0,0,32,0,0,0,53,0,0,0,44,0,0,0,32,0,0,0,55,0,0,0,45,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,87,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,112,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,65,0,0,0,53,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,54,0,0,0,32,0,0,0,73,0,0,0,65,0,0,0,53,0,0,0,32,0,0,0,71,0,0,0,101,0,0,0,114,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,55,0,0,0,45,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,55,0,0,0,32,0,0,0,73,0,0,0,65,0,0,0,53,0,0,0,32,0,0,0,83,0,0,0,119,0,0,0,101,0,0,0,100,0,0,0,105,0,0,0,115,0,0,0,104,0,0,0,32,0,0,0,40,0,0,0,55,0,0,0,45,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,56,0,0,0,32,0,0,0,73,0,0,0,65,0,0,0,53,0,0,0,32,0,0,0,78,0,0,0,111,0,0,0,114,0,0,0,119,0,0,0,101,0,0,0,103,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,55,0,0,0,45,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,54,0,0,0,49,0,0,0,32,0,0,0,84,0,0,0,46,0,0,0,54,0,0,0,49,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,54,0,0,0,57,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,54,0,0,0,57,0,0,0,51,0,0,0,55,0,0,0,32,0,0,0,78,0,0,0,111,0,0,0,110,0,0,0,45,0,0,0,83,0,0,0,112,0,0,0,97,0,0,0,99,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,65,0,0,0,99,0,0,0,99,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,51,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,71,0,0,0,101,0,0,0,114,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,121,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,55,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,68,0,0,0,101,0,0,0,110,0,0,0,109,0,0,0,97,0,0,0,114,0,0,0,107,0,0,0,45,0,0,0,78,0,0,0,111,0,0,0,114,0,0,0,119,0,0,0,97,0,0,0,121,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,56,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,70,0,0,0,105,0,0,0,110,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,45,0,0,0,83,0,0,0,119,0,0,0,101,0,0,0,100,0,0,0,101,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,48,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,73,0,0,0,116,0,0,0,97,0,0,0,108,0,0,0,121,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,52,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,65,0,0,0,109,0,0,0,101,0,0,0,114,0,0,0,105,0,0,0,99,0,0,0,97,0,0,0,45,0,0,0,83,0,0,0,112,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,85,0,0,0,110,0,0,0,105,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,75,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,100,0,0,0,111,0,0,0,109,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,57,0,0,0,48,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,116,0,0,0,97,0,0,0,107,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,32,0,0,0,69,0,0,0,120,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,100,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,57,0,0,0,55,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,70,0,0,0,114,0,0,0,97,0,0,0,110,0,0,0,99,0,0,0,101,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,52,0,0,0,50,0,0,0,48,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,65,0,0,0,114,0,0,0,97,0,0,0,98,0,0,0,105,0,0,0,99,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,52,0,0,0,50,0,0,0,51,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,71,0,0,0,114,0,0,0,101,0,0,0,101,0,0,0,107,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,52,0,0,0,50,0,0,0,52,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,72,0,0,0,101,0,0,0,98,0,0,0,114,0,0,0,101,0,0,0,119,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,51,0,0,0,51,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,75,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,69,0,0,0,120,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,100,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,51,0,0,0,56,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,97,0,0,0,105,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,54,0,0,0,54,0,0,0,32,0,0,0,82,0,0,0,117,0,0,0,115,0,0,0,115,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,75,0,0,0,79,0,0,0,73,0,0,0,56,0,0,0,45,0,0,0,82,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,121,0,0,0,114,0,0,0,105,0,0,0,108,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,75,0,0,0,79,0,0,0,73,0,0,0,56,0,0,0,45,0,0,0,82,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,55,0,0,0,49,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,73,0,0,0,99,0,0,0,101,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,105,0,0,0,99,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,56,0,0,0,48,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,67,0,0,0,121,0,0,0,114,0,0,0,105,0,0,0,108,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,82,0,0,0,117,0,0,0,115,0,0,0,115,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,48,0,0,0,53,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,84,0,0,0,117,0,0,0,114,0,0,0,107,0,0,0,105,0,0,0,115,0,0,0,104,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,50,0,0,0,52,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,49,0,0,0,47,0,0,0,79,0,0,0,112,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,83,0,0,0,121,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,109,0,0,0,32,0,0,0,40,0,0,0,49,0,0,0,48,0,0,0,52,0,0,0,55,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,51,0,0,0,50,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,74,0,0,0,73,0,0,0,83,0,0,0,32,0,0,0,48,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,45,0,0,0,49,0,0,0,57,0,0,0,57,0,0,0,48,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,48,0,0,0,50,0,0,0,49,0,0,0,50,0,0,0,45,0,0,0,49,0,0,0,57,0,0,0,57,0,0,0,48,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,71,0,0,0,66,0,0,0,50,0,0,0,51,0,0,0,49,0,0,0,50,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,40,0,0,0,71,0,0,0,66,0,0,0,50,0,0,0,51,0,0,0,49,0,0,0,50,0,0,0,45,0,0,0,56,0,0,0,48,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,52,0,0,0,57,0,0,0,32,0,0,0,75,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,87,0,0,0,97,0,0,0,110,0,0,0,115,0,0,0,117,0,0,0,110,0,0,0,103,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,49,0,0,0,48,0,0,0,50,0,0,0,53,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,67,0,0,0,121,0,0,0,114,0,0,0,105,0,0,0,108,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,83,0,0,0,101,0,0,0,114,0,0,0,98,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,45,0,0,0,66,0,0,0,117,0,0,0,108,0,0,0,103,0,0,0,97,0,0,0,114,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,49,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,32,0,0,0,69,0,0,0,120,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,100,0,0,0,47,0,0,0,69,0,0,0,120,0,0,0,116,0,0,0,32,0,0,0,65,0,0,0,108,0,0,0,112,0,0,0,104,0,0,0,97,0,0,0,32,0,0,0,76,0,0,0,111,0,0,0,119,0,0,0,101,0,0,0,114,0,0,0,99,0,0,0,97,0,0,0,115,0,0,0,101,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,49,0,0,0,56,0,0,0,54,0,0,0,54,0,0,0,32,0,0,0,85,0,0,0,107,0,0,0,114,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,75,0,0,0,79,0,0,0,73,0,0,0,56,0,0,0,45,0,0,0,85,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,121,0,0,0,114,0,0,0,105,0,0,0,108,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,75,0,0,0,79,0,0,0,73,0,0,0,56,0,0,0,45,0,0,0,85,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,57,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,112,0,0,0,97,0,0,0,32,0,0,0,51,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,51,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,56,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,56,0,0,0,32,0,0,0,72,0,0,0,101,0,0,0,98,0,0,0,114,0,0,0,101,0,0,0,119,0,0,0,59,0,0,0,32,0,0,0,72,0,0,0,101,0,0,0,98,0,0,0,114,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,45,0,0,0,76,0,0,0,111,0,0,0,103,0,0,0,105,0,0,0,99,0,0,0,97,0,0,0,108,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,48,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,32,0,0,0,104,0,0,0,97,0,0,0,108,0,0,0,102,0,0,0,119,0,0,0,105,0,0,0,100,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,116,0,0,0,97,0,0,0,107,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,59,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,74,0,0,0,73,0,0,0,83,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,49,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,104,0,0,0,97,0,0,0,108,0,0,0,102,0,0,0,119,0,0,0,105,0,0,0,100,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,116,0,0,0,97,0,0,0,107,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,59,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,74,0,0,0,73,0,0,0,83,0,0,0,45,0,0,0,65,0,0,0,108,0,0,0,108,0,0,0,111,0,0,0,119,0,0,0,32,0,0,0,49,0,0,0,32,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,50,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,74,0,0,0,73,0,0,0,83,0,0,0,32,0,0,0,88,0,0,0,32,0,0,0,48,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,45,0,0,0,49,0,0,0,57,0,0,0,56,0,0,0,57,0,0,0,59,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,74,0,0,0,73,0,0,0,83,0,0,0,45,0,0,0,65,0,0,0,108,0,0,0,108,0,0,0,111,0,0,0,119,0,0,0,32,0,0,0,49,0,0,0,32,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,32,0,0,0,45,0,0,0,32,0,0,0,83,0,0,0,79,0,0,0,47,0,0,0,83,0,0,0,73,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,53,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,32,0,0,0,75,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,55,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,49,0,0,0,57,0,0,0,51,0,0,0,50,0,0,0,32,0,0,0,69,0,0,0,85,0,0,0,67,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,49,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,32,0,0,0,69,0,0,0,85,0,0,0,67,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,40,0,0,0,69,0,0,0,85,0,0,0,67,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,49,0,0,0,57,0,0,0,52,0,0,0,57,0,0,0,32,0,0,0,69,0,0,0,85,0,0,0,67,0,0,0,32,0,0,0,75,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,50,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,32,0,0,0,72,0,0,0,90,0,0,0,45,0,0,0,71,0,0,0,66,0,0,0,50,0,0,0,51,0,0,0,49,0,0,0,50,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,40,0,0,0,72,0,0,0,90,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,52,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,32,0,0,0,87,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,32,0,0,0,88,0,0,0,80,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,108,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,58,0,0,0,32,0,0,0,71,0,0,0,66,0,0,0,49,0,0,0,56,0,0,0,48,0,0,0,51,0,0,0,48,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,52,0,0,0,32,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,40,0,0,0,71,0,0,0,66,0,0,0,49,0,0,0,56,0,0,0,48,0,0,0,51,0,0,0,48,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,68,0,0,0,101,0,0,0,118,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,103,0,0,0,97,0,0,0,114,0,0,0,105,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,51,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,66,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,97,0,0,0,108,0,0,0,105,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,52,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,109,0,0,0,105,0,0,0,108,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,53,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,84,0,0,0,101,0,0,0,108,0,0,0,117,0,0,0,103,0,0,0,117,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,54,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,65,0,0,0,115,0,0,0,115,0,0,0,97,0,0,0,109,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,55,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,79,0,0,0,114,0,0,0,105,0,0,0,121,0,0,0,97,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,110,0,0,0,110,0,0,0,97,0,0,0,100,0,0,0,97,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,57,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,77,0,0,0,97,0,0,0,108,0,0,0,97,0,0,0,121,0,0,0,97,0,0,0,108,0,0,0,97,0,0,0,109,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,71,0,0,0,117,0,0,0,106,0,0,0,97,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,49,0,0,0,49,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,80,0,0,0,117,0,0,0,110,0,0,0,106,0,0,0,97,0,0,0,98,0,0,0,105,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,10,0,0,0,55,0,0,0,48,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,55,0,0,0,50,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,53,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,55,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,52,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,57,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,51,0,0,0,54,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,54,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,54,0,0,0,57,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,57,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,57,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,52,0,0,0,50,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,52,0,0,0,50,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,52,0,0,0,50,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,51,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,51,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,54,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,55,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,56,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,48,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,50,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,51,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,52,0,0,0,57,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,49,0,0,0,48,0,0,0,50,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,49,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,49,0,0,0,56,0,0,0,54,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,57,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,51,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,49,0,0,0,57,0,0,0,51,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,49,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,49,0,0,0,57,0,0,0,52,0,0,0,57,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,50,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,52,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,57,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,49,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,108,0,0,0,108,0,0,0,111,0,0,0,119,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,86,0,0,0,105,0,0,0,115,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,70,0,0,0,111,0,0,0,120,0,0,0,80,0,0,0,114,0,0,0,111,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,54,0,0,0,50,0,0,0,48,0,0,0,32,0,0,0,77,0,0,0,97,0,0,0,122,0,0,0,111,0,0,0,118,0,0,0,105,0,0,0,97,0,0,0,32,0,0,0,40,0,0,0,80,0,0,0,111,0,0,0,108,0,0,0,105,0,0,0,115,0,0,0,104,0,0,0,41,0,0,0,32,0,0,0,77,0,0,0,83,0,0,0,45,0,0,0,68,0,0,0,79,0,0,0,83,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,56,0,0,0,57,0,0,0,53,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,109,0,0,0,101,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,107,0,0,0,253,0,0,0,32,0,0,0,40,0,0,0,67,0,0,0,122,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,41,0,0,0,32,0,0,0,77,0,0,0,83,0,0,0,45,0,0,0,68,0,0,0,79,0,0,0,83,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,10,0,0,0,54,0,0,0,50,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,57,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,107,0,0,0,110,0,0,0,111,0,0,0,119,0,0,0,110,0,0,0,32,0,0,0,109,0,0,0,105,0,0,0,115,0,0,0,115,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,117,0,0,0,109,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,82,0,0,0,69,0,0,0,65,0,0,0,68,0,0,0,77,0,0,0,69,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,35,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,105,0,0,0,108,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,78,0,0,0,111,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,112,0,0,0,116,0,0,0,32,0,0,0,96,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,96,0,0,0,32,0,0,0,40,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,108,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,41,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,108,0,0,0,108,0,0,0,32,0,0,0,103,0,0,0,101,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,97,0,0,0,116,0,0,0,97,0,0,0,10,0,0,0,40,0,0,0,112,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,45,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,105,0,0,0,114,0,0,0,115,0,0,0,41,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,118,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,114,0,0,0,111,0,0,0,112,0,0,0,112,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,116,0,0,0,104,0,0,0,115,0,0,0,10,0,0,0,96,0,0,0,46,0,0,0,47,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,47,0,0,0,60,0,0,0,67,0,0,0,79,0,0,0,68,0,0,0,69,0,0,0,80,0,0,0,65,0,0,0,71,0,0,0,69,0,0,0,62,0,0,0,46,0,0,0,84,0,0,0,66,0,0,0,76,0,0,0,96,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,70,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,97,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,108,0,0,0,97,0,0,0,115,0,0,0,116,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,32,0,0,0,108,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,96,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,66,0,0,0,76,0,0,0,96,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,54,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,67,0,0,0,54,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,55,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,67,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,56,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,48,0,0,0,65,0,0,0,70,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,57,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,56,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,65,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,57,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,66,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,65,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,67,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,48,0,0,0,66,0,0,0,56,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,68,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,68,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,69,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,66,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,67,0,0,0,55,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,119,0,0,0,104,0,0,0,105,0,0,0,99,0,0,0,104,0,0,0,32,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,54,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,96,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,67,0,0,0,54,0,0,0,41,0,0,0,96,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,118,0,0,0,105,0,0,0,99,0,0,0,101,0,0,0,32,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,97,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,35,0,0,0,32,0,0,0,87,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,45,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,32,0,0,0,98,0,0,0,117,0,0,0,105,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,112,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,111,0,0,0,32,0,0,0,98,0,0,0,117,0,0,0,105,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,111,0,0,0,117,0,0,0,114,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,44,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,117,0,0,0,108,0,0,0,116,0,0,0,32,0,0,0,96,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,47,0,0,0,77,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,69,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,96,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,65,0,0,0,102,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,115,0,0,0,97,0,0,0,118,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,97,0,0,0,114,0,0,0,100,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,96,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,96,0,0,0,44,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,115,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,119,0,0,0,107,0,0,0,32,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,112,0,0,0,116,0,0,0,32,0,0,0,40,0,0,0,96,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,96,0,0,0,41,0,0,0,32,0,0,0,116,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,99,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,10,0,0,0,35,0,0,0,33,0,0,0,47,0,0,0,98,0,0,0,105,0,0,0,110,0,0,0,47,0,0,0,98,0,0,0,97,0,0,0,115,0,0,0,104,0,0,0,10,0,0,0,105,0,0,0,102,0,0,0,32,0,0,0,91,0,0,0,32,0,0,0,33,0,0,0,32,0,0,0,45,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,47,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,32,0,0,0,93,0,0,0,59,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,105,0,0,0,116,0,0,0,59,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,10,0,0,0,60,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,47,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,45,0,0,0,115,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,39,0,0,0,92,0,0,0,116,0,0,0,39,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,97,0,0,0,119,0,0,0,107,0,0,0,32,0,0,0,39,0,0,0,78,0,0,0,70,0,0,0,62,0,0,0,50,0,0,0,32,0,0,0,123,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,41,0,0,0,32,0,0,0,99,0,0,0,108,0,0,0,111,0,0,0,115,0,0,0,101,0,0,0,40,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,61,0,0,0,34,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,47,0,0,0,34,0,0,0,32,0,0,0,36,0,0,0,49,0,0,0,32,0,0,0,34,0,0,0,46,0,0,0,84,0,0,0,66,0,0,0,76,0,0,0,34,0,0,0,125,0,0,0,32,0,0,0,78,0,0,0,70,0,0,0,61,0,0,0,61,0,0,0,50,0,0,0,32,0,0,0,123,0,0,0,112,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,32,0,0,0,62,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,125,0,0,0,39,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,105,0,0,0,108,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,112,0,0,0,116,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,110,0,0,0,106,0,0,0,115,0,0,0,96,0,0,0,32,0,0,0,116,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,103,0,0,0,117,0,0,0,109,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,44,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,114,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,112,0,0,0,111,0,0,0,110,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,10,0,0,0,103,0,0,0,101,0,0,0,110,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,35,0,0,0,32,0,0,0,82,0,0,0,97,0,0,0,119,0,0,0,32,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,110,0,0,0,106,0,0,0,115,0,0,0,10,0,0,0,35,0,0,0,33,0,0,0,47,0,0,0,117,0,0,0,115,0,0,0,114,0,0,0,47,0,0,0,98,0,0,0,105,0,0,0,110,0,0,0,47,0,0,0,101,0,0,0,110,0,0,0,118,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,103,0,0,0,118,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,46,0,0,0,97,0,0,0,114,0,0,0,103,0,0,0,118,0,0,0,46,0,0,0,115,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,101,0,0,0,40,0,0,0,49,0,0,0,41,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,105,0,0,0,114,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,102,0,0,0,115,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,97,0,0,0,114,0,0,0,103,0,0,0,118,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,60,0,0,0,32,0,0,0,50,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,111,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,114,0,0,0,114,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,34,0,0,0,117,0,0,0,115,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,58,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,110,0,0,0,106,0,0,0,115,0,0,0,32,0,0,0,60,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,95,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,120,0,0,0,62,0,0,0,32,0,0,0,91,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,105,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,93,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,120,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,50,0,0,0,50,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,47,0,0,0,42,0,0,0,32,0,0,0,69,0,0,0,73,0,0,0,78,0,0,0,86,0,0,0,65,0,0,0,76,0,0,0,32,0,0,0,42,0,0,0,47,0,0,0,10,0,0,0,125,0,0,0,10,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,103,0,0,0,118,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,106,0,0,0,115,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,103,0,0,0,118,0,0,0,91,0,0,0,50,0,0,0,93,0,0,0,32,0,0,0,124,0,0,0,124,0,0,0,32,0,0,0,34,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,34,0,0,0,59,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,120,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,70,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,34,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,47,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,46,0,0,0,84,0,0,0,66,0,0,0,76,0,0,0,34,0,0,0,44,0,0,0,34,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,10,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,121,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,120,0,0,0,46,0,0,0,115,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,34,0,0,0,92,0,0,0,110,0,0,0,34,0,0,0,41,0,0,0,46,0,0,0,109,0,0,0,97,0,0,0,112,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,122,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,119,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,122,0,0,0,46,0,0,0,115,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,34,0,0,0,92,0,0,0,116,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,119,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,60,0,0,0,32,0,0,0,50,0,0,0,41,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,119,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,91,0,0,0,78,0,0,0,117,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,119,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,41,0,0,0,44,0,0,0,32,0,0,0,78,0,0,0,117,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,119,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,41,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,41,0,0,0,46,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,122,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,122,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,62,0,0,0,32,0,0,0,49,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,68,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,83,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,103,0,0,0,101,0,0,0,110,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,103,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,105,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,105,0,0,0,109,0,0,0,117,0,0,0,109,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,10,0,0,0,117,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,100,0,0,0,105,0,0,0,115,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,117,0,0,0,105,0,0,0,115,0,0,0,104,0,0,0,32,0,0,0,40,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,83,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,41,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,121,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,121,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,32,0,0,0,62,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,99,0,0,0,112,0,0,0,41,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,121,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,123,0,0,0,125,0,0,0,44,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,40,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,60,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,32,0,0,0,63,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,123,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,121,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,91,0,0,0,121,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,121,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,91,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,121,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,41,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,121,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,10,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,44,0,0,0,32,0,0,0,111,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,44,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,59,0,0,0,10,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,60,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,97,0,0,0,99,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,96,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,70,0,0,0,68,0,0,0,96,0,0,0,32,0,0,0,40,0,0,0,82,0,0,0,69,0,0,0,80,0,0,0,76,0,0,0,65,0,0,0,67,0,0,0,69,0,0,0,77,0,0,0,69,0,0,0,78,0,0,0,84,0,0,0,32,0,0,0,67,0,0,0,72,0,0,0,65,0,0,0,82,0,0,0,65,0,0,0,67,0,0,0,84,0,0,0,69,0,0,0,82,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,117,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,112,0,0,0,108,0,0,0,97,0,0,0,99,0,0,0,101,0,0,0,104,0,0,0,111,0,0,0,108,0,0,0,100,0,0,0,101,0,0,0,114,0,0,0,10,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,97,0,0,0,99,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,116,0,0,0,32,0,0,0,115,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,112,0,0,0,32,0,0,0,40,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,97,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,96,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,48,0,0,0,96,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,116,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,10,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,41,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,70,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,83,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,44,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,100,0,0,0,101,0,0,0,97,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,101,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,114,0,0,0,97,0,0,0,119,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,96,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,109,0,0,0,101,0,0,0,114,0,0,0,101,0,0,0,108,0,0,0,121,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,115,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,44,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,96,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,96,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,101,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,34,0,0,0,41,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,70,0,0,0,68,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,79,0,0,0,78,0,0,0,46,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,105,0,0,0,102,0,0,0,121,0,0,0,40,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,46,0,0,0,106,0,0,0,111,0,0,0,105,0,0,0,110,0,0,0,40,0,0,0,34,0,0,0,34,0,0,0,41,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,123,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,100,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,68,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,101,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,123,0,0,0,125,0,0,0,59,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,61,0,0,0,48,0,0,0,59,0,0,0,105,0,0,0,33,0,0,0,61,0,0,0,100,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,59,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,100,0,0,0,46,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,65,0,0,0,116,0,0,0,40,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,70,0,0,0,68,0,0,0,41,0,0,0,32,0,0,0,101,0,0,0,91,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,105,0,0,0,59,0,0,0,32,0,0,0,68,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,100,0,0,0,46,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,65,0,0,0,116,0,0,0,40,0,0,0,105,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,123,0,0,0,34,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,68,0,0,0,32,0,0,0,125,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,39,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,32,0,0,0,101,0,0,0,108,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,68,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,115,0,0,0,105,0,0,0,109,0,0,0,105,0,0,0,108,0,0,0,97,0,0,0,114,0,0,0,44,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,99,0,0,0,101,0,0,0,112,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,112,0,0,0,97,0,0,0,99,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,115,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,45,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,117,0,0,0,110,0,0,0,107,0,0,0,115,0,0,0,32,0,0,0,40,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,115,0,0,0,10,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,111,0,0,0,110,0,0,0,108,0,0,0,121,0,0,0,32,0,0,0,103,0,0,0,101,0,0,0,110,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,111,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,104,0,0,0,105,0,0,0,103,0,0,0,104,0,0,0,45,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,112,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,41,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,103,0,0,0,121,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,117,0,0,0,99,0,0,0,116,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,97,0,0,0,121,0,0,0,45,0,0,0,111,0,0,0,102,0,0,0,45,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,97,0,0,0,121,0,0,0,115,0,0,0,32,0,0,0,115,0,0,0,111,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,96,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,104,0,0,0,105,0,0,0,103,0,0,0,104,0,0,0,93,0,0,0,91,0,0,0,108,0,0,0,111,0,0,0,119,0,0,0,93,0,0,0,96,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,10,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,97,0,0,0,99,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,111,0,0,0,99,0,0,0,105,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,97,0,0,0,121,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,98,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,114,0,0,0,117,0,0,0,110,0,0,0,116,0,0,0,105,0,0,0,109,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,121,0,0,0,105,0,0,0,101,0,0,0,108,0,0,0,100,0,0,0,10,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,101,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,111,0,0,0,98,0,0,0,106,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,32,0,0,0,40,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,111,0,0,0,98,0,0,0,106,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,101,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,41,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,100,0,0,0,100,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,46,0,0,0,104,0,0,0,97,0,0,0,115,0,0,0,79,0,0,0,119,0,0,0,110,0,0,0,80,0,0,0,114,0,0,0,111,0,0,0,112,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,121,0,0,0,40,0,0,0,105,0,0,0,41,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,56,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,34,0,0,0,41,0,0,0,32,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,56,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,56,0,0,0,93,0,0,0,91,0,0,0,105,0,0,0,32,0,0,0,37,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,123,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,100,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,101,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,123,0,0,0,125,0,0,0,44,0,0,0,32,0,0,0,68,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,106,0,0,0,59,0,0,0,92,0,0,0,110,0,0,0,39,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,106,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,106,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,106,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,106,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,34,0,0,0,41,0,0,0,32,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,106,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,70,0,0,0,68,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,43,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,68,0,0,0,91,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,79,0,0,0,78,0,0,0,46,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,105,0,0,0,102,0,0,0,121,0,0,0,40,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,46,0,0,0,106,0,0,0,111,0,0,0,105,0,0,0,110,0,0,0,40,0,0,0,34,0,0,0,34,0,0,0,41,0,0,0,41,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,46,0,0,0,115,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,34,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,92,0,0,0,110,0,0,0,39,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,43,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,106,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,106,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,68,0,0,0,91,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,93,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,106,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,68,0,0,0,91,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,93,0,0,0,91,0,0,0,106,0,0,0,93,0,0,0,46,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,65,0,0,0,116,0,0,0,40,0,0,0,48,0,0,0,41,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,70,0,0,0,68,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,101,0,0,0,91,0,0,0,68,0,0,0,91,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,93,0,0,0,91,0,0,0,106,0,0,0,93,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,40,0,0,0,105,0,0,0,42,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,41,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,106,0,0,0,59,0,0,0,32,0,0,0,100,0,0,0,91,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,40,0,0,0,105,0,0,0,42,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,41,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,106,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,68,0,0,0,91,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,93,0,0,0,91,0,0,0,106,0,0,0,93,0,0,0,59,0,0,0,125,0,0,0,92,0,0,0,110,0,0,0,39,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,43,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,123,0,0,0,34,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,100,0,0,0,32,0,0,0,125,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,39,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,10,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,111,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,108,0,0,0,111,0,0,0,103,0,0,0,40,0,0,0,106,0,0,0,115,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,91,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,96,0,0,0,32,0,0,0,103,0,0,0,101,0,0,0,110,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,117,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,98,0,0,0,121,0,0,0,32,0,0,0,96,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,110,0,0,0,106,0,0,0,115,0,0,0,96,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,114,0,0,0,97,0,0,0,119,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,10,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,108,0,0,0,117,0,0,0,109,0,0,0,110,0,0,0,97,0,0,0,114,0,0,0,58,0,0,0,32,0,0,0,96,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,35,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,109,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,115,0,0,0,96,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,70,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,97,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,108,0,0,0,97,0,0,0,115,0,0,0,116,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,32,0,0,0,108,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,10,0,0,0,116,0,0,0,101,0,0,0,120,0,0,0,116,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,82,0,0,0,79,0,0,0,77,0,0,0,65,0,0,0,78,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,32,0,0,0,40,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,67,0,0,0,80,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,41,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,54,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,67,0,0,0,54,0,0,0,9,0,0,0,35,0,0,0,77,0,0,0,79,0,0,0,68,0,0,0,73,0,0,0,70,0,0,0,73,0,0,0,69,0,0,0,82,0,0,0,32,0,0,0,76,0,0,0,69,0,0,0,84,0,0,0,84,0,0,0,69,0,0,0,82,0,0,0,32,0,0,0,67,0,0,0,73,0,0,0,82,0,0,0,67,0,0,0,85,0,0,0,77,0,0,0,70,0,0,0,76,0,0,0,69,0,0,0,88,0,0,0,32,0,0,0,65,0,0,0,67,0,0,0,67,0,0,0,69,0,0,0,78,0,0,0,84,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,55,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,67,0,0,0,9,0,0,0,35,0,0,0,83,0,0,0,77,0,0,0,65,0,0,0,76,0,0,0,76,0,0,0,32,0,0,0,84,0,0,0,73,0,0,0,76,0,0,0,68,0,0,0,69,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,56,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,48,0,0,0,65,0,0,0,70,0,0,0,9,0,0,0,35,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,82,0,0,0,79,0,0,0,78,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,57,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,56,0,0,0,9,0,0,0,35,0,0,0,66,0,0,0,82,0,0,0,69,0,0,0,86,0,0,0,69,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,65,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,57,0,0,0,9,0,0,0,35,0,0,0,68,0,0,0,79,0,0,0,84,0,0,0,32,0,0,0,65,0,0,0,66,0,0,0,79,0,0,0,86,0,0,0,69,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,66,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,65,0,0,0,9,0,0,0,35,0,0,0,82,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,32,0,0,0,65,0,0,0,66,0,0,0,79,0,0,0,86,0,0,0,69,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,67,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,48,0,0,0,66,0,0,0,56,0,0,0,9,0,0,0,35,0,0,0,67,0,0,0,69,0,0,0,68,0,0,0,73,0,0,0,76,0,0,0,76,0,0,0,65,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,68,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,68,0,0,0,9,0,0,0,35,0,0,0,68,0,0,0,79,0,0,0,85,0,0,0,66,0,0,0,76,0,0,0,69,0,0,0,32,0,0,0,65,0,0,0,67,0,0,0,85,0,0,0,84,0,0,0,69,0,0,0,32,0,0,0,65,0,0,0,67,0,0,0,67,0,0,0,69,0,0,0,78,0,0,0,84,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,69,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,66,0,0,0,9,0,0,0,35,0,0,0,79,0,0,0,71,0,0,0,79,0,0,0,78,0,0,0,69,0,0,0,75,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,67,0,0,0,55,0,0,0,9,0,0,0,35,0,0,0,67,0,0,0,65,0,0,0,82,0,0,0,79,0,0,0,78,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,73,0,0,0,110,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,97,0,0,0,116,0,0,0,97,0,0,0,44,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,109,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,40,0,0,0,97,0,0,0,102,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,35,0,0,0,96,0,0,0,41,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,112,0,0,0,112,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,10,0,0,0,101,0,0,0,108,0,0,0,101,0,0,0,109,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,40,0,0,0,108,0,0,0,105,0,0,0,107,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,48,0,0,0,120,0,0,0,55,0,0,0,70,0,0,0,96,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,67,0,0,0,80,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,41,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,109,0,0,0,111,0,0,0,118,0,0,0,101,0,0,0,100,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,10,0,0,0,35,0,0,0,33,0,0,0,47,0,0,0,98,0,0,0,105,0,0,0,110,0,0,0,47,0,0,0,98,0,0,0,97,0,0,0,115,0,0,0,104,0,0,0,10,0,0,0,73,0,0,0,78,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,61,0,0,0,36,0,0,0,123,0,0,0,49,0,0,0,58,0,0,0,45,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,125,0,0,0,10,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,61,0,0,0,36,0,0,0,123,0,0,0,50,0,0,0,58,0,0,0,45,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,125,0,0,0,10,0,0,0,74,0,0,0,83,0,0,0,86,0,0,0,65,0,0,0,82,0,0,0,61,0,0,0,36,0,0,0,123,0,0,0,51,0,0,0,58,0,0,0,45,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,125,0,0,0,10,0,0,0,86,0,0,0,69,0,0,0,82,0,0,0,83,0,0,0,73,0,0,0,79,0,0,0,78,0,0,0,61,0,0,0,36,0,0,0,40,0,0,0,99,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,99,0,0,0,107,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,103,0,0,0,114,0,0,0,101,0,0,0,112,0,0,0,32,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,45,0,0,0,100,0,0,0,99,0,0,0,32,0,0,0,91,0,0,0,48,0,0,0,45,0,0,0,57,0,0,0,46,0,0,0,93,0,0,0,41,0,0,0,10,0,0,0,10,0,0,0,109,0,0,0,107,0,0,0,100,0,0,0,105,0,0,0,114,0,0,0,32,0,0,0,45,0,0,0,112,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,10,0,0,0,114,0,0,0,109,0,0,0,32,0,0,0,45,0,0,0,102,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,111,0,0,0,32,0,0,0,34,0,0,0,47,0,0,0,42,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,32,0,0,0,40,0,0,0,67,0,0,0,41,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,51,0,0,0,45,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,52,0,0,0,32,0,0,0,83,0,0,0,104,0,0,0,101,0,0,0,101,0,0,0,116,0,0,0,74,0,0,0,83,0,0,0,32,0,0,0,45,0,0,0,45,0,0,0,32,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,115,0,0,0,104,0,0,0,101,0,0,0,101,0,0,0,116,0,0,0,106,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,32,0,0,0,42,0,0,0,47,0,0,0,34,0,0,0,32,0,0,0,62,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,111,0,0,0,32,0,0,0,34,0,0,0,47,0,0,0,42,0,0,0,106,0,0,0,115,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,32,0,0,0,45,0,0,0,87,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,32,0,0,0,42,0,0,0,47,0,0,0,34,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,111,0,0,0,32,0,0,0,34,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,36,0,0,0,74,0,0,0,83,0,0,0,86,0,0,0,65,0,0,0,82,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,123,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,58,0,0,0,92,0,0,0,34,0,0,0,36,0,0,0,86,0,0,0,69,0,0,0,82,0,0,0,83,0,0,0,73,0,0,0,79,0,0,0,78,0,0,0,92,0,0,0,34,0,0,0,125,0,0,0,59,0,0,0,34,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,105,0,0,0,102,0,0,0,32,0,0,0,91,0,0,0,32,0,0,0,45,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,32,0,0,0,93,0,0,0,59,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,98,0,0,0,97,0,0,0,115,0,0,0,104,0,0,0,32,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,59,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,10,0,0,0,97,0,0,0,119,0,0,0,107,0,0,0,32,0,0,0,45,0,0,0,70,0,0,0,44,0,0,0,32,0,0,0,39,0,0,0,123,0,0,0,112,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,32,0,0,0,36,0,0,0,49,0,0,0,44,0,0,0,32,0,0,0,36,0,0,0,50,0,0,0,44,0,0,0,32,0,0,0,36,0,0,0,51,0,0,0,125,0,0,0,39,0,0,0,32,0,0,0,36,0,0,0,73,0,0,0,78,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,119,0,0,0,104,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,117,0,0,0,114,0,0,0,108,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,59,0,0,0,32,0,0,0,100,0,0,0,111,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,111,0,0,0,32,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,36,0,0,0,117,0,0,0,114,0,0,0,108,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,32,0,0,0,91,0,0,0,32,0,0,0,33,0,0,0,32,0,0,0,45,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,84,0,0,0,66,0,0,0,76,0,0,0,32,0,0,0,93,0,0,0,59,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,110,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,117,0,0,0,114,0,0,0,108,0,0,0,32,0,0,0,36,0,0,0,117,0,0,0,114,0,0,0,108,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,39,0,0,0,115,0,0,0,47,0,0,0,35,0,0,0,46,0,0,0,42,0,0,0,47,0,0,0,47,0,0,0,103,0,0,0,39,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,97,0,0,0,119,0,0,0,107,0,0,0,32,0,0,0,39,0,0,0,78,0,0,0,70,0,0,0,61,0,0,0,61,0,0,0,50,0,0,0,39,0,0,0,32,0,0,0,62,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,84,0,0,0,66,0,0,0,76,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,111,0,0,0,32,0,0,0,34,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,36,0,0,0,74,0,0,0,83,0,0,0,86,0,0,0,65,0,0,0,82,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,39,0,0,0,41,0,0,0,32,0,0,0,36,0,0,0,74,0,0,0,83,0,0,0,86,0,0,0,65,0,0,0,82,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,123,0,0,0,125,0,0,0,59,0,0,0,34,0,0,0,32,0,0,0,62,0,0,0,32,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,110,0,0,0,106,0,0,0,115,0,0,0,32,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,36,0,0,0,74,0,0,0,83,0,0,0,86,0,0,0,65,0,0,0,82,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,101,0,0,0,32,0,0,0,45,0,0,0,97,0,0,0,32,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,39,0,0,0,115,0,0,0,47,0,0,0,34,0,0,0,92,0,0,0,40,0,0,0,91,0,0,0,48,0,0,0,45,0,0,0,57,0,0,0,93,0,0,0,43,0,0,0,92,0,0,0,41,0,0,0,34,0,0,0,58,0,0,0,47,0,0,0,92,0,0,0,49,0,0,0,58,0,0,0,47,0,0,0,103,0,0,0,39,0,0,0,32,0,0,0,60,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,32,0,0,0,62,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,114,0,0,0,109,0,0,0,32,0,0,0,45,0,0,0,102,0,0,0,32,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,100,0,0,0,111,0,0,0,110,0,0,0,101,0,0,0,10,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,111,0,0,0,32,0,0,0,34,0,0,0,105,0,0,0,102,0,0,0,32,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,109,0,0,0,111,0,0,0,100,0,0,0,117,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,39,0,0,0,32,0,0,0,38,0,0,0,38,0,0,0,32,0,0,0,109,0,0,0,111,0,0,0,100,0,0,0,117,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,111,0,0,0,114,0,0,0,116,0,0,0,115,0,0,0,41,0,0,0,32,0,0,0,109,0,0,0,111,0,0,0,100,0,0,0,117,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,111,0,0,0,114,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,36,0,0,0,74,0,0,0,83,0,0,0,86,0,0,0,65,0,0,0,82,0,0,0,59,0,0,0,34,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,39,0,0,0,115,0,0,0,47,0,0,0,34,0,0,0,92,0,0,0,40,0,0,0,91,0,0,0,48,0,0,0,45,0,0,0,57,0,0,0,93,0,0,0,43,0,0,0,92,0,0,0,41,0,0,0,34,0,0,0,58,0,0,0,47,0,0,0,92,0,0,0,49,0,0,0,58,0,0,0,47,0,0,0,103,0,0,0,39,0,0,0,32,0,0,0,60,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,32,0,0,0,62,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,10,0,0,0,114,0,0,0,109,0,0,0,32,0,0,0,45,0,0,0,102,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,35,0,0,0,32,0,0,0,85,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,107,0,0,0,101,0,0,0,112,0,0,0,116,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,115,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,112,0,0,0,116,0,0,0,32,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,41,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,66,0,0,0,111,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,100,0,0,0,97,0,0,0,116,0,0,0,97,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,112,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,40,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,44,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,112,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,85,0,0,0,67,0,0,0,83,0,0,0,50,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,65,0,0,0,114,0,0,0,114,0,0,0,97,0,0,0,121,0,0,0,32,0,0,0,40,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,97,0,0,0,121,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,32,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,97,0,0,0,99,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,44,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,110,0,0,0,117,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,40,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,45,0,0,0,56,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,44,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,111,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,41,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,111,0,0,0,102,0,0,0,109,0,0,0,116,0,0,0,96,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,105,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,116,0,0,0,114,0,0,0,111,0,0,0,108,0,0,0,115,0,0,0,32,0,0,0,96,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,96,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,32,0,0,0,40,0,0,0,96,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,96,0,0,0,44,0,0,0,32,0,0,0,96,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,96,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,118,0,0,0,101,0,0,0,108,0,0,0,121,0,0,0,41,0,0,0,10,0,0,0,119,0,0,0,104,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,109,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,117,0,0,0,116,0,0,0,111,0,0,0,109,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,99,0,0,0,97,0,0,0,108,0,0,0,108,0,0,0,121,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,109,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,32,0,0,0,84,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,115,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,99,0,0,0,108,0,0,0,117,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,108,0,0,0,105,0,0,0,100,0,0,0,105,0,0,0,116,0,0,0,121,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,40,0,0,0,114,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,105,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,101,0,0,0,118,0,0,0,97,0,0,0,108,0,0,0,39,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,41,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,105,0,0,0,114,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,102,0,0,0,115,0,0,0,39,0,0,0,41,0,0,0,44,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,105,0,0,0,114,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,39,0,0,0,41,0,0,0,44,0,0,0,32,0,0,0,118,0,0,0,109,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,105,0,0,0,114,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,118,0,0,0,109,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,115,0,0,0,98,0,0,0,99,0,0,0,115,0,0,0,59,0,0,0,10,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,111,0,0,0,117,0,0,0,114,0,0,0,99,0,0,0,101,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,108,0,0,0,111,0,0,0,97,0,0,0,100,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,105,0,0,0,114,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,108,0,0,0,111,0,0,0,97,0,0,0,100,0,0,0,32,0,0,0,115,0,0,0,98,0,0,0,99,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,115,0,0,0,98,0,0,0,99,0,0,0,115,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,105,0,0,0,114,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,115,0,0,0,98,0,0,0,99,0,0,0,115,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,108,0,0,0,111,0,0,0,97,0,0,0,100,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,99,0,0,0,101,0,0,0,108,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,99,0,0,0,101,0,0,0,108,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,105,0,0,0,114,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,99,0,0,0,112,0,0,0,101,0,0,0,120,0,0,0,99,0,0,0,101,0,0,0,108,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,32,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,100,0,0,0,105,0,0,0,114,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,39,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,39,0,0,0,41,0,0,0,46,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,120,0,0,0,41,0,0,0,123,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,120,0,0,0,46,0,0,0,115,0,0,0,117,0,0,0,98,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,40,0,0,0,45,0,0,0,51,0,0,0,41,0,0,0,61,0,0,0,61,0,0,0,34,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,34,0,0,0,59,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,69,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,120,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,109,0,0,0,46,0,0,0,114,0,0,0,117,0,0,0,110,0,0,0,73,0,0,0,110,0,0,0,84,0,0,0,104,0,0,0,105,0,0,0,115,0,0,0,67,0,0,0,111,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,120,0,0,0,116,0,0,0,40,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,70,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,120,0,0,0,41,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,82,0,0,0,69,0,0,0,65,0,0,0,68,0,0,0,77,0,0,0,69,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,105,0,0,0,102,0,0,0,121,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,110,0,0,0,105,0,0,0,112,0,0,0,112,0,0,0,101,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,82,0,0,0,69,0,0,0,65,0,0,0,68,0,0,0,77,0,0,0,69,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,10,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,82,0,0,0,69,0,0,0,65,0,0,0,68,0,0,0,77,0,0,0,69,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,109,0,0,0,101,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,95,0,0,0,99,0,0,0,112,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,95,0,0,0,50,0,0,0,53,0,0,0,53,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,91,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,93,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,91,0,0,0,50,0,0,0,53,0,0,0,53,0,0,0,93,0,0,0,59,0,0,0,32,0,0,0,47,0,0,0,47,0,0,0,32,0,0,0,199,2,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,95,0,0,0,99,0,0,0,112,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,95,0,0,0,50,0,0,0,53,0,0,0,53,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,199,2,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,95,0,0,0,55,0,0,0,49,0,0,0,49,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,91,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,93,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,91,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,55,0,0,0,49,0,0,0,49,0,0,0,41,0,0,0,93,0,0,0,59,0,0,0,32,0,0,0,47,0,0,0,47,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,53,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,95,0,0,0,55,0,0,0,49,0,0,0,49,0,0,0,44,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,53,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,98,0,0,0,49,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,48,0,0,0,120,0,0,0,98,0,0,0,98,0,0,0,44,0,0,0,48,0,0,0,120,0,0,0,101,0,0,0,51,0,0,0,44,0,0,0,48,0,0,0,120,0,0,0,100,0,0,0,55,0,0,0,44,0,0,0,48,0,0,0,120,0,0,0,100,0,0,0,99,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,71,108,0,0,59,96,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,32,0,0,0,98,0,0,0,49,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,98,0,0,0,117,0,0,0,102,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,32,0,0,0,32,0,0,0,71,108,0,0,59,96,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,71,108,0,0,59,96,0,0,44,0,0,0,34,0,0,0,71,108,0,0,59,96,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,98,0,0,0,117,0,0,0,102,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,44,0,0,0,32,0,0,0,52,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,52,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,98,0,0,0,49,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,98,0,0,0,117,0,0,0,102,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,98,0,0,0,50,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,48,0,0,0,120,0,0,0,102,0,0,0,48,0,0,0,44,0,0,0,48,0,0,0,120,0,0,0,57,0,0,0,102,0,0,0,44,0,0,0,48,0,0,0,120,0,0,0,56,0,0,0,100,0,0,0,44,0,0,0,48,0,0,0,120,0,0,0,97,0,0,0,51,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,115,0,0,0,117,0,0,0,115,0,0,0,104,0,0,0,105,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,54,0,0,0,53,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,44,0,0,0,32,0,0,0,98,0,0,0,50,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,115,0,0,0,98,0,0,0,117,0,0,0,102,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,54,0,0,0,53,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,44,0,0,0,32,0,0,0,115,0,0,0,117,0,0,0,115,0,0,0,104,0,0,0,105,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,115,0,0,0,117,0,0,0,115,0,0,0,104,0,0,0,105,0,0,0,44,0,0,0,34,0,0,0,99,243,1,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,115,0,0,0,98,0,0,0,117,0,0,0,102,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,44,0,0,0,32,0,0,0,52,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,52,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,98,0,0,0,50,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,115,0,0,0,98,0,0,0,117,0,0,0,102,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,98,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,114,0,0,0,114,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,109,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,109,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,105,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,121,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,117,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,115,0,0,0,101,0,0,0,117,0,0,0,100,0,0,0,111,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,101,0,0,0,115,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,10,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,105,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,121,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,105,0,0,0,114,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,85,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,105,0,0,0,116,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,120,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,105,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,108,0,0,0,121,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,32,0,0,0,67,0,0,0,80,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,120,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,91,0,0,0,120,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,68,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,44,0,0,0,32,0,0,0,69,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,105,0,0,0,116,0,0,0,41,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,101,0,0,0,108,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,79,0,0,0,98,0,0,0,106,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,46,0,0,0,107,0,0,0,101,0,0,0,121,0,0,0,115,0,0,0,40,0,0,0,68,0,0,0,41,0,0,0,46,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,69,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,100,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,69,0,0,0,91,0,0,0,68,0,0,0,91,0,0,0,100,0,0,0,93,0,0,0,93,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,100,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,69,0,0,0,91,0,0,0,68,0,0,0,91,0,0,0,100,0,0,0,93,0,0,0,93,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,34,0,0,0,41,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,68,0,0,0,91,0,0,0,100,0,0,0,93,0,0,0,46,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,65,0,0,0,116,0,0,0,40,0,0,0,48,0,0,0,41,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,70,0,0,0,68,0,0,0,41,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,68,0,0,0,91,0,0,0,69,0,0,0,91,0,0,0,68,0,0,0,91,0,0,0,100,0,0,0,93,0,0,0,93,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,68,0,0,0,91,0,0,0,100,0,0,0,93,0,0,0,41,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,32,0,0,0,110,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,69,0,0,0,114,0,0,0,114,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,120,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,101,0,0,0,46,0,0,0,100,0,0,0,91,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,100,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,91,0,0,0,68,0,0,0,91,0,0,0,100,0,0,0,93,0,0,0,93,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,59,0,0,0,32,0,0,0,100,0,0,0,91,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,100,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,93,0,0,0,61,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,68,0,0,0,91,0,0,0,100,0,0,0,93,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,59,0,0,0,32,0,0,0,100,0,0,0,46,0,0,0,101,0,0,0,46,0,0,0,100,0,0,0,91,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,100,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,68,0,0,0,91,0,0,0,69,0,0,0,91,0,0,0,68,0,0,0,91,0,0,0,100,0,0,0,93,0,0,0,93,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,79,0,0,0,98,0,0,0,106,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,46,0,0,0,107,0,0,0,101,0,0,0,121,0,0,0,115,0,0,0,40,0,0,0,69,0,0,0,41,0,0,0,46,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,69,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,101,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,68,0,0,0,91,0,0,0,69,0,0,0,91,0,0,0,101,0,0,0,93,0,0,0,93,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,101,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,32,0,0,0,110,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,69,0,0,0,114,0,0,0,114,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,120,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,100,0,0,0,46,0,0,0,101,0,0,0,91,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,101,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,68,0,0,0,91,0,0,0,69,0,0,0,91,0,0,0,101,0,0,0,93,0,0,0,93,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,59,0,0,0,32,0,0,0,101,0,0,0,91,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,101,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,93,0,0,0,61,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,91,0,0,0,101,0,0,0,93,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,59,0,0,0,32,0,0,0,101,0,0,0,46,0,0,0,100,0,0,0,46,0,0,0,101,0,0,0,91,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,101,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,91,0,0,0,68,0,0,0,91,0,0,0,69,0,0,0,91,0,0,0,101,0,0,0,93,0,0,0,93,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,114,0,0,0,112,0,0,0,117,0,0,0,115,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,34,0,0,0,102,0,0,0,111,0,0,0,111,0,0,0,98,0,0,0,97,0,0,0,114,0,0,0,34,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,114,0,0,0,112,0,0,0,117,0,0,0,115,0,0,0,46,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,69,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,119,0,0,0,41,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,85,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,120,0,0,0,44,0,0,0,85,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,120,0,0,0,44,0,0,0,119,0,0,0,41,0,0,0,41,0,0,0,44,0,0,0,119,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,100,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,79,0,0,0,98,0,0,0,106,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,46,0,0,0,107,0,0,0,101,0,0,0,121,0,0,0,115,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,41,0,0,0,46,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,119,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,119,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,43,0,0,0,119,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,46,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,69,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,40,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,116,0,0,0,114,0,0,0,117,0,0,0,101,0,0,0,41,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,100,0,0,0,105,0,0,0,114,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,79,0,0,0,98,0,0,0,106,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,46,0,0,0,107,0,0,0,101,0,0,0,121,0,0,0,115,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,41,0,0,0,46,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,119,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,119,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,43,0,0,0,119,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,46,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,69,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,40,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,97,0,0,0,108,0,0,0,115,0,0,0,101,0,0,0,41,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,110,0,0,0,101,0,0,0,120,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,108,0,0,0,111,0,0,0,111,0,0,0,107,0,0,0,32,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,112,0,0,0,111,0,0,0,115,0,0,0,115,0,0,0,105,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,114,0,0,0,121,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,100,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,114,0,0,0,121,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,100,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,102,0,0,0,97,0,0,0,105,0,0,0,108,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,108,0,0,0,111,0,0,0,97,0,0,0,100,0,0,0,32,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,97,0,0,0,118,0,0,0,97,0,0,0,105,0,0,0,108,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,115,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,98,0,0,0,111,0,0,0,120,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,123,0,0,0,125,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,116,0,0,0,120,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,118,0,0,0,109,0,0,0,46,0,0,0,99,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,67,0,0,0,111,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,120,0,0,0,116,0,0,0,40,0,0,0,115,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,98,0,0,0,111,0,0,0,120,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,109,0,0,0,46,0,0,0,114,0,0,0,117,0,0,0,110,0,0,0,73,0,0,0,110,0,0,0,67,0,0,0,111,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,120,0,0,0,116,0,0,0,40,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,70,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,39,0,0,0,99,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,39,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,39,0,0,0,41,0,0,0,44,0,0,0,99,0,0,0,116,0,0,0,120,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,108,0,0,0,111,0,0,0,97,0,0,0,100,0,0,0,32,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,118,0,0,0,97,0,0,0,105,0,0,0,108,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,115,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,98,0,0,0,111,0,0,0,120,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,123,0,0,0,125,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,116,0,0,0,120,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,118,0,0,0,109,0,0,0,46,0,0,0,99,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,67,0,0,0,111,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,120,0,0,0,116,0,0,0,40,0,0,0,115,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,98,0,0,0,111,0,0,0,120,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,109,0,0,0,46,0,0,0,114,0,0,0,117,0,0,0,110,0,0,0,73,0,0,0,110,0,0,0,67,0,0,0,111,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,120,0,0,0,116,0,0,0,40,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,70,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,39,0,0,0,99,0,0,0,112,0,0,0,101,0,0,0,120,0,0,0,99,0,0,0,101,0,0,0,108,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,39,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,39,0,0,0,41,0,0,0,44,0,0,0,99,0,0,0,116,0,0,0,120,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,109,0,0,0,46,0,0,0,114,0,0,0,117,0,0,0,110,0,0,0,73,0,0,0,110,0,0,0,67,0,0,0,111,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,120,0,0,0,116,0,0,0,40,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,70,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,39,0,0,0,99,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,39,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,39,0,0,0,41,0,0,0,44,0,0,0,99,0,0,0,116,0,0,0,120,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,32,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,32,0,0,0,105,0,0,0,44,0,0,0,32,0,0,0,101,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,44,0,0,0,101,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,46,0,0,0,115,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,34,0,0,0,34,0,0,0,41,0,0,0,44,0,0,0,101,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,101,0,0,0,112,0,0,0,69,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,44,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,39,0,0,0,41,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,98,0,0,0,117,0,0,0,102,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,110,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,105,0,0,0,41,0,0,0,44,0,0,0,101,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,101,0,0,0,112,0,0,0,69,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,44,0,0,0,98,0,0,0,117,0,0,0,102,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,44,0,0,0,39,0,0,0,98,0,0,0,117,0,0,0,102,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,44,0,0,0,39,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,44,0,0,0,39,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,44,0,0,0,39,0,0,0,98,0,0,0,117,0,0,0,102,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,44,0,0,0,39,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,44,0,0,0,39,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,67,0,0,0,80,0,0,0,32,0,0,0,32,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,50,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,115,0,0,0,98,0,0,0,99,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,101,0,0,0,110,0,0,0,40,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,50,0,0,0,44,0,0,0,34,0,0,0,102,0,0,0,111,0,0,0,111,0,0,0,98,0,0,0,97,0,0,0,114,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,67,0,0,0,80,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,55,0,0,0,48,0,0,0,56,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,115,0,0,0,98,0,0,0,99,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,101,0,0,0,110,0,0,0,40,0,0,0,55,0,0,0,48,0,0,0,56,0,0,0,44,0,0,0,34,0,0,0,42,6,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,43,6,0,0,32,0,0,0,115,0,0,0,109,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,121,0,0,0,32,0,0,0,102,0,0,0,97,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,67,0,0,0,80,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,100,0,0,0,98,0,0,0,99,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,101,0,0,0,110,0,0,0,40,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,217,143,0,0,47,102,0,0,45,78,0,0,135,101,0,0,87,91,0,0,38,123,0,0,75,109,0,0,213,139,0,0,34,0,0,0,41,0,0,0,59,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,32,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,32,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,115,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,39,0,0,0,32,0,0,0,38,0,0,0,38,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,115,0,0,0,116,0,0,0,97,0,0,0,110,0,0,0,99,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,41,0,0,0,32,0,0,0,115,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,46,0,0,0,109,0,0,0,97,0,0,0,112,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,108,0,0,0,108,0,0,0,40,0,0,0,105,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,115,0,0,0,41,0,0,0,123,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,115,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,101,0,0,0,108,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,61,0,0,0,40,0,0,0,105,0,0,0,46,0,0,0,109,0,0,0,97,0,0,0,112,0,0,0,41,0,0,0,32,0,0,0,63,0,0,0,32,0,0,0,105,0,0,0,46,0,0,0,109,0,0,0,97,0,0,0,112,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,115,0,0,0,41,0,0,0,123,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,115,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,105,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,115,0,0,0,46,0,0,0,106,0,0,0,111,0,0,0,105,0,0,0,110,0,0,0,63,0,0,0,115,0,0,0,46,0,0,0,106,0,0,0,111,0,0,0,105,0,0,0,110,0,0,0,40,0,0,0,34,0,0,0,34,0,0,0,41,0,0,0,58,0,0,0,115,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,101,0,0,0,112,0,0,0,69,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,44,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,39,0,0,0,41,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,98,0,0,0,117,0,0,0,102,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,110,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,105,0,0,0,41,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,101,0,0,0,112,0,0,0,69,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,44,0,0,0,98,0,0,0,117,0,0,0,102,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,105,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,67,0,0,0,80,0,0,0,32,0,0,0,32,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,50,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,115,0,0,0,98,0,0,0,99,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,50,0,0,0,44,0,0,0,91,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,54,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,102,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,102,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,50,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,49,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,55,0,0,0,50,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,47,0,0,0,42,0,0,0,32,0,0,0,34,0,0,0,102,0,0,0,111,0,0,0,111,0,0,0,98,0,0,0,97,0,0,0,114,0,0,0,34,0,0,0,32,0,0,0,42,0,0,0,47,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,39,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,67,0,0,0,80,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,55,0,0,0,48,0,0,0,56,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,115,0,0,0,98,0,0,0,99,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,55,0,0,0,48,0,0,0,56,0,0,0,44,0,0,0,32,0,0,0,110,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,91,0,0,0,48,0,0,0,120,0,0,0,99,0,0,0,97,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,50,0,0,0,48,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,49,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,52,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,50,0,0,0,48,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,99,0,0,0,98,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,50,0,0,0,48,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,55,0,0,0,51,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,100,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,57,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,99,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,53,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,55,0,0,0,57,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,50,0,0,0,48,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,54,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,49,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,51,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,54,0,0,0,53,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,55,0,0,0,51,0,0,0,93,0,0,0,41,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,47,0,0,0,42,0,0,0,32,0,0,0,40,0,0,0,34,0,0,0,42,6,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,43,6,0,0,32,0,0,0,115,0,0,0,109,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,121,0,0,0,32,0,0,0,102,0,0,0,97,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,34,0,0,0,41,0,0,0,32,0,0,0,42,0,0,0,47,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,67,0,0,0,80,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,100,0,0,0,98,0,0,0,99,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,32,0,0,0,91,0,0,0,48,0,0,0,120,0,0,0,100,0,0,0,53,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,101,0,0,0,50,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,99,0,0,0,97,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,99,0,0,0,55,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,100,0,0,0,54,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,100,0,0,0,48,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,99,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,99,0,0,0,52,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,100,0,0,0,55,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,100,0,0,0,54,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,98,0,0,0,55,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,102,0,0,0,98,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,98,0,0,0,50,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,101,0,0,0,50,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,99,0,0,0,97,0,0,0,44,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,100,0,0,0,52,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,47,0,0,0,42,0,0,0,32,0,0,0,34,0,0,0,217,143,0,0,47,102,0,0,45,78,0,0,135,101,0,0,87,91,0,0,38,123,0,0,75,109,0,0,213,139,0,0,34,0,0,0,32,0,0,0,42,0,0,0,47,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,96,0,0,0,32,0,0,0,104,0,0,0,101,0,0,0,108,0,0,0,112,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,112,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,39,0,0,0,115,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,32,0,0,0,102,0,0,0,97,0,0,0,99,0,0,0,105,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,10,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,40,0,0,0,102,0,0,0,44,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,44,0,0,0,115,0,0,0,107,0,0,0,105,0,0,0,112,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,100,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,70,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,102,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,120,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,70,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,102,0,0,0,44,0,0,0,32,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,120,0,0,0,46,0,0,0,115,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,34,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,121,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,32,0,0,0,100,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,120,0,0,0,44,0,0,0,121,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,122,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,32,0,0,0,120,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,122,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,100,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,41,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,32,0,0,0,110,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,69,0,0,0,114,0,0,0,114,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,102,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,79,0,0,0,78,0,0,0,46,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,105,0,0,0,102,0,0,0,121,0,0,0,40,0,0,0,122,0,0,0,41,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,79,0,0,0,78,0,0,0,46,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,105,0,0,0,102,0,0,0,121,0,0,0,40,0,0,0,100,0,0,0,41,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,122,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,100,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,100,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,122,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,41,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,32,0,0,0,110,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,69,0,0,0,114,0,0,0,114,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,34,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,33,0,0,0,61,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,122,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,115,0,0,0,107,0,0,0,105,0,0,0,112,0,0,0,41,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,122,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,44,0,0,0,32,0,0,0,97,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,122,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,100,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,41,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,32,0,0,0,110,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,69,0,0,0,114,0,0,0,114,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,102,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,79,0,0,0,78,0,0,0,46,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,105,0,0,0,102,0,0,0,121,0,0,0,40,0,0,0,122,0,0,0,41,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,79,0,0,0,78,0,0,0,46,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,105,0,0,0,102,0,0,0,121,0,0,0,40,0,0,0,100,0,0,0,41,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,122,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,100,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,100,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,122,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,41,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,32,0,0,0,110,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,69,0,0,0,114,0,0,0,114,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,34,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,33,0,0,0,61,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,122,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,115,0,0,0,107,0,0,0,105,0,0,0,112,0,0,0,41,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,107,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,96,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,105,0,0,0,102,0,0,0,121,0,0,0,32,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,99,0,0,0,116,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,32,0,0,0,115,0,0,0,111,0,0,0,117,0,0,0,114,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,10,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,110,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,110,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,118,0,0,0,101,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,91,0,0,0,54,0,0,0,53,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,44,0,0,0,32,0,0,0,39,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,39,0,0,0,44,0,0,0,49,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,91,0,0,0,49,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,44,0,0,0,32,0,0,0,39,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,49,0,0,0,54,0,0,0,108,0,0,0,101,0,0,0,39,0,0,0,44,0,0,0,49,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,91,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,50,0,0,0,55,0,0,0,44,0,0,0,32,0,0,0,39,0,0,0,97,0,0,0,115,0,0,0,99,0,0,0,105,0,0,0,105,0,0,0,39,0,0,0,44,0,0,0,48,0,0,0,93,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,39,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,46,0,0,0,109,0,0,0,100,0,0,0,39,0,0,0,44,0,0,0,39,0,0,0,82,0,0,0,69,0,0,0,65,0,0,0,68,0,0,0,77,0,0,0,69,0,0,0,46,0,0,0,109,0,0,0,100,0,0,0,39,0,0,0,44,0,0,0,39,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,39,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,99,0,0,0,105,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,39,0,0,0,99,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,39,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,69,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,119,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,40,0,0,0,119,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,105,0,0,0,114,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,99,0,0,0,105,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,69,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,102,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,102,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,109,0,0,0,105,0,0,0,115,0,0,0,99,0,0,0,47,0,0,0,39,0,0,0,43,0,0,0,102,0,0,0,43,0,0,0,39,0,0,0,46,0,0,0,39,0,0,0,43,0,0,0,119,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,44,0,0,0,119,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,44,0,0,0,119,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,33,0,0,0,119,0,0,0,91,0,0,0,50,0,0,0,93,0,0,0,41,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,69,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,102,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,102,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,109,0,0,0,105,0,0,0,115,0,0,0,99,0,0,0,47,0,0,0,39,0,0,0,43,0,0,0,102,0,0,0,43,0,0,0,39,0,0,0,46,0,0,0,39,0,0,0,43,0,0,0,119,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,44,0,0,0,119,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,44,0,0,0,119,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,119,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,39,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,32,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,100,0,0,0,105,0,0,0,114,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,39,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,39,0,0,0,41,0,0,0,46,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,120,0,0,0,41,0,0,0,123,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,120,0,0,0,46,0,0,0,115,0,0,0,117,0,0,0,98,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,40,0,0,0,45,0,0,0,51,0,0,0,41,0,0,0,61,0,0,0,61,0,0,0,34,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,34,0,0,0,59,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,69,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,102,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,102,0,0,0,44,0,0,0,119,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,44,0,0,0,119,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,44,0,0,0,116,0,0,0,114,0,0,0,117,0,0,0,101,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,42,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,99,0,0,0,105,0,0,0,105,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,116,0,0,0,116,0,0,0,101,0,0,0,109,0,0,0,112,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,32,0,0,0,111,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,103,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,109,0,0,0,97,0,0,0,116,0,0,0,115,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,109,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,109,0,0,0,97,0,0,0,103,0,0,0,105,0,0,0,99,0,0,0,59,0,0,0,10,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,99,0,0,0,109,0,0,0,112,0,0,0,40,0,0,0,120,0,0,0,44,0,0,0,122,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,120,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,44,0,0,0,32,0,0,0,122,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,122,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,105,0,0,0,43,0,0,0,34,0,0,0,47,0,0,0,34,0,0,0,43,0,0,0,120,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,43,0,0,0,34,0,0,0,34,0,0,0,43,0,0,0,120,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,105,0,0,0,43,0,0,0,34,0,0,0,47,0,0,0,34,0,0,0,43,0,0,0,122,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,43,0,0,0,34,0,0,0,34,0,0,0,43,0,0,0,122,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,10,0,0,0,79,0,0,0,98,0,0,0,106,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,46,0,0,0,107,0,0,0,101,0,0,0,121,0,0,0,115,0,0,0,40,0,0,0,109,0,0,0,41,0,0,0,46,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,69,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,116,0,0,0,41,0,0,0,123,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,49,0,0,0,54,0,0,0,57,0,0,0,54,0,0,0,57,0,0,0,41,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,40,0,0,0,109,0,0,0,91,0,0,0,116,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,34,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,46,0,0,0,109,0,0,0,100,0,0,0,46,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,109,0,0,0,91,0,0,0,116,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,120,0,0,0,105,0,0,0,115,0,0,0,116,0,0,0,115,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,109,0,0,0,105,0,0,0,115,0,0,0,99,0,0,0,47,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,46,0,0,0,109,0,0,0,100,0,0,0,46,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,109,0,0,0,91,0,0,0,116,0,0,0,93,0,0,0,41,0,0,0,32,0,0,0,63,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,98,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,70,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,109,0,0,0,105,0,0,0,115,0,0,0,99,0,0,0,47,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,46,0,0,0,109,0,0,0,100,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,109,0,0,0,91,0,0,0,116,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,97,0,0,0,115,0,0,0,99,0,0,0,105,0,0,0,105,0,0,0,34,0,0,0,41,0,0,0,32,0,0,0,98,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,98,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,112,0,0,0,108,0,0,0,97,0,0,0,99,0,0,0,101,0,0,0,40,0,0,0,47,0,0,0,91,0,0,0,92,0,0,0,117,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,48,0,0,0,45,0,0,0,92,0,0,0,117,0,0,0,102,0,0,0,102,0,0,0,102,0,0,0,102,0,0,0,93,0,0,0,42,0,0,0,47,0,0,0,103,0,0,0,44,0,0,0,34,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,120,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,70,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,109,0,0,0,105,0,0,0,115,0,0,0,99,0,0,0,47,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,46,0,0,0,109,0,0,0,100,0,0,0,46,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,109,0,0,0,91,0,0,0,116,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,121,0,0,0,44,0,0,0,32,0,0,0,122,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,121,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,116,0,0,0,44,0,0,0,32,0,0,0,120,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,121,0,0,0,44,0,0,0,98,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,122,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,116,0,0,0,44,0,0,0,32,0,0,0,121,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,54,0,0,0,53,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,41,0,0,0,32,0,0,0,99,0,0,0,109,0,0,0,112,0,0,0,40,0,0,0,120,0,0,0,44,0,0,0,122,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,101,0,0,0,108,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,121,0,0,0,44,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,116,0,0,0,44,0,0,0,32,0,0,0,122,0,0,0,41,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,121,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,116,0,0,0,44,0,0,0,32,0,0,0,120,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,121,0,0,0,44,0,0,0,98,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,122,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,116,0,0,0,44,0,0,0,32,0,0,0,121,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,54,0,0,0,53,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,41,0,0,0,32,0,0,0,99,0,0,0,109,0,0,0,112,0,0,0,40,0,0,0,120,0,0,0,44,0,0,0,122,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,101,0,0,0,108,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,121,0,0,0,44,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,116,0,0,0,44,0,0,0,32,0,0,0,122,0,0,0,41,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,110,0,0,0,117,0,0,0,108,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,34,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,32,0,0,0,82,0,0,0,69,0,0,0,65,0,0,0,68,0,0,0,77,0,0,0,69,0,0,0,46,0,0,0,109,0,0,0,100,0,0,0,46,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,109,0,0,0,91,0,0,0,116,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,120,0,0,0,105,0,0,0,115,0,0,0,116,0,0,0,115,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,109,0,0,0,105,0,0,0,115,0,0,0,99,0,0,0,47,0,0,0,82,0,0,0,69,0,0,0,65,0,0,0,68,0,0,0,77,0,0,0,69,0,0,0,46,0,0,0,109,0,0,0,100,0,0,0,46,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,109,0,0,0,91,0,0,0,116,0,0,0,93,0,0,0,41,0,0,0,32,0,0,0,63,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,98,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,70,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,109,0,0,0,105,0,0,0,115,0,0,0,99,0,0,0,47,0,0,0,82,0,0,0,69,0,0,0,65,0,0,0,68,0,0,0,77,0,0,0,69,0,0,0,46,0,0,0,109,0,0,0,100,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,109,0,0,0,91,0,0,0,116,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,97,0,0,0,115,0,0,0,99,0,0,0,105,0,0,0,105,0,0,0,34,0,0,0,41,0,0,0,32,0,0,0,98,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,98,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,112,0,0,0,108,0,0,0,97,0,0,0,99,0,0,0,101,0,0,0,40,0,0,0,47,0,0,0,91,0,0,0,92,0,0,0,117,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,48,0,0,0,45,0,0,0,92,0,0,0,117,0,0,0,102,0,0,0,102,0,0,0,102,0,0,0,102,0,0,0,93,0,0,0,42,0,0,0,47,0,0,0,103,0,0,0,44,0,0,0,34,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,120,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,70,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,39,0,0,0,46,0,0,0,47,0,0,0,109,0,0,0,105,0,0,0,115,0,0,0,99,0,0,0,47,0,0,0,82,0,0,0,69,0,0,0,65,0,0,0,68,0,0,0,77,0,0,0,69,0,0,0,46,0,0,0,109,0,0,0,100,0,0,0,46,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,109,0,0,0,91,0,0,0,116,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,120,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,46,0,0,0,115,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,101,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,108,0,0,0,108,0,0,0,40,0,0,0,120,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,121,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,116,0,0,0,44,0,0,0,32,0,0,0,120,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,121,0,0,0,44,0,0,0,98,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,121,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,116,0,0,0,44,0,0,0,32,0,0,0,120,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,40,0,0,0,121,0,0,0,44,0,0,0,98,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,110,0,0,0,117,0,0,0,108,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,54,0,0,0,57,0,0,0,54,0,0,0,57,0,0,0,96,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,116,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,44,0,0,0,32,0,0,0,115,0,0,0,111,0,0,0,32,0,0,0,111,0,0,0,112,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,32,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,102,0,0,0,97,0,0,0,105,0,0,0,108,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,10,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,102,0,0,0,97,0,0,0,105,0,0,0,108,0,0,0,117,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,102,0,0,0,97,0,0,0,105,0,0,0,108,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,80,0,0,0,32,0,0,0,54,0,0,0,57,0,0,0,54,0,0,0,57,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,123,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,91,0,0,0,54,0,0,0,57,0,0,0,54,0,0,0,57,0,0,0,93,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,123,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,91,0,0,0,54,0,0,0,57,0,0,0,54,0,0,0,57,0,0,0,93,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,102,0,0,0,97,0,0,0,105,0,0,0,108,0,0,0,32,0,0,0,117,0,0,0,115,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,40,0,0,0,33,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,104,0,0,0,97,0,0,0,115,0,0,0,99,0,0,0,112,0,0,0,40,0,0,0,54,0,0,0,57,0,0,0,54,0,0,0,57,0,0,0,41,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,123,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,54,0,0,0,57,0,0,0,54,0,0,0,57,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,102,0,0,0,111,0,0,0,111,0,0,0,98,0,0,0,97,0,0,0,114,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,123,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,54,0,0,0,57,0,0,0,54,0,0,0,57,0,0,0,44,0,0,0,32,0,0,0,91,0,0,0,48,0,0,0,120,0,0,0,50,0,0,0,48,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,102,0,0,0,97,0,0,0,105,0,0,0,108,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,98,0,0,0,108,0,0,0,97,0,0,0,99,0,0,0,107,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,103,0,0,0,105,0,0,0,99,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,104,0,0,0,97,0,0,0,115,0,0,0,99,0,0,0,112,0,0,0,40,0,0,0,49,0,0,0,54,0,0,0,57,0,0,0,54,0,0,0,57,0,0,0,41,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,123,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,49,0,0,0,54,0,0,0,57,0,0,0,54,0,0,0,57,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,102,0,0,0,111,0,0,0,111,0,0,0,98,0,0,0,97,0,0,0,114,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,123,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,49,0,0,0,54,0,0,0,57,0,0,0,54,0,0,0,57,0,0,0,44,0,0,0,32,0,0,0,91,0,0,0,48,0,0,0,120,0,0,0,50,0,0,0,48,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,39,0,0,0,115,0,0,0,104,0,0,0,111,0,0,0,117,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,102,0,0,0,97,0,0,0,105,0,0,0,108,0,0,0,32,0,0,0,119,0,0,0,104,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,118,0,0,0,97,0,0,0,108,0,0,0,105,0,0,0,100,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,46,0,0,0,116,0,0,0,104,0,0,0,114,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,123,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,46,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,50,0,0,0,55,0,0,0,44,0,0,0,32,0,0,0,91,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,48,0,0,0,120,0,0,0,65,0,0,0,65,0,0,0,41,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,32,0,0,0,78,0,0,0,105,0,0,0,116,0,0,0,116,0,0,0,121,0,0,0,32,0,0,0,71,0,0,0,114,0,0,0,105,0,0,0,116,0,0,0,116,0,0,0,121,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,106,0,0,0,115,0,0,0,111,0,0,0,110,0,0,0,62,0,0,0,112,0,0,0,97,0,0,0,99,0,0,0,107,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,111,0,0,0,110,0,0,0,10,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,110,0,0,0,97,0,0,0,109,0,0,0,101,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,49,0,0,0,46,0,0,0,51,0,0,0,46,0,0,0,56,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,97,0,0,0,117,0,0,0,116,0,0,0,104,0,0,0,111,0,0,0,114,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,83,0,0,0,104,0,0,0,101,0,0,0,101,0,0,0,116,0,0,0,74,0,0,0,83,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,112,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,112,0,0,0,117,0,0,0,114,0,0,0,101,0,0,0,45,0,0,0,74,0,0,0,83,0,0,0,32,0,0,0,108,0,0,0,105,0,0,0,98,0,0,0,114,0,0,0,97,0,0,0,114,0,0,0,121,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,104,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,107,0,0,0,101,0,0,0,121,0,0,0,119,0,0,0,111,0,0,0,114,0,0,0,100,0,0,0,115,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,91,0,0,0,32,0,0,0,34,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,34,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,118,0,0,0,34,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,34,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,115,0,0,0,34,0,0,0,32,0,0,0,93,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,98,0,0,0,105,0,0,0,110,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,46,0,0,0,47,0,0,0,98,0,0,0,105,0,0,0,110,0,0,0,47,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,46,0,0,0,110,0,0,0,106,0,0,0,115,0,0,0,34,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,91,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,76,0,0,0,73,0,0,0,67,0,0,0,69,0,0,0,78,0,0,0,83,0,0,0,69,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,82,0,0,0,69,0,0,0,65,0,0,0,68,0,0,0,77,0,0,0,69,0,0,0,46,0,0,0,109,0,0,0,100,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,98,0,0,0,105,0,0,0,110,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,99,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,100,0,0,0,105,0,0,0,115,0,0,0,116,0,0,0,47,0,0,0,99,0,0,0,112,0,0,0,101,0,0,0,120,0,0,0,99,0,0,0,101,0,0,0,108,0,0,0,46,0,0,0,102,0,0,0,117,0,0,0,108,0,0,0,108,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,34,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,93,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,109,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,99,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,118,0,0,0,111,0,0,0,99,0,0,0,34,0,0,0,58,0,0,0,34,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,99,0,0,0,97,0,0,0,116,0,0,0,45,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,109,0,0,0,34,0,0,0,58,0,0,0,34,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,114,0,0,0,34,0,0,0,58,0,0,0,34,0,0,0,34,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,100,0,0,0,101,0,0,0,118,0,0,0,68,0,0,0,101,0,0,0,112,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,109,0,0,0,111,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,34,0,0,0,58,0,0,0,34,0,0,0,34,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,114,0,0,0,101,0,0,0,112,0,0,0,111,0,0,0,115,0,0,0,105,0,0,0,116,0,0,0,111,0,0,0,114,0,0,0,121,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,34,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,34,0,0,0,58,0,0,0,34,0,0,0,103,0,0,0,105,0,0,0,116,0,0,0,34,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,117,0,0,0,114,0,0,0,108,0,0,0,34,0,0,0,58,0,0,0,34,0,0,0,103,0,0,0,105,0,0,0,116,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,103,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,117,0,0,0,98,0,0,0,46,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,47,0,0,0,83,0,0,0,104,0,0,0,101,0,0,0,101,0,0,0,116,0,0,0,74,0,0,0,83,0,0,0,47,0,0,0,106,0,0,0,115,0,0,0,45,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,46,0,0,0,103,0,0,0,105,0,0,0,116,0,0,0,34,0,0,0,125,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,112,0,0,0,116,0,0,0,115,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,112,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,103,0,0,0,105,0,0,0,116,0,0,0,32,0,0,0,115,0,0,0,117,0,0,0,98,0,0,0,109,0,0,0,111,0,0,0,100,0,0,0,117,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,105,0,0,0,116,0,0,0,32,0,0,0,38,0,0,0,38,0,0,0,32,0,0,0,103,0,0,0,105,0,0,0,116,0,0,0,32,0,0,0,115,0,0,0,117,0,0,0,98,0,0,0,109,0,0,0,111,0,0,0,100,0,0,0,117,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,117,0,0,0,112,0,0,0,100,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,98,0,0,0,117,0,0,0,105,0,0,0,108,0,0,0,100,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,32,0,0,0,106,0,0,0,115,0,0,0,34,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,102,0,0,0,105,0,0,0,103,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,98,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,107,0,0,0,101,0,0,0,116,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,112,0,0,0,97,0,0,0,116,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,110,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,91,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,44,0,0,0,99,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,44,0,0,0,99,0,0,0,112,0,0,0,101,0,0,0,120,0,0,0,99,0,0,0,101,0,0,0,108,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,93,0,0,0,34,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,98,0,0,0,117,0,0,0,103,0,0,0,115,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,34,0,0,0,117,0,0,0,114,0,0,0,108,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,115,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,103,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,117,0,0,0,98,0,0,0,46,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,47,0,0,0,83,0,0,0,104,0,0,0,101,0,0,0,101,0,0,0,116,0,0,0,74,0,0,0,83,0,0,0,47,0,0,0,106,0,0,0,115,0,0,0,45,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,47,0,0,0,105,0,0,0,115,0,0,0,115,0,0,0,117,0,0,0,101,0,0,0,115,0,0,0,34,0,0,0,32,0,0,0,125,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,101,0,0,0,110,0,0,0,115,0,0,0,101,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,65,0,0,0,112,0,0,0,97,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,45,0,0,0,50,0,0,0,46,0,0,0,48,0,0,0,34,0,0,0,44,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,34,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,34,0,0,0,110,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,62,0,0,0,61,0,0,0,48,0,0,0,46,0,0,0,56,0,0,0,34,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,125,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,46,0,0,0,118,0,0,0,111,0,0,0,99,0,0,0,114,0,0,0,99,0,0,0,10,0,0,0,123,0,0,0,32,0,0,0,34,0,0,0,112,0,0,0,111,0,0,0,115,0,0,0,116,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,34,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,32,0,0,0,106,0,0,0,115,0,0,0,34,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,46,0,0,0,103,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,103,0,0,0,110,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,10,0,0,0,46,0,0,0,103,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,103,0,0,0,110,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,10,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,47,0,0,0,10,0,0,0,46,0,0,0,118,0,0,0,111,0,0,0,99,0,0,0,114,0,0,0,99,0,0,0,10,0,0,0,110,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,95,0,0,0,109,0,0,0,111,0,0,0,100,0,0,0,117,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,47,0,0,0,10,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,10,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,110,0,0,0,106,0,0,0,115,0,0,0,10,0,0,0,109,0,0,0,105,0,0,0,115,0,0,0,99,0,0,0,47,0,0,0,99,0,0,0,111,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,46,0,0,0,104,0,0,0,116,0,0,0,109,0,0,0,108,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0]
;
README['utf32be'] = 
[0,0,0,35,0,0,0,32,0,0,0,71,0,0,0,101,0,0,0,116,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,108,0,0,0,100,0,0,0,115,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,105,0,0,0,102,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,44,0,0,0,117,0,0,0,114,0,0,0,108,0,0,0,44,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,96,0,0,0,32,0,0,0,40,0,0,0,83,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,61,0,0,0,49,0,0,0,44,0,0,0,32,0,0,0,68,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,61,0,0,0,50,0,0,0,41,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,10,0,0,0,51,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,48,0,0,0,51,0,0,0,55,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,52,0,0,0,51,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,52,0,0,0,51,0,0,0,55,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,53,0,0,0,48,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,53,0,0,0,48,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,55,0,0,0,51,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,55,0,0,0,51,0,0,0,55,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,55,0,0,0,55,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,55,0,0,0,55,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,53,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,53,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,53,0,0,0,50,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,53,0,0,0,50,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,53,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,53,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,53,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,53,0,0,0,55,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,49,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,49,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,50,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,50,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,51,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,51,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,52,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,52,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,54,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,54,0,0,0,57,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,80,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,54,0,0,0,57,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,55,0,0,0,52,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,55,0,0,0,52,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,55,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,56,0,0,0,55,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,57,0,0,0,51,0,0,0,50,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,57,0,0,0,51,0,0,0,50,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,57,0,0,0,52,0,0,0,57,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,57,0,0,0,52,0,0,0,57,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,57,0,0,0,53,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,57,0,0,0,53,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,50,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,48,0,0,0,50,0,0,0,54,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,49,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,49,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,50,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,50,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,51,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,51,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,52,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,52,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,55,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,56,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,87,0,0,0,73,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,87,0,0,0,83,0,0,0,47,0,0,0,67,0,0,0,80,0,0,0,49,0,0,0,50,0,0,0,53,0,0,0,56,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,78,0,0,0,111,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,87,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,117,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,105,0,0,0,109,0,0,0,97,0,0,0,114,0,0,0,121,0,0,0,10,0,0,0,100,0,0,0,105,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,117,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,105,0,0,0,118,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,56,0,0,0,70,0,0,0,70,0,0,0,96,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,119,0,0,0,104,0,0,0,105,0,0,0,99,0,0,0,104,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,65,0,0,0,112,0,0,0,112,0,0,0,108,0,0,0,101,0,0,0,10,0,0,0,108,0,0,0,111,0,0,0,103,0,0,0,111,0,0,0,32,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,99,0,0,0,115,0,0,0,32,0,0,0,98,0,0,0,117,0,0,0,116,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,32,0,0,0,103,0,0,0,97,0,0,0,114,0,0,0,98,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,32,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,111,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,111,0,0,0,112,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,109,0,0,0,115,0,0,0,41,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,116,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,121,0,0,0,32,0,0,0,98,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,105,0,0,0,114,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,10,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,102,0,0,0,97,0,0,0,108,0,0,0,108,0,0,0,32,0,0,0,98,0,0,0,97,0,0,0,99,0,0,0,107,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,98,0,0,0,101,0,0,0,104,0,0,0,97,0,0,0,118,0,0,0,105,0,0,0,111,0,0,0,114,0,0,0,44,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,119,0,0,0,104,0,0,0,105,0,0,0,99,0,0,0,104,0,0,0,32,0,0,0,99,0,0,0,97,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,76,0,0,0,69,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,116,0,0,0,10,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,97,0,0,0,98,0,0,0,115,0,0,0,111,0,0,0,108,0,0,0,117,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,58,0,0,0,47,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,47,0,0,0,82,0,0,0,79,0,0,0,77,0,0,0,65,0,0,0,78,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,47,0,0,0,71,0,0,0,82,0,0,0,69,0,0,0,69,0,0,0,75,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,47,0,0,0,67,0,0,0,89,0,0,0,82,0,0,0,73,0,0,0,76,0,0,0,76,0,0,0,73,0,0,0,67,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,57,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,47,0,0,0,76,0,0,0,65,0,0,0,84,0,0,0,73,0,0,0,78,0,0,0,50,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,55,0,0,0,57,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,47,0,0,0,73,0,0,0,67,0,0,0,69,0,0,0,76,0,0,0,65,0,0,0,78,0,0,0,68,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,49,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,86,0,0,0,69,0,0,0,78,0,0,0,68,0,0,0,79,0,0,0,82,0,0,0,83,0,0,0,47,0,0,0,77,0,0,0,73,0,0,0,67,0,0,0,83,0,0,0,70,0,0,0,84,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,47,0,0,0,84,0,0,0,85,0,0,0,82,0,0,0,75,0,0,0,73,0,0,0,83,0,0,0,72,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,110,0,0,0,117,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,115,0,0,0,99,0,0,0,104,0,0,0,101,0,0,0,109,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,45,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,88,0,0,0,96,0,0,0,32,0,0,0,115,0,0,0,101,0,0,0,114,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,96,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,48,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,88,0,0,0,96,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,49,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,50,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,50,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,51,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,51,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,52,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,52,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,54,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,55,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,55,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,56,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,56,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,57,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,57,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,48,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,49,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,49,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,51,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,51,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,52,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,52,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,53,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,53,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,56,0,0,0,54,0,0,0,48,0,0,0,54,0,0,0,44,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,119,0,0,0,119,0,0,0,119,0,0,0,46,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,111,0,0,0,114,0,0,0,103,0,0,0,47,0,0,0,80,0,0,0,117,0,0,0,98,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,47,0,0,0,77,0,0,0,65,0,0,0,80,0,0,0,80,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,83,0,0,0,47,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,47,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,49,0,0,0,54,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,35,0,0,0,32,0,0,0,71,0,0,0,101,0,0,0,110,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,108,0,0,0,108,0,0,0,111,0,0,0,119,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,118,0,0,0,97,0,0,0,105,0,0,0,108,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,46,0,0,0,78,0,0,0,69,0,0,0,84,0,0,0,32,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,87,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,55,0,0,0,48,0,0,0,56,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,65,0,0,0,114,0,0,0,97,0,0,0,98,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,65,0,0,0,83,0,0,0,77,0,0,0,79,0,0,0,32,0,0,0,55,0,0,0,48,0,0,0,56,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,55,0,0,0,50,0,0,0,48,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,65,0,0,0,114,0,0,0,97,0,0,0,98,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,84,0,0,0,114,0,0,0,97,0,0,0,110,0,0,0,115,0,0,0,112,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,32,0,0,0,65,0,0,0,83,0,0,0,77,0,0,0,79,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,65,0,0,0,114,0,0,0,97,0,0,0,98,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,68,0,0,0,79,0,0,0,83,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,56,0,0,0,53,0,0,0,56,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,79,0,0,0,69,0,0,0,77,0,0,0,32,0,0,0,77,0,0,0,117,0,0,0,108,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,49,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,56,0,0,0,55,0,0,0,48,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,77,0,0,0,117,0,0,0,108,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,47,0,0,0,82,0,0,0,79,0,0,0,69,0,0,0,67,0,0,0,69,0,0,0,32,0,0,0,40,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,50,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,77,0,0,0,117,0,0,0,108,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,50,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,52,0,0,0,55,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,49,0,0,0,47,0,0,0,79,0,0,0,112,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,83,0,0,0,121,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,109,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,48,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,85,0,0,0,83,0,0,0,45,0,0,0,67,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,100,0,0,0,97,0,0,0,32,0,0,0,40,0,0,0,48,0,0,0,51,0,0,0,55,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,85,0,0,0,83,0,0,0,45,0,0,0,67,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,100,0,0,0,97,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,49,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,71,0,0,0,101,0,0,0,114,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,121,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,51,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,71,0,0,0,101,0,0,0,114,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,121,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,50,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,68,0,0,0,101,0,0,0,110,0,0,0,109,0,0,0,97,0,0,0,114,0,0,0,107,0,0,0,45,0,0,0,78,0,0,0,111,0,0,0,114,0,0,0,119,0,0,0,97,0,0,0,121,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,55,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,68,0,0,0,101,0,0,0,110,0,0,0,109,0,0,0,97,0,0,0,114,0,0,0,107,0,0,0,45,0,0,0,78,0,0,0,111,0,0,0,114,0,0,0,119,0,0,0,97,0,0,0,121,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,51,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,70,0,0,0,105,0,0,0,110,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,45,0,0,0,83,0,0,0,119,0,0,0,101,0,0,0,100,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,56,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,70,0,0,0,105,0,0,0,110,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,45,0,0,0,83,0,0,0,119,0,0,0,101,0,0,0,100,0,0,0,101,0,0,0,110,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,52,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,73,0,0,0,116,0,0,0,97,0,0,0,108,0,0,0,121,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,48,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,116,0,0,0,97,0,0,0,108,0,0,0,121,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,53,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,65,0,0,0,109,0,0,0,101,0,0,0,114,0,0,0,105,0,0,0,99,0,0,0,97,0,0,0,45,0,0,0,83,0,0,0,112,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,52,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,83,0,0,0,112,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,54,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,85,0,0,0,110,0,0,0,105,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,75,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,100,0,0,0,111,0,0,0,109,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,85,0,0,0,75,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,55,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,70,0,0,0,114,0,0,0,97,0,0,0,110,0,0,0,99,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,57,0,0,0,55,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,70,0,0,0,114,0,0,0,97,0,0,0,110,0,0,0,99,0,0,0,101,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,56,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,73,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,110,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,40,0,0,0,53,0,0,0,48,0,0,0,48,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,110,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,57,0,0,0,32,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,73,0,0,0,99,0,0,0,101,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,55,0,0,0,49,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,99,0,0,0,101,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,105,0,0,0,99,0,0,0,45,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,51,0,0,0,54,0,0,0,49,0,0,0,32,0,0,0,32,0,0,0,75,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,74,0,0,0,111,0,0,0,104,0,0,0,97,0,0,0,98,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,32,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,32,0,0,0,84,0,0,0,114,0,0,0,97,0,0,0,100,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,66,0,0,0,105,0,0,0,103,0,0,0,53,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,84,0,0,0,114,0,0,0,97,0,0,0,100,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,51,0,0,0,32,0,0,0,75,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,52,0,0,0,32,0,0,0,65,0,0,0,114,0,0,0,97,0,0,0,98,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,53,0,0,0,32,0,0,0,72,0,0,0,101,0,0,0,98,0,0,0,114,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,32,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,71,0,0,0,66,0,0,0,32,0,0,0,50,0,0,0,51,0,0,0,49,0,0,0,50,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,32,0,0,0,82,0,0,0,111,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,55,0,0,0,32,0,0,0,85,0,0,0,107,0,0,0,114,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,49,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,97,0,0,0,105,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,50,0,0,0,32,0,0,0,67,0,0,0,114,0,0,0,111,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,77,0,0,0,97,0,0,0,99,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,32,0,0,0,67,0,0,0,78,0,0,0,83,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,105,0,0,0,119,0,0,0,97,0,0,0,110,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,84,0,0,0,114,0,0,0,97,0,0,0,100,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,40,0,0,0,67,0,0,0,78,0,0,0,83,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,32,0,0,0,84,0,0,0,67,0,0,0,65,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,105,0,0,0,119,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,32,0,0,0,69,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,105,0,0,0,119,0,0,0,97,0,0,0,110,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,84,0,0,0,114,0,0,0,97,0,0,0,100,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,40,0,0,0,69,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,51,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,53,0,0,0,53,0,0,0,53,0,0,0,48,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,105,0,0,0,119,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,52,0,0,0,32,0,0,0,84,0,0,0,101,0,0,0,108,0,0,0,101,0,0,0,84,0,0,0,101,0,0,0,120,0,0,0,116,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,105,0,0,0,119,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,53,0,0,0,32,0,0,0,87,0,0,0,97,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,105,0,0,0,119,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,53,0,0,0,32,0,0,0,73,0,0,0,65,0,0,0,53,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,82,0,0,0,86,0,0,0,32,0,0,0,73,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,110,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,65,0,0,0,108,0,0,0,112,0,0,0,104,0,0,0,97,0,0,0,98,0,0,0,101,0,0,0,116,0,0,0,32,0,0,0,78,0,0,0,111,0,0,0,46,0,0,0,32,0,0,0,53,0,0,0,44,0,0,0,32,0,0,0,55,0,0,0,45,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,87,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,112,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,65,0,0,0,53,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,54,0,0,0,32,0,0,0,73,0,0,0,65,0,0,0,53,0,0,0,32,0,0,0,71,0,0,0,101,0,0,0,114,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,55,0,0,0,45,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,55,0,0,0,32,0,0,0,73,0,0,0,65,0,0,0,53,0,0,0,32,0,0,0,83,0,0,0,119,0,0,0,101,0,0,0,100,0,0,0,105,0,0,0,115,0,0,0,104,0,0,0,32,0,0,0,40,0,0,0,55,0,0,0,45,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,56,0,0,0,32,0,0,0,73,0,0,0,65,0,0,0,53,0,0,0,32,0,0,0,78,0,0,0,111,0,0,0,114,0,0,0,119,0,0,0,101,0,0,0,103,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,55,0,0,0,45,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,54,0,0,0,49,0,0,0,32,0,0,0,84,0,0,0,46,0,0,0,54,0,0,0,49,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,54,0,0,0,57,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,54,0,0,0,57,0,0,0,51,0,0,0,55,0,0,0,32,0,0,0,78,0,0,0,111,0,0,0,110,0,0,0,45,0,0,0,83,0,0,0,112,0,0,0,97,0,0,0,99,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,65,0,0,0,99,0,0,0,99,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,51,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,71,0,0,0,101,0,0,0,114,0,0,0,109,0,0,0,97,0,0,0,110,0,0,0,121,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,55,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,68,0,0,0,101,0,0,0,110,0,0,0,109,0,0,0,97,0,0,0,114,0,0,0,107,0,0,0,45,0,0,0,78,0,0,0,111,0,0,0,114,0,0,0,119,0,0,0,97,0,0,0,121,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,56,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,70,0,0,0,105,0,0,0,110,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,45,0,0,0,83,0,0,0,119,0,0,0,101,0,0,0,100,0,0,0,101,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,48,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,73,0,0,0,116,0,0,0,97,0,0,0,108,0,0,0,121,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,52,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,65,0,0,0,109,0,0,0,101,0,0,0,114,0,0,0,105,0,0,0,99,0,0,0,97,0,0,0,45,0,0,0,83,0,0,0,112,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,85,0,0,0,110,0,0,0,105,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,75,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,100,0,0,0,111,0,0,0,109,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,57,0,0,0,48,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,116,0,0,0,97,0,0,0,107,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,32,0,0,0,69,0,0,0,120,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,100,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,57,0,0,0,55,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,70,0,0,0,114,0,0,0,97,0,0,0,110,0,0,0,99,0,0,0,101,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,52,0,0,0,50,0,0,0,48,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,65,0,0,0,114,0,0,0,97,0,0,0,98,0,0,0,105,0,0,0,99,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,52,0,0,0,50,0,0,0,51,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,71,0,0,0,114,0,0,0,101,0,0,0,101,0,0,0,107,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,52,0,0,0,50,0,0,0,52,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,72,0,0,0,101,0,0,0,98,0,0,0,114,0,0,0,101,0,0,0,119,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,51,0,0,0,51,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,75,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,69,0,0,0,120,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,100,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,51,0,0,0,56,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,97,0,0,0,105,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,54,0,0,0,54,0,0,0,32,0,0,0,82,0,0,0,117,0,0,0,115,0,0,0,115,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,75,0,0,0,79,0,0,0,73,0,0,0,56,0,0,0,45,0,0,0,82,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,121,0,0,0,114,0,0,0,105,0,0,0,108,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,75,0,0,0,79,0,0,0,73,0,0,0,56,0,0,0,45,0,0,0,82,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,55,0,0,0,49,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,73,0,0,0,99,0,0,0,101,0,0,0,108,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,105,0,0,0,99,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,56,0,0,0,48,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,67,0,0,0,121,0,0,0,114,0,0,0,105,0,0,0,108,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,82,0,0,0,117,0,0,0,115,0,0,0,115,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,48,0,0,0,53,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,84,0,0,0,117,0,0,0,114,0,0,0,107,0,0,0,105,0,0,0,115,0,0,0,104,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,50,0,0,0,52,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,76,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,49,0,0,0,47,0,0,0,79,0,0,0,112,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,83,0,0,0,121,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,109,0,0,0,32,0,0,0,40,0,0,0,49,0,0,0,48,0,0,0,52,0,0,0,55,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,32,0,0,0,115,0,0,0,121,0,0,0,109,0,0,0,98,0,0,0,111,0,0,0,108,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,51,0,0,0,50,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,74,0,0,0,73,0,0,0,83,0,0,0,32,0,0,0,48,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,45,0,0,0,49,0,0,0,57,0,0,0,57,0,0,0,48,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,48,0,0,0,50,0,0,0,49,0,0,0,50,0,0,0,45,0,0,0,49,0,0,0,57,0,0,0,57,0,0,0,48,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,71,0,0,0,66,0,0,0,50,0,0,0,51,0,0,0,49,0,0,0,50,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,40,0,0,0,71,0,0,0,66,0,0,0,50,0,0,0,51,0,0,0,49,0,0,0,50,0,0,0,45,0,0,0,56,0,0,0,48,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,52,0,0,0,57,0,0,0,32,0,0,0,75,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,87,0,0,0,97,0,0,0,110,0,0,0,115,0,0,0,117,0,0,0,110,0,0,0,103,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,49,0,0,0,48,0,0,0,50,0,0,0,53,0,0,0,32,0,0,0,73,0,0,0,66,0,0,0,77,0,0,0,32,0,0,0,69,0,0,0,66,0,0,0,67,0,0,0,68,0,0,0,73,0,0,0,67,0,0,0,32,0,0,0,67,0,0,0,121,0,0,0,114,0,0,0,105,0,0,0,108,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,83,0,0,0,101,0,0,0,114,0,0,0,98,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,45,0,0,0,66,0,0,0,117,0,0,0,108,0,0,0,103,0,0,0,97,0,0,0,114,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,49,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,32,0,0,0,69,0,0,0,120,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,100,0,0,0,47,0,0,0,69,0,0,0,120,0,0,0,116,0,0,0,32,0,0,0,65,0,0,0,108,0,0,0,112,0,0,0,104,0,0,0,97,0,0,0,32,0,0,0,76,0,0,0,111,0,0,0,119,0,0,0,101,0,0,0,114,0,0,0,99,0,0,0,97,0,0,0,115,0,0,0,101,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,49,0,0,0,56,0,0,0,54,0,0,0,54,0,0,0,32,0,0,0,85,0,0,0,107,0,0,0,114,0,0,0,97,0,0,0,105,0,0,0,110,0,0,0,105,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,40,0,0,0,75,0,0,0,79,0,0,0,73,0,0,0,56,0,0,0,45,0,0,0,85,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,121,0,0,0,114,0,0,0,105,0,0,0,108,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,32,0,0,0,40,0,0,0,75,0,0,0,79,0,0,0,73,0,0,0,56,0,0,0,45,0,0,0,85,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,50,0,0,0,57,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,32,0,0,0,69,0,0,0,117,0,0,0,114,0,0,0,111,0,0,0,112,0,0,0,97,0,0,0,32,0,0,0,51,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,51,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,56,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,56,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,45,0,0,0,56,0,0,0,32,0,0,0,72,0,0,0,101,0,0,0,98,0,0,0,114,0,0,0,101,0,0,0,119,0,0,0,59,0,0,0,32,0,0,0,72,0,0,0,101,0,0,0,98,0,0,0,114,0,0,0,101,0,0,0,119,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,45,0,0,0,76,0,0,0,111,0,0,0,103,0,0,0,105,0,0,0,99,0,0,0,97,0,0,0,108,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,48,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,32,0,0,0,104,0,0,0,97,0,0,0,108,0,0,0,102,0,0,0,119,0,0,0,105,0,0,0,100,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,116,0,0,0,97,0,0,0,107,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,59,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,74,0,0,0,73,0,0,0,83,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,49,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,104,0,0,0,97,0,0,0,108,0,0,0,102,0,0,0,119,0,0,0,105,0,0,0,100,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,116,0,0,0,97,0,0,0,107,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,59,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,74,0,0,0,73,0,0,0,83,0,0,0,45,0,0,0,65,0,0,0,108,0,0,0,108,0,0,0,111,0,0,0,119,0,0,0,32,0,0,0,49,0,0,0,32,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,50,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,74,0,0,0,73,0,0,0,83,0,0,0,32,0,0,0,88,0,0,0,32,0,0,0,48,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,45,0,0,0,49,0,0,0,57,0,0,0,56,0,0,0,57,0,0,0,59,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,74,0,0,0,73,0,0,0,83,0,0,0,45,0,0,0,65,0,0,0,108,0,0,0,108,0,0,0,111,0,0,0,119,0,0,0,32,0,0,0,49,0,0,0,32,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,32,0,0,0,45,0,0,0,32,0,0,0,83,0,0,0,79,0,0,0,47,0,0,0,83,0,0,0,73,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,53,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,32,0,0,0,75,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,55,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,40,0,0,0,73,0,0,0,83,0,0,0,79,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,49,0,0,0,57,0,0,0,51,0,0,0,50,0,0,0,32,0,0,0,69,0,0,0,85,0,0,0,67,0,0,0,32,0,0,0,74,0,0,0,97,0,0,0,112,0,0,0,97,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,49,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,32,0,0,0,69,0,0,0,85,0,0,0,67,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,40,0,0,0,69,0,0,0,85,0,0,0,67,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,49,0,0,0,57,0,0,0,52,0,0,0,57,0,0,0,32,0,0,0,69,0,0,0,85,0,0,0,67,0,0,0,32,0,0,0,75,0,0,0,111,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,110,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,50,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,32,0,0,0,72,0,0,0,90,0,0,0,45,0,0,0,71,0,0,0,66,0,0,0,50,0,0,0,51,0,0,0,49,0,0,0,50,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,40,0,0,0,72,0,0,0,90,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,52,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,32,0,0,0,87,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,32,0,0,0,88,0,0,0,80,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,108,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,58,0,0,0,32,0,0,0,71,0,0,0,66,0,0,0,49,0,0,0,56,0,0,0,48,0,0,0,51,0,0,0,48,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,40,0,0,0,52,0,0,0,32,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,67,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,83,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,40,0,0,0,71,0,0,0,66,0,0,0,49,0,0,0,56,0,0,0,48,0,0,0,51,0,0,0,48,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,68,0,0,0,101,0,0,0,118,0,0,0,97,0,0,0,110,0,0,0,97,0,0,0,103,0,0,0,97,0,0,0,114,0,0,0,105,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,51,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,66,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,97,0,0,0,108,0,0,0,105,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,52,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,84,0,0,0,97,0,0,0,109,0,0,0,105,0,0,0,108,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,53,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,84,0,0,0,101,0,0,0,108,0,0,0,117,0,0,0,103,0,0,0,117,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,54,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,65,0,0,0,115,0,0,0,115,0,0,0,97,0,0,0,109,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,55,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,79,0,0,0,114,0,0,0,105,0,0,0,121,0,0,0,97,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,110,0,0,0,110,0,0,0,97,0,0,0,100,0,0,0,97,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,57,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,77,0,0,0,97,0,0,0,108,0,0,0,97,0,0,0,121,0,0,0,97,0,0,0,108,0,0,0,97,0,0,0,109,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,71,0,0,0,117,0,0,0,106,0,0,0,97,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,49,0,0,0,49,0,0,0,32,0,0,0,73,0,0,0,83,0,0,0,67,0,0,0,73,0,0,0,73,0,0,0,32,0,0,0,80,0,0,0,117,0,0,0,110,0,0,0,106,0,0,0,97,0,0,0,98,0,0,0,105,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,10,0,0,0,55,0,0,0,48,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,55,0,0,0,50,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,53,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,55,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,52,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,49,0,0,0,52,0,0,0,57,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,51,0,0,0,54,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,54,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,54,0,0,0,57,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,56,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,57,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,50,0,0,0,57,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,52,0,0,0,50,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,52,0,0,0,50,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,52,0,0,0,50,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,51,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,51,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,54,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,55,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,56,0,0,0,56,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,48,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,50,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,51,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,48,0,0,0,57,0,0,0,52,0,0,0,57,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,50,0,0,0,49,0,0,0,48,0,0,0,50,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,49,0,0,0,48,0,0,0,50,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,49,0,0,0,56,0,0,0,54,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,50,0,0,0,57,0,0,0,48,0,0,0,48,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,51,0,0,0,56,0,0,0,53,0,0,0,57,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,48,0,0,0,50,0,0,0,50,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,49,0,0,0,57,0,0,0,51,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,49,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,49,0,0,0,57,0,0,0,52,0,0,0,57,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,50,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,52,0,0,0,57,0,0,0,51,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,50,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,51,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,52,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,54,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,55,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,56,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,48,0,0,0,57,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,49,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,53,0,0,0,55,0,0,0,48,0,0,0,49,0,0,0,49,0,0,0,44,0,0,0,44,0,0,0,50,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,108,0,0,0,108,0,0,0,111,0,0,0,119,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,86,0,0,0,105,0,0,0,115,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,70,0,0,0,111,0,0,0,120,0,0,0,80,0,0,0,114,0,0,0,111,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,54,0,0,0,50,0,0,0,48,0,0,0,32,0,0,0,77,0,0,0,97,0,0,0,122,0,0,0,111,0,0,0,118,0,0,0,105,0,0,0,97,0,0,0,32,0,0,0,40,0,0,0,80,0,0,0,111,0,0,0,108,0,0,0,105,0,0,0,115,0,0,0,104,0,0,0,41,0,0,0,32,0,0,0,77,0,0,0,83,0,0,0,45,0,0,0,68,0,0,0,79,0,0,0,83,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,56,0,0,0,57,0,0,0,53,0,0,0,32,0,0,0,75,0,0,0,97,0,0,0,109,0,0,0,101,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,107,0,0,0,253,0,0,0,32,0,0,0,40,0,0,0,67,0,0,0,122,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,41,0,0,0,32,0,0,0,77,0,0,0,83,0,0,0,45,0,0,0,68,0,0,0,79,0,0,0,83,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,10,0,0,0,54,0,0,0,50,0,0,0,48,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,56,0,0,0,57,0,0,0,53,0,0,0,44,0,0,0,44,0,0,0,49,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,107,0,0,0,110,0,0,0,111,0,0,0,119,0,0,0,110,0,0,0,32,0,0,0,109,0,0,0,105,0,0,0,115,0,0,0,115,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,117,0,0,0,109,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,82,0,0,0,69,0,0,0,65,0,0,0,68,0,0,0,77,0,0,0,69,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,35,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,105,0,0,0,108,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,78,0,0,0,111,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,112,0,0,0,116,0,0,0,32,0,0,0,96,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,96,0,0,0,32,0,0,0,40,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,98,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,108,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,41,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,108,0,0,0,108,0,0,0,32,0,0,0,103,0,0,0,101,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,97,0,0,0,116,0,0,0,97,0,0,0,10,0,0,0,40,0,0,0,112,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,45,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,105,0,0,0,114,0,0,0,115,0,0,0,41,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,118,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,117,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,114,0,0,0,111,0,0,0,112,0,0,0,112,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,116,0,0,0,104,0,0,0,115,0,0,0,10,0,0,0,96,0,0,0,46,0,0,0,47,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,47,0,0,0,60,0,0,0,67,0,0,0,79,0,0,0,68,0,0,0,69,0,0,0,80,0,0,0,65,0,0,0,71,0,0,0,69,0,0,0,62,0,0,0,46,0,0,0,84,0,0,0,66,0,0,0,76,0,0,0,96,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,70,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,97,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,108,0,0,0,97,0,0,0,115,0,0,0,116,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,32,0,0,0,108,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,96,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,46,0,0,0,84,0,0,0,66,0,0,0,76,0,0,0,96,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,54,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,67,0,0,0,54,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,55,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,67,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,56,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,48,0,0,0,65,0,0,0,70,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,57,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,56,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,65,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,57,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,66,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,65,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,67,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,48,0,0,0,66,0,0,0,56,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,68,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,68,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,69,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,66,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,67,0,0,0,55,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,119,0,0,0,104,0,0,0,105,0,0,0,99,0,0,0,104,0,0,0,32,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,54,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,96,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,67,0,0,0,54,0,0,0,41,0,0,0,96,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,118,0,0,0,105,0,0,0,99,0,0,0,101,0,0,0,32,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,97,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,35,0,0,0,32,0,0,0,87,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,45,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,101,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,32,0,0,0,98,0,0,0,117,0,0,0,105,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,101,0,0,0,112,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,111,0,0,0,32,0,0,0,98,0,0,0,117,0,0,0,105,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,111,0,0,0,117,0,0,0,114,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,111,0,0,0,119,0,0,0,115,0,0,0,44,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,117,0,0,0,108,0,0,0,116,0,0,0,32,0,0,0,96,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,47,0,0,0,77,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,69,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,96,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,65,0,0,0,102,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,115,0,0,0,97,0,0,0,118,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,97,0,0,0,114,0,0,0,100,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,96,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,96,0,0,0,44,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,115,0,0,0,105,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,119,0,0,0,107,0,0,0,32,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,112,0,0,0,116,0,0,0,32,0,0,0,40,0,0,0,96,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,96,0,0,0,41,0,0,0,32,0,0,0,116,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,99,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,116,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,10,0,0,0,35,0,0,0,33,0,0,0,47,0,0,0,98,0,0,0,105,0,0,0,110,0,0,0,47,0,0,0,98,0,0,0,97,0,0,0,115,0,0,0,104,0,0,0,10,0,0,0,105,0,0,0,102,0,0,0,32,0,0,0,91,0,0,0,32,0,0,0,33,0,0,0,32,0,0,0,45,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,47,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,32,0,0,0,93,0,0,0,59,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,105,0,0,0,116,0,0,0,59,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,10,0,0,0,60,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,47,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,45,0,0,0,115,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,39,0,0,0,92,0,0,0,116,0,0,0,39,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,97,0,0,0,119,0,0,0,107,0,0,0,32,0,0,0,39,0,0,0,78,0,0,0,70,0,0,0,62,0,0,0,50,0,0,0,32,0,0,0,123,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,41,0,0,0,32,0,0,0,99,0,0,0,108,0,0,0,111,0,0,0,115,0,0,0,101,0,0,0,40,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,61,0,0,0,34,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,47,0,0,0,34,0,0,0,32,0,0,0,36,0,0,0,49,0,0,0,32,0,0,0,34,0,0,0,46,0,0,0,84,0,0,0,66,0,0,0,76,0,0,0,34,0,0,0,125,0,0,0,32,0,0,0,78,0,0,0,70,0,0,0,61,0,0,0,61,0,0,0,50,0,0,0,32,0,0,0,123,0,0,0,112,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,32,0,0,0,62,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,125,0,0,0,39,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,105,0,0,0,108,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,112,0,0,0,116,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,110,0,0,0,106,0,0,0,115,0,0,0,96,0,0,0,32,0,0,0,116,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,103,0,0,0,117,0,0,0,109,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,44,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,114,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,112,0,0,0,111,0,0,0,110,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,10,0,0,0,103,0,0,0,101,0,0,0,110,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,35,0,0,0,32,0,0,0,82,0,0,0,97,0,0,0,119,0,0,0,32,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,110,0,0,0,106,0,0,0,115,0,0,0,10,0,0,0,35,0,0,0,33,0,0,0,47,0,0,0,117,0,0,0,115,0,0,0,114,0,0,0,47,0,0,0,98,0,0,0,105,0,0,0,110,0,0,0,47,0,0,0,101,0,0,0,110,0,0,0,118,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,103,0,0,0,118,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,46,0,0,0,97,0,0,0,114,0,0,0,103,0,0,0,118,0,0,0,46,0,0,0,115,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,101,0,0,0,40,0,0,0,49,0,0,0,41,0,0,0,44,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,113,0,0,0,117,0,0,0,105,0,0,0,114,0,0,0,101,0,0,0,40,0,0,0,39,0,0,0,102,0,0,0,115,0,0,0,39,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,97,0,0,0,114,0,0,0,103,0,0,0,118,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,60,0,0,0,32,0,0,0,50,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,111,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,114,0,0,0,114,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,34,0,0,0,117,0,0,0,115,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,58,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,110,0,0,0,106,0,0,0,115,0,0,0,32,0,0,0,60,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,95,0,0,0,105,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,120,0,0,0,62,0,0,0,32,0,0,0,91,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,105,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,93,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,46,0,0,0,101,0,0,0,120,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,50,0,0,0,50,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,47,0,0,0,42,0,0,0,32,0,0,0,69,0,0,0,73,0,0,0,78,0,0,0,86,0,0,0,65,0,0,0,76,0,0,0,32,0,0,0,42,0,0,0,47,0,0,0,10,0,0,0,125,0,0,0,10,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,103,0,0,0,118,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,106,0,0,0,115,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,103,0,0,0,118,0,0,0,91,0,0,0,50,0,0,0,93,0,0,0,32,0,0,0,124,0,0,0,124,0,0,0,32,0,0,0,34,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,34,0,0,0,59,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,120,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,102,0,0,0,115,0,0,0,46,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,70,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,83,0,0,0,121,0,0,0,110,0,0,0,99,0,0,0,40,0,0,0,34,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,47,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,46,0,0,0,84,0,0,0,66,0,0,0,76,0,0,0,34,0,0,0,44,0,0,0,34,0,0,0,117,0,0,0,116,0,0,0,102,0,0,0,56,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,10,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,121,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,120,0,0,0,46,0,0,0,115,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,34,0,0,0,92,0,0,0,110,0,0,0,34,0,0,0,41,0,0,0,46,0,0,0,109,0,0,0,97,0,0,0,112,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,122,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,119,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,122,0,0,0,46,0,0,0,115,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,34,0,0,0,92,0,0,0,116,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,119,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,60,0,0,0,32,0,0,0,50,0,0,0,41,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,119,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,91,0,0,0,78,0,0,0,117,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,119,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,41,0,0,0,44,0,0,0,32,0,0,0,78,0,0,0,117,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,119,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,41,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,41,0,0,0,46,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,122,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,122,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,62,0,0,0,32,0,0,0,49,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,68,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,83,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,103,0,0,0,101,0,0,0,110,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,103,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,105,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,105,0,0,0,109,0,0,0,117,0,0,0,109,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,10,0,0,0,117,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,100,0,0,0,105,0,0,0,115,0,0,0,116,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,117,0,0,0,105,0,0,0,115,0,0,0,104,0,0,0,32,0,0,0,40,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,83,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,41,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,121,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,121,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,32,0,0,0,62,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,99,0,0,0,112,0,0,0,41,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,121,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,123,0,0,0,125,0,0,0,44,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,40,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,60,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,32,0,0,0,63,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,32,0,0,0,58,0,0,0,32,0,0,0,123,0,0,0,125,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,121,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,91,0,0,0,121,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,121,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,91,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,121,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,49,0,0,0,93,0,0,0,41,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,121,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,48,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,10,0,0,0,10,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,44,0,0,0,32,0,0,0,111,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,44,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,59,0,0,0,10,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,109,0,0,0,97,0,0,0,120,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,60,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,97,0,0,0,99,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,96,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,70,0,0,0,68,0,0,0,96,0,0,0,32,0,0,0,40,0,0,0,82,0,0,0,69,0,0,0,80,0,0,0,76,0,0,0,65,0,0,0,67,0,0,0,69,0,0,0,77,0,0,0,69,0,0,0,78,0,0,0,84,0,0,0,32,0,0,0,67,0,0,0,72,0,0,0,65,0,0,0,82,0,0,0,65,0,0,0,67,0,0,0,84,0,0,0,69,0,0,0,82,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,117,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,112,0,0,0,108,0,0,0,97,0,0,0,99,0,0,0,101,0,0,0,104,0,0,0,111,0,0,0,108,0,0,0,100,0,0,0,101,0,0,0,114,0,0,0,10,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,97,0,0,0,99,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,116,0,0,0,32,0,0,0,115,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,105,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,112,0,0,0,32,0,0,0,40,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,97,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,96,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,48,0,0,0,96,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,116,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,10,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,41,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,70,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,83,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,44,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,100,0,0,0,101,0,0,0,97,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,101,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,114,0,0,0,97,0,0,0,119,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,96,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,101,0,0,0,108,0,0,0,100,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,109,0,0,0,101,0,0,0,114,0,0,0,101,0,0,0,108,0,0,0,121,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,115,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,44,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,96,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,96,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,101,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,34,0,0,0,41,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,70,0,0,0,68,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,79,0,0,0,78,0,0,0,46,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,105,0,0,0,102,0,0,0,121,0,0,0,40,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,46,0,0,0,106,0,0,0,111,0,0,0,105,0,0,0,110,0,0,0,40,0,0,0,34,0,0,0,34,0,0,0,41,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,123,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,100,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,44,0,0,0,32,0,0,0,68,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,101,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,123,0,0,0,125,0,0,0,59,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,61,0,0,0,48,0,0,0,59,0,0,0,105,0,0,0,33,0,0,0,61,0,0,0,100,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,59,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,100,0,0,0,46,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,65,0,0,0,116,0,0,0,40,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,70,0,0,0,68,0,0,0,41,0,0,0,32,0,0,0,101,0,0,0,91,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,105,0,0,0,59,0,0,0,32,0,0,0,68,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,100,0,0,0,46,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,65,0,0,0,116,0,0,0,40,0,0,0,105,0,0,0,41,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,123,0,0,0,34,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,68,0,0,0,32,0,0,0,125,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,39,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,32,0,0,0,101,0,0,0,108,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,68,0,0,0,66,0,0,0,67,0,0,0,83,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,115,0,0,0,105,0,0,0,109,0,0,0,105,0,0,0,108,0,0,0,97,0,0,0,114,0,0,0,44,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,99,0,0,0,101,0,0,0,112,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,112,0,0,0,97,0,0,0,99,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,115,0,0,0,108,0,0,0,105,0,0,0,99,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,45,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,117,0,0,0,110,0,0,0,107,0,0,0,115,0,0,0,32,0,0,0,40,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,115,0,0,0,10,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,111,0,0,0,110,0,0,0,108,0,0,0,121,0,0,0,32,0,0,0,103,0,0,0,101,0,0,0,110,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,111,0,0,0,115,0,0,0,101,0,0,0,32,0,0,0,104,0,0,0,105,0,0,0,103,0,0,0,104,0,0,0,45,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,112,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,41,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,103,0,0,0,121,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,117,0,0,0,99,0,0,0,116,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,97,0,0,0,121,0,0,0,45,0,0,0,111,0,0,0,102,0,0,0,45,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,97,0,0,0,121,0,0,0,115,0,0,0,32,0,0,0,115,0,0,0,111,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,96,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,104,0,0,0,105,0,0,0,103,0,0,0,104,0,0,0,93,0,0,0,91,0,0,0,108,0,0,0,111,0,0,0,119,0,0,0,93,0,0,0,96,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,10,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,97,0,0,0,99,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,115,0,0,0,111,0,0,0,99,0,0,0,105,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,97,0,0,0,121,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,98,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,114,0,0,0,117,0,0,0,110,0,0,0,116,0,0,0,105,0,0,0,109,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,111,0,0,0,32,0,0,0,121,0,0,0,105,0,0,0,101,0,0,0,108,0,0,0,100,0,0,0,10,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,101,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,111,0,0,0,98,0,0,0,106,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,32,0,0,0,40,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,111,0,0,0,98,0,0,0,106,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,32,0,0,0,105,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,32,0,0,0,101,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,41,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,100,0,0,0,100,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,46,0,0,0,104,0,0,0,97,0,0,0,115,0,0,0,79,0,0,0,119,0,0,0,110,0,0,0,80,0,0,0,114,0,0,0,111,0,0,0,112,0,0,0,101,0,0,0,114,0,0,0,116,0,0,0,121,0,0,0,40,0,0,0,105,0,0,0,41,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,56,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,34,0,0,0,41,0,0,0,32,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,56,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,56,0,0,0,93,0,0,0,91,0,0,0,105,0,0,0,32,0,0,0,37,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,40,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,40,0,0,0,41,0,0,0,123,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,100,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,101,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,123,0,0,0,125,0,0,0,44,0,0,0,32,0,0,0,68,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,91,0,0,0,93,0,0,0,44,0,0,0,32,0,0,0,106,0,0,0,59,0,0,0,92,0,0,0,110,0,0,0,39,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,105,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,106,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,106,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,106,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,106,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,34,0,0,0,41,0,0,0,32,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,91,0,0,0,106,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,46,0,0,0,102,0,0,0,114,0,0,0,111,0,0,0,109,0,0,0,67,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,40,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,70,0,0,0,68,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,43,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,68,0,0,0,91,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,79,0,0,0,78,0,0,0,46,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,105,0,0,0,102,0,0,0,121,0,0,0,40,0,0,0,100,0,0,0,100,0,0,0,91,0,0,0,105,0,0,0,93,0,0,0,46,0,0,0,106,0,0,0,111,0,0,0,105,0,0,0,110,0,0,0,40,0,0,0,34,0,0,0,34,0,0,0,41,0,0,0,41,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,46,0,0,0,115,0,0,0,112,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,40,0,0,0,34,0,0,0,34,0,0,0,41,0,0,0,59,0,0,0,92,0,0,0,110,0,0,0,39,0,0,0,59,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,43,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,40,0,0,0,106,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,59,0,0,0,32,0,0,0,106,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,32,0,0,0,68,0,0,0,91,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,93,0,0,0,46,0,0,0,108,0,0,0,101,0,0,0,110,0,0,0,103,0,0,0,116,0,0,0,104,0,0,0,59,0,0,0,32,0,0,0,43,0,0,0,43,0,0,0,106,0,0,0,41,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,68,0,0,0,91,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,93,0,0,0,91,0,0,0,106,0,0,0,93,0,0,0,46,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,67,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,65,0,0,0,116,0,0,0,40,0,0,0,48,0,0,0,41,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,70,0,0,0,68,0,0,0,41,0,0,0,32,0,0,0,123,0,0,0,32,0,0,0,101,0,0,0,91,0,0,0,68,0,0,0,91,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,93,0,0,0,91,0,0,0,106,0,0,0,93,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,40,0,0,0,105,0,0,0,42,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,41,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,106,0,0,0,59,0,0,0,32,0,0,0,100,0,0,0,91,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,40,0,0,0,105,0,0,0,42,0,0,0,50,0,0,0,53,0,0,0,54,0,0,0,41,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,106,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,68,0,0,0,91,0,0,0,39,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,105,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,39,0,0,0,93,0,0,0,91,0,0,0,106,0,0,0,93,0,0,0,59,0,0,0,125,0,0,0,92,0,0,0,110,0,0,0,39,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,125,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,43,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,117,0,0,0,114,0,0,0,110,0,0,0,32,0,0,0,123,0,0,0,34,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,34,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,34,0,0,0,58,0,0,0,32,0,0,0,100,0,0,0,32,0,0,0,125,0,0,0,59,0,0,0,32,0,0,0,125,0,0,0,41,0,0,0,40,0,0,0,41,0,0,0,59,0,0,0,39,0,0,0,59,0,0,0,10,0,0,0,125,0,0,0,10,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,111,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,108,0,0,0,111,0,0,0,103,0,0,0,40,0,0,0,106,0,0,0,115,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,91,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,34,0,0,0,93,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,34,0,0,0,32,0,0,0,43,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,41,0,0,0,59,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,96,0,0,0,32,0,0,0,103,0,0,0,101,0,0,0,110,0,0,0,101,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,117,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,98,0,0,0,121,0,0,0,32,0,0,0,96,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,110,0,0,0,106,0,0,0,115,0,0,0,96,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,114,0,0,0,97,0,0,0,119,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,115,0,0,0,10,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,108,0,0,0,117,0,0,0,109,0,0,0,110,0,0,0,97,0,0,0,114,0,0,0,58,0,0,0,32,0,0,0,96,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,105,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,35,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,109,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,115,0,0,0,96,0,0,0,46,0,0,0,32,0,0,0,32,0,0,0,70,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,97,0,0,0,109,0,0,0,112,0,0,0,108,0,0,0,101,0,0,0,44,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,108,0,0,0,97,0,0,0,115,0,0,0,116,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,32,0,0,0,108,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,10,0,0,0,116,0,0,0,101,0,0,0,120,0,0,0,116,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,82,0,0,0,79,0,0,0,77,0,0,0,65,0,0,0,78,0,0,0,46,0,0,0,84,0,0,0,88,0,0,0,84,0,0,0,32,0,0,0,40,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,67,0,0,0,80,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,41,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,54,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,67,0,0,0,54,0,0,0,9,0,0,0,35,0,0,0,77,0,0,0,79,0,0,0,68,0,0,0,73,0,0,0,70,0,0,0,73,0,0,0,69,0,0,0,82,0,0,0,32,0,0,0,76,0,0,0,69,0,0,0,84,0,0,0,84,0,0,0,69,0,0,0,82,0,0,0,32,0,0,0,67,0,0,0,73,0,0,0,82,0,0,0,67,0,0,0,85,0,0,0,77,0,0,0,70,0,0,0,76,0,0,0,69,0,0,0,88,0,0,0,32,0,0,0,65,0,0,0,67,0,0,0,67,0,0,0,69,0,0,0,78,0,0,0,84,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,55,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,67,0,0,0,9,0,0,0,35,0,0,0,83,0,0,0,77,0,0,0,65,0,0,0,76,0,0,0,76,0,0,0,32,0,0,0,84,0,0,0,73,0,0,0,76,0,0,0,68,0,0,0,69,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,56,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,48,0,0,0,65,0,0,0,70,0,0,0,9,0,0,0,35,0,0,0,77,0,0,0,65,0,0,0,67,0,0,0,82,0,0,0,79,0,0,0,78,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,57,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,56,0,0,0,9,0,0,0,35,0,0,0,66,0,0,0,82,0,0,0,69,0,0,0,86,0,0,0,69,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,65,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,57,0,0,0,9,0,0,0,35,0,0,0,68,0,0,0,79,0,0,0,84,0,0,0,32,0,0,0,65,0,0,0,66,0,0,0,79,0,0,0,86,0,0,0,69,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,66,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,65,0,0,0,9,0,0,0,35,0,0,0,82,0,0,0,73,0,0,0,78,0,0,0,71,0,0,0,32,0,0,0,65,0,0,0,66,0,0,0,79,0,0,0,86,0,0,0,69,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,67,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,48,0,0,0,66,0,0,0,56,0,0,0,9,0,0,0,35,0,0,0,67,0,0,0,69,0,0,0,68,0,0,0,73,0,0,0,76,0,0,0,76,0,0,0,65,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,68,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,68,0,0,0,9,0,0,0,35,0,0,0,68,0,0,0,79,0,0,0,85,0,0,0,66,0,0,0,76,0,0,0,69,0,0,0,32,0,0,0,65,0,0,0,67,0,0,0,85,0,0,0,84,0,0,0,69,0,0,0,32,0,0,0,65,0,0,0,67,0,0,0,67,0,0,0,69,0,0,0,78,0,0,0,84,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,69,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,68,0,0,0,66,0,0,0,9,0,0,0,35,0,0,0,79,0,0,0,71,0,0,0,79,0,0,0,78,0,0,0,69,0,0,0,75,0,0,0,10,0,0,0,48,0,0,0,120,0,0,0,70,0,0,0,70,0,0,0,9,0,0,0,48,0,0,0,120,0,0,0,48,0,0,0,50,0,0,0,67,0,0,0,55,0,0,0,9,0,0,0,35,0,0,0,67,0,0,0,65,0,0,0,82,0,0,0,79,0,0,0,78,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,73,0,0,0,110,0,0,0,32,0,0,0,112,0,0,0,114,0,0,0,111,0,0,0,99,0,0,0,101,0,0,0,115,0,0,0,115,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,97,0,0,0,116,0,0,0,97,0,0,0,44,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,109,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,40,0,0,0,97,0,0,0,102,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,35,0,0,0,96,0,0,0,41,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,112,0,0,0,112,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,10,0,0,0,101,0,0,0,108,0,0,0,101,0,0,0,109,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,40,0,0,0,108,0,0,0,105,0,0,0,107,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,48,0,0,0,120,0,0,0,55,0,0,0,70,0,0,0,96,0,0,0,32,0,0,0,102,0,0,0,111,0,0,0,114,0,0,0,32,0,0,0,67,0,0,0,80,0,0,0,32,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,48,0,0,0,41,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,109,0,0,0,111,0,0,0,118,0,0,0,101,0,0,0,100,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,62,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,10,0,0,0,35,0,0,0,33,0,0,0,47,0,0,0,98,0,0,0,105,0,0,0,110,0,0,0,47,0,0,0,98,0,0,0,97,0,0,0,115,0,0,0,104,0,0,0,10,0,0,0,73,0,0,0,78,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,61,0,0,0,36,0,0,0,123,0,0,0,49,0,0,0,58,0,0,0,45,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,115,0,0,0,118,0,0,0,125,0,0,0,10,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,61,0,0,0,36,0,0,0,123,0,0,0,50,0,0,0,58,0,0,0,45,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,125,0,0,0,10,0,0,0,74,0,0,0,83,0,0,0,86,0,0,0,65,0,0,0,82,0,0,0,61,0,0,0,36,0,0,0,123,0,0,0,51,0,0,0,58,0,0,0,45,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,125,0,0,0,10,0,0,0,86,0,0,0,69,0,0,0,82,0,0,0,83,0,0,0,73,0,0,0,79,0,0,0,78,0,0,0,61,0,0,0,36,0,0,0,40,0,0,0,99,0,0,0,97,0,0,0,116,0,0,0,32,0,0,0,112,0,0,0,97,0,0,0,99,0,0,0,107,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,103,0,0,0,114,0,0,0,101,0,0,0,112,0,0,0,32,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,116,0,0,0,114,0,0,0,32,0,0,0,45,0,0,0,100,0,0,0,99,0,0,0,32,0,0,0,91,0,0,0,48,0,0,0,45,0,0,0,57,0,0,0,46,0,0,0,93,0,0,0,41,0,0,0,10,0,0,0,10,0,0,0,109,0,0,0,107,0,0,0,100,0,0,0,105,0,0,0,114,0,0,0,32,0,0,0,45,0,0,0,112,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,32,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,10,0,0,0,114,0,0,0,109,0,0,0,32,0,0,0,45,0,0,0,102,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,111,0,0,0,32,0,0,0,34,0,0,0,47,0,0,0,42,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,32,0,0,0,40,0,0,0,67,0,0,0,41,0,0,0,32,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,51,0,0,0,45,0,0,0,50,0,0,0,48,0,0,0,49,0,0,0,52,0,0,0,32,0,0,0,83,0,0,0,104,0,0,0,101,0,0,0,101,0,0,0,116,0,0,0,74,0,0,0,83,0,0,0,32,0,0,0,45,0,0,0,45,0,0,0,32,0,0,0,104,0,0,0,116,0,0,0,116,0,0,0,112,0,0,0,58,0,0,0,47,0,0,0,47,0,0,0,115,0,0,0,104,0,0,0,101,0,0,0,101,0,0,0,116,0,0,0,106,0,0,0,115,0,0,0,46,0,0,0,99,0,0,0,111,0,0,0,109,0,0,0,32,0,0,0,42,0,0,0,47,0,0,0,34,0,0,0,32,0,0,0,62,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,111,0,0,0,32,0,0,0,34,0,0,0,47,0,0,0,42,0,0,0,106,0,0,0,115,0,0,0,104,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,32,0,0,0,45,0,0,0,87,0,0,0,49,0,0,0,48,0,0,0,48,0,0,0,32,0,0,0,42,0,0,0,47,0,0,0,34,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,111,0,0,0,32,0,0,0,34,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,32,0,0,0,36,0,0,0,74,0,0,0,83,0,0,0,86,0,0,0,65,0,0,0,82,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,123,0,0,0,118,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,58,0,0,0,92,0,0,0,34,0,0,0,36,0,0,0,86,0,0,0,69,0,0,0,82,0,0,0,83,0,0,0,73,0,0,0,79,0,0,0,78,0,0,0,92,0,0,0,34,0,0,0,125,0,0,0,59,0,0,0,34,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,105,0,0,0,102,0,0,0,32,0,0,0,91,0,0,0,32,0,0,0,45,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,32,0,0,0,93,0,0,0,59,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,110,0,0,0,32,0,0,0,98,0,0,0,97,0,0,0,115,0,0,0,104,0,0,0,32,0,0,0,100,0,0,0,111,0,0,0,116,0,0,0,110,0,0,0,101,0,0,0,116,0,0,0,46,0,0,0,115,0,0,0,104,0,0,0,59,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,10,0,0,0,97,0,0,0,119,0,0,0,107,0,0,0,32,0,0,0,45,0,0,0,70,0,0,0,44,0,0,0,32,0,0,0,39,0,0,0,123,0,0,0,112,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,32,0,0,0,36,0,0,0,49,0,0,0,44,0,0,0,32,0,0,0,36,0,0,0,50,0,0,0,44,0,0,0,32,0,0,0,36,0,0,0,51,0,0,0,125,0,0,0,39,0,0,0,32,0,0,0,36,0,0,0,73,0,0,0,78,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,119,0,0,0,104,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,97,0,0,0,100,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,117,0,0,0,114,0,0,0,108,0,0,0,32,0,0,0,99,0,0,0,112,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,59,0,0,0,32,0,0,0,100,0,0,0,111,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,111,0,0,0,32,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,36,0,0,0,117,0,0,0,114,0,0,0,108,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,105,0,0,0,102,0,0,0,32,0,0,0,91,0,0,0,32,0,0,0,33,0,0,0,32,0,0,0,45,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,84,0,0,0,66,0,0,0,76,0,0,0,32,0,0,0,93,0,0,0,59,0,0,0,32,0,0,0,116,0,0,0,104,0,0,0,101,0,0,0,110,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,99,0,0,0,117,0,0,0,114,0,0,0,108,0,0,0,32,0,0,0,36,0,0,0,117,0,0,0,114,0,0,0,108,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,39,0,0,0,115,0,0,0,47,0,0,0,35,0,0,0,46,0,0,0,42,0,0,0,47,0,0,0,47,0,0,0,103,0,0,0,39,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,97,0,0,0,119,0,0,0,107,0,0,0,32,0,0,0,39,0,0,0,78,0,0,0,70,0,0,0,61,0,0,0,61,0,0,0,50,0,0,0,39,0,0,0,32,0,0,0,62,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,103,0,0,0,101,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,84,0,0,0,66,0,0,0,76,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,102,0,0,0,105,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,111,0,0,0,32,0,0,0,34,0,0,0,105,0,0,0,102,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,36,0,0,0,74,0,0,0,83,0,0,0,86,0,0,0,65,0,0,0,82,0,0,0,32,0,0,0,61,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,39,0,0,0,41,0,0,0,32,0,0,0,36,0,0,0,74,0,0,0,83,0,0,0,86,0,0,0,65,0,0,0,82,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,123,0,0,0,125,0,0,0,59,0,0,0,34,0,0,0,32,0,0,0,62,0,0,0,32,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,110,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,109,0,0,0,97,0,0,0,107,0,0,0,101,0,0,0,46,0,0,0,110,0,0,0,106,0,0,0,115,0,0,0,32,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,32,0,0,0,36,0,0,0,74,0,0,0,83,0,0,0,86,0,0,0,65,0,0,0,82,0,0,0,32,0,0,0,124,0,0,0,32,0,0,0,116,0,0,0,101,0,0,0,101,0,0,0,32,0,0,0,45,0,0,0,97,0,0,0,32,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,39,0,0,0,115,0,0,0,47,0,0,0,34,0,0,0,92,0,0,0,40,0,0,0,91,0,0,0,48,0,0,0,45,0,0,0,57,0,0,0,93,0,0,0,43,0,0,0,92,0,0,0,41,0,0,0,34,0,0,0,58,0,0,0,47,0,0,0,92,0,0,0,49,0,0,0,58,0,0,0,47,0,0,0,103,0,0,0,39,0,0,0,32,0,0,0,60,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,32,0,0,0,62,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,10,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,114,0,0,0,109,0,0,0,32,0,0,0,45,0,0,0,102,0,0,0,32,0,0,0,98,0,0,0,105,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,36,0,0,0,99,0,0,0,112,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,100,0,0,0,111,0,0,0,110,0,0,0,101,0,0,0,10,0,0,0,101,0,0,0,99,0,0,0,104,0,0,0,111,0,0,0,32,0,0,0,34,0,0,0,105,0,0,0,102,0,0,0,32,0,0,0,40,0,0,0,116,0,0,0,121,0,0,0,112,0,0,0,101,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,109,0,0,0,111,0,0,0,100,0,0,0,117,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,33,0,0,0,61,0,0,0,61,0,0,0,32,0,0,0,39,0,0,0,117,0,0,0,110,0,0,0,100,0,0,0,101,0,0,0,102,0,0,0,105,0,0,0,110,0,0,0,101,0,0,0,100,0,0,0,39,0,0,0,32,0,0,0,38,0,0,0,38,0,0,0,32,0,0,0,109,0,0,0,111,0,0,0,100,0,0,0,117,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,111,0,0,0,114,0,0,0,116,0,0,0,115,0,0,0,41,0,0,0,32,0,0,0,109,0,0,0,111,0,0,0,100,0,0,0,117,0,0,0,108,0,0,0,101,0,0,0,46,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,111,0,0,0,114,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,61,0,0,0,32,0,0,0,36,0,0,0,74,0,0,0,83,0,0,0,86,0,0,0,65,0,0,0,82,0,0,0,59,0,0,0,34,0,0,0,32,0,0,0,62,0,0,0,62,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,115,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,39,0,0,0,115,0,0,0,47,0,0,0,34,0,0,0,92,0,0,0,40,0,0,0,91,0,0,0,48,0,0,0,45,0,0,0,57,0,0,0,93,0,0,0,43,0,0,0,92,0,0,0,41,0,0,0,34,0,0,0,58,0,0,0,47,0,0,0,92,0,0,0,49,0,0,0,58,0,0,0,47,0,0,0,103,0,0,0,39,0,0,0,32,0,0,0,60,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,32,0,0,0,62,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,10,0,0,0,114,0,0,0,109,0,0,0,32,0,0,0,45,0,0,0,102,0,0,0,32,0,0,0,36,0,0,0,79,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,73,0,0,0,76,0,0,0,69,0,0,0,46,0,0,0,116,0,0,0,109,0,0,0,112,0,0,0,10,0,0,0,96,0,0,0,96,0,0,0,96,0,0,0,10,0,0,0,10,0,0,0,35,0,0,0,35,0,0,0,32,0,0,0,85,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,105,0,0,0,116,0,0,0,105,0,0,0,101,0,0,0,115,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,102,0,0,0,117,0,0,0,110,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,111,0,0,0,110,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,101,0,0,0,32,0,0,0,107,0,0,0,101,0,0,0,112,0,0,0,116,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,32,0,0,0,97,0,0,0,32,0,0,0,115,0,0,0,101,0,0,0,112,0,0,0,97,0,0,0,114,0,0,0,97,0,0,0,116,0,0,0,101,0,0,0,32,0,0,0,115,0,0,0,99,0,0,0,114,0,0,0,105,0,0,0,112,0,0,0,116,0,0,0,32,0,0,0,40,0,0,0,99,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,105,0,0,0,108,0,0,0,115,0,0,0,46,0,0,0,106,0,0,0,115,0,0,0,41,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,66,0,0,0,111,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,97,0,0,0,110,0,0,0,100,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,97,0,0,0,108,0,0,0,32,0,0,0,119,0,0,0,105,0,0,0,116,0,0,0,104,0,0,0,32,0,0,0,100,0,0,0,97,0,0,0,116,0,0,0,97,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,112,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,101,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,100,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,58,0,0,0,10,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,40,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,44,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,112,0,0,0,114,0,0,0,101,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,85,0,0,0,67,0,0,0,83,0,0,0,50,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,115,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,115,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,65,0,0,0,114,0,0,0,114,0,0,0,97,0,0,0,121,0,0,0,32,0,0,0,40,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,97,0,0,0,121,0,0,0,32,0,0,0,111,0,0,0,102,0,0,0,32,0,0,0,74,0,0,0,83,0,0,0,32,0,0,0,83,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,32,0,0,0,99,0,0,0,104,0,0,0,97,0,0,0,114,0,0,0,97,0,0,0,99,0,0,0,116,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,44,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,110,0,0,0,117,0,0,0,109,0,0,0,98,0,0,0,101,0,0,0,114,0,0,0,115,0,0,0,41,0,0,0,10,0,0,0,45,0,0,0,32,0,0,0,66,0,0,0,117,0,0,0,102,0,0,0,102,0,0,0,101,0,0,0,114,0,0,0,32,0,0,0,40,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,85,0,0,0,84,0,0,0,70,0,0,0,45,0,0,0,56,0,0,0,32,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,105,0,0,0,110,0,0,0,103,0,0,0,44,0,0,0,32,0,0,0,100,0,0,0,101,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,32,0,0,0,101,0,0,0,120,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,115,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,112,0,0,0,111,0,0,0,105,0,0,0,110,0,0,0,116,0,0,0,115,0,0,0,47,0,0,0,98,0,0,0,121,0,0,0,116,0,0,0,101,0,0,0,115,0,0,0,41,0,0,0,46,0,0,0,10,0,0,0,10,0,0,0,84,0,0,0,104,0,0,0,101,0,0,0,32,0,0,0,96,0,0,0,111,0,0,0,102,0,0,0,109,0,0,0,116,0,0,0,96,0,0,0,32,0,0,0,118,0,0,0,97,0,0,0,114,0,0,0,105,0,0,0,97,0,0,0,98,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,99,0,0,0,111,0,0,0,110,0,0,0,116,0,0,0,114,0,0,0,111,0,0,0,108,0,0,0,115,0,0,0,32,0,0,0,96,0,0,0,101,0,0,0,110,0,0,0,99,0,0,0,111,0,0,0,100,0,0,0,101,0,0,0,96,0,0,0,32,0,0,0,111,0,0,0,117,0,0,0,116,0,0,0,112,0,0,0,117,0,0,0,116,0,0,0,32,0,0,0,40,0,0,0,96,0,0,0,115,0,0,0,116,0,0,0,114,0,0,0,96,0,0,0,44,0,0,0,32,0,0,0,96,0,0,0,97,0,0,0,114,0,0,0,114,0,0,0,96,0,0,0,32,0,0,0,114,0,0,0,101,0,0,0,115,0,0,0,112,0,0,0,101,0,0,0,99,0,0,0,116,0,0,0,105,0,0,0,118,0,0,0,101,0,0,0,108,0,0,0,121,0,0,0,41,0,0,0,10,0,0,0,119,0,0,0,104,0,0,0,105,0,0,0,108,0,0,0,101,0,0,0,32,0,0,0,116,0,