/**
 * @license
 * Copyright (c) 2017 The expand.js authors. All rights reserved.
 * This code may only be used under the BSD style license found at https://expandjs.github.io/LICENSE.txt
 * The complete set of authors may be found at https://expandjs.github.io/AUTHORS.txt
 * The complete set of contributors may be found at https://expandjs.github.io/CONTRIBUTORS.txt
 */

const _mapValues   = require('lodash/mapValues'),
    assertArgument = require('./assertArgument'),
    isFunction     = require('./isFunction'),
    isObject       = require('./isObject');

/**
 * Returns an object with keys from `object` and values generated by running each own enumerable property of `object` through `iteratee`.
 * The `iteratee` is invoked with three arguments: (`value`, `key`, `object`).
 *
 * ```js
 * XP.mapValues({a: 1, b: 2}, n => n * 3);
 * // => {a: 3, b: 6}
 * ```
 *
 * @function mapValues
 * @since 1.0.0
 * @category object
 * @description Returns an object with keys from `object` and values generated by running each own enumerable property of `object` through `iteratee`
 * @source https://github.com/expandjs/expandjs/blog/master/lib/mapValues.js
 *
 * @param {Object} object The target object
 * @param {Function} iteratee The function invoked per iteration
 * @returns {Object} Returns the new mapped object
 */
module.exports = function mapValues(object, iteratee) {
    assertArgument(isObject(object), 1, 'Object');
    assertArgument(isFunction(iteratee), 2, 'Function');
    return _mapValues(object, iteratee);
};
