/**
 * @license  Highcharts JS v6.0.2 (2017-10-20)
 *
 * Indicator series type for Highstock
 *
 * (c) 2010-2017 Sebastian Bochan
 *
 * License: www.highcharts.com/license
 */
'use strict';
(function(factory) {
    if (typeof module === 'object' && module.exports) {
        module.exports = factory;
    } else {
        factory(Highcharts);
    }
}(function(Highcharts) {
    (function(H) {

        var isArray = H.isArray,
            seriesType = H.seriesType;

        // Utils:
        function accumulateAverage(points, xVal, yVal, i, index) {
            var xValue = xVal[i],
                yValue = index < 0 ? yVal[i] : yVal[i][index];

            points.push([xValue, yValue]);
        }

        function populateAverage(points, xVal, yVal, i, EMApercent, calEMA, index, SMA) {
            var x = xVal[i - 1],
                yValue = index < 0 ? yVal[i - 1] : yVal[i - 1][index],
                y;

            y = calEMA === 0 ? SMA : ((yValue * EMApercent) + (calEMA * (1 - EMApercent)));

            return [x, y];
        }
        /**
         * The EMA series type.
         *
         * @constructor seriesTypes.ema
         * @augments seriesTypes.sma
         */
        seriesType('ema', 'sma',
            /**
             * Exponential moving average indicator (EMA). This series requires `linkedTo` option to be set.
             * 
             * @extends {plotOptions.sma}
             * @product highstock
             * @sample {highstock} stock/indicators/ema
             * 					   Exponential moving average indicator
             * @since 6.0.0
             * @optionparent plotOptions.ema
             */
            {
                name: 'EMA (14)',
                params: {
                    index: 0,
                    period: 14
                }
            }, {
                getValues: function(series, params) {
                    var period = params.period,
                        xVal = series.xData,
                        yVal = series.yData,
                        yValLen = yVal ? yVal.length : 0,
                        EMApercent = (2 / (period + 1)),
                        calEMA = 0,
                        range = 0,
                        sum = 0,
                        EMA = [],
                        xData = [],
                        yData = [],
                        index = -1,
                        points = [],
                        SMA = 0,
                        i,
                        EMAPoint;

                    // Check period, if bigger than points length, skip
                    if (xVal.length < period) {
                        return false;
                    }

                    // Switch index for OHLC / Candlestick / Arearange
                    if (isArray(yVal[0])) {
                        index = params.index ? params.index : 0;
                    }

                    // Accumulate first N-points
                    while (range < period) {
                        accumulateAverage(points, xVal, yVal, range, index);
                        sum += index < 0 ? yVal[range] : yVal[range][index];
                        range++;
                    }

                    // first point 
                    SMA = sum / period;

                    // Calculate value one-by-one for each period in visible data
                    for (i = range; i < yValLen; i++) {
                        EMAPoint = populateAverage(points, xVal, yVal, i, EMApercent, calEMA, index, SMA);
                        EMA.push(EMAPoint);
                        xData.push(EMAPoint[0]);
                        yData.push(EMAPoint[1]);
                        calEMA = EMAPoint[1];

                        accumulateAverage(points, xVal, yVal, i, index);
                    }

                    EMAPoint = populateAverage(points, xVal, yVal, i, EMApercent, calEMA, index);
                    EMA.push(EMAPoint);
                    xData.push(EMAPoint[0]);
                    yData.push(EMAPoint[1]);

                    return {
                        values: EMA,
                        xData: xData,
                        yData: yData
                    };
                }
            });

        /**
         * A `EMA` series. If the [type](#series.ema.type) option is not
         * specified, it is inherited from [chart.type](#chart.type).
         * 
         * For options that apply to multiple series, it is recommended to add
         * them to the [plotOptions.series](#plotOptions.series) options structure.
         * To apply to all series of this specific type, apply it to 
         * [plotOptions.ema](#plotOptions.ema).
         * 
         * @type {Object}
         * @since 6.0.0
         * @extends series,plotOptions.ema
         * @excluding data,dataParser,dataURL
         * @product highstock
         * @apioption series.ema
         */

        /**
         * @type {Array<Object|Array>}
         * @since 6.0.0
         * @extends series.sma.data
         * @product highstock
         * @apioption series.ema.data
         */

    }(Highcharts));
}));
