/**
 * ag-grid-community - Advanced Data Grid / Data Table supporting Javascript / React / AngularJS / Web Components
 * @version v19.1.4
 * @link http://www.ag-grid.com/
 * @license MIT
 */
"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
var context_1 = require("./context/context");
var constants_1 = require("./constants");
var columnController_1 = require("./columnController/columnController");
var utils_1 = require("./utils");
var gridRow_1 = require("./entities/gridRow");
var gridCell_1 = require("./entities/gridCell");
var gridOptionsWrapper_1 = require("./gridOptionsWrapper");
var pinnedRowModel_1 = require("./rowModels/pinnedRowModel");
var CellNavigationService = /** @class */ (function () {
    function CellNavigationService() {
    }
    // returns null if no cell to focus on, ie at the end of the grid
    CellNavigationService.prototype.getNextCellToFocus = function (key, lastCellToFocus) {
        // starting with the provided cell, we keep moving until we find a cell we can
        // focus on.
        var pointer = lastCellToFocus;
        var finished = false;
        // finished will be true when either:
        // a) cell found that we can focus on
        // b) run out of cells (ie the method returns null)
        while (!finished) {
            switch (key) {
                case constants_1.Constants.KEY_UP:
                    pointer = this.getCellAbove(pointer);
                    break;
                case constants_1.Constants.KEY_DOWN:
                    pointer = this.getCellBelow(pointer);
                    break;
                case constants_1.Constants.KEY_RIGHT:
                    if (this.gridOptionsWrapper.isEnableRtl()) {
                        pointer = this.getCellToLeft(pointer);
                    }
                    else {
                        pointer = this.getCellToRight(pointer);
                    }
                    break;
                case constants_1.Constants.KEY_LEFT:
                    if (this.gridOptionsWrapper.isEnableRtl()) {
                        pointer = this.getCellToRight(pointer);
                    }
                    else {
                        pointer = this.getCellToLeft(pointer);
                    }
                    break;
                default:
                    pointer = null;
                    console.log('ag-Grid: unknown key for navigation ' + key);
                    break;
            }
            if (pointer) {
                finished = this.isCellGoodToFocusOn(pointer);
            }
            else {
                finished = true;
            }
        }
        return pointer;
    };
    CellNavigationService.prototype.isCellGoodToFocusOn = function (gridCell) {
        var column = gridCell.column;
        var rowNode;
        switch (gridCell.floating) {
            case constants_1.Constants.PINNED_TOP:
                rowNode = this.pinnedRowModel.getPinnedTopRow(gridCell.rowIndex);
                break;
            case constants_1.Constants.PINNED_BOTTOM:
                rowNode = this.pinnedRowModel.getPinnedBottomRow(gridCell.rowIndex);
                break;
            default:
                rowNode = this.rowModel.getRow(gridCell.rowIndex);
                break;
        }
        var suppressNavigable = column.isSuppressNavigable(rowNode);
        return !suppressNavigable;
    };
    CellNavigationService.prototype.getCellToLeft = function (lastCell) {
        if (!lastCell) {
            return null;
        }
        var colToLeft = this.columnController.getDisplayedColBefore(lastCell.column);
        if (!colToLeft) {
            return null;
        }
        else {
            var gridCellDef = {
                rowIndex: lastCell.rowIndex,
                column: colToLeft,
                floating: lastCell.floating
            };
            return new gridCell_1.GridCell(gridCellDef);
        }
    };
    CellNavigationService.prototype.getCellToRight = function (lastCell) {
        if (!lastCell) {
            return null;
        }
        var colToRight = this.columnController.getDisplayedColAfter(lastCell.column);
        // if already on right, do nothing
        if (!colToRight) {
            return null;
        }
        else {
            var gridCellDef = {
                rowIndex: lastCell.rowIndex,
                column: colToRight,
                floating: lastCell.floating
            };
            return new gridCell_1.GridCell(gridCellDef);
        }
    };
    CellNavigationService.prototype.getRowBelow = function (lastRow) {
        // if already on top row, do nothing
        if (this.isLastRowInContainer(lastRow)) {
            if (lastRow.isFloatingBottom()) {
                return null;
            }
            else if (lastRow.isNotFloating()) {
                if (this.pinnedRowModel.isRowsToRender(constants_1.Constants.PINNED_BOTTOM)) {
                    return new gridRow_1.GridRow(0, constants_1.Constants.PINNED_BOTTOM);
                }
                else {
                    return null;
                }
            }
            else {
                if (this.rowModel.isRowsToRender()) {
                    return new gridRow_1.GridRow(0, null);
                }
                else if (this.pinnedRowModel.isRowsToRender(constants_1.Constants.PINNED_BOTTOM)) {
                    return new gridRow_1.GridRow(0, constants_1.Constants.PINNED_BOTTOM);
                }
                else {
                    return null;
                }
            }
        }
        else {
            return new gridRow_1.GridRow(lastRow.rowIndex + 1, lastRow.floating);
        }
    };
    CellNavigationService.prototype.getCellBelow = function (lastCell) {
        if (!lastCell) {
            return null;
        }
        var rowBelow = this.getRowBelow(lastCell.getGridRow());
        if (rowBelow) {
            var gridCellDef = {
                rowIndex: rowBelow.rowIndex,
                column: lastCell.column,
                floating: rowBelow.floating
            };
            return new gridCell_1.GridCell(gridCellDef);
        }
        else {
            return null;
        }
    };
    CellNavigationService.prototype.isLastRowInContainer = function (gridRow) {
        if (gridRow.isFloatingTop()) {
            var lastTopIndex = this.pinnedRowModel.getPinnedTopRowData().length - 1;
            return lastTopIndex <= gridRow.rowIndex;
        }
        else if (gridRow.isFloatingBottom()) {
            var lastBottomIndex = this.pinnedRowModel.getPinnedBottomRowData().length - 1;
            return lastBottomIndex <= gridRow.rowIndex;
        }
        else {
            var lastBodyIndex = this.rowModel.getPageLastRow();
            return lastBodyIndex <= gridRow.rowIndex;
        }
    };
    CellNavigationService.prototype.getRowAbove = function (lastRow) {
        // if already on top row, do nothing
        if (lastRow.rowIndex === 0) {
            if (lastRow.isFloatingTop()) {
                return null;
            }
            else if (lastRow.isNotFloating()) {
                if (this.pinnedRowModel.isRowsToRender(constants_1.Constants.PINNED_TOP)) {
                    return this.getLastFloatingTopRow();
                }
                else {
                    return null;
                }
            }
            else {
                // last floating bottom
                if (this.rowModel.isRowsToRender()) {
                    return this.getLastBodyCell();
                }
                else if (this.pinnedRowModel.isRowsToRender(constants_1.Constants.PINNED_TOP)) {
                    return this.getLastFloatingTopRow();
                }
                else {
                    return null;
                }
            }
        }
        else {
            return new gridRow_1.GridRow(lastRow.rowIndex - 1, lastRow.floating);
        }
    };
    CellNavigationService.prototype.getCellAbove = function (lastCell) {
        if (!lastCell) {
            return null;
        }
        var rowAbove = this.getRowAbove(lastCell.getGridRow());
        if (rowAbove) {
            var gridCellDef = {
                rowIndex: rowAbove.rowIndex,
                column: lastCell.column,
                floating: rowAbove.floating
            };
            return new gridCell_1.GridCell(gridCellDef);
        }
        else {
            return null;
        }
    };
    CellNavigationService.prototype.getLastBodyCell = function () {
        var lastBodyRow = this.rowModel.getPageLastRow();
        return new gridRow_1.GridRow(lastBodyRow, null);
    };
    CellNavigationService.prototype.getLastFloatingTopRow = function () {
        var lastFloatingRow = this.pinnedRowModel.getPinnedTopRowData().length - 1;
        return new gridRow_1.GridRow(lastFloatingRow, constants_1.Constants.PINNED_TOP);
    };
    CellNavigationService.prototype.getNextTabbedCell = function (gridCell, backwards) {
        if (backwards) {
            return this.getNextTabbedCellBackwards(gridCell);
        }
        else {
            return this.getNextTabbedCellForwards(gridCell);
        }
    };
    CellNavigationService.prototype.getNextTabbedCellForwards = function (gridCell) {
        var displayedColumns = this.columnController.getAllDisplayedColumns();
        var newRowIndex = gridCell.rowIndex;
        var newFloating = gridCell.floating;
        // move along to the next cell
        var newColumn = this.columnController.getDisplayedColAfter(gridCell.column);
        // check if end of the row, and if so, go forward a row
        if (!newColumn) {
            newColumn = displayedColumns[0];
            var rowBelow = this.getRowBelow(gridCell.getGridRow());
            if (utils_1.Utils.missing(rowBelow)) {
                return null;
            }
            newRowIndex = rowBelow ? rowBelow.rowIndex : null;
            newFloating = rowBelow ? rowBelow.floating : null;
        }
        var gridCellDef = { rowIndex: newRowIndex, column: newColumn, floating: newFloating };
        return new gridCell_1.GridCell(gridCellDef);
    };
    CellNavigationService.prototype.getNextTabbedCellBackwards = function (gridCell) {
        var displayedColumns = this.columnController.getAllDisplayedColumns();
        var newRowIndex = gridCell.rowIndex;
        var newFloating = gridCell.floating;
        // move along to the next cell
        var newColumn = this.columnController.getDisplayedColBefore(gridCell.column);
        // check if end of the row, and if so, go forward a row
        if (!newColumn) {
            newColumn = displayedColumns[displayedColumns.length - 1];
            var rowAbove = this.getRowAbove(gridCell.getGridRow());
            if (utils_1.Utils.missing(rowAbove)) {
                return null;
            }
            newRowIndex = rowAbove ? rowAbove.rowIndex : null;
            newFloating = rowAbove ? rowAbove.floating : null;
        }
        var gridCellDef = { rowIndex: newRowIndex, column: newColumn, floating: newFloating };
        return new gridCell_1.GridCell(gridCellDef);
    };
    __decorate([
        context_1.Autowired('columnController'),
        __metadata("design:type", columnController_1.ColumnController)
    ], CellNavigationService.prototype, "columnController", void 0);
    __decorate([
        context_1.Autowired('rowModel'),
        __metadata("design:type", Object)
    ], CellNavigationService.prototype, "rowModel", void 0);
    __decorate([
        context_1.Autowired('pinnedRowModel'),
        __metadata("design:type", pinnedRowModel_1.PinnedRowModel)
    ], CellNavigationService.prototype, "pinnedRowModel", void 0);
    __decorate([
        context_1.Autowired('gridOptionsWrapper'),
        __metadata("design:type", gridOptionsWrapper_1.GridOptionsWrapper)
    ], CellNavigationService.prototype, "gridOptionsWrapper", void 0);
    CellNavigationService = __decorate([
        context_1.Bean('cellNavigationService')
    ], CellNavigationService);
    return CellNavigationService;
}());
exports.CellNavigationService = CellNavigationService;
