import { AgoraRTCError } from '@agora-js/shared';
import { CheckVisibleResult } from '@agora-js/shared';
import { ElectronDesktopCapturerSource } from '@agora-js/shared';
import { EventEmitter } from '@agora-js/shared';
import { IAudioProcessor } from 'agora-rte-extension';
import type { IAudioProcessorContext } from 'agora-rte-extension';
import { IBaseProcessor } from 'agora-rte-extension';
import type { IProcessorContext } from 'agora-rte-extension';
import { isElectron } from '@agora-js/shared';
import type { Kind } from 'agora-rte-extension';
import type { NetworkQuality } from '@agora-js/shared';
import type { ProcessorStats } from 'agora-rte-extension';
import { PromiseMutex } from '@agora-js/shared';
import { RequiredOnlyOneOf } from '@agora-js/shared';
import { SDKStore } from '@agora-js/shared';
import { UID } from '@agora-js/shared';
import { Usage } from 'agora-rte-extension';
import type { UsageWithDirection } from 'agora-rte-extension';

export declare const __TRACK_LIST__: Track[];

export declare function addTrack(track: Track): void;

export declare interface AgoraRTCCompatibility {
    /** 是否支持标准的无插件屏幕共享 API */
    getDisplayMedia: boolean;
    /** 是否支持通过 chrome 插件采集屏幕 */
    getStreamFromExtension: boolean;
    /** 是否支持 unified-plan */
    supportUnifiedPlan: boolean;
    /** 是否支持配置最小码率 */
    supportMinBitrate: boolean;
    /** 是否支持修改 RTCRtpSender 的参数 */
    supportSetRtpSenderParameters: boolean;
    /** 浏览器是否支持开启大小流 */
    supportDualStream: boolean;
    /** WebAudio 是否支持输出 MediaStream */
    webAudioMediaStreamDest: boolean;
    /** RTCRtpSender 是否支持 replaceTrack */
    supportReplaceTrack: boolean;
    /** 是否支持 WebGL */
    supportWebGL: boolean;
    /** 是否支持通过 WebAudio 实现回声消除 */
    webAudioWithAEC: boolean;
    /** 是否支持 CanvasCaptureMediaStreamTrack.requestFrame */
    supportRequestFrame: boolean;
    /** 是否支持屏幕共享音频 */
    supportShareAudio: boolean;
    /** 是否支持使用RTCRtpEncodingParameters设置小流参数*/
    supportDualStreamEncoding: boolean;
    /** 是否支持Datachannel*/
    supportDataChannel: boolean;
    /** 是否支持PeerConnection.setConfiguration*/
    supportPCSetConfiguration: boolean;
    /** 是否支持WebRTC Encoded Transform*/
    supportWebRTCEncodedTransform: boolean;
    /** 是否支持WebRTC Insertable Stream */
    supportWebRTCInsertableStream: boolean;
    /** 是否支持requestVideoFrameCallback, 用于计算卡顿率 */
    supportRequestVideoFrameCallback: boolean;
    /** 是否支持WebCrypto */
    supportWebCrypto: boolean;
}

/**
 * @internal
 */
declare class AgoraRTCPlayer extends VideoPlayer {
    private container?;
    private slot;
    constructor(config: PlayerConfig);
    updateConfig(config: PlayerConfig): void;
    updateVideoTrack(track?: MediaStreamTrack): void;
    play(sessionId?: string): void;
    getCurrentFrame(): ImageData;
    getCurrentFrameToUint8Array(type: string, quality?: number): Promise<ImageTypedData>;
    destroy(): void;
    private createElements;
    private mountedVideoElement;
    private unmountedVideoElement;
    protected resetVideoElement(): void;
    getContainerElement(): HTMLDivElement | undefined;
}

/**
 * 当前通话的统计信息，可以通过 [AgoraRTCClient.getRTCStats]{@link IAgoraRTCClient.getRTCStats} 获取。
 * @public
 */
/** @en
 * Statistics of the call, which can be retrieved by calling [AgoraRTCClient.getRTCStats]{@link IAgoraRTCClient.getRTCStats}.
 */
export declare interface AgoraRTCStats {
    /**
     * 在当前频道内的时长，单位为秒。
     */
    /** @en
     * Call duration in seconds.
     */
    Duration: number;
    /**
     * 音视频总接收码率，单位为 bps，瞬间值。
     */
    /** @en
     * The total bitrate (bps) of the received audio and video, represented by an instantaneous value.
     */
    RecvBitrate: number;
    /**
     * 接收字节数，累计值。
     */
    /** @en
     * The total number of bytes received, represented by an aggregate value.
     */
    RecvBytes: number;
    /**
     * 音视频总发送码率，单位为 bps，瞬间值。
     */
    /** @en
     * The total bitrate (bps) of the sent audio and video, represented by an instantaneous value.
     */
    SendBitrate: number;
    /**
     * 发送字节数，累计值。
     */
    /** @en
     * The total number of bytes sent, represented by an aggregate value.
     */
    SendBytes: number;
    /**
     * 通信场景下，该值为当前频道内的用户人数。
     *
     * 直播场景下，如果本地用户为主播，该值为当前频道内的主播人数；如果本地用户为观众，该值为当前频道内的主播人数 + 1。
     */
    /** @en
     * The number of users in the channel.
     *
     * - Communication profile: The number of users in the channel.
     * - Live Broadcast profile:
     *   - If the local user is an audience: The number of users in the channel = The number of hosts in the channel + 1.
     *   - If the local user is a host: The number of users in the channel = The number of hosts in the channel.
     */
    UserCount: number;
    /**
     * SDK 到声网边缘服务器的 RTT (Round-Trip Time)，单位 ms。
     */
    /** @en
     * RTT (Round-Trip Time) between the SDK and Agora's edge server, in ms.
     */
    RTT: number;
    /**
     * 上行可用带宽估计，单位为 Kbps。
     */
    /** @en
     * The estimated bandwidth (Kbps) of the uplink network.
     */
    OutgoingAvailableBandwidth: number;
}

export declare enum AUDIO_CONTEXT_EVENT {
    IOS_15_16_INTERRUPTION_START = "ios15_16-interruption-start",
    IOS_15_16_INTERRUPTION_END = "ios15_16-interruption-end",
    IOS_INTERRUPTION_START = "ios-interruption-start",
    IOS_INTERRUPTION_END = "ios-interruption-end",
    STATE_CHANGE = "state-change"
}

/** @en
 * @ignore
 */
export declare const AUDIO_ENCODER_CONFIG_SETTINGS: {
    speech_low_quality: AudioEncoderConfiguration;
    speech_standard: AudioEncoderConfiguration;
    music_standard: AudioEncoderConfiguration;
    standard_stereo: AudioEncoderConfiguration;
    high_quality: AudioEncoderConfiguration;
    high_quality_stereo: AudioEncoderConfiguration;
};

export declare enum AUDIO_TRACK_EVENT {
    UPDATE_TRACK_SOURCE = "update-track-source"
}

declare class AudioBufferSource extends AudioSource {
    private audioBuffer;
    protected sourceNode?: AudioBufferSourceNode;
    /**
     * 开始播放的时间（context 时间）
     */
    private startPlayTime;
    /**
     * 开始播放的时间 （buffer 时间）
     */
    private startPlayOffset;
    /**
     * 暂停播放的时间 （buffer 时间）
     */
    private pausePlayTime;
    private options;
    private currentLoopCount;
    private currentPlaybackSpeed;
    set currentState(state: AudioSourceState);
    get currentState(): AudioSourceState;
    private _currentState;
    constructor(buffer: AudioBuffer, options?: AudioSourceOptions);
    createWebAudioDiagram(): GainNode;
    get duration(): number;
    get playbackSpeed(): number;
    /**
     * 计算当前播放时间的公式是
     * （(当前时间 - 开始播放时间) * 当前播放速度 + 开始播放的 offset）% 总时长
     * 如果发现播放被暂停了，返回暂停那一刻记录的 currentTime
     * 如果发现播放还没开始，返回 0
     */
    get currentTime(): number;
    updateOptions(options: AudioSourceOptions): void;
    startProcessAudioBuffer(): void;
    /**
     * AudioSourceNode 并没有提供 pause 方法
     * 一个 AudioSourceNode stop 后也不能再 start 了
     * 所以只能删掉这个 AudioSourceNode 创建新的来实现
     */
    pauseProcessAudioBuffer(): void;
    /**
     * 同 pause，需要重新创建 AudioSourceNode
     * @time 单位是秒
     */
    seekAudioBuffer(time: number): void;
    /**
     * 会从暂停的位置开始播放
     */
    resumeProcessAudioBuffer(): void;
    stopProcessAudioBuffer(): void;
    destroy(): void;
    /**
     * 与native setAudioMixingPlaybackSpeed对齐，推荐范围[50-400]
     */
    setAudioBufferPlaybackSpeed(speed: number): void;
    private startSourceNode;
    private createSourceNode;
    private handleSourceNodeEnded;
    private reset;
}

export declare const audioContextState: AudioState;

/**
 * 理论上来说任何情况下 SDK 都不会使用 <audio> 标签去播放音频，统一使用 WebAudio
 *
 * 但是在 Chrome 上，使用 WebAudio 会导致回声消除失效
 *
 * 在找到可行方案之前，仅在这个情况下会使用 <audio> 标签
 * https://bugs.chromium.org/p/chromium/issues/detail?id=687574
 */
declare class AudioElementPlayCenter {
    onAutoplayFailed?: () => void;
    private elementMap;
    private elementStateMap;
    private elementsNeedToResume;
    private sinkIdMap;
    constructor();
    setSinkID(trackId: string, deviceID: string): Promise<void>;
    play(track: MediaStreamTrack, trackId: string, volume: number, sessionId?: string): void;
    updateTrack(trackId: string, track: MediaStreamTrack): void;
    isPlaying(trackId: string): boolean;
    setVolume(trackId: string, volume: number): void;
    stop(trackId: string): void;
    private bindAudioElementEvents;
    getPlayerState(trackId: string): string;
    /**
     * 监听页面事件，自动恢复音频播放
     */
    private autoResumeAudioElement;
    autoResumeAfterInterruption: (force?: boolean) => void;
    private autoResumeAfterInterruptionOnIOS15_16;
}

export declare const audioElementPlayCenter: AudioElementPlayCenter;

/**
 * 定义音频编码配置的对象。
 *
 * 用于创建音频流时指定自定义的编码配置。
 *
 * 你可以在 [AgoraRTC.createCustomAudioTrack]{@link IAgoraRTC.createCustomAudioTrack}、
 * [AgoraRTC.createMicrophoneAudioTrack]{@link IAgoraRTC.createMicrophoneAudioTrack} 或
 * [AgoraRTC.createBufferSourceAudioTrack]{@link IAgoraRTC.createBufferSourceAudioTrack} 方法中传入该配置对象来自定义本地音频的编码配置。
 */
/** @en
 *
 * `AudioEncoderConfiguration` is the interface that defines the audio encoder configurations.
 *
 * You can customize the audio encoder configurations when calling [AgoraRTC.createCustomAudioTrack]{@link IAgoraRTC.createCustomAudioTrack}, [AgoraRTC.createMicrophoneAudioTrack]{@link IAgoraRTC.createMicrophoneAudioTrack} or [AgoraRTC.createBufferSourceAudioTrack]{@link IAgoraRTC.createBufferSourceAudioTrack}.
 */
export declare interface AudioEncoderConfiguration {
    /**
     * 音频采样率，单位为 Hz。
     */
    /** @en
     * Sample rate of the audio (Hz).
     */
    sampleRate?: number;
    /**
     * 音频采样大小。
     */
    /** @en
     * Sample size of the audio.
     */
    sampleSize?: number;
    /**
     * 是否开启立体声。
     */
    /** @en
     * Whether to enable stereo.
     */
    stereo?: boolean;
    /**
     * 音频码率，单位为 Kbps。
     */
    /** @en
     * Bitrate of the audio (Kbps).
     */
    bitrate?: number;
}

/**
 * SDK 预设的 [AudioEncoderConfiguration]{@link AudioEncoderConfiguration} 配置。
 *
 * 你可以在以下方法中传入预设值来控制本地音频的编码配置：
 * - [AgoraRTC.createCustomAudioTrack]{@link IAgoraRTC.createCustomAudioTrack}
 * - [AgoraRTC.createMicrophoneAudioTrack]{@link IAgoraRTC.createMicrophoneAudioTrack}
 * - [AgoraRTC.createBufferSourceAudioTrack]{@link IAgoraRTC.createBufferSourceAudioTrack}
 *
 * 下表列出了 SDK 所有内置的音频属性配置，SDK 默认使用 `"music_standard"`。
 *
 * | 音频属性 | 配置 |
 * | -------- | --------------- |
 * |`"speech_low_quality"`|16 kHz 采样率，单声道，编码码率约 24 Kbps|
 * |`"speech_standard"`|32 kHz 采样率，单声道，编码码率约 24 Kbps|
 * |`"music_standard"`|48 kHz 采样率，单声道，编码码率约 32 Kbps|
 * |`"standard_stereo"`|48 kHz 采样率，双声道，编码码率约 64 Kbps|
 * |`"high_quality"`|48 kHz 采样率，单声道， 编码码率约 128 Kbps|
 * |`"high_quality_stereo"`|48 kHz 采样率，双声道，编码码率约 192 Kbps|
 * @public
 */
/** @en
 * The preset audio encoder configurations.
 *
 * You can pass the preset video encoder configurations when calling the following methods:
 * - [AgoraRTC.createCustomAudioTrack]{@link IAgoraRTC.createCustomAudioTrack}
 * - [AgoraRTC.createMicrophoneAudioTrack]{@link IAgoraRTC.createMicrophoneAudioTrack}
 * - [AgoraRTC.createBufferSourceAudioTrack]{@link IAgoraRTC.createBufferSourceAudioTrack}
 *
 * The following table lists all the preset audio profiles. The SDK uses `"music_standard"` by default.
 *
 * | Audio Profile | Configurations |
 * | -------- | --------------- |
 * |`"speech_low_quality"`|Sample rate 16 kHz, mono, encoding bitrate 24 Kbps|
 * |`"speech_standard"`|Sample rate 32 kHz, mono, encoding bitrate 24 Kbps|
 * |`"music_standard"`|Sample rate 48 kHz, mono, encoding bitrate 32 Kbps|
 * |`"standard_stereo"`|Sample rate 48 kHz, stereo, encoding bitrate 64 Kbps|
 * |`"high_quality"`|Sample rate 48 kHz, mono, encoding bitrate 128 Kbps|
 * |`"high_quality_stereo"`|Sample rate 48 kHz, stereo, encoding bitrate 192 Kbps| Kbps.
 * @public
 */
export declare type AudioEncoderConfigurationPreset = keyof typeof AUDIO_ENCODER_CONFIG_SETTINGS;

export declare interface AudioPlaybackOptions {
    /**
     * 控制人声是否要被播放，默认为 `true`
     */
    origin?: boolean;
    /**
     * 控制混音背景音乐是否要被播放，默认为 `true`
     */
    mixing?: boolean;
    /**
     * 控制音效是否要被播放，默认为 `true`
     */
    effect?: boolean;
}

export declare class AudioProcessorContext extends EventEmitter implements IAudioProcessorContext {
    private constraintsMap;
    private statsRegistry;
    private readonly audioContext;
    private readonly trackId;
    private readonly direction;
    private usageRegistry;
    private _chained;
    set chained(chained: boolean);
    get chained(): boolean;
    constructor(audioContext: AudioContext, trackId: string, direction: "local" | "remote");
    getConstraints(): Promise<MediaTrackConstraints>;
    getAudioContext(): AudioContext;
    requestApplyConstraints(constraints: MediaTrackConstraints, processor: IBaseProcessor): Promise<void>;
    requestRevertConstraints(processor: IBaseProcessor): Promise<void>;
    registerStats(processor: IBaseProcessor, type: string, cb: () => any): void;
    unregisterStats(processor: IBaseProcessor, type: string): void;
    gatherStats(): ProcessorStats[];
    registerUsage(processor: IBaseProcessor, cb: () => Usage): void;
    unregisterUsage(processor: IBaseProcessor): void;
    gatherUsage(): Promise<UsageWithDirection[]>;
    getDirection(): "local" | "remote";
}

export declare class AudioProcessorDestination extends EventEmitter implements IAudioProcessor {
    name: string;
    ID: string;
    private inputTrack?;
    private inputNode?;
    private readonly audioProcessorContext;
    _source?: IAudioProcessor;
    constructor(audioProcessorContext: AudioProcessorContext);
    get kind(): Kind;
    get enabled(): boolean;
    pipe(): IAudioProcessor;
    unpipe(): void;
    enable(): void;
    disable(): void;
    reset(): void;
    updateInput(inputOptions: {
        track?: MediaStreamTrack;
        node?: AudioNode;
        context: IAudioProcessorContext;
    }): void;
}

declare abstract class AudioSource extends EventEmitter {
    outputNode: GainNode;
    outputTrack?: MediaStreamTrack;
    isPlayed: boolean;
    protected abstract sourceNode?: AudioNode;
    context: AudioContext;
    private audioBufferNode?;
    private destNode?;
    private audioOutputLevel;
    protected volumeLevelAnalyser: VolumeLevelAnalyser;
    private _processedNode;
    get processSourceNode(): AudioNode | undefined;
    set processedNode(node: AudioNode | undefined);
    get processedNode(): AudioNode | undefined;
    /**
     * 用于播放的节点，默认情况下直接连到音频输出
     */
    protected playNode: AudioNode;
    protected isDestroyed: boolean;
    protected onNoAudioInput?: () => void;
    /**
     * 表示当前输入音频是否有问题
     * 目前只会在 Safari 下判断这个情况
     *
     * 连续 5 帧出现数据为 0 的情况
     */
    protected isNoAudioInput: boolean;
    private _noAudioInputCount;
    constructor();
    startGetAudioBuffer(bufferSize: number): void;
    stopGetAudioBuffer(): void;
    createOutputTrack(): MediaStreamTrack;
    play(dest?: AudioNode): void;
    stop(): void;
    /**
     * 获取当前音量
     * @returns 当前音量，范围 [0, 1]
     */
    getAccurateVolumeLevel(): number;
    /**
     * 在检查 1s 内这个 track 是否有音量
     * @param times 递归时候使用的参数，计算递归次数
     * @returns 是否有音量的检查结果
     */
    checkHasAudioInput(times?: number): Promise<boolean>;
    /**
     * 获取音源的设置音量（通过 GainNode 得出的
     *  范围是 [0. Infinity], 1 表示原始音量
     */
    getAudioVolume(): number;
    setVolume(level: number): void;
    destroy(): void;
    protected disconnect(): void;
    protected connect(): void;
}

export declare enum AudioSourceEvents {
    AUDIO_SOURCE_STATE_CHANGE = "audio_source_state_change",
    /** 第一次收到 MediaStreamTrack 的数据 */
    RECEIVE_TRACK_BUFFER = "receive_track_buffer",
    ON_AUDIO_BUFFER = "on_audio_buffer",
    UPDATE_SOURCE = "update_source"
}

/**
 * 音频数据源处理的设置。用于 [startProcessAudioBuffer]{@link IBufferSourceAudioTrack.startProcessAudioBuffer}。
 */
/** @en
 * Options for processing the audio buffer. You need to set the options for processing the audio buffer when calling [startProcessAudioBuffer]{@link IBufferSourceAudioTrack.startProcessAudioBuffer}.
 */
export declare interface AudioSourceOptions {
    /**
     * 设置循环播放的次数。
     */
    /** @en
     * How many times the audio loops.
     */
    cycle?: number;
    /**
     * 设置是否无限循环。
     */
    /** @en
     * Whether to loop the audio infinitely.
     */
    loop?: boolean;
    /**
     * 设置开始播放的时间。
     */
    /** @en
     * The playback position (seconds).
     */
    startPlayTime?: number;
}

/**
 * 音频源数据处理状态，通过 [BufferSourceAudioTrack.on("source-state-change")]{@link IBufferSourceAudioTrack.event_source_state_change} 获取。
 *
 * - `"stopped"`: 音频源数据处理停止。可能是因为数据处理完毕，也可能是手动触发了停止。
 * - `"playing"`: 音频源数据正在处理。
 * - `"paused"`: 音频源数据暂停处理。
 * @public
 */
/** @en
 * Processing state of the audio buffer:
 * - `"stopped"`: The SDK stops processing the audio buffer. Reasons may include:
 *  - The SDK finishes processing the audio buffer.
 *  - The user manually stops the processing of the audio buffer.
 * - `"playing"`: The SDK is processing the audio buffer.
 * - `"paused"`: The SDK pauses processing the audio buffer.
 *
 * You can get the state with [BufferSourceAudioTrack.on("source-state-change")]{@link IBufferSourceAudioTrack.event_source_state_change}.
 */
export declare type AudioSourceState = "stopped" | "playing" | "paused";

declare class AudioState extends EventEmitter {
    prevState: AudioContextState | "interrupted" | undefined;
    curState: AudioContextState | "interrupted" | undefined;
    currentTime?: number;
    currentTimeStuckAt?: number;
    private interruptDetectorTrack?;
    get duringInterruption(): boolean;
    private onLocalAudioTrackMute;
    private onLocalAudioTrackUnmute;
    /**
     * only be called on iOS 15, 16
     * */
    bindInterruptDetectorTrack(track: MicrophoneAudioTrack): void;
    /**
     * only be called on iOS 15, 16
     * */
    unbindInterruptDetectorTrack(track: MicrophoneAudioTrack): void;
}

export declare function audioTimerLoop(callback: (time: number) => any, frequency: number): () => void;

/**
 * 创建音频轨道时的配置参数。
 */
/** @en
 * Configurations for the audio track of screen sharing.
 */
export declare interface AudioTrackInitConfig {
    /**
     * 是否开启回声消除：
     * - `true`: 开启回声消除。
     * - `false`: 不开启回声消除。
     */
    /** @en
     * Whether to enable acoustic echo cancellation:
     * - `true`: Enable acoustic echo cancellation.
     * - `false`: Do not enable acoustic echo cancellation.
     */
    AEC?: boolean;
    /**
     * 是否开启自动增益：
     * - `true`: 开启自动增益。
     * - `false`: 不开启自动增益。
     */
    /** @en
     * Whether to enable audio gain control:
     * - `true`: Enable audio gain control.
     * - `false`: Do not enable audio gain control.
     */
    AGC?: boolean;
    /**
     * 是否开启噪声抑制：
     * - `true`: 开启噪声抑制。
     * - `false`: 不开启噪声抑制。
     */
    /** @en
     * Whether to enable automatic noise suppression:
     * - `true`: Enable automatic noise suppression.
     * - `false`: Do not automatic noise suppression.
     */
    ANS?: boolean;
}

declare class AudioTrackSource extends AudioSource {
    protected sourceNode: MediaStreamAudioSourceNode;
    track: MediaStreamTrack;
    clonedTrack?: MediaStreamTrack;
    private audioElement;
    private isCurrentTrackCloned;
    private isRemoteTrack;
    private originVolumeLevelAnalyser?;
    get isFreeze(): boolean;
    constructor(track: MediaStreamTrack, isRemoteTrack?: boolean, originTrack?: MediaStreamTrack);
    /**
     * 在 ios 12 上，如果 context 在 suspend 的状态下创建了 MediaStreamSource
     * 当 context running 后 MediaStreamSource 是无法继续使用的
     * 需要重新构建 WebAudio 节点
     */
    private rebuildWebAudio;
    updateTrack(track: MediaStreamTrack): void;
    destroy(): void;
    createMediaStreamSourceNode(track: MediaStreamTrack): MediaStreamAudioSourceNode;
    updateOriginTrack(originTrack: MediaStreamTrack): void;
    getOriginVolumeLevel(): number;
}

declare class AutoPlayGestureEventEmitter extends EventEmitter {
    onAutoplayFailed?: () => void;
    onAudioAutoplayFailed?: () => void;
}

export declare const autoPlayGestureEventEmitter: AutoPlayGestureEventEmitter;

/**
 * @ignore
 *
 * 美颜选项，用于 [setBeautyEffect]{@link ILocalVideoTrack.setBeautyEffect}。
 * @public
 */
/** @en
 * @ignore
 *
 * Image enhancement options. You need to set the image enhancement options when calling [setBeautyEffect]{@link ILocalVideoTrack.setBeautyEffect}.
 */
export declare interface BeautyEffectOptions {
    /**
     * 平滑度。
     *
     * 取值范围为 [0.0, 1.0]，其中 0.0 表示原始平滑等级，默认值为 0.5。可用来实现祛痘、磨皮等视觉效果。
     */
    /** @en
     *
     * The smoothness level.
     *
     * The value range is [0.0, 1.0]. The original smoothness level is 0.0. The default value is 0.5. This parameter is usually used to remove blemishes.
     */
    smoothnessLevel?: number;
    /**
     * 亮度。
     *
     * 取值范围为 [0.0, 1.0]，其中 0.0 表示原始亮度，默认值为 0.7。可用来实现美白等视觉效果。
     */
    /** @en
     * The brightness level.
     *
     * The value range is [0.0, 1.0]. The original brightness level is 0.0. The default value is 0.7.
     */
    lighteningLevel?: number;
    /**
     * 红色度。
     *
     * 取值范围为 [0.0, 1.0]，其中 0.0 表示原始红色度，默认值为 0.1。可用来实现红润肤色等视觉效果。
     */
    /** @en
     * The redness level.
     *
     * The value range is [0.0, 1.0]. The original redness level is 0.0. The default value is 0.1. This parameter adjusts the red saturation level.
     */
    rednessLevel?: number;
    /**
     * 亮度明暗对比度，与 {@link lighteningLevel} 参数搭配使用。可设为：
     * - 0: 明暗对比弱。
     * - 1: (默认) 原始明暗对比度。
     * - 2: 明暗对比强。
     */
    /** @en
     * The contrast level. Use this together with {@link lighteningLevel}.
     * - 0: Low contrast level.
     * - 1: (Default) The original contrast level.
     * - 2: High contrast level.
     */
    lighteningContrastLevel?: 0 | 1 | 2;
}

export declare interface BeautyWebGLParameters {
    denoiseLevel?: number;
    lightLevel?: number;
    rednessLevel?: number;
    lighteningContrastLevel?: number;
}

export declare const blob2Uint8Array: (blob: Blob) => Promise<Uint8Array>;

export declare class BufferSourceAudioTrack extends LocalAudioTrack implements IBufferSourceAudioTrack {
    source: string | File | AudioBuffer | null;
    private _bufferSource;
    get __className__(): string;
    constructor(source: string | File | AudioBuffer, bufferSource: AudioBufferSource, encodingConfig?: AudioEncoderConfiguration, trackId?: string);
    get currentState(): AudioSourceState;
    get duration(): number;
    get playbackSpeed(): number;
    getCurrentTime(): number;
    startProcessAudioBuffer(options?: AudioSourceOptions): void;
    pauseProcessAudioBuffer(): void;
    seekAudioBuffer(time: number): void;
    resumeProcessAudioBuffer(): void;
    stopProcessAudioBuffer(): void;
    close(): void;
    setAudioBufferPlaybackSpeed(speed: number): void;
}

/**
 * 通过本地音频文件/在线音频文件/`AudioBuffer`方式创建音频流时的配置参数，用于 [AgoraRTC.createBufferSourceAudioTrack]{@link IAgoraRTC.createBufferSourceAudioTrack}。
 */
/** @en
 * Configurations for the audio track from an audio file or `AudioBuffer` object. Set these configurations when calling [AgoraRTC.createBufferSourceAudioTrack]{@link IAgoraRTC.createBufferSourceAudioTrack}.
 */
export declare interface BufferSourceAudioTrackInitConfig {
    /**
     * 音频数据源，支持 3 种类型：
     * - `File`: 浏览器标准的 [File](https://developer.mozilla.org/en-US/docs/Web/API/File) 对象，表示一个本地文件。
     * - `string`: 表示从线上 HTTPS 地址获取在线音频文件(请确保音频在线地址支持 HTTPS 和 CORS)。
     * - `AudioBuffer`: 浏览器标准的 [AudioBuffer](https://developer.mozilla.org/en-US/docs/Web/API/AudioBuffer) 对象表示 PCM 原始数据。
     */
    /** @en
     * The type of the audio source:
     * - `File`: An [File](https://developer.mozilla.org/en-US/docs/Web/API/File) object, representing a local audio file.
     * - `string`: The online audio file retrieved from an HTTPS address. Ensure the address supports HTTPS and CORS.
     * - `AudioBuffer`: An [AudioBuffer](https://developer.mozilla.org/en-US/docs/Web/API/AudioBuffer) object, representing the raw data in PCM format.
     */
    source: File | string | AudioBuffer;
    /**
     * 是否缓存线上文件:
     * - `true`: 缓存线上文件。
     * - `false`: （默认）不缓存线上文件。
     */
    /** @en
     * Whether to cache the online file:
     * - `true`: Cache the online file.
     * - `false`: (default) Do not cache the online file.
     */
    cacheOnlineFile?: boolean;
    /**
     * 控制音频的编码配置。
     *
     * 你可以通过 [[AudioEncoderConfigurationPreset]] 传入 SDK 内置的编码配置；也可以通过传入一个 [[AudioEncoderConfiguration]] 来自定义音频编码配置。
     *
     * > Firefox 不支持设置音频编码码率。
     */
    /** @en
     * The audio encoder configurations.
     *
     * You can set the audio encoder configurations in either of the following ways:
     * - Pass the preset audio encoder configurations by using [[AudioEncoderConfigurationPreset]].
     * - Pass your customized audio encoder configurations by using [[AudioEncoderConfiguration]].
     *
     * > Firefox does not support setting the audio encoding rate.
     */
    encoderConfig?: AudioEncoderConfiguration | AudioEncoderConfigurationPreset;
}

export declare class CameraVideoTrack extends LocalVideoTrack implements ICameraVideoTrack {
    private _config;
    private _originalConstraints;
    private _constraints;
    _enabled: boolean;
    _deviceName: string;
    get __className__(): string;
    constructor(track: MediaStreamTrack, config: CameraVideoTrackInitConfig, constraints: MediaTrackConstraints, scalabilityConfig?: SVCConfiguration, optimizationMode?: OptimizationMode | "balanced", trackId?: string);
    setDevice(deviceId: string | RequiredOnlyOneOf<{
        facingMode: VideoFacingModeEnum;
        deviceId: string;
    }>): Promise<void>;
    private _setDeviceById;
    /**
     * 通过 facingMode 设置设备
     * @param facingMode https://developer.mozilla.org/en-US/docs/Web/API/MediaTrackSettings/facingMode
     */
    private _setDeviceByFacingModel;
    setEnabled(enabled: boolean, skipChangeState?: boolean): Promise<void>;
    setEncoderConfiguration(config: VideoEncoderConfiguration | VideoEncoderConfigurationPreset, doNotRenegoation?: boolean): Promise<void>;
    protected _getDefaultPlayerConfig(): Partial<PlayerConfig>;
    protected onTrackEnded(): void;
    renewMediaStreamTrack(newConstraints?: MediaTrackConstraints): Promise<void>;
    tryResumeVideoForIOS15_16WeChat: () => Promise<void>;
    close(): void;
    clone(config?: VideoEncoderConfiguration | VideoEncoderConfigurationPreset, cloneTrack?: boolean): CameraVideoTrack;
    bindProcessorContextEvents(): void;
}

/**
 * 创建摄像头视频流时的配置对象，用于 [AgoraRTC.createCameraVideoTrack]{@link IAgoraRTC.createCameraVideoTrack}。
 */
/** @en
 * Configurations for the video track from the video captured by a camera. Set these configurations when calling [AgoraRTC.createCameraVideoTrack]{@link IAgoraRTC.createCameraVideoTrack}.
 */
export declare interface CameraVideoTrackInitConfig {
    /**
     * 控制视频的编码配置。
     *
     * 你可以通过以下方式设置该属性：
     * - 通过 [[VideoEncoderConfigurationPreset]] 传入 SDK 内置的编码配置。
     * - 通过传入一个 [[VideoEncoderConfiguration]] 对象来自定义视频编码配置。
     * - 不传或留空，使用 SDK 的默认值 `"480p_1"`（分辨率 640 × 480、帧率 15 fps、码率 500 Kbps）。
     */
    /** @en
     * The video encoder configurations.
     *
     * You can set the video encoder configurations in either of the following ways:
     * - Pass the preset video encoder configurations by using [[VideoEncoderConfigurationPreset]].
     * - Pass your customized video encoder configurations by using [[VideoEncoderConfiguration]].
     * - Leave this property empty to use the SDK's default value, `"480p_1"` (resolution: 640 × 480, frame rate: 15 fps, bitrate: 500 Kbps).
     */
    encoderConfig?: VideoEncoderConfiguration | VideoEncoderConfigurationPreset;
    /**
     * 指定使用前置/后置摄像头来采集视频。
     *
     * 在移动设备上，可以设置该参数选择使用前置或后置摄像头：
     * - `"user"`: 前置摄像头
     * - `"environment"`: 后置摄像头
     */
    /** @en
     * Whether to user the front camera or the rear camera.
     *
     * You can use this parameter to choose between the front camera and the rear camera on a mobile device:
     * - `"user"`: The front camera.
     * - `"environment"`: The rear camera.
     */
    facingMode?: VideoFacingModeEnum;
    /**
     * 指定摄像头的设备 ID。
     *
     * 你可以通过 [AgoraRTC.getCameras]{@link IAgoraRTC.getCameras} 来获取当前的摄像头设备列表。
     */
    /** @en
     * Specifies the camera ID.
     *
     * You can get a list of the available cameras by calling [AgoraRTC.getCameras]{@link IAgoraRTC.getCameras}.
     */
    cameraId?: string;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.2.0*
     *
     * 设置视频传输优化模式。
     *
     * 你可以在视频通话、视频直播或屏幕共享过程中调用此方法动态调整视频的传输优化模式。例如你想要把屏幕共享内容从演示文稿切换为视频时，你可以将传输优化模式从 `"detail"` 切换为 `"motion"`，确保视频画面在网络波动时不会出现卡顿。
     *
     * > 注意事项：该方法只支持 Chrome 浏览器。
     *
     * @param mode 视频传输优化模式：
     * - `"detail"`: 清晰优先。
     *   - 浏览器会自动根据你的采集分辨率和帧率设定一个最小码率。即使遭遇网络波动，发送码率也不会低于这个值，从而确保清晰的视频画面。
     *   - 大部分情况下，浏览器不会降低发送分辨率，但是可能会降低帧率。
     * - `"motion"`: 自 4.21.0 开始，浏览器默认开启流畅优先模式。
     *   - 浏览器不会启用最小码率策略。遭遇网络波动时，发送端会降低码率来确保接收端的视频画面不会出现中断和卡顿。
     *   - 大部分情况下，浏览器不会降低帧率，但是可能会降低发送分辨率。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.2.0*
     *
     * Sets the video transmission optimization mode.
     *
     * You can call this method during a video call, a live streaming or screen sharing to dynamically change the optimization mode. For example, during the screen sharing, before you change the shared content from text to video, you can change the optimization mode from `"detail"` to `"motion"` to ensure smoothness in poor network conditions.
     *
     * > Note: This method supports Chrome only.
     *
     * @param mode The video transmission optimization mode:
     * - `"detail"`: Prioritizes video quality.
     *   - The browser ensures high-quality images by automatically calculating a minimum bitrate based on the capturing resolution and frame rate. No matter how poor the network condition is, the sending bitrate will never be lower than the minimum value.
     *   - In most cases, the browser does not reduce the sending resolution, but may reduce the frame rate.
     * -  `"motion"`: Since v4.21.0, the browser prioritizes video smoothness.
     *   - In poor network conditions, the browser reduces the sending bitrate to minimize video freezes.
     *   - In most cases, the browser does not reduce the frame rate, but may reduce the sending resolution.
     */
    optimizationMode?: OptimizationMode;
    /**
     * @ignore
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.18.0*
     *
     * SVC （可伸缩视频编码）配置。
     *
     * 你可以通过 {@link SVCConfigurationPreset} 传入 SDK 预设的 SVC 编码配置，也可以通过 {@link SVCConfiguration} 传入自定义的 SVC 编码配置。
     */
    /** @en
     * @ignore
     *
     * @since
     * <br>&emsp;&emsp;&emsp;*4.18.0*
     *
     * Configurations for Scalable Video Coding (SVC).
     *
     * You can set the configurations using one of the following options:
     * - Use the preset SVC configurations provided by the SDK through {@link SVCConfigurationPreset}.
     * - Use your custom SVC configurations through {@link SVCConfiguration}.
     */
    scalabiltyMode?: SVCConfiguration | SVCConfigurationPreset;
}

export declare function checkMediaStreamTrackResolution(track: MediaStreamTrack): Promise<[number, number]>;

export declare function checkTrackState(): (target: any, propertyKey: any, descriptor: PropertyDescriptor) => TypedPropertyDescriptor<any> | undefined;

/**
 * `<video>` 标签的可见状态信息。
 *
 * 可通过以下方法或事件获取：
 * - [localVideoTrack.on("video-element-visible-status")]{@link ILocalVideoTrack.event_video_element_visible_status}
 * - [localVideoTrack.getVideoElementVisibleStatus]{@link ILocalVideoTrack.getVideoElementVisibleStatus}
 * - [remoteVideoTrack.on("video-element-visible-status")]{@link IRemoteVideoTrack.event_video_element_visible_status_2}
 * - [remoteVideoTrack.getVideoElementVisibleStatus]{@link IRemoteVideoTrack.getVideoElementVisibleStatus}
 *
 * 包含以下字段：
 * - `visible`: Boolean 型，`<video>` 标签是否可见。
 * - `reason`: 仅当 `visible` 为 `false` 时有效，不可见原因：
 *   - `"SIZE"`: `<video>` 尺寸太小，导致 `<video>` 完全不可见或很难看到。
 *   - `"STYLE"`: `<video>` 或其祖先元素的 CSS 样式导致 `<video>` 不可见、颜色暗淡或模糊不清。
 *   - `"POSITION"`: `<video>` 或其祖先元素被定位到视口区域外，导致 `<video>` 只有小面积区域在视口内或全部不在视口内。
 *   - `"COVERED"`: `<video>` 被其他元素遮挡。
 */
/** @en
 * The visibility of the `<video>` tag.
 *
 * Get the visibility of the `<video>` tag through the following methods and events:
 * - [localVideoTrack.on("video-element-visible-status")]{@link ILocalVideoTrack.event_video_element_visible_status}
 * - [localVideoTrack.getVideoElementVisibleStatus]{@link ILocalVideoTrack.getVideoElementVisibleStatus}
 * - [remoteVideoTrack.on("video-element-visible-status")]{@link IRemoteVideoTrack.event_video_element_visible_status_2}
 * - [remoteVideoTrack.getVideoElementVisibleStatus]{@link IRemoteVideoTrack.getVideoElementVisibleStatus}
 *
 * This object contains the following parameters:
 * - `visible`: Boolean, whether the `<video>` tag is visible or not.
 * - `reason`: This parameter is only valid when `visible` is `false`, which indicates the reason:
 *    - `"SIZE"`: The size of `<video>` is too small to see.
 *    - `"STYLE"`: The CSS styles of `<video>` or its ancestor element cause `<video>` to be invisible.
 *    - `"POSITION"`: The `<video>` tag or its ancestor element are positioned outside the viewport.
 *    - `"COVERED"`: The `<video>` tag is covered by other elements.
 */
export declare type CheckVideoVisibleResult = CheckVisibleResult;

/**
 * 约束对象，一般用于在 [[VideoEncoderConfiguration]] 中指定采集分辨率/帧率的范围。
 */
/** @en
 * Specifies a constraint for a property, such as the resolution or bitrate for video capture in [[VideoEncoderConfiguration]].
 */
export declare interface ConstrainLong {
    /**
     * 采集设备最终输出的值下限。
     */
    /** @en
     * The lower limit of the property.
     */
    min?: number;
    /**
     * 采集设备最终输出的值上限。
     */
    /** @en
     * The upper limit of the property.
     */
    max?: number;
    /**
     * 期望采集设备最终输出的值，如果设备不支持指定的值，会尽量输出一个最靠近的值。
     */
    /** @en
     * An ideal value of a property. If the video capture device cannot output this value, it outputs the closest value instead.
     */
    ideal?: number;
    /**
     * 严格指定采集设备最终输出的值，如果设备不支持指定的值，采集会失败。
     */
    /** @en
     * A required value of a property. If the video capture device cannot output this value, the video capture fails.
     */
    exact?: number;
}

export declare function createBufferSourceAudioTrack(config: BufferSourceAudioTrackInitConfig): Promise<BufferSourceAudioTrack>;

export declare function createCameraVideoTrack(config?: CameraVideoTrackInitConfig): Promise<CameraVideoTrack>;

export declare function createCustomAudioTrack(config: CustomAudioTrackInitConfig): LocalAudioTrack;

export declare function createCustomVideoTrack(config: CustomVideoTrackInitConfig): LocalVideoTrack;

export declare function createMicrophoneAndCameraTracks(audioConfig?: MicrophoneAudioTrackInitConfig, videoConfig?: CameraVideoTrackInitConfig): Promise<[MicrophoneAudioTrack, CameraVideoTrack]>;

export declare function createMicrophoneAudioTrack(config?: MicrophoneAudioTrackInitConfig): Promise<MicrophoneAudioTrack>;

export declare function createMixingAudioTrack(trackList: LocalAudioTrack[]): MixingAudioTrack;

export declare function createScreenVideoTrack(config: ScreenVideoTrackInitConfig, withAudio: "enable"): Promise<[LocalVideoTrack, LocalAudioTrack]>;

export declare function createScreenVideoTrack(config?: ScreenVideoTrackInitConfig, withAudio?: "disable"): Promise<LocalVideoTrack>;

export declare function createScreenVideoTrack(config: ScreenVideoTrackInitConfig, withAudio: "auto" | ScreenAudioTrackInitConfig): Promise<[LocalVideoTrack, LocalAudioTrack] | LocalVideoTrack>;

/**
 * 创建自定义音频轨道时的配置参数，用于 [AgoraRTC.createCustomAudioTrack]{@link IAgoraRTC.createCustomAudioTrack}。
 */
/** @en
 * Configurations for the custom audio track. Set these configurations when calling [AgoraRTC.createCustomAudioTrack]{@link IAgoraRTC.createCustomAudioTrack}.
 */
export declare interface CustomAudioTrackInitConfig {
    /**
     * 你自己维护的 [MediaStreamTrack](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack) 对象。
     */
    /** @en
     * Your [MediaStreamTrack](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack) object.
     */
    mediaStreamTrack: MediaStreamTrack;
    /**
     * 控制音频的编码配置。
     *
     * 你可以通过 [[AudioEncoderConfigurationPreset]] 传入 SDK 内置的编码配置；也可以通过传入一个 [[AudioEncoderConfiguration]] 来自定义音频编码配置。
     *
     * > Firefox 不支持设置音频编码码率。
     */
    /** @en
     * The audio encoder configurations.
     *
     * You can set the audio encoder configurations in either of the following ways:
     * - Pass the preset audio encoder configurations by using [[AudioEncoderConfigurationPreset]].
     * - Pass your customized audio encoder configurations by using [[AudioEncoderConfiguration]].
     *
     * > Firefox does not support setting the audio encoding rate.
     */
    encoderConfig?: AudioEncoderConfiguration | AudioEncoderConfigurationPreset;
}

/**
 * 创建自定义视频轨道时的配置参数，用于 [AgoraRTC.createCustomVideoTrack]{@link IAgoraRTC.createCustomVideoTrack}。
 */
/** @en
 * Configurations for the custom video track. Set these configurations when calling [AgoraRTC.createCustomVideoTrack]{@link IAgoraRTC.createCustomVideoTrack}.
 */
export declare interface CustomVideoTrackInitConfig {
    /**
     * 这里填写您自己维护的 [MediaStreamTrack](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack) 对象
     */
    /** @en
     * Your [MediaStreamTrack](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack) object.
     */
    mediaStreamTrack: MediaStreamTrack;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * 视频的分辨率宽。
     *
     * 支持传入 `number`，或一个约束对象，如 `{ max: 1280, min: 720 }`。
     *
     * 关于约束对象的详细说明可以参考 [ConstrainLong]{@link ConstrainLong}。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * Width of the video.
     *
     * You can pass a `number`, or a constraint such as `{ max: 1280, min: 720 }`.
     *
     * For more details about the constraint, see [ConstrainLong]{@link ConstrainLong}.
     */
    width?: number | ConstrainLong;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * 视频的分辨率高。
     *
     * 支持传入 `number`，或一个约束对象，如 `{ max: 1280, min: 720 }`。
     *
     * 关于约束对象的详细说明可以参考 [ConstrainLong]{@link ConstrainLong}。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * Height of the video.
     *
     * You can pass a `number`, or a constraint such as `{ max: 1280, min: 720 }`.
     *
     * For more details about the constraint, see [ConstrainLong]{@link ConstrainLong}.
     */
    height?: number | ConstrainLong;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * 视频帧率，单位为 fps。
     *
     * 支持传入 `number`，或一个约束对象，如 `{ max: 30, min: 5 }`。
     *
     * 关于约束对象的详细说明可以参考 [ConstrainLong]{@link ConstrainLong}。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * Frame rate of the video (fps).
     *
     * You can pass a `number`, or a constraint such as `{ max: 30, min: 5 }`.
     *
     * For details about the constraint, see [ConstrainLong]{@link ConstrainLong}.
     */
    frameRate?: number | ConstrainLong;
    /**
     * 指定发送这个视频轨道时的最小码率，单位为 Kbps。
     */
    /** @en
     * The minimum bitrate of sending the video track (Kbps).
     */
    bitrateMin?: number;
    /**
     * 指定发送这个视频轨道时的最大码率，单位为 Kbps。
     */
    /** @en
     * The maximum bitrate of sending the video track (Kbps).
     */
    bitrateMax?: number;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.2.0*
     *
     * 设置视频传输优化模式。
     *
     * 你可以在视频通话、视频直播或屏幕共享过程中调用此方法动态调整视频的传输优化模式。例如你想要把屏幕共享内容从演示文稿切换为视频时，你可以将传输优化模式从 `"detail"` 切换为 `"motion"`，确保视频画面在网络波动时不会出现卡顿。
     *
     * > 注意事项：该方法只支持 Chrome 浏览器。
     *
     * @param mode 视频传输优化模式：
     * - `"balanced"`: 使用默认的传输优化模式：
     *   - 对于屏幕共享视频流，浏览器默认的优化策略为清晰优先。
     *   - 对于其他视频流，浏览器默认的优化策略为兼顾清晰和流畅，弱网条件下，帧率和分辨率都会被调整。
     * - `"detail"`: 清晰优先。
     *   - 浏览器会自动根据你的采集分辨率和帧率设定一个最小码率。即使遭遇网络波动，发送码率也不会低于这个值，从而确保清晰的视频画面。
     *   - 大部分情况下，浏览器不会降低发送分辨率，但是可能会降低帧率。
     * - `"motion"`: 自 4.21.0 开始，浏览器默认开启流畅优先模式
     *   - SDK 不会启用最小码率策略。遭遇网络波动时，发送端会降低码率来确保接收端的视频画面不会出现中断和卡顿。
     *   - 大部分情况下，SDK 不会降低帧率，但是可能会降低发送分辨率。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.2.0*
     *
     * Sets the video transmission optimization mode.
     *
     * You can call this method during a video call, a live streaming or screen sharing to dynamically change the optimization mode. For example, during the screen sharing, before you change the shared content from text to video, you can change the optimization mode from `"detail"` to `"motion"` to ensure smoothness in poor network conditions.
     *
     * > Note: This method supports Chrome only.
     *
     * @param mode The video transmission optimization mode:
     * - `"balanced"`: Uses the default optimization mode.
     *   - For a screen-sharing video track, the default transmission optimization strategy is to prioritizes clarity.
     *   - For the other types of video tracks, the SDK may reduce the frame rate or the sending resolution in poor network conditions.
     * - `"detail"`: Prioritizes video quality.
     *   - The SDK ensures high-quality images by automatically calculating a minimum bitrate based on the capturing resolution and frame rate. No matter how poor the network condition is, the sending bitrate will never be lower than the minimum value.
     *   - In most cases, the SDK does not reduce the sending resolution, but may reduce the frame rate.
     * -  `"motion"`: Since 4.21.0, the SDK prioritizes video smoothness.
     *   - In poor network conditions, the SDK reduces the sending bitrate to minimize video freezes.
     *   - In most cases, the SDK does not reduce the frame rate, but may reduce the sending resolution.
     */
    optimizationMode?: "motion" | "detail";
    /**
     * @ignore
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.18.0*
     *
     * SVC （可伸缩视频编码）配置。
     *
     * 你可以通过 {@link SVCConfigurationPreset} 传入 SDK 预设的 SVC 编码配置，也可以通过 {@link SVCConfiguration} 传入自定义的 SVC 编码配置。
     */
    /** @en
     * @ignore
     *
     * @since
     * <br>&emsp;&emsp;&emsp;*4.18.0*
     *
     * Configurations for Scalable Video Coding (SVC).
     *
     * You can set `scalabiltyMode` using one of the following options:
     * - Use the preset SVC configurations provided by the SDK through {@link SVCConfigurationPreset}.
     * - Use your custom SVC configurations through {@link SVCConfiguration}.
     */
    scalabiltyMode?: SVCConfiguration | SVCConfigurationPreset;
}

export declare abstract class DataChannel extends EventEmitter implements IDataChannel {
    readonly trackMediaType: TrackMediaType.DATA;
    private _version;
    private _type;
    _config: IDataChannelConfig;
    /** 原始的 DataChannel 对象 */
    _originDataChannel?: RTCDataChannel;
    protected _dataStreamPacketHeader: ArrayBuffer;
    protected _dataStreamPacketHandler: IDataStream;
    private _datachannelEventMap;
    constructor(config: IDataChannelConfig, datachannel?: RTCDataChannel);
    useDataStream(dataStream: IDataStream): void;
    get id(): number;
    get ordered(): boolean;
    get maxRetransmits(): number;
    get metadata(): string;
    get readyState(): RTCDataChannelState;
    get _originDataChannelId(): number | null;
    getChannelId(): number;
    getConfig(): IDataChannelConfig;
    _close(): void;
    _waitTillOpen(): Promise<void>;
    _updateOriginDataChannel(datachannel: RTCDataChannel): void;
    private _initPacketHeader;
    private _bandDataChannelEvents;
    private _unbindDataChannelEvents;
}

export declare enum DataChannelEvents {
    OPEN = "open",
    MESSAGE = "message",
    CLOSE = "close",
    CLOSING = "closing",
    ERROR = "error"
}

export declare function decodeAudioData(buffer: ArrayBuffer): Promise<AudioBuffer>;

export declare const DEFAULT_LOCAL_AUDIO_TRACK_STATS: LocalAudioTrackStats;

export declare const DEFAULT_LOCAL_VIDEO_TRACK_STATS: LocalVideoTrackStats;

export declare const DEFAULT_NETWORK_QUALITY_STATS: NetworkQuality;

export declare const DEFAULT_REMOTE_AUDIO_TRACK_STATS: RemoteAudioTrackStats;

export declare const DEFAULT_REMOTE_VIDEO_TRACK_STATS: RemoteVideoTrackStats;

export declare interface DenoiserStats {
    ns: number;
    ebn: number;
    ean: number;
    vl: number;
}

/**
 * 是否支持Audio标签 setSinkId
 * Firefox 116 以上也支持此接口，但是会出现回声问题
 * 仅支持PC 端 Chrome / Edge，且不支持移动端
 */
export declare function detectSupportAudioElementSetSinkId(): boolean;

/**
 * 音视频采集设备的信息。
 *
 * - 音频采集设备信息可通过 [onMicrophoneChanged]{@link onMicrophoneChanged} 获取。
 * - 视频采集设备信息可通过 [onCameraChanged]{@link onCameraChanged} 获取。
 * - 音频播放设备信息可通过 [onPlaybackDeviceChanged]{@link onPlaybackDeviceChanged} 获取。
 * @public
 */
/** @en
 * Information of the media input device.
 *
 * - You can get the audio sampling device information through [onMicrophoneChanged]{@link onMicrophoneChanged}.
 * - You can get the video capture device information through [onCameraChanged]{@link onCameraChanged}.
 * - You can get the audio playback device information through [onPlaybackDeviceChanged]{@link onPlaybackDeviceChanged}.
 */
export declare interface DeviceInfo {
    /**
     * 采集设备的插拔状态更新时间，UNIX 时间戳，单位为 ms。
     */
    /** @en
     * The latest time when the state of the media input device was updated.
     *
     * A Unix timestamp in milliseconds.
     */
    updateAt: number;
    /**
     * SDK 首次检测到采集设备的时间，UNIX 时间戳，单位为 ms。
     */
    /** @en
     * The time when the SDK first detects the media input device.
     *
     * A Unix timestamp in milliseconds.
     */
    initAt: number;
    /**
     * 采集设备当前的插拔状态。
     */
    /** @en
     * The state of the capture device.
     */
    state: DeviceState;
    /**
     * 采集设备的设备信息，详见 [MediaDeviceInfo](https://developer.mozilla.org/en-US/docs/Web/API/MediaDeviceInfo)。
     */
    /** @en
     * Device information of the media input device. See [MediaDeviceInfo](https://developer.mozilla.org/en-US/docs/Web/API/MediaDeviceInfo) for details.
     */
    device: MediaDeviceInfo;
}

export declare class DeviceManager extends EventEmitter {
    private _state;
    get state(): DeviceManagerState;
    set state(state: DeviceManagerState);
    private isAccessMicrophonePermission;
    private isAccessCameraPermission;
    private lastAccessMicrophonePermission;
    private lastAccessCameraPermission;
    private checkdeviceMatched;
    /**
     * 记录所有枚举到的设备和这些设备最后出现的时间
     */
    private deviceInfoMap;
    constructor();
    enumerateDevices(audio: boolean, video: boolean, skipPermissionCheck?: boolean): Promise<MediaDeviceInfo[]>;
    getRecordingDevices(skipPermissionCheck?: boolean): Promise<MediaDeviceInfo[]>;
    getCamerasDevices(skipPermissionCheck?: boolean): Promise<MediaDeviceInfo[]>;
    getSpeakers(skipPermissionCheck?: boolean): Promise<MediaDeviceInfo[]>;
    searchDeviceIdByName(deviceName: string): string | null;
    getDeviceById(deviceId: string): Promise<MediaDeviceInfo>;
    private init;
    private updateDevicesInfo;
    /** 检查媒体设备信息是否可用，判断标准为 deviceId 和 label 不能为空 */
    private checkMediaDeviceInfoIsOk;
}

export declare const deviceManager: DeviceManager;

export declare enum DeviceManagerEvent {
    STATE_CHANGE = "state_change",
    RECORDING_DEVICE_CHANGED = "recordingDeviceChanged",
    PLAYOUT_DEVICE_CHANGED = "playoutDeviceChanged",
    CAMERA_DEVICE_CHANGED = "cameraDeviceChanged"
}

export declare enum DeviceManagerState {
    IDLE = "IDLE",
    INITING = "INITING",
    INITEND = "INITEND"
}

/**
 * 采集设备当前的插拔状态。
 * - `"ACTIVE"`: 该设备已经插入。
 * - `"INACTIVE"`: 该设备已经被拔出。
 * @public
 */
/** @en
 * The state of the media input device.
 * - `"ACTIVE"`: The device is plugged in.
 * - `"INACTIVE"`: The device is unplugged.
 */
export declare type DeviceState = "ACTIVE" | "INACTIVE";

export declare const emptyImage2TypedArray: (type: string) => Promise<ImageTypedData>;

/**
 * @ignore
 *
 * 通过 [setBeautyEffect]{@link ILocalVideoTrack.setBeautyEffect} 开启美颜后，如果检测到美颜对设备性能消耗过大就会触发该事件。
 *
 * 你可以监听这个事件提示用户美颜过载，并关闭美颜。
 *
 * ```javascript
 * localVideoTrack.on("beauty-effect-overload", () => {
 *   console.log("beauty effect overload, disable beauty effect");
 *   localVideoTrack.setBeautyEffect(false);
 * });
 * ```
 * @event
 * @asMemberOf ILocalVideoTrack
 */
/** @en
 * @ignore
 *
 * Occurs when the device is overloaded after you call [setBeautyEffect]{@link ILocalVideoTrack.setBeautyEffect} to enable image enhancement.
 *
 * You can listen for this event to notify users of the device overload and disable image enhancement.
 *
 * ```javascript
 * localVideoTrack.on("beauty-effect-overload", () => {
 *   console.log("beauty effect overload, disable beauty effect");
 *   localVideoTrack.setBeautyEffect(false);
 * });
 * ```
 * @event
 * @asMemberOf ILocalVideoTrack
 */
declare function event_beauty_effect_overload(): void;

/**
 * 首帧音频或视频解码时触发。
 *
 * @event
 * @asMemberOf IRemoteTrack
 */
/** @en
 * Occurs when the first remote audio or video frame is decoded.
 *
 * @event
 * @asMemberOf IRemoteTrack
 */
declare function event_first_frame_decoded(): void;

/**
 * @自从
 * <br>&emsp;&emsp;&emsp;*4.20.1*
 *
 * 从 H.264 视频流中解析出 SEI 数据时触发，此事件会返回 SEI 数据。
 *
 * @param sei Uint8Array 类型的 SEI 数据
 * @asMemberOf ILocalVideoTrack
 * @event
 */
/** @en
 * @since
 * <br>&emsp;&emsp;&emsp;*4.20.1*
 *
 * Parsing SEI data from the H.264 video stream triggers this event, which returns the SEI data.
 *
 * @param sei SEI data in Uint8Array
 * @asMemberOf ILocalVideoTrack
 * @event
 */
declare function event_sei_received(sei: Uint8Array): void;

/**
 * 当 [BufferSourceAudioTrack]{@link IBufferSourceAudioTrack} 的音频数据源处理状态变化时，会触发此事件。
 *
 * @param currentState 当前音频数据处理状态：
 * - `"stopped"`: 音频源数据处理停止。可能是因为数据处理完毕，也可能是手动触发了停止。
 * - `"paused"`: 音频源数据暂停处理。
 * - `"playing"`: 音频源数据正在处理。
 *
 * @event
 * @asMemberOf IBufferSourceAudioTrack
 */
/** @en
 * Occurs when the state of processing the audio buffer in [BufferSourceAudioTrack]{@link IBufferSourceAudioTrack} changes.
 *
 * @param currentState The state of processing the audio buffer:
 * - `"stopped"`: The SDK stops processing the audio buffer. Reasons may include:
 *  - The SDK finishes processing the audio buffer.
 *  - The user manually stops the processing of the audio buffer.
 * - `"paused"`: The SDK pauses the processing of the audio buffer.
 * - `"playing"`: The SDK is processing the audio buffer.
 *
 * @event
 * @asMemberOf IBufferSourceAudioTrack
 */
declare function event_source_state_change(currentState: AudioSourceState): void;

/**
 * 音频或视频轨道被终止，终止的原因可能是：
 * - 摄像头被拔出
 * - 麦克风被拔出
 * - 用户主动停止了屏幕共享
 * - 用户主动关闭了底层的 `MediaStreamTrack`
 * - 媒体设备故障
 * - 正在使用中的媒体设备的权限被收回
 *
 * @event
 * @asMemberOf ILocalTrack
 */
/** @en
 * Occurs when a audio or video track ends.
 *
 * Reasons may include:
 * - Camera is unplugged.
 * - Microphone is unplugged.
 * - The local user stops screen sharing.
 * - The local user closes the underlying `MediaStreamTrack`.
 * - A local media device malfunctions.
 * - The device permission is revoked.
 *
 * @event
 * @asMemberOf ILocalTrack
 */
declare function event_track_ended(): void;

/**
 * 媒体轨道有更新时触发。
 * @param track 媒体轨道。详见 [MediaStreamTrack](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack)。
 * @event
 */
/** @en
 * Triggers when a media track is updated.
 * @param track The media stream track. See [MediaStreamTrack](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack).
 * @event
 */
export declare function event_track_updated(track: MediaStreamTrack): void;

/**
 * 当前轨道的 `RTCRtpTransceiver` 实例更新回调。
 *
 * @param transceiver 新的 `RTCRtpTransceiver` 实例。
 * @param type 当前轨道所属视频流的类型。详见 {@link StreamType}。
 *
 * @event
 * @asMemberOf ILocalTrack
 */
/** @en
 * Occurs when the `RTCRtpTransceiver` instance of the current track is updated.
 *
 * @param transceiver The new `RTCRtpTransceiver` instance.
 * @param type The type of the video stream to which the current track belongs. See {@link StreamType}.
 *
 * @event
 * @asMemberOf ILocalTrack
 */
declare function event_transceiver_updated(transceiver: RTCRtpTransceiver, type?: StreamType): void;

/**
 * 当前轨道的 `RTCRtpTransceiver` 实例更新回调。
 *
 * @param transceiver 新的 `RTCRtpTransceiver` 实例。
 *
 * @event
 * @asMemberOf IRemoteTrack
 */
/** @en
 * Occurs when the `RTCRtpTransceiver` instance of the current track is updated.
 *
 * @param transceiver The new `RTCRtpTransceiver` instance.
 * @event
 * @asMemberOf IRemoteTrack
 */
declare function event_transceiver_updated_2(transceiver: RTCRtpTransceiver): void;

/**
 * @ignore
 */
/** @en
 * @ignore
 */
declare function event_user_datachannel_close(): void;

/**
 * @ignore
 */
/** @en
 * @ignore
 */
declare function event_user_datachannel_error(ev: Event): void;

/**
 * @ignore
 */
/** @en
 * @ignore
 */
declare function event_user_datachannel_message(data: ArrayBuffer): void;

/**
 * @ignore
 */
/** @en
 * @ignore
 */
declare function event_user_datachannel_open(): void;

/**
 * @自从
 * <br>&emsp;&emsp;&emsp;*4.8.0*
 *
 * 提示 HTML `<video>` 标签的可见状态和不可见原因。SDK 每 30 秒触发一次该事件。
 *
 * 调用 `localVideoTrack.play` 后 SDK 会创建 [HTML `<video>`](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video) 标签用于播放视频轨道。当 `localVideoTrack.isPlaying` 为 `true` 却看不到图像时，你可通过此事件来进行问题排查。
 *
 * @param data `<video>` 标签的可见状态信息。
 * @asMemberOf ILocalVideoTrack
 * @event
 */
/** @en
 * @since
 * <br>&emsp;&emsp;&emsp;*4.8.0*
 *
 * Indicates the visibility of the `<video>` HTML tag.
 *
 * The SDK triggers this event every 30 seconds.
 *
 * After you call `localVideoTrack.play`, the SDK creates an [`<video>`](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video) tag for playing video tracks. When `localVideoTrack.isPlaying` is `true` but you cannot see any video, this event helps you check whether the `<video>` tag is visible or not and learn the reason when the `<video>` tag is invisible.
 *
 * @param data The visibility of the `<video>` tag.
 * @asMemberOf ILocalVideoTrack
 * @event
 */
declare function event_video_element_visible_status(data?: CheckVideoVisibleResult): void;

/**
 * @自从
 * <br>&emsp;&emsp;&emsp;*4.8.0*
 *
 * 提示 HTML `<video>` 标签的可见状态和不可见原因。SDK 每 30 秒触发一次该事件。
 *
 * 调用 `remoteVideoTrack.play` 后 SDK 会创建 [HTML `<video>`](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video) 标签用于播放视频轨道。当 `remoteVideoTrack.isPlaying` 为 `true` 却看不到图像时，你可通过此事件来进行问题排查。
 *
 * @param data `<video>` 标签的可见状态信息。
 * @asMemberOf IRemoteVideoTrack
 * @event
 */
/** @en
 * @since
 * <br>&emsp;&emsp;&emsp;*4.8.0*
 *
 * Indicates the visibility of the `<video>` HTML tag.
 *
 * The SDK triggers this event every 30 seconds.
 *
 * After you call `remoteVideoTrack.play`, the SDK creates an [`<video>`](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video) tag for playing video tracks. When `remoteVideoTrack.isPlaying` is `true` but you cannot see any video, this event helps you check whether the `<video>` tag is visible or not and learn the reason when the `<video>` tag is invisible.
 *
 * @param data The visibility of the `<video>` tag.
 * @asMemberOf IRemoteVideoTrack
 * @event
 */
declare function event_video_element_visible_status_2(data?: CheckVideoVisibleResult): void;

/**
 * 视频状态发生变化时触发。
 *
 * @event
 * @asMemberOf IRemoteTrack
 */
/** @en
 * Occurs when the video state changes.
 *
 * @event
 * @asMemberOf IRemoteTrack
 */
declare function event_video_state_changed(videoState: VideoState): void;

export declare interface ExtendedVideoFrameCallbackMetadata extends VideoFrameCallbackMetadata {
    timestamp: DOMHighResTimeStamp;
}

export declare interface ExternalMethods {
    getDenoiserStats?: () => DenoiserStats | undefined;
}

declare class FakeAudioNode {
    disconnect(): void;
    connect(): void;
}

declare class FakeTrackSource extends EventEmitter {
    context: any;
    processSourceNode: undefined;
    outputTrack: undefined;
    processedNode: undefined;
    clonedTrack: undefined;
    outputNode: FakeAudioNode;
    get isPlayed(): boolean;
    get isFreeze(): boolean;
    constructor();
    setVolume(): void;
    createOutputTrack(): MediaStreamTrack;
    getOriginVolumeLevel(): number;
    getAccurateVolumeLevel(): number;
    stopGetAudioBuffer(): void;
    startGetAudioBuffer(): void;
    play(): void;
    stop(): void;
    destroy(): void;
    updateTrack(): void;
    updateOriginTrack(): void;
    createMediaStreamSourceNode(): undefined;
}

export declare const frameData2CryptoBuffer: (imageData: ImageTypedData, appid: string, channelName: string) => Promise<Uint8Array>;

export declare function getAudioContext(): AudioContext;

export declare function getAudioEncoderConfiguration(profile: AudioEncoderConfigurationPreset | AudioEncoderConfiguration): AudioEncoderConfiguration;

export declare function getBitrateConstrainRange(width: number, height: number, frameRate: number, bitrateMin?: number, bitrateMax?: number): DoubleRange;

/**
 * get bitrate, use STANDARD_BITRATE to make sure the response
 * @param width
 * @param height
 * @param fps
 */
export declare function getBitrateFromResAndFps(width: number, height: number, fps: number): Required<DoubleRange>;

export declare function getCompatibility(): AgoraRTCCompatibility;

export declare function getConstraintsFromAudioConfig(config: AudioTrackInitConfig): MediaTrackConstraints;

export declare function getConstraintsFromCameraConfig(config: CameraVideoTrackInitConfig): MediaTrackConstraints;

export declare function getConstraintsFromMicrophoneConfig(config: MicrophoneAudioTrackInitConfig): MediaTrackConstraints;

export declare function getConstraintsFromScreenAudioConfig(config: ScreenAudioTrackInitConfig): MediaTrackConstraints;

export declare function getConstraintsFromScreenConfig(config: ScreenVideoTrackInitConfig): ScreenConstraintsWithElectron;

export declare function getElectronScreenSources(type?: ScreenSourceType): Promise<ElectronDesktopCapturerSource[]>;

export declare function getElectronScreenStream(sourceId: string, config: ScreenConstraintsWithElectron, captureAudio?: boolean): Promise<MediaStream>;

export declare function getElectronScreenStreamByUserSelect(config: ScreenConstraintsWithElectron, captureAudio?: boolean): Promise<MediaStream>;

export declare function getLocalStream(config: GetUserMediaConfig, id: string): Promise<MediaStream>;

/**
 * 获取初始配置
 * 分辨率：track.getMediaStreamTrackSettings > 客户设置
 * 帧率: track.getMediaStreamTrackSettings > 客户设置
 * 码率： 客户设置 > 根据分辨率计算
 *
 * ⚠️：理论上不存在没有分辨率的情况，如果出现不足任何处理
 */
export declare const getOriginSenderConfig: (track: LocalVideoTrack) => {
    frameRate: number;
    bitrateMax: number;
    bitrateMin: number;
    scaleResolutionDownBy: number;
    scale: number;
};

export declare function getScalabilityConfiguration(profile: SVCConfiguration | SVCConfigurationPreset): SVCConfiguration;

export declare function getScreenEncoderConfiguration(profile: VideoEncoderConfiguration | ScreenEncoderConfigurationPreset | string): VideoEncoderConfiguration;

export declare function getSilenceAudioTrack(): MediaStreamTrack;

/**
 * 当音频解码完成后，PCM 会被扔给 WebAudio，此时如果解码失败，PCM 的长度就会减少。
 * WebAudio 为了保证自己音频循环不会延迟或者中断，当 PCM 不足时，会 padding zero value 到 Buffer 中。
 *
 * 这个函数的目的就是找出 WebAudio padding 的样本时长，从而估算出丢失了多少音频数据
 */
export declare function getSilenceSamplesDuration(buffer: AudioBuffer): number;

/**
 * 获取一个稳定流，缓存一帧数据，当流停止超过1s时，会不断发送上一帧数据
 */
export declare function getStaticTrackStream(track: MediaStreamTrack, interval?: number): MediaStreamTrack | void;

export declare interface GetUserMediaConfig {
    video?: MediaTrackConstraints;
    audio?: MediaTrackConstraints;
    screen?: ScreenConstraintsWithElectron;
    videoSource?: MediaStreamTrack;
    audioSource?: MediaStreamTrack;
    screenAudio?: boolean | MediaTrackConstraints;
}

export declare function getVideoEncoderConfiguration(profile: VideoEncoderConfigurationPreset | VideoEncoderConfiguration | undefined): VideoEncoderConfiguration;

/**
 * 将原生的 DomException 转成对应的 AgoraRTCError
 */
export declare function handleGetUserMediaError(errorName: string, message?: string): AgoraRTCError;

export declare let HAS_GUM_AUDIO: boolean;

export declare let HAS_GUM_VIDEO: boolean;

export declare function hasAudioContext(): boolean;

/**
 * 直接通过音频数据源创建的音频轨道，继承于 [LocalAudioTrack]{@link ILocalAudioTrack}, 在此之上提供了控制音频数据处理流程的功能（开始/暂停/跳转等等）。
 *
 * 可以通过 [AgoraRTC.createBufferSourceAudioTrack]{@link IAgoraRTC.createBufferSourceAudioTrack} 来获取。
 */
/** @en
 * Inherited from [LocalAudioTrack]{@link ILocalAudioTrack}, `BufferSourceAudioTrack` is an interface for the audio from a local audio file and adds several functions for controlling the processing of the audio buffer, such as starting processing, stopping processing, and seeking a specified time location.
 *
 * You can create an audio track from an audio file by calling [AgoraRTC.createBufferSourceAudioTrack]{@link IAgoraRTC.createBufferSourceAudioTrack}.
 */
export declare interface IBufferSourceAudioTrack extends ILocalAudioTrack {
    /**
     * 创建时指定的 [source]{@link BufferSourceAudioTrackInitConfig.source} 值。
     */
    /** @en
     * The [source]{@link BufferSourceAudioTrackInitConfig.source} specified when creating an audio track.
     */
    source: string | File | AudioBuffer | null;
    /**
     * 当前音频数据源的处理状态（开始/暂停/停止）。
     */
    /** @en
     * The current state of audio processing, such as start, pause, or stop.
     */
    currentState: AudioSourceState;
    /**
     * 音频总时长，单位为秒。
     */
    /** @en
     * The total duration of the audio (seconds).
     */
    duration: number;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.18.0*
     *
     * 当前音频文件的播放速度。取值范围为 [50,400]，其中：
     * - `50`: 0.5 倍速。
     * - `100`: （默认）原始速度。
     * - `400`: 4 倍速。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.18.0*
     *
     * The playback speed of the current audio file. Valid range is [50, 400], where:
     * - `50`: Half the original speed.
     * - `100`: (Default) The original speed.
     * - `400`: Four times the original speed.
     */
    playbackSpeed: number;
    /**
     * @param event 事件名称。
     * @param listener 详见 [source-state-change]{@link event_source_state_change}。
     */
    /** @en
     * @param event The event name.
     * @param listener See [source-state-change]{@link event_source_state_change}.
     */
    on(event: "source-state-change", listener: typeof event_source_state_change): void;
    /**
     * 当指定的事件触发时，会调用传入的回调函数。
     * @param event 事件名称。
     * @param listener 回调函数。
     */
    /** @en
     * When the specified event happens, the SDK triggers the callback that you pass.
     *
     * @param event The event name.
     * @param listener The callback function.
     */
    on(event: string, listener: Function): void;
    /**
     * 获取当前的音频处理的进度，单位为秒。
     *
     * @returns 当前音频处理的进度。
     */
    /** @en
     * Gets the progress (seconds) of the audio buffer processing.
     *
     * @returns The progress (seconds) of the audio buffer processing.
     */
    getCurrentTime(): number;
    /**
     * 开始处理音频数据源。
     *
     * > 开始处理音频数据源说明我们已经将音频数据送入 SDK 的处理节点。如果此时该音频轨道已发布，远端可以听到该音源的声音。
     * > 只有在调用 [[play]] 之后，音频数据才会送入声卡，此时本地才能听到该音源的声音。
     *
     * @param options 音频数据源处理的相关设置，包括是否需要循环播放/开始播放的位置。详见 [[AudioSourceOptions]]。
     */
    /** @en
     * Starts processing the audio buffer.
     *
     * > Starting processing the audio buffer means that the processing unit in the SDK has received the audio data. If the audio track has been published, the remote user can hear the audio.
     * > Whether the local user can hear the audio depends on whether the SDK calls the [[play]] method and sends the audio data to the sound card.
     *
     * @param options Options for processing the audio buffer. See [[AudioSourceOptions]].
     */
    startProcessAudioBuffer(options?: AudioSourceOptions): void;
    /**
     * 暂停处理音频数据。
     */
    /** @en
     * Pauses processing the audio buffer.
     */
    pauseProcessAudioBuffer(): void;
    /**
     * 跳转到指定时间。
     *
     * @param time 指定的音频时间，单位为秒。
     */
    /** @en
     * Jumps to a specified time point.
     *
     * @param time The specified time point (seconds).
     */
    seekAudioBuffer(time: number): void;
    /**
     * 恢复被暂停的音频数据处理。
     */
    /** @en
     * Resumes processing the audio buffer.
     */
    resumeProcessAudioBuffer(): void;
    /**
     * 停止处理音频数据。
     */
    /** @en
     * Stops processing the audio buffer.
     */
    stopProcessAudioBuffer(): void;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.18.0*
     *
     * 设置当前音频文件的播放速度。
     *
     * 在加入频道前和加入频道后，你都可以调用该方法。
     *
     * @param speed 播放速度。取值范围为 [50,400]，其中：
     * - `50`: 0.5 倍速。
     * - `100`: （默认）原始速度。
     * - `400`: 4 倍速。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.18.0*
     *
     * Sets the playback speed for the current audio file.
     *
     * You can call this method before or after joining a channel.
     *
     * @param speed The playback speed. Valid range is [50, 400], where:
     * - `50`: Half the original speed.
     * - `100`: (Default) The original speed.
     * - `400`: Four times the original speed.
     */
    setAudioBufferPlaybackSpeed(speed: number): void;
}

/**
 * 本地摄像头视频轨道，继承于 [LocalVideoTrack]{@link ILocalVideoTrack}，并在此基础上提供了更换采集设备/调整编码参数的功能。
 *
 * 可以通过 [AgoraRTC.createCameraVideoTrack]{@link IAgoraRTC.createCameraVideoTrack} 来获取。
 */
/** @en
 *
 * Inherited from [LocalVideoTrack]{@link ILocalVideoTrack}, `CameraVideoTrack` is an interface for the video captured by a local camera and adds functions such as switching devices and adjusting video encoder configurations.
 *
 * You can create a local camera video track by calling [AgoraRTC.createCameraVideoTrack]{@link IAgoraRTC.createCameraVideoTrack}.
 */
export declare interface ICameraVideoTrack extends ILocalVideoTrack {
    /**
     * 设置负责采集视频数据的摄像头设备。
     *
     * > 在发布前和发布后均可调用此方法。
     *
     * @param deviceId 设备 ID，支持通过以下方式传入：
     * - 传入字符串：传入通过 [AgoraRTC.getCameras]{@link IAgoraRTC.getCameras} 获取的 `deviceId`。
     * - 传入对象：自 4.19.0 起，你可以传入一个包含 `facingMode` 或 `deviceId` 的对象，但只能指定其中一个属性的值。其中 `deviceId` 通过 [AgoraRTC.getCameras]{@link IAgoraRTC.getCameras} 获取，`facingMode` 支持设为以下值：
     *   - `"environment"`: 使用后置摄像头。
     *   - `"user"`: 使用前置摄像头。
     */
    /** @en
     * Sets the device for capturing video.
     *
     * > You can call this method either before or after publishing the video track.
     *
     * @param deviceId Device ID, which can be passed in using the following ways:
     * - Pass a string: Pass the `deviceId` obtained using [AgoraRTC.getCameras]{@link IAgoraRTC.getCameras}.
     * - Pass an object: Starting from version 4.19.0, you can pass an object
     * containing `facingMode` or `deviceId`, but only one of these properties
     * can be specified. `deviceId` can be obtained through [AgoraRTC.getCameras]{@link IAgoraRTC.getCameras}, and `facingMode` supports the following values:
     *    - `"environment"`: Use the rear camera.
     *    - `"user"`: Use the front camera.
     */
    setDevice(deviceId: string | RequiredOnlyOneOf<{
        facingMode: VideoFacingModeEnum;
        deviceId: string;
    }>): Promise<void>;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.0.0*
     *
     * 启用/禁用该轨道。
     *
     * 轨道禁用后，播放和发布都将被停止。
     *
     * > - 禁用轨道不会触发 [LocalTrack.on("track-ended")]{@link event_track_ended} 事件。
     * > - 如果该轨道已发布，禁用轨道后，远端会触发 [user-unpublished]{@link IAgoraRTCClient.event_user_unpublished} 事件。重新启用后，远端会触发 [user-published]{@link IAgoraRTCClient.event_user_published} 事件。
     * > - `setEnabled` 和 `setMuted` 不能同时调用。
     *
     * @param enabled 是否启用该轨道:
     * - `true`: 启用该轨道.
     * - `false`: 禁用该轨道.
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.0.0*
     *
     * Enables/Disables the track.
     *
     * After a track is disabled, the SDK stops playing and publishing the track.
     *
     * > - Disabling a track does not trigger the [LocalTrack.on("track-ended")]{@link event_track_ended} event.
     * > - If a track is published, disabling this track triggers the [user-unpublished]{@link IAgoraRTCClient.event_user_unpublished} event on the remote client, and re-enabling this track triggers the [user-published]{@link IAgoraRTCClient.event_user_published} event.
     * > - Do not call `setEnabled` and `setMuted` together.
     *
     * @param enabled Whether to enable the track:
     * - `true`: Enable the track.
     * - `false`: Disable the track.
     */
    setEnabled(enabled: boolean): Promise<void>;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.19.0*
     *
     * 复制当前的视频轨道对象，生成一个新的视频轨道。
     *
     * 在视频会议、在线教育等场景中，你可以用该方法实现同一视频流的两路显示，并为每路显示设置不同的分辨率、帧率、码率等参数。
     *
     * @param config 新生成视频轨道的编码配置。你可以通过 [[VideoEncoderConfiguration]] 传入 SDK 内置的编码配置；也可以通过传入一个 [[VideoEncoderConfigurationPreset]] 来自定义视频编码配置。
     * @param cloneTrack 是否克隆当前轨道。默认为 `true`。
     * @returns 新生成的视频轨道。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.19.0*
     *
     * Clones the current video track to create a new video track.
     *
     * In scenarios such as video conferencing and online education, you can use this method to display the same video stream with two sets of display parameters, including resolution, frame rate, and bitrate. For example, you can have one display set to high-definition and the other to low-definition.
     *
     * @param config The encoding configuration for the new video track. You can pass in the SDK's built-in encoding configuration through [[VideoEncoderConfiguration]], or customize the video encoding configuration by passing in a [[VideoEncoderConfigurationPreset]].
     * @param cloneTrack Whether to clone the current track. Default is `true`.
     * @returns The newly generated video track.
     */
    clone(config?: VideoEncoderConfiguration | VideoEncoderConfigurationPreset, cloneTrack?: boolean): ICameraVideoTrack;
}

/** @ignore */
/** @en
 * @ignore
 */
export declare interface IDataChannel extends EventEmitter {
    readonly id: number;
    readonly maxRetransmits: number | null;
    readonly ordered: boolean;
    readonly readyState: RTCDataChannelState;
    readonly metadata: string;
    getChannelId(): number;
    getConfig(): IDataChannelConfig;
}

/** @ignore */
/** @en
 * @ignore
 */
export declare interface IDataChannelConfig {
    id: number;
    /**
     * 表示通过 RTCDataChannel 的信息的到达顺序需要和发送顺序一致 (true), 或者到达顺序不需要和发送顺序一致 (false). 默认：true.
     */
    ordered: boolean;
    /**
     * 0~512
     */
    metadata: string;
}

declare interface IDataStream {
    serialize: (payload: ArrayBuffer) => ArrayBuffer;
    deserialize: (packet: ArrayBuffer) => ArrayBuffer;
}

/**
 * 基础本地音频轨道，提供了本地音频轨道的主要功能。
 *
 * 可以通过 [AgoraRTC.createCustomAudioTrack]{@link IAgoraRTC.createCustomAudioTrack} 来获取。
 *
 * - 通过麦克风采集到的本地音频轨道对象 [MicrophoneAudioTrack]{@link IMicrophoneAudioTrack} 继承于此接口，并在本接口的基础上提供了一些麦克风独有的功能。
 * - 通过音频文件获取到的本地音频轨道对象 [BufferSourceAudioTrack]{@link IBufferSourceAudioTrack} 继承于此接口，并在本接口的基础上提供了一些音频文件独有的功能。
 */
/** @en
 * `LocalAudioTrack` is the basic interface for local audio tracks, providing main methods of local audio tracks.
 *
 * You can create a local audio track by calling [AgoraRTC.createCustomAudioTrack]{@link IAgoraRTC.createCustomAudioTrack}.
 *
 * The following interfaces are inherited from `LocalAudioTrack`:
 * - [MicrophoneAudioTrack]{@link IMicrophoneAudioTrack}, the interface for the audio sampled by a local microphone, which adds several microphone-related functions.
 * - [BufferSourceAudioTrack]{@link IBufferSourceAudioTrack}, the interface for the audio from a local audio file, which adds several audio-file-related functions.
 */
export declare interface ILocalAudioTrack extends ILocalTrack {
    /**
     * 设置本地音频轨道的音量。
     * @param volume 音量值，范围 [0, 1000], 0 代表静音，100 代表原始音量。100 以上会使用 WebAudio 进行音量增益。如果本地音频轨道已经发布，修改音量会影响远端听到的音量大小。
     */
    /** @en
     * Sets the volume of a local audio track.
     *
     * @param volume The volume. The value ranges from 0 (mute) to 1000 (maximum). A value of 100 is the original volume。 The volume change may not be obvious to human ear. If local track has been published, setting volume will affect the volume heard by remote users.
     */
    setVolume(volume: number): void;
    /**
     * 获取本地音频轨道的音量等级。
     *
     * @returns 音量等级值，范围 [0, 1]，1 代表理论最大音量。通常该值大于 0.6 代表用户在持续说话。
     */
    /** @en
     * Gets the audio level of a local audio track.
     *
     * @returns The audio level. The value range is [0,1]. 1 is the highest audio level.
     * Usually a user with audio level above 0.6 is a speaking user.
     *
     */
    getVolumeLevel(): number;
    /**
     * 设置原始音频数据（PCM）回调。
     *
     * 设置成功后，SDK 会不断地将本地音频轨道的音频帧以 [AudioBuffer](https://developer.mozilla.org/en-US/docs/Web/API/AudioBuffer) 的形式通过回调返回。
     *
     * > 你可以通过 `frameSize` 来设置每次回调中音频帧的大小。该设置也会影响回调的间隔，`frameSize` 越大，每次回调的音频数据越多，回调间隔越长。
     *
     * ```js
     * track.setAudioFrameCallback((buffer) => {
     *   for (let channel = 0; channel < buffer.numberOfChannels; channel += 1) {
     *     // Float32Array with PCM data
     *     const currentChannelData = buffer.getChannelData(channel);
     *     console.log("PCM data in channel", channel, currentChannelData);
     *   }
     * }, 2048);
     *
     * // ....
     * // Stop getting the raw audio data
     * track.setAudioFrameCallback(null);
     * ```
     *
     * @param audioFrameCallback 用于接收 [AudioBuffer](https://developer.mozilla.org/en-US/docs/Web/API/AudioBuffer) 的回调函数。设为 `null` 后，SDK 就会停止获取音频原始数据。
     * @param frameSize 每次回调的 `AudioBuffer` 中每个声道包含的采样数据个数，只能设为下列值：256, 512, 1024, 2048, 4096, 8192, 16384。默认为 4096。
     */
    /** @en
     * Sets the callback for getting raw audio data in PCM format.
     *
     * After you successfully set the callback, the SDK constantly returns the audio frames of a local audio track in this callback by using [AudioBuffer](https://developer.mozilla.org/en-US/docs/Web/API/AudioBuffer).
     *
     * > You can set the `frameSize` parameter to determine the frame size in each callback, which affects the interval between the callbacks. The larger the frame size, the longer the interval between them.
     *
     * ```js
     * track.setAudioFrameCallback((buffer) => {
     *   for (let channel = 0; channel < buffer.numberOfChannels; channel += 1) {
     *     // Float32Array with PCM data
     *     const currentChannelData = buffer.getChannelData(channel);
     *     console.log("PCM data in channel", channel, currentChannelData);
     *   }
     * }, 2048);
     *
     * // ....
     * // Stop getting the raw audio data
     * track.setAudioFrameCallback(null);
     * ```
     *
     * @param audioFrameCallback The callback function for receiving the [AudioBuffer](https://developer.mozilla.org/en-US/docs/Web/API/AudioBuffer) object. If you set `audioBufferCallback` as `null`, the SDK stops getting raw audio data.
     * @param frameSize The number of samples of each audio channel that an `AudioBuffer` object contains. You can set `frameSize` as 256, 512, 1024, 2048, 4096, 8192, or 16384. The default value is 4096.
     */
    setAudioFrameCallback(audioFrameCallback: null | ((buffer: AudioBuffer) => void), frameSize?: number): void;
    /**
     * 播放本地音频轨道。
     *
     * > 播放音频时 SDK 不会创建任何 DOM 元素，所以无需像视频一样提供 DOM 元素。
     */
    /** @en
     * Plays a local audio track.
     *
     * > When playing a audio track, you do not need to pass any DOM element.
     */
    play(): void;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.1.0*
     *
     * > 注意事项：
     * > - 自 v4.7.0 起，该方法不再生效。请改用 [IRemoteAudioTrack.setPlaybackDevice]{@link IRemoteAudioTrack.setPlaybackDevice}。
     *
     * 设置本地播放远端音频流的设备（扬声器）。
     *
     * @param deviceId 设备 ID，可以通过 [[getPlaybackDevices]] 方法获取。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.1.0*
     *
     * > Note:
     * > - As of v4.7.0, this method no longer takes effect. Use [IRemoteAudioTrack.setPlaybackDevice]{@link IRemoteAudioTrack.setPlaybackDevice} instead.
     *
     * Sets the playback device (speaker) for the remote audio stream.
     *
     * @param deviceId The device ID, which can be retrieved by calling [[getPlaybackDevices]].
     */
    setPlaybackDevice(deviceId: string): Promise<void>;
    /**
     * 获取本地音频轨道的质量相关信息。
     *
     * @deprecated
     * 自 v4.1.0 起废弃，请使用 [AgoraRTCClient.getLocalVideoStats]{@link IAgoraRTCClient.getLocalVideoStats} 和 [AgoraRTCClient.getLocalAudioStats]{@link IAgoraRTCClient.getLocalAudioStats}。
     */
    /** @en
     * Gets the statistics of a local audio track.
     *
     * @deprecated from v4.1.0. Use [AgoraRTCClient.getLocalVideoStats]{@link IAgoraRTCClient.getLocalVideoStats} and [AgoraRTCClient.getLocalAudioStats]{@link IAgoraRTCClient.getLocalAudioStats} instead.
     */
    getStats(): LocalAudioTrackStats;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.10.0*
     *
     * 将 `Processor` 注入本地音频轨道。
     * @param processor `Processor` 实例。每个插件均有对应类型的 `Processor`。
     *
     * @returns `Processor` 实例。
     */
    /** @en
     * Inserts a `Processor` to the local audio track.
     *
     * @param processor The `Processor` instance. Each extension has a corresponding type of `Processor`.
     *
     * @returns The `Processor` instance.
     */
    pipe(processor: IAudioProcessor): IAudioProcessor;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.10.0*
     *
     * 取消本地音频轨道上注入的 `Processor`。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.10.0*
     *
     * Removes the `Processor` inserted to the local audio track.
     */
    unpipe(): void;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.10.0*
     *
     * 本地音频轨道上当前媒体处理链路的最终节点。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.10.0*
     *
     * The destination of the current processing pipeline on the local audio track.
     */
    processorDestination: IAudioProcessor;
}

/** @ignore */
/** @en
 * @ignore
 */
export declare interface ILocalDataChannel extends IDataChannel {
    send(data: ArrayBuffer): void;
    /** @internal */
    on(event: "open", listener: typeof event_user_datachannel_open): void;
    /** @internal */
    on(event: "close", listener: typeof event_user_datachannel_close): void;
    /** @internal */
    on(event: "error", listener: typeof event_user_datachannel_error): void;
}

/**
 * `LocalTrack` 为本地轨道的基础类，为音频轨道 [LocalAudioTrack]{@link ILocalAudioTrack} 和视频轨道 [LocalVideoTrack]{@link ILocalVideoTrack} 提供一些公共的方法。
 *
 */
/** @en
 * `LocalTrack` is the basic interface for local tracks, providing public methods for [LocalAudioTrack]{@link ILocalAudioTrack} and [LocalVideoTrack]{@link ILocalVideoTrack}.
 */
export declare interface ILocalTrack extends ITrack {
    /**
     * @param event 事件名称。
     * @param listener 详见 [track-updated]{@link event_track_updated}。
     */
    /** @en
     * @param event The event name.
     * @param listener See [track-updated]{@link event_track_updated}.
     */
    on(event: "track-updated", listener: typeof event_track_updated): void;
    /**
     * @param event 事件名称。
     * @param listener 详见 [track-ended]{@link event_track_ended}。
     */
    /** @en
     * @param event The event name.
     * @param listener See [track-ended]{@link event_track_ended}.
     */
    on(event: "track-ended", listener: typeof event_track_ended): void;
    /**
     * 添加事件监听器。
     * @param event 事件名称。
     * @param listener 详见 [ILocalTrack.transceiver-updated]{@link event_transceiver_updated}。
     */
    /** @en
     * Adds an event listener.
     * @param event The event name.
     * @param listener See [ILocalTrack.transceiver-updated]{@link event_transceiver_updated}.
     */
    on(event: "transceiver-updated", listener: typeof event_transceiver_updated): void;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.0.0*
     *
     * 启用/禁用该轨道。
     *
     * 轨道禁用后，播放和发布都将被停止。
     *
     * > - 禁用轨道不会触发 [LocalTrack.on("track-ended")]{@link event_track_ended} 事件。
     * > - 如果该轨道已发布，禁用轨道后，远端会触发 [user-unpublished]{@link IAgoraRTCClient.event_user_unpublished} 事件。重新启用后，远端会触发 [user-published]{@link IAgoraRTCClient.event_user_published} 事件。
     * > - `setEnabled` 和 `setMuted` 不能同时调用。
     *
     * @param enabled 是否启用该轨道:
     * - `true`: 启用该轨道。
     * - `false`: 禁用该轨道。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.0.0*
     *
     * Enables/Disables the track.
     *
     * After a track is disabled, the SDK stops playing and publishing the track.
     *
     * > - Disabling a track does not trigger the [LocalTrack.on("track-ended")]{@link event_track_ended} event.
     * > - If a track is published, disabling this track triggers the [user-unpublished]{@link IAgoraRTCClient.event_user_unpublished} event on the remote client, and re-enabling this track triggers the [user-published]{@link IAgoraRTCClient.event_user_published} event.
     * > - Do not call `setEnabled` and `setMuted` together.
     *
     * @param enabled Whether to enable the track:
     * - `true`: Enable the track.
     * - `false`: Disable the track.
     */
    setEnabled(enabled: boolean): Promise<void>;
    /**
     * @deprecated
     * 自 v4.1.0 起废弃，请使用 [AgoraRTCClient.getLocalVideoStats]{@link IAgoraRTCClient.getLocalVideoStats} 和 [AgoraRTCClient.getLocalAudioStats]{@link IAgoraRTCClient.getLocalAudioStats}。
     *
     * 获取本地轨道相关信息。
     *
     * > 注意事项：获取本地视频轨道数据时，iOS 上无法获取到 `encodeDelay` 字段。
     */
    /** @en
     * @deprecated from v4.1.0. Use [AgoraRTCClient.getLocalVideoStats]{@link IAgoraRTCClient.getLocalVideoStats} and [AgoraRTCClient.getLocalAudioStats]{@link IAgoraRTCClient.getLocalAudioStats} instead.
     *
     * Gets the statistics of a local track.
     *
     * > Note: When getting the statistics of a local video track, you cannot get the `encodeDelay` property on iOS.
     */
    getStats(): LocalVideoTrackStats | LocalAudioTrackStats;
    /**
     * 获取本地轨道的来源描述。
     *
     * @return 可能返回以下值：
     * - 如果是通过 `createMicrophoneAudioTrack` 或 `createCameraVideoTrack` 创建的轨道，返回 [MediaDeviceInfo.label](https://developer.mozilla.org/en-US/docs/Web/API/MediaDeviceInfo/label) 字段。
     * - 如果是通过 `createScreenVideoTrack` 创建的轨道，返回屏幕共享的 `sourceId`。
     * - 如果是 `createCustomAudioTrack` 或 `createCustomVideoTrack` 创建的轨道，返回 [MediaStreamTrack.label](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack/label) 字段。
     */
    /** @en
     * Gets the label of a local track.
     *
     * @return The label that the SDK returns may include:
     * - The [MediaDeviceInfo.label](https://developer.mozilla.org/en-US/docs/Web/API/MediaDeviceInfo/label) property, if the track is created by calling `createMicrophoneAudioTrack` or `createCameraVideoTrack`.
     * - The `sourceId` property, if the track is created by calling `createScreenVideoTrack`.
     * - The [MediaStreamTrack.label](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack/label) property, if the track is created by calling `createCustomAudioTrack` or `createCustomVideoTrack`.
     */
    getTrackLabel(): string;
    /**
     * 发送或暂停发送该轨道的媒体数据。
     *
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.6.0*
     *
     * 如果该轨道已发布，调用 `setMuted(true)` 后，远端会触发 [user-unpublished]{@link IAgoraRTCClient.event_user_unpublished} 事件。再调用 `setMuted(false)` 后，远端会触发 [user-published]{@link IAgoraRTCClient.event_user_published} 事件。
     *
     * > - 与 {@link setEnabled} 相比，调用该方法响应速度更快且不影响视频采集状态，详见 [setEnabled 和 setMuted 有什么区别？](http://doc.shengwang.cn/faq/integration-issues/diff-setenabled-setmuted)。
     * > - `setEnabled` 和 `setMuted` 不能同时调用。
     *
     * @param muted 是否发送该轨道的媒体数据:
     * - `true`: 暂停发送该轨道的媒体数据。
     * - `false`: 恢复发送该轨道的媒体数据。
     */
    /** @en
     * Sends or stops sending the media data of the track.
     *
     * @since
     * <br>&emsp;&emsp;&emsp;*4.6.0*
     *
     * If the track is published, a successful call of `setMuted(true)` triggers the [user-unpublished]{@link IAgoraRTCClient.event_user_unpublished} event on the remote client, and a successful call of `setMuted(false)` triggers the [user-published]{@link IAgoraRTCClient.event_user_published} event.
     *
     * > - Calling `setMuted(true)` does not stop capturing audio or video and takes shorter time to take effect than [[setEnabled]]. For details, see [What are the differences between setEnabled and setMuted?](https://docs.agora.io/en/interactive-live-streaming/develop/product-workflow?platform=web#setenabled-and-setmuted).
     * > - Do not call `setEnabled` and `setMuted` together.
     *
     * @param muted Whether to stop sending the media data of the track:
     * - `true`: Stop sending the media data of the track.
     * - `false`: Resume sending the media data of the track.
     */
    setMuted(muted: boolean): Promise<void>;
    /**
     * 关闭本地轨道，并释放相关采集设备。
     *
     * 一旦本地轨道被关闭，就无法再次使用。如需再次使用本地轨道，需要重新创建。
     */
    /** @en
     * Closes a local track and releases the audio and video resources that it occupies.
     *
     * Once you close a local track, you can no longer reuse it.
     */
    close(): void;
    /**
     * 本地媒体轨道当前的静音状态。
     */
    muted: boolean;
    /**
     * 本地轨道当前的启用状态。
     */
    enabled: boolean;
}

/**
 * `LocalVideoTrack` 为本地视频轨道的基础类，提供了本地视频轨道的主要功能。
 *
 * 你可以通过 [AgoraRTC.createCustomVideoTrack]{@link IAgoraRTC.createCustomVideoTrack} 或者 [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack} 来获取 `LocalVideoTrack`。
 *
 * 通过摄像头采集到的本地视频轨道对象 [CameraVideoTrack]{@link ICameraVideoTrack} 继承于此接口，并在本接口的基础上提供了一些摄像头独有的功能。
 */
/** @en
 * `LocalVideoTrack` is the basic interface for local video tracks, providing the main methods for local video tracks.
 *
 * You can get create a local video track by calling [AgoraRTC.createCustomVideoTrack]{@link IAgoraRTC.createCustomVideoTrack} or [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack} method.
 *
 * Inherited from `LocalVideoTrack`, [CameraVideoTrack]{@link ICameraVideoTrack} is an interface for the video captured by a local camera and adds several camera-related functions.
 */
export declare interface ILocalVideoTrack extends ILocalTrack {
    /**
     * @param event 事件名称。
     * @param listener 详见 [track-updated]{@link event_track_updated}。
     */
    /** @en
     * @param event The event name.
     * @param listener See [track-updated]{@link event_track_updated}.
     */
    on(event: "track-updated", listener: typeof event_track_updated): void;
    /** @internal */
    on(event: "beauty-effect-overload", listener: typeof event_beauty_effect_overload): void;
    /**
     * @param event 事件名称。
     * @param listener 详见 [track-ended]{@link event_track_ended}。
     */
    /** @en
     * @param event The event name.
     * @param listener See [track-ended]{@link event_track_ended}.
     */
    on(event: "track-ended", listener: typeof event_track_ended): void;
    /**
     * @param event 事件名称。
     * @param listener 详见 [video-element-visible-status]{@link event_video_element_visible_status}。
     */
    /** @en
     * @param event The event name.
     * @param listener See [video-element-visible-status]{@link event_video_element_visible_status}.
     */
    on(event: "video-element-visible-status", listener: typeof event_video_element_visible_status): void;
    /**
     * 添加事件监听器。
     * @param event 事件名称。
     * @param listener 详见 [ILocalTrack.transceiver-updated]{@link event_transceiver_updated}。
     */
    /** @en
     * Adds an event listener.
     * @param event The event name.
     * @param listener See [ILocalTrack.transceiver-updated]{@link event_transceiver_updated}.
     */
    on(event: "transceiver-updated", listener: typeof event_transceiver_updated): void;
    /**
     * 在页面上播放远端视频轨道。
     *
     * @param element 指定一个 DOM 元素，SDK 将在这个元素下创建 `<video>` 元素播放视频轨道，支持 2 种类型：
     * - `string`: 指定该 DOM 元素的 ID 值。
     * - `HTMLElement`: 直接传入一个 DOM 元素对象。
     * @param config 设置播放参数（镜像/显示模式）。详见 [[VideoPlayerConfig]]。对于本地视频轨道，镜像模式默认开启。
     */
    /** @en
     * Plays a remote video track on the web page.
     *
     * @param element Specifies a DOM element. The SDK will create a `<video>` element under the specified DOM element to play the video track. You can specify a DOM element in either of the following ways:
     * - `string`: Specify the ID of the DOM element.
     * - `HTMLElement`: Pass a DOM object.
     * @param config Sets the playback configurations, such as display mode and mirror mode. See [[VideoPlayerConfig]]. By default, the SDK enables mirror mode for a local video track.
     */
    play(element: string | HTMLElement, config?: VideoPlayerConfig): void;
    /**
     * 获取本地视频轨道的相关信息。
     *
     * @deprecated
     * 自 v4.1.0 起废弃，请使用 [AgoraRTCClient.getLocalVideoStats]{@link IAgoraRTCClient.getLocalVideoStats} 和 [AgoraRTCClient.getLocalAudioStats]{@link IAgoraRTCClient.getLocalAudioStats}。
     *
     */
    /** @en
     * Gets the statistics of a local video track.
     *
     * @deprecated from v4.1.0. Use [AgoraRTCClient.getLocalVideoStats]{@link IAgoraRTCClient.getLocalVideoStats} and [AgoraRTCClient.getLocalAudioStats]{@link IAgoraRTCClient.getLocalAudioStats} instead.
     */
    getStats(): LocalVideoTrackStats;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.19.0*
     *
     * 复制当前的视频轨道对象，生成一个新的视频轨道。
     *
     * 在视频会议、在线教育等场景中，你可以用该方法实现同一视频流的两路显示，并为每路显示设置不同的分辨率、帧率、码率等参数。
     *
     * @param config 新生成视频轨道的编码配置。你可以通过 [[VideoEncoderConfiguration]] 传入 SDK 内置的编码配置；也可以通过传入一个 [[VideoEncoderConfigurationPreset]] 来自定义视频编码配置。
     * @param cloneTrack 是否克隆当前轨道。默认为 `true`。
     * @returns 新生成的视频轨道。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.19.0*
     *
     * Clones the current video track to create a new video track.
     *
     * In scenarios such as video conferencing and online education, you can use this method to display the same video stream with two sets of display parameters, including resolution, frame rate, and bitrate. For example, you can have one display set to high-definition and the other to low-definition.
     *
     * @param config The encoding configuration for the new video track. You can pass in the SDK's built-in encoding configuration through [[VideoEncoderConfiguration]], or customize the video encoding configuration by passing in a [[VideoEncoderConfigurationPreset]].
     * @param cloneTrack Whether to clone the current track. Default is `true`.
     * @returns The newly generated video track.
     */
    clone(config?: VideoEncoderConfiguration | VideoEncoderConfigurationPreset, cloneTrack?: boolean): ILocalVideoTrack;
    /**
     * @ignore
     *
     * 开启或关闭美颜。
     *
     * > 注意事项：
     * > - 该功能是 SDK 内置的美颜功能，将逐步停止更新。你可以改用升级的美颜插件，详见[使用美颜插件](https://docs.agora.io/cn/Video/beauty_effect_web_ng?platform=Web)。
     * > - 该方法支持以下浏览器：
     * >  - Safari 12 及以上版本。
     * >  - Chrome 65 及以上版本。
     * >  - Firefox 70.0.1 及以上版本。
     * > - 该功能不支持移动端设备。
     * > - 如果开启了双流模式，美颜选项仅对大流生效。
     *
     * @param enabled 是否开启本地美颜功能：
     * - `true`: 开启本地美颜。
     * - `false`: 关闭本地美颜。
     * @param options 美颜效果选项。详见 [[BeautyEffectOptions]]。
     */
    /** @en
     * @ignore
     *
     * Enables/Disables image enhancement and sets the options.
     *
     * > Notes:
     * > - Agora is planning to sunset this built-in image enhancement feature.
     * You can use the upgraded Image Enhancement Extension instead. For details, see
     * [Use the Image Enhancement Extension](https://docs.agora.io/en/Video/beauty_effect_web_ng?platform=Web).
     * > - This method supports the following browsers:
     * >  - Safari 12 or later.
     * >  - Chrome 65 or later.
     * >  - Firefox 70.0.1 or later.
     * > - This function is not supported on mobile devices.
     * > - If you enable dual-stream mode, the image enhancement options only apply to the high-quality video stream.
     *
     * @param enabled Whether to enable image enhancement:
     * - `true`: Enable image enhancement.
     * - `false`: Disable image enhancement.
     * @param options Sets image enhancement options. See [[BeautyEffectOptions]].
     */
    setBeautyEffect(enabled: boolean, options?: BeautyEffectOptions): Promise<void>;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.1.0*
     *
     * 获取当前渲染的视频帧数据。
     *
     * > 只有在播放后才能调用此方法，如果没有调用 [[play]] 则返回空数据。
     *
     * @returns 存储了 RGBA 数据的 `ImageData` 对象。该对象为浏览器原生对象，详见 [ImageData](https://developer.mozilla.org/zh-CN/docs/Web/API/ImageData)。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.1.0*
     *
     * Gets the data of the video frame being rendered.
     *
     * > You should call this method after calling [[play]]. Otherwise, the method call returns null.
     *
     * @returns An `ImageData` object that stores RGBA data. `ImageData` is a web API supported by the browser. For details, see [ImageData](https://developer.mozilla.org/en-US/docs/Web/API/ImageData).
     */
    getCurrentFrameData(): ImageData;
    /**
     * @ignore
     */
    /** @en
     *
     * @ignore
     */
    getCurrentFrameImage(imageType: string, quality: number): Promise<ImageTypedData>;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.2.0*
     *
     * 设置视频传输优化模式。
     *
     * 你可以在视频通话、视频直播或屏幕共享过程中调用此方法动态调整视频的传输优化模式。例如你想要把屏幕共享内容从演示文稿切换为视频时，你可以将传输优化模式从 `"detail"` 切换为 `"motion"`，确保视频画面在网络波动时不会出现卡顿。
     *
     * > 注意事项：该方法只支持 Chrome 浏览器。
     *
     * @param mode 视频传输优化模式：
     * - `"balanced"`: 使用默认的传输优化模式：
     *   - 对于屏幕共享视频流，SDK 默认的优化策略为清晰优先。
     *   - 对于其他视频流，SDK 默认的优化策略为兼顾清晰和流畅，弱网条件下，帧率和分辨率都会被调整。
     * - `"detail"`: 清晰优先。
     *   - 浏览器会自动根据你的采集分辨率和帧率设定一个最小码率。即使遭遇网络波动，发送码率也不会低于这个值，从而确保清晰的视频画面。
     *   - 大部分情况下，SDK 不会降低发送分辨率，但是可能会降低帧率。
     * - `"motion"`: 自 4.21.0 开始，浏览器默认开启流畅优先模式。
     *   - 浏览器不会启用最小码率策略。遭遇网络波动时，发送端会降低码率来确保接收端的视频画面不会出现中断和卡顿。
     *   - 大部分情况下，浏览器不会降低帧率，但是可能会降低发送分辨率。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.2.0*
     *
     * Sets the video transmission optimization mode.
     *
     * You can call this method during a video call, a live streaming or screen sharing to dynamically change the optimization mode. For example, during the screen sharing, before you change the shared content from text to video, you can change the optimization mode from `"detail"` to `"motion"` to ensure smoothness in poor network conditions.
     *
     * > Note: This method supports Chrome only.
     *
     * @param mode The video transmission optimization mode:
     * - `"balanced"`: Uses the default optimization mode.
     *   - For a screen-sharing video track, the default transmission optimization strategy is to prioritizes clarity.
     *   - For the other types of video tracks, the browser may reduce the frame rate or the sending resolution in poor network conditions.
     * - `"detail"`: Prioritizes video quality.
     *   - The browser ensures high-quality images by automatically calculating a minimum bitrate based on the capturing resolution and frame rate. No matter how poor the network condition is, the sending bitrate will never be lower than the minimum value.
     *   - In most cases, the browser does not reduce the sending resolution, but may reduce the frame rate.
     * -  `"motion"`: Since v4.21.0, the browser prioritizes video smoothness.
     *   - In poor network conditions, the browser reduces the sending bitrate to minimize video freezes.
     *   - In most cases, the browser does not reduce the frame rate, but may reduce the sending resolution.
     */
    setOptimizationMode(mode: "balanced" | "motion" | "detail"): Promise<void>;
    /**
     * * @自从
     * <br>&emsp;&emsp;&emsp;*4.8.0*
     *
     * 获取 HTML `<video>` 标签的可见状态和不可见原因。
     *
     * 调用 `localVideoTrack.play` 后 SDK 会创建 [HTML `<video>`](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video) 标签用于播放视频轨道。当 `localVideoTrack.isPlaying` 为 `true` 却看不到图像时，你可调用该方法进行问题排查。
     *
     * @returns [[CheckVideoVisibleResult]] 对象。如果该方法返回 `undefined`，可能为以下原因：
     * - `localVideoTrack.isPlaying` 为 `false`。
     * - `<video>` 标签不存在。
     * - `<video>` 标签不是通过 `play` 方法调用产生的。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.8.0*
     *
     * Gets the visibility of the `<video>` HTML tag.
     *
     * After you call `localVideoTrack.play`, the SDK creates an [`<video>`](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video) tag for playing video tracks. When `localVideoTrack.isPlaying` is `true` but you cannot see any video, call this method to check whether the `<video>` tag is visible or not and learn the reason when the `<video>` tag is invisible.
     *
     * @returns The [[CheckVideoVisibleResult]] object. If this method returns `undefined`, it may be due to the following reasons:
     * - `localVideoTrack.isPlaying` is `false`.
     * - The `<video>` tag does not exist.
     * - The `<video>` tag is not created by calling the `play` method.
     */
    getVideoElementVisibleStatus(): CheckVideoVisibleResult | undefined;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.10.0*
     *
     * 将 `Processor` 注入本地视频轨道。
     * @param processor `Processor` 实例。每个插件均有对应类型的 `Processor`。
     *
     * @returns `Processor` 实例。
     */
    /** @en
     * Inserts a `Processor` to the local video track.
     *
     * @param processor The `Processor` instance. Each extension has a corresponding type of `Processor`.
     *
     * @returns The `Processor` instance.
     */
    pipe(processor: IBaseProcessor): IBaseProcessor;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.10.0*
     *
     * 取消本地视频轨道上注入的 `Processor`。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.10.0*
     *
     * Removes the `Processor` inserted to the local video track.
     */
    unpipe(): void;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.10.0*
     *
     * 本地视频轨道上当前媒体处理链路的最终节点。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.10.0*
     *
     * The destination of the current processing pipeline on the local video track.
     */
    processorDestination: IBaseProcessor;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.17.0*
     *
     * 替换本地视频轨道。
     *
     * 该方法在本地视频流发布前或发布后都可以调用：
     * - 在发流前调用，新的视频轨道会在本地播放。
     * - 在发流后调用，远端会接收到新的视频轨道。
     *
     * 新的视频轨道可以通过 {@link ILocalVideoTrack.getMediaStreamTrack} 或者 `mediaStream.getVideoTracks` 方法获取。你可以选择是否停止被替换的视频轨道。
     *
     * > 注意事项：
     * > - 支持 Chrome 65+，Safari 以及最新版 Firefox 浏览器。
     * > - 部分移动设备上该方法可能不生效。
     * > - 建议在相同类型、相同编码配置的视频轨道之间进行替换，原因如下：
     * >   - 如果视频轨道类型不同，例如用 `ScreenVideoTrack` 对象替换 `CameraVideoTrack` 对象，由于 `CameraVideoTrack` 默认开启镜像（详见 {@link VideoPlayerConfig.mirror}），会导致画面发生镜像翻转。
     * >   - 如果编码配置（`encoderConfig`）不同，可能导致实际发送的分辨率或帧率不准。
     * > - 订阅端无法知悉视频轨道被替换。
     * > - 如果需要切换媒体输入设备，推荐使用 {@link ICameraVideoTrack.setDevice} 方法。
     *
     * **示例代码**
     * ```javascript
     * // 原有的本地视频轨道
     * const localVideoTrack = await AgoraRTC.createCameraVideoTrack();
     * // 获取新的视频轨道（方式一）
     * const newTrack = localVideoTrack.getMediaStreamTrack();
     * // 获取新的视频轨道（方式二）
     * const newTrack = await navigator.mediaDevices.getUserMedia({audio: true, video: true}).then(mediaStream => mediaStream.getVideoTracks()[0]);
     * // 替换本地视频轨道，并且停止原有的视频轨道
     * await localVideoTrack.replaceTrack(newTrack, true);
     * ```
     *
     * @param track 新的视频轨道。需要传入一个 [MediaStreamTrack](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack) 对象。
     * @param stopOldTrack 是否停止原有的视频轨道：
     * - 停止原有的视频轨道。
     * - 保留原有的视频轨道。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.17.0*
     *
     * Replaces the local video track.
     *
     * You can call this method before or after publishing the local video stream:
     * - If you call this method before publishing, the new video track is played locally.
     * - If you call this method after publishing, the new video track is received by the remote user.
     *
     * The new video track can be retrieved by the {@link ILocalVideoTrack.getMediaStreamTrack} or `mediaStream.getVideoTracks` method. You can choose to either stop or retain the replaced track.
     *
     * > Notes:
     * > - This method supports Chrome 65+, Safari, and the latest Firefox.
     * > - This method might not take effect on some mobile devices.
     * > - Agora recommends switching between video tracks that are of the same type and have the same encoder configurations for the following reasons:
     * >   - If the video track types are different, such as replacing a `CameraVideoTrack` object with a `ScreenVideoTrack` object, the video is flipped horizontally due to the mirror effect enabled by default on `CameraVideoTrack` (see {@link VideoPlayerConfig.mirror} for details).
     * >   - If the encoder configurations (`encoderConfig`) are different, the actual sending resolution or frame rate might be different from what you set.
     * > - The subscriber will not be notified if the track gets replaced.
     * > - To switch the media input devices, Agora recommends using {@link ICameraVideoTrack.setDevice}.
     *
     * **Example**
     * ```javascript
     * // Current video track
     * const localVideoTrack = await AgoraRTC.createCameraVideoTrack();
     * // Gets the new video track (option one)
     * const newTrack = localVideoTrack.getMediaStreamTrack();
     * // Gets the new video track (option two)
     * const newTrack = await navigator.mediaDevices.getUserMedia({audio: true, video: true}).then(mediaStream => mediaStream.getVideoTracks()[0]);
     * // Replaces and stops the current video track
     * await localVideoTrack.replaceTrack(newTrack, true);
     * ```
     * @param track The new video track, which is a [MediaStreamTrack](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack) object.
     * @param stopOldTrack Whether to stop the old video track:
     *  - true: Stops the old video track.
     *  - false: Retains the old video track.
     */
    replaceTrack(track: MediaStreamTrack, stopOldTrack: boolean): Promise<void>;
    /**
     * 设置视频的编码参数，包括分辨率、帧率、码率。
     *
     * @param config 你可以传入 SDK 预设的视频编码配置 [[VideoEncoderConfigurationPreset]]，也可以传入自定义的 [[VideoEncoderConfiguration]] 对象。
     */
    /** @en
     * Sets the video encoder configurations, such as resolution, frame rate, and bitrate.
     *
     * @param config The video encoder configurations. You can pass either [[VideoEncoderConfigurationPreset]] or a customized [[VideoEncoderConfiguration]] object.
     */
    setEncoderConfiguration(config: VideoEncoderConfiguration | VideoEncoderConfigurationPreset): Promise<void>;
    /**
     * 在 H.264 视频流中添加 SEI 数据。
     *
     * @param sei SEI 数据。
     */
    /** @en
     * Add the SEI data to the H.264 video stream.
     *
     * @param config SEI data.
     */
    sendSeiData(sei: Uint8Array): void;
}

export declare interface ImageTypedData {
    buffer: Uint8Array;
    width: number;
    height: number;
}

/**
 * 本地麦克风音频轨道，继承于 [LocalAudioTrack]{@link ILocalAudioTrack}，在此之上提供了更换采集设备的功能。
 *
 * 可以通过 [AgoraRTC.createMicrophoneAudioTrack]{@link IAgoraRTC.createMicrophoneAudioTrack} 来获取。
 */
/** @en
 * Inherited from [LocalAudioTrack]{@link ILocalAudioTrack}, `MicrophoneAudioTrack` is an interface for the audio sampled by a local microphone and adds several functions such as switching devices.
 *
 * You can create a local microphone audio track by calling [AgoraRTC.createMicrophoneAudioTrack]{@link IAgoraRTC.createMicrophoneAudioTrack}.
 */
export declare interface IMicrophoneAudioTrack extends ILocalAudioTrack {
    /**
     * 设置负责采集音频数据的麦克风设备。
     *
     * > 在发布前和发布后均可调用此方法。
     *
     * @param deviceId 指定设备的设备 Id，可以通过 [AgoraRTC.getMicrophones]{@link IAgoraRTC.getMicrophones} 获取。
     */
    /** @en
     * Sets the device for sampling audio.
     *
     * > You can call the method either before or after publishing an audio track.
     *
     * @param deviceId The ID of the specified device. You can get the `deviceId` by calling [AgoraRTC.getMicrophones]{@link IAgoraRTC.getMicrophones}.
     */
    setDevice(deviceId: string): Promise<void>;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.0.0*
     *
     * 启用/禁用该轨道。
     *
     * 轨道禁用后，播放和发布都将被停止。
     *
     * > - 禁用轨道不会触发 [LocalTrack.on("track-ended")]{@link event_track_ended} 事件。
     * > - 如果该轨道已发布，禁用轨道后，远端会触发 [user-unpublished]{@link IAgoraRTCClient.event_user_unpublished} 事件。重新启用后，远端会触发 [user-published]{@link IAgoraRTCClient.event_user_published} 事件。
     * > - `setEnabled` 和 `setMuted` 不能同时调用。
     *
     * @param enabled 是否启用该轨道:
     * - `true`: 启用该轨道.
     * - `false`: 禁用该轨道.
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.0.0*
     *
     * Enables/Disables the track.
     *
     * After a track is disabled, the SDK stops playing and publishing the track.
     *
     * > - Disabling a track does not trigger the [LocalTrack.on("track-ended")]{@link event_track_ended} event.
     * > - If a track is published, disabling this track triggers the [user-unpublished]{@link IAgoraRTCClient.event_user_unpublished} event on the remote client, and re-enabling this track triggers the [user-published]{@link IAgoraRTCClient.event_user_published} event.
     * > - Do not call `setEnabled` and `setMuted` together.
     *
     * @param enabled Whether to enable the track:
     * - `true`: Enable the track.
     * - `false`: Disable the track.
     */
    setEnabled(enabled: boolean): Promise<void>;
}

export declare function interceptLocalAudioFrame(sender: RTCRtpSender): Promise<void>;

export declare function interceptLocalVideoFrame(sender: RTCRtpSender, localVideoTrack: LocalVideoTrack): Promise<void>;

export declare function interceptRemoteAudioFrame(receiver: RTCRtpReceiver): Promise<void>;

export declare function interceptRemoteVideoFrame(receiver: RTCRtpReceiver, options?: InterceptVideoFrameOptions): Promise<void>;

declare interface InterceptVideoFrameOptions {
    onSei?: (sei: Uint8Array) => void;
}

/**
 * 远端音频轨道，调用 [subscribe]{@link IAgoraRTCClient.subscribe} 后通过 [AgoraRTCRemoteUser.audioTrack]{@link IAgoraRTCRemoteUser.audioTrack} 对象获取
 */
/** @en
 * `RemoteAudioTrack` is the basic interface for the remote audio track.
 *
 * You can get create a remote audio track by the [AgoraRTCRemoteUser.audioTrack]{@link IAgoraRTCRemoteUser.audioTrack} object after calling [subscribe]{@link IAgoraRTCClient.subscribe}.
 */
export declare interface IRemoteAudioTrack extends IRemoteTrack {
    /**
     * 获取远端音频轨道相关信息。
     *
     * **不推荐使用** 请使用 [AgoraRTCClient.getRemoteAudioStats]{@link IAgoraRTCClient.getRemoteAudioStats} 获取远端音频轨道相关信息。
     *
     * @return 一个 [[RemoteAudioTrackStats]] 对象。
     */
    /** @en
     * Gets the statistics of a remote audio track.
     *
     * @return An [[RemoteAudioTrackStats]] object.
     */
    getStats(): RemoteAudioTrackStats;
    /**
     * 播放远端音频轨道。
     *
     * > 播放音频时 SDK 不会创建任何 DOM 元素，所以无需像视频一样提供 DOM 元素。
     */
    /** @en
     * Plays a remote audio track.
     *
     * > When playing the audio track, you do not need to pass any DOM element.
     */
    play(): void;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.1.0*
     *
     * 设置音频播放设备，比如扬声器。
     *
     * > 该方法只支持桌面端的 Chrome 和 Edge 浏览器，其他浏览器调用将会抛出 `NOT_SUPPORTED` 错误。
     * @param deviceId 设备 ID，可以通过 [[getPlaybackDevices]] 方法获取。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.1.0*
     *
     * Sets the audio playback device, for example, the speaker.
     *
     * > This method supports Chrome and Edge on desktop devices only. Other browsers throw a `NOT_SUPPORTED` error when calling this method.
     * @param deviceId Device ID, which can be retrieved by calling [[getPlaybackDevices]].
     */
    setPlaybackDevice(deviceId: string): Promise<void>;
    /**
     * 设置原始音频数据（PCM）回调。
     *
     * 设置成功后，SDK 会不断地将远端音频轨道的音频帧以 [AudioBuffer](https://developer.mozilla.org/en-US/docs/Web/API/AudioBuffer) 的形式通过回调返回。
     *
     * > 你可以通过 `frameSize` 来设置每次回调中音频帧的大小。该设置也会影响回调的间隔，`frameSize` 越大，每次回调的音频数据越多，回调间隔越长。
     *
     * ```js
     * track.setAudioFrameCallback((buffer) => {
     *   for (let channel = 0; channel < buffer.numberOfChannels; channel += 1) {
     *     // Float32Array with PCM data
     *     const currentChannelData = buffer.getChannelData(channel);
     *     console.log("PCM data in channel", channel, currentChannelData);
     *   }
     * }, 2048);
     *
     * // ....
     * // Stop getting the raw audio data
     * track.setAudioFrameCallback(null);
     * ```
     *
     * @param audioFrameCallback 用于接收 [AudioBuffer](https://developer.mozilla.org/en-US/docs/Web/API/AudioBuffer) 的回调函数。设为 `null` 后，SDK 就会停止获取音频原始数据。
     * @param frameSize 每次回调的 `AudioBuffer` 中每个声道包含的采样数据个数，只能设为下列值：256, 512, 1024, 2048, 4096, 8192, 16384。默认为 4096。
     */
    /** @en
     * Sets the callback for getting raw audio data in PCM format.
     *
     * After you successfully set the callback, the SDK constantly returns the audio frames of a remote audio track in this callback by using [AudioBuffer](https://developer.mozilla.org/en-US/docs/Web/API/AudioBuffer).
     *
     * > You can set the `frameSize` parameter to determine the frame size in each callback, which affects the interval between the callbacks. The larger the frame size, the longer the interval between them.
     *
     * ```js
     * track.setAudioFrameCallback((buffer) => {
     *   for (let channel = 0; channel < buffer.numberOfChannels; channel += 1) {
     *     // Float32Array with PCM data
     *     const currentChannelData = buffer.getChannelData(channel);
     *     console.log("PCM data in channel", channel, currentChannelData);
     *   }
     * }, 2048);
     *
     * // ....
     * // Stop getting the raw audio data
     * track.setAudioFrameCallback(null);
     * ```
     *
     * @param audioFrameCallback The callback function for receiving the [AudioBuffer](https://developer.mozilla.org/en-US/docs/Web/API/AudioBuffer) object. If you set `audioBufferCallback` as `null`, the SDK stops getting raw audio data.
     * @param frameSize The number of samples of each audio channel that an `AudioBuffer` object contains. You can set `frameSize` as 256, 512, 1024, 2048, 4096, 8192, or 16384. The default value is 4096.
     */
    setAudioFrameCallback(audioFrameCallback: null | ((buffer: AudioBuffer) => void), frameSize?: number): void;
    /**
     * 设置远端音频轨道的播放音量
     * @param volume 音量值，范围 [0, 100]，0 代表静音，100 代表原始音量。
     */
    /** @en
     * Sets the volume of a remote audio track.
     *
     * @param volume The volume. The value ranges from 0 (mute) to 100 (maximum). A value of 100 is the current volume.
     */
    setVolume(volume: number): void;
    /**
     * 获取远端音频轨道的音量等级。
     *
     * @returns 音量等级值，范围 [0, 1]，1 代表理论最大音量。通常该值大于 0.6 代表用户在持续说话。
     */
    /** @en
     * Gets the audio level of a remote audio track.
     *
     * @returns The audio level. The value range is [0,1]. 1 is the highest audio level.
     * Usually a user with audio level above 0.6 is a speaking user.
     */
    getVolumeLevel(): number;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.13.0*
     *
     * 将 `Processor` 注入远端音频轨道。
     *
     * @param processor `Processor` 实例。每个插件均有对应类型的 `Processor`。
     *
     * @returns `Processor` 实例。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.13.0*
     *
     * Inserts a `Processor` to the remote audio track.
     *
     * @param processor The `Processor` instance. Each extension has a corresponding type of `Processor`.
     *
     * @returns The `Processor` instance.
     */
    pipe(processor: IAudioProcessor): IAudioProcessor;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.13.0*
     *
     * 取消远端音频轨道上注入的 `Processor`。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.13.0*
     *
     * Removes the `Processor` inserted to the remote audio track.
     */
    unpipe(): void;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.13.0*
     *
     * 远端音频轨道上当前媒体处理链路的最终节点。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.13.0*
     *
     * The destination of the current processing pipeline on the remote audio track.
     */
    processorDestination: IAudioProcessor;
}

/** @ignore */
/** @en
 * @ignore
 */
export declare interface IRemoteDataChannel extends IDataChannel {
    /** @internal */
    on(event: "open", listener: typeof event_user_datachannel_open): void;
    /** @internal */
    on(event: "close", listener: typeof event_user_datachannel_close): void;
    /** @internal */
    on(event: "error", listener: typeof event_user_datachannel_error): void;
    /** @internal */
    on(event: "message", listener: typeof event_user_datachannel_message): void;
}

/**
 * 远端轨道对象的基础类，为远端音频轨道 [RemoteAudioTrack]{@link IRemoteAudioTrack} 和远端视频轨道 [RemoteVideoTrack]{@link IRemoteVideoTrack} 提供一些公共的方法。
 */
/** @en
 * `RemoteTrack` is the basic interface for remote tracks, providing public methods for [RemoteAudioTrack]{@link IRemoteAudioTrack} and [RemoteVideoTrack]{@link IRemoteVideoTrack}.
 */
export declare interface IRemoteTrack extends ITrack {
    /**
     * @param event 事件名称。
     * @param listener 详见 [track-updated]{@link event_track_updated}。
     */
    /** @en
     * @param event The event name.
     * @param listener See [track-updated]{@link event_track_updated}.
     */
    on(event: "track-updated", listener: typeof event_track_updated): void;
    /**
     * @param event 事件名称。
     * @param listener 详见 [first-frame-decoded]{@link event_first_frame_decoded}。
     */
    /** @en
     * @param event The event name.
     * @param listener See [first-frame-decoded]{@link event_first_frame_decoded}.
     */
    on(event: "first-frame-decoded", listener: typeof event_first_frame_decoded): void;
    /**
     * 添加事件监听器。
     * @param event 事件名称。
     * @param listener 详见 [IRemoteTrack.transceiver-updated]{@link event_transceiver_updated_2}。
     */
    /** @en
     * Adds an event listener.
     * @param event The event name.
     * @param listener See [IRemoteTrack.transceiver-updated]{@link event_transceiver_updated_2}.
     */
    on(event: "transceiver-updated", listener: typeof event_transceiver_updated_2): void;
    /**
     * 获取发布远端轨道的远端用户 ID。
     *
     * @return 远端轨道对应的 `uid`。
     */
    /** @en
     * Gets the `uid` of the remote user who publishes the remote track.
     *
     * @return The `uid` of the remote user.
     */
    getUserId(): UID;
    /**
     * 获取远端轨道相关信息。
     *
     * @deprecated
     *
     * 自 v4.1.0 起废弃，请使用 [AgoraRTCClient.getRemoteVideoStats]{@link IAgoraRTCClient.getRemoteVideoStats} 和 [AgoraRTCClient.getRemoteAudioStats]{@link IAgoraRTCClient.getRemoteAudioStats}。
     * @return [[RemoteAudioTrackStats]] 或 [[RemoteVideoTrackStats]] 对象。
     */
    /** @en
     * Gets the statistics of a remote track.
     *
     * @deprecated from v4.1.0. Use [AgoraRTCClient.getRemoteVideoStats]{@link IAgoraRTCClient.getRemoteVideoStats} and [AgoraRTCClient.getRemoteAudioStats]{@link IAgoraRTCClient.getRemoteAudioStats} instead.
     * @return An [[RemoteAudioTrackStats]] or [[RemoteVideoTrackStats]] object.
     */
    getStats(): RemoteAudioTrackStats | RemoteVideoTrackStats;
}

/**
 * 远端视频轨道。
 *
 * 调用 [subscribe]{@link IAgoraRTCClient.subscribe} 后通过 [AgoraRTCRemoteUser.videoTrack]{@link IAgoraRTCRemoteUser.videoTrack} 对象获取。
 */
/** @en
 * `RemoteVideoTrack` is the basic interface for the remote video track.
 *
 * You can get create a remote video track by the [AgoraRTCRemoteUser.videoTrack]{@link IAgoraRTCRemoteUser.videoTrack} object after calling [subscribe]{@link IAgoraRTCClient.subscribe}.
 */
export declare interface IRemoteVideoTrack extends IRemoteTrack {
    /**
     * @param event 事件名称。
     * @param listener 详见 [track-updated]{@link event_track_updated}。
     */
    /** @en
     * @param event The event name.
     * @param listener See [track-updated]{@link event_track_updated}.
     */
    on(event: "track-updated", listener: typeof event_track_updated): void;
    /**
     * @param event 事件名称。
     * @param listener 详见 [video-state-changed]{@link event_video_state_changed}。
     */
    /** @en
     * @param event The event name.
     * @param listener See [video-state-changed]{@link event_video_state_changed}.
     */
    on(event: "video-state-changed", listener: typeof event_video_state_changed): void;
    /**
     * @param event 事件名称。
     * @param listener 详见 [first-frame-decoded]{@link event_first_frame_decoded}。
     */
    /** @en
     * @param event The event name.
     * @param listener See [first-frame-decoded]{@link event_first_frame_decoded}.
     */
    on(event: "first-frame-decoded", listener: typeof event_first_frame_decoded): void;
    /**
     * @param event 事件名称。
     * @param listener 详见 [video-element-visible-status]{@link event_video_element_visible_status_2}。
     */
    /** @en
     * @param event The event name.
     * @param listener See [video-element-visible-status]{@link event_video_element_visible_status_2}.
     */
    on(event: "video-element-visible-status", listener: typeof event_video_element_visible_status_2): void;
    /**
     * 添加事件监听器。
     * @param event 事件名称。
     * @param listener 详见 [IRemoteTrack.transceiver-updated]{@link event_transceiver_updated_2}。
     */
    /** @en
     * Adds an event listener.
     * @param event The event name.
     * @param listener See [IRemoteTrack.transceiver-updated]{@link event_transceiver_updated_2}.
     */
    on(event: "transceiver-updated", listener: typeof event_transceiver_updated_2): void;
    /**
     * 添加事件监听器。
     * @param event 事件名称。
     * @param listener 详见 [ILocalTrack.sei-received]{@link event_sei_received}。
     */
    /** @en
     * Adds an event listener.
     * @param event The event name.
     * @param listener See [ILocalTrack.sei-received]{@link event_sei_received}.
     */
    on(event: "sei-received", listener: typeof event_sei_received): void;
    /**
     * 获取远端视频轨道相关信息。
     *
     * **不推荐使用** 请使用 [AgoraRTCClient.getRemoteVideoStats]{@link IAgoraRTCClient.getRemoteVideoStats} 获取远端视频轨道相关信息。
     *
     * @return 一个 [[RemoteVideoTrackStats]] 对象。
     */
    /** @en
     * Gets the statistics of a remote video track.
     *
     * @return An [[RemoteVideoTrackStats]] object。
     */
    getStats(): RemoteVideoTrackStats;
    /**
     * 在页面上播放本地视频轨道。
     * @param element 指定一个 DOM 元素，SDK 将在这个元素下创建 `<video>` 元素播放视频轨道，支持 2 种类型：
     * - `string`: 指定该 DOM 元素的 ID 值。
     * - `HTMLElement`: 直接传入一个 DOM 元素对象。
     * @param config 设置播放参数（镜像/显示模式）。详见 [[VideoPlayerConfig]]。对于本地视频轨道，镜像模式默认开启。
     */
    /** @en
     * Plays a remote video track on the web page.
     *
     * @param element Specifies a DOM element. The SDK will create a `<video>` element under the specified DOM element to play the video track. You can specify a DOM element in either of following ways:
     * - `string`: Specify the ID of the DOM element.
     * - `HTMLElement`: Pass a DOM object.
     * @param config Sets the playback configurations, such as display mode and mirror mode. See [[VideoPlayerConfig]]. By default, the SDK enables mirror mode for a local video track.
     */
    play(element: string | HTMLElement, config?: VideoPlayerConfig): void;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.1.0*
     *
     * 获取当前渲染的视频帧数据。
     *
     * > 只有在播放后才能调用此方法，如果没有调用 [[play]] 则返回空数据。
     *
     * @returns 存储了 RGBA 数据的 `ImageData` 对象。该对象为浏览器原生对象，详见 [ImageData](https://developer.mozilla.org/zh-CN/docs/Web/API/ImageData)。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.1.0*
     *
     * Gets the data of the video frame being rendered.
     *
     * > You should call this method after calling [[play]]. Otherwise, the method call returns null.
     *
     * @returns An `ImageData` object that stores RGBA data. `ImageData` is a web API supported by the browser. For details, see [ImageData](https://developer.mozilla.org/en-US/docs/Web/API/ImageData).
     */
    getCurrentFrameData(): ImageData;
    /**
     * * @自从
     * <br>&emsp;&emsp;&emsp;*4.8.0*
     *
     * 获取 HTML `<video>` 标签的可见状态和不可见原因。
     *
     * 调用 `remoteVideoTrack.play` 后 SDK 会创建 [HTML `<video>`](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video) 标签用于播放视频轨道。当 `remoteVideoTrack.isPlaying` 为 `true` 却看不到图像时，你可调用该方法进行问题排查。
     *
     * @return [[CheckVideoVisibleResult]] 对象。如果该方法返回 `undefined`，可能为以下原因：
     * - `remoteVideoTrack.isPlaying` 为 `false`。
     * - `<video>` 标签不存在。
     * - `<video>` 标签不是通过 `play` 方法调用产生的。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.8.0*
     *
     * Gets the visibility of the `<video>` HTML tag.
     *
     * After you call `remoteVideoTrack.play`, the SDK creates an [`<video>`](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video) tag for playing video tracks. When `remoteVideoTrack.isPlaying` is `true` but you cannot see any video, call this method to check whether the `<video>` tag is visible or not and learn the reason when the `<video>` tag is invisible.
     *
     * @returns The [[CheckVideoVisibleResult]] object. If this method returns `undefined`, it may be due to the following reasons:
     * - `remoteVideoTrack.isPlaying` is `false`.
     * - The `<video>` tag does not exist.
     * - The `<video>` tag is not created by calling the `play` method.
     */
    getVideoElementVisibleStatus(): CheckVideoVisibleResult | undefined;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.13.0*
     *
     * 将 `Processor` 注入远端视频轨道。
     *
     * @param processor `Processor` 实例。每个插件均有对应类型的 `Processor`。
     *
     * @returns `Processor` 实例。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.13.0*
     *
     * Inserts a `Processor` to the remote video track.
     *
     * @param processor The `Processor` instance. Each extension has a corresponding type of `Processor`.
     *
     * @returns The `Processor` instance.
     */
    pipe(processor: IBaseProcessor): IBaseProcessor;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.13.0*
     *
     * 取消远端视频轨道上注入的 `Processor`。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.13.0*
     *
     * Removes the `Processor` inserted to the remote video track.
     */
    unpipe(): void;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.13.0*
     *
     * 远端视频轨道上当前媒体处理链路的最终节点。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.13.0*
     *
     * The destination of the current processing pipeline on the remote video track.
     */
    processorDestination: IBaseProcessor;
}

export declare function isAudioEncoderConfiguration(config: AudioEncoderConfiguration): config is AudioEncoderConfiguration;

/**
 * AudioProfilePreset Type Guard
 */
export declare function isAudioEncoderConfigurationOrPreset(profile: any): profile is AudioEncoderConfigurationPreset | AudioEncoderConfiguration;

export declare function isBeautyEffectOptions(options: BeautyEffectOptions): options is BeautyEffectOptions;

export { isElectron }

export declare function isLowStreamParameter(streamParameter: any): streamParameter is LowStreamParameter;

export declare function isPlanB(): boolean;

export declare function isScreenSourceType(mediaSource: any): mediaSource is ScreenSourceType;

export declare function isVideoEncoderConfiguration(config: VideoEncoderConfiguration): config is VideoEncoderConfiguration;

/**
 * VideoProfilePreset Type Guard
 */
export declare function isVideoEncoderConfigurationOrPreset(profile: any): profile is VideoEncoderConfigurationPreset | VideoEncoderConfiguration;

/**
 * @ignore
 */
export declare interface ITrack extends EventEmitter {
    /**
     * 媒体轨道的类型：
     * - `"audio"`: 音频轨道。
     * - `"video"`: 视频轨道。
     */
    /** @en
     * The type of a media track:
     * - `"audio"`: Audio track.
     * - `"video"`: Video track.
     */
    trackMediaType: "audio" | "video";
    /**
     * 媒体轨道是否正在页面上播放。
     * - `true`: 媒体轨道正在页面上播放。
     * - `false`: 媒体轨道没有在页面上播放。
     */
    /** @en
     * Whether a media track is playing on the webpage:
     * - `true`: The media track is playing on the webpage.
     * - `false`: The media track is not playing on the webpage.
     */
    isPlaying: boolean;
    /**
     * 获取由 SDK 生成的对于媒体轨道来说的唯一 ID。
     *
     * @return 媒体轨道 ID。
     */
    /** @en
     * Gets the ID of a media track, a unique identifier generated by the SDK.
     *
     * @return The media track ID.
     */
    getTrackId(): string;
    /**
     * 获取浏览器原生的 [MediaStreamTrack](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack) 对象。
     *
     * @return 一个 [MediaStreamTrack](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack) 对象。
     */
    /** @en
     * Gets an [MediaStreamTrack](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack) object.
     *
     * @return An [MediaStreamTrack](https://developer.mozilla.org/en-US/docs/Web/API/MediaStreamTrack) object.
     */
    getMediaStreamTrack(): MediaStreamTrack;
    /**
     * 获取当前轨道的 [RTCRtpTransceiver](https://developer.mozilla.org/en-US/docs/Web/API/RTCRtpTransceiver) 实例。
     *
     * 该方法目前主要用于视频流的[端到端加密（Beta）](https://doc.shengwang.cn/doc/rtc/javascript/advanced-features/e2e-encryption)。
     *
     * > 如果 SDK 发生断线重连，当前轨道对应的 `RTCRtpTransceiver` 实例可能会发生变化。你可以通过以下回调来获取新的 `RTCRtpTransceiver` 实例：
     * >  - 本地轨道：[ILocalTrack.transceiver-updated]{@link event_transceiver_updated}
     * >  - 远端轨道：[IRemoteTrack.transceiver-updated]{@link event_transceiver_updated_2}
     *
     * @param type 视频流的类型。详见 {@link StreamType}。
     * @returns 当前轨道的 `RTCRtpTransceiver` 实例。
     */
    /** @en
     * Gets the [RTCRtpTransceiver](https://developer.mozilla.org/en-US/docs/Web/API/RTCRtpTransceiver) instance of the current track.
     *
     * This method is currently mainly used for end-to-end encryption of video streams (Beta).
     *
     * > If the SDK experiences a reconnection, the `RTCRtpTransceiver` instance corresponding to the current track might change. You can obtain the new `RTCRtpTransceiver` instance through the following callbacks:
     * >  - For a local track: [ILocalTrack.transceiver-updated]{@link event_transceiver_updated}
     * >  - For a remote track: [IRemoteTrack.transceiver-updated]{@link event_transceiver_updated_2}
     *
     * @param type The type of the video stream. See {@link StreamType}.
     * @returns The [RTCRtpTransceiver](https://developer.mozilla.org/en-US/docs/Web/API/RTCRtpTransceiver) instance of the current track.
     */
    getRTCRtpTransceiver(type?: StreamType): RTCRtpTransceiver | undefined;
    /**
     * 在页面上播放媒体轨道。
     *
     * @param element 指定一个 DOM 元素，SDK 将在这个元素下创建 `<video>` 元素播放视频轨道，支持 2 种类型：
     * - `string`: 指定该 DOM 元素的 ID 值。
     * - `HTMLElement`: 直接传入一个 DOM 元素对象。
     */
    /** @en
     * Plays a media track on the webpage.
     *
     * @param element Specifies a DOM element. The SDK will create a `<video>` element under the specified DOM element to play the video track. You can specify a DOM element in either of following ways:
     * - `string`: Specify the ID of the DOM element.
     * - `HTMLElement`: Pass a DOM object.
     */
    play(element?: string | HTMLElement): void;
    /**
     * 停止播放。
     */
    /** @en
     * Stops playing the media track.
     */
    stop(): void;
}

export declare class LocalAudioTrack extends LocalTrack implements ILocalAudioTrack {
    readonly trackMediaType: "audio" | "video";
    _encoderConfig?: AudioEncoderConfiguration;
    _trackSource: AudioTrackSource | FakeTrackSource;
    /**
     * @description This method is used to init WebAudio, if you're sure you don't want to use WebAudio, call _trackSource
     */
    get _source(): AudioTrackSource;
    set _source(source: AudioTrackSource | FakeTrackSource);
    _enabled: boolean;
    private _volume;
    _useAudioElement: boolean;
    _bypassWebAudio: boolean;
    protected processor?: IAudioProcessor;
    protected _processorContext: AudioProcessorContext | undefined;
    protected get processorContext(): AudioProcessorContext;
    _processorDestination: AudioProcessorDestination | undefined;
    get processorDestination(): AudioProcessorDestination;
    protected _getOriginVolumeLevel: boolean;
    get isPlaying(): boolean;
    get __className__(): string;
    constructor(track: MediaStreamTrack, encoderConfig?: AudioEncoderConfiguration, trackId?: string, getOriginVolumeLevel?: boolean);
    setVolume(volume: number): void;
    getVolumeLevel(): number;
    setPlaybackDevice(deviceId: string): Promise<void>;
    setEnabled(enabled: boolean, _?: any, skipChangeState?: boolean): Promise<void>;
    protected _setEnabled(enabled: boolean, _?: any, skipChangeState?: boolean): Promise<void>;
    setMuted(muted: boolean): Promise<void>;
    getStats(): LocalAudioTrackStats;
    setAudioFrameCallback(callback: null | ((buffer: AudioBuffer) => void), frameSize?: number): void;
    play(): void;
    stop(): void;
    close(): void;
    protected _updatePlayerSource(updateWebAudioSource?: boolean): void;
    protected _updateOriginMediaStreamTrack(track: MediaStreamTrack, stopOldTrack: boolean): Promise<void>;
    renewMediaStreamTrack(newConstraints?: MediaTrackConstraints): Promise<void>;
    pipe(processor: IAudioProcessor): IAudioProcessor;
    unpipe(): void;
    private bindProcessorDestinationEvents;
    private unbindProcessorDestinationEvents;
    protected bindProcessorContextEvents(processorContext: AudioProcessorContext): void;
    protected unbindProcessorContextEvents(processorContext: AudioProcessorContext): void;
    private initWebAudio;
    private initProcessor;
}

/**
 * 本地音频轨道的基本信息，可以通过 [AgoraRTCClient.getLocalAudioStats]{@link IAgoraRTCClient.getLocalAudioStats} 获取。
 * @public
 */
/** @en
 * Information of the local audio track, which can be retrieved by calling [AgoraRTCClient.getLocalAudioStats]{@link IAgoraRTCClient.getLocalAudioStats}.
 */
export declare interface LocalAudioTrackStats {
    /**
     * 音频的编码格式。
     *
     * - `"opus"`: 音频编码格式为 OPUS。
     * - `"aac"`: 音频编码格式为 AAC。
     * - `"pcmu"`: 预留参数，暂不支持。
     * - `"pcma"`: 预留参数，暂不支持。
     * - `"g722"`: 预留参数，暂不支持。
     *
     * > Firefox 上无法获取该值。
     */
    /** @en
     * The audio codec.
     *
     * - `"opus"`: The audio codec is OPUS。
     * - `"aac"`: The audio codec is AAC。
     * - `"pcmu"`: Reserved for future use.
     * - `"pcma"`: Reserved for future use.
     * - `"g722"`: Reserved for future use.
     *
     * > Firefox does not support this property.
     */
    codecType?: "opus" | "aac" | "PCMU" | "PCMA" | "G722";
    /**
     * 发送的音频音量等级，范围为 [0,32767]。
     *
     * > 该值通过 WebRTC-Stats 获取，不能确保实时性。如果你想获取当前实时的音量等级，调用 [LocalAudioTrack.getVolumeLevel]{@link ILocalAudioTrack.getVolumeLevel}。
     */
    /** @en
     * The energy level of the sent audio.
     *
     * The value range is [0,32767].
     *
     * > This value is retrieved by calling WebRTC-Stats and may not be up-to-date. To get the real-time sound volume, call [LocalAudioTrack.getVolumeLevel]{@link ILocalAudioTrack.getVolumeLevel}.
     */
    sendVolumeLevel: number;
    /**
     * 发送的音频码率 (bps)。
     */
    /** @en
     * The bitrate (bps) of the sent audio.
     */
    sendBitrate: number;
    /**
     * 发送的音频总字节数 (bytes)。
     */
    /** @en
     * The total bytes of the sent audio.
     */
    sendBytes: number;
    /**
     * 发送的音频总包数。
     */
    /** @en
     * The total packets of the sent audio.
     */
    sendPackets: number;
    /**
     * 发送的音频总丢包数。
     *
     * > Safari 上无法获取该值。
     */
    /** @en
     * The total number of lost audio packets that were sent.
     *
     * > You can not get this property on Safari.
     */
    sendPacketsLost: number;
    /**
     * 发送音频数据包的抖动 (ms)。
     */
    /** @en
     * Jitter (ms) of the audio packets that were sent.
     */
    sendJitterMs: number;
    /**
     * 发送音频数据包的往返时延 (ms)。
     */
    /** @en
     * Round-trip time delay (ms) of the audio packets that were sent.
     */
    sendRttMs: number;
    /**
     * 发送的音频 400ms 内的丢包率。
     */
    /** @en
     * The packet loss rate of the sent audio in 400ms.
     */
    currentPacketLossRate: number;
}

export declare class LocalDataChannel extends DataChannel implements ILocalDataChannel {
    send(data: ArrayBuffer): void;
}

export declare abstract class LocalTrack extends Track implements ILocalTrack {
    _enabled: boolean;
    _muted: boolean;
    /**
     * @description Used to determine whether the user switches streams
     */
    _isExternalTrack: boolean;
    get isExternalTrack(): boolean;
    get muted(): boolean;
    get enabled(): boolean;
    _isClosed: boolean;
    protected _enabledMutex: PromiseMutex;
    protected processor?: IBaseProcessor;
    protected abstract _processorContext: IProcessorContext | undefined;
    protected get processorContext(): IProcessorContext | undefined;
    abstract get processorDestination(): IBaseProcessor;
    constructor(track: MediaStreamTrack, trackId?: string);
    abstract getStats(): LocalVideoTrackStats | LocalAudioTrackStats;
    abstract setMuted(enabled: boolean): Promise<void>;
    abstract setEnabled(enabled: boolean): Promise<void>;
    getTrackLabel(): string;
    close(): void;
    protected _updateOriginMediaStreamTrack(track: MediaStreamTrack, stopOldTrack: boolean, isExternalTrack?: boolean): Promise<void>;
    protected abstract _updatePlayerSource(): void;
    protected _getDefaultPlayerConfig(): Partial<PlayerConfig>;
    protected _handleTrackEnded: () => void;
    protected onTrackEnded(): void;
    protected stateCheck(stateName: "enabled" | "muted", state: boolean): void;
    abstract renewMediaStreamTrack(): Promise<void>;
    getProcessorStats(): ProcessorStats[];
    getProcessorUsage(): Promise<UsageWithDirection[]>;
}

export declare enum LocalTrackEvents {
    SOURCE_STATE_CHANGE = "source-state-change",
    TRACK_ENDED = "track-ended",
    /** @ignore */
    BEAUTY_EFFECT_OVERLOAD = "beauty-effect-overload",
    VIDEO_ELEMENT_VISIBLE_STATUS = "video-element-visible-status",
    CLOSED = "closed"
}

export declare class LocalVideoTrack extends LocalTrack implements ILocalVideoTrack {
    readonly trackMediaType: "audio" | "video";
    _player?: AgoraRTCPlayer | VideoPlayer;
    isUseScaleResolutionDownBy: boolean;
    private _videoVisibleTimer;
    private _previousVideoVisibleStatus;
    private _clearPreviousVideoVisibleStatus;
    _encoderConfig?: Partial<VideoEncoderConfiguration>;
    _scalabilityMode?: SVCConfiguration;
    _optimizationMode?: OptimizationMode | "balanced";
    private _videoHeight?;
    private _videoWidth?;
    get videoHeight(): number | undefined;
    get videoWidth(): number | undefined;
    _forceBitrateLimit?: {
        max_bitrate: number;
        min_bitrate: number;
    };
    _enabled: boolean;
    get isPlaying(): boolean;
    protected _processorDestination: VideoProcessorDestination;
    get processorDestination(): VideoProcessorDestination;
    protected _processorContext: VideoProcessorContext;
    protected get processorContext(): VideoProcessorContext;
    protected set processorContext(ctx: VideoProcessorContext);
    get __className__(): string;
    constructor(track: MediaStreamTrack, encoderConfig?: Partial<VideoEncoderConfiguration>, scalabilityConfig?: SVCConfiguration, optimizationMode?: OptimizationMode | "balanced", trackId?: string, hints?: TrackHint[]);
    play(element: HTMLElement | HTMLVideoElement | string, config?: VideoPlayerConfig): void;
    stop(): void;
    setEnabled(enabled: boolean, skipChangeState?: boolean): Promise<void>;
    setMuted(muted: boolean): Promise<void>;
    setEncoderConfiguration(config: VideoEncoderConfiguration | VideoEncoderConfigurationPreset, doNotRenegoation?: boolean): Promise<void>;
    getStats(): LocalVideoTrackStats;
    setBeautyEffect(enabled: boolean, options?: BeautyEffectOptions): Promise<void>;
    getCurrentFrameData(): ImageData;
    getCurrentFrameImage(imageType: string, quality?: number): Promise<ImageTypedData>;
    setBitrateLimit(bitrateLimit: {
        max_bitrate: number;
        min_bitrate: number;
    }): Promise<void>;
    setOptimizationMode(mode: OptimizationMode | "balanced"): Promise<void>;
    setScalabiltyMode(mode: SVCConfiguration): void;
    updateMediaStreamTrackResolution(): void;
    protected _updatePlayerSource(): void;
    protected _getDefaultPlayerConfig(): Partial<VideoPlayerConfig>;
    protected setSenderConfiguration(config: VideoEncoderConfiguration | VideoEncoderConfigurationPreset): Promise<void>;
    updateBitrateFromProfile(): void;
    getVideoElementVisibleStatus(): CheckVideoVisibleResult | undefined;
    renewMediaStreamTrack(newConstraints?: MediaTrackConstraints): Promise<void>;
    pipe(processor: IBaseProcessor): IBaseProcessor;
    unpipe(): void;
    close(): void;
    clone(config?: VideoEncoderConfiguration | VideoEncoderConfigurationPreset, cloneTrack?: boolean): LocalVideoTrack;
    replaceTrack(track: MediaStreamTrack, stopOldTrack: boolean): Promise<void>;
    sendSeiData(sei: Uint8Array): never;
    private bindProcessorDestinationEvents;
    private unbindProcessorDestinationEvents;
    private unbindProcessorContextEvents;
}

/**
 * 本地视频轨道的基本信息，可以通过 [AgoraRTCClient.getLocalVideoStats]{@link IAgoraRTCClient.getLocalVideoStats} 获取。
 * @public
 */
/** @en
 * Information of the local video track, which can be retrieved by calling [AgoraRTCClient.getLocalVideoStats]{@link IAgoraRTCClient.getLocalVideoStats}.
 */
export declare interface LocalVideoTrackStats {
    /**
     * 视频编码格式。
     *
     * - `"H264"`: 视频编码格式为 H.264。
     * - `"VP8"`: 视频编码格式为 VP8。
     * - `"VP9"`: 视频编码格式为 VP9。
     * - `"AV1X"`: 预留参数，暂不支持。
     * - `"AV1"`: 视频编码格式为 AV1。
     *
     * > Firefox 上无法获取该值。
     */
    /** @en
     * The video codec.
     *
     * - `"H264"`: The video codec is H.264.
     * - `"VP8"`: The video codec is VP8.
     * - `"VP9"`: The video codec is VP9.
     * - `"AV1X"`: Reserved for future use.
     * - `"AV1"`: The video codec is AV1.
     *
     * > You can not get this property on Firefox.
     */
    codecType?: "H264" | "H265" | "VP8" | "VP9" | "AV1X" | "AV1";
    /**
     * 发送的视频总字节数 (bytes)。
     */
    /** @en
     * The total bytes of the sent video.
     */
    sendBytes: number;
    /**
     * 视频发送帧率（fps）。
     *
     * > Firefox 上无法获取该值。
     */
    /** @en
     * The frame rate (fps) of the sent video.
     *
     * > You can not get this property on Firefox.
     */
    sendFrameRate?: number;
    /**
     * 视频采集帧率（fps）。
     *
     * > Safari 和 Firefox 上无法获取该值。
     */
    /** @en
     * The frame rate (fps) of the captured video.
     *
     * > You can not get this property on Safari and Firefox.
     */
    captureFrameRate?: number;
    /**
     * 发送的视频总包数。
     */
    /** @en
     * The total packets of the sent video.
     */
    sendPackets: number;
    /**
     * 发送的视频总丢包数。
     *
     * > - Safari 上无法获取该值。
     * > - Firefox 上该字段不准确。
     */
    /** @en
     * The total number of lost video packets that were sent.
     *
     * > - You can not get this property on Safari.
     * > - This property is inaccurate on Firefox.
     */
    sendPacketsLost: number;
    /**
     * 发送视频数据包的抖动 (ms)。
     */
    /** @en
     * Jitter (ms) of the video packets that were sent.
     */
    sendJitterMs: number;
    /**
     * 发送视频数据包的往返时延 (ms)。
     */
    /** @en
     * Round-trip time delay (ms) of the video packets that were sent.
     */
    sendRttMs: number;
    /**
     * 视频发送的分辨率高度。
     */
    /** @en
     * The resolution height (pixel) of the sent video.
     */
    sendResolutionHeight: number;
    /**
     * 视频发送的分辨率宽度。
     */
    /** @en
     * The resolution width (pixel) of the sent video.
     */
    sendResolutionWidth: number;
    /**
     * 视频采集的分辨率高度。
     */
    /** @en
     * The resolution height (pixel) of the captured video.
     */
    captureResolutionHeight: number;
    /**
     * 视频采集的分辨率宽度。
     */
    /** @en
     * The resolution width (pixel) of the captured video.
     */
    captureResolutionWidth: number;
    /**
     * 视频编码延迟（ms）。
     */
    /** @en
     * The time (ms) required for encoding the captured video.
     */
    encodeDelay?: number;
    /**
     * 视频发送码率 (bps)。
     */
    /** @en
     * The bitrate (bps) of the sent video.
     */
    sendBitrate: number;
    /**
     * 视频目标发送码率 (bps)，即 {@link VideoEncoderConfiguration} 中设置的码率。
     */
    /** @en
     * The target bitrate (bps) of the sent video, namely the bitrate set in {@link VideoEncoderConfiguration}.
     */
    targetSendBitrate: number;
    /**
     * 视频总时长，单位为秒。
     */
    /** @en
     * The total duration of the sent video in seconds.
     */
    totalDuration: number;
    /**
     * 视频总卡顿时长，单位为秒。
     */
    /** @en
     * The total freeze time of the encoded video in seconds.
     */
    totalFreezeTime: number;
    /**
     * 发送的视频 400ms 内的丢包率。
     */
    /** @en
     * The packet loss rate of the sent video in 400ms.
     */
    currentPacketLossRate: number;
}

/**
 * 小流的视频属性，用于 [setLowStreamParameter]{@link IAgoraRTCClient.setLowStreamParameter}。
 * @public
 */
/** @en
 * The video profile of the low-quality video stream. Set the the video profile of the low-quality video stream when calling [setLowStreamParameter]{@link IAgoraRTCClient.setLowStreamParameter}.
 */
export declare interface LowStreamParameter {
    /**
     * 小流视频的分辨率宽。
     *
     * 可以传入一个 `number`，或一个约束对象，如 `{ max: 1280, min: 720 }`。
     *
     * 关于约束对象的详细说明可以参考 [ConstrainLong]{@link ConstrainLong}。
     */
    /** @en
     * Width of the video.
     *
     * You can pass a `number`, or a constraint such as `{ max: 1280, min: 720 }`.
     *
     * For more details about the constraint, see [ConstrainLong]{@link ConstrainLong}.
     */
    width: ConstrainULong;
    /**
     * 小流视频的分辨率高。
     *
     * 可以传入一个 `number`，或一个约束对象，如 `{ max: 720, min: 480 }`。
     *
     * 关于约束对象的详细说明可以参考 [ConstrainLong]{@link ConstrainLong}。
     */
    /** @en
     * Height of the video.
     *
     * You can pass a `number`, or a constraint such as `{ max: 1280, min: 720 }`.
     *
     * For more details about the constraint, see [ConstrainLong]{@link ConstrainLong}.
     */
    height: ConstrainULong;
    /**
     * 小流视频的帧率。
     *
     * 可以传入一个 `number`，或一个约束对象，如 `{ max: 30, min: 5 }`。
     *
     * 关于约束对象的详细说明可以参考 [ConstrainLong]{@link ConstrainLong}。
     */
    /** @en
     * Frame rate of the video (fps).
     *
     * You can pass a `number`, or a constraint such as `{ max: 30, min: 5 }`.
     *
     * For details about the constraint, see [ConstrainLong]{@link ConstrainLong}.
     */
    framerate?: ConstrainULong;
    /**
     * 小流视频的码率，单位为 Kbps。
     */
    /** @en
     * Bitrate of the video (Kbps).
     */
    bitrate?: number;
}

export declare const MediaElementNumStatus: {
    [n in MediaElementStatus | "uninit"]: number;
};

export declare enum MediaElementStatus {
    NONE = "none",
    INIT = "init",
    CANPLAY = "canplay",
    PLAYING = "playing",
    PAUSED = "paused",
    SUSPEND = "suspend",
    STALLED = "stalled",
    WAITING = "waiting",
    ERROR = "error",
    DESTROYED = "destroyed",
    ABORT = "abort",
    ENDED = "ended",
    EMPTIED = "emptied",
    LOADEDDATA = "loadeddata"
}

export declare interface MediaStats {
    resolution: Resolution;
}

export declare class MicrophoneAudioTrack extends LocalAudioTrack implements IMicrophoneAudioTrack {
    _config: MicrophoneAudioTrackInitConfig;
    _deviceName: string;
    private _constraints;
    private readonly _originalConstraints;
    _enabled: boolean;
    get __className__(): string;
    constructor(track: MediaStreamTrack, config: MicrophoneAudioTrackInitConfig, constraints: MediaTrackConstraints, trackId?: string);
    setDevice(deviceId: string): Promise<void>;
    setEnabled(enabled: boolean, notCloseDevice?: boolean, skipChangeState?: boolean): Promise<void>;
    close(): void;
    protected onTrackEnded(): void;
    renewMediaStreamTrack(newConstraints?: MediaTrackConstraints): Promise<void>;
    protected bindProcessorContextEvents(processorContext: AudioProcessorContext): void;
    protected unbindProcessorContextEvents(processorContext: AudioProcessorContext): void;
}

/**
 * 创建麦克风音频流时的配置对象，用于 [AgoraRTC.createMicrophoneAudioTrack]{@link IAgoraRTC.createMicrophoneAudioTrack}。
 */
/** @en
 * Configurations for the audio track from the audio captured by a microphone. Set these configurations when calling [AgoraRTC.createMicrophoneAudioTrack]{@link IAgoraRTC.createMicrophoneAudioTrack}.
 */
export declare interface MicrophoneAudioTrackInitConfig extends AudioTrackInitConfig {
    /**
     * 控制音频的编码配置。
     *
     * 你可以通过 [[AudioEncoderConfigurationPreset]] 传入 SDK 内置的编码配置；也可以通过传入一个 [[AudioEncoderConfiguration]] 来自定义音频编码配置。
     *
     * > Firefox 不支持设置音频编码码率。
     */
    /** @en
     * The audio encoder configurations.
     *
     * You can set the audio encoder configurations in either of the following ways:
     * - Pass the preset audio encoder configurations by using [[AudioEncoderConfigurationPreset]].
     * - Pass your customized audio encoder configurations by using [[AudioEncoderConfiguration]].
     *
     * > Firefox does not support setting the audio encoding rate.
     */
    encoderConfig?: AudioEncoderConfiguration | AudioEncoderConfigurationPreset;
    /**
     * @ignore
     */
    /** @en
     * @ignore
     */
    DTX?: boolean;
    /**
     * 指定麦克风的设备 ID。
     *
     * 你可以通过 [AgoraRTC.getMicrophones]{@link IAgoraRTC.getMicrophones} 来获取当前的麦克风设备列表。
     */
    /** @en
     * Specifies the microphone ID.
     *
     * You can get a list of the available microphones by calling [AgoraRTC.getMicrophones]{@link IAgoraRTC.getMicrophones}.
     */
    microphoneId?: string;
    /**
     * @ignore
     * 指定是否音频不路由通过WebAudio
     */
    /** @en
     * @ignore
     * Specifies whether or not audio track pass through WebAudio.
     */
    bypassWebAudio?: boolean;
}

export declare class MixingAudioTrack extends LocalAudioTrack {
    get __className__(): string;
    trackList: LocalAudioTrack[];
    private destNode;
    get isActive(): boolean;
    constructor();
    hasAudioTrack(track: LocalAudioTrack): boolean;
    addAudioTrack(track: LocalAudioTrack): void;
    removeAudioTrack(track: LocalAudioTrack): void;
    private updateEncoderConfig;
    _updateRtpTransceiver(transceiver: RTCRtpTransceiver): void;
}

/**
 * 视频传输优化模式：
 * - `"detail"`: 清晰优先。
 *   - 浏览器会自动根据你的采集分辨率和帧率设定一个最小码率。即使遭遇网络波动，发送码率也不会低于这个值，从而确保清晰的视频画面。
 *   - 大部分情况下，浏览器不会降低发送分辨率，但是可能会降低帧率。
 * - `"motion"`: 自 4.21.0 开始，浏览器默认开启流畅优先模式。
 *   - 浏览器不会启用最小码率策略。遭遇网络波动时，发送端会降低码率来确保接收端的视频画面不会出现中断和卡顿。
 *   - 大部分情况下，浏览器不会降低帧率，但是可能会降低发送分辨率。
 * @public
 */
/** @en
 * The video transmission optimization mode:
 * - `"detail"`: Prioritizes video quality.
 *   - The browser ensures high-quality images by automatically calculating a minimum bitrate based on the capturing resolution and frame rate. No matter how poor the network condition is, the sending bitrate will never be lower than the minimum value.
 *   - In most cases, the browser does not reduce the sending resolution, but may reduce the frame rate.
 * -  `"motion"`: Since v4.21.0, the browser prioritizes video smoothness.
 *   - In poor network conditions, the browser reduces the sending bitrate to minimize video freezes.
 *   - In most cases, the browser does not reduce the frame rate, but may reduce the sending resolution.
 */
export declare type OptimizationMode = "motion" | "detail";

/**
 * @internal
 */
export declare interface PlayerConfig extends VideoPlayerConfig {
    trackId: string;
    element: HTMLElement;
}

/**
 * Safari 的 AudioNode.disconnect 不支持 disconnect 指定节点，只能操作全部节点
 * 这里的 polyfill 是通过 _inputNodes 保存全部已经 connect 的节点
 * 然后在 disconnect 后，将那些预期不应该被 disconnect 的节点重新 connect
 */
export declare function polyfillAudioNode(node: AudioNode): void;

export declare enum PROCESSOR_CONTEXT_EVENTS {
    REQUEST_UPDATE_CONSTRAINTS = "request_update_constraints",
    REQUEST_CONSTRAINTS = "request_constraints"
}

export declare enum PROCESSOR_DESTINATION_EVENTS {
    ON_TRACK = "on_track",
    ON_NODE = "on_node"
}

export declare interface ProcessorUsageStats {
    id: string;
    value: number;
    level: number;
    totalTs: number;
}

export declare class RemoteAudioTrack extends RemoteTrack implements IRemoteAudioTrack {
    readonly trackMediaType: TrackMediaType.AUDIO;
    _source: AudioTrackSource | FakeTrackSource;
    _useAudioElement: boolean;
    private _volume;
    protected processorContext: AudioProcessorContext;
    processorDestination: AudioProcessorDestination;
    private _played;
    private _bypassWebAudio;
    get isPlaying(): boolean;
    get __className__(): string;
    constructor(track: MediaStreamTrack, userId: UID, uintId: number, store: SDKStore);
    setAudioFrameCallback(callback: null | ((buffer: AudioBuffer) => void), frameSize?: number): void;
    setVolume(volume: number): void;
    setPlaybackDevice(deviceId: string): Promise<void>;
    getVolumeLevel(): number;
    getStats(): RemoteAudioTrackStats;
    play(): void;
    stop(): void;
    _destroy(): void;
    _isFreeze(): boolean;
    protected _updatePlayerSource(updateWebAudioSource?: boolean): void;
    pipe(processor: IAudioProcessor): IAudioProcessor;
    unpipe(): void;
    private bindProcessorDestinationEvents;
    private unbindProcessorDestinationEvents;
}

/**
 * 远端音频轨道的统计数据，包括连接、传输等基本信息，可以通过 [AgoraRTCClient.getRemoteAudioStats]{@link IAgoraRTCClient.getRemoteAudioStats} 获取。
 * @public
 */
/** @en
 * Statistics of the remote audio track, such as connection and transmission statistics, which can be retrieved by calling [AgoraRTCClient.getRemoteAudioStats]{@link IAgoraRTCClient.getRemoteAudioStats}.
 */
export declare interface RemoteAudioTrackStats {
    /**
     * 传输延迟（ms)。
     *
     * 从远端发送音频到本地接收音频的延迟。
     */
    /** @en
     * Transmission delay (ms).
     *
     * The delay (ms) between a remote client sending the audio and the local client receiving the audio.
     */
    transportDelay: number;
    /**
     * 音频的编码格式。
     *
     * - `"opus"`: 音频编码格式为 OPUS。
     * - `"aac"`: 音频编码格式为 AAC。
     * - `"pcmu"`: 预留参数，暂不支持。
     * - `"pcma"`: 预留参数，暂不支持。
     * - `"g722"`: 预留参数，暂不支持。
     *
     * > Firefox 上无法获取该值。
     */
    /** @en
     * The audio codec.
     *
     * - `"opus"`: The audio codec is OPUS。
     * - `"aac"`: The audio codec is AAC。
     * - `"pcmu"`: Reserved for future use.
     * - `"pcma"`: Reserved for future use.
     * - `"g722"`: Reserved for future use.
     *
     * > Firefox does not support this property.
     */
    codecType?: "opus" | "aac" | "PCMU" | "PCMA" | "G722";
    /**
     * 音频端到端延迟（ms）。
     *
     * 从远端采集音频到本地播放音频的延迟。该延迟不包括发送端编码和接收端解码的时间。
     */
    /** @en
     * End-to-end delay (ms).
     *
     * The delay (ms) between a remote client sampling the audio and the local client playing the audio.
     * This delay does not include the time spent in encoding at the remote client and the time spent in decoding at the local client.
     */
    end2EndDelay: number;
    /**
     * 接收的音频码率（bps）。
     */
    /** @en
     * The bitrate (bps) of the received audio.
     */
    receiveBitrate: number;
    /**
     * 接收的音频音量等级，范围为 [0,32767]。
     *
     * > 该值通过 WebRTC-Stats 获取，不能确保实时性。如果你想获取当前实时的音量等级，调用 [RemoteAudioTrack.getVolumeLevel]{@link IRemoteAudioTrack.getVolumeLevel}。
     */
    /** @en
     * The energy level of the received audio.
     *
     * The value range is [0,32767].
     *
     * > This value is retrieved by calling WebRTC-Stats and may not be up-to-date. To get the real-time sound volume, call [RemoteAudioTrack.getVolumeLevel]{@link IRemoteAudioTrack.getVolumeLevel}.
     */
    receiveLevel: number;
    /**
     * 接收的音频总字节数。
     */
    /** @en
     * The total bytes of the received audio.
     */
    receiveBytes: number;
    /**
     * 接收音频延迟 (ms)。
     *
     * 从发送到接收端播放音频的延迟时间。
     *
     * > Safari 和 Firefox 上该字段不准确。
     */
    /** @en
     * The delay (ms) between a remote client sending the audio and the local client playing the audio.
     *
     * > This property is inaccurate on Safari and Firefox.
     */
    receiveDelay: number;
    /**
     * 接收的音频总包数。
     */
    /** @en
     * The total packets of the received audio.
     */
    receivePackets: number;
    /**
     * 接收的音频总丢包数。
     */
    /** @en
     * The total number of lost audio packets that should be received.
     */
    receivePacketsLost: number;
    /**
     * 因过早或过晚到达抖动缓冲器而被丢弃的数据包数量。
     */
    /** @en
     * The number of packets discarded by the jitter buffer due to early or late arrival.
     */
    receivePacketsDiscarded: number;
    /**
     * 接收的音频总丢包率。
     */
    /** @en
     * The packet loss rate of the received audio.
     */
    packetLossRate: number;
    /**
     * 接收的音频400ms内的丢包率。
     */
    /** @en
     * The packet loss rate of the received audio.
     */
    currentPacketLossRate: number;
    /**
     * 接收的音频总时长，单位为秒。
     */
    /** @en
     * The total duration of the received audio in seconds.
     */
    totalDuration: number;
    /**
     * 接收的音频总卡顿时长，单位为秒。
     */
    /** @en
     * The total freeze time of the received audio in seconds.
     */
    totalFreezeTime: number;
    /**
     * 接收的音频卡顿率。
     */
    /** @en
     * The freeze rate of the received audio.
     */
    freezeRate: number;
    /**
     * 远端发布音频时长
     * 如果为 -1，说明暂时还无法获取发布时长
     */
    publishDuration: number;
}

export declare class RemoteDataChannel extends DataChannel implements IRemoteDataChannel {
    private _messageListener;
    constructor(config: IDataChannelConfig);
    _updateOriginDataChannel(datachannel: RTCDataChannel): void;
    _close(): void;
    private _bandRemoteDataChannelEvents;
}

/**
 * 媒体流的回退策略选项，用于 [setStreamFallbackOption]{@link IAgoraRTCClient.setStreamFallbackOption}。
 * @public
 */
/** @en
 * The stream fallback option. Set the stream fallback option when calling [setStreamFallbackOption]{@link IAgoraRTCClient.setStreamFallbackOption}.
 *
 */
export declare enum RemoteStreamFallbackType {
    /** 0: 关闭回退策略，弱网时不对音视频流作回退处理。 */
    /** @en
     * 0: Disable the fallback.
     */
    DISABLE = 0,
    /** 1: 在网络条件较差的情况下，自动订阅视频小流。 */
    /** @en
     * 1: Automatically subscribe to the low-video stream under poor network conditions. */
    LOW_STREAM = 1,
    /** 2: 网络较弱时，先尝试订阅视频小流。如果网络环境无法显示视频，则再回退到订阅纯音频流。 */
    /** @en
     * 2: Subscribe to the low-quality video stream when the network conditions worsen, and subscribe to audio only when the conditions become too poor to support video transmission.
     */
    AUDIO_ONLY = 2,
    /**
     * ABR 拓展, 暂时不对外，以实际客户为主
     */
    HIGH_STREAM_LAYER1 = 3,
    HIGH_STREAM_LAYER2 = 4,
    HIGH_STREAM_LAYER3 = 5,
    HIGH_STREAM_LAYER4 = 6,
    HIGH_STREAM_LAYER5 = 7,
    HIGH_STREAM_LAYER6 = 8
}

/**
 * 远端视频流的大小类型，可以在 [setRemoteVideoStreamType]{@link IAgoraRTCClient.setRemoteVideoStreamType} 里使用。
 * @public
 */
/** @en
 * The video type of the remote stream. Set the video type of the remote stream when calling [setRemoteVideoStreamType]{@link IAgoraRTCClient.setRemoteVideoStreamType}.
 */
export declare enum RemoteStreamType {
    /** 0: 高分辨率、高码率的视频大流。 */
    /** @en
     * 0: High-quality video stream (high-bitrate, high-resolution).
     */
    HIGH_STREAM = 0,
    /** 1: 低分辨率、低码率的视频小流。 */
    /** @en
     * 1: Low-quality video stream (low-bitrate, low-resolution).
     */
    LOW_STREAM = 1,
    /**
     * ABR 拓展
     * 暂时不对外，以实际客户为主
     */
    HIGH_STREAM_LAYER1 = 4,
    HIGH_STREAM_LAYER2 = 5,
    HIGH_STREAM_LAYER3 = 6,
    HIGH_STREAM_LAYER4 = 7,
    HIGH_STREAM_LAYER5 = 8,
    HIGH_STREAM_LAYER6 = 9
}

export declare abstract class RemoteTrack extends Track implements IRemoteTrack {
    private _userId;
    /** @internal */
    _uintId: number;
    _isDestroyed: boolean;
    protected store: SDKStore;
    protected processor?: IBaseProcessor;
    protected abstract processorContext: IProcessorContext;
    getUserId(): UID;
    abstract getStats(): RemoteAudioTrackStats | RemoteVideoTrackStats;
    constructor(track: MediaStreamTrack, userId: UID, uintId: number, store: SDKStore);
    _updateOriginMediaStreamTrack(track: MediaStreamTrack): void;
    _destroy(): void;
    protected abstract _updatePlayerSource(): void;
    getProcessorStats(): ProcessorStats[];
    getProcessorUsage(): Promise<UsageWithDirection[]>;
}

export declare enum RemoteTrackEvents {
    FIRST_FRAME_DECODED = "first-frame-decoded",
    VIDEO_ELEMENT_VISIBLE_STATUS = "video-element-visible-status",
    VIDEO_STATE_CHANGED = "video-state-changed"
}

export declare class RemoteVideoTrack extends RemoteTrack implements IRemoteVideoTrack {
    private _videoVisibleTimer;
    private _previousVideoVisibleStatus;
    private _clearPreviousVideoVisibleStatus;
    readonly trackMediaType: TrackMediaType.VIDEO;
    _videoWidth?: number;
    _videoHeight?: number;
    _player?: AgoraRTCPlayer | VideoPlayer;
    processorDestination: VideoProcessorDestination;
    protected processorContext: VideoProcessorContext;
    get isPlaying(): boolean;
    get __className__(): string;
    constructor(track: MediaStreamTrack, userId: UID, uintId: number, store: SDKStore);
    getStats(): RemoteVideoTrackStats;
    play(element: string | HTMLElement | HTMLVideoElement, config?: VideoPlayerConfig): void;
    stop(): void;
    getCurrentFrameData(): ImageData;
    updateMediaStreamTrackResolution(): void;
    protected _updatePlayerSource(): void;
    getVideoElementVisibleStatus(): CheckVideoVisibleResult | undefined;
    pipe(processor: IBaseProcessor): IBaseProcessor;
    unpipe(): void;
    private bindProcessorDestinationEvents;
    private unbindProcessorDestinationEvents;
    _destroy(): void;
    _onSei(sei: Uint8Array): void;
}

/**
 * 远端视频轨道的统计数据，包括连接、传输等基本信息，可以通过 [AgoraRTCClient.getRemoteVideoStats]{@link IAgoraRTCClient.getRemoteVideoStats} 获取。
 * @public
 */
/** @en
 * Statistics of the remote video track, such as connection and transmission statistics, which can be retrieved by calling [AgoraRTCClient.getRemoteVideoStats]{@link IAgoraRTCClient.getRemoteVideoStats}.
 */
export declare interface RemoteVideoTrackStats {
    /**
     * 传输延迟（ms)。
     *
     * 从远端发送视频到本地接收视频的延迟。
     */
    /** @en
     * Transmission delay (ms).
     *
     * The delay (ms) between a remote client sending the video and the local client receiving the video.
     */
    transportDelay: number;
    /**
     * 视频编码格式。
     *
     * - `"H264"`: 视频编码格式为 H.264。
     * - `"VP8"`: 视频编码格式为 VP8。
     * - `"VP9"`: 视频编码格式为 VP9。
     * - `"AV1X"`: 视频编码格式为 AV1，在 Chrome M96 版本之前。
     * - `"AV1"`: 视频编码格式为 AV1。
     *
     * > Firefox 上无法获取该值。
     */
    /** @en
     * The video codec.
     *
     * - `"H264"`: The video codec is H.264.
     * - `"VP8"`: The video codec is VP8.
     * - `"VP9"`: The video codec is VP9.
     * - `"AV1X"`: Reserved for future use.
     * - `"AV1"`: The video codec is AV1.
     *
     * > You can not get this property on Firefox.
     */
    codecType?: "H264" | "H265" | "VP8" | "VP9" | "AV1X" | "AV1";
    /**
     * 视频端到端延迟（ms）。
     *
     * 从远端采集视频到本地播放视频的延迟。该延迟不包括发送端编码和接收端解码的时间。
     */
    /** @en
     * End-to-end delay (ms).
     *
     * The delay (ms) a remote client capturing the video and the local client playing the video.
     * This delay does not include the time spent in encoding at the remote client and the time spent in decoding at the local client.
     */
    end2EndDelay: number;
    /**
     * 接收的视频码率（bps）。
     */
    /** @en
     * The bitrate (bps) of the received video.
     */
    receiveBitrate: number;
    /**
     * 接收视频延迟 (ms)。
     *
     * 从发送到接收端播放视频的延迟时间。
     *
     * > Safari 和 Firefox 上该字段不准确。
     */
    /** @en
     * The delay (ms) between a remote client sending the video and the local client playing the video.
     *
     * > This property is inaccurate on Safari and Firefox.
     */
    receiveDelay: number;
    /**
     * 接收的视频总字节数。
     */
    /** @en
     * The total byes of the received video.
     */
    receiveBytes: number;
    /**
     * 视频解码帧率（fps）。
     */
    /** @en
     * The frame rate (fps) of the decoded video.
     */
    decodeFrameRate?: number;
    /**
     * 视频接收帧率（fps）。
     */
    /** @en
     * The frame rate (fps) of the received video.
     */
    receiveFrameRate?: number;
    /**
     * 视频渲染帧率（fps）。
     */
    /** @en
     * The rendering frame rate (fps) of the decoded video.
     */
    renderFrameRate?: number;
    /**
     * 接收的视频总包数。
     */
    /** @en
     * The total bytes of the received video.
     */
    receivePackets: number;
    /**
     * 接收的视频总丢包数。
     */
    /** @en
     * The total number of lost video packets that should be received.
     */
    receivePacketsLost: number;
    /**
     * 接收的视频丢包率。
     */
    /** @en
     * The packet loss rate of the received video.
     */
    packetLossRate: number;
    /**
     * 接收的视频在400ms内的丢包率。
     */
    /** @en
     * The packet loss rate of the received video.
     */
    currentPacketLossRate: number;
    /**
     * 接收的视频分辨率高度。
     */
    /** @en
     * The resolution height (pixel) of the received video.
     */
    receiveResolutionHeight: number;
    /**
     * 接收的视频分辨率宽度。
     */
    /** @en
     * The resolution width (pixel) of the received video.
     */
    receiveResolutionWidth: number;
    /**
     * 接收的视频总时长，单位为秒。
     */
    /** @en
     * The total duration of the received video in seconds.
     */
    totalDuration: number;
    /**
     * 接收的视频总卡顿时长，单位为秒。
     */
    /** @en
     * The total freeze time of the received video in seconds.
     */
    totalFreezeTime: number;
    /**
     * 接收的视频卡顿率。
     */
    /** @en
     * The freeze rate of the received video.
     */
    freezeRate: number;
    /**
     * 远端发布视频时长
     * 如果为 -1，说明暂时还无法获取发布时长
     */
    publishDuration: number;
}

export declare function removeTrack(track: Track): void;

export declare function requestAutoplayGesture(): void;

export declare interface Resolution {
    width: number;
    height: number;
}

/**
 * 全局 getUserMedia 锁，确保每次 GUM 时上一个 GUM 已经完成
 *
 * 该目前只用于 Safari 和 iOS 浏览器，https://bugs.webkit.org/show_bug.cgi?id=179363
 *
 * 防止 SDK Device 模块和用户同时触发 GUM 导致用户 GUN 返回的 track 被 mute。
 */
export declare const SAFARI_GLOBAL_GUM_LOCK: PromiseMutex;

/**
 * 创建屏幕共享音频轨道时的配置参数。当使用 [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack} 创建屏幕共享视频轨道时，可以通过此配置共享音频轨道。
 */
/** @en
 * Configurations for the audio track of screen sharing. Set these configurations when calling [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack} to enable audio during screen sharing.
 */
export declare type ScreenAudioTrackInitConfig = AudioTrackInitConfig;

export declare interface ScreenConstraintsWithElectron extends MediaTrackConstraints {
    /**
     * Electron 屏幕共享时用户传入的 sourceID。
     */
    sourceId?: string;
    /**
     * Chrome 插件屏幕共享的 id。
     */
    extensionId?: string;
    /**
     * Chrome 插件屏幕共享的 options。
     */
    mandatory?: any;
    /**
     * 屏幕共享的 source 类型，目前只对 Firefox 有效。
     */
    mediaSource?: ScreenSourceType;
    width?: ConstrainULong;
    height?: ConstrainULong;
    /**
     * "browser" for tabs.
     * "window" for windows.
     * "monitor" for screens.
     */
    displaySurface?: "browser" | "window" | "monitor";
    /** 是否屏蔽本地tag页面, 默认include(和Chrome 107默认exclude不同，为了保持与过去浏览器版本一致), Chrome 107以上 Opera 93以上*/
    selfBrowserSurface?: "include" | "exclude";
    /** 是否允许tag切换分享, 默认允许include, Chrome 107以上 Opera 93以上*/
    surfaceSwitching?: "include" | "exclude";
    /** 是否开启系统音频，默认开启include, Chrome 105以上, Opera 91以上 */
    systemAudio?: "include" | "exclude";
}

/**
 * SDK 预设的 [[VideoEncoderConfiguration]] 配置。
 *
 * 你可以在 [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack} 方法中传入以下预设值来控制屏幕共享视频编码配置。
 *
 * 下表列出了 SDK 所有预设的屏幕共享视频属性配置。
 *
 * | 视频属性 | 分辨率（宽×高） | 帧率（fps） |
 * | -------- | --------------- | ----------- |
 * | "480p"     | 640 × 480       | 5           |
 * | "480p_1"   | 640 × 480       | 5           |
 * | "480p_2"   | 640 × 480       | 30          |
 * | "480p_3"   | 640 × 480       | 15          |
 * | "720p"     | 1280 × 720      | 5           |
 * | "720p_1"   | 1280 × 720      | 5           |
 * | "720p_2"   | 1280 × 720      | 30          |
 * | "720p_3"   | 1280 × 720      | 15          |
 * | "720p_auto" <sup>①</sup>  | 1280 × 720       | 30          |
 * | "1080p"    | 1920 × 1080     | 5           |
 * | "1080p_1"  | 1920 × 1080     | 5           |
 * | "1080p_2"  | 1920 × 1080     | 30          |
 * | "1080p_3"  | 1920 × 1080     | 15          |
 *
 * > <sup>①</sup> `"720p_auto"` 仅推荐在 Safari 上设置，用于确保能够动态调整编码分辨率，详见[发版说明](https://doc.shengwang.cn/doc/rtc/javascript/overview/release-notes)。
 *
 */
/** @en
 * The preset video encoder configurations for screen sharing.
 *
 * You can pass the preset video encoder configurations when calling [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack}.
 *
 * The following table lists all the preset video profiles for screen sharing.
 *
 * | Video Profile | Resolution (Width×Height) | Frame Rate (fps) |
 * | -------- | --------------- | ----------- |
 * | "480p"     | 640 × 480       | 5           |
 * | "480p_1"   | 640 × 480       | 5           |
 * | "480p_2"   | 640 × 480       | 30          |
 * | "480p_3"   | 640 × 480       | 15          |
 * | "720p"     | 1280 × 720      | 5           |
 * | "720p_1"   | 1280 × 720      | 5           |
 * | "720p_2"   | 1280 × 720      | 30          |
 * | "720p_3"   | 1280 × 720      | 15          |
 * | "720p_auto" <sup>①</sup>  | 1280 × 720       | 30          |
 * | "1080p"    | 1920 × 1080     | 5           |
 * | "1080p_1"  | 1920 × 1080     | 5           |
 * | "1080p_2"  | 1920 × 1080     | 30          |
 * | "1080p_3"  | 1920 × 1080     | 15          |
 *
 * > <sup>①</sup> `"720p_auto"` is only recommended to be set on Safari to ensure dynamic adjustment of the encoding resolution. For details, see the release notes.
 */
export declare type ScreenEncoderConfigurationPreset = keyof typeof SUPPORT_SCREEN_ENCODER_CONFIG_LIST;

/**
 * 屏幕共享源的类型。
 * - `"screen"`: 共享屏幕。
 * - `"application"`: 共享某一个 app 的所有窗口。
 * - `"window"`: 共享某一个 app 的某一个窗口。
 * @public
 */
/** @en
 * The type of the source for screen sharing.
 * - `"screen"`: Sharing the whole screen.
 * - `"application"`: Sharing all windows of an app.
 * - `"window"`: Sharing a window of an app.
 */
export declare type ScreenSourceType = "screen" | "window" | "application";

/**
 * 创建屏幕共享视频轨道时的配置对象，用于 [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack}。
 */
/** @en
 * Configurations for the video track for screen sharing. Set these configurations when calling [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack}.
 */
export declare interface ScreenVideoTrackInitConfig {
    /**
     * 控制屏幕共享视频的编码配置。
     *
     * 你可以通过以下方式设置该属性：
     * - 通过 [[ScreenEncoderConfigurationPreset]] 传入 SDK 内置的编码配置。
     * - 通过传入一个 [[VideoEncoderConfiguration]] 对象来自定义视频编码配置。
     * - 不传或留空，使用 SDK 的默认值 `"1080p_2"`（分辨率 1920 × 1080、帧率 30 fps、码率 	3000 Kbps）。
     */
    /** @en
     * The video encoder configurations for screen sharing.
     *
     * You can set the video encoder configurations in either of the following ways:
     * - Pass the preset video encoder configurations by using [[ScreenEncoderConfigurationPreset]].
     * - Pass your customized video encoder configurations by using [[VideoEncoderConfiguration]].
     * - Leave this property empty to use the SDK's default value, `"1080p_2"` (resolution: 1920 × 1080, frame rate: 30 fps, bitrate: 3000 Kbps).
     */
    encoderConfig?: VideoEncoderConfiguration | ScreenEncoderConfigurationPreset;
    /**
     * 使用 Electron 屏幕共享时，传入 `sourceId`。
     */
    /** @en
     * The `sourceId` when you share the screen through Electron.
     */
    electronScreenSourceId?: string;
    /**
     * 使用 Chrome 浏览器插件进行屏幕共享时，传入插件的 ID。
     */
    /** @en
     * The `extensionId` when you share the screen with a Chrome extension.
     */
    extensionId?: string;
    /**
     * @deprecated
     *
     * 自 v4.17.1 起废弃，请改用 {@link displaySurface}。
     *
     * 屏幕共享源的类型。
     */
    /** @en
     *
     * @deprecated from v4.17.1. Use {@link displaySurface} instead.
     *
     * The type of the source for screen sharing.
     */
    screenSourceType?: ScreenSourceType;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.2.0*
     *
     * 设置视频传输优化模式。
     *
     * 你可以在视频通话、视频直播或屏幕共享过程中调用此方法动态调整视频的传输优化模式。例如你想要把屏幕共享内容从演示文稿切换为视频时，你可以将传输优化模式从 `"detail"` 切换为 `"motion"`，确保视频画面在网络波动时不会出现卡顿。
     *
     * > 注意事项：该方法只支持 Chrome 浏览器。
     *
     * @param mode 视频传输优化模式：
     * - `"detail"`: 清晰优先。
     *   - 浏览器会自动根据你的采集分辨率和帧率设定一个最小码率。即使遭遇网络波动，发送码率也不会低于这个值，从而确保清晰的视频画面。
     *   - 大部分情况下，浏览器不会降低发送分辨率，但是可能会降低帧率。
     * - `"motion"`: 自 4.21.0 开始，浏览器默认开启流畅优先模式。
     *   - 浏览器不会启用最小码率策略。遭遇网络波动时，发送端会降低码率来确保接收端的视频画面不会出现中断和卡顿。
     *   - 大部分情况下，浏览器不会降低帧率，但是可能会降低发送分辨率。
     */
    /** @en
     * @since
     * <br>&emsp;&emsp;&emsp;*4.2.0*
     *
     * Sets the video transmission optimization mode.
     *
     * You can call this method during a video call, a live streaming or screen sharing to dynamically change the optimization mode. For example, during the screen sharing, before you change the shared content from text to video, you can change the optimization mode from `"detail"` to `"motion"` to ensure smoothness in poor network conditions.
     *
     * > Note: This method supports Chrome only.
     *
     * @param mode The video transmission optimization mode:
     * - `"detail"`: Prioritizes video quality.
     *   - The browser ensures high-quality images by automatically calculating a minimum bitrate based on the capturing resolution and frame rate. No matter how poor the network condition is, the sending bitrate will never be lower than the minimum value.
     *   - In most cases, the browser does not reduce the sending resolution, but may reduce the frame rate.
     * -  `"motion"`: Since v4.21.0, the browser prioritizes video smoothness.
     *   - In poor network conditions, the browser reduces the sending bitrate to minimize video freezes.
     *   - In most cases, the browser does not reduce the frame rate, but may reduce the sending resolution.
     */
    optimizationMode?: "motion" | "detail";
    /**
     * @ignore
     *
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.18.0*
     *
     * SVC （可伸缩视频编码）配置。
     *
     * 你可以通过 {@link SVCConfigurationPreset} 传入 SDK 预设的 SVC 编码配置，也可以通过 {@link SVCConfiguration} 传入自定义的 SVC 编码配置。
     */
    /** @en
     * @ignore
     *
     * @since
     * <br>&emsp;&emsp;&emsp;*4.18.0*
     *
     * Configurations for Scalable Video Coding (SVC).
     *
     * You can set the configurations using one of the following options:
     * - Use the preset SVC configurations provided by the SDK through {@link SVCConfigurationPreset}.
     * - Use your custom SVC configurations through {@link SVCConfiguration}.
     */
    scalabiltyMode?: SVCConfiguration | SVCConfigurationPreset;
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * 默认选中的共享内容类型，支持设为标签页、应用窗口（如下图）、或屏幕。详见 [displaySurface](https://developer.chrome.com/docs/web-platform/screen-sharing-controls/#displaySurface)。
     *
     * ![](https://wd.imgix.net/image/vvhSqZboQoZZN9wBvoXq72wzGAf1/AaQIUrKKCvoNuaBjvGOM.png?auto=format&w=1600)
     *
     * > 注意事项：该属性仅支持 Chrome 107 及以上版本、Edge 107 及以上版本。
     *
     */
    /** @en
     *
     * @since
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * The pre-selected pane in the media picker. See [displaySurface](https://developer.chrome.com/docs/web-platform/screen-sharing-controls/#displaySurface) for details.
     *
     * > Note: This property is supported on Chrome 107 and later, as well as Edge 107 and later.
     */
    displaySurface?: "browser" | "window" | "monitor";
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * 是否允许用户共享当前标签页：
     * - `"include"`：（默认）允许用户共享当前标签页。
     * - `"exclude"`：禁止用户共享当前标签页。
     *
     * 详见 [selfBrowserSurface](https://developer.chrome.com/docs/web-platform/screen-sharing-controls/#selfBrowserSurface)。
     *
     * > 注意事项：
     * > - Chrome 107 上该属性的默认值为 `"exclude"`，但 SDK 为保证兼容性将默认值改为了 `"include"`，以免用户升级到 Chrome 107 后无法共享当前标签页。
     * > - 该属性仅支持 Chrome 107 及以上版本、Edge 107 及以上版本。
     */
    /** @en
     *
     * @since
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * Whether to allow the user to share the current tab:
     * - `"include"`: (Default) Allows the user to share the current tab.
     * - `"exclude"`: Prevents the user from sharing the current tab.
     *
     * See [displaySurface](https://developer.chrome.com/docs/web-platform/screen-sharing-controls/#displaySurface) for details.
     *
     * > Note:
     * > - This property is defaulted to `"exclude"` on Chrome 107. For better compatibility with earlier versions, the SDK changes the default value to "include", which ensures that users can still share the current tab after upgrading to Chrome 107.
     * > - This property is supported on Chrome 107 and later, as well as Edge 107 and later.
     */
    selfBrowserSurface?: "include" | "exclude";
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * 是否允许用户动态切换共享的标签页：
     * - `"include"`：（默认）允许用户动态切换共享的标签页。
     * - `"exclude"`：禁止用户动态切换共享的标签页。
     *
     * 详见 [surfaceSwitching](https://developer.chrome.com/docs/web-platform/screen-sharing-controls/#surfaceSwitching)。
     *
     * > 注意事项：该属性仅支持 Chrome 107 及以上版本、Edge 107 及以上版本。
     */
    /** @en
     *
     * @since
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * Whether to allow the user to dynamically switch between shared tabs:
     * - `"include"`: (Default) The user can dynamically switch between shared tabs.
     * - `"exclude"`: The user cannot dynamically switch between shared tabs.
     *
     * See [surfaceSwitching](https://developer.chrome.com/docs/web-platform/screen-sharing-controls/#surfaceSwitching) for details.
     *
     * > Note: This property is supported on Chrome 107 and later, as well as Edge 107 and later.
     */
    surfaceSwitching?: "include" | "exclude";
    /**
     * @自从
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * 是否允许采集系统音频：
     * - `"include"`：（默认）允许采集系统音频。
     * - `"exclude"`：禁止采集系统音频。
     *
     * 详见 [systemAudio](https://developer.chrome.com/docs/web-platform/screen-sharing-controls/#systemAudio)。
     *
     * > 注意事项：该属性仅支持 Windows 设备上的 Chrome 105 及以上版本、Edge 105 及以上版本。
     */
    /** @en
     *
     * @since
     * <br>&emsp;&emsp;&emsp;*4.17.1*
     *
     * Whether to capture system audio:
     * - `"include"`: (Default) Captures system audio.
     * - `"exclude"`: Avoids capturing system audio
     *
     * See [systemAudio](https://developer.chrome.com/docs/web-platform/screen-sharing-controls/#systemAudio) for details.
     *
     * > Note: This property is supported on Chrome 105 and later (Windows only), as well as Edge 105 and later (Windows only).
     */
    systemAudio?: "include" | "exclude";
}

export declare type SenderConfig = {
    bitrateMax: number;
    bitrateMin: number;
    scaleResolutionDownBy: number;
};

/**
 * 将节点输入的 AudioBuffer 返回，将节点输出的 AudioBuffer 设置成 silence
 */
export declare function silenceScriptProcessHandler(e: AudioProcessingEvent): AudioBuffer;

export declare interface StatsRegistry {
    processorID: string;
    processorName: string;
    type: string;
    cb: Function;
}

export declare enum StreamType {
    /** 0: 高分辨率、高码率的视频大流。 */
    /** @en
     * 0: High-quality video stream (high-bitrate, high-resolution).
     */
    HIGH_STREAM = 0,
    /** 1: 低分辨率、低码率的视频小流。 */
    /** @en
     * 1: Low-quality video stream (low-bitrate, low-resolution).
     */
    LOW_STREAM = 1
}

export declare const SUPPORT_720P_AUTO_CONFIG_LIST: Required<{
    scaleResolutionDownBy: number;
    width: number;
    height: number;
    frameRate: number;
    bitrateMin: number;
    bitrateMax: number;
}>[];

/** @en
 * @ignore
 */
export declare const SUPPORT_SCREEN_ENCODER_CONFIG_LIST: Record<string, VideoEncoderConfiguration>;

/** @en
 * @ignore
 */
export declare const SUPPORT_SVC_CONFIG_LIST: Record<string, SVCConfiguration>;

/** @en
 * @ignore
 */
export declare const SUPPORT_VIDEO_ENCODER_CONFIG_LIST: Record<string, VideoEncoderConfiguration>;

/**
 * @ignore
 * @自从
 * <br>&emsp;&emsp;&emsp;*4.18.0*
 *
 * 自定义的 SVC 编码配置。
 *
 * 你可以在以下方法中传入 `SVCConfiguration` 来控制本地视频的 SVC 编码配置：
 * - [AgoraRTC.createCameraVideoTrack]{@link IAgoraRTC.createCameraVideoTrack}
 * - [AgoraRTC.createCustomVideoTrack]{@link IAgoraRTC.createCustomVideoTrack}
 * - [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack}
 */
/** @en
 * @ignore
 * @since
 * <br>&emsp;&emsp;&emsp;*4.18.0*
 *
 * Custom SVC encoding configurations.
 *
 * You can control the SVC configurations for local video by passing `SVCConfiguration` in the following methods:
 * - [AgoraRTC.createCameraVideoTrack]{@link IAgoraRTC.createCameraVideoTrack}
 * - [AgoraRTC.createCustomVideoTrack]{@link IAgoraRTC.createCustomVideoTrack}
 * - [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack}
 */
export declare interface SVCConfiguration {
    /**
     * SVC 空域层数。
     */
    /** @en
     * Specifies the number of spatial layer in SVC.
     */
    numSpatialLayers: 1 | 2 | 3;
    /**
     * SVC 时域层数。
     */
    /** @en
     * Specifies the number of temporal layer in SVC.
     */
    numTemporalLayers: 1 | 3;
}

/**
 * @ignore
 *
 * @自从
 * <br>&emsp;&emsp;&emsp;*4.18.0*
 *
 * SDK 预设的 SVC 编码配置：
 * - `"1SL1TL"`: 空域层数为 1，时域层数为 1。
 * - `"3SL3TL"`: 空域层数为 3，时域层数为 3。
 * - `"2SL3TL"`: 空域层数为 2，时域层数为 3。
 *
 * 你可以在以下方法中传入这些预设值，来控制本地视频的 SVC 编码配置：
 * - [AgoraRTC.createCameraVideoTrack]{@link IAgoraRTC.createCameraVideoTrack}
 * - [AgoraRTC.createCustomVideoTrack]{@link IAgoraRTC.createCustomVideoTrack}
 * - [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack}
 *
 */
/** @en
 * @ignore
 *
 * @since
 * <br>&emsp;&emsp;&emsp;*4.18.0*
 *
 * The preset SVC configurations provided by the SDK.
 * - `"1SL1TL"`: 1 spatial layer, 1 temporal layer.
 * - `"3SL3TL"`: 3 spatial layers, 3 temporal layers.
 * - `"2SL3TL"`: 2 spatial layers, 3 temporal layers.
 *
 * You can control the SVC configurations for local video by passing these preset values in the following methods:
 * - [AgoraRTC.createCameraVideoTrack]{@link IAgoraRTC.createCameraVideoTrack}
 * - [AgoraRTC.createCustomVideoTrack]{@link IAgoraRTC.createCustomVideoTrack}
 * - [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack}
 *
 */
export declare type SVCConfigurationPreset = keyof typeof SUPPORT_SVC_CONFIG_LIST;

export declare abstract class Track extends EventEmitter implements ITrack {
    abstract readonly trackMediaType: "audio" | "video";
    private _ID;
    protected _rtpTransceiver?: RTCRtpTransceiver;
    protected _lowRtpTransceiver?: RTCRtpTransceiver;
    abstract get isPlaying(): boolean;
    /** 用于描述这个 Track 的一些额外信息 */
    _hints: TrackHint[];
    _isClosed: boolean;
    /** 原始的 Track 对象，来自于采集/用户传入/订阅返回 */
    _originMediaStreamTrack: MediaStreamTrack;
    /** 用于播放和发布的 Track 对象，可能被各种模块修改，一般来说等于 `_originMediaStreamTrack` */
    mediaStreamTrack: MediaStreamTrack;
    set _mediaStreamTrack(track: MediaStreamTrack);
    get _mediaStreamTrack(): MediaStreamTrack;
    /** 提供给外部插件以同步数据 */
    _external: ExternalMethods;
    constructor(track: MediaStreamTrack, trackId?: string);
    toString(): string;
    getTrackId(): string;
    getMediaStreamTrack(fromInternal?: boolean): MediaStreamTrack;
    getRTCRtpTransceiver(type?: StreamType): RTCRtpTransceiver | undefined;
    getMediaStreamTrackSettings(): MediaTrackSettings;
    protected close(): void;
    abstract play(element?: HTMLElement | string): void;
    abstract stop(): void;
    _updateRtpTransceiver(transceiver?: RTCRtpTransceiver, type?: StreamType): void;
}

export declare enum TrackEvents {
    TRANSCEIVER_UPDATED = "transceiver-updated",
    SEI_TO_SEND = "sei-to-send",
    SEI_RECEIVED = "sei-received",
    TRACK_UPDATED = "track-updated"
}

/**
 * 标识新生成轨道的来源。用于 track.clone 方法。
 */
export declare enum TrackHint {
    /** 该轨道是克隆屏幕共享视频轨道生成的。 */
    SCREEN_TRACK = "screen_track",
    /** 该轨道是克隆自定义视频轨道生成的。 */
    CUSTOM_TRACK = "custome_track",
    /** 该轨道是克隆小流视频轨道生成的。 */
    LOW_STREAM = "low_stream",
    /** 标记这个 Track 是否来自于屏幕共享小流 */
    SCREEN_LOW_TRACK = "screen_low_track"
}

export declare enum TrackInternalEvent {
    NEED_RENEGOTIATE = "@need_renegotiate",//deprecated
    NEED_REPLACE_TRACK = "@need_replace_track",
    NEED_REPLACE_MIXING_TRACK = "@need_replace_mixing_track",
    NEED_CLOSE = "@need_close",
    NEED_ENABLE_TRACK = "@need_enable_track",
    NEED_DISABLE_TRACK = "@need_disable_track",
    NEED_SESSION_ID = "@need_sid",
    SET_OPTIMIZATION_MODE = "@set_optimization_mode",
    GET_STATS = "@get_stats",
    GET_RTC_STATS = "@get_rtc_stats",
    GET_LOW_VIDEO_TRACK = "@get_low_video_track",
    NEED_RESET_REMOTE_SDP = "@need_reset_remote_sdp",//deprecated
    NEED_UPDATE_VIDEO_ENCODER = "@need_update_video_encoder",
    NEED_UPDATE_VIDEO_SEND_PARAMETERS = "@need_update_video_send_parameters",
    NEED_MUTE_TRACK = "@need_mute_track",
    NEED_UNMUTE_TRACK = "@need_unmute_track"
}

export declare enum TrackMediaType {
    AUDIO = "audio",
    VIDEO = "video",
    DATA = "data"
}

export declare function updateAgoraRTCCompatibility(): void;

export declare interface UsageRegistry {
    processorID: string;
    processorName: string;
    cb: () => Usage | Promise<Usage>;
}

/**
 * 定义视频编码参数的配置对象。
 *
 * 你可以在 [AgoraRTC.createCameraVideoTrack]{@link IAgoraRTC.createCameraVideoTrack} 或者 [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack} 方法中传入自定义的视频编码配置。
 *
 * SDK 提供了一些预设视频编码配置，详见 [[VideoEncoderConfigurationPreset]]。
 *
 * > 受操作系统、浏览器影响，实际的码率可能会和设置的取值范围不同。我们建议将码率设置在 100 Kbps 到 5000 Kbps 之间。
 *
 * @public
 */
/** @en
 * `VideoEncoderConfiguration` is the interface that defines the video encoder configurations.
 *
 * You can customize the video encoder configurations when calling [AgoraRTC.createCameraVideoTrack]{@link IAgoraRTC.createCameraVideoTrack} or [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack}.
 *
 * The SDK provides the preset video encoder configurations. For more information, see [[VideoEncoderConfigurationPreset]].
 *
 * > The actual bitrate may differ slightly from the value you set due to the limitations of the operation system or the web browser. Agora recommends setting the bitrate between 100 Kbps and 5000 Kbps.
 */
export declare interface VideoEncoderConfiguration {
    /**
     * 视频的分辨率宽。
     *
     * 支持传入 `number`，或一个约束对象，如 `{ max: 1280, min: 720 }`。
     *
     * 关于约束对象的详细说明可以参考 [ConstrainLong]{@link ConstrainLong}。
     */
    /** @en
     * Width of the video.
     *
     * You can pass a `number`, or a constraint such as `{ max: 1280, min: 720 }`.
     *
     * For more details about the constraint, see [ConstrainLong]{@link ConstrainLong}.
     */
    width?: number | ConstrainLong;
    /**
     * 视频的分辨率高。
     *
     * 支持传入 `number`，或一个约束对象，如 `{ max: 1280, min: 720 }`。
     *
     * 关于约束对象的详细说明可以参考 [ConstrainLong]{@link ConstrainLong}。
     */
    /** @en
     * Height of the video.
     *
     * You can pass a `number`, or a constraint such as `{ max: 1280, min: 720 }`.
     *
     * For more details about the constraint, see [ConstrainLong]{@link ConstrainLong}.
     */
    height?: number | ConstrainLong;
    /**
     * 视频帧率，单位为 fps。
     *
     * 支持传入 `number`，或一个约束对象，如 `{ max: 30, min: 5 }`。
     *
     * 关于约束对象的详细说明可以参考 [ConstrainLong]{@link ConstrainLong}。
     */
    /** @en
     * Frame rate of the video (fps).
     *
     * You can pass a `number`, or a constraint such as `{ max: 30, min: 5 }`.
     *
     * For details about the constraint, see [ConstrainLong]{@link ConstrainLong}.
     */
    frameRate?: number | ConstrainLong;
    /**
     * 传输过程中的最小码率，单位为 Kbps。
     */
    /** @en
     * The minimum bitrate of the video (Kbps).
     */
    bitrateMin?: number;
    /**
     * 传输过程中的最大码率，单位为 Kbps。
     */
    /** @en
     * The maximum bitrate of the video (Kbps).
     */
    bitrateMax?: number;
    /**
     * @ignore
     */
    /** @en
     * @ignore
     */
    scaleResolutionDownBy?: number;
}

/**
 * SDK 预设的 [[VideoEncoderConfiguration]] 配置。
 *
 * 你可以在 [AgoraRTC.createCameraVideoTrack]{@link IAgoraRTC.createCameraVideoTrack} 方法中传入预设的视频编码配置。
 *
 * 下表列出了 SDK 所有内置的视频属性配置，SDK 默认使用 `"480p_1"`。
 *
 * | 视频属性 | 分辨率（宽×高） | 帧率（fps） | Chrome | Firefox | Safari |
 * | -------- | --------------- | ----------- | ------ | ------- | ------ |
 * | 120p     | 160 × 120       | 15          | ✓      |         |        |
 * | 120p_1   | 160 × 120       | 15          | ✓      |         |        |
 * | 120p_3   | 120 × 120       | 15          | ✓      |         |        |
 * | 180p     | 320 × 180       | 15          | ✓      |         |        |
 * | 180p_1   | 320 × 180       | 15          | ✓      |         |        |
 * | 180p_3   | 180 × 180       | 15          | ✓      |         |        |
 * | 180p_4   | 240 × 180       | 15          | ✓      |         |        |
 * | 240p     | 320 × 240       | 15          | ✓      |         |        |
 * | 240p_1   | 320 × 240       | 15          | ✓      |         |        |
 * | 240p_3   | 240 × 240       | 15          | ✓      |         |        |
 * | 240p_4   | 424 × 240       | 15          | ✓      |         |        |
 * | 360p     | 640 × 360       | 15          | ✓      |         |        |
 * | 360p_1   | 640 × 360       | 15          | ✓      |         |        |
 * | 360p_3   | 360 × 360       | 15          | ✓      |         |        |
 * | 360p_4   | 640 × 360       | 30          | ✓      |         |        |
 * | 360p_6   | 360 × 360       | 30          | ✓      |         |        |
 * | 360p_7   | 480 × 360       | 15          | ✓      |         |        |
 * | 360p_8   | 480 × 360       | 30          | ✓      |         |        |
 * | 360p_9   | 640 × 360       | 15          | ✓      |         |        |
 * | 360p_10  | 640 × 360       | 24          | ✓      |         |        |
 * | 360p_11  | 640 × 360       | 24          | ✓      |         |        |
 * | 480p     | 640 × 480       | 15          | ✓      | ✓       | ✓      |
 * | 480p_1   | 640 × 480       | 15          | ✓      | ✓       | ✓      |
 * | 480p_2   | 640 × 480       | 30          | ✓      | ✓       | ✓      |
 * | 480p_3   | 480 × 480       | 15          | ✓      | ✓       | ✓      |
 * | 480p_4   | 640 × 480       | 30          | ✓      | ✓       | ✓      |
 * | 480p_6   | 480 × 480       | 30          | ✓      | ✓       | ✓      |
 * | 480p_8   | 848 × 480       | 15          | ✓      | ✓       | ✓      |
 * | 480p_9   | 848 × 480       | 30          | ✓      | ✓       | ✓      |
 * | 480p_10  | 640 × 480       | 10          | ✓      | ✓       | ✓      |
 * | 720p     | 1280 × 720      | 15          | ✓      | ✓       | ✓      |
 * | 720p_1   | 1280 × 720      | 15          | ✓      | ✓       | ✓      |
 * | 720p_2   | 1280 × 720      | 30          | ✓      | ✓       | ✓      |
 * | 720p_3   | 1280 × 720      | 30          | ✓      | ✓       | ✓      |
 * | 720p_auto <sup>①</sup>  | 1280 × 720       | 30          |✓      | ✓       | ✓      |
 * | 720p_5   | 960 × 720       | 15          | ✓      | ✓       | ✓      |
 * | 720p_6   | 960 × 720       | 30          | ✓      | ✓       | ✓      |
 * | 1080p    | 1920 × 1080     | 15          | ✓      |         | ✓      |
 * | 1080p_1  | 1920 × 1080     | 15          | ✓      |         | ✓      |
 * | 1080p_2  | 1920 × 1080     | 30          | ✓      |         | ✓      |
 * | 1080p_3  | 1920 × 1080     | 30          | ✓      |         | ✓      |
 * | 1080p_5  | 1920 × 1080     | 60          | ✓      |         | ✓      |
 *
 * > <sup>①</sup> `"720p_auto"` 仅推荐在 Safari 上设置，用于确保能够动态调整编码分辨率，详见[发版说明](https://doc.shengwang.cn/doc/rtc/javascript/overview/release-notes)。
 *
 * @public
 */
/** @en
 *
 * The preset video encoder configurations.
 *
 * You can pass the preset video encoder configurations when calling [AgoraRTC.createCameraVideoTrack]{@link IAgoraRTC.createCameraVideoTrack} or [AgoraRTC.createScreenVideoTrack]{@link IAgoraRTC.createScreenVideoTrack}.
 *
 * The following table lists all the preset video profiles. The SDK uses `"480p_1"` by default.
 *
 * | Video Profile | Resolution (Width×Height) | Frame Rate (fps) | Chrome | Firefox | Safari |
 * | -------- | --------------- | ----------- | ------ | ------- | ------ |
 * | 120p     | 160 × 120       | 15          | ✓      |         |        |
 * | 120p_1   | 160 × 120       | 15          | ✓      |         |        |
 * | 120p_3   | 120 × 120       | 15          | ✓      |         |        |
 * | 180p     | 320 × 180       | 15          | ✓      |         |        |
 * | 180p_1   | 320 × 180       | 15          | ✓      |         |        |
 * | 180p_3   | 180 × 180       | 15          | ✓      |         |        |
 * | 180p_4   | 240 × 180       | 15          | ✓      |         |        |
 * | 240p     | 320 × 240       | 15          | ✓      |         |        |
 * | 240p_1   | 320 × 240       | 15          | ✓      |         |        |
 * | 240p_3   | 240 × 240       | 15          | ✓      |         |        |
 * | 240p_4   | 424 × 240       | 15          | ✓      |         |        |
 * | 360p     | 640 × 360       | 15          | ✓      |         |        |
 * | 360p_1   | 640 × 360       | 15          | ✓      |         |        |
 * | 360p_3   | 360 × 360       | 15          | ✓      |         |        |
 * | 360p_4   | 640 × 360       | 30          | ✓      |         |        |
 * | 360p_6   | 360 × 360       | 30          | ✓      |         |        |
 * | 360p_7   | 480 × 360       | 15          | ✓      |         |        |
 * | 360p_8   | 480 × 360       | 30          | ✓      |         |        |
 * | 360p_9   | 640 × 360       | 15          | ✓      |         |        |
 * | 360p_10  | 640 × 360       | 24          | ✓      |         |        |
 * | 360p_11  | 640 × 360       | 24          | ✓      |         |        |
 * | 480p     | 640 × 480       | 15          | ✓      | ✓       | ✓      |
 * | 480p_1   | 640 × 480       | 15          | ✓      | ✓       | ✓      |
 * | 480p_2   | 640 × 480       | 30          | ✓      | ✓       | ✓      |
 * | 480p_3   | 480 × 480       | 15          | ✓      | ✓       | ✓      |
 * | 480p_4   | 640 × 480       | 30          | ✓      | ✓       | ✓      |
 * | 480p_6   | 480 × 480       | 30          | ✓      | ✓       | ✓      |
 * | 480p_8   | 848 × 480       | 15          | ✓      | ✓       | ✓      |
 * | 480p_9   | 848 × 480       | 30          | ✓      | ✓       | ✓      |
 * | 480p_10  | 640 × 480       | 10          | ✓      | ✓       | ✓      |
 * | 720p     | 1280 × 720      | 15          | ✓      | ✓       | ✓      |
 * | 720p_1   | 1280 × 720      | 15          | ✓      | ✓       | ✓      |
 * | 720p_2   | 1280 × 720      | 30          | ✓      | ✓       | ✓      |
 * | 720p_3   | 1280 × 720      | 30          | ✓      | ✓       | ✓      |
 * | 720p_auto <sup>①</sup>  | 1280 × 720       | 30          | ✓      | ✓       | ✓      |
 * | 720p_5   | 960 × 720       | 15          | ✓      | ✓       | ✓      |
 * | 720p_6   | 960 × 720       | 30          | ✓      | ✓       | ✓      |
 * | 1080p    | 1920 × 1080     | 15          | ✓      |         | ✓      |
 * | 1080p_1  | 1920 × 1080     | 15          | ✓      |         | ✓      |
 * | 1080p_2  | 1920 × 1080     | 30          | ✓      |         | ✓      |
 * | 1080p_3  | 1920 × 1080     | 30          | ✓      |         | ✓      |
 * | 1080p_5  | 1920 × 1080     | 60          | ✓      |         | ✓      |
 *
 * > <sup>①</sup> `"720p_auto"` is only recommended to be set on Safari to ensure dynamic adjustment of the encoding resolution. For details, see the release notes.
 */
export declare type VideoEncoderConfigurationPreset = keyof typeof SUPPORT_VIDEO_ENCODER_CONFIG_LIST;

/**
 * @internal
 */
declare class VideoPlayer {
    trackId: string;
    config: PlayerConfig;
    onFirstVideoFrameDecoded?: () => void;
    onVideoStateChanged?: (state: VideoState) => void;
    freezeTimeCounterList: number[];
    renderFreezeAccTime: number;
    isKeepLastFrame: boolean;
    /** 计算 500ms 卡顿时间用 */
    private timeUpdatedCount;
    private freezeTime;
    private playbackTime;
    private lastTimeUpdatedTime;
    /** 记录初始autoplay 情况，用于safari异常状态处理 */
    private autoplayFailed;
    protected videoTrack?: MediaStreamTrack;
    protected videoElement: HTMLVideoElement;
    protected cacheVideoElement?: HTMLVideoElement;
    private renderStats?;
    get rendFrameRate(): number;
    get videoElementStatus(): MediaElementStatus;
    set videoElementStatus(status: MediaElementStatus);
    get videoState(): VideoState;
    set videoState(state: VideoState);
    private _videoState;
    private videoElementCheckInterval?;
    private videoElementFreezeTimeout?;
    private _videoElementStatus;
    private isGettingVideoDimensions;
    constructor(config: PlayerConfig);
    getVideoElement(): HTMLVideoElement | undefined;
    getContainerElement(): HTMLDivElement | undefined;
    updateConfig(config: PlayerConfig): void;
    updateVideoTrack(track?: MediaStreamTrack): void;
    play(sessionId?: string): void;
    getCurrentFrame(): ImageData;
    getCurrentFrameToUint8Array(type: string, quality?: number): Promise<ImageTypedData>;
    destroy(): void;
    protected initVideoElement(): void;
    protected resetVideoElement(): void;
    private startGetVideoDimensions;
    private handleAutoPlayFailed;
    private autoResumeAfterInterruption;
    private handleVideoEvents;
    private autoResumeAfterInterruptionOnIOS15_16;
}

/**
 * 视频轨道的播放配置。用于 [LocalVideoTrack.play]{@link ILocalVideoTrack.play}。
 */
/** @en
 * Playback configurations for a video track. Set the playback configurations for a video track when calling [ILocalVideoTrack.play]{@link ILocalVideoTrack.play}.
 */
export declare interface VideoPlayerConfig {
    /**
     * 设置是否开启镜像模式：
     * - `true`: 开启镜像模式。
     * - `false`: 关闭镜像模式。
     *
     * > 注意事项：
     * > - 对于本地视频轨道，镜像模式默认开启。
     * > - 对于远端视频轨道，镜像模式默认关闭。
     */
    /** @en
     * Sets whether to enable mirror mode:
     * - `true`: Enable mirror mode.
     * - `false`: Disable mirror mode.
     *
     * > Notes:
     * > - The SDK enables mirror mode for the local video track by default.
     * > - The SDK disables mirror mode for the remote video track by default.
     */
    mirror?: boolean;
    /**
     * 设置视频播放时的显示模式：
     * - `"cover"`: 优先保证视窗被填满。视频尺寸等比缩放，直至整个视窗被视频填满。如果视频长宽与显示窗口不同，则视频流会按照显示视窗的比例进行周边裁剪或图像拉伸后填满视窗。可以参考 CSS 属性中 `object-fit` 的 `cover` 选项。
     * - `"contain"`: 优先保证视频内容全部显示。视频尺寸等比缩放，直至视频窗口的一边与视窗边框对齐。如果视频尺寸与显示视窗尺寸不一致，在保持长宽比的前提下，将视频进行缩放后填满视窗，缩放后的视频四周会有一圈黑边。可以参考 CSS 属性中 `object-fit` 的 `contain` 选项。
     * - `"fill"`: 保证视窗被填满的同时保证视频内容全部显示，但是不保证视频尺寸比例不变。视频的宽高会被拉伸至和视窗尺寸一致。可以参考 CSS 属性中 `object-fit` 的 `fill` 选项。
     *
     * > 注意事项：
     * > - 播放由摄像头采集的本地视频轨道，默认使用 cover 模式；播放用于屏幕共享的视频轨道，默认使用 contain 模式。
     * > - 对于远端视频轨道，因为不知道轨道的类型，默认使用 cover 模式。
     */
    /** @en
     * Sets video display mode:
     * - `"cover"`: The image files the height and width of the box, while maintaining its aspect ratio but often cropping the image in the process. For more information, see the `cover` option of `object-fit` in CSS.
     * - `"contain"`: The size of the image increases or decreases to fill the box while preserving its aspect-ratio. Areas that are not filled due to the disparity in the aspect ratio are filled with black. For more information, see the `contain` option of `object-fit` in CSS.
     * - `"fill"`: The image stretches to fit the box, regardless of its aspect-ratio. For more information, see the `fill` option of `object-fit` in CSS.
     *
     * > Notes:
     * > - When playing the local camera video track, the SDK uses cover mode by default; when playing the local video track of screen sharing, the SDK uses contain mode by default.
     * > - When playing the remote video track, the SDK uses cover mode by default.
     */
    fit?: "cover" | "contain" | "fill";
}

export declare class VideoProcessorContext extends EventEmitter implements IProcessorContext {
    private constraintsMap;
    private statsRegistry;
    private usageRegistry;
    private readonly trackId;
    private readonly direction;
    private _chained;
    set chained(chained: boolean);
    get chained(): boolean;
    constructor(trackId: string, direction: "local" | "remote");
    getConstraints(): Promise<MediaTrackConstraints>;
    requestApplyConstraints(constraints: MediaTrackConstraints, processor: IBaseProcessor): Promise<void>;
    requestRevertConstraints(processor: IBaseProcessor): Promise<void>;
    registerStats(processor: IBaseProcessor, type: string, cb: () => any): void;
    unregisterStats(processor: IBaseProcessor, type: string): void;
    gatherStats(): ProcessorStats[];
    registerUsage(processor: IBaseProcessor, cb: () => Usage): void;
    unregisterUsage(processor: IBaseProcessor): void;
    gatherUsage(): Promise<UsageWithDirection[]>;
    getDirection(): "local" | "remote";
}

export declare class VideoProcessorDestination extends EventEmitter implements IBaseProcessor {
    name: string;
    ID: string;
    _source?: IBaseProcessor;
    private readonly videoContext;
    constructor(videoContext: VideoProcessorContext);
    private inputTrack?;
    get kind(): Kind;
    get enabled(): boolean;
    pipe(): IBaseProcessor;
    unpipe(): void;
    enable(): void;
    disable(): void;
    updateInput(inputOptions: {
        track?: MediaStreamTrack;
        node?: AudioNode;
        context: IProcessorContext;
    }): void;
    reset(): void;
}

/**
 * 视频流状态。
 * @public
 */
/** @en
 * The state of the video stream.
 */
export declare enum VideoState {
    /** 0: 视频默认初始状态。 */
    /** @en
     * 0: The initial state of the video.
     */
    VideoStateStopped = 0,
    /** 1: 本地用户已接收视频首包。 */
    /** @en
     * 1: The local user has received the first video packet.
     */
    VideoStateStarting = 1,
    /** 2: 视频流正在解码，正常播放。 */
    /** @en
     * 2: The video stream is being decoded and played normally.
     */
    VideoStateDecoding = 2,
    /** 3: 视频流卡顿。 */
    /** @en
     * 3: The video stream is frozen.
     */
    VideoStateFrozen = 3
}

declare class VisibilityWatcher extends EventEmitter {
    private _lastHiddenTime;
    private _lastVisibleTime;
    get visibility(): DocumentVisibilityState;
    get lastHiddenTime(): DOMHighResTimeStamp;
    get lastVisibleTime(): DOMHighResTimeStamp;
    constructor();
}

export declare const visibilityWatcher: VisibilityWatcher;

declare class VolumeLevelAnalyser {
    private readonly context;
    private analyserNode;
    private sourceNode?;
    constructor();
    updateSource(sourceNode?: AudioNode): void;
    getVolumeLevel(): number;
    getAnalyserNode(): AnalyserNode;
    rebuildAnalyser(): void;
    destroy(): void;
}

export { }
