function createAuth(appId, apiKey, authMode = 'WithinHeaders') {
  const credentials = {
    'x-algolia-api-key': apiKey,
    'x-algolia-application-id': appId
  };
  return {
    headers() {
      return authMode === 'WithinHeaders' ? credentials : {};
    },

    queryParameters() {
      return authMode === 'WithinQueryParameters' ? credentials : {};
    }

  };
}

function createBrowserLocalStorageCache(options) {
  let storage; // We've changed the namespace to avoid conflicts with v4, as this version is a huge breaking change

  const namespaceKey = `algolia-client-js-${options.key}`;

  function getStorage() {
    if (storage === undefined) {
      storage = options.localStorage || window.localStorage;
    }

    return storage;
  }

  function getNamespace() {
    return JSON.parse(getStorage().getItem(namespaceKey) || '{}');
  }

  return {
    get(key, defaultValue, events = {
      miss: () => Promise.resolve()
    }) {
      return Promise.resolve().then(() => {
        const keyAsString = JSON.stringify(key);
        const value = getNamespace()[keyAsString];
        return Promise.all([value || defaultValue(), value !== undefined]);
      }).then(([value, exists]) => {
        return Promise.all([value, exists || events.miss(value)]);
      }).then(([value]) => value);
    },

    set(key, value) {
      return Promise.resolve().then(() => {
        const namespace = getNamespace();
        namespace[JSON.stringify(key)] = value;
        getStorage().setItem(namespaceKey, JSON.stringify(namespace));
        return value;
      });
    },

    delete(key) {
      return Promise.resolve().then(() => {
        const namespace = getNamespace();
        delete namespace[JSON.stringify(key)];
        getStorage().setItem(namespaceKey, JSON.stringify(namespace));
      });
    },

    clear() {
      return Promise.resolve().then(() => {
        getStorage().removeItem(namespaceKey);
      });
    }

  };
}

function createNullCache() {
  return {
    get(_key, defaultValue, events = {
      miss: () => Promise.resolve()
    }) {
      const value = defaultValue();
      return value.then(result => Promise.all([result, events.miss(result)])).then(([result]) => result);
    },

    set(_key, value) {
      return Promise.resolve(value);
    },

    delete(_key) {
      return Promise.resolve();
    },

    clear() {
      return Promise.resolve();
    }

  };
}

function createFallbackableCache(options) {
  const caches = [...options.caches];
  const current = caches.shift();

  if (current === undefined) {
    return createNullCache();
  }

  return {
    get(key, defaultValue, events = {
      miss: () => Promise.resolve()
    }) {
      return current.get(key, defaultValue, events).catch(() => {
        return createFallbackableCache({
          caches
        }).get(key, defaultValue, events);
      });
    },

    set(key, value) {
      return current.set(key, value).catch(() => {
        return createFallbackableCache({
          caches
        }).set(key, value);
      });
    },

    delete(key) {
      return current.delete(key).catch(() => {
        return createFallbackableCache({
          caches
        }).delete(key);
      });
    },

    clear() {
      return current.clear().catch(() => {
        return createFallbackableCache({
          caches
        }).clear();
      });
    }

  };
}

function createMemoryCache(options = {
  serializable: true
}) {
  let cache = {};
  return {
    get(key, defaultValue, events = {
      miss: () => Promise.resolve()
    }) {
      const keyAsString = JSON.stringify(key);

      if (keyAsString in cache) {
        return Promise.resolve(options.serializable ? JSON.parse(cache[keyAsString]) : cache[keyAsString]);
      }

      const promise = defaultValue();
      return promise.then(value => events.miss(value)).then(() => promise);
    },

    set(key, value) {
      cache[JSON.stringify(key)] = options.serializable ? JSON.stringify(value) : value;
      return Promise.resolve(value);
    },

    delete(key) {
      delete cache[JSON.stringify(key)];
      return Promise.resolve();
    },

    clear() {
      cache = {};
      return Promise.resolve();
    }

  };
}

// By default, API Clients at Algolia have expiration delay of 5 mins.
// In the JavaScript client, we have 2 mins.
const EXPIRATION_DELAY = 2 * 60 * 1000;
function createStatefulHost(host, status = 'up') {
  const lastUpdate = Date.now();

  function isUp() {
    return status === 'up' || Date.now() - lastUpdate > EXPIRATION_DELAY;
  }

  function isTimedOut() {
    return status === 'timed out' && Date.now() - lastUpdate <= EXPIRATION_DELAY;
  }

  return { ...host,
    status,
    lastUpdate,
    isUp,
    isTimedOut
  };
}

function _defineProperty(obj, key, value) {
  if (key in obj) {
    Object.defineProperty(obj, key, {
      value: value,
      enumerable: true,
      configurable: true,
      writable: true
    });
  } else {
    obj[key] = value;
  }

  return obj;
}

class AlgoliaError extends Error {
  constructor(message, name) {
    super(message);

    _defineProperty(this, "name", 'AlgoliaError');

    if (name) {
      this.name = name;
    }
  }

}
class ErrorWithStackTrace extends AlgoliaError {
  constructor(message, stackTrace, name) {
    super(message, name); // the array and object should be frozen to reflect the stackTrace at the time of the error

    _defineProperty(this, "stackTrace", void 0);

    this.stackTrace = stackTrace;
  }

}
class RetryError extends ErrorWithStackTrace {
  constructor(stackTrace) {
    super('Unreachable hosts - your application id may be incorrect. If the error persists, contact support@algolia.com.', stackTrace, 'RetryError');
  }

}
class ApiError extends ErrorWithStackTrace {
  constructor(message, status, stackTrace) {
    super(message, stackTrace, 'ApiError');

    _defineProperty(this, "status", void 0);

    this.status = status;
  }

}
class DeserializationError extends AlgoliaError {
  constructor(message, response) {
    super(message, 'DeserializationError');

    _defineProperty(this, "response", void 0);

    this.response = response;
  }

}
function serializeUrl(host, path, queryParameters) {
  const queryParametersAsString = serializeQueryParameters(queryParameters);
  let url = `${host.protocol}://${host.url}/${path.charAt(0) === '/' ? path.substr(1) : path}`;

  if (queryParametersAsString.length) {
    url += `?${queryParametersAsString}`;
  }

  return url;
}
function serializeQueryParameters(parameters) {
  const isObjectOrArray = value => Object.prototype.toString.call(value) === '[object Object]' || Object.prototype.toString.call(value) === '[object Array]';

  return Object.keys(parameters).map(key => `${key}=${isObjectOrArray(parameters[key]) ? JSON.stringify(parameters[key]) : parameters[key]}`).join('&');
}
function serializeData(request, requestOptions) {
  if (request.method === 'GET' || request.data === undefined && requestOptions.data === undefined) {
    return undefined;
  }

  const data = Array.isArray(request.data) ? request.data : { ...request.data,
    ...requestOptions.data
  };
  return JSON.stringify(data);
}
function serializeHeaders(baseHeaders, requestHeaders, requestOptionsHeaders) {
  const headers = {
    Accept: 'application/json',
    ...baseHeaders,
    ...requestHeaders,
    ...requestOptionsHeaders
  };
  const serializedHeaders = {};
  Object.keys(headers).forEach(header => {
    const value = headers[header];
    serializedHeaders[header.toLowerCase()] = value;
  });
  return serializedHeaders;
}
function deserializeSuccess(response) {
  try {
    return JSON.parse(response.content);
  } catch (e) {
    throw new DeserializationError(e.message, response);
  }
}
function deserializeFailure({
  content,
  status
}, stackFrame) {
  let message = content;

  try {
    message = JSON.parse(content).message;
  } catch (e) {// ..
  }

  return new ApiError(message, status, stackFrame);
}

function isNetworkError({
  isTimedOut,
  status
}) {
  return !isTimedOut && ~~status === 0;
}
function isRetryable({
  isTimedOut,
  status
}) {
  return isTimedOut || isNetworkError({
    isTimedOut,
    status
  }) || ~~(status / 100) !== 2 && ~~(status / 100) !== 4;
}
function isSuccess({
  status
}) {
  return ~~(status / 100) === 2;
}

function stackTraceWithoutCredentials(stackTrace) {
  return stackTrace.map(stackFrame => stackFrameWithoutCredentials(stackFrame));
}
function stackFrameWithoutCredentials(stackFrame) {
  const modifiedHeaders = stackFrame.request.headers['x-algolia-api-key'] ? {
    'x-algolia-api-key': '*****'
  } : {};
  return { ...stackFrame,
    request: { ...stackFrame.request,
      headers: { ...stackFrame.request.headers,
        ...modifiedHeaders
      }
    }
  };
}

function createTransporter({
  hosts,
  hostsCache,
  baseHeaders,
  baseQueryParameters,
  algoliaAgent,
  timeouts,
  requester,
  requestsCache,
  responsesCache
}) {
  async function createRetryableOptions(compatibleHosts) {
    const statefulHosts = await Promise.all(compatibleHosts.map(compatibleHost => {
      return hostsCache.get(compatibleHost, () => {
        return Promise.resolve(createStatefulHost(compatibleHost));
      });
    }));
    const hostsUp = statefulHosts.filter(host => host.isUp());
    const hostsTimedOut = statefulHosts.filter(host => host.isTimedOut()); // Note, we put the hosts that previously timed out on the end of the list.

    const hostsAvailable = [...hostsUp, ...hostsTimedOut];
    const compatibleHostsAvailable = hostsAvailable.length > 0 ? hostsAvailable : compatibleHosts;
    return {
      hosts: compatibleHostsAvailable,

      getTimeout(timeoutsCount, baseTimeout) {
        /**
         * Imagine that you have 4 hosts, if timeouts will increase
         * on the following way: 1 (timed out) > 4 (timed out) > 5 (200).
         *
         * Note that, the very next request, we start from the previous timeout.
         *
         *  5 (timed out) > 6 (timed out) > 7 ...
         *
         * This strategy may need to be reviewed, but is the strategy on the our
         * current v3 version.
         */
        const timeoutMultiplier = hostsTimedOut.length === 0 && timeoutsCount === 0 ? 1 : hostsTimedOut.length + 3 + timeoutsCount;
        return timeoutMultiplier * baseTimeout;
      }

    };
  }

  async function retryableRequest(request, requestOptions, isRead = true) {
    const stackTrace = [];
    /**
     * First we prepare the payload that do not depend from hosts.
     */

    const data = serializeData(request, requestOptions);
    const headers = serializeHeaders(baseHeaders, request.headers, requestOptions.headers); // On `GET`, the data is proxied to query parameters.

    const dataQueryParameters = request.method === 'GET' ? { ...request.data,
      ...requestOptions.data
    } : {};
    const queryParameters = { ...baseQueryParameters,
      ...request.queryParameters,
      ...dataQueryParameters
    };

    if (algoliaAgent.value) {
      queryParameters['x-algolia-agent'] = algoliaAgent.value;
    }

    if (requestOptions && requestOptions.queryParameters) {
      for (const key of Object.keys(requestOptions.queryParameters)) {
        // We want to keep `undefined` and `null` values,
        // but also avoid stringifying `object`s, as they are
        // handled in the `serializeUrl` step right after.
        if (!requestOptions.queryParameters[key] || Object.prototype.toString.call(requestOptions.queryParameters[key]) === '[object Object]') {
          queryParameters[key] = requestOptions.queryParameters[key];
        } else {
          queryParameters[key] = requestOptions.queryParameters[key].toString();
        }
      }
    }

    let timeoutsCount = 0;

    const retry = async (retryableHosts, getTimeout) => {
      /**
       * We iterate on each host, until there is no host left.
       */
      const host = retryableHosts.pop();

      if (host === undefined) {
        throw new RetryError(stackTraceWithoutCredentials(stackTrace));
      }

      let responseTimeout = requestOptions.timeout;

      if (responseTimeout === undefined) {
        responseTimeout = isRead ? timeouts.read : timeouts.write;
      }

      const payload = {
        data,
        headers,
        method: request.method,
        url: serializeUrl(host, request.path, queryParameters),
        connectTimeout: getTimeout(timeoutsCount, timeouts.connect),
        responseTimeout: getTimeout(timeoutsCount, responseTimeout)
      };
      /**
       * The stackFrame is pushed to the stackTrace so we
       * can have information about onRetry and onFailure
       * decisions.
       */

      const pushToStackTrace = response => {
        const stackFrame = {
          request: payload,
          response,
          host,
          triesLeft: retryableHosts.length
        };
        stackTrace.push(stackFrame);
        return stackFrame;
      };

      const response = await requester.send(payload);

      if (isRetryable(response)) {
        const stackFrame = pushToStackTrace(response); // If response is a timeout, we increase the number of timeouts so we can increase the timeout later.

        if (response.isTimedOut) {
          timeoutsCount++;
        }
        /**
         * Failures are individually sent to the logger, allowing
         * the end user to debug / store stack frames even
         * when a retry error does not happen.
         */
        // eslint-disable-next-line no-console -- this will be fixed by exposing a `logger` to the transporter


        console.log('Retryable failure', stackFrameWithoutCredentials(stackFrame));
        /**
         * We also store the state of the host in failure cases. If the host, is
         * down it will remain down for the next 2 minutes. In a timeout situation,
         * this host will be added end of the list of hosts on the next request.
         */

        await hostsCache.set(host, createStatefulHost(host, response.isTimedOut ? 'timed out' : 'down'));
        return retry(retryableHosts, getTimeout);
      }

      if (isSuccess(response)) {
        return deserializeSuccess(response);
      }

      pushToStackTrace(response);
      throw deserializeFailure(response, stackTrace);
    };
    /**
     * Finally, for each retryable host perform request until we got a non
     * retryable response. Some notes here:
     *
     * 1. The reverse here is applied so we can apply a `pop` later on => more performant.
     * 2. We also get from the retryable options a timeout multiplier that is tailored
     * for the current context.
     */


    const compatibleHosts = hosts.filter(host => host.accept === 'readWrite' || (isRead ? host.accept === 'read' : host.accept === 'write'));
    const options = await createRetryableOptions(compatibleHosts);
    return retry([...options.hosts].reverse(), options.getTimeout);
  }

  function createRequest(request, requestOptions = {}) {
    /**
     * A read request is either a `GET` request, or a request that we make
     * via the `read` transporter (e.g. `search`).
     */
    const isRead = request.useReadTransporter || request.method === 'GET';

    if (!isRead) {
      /**
       * On write requests, no cache mechanisms are applied, and we
       * proxy the request immediately to the requester.
       */
      return retryableRequest(request, requestOptions, isRead);
    }

    const createRetryableRequest = () => {
      /**
       * Then, we prepare a function factory that contains the construction of
       * the retryable request. At this point, we may *not* perform the actual
       * request. But we want to have the function factory ready.
       */
      return retryableRequest(request, requestOptions);
    };
    /**
     * Once we have the function factory ready, we need to determine of the
     * request is "cacheable" - should be cached. Note that, once again,
     * the user can force this option.
     */


    const cacheable = requestOptions.cacheable || request.cacheable;
    /**
     * If is not "cacheable", we immediately trigger the retryable request, no
     * need to check cache implementations.
     */

    if (cacheable !== true) {
      return createRetryableRequest();
    }
    /**
     * If the request is "cacheable", we need to first compute the key to ask
     * the cache implementations if this request is on progress or if the
     * response already exists on the cache.
     */


    const key = {
      request,
      requestOptions,
      transporter: {
        queryParameters: baseQueryParameters,
        headers: baseHeaders
      }
    };
    /**
     * With the computed key, we first ask the responses cache
     * implementation if this request was been resolved before.
     */

    return responsesCache.get(key, () => {
      /**
       * If the request has never resolved before, we actually ask if there
       * is a current request with the same key on progress.
       */
      return requestsCache.get(key, () =>
      /**
       * Finally, if there is no request in progress with the same key,
       * this `createRetryableRequest()` will actually trigger the
       * retryable request.
       */
      requestsCache.set(key, createRetryableRequest()).then(response => Promise.all([requestsCache.delete(key), response]), err => Promise.all([requestsCache.delete(key), Promise.reject(err)])).then(([_, response]) => response));
    }, {
      /**
       * Of course, once we get this response back from the server, we
       * tell response cache to actually store the received response
       * to be used later.
       */
      miss: response => responsesCache.set(key, response)
    });
  }

  return {
    hostsCache,
    requester,
    timeouts,
    algoliaAgent,
    baseHeaders,
    baseQueryParameters,
    hosts,
    request: createRequest,
    requestsCache,
    responsesCache
  };
}

function createAlgoliaAgent(version) {
  const algoliaAgent = {
    value: `Algolia for JavaScript (${version})`,

    add(options) {
      const addedAlgoliaAgent = `; ${options.segment}${options.version !== undefined ? ` (${options.version})` : ''}`;

      if (algoliaAgent.value.indexOf(addedAlgoliaAgent) === -1) {
        algoliaAgent.value = `${algoliaAgent.value}${addedAlgoliaAgent}`;
      }

      return algoliaAgent;
    }

  };
  return algoliaAgent;
}

function getAlgoliaAgent({
  algoliaAgents,
  client,
  version
}) {
  const defaultAlgoliaAgent = createAlgoliaAgent(version).add({
    segment: client,
    version
  });
  algoliaAgents.forEach(algoliaAgent => defaultAlgoliaAgent.add(algoliaAgent));
  return defaultAlgoliaAgent;
}

const DEFAULT_CONNECT_TIMEOUT_BROWSER = 1000;
const DEFAULT_READ_TIMEOUT_BROWSER = 2000;
const DEFAULT_WRITE_TIMEOUT_BROWSER = 30000;

function createXhrRequester() {
    function send(request) {
        return new Promise((resolve) => {
            const baseRequester = new XMLHttpRequest();
            baseRequester.open(request.method, request.url, true);
            Object.keys(request.headers).forEach((key) => baseRequester.setRequestHeader(key, request.headers[key]));
            const createTimeout = (timeout, content) => {
                return setTimeout(() => {
                    baseRequester.abort();
                    resolve({
                        status: 0,
                        content,
                        isTimedOut: true,
                    });
                }, timeout);
            };
            const connectTimeout = createTimeout(request.connectTimeout, 'Connection timeout');
            let responseTimeout;
            baseRequester.onreadystatechange = () => {
                if (baseRequester.readyState > baseRequester.OPENED &&
                    responseTimeout === undefined) {
                    clearTimeout(connectTimeout);
                    responseTimeout = createTimeout(request.responseTimeout, 'Socket timeout');
                }
            };
            baseRequester.onerror = () => {
                // istanbul ignore next
                if (baseRequester.status === 0) {
                    clearTimeout(connectTimeout);
                    clearTimeout(responseTimeout);
                    resolve({
                        content: baseRequester.responseText || 'Network request failed',
                        status: baseRequester.status,
                        isTimedOut: false,
                    });
                }
            };
            baseRequester.onload = () => {
                clearTimeout(connectTimeout);
                clearTimeout(responseTimeout);
                resolve({
                    content: baseRequester.responseText,
                    status: baseRequester.status,
                    isTimedOut: false,
                });
            };
            baseRequester.send(request.data);
        });
    }
    return { send };
}

// This file is generated, manual changes will be lost - read more on https://github.com/algolia/api-clients-automation.
const apiClientVersion = '5.0.0-alpha.9';
const REGIONS = ['de', 'us'];
function getDefaultHosts(region) {
    const url = !region
        ? 'analytics.algolia.com'
        : 'analytics.{region}.algolia.com'.replace('{region}', region);
    return [{ url, accept: 'readWrite', protocol: 'https' }];
}
// eslint-disable-next-line @typescript-eslint/explicit-function-return-type
function createAbtestingClient({ appId: appIdOption, apiKey: apiKeyOption, authMode, algoliaAgents, region: regionOption, ...options }) {
    const auth = createAuth(appIdOption, apiKeyOption, authMode);
    const transporter = createTransporter({
        hosts: getDefaultHosts(regionOption),
        ...options,
        algoliaAgent: getAlgoliaAgent({
            algoliaAgents,
            client: 'Abtesting',
            version: apiClientVersion,
        }),
        baseHeaders: {
            'content-type': 'text/plain',
            ...auth.headers(),
            ...options.baseHeaders,
        },
        baseQueryParameters: {
            ...auth.queryParameters(),
            ...options.baseQueryParameters,
        },
    });
    return {
        transporter,
        /**
         * The `appId` currently in use.
         */
        appId: appIdOption,
        /**
         * Clears the cache of the transporter for the `requestsCache` and `responsesCache` properties.
         */
        clearCache() {
            return Promise.all([
                transporter.requestsCache.clear(),
                transporter.responsesCache.clear(),
            ]).then(() => undefined);
        },
        /**
         * Get the value of the `algoliaAgent`, used by our libraries internally and telemetry system.
         */
        get _ua() {
            return transporter.algoliaAgent.value;
        },
        /**
         * Adds a `segment` to the `x-algolia-agent` sent with every requests.
         *
         * @param segment - The algolia agent (user-agent) segment to add.
         * @param version - The version of the agent.
         */
        addAlgoliaAgent(segment, version) {
            transporter.algoliaAgent.add({ segment, version });
        },
        /**
         * Creates a new A/B test with provided configuration. You can set an A/B test on two different indices with different settings, or on the same index with different search parameters by providing a customSearchParameters setting on one of the variants.
         *
         * @summary Create a test.
         * @param addABTestsRequest - The addABTestsRequest object.
         * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
         */
        addABTests(addABTestsRequest, requestOptions) {
            if (!addABTestsRequest) {
                throw new Error('Parameter `addABTestsRequest` is required when calling `addABTests`.');
            }
            if (!addABTestsRequest.name) {
                throw new Error('Parameter `addABTestsRequest.name` is required when calling `addABTests`.');
            }
            if (!addABTestsRequest.variant) {
                throw new Error('Parameter `addABTestsRequest.variant` is required when calling `addABTests`.');
            }
            if (!addABTestsRequest.endAt) {
                throw new Error('Parameter `addABTestsRequest.endAt` is required when calling `addABTests`.');
            }
            const requestPath = '/2/abtests';
            const headers = {};
            const queryParameters = {};
            const request = {
                method: 'POST',
                path: requestPath,
                queryParameters,
                headers,
                data: addABTestsRequest,
            };
            return transporter.request(request, requestOptions);
        },
        /**
         * This method allow you to send requests to the Algolia REST API.
         *
         * @summary Send requests to the Algolia REST API.
         * @param del - The del object.
         * @param del.path - The path of the API endpoint to target, anything after the /1 needs to be specified.
         * @param del.parameters - Query parameters to be applied to the current query.
         * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
         */
        del({ path, parameters }, requestOptions) {
            if (!path) {
                throw new Error('Parameter `path` is required when calling `del`.');
            }
            const requestPath = '/1{path}'.replace('{path}', path);
            const headers = {};
            const queryParameters = parameters ? parameters : {};
            const request = {
                method: 'DELETE',
                path: requestPath,
                queryParameters,
                headers,
            };
            return transporter.request(request, requestOptions);
        },
        /**
         * Delete a test.
         *
         * @summary Delete a test.
         * @param deleteABTest - The deleteABTest object.
         * @param deleteABTest.id - The A/B test ID.
         * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
         */
        deleteABTest({ id }, requestOptions) {
            if (!id) {
                throw new Error('Parameter `id` is required when calling `deleteABTest`.');
            }
            const requestPath = '/2/abtests/{id}'.replace('{id}', encodeURIComponent(id));
            const headers = {};
            const queryParameters = {};
            const request = {
                method: 'DELETE',
                path: requestPath,
                queryParameters,
                headers,
            };
            return transporter.request(request, requestOptions);
        },
        /**
         * This method allow you to send requests to the Algolia REST API.
         *
         * @summary Send requests to the Algolia REST API.
         * @param get - The get object.
         * @param get.path - The path of the API endpoint to target, anything after the /1 needs to be specified.
         * @param get.parameters - Query parameters to be applied to the current query.
         * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
         */
        get({ path, parameters }, requestOptions) {
            if (!path) {
                throw new Error('Parameter `path` is required when calling `get`.');
            }
            const requestPath = '/1{path}'.replace('{path}', path);
            const headers = {};
            const queryParameters = parameters ? parameters : {};
            const request = {
                method: 'GET',
                path: requestPath,
                queryParameters,
                headers,
            };
            return transporter.request(request, requestOptions);
        },
        /**
         * Returns metadata and metrics for an A/B test.
         *
         * @summary Get a test.
         * @param getABTest - The getABTest object.
         * @param getABTest.id - The A/B test ID.
         * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
         */
        getABTest({ id }, requestOptions) {
            if (!id) {
                throw new Error('Parameter `id` is required when calling `getABTest`.');
            }
            const requestPath = '/2/abtests/{id}'.replace('{id}', encodeURIComponent(id));
            const headers = {};
            const queryParameters = {};
            const request = {
                method: 'GET',
                path: requestPath,
                queryParameters,
                headers,
            };
            return transporter.request(request, requestOptions);
        },
        /**
         * Fetch all existing A/B tests for App that are available for the current API Key. When no data has been processed, the metrics will be returned as null.
         *
         * @summary List all tests.
         * @param listABTests - The listABTests object.
         * @param listABTests.offset - Position of the starting record. Used for paging. 0 is the first record.
         * @param listABTests.limit - Number of records to return. Limit is the size of the page.
         * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
         */
        listABTests({ offset, limit }, requestOptions) {
            const requestPath = '/2/abtests';
            const headers = {};
            const queryParameters = {};
            if (offset !== undefined) {
                queryParameters.offset = offset.toString();
            }
            if (limit !== undefined) {
                queryParameters.limit = limit.toString();
            }
            const request = {
                method: 'GET',
                path: requestPath,
                queryParameters,
                headers,
            };
            return transporter.request(request, requestOptions);
        },
        /**
         * This method allow you to send requests to the Algolia REST API.
         *
         * @summary Send requests to the Algolia REST API.
         * @param post - The post object.
         * @param post.path - The path of the API endpoint to target, anything after the /1 needs to be specified.
         * @param post.parameters - Query parameters to be applied to the current query.
         * @param post.body - The parameters to send with the custom request.
         * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
         */
        post({ path, parameters, body }, requestOptions) {
            if (!path) {
                throw new Error('Parameter `path` is required when calling `post`.');
            }
            const requestPath = '/1{path}'.replace('{path}', path);
            const headers = {};
            const queryParameters = parameters ? parameters : {};
            const request = {
                method: 'POST',
                path: requestPath,
                queryParameters,
                headers,
                data: body ? body : {},
            };
            return transporter.request(request, requestOptions);
        },
        /**
         * This method allow you to send requests to the Algolia REST API.
         *
         * @summary Send requests to the Algolia REST API.
         * @param put - The put object.
         * @param put.path - The path of the API endpoint to target, anything after the /1 needs to be specified.
         * @param put.parameters - Query parameters to be applied to the current query.
         * @param put.body - The parameters to send with the custom request.
         * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
         */
        put({ path, parameters, body }, requestOptions) {
            if (!path) {
                throw new Error('Parameter `path` is required when calling `put`.');
            }
            const requestPath = '/1{path}'.replace('{path}', path);
            const headers = {};
            const queryParameters = parameters ? parameters : {};
            const request = {
                method: 'PUT',
                path: requestPath,
                queryParameters,
                headers,
                data: body ? body : {},
            };
            return transporter.request(request, requestOptions);
        },
        /**
         * Marks the A/B test as stopped. At this point, the test is over and cannot be restarted. As a result, your application is back to normal: index A will perform as usual, receiving 100% of all search requests. Associated metadata and metrics are still stored.
         *
         * @summary Stop a test.
         * @param stopABTest - The stopABTest object.
         * @param stopABTest.id - The A/B test ID.
         * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
         */
        stopABTest({ id }, requestOptions) {
            if (!id) {
                throw new Error('Parameter `id` is required when calling `stopABTest`.');
            }
            const requestPath = '/2/abtests/{id}/stop'.replace('{id}', encodeURIComponent(id));
            const headers = {};
            const queryParameters = {};
            const request = {
                method: 'POST',
                path: requestPath,
                queryParameters,
                headers,
            };
            return transporter.request(request, requestOptions);
        },
    };
}

// This file is generated, manual changes will be lost - read more on https://github.com/algolia/api-clients-automation.
function abtestingClient(appId, apiKey, region, options) {
    if (!appId || typeof appId !== 'string') {
        throw new Error('`appId` is missing.');
    }
    if (!apiKey || typeof apiKey !== 'string') {
        throw new Error('`apiKey` is missing.');
    }
    if (region && (typeof region !== 'string' || !REGIONS.includes(region))) {
        throw new Error(`\`region\` must be one of the following: ${REGIONS.join(', ')}`);
    }
    return createAbtestingClient({
        appId,
        apiKey,
        region,
        timeouts: {
            connect: DEFAULT_CONNECT_TIMEOUT_BROWSER,
            read: DEFAULT_READ_TIMEOUT_BROWSER,
            write: DEFAULT_WRITE_TIMEOUT_BROWSER,
        },
        requester: createXhrRequester(),
        algoliaAgents: [{ segment: 'Browser' }],
        authMode: 'WithinQueryParameters',
        responsesCache: createMemoryCache(),
        requestsCache: createMemoryCache({ serializable: false }),
        hostsCache: createFallbackableCache({
            caches: [
                createBrowserLocalStorageCache({ key: `${apiClientVersion}-${appId}` }),
                createMemoryCache(),
            ],
        }),
        ...options,
    });
}

export { abtestingClient, apiClientVersion };
