/**
  * alova 2.17.1 (https://alova.js.org)
  * Document https://alova.js.org
  * Copyright 2024 JOU-amjs. All Rights Reserved
  * Licensed under MIT (https://httpshub.com/alovajs/alova/blob/main/LICENSE)
  */

'use strict';

// 以下为减少编译代码量而添加的统一处理函数或变量
const PromiseCls = Promise, promiseReject = (value) => PromiseCls.reject(value), ObjectCls = Object, trueValue = true, falseValue = false, promiseThen = (promise, onFulfilled, onrejected) => promise.then(onFulfilled, onrejected), JSONStringify = (value) => JSON.stringify(value), setTimeoutFn = (fn, delay = 0) => setTimeout(fn, delay), clearTimeoutTimer = (timer) => clearTimeout(timer);

/**
 * 空函数，做兼容处理
 */
const /**
 * 判断参数是否为字符串
 * @param arg 任意参数
 * @returns 该参数是否为字符串
 */
isString = (arg) => typeof arg === 'string', 
/**
 * 全局的toString
 * @param arg 任意参数
 * @returns 字符串化的参数
 */
globalToString = (arg) => ObjectCls.prototype.toString.call(arg), 
/**
 * 判断是否为某个类的实例
 * @param arg 任意参数
 * @returns 判断结果
 */
instanceOf = (arg, cls) => arg instanceof cls, 
/**
 * 是否为特殊数据
 * @param data 提交数据
 * @returns 判断结果
 */
isSpecialRequestBody = (data) => {
    const dataTypeString = globalToString(data);
    return (/^\[object (Blob|FormData|ReadableStream|URLSearchParams)\]$/i.test(dataTypeString) ||
        instanceOf(data, ArrayBuffer));
}, 
/**
 * 创建类实例
 * @param cls 构造函数
 * @param args 构造函数参数
 * @returns 类实例
 */
newInstance = (cls, ...args) => new cls(...args);

/**
 * 构建错误信息
 * @param msg 错误信息
 * @returns 构建后的错误信息
 */
const buildErrorMsg = (msg) => `[alova]${msg}`;
/**
 * 创建一个Alova错误对象
 * @param msg 错误消息
 * @returns 错误对象
 */
var alovaError = (msg, code) => {
    const err = newInstance(Error, buildErrorMsg(msg));
    code && (err.name = code);
    return err;
};

const isBodyData = (data) => isString(data) || isSpecialRequestBody(data);
function GlobalFetch() {
    return function (elements, method) {
        const adapterConfig = method.config, timeout = adapterConfig.timeout || 0, ctrl = new AbortController(), { data, headers } = elements, isContentTypeSet = /content-type/i.test(ObjectCls.keys(headers).join()), isDataFormData = data && data.toString() === '[object FormData]';
        // 未设置Content-Type并且data不是FormData对象时，默认设置Content-Type为application/json
        if (!isContentTypeSet && !isDataFormData) {
            headers['Content-Type'] = 'application/json;charset=UTF-8';
        }
        const fetchPromise = fetch(elements.url, {
            ...adapterConfig,
            method: elements.type,
            signal: ctrl.signal,
            body: isBodyData(data) ? data : JSONStringify(data)
        });
        // 设置了中断时间，则在指定时间后中断请求
        let abortTimer, isTimeout = falseValue;
        if (timeout > 0) {
            abortTimer = setTimeoutFn(() => {
                isTimeout = trueValue;
                ctrl.abort();
            }, timeout);
        }
        return {
            response: () => promiseThen(fetchPromise, response => {
                // 请求成功后清除中断处理
                clearTimeoutTimer(abortTimer);
                // Response的Readable只能被读取一次，需要克隆才可重复使用
                return response.clone();
            }, err => promiseReject(alovaError(isTimeout ? 'fetchError: network timeout' : err.message))),
            // headers函数内的then需捕获异常，否则会导致内部无法获取到正确的错误对象
            headers: () => promiseThen(fetchPromise, ({ headers }) => headers, () => ({})),
            // 因nodeFetch库限制，这块代码无法进行单元测试，但已在浏览器中通过测试
            /* c8 ignore start */
            onDownload: async (cb) => {
                const response = await fetchPromise;
                const { headers, body } = response.clone(), reader = body === null || body === void 0 ? void 0 : body.getReader(), total = Number(headers.get('Content-Length') || headers.get('content-length') || 0);
                if (total <= 0) {
                    return;
                }
                let loaded = 0;
                if (reader) {
                    const progressTimer = setInterval(() => {
                        promiseThen(reader.read(), ({ done, value = new Uint8Array() }) => {
                            if (done) {
                                clearInterval(progressTimer);
                            }
                            else {
                                loaded += value.byteLength;
                                cb(total, loaded);
                            }
                        });
                    }, 0);
                }
            },
            /* c8 ignore start */
            abort: () => {
                ctrl.abort();
                clearTimeoutTimer(abortTimer);
            }
        };
    };
}

module.exports = GlobalFetch;
