/**
  * alova 2.17.1 (https://alova.js.org)
  * Document https://alova.js.org
  * Copyright 2024 JOU-amjs. All Rights Reserved
  * Licensed under MIT (https://httpshub.com/alovajs/alova/blob/main/LICENSE)
  */

import { useState, useRef, useCallback, useEffect } from 'react';

// 以下为减少编译代码量而添加的统一处理函数或变量
const ObjectCls = Object, undefinedValue = undefined, trueValue = true, objectKeys = (obj) => ObjectCls.keys(obj), forEach = (ary, fn) => ary.forEach(fn), mapItem = (ary, callbackfn) => ary.map(callbackfn);

/**
 * 空函数，做兼容处理
 */
const noop = () => { }, 
/**
 * 判断参数是否为数字
 * @param arg 任意参数
 * @returns 该参数是否为数字
 */
isNumber = (arg) => typeof arg === 'number' && !isNaN(arg);

const stateToData = ([state]) => state, refCurrent = (ref) => ref.current, setRef = (ref, newVal) => (ref.current = newVal);
// React的预定义hooks
var ReactHook = {
    create: (initialValue) => useState(initialValue),
    export: stateToData,
    dehydrate: stateToData,
    update: (newVal, states) => forEach(objectKeys(newVal), key => {
        states[key][1](newVal[key]);
    }),
    memorize: (fn) => {
        // 使用useCallback使用同一个引用，同事通过useRef来引用最新函数
        const fnRef = useRef(noop);
        setRef(fnRef, fn);
        return useCallback((...args) => refCurrent(fnRef)(...args), []);
    },
    ref: (initialValue) => {
        const refObj = useRef(initialValue);
        refCurrent(refObj) === undefinedValue && setRef(refObj, initialValue);
        return refObj;
    },
    effectRequest({ handler, removeStates, saveStates, immediate, frontStates, watchingStates = [] }) {
        // 当有监听状态时，状态变化再触发
        const oldStates = mapItem(watchingStates, s => useRef(s)); // 用于对比新旧值
        useEffect(() => {
            // 对比新旧状态，获取变化的状态索引
            let changedIndex = undefinedValue;
            forEach(watchingStates, (newState, i) => {
                if (!Object.is(refCurrent(oldStates[i]), newState)) {
                    changedIndex = i;
                    setRef(oldStates[i], newState);
                }
            });
            if (immediate || isNumber(changedIndex)) {
                handler(changedIndex);
            }
            // 组件卸载时移除对应状态
            return removeStates;
        }, watchingStates);
        // 因为react每次刷新都会重新调用usehook，因此每次会让状态缓存失效
        // 因此每次都需要更新管理的状态
        const needSave = useRef(false);
        useEffect(() => {
            refCurrent(needSave) ? saveStates(frontStates) : setRef(needSave, trueValue);
        });
    }
};

export { ReactHook as default };
