"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;
var _eventify = _interopRequireDefault(require("./lib/eventify"));
var _constructEvent = _interopRequireDefault(require("./lib/constructEvent"));
var _constants = require("./lib/constants");
var _punycode = _interopRequireDefault(require("punycode"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }
function _typeof(obj) { "@babel/helpers - typeof"; return _typeof = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function (obj) { return typeof obj; } : function (obj) { return obj && "function" == typeof Symbol && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; }, _typeof(obj); }
function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }
function _defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, _toPropertyKey(descriptor.key), descriptor); } }
function _createClass(Constructor, protoProps, staticProps) { if (protoProps) _defineProperties(Constructor.prototype, protoProps); if (staticProps) _defineProperties(Constructor, staticProps); Object.defineProperty(Constructor, "prototype", { writable: false }); return Constructor; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return _typeof(key) === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (_typeof(input) !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (_typeof(res) !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }
/**
 * Embedding options.
 * @typedef {Object} EmbeddingOptions
 * @property {string} url - url of the session or dashboard to embed
 * @property {HTMLElement | string} container - parent html element or query selector string
 * @property {Function} errorCallback - callback when error occurs
 * @property {Function} loadCallback - callback when visualization data load complete
 * @property {Function} parametersChangeCallback - callback when parameters change occurs
 * @property {Function} getActiveParametersCallback - callback to get active parameter values
 * @property {Function} getSheetsCallback - callback to get sheet details
 * @property {Function} selectedSheetChangeCallback - callback when current sheet is changed
 * @property {Object} parameters
 * @property {string} width - width of the iframe
 * @property {string} height - height of the iframe
 * @property {string} loadingHeight - when height is set to be "AutoFit",
 *                                   loadingHeight is used before actual height is received
 * @property {string} scrolling
 * @property {string} locale
 */
/**
 * Q SearchBar embedding options. 
 * @typedef {Object} QSearchBarOptions
 * @property {Function} expandCallback - callback when Q search bar is expanded
 * @property {Function} collapseCallback - callback when Q search bar is collapsed
 * @property {boolean} iconDisabled - disable Q icon in search bar (only for single topic set)
 * @property {boolean} topicNameDisabled - disable topic name in search bar (only for single topic set)
 * @property {string} themeId - themeId to apply to search bar (theme must be shared with user)
 * @property {boolean} allowTopicSelection - allow user to change selected topic (only when initialTopicId is set from API)
 */
/**
 * Embeddable Object class.
 * @class
 * @name EmbeddableObject
 * @param {EmbeddingOptions} options - options set by customers to embed the session or dashboard.
 */
var EmbeddableObject = /*#__PURE__*/function () {
  // Q specific members

  /* eslint-disable complexity */
  function EmbeddableObject(options) {
    _classCallCheck(this, EmbeddableObject);
    if (!options) {
      throw new Error('options is required');
    }
    if (!options.url) {
      throw new Error('url is required');
    }
    var url = options.url,
      container = options.container,
      parameters = options.parameters,
      defaultEmbeddingVisualType = options.defaultEmbeddingVisualType,
      errorCallback = options.errorCallback,
      loadCallback = options.loadCallback,
      parametersChangeCallback = options.parametersChangeCallback,
      selectedSheetChangeCallback = options.selectedSheetChangeCallback,
      isQEmbedded = options.isQEmbedded;
    this.url = url;
    if (container instanceof HTMLElement) {
      this.container = container;
    } else if (typeof container === 'string') {
      this.container = document.querySelector(container);
    }
    if (!this.container) {
      throw new Error('can\'t find valid container');
    }
    this.parameters = parameters;
    this.defaultEmbeddingVisualType = defaultEmbeddingVisualType;
    this.iframe = createIframe(options);
    (0, _eventify["default"])(this);
    if (typeof errorCallback === 'function') {
      this.on(_constants.CLIENT_FACING_EVENT_NAMES.error, errorCallback);
    }
    if (typeof loadCallback === 'function') {
      this.on(_constants.CLIENT_FACING_EVENT_NAMES.load, loadCallback);
    }
    if (typeof parametersChangeCallback === 'function') {
      this.on(_constants.CLIENT_FACING_EVENT_NAMES.parametersChange, parametersChangeCallback);
    }
    if (typeof selectedSheetChangeCallback === 'function') {
      this.on(_constants.CLIENT_FACING_EVENT_NAMES.selectedSheetChange, selectedSheetChangeCallback);
    }
    window.addEventListener('message', function (event) {
      if (!event) {
        return;
      }
      if (event.source === (this.iframe && this.iframe.contentWindow)) {
        this.handleMessageEvent(event, options);
      }
    }.bind(this), false);
    if (isQEmbedded) {
      this.qBarOpen = false;
      this.isQEmbedded = isQEmbedded;
      this.qOptions = options.qSearchBarOptions;
      window.addEventListener('click', function (event) {
        var isClickInside = this.container ? this.container.contains(event.target) : true;
        if (!isClickInside) {
          this.closeQPopover();
        }
      }.bind(this), false);
    }
    this.getContainer = this.getContainer.bind(this);
    this.getParameters = this.getParameters.bind(this);
    this.getActiveParameterValues = this.getActiveParameterValues.bind(this);
    this.getSheets = this.getSheets.bind(this);
    this.getDefaultEmbeddingVisualType = this.getDefaultEmbeddingVisualType.bind(this);
    this.getUrl = this.getUrl.bind(this);
    this.handleMessageEvent = this.handleMessageEvent.bind(this);
    this.setParameters = this.setParameters.bind(this);
    this.setDefaultEmbeddingVisualType = this.setDefaultEmbeddingVisualType.bind(this);
    this.setQBarQuestion = this.setQBarQuestion.bind(this);
    this.closeQPopover = this.closeQPopover.bind(this);
  }
  _createClass(EmbeddableObject, [{
    key: "getUrl",
    value: function getUrl() {
      return this.url;
    }
  }, {
    key: "getContainer",
    value: function getContainer() {
      return this.container;
    }
  }, {
    key: "getParameters",
    value: function getParameters() {
      return this.parameters;
    }
  }, {
    key: "getActiveParameterValues",
    value: function getActiveParameterValues(callback) {
      if (typeof callback !== 'function') {
        return;
      }
      if (this.getActiveParametersCallback) {
        this.off(_constants.CLIENT_FACING_EVENT_NAMES.GET_ACTIVE_PARAMETER_VALUES, this.getActiveParametersCallback);
      }
      this.getActiveParametersCallback = callback;
      this.on(_constants.CLIENT_FACING_EVENT_NAMES.GET_ACTIVE_PARAMETER_VALUES, callback);
      var event = (0, _constructEvent["default"])(_constants.OUT_GOING_POST_MESSAGE_EVENT_NAMES.GET_ACTIVE_PARAMETER_VALUES, {});
      this.iframe.contentWindow.postMessage(event, this.url);
    }
  }, {
    key: "getSheets",
    value: function getSheets(callback) {
      if (typeof callback !== 'function') {
        return;
      }
      if (this.getSheetsCallback) {
        this.off(_constants.CLIENT_FACING_EVENT_NAMES.GET_SHEETS, this.getSheetsCallback);
      }
      this.getSheetsCallback = callback;
      this.on(_constants.CLIENT_FACING_EVENT_NAMES.GET_SHEETS, callback);
      var event = (0, _constructEvent["default"])(_constants.OUT_GOING_POST_MESSAGE_EVENT_NAMES.GET_SHEETS, {});
      this.iframe.contentWindow.postMessage(event, this.url);
    }
  }, {
    key: "handleShowQ",
    value: function handleShowQ(payload) {
      if (this.qOptions && this.qOptions.expandCallback && typeof this.qOptions.expandCallback === 'function' && !this.qBarOpen) {
        this.qOptions.expandCallback();
      }
      if (payload && payload.height) {
        this.iframe.height = payload.height;
      }
      this.qBarOpen = true;
    }
  }, {
    key: "handleHideQ",
    value: function handleHideQ(payload) {
      if (this.qOptions && this.qOptions.collapseCallback && typeof this.qOptions.collapseCallback === 'function' && this.qBarOpen) {
        this.qOptions.collapseCallback();
      }
      if (payload && payload.height) {
        this.iframe.height = payload.height;
      }
      this.qBarOpen = false;
    }
  }, {
    key: "handleResizeQ",
    value: function handleResizeQ(payload) {
      if (payload && payload.height) {
        this.iframe.height = payload.height;
      }
    }
  }, {
    key: "handleMessageEvent",
    value: function handleMessageEvent(event, options) {
      var _event$data = event.data,
        eventName = _event$data.eventName,
        payload = _event$data.payload;
      this.trigger(_constants.CLIENT_FACING_EVENT_NAMES[eventName], payload);
      if (eventName === _constants.IN_COMING_POST_MESSAGE_EVENT_NAMES.RESIZE_EVENT) {
        var height = options.height;
        if (height === _constants.DASHBOARD_SIZE_OPTIONS.AUTO_FIT) {
          this.iframe.height = payload.height;
        }
      } else if (eventName === _constants.CLIENT_FACING_EVENT_NAMES.SHOW_Q_BAR) {
        this.handleShowQ(payload);
      } else if (eventName === _constants.CLIENT_FACING_EVENT_NAMES.HIDE_Q_BAR) {
        this.handleHideQ(payload);
      } else if (eventName === _constants.CLIENT_FACING_EVENT_NAMES.RESIZE_Q_BAR) {
        this.handleResizeQ(payload);
      }
    }
  }, {
    key: "getDefaultEmbeddingVisualType",
    value: function getDefaultEmbeddingVisualType() {
      return this.defaultEmbeddingVisualType;
    }
  }, {
    key: "setParameters",
    value: function setParameters(parameters) {
      var event = this.getParameterEvent(parameters);
      this.iframe.contentWindow.postMessage(event, this.url);
    }
  }, {
    key: "getParameterEvent",
    value: function getParameterEvent(parameters) {
      var eventName = _constants.OUT_GOING_POST_MESSAGE_EVENT_NAMES.UPDATE_PARAMETER_VALUES;
      var payload = {};
      var parameterNames = Object.keys(parameters);
      parameterNames.map(function (name) {
        var value = parameters[name];
        var values = [].concat(value);
        var encodedName = encodeURIComponent(name);
        payload[encodedName] = values.map(function (paramValue) {
          return encodeURIComponent(paramValue);
        });
      });
      return (0, _constructEvent["default"])(eventName, {
        parameters: payload
      });
    }
  }, {
    key: "setDefaultEmbeddingVisualType",
    value: function setDefaultEmbeddingVisualType(defaultEmbeddingVisualType) {
      var event = this.generateDefaultEmbeddingVisualTypeEvent(defaultEmbeddingVisualType);
      this.iframe.contentWindow.postMessage(event, this.url);
    }
  }, {
    key: "generateDefaultEmbeddingVisualTypeEvent",
    value: function generateDefaultEmbeddingVisualTypeEvent(defaultEmbeddingVisualType) {
      var eventName = _constants.OUT_GOING_POST_MESSAGE_EVENT_NAMES.DEFAULT_EMBEDDING_VISUAL_TYPE_OPTIONS;
      if (defaultEmbeddingVisualType == null || !(defaultEmbeddingVisualType in _constants.DEFAULT_EMBEDDING_VISUAL_TYPE_OPTIONS)) {
        defaultEmbeddingVisualType = _constants.DEFAULT_EMBEDDING_VISUAL_TYPE_OPTIONS.AUTO_GRAPH;
      }
      var payload = {
        defaultEmbeddingVisualType: defaultEmbeddingVisualType
      };
      return (0, _constructEvent["default"])(eventName, payload);
    }
  }, {
    key: "getQuestionEvent",
    value: function getQuestionEvent(question) {
      var eventName = _constants.OUT_GOING_POST_MESSAGE_EVENT_NAMES.SET_Q_BAR_QUESTION;
      return (0, _constructEvent["default"])(eventName, {
        question: question
      });
    }
  }, {
    key: "setQBarQuestion",
    value: function setQBarQuestion(question) {
      var event = this.getQuestionEvent(question);
      this.iframe.contentWindow.postMessage(event, this.url);
    }
  }, {
    key: "closeQPopover",
    value: function closeQPopover() {
      var _this$iframe$contentW;
      var closeQPopoverEvent = (0, _constructEvent["default"])(_constants.OUT_GOING_POST_MESSAGE_EVENT_NAMES.HIDE_Q_BAR, {});
      (_this$iframe$contentW = this.iframe.contentWindow) === null || _this$iframe$contentW === void 0 ? void 0 : _this$iframe$contentW.postMessage(closeQPopoverEvent, this.url);
    }
  }]);
  return EmbeddableObject;
}();
function createIframe(options) {
  var width = options.width,
    height = options.height,
    isQEmbedded = options.isQEmbedded;
  var loadingHeight = options.loadingHeight,
    url = options.url,
    scrolling = options.scrolling,
    className = options.className;
  if (height === _constants.DASHBOARD_SIZE_OPTIONS.AUTO_FIT) {
    height = loadingHeight;
  }
  var iframe = document.createElement('iframe');
  iframe.className = ['quicksight-embedding-iframe', className].join(' ').trim();
  iframe.width = width || '100%';
  iframe.height = height || '100%';
  iframe.scrolling = scrolling || 'no';
  iframe.onload = sendInitialPostMessage.bind(null, iframe, url);
  iframe.src = getIframeSrc(options);
  iframe.style.border = '0px';
  iframe.style.padding = '0px';
  if (isQEmbedded) {
    iframe.setAttribute('allowtransparency', 'true');
  }
  return iframe;
}
function getIframeSrc(options) {
  var url = options.url,
    parameters = options.parameters,
    locale = options.locale,
    footerPaddingEnabled = options.footerPaddingEnabled,
    iframeResizeOnSheetChange = options.iframeResizeOnSheetChange,
    printEnabled = options.printEnabled,
    resetDisabled = options.resetDisabled,
    sheetId = options.sheetId,
    sheetTabsDisabled = options.sheetTabsDisabled,
    undoRedoDisabled = options.undoRedoDisabled,
    isQEmbedded = options.isQEmbedded,
    qSearchBarOptions = options.qSearchBarOptions;
  var src = url + '&punyCodeEmbedOrigin=' + _punycode["default"].encode(window.location.origin + '/');
  src = src + '&printEnabled=' + String(!!printEnabled);
  if (locale) {
    src = src + '&locale=' + locale;
  }
  if (sheetTabsDisabled) {
    src = src + '&sheetTabsDisabled=' + String(sheetTabsDisabled);
  }
  if (sheetId) {
    src = src + '&sheetId=' + sheetId;
  }
  if (footerPaddingEnabled) {
    src = src + '&footerPaddingEnabled=' + String(footerPaddingEnabled);
  }
  if (undoRedoDisabled) {
    src = src + '&undoRedoDisabled=' + String(undoRedoDisabled);
  }
  if (resetDisabled) {
    src = src + '&resetDisabled=' + String(resetDisabled);
  }
  if (iframeResizeOnSheetChange) {
    src = src + '&resizeOnSheetChange=' + String(iframeResizeOnSheetChange);
  }
  if (parameters) {
    return useParameterValuesInUrl(src, parameters);
  }
  if (isQEmbedded && qSearchBarOptions) {
    if (qSearchBarOptions.iconDisabled !== undefined) {
      src = src + '&qBarIconDisabled=' + String(qSearchBarOptions.iconDisabled);
    }
    if (qSearchBarOptions.topicNameDisabled !== undefined) {
      src = src + '&qBarTopicNameDisabled=' + String(qSearchBarOptions.topicNameDisabled);
    }
    if (qSearchBarOptions.themeId) {
      src = src + '&themeId=' + qSearchBarOptions.themeId;
    }
    if (qSearchBarOptions.allowTopicSelection !== undefined) {
      src = src + '&allowTopicSelection=' + String(qSearchBarOptions.allowTopicSelection);
    }
  }
  return src;
}

/**
 * Use parameter values in url.
 * @function
 * @name useParameterValuesInUrl
 * @param {string} url - url of the session or dashboard to embed.
 * @param {Object} parameters
 */
function useParameterValuesInUrl(url, parameters) {
  var parameterNames = Object.keys(parameters);
  var parameterStrings = parameterNames.map(function (name) {
    var value = parameters[name];
    var values = [].concat(value);
    var encodedName = encodeURIComponent(name);
    return values.map(function (paramValue) {
      return encodeURIComponent(paramValue);
    }).map(function (encodedValue) {
      return "p.".concat(encodedName, "=").concat(encodedValue);
    }).join('&');
  });
  return "".concat(url, "#").concat(parameterStrings.join('&'));
}
function sendInitialPostMessage(iframe, domain) {
  if (iframe.contentWindow === null) {
    setTimeout(sendInitialPostMessage.bind(null, iframe, domain), 100);
    return;
  }
  var eventName = _constants.OUT_GOING_POST_MESSAGE_EVENT_NAMES.ESTABLISH_MESSAGE_CHANNEL;
  var event = (0, _constructEvent["default"])(eventName);
  // wait until iframe.contentWindow exists and send message to iframe window
  iframe.contentWindow.postMessage(event, domain);
}
var _default = EmbeddableObject;
exports["default"] = _default;