"use strict";
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@angular-devkit/core");
const node_1 = require("@angular-devkit/core/node");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
class TestProjectHost extends node_1.NodeJsSyncHost {
    constructor(_templateRoot) {
        super();
        this._templateRoot = _templateRoot;
        this._currentRoot = null;
        this._scopedSyncHost = null;
    }
    root() {
        if (this._currentRoot === null) {
            throw new Error('TestProjectHost must be initialized before being used.');
        }
        return this._currentRoot;
    }
    scopedSync() {
        if (this._currentRoot === null || this._scopedSyncHost === null) {
            throw new Error('TestProjectHost must be initialized before being used.');
        }
        return this._scopedSyncHost;
    }
    initialize() {
        const recursiveList = (path) => this.list(path).pipe(
        // Emit each fragment individually.
        operators_1.concatMap(fragments => rxjs_1.from(fragments)), 
        // Join the path with fragment.
        operators_1.map(fragment => core_1.join(path, fragment)), 
        // Emit directory content paths instead of the directory path.
        operators_1.mergeMap(path => this.isDirectory(path).pipe(operators_1.concatMap(isDir => isDir ? recursiveList(path) : rxjs_1.of(path)))));
        // Find a unique folder that we can write to to use as current root.
        return this.findUniqueFolderPath().pipe(
        // Save the path and create a scoped host for it.
        operators_1.tap(newFolderPath => {
            this._currentRoot = newFolderPath;
            this._scopedSyncHost = new core_1.virtualFs.SyncDelegateHost(new core_1.virtualFs.ScopedHost(this, this.root()));
        }), 
        // List all files in root.
        operators_1.concatMap(() => recursiveList(this._templateRoot)), 
        // Copy them over to the current root.
        operators_1.concatMap(from => {
            const to = core_1.join(this.root(), core_1.relative(this._templateRoot, from));
            return this.read(from).pipe(operators_1.concatMap(buffer => this.write(to, buffer)));
        }), operators_1.map(() => { }));
    }
    restore() {
        if (this._currentRoot === null) {
            return rxjs_1.EMPTY;
        }
        // Delete the current root and clear the variables.
        // Wait 50ms and retry up to 10 times, to give time for file locks to clear.
        return this.exists(this.root()).pipe(operators_1.delay(50), operators_1.concatMap(exists => exists ? this.delete(this.root()) : rxjs_1.EMPTY), operators_1.retry(10), operators_1.finalize(() => {
            this._currentRoot = null;
            this._scopedSyncHost = null;
        }));
    }
    writeMultipleFiles(files) {
        Object.keys(files).forEach(fileName => {
            let content = files[fileName];
            if (typeof content == 'string') {
                content = core_1.virtualFs.stringToFileBuffer(content);
            }
            else if (content instanceof Buffer) {
                content = content.buffer.slice(content.byteOffset, content.byteOffset + content.byteLength);
            }
            this.scopedSync().write(core_1.normalize(fileName), content);
        });
    }
    replaceInFile(path, match, replacement) {
        const content = core_1.virtualFs.fileBufferToString(this.scopedSync().read(core_1.normalize(path)));
        this.scopedSync().write(core_1.normalize(path), core_1.virtualFs.stringToFileBuffer(content.replace(match, replacement)));
    }
    appendToFile(path, str) {
        const content = core_1.virtualFs.fileBufferToString(this.scopedSync().read(core_1.normalize(path)));
        this.scopedSync().write(core_1.normalize(path), core_1.virtualFs.stringToFileBuffer(content.concat(str)));
    }
    fileMatchExists(dir, regex) {
        const [fileName] = this.scopedSync().list(core_1.normalize(dir)).filter(name => name.match(regex));
        return fileName || undefined;
    }
    copyFile(from, to) {
        const content = this.scopedSync().read(core_1.normalize(from));
        this.scopedSync().write(core_1.normalize(to), content);
    }
    findUniqueFolderPath() {
        // 11 character alphanumeric string.
        const randomString = Math.random().toString(36).slice(2);
        const newFolderName = `test-project-host-${core_1.basename(this._templateRoot)}-${randomString}`;
        const newFolderPath = core_1.join(core_1.dirname(this._templateRoot), newFolderName);
        return this.exists(newFolderPath).pipe(operators_1.concatMap(exists => exists ? this.findUniqueFolderPath() : rxjs_1.of(newFolderPath)));
    }
}
exports.TestProjectHost = TestProjectHost;
//# sourceMappingURL=data:application/json;base64,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