"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const fs_1 = require("fs");
const path_1 = require("path");
const ts = require("typescript");
function transformJavascript(options) {
    const { content, getTransforms, emitSourceMap, inputFilePath, outputFilePath, strict, } = options;
    // Bail if there's no transform to do.
    if (getTransforms.length === 0) {
        return {
            content: null,
            sourceMap: null,
            emitSkipped: true,
        };
    }
    // Print error diagnostics.
    const checkDiagnostics = (diagnostics) => {
        if (diagnostics && diagnostics.length > 0) {
            let errors = '';
            errors = errors + '\n' + ts.formatDiagnostics(diagnostics, {
                getCurrentDirectory: () => ts.sys.getCurrentDirectory(),
                getNewLine: () => ts.sys.newLine,
                getCanonicalFileName: (f) => f,
            });
            return errors;
        }
    };
    // Make a in-memory host and populate it with a single file
    const fileMap = new Map();
    const sourcesMap = new Map();
    const outputs = new Map();
    // We're not actually writing anything to disk, but still need to define an outDir
    // because otherwise TS will fail to emit JS since it would overwrite the original.
    const tempOutDir = '$$_temp/';
    const tempFilename = 'bo-default-file.js';
    fileMap.set(tempFilename, content);
    // We need to load the default lib for noEmitOnError to work properly.
    const defaultLibFileName = 'lib.d.ts';
    const defaultLibContent = fs_1.readFileSync(path_1.join(path_1.dirname(require.resolve('typescript')), defaultLibFileName), 'UTF-8');
    fileMap.set(defaultLibFileName, defaultLibContent);
    fileMap.forEach((v, k) => sourcesMap.set(k, ts.createSourceFile(k, v, ts.ScriptTarget.ES2015)));
    const host = {
        getSourceFile: (fileName) => {
            const sourceFile = sourcesMap.get(fileName);
            if (!sourceFile) {
                throw new Error(`File ${fileName} does not have a sourceFile.`);
            }
            return sourceFile;
        },
        getDefaultLibFileName: () => defaultLibFileName,
        getCurrentDirectory: () => '',
        getDirectories: () => [],
        getCanonicalFileName: (fileName) => fileName,
        useCaseSensitiveFileNames: () => true,
        getNewLine: () => '\n',
        fileExists: (fileName) => fileMap.has(fileName),
        readFile: (fileName) => {
            const content = fileMap.get(fileName);
            if (!content) {
                throw new Error(`File ${fileName} does not exist.`);
            }
            return content;
        },
        writeFile: (fileName, text) => outputs.set(fileName, text),
    };
    const tsOptions = {
        noEmitOnError: true,
        allowJs: true,
        // Using just line feed makes test comparisons easier, and doesn't matter for generated files.
        newLine: ts.NewLineKind.LineFeed,
        // We target next so that there is no downleveling.
        target: ts.ScriptTarget.ESNext,
        skipLibCheck: true,
        outDir: '$$_temp/',
        sourceMap: emitSourceMap,
        inlineSources: emitSourceMap,
        inlineSourceMap: false,
    };
    const program = ts.createProgram(Array.from(fileMap.keys()), tsOptions, host);
    // We need the checker inside transforms.
    const transforms = getTransforms.map((getTf) => getTf(program));
    const { emitSkipped, diagnostics } = program.emit(undefined, host.writeFile, undefined, undefined, { before: transforms, after: [] });
    let transformedContent = outputs.get(`${tempOutDir}${tempFilename}`);
    if (emitSkipped || !transformedContent) {
        // Throw only if we're in strict mode, otherwise return original content.
        if (strict) {
            throw new Error(`
        TS failed with the following error messages:

        ${checkDiagnostics(diagnostics)}
      `);
        }
        else {
            return {
                content: null,
                sourceMap: null,
                emitSkipped: true,
            };
        }
    }
    let sourceMap = null;
    if (emitSourceMap) {
        const tsSourceMap = outputs.get(`${tempOutDir}${tempFilename}.map`);
        const urlRegExp = /^\/\/# sourceMappingURL=[^\r\n]*/gm;
        sourceMap = JSON.parse(tsSourceMap);
        // Fix sourcemaps file references.
        if (outputFilePath) {
            sourceMap.file = outputFilePath;
            transformedContent = transformedContent.replace(urlRegExp, `//# sourceMappingURL=${sourceMap.file}.map\n`);
            if (inputFilePath) {
                sourceMap.sources = [inputFilePath];
            }
            else {
                sourceMap.sources = [''];
            }
        }
        else {
            // TODO: figure out if we should inline sources here.
            transformedContent = transformedContent.replace(urlRegExp, '');
            sourceMap.file = '';
            sourceMap.sources = [''];
        }
    }
    return {
        content: transformedContent,
        sourceMap,
        emitSkipped: false,
    };
}
exports.transformJavascript = transformJavascript;
//# sourceMappingURL=data:application/json;base64,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