"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const ts = require("typescript");
function getFoldFileTransformer(program) {
    const checker = program.getTypeChecker();
    return (context) => {
        const transformer = (sf) => {
            const classes = findClassDeclarations(sf);
            const statements = findClassStaticPropertyAssignments(sf, checker, classes);
            const visitor = (node) => {
                // Check if node is a statement to be dropped.
                if (statements.find((st) => st.expressionStatement === node)) {
                    // According to @mhegazy returning undefined is supported.
                    // https://github.com/Microsoft/TypeScript/pull/17044
                    // tslint:disable-next-line:no-any
                    return undefined;
                }
                // Check if node is a class to add statements to.
                const clazz = classes.find((cl) => cl.classFunction === node);
                if (clazz) {
                    const functionExpression = node;
                    const newExpressions = clazz.statements.map((st) => ts.createStatement(st.expressionStatement.expression));
                    // Create a new body with all the original statements, plus new ones,
                    // plus return statement.
                    const newBody = ts.createBlock([
                        ...functionExpression.body.statements.slice(0, -1),
                        ...newExpressions,
                        ...functionExpression.body.statements.slice(-1),
                    ]);
                    const newNode = ts.createFunctionExpression(functionExpression.modifiers, functionExpression.asteriskToken, functionExpression.name, functionExpression.typeParameters, functionExpression.parameters, functionExpression.type, newBody);
                    // Replace node with modified one.
                    return ts.visitEachChild(newNode, visitor, context);
                }
                // Otherwise return node as is.
                return ts.visitEachChild(node, visitor, context);
            };
            return ts.visitNode(sf, visitor);
        };
        return transformer;
    };
}
exports.getFoldFileTransformer = getFoldFileTransformer;
function findClassDeclarations(node) {
    const classes = [];
    // Find all class declarations, build a ClassData for each.
    ts.forEachChild(node, (child) => {
        if (child.kind !== ts.SyntaxKind.VariableStatement) {
            return;
        }
        const varStmt = child;
        if (varStmt.declarationList.declarations.length > 1) {
            return;
        }
        const varDecl = varStmt.declarationList.declarations[0];
        if (varDecl.name.kind !== ts.SyntaxKind.Identifier) {
            return;
        }
        const name = varDecl.name.text;
        const expr = varDecl.initializer;
        if (!expr || expr.kind !== ts.SyntaxKind.ParenthesizedExpression) {
            return;
        }
        if (expr.expression.kind !== ts.SyntaxKind.CallExpression) {
            return;
        }
        const callExpr = expr.expression;
        if (callExpr.expression.kind !== ts.SyntaxKind.FunctionExpression) {
            return;
        }
        const fn = callExpr.expression;
        if (fn.body.statements.length < 2) {
            return;
        }
        if (fn.body.statements[0].kind !== ts.SyntaxKind.FunctionDeclaration) {
            return;
        }
        const innerFn = fn.body.statements[0];
        if (fn.body.statements[fn.body.statements.length - 1].kind !== ts.SyntaxKind.ReturnStatement) {
            return;
        }
        if (!innerFn.name || innerFn.name.kind !== ts.SyntaxKind.Identifier) {
            return;
        }
        if (innerFn.name.text !== name) {
            return;
        }
        classes.push({
            name,
            class: varDecl,
            classFunction: fn,
            statements: [],
        });
    });
    return classes;
}
function findClassStaticPropertyAssignments(node, checker, classes) {
    const statements = [];
    // Find each assignment outside of the declaration.
    ts.forEachChild(node, (child) => {
        if (child.kind !== ts.SyntaxKind.ExpressionStatement) {
            return;
        }
        const expressionStatement = child;
        if (expressionStatement.expression.kind !== ts.SyntaxKind.BinaryExpression) {
            return;
        }
        const binEx = expressionStatement.expression;
        if (binEx.left.kind !== ts.SyntaxKind.PropertyAccessExpression) {
            return;
        }
        const propAccess = binEx.left;
        if (propAccess.expression.kind !== ts.SyntaxKind.Identifier) {
            return;
        }
        const symbol = checker.getSymbolAtLocation(propAccess.expression);
        if (!symbol) {
            return;
        }
        const decls = symbol.declarations;
        if (decls === undefined || decls.length !== 1) {
            return;
        }
        const classIdx = classes
            .map((clazz) => clazz.class).indexOf(decls[0]);
        if (classIdx === -1) {
            return;
        }
        const hostClass = classes[classIdx];
        const statement = { expressionStatement, hostClass };
        hostClass.statements.push(statement);
        statements.push(statement);
    });
    return statements;
}
//# sourceMappingURL=data:application/json;base64,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