"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const ts = require("typescript");
function testPrefixClasses(content) {
    const regexes = [
        // tslint:disable-next-line:max-line-length
        /^(var (\S+) = )(\(function \(\) \{\r?\n(?:    (?:\/\*\*| \*|\*\/|\/\/)[^\r?\n]*\r?\n)*    function \2\([^\)]*\) \{\r?\n)/,
        /^(var (\S+) = )(\(function \(_super\) \{\r?\n    \w*__extends\(\w+, _super\);\r?\n)/,
    ];
    return regexes.some((regex) => regex.test(content));
}
exports.testPrefixClasses = testPrefixClasses;
function getPrefixClassesTransformer() {
    return (context) => {
        const transformer = (sf) => {
            const pureFunctionComment = '@__PURE__';
            const visitor = (node) => {
                // Add pure comment to downleveled classes.
                if (isDownleveledClass(node)) {
                    const varDecl = node;
                    const varDeclInit = varDecl.initializer;
                    // Create a new node with the pure comment before the variable declaration initializer.
                    const newNode = ts.createVariableDeclaration(varDecl.name, undefined, ts.addSyntheticLeadingComment(varDeclInit, ts.SyntaxKind.MultiLineCommentTrivia, pureFunctionComment, false));
                    // Replace node with modified one.
                    return ts.visitEachChild(newNode, visitor, context);
                }
                // Otherwise return node as is.
                return ts.visitEachChild(node, visitor, context);
            };
            return ts.visitNode(sf, visitor);
        };
        return transformer;
    };
}
exports.getPrefixClassesTransformer = getPrefixClassesTransformer;
// Determine if a node matched the structure of a downleveled TS class.
function isDownleveledClass(node) {
    let isExtendedClass = false;
    if (node.kind !== ts.SyntaxKind.VariableDeclaration) {
        return false;
    }
    const varDecl = node;
    if (varDecl.name.kind !== ts.SyntaxKind.Identifier) {
        return false;
    }
    // The variable name should be the class name.
    const className = varDecl.name.text;
    if (!varDecl.initializer || varDecl.initializer.kind !== ts.SyntaxKind.ParenthesizedExpression) {
        return false;
    }
    const parenExpr = varDecl.initializer;
    if (parenExpr.expression.kind !== ts.SyntaxKind.CallExpression) {
        return false;
    }
    const callExpr = parenExpr.expression;
    if (callExpr.expression.kind !== ts.SyntaxKind.FunctionExpression) {
        return false;
    }
    const funcExpr = callExpr.expression;
    // Extended classes have the `_super` parameter.
    if (funcExpr.parameters.length === 1
        && funcExpr.parameters[0].name.text === '_super') {
        isExtendedClass = true;
    }
    // IIFE inner parameters should be empty or `_super`.
    if (funcExpr.parameters.length !== 0 && !isExtendedClass) {
        return false;
    }
    const stmts = funcExpr.body.statements;
    if (stmts.length === 0) {
        return false;
    }
    const firstStatement = stmts[0];
    // Check if `node` is a FunctionDeclaration named `name`.
    function isFunDeclNamed(node, name) {
        if (node.kind === ts.SyntaxKind.FunctionDeclaration) {
            const funcDecl = node;
            if (funcDecl.name && funcDecl.name.text === name) {
                return true;
            }
        }
        else {
            return false;
        }
    }
    // If the class is extending another, the first statement is a _extends(..., _super) call.
    if (isExtendedClass) {
        if (firstStatement.kind !== ts.SyntaxKind.ExpressionStatement) {
            return false;
        }
        const exprStmt = firstStatement;
        if (exprStmt.expression.kind !== ts.SyntaxKind.CallExpression) {
            return false;
        }
        const extendsCallExpr = exprStmt.expression;
        // Function should be called `__extends`.
        if (extendsCallExpr.expression.kind !== ts.SyntaxKind.Identifier) {
            return false;
        }
        const callExprName = extendsCallExpr.expression.text;
        // Reserved TS names are retrieved with three underscores instead of two.
        if (callExprName !== '___extends') {
            return false;
        }
        // Function should have 1+ arguments, with the last being named `_super`.
        if (extendsCallExpr.arguments.length === 0) {
            return false;
        }
        const lastArg = extendsCallExpr.arguments[extendsCallExpr.arguments.length - 1];
        if (lastArg.kind !== ts.SyntaxKind.Identifier) {
            return false;
        }
        const lastArgName = lastArg.text;
        if (lastArgName !== '_super') {
            return false;
        }
        const secondStatement = stmts[1];
        if (secondStatement && isFunDeclNamed(secondStatement, className)) {
            // This seems to be downleveled class that extends another class.
            return true;
        }
    }
    else if (isFunDeclNamed(firstStatement, className)) {
        // This seems to be downleveled class.
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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