"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const ts = require("typescript");
const ast_utils_1 = require("../helpers/ast-utils");
function testWrapEnums(content) {
    const regexes = [
        // tslint:disable:max-line-length
        /var (\S+) = \{\};\r?\n(\1\.(\S+) = \d+;\r?\n)+\1\[\1\.(\S+)\] = "\4";\r?\n(\1\[\1\.(\S+)\] = "\S+";\r?\n*)+/,
        /var (\S+);(\/\*@__PURE__\*\/)*\r?\n\(function \(\1\) \{\s+(\1\[\1\["(\S+)"\] = 0\] = "\4";(\s+\1\[\1\["\S+"\] = \d\] = "\S+";)*\r?\n)\}\)\(\1 \|\| \(\1 = \{\}\)\);/,
    ];
    return regexes.some((regex) => regex.test(content));
}
exports.testWrapEnums = testWrapEnums;
function getWrapEnumsTransformer() {
    return (context) => {
        const transformer = (sf) => {
            const enums = findEnumDeclarations(sf);
            const dropNodes = enums.reduce((acc, curr) => acc.concat(curr.dropNodes), []);
            const visitor = (node) => {
                const enumData = enums.find((e) => e.hostNode === node);
                if (enumData) {
                    // Replace node with a wrapped enum.
                    return ts.visitEachChild(createWrappedEnum(enumData), visitor, context);
                }
                // Drop enum nodes we relocated.
                if (dropNodes.find((n) => n === node)) {
                    // According to @mhegazy returning undefined is supported.
                    // https://github.com/Microsoft/TypeScript/pull/17044
                    // tslint:disable-next-line:no-any
                    return undefined;
                }
                // Otherwise return node as is.
                return ts.visitEachChild(node, visitor, context);
            };
            return ts.visitNode(sf, visitor);
        };
        return transformer;
    };
}
exports.getWrapEnumsTransformer = getWrapEnumsTransformer;
// Find all enum declarations, build a EnumData for each.
function findEnumDeclarations(sourceFile) {
    const enums = [];
    const enumHoldingNodes = [
        sourceFile,
        ...ast_utils_1.collectDeepNodes(sourceFile, ts.SyntaxKind.Block),
    ];
    enumHoldingNodes.forEach((node) => {
        const stmts = node.statements;
        stmts.forEach((stmt, idx) => {
            // We're looking for a variable statement with more statements after it.
            if (idx >= stmts.length - 1
                || stmt.kind !== ts.SyntaxKind.VariableStatement) {
                return;
            }
            const varStmt = stmt;
            if (varStmt.declarationList.declarations.length !== 1) {
                return;
            }
            // We've found a single variable declaration statement, it might be the start of an enum.
            const maybeHostNode = varStmt;
            const varDecl = maybeHostNode.declarationList.declarations[0];
            if (varDecl.name.kind !== ts.SyntaxKind.Identifier) {
                return;
            }
            const maybeName = varDecl.name.text;
            const enumStatements = [], enumDropNodes = [];
            // Try to figure out the enum type from the variable declaration.
            if (!varDecl.initializer) {
                // Typescript 2.3 enums have no initializer.
                const nextStatement = stmts[idx + 1];
                enumStatements.push(...findTs2_3EnumStatements(maybeName, nextStatement));
                enumDropNodes.push(nextStatement);
            }
            else if (varDecl.initializer
                && varDecl.initializer.kind === ts.SyntaxKind.ObjectLiteralExpression
                && varDecl.initializer.properties.length === 0) {
                // Typescript 2.2 enums have a {} initializer.
                const nextStatements = stmts.slice(idx + 1);
                const statements = findTs2_2EnumStatements(maybeName, nextStatements);
                // We have to create new statements so we can keep new ones and drop old ones.
                enumStatements.push(...statements.map(stmt => ts.createStatement(stmt.expression)));
                enumDropNodes.push(...statements);
            }
            else {
                return;
            }
            if (enumStatements.length === 0) {
                return;
            }
            enums.push({
                name: maybeName,
                hostNode: maybeHostNode,
                statements: enumStatements,
                dropNodes: enumDropNodes,
            });
        });
    });
    return enums;
}
// TS 2.3 enums have statements are inside a IIFE.
function findTs2_3EnumStatements(name, statement) {
    const enumStatements = [];
    const noNodes = [];
    const funcExpr = ast_utils_1.drilldownNodes(statement, [
        { prop: null, kind: ts.SyntaxKind.ExpressionStatement },
        { prop: 'expression', kind: ts.SyntaxKind.CallExpression },
        { prop: 'expression', kind: ts.SyntaxKind.ParenthesizedExpression },
        { prop: 'expression', kind: ts.SyntaxKind.FunctionExpression },
    ]);
    if (funcExpr === null) {
        return noNodes;
    }
    if (!(funcExpr.parameters.length === 1
        && funcExpr.parameters[0].name.kind === ts.SyntaxKind.Identifier
        && funcExpr.parameters[0].name.text === name)) {
        return noNodes;
    }
    // In TS 2.3 enums, the IIFE contains only expressions with a certain format.
    // If we find any that is different, we ignore the whole thing.
    for (const innerStmt of funcExpr.body.statements) {
        const innerBinExpr = ast_utils_1.drilldownNodes(innerStmt, [
            { prop: null, kind: ts.SyntaxKind.ExpressionStatement },
            { prop: 'expression', kind: ts.SyntaxKind.BinaryExpression },
        ]);
        if (innerBinExpr === null) {
            return noNodes;
        }
        const exprStmt = innerStmt;
        if (!(innerBinExpr.operatorToken.kind === ts.SyntaxKind.FirstAssignment
            && innerBinExpr.left.kind === ts.SyntaxKind.ElementAccessExpression)) {
            return noNodes;
        }
        const innerElemAcc = innerBinExpr.left;
        if (!(innerElemAcc.expression.kind === ts.SyntaxKind.Identifier
            && innerElemAcc.expression.text === name
            && innerElemAcc.argumentExpression
            && innerElemAcc.argumentExpression.kind === ts.SyntaxKind.BinaryExpression)) {
            return noNodes;
        }
        const innerArgBinExpr = innerElemAcc.argumentExpression;
        if (innerArgBinExpr.left.kind !== ts.SyntaxKind.ElementAccessExpression) {
            return noNodes;
        }
        const innerArgElemAcc = innerArgBinExpr.left;
        if (!(innerArgElemAcc.expression.kind === ts.SyntaxKind.Identifier
            && innerArgElemAcc.expression.text === name)) {
            return noNodes;
        }
        enumStatements.push(exprStmt);
    }
    return enumStatements;
}
// TS 2.2 enums have statements after the variable declaration, with index statements followed
// by value statements.
function findTs2_2EnumStatements(name, statements) {
    const enumStatements = [];
    let beforeValueStatements = true;
    for (const stmt of statements) {
        // Ensure all statements are of the expected format and using the right identifer.
        // When we find a statement that isn't part of the enum, return what we collected so far.
        const binExpr = ast_utils_1.drilldownNodes(stmt, [
            { prop: null, kind: ts.SyntaxKind.ExpressionStatement },
            { prop: 'expression', kind: ts.SyntaxKind.BinaryExpression },
        ]);
        if (binExpr === null
            || (binExpr.left.kind !== ts.SyntaxKind.PropertyAccessExpression
                && binExpr.left.kind !== ts.SyntaxKind.ElementAccessExpression)) {
            return beforeValueStatements ? [] : enumStatements;
        }
        const exprStmt = stmt;
        const leftExpr = binExpr.left;
        if (!(leftExpr.expression.kind === ts.SyntaxKind.Identifier
            && leftExpr.expression.text === name)) {
            return beforeValueStatements ? [] : enumStatements;
        }
        if (!beforeValueStatements && leftExpr.kind === ts.SyntaxKind.PropertyAccessExpression) {
            // We shouldn't find index statements after value statements.
            return [];
        }
        else if (beforeValueStatements && leftExpr.kind === ts.SyntaxKind.ElementAccessExpression) {
            beforeValueStatements = false;
        }
        enumStatements.push(exprStmt);
    }
    return enumStatements;
}
function createWrappedEnum(enumData) {
    const pureFunctionComment = '@__PURE__';
    const { name, statements } = enumData;
    const innerVarStmt = ts.createVariableStatement(undefined, ts.createVariableDeclarationList([
        ts.createVariableDeclaration(name, undefined, ts.createObjectLiteral()),
    ]));
    const innerReturn = ts.createReturn(ts.createIdentifier(name));
    const iife = ts.createCall(ts.createParen(ts.createFunctionExpression(undefined, undefined, undefined, undefined, [], undefined, ts.createBlock([
        innerVarStmt,
        ...statements,
        innerReturn,
    ]))), undefined, []);
    // Create a new node with the pure comment before the variable declaration initializer.
    const outerVarStmt = ts.createVariableStatement(undefined, ts.createVariableDeclarationList([
        ts.createVariableDeclaration(name, undefined, ts.addSyntheticLeadingComment(iife, ts.SyntaxKind.MultiLineCommentTrivia, pureFunctionComment, false)),
    ]));
    return outerVarStmt;
}
//# sourceMappingURL=data:application/json;base64,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