"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ts = require("typescript");
function validateDiagnostics(diagnostics, strict) {
    // Print error diagnostics.
    const checkDiagnostics = (diagnostics) => {
        if (diagnostics && diagnostics.length > 0) {
            let errors = '';
            errors = errors + '\n' + ts.formatDiagnostics(diagnostics, {
                getCurrentDirectory: () => ts.sys.getCurrentDirectory(),
                getNewLine: () => ts.sys.newLine,
                getCanonicalFileName: (f) => f,
            });
            return errors;
        }
    };
    const hasError = diagnostics.some(diag => diag.category === ts.DiagnosticCategory.Error);
    if (hasError) {
        // Throw only if we're in strict mode, otherwise return original content.
        if (strict) {
            throw new Error(`
        TS failed with the following error messages:

        ${checkDiagnostics(diagnostics)}
      `);
        }
        else {
            return false;
        }
    }
    return true;
}
function transformJavascript(options) {
    const { content, getTransforms, emitSourceMap, inputFilePath, outputFilePath, strict, } = options;
    // Bail if there's no transform to do.
    if (getTransforms.length === 0) {
        return {
            content: null,
            sourceMap: null,
            emitSkipped: true,
        };
    }
    const allowFastPath = options.typeCheck === false && !emitSourceMap;
    const outputs = new Map();
    const tempFilename = 'bo-default-file.js';
    const tempSourceFile = ts.createSourceFile(tempFilename, content, ts.ScriptTarget.Latest, allowFastPath);
    const parseDiagnostics = tempSourceFile.parseDiagnostics;
    const tsOptions = {
        // We target latest so that there is no downleveling.
        target: ts.ScriptTarget.Latest,
        isolatedModules: true,
        suppressOutputPathCheck: true,
        allowNonTsExtensions: true,
        noLib: true,
        noResolve: true,
        sourceMap: emitSourceMap,
        inlineSources: emitSourceMap,
        inlineSourceMap: false,
    };
    if (allowFastPath && parseDiagnostics) {
        if (!validateDiagnostics(parseDiagnostics, strict)) {
            return {
                content: null,
                sourceMap: null,
                emitSkipped: true,
            };
        }
        const transforms = getTransforms.map((getTf) => getTf(undefined));
        const result = ts.transform(tempSourceFile, transforms, tsOptions);
        if (result.transformed.length === 0 || result.transformed[0] === tempSourceFile) {
            return {
                content: null,
                sourceMap: null,
                emitSkipped: true,
            };
        }
        const printer = ts.createPrinter(undefined, {
            onEmitNode: result.emitNodeWithNotification,
            substituteNode: result.substituteNode,
        });
        const output = printer.printFile(result.transformed[0]);
        result.dispose();
        return {
            content: output,
            sourceMap: null,
            emitSkipped: false,
        };
    }
    const host = {
        getSourceFile: (fileName) => {
            if (fileName !== tempFilename) {
                throw new Error(`File ${fileName} does not have a sourceFile.`);
            }
            return tempSourceFile;
        },
        getDefaultLibFileName: () => 'lib.d.ts',
        getCurrentDirectory: () => '',
        getDirectories: () => [],
        getCanonicalFileName: (fileName) => fileName,
        useCaseSensitiveFileNames: () => true,
        getNewLine: () => '\n',
        fileExists: (fileName) => fileName === tempFilename,
        readFile: (_fileName) => '',
        writeFile: (fileName, text) => outputs.set(fileName, text),
    };
    const program = ts.createProgram([tempFilename], tsOptions, host);
    const diagnostics = program.getSyntacticDiagnostics(tempSourceFile);
    if (!validateDiagnostics(diagnostics, strict)) {
        return {
            content: null,
            sourceMap: null,
            emitSkipped: true,
        };
    }
    // We need the checker inside transforms.
    const transforms = getTransforms.map((getTf) => getTf(program));
    program.emit(undefined, undefined, undefined, undefined, { before: transforms, after: [] });
    let transformedContent = outputs.get(tempFilename);
    if (!transformedContent) {
        return {
            content: null,
            sourceMap: null,
            emitSkipped: true,
        };
    }
    let sourceMap = null;
    const tsSourceMap = outputs.get(`${tempFilename}.map`);
    if (emitSourceMap && tsSourceMap) {
        const urlRegExp = /^\/\/# sourceMappingURL=[^\r\n]*/gm;
        sourceMap = JSON.parse(tsSourceMap);
        // Fix sourcemaps file references.
        if (outputFilePath) {
            sourceMap.file = outputFilePath;
            transformedContent = transformedContent.replace(urlRegExp, `//# sourceMappingURL=${sourceMap.file}.map\n`);
            if (inputFilePath) {
                sourceMap.sources = [inputFilePath];
            }
            else {
                sourceMap.sources = [''];
            }
        }
        else {
            // TODO: figure out if we should inline sources here.
            transformedContent = transformedContent.replace(urlRegExp, '');
            sourceMap.file = '';
            sourceMap.sources = [''];
        }
    }
    return {
        content: transformedContent,
        sourceMap,
        emitSkipped: false,
    };
}
exports.transformJavascript = transformJavascript;
//# sourceMappingURL=data:application/json;base64,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