"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const webpack_sources_1 = require("webpack-sources");
// This matches a comment left by the build-optimizer that contains pure import paths
const importCommentRegex = /\/\*\* PURE_IMPORTS_START (\S+) PURE_IMPORTS_END \*\//mg;
function purifyReplacements(content) {
    const pureImportMatches = getMatches(content, importCommentRegex, 1)
        .map(match => match.replace(/^\.+/, ''))
        .join('|');
    if (!pureImportMatches) {
        return [];
    }
    const inserts = [];
    /* Prefix safe imports with pure */
    const regex = new RegExp(`(_(${pureImportMatches})__(_default)? = )(__webpack_require__(\\.\\w)?\\(\\S+\\);)`, 'mg');
    let match;
    // tslint:disable-next-line:no-conditional-assignment
    while (match = regex.exec(content)) {
        inserts.push({
            pos: match.index + match[1].length,
            content: '/*@__PURE__*/',
        });
    }
    return inserts;
}
exports.purifyReplacements = purifyReplacements;
function purify(content) {
    const rawSource = new webpack_sources_1.RawSource(content);
    const replaceSource = new webpack_sources_1.ReplaceSource(rawSource, 'file.js');
    const inserts = purifyReplacements(content);
    inserts.forEach((insert) => {
        replaceSource.insert(insert.pos, insert.content);
    });
    return replaceSource.source();
}
exports.purify = purify;
function getMatches(str, regex, index) {
    let matches = [];
    let match;
    // tslint:disable-next-line:no-conditional-assignment
    while (match = regex.exec(str)) {
        matches = matches.concat(match[index].split(','));
    }
    return matches;
}
//# sourceMappingURL=data:application/json;base64,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