"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const ts = require("typescript");
function getFoldFileTransformer(program) {
    const checker = program.getTypeChecker();
    return (context) => {
        const transformer = (sf) => {
            const classes = findClassDeclarations(sf);
            const statements = findClassStaticPropertyAssignments(sf, checker, classes);
            const visitor = (node) => {
                // Check if node is a statement to be dropped.
                if (statements.find((st) => st.expressionStatement === node)) {
                    return undefined;
                }
                // Check if node is a class to add statements to.
                const clazz = classes.find((cl) => cl.classFunction === node);
                if (clazz) {
                    const functionExpression = node;
                    const newExpressions = clazz.statements.map((st) => ts.createStatement(st.expressionStatement.expression));
                    // Create a new body with all the original statements, plus new ones,
                    // plus return statement.
                    const newBody = ts.createBlock([
                        ...functionExpression.body.statements.slice(0, -1),
                        ...newExpressions,
                        ...functionExpression.body.statements.slice(-1),
                    ]);
                    const newNode = ts.createFunctionExpression(functionExpression.modifiers, functionExpression.asteriskToken, functionExpression.name, functionExpression.typeParameters, functionExpression.parameters, functionExpression.type, newBody);
                    // Replace node with modified one.
                    return ts.visitEachChild(newNode, visitor, context);
                }
                // Otherwise return node as is.
                return ts.visitEachChild(node, visitor, context);
            };
            return ts.visitNode(sf, visitor);
        };
        return transformer;
    };
}
exports.getFoldFileTransformer = getFoldFileTransformer;
function findClassDeclarations(node) {
    const classes = [];
    // Find all class declarations, build a ClassData for each.
    ts.forEachChild(node, (child) => {
        if (child.kind !== ts.SyntaxKind.VariableStatement) {
            return;
        }
        const varStmt = child;
        if (varStmt.declarationList.declarations.length > 1) {
            return;
        }
        const varDecl = varStmt.declarationList.declarations[0];
        if (varDecl.name.kind !== ts.SyntaxKind.Identifier) {
            return;
        }
        const name = varDecl.name.text;
        const expr = varDecl.initializer;
        if (!expr || expr.kind !== ts.SyntaxKind.ParenthesizedExpression) {
            return;
        }
        if (expr.expression.kind !== ts.SyntaxKind.CallExpression) {
            return;
        }
        const callExpr = expr.expression;
        if (callExpr.expression.kind !== ts.SyntaxKind.FunctionExpression) {
            return;
        }
        const fn = callExpr.expression;
        if (fn.body.statements.length < 2) {
            return;
        }
        if (fn.body.statements[0].kind !== ts.SyntaxKind.FunctionDeclaration) {
            return;
        }
        const innerFn = fn.body.statements[0];
        if (fn.body.statements[fn.body.statements.length - 1].kind !== ts.SyntaxKind.ReturnStatement) {
            return;
        }
        if (!innerFn.name || innerFn.name.kind !== ts.SyntaxKind.Identifier) {
            return;
        }
        if (innerFn.name.text !== name) {
            return;
        }
        classes.push({
            name,
            class: varDecl,
            classFunction: fn,
            statements: [],
        });
    });
    return classes;
}
function findClassStaticPropertyAssignments(node, checker, classes) {
    const statements = [];
    // Find each assignment outside of the declaration.
    ts.forEachChild(node, (child) => {
        if (child.kind !== ts.SyntaxKind.ExpressionStatement) {
            return;
        }
        const expressionStatement = child;
        if (expressionStatement.expression.kind !== ts.SyntaxKind.BinaryExpression) {
            return;
        }
        const binEx = expressionStatement.expression;
        if (binEx.left.kind !== ts.SyntaxKind.PropertyAccessExpression) {
            return;
        }
        const propAccess = binEx.left;
        if (propAccess.expression.kind !== ts.SyntaxKind.Identifier) {
            return;
        }
        const symbol = checker.getSymbolAtLocation(propAccess.expression);
        if (!symbol) {
            return;
        }
        const decls = symbol.declarations;
        if (decls === undefined || decls.length !== 1) {
            return;
        }
        const classIdx = classes
            .map((clazz) => clazz.class).indexOf(decls[0]);
        if (classIdx === -1) {
            return;
        }
        const hostClass = classes[classIdx];
        const statement = { expressionStatement, hostClass };
        hostClass.statements.push(statement);
        statements.push(statement);
    });
    return statements;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xhc3MtZm9sZC5qcyIsInNvdXJjZVJvb3QiOiIvVXNlcnMvaGFuc2wvU291cmNlcy9oYW5zbC9kZXZraXQvIiwic291cmNlcyI6WyJwYWNrYWdlcy9hbmd1bGFyX2RldmtpdC9idWlsZF9vcHRpbWl6ZXIvc3JjL3RyYW5zZm9ybXMvY2xhc3MtZm9sZC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBOzs7Ozs7R0FNRztBQUNILGlDQUFpQztBQWNqQyxnQ0FBdUMsT0FBbUI7SUFDeEQsTUFBTSxPQUFPLEdBQUcsT0FBTyxDQUFDLGNBQWMsRUFBRSxDQUFDO0lBRXpDLE1BQU0sQ0FBQyxDQUFDLE9BQWlDLEVBQWlDLEVBQUU7UUFFMUUsTUFBTSxXQUFXLEdBQWtDLENBQUMsRUFBaUIsRUFBRSxFQUFFO1lBRXZFLE1BQU0sT0FBTyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQzFDLE1BQU0sVUFBVSxHQUFHLGtDQUFrQyxDQUFDLEVBQUUsRUFBRSxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFFNUUsTUFBTSxPQUFPLEdBQWUsQ0FBQyxJQUFhLEVBQTJCLEVBQUU7Z0JBQ3JFLDhDQUE4QztnQkFDOUMsRUFBRSxDQUFDLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDLG1CQUFtQixLQUFLLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztvQkFDN0QsTUFBTSxDQUFDLFNBQVMsQ0FBQztnQkFDbkIsQ0FBQztnQkFFRCxpREFBaUQ7Z0JBQ2pELE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEtBQUssSUFBSSxDQUFDLENBQUM7Z0JBQzlELEVBQUUsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7b0JBQ1YsTUFBTSxrQkFBa0IsR0FBRyxJQUE2QixDQUFDO29CQUV6RCxNQUFNLGNBQWMsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQ2pELEVBQUUsQ0FBQyxlQUFlLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7b0JBRXpELHFFQUFxRTtvQkFDckUseUJBQXlCO29CQUN6QixNQUFNLE9BQU8sR0FBRyxFQUFFLENBQUMsV0FBVyxDQUFDO3dCQUM3QixHQUFHLGtCQUFrQixDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQzt3QkFDbEQsR0FBRyxjQUFjO3dCQUNqQixHQUFHLGtCQUFrQixDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO3FCQUNoRCxDQUFDLENBQUM7b0JBRUgsTUFBTSxPQUFPLEdBQUcsRUFBRSxDQUFDLHdCQUF3QixDQUN6QyxrQkFBa0IsQ0FBQyxTQUFTLEVBQzVCLGtCQUFrQixDQUFDLGFBQWEsRUFDaEMsa0JBQWtCLENBQUMsSUFBSSxFQUN2QixrQkFBa0IsQ0FBQyxjQUFjLEVBQ2pDLGtCQUFrQixDQUFDLFVBQVUsRUFDN0Isa0JBQWtCLENBQUMsSUFBSSxFQUN2QixPQUFPLENBQ1IsQ0FBQztvQkFFRixrQ0FBa0M7b0JBQ2xDLE1BQU0sQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLE9BQU8sRUFBRSxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQUM7Z0JBQ3RELENBQUM7Z0JBRUQsK0JBQStCO2dCQUMvQixNQUFNLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ25ELENBQUMsQ0FBQztZQUVGLE1BQU0sQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNuQyxDQUFDLENBQUM7UUFFRixNQUFNLENBQUMsV0FBVyxDQUFDO0lBQ3JCLENBQUMsQ0FBQztBQUNKLENBQUM7QUF2REQsd0RBdURDO0FBRUQsK0JBQStCLElBQWE7SUFDMUMsTUFBTSxPQUFPLEdBQWdCLEVBQUUsQ0FBQztJQUNoQywyREFBMkQ7SUFDM0QsRUFBRSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxLQUFLLEVBQUUsRUFBRTtRQUM5QixFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsSUFBSSxLQUFLLEVBQUUsQ0FBQyxVQUFVLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDO1lBQ25ELE1BQU0sQ0FBQztRQUNULENBQUM7UUFDRCxNQUFNLE9BQU8sR0FBRyxLQUE2QixDQUFDO1FBQzlDLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsWUFBWSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3BELE1BQU0sQ0FBQztRQUNULENBQUM7UUFDRCxNQUFNLE9BQU8sR0FBRyxPQUFPLENBQUMsZUFBZSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN4RCxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7WUFDbkQsTUFBTSxDQUFDO1FBQ1QsQ0FBQztRQUNELE1BQU0sSUFBSSxHQUFJLE9BQU8sQ0FBQyxJQUFzQixDQUFDLElBQUksQ0FBQztRQUNsRCxNQUFNLElBQUksR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO1FBQ2pDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQyxJQUFJLEtBQUssRUFBRSxDQUFDLFVBQVUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDLENBQUM7WUFDakUsTUFBTSxDQUFDO1FBQ1QsQ0FBQztRQUNELEVBQUUsQ0FBQyxDQUFFLElBQW1DLENBQUMsVUFBVSxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsVUFBVSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUM7WUFDMUYsTUFBTSxDQUFDO1FBQ1QsQ0FBQztRQUNELE1BQU0sUUFBUSxHQUFJLElBQW1DLENBQUMsVUFBK0IsQ0FBQztRQUN0RixFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsVUFBVSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUNsRSxNQUFNLENBQUM7UUFDVCxDQUFDO1FBQ0QsTUFBTSxFQUFFLEdBQUcsUUFBUSxDQUFDLFVBQW1DLENBQUM7UUFDeEQsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDbEMsTUFBTSxDQUFDO1FBQ1QsQ0FBQztRQUNELEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsVUFBVSxDQUFDLG1CQUFtQixDQUFDLENBQUMsQ0FBQztZQUNyRSxNQUFNLENBQUM7UUFDVCxDQUFDO1FBQ0QsTUFBTSxPQUFPLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUEyQixDQUFDO1FBQ2hFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLEtBQUssRUFBRSxDQUFDLFVBQVUsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDO1lBQzdGLE1BQU0sQ0FBQztRQUNULENBQUM7UUFDRCxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssRUFBRSxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1lBQ3BFLE1BQU0sQ0FBQztRQUNULENBQUM7UUFDRCxFQUFFLENBQUMsQ0FBRSxPQUFPLENBQUMsSUFBc0IsQ0FBQyxJQUFJLEtBQUssSUFBSSxDQUFDLENBQUMsQ0FBQztZQUNsRCxNQUFNLENBQUM7UUFDVCxDQUFDO1FBQ0QsT0FBTyxDQUFDLElBQUksQ0FBQztZQUNYLElBQUk7WUFDSixLQUFLLEVBQUUsT0FBTztZQUNkLGFBQWEsRUFBRSxFQUFFO1lBQ2pCLFVBQVUsRUFBRSxFQUFFO1NBQ2YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsT0FBTyxDQUFDO0FBQ2pCLENBQUM7QUFFRCw0Q0FDRSxJQUFhLEVBQ2IsT0FBdUIsRUFDdkIsT0FBb0I7SUFFcEIsTUFBTSxVQUFVLEdBQW9CLEVBQUUsQ0FBQztJQUV2QyxtREFBbUQ7SUFDbkQsRUFBRSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxLQUFLLEVBQUUsRUFBRTtRQUM5QixFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsSUFBSSxLQUFLLEVBQUUsQ0FBQyxVQUFVLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDO1lBQ3JELE1BQU0sQ0FBQztRQUNULENBQUM7UUFDRCxNQUFNLG1CQUFtQixHQUFHLEtBQStCLENBQUM7UUFDNUQsRUFBRSxDQUFDLENBQUMsbUJBQW1CLENBQUMsVUFBVSxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsVUFBVSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQztZQUMzRSxNQUFNLENBQUM7UUFDVCxDQUFDO1FBQ0QsTUFBTSxLQUFLLEdBQUcsbUJBQW1CLENBQUMsVUFBaUMsQ0FBQztRQUNwRSxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsVUFBVSxDQUFDLHdCQUF3QixDQUFDLENBQUMsQ0FBQztZQUMvRCxNQUFNLENBQUM7UUFDVCxDQUFDO1FBQ0QsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLElBQW1DLENBQUM7UUFDN0QsRUFBRSxDQUFDLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEtBQUssRUFBRSxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1lBQzVELE1BQU0sQ0FBQztRQUNULENBQUM7UUFFRCxNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsbUJBQW1CLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2xFLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztZQUNaLE1BQU0sQ0FBQztRQUNULENBQUM7UUFFRCxNQUFNLEtBQUssR0FBRyxNQUFNLENBQUMsWUFBWSxDQUFDO1FBQ2xDLEVBQUUsQ0FBQyxDQUFDLEtBQUssS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzlDLE1BQU0sQ0FBQztRQUNULENBQUM7UUFDRCxNQUFNLFFBQVEsR0FBRyxPQUFPO2FBQ3JCLEdBQUcsQ0FBQyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUEyQixDQUFDLENBQUM7UUFDM0UsRUFBRSxDQUFDLENBQUMsUUFBUSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNwQixNQUFNLENBQUM7UUFDVCxDQUFDO1FBQ0QsTUFBTSxTQUFTLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3BDLE1BQU0sU0FBUyxHQUFrQixFQUFFLG1CQUFtQixFQUFFLFNBQVMsRUFBRSxDQUFDO1FBRXBFLFNBQVMsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3JDLFVBQVUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDN0IsQ0FBQyxDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsVUFBVSxDQUFDO0FBQ3BCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIEBsaWNlbnNlXG4gKiBDb3B5cmlnaHQgR29vZ2xlIEluYy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuaW8vbGljZW5zZVxuICovXG5pbXBvcnQgKiBhcyB0cyBmcm9tICd0eXBlc2NyaXB0JztcblxuaW50ZXJmYWNlIENsYXNzRGF0YSB7XG4gIG5hbWU6IHN0cmluZztcbiAgY2xhc3M6IHRzLlZhcmlhYmxlRGVjbGFyYXRpb247XG4gIGNsYXNzRnVuY3Rpb246IHRzLkZ1bmN0aW9uRXhwcmVzc2lvbjtcbiAgc3RhdGVtZW50czogU3RhdGVtZW50RGF0YVtdO1xufVxuXG5pbnRlcmZhY2UgU3RhdGVtZW50RGF0YSB7XG4gIGV4cHJlc3Npb25TdGF0ZW1lbnQ6IHRzLkV4cHJlc3Npb25TdGF0ZW1lbnQ7XG4gIGhvc3RDbGFzczogQ2xhc3NEYXRhO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gZ2V0Rm9sZEZpbGVUcmFuc2Zvcm1lcihwcm9ncmFtOiB0cy5Qcm9ncmFtKTogdHMuVHJhbnNmb3JtZXJGYWN0b3J5PHRzLlNvdXJjZUZpbGU+IHtcbiAgY29uc3QgY2hlY2tlciA9IHByb2dyYW0uZ2V0VHlwZUNoZWNrZXIoKTtcblxuICByZXR1cm4gKGNvbnRleHQ6IHRzLlRyYW5zZm9ybWF0aW9uQ29udGV4dCk6IHRzLlRyYW5zZm9ybWVyPHRzLlNvdXJjZUZpbGU+ID0+IHtcblxuICAgIGNvbnN0IHRyYW5zZm9ybWVyOiB0cy5UcmFuc2Zvcm1lcjx0cy5Tb3VyY2VGaWxlPiA9IChzZjogdHMuU291cmNlRmlsZSkgPT4ge1xuXG4gICAgICBjb25zdCBjbGFzc2VzID0gZmluZENsYXNzRGVjbGFyYXRpb25zKHNmKTtcbiAgICAgIGNvbnN0IHN0YXRlbWVudHMgPSBmaW5kQ2xhc3NTdGF0aWNQcm9wZXJ0eUFzc2lnbm1lbnRzKHNmLCBjaGVja2VyLCBjbGFzc2VzKTtcblxuICAgICAgY29uc3QgdmlzaXRvcjogdHMuVmlzaXRvciA9IChub2RlOiB0cy5Ob2RlKTogdHMuVmlzaXRSZXN1bHQ8dHMuTm9kZT4gPT4ge1xuICAgICAgICAvLyBDaGVjayBpZiBub2RlIGlzIGEgc3RhdGVtZW50IHRvIGJlIGRyb3BwZWQuXG4gICAgICAgIGlmIChzdGF0ZW1lbnRzLmZpbmQoKHN0KSA9PiBzdC5leHByZXNzaW9uU3RhdGVtZW50ID09PSBub2RlKSkge1xuICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBDaGVjayBpZiBub2RlIGlzIGEgY2xhc3MgdG8gYWRkIHN0YXRlbWVudHMgdG8uXG4gICAgICAgIGNvbnN0IGNsYXp6ID0gY2xhc3Nlcy5maW5kKChjbCkgPT4gY2wuY2xhc3NGdW5jdGlvbiA9PT0gbm9kZSk7XG4gICAgICAgIGlmIChjbGF6eikge1xuICAgICAgICAgIGNvbnN0IGZ1bmN0aW9uRXhwcmVzc2lvbiA9IG5vZGUgYXMgdHMuRnVuY3Rpb25FeHByZXNzaW9uO1xuXG4gICAgICAgICAgY29uc3QgbmV3RXhwcmVzc2lvbnMgPSBjbGF6ei5zdGF0ZW1lbnRzLm1hcCgoc3QpID0+XG4gICAgICAgICAgICB0cy5jcmVhdGVTdGF0ZW1lbnQoc3QuZXhwcmVzc2lvblN0YXRlbWVudC5leHByZXNzaW9uKSk7XG5cbiAgICAgICAgICAvLyBDcmVhdGUgYSBuZXcgYm9keSB3aXRoIGFsbCB0aGUgb3JpZ2luYWwgc3RhdGVtZW50cywgcGx1cyBuZXcgb25lcyxcbiAgICAgICAgICAvLyBwbHVzIHJldHVybiBzdGF0ZW1lbnQuXG4gICAgICAgICAgY29uc3QgbmV3Qm9keSA9IHRzLmNyZWF0ZUJsb2NrKFtcbiAgICAgICAgICAgIC4uLmZ1bmN0aW9uRXhwcmVzc2lvbi5ib2R5LnN0YXRlbWVudHMuc2xpY2UoMCwgLTEpLFxuICAgICAgICAgICAgLi4ubmV3RXhwcmVzc2lvbnMsXG4gICAgICAgICAgICAuLi5mdW5jdGlvbkV4cHJlc3Npb24uYm9keS5zdGF0ZW1lbnRzLnNsaWNlKC0xKSxcbiAgICAgICAgICBdKTtcblxuICAgICAgICAgIGNvbnN0IG5ld05vZGUgPSB0cy5jcmVhdGVGdW5jdGlvbkV4cHJlc3Npb24oXG4gICAgICAgICAgICBmdW5jdGlvbkV4cHJlc3Npb24ubW9kaWZpZXJzLFxuICAgICAgICAgICAgZnVuY3Rpb25FeHByZXNzaW9uLmFzdGVyaXNrVG9rZW4sXG4gICAgICAgICAgICBmdW5jdGlvbkV4cHJlc3Npb24ubmFtZSxcbiAgICAgICAgICAgIGZ1bmN0aW9uRXhwcmVzc2lvbi50eXBlUGFyYW1ldGVycyxcbiAgICAgICAgICAgIGZ1bmN0aW9uRXhwcmVzc2lvbi5wYXJhbWV0ZXJzLFxuICAgICAgICAgICAgZnVuY3Rpb25FeHByZXNzaW9uLnR5cGUsXG4gICAgICAgICAgICBuZXdCb2R5LFxuICAgICAgICAgICk7XG5cbiAgICAgICAgICAvLyBSZXBsYWNlIG5vZGUgd2l0aCBtb2RpZmllZCBvbmUuXG4gICAgICAgICAgcmV0dXJuIHRzLnZpc2l0RWFjaENoaWxkKG5ld05vZGUsIHZpc2l0b3IsIGNvbnRleHQpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gT3RoZXJ3aXNlIHJldHVybiBub2RlIGFzIGlzLlxuICAgICAgICByZXR1cm4gdHMudmlzaXRFYWNoQ2hpbGQobm9kZSwgdmlzaXRvciwgY29udGV4dCk7XG4gICAgICB9O1xuXG4gICAgICByZXR1cm4gdHMudmlzaXROb2RlKHNmLCB2aXNpdG9yKTtcbiAgICB9O1xuXG4gICAgcmV0dXJuIHRyYW5zZm9ybWVyO1xuICB9O1xufVxuXG5mdW5jdGlvbiBmaW5kQ2xhc3NEZWNsYXJhdGlvbnMobm9kZTogdHMuTm9kZSk6IENsYXNzRGF0YVtdIHtcbiAgY29uc3QgY2xhc3NlczogQ2xhc3NEYXRhW10gPSBbXTtcbiAgLy8gRmluZCBhbGwgY2xhc3MgZGVjbGFyYXRpb25zLCBidWlsZCBhIENsYXNzRGF0YSBmb3IgZWFjaC5cbiAgdHMuZm9yRWFjaENoaWxkKG5vZGUsIChjaGlsZCkgPT4ge1xuICAgIGlmIChjaGlsZC5raW5kICE9PSB0cy5TeW50YXhLaW5kLlZhcmlhYmxlU3RhdGVtZW50KSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIGNvbnN0IHZhclN0bXQgPSBjaGlsZCBhcyB0cy5WYXJpYWJsZVN0YXRlbWVudDtcbiAgICBpZiAodmFyU3RtdC5kZWNsYXJhdGlvbkxpc3QuZGVjbGFyYXRpb25zLmxlbmd0aCA+IDEpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgY29uc3QgdmFyRGVjbCA9IHZhclN0bXQuZGVjbGFyYXRpb25MaXN0LmRlY2xhcmF0aW9uc1swXTtcbiAgICBpZiAodmFyRGVjbC5uYW1lLmtpbmQgIT09IHRzLlN5bnRheEtpbmQuSWRlbnRpZmllcikge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBjb25zdCBuYW1lID0gKHZhckRlY2wubmFtZSBhcyB0cy5JZGVudGlmaWVyKS50ZXh0O1xuICAgIGNvbnN0IGV4cHIgPSB2YXJEZWNsLmluaXRpYWxpemVyO1xuICAgIGlmICghZXhwciB8fCBleHByLmtpbmQgIT09IHRzLlN5bnRheEtpbmQuUGFyZW50aGVzaXplZEV4cHJlc3Npb24pIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgaWYgKChleHByIGFzIHRzLlBhcmVudGhlc2l6ZWRFeHByZXNzaW9uKS5leHByZXNzaW9uLmtpbmQgIT09IHRzLlN5bnRheEtpbmQuQ2FsbEV4cHJlc3Npb24pIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgY29uc3QgY2FsbEV4cHIgPSAoZXhwciBhcyB0cy5QYXJlbnRoZXNpemVkRXhwcmVzc2lvbikuZXhwcmVzc2lvbiBhcyB0cy5DYWxsRXhwcmVzc2lvbjtcbiAgICBpZiAoY2FsbEV4cHIuZXhwcmVzc2lvbi5raW5kICE9PSB0cy5TeW50YXhLaW5kLkZ1bmN0aW9uRXhwcmVzc2lvbikge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBjb25zdCBmbiA9IGNhbGxFeHByLmV4cHJlc3Npb24gYXMgdHMuRnVuY3Rpb25FeHByZXNzaW9uO1xuICAgIGlmIChmbi5ib2R5LnN0YXRlbWVudHMubGVuZ3RoIDwgMikge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBpZiAoZm4uYm9keS5zdGF0ZW1lbnRzWzBdLmtpbmQgIT09IHRzLlN5bnRheEtpbmQuRnVuY3Rpb25EZWNsYXJhdGlvbikge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBjb25zdCBpbm5lckZuID0gZm4uYm9keS5zdGF0ZW1lbnRzWzBdIGFzIHRzLkZ1bmN0aW9uRGVjbGFyYXRpb247XG4gICAgaWYgKGZuLmJvZHkuc3RhdGVtZW50c1tmbi5ib2R5LnN0YXRlbWVudHMubGVuZ3RoIC0gMV0ua2luZCAhPT0gdHMuU3ludGF4S2luZC5SZXR1cm5TdGF0ZW1lbnQpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgaWYgKCFpbm5lckZuLm5hbWUgfHwgaW5uZXJGbi5uYW1lLmtpbmQgIT09IHRzLlN5bnRheEtpbmQuSWRlbnRpZmllcikge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBpZiAoKGlubmVyRm4ubmFtZSBhcyB0cy5JZGVudGlmaWVyKS50ZXh0ICE9PSBuYW1lKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIGNsYXNzZXMucHVzaCh7XG4gICAgICBuYW1lLFxuICAgICAgY2xhc3M6IHZhckRlY2wsXG4gICAgICBjbGFzc0Z1bmN0aW9uOiBmbixcbiAgICAgIHN0YXRlbWVudHM6IFtdLFxuICAgIH0pO1xuICB9KTtcblxuICByZXR1cm4gY2xhc3Nlcztcbn1cblxuZnVuY3Rpb24gZmluZENsYXNzU3RhdGljUHJvcGVydHlBc3NpZ25tZW50cyhcbiAgbm9kZTogdHMuTm9kZSxcbiAgY2hlY2tlcjogdHMuVHlwZUNoZWNrZXIsXG4gIGNsYXNzZXM6IENsYXNzRGF0YVtdKTogU3RhdGVtZW50RGF0YVtdIHtcblxuICBjb25zdCBzdGF0ZW1lbnRzOiBTdGF0ZW1lbnREYXRhW10gPSBbXTtcblxuICAvLyBGaW5kIGVhY2ggYXNzaWdubWVudCBvdXRzaWRlIG9mIHRoZSBkZWNsYXJhdGlvbi5cbiAgdHMuZm9yRWFjaENoaWxkKG5vZGUsIChjaGlsZCkgPT4ge1xuICAgIGlmIChjaGlsZC5raW5kICE9PSB0cy5TeW50YXhLaW5kLkV4cHJlc3Npb25TdGF0ZW1lbnQpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgY29uc3QgZXhwcmVzc2lvblN0YXRlbWVudCA9IGNoaWxkIGFzIHRzLkV4cHJlc3Npb25TdGF0ZW1lbnQ7XG4gICAgaWYgKGV4cHJlc3Npb25TdGF0ZW1lbnQuZXhwcmVzc2lvbi5raW5kICE9PSB0cy5TeW50YXhLaW5kLkJpbmFyeUV4cHJlc3Npb24pIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgY29uc3QgYmluRXggPSBleHByZXNzaW9uU3RhdGVtZW50LmV4cHJlc3Npb24gYXMgdHMuQmluYXJ5RXhwcmVzc2lvbjtcbiAgICBpZiAoYmluRXgubGVmdC5raW5kICE9PSB0cy5TeW50YXhLaW5kLlByb3BlcnR5QWNjZXNzRXhwcmVzc2lvbikge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBjb25zdCBwcm9wQWNjZXNzID0gYmluRXgubGVmdCBhcyB0cy5Qcm9wZXJ0eUFjY2Vzc0V4cHJlc3Npb247XG4gICAgaWYgKHByb3BBY2Nlc3MuZXhwcmVzc2lvbi5raW5kICE9PSB0cy5TeW50YXhLaW5kLklkZW50aWZpZXIpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBjb25zdCBzeW1ib2wgPSBjaGVja2VyLmdldFN5bWJvbEF0TG9jYXRpb24ocHJvcEFjY2Vzcy5leHByZXNzaW9uKTtcbiAgICBpZiAoIXN5bWJvbCkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IGRlY2xzID0gc3ltYm9sLmRlY2xhcmF0aW9ucztcbiAgICBpZiAoZGVjbHMgPT09IHVuZGVmaW5lZCB8fCBkZWNscy5sZW5ndGggIT09IDEpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgY29uc3QgY2xhc3NJZHggPSBjbGFzc2VzXG4gICAgICAubWFwKChjbGF6eikgPT4gY2xhenouY2xhc3MpLmluZGV4T2YoZGVjbHNbMF0gYXMgdHMuVmFyaWFibGVEZWNsYXJhdGlvbik7XG4gICAgaWYgKGNsYXNzSWR4ID09PSAtMSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBjb25zdCBob3N0Q2xhc3MgPSBjbGFzc2VzW2NsYXNzSWR4XTtcbiAgICBjb25zdCBzdGF0ZW1lbnQ6IFN0YXRlbWVudERhdGEgPSB7IGV4cHJlc3Npb25TdGF0ZW1lbnQsIGhvc3RDbGFzcyB9O1xuXG4gICAgaG9zdENsYXNzLnN0YXRlbWVudHMucHVzaChzdGF0ZW1lbnQpO1xuICAgIHN0YXRlbWVudHMucHVzaChzdGF0ZW1lbnQpO1xuICB9KTtcblxuICByZXR1cm4gc3RhdGVtZW50cztcbn1cbiJdfQ==