"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const ts = require("typescript");
function testPrefixClasses(content) {
    const exportVarSetter = /(?:export )?(?:var|const)\s+(\S+)\s*=\s*/;
    const multiLineComment = /\s*(?:\/\*[\s\S]*?\*\/)?\s*/;
    const newLine = /\s*\r?\n\s*/;
    const regexes = [
        [
            /^/,
            exportVarSetter, multiLineComment,
            /\(/, multiLineComment,
            /\s*function \(\) {/, newLine,
            multiLineComment,
            /function \1\([^\)]*\) \{/, newLine,
        ],
        [
            /^/,
            exportVarSetter, multiLineComment,
            /\(/, multiLineComment,
            /\s*function \(_super\) {/, newLine,
            /\w*__extends\(\w+, _super\);/,
        ],
    ].map(arr => new RegExp(arr.map(x => x.source).join(''), 'm'));
    return regexes.some((regex) => regex.test(content));
}
exports.testPrefixClasses = testPrefixClasses;
const superParameterName = '_super';
const extendsHelperName = '__extends';
function getPrefixClassesTransformer() {
    return (context) => {
        const transformer = (sf) => {
            const pureFunctionComment = '@__PURE__';
            const visitor = (node) => {
                // Add pure comment to downleveled classes.
                if (ts.isVariableStatement(node) && isDownleveledClass(node)) {
                    const varDecl = node.declarationList.declarations[0];
                    const varInitializer = varDecl.initializer;
                    // Update node with the pure comment before the variable declaration initializer.
                    const newNode = ts.updateVariableStatement(node, node.modifiers, ts.updateVariableDeclarationList(node.declarationList, [
                        ts.updateVariableDeclaration(varDecl, varDecl.name, varDecl.type, ts.addSyntheticLeadingComment(varInitializer, ts.SyntaxKind.MultiLineCommentTrivia, pureFunctionComment, false)),
                    ]));
                    // Replace node with modified one.
                    return ts.visitEachChild(newNode, visitor, context);
                }
                // Otherwise return node as is.
                return ts.visitEachChild(node, visitor, context);
            };
            return ts.visitEachChild(sf, visitor, context);
        };
        return transformer;
    };
}
exports.getPrefixClassesTransformer = getPrefixClassesTransformer;
// Determine if a node matched the structure of a downleveled TS class.
function isDownleveledClass(node) {
    if (!ts.isVariableStatement(node)) {
        return false;
    }
    if (node.declarationList.declarations.length !== 1) {
        return false;
    }
    const variableDeclaration = node.declarationList.declarations[0];
    if (!ts.isIdentifier(variableDeclaration.name)
        || !variableDeclaration.initializer) {
        return false;
    }
    let potentialClass = variableDeclaration.initializer;
    // TS 2.3 has an unwrapped class IIFE
    // TS 2.4 uses a function expression wrapper
    // TS 2.5 uses an arrow function wrapper
    if (ts.isParenthesizedExpression(potentialClass)) {
        potentialClass = potentialClass.expression;
    }
    if (!ts.isCallExpression(potentialClass) || potentialClass.arguments.length > 1) {
        return false;
    }
    let wrapperBody;
    if (ts.isFunctionExpression(potentialClass.expression)) {
        wrapperBody = potentialClass.expression.body;
    }
    else if (ts.isArrowFunction(potentialClass.expression)
        && ts.isBlock(potentialClass.expression.body)) {
        wrapperBody = potentialClass.expression.body;
    }
    else {
        return false;
    }
    if (wrapperBody.statements.length === 0) {
        return false;
    }
    const functionExpression = potentialClass.expression;
    const functionStatements = wrapperBody.statements;
    // need a minimum of two for a function declaration and return statement
    if (functionStatements.length < 2) {
        return false;
    }
    // The variable name should be the class name.
    const className = variableDeclaration.name.text;
    const firstStatement = functionStatements[0];
    // find return statement - may not be last statement
    let returnStatement;
    for (let i = functionStatements.length - 1; i > 0; i--) {
        if (ts.isReturnStatement(functionStatements[i])) {
            returnStatement = functionStatements[i];
            break;
        }
    }
    if (returnStatement == undefined
        || returnStatement.expression == undefined
        || !ts.isIdentifier(returnStatement.expression)) {
        return false;
    }
    if (functionExpression.parameters.length === 0) {
        // potential non-extended class or wrapped es2015 class
        return (ts.isFunctionDeclaration(firstStatement) || ts.isClassDeclaration(firstStatement))
            && firstStatement.name !== undefined
            && firstStatement.name.text === className
            && returnStatement.expression.text === firstStatement.name.text;
    }
    else if (functionExpression.parameters.length !== 1) {
        return false;
    }
    // Potential extended class
    const functionParameter = functionExpression.parameters[0];
    if (!ts.isIdentifier(functionParameter.name)
        || functionParameter.name.text !== superParameterName) {
        return false;
    }
    if (functionStatements.length < 3) {
        return false;
    }
    if (!ts.isExpressionStatement(firstStatement)
        || !ts.isCallExpression(firstStatement.expression)) {
        return false;
    }
    const extendCallExpression = firstStatement.expression;
    if (!ts.isIdentifier(extendCallExpression.expression)
        || !extendCallExpression.expression.text.endsWith(extendsHelperName)) {
        return false;
    }
    if (extendCallExpression.arguments.length === 0) {
        return false;
    }
    const lastArgument = extendCallExpression.arguments[extendCallExpression.arguments.length - 1];
    if (!ts.isIdentifier(lastArgument) || lastArgument.text !== functionParameter.name.text) {
        return false;
    }
    const secondStatement = functionStatements[1];
    return ts.isFunctionDeclaration(secondStatement)
        && secondStatement.name !== undefined
        && className.endsWith(secondStatement.name.text)
        && returnStatement.expression.text === secondStatement.name.text;
}
//# sourceMappingURL=data:application/json;base64,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